/****************************************************************************** 
 *
 * File:        smfcheck.c
 * Version:     $Id: smfcheck.c,v 1.3 1995/08/06 18:39:51 burgaard Exp $
 *              $Version: 0.5$
 *
 * Purpose:     Standard MIDI File Check.
 *
 * Project:     MIDI/Sequencer library.
 *              Roland MPU-401 Device driver.
 * Authors:     Kim Burgaard, <burgaard@daimi.aau.dk>
 * Copyrights:  Copyright (c) 1994 Kim Burgaard.
 *
 *      This package is free software; you can redistribute it and/or modify it
 *      under the terms of the GNU General Public License as published by the
 *      Free Software Foundation; either version 2, or (at your option) any
 *      later version.
 *
 *      This package is distributed in the hope that it will be useful, but
 *      WITHOUT ANY WARRANTY; without even the implied warranty of
 *      MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General
 *      Public License for more details.
 *
 *      You should have received a copy of the GNU General Public License along
 *      with this program; see the file COPYING. If not, write to the Free
 *      Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 ******************************************************************************/

/*** INCLUDES & DEFINES *******************************************************/

#include <errno.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include "miditypes.h"
#include "midiqueue.h"
#include "midiprint.h"
#include "midifile.h"

/*** GLOBAL VARIABLES *********************************************************/

static char verbose = 0;
static midi_file midifile;
static midi_file_timing timing;
static midi_file_data queues;

/*** ARGUMENT PARSING *********************************************************/

void usage(int exitcode)
{
  printf("Standard MIDI File Check version 0.5\n");
  printf("Copyright (c) 1994 Kim Burgaard\n");
  printf("\n");
  printf("Usage: smfcheck [options] filename(s) ...\n");
  printf("\n");
  printf("   -v --verbose   Verbose output to console.\n");
  printf("   -h --help      This help text.\n");
  exit(exitcode);
}

inline char isfilename(char *s)
{
  return  ( s && strlen(s) > 0 && s[0] != '-' );
}

inline char isoption(char *s)
{
  return ( s && strlen(s) > 0 && s[0] == '-' ); 
}

static int
parsearg(int argc, char *argv[])
{
  int i = 0;
  int fc = 0;
  if (argc < 2) usage(1);

  i = 1;
  while (i < argc)
    {
      if ( isoption(argv[i]) )
	{
	  if ( !strcmp("-v", argv[i]) || !strcmp("--verbose", argv[i]) )
	    {
	      verbose++;
	      if (verbose > MIDI_PRINT_ALL)
		fprintf(stderr, "Repeating %s more than %d times will have no effect.\n",
			argv[i], MIDI_PRINT_ALL);
	    }
	  else if ( !strcmp("-h", argv[i]) || !strcmp("--help", argv[i]) )
	    {
	      usage(0);
	    } 
	  else
	    {
	      fprintf(stderr, "Unknown option `%s' :\n\n", argv[i]);
	      fprintf(stderr, "Try `smfcheck --help' for more information\n");
	      exit(-1);
	    };
	}
      else
	{
	  fc++;
	};
      i++;
    };
  
  if (!fc)
    { 
      fprintf(stderr, "*** No files to process ***\n\n");
      usage(1);
    };

  return 0;
}

/*** MIDI FILE HANDLING *******************************************************/

void check(char *source)
{
  char * name = NULL;
  int h = 0;
  int m = 0;
  int s = 0;

  if ( !(midifile.file = fopen(source, "rb")) )
    {
      fprintf(stderr, "Could not open `%s': ", source);
      perror("");
      return;
    };

  if ( (name = strrchr(source, '/')) )  name++;
  else name = source;

  midifile.name = name;
  midifile.title = NULL;
  midifile.copyright = NULL;

  timing.newdivision = 0;
  timing.time = 0.0;

  midi_queue_flush(&queues.voice);
  midi_queue_flush(&queues.sysex);
  midi_queue_flush(&queues.meta);

  queues.dovoice = queues.dosysex = queues.dometa = 0;

  if ( !midi_load(&midifile, &timing, &queues, verbose) )
    {
      h = timing.time / (60*60);
      m = (timing.time / 60) - (h*60);
      s = ((byte)timing.time % 60);

      if (h > 0) printf("%02hd:", h);
      printf("%02hd:%02hd  ", m, s);
      
      if (midifile.title && midifile.copyright)
	{
	  printf("[ %s, %s, %s ]\n", midifile.name, midifile.title, midifile.copyright);
	  free(midifile.title);
	  free(midifile.copyright);
	}
      else if (midifile.copyright)
	{
	  printf("[ %s, %s ]\n", midifile.name, midifile.copyright);
	  free(midifile.copyright);
	}
      else if (midifile.title)
	{
	  printf("[ %s, %s ]\n", midifile.name, midifile.title);
	  free(midifile.title);
	}
      else
	printf("[ %s ]\n", midifile.name);
      
      midifile.title = midifile.copyright = NULL;
    }
  else if (ferror(midifile.file))
    {
      fprintf(stderr, "Error while reading `%s': ", source);
      perror("");
    };

  /* play it safe ... they are/should be empty */
  midi_queue_flush(&queues.voice);
  midi_queue_flush(&queues.sysex);
  midi_queue_flush(&queues.meta);

  fclose(midifile.file);
}

/*** MAIN *********************************************************************/

int main(int argc, char *argv[])
{
  int i = 0;
  if ( parsearg(argc, argv) ) return 1;

  midi_queue_reset(&queues.voice);
  midi_queue_reset(&queues.sysex);
  midi_queue_reset(&queues.meta);

  for (i = 1; i < argc; i++) if (isfilename(argv[i])) check(argv[i]);

  return 0;
}

/*** END OF FILE **************************************************************/
