/*
 * 3Dc.h
 *
 * Definitions file for 3Dc
 */
/*

    3Dc, a game of 3-Dimensional Chess
    Copyright (C) 1995  Paul Hicks

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

    E-Mail: P.Hicks@net-cs.ucd.ie
*/
#ifndef __3Dc_H
#define __3Dc_H

#include <X11/X.h> /* For Pixmap */
#include <X11/Intrinsic.h> /* For all other X stuff */
/*****************************************************************************
 * A miscellany of useful tidbits
 */
#ifndef Boolean
#define Boolean int
#endif Boolean
#ifndef True
#define True 1
#define False 0
#endif True
#ifndef NULL
#define NULL 0
#endif NULL
#ifndef ABS
#define ABS(a) ((a) < 0 ? -(a) : (a))
#endif /* ABS */
#ifndef MAX
#define MAX(a,b) ((a) < (b) ? (b) : (a))
#define MIN(a,b) ((a) > (b) ? (b) : (a))
#endif /* MAX */

/* Returns from pieceMayMove */
#define Castle 2
#define EnPassant 3
#define Promote 4

typedef enum
{
  king, queen, bishop, knight, rook,
  prince, princess, abbey, cannon, galley,
  pawn, none
} Title;

typedef enum
{
  white, black
} Colour;

typedef struct
{
  unsigned int xFile :3,
               yRank :3,
               zLevel :2;
} Coord;

typedef int File;
typedef int Rank;
typedef int Level;

/*
 * End miscellany
 ****************************************************************************/

/*****************************************************************************
 * Piece definitions for all pieces
 */
typedef struct
{
  Coord xyzPos;
  Colour bwSide;
  Title nName;
  Boolean bVisible;
  Boolean bHasMoved; /* For king, rook, pawn only */
  Boolean bInCheck; /* For king only */
} Piece;

extern void pieceDisplay(Piece *, const Boolean);
extern Boolean pieceMayMove(Piece *, const File, const Rank, const Level);
extern Boolean pieceMove(Piece *, const File, const Rank, const Level);
extern Boolean pieceUndo(void);
extern Piece *pieceThreatened(Piece *, const File, const Rank, const Level);
extern Piece *pieceNew(const Title, const File, const Rank, const Level,
                       const Colour);
extern void pieceDelete(Piece *);
extern void piecePromote(Piece *);
/*
 * End piece definitions
 ****************************************************************************/

/*****************************************************************************
 * The move-stack (for undos, checking for en passant, etc)
 */

/* This structure is enough to rebuild all moves, but it should
 * be extended to treat Castling, En Passant, and Promotion specially */
typedef struct
{
  Coord xyzBefore;
  Coord xyzAfter;
  Piece *pVictim;
  Boolean bHadMoved; /*  Status of bHasMoved before move.  Relevant to Kings, Rooks, and Pawns */
} Move;

struct stack_el
{
  Move *mvt;
  struct stack_el *below;
};

typedef struct
{
  int nSize;
  struct stack_el *top;
} stack;

extern void stackPush(stack *, const Move *);
extern Move *stackPop(stack *);
extern Move *stackPeek(stack *, int); /* Returns move n places down stack.  Do not free! */

extern stack *findAllMoves(const Piece *);

extern stack *MoveStack;
/*
 * End of the move stack
 ****************************************************************************/
/*****************************************************************************
 * Platform-dependent graphics info
 */
typedef struct
{
  Pixmap face[2][11]; /* The pixmap for the pieces */
  Pixmap mask[2][11]; /* The mask for the pieces */
  Pixel blackPixel, whitePixel, greyPixel; /* Colours */
  Widget mainWindow; /* Main widget, and playing squares */
  Widget remark;      /* Widget for the status line */
  Widget undo;    /* Buttons in mainWindow */
} GfxInfo;

/*
 * End of graphics info
 ****************************************************************************/

/****************************************************************************/
extern Piece *Board[3][8][8];
extern Piece *Muster[2][48]; /* Database of all pieces */
extern Colour bwToMove;
#include "3DcErr.h"

/* And finally the function prototypes for the game itself */
extern void Init3Dc(void);
extern int idx(const Title, const int);

/* Interface stuff */
extern GfxInfo gfxInfo;
extern int Init3DcGFX(int, char **);
extern void Quit3DcGFX(void); /* IF-defined */
extern void DrawBoard(void); /* IF-defined */
extern void undoMove(Widget, XtPointer, XtPointer);

/* ComputerPlay stuff */
extern Colour AutoPlay;
extern int DontMove;
extern Move *genMove(const Colour);

/* Net stuff */
extern int SetupClient(const char *, const int);
extern int SetupServer(const int);
extern int TCP_FD, SERVER, CLIENT;

#endif /* __3Dc_h */
