/* $Id: macro.c,v 30000.23 1993/05/25 00:57:32 kkeys Exp $ */
/******************************************************************
 * Copyright 1993 by Ken Keys.
 * Permission is granted to obtain and redistribute this code freely.
 * All redistributions must contain this header.  You may modify this
 * software, but any redistributions of modified code must be clearly
 * marked as having been modified.
 ******************************************************************/


/**********************************************
 * Fugue macro package                        *
 *                                            *
 * Macros, hooks, triggers, hilites and gags  *
 * are all processed here.                    *
 **********************************************/

#include <ctype.h>
#include "port.h"
#include "dstring.h"
#include "tf.h"
#include "util.h"
#include "search.h"
#include "world.h"
#include "macro.h"
#include "keyboard.h"
#include "expand.h"
#include "socket.h"
#include "command1.h"

static int     FDECL(macro_match,(Macro *spec, Macro *macro,
               int FDECL((*cmp),(char *s1, char *s2))));
static Macro  *FDECL(match_exact,(int hook, char *str, int flags));
static Macro  *FDECL(find_match,(int hook, char *text));
static int     FDECL(list_defs,(TFILE *file, Macro *spec, int abbrev));
static String *FDECL(escape,(char *src));
static int     FDECL(parse_hook,(char *args));
static int     FDECL(run_match,(Macro *macro, char *text));
static char   *FDECL(world_subs,(char *src));
static String *FDECL(hook_name,(int hook));

void   NDECL(init_macros);
Macro *FDECL(find_macro,(char *name));
Macro *FDECL(new_macro,(char *name, char *trig, char *bind, int hook,
       char *hargs, char *body, World *world, int pri, int prob,
       int attr, int shots, int invis));
void   FDECL(add_macro,(Macro *macro));
int    FDECL(add_hook,(char *proto, char *body));
int    FDECL(install_bind,(Macro *spec));
int    FDECL(do_add,(Macro *spec));
int    FDECL(do_edit,(Macro *spec));
void   NDECL(nuke_dead_macros);
void   FDECL(nuke_macro,(Macro *macro));
void   FDECL(kill_macro,(Macro *macro));
int    FDECL(remove_macro,(char *args, int attr, int byhook));
int    FDECL(purge_macro,(Macro *spec));
int    FDECL(remove_by_number,(char *args));
void   FDECL(remove_world_macros,(World *w));
int    FDECL(save_macros,(char *args));
int    FDECL(list_macros,(char *args));
int    FDECL(do_macro,(Macro *macro, char *args));
short  VDECL(do_hook,(int hook, char *fmt, char *argfmt, ...));
char  *FDECL(macro_body,(char *name));
short  FDECL(check_trigger,(char *text));

#define HASH_SIZE 197

static List maclist[1];                /* complete macro list */
static List triglist[1];               /* head of trigger macro list */
static List hooklist[1];               /* head of hook macro list */
static HashTable macro_table[1];       /* macros hashed by name */
static World NoWorld, AnyWorld;        /* explicit "no" and "don't care" */
static int mnum = 1;                   /* macro ID number */
static Macro *dead_macros = NULL;      /* head of dead macro list */

/* It is IMPORTANT that these be in the same order as enum Hooks */
static char *hook_table[] = {
  "ACTIVITY",   
  "BACKGROUND",
  "BAMF",
  "CONFAIL",
  "CONFLICT",
  "CONNECT",
  "DISCONNECT",
  "HISTORY",
  "KILL",
  "LOAD",
  "LOADFAIL",
  "LOG",
  "LOGIN",
  "MAIL",
  "MORE",
  "PENDING",
  "PROCESS",
  "REDEF",
  "RESIZE",
  "RESUME",
  "SEND",
  "SHADOW",
  "SHELL",
  "WORLD"
};

#define macmatch(spec, macro) macro_match((spec), (macro), smatch)
#define maccmp(spec, macro)   macro_match((spec), (macro), cstrcmp)

/* These macros allow easy sharing of trigger and hook code. The address and
 * dereference is necessary to allow the expression to be used portably as
 * the Left Hand Side of an assignment.
 */
#define Mac(Node)   ((Macro *)(Node->data))
#define Pattern(Node)   (*(hook ? &Mac(Node)->hargs : &Mac(Node)->trig))
#define Flag(mac)       ( (hook ? Mac(node)->hook   : (int)Mac(node)->attr))
    /* Flag() can't be lhs */


void init_macros()
{
    init_hashtable(macro_table, HASH_SIZE, cstrcmp);
    init_list(maclist);
    init_list(triglist);
    init_list(hooklist);
}

/***************************************
 * Routines for parsing macro commands *
 ***************************************/

short parse_attrs(args)            /* convert attribute string to bitfields */
    char *args;
{
    short attrs = 0;

    while (*args) {
        switch(*args) {
        case 'n':  break;
        case 'G':  attrs |= F_NOHISTORY; break;
        case 'g':  attrs |= F_GAG;       break;
        case 'u':  attrs |= F_UNDERLINE; break;
        case 'r':  attrs |= F_REVERSE;   break;
        case 'f':  attrs |= F_FLASH;     break;
        case 'd':  attrs |= F_DIM;       break;
        case 'B':  attrs |= F_BOLD;      break;
        case 'b':  attrs |= F_BELL;      break;
        case 'h':  attrs |= F_HILITE;    break;
        default:
            tfprintf(tferr, "%% Invalid display attribute '%c'", *args);
            return -1;
        }
        args++;
    }
    return attrs;
}

static int parse_hook(str)            /* convert hook string to bitfields */
    char *str;
{
    char *in;
    int hook, result = 0, done;

    if (strcmp(str, "*") == 0) return ~0L;
    for (done = FALSE; !done; str = in) {
        if (!(in = strchr(str, '|'))) done = TRUE;
        else *in++ = '\0';
        hook = binsearch(str, (GENERIC *)hook_table, NUM_HOOKS, sizeof(char*), cstrcmp);
        if (hook < 0) {
            tfprintf(tferr, "%% %s: invalid hook name", str);
            return 0;
        }
        result |= (1 << hook);
    }
    return result;
}

Macro *macro_spec(args)           /* convert user macro string to Macro */
    char *args;
{
    Macro *spec;
    char opt, *ptr, *trig = NULL, *bind = NULL, *hargs = NULL;
    int num, error = FALSE;

    spec = (Macro *)MALLOC(sizeof(struct Macro));
    spec->numnode = spec->trignode = spec->hooknode = spec->bucketnode = NULL;
    spec->name = spec->trig = spec->bind = spec->hargs = spec->body = NULL;
    spec->hook = 0;
    spec->func = NULL;
    spec->world = NULL;
    spec->pri = spec->prob = spec->shots = -1;
    spec->invis = 0;
    spec->attr = 0;
    spec->temp = TRUE;
    spec->dead = FALSE;

    startopt(args, "p#c#b:t:w:h:a:f:iIn#1:");
    while ((opt = nextopt(&ptr, &num))) {
        switch (opt) {
        case 'p':
            spec->pri = num;
            break;
        case 'c':
            spec->prob = num;
            break;
        case 'i':
            spec->invis = 1;
            break;
        case 'I':
            spec->invis = 2;
            break;
        case 'b':
            if (bind) FREE(bind);
            else bind = STRDUP(print_to_ascii(ptr));
            break;
        case 't':
            if (trig) FREE(trig);
            if (!smatch_check(trig = STRDUP(ptr))) error = TRUE;
            break;
        case 'w':
            if (!*ptr || strcmp(ptr, "+") == 0) spec->world = &AnyWorld;
            else if (strcmp(ptr, "-") == 0) spec->world = &NoWorld;
            else if (!(spec->world = find_world(ptr))) {
                tfprintf(tferr, "%% No world %s", ptr);
                error = TRUE;
            }
            break;
        case 'h':
            if (hargs) FREE(hargs);
            if (!*ptr || strcmp(ptr, "+") == 0) spec->hook = ~0L;
            else if (strcmp(ptr, "-") == 0) spec->hook = 0;
            else {
                if ((hargs = strchr(ptr, ' '))) *hargs++ = '\0';
                if (!(spec->hook = parse_hook(ptr))) error = TRUE;
                else hargs = STRDUP(hargs ? stripstr(hargs) : "");
            }
            break;
        case 'a': case 'f':
            if ((spec->attr |= parse_attrs(ptr)) < 0) error = TRUE;
            break;
        case 'n':
            spec->shots = num;
            break;
        case '1':
            if (*ptr && *(ptr + 1)) error = TRUE;
            else if (!*ptr || *ptr == '+') spec->shots = 1;
            else if (*ptr == '-') spec->shots = 0;
            else error = TRUE;
            if (error) tfputs("% Invalid argument to 1 option", tferr);
            break;
        default:
            error = TRUE;
        }

        if (error) {
            nuke_macro(spec);
            return NULL;
        }
    }

    if (trig) spec->trig = trig;
    if (bind) spec->bind = bind;
    spec->hargs = (hargs) ? hargs : STRDUP("");
    if (!*ptr) return spec;
    spec->name = ptr;
    if ((ptr = strchr(ptr, '='))) {
        *ptr = '\0';
        spec->body = ptr + 1;
        stripstr(spec->body);
        if (*spec->body) {
            spec->body = STRDUP(spec->body);
            if (cstrncmp(spec->body, "/DOKEY ", 7) == 0)
                spec->func = find_efunc(stripstr(spec->body + 7));
        } else spec->body = NULL;
    }
    stripstr(spec->name);
    spec->name = *spec->name ? STRDUP(spec->name) : NULL;
    return spec;
}


/********************************
 * Routines for locating macros *
 ********************************/

static int macro_match(spec, macro, cmp)     /* compare spec to macro */
    Macro *spec, *macro;
    int FDECL((*cmp),(char *s1, char *s2));  /* string comparison function */
{
    if (!spec->invis && macro->invis) return 1;
    if (spec->invis == 2 && !macro->invis) return 1;
    if (spec->shots >= 0 && spec->shots != macro->shots) return 1;
    if (spec->prob >= 0 && spec->prob != macro->prob) return 1;
    if (spec->pri >= 0 && spec->pri != macro->pri) return 1;
    if (spec->world) {
        if (spec->world == &NoWorld && macro->world) return 1;
        else if (spec->world == &AnyWorld && !macro->world) return 1;
        else if (spec->world != macro->world) return 1;
    }
    if (spec->hook) {
        if ((spec->hook & macro->hook) == 0) return 1;
        if (*spec->hargs && (*cmp)(spec->hargs, macro->hargs) != 0) return 1;
    }
    if (spec->attr && (spec->attr & macro->attr) == 0) return 1;
    if (spec->trig) {
        if (strcmp(spec->trig, "+") == 0 || strcmp(spec->trig, "") == 0) {
            if (!*macro->trig) return 1;
        } else if (strcmp(spec->trig, "-") == 0) {
            if (*macro->trig) return 1;
        } else if ((*cmp)(spec->trig, macro->trig) != 0) return 1;
    }
    if (spec->bind) {
        if (strcmp(spec->bind, "+") == 0 || strcmp(spec->bind, "") == 0) {
            if (!*macro->bind) return 1;
        } else if (strcmp(spec->bind, "-") == 0) {
            if (*macro->bind) return 1;
        } else if ((*cmp)(spec->bind, macro->bind) != 0) return 1;
    }
    if (spec->name && (*cmp)(spec->name, macro->name) != 0) return 1;
    if (spec->body && (*cmp)(spec->body, macro->body) != 0) return 1;
    return 0;
}

Macro *find_macro(name)              /* find Macro by name */
    char *name;
{
    if (!*name) return NULL;
    return (Macro *)hash_find(name, macro_table);
}

static Macro *match_exact(hook, str, flags)   /* find single exact match */
    int hook, flags;
    char *str;
{
    ListEntry *node;
  
    if ((hook && !flags) || (!hook && !*str)) return NULL;
    node = hook ? hooklist->head : triglist->head;
    while (node) {
        if ((Flag(node) & flags) && !cstrcmp(Pattern(node), str)) break;
        node = node->next;
    }
    return node ? (Macro *)node->data : NULL;
}

/* find one or more matches for a hook or trig */
static Macro *find_match(hook, text)
    char *text;
    int hook;
{
    Macro *first = NULL;
    int num = 0, lowerlimit = -1;
    ListEntry *node;

    /* Macros are sorted by decreasing priority, so we can stop looking
     * after we find one match and check other macros of equal priority.
     */
    node = hook ? hooklist->head : triglist->head;
    for ( ; node && Mac(node)->pri >= lowerlimit; node = node->next) {
        if (hook && !(Mac(node)->hook & hook)) continue;
        if (Mac(node)->world && Mac(node)->world != xworld()) continue;
        if ((hook && !*Mac(node)->hargs) || equalstr(Pattern(node), text)) {
            lowerlimit = Mac(node)->pri;
            num++;
            Mac(node)->tnext = first;
            first = Mac(node);
        }
    }
    if (!first) return NULL;
    for (num = (int)(RANDOM() % num); num--; first = first->tnext);
    return first;
}


/**************************
 * Routines to add macros *
 **************************/

/* create a Macro */
Macro *new_macro(name, trig, bind, hook, hargs, body, world, pri, prob,
  attr, shots, invis)
    char *name, *trig, *bind, *body, *hargs;
    int hook, pri, prob, attr, shots, invis;
    World *world;
{
    Macro *new;

    new = (Macro *) MALLOC(sizeof(struct Macro));
    new->numnode = new->trignode = new->hooknode = new->bucketnode = NULL;
    new->name = STRDUP(name);
    new->trig = STRDUP(trig);
    new->bind = STRDUP(bind);
    new->hook = hook;
    new->hargs = STRDUP(hargs);
    new->body = STRDUP(body);
    new->world = world;
    new->hook = hook;
    new->pri = pri;
    new->prob = prob;
    new->attr = attr;
    new->shots = shots;
    new->invis = invis;
    new->temp = TRUE;
    new->dead = FALSE;
    if (cstrncmp(new->body, "/DOKEY ", 7) == 0)
        new->func = find_efunc(stripstr(new->body + 7));
    else new->func = NULL;

    return new;
}

/* add permanent Macro to appropriate structures */
/* add_macro() assumes there is no confict with existing names */
void add_macro(macro)
    Macro *macro;
{
    macro->num = mnum++;
    macro->numnode = inlist((GENERIC *)macro, maclist, NULL);
    if (*macro->name)
        macro->bucketnode = hash_insert((GENERIC *)macro, macro_table);
    if (*macro->trig)
        macro->trignode = sinsert((GENERIC *)macro, triglist, (Cmp *)rpricmp);
    if (macro->hook)
        macro->hooknode = sinsert((GENERIC *)macro, hooklist, (Cmp *)rpricmp);
    macro->temp = FALSE;
}

int rpricmp(m1, m2)
    CONST Macro *m1, *m2;
{
    return m2->pri - m1->pri;
}

int install_bind(spec)          /* install Macro's binding in key structures */
    Macro *spec;
{
    Macro *macro;

    if ((macro = find_key(spec->bind))) {
        if (redef) {
            do_hook(H_REDEF, "%% Redefined %s %s", "%s %s",
                "binding", ascii_to_print(spec->bind));
            kill_macro(macro);
        } else {
            tfprintf(tferr, "%% Binding %s already exists.",
                ascii_to_print(spec->bind));
            nuke_macro(spec);
            return 0;
        }
    }
    if (bind_key(spec)) return 1;  /* fails if is prefix or has prefix */
    nuke_macro(spec);
    return 0;
}

int do_add(spec)                          /* define a new Macro (general) */
    Macro *spec;
{
    Macro *macro = NULL;

    if (spec->name && *spec->name == '@') {
        tfputs("% Macro names may not begin with '@'.", tferr);
        nuke_macro(spec);
        return 0;
    }
    if (spec->name && find_command(spec->name)) {
        do_hook(H_CONFLICT,
        "%% warning: /%s conflicts with the builtin command of the same name.",
        "%s", spec->name);
    }

    if (spec->world == &AnyWorld) spec->world = NULL;
    if (spec->pri < 0) spec->pri = 1;
    if (spec->prob < 0) spec->prob = 100;
    if (spec->shots < 0) spec->shots = 0;
    if (spec->invis) spec->invis = 1;
    if (spec->attr & F_NORM || !spec->attr) spec->attr = F_NORM;
    if (!spec->trig) spec->trig = STRDUP("");
    if (!spec->bind) spec->bind = STRDUP("");
    if (!spec->name) spec->name = STRDUP("");
    if (!spec->body) spec->body = STRDUP("");

    if (*spec->name && (macro = find_macro(spec->name)) && !redef) {
        tfprintf(tferr, "%% Macro %s already exists", spec->name);
        nuke_macro(spec);
        return 0;
    }
    if (*spec->bind && !install_bind(spec)) return 0;
    add_macro(spec);
    if (macro) {
        do_hook(H_REDEF, "%% Redefined %s %s", "%s %s", "macro", spec->name);
        kill_macro(macro);
    }
    return 1;
}

int add_hook(name, body)                  /* define a new Macro with hook */
    char *name, *body;
{
    int hook;
    char *hargs;

    if ((hargs = strchr(name, ' '))) *hargs++ = '\0';
    if (!(hook = parse_hook(name))) return 0;
    hargs = STRDUP(hargs ? stripstr(hargs) : "");
    add_macro(new_macro("", "", "", hook, hargs, body, NULL, 0, 100, 1, 0, 0));
    return 1;
}

int do_edit(spec)                         /* edit an existing Macro */
    Macro *spec;
{
    Macro *macro;
    ListEntry *node;
    int number;

    if (!spec->name) {
        tfputs("% You must specify a macro.", tferr);
        nuke_macro(spec);
        return 0;
    } else if (spec->name[0] == '#') {
        number = atoi(spec->name + 1);
        for (node = maclist->head; node; node = node->next)
            if (Mac(node)->num == number) break;
        if (!node) {
            tfprintf(tferr, "%% Macro #%d does not exist", number);
            nuke_macro(spec);
            return 0;
        }
        macro = Mac(node);
    } else if (spec->name[0] == '$') {
        if (!(macro = match_exact(FALSE, spec->name + 1, F_ALL))) {
            tfprintf(tferr, "%% Trigger %s does not exist", spec->name + 1);
            nuke_macro(spec);
            return 0;
        }
    } else if (!(macro = find_macro(spec->name))) {
        tfprintf(tferr, "%% Macro %s does not exist", spec->name);
        nuke_macro(spec);
        return 0;
    }

    if (spec->body && *spec->body) {
        FREE(macro->body);
        macro->body = STRDUP(spec->body);
    }
    if (spec->trig && *spec->trig) {
        FREE(macro->trig);
        macro->trig = STRDUP(spec->trig);
        if (spec->pri >= 0) {                    /* re-link if pri changed */
            unlist(macro->trignode, triglist);
            macro->trignode = sinsert((GENERIC*)macro, triglist, (Cmp*)rpricmp);
        }
    }
    if (spec->hook) {
        macro->hook = spec->hook;
        if (*macro->hargs) FREE(macro->hargs);
        macro->hargs = STRDUP((spec->hargs && *spec->hargs) ? spec->hargs : "");
        if (spec->pri >= 0) {                    /* re-link if pri changed */
            unlist(macro->hooknode, hooklist);
            macro->hooknode = sinsert((GENERIC*)macro, hooklist, (Cmp*)rpricmp);
        }
    }
    if (spec->world && spec->world != &AnyWorld) macro->world = spec->world;
    if (spec->pri >= 0) macro->pri = spec->pri;
    if (spec->prob >= 0) macro->prob = spec->prob;
    if (spec->shots >= 0) macro->shots = spec->shots;
    if (spec->attr & F_NORM) macro->attr = F_NORM;
    else if (spec->attr & F_GAG) macro->attr = spec->attr & F_SUPERGAG;
    else if (spec->attr) macro->attr = spec->attr;
    macro->invis = (spec->invis) ? 1 : 0;
    nuke_macro(spec);
    return 1;
}


/********************************
 * Routines for removing macros *
 ********************************/

void kill_macro(macro)            /* remove all references to macro */
    Macro *macro;
{
    if (macro->dead) return;
    macro->dead = TRUE;
    macro->tnext = dead_macros;
    dead_macros = macro;
    unlist(macro->numnode, maclist);
    if (*macro->name) hash_remove(macro->bucketnode, macro_table);
    if (macro->trig && *macro->trig) unlist(macro->trignode, triglist);
    if (macro->hook) unlist(macro->hooknode, hooklist);
    if (*macro->bind) unbind_key(macro);
}

void nuke_dead_macros()
{
    Macro *macro;

    while ((macro = dead_macros)) {
        dead_macros = dead_macros->tnext;
        nuke_macro(macro);
    }
}

void nuke_macro(macro)            /* free macro structure */
    Macro *macro;
{
    if (!macro->dead && !macro->temp) {
        kill_macro(macro);
    }

    if (macro->name) FREE(macro->name);
    if (macro->body) FREE(macro->body);
    if (macro->trig) FREE(macro->trig);
    if (macro->hargs) FREE(macro->hargs);
    if (macro->bind) FREE(macro->bind);
    FREE(macro);
}

int remove_macro(str, flags, byhook)        /* delete a macro */
    char *str;
    int flags;
    int byhook;
{
    Macro *macro = NULL;
    char *args;

    if (byhook) {
        if ((args = strchr(str, ' '))) *args++ = '\0';
        if (!(flags = parse_hook(str))) return 0;
        if (!(macro = match_exact(byhook, args ? args : "", flags)))
            tfprintf(tferr, "%% Hook on \"%s\" was not defined.", str);
    } else if (flags) {
        if (!(macro = match_exact(byhook, str, flags)))
            tfprintf(tferr, "%% Trigger on \"%s\" was not defined.", str);
    } else {
        if (!(macro = find_macro(str)))
            tfprintf(tferr, "%% Macro \"%s\" was not defined.", str);
    }
    if (!macro) return 0;
    kill_macro(macro);
    return 1;
}

int purge_macro(spec)                   /* delete all specified macros */
    Macro *spec;
{
    ListEntry *node, *next;

    if (!spec) return 0;
    if (spec->name && !smatch_check(spec->name)) return 0;
    if (spec->trig && !smatch_check(spec->trig)) return 0;
    if (spec->hargs && !smatch_check(spec->hargs)) return 0;
    if (spec->bind && !smatch_check(spec->bind)) return 0;
    if (spec->body && !smatch_check(spec->body)) return 0;
    for (node = maclist->head; node; node = next) {
        next = node->next;
        if (macmatch(spec, Mac(node)) == 0) kill_macro(Mac(node));
    }
    nuke_macro(spec);
    return 1;
}

int remove_by_number(args)                 /* delete macro by number */
    char *args;
{
    char *ptr;
    int num, result = 1;
    ListEntry *node;

    for (ptr = args; ptr; ptr = strchr(ptr, ' ')) {
        while (isspace(*ptr)) ptr++;
        num = atoi(ptr);
        if (num > 0 && num < mnum) {
            for (node = maclist->head; node; node = node->next) {
                /* We know the macros are in decending numeric order. */
                if (Mac(node)->num <= num) break;
            }
            if (node && Mac(node)->num == num) kill_macro(Mac(node));
            else {
                tfprintf(tferr, "%% No macro with number %d", num);
                result = 0;
            }
        } else {
            tfprintf(tferr, "%% Invalid number %d", num);
            result = 0;
        }
    }
    return result;
}

void remove_world_macros(w)
    World *w;
{
    ListEntry *node, *next;

    for (node = maclist->head; node; node = next) {
        next = node->next;
        if (Mac(node)->world == w) kill_macro(Mac(node));
    }
}


/**************************
 * Routine to list macros *
 **************************/

static String *hook_name(hook)        /* convert hook bitfield to string */
    int hook;
{
    int n;
    STATIC_BUFFER(buf)

    Stringterm(buf, 0);
    for (n = 0; n < NUM_HOOKS; n++) {
        if (!((1 << n) & hook)) continue;
        if (buf->len) Stringadd(buf, '|');
        Stringcat(buf, hook_table[n]);
    }
    return buf;
}

static String *escape(src)                /* insert '\' before each '"' */
    char *src;
{
    char *ptr;
    STATIC_BUFFER(dest)

    for (Stringterm(dest, 0); *src; src = ptr) {
        if (*src == '\"' || *src == '\\') Sprintf(dest, "\200\\%c", *src++);
        for (ptr = src; *ptr && *ptr != '\"' && *ptr != '\\'; ptr++);
        Stringncat(dest, src, ptr - src);
    }
    return dest;
}

static int list_defs(file, spec, abbrev)    /* list all specified macros */
    TFILE *file;
    Macro *spec;
    int abbrev;
{
    Macro *p;
    ListEntry *node;
    STATIC_BUFFER(buffer)

    if (spec->name && !smatch_check(spec->name)) return 0;
    if (spec->trig && !smatch_check(spec->trig)) return 0;
    if (spec->hargs && !smatch_check(spec->hargs)) return 0;
    if (spec->bind && !smatch_check(spec->bind)) return 0;
    if (spec->body && !smatch_check(spec->body)) return 0;

    for (node = maclist->tail; node; node = node->prev) {
        p = Mac(node);
        if (macmatch(spec, p) != 0) continue;

        if (abbrev) {
            Sprintf(buffer, "%% %d: ", p->num);
            if (p->attr & F_NOHISTORY) Stringcat(buffer, "(norecord) ");
            if (p->attr & F_GAG) Stringcat(buffer, "(gag) ");
            else if (p->attr & ~F_NORM) {
                if (p->attr & F_UNDERLINE) Stringcat(buffer, "(underline) ");
                if (p->attr & F_REVERSE)   Stringcat(buffer, "(reverse) ");
                if (p->attr & F_FLASH)     Stringcat(buffer, "(flash) ");
                if (p->attr & F_DIM)       Stringcat(buffer, "(dim) ");
                if (p->attr & F_BOLD)      Stringcat(buffer, "(bold) ");
                if (p->attr & F_BELL)      Stringcat(buffer, "(bell) ");
                if (p->attr & F_HILITE)    Stringcat(buffer, "(hilite) ");
            } else if (*p->trig) Stringcat(buffer, "(trig) ");
            if (*p->trig) Sprintf(buffer, "\200\"%S\" ", escape(p->trig));
            if (*p->bind)
                Sprintf(buffer, "\200(bind) \"%S\" ",
                    escape(ascii_to_print(p->bind)));
            if (p->hook) Sprintf(buffer, "\200(hook) %S ", hook_name(p->hook));
            if (*p->name) Sprintf(buffer, "\200%s ", p->name);

        } else {
            if (!file) Sprintf(buffer, "%% %d: /def ", p->num);
            else Stringcpy(buffer, "/def ");
            if (p->invis) Stringcat(buffer, "-i ");
            if ((*p->trig || p->hook) && p->pri)
                Sprintf(buffer, "\200-p%d ", p->pri);
            if (*p->trig && p->prob != 100)
                Sprintf(buffer, "\200-c%d ", p->prob);
            if (p->attr & F_GAG) {
                Stringcat(buffer, "-ag");
                if (p->attr & F_NOHISTORY)  Stringadd(buffer, 'G');
                Stringadd(buffer, ' ');
            } else if (p->attr & ~F_NORM) {
                Stringcat(buffer, "-a");
                if (p->attr & F_NOHISTORY) Stringadd(buffer, 'G');
                if (p->attr & F_UNDERLINE) Stringadd(buffer, 'u');
                if (p->attr & F_REVERSE)   Stringadd(buffer, 'r');
                if (p->attr & F_FLASH)     Stringadd(buffer, 'f');
                if (p->attr & F_DIM)       Stringadd(buffer, 'd');
                if (p->attr & F_BOLD)      Stringadd(buffer, 'B');
                if (p->attr & F_BELL)      Stringadd(buffer, 'b');
                if (p->attr & F_HILITE)    Stringadd(buffer, 'h');
                Stringadd(buffer, ' ');
            }
            if (p->world) Sprintf(buffer, "\200-w%s ", p->world->name);
            if (p->shots) Sprintf(buffer, "\200-n%d ", p->shots);
            if (*p->trig) Sprintf(buffer, "\200-t\"%S\" ", escape(p->trig));
            if (p->hook) {
                Sprintf(buffer, "\200-h\"%S", hook_name(p->hook));
                if (*p->hargs) Sprintf(buffer, "\200 %S", escape(p->hargs));
                Stringcat(buffer, "\" ");
            }
            if (*p->bind) 
                Sprintf(buffer, "\200-b\"%S\" ",
                    escape(ascii_to_print(p->bind)));
            if (*p->name) Sprintf(buffer, "\200%s ", p->name);
            if (*p->body) Sprintf(buffer, "\200= %s", p->body);
        }

        if (file) tfputs(buffer->s, file);
        else oputs(buffer->s);
    }
    return 1;
}

int save_macros(args)              /* write specified macros to file */
    char *args;
{
    char *p;
    Macro *spec;
    TFILE *file;
    int result;

    for (p = args; *p && !isspace(*p); p++);
    if (*p) *p++ = '\0';
    if (!(spec = macro_spec(p))) return 0;
    if (!(file = tfopen(tfname(args, NULL), "w"))) {
        operror(args);
        return 0;
    }
    oprintf("%% Saving macros to %s", file->name);
    result = list_defs(file, spec, FALSE);
    tfclose(file);
    nuke_macro(spec);
    return result;
}

int list_macros(args)                    /* list specified macros on screen */
    char *args;
{
    Macro *spec;
    char *p;
    int abbrev = FALSE;
    int result;

    for (p = args; isspace(*p); p++);
    if (p[0] == '-' && p[1] == 's' && (p[2] == '\0' || isspace(p[2]))) {
        args = p + 2;
        abbrev = TRUE;
    }
    if (!(spec = macro_spec(args))) return 0;
    result = list_defs(NULL, spec, abbrev);
    nuke_macro(spec);
    return result;
}


/**************************
 * Routines to use macros *
 **************************/

static char *world_subs(src)            /* "world_*" subtitutions */
    char *src;
{
    char *ptr = src + 6;
    World *world, *def;

    if (cstrncmp("world_", src, 6) != 0) return NULL;
    if (!(world = xworld())) return "";
    def = get_default_world();

    if (!cstrcmp("name", ptr)) return world->name;
    else if (!cstrcmp("character", ptr)) {
        if (*world->character) return world->character;
        else if (def) return def->character;
    } else if (!cstrcmp("password", ptr)) {
        if (*world->pass) return world->pass;
        else if (def) return def->pass;
    } else if (!cstrcmp("host", ptr)) return world->address;
    else if (!cstrcmp("port", ptr)) return world->port;
    else if (!cstrcmp("mfile", ptr)) {
        if (*world->mfile) return world->mfile;
        else if (def) return def->mfile;
    } else return NULL;
    return "";
}

int do_macro(macro, args)       /* Do a macro! */
    Macro *macro;
    char *args;
{
    char *body;
    int result;

    if (macro->func) {
        (*macro->func)();
        return 1;
    } else {
        body = STRDUP(macro->body);         /* in case macro undefines itself */
        result = process_macro(body, args);
        FREE(body);
    }
    return result;
}

char *macro_body(name)                            /* get body of macro */
    char *name;
{
    Macro *m;
    char *result;

    if ((result = world_subs(name))) return result;
    if (!(m = find_macro(name))) return NULL;
    return m->body;
}


/****************************************
 * Routines to check triggers and hooks *
 ****************************************/

#ifdef HAVE_STDARG
short do_hook(int idx, char *fmt, char *argfmt, ...)     /* do a hook event */
#else
/* VARARGS */
short do_hook(va_alist)
va_dcl
#endif
{
#ifndef HAVE_STDARG
    int idx;
    char *fmt, *argfmt;
#endif
    va_list ap;
    Macro *macro;
    short attr = 0;
    static Stringp buf, args;
    static int buffers_initted = FALSE;

    if (!buffers_initted) {
        Stringinit(buf);
        Stringinit(args);
        buffers_initted = TRUE;
    }

#ifdef HAVE_STDARG
    va_start(ap, argfmt);
#else
    va_start(ap);
    idx = va_arg(ap, int);
    fmt = va_arg(ap, char *);
    argfmt = va_arg(ap, char *);
#endif
    Stringterm(args, 0);
    vSprintf(args, argfmt, ap);
    va_end(ap);

    macro = find_match((1<<idx), args->s);
    if (macro) attr = macro->attr;

    if (fmt) {
#ifdef HAVE_STDARG
        va_start(ap, argfmt);
#else
        va_start(ap);
        idx = va_arg(ap, int);
        fmt = va_arg(ap, char *);
        argfmt = va_arg(ap, char *);
#endif
        vSprintf(buf, fmt, ap);
        oputa(new_aline(buf->s, attr | F_NEWLINE));
    }
    va_end(ap);

    if (macro && hookflag) run_match(macro, args->s);
    return attr;
}

short check_trigger(text)                 /* look for trigger, and call it */
    char *text;
{
    Macro *macro;
    short attr;

    if (!(borg || hilite || gag)) return 0;
    macro = find_match(0, text);
    attr = macro ? macro->attr : F_NORM;
    if (macro && borg && run_match(macro, text)) background_hook(text);
    return attr;
}

/* run trigger or hook that has been matched */
static int run_match(macro, text)
    Macro *macro;
    char *text;
{
    if (macro->prob < 100 && RANDOM() % 100 >= macro->prob) return 0;
    if (macro->shots && !--macro->shots) kill_macro(macro);
    if (macro->func) {
        (*macro->func)();
        return 1;
    } else if (*macro->body) {
        return process_macro(macro->body, text);
    } else {
        return 0;
    }
}

void change_hilite()
{
    char *value;

    if (!(value = getvar("hiliteattr"))) return;
    if ((hiliteattr = parse_attrs(value)) < 0) hiliteattr = 0;
}

#ifdef DMALLOC
void free_macros()
{
    while (maclist->head) nuke_macro((Macro *)maclist->head->data);
    free_hash(macro_table);
}
#endif

