/****************************************************************/
/*  dsputil.c							*/
/* DSP utility functions for Csound - dispfft and pvoc		*/
/* 20apr90 dpwe							*/
/****************************************************************/

#include "dpwelib.h"
#include "dsputil.h"
#include <math.h>

/* Do we do the whole buffer, or just indep vals? */
#define someof(s)	(s)
    /* (1L+s/2L) is the indep vals in an FFT frame of length (s).. well */
    /* (change to just (s) to work on whole buffer) */
#define actual(s)	((s-1L)*2L)
    /* if callers are passing whole frame size, make this (s) */
    /* where callers pass s/2+1, this recreates the parent fft frame size */

/* #ifndef __STDC__	/* needed for difference in libc.a .. ? */
/* long labs(l1)
/*     long l1;
/* {
/*     if(l1<0L)
/*       return(-l1);
/*     else
/*       return(l1);
/* }
/* #endif
 */

void CopySamps(sce,dst,size)	/* just move samples */
    float	*sce;
    float	*dst;
    long 	size;
    {
    ++size;
    while(--size)
	*dst++ = *sce++;
    }

/* Allocate a cleared buffer */
float *MakeBuf(size)
    long     size;
    {
    float *res,*p;
    long	  i;
    
    res = (float *) malloc((long)size * sizeof(float));
    p = res;
    for(i=0; i<size; ++i)
    	*p++ = 0.0;
	return(res);
    }

/* Write window coefs into buffer, don't malloc */
void FillHalfWin(wBuf,size, max, hannq)
    float	*wBuf;	
    long	size;
    FLOATARG	max;
    int 	hannq;		/* 1 => hanning window else hamming */
    {
    float	a,b;
    long	i;

    if(hannq)
	{	a = .50;	b = .50;	}
    else
	{	a = .54;	b = .46;	}
		
    if (wBuf!= NULL)
	{			/* NB: size/2 + 1 long - just indep terms */
	size /= 2; 		/* to fix scaling */
	for(i=0; i<=size;++i)
	    wBuf[i] = max * (a-b*cos(PI*(float)i/(float)size ) );
    	}
    return;
    }
    
/* Make 1st half of hamming window for those as desire */
float *MakeHalfWin(size, max, hannq)
    long		size;		/* Effective window size (altho only s/2+1 alloc'd */
    FLOATARG	max;
    int 	hannq;		/* 1 => hanning window else hamming */
    {
    float 	*wBuf;
    float	a,b;
    long	i;

    wBuf = (float *) malloc((long)(size/2+1) * sizeof(float));
				/* NB: size/2 + 1 long - just indep terms */
    FillHalfWin(wBuf,size,max,hannq);
    return(wBuf);
    }
    
void UnpackReals(dst,size)		/* expand out size to re,0,re,0 etc */
    float	*dst;
    long 	size;
    {
    float	*d2;

    d2 = dst + 2*size - 1;	/* start at ends .. */
    dst += size - 1;
    ++size;
    while(--size)		/* .. & copy backwards */
	{
	*d2-- = 0.0;
	*d2-- = *dst--;
	}
    }

void PackReals(buffer,size)	/* pack re,im,re,im into re,re */
    float *buffer;
    long size;
    {
    float	*b2 = buffer;

    ++size;
    while(--size)
	{
	*b2++ = *buffer++;
	++buffer;
	}
    }

/* Convert Real & Imaginary spectra into Amplitude & Phase */
void Rect2Polar(buffer,size)
    float *buffer;
    long size;
    {
    long	i;
    float	*real,*imag;
    float	re,im;
    float	mag, pha;

    real = buffer;	imag = buffer+1;
    for(i = 0; i< someof(size); ++i)
	{
	re = real[2L*i];
	im = imag[2L*i];
	real[2L*i] = mag = hypot(re,im);
	if(mag == 0.0)
	    imag[2L*i] = 0.0;
	else
	    imag[2L*i] = atan2(im,re);
	} 
    }


void Polar2Rect(buffer,size)
    float *buffer;
    long size;	    /* reverse above */
    {
    long	    i;
    float   *magn,*phas;
    float   mag, pha;

    magn = buffer;  phas = buffer+1;
    for(i = 0; i< someof(size); ++i)
	{
	mag = magn[2L*i];
	pha = phas[2L*i];
	magn[2L*i] = mag*cos(pha);
	phas[2L*i] = mag*sin(pha);
	}
    }

void Lin2DB(buffer, size)	/* packed buffer i.e. reals, not complex */
    float *buffer;
    long size;
    {
    while(size--)
	*buffer++ = /* 20.0*log10 */ 8.68589*log(*buffer);
    }

void DB2Lin(buffer, size)	/* packed buffer i.e. reals, not complex */
    float *buffer;
    long size;
    {
    while(size--)
	*buffer++ = exp( /* 1/20.0*log10 */ .1151292*(*buffer) );
    }

float maskPhs(phs)	/* do it inline instead! */
    FLOATARG   phs;
    {
    while(phs > PI)
	{
/*	putchar('-'); fflush(stdout);	*/
	phs -= 2*PI;
	}
    while(phs < -PI)
	{
/*	putchar('+'); fflush(stdout);	*/
	phs += 2*PI;
	}
    return(phs);
    }

#define MmaskPhs(phs)	/* macro version of phase masking */ \
    while(phs > PI)	\
	phs -= 2*PI;	\
    while(phs < -PI)	\
	phs += 2*PI;	\

#define MMmaskPhs(p,q,r,s) /* p is pha, q is as int, r is PI, s is 1/PI */ \
    q = (int)(s*p);	\
    p -= r*(float)((int)((q+((q>=0)?(q&1):-(q&1) )))); \


void UnwrapPhase(buf,size,oldPh)
    float *buf;
    long size;
    float *oldPh;
    {
    long    i;
    float   *pha;
    float   p,pi,oneOnPi;
    int     z;

    pha = buf + 1;
    pi = PI; oneOnPi = 1/PI;
    for(i=0; i<someof(size); ++i)
	{
	p = pha[2L*i];
	p -= oldPh[i];		/* find change since last frame */
	MMmaskPhs(p,z,pi,oneOnPi);
	/* MmaskPhs(p); */
	oldPh[i] = pha[2L*i];	/* hold actual phase for next diffce */
	pha[2L*i] = p;		/* .. but write back phase change */
	}
    }

void RewrapPhase(buf,size,oldPh)
    float   *buf;
    long	    size;
    float   *oldPh;
    {
    long	    i;
    float   *pha;
    register float  p,pi,twoPi,oneOnPi;
    int		z;

    /* Phase angle was properly scaled when it came out of frqspace */
    /* .. so just add it */
    pha = buf + 1;
    pi = PI; twoPi = 2*PI; oneOnPi = 1/PI;
    for(i=0; i<someof(size); ++i)
	{
	p = (pha[2L*i]+oldPh[i]);
	MMmaskPhs(p,z,pi,oneOnPi);
	/* MmaskPhs(p);	*/
	oldPh[i] = pha[2L*i] = p;
	}
    }

/* Convert a batch of phase differences to actual target freqs */
void PhaseToFrq(buf,size,incr,sampRate)
    float   *buf;
    long    size;
    FLOATARG   incr;
    FLOATARG   sampRate;
    {
    long    i;
    float   *pha,p,q,pi,oneOnPi,twoPi;
    int     z;
    float   srOn2pi, binMidFrq, frqPerBin;
    float   expectedDphas,eDphIncr;

    pha = buf + 1;
    srOn2pi = sampRate/(2*PI*incr);
    frqPerBin = sampRate/((float)actual(size));
    binMidFrq = 0;
    /* Of course, you get some phase shift with spot-on frq coz time shift */
    expectedDphas = 0;
    eDphIncr = 2.0*PI*incr/((float)actual(size));
    pi = PI; oneOnPi = 1/PI; twoPi = 2*PI;
    for(i=0; i<someof(size); ++i)
	{
	q = p = pha[2L*i]-expectedDphas;
	MMmaskPhs(p,z,pi,oneOnPi);
/*	MmaskPhs(q);	*/
	pha[2L*i] = p;
	pha[2L*i] *= srOn2pi;
	pha[2L*i] += binMidFrq;

	expectedDphas += eDphIncr;
	expectedDphas -= twoPi*(float)((int)(expectedDphas*oneOnPi)); 
	binMidFrq += frqPerBin;
	}
    /* Doesn't deal with 'phases' of DC & fs/2 any different */
    }

/* Undo a pile of frequencies back into phase differences */
void FrqToPhase(buf,size,incr,sampRate,fixUp)
    float   *buf;
    long	    size;
    FLOATARG   incr;
    FLOATARG   sampRate;
    FLOATARG   fixUp;	    /* the fixup phase shift ... ? */
    {
    float   *pha;
    float   twoPiOnSr, binMidFrq, frqPerBin;
    float   expectedDphas,eDphIncr;
    register float  p,pi;
    register long    i;
    register int     j;
    float   twoPi, oneOnPi;

    twoPi = 2*PI;
    oneOnPi = 1/PI;
    pha = buf + 1;
    pi = PI;
    twoPiOnSr = 2.*PI*((float)incr)/sampRate;
    frqPerBin = sampRate/((float)actual(size));
    binMidFrq = 0;
    /* Of course, you get some phase shift with spot-on frq coz time shift */
    expectedDphas = 0;
    eDphIncr = 2.*PI*((incr)/((float)actual(size)) + fixUp);
    for(i=0; i<someof(size); ++i)
	{
	p = pha[2L*i];
	p -= binMidFrq;
	p *= twoPiOnSr;
	p += expectedDphas;
        MMmaskPhs(p,j,pi,oneOnPi);
	/* MmaskPhs(p);	*/
	pha[2L*i] = p;
	expectedDphas += eDphIncr;
	expectedDphas -= twoPi*(float)((int)(expectedDphas*oneOnPi)); 
	binMidFrq += frqPerBin;
	}
	/* Doesn't deal with 'phases' of DC & fs/2 any different */
    }

/* Unpack stored mag/pha data into buffer */
void FetchIn(inp, buf, fsize, pos)
    float   *inp;	/* pointer to input data */
    float   *buf;	/* where to put our nice mag/pha pairs */
    long    fsize;	/* frame size we're working with */
    FLOATARG   pos;	/* fractional frame we want */
    {
    long    j;
    float   *frm0,*frm1;
    long    base;
    float   frac;

    /***** WITHOUT INFO ON WHERE LAST FRAME IS, MAY 'INTERP' BEYOND IT ****/
    base = (long)pos;		    /* index of basis frame of interpolation */
    frac = ((float)(pos - (float)base));
    /* & how close to get to next */
    frm0 = inp + ((long)fsize+2L)*base;
    frm1 = frm0 + ((long)fsize+2L);	    /* addresses of both frames */
	if(frac != 0.0)		/* must have 2 cases to avoid poss seg vlns */
		{		/* and failed computes, else may interp   */
		                /* bd valid data */
	    for(j=0; j<(fsize/2L + 1L); ++j)  /* mag/pha for just over 1/2 */
			{		/* Interpolate both mag and phase */
			buf[2L*j] = frm0[2L*j] + frac*(frm1[2L*j]-frm0[2L*j]);
			buf[2L*j+1L] = frm0[2L*j+1L] 
			    + frac*(frm1[2L*j+1L]-frm0[2L*j+1L]);
			}
		}
	else
		{	/* frac is 0.0 i.e. just copy the source frame */
	    for(j=0; j<(fsize/2L + 1L); ++j)
			{		/* no need to interpolate */
			buf[2L*j] = frm0[2L*j];
			buf[2L*j+1] = frm0[2L*j+1L];
			}
		}
/*    if(someof(fsize) == fsize)    /* i.e. if dealing whole frames.. */
/*		{
/*		for(j=(fsize/2L + 1L); j<fsize; ++j)	/* rest mirrored .. */
/*		    {
/*		    buf[2L*j] = buf[2L*(fsize-j)];
/*		    buf[2L*j+1L] = -buf[2L*(fsize-j)+1L]; /* ..w/ imag conj */
/*		    }
/*		}
 */
    }

/* Fill out the dependent 2nd half of iFFT data; scale down & opt conjgt */
void FillFFTnConj(buf,size,scale,conj)
    float   *buf;
    long     size;		/* full length of FFT ie 2^n */
    FLOATARG   scale;		/* can apply a scale factor.. */
    int     conj;		/* flag to conjugate at same time */
    {
    float   miscale;		/* scaling for poss. conj part */
    float   *mag,*pha;
    long    j;
    long    hasiz = 1L + size/2L; /* the indep values */

    if(scale == 0.0)	scale = 1.0;
    if(conj)
	miscale = -scale;
    else
	miscale = scale;
    mag = buf;	    pha = buf+1;
    for(j=0; j<hasiz; ++j)	    /* i.e. mag/pha for just over 1/2 */
	{
	mag[2L*j] *= scale;
	pha[2L*j] *= miscale;
	}
    for(j=hasiz; j<size; ++j)		/* .. the rest is mirrored .. */
	{
	mag[2L*j] = mag[2L*(size-j)];	/* For the symmetry extension, */
	pha[2L*j] = -pha[2L*(size-j)];	/*  conjugate of 1st 1/2 rgdls */
	}
    }

void ApplyHalfWin(buf,win,len)	/* Window only store 1st half, is symmetric */
    register float   *buf;
    register float   *win;
    long    len;
    {
    register long j;
    long    lenOn2 = (len/2L);

/*    for(j=0; j<len; ++j)                            */
/*	buf[j] *= win[lenOn2 - labs(lenOn2 -j)];      */
    for (j = lenOn2 + 1; j--; )
	*buf++ *= *win++;
    for (j = len - lenOn2 - 1, win--; j--; )
	*buf++ *= *--win;
    }	

/* Overlap (some of) new data window with stored previous data in circular buffer */
void addToCircBuf(sce, dst, dstStart, numToDo, circBufSize)
    float   *sce, *dst;     /* linear source and circular destination */
    long    dstStart;	    /* Current starting point index in circular dst */
    long    numToDo;	    /* How many points to add ( <= circBufSize ) */
    long    circBufSize;    /* Size of circ buf i.e. dst[0..circBufSize-1] */
    {
    long    i;
    long    breakPoint;     /* how many points to add before having to wrap */

    breakPoint = circBufSize-dstStart;	/* i.e. if we start at (dIndx = lim -2) */
    if(numToDo > breakPoint)		/*   we will do 2 in 1st loop, rest in 2nd. */
	{
	for(i=0; i<breakPoint; ++i)
	    dst[dstStart+i] += sce[i];
	dstStart -= circBufSize;
	for(i=breakPoint; i<numToDo; ++i)
	    dst[dstStart+i] += sce[i];
	}
    else				/* All fits without wraparound */
	for(i=0; i<numToDo; ++i)
	    dst[dstStart+i] += sce[i];
    return;
    }

/* Write from a circular buffer into a linear output buffer CLEARING DATA */
void writeClrFromCircBuf(sce, dst, sceStart, numToDo, circBufSize)
    float   *sce, *dst;     /* Circular source and linear destination */
    long    sceStart;	    /* Current starting point index in circular sce */
    long    numToDo;	    /* How many points to write ( <= circBufSize ) */
    long    circBufSize;    /* Size of circ buf i.e. sce[0..circBufSize-1] */
    {
    long    i;
    long    breakPoint;     /* how many points to add before having to wrap */

    breakPoint = circBufSize-sceStart;	/* i.e. if we start at (Indx = lim -2)	*/
    if(numToDo > breakPoint)		/*  we will do 2 in 1st loop, rest in 2nd. */
	{
	for(i=0; i<breakPoint; ++i)
	    {
	    dst[i] = sce[sceStart+i];
	    sce[sceStart+i] = 0.0;
	    }
	sceStart -= circBufSize;
	for(i=breakPoint; i<numToDo; ++i)
	    {
	    dst[i] = sce[sceStart+i];
	    sce[sceStart+i] = 0.0;
	    }
	}
    else				/* All fits without wraparound */
	for(i=0; i<numToDo; ++i)
	    {
	    dst[i] = sce[sceStart+i];
	    sce[sceStart+i] = 0.0;
	    }
    return;
    }

/* Add source array to dest array, results in dest */
void	FixAndAdd(samplSce, shortDest, size)
    float   *samplSce;
    short   *shortDest;
    long     size;
    {
    long i;
    for (i = 0; i < size; i++)
	shortDest[i] += (short)samplSce[i];
    }

/* Rules to convert between samples and frames, given frSiz & frIncr */
long NumFrames(dataSmps,frSiz,frInc)
    long    dataSmps;
    long     frSiz;
    long     frInc;
    {
    return( 1L + (dataSmps - (long)frSiz)/(long)frInc );
    }

long NumSampls(frames, frSiz, frIncr)
    long    frames;
    long     frSiz;
    long     frIncr;
    {
    return(((long)frSiz)+((long)frIncr)*(frames-1L));
    }

/********************************************************************/
/*  udsample.c	    -	    from dsampip.c			    */
/*  Performs sample rate conversion by interpolated FIR LPF approx  */
/*  VAX, CSOUND VERSION						    */
/*  1700 07feb90 taken from rational-only version		    */
/*  1620 06dec89 started dpwe					    */
/********************************************************************/

/* (ugens7.h) #define	SPDS (8)    /* How many sinc lobes to go out */
/* Static function prototypes - moved to top of file */
static float *sncTab = NULL;	/* point to our sin(x)/x lookup table */

void UDSample(inSnd, stindex, outSnd, inLen, outLen, fex)
    float   *inSnd;
    FLOATARG   stindex;
    float   *outSnd;
    long     inLen;
    long     outLen;
    FLOATARG   fex;
/*  Perform the sample rate conversion:
    inSnd   is the existing sample to be converted
    outSnd  is a pointer to the (pre-allocated) new soundspace
    inLen   is the number of points in the input sequence
    outLen  is the number of output sample points.  e.g inLen/fex
    fex is the factor by which frequencies are increased
    1/fex = lex, the factor by which the output will be longer
    i.e. if the input sample is at 12kHz and we want to produce an
    8kHz sample, we will want it to be 8/12ths as many samples, so
    lex will be 0.75
 */
    {
    int     in2out;
    long    i,j,x;
    float   a;
    float   phasePerInStep, fracInStep;
    float   realInStep, stepInStep;
    long    nrstInStep;
    float   posPhase, negPhase;
    float   lex = 1.0/fex;
    int     nrst;
    float   frac;

    phasePerInStep = ((lex>1)? 1.0 : lex)* (float)SPTS;
    /* If we are upsampling, LPF is at input frq => sinc pd matches */
    /*	downsamp => lpf at output rate; input steps at some fraction */
    in2out = (int)( ((float)SPDS) * ( (fex<1)? 1.0 : fex ) );
    /* number of input points contributing to each op: depends on LPF */
    realInStep = stindex;	    stepInStep = fex;
    for(i = 0; i<outLen; ++i)		    /* output sample loop */
	{				    /* i = lex*nrstIp, so .. */
	nrstInStep = (long)realInStep;		    /* imm. prec actual sample */
	fracInStep = realInStep-(float)nrstInStep;  /* Fractional part */
	negPhase = phasePerInStep * fracInStep;
	posPhase = -negPhase;
	/* cum. sinc arguments for +ve & -ve going spans into input */
	nrst = (int)negPhase;	frac = negPhase - (float)nrst;
	a = (sncTab[nrst]+frac*(sncTab[nrst+1]-sncTab[nrst]))*(float)inSnd[nrstInStep];
	for(j=1L; j<in2out; ++j)		    /* inner FIR convolution loop */
	    {
	    posPhase += phasePerInStep;
	    negPhase += phasePerInStep;
	    if( (x = nrstInStep-j)>=0L )
	    nrst = (int)negPhase;   frac = negPhase - (float)nrst;
	    a += (sncTab[nrst]+frac*(sncTab[nrst+1]-sncTab[nrst]))
		    * (float)inSnd[x];
	    if( (x = nrstInStep+j)<inLen )
	    nrst = (int)posPhase;   frac = posPhase - (float)nrst;
	    a += (sncTab[nrst]+frac*(sncTab[nrst+1]-sncTab[nrst]))
		    * (float)inSnd[x];
	    }
	outSnd[i] = (float)a;
	realInStep += stepInStep;
	}
    }

void    FloatAndCopy(sce,dst,size)
    short  *sce;
    float  *dst;
    long   size;
    {
    while(size--)
	*dst++ = (float)*sce++;
    }

/* Copy converted frame to the output data */
void	WriteOut(sce,pdst,fsize)
    float *sce;
    float **pdst;
    long  fsize;	/* the frame size - but we may not copy them all! */
    {
    int     j;

    for(j=0; j<(2L*(fsize/2L + 1L)); ++j ) /* i.e. mg/ph for just over 1/2 */
	*(*pdst)++ = sce[j];		    /* pointer updated for next time */
    }	

/*--------------------------------------------------------------------*/
/*---------------------------- sinc module ---------------------------*/
/*--------------------------------------------------------------------*/

/* (ugens7.h) #define SPTS (16) /* How many points in each lobe */

void MakeSinc()		/* initialise our static sinc table */
    {
    int     i;
    int     stLen = SPDS*SPTS;	/* sinc table is SPDS/2 periods of sinc */
    float   theta   = 0.0;	/* theta (sinc arg) reaches pi in SPTS */
    float   dtheta  = SBW*PI/(float)SPTS;   /* SBW lowers cutoff to redc ali */
    float   phi     = 0.0;	/* phi (hamm arg) reaches pi at max ext */
    float   dphi    = PI/(float)(SPDS*SPTS);


    if(sncTab == NULL)
	sncTab = (float *)malloc((long)(stLen+1) * sizeof(float));
    /* (stLen+1 to include final zero; better for interpolation etc) */
/*    printf("Make sinc : pts = %d, table = %lx \n",stLen,sncTab);   */
    sncTab[0] =  1.0;
    for(i=1; i<=stLen; ++i) /* build table of sin x / x */
	{
	theta += dtheta;
	phi   += dphi;
	sncTab[i] = sin(theta)/theta * (.54 + .46*cos(phi));
	/* hamming window on top of sinc */
	}
    }

void DestroySinc()	/* relase the lookup table */
    {
    free(sncTab);
    }

float SincIntp(index)
    FLOATARG index;
/* Calculate the sinc of the 'index' value by interpolating the table */
/* <index> is scaled s.t. 1.0 is first zero crossing */
/* ! No checks ! */
    {
    int     nrst;
    float   frac,scaledUp;

    scaledUp = index * SPTS;
    nrst = (int)scaledUp;
    frac = scaledUp - (float)nrst;
    return(sncTab[nrst] + frac*(sncTab[nrst+1]-sncTab[nrst]) );
    }	

/****************************************/
/** prewarp.c module			*/
/****************************************/

/* spectral envelope detection: this is a very crude peak picking algorithm
	which is used to detect and pre-warp the spectral envelope so that
	pitch transposition can be performed without altering timbre.
	The basic idea is to disallow large negative slopes between
	successive values of magnitude vs. frequency. */

#ifndef NULL
#define NULL 0x0L
#endif


static	float	*env = (float *)NULL;	/* Scratch buffer to hold 'envelope' */

void PreWarpSpec(spec, size, warpFactor)
    float   *spec;	/* spectrum as magnitude,phase */
    long     size;	/* full frame size, tho' we only use n/2+1 */
    FLOATARG   warpFactor; /* How much pitches are being multd by */
    {
    float   eps,slope;
    float   mag, lastmag, nextmag, pkOld;
    long     pkcnt, i, j;

    if(env==(float *)NULL)
	env = (float *)malloc((long)size * sizeof(float));
    /*!! Better hope <size> in first call is as big as it gets !! */
    eps = -64. / size;		    /* for spectral envelope estimation */
    lastmag = *spec;
    mag = spec[2*1];
    pkOld = lastmag;
    *env = pkOld;
    pkcnt = 1;

    for (i = 1; i < someof(size); i++)	/* step thru spectrum */
	{
	if (i < someof(size)-1)
	    nextmag = spec[2*(i+1)];
	else nextmag = 0.;

	if (pkOld != 0.)
	    slope =
		((float) (mag - pkOld)/(pkOld * pkcnt));
	else
	    slope = -10.;

	/* look for peaks */
	if ((mag>=lastmag)&&(mag>nextmag)&&(slope>eps))
	    {
	    env[i] = mag;
	    pkcnt--;
	    for (j = 1; j <= pkcnt; j++)
		{
		env[i - pkcnt + j - 1]
		    = pkOld * (1. + slope * j);
		}
	    pkOld = mag;
	    pkcnt = 1;
	    }
	else
	    pkcnt++;		    /* not a peak */

	lastmag = mag;
	mag = nextmag;
	}

    if (pkcnt > 1)		    /* get final peak */
	{
	mag = spec[2*(size/2)];
	slope = ((float) (mag - pkOld) / pkcnt);
	env[size/2] = mag;
	pkcnt--;
	for (j = 1; j <= pkcnt; j++)
	    {
	    env[size/2 - pkcnt + j - 1] = pkOld + slope * j;
	    }
	}

    for (i = 0; i < someof(size); i++)	/* warp spectral env.*/
	{
	j = ((float) i * warpFactor);
	mag = spec[2*i];
	if ((j < someof(size)) && (env[i] != 0.))
	    spec[2*i] *= env[j]/env[i];
	else
	    spec[2*i] = 0.;
/*	printf("I<%d>J<%d>S<%.0f>E<%.0f>F<%.0f>T<%0.f>",
	    i, j, mag, env[i], env[j], spec[2*i]);  */
	}
    }


