/* Signals stubs.
   This file is part of Checker.
   Copyright 1994 Tristan Gingold
		  Written January 1994 by Tristan Gingold

This program is free software; you can redistribute it and/or
modify it under the terms of the GNU General Public License as
published by the Free Software Foundation; either version 2 of the
License, or (at your option) any later version.

This library is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
General Public License for more details.

You should have received a copy of the GNU General Public License 
along with this program; see the file COPYING.  If not, write to
the Free Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

 The author may be reached (Email) at the address gingold@amoco.saclay.cea.fr,
 or (US/French mail) as Tristan Gingold 
   			  8 rue Parmentier
   			  F91120 PALAISEAU
   			  FRANCE
*/
#include "checker.h"
#include "errlist.h"
#include <signal.h>

#ifdef CHKR_USE_BITMAP

#ifndef _NSIG
#define NSIGNALS 32
#else
#define NSIGNALS _NSIG
#endif

void _sig_jump_to(void *, void *);
void _sig_save(void *ptr, void *func, unsigned int mask);
static struct sigaction sig_tab[NSIGNALS+1];
int chkr_in_checker = 0;

/* The really used handler.
 * It calls your handler after changing the rights of the stack.
 */
static void 
chkr_sig_handler(int nsignal)
{
 if (chkr_in_checker)
   {
     /* The processus has received a signal, but it must not be delivered
      * now, since we are in Checker and Checker is not recursiv. So, we
      * have to store all information and to deliver the signal as soon as
      * possible
      */
     _sig_save(&nsignal, sig_tab[nsignal].sa_handler, sig_tab[nsignal].sa_mask);
     chkr_abort();
   }
 known_stack_limit = &nsignal;
 chkr_set_right(&nsignal, SIG_FRAME_SIZE, CHKR_RW);
 /* restore the true sa_mask */
 sigprocmask(SIG_SETMASK, &(sig_tab[nsignal].sa_mask), 0);
 _sig_jump_to(sig_tab[nsignal].sa_handler, &nsignal);	/* A jump */
}

/*
 * Save signal before main.
 */
void
save_signals(void)
{
 int i;
 struct sigaction t;
 
 for (i=1; i < NSIGNALS; i++)
   {
     chkr_sigaction(i, (struct sigaction*)0, &sig_tab[i]);
     if (i != SIGSTOP && i != SIGKILL 
         && sig_tab[i].sa_handler != SIG_DFL 
         && sig_tab[i].sa_handler != SIG_IGN)
       {
         t = sig_tab[i];
         t.sa_handler = chkr_sig_handler;
         t.sa_mask = ~0L;	/* block all signal */
         chkr_sigaction(i, &t, (struct sigaction*)0);
       }
   }
}

/*
 * The new sigaction syscall.
 */
int
__sigaction(int sig, struct sigaction *act, struct sigaction *oldact)
{
  int ret;
  struct sigaction action;

  chkr_check_addr(&sig, sizeof(int), CHKR_RO);
  chkr_check_addr(&act, sizeof(struct sigaction*), CHKR_RO);
  chkr_check_addr(&oldact, sizeof(struct sigaction*), CHKR_RO);
    
  if (act)
#ifdef linux
    chkr_check_addr(act, sizeof(struct sigaction) - sizeof(int), CHKR_RO);
#else
    chkr_check_addr(act, sizeof(struct sigaction), CHKR_RO);
#endif
  if (oldact)
    chkr_check_addr(oldact, sizeof(struct sigaction), CHKR_WO);  
  if (sig == SIGSTOP || sig == SIGKILL || act == (struct sigaction*)0)
    return chkr_sigaction(sig, act, oldact);
  if (act->sa_handler == SIG_DFL || act->sa_handler == SIG_IGN)
    {
      ret = chkr_sigaction(sig, act, oldact);
      if (oldact)
        {
          oldact->sa_handler = sig_tab[sig].sa_handler;
          oldact->sa_mask = sig_tab[sig].sa_mask;
        }
      return ret;
    }
   
  action = *act;
  action.sa_handler = chkr_sig_handler;		/* use the checker handler */
  action.sa_mask = ~0L;		/* mask all signals */
  ret = chkr_sigaction(sig, &action, oldact);
  if (oldact)
    {
      oldact->sa_handler = sig_tab[sig].sa_handler;
      oldact->sa_mask = sig_tab[sig].sa_mask;
    }
  sig_tab[sig] = *act;
  return ret;
}

#endif /* CHKR_USE_BITMAP */