/*
 * $Id: rip.c,v 1.1 1995/01/05 09:14:58 root Exp root $
 */
#include "ipxroute.h"
#include <linux/route.h>

/*
 * Adding a local route involves binding the network number to the
 * requested interface.  The 'unusual' thing here is when we try to
 * bind a number to the 'lo' device.  I have assumed that the 'lo'
 * device is the internal network number of the router.  It needs to
 * be set in the kernel, but we don't want it broadcast around to
 * other routers.
 */
IPXRouteEnt *add_local_route(char *dev,int frame,unsigned long net,IPXRouteEnt *p) {
  int skfd = socket(AF_IPX,SOCK_DGRAM,0); 
  struct sockaddr_ipx ipx_addr;
  struct ifreq intfc;
  IPXRouteEnt *old = p;
  int val = 1;

  if (skfd < 0) { 
    sysMsg(MSG_WARN,ENOENT,dev); 
    return p; 
  }

  ipx_addr.sipx_network = htonl(net);
  strcpy(intfc.ifr_name, dev);
  ipx_addr.sipx_type = frame;
  ipx_addr.sipx_port = 0;
  ipx_addr.sipx_family = AF_IPX;
  memset(&ipx_addr.sipx_node, 0, sizeof(ipx_addr.sipx_node));
  ipx_addr.sipx_special = 0;
  ipx_addr.sipx_action = IPX_CRTITF;

  memcpy(&intfc.ifr_addr, &ipx_addr, sizeof(ipx_addr));
  if (ioctl(skfd, SIOCSIFADDR, &intfc) < 0) {
    sysMsg(MSG_WARN, ERR_ERRNO, "binding network to interface");
  }
  /* The first network will be the primary...I hope the user put the
   * 'lo' device first..!
   */
  ioctl(skfd, SIOCAIPXPRISLT, &val);

  if (verbose)
    sysMsg(MSG_WARN,ERR_OK,"Adding local route: %s %08x %x",
	   dev,net,2);

  /* If this is the lo device, don't add it to the table, just
   * enter the route in the kernel.
   */
  if (!strcmp(dev, "lo"))
    return(p);

  /* Ok, everything succeeded, lets create a new entry */
  p = (IPXRouteEnt *)malloc(sizeof(IPXRouteEnt));
  if (!p) {
    sysMsg(MSG_WARN,ENOMEM,"malloc");
    ipx_addr.sipx_action = IPX_DLTITF;
    memcpy(&intfc.ifr_addr, &ipx_addr, sizeof(ipx_addr));
    ioctl(skfd, SIOCSIFADDR, &intfc);
    close(skfd);
    return old;
  }  
  p->network = net;
  p->rnetwork = IPX_ROUTE_NO_ROUTER;
  memset(p->rnode,0,sizeof p->rnode);
  p->hops = 1;
  p->ticks = 1;
  p->age = 0;
  p->frame = frame;
  p->next = old;

  close(skfd);
  return p;
}

void down_local_route(char *dev,int frame,unsigned long net) 
{
  int skfd = socket(AF_IPX, SOCK_DGRAM, 0);
  struct sockaddr_ipx ipx_addr;
  struct ifreq intfc;

  if (skfd < 0)
    {
      sysMsg(MSG_WARN, ENOENT, dev);
      return;
    }
  ipx_addr.sipx_network = htonl(net);
  ipx_addr.sipx_port = 0;
  strcpy(intfc.ifr_name, dev);
  ipx_addr.sipx_type = frame;
  ipx_addr.sipx_family = AF_IPX;
  memset(&ipx_addr.sipx_node, 0, sizeof(ipx_addr.sipx_node));
  ipx_addr.sipx_special = 0;
  ipx_addr.sipx_action = IPX_DLTITF;
  memcpy(&intfc.ifr_addr, &ipx_addr, sizeof(ipx_addr));
  ioctl(skfd, SIOCSIFADDR, &intfc);
}

void down_route(IPXRouteEnt *p) {
  int skfd = socket(AF_IPX,SOCK_DGRAM,0);
  struct sockaddr_ipx st,sg;
  struct rtentry routentry;

  if (skfd < 0) 
    { 
      sysMsg(MSG_WARN,ERR_ERRNO,"socket"); 
      return; 
    }
  routentry.rt_flags = RTF_GATEWAY;
  st.sipx_family = AF_IPX;
  sg.sipx_family = AF_IPX;
  st.sipx_network = htonl(p->network);
  memcpy(&routentry.rt_dst, &st, sizeof(st));
  memcpy(&routentry.rt_gateway, &sg, sizeof(sg));
  if (verbose)
    sysMsg(MSG_WARN,ERR_OK,"Removing route: %08X %X", p->network,
	   routentry.rt_flags);

  if (ioctl(skfd,SIOCDELRT,&routentry)<0)  
    sysMsg(MSG_WARN,ERR_ERRNO,"ipxroute-del");
  close(skfd);
}

void write_rip(struct sockaddr_ipx *skp, unsigned char iobuff[], int size)
{
  struct sockaddr_ipx sk;
  int sock;

  sock = socket(AF_IPX, SOCK_DGRAM, 0);
  sk.sipx_family = AF_IPX;
  sk.sipx_port = htons(IPXS_RIP);
  sk.sipx_network = skp->sipx_network;
  memset(&sk.sipx_node, 0, sizeof(sk.sipx_node));
  sk.sipx_type = 1;
  skp->sipx_type = 1;
  bind(sock, (struct sockaddr *)&sk, sizeof(sk));
  connect(sock, (struct sockaddr *)skp, sizeof(*skp));
  write(sock, iobuff, size);
  close(sock);
}

void broadcast_ipx(unsigned char iobuf[],unsigned long network,
		   unsigned port,int type,int size) {
  struct sockaddr_ipx sk;
  long soopt;
  int sock = socket(AF_IPX,SOCK_DGRAM,0);
  if (sock < 0) sysMsg(MSG_FATAL,ERR_ERRNO,"sock-out");

  sk.sipx_family = AF_IPX;
  sk.sipx_network = htonl(network);
  memset(sk.sipx_node,0xff,sizeof sk.sipx_node);
  sk.sipx_port = htons(IPXS_DYNAMIC);
  sk.sipx_type = type;

  soopt = bind(sock,(struct sockaddr *)&sk,sizeof sk);
  if (soopt < 0) sysMsg(MSG_FATAL,ERR_ERRNO,"broadcast_ipx:bind");
  soopt = TRUE;

  sk.sipx_family = AF_IPX;	/* Set IPX dest socket */
  sk.sipx_network = htonl(network);
  sk.sipx_port = htons(port);
  sk.sipx_type = type;
  memset(sk.sipx_node,0xff,sizeof sk.sipx_node); 

  if (connect(sock, (struct sockaddr *)&sk, sizeof(sk)) < 0)
    sysMsg(MSG_FATAL, ERR_ERRNO, "connect");
  if (write(sock, iobuf, size) < 0)
    sysMsg(MSG_FATAL, ERR_ERRNO, "write");
  close (sock);
}

void send_ripinfo(IPXRouteEnt *rtes,unsigned long net) {
  unsigned char iobuf[MAXNETBUF];
  IPXRipEnt *riptab = (IPXRipEnt *)(&iobuf[2]);
  int i=0;
  iobuf[0] = 0; iobuf[1] = IPX_RESP;	/* Header info */
  while (rtes) {
    if (rtes->network != net && rtes->rnetwork != net) {
      riptab[i].network = htonl(rtes->network);
      riptab[i].hops = htons(rtes->hops);
      riptab[i].ticks = htons(rtes->ticks);
      if (++i >= MAX_IPX_INFO) {
	broadcast_ipx(iobuf,net,IPXS_RIP,IPXT_RIP, (sizeof *riptab)*i+2);
	i = 0;
      }
    }
    rtes = rtes->next;
  }
  if (i) {
    broadcast_ipx(iobuf,net,IPXS_RIP,IPXT_RIP, (sizeof *riptab)*i+2);
  }
}

void broadcast_rip(IPXRouteEnt *rtes) {
  IPXRouteEnt *nets = net_table;
  while (nets) {
    send_ripinfo(rtes,nets->network);
    nets = nets->next;
  }
}

/*
 * Open socket
 */
int opensock(unsigned int port,int type) {
  struct sockaddr_ipx sk;
  long soopt=TRUE;
  int sock=socket(AF_IPX,SOCK_DGRAM,0);
  if (sock < 0) sysMsg(MSG_FATAL,ERR_ERRNO,"socket");

  memset(&sk,0,sizeof sk);
  sk.sipx_family = AF_IPX;
  sk.sipx_port = htons(port);
  sk.sipx_type = type;

  if (bind(sock,(struct sockaddr *)&sk,sizeof sk) < 0)
    sysMsg(MSG_FATAL,ERR_ERRNO,"opensock:bind"); 
  /*
   * Set the sock options
   */
  if ( setsockopt(sock,SOL_SOCKET,SO_BROADCAST,&soopt,sizeof soopt) < 0)
    sysMsg(MSG_FATAL,ERR_ERRNO,"setsockopt");
  return sock;
}

IPXRouteEnt *findRoute(unsigned long net,IPXRouteEnt *p) {
  while (p) {
    if (p->network == net) return p;
    p = p->next;
  }
  return NULL;
}

IPXRouteEnt *delRoute(IPXRouteEnt *todel,IPXRouteEnt *net) {
  IPXRouteEnt *prev = NULL, *p = net;

  while (p) {
    if (todel == p) {
      if (prev) {
	prev->next = p->next;
	return net;
      }
      else
	return p->next;
    }
    prev = p;
    p = p->next;
  }
  return net;
}

IPXRouteEnt *new_route(IPXRouteEnt *net,unsigned long rnet,int hops,int ticks,
		       unsigned char *node) {
  int skfd = socket(AF_IPX,SOCK_DGRAM,0);
  struct sockaddr_ipx sg, st;
  struct rtentry routentry;
  IPXRouteEnt *p = NULL;

  if (skfd < 0) { sysMsg(MSG_WARN,ERR_ERRNO,"socket"); return NULL; }
  routentry.rt_flags = RTF_GATEWAY;
  sg.sipx_family = AF_IPX;
  st.sipx_family = AF_IPX;
  st.sipx_network = htonl(rnet);
  sg.sipx_network = htonl(net->network);
  memcpy(sg.sipx_node, node, sizeof(sg.sipx_node));
  memcpy(&routentry.rt_gateway, &sg, sizeof(sg));
  memcpy(&routentry.rt_dst, &st, sizeof(sg));
  if (verbose)
    sysMsg(MSG_WARN,ERR_OK,"Adding remote route: %08x %x %08x:%s",
	   rnet,routentry.rt_flags,net->network,ipx_ntoa(node));

  if (ioctl(skfd,SIOCADDRT,&routentry) < 0) {
    sysMsg(MSG_WARN,ERR_ERRNO,"ipxroute-add");
    close(skfd);
    return NULL;
  }
  p = (IPXRouteEnt *)malloc(sizeof(IPXRouteEnt));
  if (!p) {
    sysMsg(MSG_WARN,ENOMEM,"addrte");
    ioctl(skfd,SIOCDELRT,&routentry);
    close(skfd);
    return NULL;
  }
  p->network = rnet;
  p->rnetwork = net->network;
  memcpy(p->rnode,node,sizeof p->rnode);
  p->hops = hops;
  p->ticks = ticks;
  p->age = time(NULL);
  p->frame = net->frame;
  p->next = NULL;
  close(skfd);
  return p;
}


void process_req(struct sockaddr_ipx *sk,IPXRipEnt *p,int k,IPXRouteEnt *rtes) {
  unsigned char pkt[MAXNETBUF];
  IPXRipEnt *op = (IPXRipEnt *)&pkt[2];
  int i,j;
  unsigned long srcnet = ntohl(sk->sipx_network);
  int loop;

  pkt[0] = 0; pkt[1] = IPX_RESP;
  if (!k) k = 1;

  for (j=i=0;i<k;i++) {
    IPXRouteEnt *rte = net_table;
    loop = TRUE;

    while (rte) {
      if ((srcnet != rte->network && srcnet != rte->rnetwork &&
	   p[i].network==IPX_GEN_REQ) || ntohl(p[i].network)==rte->network) {
	op[j].network = htonl(rte->network);
	op[j].hops = htons(rte->hops);
	op[j].ticks = htons(rte->ticks);
	if (++j >= MAX_IPX_INFO) {
	  write_rip(sk,pkt,j*sizeof(IPXRipEnt)+2);
	  j = 0;
	}
      }
      rte = rte->next;
      if (loop && !rte) {
	loop = FALSE;
	rte = rtes;
      }
    }
  }
  if (j) write_rip(sk,pkt,j*sizeof(IPXRipEnt)+2);
}

IPXRouteEnt *read_rip(int sock,IPXRouteEnt *rtes) {
  unsigned char pkt[MAXNETBUF];
  struct sockaddr_ipx sk;
  int frlen = sizeof sk;
  IPXRouteEnt *modRtes = NULL;
  int doreq = FALSE;

  int read_ret=recvfrom(sock,pkt,sizeof pkt,0,(struct sockaddr *)&sk,&frlen);

  if (read_ret < 0) { sysMsg(MSG_WARN,ERR_ERRNO,"recv-rip"); return rtes; }

  if (read_ret > 0) {
    int pktype = pkt[0] * 256 + pkt[1], i = 2;
    IPXRouteEnt *netinfo = findRoute(ntohl(sk.sipx_network),net_table);

    switch (pktype) {
    case IPX_RESP:
      if (!netinfo) return rtes;
      while (i < read_ret) {
	IPXRipEnt *rip = (IPXRipEnt *)(pkt+i);
	unsigned long network =ntohl(rip->network);
	unsigned short hops = ntohs(rip->hops)+1;
	unsigned short ticks = ntohs(rip->ticks)+1;
	IPXRouteEnt *rp = findRoute(network,net_table);

	i += sizeof(IPXRipEnt);

	if (rp) return rtes;
	rp = findRoute(network,rtes);
	if (rp) {	/* A route to that network exists already */
	  if ( ntohl(sk.sipx_network) == rp->rnetwork &&
	      !memcmp(sk.sipx_node,rp->rnode,sizeof sk.sipx_node)) {
	    /* Ok, this is info about the current route */
	    if (hops >= IPX_HOP_LIMIT) {	/* We wanna delete it... */
	      rtes = delRoute(rp,rtes);
	      rp->next = modRtes;
	      rp->hops = IPX_HOP_LIMIT;
	      modRtes = rp;
	      doreq = TRUE;
	      down_route(rp);
	    }
	    else {
	      rp->hops = hops+1;	/* Refresh route */
	      rp->ticks = ticks+1;
	      rp->age = time(NULL);
	    }
	    continue;
	  }
	  /* Hey a better route! */
	  else if (ticks+1 < rp->ticks || 
		   (ticks+1 == rp->ticks && hops+1 < rp->hops)) {
	    rtes = delRoute(rp,rtes);
	    down_route(rp);
	    free(rp);
	    /* We just get rid of the entry and add it as new... */
	  }
	  else continue;	/* Useless broadcast for us...*/
	}
	else if (hops >= IPX_HOP_LIMIT) continue;

	rp = new_route(netinfo,network,hops,ticks,sk.sipx_node);
	if (rp) {	/* Created a new route... */
	  rp->next = rtes;
	  rtes = rp;
	  rp = (IPXRouteEnt *)malloc(sizeof(IPXRouteEnt));
	  if (rp) {	/* Add it to the modification table */
	    *rp = *rtes;
	    rp->next = modRtes;
	    modRtes = rp;
	  }
	}
      }
      if (modRtes) {
	broadcast_rip(modRtes);	/* Broadcast changes */
	/* and delete the modified routes table */
	for (netinfo = modRtes; netinfo ; netinfo = modRtes) {
	  modRtes = netinfo->next;
	  free(netinfo);
	}
      }
      break;
    case IPX_REQ:
      process_req(&sk,(IPXRipEnt *)&pkt[2],(read_ret-2)/sizeof(IPXRipEnt),rtes);
      break;
    default:
      sysMsg(MSG_WARN,EINVAL,"Unrecognized IPX/RIP packet: %04x",pktype);
    }
  }
  if (doreq) {
    alarm(1);
    signal(SIGALRM,general_request);
  }
  dump_ipxtab(rtes);
  return rtes;
}

IPXRouteEnt *maxhop_rip(IPXRouteEnt *rtes)
{
  IPXRouteEnt *p = rtes;

  while (p)
    {
      p->hops = IPX_HOP_LIMIT;
      p = p->next;
    }
  dump_ipxtab(rtes);
  return(rtes);
}

IPXRouteEnt *age_rip(IPXRouteEnt *rtes) {
  unsigned long now = time(NULL);
  IPXRouteEnt *rmved = NULL, *p = rtes;

  while (p) {
    if (now - p->age > IPX_AGE_LIMIT) {
      IPXRouteEnt *rp = p;
      p = p->next;
      rtes = delRoute(rp,rtes);
      down_route(rp);
      rp->next = rmved;
      rp->hops = IPX_HOP_LIMIT;
      rmved = rp;
    }
    else p = p->next;
  }
  if (rmved) {
    IPXRouteEnt *next;
    broadcast_rip(rmved);
    /* and free routes table */
    for (p = rmved; p ; p = next) {
      next = p->next;
      free(p);
    }
    alarm(1);
    signal(SIGALRM,general_request);
  }
  dump_ipxtab(rtes);
  return rtes;
}

/* Dump IPX table... */
void dump_ipx(IPXRouteEnt *p, FILE *fp) {
  while (p) {
    char rte[23], *tm, *zm;
    if (p->rnetwork) {
      zm = rte;
      sprintf(rte,"%08lX:%s",p->rnetwork,ipx_ntoa(p->rnode));
    }
    else  zm = "(local)";

    tm = p->age ? ctime(&p->age) : "(static)\n";
    fprintf(fp,"%08lX %-21s %4d %4d %s", p->network, zm, p->hops,p->ticks,tm);
    p = p->next;
  }
}

void dump_ipxtab(IPXRouteEnt *rtes) {
  FILE *fp = fopen(IPXDUMP_FILE,"w");
  if (!fp) return;

  fprintf(fp,"IPXRIPD PID=%d\n",getpid());
  fputs("Network  Router                Hops Tcks Age\n",fp);
  fputs("======== ===================== ==== ==== ====================\n",fp);
  dump_ipx(net_table,fp);
  dump_ipx(rtes,fp);
  fclose(fp);
}

void display_ipxtab(IPXRouteEnt *rtes) {
  fprintf(stdout,"IPXRIPD PID=%d\n",getpid());
  fputs("Network  Router                Hops Tcks Age\n",stdout);
  fputs("======== ===================== ==== ==== ====================\n",stdout);
  dump_ipx(net_table,stdout);
  dump_ipx(rtes,stdout);
}
