/*
 * Author: Paul Ashton
 * Date written: September 12th 1992
 *
 * Files: stdin - team's solution.
 *        progin - input used to produce solution
 *        judgeout - output of judging program where progin was the
 *                   input file.
 *
 * Algorithm: For each set of boxes in progin:
 *            1. The set of boxes are read in from progin.
 *            2. The judge's and team's solutions are read in.
 *            3. The team's number of boxes is compared with the
 *               judges.
 *            4. It is checked to see whether the team's set of boxes
 *               nests.
 */

#include <stdio.h>
#include <ctype.h>

#define PROGIN "../../input/d"
#define JUDGEOUT "../../answers/d"

#define MAXDIMS 10
#define MAXBOXES 30

typedef int BoxType[MAXDIMS];	/* represents one box */
BoxType boxlist[MAXBOXES+1];	/* represents all the boxes 1..numboxes */

void SortBox();
void ReadAnswer();

static char *progname;
static int boxsetnum = 0;              /* Used in diagnostics */
int error = 0;

main(argc, argv)
        int argc;
        char *argv[];
{
	FILE *progin, *judgeout;
	int numboxes;			/* number of boxes */
	int dims;			/* number of dimensions */
	int b, d;			/* box number, dimension number */
	int maxjudge, maxteam;          /* team and judge's maximum */
	int teamorder[MAXBOXES];        /* team's order */
	
	progname = argv[0];

	if ((progin = fopen(PROGIN, "r")) == 0) {
		fprintf(stderr, "%s: failed to open %s\n", progname, PROGIN);
		exit(1);
	}
	if ((judgeout = fopen(JUDGEOUT, "r")) == 0) {
		fprintf(stderr, "%s: failed to open %s\n", progname, JUDGEOUT);
		exit(1);
	}

	/* Repeatedly read data sets */
	while (fscanf(progin, "%d %d\n", &numboxes, &dims) == 2) {
		boxsetnum++;
		for (b = 1; b <= numboxes; b++) {
			for (d = 0; d < dims; d++)
			  fscanf(progin, "%d ", &boxlist[b][d]);
			SortBox(boxlist[b], dims);
		}
		ReadAnswer(stdin, &maxteam, teamorder, "Team");
		ReadAnswer(judgeout, &maxjudge, (int *)0, "Judge");

		if (maxteam != maxjudge) {
			printf(
	    "Team (%d boxes) disagrees with judge (%d boxes) for box set %d\n",
			       maxteam, maxjudge, boxsetnum);
			error = 1;
		} else {
			for (b = 0 ; b < maxteam - 1 ; b++) {
				if (!DoesNestIn(boxlist[teamorder[b]], 
					     boxlist[teamorder[b+1]], dims)) {
					error = 1;
					printf("Error in Team answer for box set %d\n",
					       boxsetnum);
					printf("\t%d does not nest in %d\n",
					       teamorder[b], teamorder[b+1]);
				}
			}
		}
	}
	CheckEof(stdin, "Team");
	if (!error) {
		printf("Team's solution is correct\n");
	}
}


/* Sort a box's dimensions */
void
SortBox(box, dims)
	BoxType box;			/* a box */
	int dims;				/* number of dimensions */
{
    int i,j;
    int temp;

    for (i = 0; i < dims-1; i++) {
	   for (j = i+1; j < dims; j++) {
		  if (box[i] > box[j]) {
			 temp = box[i];
			 box[i] = box[j];
			 box[j] = temp;
		  }
	   }
    }
}


char *curpos;
char buffer[BUFSIZ];

	void
ReadLine(infile, name)
        FILE *infile;
        char *name;
{
        if (fgets(buffer, sizeof(buffer), infile) == 0) {
                printf("%s: read line failed for file %s\n",
                       progname, name);
                exit(1);
        }
	if (buffer[strlen(buffer)-1] != '\n') {
                printf("%s: line too long for file %s (longer than %d chars)\n",
                       progname, name, BUFSIZ-1);
                exit(1);
        }
	curpos = buffer;
}


        char *
skipspace(cp)
        char *cp;
{
	while (*cp && isspace(*cp)) {
		cp++;
	}
	return cp;
}


        void
CheckRestEmpty(name)
        char *name;
{
	if (*skipspace(curpos) != '\0') {
		printf("Trailing garbage reading box set %d for %s\n",
		       boxsetnum, name);
		error = 1;
	}
}


CheckEof(infile, name)
        FILE *infile;
        char *name;
{
	if (fgetc(infile) != EOF) {
		printf("Trailing garbage after expected EOF for %s\n", name);
		error = 1;
	}
}


        void
ReadAnswer(infile, max, order, name)
        FILE *infile;
        int *max;
        int *order;
{
	int i, temp;

	ReadLine(infile, name);
	*max = ReadInt(name);
	CheckRestEmpty(name);

	ReadLine(infile, name);
	for (i = 0 ; i < *max ; i++) {
		temp = ReadInt(name);
		if (order) {
			order[i] = temp;
		}
	}
	CheckRestEmpty(name);
}


        int
ReadInt(name)
        char *name;
{
	int temp;
	char *cp;

	curpos = skipspace(curpos);
	if (*curpos == 0) {
              printf("Missing value for box set %d for %s\n",
                       boxsetnum, name);
	      exit(1);
        }

        temp = strtol(curpos, &cp, 10);
	if (curpos == cp) {
                printf("Illegal integer for box set %d for %s\n",
                       boxsetnum, name);
		exit(1);
        }
	curpos = cp;
        return temp;
}

		  
int					/* does block a nest in block b? */
DoesNestIn(a,b, dims)
	BoxType a,b;			/* the two boxes */
	int dims;				/* number of dimensions */
{
    int i;

    for (i = 0; i < dims; i++) {
	   if (a[i] >= b[i])
		return(0);
    }
    return(1);
}
