/*  
    Status line management file
    Copyright (c) Tudor Hulubei & Andrei Pitis, May 1994

This file is part of UIT (UNIX Interactive Tools)

UIT is free software; you can redistribute it and/or modify it under the
terms of the GNU General Public License as published by the Free Software
Foundation; either version 2, or (at your option) any later version.

UIT is distributed in the hope that it will be useful, but WITHOUT ANY
WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
FOR A PARTICULAR PURPOSE.  See the GNU General Public License for more
details .

You should have received a copy of the GNU General Public License along with
UIT; see the file COPYING.  If not, write to the Free Software Foundation,
675 Mass Ave, Cambridge, MA 02139, USA.  */


#include <string.h>
#include "tty.h"
#include "window.h"
#include "status.h"


extern int  LinuxConsole;
extern int  ColorMonitor;
extern char cSection[];
extern char bwSection[];


static window *status_win;
static char stemp[256];
static int columns;
static int line;
static char *default_msg;
static char dummy[] = "";


#define min(a, b) ((a) <= (b) ? (a) : (b))


#define STATUSBAR_FIELDS	9

static char StatusBarFields[STATUSBAR_FIELDS][40] = 
{
    "StatusBarForeground",
    "StatusBarBackground",
    "StatusBarBrightness",
    "StatusBarWarningForeground",
    "StatusBarWarningBackground",
    "StatusBarWarningBrightness",
    "StatusBarErrorForeground",
    "StatusBarErrorBackground",
    "StatusBarErrorBrightness"
};

#ifdef HAVE_LINUX
static int StatusBarColors[STATUSBAR_FIELDS] =
{
    BLACK, CYAN, OFF, BLACK, WHITE, OFF, WHITE, RED, ON
};
#else
static int StatusBarColors[STATUSBAR_FIELDS] =
{
    BLACK, WHITE, OFF, BLACK, WHITE, OFF, BLACK, WHITE, ON
};
#endif

#define StatusBarForeground 		StatusBarColors[0]
#define StatusBarBackground 		StatusBarColors[1]
#define StatusBarBrightness 		StatusBarColors[2]
#define StatusBarWarningForeground 	StatusBarColors[3]
#define StatusBarWarningBackground 	StatusBarColors[4]
#define StatusBarWarningBrightness 	StatusBarColors[5]
#define StatusBarErrorForeground 	StatusBarColors[6]
#define StatusBarErrorBackground 	StatusBarColors[7]
#define StatusBarErrorBrightness 	StatusBarColors[8]


void status_init(int _columns, int _begin_y, 
                 char *def_msg, configuration *config)
{
    char *data, *section;
    int sectionptr, index, i;
    
    columns = _columns;
    line    = _begin_y;
    default_msg = def_msg;
    status_win = window_init(0, _begin_y, 1, _columns);

    if (configuration_getstatus(config) == STATUS_OK)
    {
	section = (LinuxConsole && ColorMonitor) ? cSection : bwSection;

	sectionptr = configuration_getsectionptr(config, section);
        if (sectionptr != -1)
            for (i = 0; i < STATUSBAR_FIELDS; i++)
            {
                configuration_getfielddata(config, sectionptr,
					   StatusBarFields[i], &data,
					   1, DO_SEEK);
                if (!data || (index = tty_getcolorindex(data)) == -1)
                    fprintf(stderr, "Invalid %s (%s).\n",
			    StatusBarFields[i], data);
	        else
                    StatusBarColors[i] = index;
            }
    }
} 


void status_end(void)
{
    free(status_win);
}


int status(char *msg_name, int wait, int sound, int restore, int msg_type)
{
    tty_status status;
    int i, len, key = 0;
    struct key_struct *ks;
    

    tty_save(&status);
    
    memset(stemp, ' ', columns);
    if (msg_name == NULL) msg_name = dummy;
    len = strlen(msg_name);
    if (len < columns)
        memcpy(stemp + ((columns - len) >> 1), msg_name, len);
    else
        memcpy(stemp, msg_name, min(len, columns));

    for (i = 0; i < columns; i++)
        if (stemp[i] == '\r' || stemp[i] == '\n' || stemp[i] == '\t')
            stemp[i] = ' ';

    if (sound) tty_beep();
    switch (msg_type)
    {
        case MSG_WARNING:

	    tty_bright(StatusBarWarningBrightness);
	    tty_foreground(StatusBarWarningForeground);
	    tty_background(StatusBarWarningBackground);
	    break;
             
        case MSG_ERROR:

	    tty_bright(StatusBarErrorBrightness);
	    tty_foreground(StatusBarErrorForeground);
	    tty_background(StatusBarErrorBackground);
	    break;
             
        default:

	    tty_bright(StatusBarBrightness);
	    tty_foreground(StatusBarForeground);
	    tty_background(StatusBarBackground);
	    break;
    }
    window_cursormove(status_win, 0, 0);
    window_write(stemp, columns);

    if (wait)
    {
        while ((ks = tty_getkey(NULL)) == KEY_NOKEY)
            window_cursormove(status_win, line, columns);
	key = ks->key_seq[0];
    }

    if (restore)
    {
        memset(stemp, ' ', columns);
        len = strlen(default_msg);
        if (len < columns)    
            memcpy(stemp + ((columns - len) >> 1), default_msg, len);
        else
            memcpy(stemp, default_msg, columns);

        tty_bright(StatusBarBrightness);
        tty_foreground(StatusBarForeground);
        tty_background(StatusBarBackground);
        window_cursormove(status_win, 0, 0);
        window_write(stemp, columns);
    }
    
    tty_restore(&status);
    return key;
}
