/* Main program for the MouseLess Commander
   Copyright (C) 1994 Miguel de Icaza, Mauricio Plaza
   
   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.
   
   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.  */

#include <ncurses.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <sys/param.h>
#include <errno.h>
#include <pwd.h>

#include "dir.h"
#include "color.h"
#include "panel.h"
#include "input.h"
#include "global.h"
#include "dialog.h"
#include "menu.h"
#include "file.h"
#include "main.h"
#include "util.h"
#include "win.h"
#include "user.h"

static char rcsid [] = "$Header: /usr/users/miguel/c/CVS/nc/main.c,v 1.2 1994/05/10 20:49:07 miguel Exp $";

/* The structures for the panels */
Panel left_panel, right_panel;

/* Pointers to the selected and unselected panel */
Panel *current_panel, *other_panel;

Input cmdline;
char  input_buffer [255];

/* Usefull macros to avoid too much typing */
#define cpanel current_panel
#define opanel other_panel

/* Set when main loop should be terminated */
int   quit = 0;

/* If true, marking a files moves the cursor down */
int   mark_moves_down = 1;

/* If true, at startup the user-menu is invoked */
int   auto_menu = 0;

/* If true, after executing a command, wait for a keystroke */
int   pause_after_run = 1;

/* It true saves the setup when quitting */
int auto_save_setup = 0;

/* If true, next character is quoted */
int quote = 0;

char *prompt = "$ ";

/* WINDOW handles for the command line and the function keys */
WINDOW *cmdline_win;
WINDOW *fkeys;
WINDOW *clean_screen;

/* The currently selected file */
file_entry *selection;

/* The home directory */
char *home_dir;

/* The value of the other directory, only used when loading the setup */
char *other_dir = 0;

/* Ugly hack in order to distinguish between left and right panel in menubar */
int is_right;
#define MENU_PANEL (is_right ? &right_panel : &left_panel)

/* Quick search status */
char search_buffer [256] = { 0 };
char cmd_buf [512];

extern char *sys_errlist [];

void save_setup_cmd ();

inline void unselect_item (Panel *panel)
{
    set_attr (cpanel, 0, selection->f.marked);
    repaint_file (cpanel, cpanel->selected);
    wrefresh (cpanel->win_file);
}

inline void select_item (Panel *panel)
{
    selection = &current_panel->dir [current_panel->selected];
    set_attr (cpanel, 1, selection->f.marked);
    repaint_file (cpanel, cpanel->selected);
    wrefresh (cpanel->win_file);
    display_mini_info (cpanel);
}

init_panels ()
{
    Panel *x = &left_panel;

    init_panel (&left_panel, 0, 0, COLS/2, LINES-2);
    left_panel.active = 1;
    left_panel.count  = do_load_dir (&left_panel.dir, left_panel.sort_type);
    if (other_dir)
	chdir (other_dir);
    init_panel (&right_panel, COLS/2, 0, COLS, LINES-2);
    right_panel.count = do_load_dir (&right_panel.dir, right_panel.sort_type);
    if (other_dir)
	chdir (left_panel.cwd);

    current_panel = &left_panel;
    other_panel   = &right_panel;
    
    select_item (current_panel);/* Inits selection variable */
    display_mini_info (cpanel);	/* It needs selection defined */
    display_mini_info (opanel);	/* Same here */

    /* Now, update the screen */
    paint_panel (&left_panel);
    paint_panel (&right_panel);
}

init_entry ()
{
    cmdline_win = newwin (1, COLS, LINES-2, 0);
    wprintw (cmdline_win, prompt);
    raw ();
    noecho ();
    keypad (stdscr, TRUE);
    wrefresh (cmdline_win);
}

init_labels ()
{
    fkeys   = new_fkey ();
    define_label (fkeys, 1, "Help");
    define_label (fkeys, 2, "Menu");
    define_label (fkeys, 3, "View");
    define_label (fkeys, 4, "Edit");
    define_label (fkeys, 5, "Copy");
    define_label (fkeys, 6, "RenMov");
    define_label (fkeys, 7, "Mkdir");
    define_label (fkeys, 8, "Delete");
    define_label (fkeys, 9, "PullDn");
    define_label (fkeys, 10, "Quit");

    /* workaround ncurses 1.8.5 bug */
    wmove (fkeys, 0, 0);
    waddch (fkeys, '1');
    
    wrefresh (fkeys);
}

void update_panels (char *current_file, char *other_file)
{
    int free_pointers = 0;
    
    /* If current_file == -1 (an invalid pointer, then preserve selection */
    if (current_file == (char *) -1){
	free_pointers = 1;
	current_file = strdup (cpanel->dir [cpanel->selected].fname);
	other_file   =  strdup (opanel->dir [opanel->selected].fname);
    }
    panel_reload (cpanel);
    try_to_select (cpanel, current_file);
    panel_reload (opanel);
    try_to_select (opanel, other_file);
    paint_dir (cpanel);
    paint_dir (opanel);
    if (free_pointers){
	free (current_file);
	free (other_file);
    }
    chdir (cpanel->cwd);
}

void pre_exec ()
{
    wstandend (clean_screen);
    werase (clean_screen);
    touchwin (clean_screen);
    wrefresh (clean_screen);
    keypad (stdscr, FALSE);
    reset_shell_mode ();
}

void post_exec ()
{
    reset_prog_mode ();
    cbreak ();
    keypad (stdscr, TRUE);
}

void execute (char *command)
{
    int  i;
    char buf [4096];
#ifdef OLD_CODE
    char *current_file = strdup (selection->fname);
    char *other_file   = strdup (selection->fname);
#endif
    struct passwd *pw_data;

    pw_data = getpwuid (geteuid ());
    sprintf (buf, "%s -c \"%s\"", pw_data->pw_shell, command);

    pre_exec ();
    system (buf);
    post_exec ();

    if (pause_after_run){
	printf ("Press any key to continue...");
	fflush (stdout);
	getch ();
    }
#ifdef OLD_CODE
    update_panels (current_file, other_file);
#endif
    update_panels ((char *) -1, 0);
    repaint_screen (RP_CLEAR);
#ifdef OLD_CODE
    free (current_file);
    free (other_file);
#endif
}

void change_panel ()
{
    if (other_panel->is_status)
	return;

    current_panel->active = 0;
    show_dir (current_panel);
    unselect_item (current_panel);
    other_panel = current_panel;
    if (current_panel == &left_panel)
	current_panel = &right_panel;
    else
	current_panel = &left_panel;
    current_panel->active = 1;
    if (chdir (current_panel->cwd) != 0){
	message (1, " Error ", " Can't chdir to %s ", current_panel->cwd);
    }
    show_dir (current_panel);
    select_item (current_panel);
}

void cmd_quit ()
{
    if (query_dialog (" The MouseLess Commander  ",
		      " Do you really want to quit the Mouse-less commander? ",
		      0, 2, " Yes ", " No ") == 0)
	quit = 1;
}

int ITEMS (Panel *p)
{
    if (p->view_type == view_brief)
	return p->lines * 2;
    else
	return p->lines;
}

void move_down ()
{
    if (cpanel->selected+1 == cpanel->count)
	return;
    
    unselect_item (cpanel);
    cpanel->selected++;
    
    if (cpanel->selected - cpanel->top_file == ITEMS (cpanel)){
	/* Scroll window half screen */
	cpanel->top_file += ITEMS (cpanel)/2;
	paint_dir (cpanel);
	select_item (cpanel);
    } else 
	select_item (cpanel);
    panel_refresh (cpanel);
}

void move_up ()
{
    if (cpanel->selected == 0)
	return;
    
    unselect_item (cpanel);
    cpanel->selected--;
    if (cpanel->selected < cpanel->top_file){
	/* Scroll window half screen */
	cpanel->top_file -= ITEMS (cpanel)/2;
	if (cpanel->top_file < 0) cpanel->top_file = 0;
	paint_dir (cpanel);
    } else
	select_item (cpanel);
    panel_refresh (cpanel);
}

void move_home ()
{
    if (cpanel->selected == 0)
	return;
    cpanel->top_file = 0;
    cpanel->selected = 0;
    paint_dir (cpanel);
    panel_refresh (cpanel);
}

void move_end ()
{
    if (cpanel->selected == cpanel->count-1)
	return;
    cpanel->selected = cpanel->count-1;
    cpanel->top_file = cpanel->selected - ITEMS (cpanel) + 1;
    if (cpanel->top_file < 0) cpanel->top_file = 0;
    paint_dir (cpanel);
    panel_refresh (cpanel);
}

void prev_page ()
{
    cpanel->selected -= ITEMS (cpanel);
    cpanel->top_file -= ITEMS (cpanel);
    if (cpanel->selected < 0) cpanel->selected = 0;
    if (cpanel->top_file < 0) cpanel->top_file = 0;
    paint_dir (cpanel);
    panel_refresh (cpanel);
}

void next_page ()
{
    cpanel->selected += ITEMS (cpanel);
    cpanel->top_file += ITEMS (cpanel);
    if (cpanel->selected >= cpanel->count)
	cpanel->selected = cpanel->count - 1;
    if (cpanel->top_file >= cpanel->count)
	cpanel->top_file = cpanel->count - 1;
    paint_dir (cpanel);
    panel_refresh (cpanel);
}

void mark_file ()
{
    if (S_ISDIR (selection->buf.st_mode))
	return;
    
    selection->f.marked = selection->f.marked ? 0 : 1;
    set_attr  (cpanel, 1, selection->f.marked);
    repaint_file (cpanel, cpanel->selected);
    if (selection->f.marked){
	cpanel->marked++;
	cpanel->total += selection->buf.st_size;
    } else {
	cpanel->marked--;
	cpanel->total -= selection->buf.st_size;
    }
    if (mark_moves_down)
	move_down ();
    wrefresh (cpanel->win_file);
}

/* This routine untouches the first line on both panels in order */
/* to avoid the refreshing the menu bar */
void untouch_bar ()
{
    touchwin (cpanel->win_file);
    touchwin (opanel->win_file);
    wtouchln (cpanel->win_file, 0, 1, 0);
    wtouchln (opanel->win_file, 0, 1, 0);
    wrefresh (cpanel->win_file);
    wrefresh (opanel->win_file);
}

void clr_scr ()
{
	wstandend (clean_screen);
	wclear (clean_screen);
	touchwin (clean_screen);
	wrefresh (clean_screen);
}

void repaint_screen (int clear)
{
    if (clear){
	clr_scr ();
    }
    touchwin (current_panel->win_file);
    panel_refresh (current_panel);
    touchwin (other_panel->win_file);
    panel_refresh (other_panel);
    
    touchwin (cmdline_win);
    touchwin (fkeys);
    
    wrefresh (cmdline_win);
    wrefresh (fkeys);
}

void repaint_screen_cmd ()
{
    repaint_screen (RP_CLEAR);
}

void refresh_screen ()
{
    touchwin (current_panel->win_file);
    touchwin (other_panel->win_file);
    touchwin (cmdline_win);
    touchwin (fkeys);
    wrefresh (current_panel->win_file);
    wrefresh (other_panel->win_file);
    wrefresh (cmdline_win);
    wrefresh (fkeys);
}

void help_cmd ()
{
    interactive_display (HELPFILE);
}

int do_cd (char *new_dir)
{
    int ret;

    if (!*new_dir)
	ret = chdir (home_dir);
    else
	ret = chdir (new_dir);
    if (ret == -1)
	return 0;
    clean_dir (&cpanel->dir, cpanel->count);
    cpanel->count = do_load_dir (&cpanel->dir, cpanel->sort_type);
    cpanel->top_file = 0;

    cpanel->selected = 0;
    cpanel->marked = 0;
    getwd (cpanel->cwd);
    return 1;
}

void enter ()
{
    char *old_dir;

    if (strlen (cmdline.buffer)){
	if (strncmp (cmdline.buffer, "cd ", 3) == 0 ||
	    strcmp (cmdline.buffer, "cd") == 0){
	    if (cmdline.buffer [2] == 0)
		cmdline.buffer [3] = 0;
	    if (!do_cd (&cmdline.buffer [3])){
		message (1, " Error ", " Can't chdir to '%s' ",
			 &cmdline.buffer [3]);
		return;
	    }
	    paint_panel (cpanel);
	    select_item (cpanel);
	    *cmdline.buffer = 0;
	    cmdline.pos = 0;
	    update_input (&cmdline);
	} else {
	    execute (cmdline.buffer);
	    cmdline.buffer [0] = 0;
	    cmdline.pos = 0;
	    update_input (&cmdline);
	}
	return;
    }
    if (S_ISDIR (selection->buf.st_mode) || link_isdir (selection)){
	if (!strcmp (selection->fname, ".."))
	    old_dir = strdup (cpanel->cwd);
	else
	    old_dir = 0;
	if (!do_cd (selection->fname))
	    return;
	try_to_select (cpanel, old_dir);
	free (old_dir);
	paint_panel (cpanel);
	select_item (cpanel);
	return;
    }
    if (is_exe (selection->buf.st_mode)){
	/* Need to append "./" to the command */
	execute (selection->fname);
	return;
    }
    /* Code for extension dependent execution goes here */
}

void default_key (int c_code)
{
    int i;
    int l;
    
    if (c_code & 0x80 || search_buffer [0]){
	l = strlen (search_buffer);
	search_buffer [l] = c_code & 0x7f;
	search_buffer [l+1] = 0;
	l++;
	
	for (i = 0; i < cpanel->count; i++){
	    if (strncmp (cpanel->dir [i].fname, search_buffer, l) == 0){
		cpanel->selected = i;
		cpanel->top_file = cpanel->selected - cpanel->lines/2;
		if (cpanel->top_file < 0)
		    cpanel->top_file = 0;
		
		select_item (cpanel);
		paint_panel (cpanel);
		break;
	    }
	}
    } else 
	handle_char (&cmdline, c_code);
}

void view_cmd ()
{
    static char *viewer = 0;
    
    if (!viewer){
	viewer = getenv ("PAGER");
	if (!viewer)
	    viewer = "view";
    }
    sprintf (cmd_buf, "%s %s", viewer, selection->fname);
    execute (cmd_buf);
}

void edit_cmd ()
{
    static char *editor = 0;

    if (!editor){
	editor = getenv ("EDITOR");
	if (!editor)
	    editor = "vi";
    }
    sprintf (cmd_buf, "%s %s", editor, selection->fname);
    execute (cmd_buf);
}

char *get_file (Panel *panel)
{
    int i;
    
    if (panel->marked){
	for (i = 0; i < panel->count; i++)
	    if (panel->dir [i].f.marked)
		return panel->dir [i].fname;
    } else
	return panel->dir [panel->selected].fname;
}

void copy_cmd ()
{
    char *dest;
    int  i, c, stat_r;
    struct stat s;

    if (cpanel->marked > 1)
	sprintf (cmd_buf, " Copy %d files to:", cpanel->marked);
    else {
	if (!S_ISREG (selection->buf.st_mode))
	    return;
	sprintf (cmd_buf," Copy \"%s\" file to:",name_trunc(selection->fname, 30));
    }
    dest = input_dialog (" Copy ", cmd_buf, other_panel->cwd);
    if (!dest)
	return;
    
    stat_r = stat (dest, &s);
    if (S_ISREG (s.st_mode) || stat_r != 0){
	if (cpanel->marked > 1){
	    message (1, " Error ", "Can't copy multiple files to one file");
	    return;
	}
	copy_file_file (get_file (cpanel), dest);
    } else {
	if (!S_ISDIR (s.st_mode)){
	    message (1, " Copy ", "Unknown copy destination");
	    return;
	}
	/* Destination is a directory */
	if (!cpanel->marked)
	    copy_file_dir (selection->fname, dest);
	else {
	    nodelay (stdscr, TRUE);
	    for (i = 0; i < cpanel->count; i++){
		if (cpanel->dir [i].f.marked){
		    copy_file_dir (cpanel->dir [i].fname, dest);
		    cpanel->dir [i].f.marked = 0;
		    cpanel->marked--;
		    cpanel->total -= cpanel->dir [i].buf.st_size;
		}
		if ((c = getch ()) != ERR)
		    if (c == '\e' || c == 3)
			break;
	    }
	    nodelay (stdscr, FALSE);
	}
    }
    update_panels ((char *) -1, 0);
    repaint_screen (RP_NOCLEAR);
}

void ren_cmd ()
{
    struct stat s;
    int i, c, stat_r;
    
    char *new_name = input_dialog (" Rename/Move ", "Enter rename mask:", "");
    if (!new_name)
	return;
    if (strchr (new_name, '*') || strchr (new_name, '?')){
	/* Mask rename */
	message (0, " Missing feature ",
		 " Mask renaming not yet implemented, sorry ");
    }
    stat_r = stat (new_name, &s);
    if (S_ISREG (s.st_mode) || stat_r != 0){
	if (cpanel->marked <= 1){
	    if (cpanel->marked)
		move_file (get_file (cpanel), new_name);
	    else
		move_file (selection->fname, new_name);
	} else {
	    message (1," Error "," Can't rename multiple files to same name ");
	    return;
	}
    } else if (!S_ISDIR (s.st_mode)){
	message (1, " Error ", " Destination is not a directory or a file ");
	return;
    } else {
	/* destination is a directory */
	if (cpanel->marked){
	    nodelay (stdscr, TRUE);
	    for (i = 0; i < cpanel->marked; i++){
		if (cpanel->dir [i].f.marked)
		    move_file_dir (cpanel->dir [i].fname, new_name);
		if ((c = getch ()) != ERR)
		    if (c == '\e' || c == 3)
			break;
	    }
	    nodelay (stdscr, FALSE);
	} else
	    move_file_dir (selection->fname, new_name);
    }
    update_panels (0, 0);
    repaint_screen (RP_NOCLEAR);
}

void mkdir_cmd ()
{
    int current_mask;
    
    char *dir = input_dialog (" Mkdir ", "Enter directoy name:" , "");
    if (!dir)
	return;

    if (mkdir (dir, 0777) == 0){
	update_panels (0, 0);
	repaint_screen (RP_NOCLEAR);
	return;
    }
    message (1, " Error ", "  %s  ", sys_errlist [errno]);
}

void delete_cmd ()
{
    char *dest;
    int  i;
    struct stat s;
    
    if (cpanel->marked > 1){
	sprintf (cmd_buf, " Do you really want to delete %d files? ",
		 cpanel->marked);
	if (query_dialog (" Delete ", cmd_buf, 3, 2, " Yes ", " No ") == 0){
	    for (i = 0; i < cpanel->count; i++){
		if (cpanel->dir [i].f.marked)
		    erase_file (cpanel->dir [i].fname);
	    }
	} else
	    return;
    } else {
	if (S_ISDIR (selection->buf.st_mode))
	    erase_dir (selection->fname);
	else
	    erase_file (selection->fname);
    }
    update_panels (0, 0);
    repaint_screen (RP_NOCLEAR);
}

void find_cmd ()
{
    do_find ();
}

void ext_cmd ()
{
}

void select_cmd ()
{
    char *reg_exp;
    int i;
    int c;

    reg_exp = input_dialog (" Select ","", "*");
    if (!reg_exp)
	return;
    for (i = 0; i < cpanel->count; i++){
	if (S_ISDIR (cpanel->dir [i].buf.st_mode))
	    continue;
	c = regexp_match (reg_exp, cpanel->dir [i].fname);
	if (c == -1){
	    message (1, " Error ", "  Malformed regular expression  ");
	    return;
	}
	if (c){
	    if (!cpanel->dir [i].f.marked){
		cpanel->marked++;
		cpanel->total += cpanel->dir [i].buf.st_size;
	    }
	    cpanel->dir [i].f.marked = 1;
	}
    }
    paint_panel(cpanel);
    panel_refresh(cpanel);
    wrefresh(cpanel->win_file);
}

void unselect_cmd ()
{
    char *reg_exp;
    int i;
    int c;

    reg_exp = input_dialog (" Unselect ","", "*");
    if (!reg_exp)
	return;
    for (i = 0; i < cpanel->count; i++){
	if (S_ISDIR (cpanel->dir [i].buf.st_mode))
	    continue;
	c = regexp_match (reg_exp, cpanel->dir [i].fname);
	if (c == -1){
	    message (1, " Error ", "  Malformed regular expression  ");
	    return;
	}
	if (c){
	    if (cpanel->dir [i].f.marked){
		cpanel->marked--;
		cpanel->total -= cpanel->dir [i].buf.st_size;
	    }
	    cpanel->dir [i].f.marked = 0;
	}
    }
    paint_panel(cpanel);
    panel_refresh(cpanel);
    wrefresh(cpanel->win_file);
}

void full_cmd ()
{
    Panel *p = MENU_PANEL;

    full_frame (p);
    p->view_type = view_full;
    paint_panel (p);
}

void brief_cmd ()
{
    Panel *p = MENU_PANEL;

    brief_frame (p);
    p->view_type = view_brief;
    paint_panel (p);
}

void info_cmd ()
{
}

void do_re_sort (Panel *panel)
{
    do_sort (&panel->dir, panel->sort_type, panel->count-1);
    paint_dir (panel);
    panel_refresh (panel);
}

void by_name_cmd ()
{
    Panel *panel = MENU_PANEL;
    panel->sort_type = (sortfn *) sort_name;
    do_re_sort (panel);
}

void by_ext_cmd ()
{
    Panel *panel = MENU_PANEL;
    panel->sort_type = (sortfn *) sort_ext;
    do_re_sort (panel);
}

void by_time_cmd ()
{
    Panel *panel = MENU_PANEL;
    panel->sort_type = (sortfn *) sort_time;
    do_re_sort (panel);
}

void by_size_cmd ()
{
    Panel *panel = MENU_PANEL;
    panel->sort_type = (sortfn *) sort_size;
    do_re_sort (panel);
}

void unsorted_cmd ()
{
    Panel *panel = MENU_PANEL;
    panel->sort_type = (sortfn *) unsorted;
    do_re_sort (panel);
}

void reread_cmd ()
{
    update_panels ((char *) -1, 0);
    repaint_screen (RP_NOCLEAR);
}

void link_cmd ()
{
}

void menu_cmd ();

menu_entry PanelMenu [] = {
    "  Full",  'F', full_cmd,
    "  Brief", 'B', brief_cmd,
/*    "  Info",  'I', info_cmd, */
    "", ' ', 0,
    "  Name",      'N', by_name_cmd,
    "  Extension", 'E', by_ext_cmd,
    "  Time",      'T', by_time_cmd,
    "  Size",      'S', by_size_cmd,
    "  Unsorted",  'U', unsorted_cmd,
    "", ' ', 0,
    "  Reread",    'R', reread_cmd,
/*    "  Filter...", 'l', reread_cmd, */
};

menu_entry RightMenu [] = {
    "  Full",  'F', full_cmd,
    "  Brief", 'B', brief_cmd,
/*    "  Info",  'I', info_cmd, */
    "", ' ', 0,
    "  Name",      'N', by_name_cmd,
    "  Extension", 'E', by_ext_cmd,
    "  Time",      'T', by_time_cmd,
    "  Size",      'S', by_size_cmd,
    "  Unsorted",  'U', unsorted_cmd,
    "", ' ', 0,
    "  Reread",    'R', reread_cmd,
    /*
    "  Filter...", 'l', reread_cmd,
    */
};

menu_entry FileMenu [] = {
    "View            F3", KEY_F(3), view_cmd,
    "Edit            F4", KEY_F(4), edit_cmd,
    "Copy            F5", KEY_F(5), copy_cmd,
/*    "Link              ", ' ', link_cmd, */
    "Rename/Move     F6", KEY_F(6), ren_cmd,
    "Mkdir           F7", KEY_F(7), mkdir_cmd,
    "Delete          F8", KEY_F(8), delete_cmd,
    "", ' ', 0,
    "Select group     +", '+', select_cmd,
    "Unselect group   \\", '\\', unselect_cmd,
    "", ' ', 0,
    "Quit           F10", KEY_F(10), cmd_quit
};

menu_entry CmdMenu [] = {
    "Find file (test)", 'F', find_cmd,
    "Save setup",       'S', save_setup_cmd,
/*    "Extension file edit", 'E', ext_cmd, */
};

/* Must keep in sync with the constants in menu_cmd */
menu_entry OptMenu [] = {
    "  Show backup files", 'B', toggle_show_backup,
    "  Show hidden files", 'H', toggle_show_hidden,
    "  Verbose operation", 'V', toggle_verbose,
    "  Mark moves down",   'D', toggle_mark_move_down,
    "  Pause after run",   'P', toggle_pause_after_run,
    "  Show mini-status",  'S', toggle_show_mini_status,
    "  Shell patterns",    'I', toggle_easy_patterns,
    "  Auto save setup",   'A', toggle_auto_save,
    "  Auto menus",        'M', toggle_auto_menu,
/*    "  Align extensions",  'A', toggle_align_extensions, */
};

#define menu_entries(x) sizeof(x)/sizeof(menu_entry)

Menu *MenuBar [5];

void init_menu ()
{
    MenuBar [0] = create_menu (" Left ", PanelMenu, menu_entries (PanelMenu));
    MenuBar [1] = create_menu (" File ", FileMenu, menu_entries (FileMenu));
    MenuBar [2] = create_menu (" Command ", CmdMenu, menu_entries (CmdMenu));
    MenuBar [3] = create_menu (" Options ", OptMenu, menu_entries (OptMenu));
    MenuBar [4] = create_menu (" Right ", RightMenu, menu_entries (PanelMenu));
}

void check_options_menu (int index, int flag)
{
    *OptMenu [index].text = flag ? '*' : ' ';
}

void check_menu_panel (Panel *panel, menu_entry PanelMenu [])
{
    PanelMenu [0].text [0] = panel->view_type == view_full ? '*' : ' ';
    PanelMenu [1].text [0] = panel->view_type == view_brief ? '*' : ' ';
    PanelMenu [2].text [0] = panel->view_type == view_info ? '*' : ' ';

    PanelMenu[3].text[0]= panel->sort_type == (sortfn *)sort_name ? '*' : ' ';
    PanelMenu[4].text[0]= panel->sort_type == (sortfn *) sort_ext ? '*' : ' ';
    PanelMenu[5].text[0]= panel->sort_type == (sortfn *) sort_time ? '*' : ' ';
    PanelMenu[6].text[0]= panel->sort_type == (sortfn *) sort_size ? '*' : ' ';
    PanelMenu[7].text[0]= panel->sort_type == (sortfn *) unsorted ? '*' : ' ';
}

void menu_cmd ()
{
    int panel;
    
    check_options_menu (0, show_backups);
    check_options_menu (1, show_dot_files);
    check_options_menu (2, verbose);
    check_options_menu (3, mark_moves_down);
    check_options_menu (4, pause_after_run);
    check_options_menu (5, show_mini_info);
    check_options_menu (6, easy_patterns);
    check_options_menu (7, auto_save_setup);
    check_options_menu (8, auto_menu);
/*     check_options_menu (7, align_extensions); */

    check_menu_panel (&left_panel, PanelMenu);
    check_menu_panel (&right_panel, RightMenu);
    panel = current_panel == &right_panel ? 4 : 0;
    run_bar (stdscr, 0, 0, 80, 12, 5, panel, MenuBar, BAR_FILL, A_BOLD, SELECTED_COLOR);
}

/* Flag toggling functions */

void toggle_auto_menu ()
{
    auto_menu = !auto_menu;
}

void toggle_show_backup ()
{
    show_backups = !show_backups;
    update_panels ((char *) -1, 0);
}

void toggle_show_hidden ()
{
    show_dot_files = !show_dot_files;
    update_panels ((char *)-1, 0);
}

void toggle_verbose ()
{
    verbose = !verbose;
}

void toggle_auto_save ()
{
    auto_save_setup = !auto_save_setup;
}

void toggle_mark_move_down ()
{
    mark_moves_down = !mark_moves_down;
}

void toggle_pause_after_run ()
{
    pause_after_run = !pause_after_run;
}

void toggle_show_mini_status ()
{
    if (show_mini_info){
	left_panel.lines += 2;
	right_panel.lines +=2;
    } else {
	left_panel.lines -= 2;
	right_panel.lines -= 2;
    }
    show_mini_info = !show_mini_info;
    paint_panel (cpanel);
    paint_panel (opanel);
}

void toggle_easy_patterns ()
{
    easy_patterns = !easy_patterns;
}

void toggle_align_extensions ()
{
    align_extensions = !align_extensions;
}

void dump_marked ()
{
    int i;
    
    system ("clear");
    move (1, 1);
    for (i = 1; i < 8; i++){
	attron (COLOR_PAIR (i));
	printw ("Este es el color %d\n", i);
    }
    getch ();
    fprintf (stderr, "Hay %d archivos marcados\n", cpanel->marked);
    for (i = 0; i < cpanel->count; i++){
	if (cpanel->dir [i].f.marked)
	    fprintf (stderr, "%s\n", cpanel->dir [i].fname);
    }
    getch ();
    repaint_screen (RP_NOCLEAR);
}

/* Inserts the selected file name into the input line */
void copy_prog_name ()
{
    char *k = selection->fname;

    while (*k)
	handle_char (&cmdline, *k++);
    handle_char (&cmdline, ' ');
}

void suspend_cmd ()
{
    pre_exec ();
    kill (getpid (), SIGTSTP);
    post_exec ();
    update_panels ((char *) -1, 0);
    repaint_screen (RP_CLEAR);
}

void quote_next ()
{
    quote = 1;
}

struct {
    int   key_code;
    void  (*fn)(int);
} key_map [] = {
    KEY_F(10), 	cmd_quit,
    KEY_F(1),   help_cmd,
    KEY_F(2),   user_menu_cmd, 
    KEY_F(3),   view_cmd,
    KEY_F(4),   edit_cmd,
    KEY_F(5),   copy_cmd,
    KEY_F(6),   ren_cmd,
    KEY_F(7),   mkdir_cmd,
    KEY_F(8),   delete_cmd,
    KEY_F(9),   menu_cmd,
    KEY_F(12),  dump_marked,
    '+',        select_cmd,
    '\\',       unselect_cmd,
    '\t',  	change_panel,
    '\n',       enter,		/* Enter */
    KEY_DOWN,   move_down,
    KEY_UP, 	move_up,
    KEY_IC,     mark_file,
    KEY_HOME,	move_home,
    KEY_C1,     move_end,
    KEY_END,    move_end,
    KEY_A1,     move_home,
    KEY_NPAGE,  next_page,
    KEY_PPAGE,  prev_page,
    CTRL('l'),  repaint_screen_cmd, /* C-l */

    0x80| '\n', copy_prog_name,
    CTRL('t'),  mark_file,
    
    /* Emacs-like bindings */
    CTRL('v'),  next_page,      /* C-v like emacs */
    CTRL('p'),  move_up,	/* C-p like emacs */
    CTRL('n'),  move_down,	/* C-n like emacs */
    CTRL('q'),  quote_next,

    /* Suspend */
    CTRL('z'),  suspend_cmd,
    
    /* Default key handling */
    0,          default_key,
};

void save_setup ()
{
    char profile [100];

    sprintf (profile, "%s/.nc.ini", home_dir);
    
    set_int (profile, "show_backups", show_backups);
    set_int (profile, "show_dot_files", show_dot_files);
    set_int (profile, "verbose", verbose);
    set_int (profile, "mark_moves_down", mark_moves_down);
    set_int (profile, "pause_after_run", pause_after_run);
    set_int (profile, "show_mini_info", show_mini_info);
    set_int (profile, "shell_patterns", easy_patterns);
    set_int (profile, "auto_save_setup", auto_save_setup);
    set_int (profile, "align_extensions", align_extensions);
    set_int (profile, "auto_menu", auto_menu);
    
    WritePrivateProfileString ("Dirs", "other_dir", opanel->cwd, profile);
    sync_profiles ();
}

void save_setup_cmd ()
{
    save_setup ();
    message (0, " Setup ", " Setup saved to ~/.nc.ini ");
}

void load_setup ()
{
    static char buffer [100];
    char   *profile;
    
    sprintf (buffer, "%s/.nc.ini", home_dir);
    
    if (exist_file (buffer)){
	profile = buffer;
    } else if (exist_file (PROFILE)){
	profile = profile;
    } else
	return;
    
    show_backups    = get_int (profile, "show_backups", 0);
    show_dot_files  = get_int (profile, "show_dot_files", 1);
    verbose         = get_int (profile, "verbose", 1);
    mark_moves_down = get_int (profile, "mark_moves_down", 1);
    pause_after_run = get_int (profile, "pause_after_run", 1);
    show_mini_info  = get_int (profile, "show_mini_info", 1);
    easy_patterns   = get_int (profile, "shell_patterns", 1);
    auto_save_setup = get_int (profile, "auto_save_setup", 1);
    align_extensions= get_int (profile, "align_extensions", 1);
    auto_menu       = get_int (profile, "auto_menu", 0);
    GetPrivateProfileString ("Dirs", "other_dir", ".", buffer,
			     sizeof (buffer), profile);
    other_dir = buffer;
}

void do_nc ()
{
    int key;
    int i;
    
    if (COLS < 70 || LINES < 12){
	fprintf (stderr, "Screen too small: you need at leas 70x12\n");
	return;
    }
    clean_screen = newwin (0, 0, 0, 0);
    home_dir = getenv ("HOME");
    home_dir = home_dir ? home_dir : "/";
    load_setup ();
    init_colors ();
    init_labels ();
    init_key ();
    init_entry ();
    init_panels ();
    init_menu ();
    create_input (&cmdline, strlen (prompt), 0, sizeof (input_buffer),
		  input_buffer, cmdline_win, 0, COLS-2-strlen (prompt), "");

    if (auto_menu)
	user_menu_cmd ();
    
    for (quit = 0; !quit;){
	key = mi_getch ();
	if (quote){
	    default_key (key);
	    quote = 0;
	    continue;
	}
	for (i = 0; key_map [i].key_code; i++){
	    if (key == key_map [i].key_code){
		(*key_map [i].fn)(key);
		search_buffer [0] = 0;
		break;
	    }
	}
	if (key_map [i].key_code)
	    continue;

	/* Default key handling */
	(*key_map [i].fn)(key);
	wrefresh (cmdline_win);
    }
    if (auto_save_setup)
	save_setup ();
    clr_scr ();
    keypad (cmdline_win, FALSE);
    keypad (stdscr, FALSE);
    reset_shell_mode ();
    noraw ();
}

main ()
{
    initscr ();
    do_nc ();
    endwin ();
}
