------------------------------------------------------------------------------
--                                                                          --
--                 GNU ADA RUNTIME LIBRARY (GNARL) COMPONENTS               --
--                                                                          --
--                     R T S  _ T i m e r _ S e r v i c e                   --
--                                                                          --
--                                  B o d y                                 --
--                                                                          --
--                             $Revision: 1.2 $                            --
--                                                                          --
--           Copyright (c) 1991,1992,1993, FSU, All Rights Reserved         --
--                                                                          --
--  GNARL is free software; you can redistribute it and/or modify it  under --
--  terms  of  the  GNU  Library General Public License as published by the --
--  Free Software Foundation; either version 2, or  (at  your  option)  any --
--  later  version.   GNARL is distributed in the hope that it will be use- --
--  ful, but but WITHOUT ANY WARRANTY; without even the implied warranty of --
--  MERCHANTABILITY  or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU Gen- --
--  eral Library Public License for more details.  You should have received --
--  a  copy of the GNU Library General Public License along with GNARL; see --
--  file COPYING. If not, write to the Free Software Foundation,  675  Mass --
--  Ave, Cambridge, MA 02139, USA.                                          --
--                                                                          --
------------------------------------------------------------------------------

with System.Task_Stages;
with System.Task_Primitives; use System.Task_Primitives;

with System.Compiler_Exceptions; use System.Compiler_Exceptions;

with Unchecked_Conversion;

with Ada.Calendar.Conv; use Ada.Calendar.Conv;
with System.Real_Time.Conv; use System.Real_Time.Conv;
package body System.Task_Timer_Service is

   type Q_Rec;
   type Q_Link is access Q_Rec;
   type Q_Rec is record

      S_O :      Signal_Object.O_Type;

      T :        Real_Time.Time;    --  wake up time
      Next :     Q_Link;
      Previous : Q_Link;
   end record;

   Q_Head : Q_Link := null;

   Timer_Condition :  Condition_Variable;
   Timer_Lock :       Task_Primitives.Lock;

--  Hand Traslated code will be provided here.

   function To_Access is new Unchecked_Conversion (
         System.Address,
         Protection_Access);

   package body Signal_Object is

      procedure Signal_Unprotected (Open : in out boolean) is
      begin
         Open := true;
      end Signal_Unprotected;

      procedure Signal (PO : in out O_Type) is
         PS : Boolean;
      begin
         Task_Protected_Objects.Lock (To_Access (PO.Object'Address));
         begin
            Signal_Unprotected (PO.Open);
         exception
            when others =>
               Service_Entries (PO, PS);
               Task_Protected_Objects.Unlock (To_Access (PO.Object'Address));
               raise;
         end;
         Service_Entries (PO, PS);
         Task_Protected_Objects.Unlock (To_Access (PO.Object'Address));
         --  barriers may have changed
      end Signal;

      function Wait_Count_Unprotected (Object : Protection) return integer is
      begin
         return Protected_Count (Object, 1);
         --  find the number of calls waiting on the specified entry
      end Wait_Count_Unprotected;

      procedure Wait_Count (PO : in out O_Type; W : out integer) is
      begin
         Lock_Read_Only (To_Access (PO.Object'Address));
         W := Wait_Count_Unprotected (PO.Object);
         Unlock (To_Access (PO.Object'Address));
      exception
         when others =>
            Unlock (To_Access (PO.Object'Address));
            raise;
      end Wait_Count;

      procedure Service_Entries
        (PO : in out O_Type;
         Pending_Serviced : out Boolean)
      is
         P : System.Address;
         subtype PO_Entry_Index is Protected_Entry_Index
               range Null_Protected_Entry .. 1;
         Barriers : Barrier_Vector (1 .. 1);
         E : PO_Entry_Index;
         PS : Boolean;
         Cumulative_PS : Boolean := False;
      begin
         loop
            begin
               Barriers (1) := PO.Open;
            exception
               when others =>
                  begin
                     Broadcast_Program_Error (To_Access (PO.Object'Address));
                  exception
                     when Program_Error =>
                        Unlock (To_Access (PO.Object'Address));
                        raise;
                  end;
            end;
            Next_Entry_Call (To_Access (PO.Object'Address), Barriers, P, E);
            begin
               case E is
                  when Null_Protected_Entry =>  --  no pending call to serve
                     exit;
                  when 1 =>
                     PO.Open := false;          --  code from the entry Wait
                     Complete_Entry_Body (To_Access (PO.Object'Address), PS);
               end case;
            exception
               when others =>
                  Exceptional_Complete_Entry_Body (
                        Object => To_Access (PO.Object'Address),
                        Ex => Current_Exception,
                        Pending_Serviced => PS);
            end;
            Cumulative_PS := Cumulative_PS or PS;
         end loop;
         Pending_Serviced := Cumulative_PS;
      end Service_Entries;

   end Signal_Object;

   package body Timer is

      procedure Service_Unprotected (T : out Real_Time.Time) is
         Q_Ptr : Q_Link := Q_Head;
         W : integer;
      begin
         while Q_Ptr /= null loop
            Signal_Object.Wait_Count (Q_Ptr.S_O, W);
            if Q_Ptr.T < Real_Time.Clock or else W = 0 then
               --  wake up the waiting task
               Signal_Object.Signal (Q_Ptr.S_O);
               --  when it is done, all the pending calls are serviced
               --  Therefore it is safe to finalize it.
               Finalize_Protection (To_Access (Q_Ptr.S_O.Object'Address));

               --  remove the entry
               if Q_Ptr = Q_Head then         --  head entry
                  Q_Head := Q_Ptr.Next;
                  if Q_Head /= null then
                     Q_Head.Previous := null;
                  end if;
               elsif Q_Ptr.Next = null then   --  tail entry
                  Q_Ptr.Previous.Next := null;
               else                           --  middle entry
                  Q_Ptr.Previous.Next := Q_Ptr.Next;
                  Q_Ptr.Next.Previous := Q_Ptr.Previous;
               end if;
            end if;
            Q_Ptr := Q_Ptr.Next;
         end loop;
         if Q_Head = null then
            T := Real_Time.Time_Of (800000000, Time_Span_Zero);
            --  **** This vaule has to be changed later
         else
            T := Q_Head.T;
         end if;
      end Service_Unprotected;

      procedure Service (T : out Real_Time.Time) is
         PS : Boolean;
      begin
         Task_Protected_Objects.Lock (To_Access (Object'Address));
         begin
            Service_Unprotected (T);
         exception
         when others =>
            Service_Entries (PS);
            Task_Protected_Objects.Unlock (To_Access (Object'Address));
            raise;
         end;
         Service_Entries (PS);
         Task_Protected_Objects.Unlock (To_Access (Object'Address));
      end Service;

      procedure Real_Time_Enqueue (
            T : in Real_Time.Time; N : in out Q_Link) is
         Q_Ptr : Q_Link := Q_Head;
      begin
         --  create a queue entry
         N := new Q_Rec;
         Initialize_Protection (
               To_Access (N.S_O.Object'Address),
               Task_Stages.Unspecified_Priority);
         --  a new protected object is created. So, initialize it

         N.T := T;

         --  if the new element becomes the head of the queue,
         --  notify the Timer Service
         if Q_Head = null then
            N.Next := null;
            N.Previous := null;
            Q_Head := N;
            Task_Primitives.Write_Lock (Timer_Lock);
            Task_Primitives.Cond_Signal (Timer_Condition);
            Task_Primitives.Unlock (Timer_Lock);
            --  Signal the timer server to wake up
         elsif N.T < Q_Head.T then
            N.Next := Q_Head;
            N.Previous := null;
            Q_Head.Previous := N;
            Q_Head := N;
            Task_Primitives.Write_Lock (Timer_Lock);
            Task_Primitives.Cond_Signal (Timer_Condition);
            Task_Primitives.Unlock (Timer_Lock);
            --  Signal the timer server to wake up
         else
         --  place in the middle
            while Q_Ptr.Next /= null loop
               if Q_Ptr.Next.T >= N.T then
                  N.Next := Q_Ptr.Next;
                  N.Previous := Q_Ptr;
                  Q_Ptr.Next.Previous := N;
                  Q_Ptr.Next := N;
                  exit;
               end if;
               Q_Ptr := Q_Ptr.Next;
            end loop;
            if Q_Ptr.Next = null then
               --  place in the last
               N.Next := null;
               N.Previous := Q_Ptr;
               Q_Ptr.Next := N;
            end if;
         end if;
      end Real_Time_Enqueue;

      procedure Service_Entries (Pending_Serviced : out Boolean) is
         P : System.Address;
         subtype PO_Entry_Index is Protected_Entry_Index
               range Null_Protected_Entry .. 4;
         Barriers : Barrier_Vector (1 .. 4)  :=  (others => true);
         --  no barriers. always true
         E : PO_Entry_Index;
         PS : Boolean;
         Cumulative_PS : Boolean := False;
      begin
         loop
            Next_Entry_Call (To_Access (Object'Address), Barriers, P, E);
            begin
               case E is
                  when Null_Protected_Entry =>  --  no pending call to serve
                     exit;
                  when 1 =>
                     --  code from entry Enqueue (T : in Time_Span);
                     --  enqueues elements in wake-up time order
                     declare
                        Params : Time_Span_Params;
                        for Params use at P;
                        T : Time_Span  := Params.Param;
                        N : Q_Link;
                     begin
                        Real_Time_Enqueue (Real_Time.Clock + T, N);
                        Task_Protected_Objects.Lock (
                              To_Access (N.S_O.Object'Address));
                        --  Lock the target object before requeueing
                        --  Param is also passed with
                        --  Object.Call_In_Progress.Parameter
--                      begin
                           Requeue_Protected_Entry (
                                 Object => To_Access (Object'Address),
                                 New_Object =>
                                    To_Access (N.S_O.Object'Address),
                                 E => 1,
                                 With_Abort => true);
                           Signal_Object.Service_Entries (N.S_O, PS);
                           Task_Protected_Objects.Unlock (
                                 To_Access (N.S_O.Object'Address));
--                      exception
--                         when others =>
--                            Signal_Object.Service_Entries (N.S_O);
--                            raise;
--                      end;
                     end;
                  when 2 =>
                     --  code from entry Enqueue (T : in Duration);
                     --  enqueues elements in wake-up time order
                     declare
                        Params : Duration_Params;
                        for Params use at P;
                        T : Duration := Params.Param;
                        N : Q_Link;
                     begin
                        Real_Time_Enqueue (
                              Real_Time.Clock + To_Time_Span (T), N);
                        Task_Protected_Objects.Lock (
                              To_Access (N.S_O.Object'Address));
                        --  Lock the target object before requeueing
                        --  Param is also passed with
                        --  Object.Call_In_Progress.Parameter
--                    begin
                           Requeue_Protected_Entry (
                                 Object => To_Access (Object'Address),
                                 New_Object =>
                                    To_Access (N.S_O.Object'Address),
                                 E => 1,
                                 With_Abort => true);
                           Signal_Object.Service_Entries (N.S_O, PS);
                           Task_Protected_Objects.Unlock (
                                 To_Access (N.S_O.Object'Address));
--                      exception
--                         when others =>
--                            Signal_Object.Service_Entries (N.S_O);
--                            raise;
--                      end;
                     end;
                  when 3 =>
                     --  code from entry Enqueue (T : in Real_Time.Time);
                     --  enqueues elements in wake-up time order
                     declare
                        Params : Real_Time_Time_Params;
                        for Params use at P;
                        T : Real_Time.Time  := Params.Param;
                        N : Q_Link;
                     begin
                        Real_Time_Enqueue (T, N);  --  Put in the Timer Queue
                        Task_Protected_Objects.Lock (
                              To_Access (N.S_O.Object'Address));
                        --  Lock the target object before requeueing
                        --  Param is also passed with
                        --  Object.Call_In_Progress.Parameter
--                      begin
                           Requeue_Protected_Entry (
                              Object => To_Access (Object'Address),
                              New_Object => To_Access (N.S_O.Object'Address),
                              E => 1,
                              With_Abort => true);
                           Signal_Object.Service_Entries (N.S_O, PS);
                           Task_Protected_Objects.Unlock (
                                 To_Access (N.S_O.Object'Address));
--                      exception
--                         when others =>
--                            Signal_Object.Service_Entries (N.S_O);
--                            raise;
--                      end;
                     end;
                  when 4 =>
                     --  code from entry Enqueue (T : in Calendar.Time);
                     --  enqueues elements in wake-up time order
                     declare
                        Params : Calendar_Time_Params;
                        for Params use at P;

                        T : Ada.Calendar.Time  := Params.Param;

                        N : Q_Link;
                     begin
                        Real_Time_Enqueue (
                              Calendar_Time_To_Real_Time_Time (T),
                              N);
                        Task_Protected_Objects.Lock (
                              To_Access (N.S_O.Object'Address));
                        --  Lock the target object before requeueing
                        --  Param is also passed with
                        --  Object.Call_In_Progress.Parameter
--                      begin
                           Requeue_Protected_Entry (
                              Object => To_Access (Object'Address),
                              New_Object => To_Access (N.S_O.Object'Address),
                              E => 1,
                              With_Abort => true);
                           Signal_Object.Service_Entries (N.S_O, PS);
                           Task_Protected_Objects.Unlock (
                                 To_Access (N.S_O.Object'Address));
--                      exception
--                         when others =>
--                            Signal_Object.Service_Entries (N.S_O);
--                            raise;
--                      end;
                     end;
               end case;
            exception
               when others =>
                  Exceptional_Complete_Entry_Body (
                        Object => To_Access (Object'Address),
                        Ex => Current_Exception,
                        Pending_Serviced => PS);
            end;
            Cumulative_PS := Cumulative_PS or PS;
         end loop;
         Pending_Serviced := Cumulative_PS;
      end Service_Entries;

   begin
      Initialize_Protection (
            To_Access (Object'Address),
            Task_Stages.Unspecified_Priority);
   end Timer;

   task Timer_Server is
      pragma priority (System.Priority'Last);
   end Timer_Server;

   task body Timer_Server is

      Next_Wakeup_Time : Real_Time.Time :=
            Real_Time.Time_Of (800000000, Time_Span_Zero);
            --  This is the maximum clock value be can handled by SPARC
            --  CHANGE IT LATER>**********************
      Result : boolean;
   begin
      Task_Stages.Make_Independent;
      Task_Primitives.Initialize_Lock (System.Priority'Last, Timer_Lock);
      Task_Primitives.Initialize_Cond (Timer_Condition);
      Task_Primitives.Write_Lock (Timer_Lock);     --  necessary for timed wait
      loop
         Task_Primitives.Cond_Timed_wait (Timer_Condition, Timer_Lock,
               Real_Time_Time_To_Stimespec (Next_Wakeup_Time), Result);
         Timer.Service (Next_Wakeup_Time);
      end loop;
   end Timer_Server;

end System.Task_Timer_Service;
