------------------------------------------------------------------------------
--                                                                          --
--                 GNU ADA RUNTIME LIBRARY (GNARL) COMPONENTS               --
--                                                                          --
--                       R T S _ T a s k _ S t a g e s                      --
--                                                                          --
--                                  B o d y                                 --
--                                                                          --
--                             $Revision: 1.2 $                            --
--                                                                          --
--           Copyright (c) 1991,1992,1993, FSU, All Rights Reserved         --
--                                                                          --
--  GNARL is free software; you can redistribute it and/or modify it  under --
--  terms  of  the  GNU  Library General Public License as published by the --
--  Free Software Foundation; either version 2, or (at  your  option)  any --
--  later  version.   GNARL is distributed in the hope that it will be use- --
--  ful, but but WITHOUT ANY WARRANTY; without even the implied warranty of --
--  MERCHANTABILITY  or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU Gen- --
--  eral Library Public License for more details.  You should have received --
--  a  copy of the GNU Library General Public License along with GNARL; see --
--  file COPYING. If not, write to the Free Software Foundation,  675  Mass --
--  Ave, Cambridge, MA 02139, USA.                                          --
--                                                                          --
------------------------------------------------------------------------------

with System.Storage_Elements;

with System.Machine_Specifics;
with System.Task_Rendezvous; use System.Task_Rendezvous;
with System.Task_IDs; use System.Task_IDs;
with System.Task_ATCB;
with System.Task_Primitives; use System.Task_Primitives;
with System.Machine; use System.Machine;
with System.Task_Abortion; use System.Task_Abortion;
with System.Error_Reporting; use System.Error_Reporting;
with System.Compiler_Exceptions;

with Unchecked_Conversion;

package body System.Task_Stages is

   function "=" (L, R : Task_ATCB.ATCB_Ptr)
         return Boolean renames Task_ATCB."=";
   function "=" (L, R : Task_ATCB.Task_Stage)
         return Boolean renames Task_ATCB."=";
   function ">=" (L, R : Task_ATCB.Task_Stage)
         return Boolean renames Task_ATCB.">=";
   function "<" (L, R : Task_ATCB.Task_Stage)
         return Boolean renames Task_ATCB."<";
   function "=" (L, R : Task_ATCB.Accepting_State)
         return Boolean renames Task_ATCB."=";

   function Activation_to_ATCB is new
         Unchecked_Conversion (Activation_Chain, Task_ATCB.ATCB_Ptr);

   function ATCB_to_Activation is new
         Unchecked_Conversion (Task_ATCB.ATCB_Ptr, Activation_Chain);

function "-" (A : System.Address; B : System.Address)
      return Storage_Elements.Storage_Offset
      renames Storage_Elements."-";
function "-" (A : System.Address; I : Storage_Elements.Storage_Offset)
      return System.Address renames Storage_Elements."-";

   function Get_LL_TCB_Offset return Storage_Elements.Storage_Count;
   LL_TCB_Offset : Storage_Elements.Storage_Count := Get_LL_TCB_Offset;
   function Get_LL_TCB_Offset return Storage_Elements.Storage_Count is

      ATCB_Record : Task_ATCB.Ada_Task_Control_Block (0);
   begin
      return ATCB_Record.LL_TCB'Address - ATCB_Record'Address;
   end Get_LL_TCB_Offset;

   function Address_To_Task_ID is new Unchecked_Conversion (
         System.Address,
         Task_ID);
   function TCB_Ptr_To_Address is new Unchecked_Conversion (
         Task_Primitives.TCB_Ptr,
         System.Address);

   --  This is an INLINE_ONLY version of Self for use in the RTS.
   function Self return Task_ID is
      S : Task_Primitives.TCB_Ptr := Task_Primitives.Self;
   begin
      return Address_To_Task_ID (TCB_Ptr_To_Address (S) - LL_TCB_Offset);
   end Self;

   procedure Init_RTS (Main_Task_Priority : System.Priority) is
      T : Task_ATCB.ATCB_Ptr;
      Init : Task_ATCB.ATCB_Init;
   begin

      Task_ATCB.All_Tasks_List := null;
      Init.Entry_Num := 0;
      Init.Parent := null;

      Init.Task_Entry_Point := null;

      Init.Stack_Size := 0;
      Init.Activator := null;
      Task_Stages.Init_Master (Init.Master_of_Task);
      Init.Elaborated := null;

      T := Task_ATCB.Unsafe_New_ATCB (Init);
      Initialize_LL_Tasks (T.LL_TCB'access);
      Task_ATCB.Initialize_ATCB (T, Init);
      --  The allocation of the initial task ATCB is different from
      --  that of subsequent ATCBs, which are allocated with ATCB.New_ATCB.
      --  New_ATCB performs all of the functions of Unsafe_New_ATCB
      --  and Initialize_ATCB.  However, it uses GNULLI operations, which
      --  should not be called until after Initialize_LL_Tasks.  Since
      --  Initialize_LL_Tasks needs the initial ATCB, New_ATCB was broken
      --  down into two parts, the first of which alloctes the ATCB without
      --  calling any GNULLI operations.

      Install_Abort_Handler (Task_Abortion.Abort_Handler'access);

      --  This is not according the the GNULLI, which specifes
      --  access procedure(Context: Pre_Call_State) for the handler.
      --  This may be a mistake in the interface.

      Install_Error_Handler (Compiler_Exceptions.Notify_Exception'Address);
      --  Install handlers for asynchronous error signals.
      --  This is not according the the GNULLI, which specifes
      --  access procedure(...) for the handler.
      --  This may be a mistake in the interface.

      Task_Abortion.Undefer_Abortion;
      --  Abortion is deferred in a new ATCB; this makes the environment task
      --  abortable.

   end Init_RTS;

   procedure Init_Master (M : out Master_ID) is
   begin
      M := 0;
   end Init_Master;

   function Increment_Master (M : Master_ID) return Master_ID is
   begin
      return M + 1;
   end Increment_Master;

   function Decrement_Master (M : Master_ID) return Master_ID is
   begin
      return M - 1;
   end Decrement_Master;

   procedure Make_Independent is
      S : Task_ATCB.ATCB_Ptr := Task_ATCB.ID_To_ATCB (Self);
   begin
      S.Master_of_Task := Master_ID (0);  --  independent of the master chain
   end Make_Independent;

   procedure Create_Task (Size : Size_Type;
         Priority : Integer;
         Num_Entries : Task_Entry_Index;
         Master : Master_ID;
         State : Machine_Specifics.Init_State;
         Discriminants : System.Address;
         Elaborated : Access_Boolean;
         Chain : in out Activation_Chain;
         Created_Task : out Task_ID) is

      LL_Entry_Point : Machine_Specifics.Init_State;
      T, P, S : Task_ATCB.ATCB_Ptr;
      Init : Task_ATCB.ATCB_Init;
      Init_Priority : Integer := Priority;
      Default_Stack_Size : constant Size_Type := 10000;
   begin
      S := Task_ATCB.ID_To_ATCB (Self);

      if Init_Priority = Unspecified_Priority then
         Init_Priority := Default_Priority;
      end if;

      --  Find parent of new task, P, via master level number.

      P := S;
      if P /= null then
         while P.Master_of_Task >= Master loop
            P := P.Parent;
            exit when P = null;
         end loop;
      end if;

      Defer_Abortion;
      if P /= null then
         Write_Lock (P.L);
         if P /= Task_ATCB.ID_To_ATCB (Self) then
            if P.Awaited_Dependent_Count /= 0 and then
                  T.Master_of_Task = P.Master_Within then
               P.Awaited_Dependent_Count := P.Awaited_Dependent_Count + 1;
            end if;
         end if;
         P.Awake_Count := P.Awake_Count + 1;
         Unlock (P.L);
      end if;
      Undefer_Abortion;

      Init.Entry_Num := Num_Entries;
      Init.Task_Arg := Discriminants;
      Init.Parent := P;
      Init.Task_Entry_Point := State;
      if Size = Unspecified_Size then
         Init.Stack_Size := Default_Stack_Size;
      else
         Init.Stack_Size := Size;
      end if;
      Init.Activator := S;
      Init.Master_of_Task := Master;
      Init.Elaborated := Elaborated;
      T := Task_ATCB.New_ATCB (Init);

      LL_Entry_Point := Machine.Task_Wrapper'access;

      Create_LL_Task (
            System.Priority (Init_Priority),
            Machine_Specifics.Task_Storage_Size (
            Integer (Init.Stack_Size) +
            Integer (Machine.Task_Wrapper_Frame) + 4),
            LL_Entry_Point,
            Task_ATCB.ATCB_To_Address (T),
             T.LL_TCB'access);
      --  Ask for 4 extra bytes of stack space so that the ATCB pointer can
      --  be stored below the stack limit, plus extra space for the frame of
      --  Task_Wrapper.  This is so the use gets the amount of stack requested
      --  exclusive of the needs of the runtime.

      T.Activation_Link := Activation_to_ATCB (Chain);
      Chain := ATCB_to_Activation (T);

      T.Aborter_Link := null;

      Created_Task := Task_ATCB.ATCB_To_ID (T);
   end Create_Task;

   procedure Activate_Tasks (Chain : in out Activation_Chain) is

      This_Task : Task_ATCB.ATCB_Ptr;

      C : Task_ATCB.ATCB_Ptr;
      All_Elaborated : Boolean := True;
   begin
      This_Task := Task_ATCB.ID_To_ATCB (Self);

      This_Task.Activation_Status := Good_Activations;
      --  Set the calling task's activation status to Good_Activations.  This
      --  is about to be checked, and it may contain garbage (e.g. a leftover
      --  activation status already used to raise an error in this task).

      C := Activation_to_ATCB (Chain);
      while (C /= null) and All_Elaborated loop
         if C.Elaborated /= null and then not C.Elaborated.all then
            All_Elaborated := False;
         end if;
         C := C.Activation_Link;
      end loop;
      --  Check that all task bodies have been elaborated.

      if not All_Elaborated then

         raise Program_Error;

      end if;
      Write_Lock (This_Task.L);
      This_Task.Activation_Count := 0;

      --  Wake up all the tasks so that they can activate themselves.

      C := Activation_to_ATCB (Chain);
      while C /= null loop

         Write_Lock (C.L);
         --  Note that the locks of the activator and created task are locked
         --  here.  This is necessary because C.Stage and
         --  This_Task.Activation_Count have to be synchronized.  This is also
         --  done in Complete_Activation and Init_Abortion.  So long as the
         --  activator lock is always locked first, this cannot lead to
         --  deadlock.

         if C.Stage = Task_ATCB.Created then
            C.Stage := Task_ATCB.Can_Activate;
            Cond_Signal (C.Cond);
            This_Task.Activation_Count := This_Task.Activation_Count + 1;
         end if;
         Unlock (C.L);

         C := C.Activation_Link;
      end loop;

      Defer_Abortion;
      while This_Task.Activation_Count > 0 and then
            This_Task.Pending_ATC_Level >= This_Task.ATC_Nesting_Level loop
         Cond_Wait (This_Task.Cond, This_Task.L);
      end loop;
      Unlock (This_Task.L);

      Undefer_Abortion;
   end Activate_Tasks;

   procedure Expunge_Unactivated_Tasks (Chain : in out Activation_Chain) is
   begin
      Unimplemented_Operation;
   end Expunge_Unactivated_Tasks;

   function  Current_Master return Master_ID is
   begin
      return Task_ATCB.ID_To_ATCB (Self).Master_Within;
   end Current_Master;

   procedure Terminate_Dependents (ML : Master_ID := Master_ID'FIRST);

   --  WARNING : Only call this procedure with priority already boosted;
   --           hence the "Vulnerable".
   procedure Vulnerable_Complete_Activation (T : Task_ATCB.ATCB_Ptr;
         Status : Activations_Status) is
      This_Task : Task_ATCB.ATCB_Ptr := Task_ATCB.ID_To_ATCB (Self);
      Activator : Task_ATCB.ATCB_Ptr;
   begin

      Activator := T.Activator;

      --  Decrement the count of tasks to be activated by the activator and
      --  wake it up so it can check to see if all tasks have been activated.
      --  Note that the locks of the activator and created task are locked
      --  here.  This is necessary because C.Stage and
      --  T.Activation_Count have to be synchronized.  This is also
      --  done in Activate_Tasks and Init_Abortion.  So long as the
      --  activator lock is always locked first, this cannot lead to deadlock.

      Write_Lock (Activator.L);
      Write_Lock (T.L);
      if T.Stage = Task_ATCB.Can_Activate then
         T.Stage := Task_ATCB.Active;
         Activator.Activation_Count := Activator.Activation_Count - 1;
      end if;

      Unlock (T.L);
      if Activator.Activation_Status = Good_Activations then
         Activator.Activation_Status := Status;
      end if;
      Cond_Signal (Activator.Cond);
      Unlock (Activator.L);

   end Vulnerable_Complete_Activation;

   --  WARNING : Only call this procedure with abortion deferred;
   --           hence the "Vulnerable".
   --  This procedure needs to have abortion deferred while it has
   --  the current task's lock locked.  This is indicated by the commented
   --  calls to abortion control calls.
   --  work to be done by task when shutting itself down,
   --  as for : normal termination via completion;
   --          termination via unhandled exception;
   --          terminate alternative;
   --          abortion.
   procedure Vulnerable_Complete_Task is
      P, T : Task_ATCB.ATCB_Ptr := Task_ATCB.ID_To_ATCB (Self);

   begin
      --  T.Stage can be safely checked for Can_Activate here without
      --  protection, since T does not get to run until Stage is Can_Activate,
      --  and Vulnerable_Complete_Activation will check to see if it has moved
      --  beyond Complete_Activation under the protection of the mutex
      --  before decrementing the activator's Activation_Count.
      if T.Stage = Task_ATCB.Can_Activate then
         Vulnerable_Complete_Activation (T, Task_Error);
      end if;

   --   Defer_Abortion;

      Complete (Task_ATCB.ATCB_To_ID (T));

      Write_Lock (T.L);

      --  If the task has been awakened due to abortion, this should
      --  cause the dependents to abort themselves and cause the awake
      --  count to go to zero.

      if T.Pending_ATC_Level < T.ATC_Nesting_Level and then
            T.Awake_Count /= 0 then
         Unlock (T.L);
         Abort_Dependents (Task_ATCB.ATCB_To_ID (T));
         Write_Lock (T.L);
      end if;

      while T.Awake_Count /= 0 loop
         Cond_Wait (T.Cond, T.L);
         if T.Pending_ATC_Level < T.ATC_Nesting_Level and then
               T.Awake_Count /= 0 then
         --  The task may have been awakened to perform abortion.
            Unlock (T.L);
            Abort_Dependents (Task_ATCB.ATCB_To_ID (T));
            Write_Lock (T.L);
         end if;
      end loop;
      Unlock (T.L);

      Terminate_Dependents;

   end Vulnerable_Complete_Task;

--  This procedure will be executed, after the
--  finalization of a task, through Task_Wrapper (GNAT only).
   procedure Leave_Task is
      P, T : Task_ATCB.ATCB_Ptr := Task_ATCB.ID_To_ATCB (Self);
      Saved_Pending_ATC_Level : ATC_Level_Base;
   begin

      Saved_Pending_ATC_Level := T.Pending_ATC_Level;
      --  We are about to loose our ATCB.  Save a few choice fields for final
      --  cleanup.

      P := T.Parent;
      if P /= null then
         Write_Lock (P.L);
         Write_Lock (T.L);

         --  If T has a parent, then setting T.Stage to Terminted and
         --  incrementing/decrementing P.Terminating_Dependent_Count
         --  have to be synchronized here and in Terminate_Dependents.
         --  This is done by locking the parent and dependent locks.  So
         --  long as the parent lock is always locked first, this should not
         --  cause deadlock.

         T.Stage := Task_ATCB.Terminated;
         if P.Terminating_Dependent_Count > 0 and then
               T.Master_of_Task = P.Master_Within then
            P.Terminating_Dependent_Count := P.Terminating_Dependent_Count - 1;
            if P.Terminating_Dependent_Count = 0 then
               Cond_Signal (P.Cond);
            end if;
         end if;

         Unlock (T.L);
         Unlock (P.L);
         --  WARNING - Once this lock is unlocked, it should be assumed that
         --            the ATCB has been deallocated. It should not be
         --            accessed again.
      else
         Write_Lock (T.L);
         T.Stage := Task_ATCB.Terminated;
         Unlock (T.L);
      end if;

      Exit_LL_Task;

   end Leave_Task;

   --  This is the entry point for compiler-generated code.  It just controls
   --  abortion and calls Vulnerable_Complete_Task.
   procedure Complete_Task is
   begin
      Defer_Abortion;
      Vulnerable_Complete_Task; --  No return for VERDIX version
      Undefer_Abortion;         --  Never executed for VERDIX version
   end Complete_Task;

   procedure Complete_Activation is
   begin
      Defer_Abortion;
      Vulnerable_Complete_Activation (Task_ATCB.ID_To_ATCB (Self),
            Good_Activations);
      Undefer_Abortion;
   end Complete_Activation;

   --  record that task T is passive.
   --  If T is the last dependent of some master in task P
   --  to become passive, then release P.
   --  A special case of this is when T has no dependents
   --  and is completed.
   --  In this case, T itself should be released.

   --  If the parent is made passive, this is repeated recursively, with C
   --  being the previous parent and P being the next parent up.

   --  Note that we have to hold the locks of both P and C (locked in that
   --  order) so that the Awake_Count of C and the Awaited_Dependent_Count of
   --  P will be synchronized.  Otherwise, an attempt by P to terminate can
   --  preempt this routine after C's Awake_Count has been decremented to zero
   --  but before C has checked the Awaited_Dependent_Count of P.  P would not
   --  count C in its Awaited_Dependent_Count since it is not awake, but it
   --  might count other awake dependents.  When C gained control again, it
   --  would decrement P's Awaited_Dependent_Count to indicate that it is
   --  passive, even though it was never counted as active.  This would cause
   --  P to wake up before all of its dependents are passive.
   --  Note : Any task with an interrupt entry should never become passive.
   --        Support for this feature needs to be added here.
   procedure Make_Passive (T : Task_ATCB.ATCB_Ptr) is
         P : Task_ATCB.ATCB_Ptr;
         --  Task whose Awaited_Dependent_Count may be decremented.
         C : Task_ATCB.ATCB_Ptr;
         --  Task whose awake-count gets decremented.
         H : Task_ATCB.ATCB_Ptr;
         --  Highest task that is ready to terminate dependents.
         Taken : Boolean;

         Activator : Task_ATCB.ATCB_Ptr;

   begin

      Vulnerable_Complete_Activation (T, Good_Activations);

      Write_Lock (T.L);
      if T.Stage >= Task_ATCB.Passive then
         Unlock (T.L);
         return;
      else
         T.Stage := Task_ATCB.Passive;
         Unlock (T.L);
      end if;
      H := null;
      P := T.Parent;
      C := T;
      while C /= null loop

         if P /= null then
            Write_Lock (P.L);
            Write_Lock (C.L);

            C.Awake_Count := C.Awake_Count - 1;
            if C.Awake_Count /= 0 then
            --  C is not passive; we cannot make anything above this point
            --  passive.
               Unlock (C.L);
               Unlock (P.L);
               exit;
            end if;

            if P.Awaited_Dependent_Count /= 0 then
            --  We have hit a non-task master; we will not be able to make
            --  anything above this point passive.
               P.Awake_Count := P.Awake_Count - 1;
               if C.Master_of_Task = P.Master_Within then
                  P.Awaited_Dependent_Count := P.Awaited_Dependent_Count - 1;
                  if P.Awaited_Dependent_Count = 0 then
                     H := P;
                  end if;
               end if;
               Unlock (C.L);
               Unlock (P.L);
               exit;
            end if;

            if C.Stage = Task_ATCB.Complete then
            --  C is both passive (Awake_Count = 0) and complete; wake it up to
            --  await termination of its dependents.  It will not be complete
            --  if it is waiting on a terminate alternative.  Such a task is
            --  not ready to wait for its dependents to terminate, though one
            --  of its ancestors may be.
               H := C;
            end if;

            Unlock (C.L);
            Unlock (P.L);
            C := P;
            P := C.Parent;

         else
            Write_Lock (C.L);

            C.Awake_Count := C.Awake_Count - 1;
            if C.Awake_Count /= 0 then
            --  C is not passive; we cannot make anything above this point
            --  passive.
               Unlock (C.L);
               exit;
            end if;

            if C.Stage = Task_ATCB.Complete then
            --  C is both passive (Awake_Count = 0) and complete; wake it up to
            --  await termination of its dependents.  It will not be complete
            --  if it is waiting on a terminate alternative.  Such a task is
            --  not ready to wait for its dependents to terminate, though one
            --  of its ancestors may be.
               H := C;
            end if;

            Unlock (C.L);
            C := P;
         end if;

      end loop;

      if H /= null then
         Cond_Signal (H.Cond);
      end if;

   end Make_Passive;

   procedure Complete (Target : Task_ID) is
      T :      Task_ATCB.ATCB_Ptr := Task_ATCB.ID_To_ATCB (Target);
      Caller : Task_ATCB.ATCB_Ptr := Task_ATCB.ID_To_ATCB (Self);
      Task1 : Task_ATCB.ATCB_Ptr;
      Task2 : Task_ATCB.ATCB_Ptr;
   begin

      --  Make_Passive used to be the last thing done in this routine in the
      --  original MRTSI code.  Make_Passive was modified not to process a
      --  completed task, so setting the complete flag conflicted with this.
      --  I don't see any reason why the task cannot be made passive before
      --  it is marked as completed, but I may find out.
      Make_Passive (T);

      Write_Lock (T.L);
      if T.Stage < Task_ATCB.Completing then
         T.Stage := Task_ATCB.Completing;
         T.Accepting := Task_ATCB.Not_Accepting;
         --  *LATER* consider new value of this type
         T.Awaited_Dependent_Count := 0;
         Unlock (T.L);
         Task_Rendezvous.Close_Entries (Task_ATCB.ATCB_To_ID (T));
         T.Stage := Task_ATCB.Complete;

         --  Wake up all the pending calls on Aborter_Link list
         Task1 := T.Aborter_Link;
         T.Aborter_Link := null;
         while (Task1 /= null) loop
            Task2 := Task1;
            Task1 := Task1.Aborter_Link;
            Task2.Aborter_Link := null;
            Cond_Signal (Task2.Cond);
         end loop;
      else

         --  Some other task is competing this task. So just wait until
         --  the completion is done. A list of such waiting tasks is maintained
         --  by Aborter_Link in ATCB.
         while T.Stage < Task_ATCB.Complete loop
            if T.Aborter_Link /= null then
               Caller.Aborter_Link := T.Aborter_Link;
            end if;
            T.Aborter_Link := Caller;
            Cond_Wait (Caller.Cond, T.L);
         end loop;
         Unlock (T.L);
      end if;
   end Complete;

   --  WARNING : Only call this procedure with abortion deferred.
   --  This procedure needs to have abortion deferred while it has
   --  the current task's lock locked.  This is indicated by the commented
   --  abortion control calls.  Since it is called from two procedures which
   --  also need abortion deferred, it is left controlled on entry to
   --  this procedure.
   --
   --  This relies that all dependents are passive.
   --  That is, they may be :
   --  1) held in COMPLETE_TASK;
   --  2) aborted, with forced-call to COMPLETE_TASK pending;
   --  3) held in terminate-alternative of SELECT.
   procedure Terminate_Dependents (ML : Master_ID := Master_ID'FIRST) is
      Failed, Taken : Boolean;
      T : Task_ATCB.ATCB_Ptr := Task_ATCB.ID_To_ATCB (Self);
      C, Previous, Temp : Task_ATCB.ATCB_Ptr;
   begin

      Write_Lock (Task_ATCB.All_Tasks_L);

   --   Defer_Abortion;

      Write_Lock (T.L);

      --  Count the number of active dependents that must terminate before
      --  proceeding.  If Terminating_Dependent_Count is not zero, then the
      --  dependents have already been counted.  This
      --  can occur when a thread executing this routine is canceled and the
      --  cancellation takes effect in the Cond_Wait () called below to
      --  wait for Terminating_Dependent_Count to go to zero.  In this case
      --  we just skip the count and continue waiting for the count to
      --  go to zero.

      if T.Terminating_Dependent_Count = 0 then
         C := Task_ATCB.All_Tasks_List;
         while C /= null loop

            --  The check for C.Stage=ATCB.Terminated and the increment of
            --  T.Terminating_Dependent_Count must be synchronized here and in
            --  Complete_Task using T.L and C.L.  So long as the parent T
            --  is locked before the dependent C, this should not lead to
            --  deadlock.
            if C /= T then
               Write_Lock (C.L);
               if C.Parent = T and then
                     C.Master_of_Task >= ML and then
                     C.Stage /= Task_ATCB.Terminated then
                  T.Terminating_Dependent_Count :=
                        T.Terminating_Dependent_Count + 1;
               end if;
               Unlock (C.L);
            end if;

            C := C.All_Tasks_Link;
         end loop;
      end if;

      Unlock (T.L);

      C := Task_ATCB.All_Tasks_List;
      while C /= null loop
         if C.Parent = T and then C.Master_of_Task >= ML then
            Complete (Task_ATCB.ATCB_To_ID (C));
            Cond_Signal (C.Cond);
         end if;
         C := C.All_Tasks_Link;
      end loop;
      Unlock (Task_ATCB.All_Tasks_L);

      Write_Lock (T.L);
      while T.Terminating_Dependent_Count /= 0 loop
         Cond_Wait (T.Cond, T.L);
      end loop;
      Unlock (T.L);
      --  I don't wake up for abortion here, since I am already
      --  terminating just as fast as I can.

   --   Undefer_Abortion;

      Write_Lock (Task_ATCB.All_Tasks_L);
      C := Task_ATCB.All_Tasks_List;
      Previous := null;
      while C /= null loop
         if (C.Parent = T and then C.Master_of_Task >= ML) then
            if Previous /= null then
               Previous.All_Tasks_Link := C.All_Tasks_Link;
            else
               Task_ATCB.All_Tasks_List := C.All_Tasks_Link;
            end if;
            Temp := C;
            C := C.All_Tasks_Link;
            Task_ATCB.Free_ATCB (Temp);
            --  It is OK to free the ATCB provided that the dependent task
            --  does not access its ATCB in Complete_Task after signalling its
            --  parent's (this task) condition variable and unlocking its lock.
         else
            Previous := C;
            C := C.All_Tasks_Link;
         end if;
      end loop;
      Unlock (Task_ATCB.All_Tasks_L);
   end Terminate_Dependents;

   --  WARNING : Only call this procedure with abortion deferred.
   --  This procedure needs to have abortion deferred while it has
   --  the current task's lock locked.  This is indicated by the commented
   --  abortion control calls.  Since it is called from two procedures which
   --  also need abortion deferred, it is left controlled on entry to
   --  this procedure.
   procedure Terminate_Alternative is
      P, T : Task_ATCB.ATCB_Ptr := Task_ATCB.ID_To_ATCB (Self);
      Taken : Boolean;
   begin
      Make_Passive (T);

   --   Defer_Abortion;

      Write_Lock (T.L);

      while T.Accepting /= Task_ATCB.Not_Accepting and then
            T.Stage /= Task_ATCB.Complete and then
            T.Pending_ATC_Level >= T.ATC_Nesting_Level loop
         Cond_Wait (T.Cond, T.L);
      end loop;

      if T.Stage = Task_ATCB.Complete then
         Unlock (T.L);
         if T.Pending_ATC_Level < T.ATC_Nesting_Level then
            Undefer_Abortion;
            Assert (False, "Continuing after being aborted!");
         end if;

         Abort_To_Level (Task_ATCB.ATCB_To_ID (T), 0);
         Undefer_Abortion;
         Assert (False, "Continuing after being aborted!");

      end if;

      T.Stage := Task_ATCB.Active;
      T.Awake_Count := T.Awake_Count + 1;

      --  At this point, T.Awake_Count and P.Awaited_Dependent_Count could be
      --  out of synchronization.  However, we know that
      --  P.Awaited_Dependent_Count cannot be zero, and cannot go to zero,
      --  since some other dependent must have just called us.  There should
      --  therefore be no danger of the parent terminating before we increment
      --  P.Awaited_Dependent_Count below.

      if T.Awake_Count = 1 then
         Unlock (T.L);
         if T.Pending_ATC_Level < T.ATC_Nesting_Level then
            Undefer_Abortion;
            Assert (False, "Continuing after being aborted!");
         end if;
         P := T.Parent;
         Write_Lock (P.L);
         if P.Awake_Count /= 0 then
            P.Awake_Count := P.Awake_Count + 1;
         else
            Unlock (P.L);

            Abort_To_Level (Task_ATCB.ATCB_To_ID (T), 0);
            Undefer_Abortion;
            Assert (False, "Continuing after being aborted!");

         end if;

         --  Conservative checks which should only matter when an interrupt
         --  entry was chosen. In this case, the current task completes if the
         --  parent has already been signaled that all children have
         --  terminated.

         if T.Master_of_Task = P.Master_Within then
            if P.Awaited_Dependent_Count /= 0 then
               P.Awaited_Dependent_Count := P.Awaited_Dependent_Count + 1;
            elsif P.Stage = Task_ATCB.Await_Dependents then
               Unlock (P.L);

               Abort_To_Level (Task_ATCB.ATCB_To_ID (T), 0);
               Undefer_Abortion;
               Assert (False, "Continuing after being aborted!");

            end if;
         end if;

         Unlock (P.L);
      else
         Unlock (T.L);
         if T.Pending_ATC_Level < T.ATC_Nesting_Level then
            Undefer_Abortion;
            Assert (False, "Continuing after being aborted!");
         end if;
      end if;

   --   Undefer_Abortion;

   end Terminate_Alternative;

   procedure Enter_Master is
      T : Task_ATCB.ATCB_Ptr := Task_ATCB.ID_To_ATCB (Self);
   begin
      T.Master_Within := Increment_Master (T.Master_Within);
   end Enter_Master;

   procedure Pop_Master is
      T : Task_ATCB.ATCB_Ptr := Task_ATCB.ID_To_ATCB (Self);
      C : Task_ATCB.ATCB_Ptr;
      CM : Master_ID := T.Master_Within;
      Taken, Asleep : Boolean;
      TAS_Result : Boolean;
   begin

      --  Shouldn't abortion be deferred here?  Why is it left until later?
      --  Ted Giering                        June 29, 1993

      Write_Lock (Task_ATCB.All_Tasks_L);

      --  Cancel threads of dependent tasks that have not yet started
      --  activation.

      C := Task_ATCB.All_Tasks_List;
      while C /= null loop
         if C.Parent = T and then C.Master_of_Task = CM then
            Write_Lock (C.L);
            if C.Stage = Task_ATCB.Created then

                  --  This is funny-looking stuff; it appears to be making
                  --  tasks abnormal that have not been aborted.  I am
                  --  duplicating this logic with GNULLI operations
                  --  nevertheless.
                  --  Ted Giering                  June 29, 1993
               if C.Pending_ATC_Level > 0 then
                  C.Pending_ATC_Level := 0;
                  Abort_Task (C.LL_TCB'access);
               end if;
            end if;
            Unlock (C.L);
         end if;
         C := C.All_Tasks_Link;
      end loop;

      Defer_Abortion;

      --  Note that Awaited_Dependent_Count must be zero at this point.  It is
      --  initialized to zero, this is the only code that can increment it
      --  when it is zero, and it will be zero again on exit from this routine.

      Write_Lock (T.L);
      C := Task_ATCB.All_Tasks_List;
      while C /= null loop
         if C.Parent = T and then C.Master_of_Task = CM then
            Write_Lock (C.L);
            if C.Awake_Count /= 0 then
               T.Awaited_Dependent_Count := T.Awaited_Dependent_Count + 1;
            end if;
            Unlock (C.L);
         end if;
         C := C.All_Tasks_Link;
      end loop;
      Unlock (Task_ATCB.All_Tasks_L);

      --  If the task has been awakened due to abortion, this should
      --  cause the dependents to abort themselves and cause
      --  Awaited_Dependent_Count count to go to zero.

      if T.Pending_ATC_Level < T.ATC_Nesting_Level and then
            T.Awaited_Dependent_Count /= 0 then
         Unlock (T.L);
         Abort_Dependents (Task_ATCB.ATCB_To_ID (T));
         Write_Lock (T.L);
      end if;

      T.Stage := Task_ATCB.Await_Dependents;
      while T.Awaited_Dependent_Count /= 0 loop
         Cond_Wait (T.Cond, T.L);

         if T.Pending_ATC_Level < T.ATC_Nesting_Level and then
               T.Awaited_Dependent_Count /= 0 then
         --  The task may have been awakened to perform abortion.
            Unlock (T.L);
            Abort_Dependents (Task_ATCB.ATCB_To_ID (T));
            Write_Lock (T.L);
         end if;

      end loop;
      Unlock (T.L);
      if T.Pending_ATC_Level < T.ATC_Nesting_Level then
         Undefer_Abortion;
         Assert (False, "Continuing after being aborted!");
      end if;

      Terminate_Dependents (CM);

      T.Stage := Task_ATCB.Active;
      --  Make next master level up active.  This needs to be done before
      --  decrementing the master level number, so that tasks finding
      --  themselves dependent on the current master level do not think that
      --  this master has been terminated (i.e. Stage=Await_Dependents and
      --  Awaited_Dependent_Count=0).  This should be safe; the only thing that
      --  can affect the stage of a task after it has become active is either
      --  the task itself or abortion, which is deferred here.

      T.Master_Within := Decrement_Master (CM);
      --  Should not need protection; can only change if T executes an
      --  Enter_Master or a Complete_Master.  T is only one task, and cannot
      --  execute these while executing this.

      Undefer_Abortion;

   end Pop_Master;

   procedure Complete_Master is
   begin
      Pop_Master;
   end Complete_Master;

   function Terminated (T : Task_ID) return Boolean is
   begin
      --  Does not need protection; access is assumed to be atomic.
      return Task_ATCB.ID_To_ATCB (T).Stage = Task_ATCB.Terminated;
   end Terminated;

begin
   Init_RTS (Default_Priority);
end System.Task_Stages;
