------------------------------------------------------------------------------
--                                                                          --
--                 GNU ADA RUNTIME LIBRARY (GNARL) COMPONENTS               --
--                                                                          --
--                 R T S _ P r o t e c t e d _ O b j e c t s                --
--                                                                          --
--                                  B o d y                                 --
--                                                                          --
--                             $Revision: 1.2 $                            --
--                                                                          --
--           Copyright (c) 1991,1992,1993, FSU, All Rights Reserved         --
--                                                                          --
--  GNARL is free software; you can redistribute it and/or modify it  under --
--  terms  of  the  GNU  Library General Public License as published by the --
--  Free Software Foundation; either version 2, or (at  your  option)  any  --
--  later  version.   GNARL is distributed in the hope that it will be use- --
--  ful, but but WITHOUT ANY WARRANTY; without even the implied warranty of --
--  MERCHANTABILITY  or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU Gen- --
--  eral Library Public License for more details.  You should have received --
--  a  copy of the GNU Library General Public License along with GNARL; see --
--  file COPYING. If not, write to the Free Software Foundation,  675  Mass --
--  Ave, Cambridge, MA 02139, USA.                                          --
--                                                                          --
------------------------------------------------------------------------------

with System.Error_Reporting; use System.Error_Reporting;
with System.Machine_Specifics;
with System.Machine; use System.Machine;
with System.POSIX_RTE;
with System.Task_ATCB;
with System.Task_Entries; use System.Task_Entries;
with System.Task_Abortion; use System.Task_Abortion;
with System.Task_Primitives; use System.Task_Primitives;
with System.Task_Stages;

package body System.Task_Protected_Objects is

   function "=" (L, R : System.Address) return Boolean renames System."=";
   function "=" (L, R : Task_ATCB.ATCB_Ptr) return Boolean
         renames Task_ATCB."=";
   function "=" (L, R : Task_IDs.Task_ID) return Boolean
         renames Task_IDs."=";
   function "=" (L, R : Compiler_Exceptions.Exception_ID) return Boolean
         renames Compiler_Exceptions."=";

   procedure Raise_Pending_Exception (Block : Communication_Block) is
         T : Task_ATCB.ATCB_Ptr := Task_ATCB.ID_To_ATCB (Block.Self);
         Ex : Compiler_Exceptions.Exception_ID := T.Exception_To_Raise;
   begin
      T.Exception_To_Raise := Compiler_Exceptions.Null_Exception;
      Compiler_Exceptions.Raise_Exception (Ex);
   end Raise_Pending_Exception;

   procedure Check_Exception is
         T : Task_ATCB.ATCB_Ptr := Task_ATCB.ID_To_ATCB (Task_Stages.Self);
         Ex : Compiler_Exceptions.Exception_ID := T.Exception_To_Raise;
   begin
      T.Exception_To_Raise := Compiler_Exceptions.Null_Exception;
      Compiler_Exceptions.Raise_Exception (Ex);
   end Check_Exception;

   procedure Initialize_Protection (
         Object : Protection_Access;
         Ceiling_Priority : Integer) is
         Init_Priority : Integer := Ceiling_Priority;
   begin
      if Init_Priority = Task_Stages.Unspecified_Priority then

         Init_Priority := System.Default_Priority;

      end if;
      Initialize_Lock (Init_Priority, Object.L);
      Object.Pending_Call := null;
      Object.Call_In_Progress := null;
      for E in Object.Entry_Queues'range loop
         Object.Entry_Queues (E).Head := null;
         Object.Entry_Queues (E).Tail := null;
      end loop;
   end Initialize_Protection;

   procedure Finalize_Protection (Object : Protection_Access) is
   begin
   --   Need to purge entry queues and pending entry call here.
      Finalize_Lock (Object.L);
   end Finalize_Protection;

   procedure Lock (Object : Protection_Access) is
   begin
      Write_Lock (Object.L);
   end Lock;

   procedure Lock_Read_Only (Object : Protection_Access) is
   begin
      Read_Lock (Object.L);
   end Lock_Read_Only;

   procedure Unlock (Object : Protection_Access) is
   begin
      Unlock (Object.L);
   end Unlock;

   procedure Protected_Entry_Call
         (Object : Protection_Access;
         E : Protected_Entry_Index;
         Parameter : System.Address;
         Mode : Call_Modes;
         Block : out Communication_Block)
      is
         Level : ATC_Level;
         Caller : Task_ATCB.ATCB_Ptr
               := Task_ATCB.ID_To_ATCB (Task_Stages.Self);
   begin
      Block.Self := Task_ATCB.ATCB_To_ID (Caller);
      Caller.ATC_Nesting_Level := Caller.ATC_Nesting_Level + 1;
      Level := Caller.ATC_Nesting_Level;

      Object.Pending_Call := Caller.Entry_Calls (Level)'access;

      --   I don't think that we need the calling task's lock here.
      --   Only the calling task will get to access this record until
      --   it is queued, since the calling task
      --   will call Next_Entry_Call before releasing the PO lock,
      --   and since Next_Entry_Call always removes Pending_Call.

      Object.Pending_Call.Next := null;
      Object.Pending_Call.Call_Claimed := False;
      Object.Pending_Call.Mode := C_M_To_EQ_Call_Modes (Mode);
      Object.Pending_Call.Abortable := True;
      Object.Pending_Call.Done := False;
      Object.Pending_Call.E := Entry_Index (E);
      Object.Pending_Call.Prio := Caller.Current_Priority;
      Object.Pending_Call.Parameter := Parameter;
      Object.Pending_Call.Called_PO := P_A_To_EQ_Protection_Access (

            Protection_Access (Object));

      Object.Pending_Call.Called_Task := Task_IDs.Null_Task;
      Object.Pending_Call.Exception_To_Raise :=
            Compiler_Exceptions.Null_Exception;

   end Protected_Entry_Call;

   procedure Vulnerable_Cancel_Protected_Entry_Call (
         Caller : Task_ATCB.ATCB_Ptr;
         Call : Entry_Call_Link;
         PO : Protection_Access;
         Call_Cancelled : out Boolean)
      is
         TAS_Result : Boolean;
   begin
      Test_And_Set (Call.Call_Claimed'Address, TAS_Result);
      if TAS_Result then

         Lock (PO);

         Dequeue (PO.Entry_Queues (Protected_Entry_Index (Call.E)), Call);
      else
         Write_Lock (Caller.L);
         while not Call.Done loop
            Cond_Wait (Caller.Rend_Cond, Caller.L);
         end loop;
         Unlock (Caller.L);
      end if;
      Caller.ATC_Nesting_Level := Caller.ATC_Nesting_Level - 1;

      Write_Lock (Caller.L);
      if Caller.Pending_ATC_Level = Caller.ATC_Nesting_Level then
         Caller.Pending_ATC_Level := ATC_Level_Infinity;
         Caller.Aborting := False;
      end if;
      Unlock (Caller.L);
      --   If we have reached the desired ATC nesting level, reset the
      --   requested level to effective infinity, to allow further calls.

      Caller.Exception_To_Raise := Call.Exception_To_Raise;

      Call_Cancelled := TAS_Result;

   end Vulnerable_Cancel_Protected_Entry_Call;

   --   New control flow procedure.
   procedure Wait_For_Completion (Call_Cancelled : out Boolean;
         Block : in out Communication_Block)
      is
         Caller : Task_ATCB.ATCB_Ptr := Task_ATCB.ID_To_ATCB (Block.Self);
         Call : Entry_Call_Link;
         PO : Protection_Access;
         TAS_Result : Boolean;
         Cancelled : Boolean;
   begin
      Defer_Abortion;
      Assert (Caller.ATC_Nesting_Level > ATC_Level_Base'FIRST,
            "Attempt to wait on a nonexistant task entry call.");
      Call := Caller.Entry_Calls (Caller.ATC_Nesting_Level)'access;
      Assert (EQ_C_M_To_Call_Modes (Call.Mode) = Simple_Call,
            "Attempt to wait on a on a conditional or asynchronous call");
      PO := EQ_P_A_To_Protection_Access (Call.Called_PO);

      Write_Lock (Caller.L);
      if Call.Abortable then
         Caller.Suspended_Abortably := True;
         while not Call.Done and then
            Caller.Pending_ATC_Level >= Caller.ATC_Nesting_Level loop
            Cond_Wait (Caller.Cond, Caller.L);
         end loop;
         Caller.Suspended_Abortably := False;
      else
         while not Call.Done loop
            Cond_Wait (Caller.Cond, Caller.L);
         end loop;
      end if;
      Unlock (Caller.L);

      Vulnerable_Cancel_Protected_Entry_Call (Caller, Call, PO, Cancelled);

      Undefer_Abortion;

      Call_Cancelled := Cancelled;

   end Wait_For_Completion;

   procedure Cancel_Protected_Entry_Call (
         Call_Cancelled : out Boolean;
         Block : in out Communication_Block)
      is
         Caller : Task_ATCB.ATCB_Ptr := Task_ATCB.ID_To_ATCB (Block.Self);
         Call : Entry_Call_Link;
         PO : Protection_Access;
         TAS_Result : Boolean;
         Cancelled : Boolean;
   begin
      Defer_Abortion;
      Assert (Caller.ATC_Nesting_Level > ATC_Level_Base'FIRST,
            "Attempt to cancel a nonexistant task entry call.");
      Call := Caller.Entry_Calls (Caller.ATC_Nesting_Level)'access;
      Assert (EQ_C_M_To_Call_Modes (Call.Mode) = Asynchronous_Call,
            "Attempt to cancel a conditional or simple call");
      Assert (Call.Called_Task = Task_IDs.Null_Task,
            "Attempt to use Cancel_Protected_Entry_Call on task entry call.");
      PO := EQ_P_A_To_Protection_Access (Call.Called_PO);
      Vulnerable_Cancel_Protected_Entry_Call (Caller, Call, PO, Cancelled);
      Undefer_Abortion;

      Call_Cancelled := Cancelled;
   end Cancel_Protected_Entry_Call;

   --   This procedure assumes that a task will have to enter the eggshell to
   --   cancel a call, so there is no need to check for cancellation here.
   --   This seems to obviate the need to lock the task at this point, since
   --   the task will be forced to wait before doing the cancellation, meaning
   --   that it will not take place.
   procedure Next_Entry_Call (
         Object : Protection_Access;
         Barriers : Barrier_Vector;
         Parameter : out System.Address;
         E : out Protected_Entry_Index)
      is
         TAS_Result : Boolean;
         Selected_Entry : Protected_Entry_Index;

         Selected_Priority : System.Any_Priority;
         Next_Priority : System.Any_Priority;

   begin
      Object.Call_In_Progress := null;
      if Object.Pending_Call /= null then

         Assert (Task_Stages.Self = Object.Pending_Call.Self,
                "Pending call handled by a task that did not pend it.");
         --   Note that the main cost of the above assertion is likely
         --   to be the call to Self.  If this is not optimized away,
         --   nulling out Assert will not be of much value.

         if Barriers (Protected_Entry_Index (Object.Pending_Call.E)) then
            Test_And_Set (
                  Object.Pending_Call.Call_Claimed'Address,
                  TAS_Result);
            if TAS_Result then
               declare
                  Caller : Task_ATCB.ATCB_Ptr :=
                  Task_ATCB.ID_To_ATCB (Object.Pending_Call.Self);
                  --   Note that Object.Pending_Call.Self has to be Self;
                  --   otherwise, this would be illegal.
                  --   The task that pends the call must keep the object locked
                  --   until it calls Next_Entry_Call, and it will not be
                  --   pending on exit from Next_Entry_Call.
               begin
                  Object.Call_In_Progress := Object.Pending_Call;
               end;
            else
               Object.Pending_Call := null;
            end if;
         else
            Enqueue (
                  Object.Entry_Queues (
                  Protected_Entry_Index (Object.Pending_Call.E)),
                  Object.Pending_Call);
         end if;
            Object.Pending_Call := null;
      end if;
      if Object.Call_In_Progress = null then
      --   The following loop attempts to claim a call on an open barrier.

         loop

            Selected_Entry := Null_Protected_Entry;
            Selected_Priority := System.Priority'First;

            --   The following loop finds the caller waiting on an open barrier
            --   with the highest base priority.  Active priority is not used,
            --   since it should be the same as base priority.  The only way
            --   that the active priority could be higher than the base
            --   priority is if the call had been made from within an eggshell.
            --   As an entry call is a potentially blocking operation, it is
            --   illegal to make one from within an eggshell.

            for B in Barriers'range loop
               if Barriers (B) and then
                  Head (Object.Entry_Queues (B)) /= null then
                  Next_Priority := Head (Object.Entry_Queues (B)).Prio;
                  if (Selected_Entry = Null_Protected_Entry or else
                     Next_Priority >= Selected_Priority) then
                     Selected_Entry := B;
                     Selected_Priority := Next_Priority;
                  end if;
               end if;
            end loop;

            exit when Selected_Entry = Null_Protected_Entry;

            Dequeue_Head (Object.Entry_Queues (Selected_Entry),
                   Object.Call_In_Progress);
            if Object.Call_In_Progress.Abortable then
               Test_And_Set (
                     Object.Call_In_Progress.Call_Claimed'Address,
                     TAS_Result);
               exit when TAS_Result;
               Object.Call_In_Progress := null;
            else
            --   If the call is not abortable, it has already been claimed
            --   for us.
               exit;
            end if;
         end loop;

      end if;

      if Object.Call_In_Progress /= null then
         E := Protected_Entry_Index (Object.Call_In_Progress.E);
         Parameter := Object.Call_In_Progress.Parameter;
      else
         E := Null_Protected_Entry;
      end if;
   end Next_Entry_Call;

   procedure Complete_Entry_Body
         (Object : Protection_Access;
         Pending_Serviced : out Boolean)
      is
   begin
      Exceptional_Complete_Entry_Body (
            Object,
            Pending_Serviced,
            Compiler_Exceptions.Null_Exception);
   end Complete_Entry_Body;

   procedure Exceptional_Complete_Entry_Body (
         Object : Protection_Access;
         Pending_Serviced : out Boolean;
         Ex : Compiler_Exceptions.Exception_ID)
      is
         Caller : Task_ATCB.ATCB_Ptr :=
               Task_ATCB.ID_To_ATCB (Object.Call_In_Progress.Self);
   begin
      Object.Call_In_Progress.Exception_To_Raise := Ex;

      if Object.Pending_Call /= null then
         Assert (Object.Pending_Call = Object.Call_In_Progress,
                "Serviced a protected entry call when another was pending");
         Pending_Serviced := True;
         Caller.ATC_Nesting_Level := Caller.ATC_Nesting_Level - 1;
         Object.Pending_Call := null;
      end if;
      --   If we have completed a pending entry call, pop it and set the
      --   Pending_Serviced flag to indicate that it is complete.

      Write_Lock (Caller.L);
      Object.Call_In_Progress.Done := True;
      Unlock (Caller.L);
      if Object.Call_In_Progress.Mode = Asynchronous_Call then
         Abort_To_Level (
               Task_ATCB.ATCB_To_ID (Caller),
               Object.Call_In_Progress.Level - 1);
      elsif Object.Call_In_Progress.Mode = Simple_Call then
         Cond_Signal (Caller.Cond);
      end if;
   end Exceptional_Complete_Entry_Body;

   procedure Requeue_Protected_Entry (
         Object, New_Object : Protection_Access;
         E : Protected_Entry_Index;
         With_Abort : Boolean) is
   begin
      Object.Call_In_Progress.Abortable := With_Abort;
      Object.Call_In_Progress.E := Entry_Index (E);
      if With_Abort then
         Object.Call_In_Progress.Abortable := True;
         Object.Call_In_Progress.Call_Claimed := False;
      end if;

      if Protection_Access (Object) =
         Protection_Access (New_Object) then

         Enqueue (New_Object.Entry_Queues (E), Object.Call_In_Progress);
      else
         New_Object.Pending_Call := Object.Call_In_Progress;
      end if;
   end Requeue_Protected_Entry;

   --   Leave this until I decide on what to do with Rendezvous.
   procedure Requeue_Task_Entry (
         New_Object : Protection_Access;
         T : in out Task_IDs.Task_ID;
         E : Protected_Entry_Index;
         With_Abort : Boolean)
      is
         Old_Acceptor : Task_ATCB.ATCB_Ptr :=
               Task_ATCB.ID_To_ATCB (Task_Stages.Self);
         Entry_Call : Entry_Call_Link;
   begin
      Write_Lock (Old_Acceptor.L);
      Entry_Call := Old_Acceptor.Call;
      Old_Acceptor.Call := null;
      Unlock (Old_Acceptor.L);
      Entry_Call.Abortable := With_Abort;
      Entry_Call.E := Entry_Index (E);
      Entry_Call.Called_PO :=

                  P_A_To_EQ_Protection_Access (Protection_Access (New_Object));

      if With_Abort then
         Entry_Call.Call_Claimed := False;
      end if;
      New_Object.Pending_Call := Entry_Call;
   end Requeue_Task_Entry;

   function Protected_Count (Object : Protection; E : Protected_Entry_Index)
         return Natural is
   begin
      return Count_Waiting (Object.Entry_Queues (E));
   end Protected_Count;

   procedure Broadcast_Program_Error (Object : Protection_Access) is
         Entry_Call : Entry_Call_Link;
         Current_Task : Task_ATCB.ATCB_Ptr;
         Raise_In_Self : Boolean := True;
   begin
      for E in Object.Entry_Queues'range loop
         Dequeue (Object.Entry_Queues (E), Entry_Call);
         while Entry_Call /= null loop
            Current_Task := Task_ATCB.ID_To_ATCB (Entry_Call.Self);
            Entry_Call.Exception_To_Raise  :=
            Compiler_Exceptions.Program_Error_ID;
            Write_Lock (Current_Task.L);
            Entry_Call.Done := True;
            Unlock (Current_Task.L);
            case EQ_C_M_To_Call_Modes (Entry_Call.Mode) is
               when Simple_Call =>
                  Abort_To_Level (
                        Task_ATCB.ATCB_To_ID (Current_Task),
                        Entry_Call.Level - 1);
               when Conditional_Call =>
                  Assert (False, "Conditional call found on entry queue.");
               when Asynchronous_Call =>
                  Abort_To_Level (
                        Task_ATCB.ATCB_To_ID (Current_Task),
                        Entry_Call.Level - 1);
            end case;
            Dequeue (Object.Entry_Queues (E), Entry_Call);
         end loop;
      end loop;
   end Broadcast_Program_Error;

   function Get_Call_In_Progress (Object : Protection)
         return Entry_Call_Link is
   begin
      return Object.Call_In_Progress;
   end Get_Call_In_Progress;
   pragma Inline (Get_Call_In_Progress);

   procedure Set_Call_In_Progress (
         Object : Protection_Access;
         Call : Entry_Call_Link) is
   begin
      Object.Call_In_Progress  := Call;
   end Set_Call_In_Progress;
   pragma Inline (Set_Call_In_Progress);

end System.Task_Protected_Objects;
