------------------------------------------------------------------------------
--                                                                          --
--                 GNU ADA RUNTIME LIBRARY (GNARL) COMPONENTS               --
--                                                                          --
--                                  A T C B                                 --
--                                                                          --
--                                  B o d y                                 --
--                                                                          --
--                             $Revision: 1.2 $                            --
--                                                                          --
--           Copyright (c) 1991,1992,1993, FSU, All Rights Reserved         --
--                                                                          --
--  GNARL is free software; you can redistribute it and/or modify it  under --
--  terms  of  the  GNU  Library General Public License as published by the --
--  Free Software Foundation; either version 2, or  (at  your  option)  any --
--  later  version.   GNARL is distributed in the hope that it will be use- --
--  ful, but but WITHOUT ANY WARRANTY; without even the implied warranty of --
--  MERCHANTABILITY  or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU Gen- --
--  eral Library Public License for more details.  You should have received --
--  a  copy of the GNU Library General Public License along with GNARL; see --
--  file COPYING. If not, write to the Free Software Foundation,  675  Mass --
--  Ave, Cambridge, MA 02139, USA.                                          --
--                                                                          --
------------------------------------------------------------------------------

with System.Task_Memory;
with System.Task_Primitives; use System.Task_Primitives;
with System.Compiler_Exceptions;

package body System.Task_ATCB is

   --  The TCB contains a variable size array whose dope vector must be
   --  initialized.  This is too complex (and changes too much with changes
   --  in the TCB record) to do explicitely, so a record of the correct size
   --  is declared here and copied into the newly allocated storage.
   --  Discriminant checking is disabled to prevent the discriminant in the
   --  newly created record from being checked before a legal value is
   --  assigned to it.
   procedure Initialize_ATCB (T : ATCB_Ptr; Init : ATCB_Init) is

      procedure Init_Fields is
      begin
         Initialize_Lock (System.Priority'LAST, T.L);
         Initialize_Cond (T.Cond);
         Initialize_Cond (T.Rend_Cond);
         T.Activation_Count := 0;
         T.Awake_Count := 1;                       --  Counting this task.
         T.Awaited_Dependent_Count := 0;
         T.Terminating_Dependent_Count := 0;
         T.Pending_ATC_Level := ATC_Level_Infinity;
         T.ATC_Nesting_Level := 1;                 --  1 deep; 0 = abnormal.
         T.Deferral_Level := 1;                    --  Start out deferred.
         T.Activation_Status := Good_Activations;
         T.Stage := Created;
         T.Exception_To_Raise := Compiler_Exceptions.Null_Exception;
         T.Accepting := Not_Accepting;
         T.Aborting := False;
         T.Suspended_Abortably := False;
         T.Call := null;
         T.Elaborated := Init.Elaborated;
         T.Parent := Init.Parent;
         T.Task_Entry_Point := Init.Task_Entry_Point;
         T.Task_Arg := Init.Task_Arg;
         T.Stack_Size := Init.Stack_Size;
         T.Activator := Init.Activator;
         T.Master_of_Task := Init.Master_of_Task;
         T.Master_Within := Increment_Master (Init.Master_of_Task);

         for I in 1 .. T.Entry_Num loop
            T.Entry_Queues (I).Head := null;
            T.Entry_Queues (I).Tail := null;
         end loop;

         for L in T.Entry_Calls'range loop
            T.Entry_Calls (L).Next := null;
            T.Entry_Calls (L).Self := ATCB_To_ID (T);
            T.Entry_Calls (L).Level := L;
         end loop;

      end Init_Fields;
      pragma Inline (Init_Fields);

   begin
      Init_Fields;

      --  Link the task into the list of all tasks.
      if T.Parent /= null then
         Defer_Abortion;
         Write_Lock (All_Tasks_L);
      end if;
      T.All_Tasks_Link := All_Tasks_List;
      All_Tasks_List := T;
      if T.Parent /= null then
         Unlock (All_Tasks_L);
         Undefer_Abortion;
      end if;
   end Initialize_ATCB;

   --  This creates a new ATCB using the low level allocation routines
   --  (essentially a protected version of malloc()).  This is done because
   --  the new operator can be changed by the user, and may involve
   --  allocation from pools (which would limit the number of tasks), might
   --  block on insufficiant memory, or might fragment the user's heap
   --  behind his back.
   function New_ATCB (Init : ATCB_Init) return ATCB_Ptr is
      subtype Constrained_ATCB is Ada_Task_Control_Block (Init.Entry_Num);
      Initialized_ATCB : Constrained_ATCB;
      T : ATCB_Ptr;
      A : System.Address;

      function Address_to_Pointer is new
            Unchecked_Conversion (System.Address, ATCB_Ptr);

   begin
      A := Task_Memory.Low_Level_New (
            Constrained_ATCB'SIZE / System.Storage_Unit);
      T := Address_to_Pointer (A);
      T.all := Initialized_ATCB;
      Initialize_ATCB (T, Init);
      return T;
   end New_ATCB;

   --  This creates a new ATCB using unprotected low level allocation routines
   --  (essentially malloc()).  This is done for allocating the ATCB for the
   --  initial task, since this must be done before initializing the low
   --  level tasking, and locks (and hence protected Low_Level_New) cannot
   --  be used until it is.
   function Unsafe_New_ATCB (Init : ATCB_Init) return ATCB_Ptr is
      subtype Constrained_ATCB is Ada_Task_Control_Block (Init.Entry_Num);
      Initialized_ATCB : Constrained_ATCB;
      T : ATCB_Ptr;
      A : System.Address;

      function Address_to_Pointer is new
            Unchecked_Conversion (System.Address, ATCB_Ptr);

   begin
      A := Task_Memory.Unsafe_Low_Level_New (
            Constrained_ATCB'SIZE / System.Storage_Unit);
      T := Address_to_Pointer (A);
      T.all := Initialized_ATCB;
      return T;
   end Unsafe_New_ATCB;

   procedure Free_ATCB (T : in out ATCB_Ptr) is
      function Pointer_to_Address is new
            Unchecked_Conversion (ATCB_Ptr, System.Address);
   begin
      Finalize_Lock (T.L);
      Finalize_Cond (T.Cond);
      Finalize_Cond (T.Rend_Cond);
      Task_Memory.Low_Level_Free (Pointer_to_Address (T));
   end Free_ATCB;

end System.Task_ATCB;
