------------------------------------------------------------------------------
--                                                                         --
--                GNU ADA RUNTIME LIBRARY (GNARL) COMPONENTS               --
--                                                                         --
--                            R T S _ A b o r t                            --
--                                                                         --
--                                 B o d y                                 --
--                                                                         --
--                            $Revision: 1.2 $                             --
--                                                                         --
--          Copyright (c) 1991,1992,1993, FSU, All Rights Reserved         --
--                                                                         --
--  GNARL is free software; you can redistribute it and/or modify it  under --
--  terms  of  the  GNU  Library General Public License as published by the --
--  Free Software Foundation; either version 2, or (at  your  option)  any  --
--  later  version.   GNARL is distributed in the hope that it will be use- --
--  ful, but but WITHOUT ANY WARRANTY; without even the implied warranty of --
--  MERCHANTABILITY  or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU Gen- --
--  eral Library Public License for more details.  You should have received --
--  a  copy of the GNU Library General Public License along with GNARL; see --
--  file COPYING. If not, write to the Free Software Foundation,  675  Mass --
--  Ave, Cambridge, MA 02139, USA.                                          --
--                                                                         --
------------------------------------------------------------------------------

with System.Task_ATCB;
with System.Task_Rendezvous;
with System.Error_Reporting; use System.Error_Reporting;
with System.Task_Primitives; use System.Task_Primitives;

package body System.Task_Abortion is

   function "=" (L, R : Task_ATCB.Task_Stage)
         return Boolean renames Task_ATCB."=";
   function "=" (L, R : Task_ATCB.ATCB_Ptr)
         return Boolean renames Task_ATCB."=";
   function "=" (L, R : Task_ATCB.Accepting_State)
         return Boolean renames Task_ATCB."=";

   procedure Defer_Abortion is
      T : Task_ATCB.ATCB_Ptr := Task_ATCB.ID_To_ATCB (Task_Stages.Self);
   begin
      T.Deferral_Level := T.Deferral_Level + 1;
   end Defer_Abortion;

   --  pre : Self does not hold any locks!
   procedure Undefer_Abortion is
      T : Task_ATCB.ATCB_Ptr := Task_ATCB.ID_To_ATCB (Task_Stages.Self);
   begin
      T.Deferral_Level := T.Deferral_Level - 1;
      if T.Deferral_Level = ATC_Level'FIRST and then
            T.Pending_ATC_Level < T.ATC_Nesting_Level then
         T.Deferral_Level := T.Deferral_Level + 1; --  go away w/ GNARLI 1.28
         raise Abort_Signal;
      end if;
   end Undefer_Abortion;

   procedure Abort_To_Level (Target : Task_ID; L : ATC_Level) is
      T : Task_ATCB.ATCB_Ptr := Task_ATCB.ID_To_ATCB (Target);
   begin
      Write_Lock (T.L);
      if T.Pending_ATC_Level > L then
         T.Pending_ATC_Level := L;

         if not T.Aborting then
            T.Aborting := True;

            if T.Suspended_Abortably then
               Cond_Signal (T.Cond);
               Cond_Signal (T.Rend_Cond);
               --  Ugly; think about ways to have tasks suspend on one
               --  condition variable.
            else

               if Same_Task (Target, Task_Stages.Self) then
                  Unlock (T.L);
                  Abort_Task (T.LL_TCB'access);
                  return;
               elsif T.Stage /= Task_ATCB.Terminated then
                  Abort_Task (T.LL_TCB'access);
               end if;
               --  If this task is aborting itself, it should unlock itself
               --  before calling abort, as it is unlikely to have the
               --  opportunity to do so afterwords.  On the other hand, if
               --  another task is being aborted, we want to make sure it is
               --  not terminated, since
               --  there is no need to abort a terminated task, and it may be
               --  illegal if it has stopped executing.  In this case, the
               --  Abort_Task must take place under the protection of the
               --  mutex, so that we know that Stage/=Terminated.

            end if;
         end if;
      end if;
      Unlock (T.L);
   end Abort_To_Level;

   procedure Abort_Handler (Context : Machine_Specifics.Pre_Call_State) is
      T : Task_ATCB.ATCB_Ptr := Task_ATCB.ID_To_ATCB (Task_Stages.Self);
   begin
      if T.Deferral_Level = 0 and then
            T.Pending_ATC_Level < T.ATC_Nesting_Level then
         raise Abort_Signal;
         --  Not a good idea; signal remains masked after the Abortion
         --  exception is handled.  There are a number of solutions :
         --  1. Change the PC to point to code that raises the exception and
         --     then jumps to the location that was interrupted.
         --  2. Longjump to the code that raises the exception.
         --  3. Unmask the signal in the Abortion exception handler
         --     (in the RTS).
      end if;
   end Abort_Handler;

   --  process abortion of child tasks.
   --  Abortion should be dererred when calling this routine.
   --  No mutexes should be locked when calling this routine.
   procedure Abort_Dependents (Abortee : Task_ID) is
      Temp_T : Task_ATCB.ATCB_Ptr;
      Temp_P : Task_ATCB.ATCB_Ptr;
      Old_Pending_ATC_Level : ATC_Level_Base;
      TAS_Result : Boolean;
      A : Task_ATCB.ATCB_Ptr := Task_ATCB.ID_To_ATCB (Abortee);
   begin
      Write_Lock (Task_ATCB.All_Tasks_L);
      Temp_T := Task_ATCB.All_Tasks_List;
      while Temp_T /= null loop
         Temp_P := Temp_T.Parent;
         while Temp_P /= null loop
            exit when Temp_P = A;
            Temp_P := Temp_P.Parent;
         end loop;
         if Temp_P = A then
            Temp_T.Accepting := Task_ATCB.Not_Accepting;
            --  send cancel signal.
            Task_Rendezvous.Complete_on_Sync_Point
                  (Task_ATCB.ATCB_To_ID (Temp_T));
            Abort_To_Level (Task_ATCB.ATCB_To_ID (Temp_T), 0);
         end if;
         Temp_T := Temp_T.All_Tasks_Link;
      end loop;
      Unlock (Task_ATCB.All_Tasks_L);
   end Abort_Dependents;

   --  is called to initiate abortion, however, the actual abortion
   --  is done by abortee by means of Abort_Handler
   procedure Abort_Tasks (Tasks : Task_List) is
      Abortee : Task_ATCB.ATCB_Ptr;
      Aborter : Task_ATCB.ATCB_Ptr;
      Activator : Task_ATCB.ATCB_Ptr;
      TAS_Result : Boolean;
      Old_Pending_ATC_Level : ATC_Level_Base;
   begin
      Defer_Abortion;
      --  begin non-abortable section
      Aborter := Task_ATCB.ID_To_ATCB (Task_Stages.Self);
      for I in Tasks'range loop
         Abortee := Task_ATCB.ID_To_ATCB (Tasks (I));
         Abortee.Accepting := Task_ATCB.Not_Accepting;
         Task_Rendezvous.Complete_on_Sync_Point
               (Task_ATCB.ATCB_To_ID (Abortee));
         Abort_To_Level (Task_ATCB.ATCB_To_ID (Abortee), 0);

         --  process abortion of child tasks
         Abort_Dependents (Task_ATCB.ATCB_To_ID (Abortee));

      end loop;
      --  end non-abortable section
      Undefer_Abortion;
   end Abort_Tasks;

end System.Task_Abortion;
