------------------------------------------------------------------------------
--                                                                          --
--                 GNU ADA RUNTIME LIBRARY (GNARL) COMPONENTS               --
--                                                                          --
--                              P O S I X _ R T E                           --
--                                                                          --
--                                  S p e c                                 --
--                                                                          --
--                             $Revision: 1.2 $                            --
--                                                                          --
--           Copyright (c) 1991,1992,1993, FSU, All Rights Reserved         --
--                                                                          --
--  GNARL is free software; you can redistribute it and/or modify it  under --
--  terms  of  the  GNU  Library General Public License as published by the --
--  Free Software Foundation; either version 2, or (at  your  option)  any  --
--  later  version.   GNARL is distributed in the hope that it will be use- --
--  ful, but but WITHOUT ANY WARRANTY; without even the implied warranty of --
--  MERCHANTABILITY  or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU Gen- --
--  eral Library Public License for more details.  You should have received --
--  a  copy of the GNU Library General Public License along with GNARL; see --
--  file COPYING. If not, write to the Free Software Foundation,  675  Mass --
--  Ave, Cambridge, MA 02139, USA.                                          --
--                                                                          --
------------------------------------------------------------------------------

--  This package interfaces with the POSIX real-time extensions.  It may
--  also implement some of them using UNIX operations.
--  It is not a complete interface; it
--  only includes what is needed to implement the Ada runtime.

--  temporarily, should really only be for 1
with System.POSIX_ERROR; use System.POSIX_ERROR;

with System.POSIX_Constants;

package System.POSIX_RTE is

   subtype time_t is Integer range 0 .. Integer'LAST;
   type Nanoseconds is new Integer;
   subtype Fractional_Second is Nanoseconds range 0 .. 10#1#E9 - 1;
   --  This is dependent on the stdtypes.h header file.

   type timespec is record
      tv_sec : time_t;
      tv_nsec : Fractional_Second;
   end record;

   timespec_First : constant timespec :=
         timespec' (time_t'FIRST, Fractional_Second'FIRST);
   timespec_Last : constant timespec :=
         timespec' (time_t'LAST, Fractional_Second'LAST);
   timespec_Zero : constant timespec :=
         timespec' (time_t'FIRST, Fractional_Second'FIRST);
   timespec_Unit : constant timespec :=
         timespec' (time_t'FIRST, Fractional_Second'FIRST + 1);
   --  This is dependent on the POSIX.4 implementation; the draft standard
   --  only says that fields of these names and types (with Integer for long)
   --  will be in the record.  There may be other fields, and these do not have
   --  to be in the indicated position.  This should really be done by
   --  getting the sizes and offsets using get_posix_constants and building
   --  the record to match using representation clauses.

   type clock_id_t is new Integer;
   --  This clock_id_t is defined as an integer in POSIX
   CLOCK_REALTIME : constant clock_id_t := 0;
   --  We currently implement only Realtime clock.

   type Signal_Number is range 0 .. 31;
   for Signal_Number'Size use 32;
   type sigset_t is new Integer;
   All_Signals : constant sigset_t := -1;

   type sigval is record
      u0 : Integer;
   end record;
   --  this is not used at the moment, need to update to reflect
   --  any changes in the Pthreads signal.h in the future

   type struct_siginfo is record
      si_signo : Signal_Number;
      si_code : Integer;
      si_value : sigval;
   end record;

   type siginfo_ptr is access struct_siginfo;

   type sigset_t_ptr is access sigset_t;

   SIG_ERR : constant := -1;
   SIG_DFL : constant := 0;
   SIG_IGN : constant := 1;
   --  constants for sa_handler

   type struct_sigaction is record
      sa_handler : System.Address; --  address of signal handler
      sa_mask : sigset_t;
      --  additional signals to be blocked during
      --  execution of signal-catching function
      sa_flags : Integer;
      --  special flags to affect behavior
      --  of signal
   end record;

   --  signal catching function (signal handler) will have the
   --  following profile :
   --  procedure Signal_Handler (signo : Signal_Number;
   --                           info : siginfo_ptr;
   --                           context : sigcontext_ptr);

   SA_NOCLDSTOP : constant := 16#0008#;  --  don't send a SIGCHLD on child stop
   SA_SIGINFO : constant := 16#0010#;
   --  sa_flags flags for struct_sigaction

   SIG_BLOCK : constant := 16#0001#;
   SIG_UNBLOCK : constant := 16#0002#;
   SIG_SETMASK : constant := 16#0004#;
   --  sigprocmask flags (how)

   type jmp_buf is array (1 .. POSIX_Constants.pthread_jmp_buf_size)
         of Integer;
   type sigjmp_buf is array (
         1 .. POSIX_Constants.pthread_sigjmp_buf_size) of Integer;
   type jmp_buf_ptr is access jmp_buf;
   type sigjmp_buf_ptr is access sigjmp_buf;
   --  environment for long jumps

   procedure sigaction (sig : Signal_Number;
         act : struct_sigaction;
         oact : out struct_sigaction;
         result : out POSIX_Error.Return_Code);
   pragma Inline (sigaction);
   --  install new sigaction structure and obtain old one

   procedure sigprocmask (how : Integer;
         set : sigset_t;
         oset : out sigset_t;
         Result : out POSIX_Error.Return_Code);
   pragma Inline (sigprocmask);
   --  install new signal mask and obtain old one

   procedure sigsuspend (mask : sigset_t;
         result : out Return_Code);
   pragma Inline (sigsuspend);
   --  suspend waiting for signals in mask and resume after
   --  executing handler or take default action

   procedure sigpending (set : out sigset_t;
         result : out Return_Code);
   pragma Inline (sigpending);
   --  get pending signals on thread and process

   procedure longjmp (env : jmp_buf; val : Integer);
   pragma Inline (longjmp);
   --  execute a jump across procedures according to setjmp

   procedure siglongjmp (env : sigjmp_buf; val : Integer);
   pragma Inline (siglongjmp);
   --  execute a jump across procedures according to sigsetjmp

   procedure setjmp (env : jmp_buf; result : out Integer);
   pragma Inline (setjmp);
   --  set up a jump across procedures and return here with longjmp

   procedure sigsetjmp (env : sigjmp_buf; savemask : Integer;
                                 result : out Integer);
   pragma Inline (sigsetjmp);
   --  set up a jump across procedures and return here with siglongjmp

   --  set signal mask for a signal number
   function Sigmask (Signal : Signal_Number) return sigset_t;

   --  temporarily, should really only be for 1

   procedure clock_gettime (ID : clock_id_t;
         CT : out timespec;
         result : out Integer);        --  use Return_Code in pthreads

   --  clock_gettime gets POSIX time

   SIGKILL : constant Signal_Number := 9;
   SIGSTOP : constant Signal_Number := 17;
   --  signals which cannot be masked

   SIGILL  : constant Signal_Number := 4;
   SIGABRT : constant Signal_Number := 6;
   SIGEMT  : constant Signal_Number := 7;
   SIGFPE  : constant Signal_Number := 8;
   SIGBUS  : constant Signal_Number := 10;
   SIGSEGV : constant Signal_Number := 11;
   SIGPIPE : constant Signal_Number := 13;
   --  some synchronous signals (cannot be used for interrupt entries)

   SIGALRM : constant Signal_Number := 14;
   --  alarm signal (cannot be used for interrupt entries)

   SIGUSR1 : constant Signal_Number := 30;
   SIGUSR2 : constant Signal_Number := 31;
   --  user-defined signals

   SIGILL_Mask  : constant sigset_t := 2#1#E3;
   SIGABRT_Mask : constant sigset_t := 2#1#E5;
   SIGEMT_Mask  : constant sigset_t := 2#1#E6;
   SIGFPE_Mask  : constant sigset_t := 2#1#E7;
   SIGBUS_Mask  : constant sigset_t := 2#1#E9;
   SIGSEGV_Mask : constant sigset_t := 2#1#E10;
   SIGPIPE_Mask : constant sigset_t := 2#1#E12;
   SIGALRM_Mask : constant sigset_t := 2#1#E13;

   Masked_Signals : constant sigset_t := -1 -
         (SIGILL_Mask + SIGABRT_Mask + SIGEMT_Mask + SIGFPE_Mask +
         SIGBUS_Mask + SIGSEGV_Mask + SIGPIPE_Mask + SIGALRM_Mask);
   --  signals which are masked initially for each task (synchronous + alarm)

end System.POSIX_RTE;
