------------------------------------------------------------------------------
--                                                                          --
--                 GNU ADA RUNTIME LIBRARY (GNARL) COMPONENTS               --
--                                                                          --
--                             P O S I X _ R T E                            --
--                                                                          --
--                                  B o d y                                 --
--                                                                          --
--                             $Revision: 1.2 $                            --
--                                                                          --
--           Copyright (c) 1991,1992,1993, FSU, All Rights Reserved         --
--                                                                          --
--  GNARL is free software; you can redistribute it and/or modify it  under --
--  terms  of  the  GNU  Library General Public License as published by the --
--  Free Software Foundation; either version 2, or (at  your  option)  any --
--  later  version.   GNARL is distributed in the hope that it will be use- --
--  ful, but but WITHOUT ANY WARRANTY; without even the implied warranty of --
--  MERCHANTABILITY  or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU Gen- --
--  eral Library Public License for more details.  You should have received --
--  a  copy of the GNU Library General Public License along with GNARL; see --
--  file COPYING. If not, write to the Free Software Foundation,  675  Mass --
--  Ave, Cambridge, MA 02139, USA.                                          --
--                                                                          --
------------------------------------------------------------------------------

--  This package interfaces with the POSIX real-time extensions.  It may
--  also implement some of them using UNIX operations.
--  It is not a complete interface; it
--  only includes what is needed to implement the Ada runtime.

with Unchecked_Conversion;

package body System.POSIX_RTE is

   type sigaction_ptr is access struct_sigaction;
   function Address_to_Pointer is new
         Unchecked_Conversion (System.Address, sigaction_ptr);
   function Address_to_Pointer is new
         Unchecked_Conversion (System.Address, sigset_t_ptr);
   type timespec_ptr is access timespec;
   function Address_to_Pointer is new
         Unchecked_Conversion (System.Address, timespec_ptr);
   function Address_to_Pointer is new
         Unchecked_Conversion (System.Address, jmp_buf_ptr);
   function Address_to_Pointer is new
         Unchecked_Conversion (System.Address, sigjmp_buf_ptr);

   function clock_gettime_base (ID : clock_id_t;
         CT_Add : System.Address) return Integer;
   pragma INTERFACE (C, clock_gettime_base);
   pragma INTERFACE_NAME (clock_gettime_base, "clock_gettime");

   procedure clock_gettime (ID : clock_id_t; CT : out timespec;
         result : out Integer) is
   begin
      result := clock_gettime_base (ID, CT'Address);
   end clock_gettime;

   function sigaction_base (sig : Signal_Number;
         act : sigaction_ptr;
         oact : sigaction_ptr) return Return_Code;
   pragma INTERFACE (C, sigaction_base);
   pragma INTERFACE_NAME (sigaction_base, "sigaction");

   function sigprocmask_base (how : Integer;
         set : sigset_t_ptr;
         oset : sigset_t_ptr) return Return_Code;
   pragma INTERFACE (C, sigprocmask_base);
   pragma INTERFACE_NAME (sigprocmask_base, "sigprocmask");

   function sigsuspend_base (mask : sigset_t_ptr) return Return_Code;
   pragma INTERFACE (C, sigsuspend_base);
   pragma INTERFACE_NAME (sigsuspend_base, "sigsuspend");

   function sigpending_base (set : sigset_t_ptr) return Return_Code;
   pragma INTERFACE (C, sigpending_base);
   pragma INTERFACE_NAME (sigpending_base, "sigpending");

   procedure longjmp_base (env : jmp_buf_ptr; val : Integer);
   pragma INTERFACE (C, longjmp_base);
   pragma INTERFACE_NAME (longjmp_base, "longjmp");

   procedure siglongjmp_base (env : sigjmp_buf_ptr; val : Integer);
   pragma INTERFACE (C, siglongjmp_base);
   pragma INTERFACE_NAME (siglongjmp_base, "siglongjmp");

   function setjmp_base (env : jmp_buf_ptr) return Integer;
   pragma INTERFACE (C, setjmp_base);
   pragma INTERFACE_NAME (setjmp_base, "setjmp");

   function sigsetjmp_base (env : sigjmp_buf_ptr; savemask : Integer)
         return Integer;
   pragma INTERFACE (C, sigsetjmp_base);
   pragma INTERFACE_NAME (sigsetjmp_base, "sigsetjmp");

   --  install new sigaction structure and obtain old one
   procedure sigaction (sig : Signal_Number;
         act : struct_sigaction;
         oact : out struct_sigaction;
         result : out POSIX_Error.Return_Code) is
   begin
      result := sigaction_base (sig, Address_to_Pointer (act'Address),
      Address_to_Pointer (oact'Address));
   end sigaction;

   --  install new signal mask and obtain old one
   procedure sigprocmask (how : Integer;
         set : sigset_t;
         oset : out sigset_t;
         Result : out POSIX_Error.Return_Code) is
   begin
         Result := sigprocmask_base (how, Address_to_Pointer (set'Address),
         Address_to_Pointer (oset'Address));
   end sigprocmask;

   --  suspend waiting for signals in mask and resume after
   --  executing handler or take default action
   procedure sigsuspend (mask : sigset_t;
         result : out Return_Code) is
   begin
      result := sigsuspend_base (Address_to_Pointer (mask'Address));
   end sigsuspend;

   --  get pending signals on thread and process
   procedure sigpending (set : out sigset_t;
         result : out Return_Code) is
   begin
      result := sigpending_base (Address_to_Pointer (set'Address));
   end sigpending;

   --  execute a jump across procedures according to setjmp
   procedure longjmp (env : jmp_buf; val : Integer) is
   begin
      longjmp_base (Address_to_Pointer (env'Address), val);
   end longjmp;

   --  execute a jump across procedures according to sigsetjmp
   procedure siglongjmp (env : sigjmp_buf; val : Integer) is
   begin
      siglongjmp_base (Address_to_Pointer (env'Address), val);
   end siglongjmp;

   --  set up a jump across procedures and return here with longjmp
   procedure setjmp (env : jmp_buf; result : out Integer) is
   begin
      result := setjmp_base (Address_to_Pointer (env'Address));
   end setjmp;

   --  set up a jump across procedures and return here with siglongjmp
   procedure sigsetjmp (env : sigjmp_buf; savemask : Integer;
         result : out Integer) is
   begin
      result := sigsetjmp_base (Address_to_Pointer (env'Address), savemask);
   end sigsetjmp;

   function Sigmask (Signal : Signal_Number) return sigset_t is
   begin
      return (sigset_t (2 ** Integer (Signal - 1)));
   end Sigmask;
   pragma Inline (Sigmask);

end System.POSIX_RTE;
