/*
 * Klondike
 *
 * Copyright (C) Evan Harris, 1991, 1994
 *
 * Permission is granted to freely redistribute and modify this code,
 * providing the author(s) get credit for having written it.
 */

#include <stdlib.h>
#include <stdio.h>
#include <time.h>
#include <vga.h>
#include <vgamouse.h>

#include "vga16.h"
#include "klondike.h"


#define KL_SCREENMODE G640x480x16
#define KL_SCREENWIDTH 640
#define KL_SCREENHEIGHT 480

#define KL_TEXTFG 1
#define KL_TEXTBG 0
#define KL_BUTTONFG 2
#define KL_BUTTONBG 6
#define KL_MOUSEFG 5
#define KL_MOUSEDARKFG 2
#define KL_OUTLINE 6

#define NUMCARDIMAGES 52
#define CARDACTUALWIDTH 53
#define CARDIMAGEWIDTH 56
#define CARDIMAGEHEIGHT 80

#define VERSION "KLONDIKE\n  v1.0"
#define VERSIONLEFT (KL_SCREENWIDTH - 72)

#ifndef CARDSFILE
#define CARDSFILE "Cards56x80"
#endif


#define CARDTOIMAGE(card)	(card % NUMCARDIMAGES)

#define BITWIDTH CARDIMAGEWIDTH /* multiple of 8 */
#define BITHEIGHT CARDIMAGEHEIGHT
#define BITSHOWLEFT 24		/* multiple of 8 */
#define BITTOP 24
#define COLWIDTH (BITWIDTH + 8)
#define TOPROWHEIGHT	(BITHEIGHT + 4)


#define KL_QUITLEFT (KL_SCREENWIDTH - 80)
#define KL_QUITRIGHT (KL_QUITLEFT + 80)
#define KL_QUITBOTTOM 104
#define KL_QUITTOP (KL_QUITBOTTOM - 16)

#define KL_NEWGAMELEFT KL_QUITLEFT
#define KL_NEWGAMERIGHT KL_QUITRIGHT
#define KL_NEWGAMEBOTTOM 48
#define KL_NEWGAMETOP (KL_NEWGAMEBOTTOM - 16)

#define KL_RESTARTLEFT KL_QUITLEFT
#define KL_RESTARTRIGHT KL_QUITRIGHT
#define KL_RESTARTBOTTOM 76
#define KL_RESTARTTOP (KL_RESTARTBOTTOM - 16)


#define STOCKLEFT 0
#define STOCKTOP 0
#define PILELEFT COLWIDTH
#define PILETOP 0
#define FOUNDLEFT (3 * COLWIDTH)
#define FOUNDTOP 0


#define KL_PALETTESIZE 7

int palette[KL_PALETTESIZE * 3] = {
    0x00, 0x20, 0x00,		/* green */
    0x3f, 0x3f, 0x3f,		/* white */
    0x00, 0x00, 0x00,		/* black */
    0x20, 0x00, 0x00,		/* red */
    0x00, 0x00, 0x20,		/* blue */
    0x3f, 0x3f, 0x00,		/* yellow */
    0x18, 0x18, 0x18,		/* grey */
};

#define MIN(a, b) ((a) < (b) ? (a) : (b))
static void LoadCards(char *file);
static long GetMouseButton(void);
static long WhileMouseButton(void);
static int ParsePos(long pos, int downcard, int buttonup);

static unsigned char ***imageCard;
static unsigned char **backCard;
static unsigned char **outlineCard;
static unsigned char *blankLine;


void
InitDisplay(int argc, char **argv)
{
    vga_disabledriverreport();
    vga_init();
    vga_setmousesupport(1);
    
    if (vga_setmode(KL_SCREENMODE) != 0) {
	fprintf(stderr, "Mode %s not available!\n",
		vga_getmodename(KL_SCREENMODE));
	exit(1);
    }

    vga16_init();

    LoadCards(CARDSFILE);

    blankLine = (unsigned char *)calloc(KL_SCREENWIDTH, sizeof(unsigned char));
    if (blankLine == NULL) {
	fprintf(stderr, "Error: cannot get memory for blankLine\n");
	exit(1);
    }

    vga_setpalvec(0, KL_PALETTESIZE, &palette[0]);

    vga16_text(KL_NEWGAMELEFT, KL_NEWGAMEBOTTOM, " NEW GAME ",
	       KL_BUTTONFG, KL_BUTTONBG);
    vga16_text(KL_RESTARTLEFT, KL_RESTARTBOTTOM, " RESTART  ",
	       KL_BUTTONFG, KL_BUTTONBG);
    vga16_text(KL_QUITLEFT, KL_QUITBOTTOM, "   QUIT   ",
	       KL_BUTTONFG, KL_BUTTONBG);

    vga16_text(VERSIONLEFT, 200, VERSION, KL_TEXTFG, KL_TEXTBG);
}


void
EndDisplay()
{
    vga_setmode(TEXT);
}


static void
LoadCards(char *file)
{
    int i, j, k, l, c, colour;
    FILE *f;

    f = fopen(file, "r");
    if (f == NULL) {
	fprintf(stderr, "Cannot find '%s'\n", file);
	exit(1);
    }

    imageCard = (unsigned char ***)malloc(NUMCARDIMAGES
					  * sizeof(unsigned char **));
    if (imageCard == NULL) {
	fprintf(stderr, "Error: cannot get memory for imageCard\n");
	exit(1);
    }
    for (i = 0; i < NUMCARDIMAGES; i++) {
	imageCard[i] = (unsigned char **)malloc(CARDIMAGEHEIGHT
						* sizeof(unsigned char *));
	if (imageCard == NULL) {
	    fprintf(stderr, "Error: cannot get memory for imageCard[%d]\n",
		    i);
	    exit(1);
	}
	for (j = 0; j < CARDIMAGEHEIGHT; j++) {
	    imageCard[i][j] = (unsigned char *)malloc(CARDIMAGEWIDTH
						      * sizeof(unsigned char));
	    if (imageCard[i][j] == NULL) { 
		fprintf(stderr,
			"Error: cannot get memory for imageCard[%d][%d]\n",
			i, j);
		exit(1);
	    }
	    
	    if (SUIT(i) == SPADES || SUIT(i) == CLUBS) {
		colour = 2;
	    } else {
		colour = 3;
	    }
	    if (TYPE(i) < JACK) {
		for (k = 0; k < CARDIMAGEWIDTH / 8; k++) {
		    if ((c = getc(f)) == EOF) {
			fprintf(stderr, "Unexpected EOF in '%s'\n", file);
			exit(1);
		    }
		    for (l = 0; l < 8; l++) {
			imageCard[i][j][8 * k + l] =
			    (c & 1 << (7 - l)) ? colour : 1;
		    }
		}
		for (k = CARDACTUALWIDTH; k < CARDIMAGEWIDTH; k++) {
		    imageCard[i][j][k] = 0;
		}
	    } else {
		for (k = 0; k < CARDIMAGEWIDTH / 2; k++) {
		    if ((c = getc(f)) == EOF) {
			fprintf(stderr, "Unexpected EOF in '%s'\n", file);
			exit(1);
		    }
		    imageCard[i][j][2 * k] = (unsigned char)c >> 4;
		    imageCard[i][j][2 * k + 1] = (unsigned char)c & 0xf;
		}
	    }
	}
    }

    fclose(f);

    backCard = malloc(CARDIMAGEHEIGHT * sizeof(unsigned char *));
    outlineCard = malloc(CARDIMAGEHEIGHT * sizeof(unsigned char *));
    if (backCard == NULL || outlineCard == NULL) {
	fprintf(stderr, "Error: cannot get memory for cards");
	exit(1);
    }
    for (i = 0; i < CARDIMAGEHEIGHT; i++) {
	backCard[i] = malloc(CARDIMAGEWIDTH * sizeof(unsigned char));
	outlineCard[i] = calloc(CARDIMAGEWIDTH, sizeof(unsigned char));
	if (backCard[i] == NULL || outlineCard[i] == NULL) {
	    fprintf(stderr, "Error: cannot get memory for cards");
	    exit(1);
	}
    }
    for (i = 0; i < CARDACTUALWIDTH; i++) {
	outlineCard[0][i] = KL_OUTLINE;
	outlineCard[CARDIMAGEHEIGHT - 1][i] = KL_OUTLINE;
    }
    for (i = 0; i < CARDIMAGEHEIGHT; i++) {
	outlineCard[i][0] = KL_OUTLINE;
	outlineCard[i][CARDACTUALWIDTH - 1] = KL_OUTLINE;
    }
    for (i = 0; i < CARDIMAGEHEIGHT; i++) {
	for (j = 0; j < CARDACTUALWIDTH; j++) {
	    if ((i / 4) % 2 == (j / 4) % 2) {
		backCard[i][j] = 4;
	    } else {
		backCard[i][j] = 1;
	    }
	}
	for (; j < CARDIMAGEWIDTH; j++) {
	    backCard[i][j] = 0;
	}
    }
}


void
InitRandom(unsigned char state)
{
    static long seed;
    
    if (state == NEW) {
	seed = time(NULL);
    }
    srand48(seed);
}


long
Random(long max)
{
    return lrand48() % max;
}


static unsigned char columnTop[COLUMNS];

/* We redisplay a whole column, rather than do anything tricky. */
void
DisplayColumn(short col)
{
    unsigned char **image;
    int card, length, row, height, line;
    int bittop = BITTOP;
	
    if (col == STOCK || col == PILE) {
	DisplayStockPile();
	return;
    }

    for (card = column[col], length = 0; card != NOCARD; card = next[card])
	length++;

    if (length > 0 && (KL_SCREENHEIGHT - TOPROWHEIGHT) / length < bittop)
	bittop = (KL_SCREENHEIGHT - TOPROWHEIGHT) / length;
    columnTop[col] = bittop;

    card = column[col];
    row = TOPROWHEIGHT;
    col = col * COLWIDTH;

    while (card != NOCARD && row < KL_SCREENHEIGHT - 1) {
	if (hidden[card])
	    image = backCard;
	else
	    image = imageCard[CARDTOIMAGE(card)];
	if (next[card] == NOCARD)
	    height = BITHEIGHT;
	else
	    height = bittop;
	if (row + height >= KL_SCREENHEIGHT)
	    height -= row + height - KL_SCREENHEIGHT + 1;

	for (line = 0; line < height; line++) {
	    vga16_drawscansegment(image[line], col, row + line, BITWIDTH);
	}

	row += height;
	card = next[card];
    }
    for (; row < KL_SCREENHEIGHT; row++) {
	vga16_drawscansegment(blankLine, col, row, BITWIDTH);
    }

    return;
}


void
DisplayStockPile()
{
    unsigned char **card;
    int line;
    
    if (stock == NOCARD) {
	card = outlineCard;
    } else {
	card = backCard;
    }
    for (line = 0; line < CARDIMAGEHEIGHT; line++) {
	vga16_drawscansegment(card[line], STOCKLEFT, STOCKTOP + line,
			      BITWIDTH);
    }
    if (pile == NOCARD) {
	card = outlineCard;
    } else {
	card = imageCard[CARDTOIMAGE(pile)];
    }
    for (line = 0; line < CARDIMAGEHEIGHT; line++) {
	vga16_drawscansegment(card[line], PILELEFT, PILETOP + line, BITWIDTH);
    }
}


void
DisplayFoundations()
{
    unsigned char **card;
    int i, line, foundleft;

    foundleft = FOUNDLEFT;
    for (i = 0; i < NUMSUITS; i++) {
	if (foundation[i] == 0) {
	    card = outlineCard;
	} else {
	    card = imageCard[CARD(i % 4, foundation[i] - 1)];
	}
	for (line = 0; line < CARDIMAGEHEIGHT; line++) {
	    vga16_drawscansegment(card[line], foundleft, FOUNDTOP + line,
				  BITWIDTH);
	}
	foundleft += COLWIDTH;
    }
}


short
GetCmd()
{
    int c, c1;
    
    for (c = NOCARD; c == NOCARD;) {
	c = GetMouseButton();
	c1 = WhileMouseButton();
	if (c >= 0) {
	    c = ParsePos(c, -1, 0);
	    c1 = ParsePos(c1, c, 1);

	    if (c == c1 && (c >= 0 || c == FROMSTOCK)) {
		return c;
	    }
	    if (c >= 0 && c1 == TOFOUNDATION) {
		return TOHINT(FOUNDATION) | c;
	    }
	    if (c >= 0 && ISCOLSPEC(c1)) {
		return TOHINT(SPECTOCOL(c1)) | c;
	    }
	    
	    c = NOCARD;
	}
    }

    return c;
}


static void
RenderMousePointer(int x, int y)
{
    vga16_setpixel(KL_MOUSEFG, x    , y);
    vga16_setpixel(KL_MOUSEFG, x + 1, y);
    vga16_setpixel(KL_MOUSEFG, x + 2, y);
    vga16_setpixel(KL_MOUSEFG, x + 3, y);
    vga16_setpixel(KL_MOUSEFG, x + 4, y);
    vga16_setpixel(KL_MOUSEFG, x + 5, y);
    vga16_setpixel(KL_MOUSEFG, x    , y + 1);
    vga16_setpixel(KL_MOUSEDARKFG, x + 1, y + 1);
    vga16_setpixel(KL_MOUSEDARKFG, x + 2, y + 1);
    vga16_setpixel(KL_MOUSEFG, x + 3, y + 1);
    vga16_setpixel(KL_MOUSEFG, x    , y + 2);
    vga16_setpixel(KL_MOUSEDARKFG, x + 1, y + 2);
    vga16_setpixel(KL_MOUSEDARKFG, x + 2, y + 2);
    vga16_setpixel(KL_MOUSEFG, x + 3, y + 2);
    vga16_setpixel(KL_MOUSEFG, x + 4, y + 2);
    vga16_setpixel(KL_MOUSEFG, x    , y + 3);
    vga16_setpixel(KL_MOUSEFG, x + 1, y + 3);
    vga16_setpixel(KL_MOUSEFG, x + 2, y + 3);
    vga16_setpixel(KL_MOUSEDARKFG, x + 3, y + 3);
    vga16_setpixel(KL_MOUSEFG, x + 4, y + 3);
    vga16_setpixel(KL_MOUSEFG, x + 5, y + 3);
    vga16_setpixel(KL_MOUSEFG, x    , y + 4);
    vga16_setpixel(KL_MOUSEFG, x + 2, y + 4);
    vga16_setpixel(KL_MOUSEFG, x + 3, y + 4);
    vga16_setpixel(KL_MOUSEDARKFG, x + 4, y + 4);
    vga16_setpixel(KL_MOUSEFG, x + 5, y + 4);
    vga16_setpixel(KL_MOUSEFG, x + 6, y + 4);
    vga16_setpixel(KL_MOUSEFG, x    , y + 5);
    vga16_setpixel(KL_MOUSEFG, x + 3, y + 5);
    vga16_setpixel(KL_MOUSEFG, x + 4, y + 5);
    vga16_setpixel(KL_MOUSEDARKFG, x + 5, y + 5);
    vga16_setpixel(KL_MOUSEFG, x + 6, y + 5);
    vga16_setpixel(KL_MOUSEFG, x + 7, y + 5);
    vga16_setpixel(KL_MOUSEFG, x + 4, y + 6);
    vga16_setpixel(KL_MOUSEFG, x + 5, y + 6);
    vga16_setpixel(KL_MOUSEFG, x + 6, y + 6);
    vga16_setpixel(KL_MOUSEFG, x + 5, y + 7);
}

static void
RestoreUnderMousePointer(int x, int y, int *colour)
{
    vga16_setpixel(colour[0], x    , y);
    vga16_setpixel(colour[1], x + 1, y);
    vga16_setpixel(colour[2], x + 2, y);
    vga16_setpixel(colour[3], x + 3, y);
    vga16_setpixel(colour[4], x + 4, y);
    vga16_setpixel(colour[5], x + 5, y);
    vga16_setpixel(colour[6], x    , y + 1);
    vga16_setpixel(colour[7], x + 1, y + 1);
    vga16_setpixel(colour[8], x + 2, y + 1);
    vga16_setpixel(colour[9], x + 3, y + 1);
    vga16_setpixel(colour[10], x    , y + 2);
    vga16_setpixel(colour[11], x + 1, y + 2);
    vga16_setpixel(colour[12], x + 2, y + 2);
    vga16_setpixel(colour[13], x + 3, y + 2);
    vga16_setpixel(colour[14], x + 4, y + 2);
    vga16_setpixel(colour[15], x    , y + 3);
    vga16_setpixel(colour[16], x + 1, y + 3);
    vga16_setpixel(colour[17], x + 2, y + 3);
    vga16_setpixel(colour[18], x + 3, y + 3);
    vga16_setpixel(colour[19], x + 4, y + 3);
    vga16_setpixel(colour[20], x + 5, y + 3);
    vga16_setpixel(colour[21], x    , y + 4);
    vga16_setpixel(colour[22], x + 2, y + 4);
    vga16_setpixel(colour[23], x + 3, y + 4);
    vga16_setpixel(colour[24], x + 4, y + 4);
    vga16_setpixel(colour[25], x + 5, y + 4);
    vga16_setpixel(colour[26], x + 6, y + 4);
    vga16_setpixel(colour[27], x    , y + 5);
    vga16_setpixel(colour[28], x + 3, y + 5);
    vga16_setpixel(colour[29], x + 4, y + 5);
    vga16_setpixel(colour[30], x + 5, y + 5);
    vga16_setpixel(colour[31], x + 6, y + 5);
    vga16_setpixel(colour[32], x + 7, y + 5);
    vga16_setpixel(colour[33], x + 4, y + 6);
    vga16_setpixel(colour[34], x + 5, y + 6);
    vga16_setpixel(colour[35], x + 6, y + 6);
    vga16_setpixel(colour[36], x + 5, y + 7);
}

static int
SaveUnderMousePointer(int x, int y, int *colour)
{
    colour[0] = vga16_getpixel(x    , y);
    colour[1] = vga16_getpixel(x + 1, y);
    colour[2] = vga16_getpixel(x + 2, y);
    colour[3] = vga16_getpixel(x + 3, y);
    colour[4] = vga16_getpixel(x + 4, y);
    colour[5] = vga16_getpixel(x + 5, y);
    colour[6] = vga16_getpixel(x    , y + 1);
    colour[7] = vga16_getpixel(x + 1, y + 1);
    colour[8] = vga16_getpixel(x + 2, y + 1);
    colour[9] = vga16_getpixel(x + 3, y + 1);
    colour[10] = vga16_getpixel(x    , y + 2);
    colour[11] = vga16_getpixel(x + 1, y + 2);
    colour[12] = vga16_getpixel(x + 2, y + 2);
    colour[13] = vga16_getpixel(x + 3, y + 2);
    colour[14] = vga16_getpixel(x + 4, y + 2);
    colour[15] = vga16_getpixel(x    , y + 3);
    colour[16] = vga16_getpixel(x + 1, y + 3);
    colour[17] = vga16_getpixel(x + 2, y + 3);
    colour[18] = vga16_getpixel(x + 3, y + 3);
    colour[19] = vga16_getpixel(x + 4, y + 3);
    colour[20] = vga16_getpixel(x + 5, y + 3);
    colour[21] = vga16_getpixel(x    , y + 4);
    colour[22] = vga16_getpixel(x + 2, y + 4);
    colour[23] = vga16_getpixel(x + 3, y + 4);
    colour[24] = vga16_getpixel(x + 4, y + 4);
    colour[25] = vga16_getpixel(x + 5, y + 4);
    colour[26] = vga16_getpixel(x + 6, y + 4);
    colour[27] = vga16_getpixel(x    , y + 5);
    colour[28] = vga16_getpixel(x + 3, y + 5);
    colour[29] = vga16_getpixel(x + 4, y + 5);
    colour[30] = vga16_getpixel(x + 5, y + 5);
    colour[31] = vga16_getpixel(x + 6, y + 5);
    colour[32] = vga16_getpixel(x + 7, y + 5);
    colour[33] = vga16_getpixel(x + 4, y + 6);
    colour[34] = vga16_getpixel(x + 5, y + 6);
    colour[35] = vga16_getpixel(x + 6, y + 6);
    colour[36] = vga16_getpixel(x + 5, y + 7);
}


static int oldx = -1, oldy, oldcolour[40];

static long
GetMouseButton()
{
    int x, y, button;

    for (;;) {
	mouse_update();
	x = mouse_getx();
	y = mouse_gety();
	button = mouse_getbutton();

	if (x != oldx || y != oldy) {
	    if (oldx != -1) {
		RestoreUnderMousePointer(oldx, oldy, oldcolour);
	    }
	    SaveUnderMousePointer(x, y, oldcolour);
	    RenderMousePointer(x, y);
	    oldx = x;
	    oldy = y;
	}
	
	if (button & MOUSE_LEFTBUTTON) {
	    break;
	}
    }

    if (x >= KL_NEWGAMELEFT && x < KL_NEWGAMERIGHT
	&& y > KL_NEWGAMETOP && y <= KL_NEWGAMEBOTTOM) {
	return NEWGAME;
    }
    if (x >= KL_RESTARTLEFT && x < KL_RESTARTRIGHT
	&& y > KL_RESTARTTOP && y <= KL_RESTARTBOTTOM) {
	return RESTART;
    }
    if (x >= KL_QUITLEFT && x < KL_QUITRIGHT
	&& y > KL_QUITTOP && y <= KL_QUITBOTTOM) {
	return QUIT;
    }

    return (x << 9) | y;
}


static long
WhileMouseButton()
{
    int x, y, button;

    for (;;) {
	mouse_update();
	x = mouse_getx();
	y = mouse_gety();
	button = mouse_getbutton();

	if (x != oldx || y != oldy) {
	    if (oldx != -1) {
		RestoreUnderMousePointer(oldx, oldy, oldcolour);
	    }
	    SaveUnderMousePointer(x, y, oldcolour);
	    RenderMousePointer(x, y);
	    oldx = x;
	    oldy = y;
	}
	
	if (!(button & MOUSE_LEFTBUTTON)) {
	    break;
	}
    }

    if (oldx != -1) {
	RestoreUnderMousePointer(oldx, oldy, oldcolour);
    }
    oldx = -1;

    return (x << 9) | y;
}


static int
ParsePos(long pos, int downcard, int buttonup)
{
    int x, y, card, row, bittop;
    
    x = pos >> 9;
    y = pos & 0x1ff;
    if (x >= STOCKLEFT && x < STOCKLEFT + BITWIDTH
	&& y >= STOCKTOP && y < STOCKTOP + CARDIMAGEHEIGHT) {
	return FROMSTOCK;
    }
    if (x >= STOCKLEFT + COLWIDTH && x < STOCKLEFT + COLWIDTH + BITWIDTH
	&& y >= STOCKTOP && y < STOCKTOP + CARDIMAGEHEIGHT) {
	return pile;
    }
    if (buttonup
	&& x >= FOUNDLEFT && x < FOUNDLEFT + NUMSUITS * COLWIDTH
	&& y >= FOUNDTOP && y < FOUNDTOP + CARDIMAGEHEIGHT) {
	return TOFOUNDATION;
    }

    if (x / COLWIDTH > 6 || y < TOPROWHEIGHT ||
	(x % COLWIDTH) / BITWIDTH > 0) {
	return NOCARD;
    }

    card = column[x / COLWIDTH];
    bittop = columnTop[x / COLWIDTH];
    row = TOPROWHEIGHT;

    while (card != NOCARD) {
	if ((next[card] != NOCARD && y >= row && y < row + bittop)
	    || (next[card] == NOCARD && y >= row && y < row + BITHEIGHT)) {
	    if (!hidden[card] && (downcard == -1 || downcard == card)) {
		return card;
	    }
	}
	card = next[card];
	row += bittop;
    }
    if (buttonup) {
	return COLTOSPEC(x / COLWIDTH);
    }

    return NOCARD;
}
