/* $Id: expand.c,v 30000.25 1993/05/25 00:56:36 kkeys Exp $ */
/******************************************************************
 * Copyright 1993 by Ken Keys.
 * Permission is granted to obtain and redistribute this code freely.
 * All redistributions must contain this header.  You may modify this
 * software, but any redistributions of modified code must be clearly
 * marked as having been modified.
 ******************************************************************/


/********************************************************************
 * Fugue macro text expander                                        *
 *                                                                  *
 * Performs argument, variable, and macro body substitution.        *
 * Interpreter rewritten from scratch by Ken Keys to work in 1 pass *
 * instead of 3, while still supporting Greg's "$mac$" and "%arg"   *
 * syntax, with the addition of "{...}" delimiters, variable        *
 * subs, arithmetic evaluation, and got rid of those useless        *
 * "/.../" evaluations within a body.                               *
 ********************************************************************/

#include <ctype.h>
#include "port.h"
#include "dstring.h"
#include "tf.h"
#include "util.h"
#include "macro.h"
#include "command1.h"
#include "signal.h"
#include "socket.h"
#include "search.h"

#define MAX_ARGS 256

#define end_of_statement(p) ((p)[0] == '%' && ((p)[1] == ';' || (p)[1] == '\\'))
#define end_of_cmdsub(p) (cmdsub_count && *(p) == ')')

#define IF       '\200'
#define THEN     '\201'
#define ELSEIF   '\202'
#define ELSE     '\203'
#define ENDIF    '\204'
#define WHILE    '\205'
#define DO       '\206'
#define DONE     '\207'
#define BREAK    '\210'

#define EQUAL    '\300'
#define NOTEQ    '\301'
#define GTE      '\302'
#define LTE      '\303'
#define STREQ    '\304'
#define STRNEQ   '\305'
#define PATEQ    '\306'
#define PATNEQ   '\307'
#define STRCMP   '\310'

#define VAL_INT  1
#define VAL_STR  2


typedef struct Arg {
    Stringp text;
    int spaces;
} Arg;

typedef struct Value {
    int type;
    union {
        int ival;
        char *sval;
    } u;
} Value;

#define STACKSIZE 128

static Value *stack[STACKSIZE];
static int stacktop;

static int recur_count = 0;
static int cmdsub_count = 0;
static int user_result = 0;       /* user result */
static int argc;                  /* argument count */
static Arg **argv;                /* argument vector */
static char *ip;                  /* instruction pointer */
static int condition = 1;         /* checked by /if and /while */
static int eflag = 1;             /* flag: should we evaluate? */
static int block = 0;             /* type of current block */
static int breaking = 0;          /* flag: are we /break'ing? */
static List *localvar;            /* local variables */

void  NDECL(init_expand);
int   FDECL(process_macro,(char *body, char *args));
void  FDECL(check_command,(int keyboard, Stringp str));
int   FDECL(test,(char *args));
int   FDECL(do_let,(char *name, char *value));
char *FDECL(getlocalvar,(char *name));

static int    NDECL(control);
static int    FDECL(list,(Stringp dest));
static int    FDECL(statement,(Stringp dest));
static Value *FDECL(newint,(int i));
static Value *FDECL(newstr,(char *str, int len));
static void   FDECL(freeval,(Value *val));
static int    FDECL(valint,(Value *val));
static char  *FDECL(valstr,(Value *val));
static int    FDECL(slashsub,(Stringp dest));
static int    FDECL(macsub,(Stringp dest));
static int    FDECL(varsub,(Stringp dest));
static int    FDECL(backsub,(Stringp dest));
static int    FDECL(mathsub,(Stringp dest));
static int    FDECL(cmdsub,(Stringp dest));
static int    NDECL(expr);
static int    NDECL(orterm);
static int    NDECL(moreorterms);
static int    NDECL(andterm);
static int    NDECL(moreandterms);
static int    NDECL(boolterm);
static int    NDECL(moreboolterms);
static int    NDECL(term);
static int    NDECL(moreterms);
static int    NDECL(factor);
static int    NDECL(morefactors);
static int    FDECL(pushval,(Value *val));
static int    FDECL(reduce,(int op));
static Var   *FDECL(findlocalvar,(char *name));
static Var   *FDECL(findlevelvar,(char *name, List *level));


void init_expand()
{
    init_list(localvar = (List *)MALLOC(sizeof(List)));
}

void check_command(keyboard, str)
    int keyboard;
    Stringp str;
{
    if (str->s[0] == '/' && str->s[1] != '/') {
        newline_package(str, 0);
        user_result = handle_command(str->s);
    } else if (str->s[0] == '/') {
        newline_package(str, 1);
        user_result = transmit(str->s + 1, str->len - 1);
    } else if (!keyboard || !(do_hook(H_SEND, NULL, "%S", str) & F_GAG)) {
        newline_package(str, 1);
        user_result = transmit(str->s, str->len);
    }
}

int test(args)
    char *args;
{
    char *saved_ip = ip;
    int result = 0;

    ip = args;
    stacktop = 0;
    if (!expr()) {
        /* error message was already given */
    } else if (*ip) {
        tfprintf(tferr,"%% illegal character in arithmetic expression: %c",*ip);
    } else if (!stacktop) {
        tfputs("% internal error: arithmetic stack underflow", tferr);
    } else result = valint(stack[0]);
    while (stacktop) freeval(stack[--stacktop]);
    ip = saved_ip;
    return result;
}

char *getlocalvar(name)
    char *name;
{
    Var *var;

    return (var = findlocalvar(name)) ? var->value : NULL;
}

static Var *findlocalvar(name)
    char *name;
{
    ListEntry *lnode;
    Var *var = NULL;

    for (lnode = localvar->head; lnode && !var; lnode = lnode->next) {
        var = findlevelvar(name, (List *)lnode->data);
    }
    return var;
}

static Var *findlevelvar(name, level)
    char *name;
    List *level;
{
    ListEntry *vnode;
    Var *var = NULL;

    for (vnode = level->head; vnode && !var; vnode = vnode->next) {
        if (strcmp(name, ((Var *)(vnode->data))->name) == 0)
            var = (Var *)vnode->data;
    }
    return var;
}

int do_let(name, value)
    char *name, *value;
{
    Var *var;

    if (!localvar->head) {
        tfputs("% /let illegal at top level.", tferr);
        return 0;
    }
    if ((var = findlevelvar(name, (List *)localvar->head->data))) {
        FREE(var->value);
        var->value = STRDUP(value);
    } else {
        if (getvar(name)) {
            do_hook(H_SHADOW, "%% Warning:  Local variable \"%s\" overshadows global variable of same name.", "%s", name);
        }
        var = (Var *)MALLOC(sizeof(Var));
        var->name = STRDUP(name);
        var->value = STRDUP(value);
        var->node = inlist((GENERIC *)var, (List*)(localvar->head->data), NULL);
    }
    return 1;
}

int process_macro(body, args)
    char *body, *args;
{
    Stringp buffer;
    int vecsize = 20;
    char *in;
    int saved_cmdsub, saved_argc, saved_breaking;
    Arg **saved_argv;
    char *saved_ip;
    List *level;              /* list of local vars created on current level */
    Var *var;

    if (++recur_count > max_recur && max_recur) {
        tfputs("% Too many recursions.", tferr);
        recur_count--;
        return 0;
    }
    saved_cmdsub = cmdsub_count;
    saved_ip = ip;
    saved_argv = argv;
    saved_argc = argc;
    saved_breaking = breaking;

    ip = body;
    cmdsub_count = 0;
    init_list(level = (List *)MALLOC(sizeof(List)));
    inlist((GENERIC *)level, localvar, NULL);

    argc = 0;
    argv = (Arg **)MALLOC(vecsize * sizeof(Arg *));
    for (in = args; isspace(*in); in++);
    while (*in) {
        if (argc == vecsize)
            argv = (Arg**)REALLOC((char*)argv, sizeof(Arg*) * (vecsize += 10));
        argv[argc] = (Arg *) MALLOC(sizeof(Arg));
        Stringinit(argv[argc]->text);
        in = getword(argv[argc]->text, in);
        for (argv[argc]->spaces = 0; isspace(*in); in++) argv[argc]->spaces++;
        argc++;
    }

    Stringinit(buffer);
    if (!list(buffer)) user_result = 0;
    Stringfree(buffer);

    while (--argc >= 0) {
        Stringfree(argv[argc]->text);
        FREE(argv[argc]);
    }
    FREE(argv);

    level = (List *)localvar->head->data;
    while (level->head) {
        var = (Var *)unlist(level->head, level);
        FREE(var->name);
        FREE(var->value);
        FREE(var);
    }
    FREE(unlist((GENERIC *)localvar->head, localvar));

    cmdsub_count = saved_cmdsub;
    ip = saved_ip;
    argv = saved_argv;
    argc = saved_argc;
    breaking = saved_breaking;
    recur_count--;
    return user_result;
}

static int list(dest)
    Stringp dest;
{
    int slash, oldcondition, oldeflag, oldblock, failed = 0;
    char *start = NULL;

    while (isspace(*ip)) ip++;         /* skip leading space */
    if (block == WHILE) start = ip;

    if (!*ip) user_result = 1;         /* empty lists return 1 */

    while(*ip) {
        if (interrupted()) {
            tfputs("% macro evaluation interrupted.", tferr);
            return 0;
        }
        if ((slash = (ip[0] == '/' && ip[1] != '/'))) ip++;
        if (slash) {
            oldblock = block;
            block = control();
            switch(block) {
            case WHILE:
                oldeflag = eflag;
                oldcondition = condition;
                if (!list(dest)) failed = 1;
                else if (block == WHILE) {
                    tfputs("% error: missing /do", tferr);
                    failed = 1;
                } else if (block == DO) {
                    tfputs("% error: missing /done", tferr);
                    failed = 1;
                }
                eflag = oldeflag;
                condition = oldcondition;
                block = oldblock;
                if (failed) return 0;
                continue;
            case DO:
                if (oldblock != WHILE) {
                    tfputs("% error: unexpected /do", tferr);
                    return 0;
                }
                eflag = eflag && condition;
                condition = user_result;
                break;
            case BREAK:
                if (!breaking && eflag && condition) {
                    breaking = 1;
                }
                block = oldblock;
                break;
            case DONE:
                if (oldblock != DO) {
                    tfputs("% error: unexpected /done", tferr);
                    return 0;
                }
                if (breaking || !condition || !eflag) {
                    breaking = 0;
                    return 1;
                } else {
                    ip = start;
                    block = WHILE;
                    break;
                }
            case IF:
                oldeflag = eflag;
                oldcondition = condition;
                if (!list(dest)) failed = 1;
                else if (block == IF || block == ELSEIF) {
                    tfputs("% error: missing /then", tferr);
                    failed = 1;
                } else if (block == THEN || block == ELSE) {
                    tfputs("% error: missing /endif", tferr);
                    failed = 1;
                }
                eflag = oldeflag;
                condition = oldcondition;
                block = oldblock;
                if (failed) return 0;
                continue;
            case THEN:
                if (oldblock != IF && oldblock != ELSEIF) {
                    tfputs("% error: unexpected /then", tferr);
                    return 0;
                }
                eflag = eflag && condition;
                condition = user_result;
                break;
            case ELSEIF:
                if (oldblock != THEN) {
                    tfputs("% error: unexpected /elseif", tferr);
                    return 0;
                }
                condition = !condition;
                break;
            case ELSE:
                if (oldblock != THEN) {
                    tfputs("% error: unexpected /else", tferr);
                    return 0;
                }
                condition = !condition;
                break;
            case ENDIF:
                if (oldblock != THEN && oldblock != ELSE) {
                    tfputs("% error: unexpected /endif", tferr);
                    return 0;
                }
                return 1;
            default:
                /* not a control statement */
                break;
            }
            if (block) continue;
            block = oldblock;
        }

        if (!statement(dest)) return 0;

        /* I/O redirection will go here someday.  %>, %<, etc. */

        if (!breaking && eflag && condition && dest->len) {
            if (slash) {
                if (mecho) tfprintf(tferr, "%s/%S", getvar("mprefix"), dest);
                user_result = handle_command(dest->s);
            } else {
                if (mecho) tfprintf(tferr, "%s%S", getvar("mprefix"), dest);
                Stringadd(dest, '\n');
                user_result = transmit(dest->s, dest->len);
            }
        }

        Stringterm(dest, 0);
        if (end_of_cmdsub(ip)) break;
        while (isspace(*ip)) ip++;        /* skip leading space for next stat */
    }
    return 1;
}

static int control()
{
    char *end, save;
    int result = 0;

    for (end = ip; isalnum(*end) || *end == '_'; end++);
    save = *end;
    *end = '\0';
    if (cstrcmp(ip, "if") == 0)          result = IF;
    else if (cstrcmp(ip, "then") == 0)   result = THEN;
    else if (cstrcmp(ip, "elseif") == 0) result = ELSEIF;
    else if (cstrcmp(ip, "else") == 0)   result = ELSE;
    else if (cstrcmp(ip, "endif") == 0)  result = ENDIF;
    else if (cstrcmp(ip, "while") == 0)  result = WHILE;
    else if (cstrcmp(ip, "do") == 0)     result = DO;
    else if (cstrcmp(ip, "done") == 0)   result = DONE;
    else if (cstrcmp(ip, "break") == 0)  result = BREAK;
    else result = 0;
    *end = save;
    if (result) for (ip = end; isspace(*ip); ip++);
    return result;
}

static int statement(dest)
    Stringp dest;
{
    char *start;

    while (*ip) {
        if (*ip == '\\') {
            ++ip;
            if (!backsub(dest)) return 0;
        } else if (*ip == '/') {
            ++ip;
            if (!slashsub(dest)) return 0;
        } else if (end_of_statement(ip)) {
            ip += 2;
            break;
        } else if (end_of_cmdsub(ip)) {
            break;
        } else if (ip[0] == '$' && ip[1] == '[') {
            ip += 2;
            if (!mathsub(dest)) return 0;
        } else if (ip[0] == '$' && ip[1] == '(') {
            ip += 2;
            if (!cmdsub(dest)) return 0;
        } else if (*ip == '$') {
            ++ip;
            if (!macsub(dest)) return 0;
        } else if (*ip == '%') {
            ++ip;
            if (!varsub(dest)) return 0;
        } else {
            for (start = ip++; *ip && !strchr("\\/$%})", *ip); ip++);
            Stringncat(dest, start, ip - start);
        }
    }

    return 1;
}

static int slashsub(dest)
    Stringp dest;
{
    if (*ip == '/' && oldslash) while (*ip == '/') Stringadd(dest, *ip++);
    else Stringadd(dest, '/');
    return 1;
}

static Value *newint(i)
    int i;
{
    Value *val;

    val = (Value *)MALLOC(sizeof(Value));
    val->type = VAL_INT;
    val->u.ival = i;
    return val;
}

static Value *newstr(str, len)
    char *str;
    int len;
{
    Value *val;

    val = (Value *)MALLOC(sizeof(Value));
    val->type = VAL_STR;
    strncpy(val->u.sval = (char *)MALLOC(len + 1), str, len);
    val->u.sval[len] = '\0';
    return val;
}

static void freeval(val)
    Value *val;
{
    if (val->type == VAL_STR) FREE(val->u.sval);
    FREE(val);
}

/* return integer value of variable */
static int valint(val)
    Value *val;
{
    if (val->type == VAL_INT) return val->u.ival;
    return atoi(val->u.sval);
}

/* return string value of variable */
static char *valstr(val)
    Value *val;
{
    STATIC_BUFFER(buffer)

    if (val->type == VAL_STR) return val->u.sval;
    Sprintf(buffer, "%d", val->u.ival);
    return buffer->s;
}

static int pushval(val)
    Value *val;
{
    if (stacktop == STACKSIZE) {
        tfputs("% error: expression stack overflow", tferr);
        return 0;
    }
    stack[stacktop++] = val;
    return 1;
}

static int reduce(op)
    int op;
{
    Value *v1, *v2;
    int ival = 0;
    int result = 1;

    if (stacktop < 2) {
        tfputs("% internal error:  arithmetic stack underflow", tferr);
        return 0;
    }
    v2 = stack[--stacktop];
    v1 = stack[--stacktop];

    switch (op) {
        case '+':    ival = (valint(v1) + valint(v2));               break;
        case '-':    ival = (valint(v1) - valint(v2));               break;
        case '*':    ival = (valint(v1) * valint(v2));               break;
        case '/':    ival = (valint(v1) / valint(v2));               break;
        case '&':    ival = (valint(v1) ? valint(v2) : valint(v1));  break;
        case '|':    ival = (valint(v1) ? valint(v1) : valint(v2));  break;
        case '>':    ival = (valint(v1) > valint(v2));               break;
        case '<':    ival = (valint(v1) < valint(v2));               break;
        case EQUAL:  ival = (valint(v1) == valint(v2));              break;
        case NOTEQ:  ival = (valint(v1) != valint(v2));              break;
        case GTE:    ival = (valint(v1) >= valint(v2));              break;
        case LTE:    ival = (valint(v1) <= valint(v2));              break;
        case STRCMP: ival = (strcmp(valstr(v1), valstr(v2)));        break;
        case STREQ:  ival = (strcmp(valstr(v1), valstr(v2)) == 0);   break;
        case STRNEQ: ival = (strcmp(valstr(v1), valstr(v2)) != 0);   break;
        case PATEQ:
          ival = (smatch_check(valstr(v2)) && smatch(valstr(v2),valstr(v1))==0);
          break;
        case PATNEQ:
          ival = (smatch_check(valstr(v2)) && smatch(valstr(v2),valstr(v1))!=0);
          break;
        default:
          tfputs("% internal error: unknown operator in reduce()", tferr);
          result = 0;
          break;
    }
    if (result) pushval(newint(ival));

    freeval(v1);
    freeval(v2);
    return result;
}

static int expr()
{
    if (!orterm()) return 0;
    if (!moreorterms()) return 0;
    return 1;
}

static int moreorterms()
{
    while (1) {
        if (ip[0] == '|') {
            ip++;
            if (!orterm()) return 0;
            reduce('|');
        } else break;
    }
    return 1;
}

static int orterm()
{
    if (!andterm()) return 0;
    if (!moreandterms()) return 0;
    return 1;
}

static int moreandterms()
{
    while (1) {
        if (ip[0] == '&') {
            ip++;
            if (!andterm()) return 0;
            reduce('&');
        } else break;
    }
    return 1;
}

static int andterm()
{
    if (!boolterm()) return 0;
    if (!moreboolterms()) return 0;
    return 1;
}

static int moreboolterms()
{
    while (1) {
        if (ip[0] == '-' && ip[1] == '~') {
            ip += 2;
            if (!factor()) return 0;
            reduce(STRCMP);
        } else if (ip[0] == '=' && ip[1] == '~') {
            ip += 2;
            if (!factor()) return 0;
            reduce(STREQ);
        } else if (ip[0] == '!' && ip[1] == '~') {
            ip += 2;
            if (!factor()) return 0;
            reduce(STRNEQ);
        } else if (ip[0] == '=' && ip[1] == '/') {
            ip += 2;
            if (!factor()) return 0;
            reduce(PATEQ);
        } else if (ip[0] == '!' && ip[1] == '/') {
            ip += 2;
            if (!factor()) return 0;
            reduce(PATNEQ);
        } else if (ip[0] == '>' && ip[1] == '=') {
            ip += 2;
            if (!boolterm()) return 0;
            reduce(GTE);
        } else if (ip[0] == '<' && ip[1] == '=') {
            ip += 2;
            if (!boolterm()) return 0;
            reduce(LTE);
        } else if (ip[0] == '=' && ip[1] == '=') {
            ip += 2;
            if (!boolterm()) return 0;
            reduce(EQUAL);
        } else if (ip[0] == '!' && ip[1] == '=') {
            ip += 2;
            if (!boolterm()) return 0;
            reduce(NOTEQ);
        } else if (*ip == '<') {
            ++ip;
            if (!boolterm()) return 0;
            reduce('<');
        } else if (*ip == '>') {
            ++ip;
            if (!boolterm()) return 0;
            reduce('>');
        } else break;
    }
    return 1;
}

static int boolterm()
{
    if (!term()) return 0;
    if (!moreterms()) return 0;
    return 1;
}

static int moreterms()
{
    while (1) {
        if (*ip == '+') {
            ++ip;
            if (!term()) return 0;
            reduce('+');
        } else if (*ip == '-') {
            ++ip;
            if (!term()) return 0;
            reduce('-');
        } else break;
    }
    return 1;
}

static int term()
{
    if (!factor()) return 0;
    if (!morefactors()) return 0;
    return 1;
}

static int morefactors()
{
    while (1) {
        if (*ip == '*') {
            ++ip;
            if (!factor()) return 0;
            reduce('*');
        } else if (*ip == '/') {
            ++ip;
            if (!factor()) return 0;
            reduce('/');
        } else break;
    }
    return 1;
}

static int factor()
{
    Stringp buffer;
    int ival, sign = 1, truth = 2;
    char *end;

    while (*ip == '!' || isspace(*ip)) {
        if (*ip == '!') truth = !truth;
        ip++;
    }
    while (*ip == '+' || *ip == '-' || isspace(*ip)) {
        if (*ip == '-') sign = -sign;
        ip++;
    }
    if (*ip == '\"') {
        end = estrchr(++ip, '\"', '\\');
        if (!end) {
            tfputs("% error: unterminated string", tferr);
            return 0;
        }
        pushval(newstr(ip, end - ip));
        ip = end + 1;
    } else if (*ip == '(') {
        ++ip;
        if (!expr()) return 0;
        if (*ip != ')') {
            tfputs("% error: missing arithmetic operator", tferr);
            return 0;
        }
        if (stack[stacktop - 1]->type == VAL_INT) {
            ival = sign * stack[stacktop - 1]->u.ival;
            if (truth != 2) ival = truth ? !!ival : !ival;
            stack[stacktop - 1]->u.ival = ival;
        }
        ip++;
    } else if (isdigit(*ip)) {
        ival = atoi(ip) * sign;
        if (truth != 2) ival = truth ? !!ival : !ival; 
        if (!pushval(newint(ival))) return 0;
        while (isdigit(*++ip));
    } else if (*ip == '%') {
        Stringinit(buffer);
        ++ip;
        if (!varsub(buffer)) return 0;
        if (isdigit(*buffer->s)) {
            ival = atoi(buffer->s) * sign;
            if (truth != 2) ival = truth ? !!ival : !ival; 
            if (!pushval(newint(ival))) return 0;
        } else {
            if (!pushval(newstr(buffer->s, buffer->len))) return 0;
        }
        Stringfree(buffer);
    } else if (strchr("+-*/&|><=!]", *ip)) {
        tfputs("% error: missing arithmetic operand", tferr);
        return 0;
    } else {
        tfprintf(tferr,"%% illegal character in arithmetic expression: %c",*ip);
        return 0;
    }
    
    while (isspace(*ip)) ip++;
    return 1;
}

static int mathsub(dest)
    Stringp dest;
{
    int result = 0;

    stacktop = 0;
    while (isspace(*ip)) ip++;
    if (!expr()) {
        /* error message was already printed */
    } else if (end_of_statement(ip)) {
        tfputs("% error: unmatched $[", tferr);
    } else if (*ip == ')') {
        tfputs("% error: unmatched )", tferr);
    } else if (isdigit(*ip) || strchr("-+%(", *ip)) {
        tfputs("% error: missing arithmetic operator", tferr);
    } else if (*ip != ']') {
        tfprintf(tferr,"%% illegal character in arithmetic expression: %c",*ip);
    } else if (!stacktop) {
        tfputs("% internal error: arithmetic stack underflow", tferr);
    } else {
        if (stack[0]->type == VAL_INT) {
            Sprintf(dest, "\200%d", stack[0]->u.ival);
        } else {
            Sprintf(dest, "\200%s", stack[0]->u.sval);
        }
        ++ip;
        result = 1;
    }
    while (stacktop) freeval(stack[--stacktop]);
    return result;
}

static int cmdsub(dest)
    Stringp dest;
{
    TFILE *oldout, *olderr, *file;
    extern TFILE *tfout, *tferr;
    Stringp buffer;
    int result, first = 1;
    Aline *aline;

    file = tfopen(NULL, "q");
    cmdsub_count++;
    oldout = tfout;
    olderr = tferr;
    tfout = file;
    /* tferr = file; */

    Stringinit(buffer);
    result = list(buffer);
    Stringfree(buffer);

    tferr = olderr;
    tfout = oldout;
    cmdsub_count--;

    if (*ip != ')') {
        tfputs("% error: unmatched (", tferr);
        tfclose(file);
        return 0;
    }

    while ((aline = dequeue(file->u.queue))) {
        if (!((aline->attrs & F_GAG) && gag)) {
            if (!first) Stringadd(dest, ' ');
            first = 0;
            Stringcat(dest, aline->str);
        }
        free_aline(aline);
    }

    tfclose(file);
    ip++;
    return result;
}

static int macsub(dest)
    Stringp dest;
{
    Stringp buffer;
    char *body, *start;
    int bracket;

    if (*ip == '$') {
        while (*ip == '$') Stringadd(dest, *ip++);
        return 1;
    }

    Stringinit(buffer);
    if ((bracket = (*ip == '{'))) ip++;
    while (*ip) {
        if (*ip == '\\') {
            ++ip;
            if (!backsub(dest)) return 0;
        } else if (end_of_statement(ip) || end_of_cmdsub(ip)) {
            break;
        } else if (*ip == '/') {
            ++ip;
            if (!slashsub(buffer)) return 0;
        } else if (bracket && *ip == '}') {
            break;
        } else if (*ip == '$') {
            if (ip[1] == '$') {
                while(*++ip == '$') Stringadd(buffer, *ip);
            } else {
                if (!bracket) break;
                else Stringadd(buffer, *ip++);
            }
        } else if (*ip == '%') {
            ++ip;
            if (!varsub(buffer)) return 0;
        } else {
            for (start = ip++; *ip && !strchr("\\/$%})", *ip); ip++);
            Stringncat(buffer, start, ip - start);
        }
    }
    if (bracket) {
        if (*ip != '}') {
            tfputs("% error: unmatched ${", tferr);
            return 0;
        } else ip++;
    } else if (*ip == '$') {
        ip++;
    }

    if (breaking || !eflag || !condition) return 1;

    if ((body = macro_body(buffer->s))) Stringcat(dest, body);
    else tfprintf(tferr, "%% macro not defined: %S", buffer);
    if (mecho) tfprintf(tferr, "%s$%S --> %s", getvar("mprefix"), buffer, body);
    Stringfree(buffer);
    return 1;
}

static int backsub(dest)
    Stringp dest;
{
    if (isdigit(*ip)) {
        Stringadd(dest, strtochr(&ip));
    } else if (!backslash) {
        Stringadd(dest, '\\');
    } else if (*ip) {
        Stringadd(dest, *ip++);
    }
    return 1;
}

static int varsub(dest)
    Stringp dest;
{
    String *buffer;
    char *value, *start;
    int bracket, except, ell, i, n = -1, needloop = TRUE;
    int first, last, done = 0;
    static Stringp selector, defalt;
    static int buffers_initted = 0;

    if (!buffers_initted) {
        buffers_initted = 1;
        Stringinit(selector);
        Stringinit(defalt);
    }

    buffer = selector;
    if (*ip == '%') {
        while (*ip == '%') Stringadd(dest, *ip++);
        return 1;
    }
    if (!*ip || *ip == ' ') {
        Stringadd(dest, '%');
        return 1;
    }

    /* this is too hairy. */

    Stringterm(selector, 0);
    if ((bracket = (*ip == '{'))) ip++;
    if (*ip == '#') {
        Sprintf(dest, "\200%d", argc);
        done = 1;
        ip++;
    } else if (*ip == '?') {
        Sprintf(dest, "\200%d", user_result);
        done = 1;
        ip++;
    } else {
        if ((except = (ip[0] == '-' && (isdigit(ip[1]) || ucase(ip[1])=='L'))))
            ip++;
        if ((ell = (ucase(*ip) == 'L'))) {
            Stringadd(selector, *ip++);
            n = 1;
        }
        if (isdigit(*ip)) {
            n = atoi(ip);
            while (isdigit(*ip)) Stringadd(selector, *ip++);
            needloop = (*ip == '-');
        } else if (*ip == '*') {
            n = 0;
            needloop = (*++ip == '-');
        }
    }
    if (!done && needloop) while (*ip) {
        if (end_of_statement(ip) || end_of_cmdsub(ip)) {
            break;
        } else if (bracket && *ip == '}') {
            break;
        } else if (!bracket && *ip == ' ') {
            break;
        } else if (*ip == '/') {
            ++ip;
            if (!slashsub(buffer)) return 0;
        } else if (*ip == '-') {
            if (buffer == selector) buffer = Stringterm(defalt, 0);
            else Stringadd(buffer, *ip);
            ip++;
        } else if (buffer == selector && !isalnum(*ip) && *ip != '_') {
            break;
        } else {
            for (start = ip++; *ip && isalnum(*ip); ip++);
            Stringncat(buffer, start, ip - start);
        }
        if (buffer == selector) n = -1, ell = FALSE;
    }
    if (bracket) {
        if (*ip != '}') {
            tfputs("% error: unmatched %{", tferr);
            return 0;
        } else ip++;
    }

    if (done || breaking || !eflag || !condition) return 1;

    if (ell) {
        if (except) first = 0, last = argc - n - 1;
        else first = last = argc - n;
    } else if (n == 0) {
        first = 0, last = argc - 1;
    } else if (n > 0) {
        if (except) first = n, last = argc - 1;
        else first = last = n - 1;
    } else if (cstrcmp(selector->s, "R") == 0) {
        if (argc > 0) n = first = last = RANDOM() % argc;
        else if (buffer == defalt) SStringcat(dest, defalt);
    } else {
        Var *var;
        var = findlocalvar(selector->s);
        if (var) value = var->value;
        else value = getvar(selector->s);
        if (value && *value) {
            Stringcat(dest, value);
        } else if (buffer == defalt) {
            SStringcat(dest, defalt);
        }
    }

    if (n >= 0) {
        if (first > last || first < 0 || last >= argc) {
            if (buffer == defalt) SStringcat(dest, defalt);
        } else {
            for (i = first; i <= last; i++) {
                SStringcat(dest, argv[i]->text);
                if (i != last) Stringnadd(dest, ' ', argv[i]->spaces);
            }
        }
    }

    return 1;
}

