/* conf.c */

/* configuration adjustment. Some of the ideas and bits and pieces of the
 * code here are based on TinyMUSH 2.0.
 */

#include <stdio.h>

#include "config.h"
#include "interface.h"
#include "db.h"

extern object_flag_type find_flag();

void cf_str();
void cf_int();
void cf_flag();
void cf_bool();

OPTTAB options;

typedef struct confparm CONF;

struct confparm {
  char *name;			/* name of option */
  void (*handler)();		/* set option with this handler */
  int *loc;			/* place to put this option */
  int max;			/* max: string length, integer value */
};

CONF conftable[] = {
{(char *)"mud_name", cf_str, (int *)options.mud_name, 128},
{(char *)"port", cf_int, &options.port, 32000},
{(char *)"input_database", cf_str, (int *)options.input_db, 256},
{(char *)"output_database", cf_str, (int *)options.output_db, 256},
{(char *)"crash_database", cf_str, (int *)options.crash_db, 256},
{(char *)"mail_database", cf_str, (int *)options.mail_db, 256},
{(char *)"guest_player", cf_int, &options.guest_player, 100000},
{(char *)"player_start", cf_int, &options.player_start, 100000},
{(char *)"master_room", cf_int, &options.master_room, 100000},
{(char *)"idle_timeout", cf_int, &options.idle_timeout, 100000},
{(char *)"dump_interval", cf_int, &options.dump_interval, 100000},
{(char *)"max_logins", cf_int, &options.max_logins, 128},
{(char *)"paycheck", cf_int, &options.paycheck, 1000},
{(char *)"starting_money", cf_int, &options.starting_money, 10000},
{(char *)"player_queue_limit", cf_int, &options.player_queue_limit, 100000},
{(char *)"money_singular", cf_str, (int *)options.money_singular, 32},
{(char *)"money_plural", cf_str, (int *)options.money_plural, 32},
{(char *)"compress_program", cf_str, (int *)options.compress, 256},
{(char *)"uncompress_program", cf_str, (int *)options.uncompress, 256},
{(char *)"help_file", cf_str, (int *)options.help_file, 256},
{(char *)"help_index", cf_str, (int *)options.help_index, 256},
{(char *)"news_file", cf_str, (int *)options.news_file, 256},
{(char *)"news_index", cf_str, (int *)options.news_index, 256},
{(char *)"events_file", cf_str, (int *)options.events_file, 256},
{(char *)"events_index", cf_str, (int *)options.events_index, 256},
{(char *)"connect_file", cf_str, (int *)options.connect_file, 256},
{(char *)"motd_file", cf_str, (int *)options.motd_file, 256},
{(char *)"wizmotd_file", cf_str, (int *)options.wizmotd_file, 256},
{(char *)"newuser_file", cf_str, (int *)options.newuser_file, 256},
{(char *)"register_create_file", cf_str, (int *)options.register_file, 256},
{(char *)"quit_file", cf_str, (int *)options.quit_file, 256},
{(char *)"down_file", cf_str, (int *)options.down_file, 256},
{(char *)"full_file", cf_str, (int *)options.full_file, 256},
{(char *)"log_commands", cf_bool, &options.log_commands, 2},
{(char *)"log_huhs", cf_bool, &options.log_huhs, 2},
{(char *)"log_forces", cf_bool, &options.log_forces, 2},
{(char *)"log_walls", cf_bool, &options.log_walls, 2},
{(char *)"logins", cf_bool, &options.login_allow, 2},
{(char *)"daytime", cf_bool, &options.daytime, 2},
{(char *)"player_flags", cf_flag, &options.player_flags, 64},
{(char *)"room_flags", cf_flag, &options.room_flags, 64},
{(char *)"exit_flags", cf_flag, &options.exit_flags, 64},
{(char *)"thing_flags", cf_flag, &options.thing_flags, 64},
{(char *)"rwho_dump_interval", cf_int, &options.rwho_interval, 32000},
{(char *)"rwho_info_port", cf_int, &options.rwho_port, 32000},
{(char *)"rwho_host", cf_str, (int *)options.rwho_host, 64},
{(char *)"rwho_password", cf_str, (int *)options.rwho_pass, 64},
{ NULL, NULL, NULL, 0}
};

void cf_bool(opt, val, loc, maxval)
     char *opt;
     char *val;
     int *loc;
     int maxval;
{
  /* enter boolean parameter */

  if (!strcasecmp(val, "yes") || !strcasecmp(val, "true") ||
      !strcasecmp(val, "1"))
    *loc = 1;
  else if (!strcasecmp(val, "no") || !strcasecmp(val, "false") ||
	   !strcasecmp(val, "0"))
    *loc = 0;
  else {
    fprintf(stderr, "CONFIGURATION: option %s value %s invalid.\n", opt, val);
    fflush(stderr);
  }
}

void cf_str(opt, val, loc, maxlen)
     char *opt;
     char *val;
     int *loc;
     int maxlen;
{
  /* enter string parameter */

  /* truncate if necessary */
  if (strlen(val) >= maxlen) {
    val[maxlen - 1] = '\0';
    fprintf(stderr, "CONFIGURATION: option %s value truncated\n", opt);
    fflush(stderr);
  }

  strcpy((char *)loc, val);
}
    
void cf_int(opt, val, loc, maxval)
     char *opt;
     char *val;
     int *loc;
     int maxval;
{
  /* enter integer parameter */
  
  int n;

  n = atoi(val);

  /* enforce limits */
  if (n > maxval) {
    n = maxval;
    fprintf(stderr, "CONFIGURATION: option %s value limited to %d\n", 
	    opt, maxval);
    fflush(stderr);
  }

  *loc = n;
}

void cf_flag(opt, val, loc, maxval)
     char *opt;
     char *val;
     int *loc;
     int maxval;
{
  /* set default flags */

  int f = -1;
  int toggle;

  /* figure out what flag type we're setting */

  switch (opt[0]) {
  case 'p':
    f = find_flag(val, TYPE_PLAYER, &toggle, 1);
    break;
  case 'r':
    f = find_flag(val, TYPE_ROOM, &toggle, 1);
    break;
  case 'e':
    f = find_flag(val, TYPE_EXIT, &toggle, 1);
    break;
  case 't':
    f = find_flag(val, TYPE_THING, &toggle, 1);
    break;
  default:
    fprintf(stderr, "CONFIGURATION: weird flag set directive '%s'\n", opt);
  }

  if (f == -1) {
    fprintf(stderr, "CONFIGURATION: flag '%s' cannot be set.\n", val);
    return;
  }

  if (f == -2) {
    fprintf(stderr, "CONFIGURATION: flag '%s' for type not found.\n", val);
    return;
  }
  
  if (!toggle)
    *loc |= f;
  else {
    switch (opt[0]) {
    case 'p':
      options.player_toggles |= f;
      break;
    case 'r':
      options.room_toggles |= f;
      break;
    case 'e':
      options.exit_toggles |= f;
      break;
    case 't':
      options.thing_toggles |= f;
      break;
    }
  }
}

void config_set(opt, val)
     char *opt;
     char *val;
{
  CONF *cp;

  /* search conf table for the option; if found, add it, if not found,
   * complain about it.
   */

  for (cp = conftable; cp->name; cp++) {
    if (!strcmp(cp->name, opt)) {
      cp->handler(opt, val, cp->loc, cp->max);
      return;
    }
  }

  fprintf(stderr, "CONFIGURATION: directive '%s' not found.\n", opt);
  fflush(stderr);
}

void conf_default_set()
{
    strcpy(options.mud_name, "TinyMUSH");
    options.port = 4201;
    strcpy(options.input_db, "data/indb.Z");
    strcpy(options.output_db, "data/outdb.Z");
    strcpy(options.crash_db, "data/PANIC.db");
    strcpy(options.mail_db, "maildb.Z");
    options.guest_player = -1;
    options.player_start = 0;
    options.master_room = 2;
    options.idle_timeout = 10801;
    options.dump_interval = 3601;
    options.max_logins = 128;
    options.paycheck = 50;
    options.starting_money = 100;
    options.player_queue_limit = 100;
    strcpy(options.money_singular, "Penny");
    strcpy(options.money_plural, "Pennies");
    strcpy(options.compress, "compress");
    strcpy(options.uncompress, "uncompress");
    strcpy(options.help_file, "txt/help.txt");
    strcpy(options.help_index, "txt/help.indx");
    strcpy(options.news_file, "txt/news.txt");
    strcpy(options.news_index, "txt/news.indx");
    strcpy(options.events_file, "txt/events.txt");
    strcpy(options.events_index, "txt/events.indx");
    strcpy(options.connect_file, "txt/connect.txt");
    strcpy(options.motd_file, "txt/motd.txt");
    strcpy(options.wizmotd_file, "txt/wizmotd.txt");
    strcpy(options.newuser_file, "txt/newuser.txt");
    strcpy(options.register_file, "txt/create_reg.txt");
    strcpy(options.quit_file, "txt/quit.txt");
    strcpy(options.down_file, "txt/down.txt");
    strcpy(options.full_file, "txt/full.txt");
    options.log_commands = 0;
    options.log_huhs = 0;
    options.log_forces = 1;
    options.log_walls = 0;
    options.login_allow = 1;
    options.daytime = 0;
    options.player_flags = 0;
    options.room_flags = 0;
    options.exit_flags = 0;
    options.thing_flags = 0;
    options.player_toggles = 0;
    options.room_toggles = 0;
    options.exit_toggles = 0;
    options.thing_toggles = 0;
    options.rwho_interval = 241;
    options.rwho_port = 6889;
    strcpy(options.rwho_host, "riemann.math.okstate.edu");
    strcpy(options.rwho_pass, "getyours");
}

int config_file_startup(conf)
     char *conf;
{
  /* read a configuration file. Return 0 on failure, 1 on success */
  
  FILE *fp;
  char tbuf1[BUFFER_LEN];
  char *p, *q, *s;
  
  fp = fopen(conf, "r");
  if (fp == NULL) {
    fprintf(stderr, "ERROR: Cannot open configuration file %s.\n", conf);
    return 0;
  }

  conf_default_set();             /* initialize defaults */

  fgets(tbuf1, BUFFER_LEN, fp);
  while (!feof(fp)) {

    p = tbuf1;

    if (*p == '#') {
      /* comment line */
      fgets(tbuf1, BUFFER_LEN, fp);
      continue;
    }

    /* this is a real line. Strip the newline and characters following it.
     * Split the line into command and argument portions. If it exists,
     * also strip off the trailing comment.
     */

    for (p = tbuf1; *p && (*p != '\n'); p++)
      ;
    *p = '\0';			                    /* strip '\n' */
    for (p = tbuf1; *p && isspace(*p); p++)         /* strip spaces */
      ;
    for (q = p; *q && !isspace(*q); q++)            /* move over command */
      ;
    if (*q)
      *q++ = '\0';		                    /* split off command */
    for (; *q && isspace(*q); q++)                  /* skip spaces */
      ;
    for (s = q; *s && (*s != '#'); s++)	            /* look for comment */
      ;
    if (*s)			                    /* if found nuke it */
      *s = '\0';
    for (s = s - 1; (s >= q) && isspace(*s); s--)   /* smash trailing stuff */
      *s = '\0';

    if (strlen(p) != 0)		/* skip blank lines */
      config_set(p, q);

    fgets(tbuf1, BUFFER_LEN, fp);
  }

  /* these directives aren't player-settable but need to be initialized */
  options.dump_counter = options.dump_interval;
#ifdef RWHO_SEND
  options.rwho_counter = options.rwho_interval;
#endif

  fclose(fp);
  return 1;
}

