/* File: gredit.c   Main routine for graphics editor
 * 
 * Based on chredit.pas
 */

/*
 * Copyright (c) 1991-1997 Mike Marcelais
 *
 * This software may be copied and distributed for educational, research, and
 * not-for-profit purposes provided that this copyright and statement are
 * included in all such copies.
 */

#include <stdio.h>
#include <string.h>
#include <ctype.h>
#include <conio.h>
#include <stdlib.h>
#include <dos.h>

#ifdef __WATCOMC__
#include <graph.h>
#else
#include <grx20.h>

/* Graphics emulation libraries to simulate missing Watcom graphics
 * calls for DJGPP
 */

/*
 * Modified From DFLTFONT.C   in the grx20.h library.
 * Font data comes from PC8X8.FNT
 */

static char far fnt8x8bits[] = {
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x7e, 0x81, 0xa5, 0x81, 
   0xbd, 0x99, 0x81, 0x7e, 0x7e, 0xff, 0xdb, 0xff, 0xc3, 0xe7, 0xff, 0x7e, 
   0x6c, 0xfe, 0xfe, 0xfe, 0x7c, 0x38, 0x10, 0x00, 0x08, 0x1c, 0x3e, 0x7f, 
   0x3e, 0x1c, 0x08, 0x00, 0x1c, 0x1c, 0x1c, 0x7f, 0x7f, 0x6b, 0x08, 0x1c, 
   0x10, 0x10, 0x38, 0x7c, 0xfe, 0x7c, 0x10, 0x38, 0x00, 0x00, 0x18, 0x3c, 
   0x3c, 0x18, 0x00, 0x00, 0xff, 0xff, 0xe7, 0xc3, 0xc3, 0xe7, 0xff, 0xff, 
   0x00, 0x3c, 0x66, 0x42, 0x42, 0x66, 0x3c, 0x00, 0xff, 0xc3, 0x99, 0xbd, 
   0xbd, 0x99, 0xc3, 0xff, 0x0f, 0x07, 0x0f, 0x7d, 0xcc, 0xcc, 0xcc, 0x78, 
   0x3c, 0x66, 0x66, 0x66, 0x3c, 0x18, 0x7e, 0x18, 0x3f, 0x33, 0x3f, 0x30, 
   0x30, 0x70, 0xf0, 0xe0, 0x7f, 0x63, 0x7f, 0x63, 0x63, 0x67, 0xe6, 0xc0, 
   0x18, 0xdb, 0x3c, 0xe7, 0xe7, 0x3c, 0xdb, 0x18, 0x80, 0xe0, 0xf8, 0xfe, 
   0xf8, 0xe0, 0x80, 0x00, 0x02, 0x0e, 0x3e, 0xfe, 0x3e, 0x0e, 0x02, 0x00, 
   0x18, 0x3c, 0x7e, 0x18, 0x18, 0x7e, 0x3c, 0x18, 0x66, 0x66, 0x66, 0x66, 
   0x66, 0x00, 0x66, 0x00, 0x7f, 0xdb, 0xdb, 0x7b, 0x1b, 0x1b, 0x1b, 0x00, 
   0x3e, 0x63, 0x38, 0x6c, 0x6c, 0x38, 0xcc, 0x78, 0x00, 0x00, 0x00, 0x00, 
   0x7e, 0x7e, 0x7e, 0x00, 0x18, 0x3c, 0x7e, 0x18, 0x7e, 0x3c, 0x18, 0xff, 
   0x18, 0x3c, 0x7e, 0x18, 0x18, 0x18, 0x18, 0x00, 0x18, 0x18, 0x18, 0x18, 
   0x7e, 0x3c, 0x18, 0x00, 0x00, 0x18, 0x0c, 0xfe, 0x0c, 0x18, 0x00, 0x00, 
   0x00, 0x30, 0x60, 0xfe, 0x60, 0x30, 0x00, 0x00, 0x00, 0x00, 0xc0, 0xc0, 
   0xc0, 0xfe, 0x00, 0x00, 0x00, 0x24, 0x66, 0xff, 0x66, 0x24, 0x00, 0x00, 
   0x00, 0x18, 0x3c, 0x7e, 0xff, 0xff, 0x00, 0x00, 0x00, 0xff, 0xff, 0x7e, 
   0x3c, 0x18, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
   0x30, 0x78, 0x78, 0x30, 0x30, 0x00, 0x30, 0x00, 0x6c, 0x6c, 0x6c, 0x00, 
   0x00, 0x00, 0x00, 0x00, 0x6c, 0x6c, 0xfe, 0x6c, 0xfe, 0x6c, 0x6c, 0x00, 
   0x18, 0x3e, 0x60, 0x3c, 0x06, 0x7c, 0x18, 0x00, 0x00, 0x63, 0x66, 0x0c, 
   0x18, 0x33, 0x63, 0x00, 0x1c, 0x36, 0x1c, 0x3b, 0x6e, 0x66, 0x3b, 0x00, 
   0x30, 0x30, 0x60, 0x00, 0x00, 0x00, 0x00, 0x00, 0x0c, 0x18, 0x30, 0x30, 
   0x30, 0x18, 0x0c, 0x00, 0x30, 0x18, 0x0c, 0x0c, 0x0c, 0x18, 0x30, 0x00, 
   0x00, 0x66, 0x3c, 0xff, 0x3c, 0x66, 0x00, 0x00, 0x00, 0x30, 0x30, 0xfc, 
   0x30, 0x30, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x18, 0x18, 0x30, 
   0x00, 0x00, 0x00, 0x7e, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
   0x00, 0x18, 0x18, 0x00, 0x03, 0x06, 0x0c, 0x18, 0x30, 0x60, 0x40, 0x00, 
   0x3e, 0x63, 0x67, 0x6f, 0x7b, 0x73, 0x3e, 0x00, 0x18, 0x38, 0x58, 0x18, 
   0x18, 0x18, 0x7e, 0x00, 0x3c, 0x66, 0x06, 0x1c, 0x30, 0x66, 0x7e, 0x00, 
   0x3c, 0x66, 0x06, 0x1c, 0x06, 0x66, 0x3c, 0x00, 0x0e, 0x1e, 0x36, 0x66, 
   0x7f, 0x06, 0x0f, 0x00, 0x7e, 0x60, 0x7c, 0x06, 0x06, 0x66, 0x3c, 0x00, 
   0x1c, 0x30, 0x60, 0x7c, 0x66, 0x66, 0x3c, 0x00, 0x7e, 0x66, 0x06, 0x0c, 
   0x18, 0x18, 0x18, 0x00, 0x3c, 0x66, 0x66, 0x3c, 0x66, 0x66, 0x3c, 0x00, 
   0x3c, 0x66, 0x66, 0x3e, 0x06, 0x0c, 0x38, 0x00, 0x00, 0x18, 0x18, 0x00, 
   0x00, 0x18, 0x18, 0x00, 0x00, 0x18, 0x18, 0x00, 0x00, 0x18, 0x18, 0x30, 
   0x0c, 0x18, 0x30, 0x60, 0x30, 0x18, 0x0c, 0x00, 0x00, 0x00, 0x7e, 0x00, 
   0x00, 0x7e, 0x00, 0x00, 0x30, 0x18, 0x0c, 0x06, 0x0c, 0x18, 0x30, 0x00, 
   0x3c, 0x66, 0x06, 0x0c, 0x18, 0x00, 0x18, 0x00, 0x3e, 0x63, 0x6f, 0x69, 
   0x6f, 0x60, 0x3e, 0x00, 0x18, 0x3c, 0x66, 0x66, 0x7e, 0x66, 0x66, 0x00, 
   0x7e, 0x33, 0x33, 0x3e, 0x33, 0x33, 0x7e, 0x00, 0x1e, 0x33, 0x60, 0x60, 
   0x60, 0x33, 0x1e, 0x00, 0x7c, 0x36, 0x33, 0x33, 0x33, 0x36, 0x7c, 0x00, 
   0x7f, 0x31, 0x34, 0x3c, 0x34, 0x31, 0x7f, 0x00, 0x7f, 0x31, 0x34, 0x3c, 
   0x34, 0x30, 0x78, 0x00, 0x1e, 0x33, 0x60, 0x60, 0x67, 0x33, 0x1f, 0x00, 
   0x66, 0x66, 0x66, 0x7e, 0x66, 0x66, 0x66, 0x00, 0x3c, 0x18, 0x18, 0x18, 
   0x18, 0x18, 0x3c, 0x00, 0x0f, 0x06, 0x06, 0x06, 0x66, 0x66, 0x3c, 0x00, 
   0x73, 0x33, 0x36, 0x3c, 0x36, 0x33, 0x73, 0x00, 0x78, 0x30, 0x30, 0x30, 
   0x31, 0x33, 0x7f, 0x00, 0x63, 0x77, 0x7f, 0x7f, 0x6b, 0x63, 0x63, 0x00, 
   0x63, 0x73, 0x7b, 0x6f, 0x67, 0x63, 0x63, 0x00, 0x3e, 0x63, 0x63, 0x63, 
   0x63, 0x63, 0x3e, 0x00, 0x7e, 0x33, 0x33, 0x3e, 0x30, 0x30, 0x78, 0x00, 
   0x3c, 0x66, 0x66, 0x66, 0x6e, 0x3c, 0x0e, 0x00, 0x7e, 0x33, 0x33, 0x3e, 
   0x36, 0x33, 0x73, 0x00, 0x3c, 0x66, 0x30, 0x18, 0x0c, 0x66, 0x3c, 0x00, 
   0x7e, 0x5a, 0x18, 0x18, 0x18, 0x18, 0x3c, 0x00, 0x66, 0x66, 0x66, 0x66, 
   0x66, 0x66, 0x7e, 0x00, 0x66, 0x66, 0x66, 0x66, 0x66, 0x3c, 0x18, 0x00, 
   0x63, 0x63, 0x63, 0x6b, 0x7f, 0x77, 0x63, 0x00, 0x63, 0x63, 0x36, 0x1c, 
   0x1c, 0x36, 0x63, 0x00, 0x66, 0x66, 0x66, 0x3c, 0x18, 0x18, 0x3c, 0x00, 
   0x7f, 0x63, 0x46, 0x0c, 0x19, 0x33, 0x7f, 0x00, 0x3c, 0x30, 0x30, 0x30, 
   0x30, 0x30, 0x3c, 0x00, 0x60, 0x30, 0x18, 0x0c, 0x06, 0x03, 0x01, 0x00, 
   0x3c, 0x0c, 0x0c, 0x0c, 0x0c, 0x0c, 0x3c, 0x00, 0x08, 0x1c, 0x36, 0x63, 
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xff, 
   0x18, 0x18, 0x0c, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x3c, 0x06, 
   0x3e, 0x66, 0x3b, 0x00, 0x70, 0x30, 0x30, 0x3e, 0x33, 0x33, 0x6e, 0x00, 
   0x00, 0x00, 0x3c, 0x66, 0x60, 0x66, 0x3c, 0x00, 0x0e, 0x06, 0x06, 0x3e, 
   0x66, 0x66, 0x3b, 0x00, 0x00, 0x00, 0x3c, 0x66, 0x7e, 0x60, 0x3c, 0x00, 
   0x1c, 0x36, 0x30, 0x78, 0x30, 0x30, 0x78, 0x00, 0x00, 0x00, 0x3b, 0x66, 
   0x66, 0x3e, 0x06, 0x7c, 0x70, 0x30, 0x36, 0x3b, 0x33, 0x33, 0x73, 0x00, 
   0x18, 0x00, 0x38, 0x18, 0x18, 0x18, 0x3c, 0x00, 0x06, 0x00, 0x06, 0x06, 
   0x06, 0x66, 0x66, 0x3c, 0x70, 0x30, 0x33, 0x36, 0x3c, 0x36, 0x73, 0x00, 
   0x38, 0x18, 0x18, 0x18, 0x18, 0x18, 0x3c, 0x00, 0x00, 0x00, 0x66, 0x7f, 
   0x7f, 0x6b, 0x63, 0x00, 0x00, 0x00, 0x7c, 0x66, 0x66, 0x66, 0x66, 0x00, 
   0x00, 0x00, 0x3c, 0x66, 0x66, 0x66, 0x3c, 0x00, 0x00, 0x00, 0x6e, 0x33, 
   0x33, 0x3e, 0x30, 0x78, 0x00, 0x00, 0x3b, 0x66, 0x66, 0x3e, 0x06, 0x0f, 
   0x00, 0x00, 0x6e, 0x3b, 0x33, 0x30, 0x78, 0x00, 0x00, 0x00, 0x3e, 0x60, 
   0x3c, 0x06, 0x7c, 0x00, 0x08, 0x18, 0x3e, 0x18, 0x18, 0x1a, 0x0c, 0x00, 
   0x00, 0x00, 0x66, 0x66, 0x66, 0x66, 0x3b, 0x00, 0x00, 0x00, 0x66, 0x66, 
   0x66, 0x3c, 0x18, 0x00, 0x00, 0x00, 0x63, 0x6b, 0x7f, 0x7f, 0x36, 0x00, 
   0x00, 0x00, 0x63, 0x36, 0x1c, 0x36, 0x63, 0x00, 0x00, 0x00, 0x66, 0x66, 
   0x66, 0x3e, 0x06, 0x7c, 0x00, 0x00, 0x7e, 0x4c, 0x18, 0x32, 0x7e, 0x00, 
   0x0e, 0x18, 0x18, 0x70, 0x18, 0x18, 0x0e, 0x00, 0x0c, 0x0c, 0x0c, 0x00, 
   0x0c, 0x0c, 0x0c, 0x00, 0x70, 0x18, 0x18, 0x0e, 0x18, 0x18, 0x70, 0x00, 
   0x3b, 0x6e, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x08, 0x1c, 0x36, 
   0x63, 0x63, 0x7f, 0x00, 0x3c, 0x66, 0x60, 0x66, 0x3c, 0x0c, 0x06, 0x3c, 
   0x00, 0x66, 0x00, 0x66, 0x66, 0x66, 0x3f, 0x00, 0x1c, 0x00, 0x78, 0xcc, 
   0xfc, 0xc0, 0x78, 0x00, 0x7e, 0xc3, 0x3c, 0x06, 0x3e, 0x66, 0x3f, 0x00, 
   0x66, 0x00, 0x3c, 0x06, 0x3e, 0x66, 0x3f, 0x00, 0x70, 0x00, 0x3c, 0x06, 
   0x3e, 0x66, 0x3f, 0x00, 0x18, 0x18, 0x3c, 0x06, 0x3e, 0x66, 0x3f, 0x00, 
   0x00, 0x00, 0x3c, 0x60, 0x60, 0x3c, 0x06, 0x1c, 0x7e, 0xc3, 0x3c, 0x66, 
   0x7e, 0x60, 0x3c, 0x00, 0xcc, 0x00, 0x78, 0xcc, 0xfc, 0xc0, 0x78, 0x00, 
   0x70, 0x00, 0x3c, 0x66, 0x7e, 0x60, 0x3c, 0x00, 0x66, 0x00, 0x38, 0x18, 
   0x18, 0x18, 0x3c, 0x00, 0x3e, 0x63, 0x1c, 0x0c, 0x0c, 0x0c, 0x1e, 0x00, 
   0x70, 0x00, 0x38, 0x18, 0x18, 0x18, 0x3c, 0x00, 0x63, 0x1c, 0x36, 0x63, 
   0x7f, 0x63, 0x63, 0x00, 0x18, 0x18, 0x00, 0x3c, 0x66, 0x7e, 0x66, 0x00, 
   0x1c, 0x00, 0xfc, 0x60, 0x78, 0x60, 0xfc, 0x00, 0x00, 0x00, 0x7f, 0x0c, 
   0x7f, 0xcc, 0x7f, 0x00, 0x1f, 0x36, 0x66, 0x7f, 0x66, 0x66, 0x67, 0x00, 
   0x3c, 0x66, 0x00, 0x3c, 0x66, 0x66, 0x3c, 0x00, 0x00, 0x66, 0x00, 0x3c, 
   0x66, 0x66, 0x3c, 0x00, 0x00, 0x70, 0x00, 0x3c, 0x66, 0x66, 0x3c, 0x00, 
   0x3c, 0x66, 0x00, 0x66, 0x66, 0x66, 0x3f, 0x00, 0x00, 0x70, 0x00, 0x66, 
   0x66, 0x66, 0x3f, 0x00, 0x00, 0xcc, 0x00, 0xcc, 0xcc, 0x7c, 0x0c, 0xf8, 
   0xc3, 0x18, 0x3c, 0x66, 0x66, 0x3c, 0x18, 0x00, 0x66, 0x00, 0x66, 0x66, 
   0x66, 0x66, 0x3c, 0x00, 0x0c, 0x0c, 0x3f, 0x60, 0x60, 0x3f, 0x0c, 0x0c, 
   0x1c, 0x36, 0x32, 0x78, 0x30, 0x73, 0x7e, 0x00, 0x66, 0x66, 0x3c, 0x7e, 
   0x18, 0x7e, 0x18, 0x18, 0xf8, 0xcc, 0xcc, 0xfa, 0xc6, 0xcf, 0xc6, 0xc7, 
   0x0e, 0x1b, 0x18, 0x3c, 0x18, 0x18, 0xd8, 0x70, 0x0e, 0x00, 0x3c, 0x06, 
   0x3e, 0x66, 0x3f, 0x00, 0x1c, 0x00, 0x38, 0x18, 0x18, 0x18, 0x3c, 0x00, 
   0x00, 0x0e, 0x00, 0x3c, 0x66, 0x66, 0x3c, 0x00, 0x00, 0x0e, 0x00, 0x66, 
   0x66, 0x66, 0x3f, 0x00, 0x00, 0x7c, 0x00, 0x7c, 0x66, 0x66, 0x66, 0x00, 
   0x7e, 0x00, 0x66, 0x76, 0x7e, 0x6e, 0x66, 0x00, 0x1e, 0x36, 0x36, 0x1f, 
   0x00, 0x3f, 0x00, 0x00, 0x1c, 0x36, 0x36, 0x1c, 0x00, 0x3e, 0x00, 0x00, 
   0x18, 0x00, 0x18, 0x30, 0x60, 0x66, 0x3c, 0x00, 0x00, 0x00, 0x00, 0x7e, 
   0x60, 0x60, 0x00, 0x00, 0x00, 0x00, 0x00, 0xfc, 0x0c, 0x0c, 0x00, 0x00, 
   0xc3, 0xc6, 0xcc, 0xde, 0x33, 0x66, 0xcc, 0x0f, 0xc3, 0xc6, 0xcc, 0xdb, 
   0x37, 0x6f, 0xcf, 0x03, 0x18, 0x18, 0x00, 0x18, 0x18, 0x18, 0x18, 0x00, 
   0x00, 0x33, 0x66, 0xcc, 0x66, 0x33, 0x00, 0x00, 0x00, 0xcc, 0x66, 0x33, 
   0x66, 0xcc, 0x00, 0x00, 0x22, 0x88, 0x22, 0x88, 0x22, 0x88, 0x22, 0x88, 
   0x55, 0xaa, 0x55, 0xaa, 0x55, 0xaa, 0x55, 0xaa, 0xdb, 0x77, 0xdb, 0xee, 
   0xdb, 0x77, 0xdb, 0xee, 0x18, 0x18, 0x18, 0x18, 0x18, 0x18, 0x18, 0x18, 
   0x18, 0x18, 0x18, 0x18, 0xf8, 0x18, 0x18, 0x18, 0x18, 0x18, 0xf8, 0x18, 
   0xf8, 0x18, 0x18, 0x18, 0x36, 0x36, 0x36, 0x36, 0xf6, 0x36, 0x36, 0x36, 
   0x00, 0x00, 0x00, 0x00, 0xfe, 0x36, 0x36, 0x36, 0x00, 0x00, 0xf8, 0x18, 
   0xf8, 0x18, 0x18, 0x18, 0x36, 0x36, 0xf6, 0x06, 0xf6, 0x36, 0x36, 0x36, 
   0x36, 0x36, 0x36, 0x36, 0x36, 0x36, 0x36, 0x36, 0x00, 0x00, 0xfe, 0x06, 
   0xf6, 0x36, 0x36, 0x36, 0x36, 0x36, 0xf6, 0x06, 0xfe, 0x00, 0x00, 0x00, 
   0x36, 0x36, 0x36, 0x36, 0xfe, 0x00, 0x00, 0x00, 0x18, 0x18, 0xf8, 0x18, 
   0xf8, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xf8, 0x18, 0x18, 0x18, 
   0x18, 0x18, 0x18, 0x18, 0x1f, 0x00, 0x00, 0x00, 0x18, 0x18, 0x18, 0x18, 
   0xff, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xff, 0x18, 0x18, 0x18, 
   0x18, 0x18, 0x18, 0x18, 0x1f, 0x18, 0x18, 0x18, 0x00, 0x00, 0x00, 0x00, 
   0xff, 0x00, 0x00, 0x00, 0x18, 0x18, 0x18, 0x18, 0xff, 0x18, 0x18, 0x18, 
   0x18, 0x18, 0x1f, 0x18, 0x1f, 0x18, 0x18, 0x18, 0x36, 0x36, 0x36, 0x36, 
   0x37, 0x36, 0x36, 0x36, 0x36, 0x36, 0x37, 0x30, 0x3f, 0x00, 0x00, 0x00, 
   0x00, 0x00, 0x3f, 0x30, 0x37, 0x36, 0x36, 0x36, 0x36, 0x36, 0xf7, 0x00, 
   0xff, 0x00, 0x00, 0x00, 0x00, 0x00, 0xff, 0x00, 0xf7, 0x36, 0x36, 0x36, 
   0x36, 0x36, 0x37, 0x30, 0x37, 0x36, 0x36, 0x36, 0x00, 0x00, 0xff, 0x00, 
   0xff, 0x00, 0x00, 0x00, 0x36, 0x36, 0xf7, 0x00, 0xf7, 0x36, 0x36, 0x36, 
   0x18, 0x18, 0xff, 0x00, 0xff, 0x00, 0x00, 0x00, 0x36, 0x36, 0x36, 0x36, 
   0xff, 0x00, 0x00, 0x00, 0x00, 0x00, 0xff, 0x00, 0xff, 0x18, 0x18, 0x18, 
   0x00, 0x00, 0x00, 0x00, 0xff, 0x36, 0x36, 0x36, 0x36, 0x36, 0x36, 0x36, 
   0x3f, 0x00, 0x00, 0x00, 0x18, 0x18, 0x1f, 0x18, 0x1f, 0x00, 0x00, 0x00, 
   0x00, 0x00, 0x1f, 0x18, 0x1f, 0x18, 0x18, 0x18, 0x00, 0x00, 0x00, 0x00, 
   0x3f, 0x36, 0x36, 0x36, 0x36, 0x36, 0x36, 0x36, 0xff, 0x36, 0x36, 0x36, 
   0x18, 0x18, 0xff, 0x18, 0xff, 0x18, 0x18, 0x18, 0x18, 0x18, 0x18, 0x18, 
   0xf8, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x1f, 0x18, 0x18, 0x18, 
   0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x00, 0x00, 0x00, 0x00, 
   0xff, 0xff, 0xff, 0xff, 0xf0, 0xf0, 0xf0, 0xf0, 0xf0, 0xf0, 0xf0, 0xf0, 
   0x0f, 0x0f, 0x0f, 0x0f, 0x0f, 0x0f, 0x0f, 0x0f, 0xff, 0xff, 0xff, 0xff, 
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x3b, 0x6e, 0x64, 0x6e, 0x3b, 0x00, 
   0x00, 0x3c, 0x66, 0x7c, 0x66, 0x7c, 0x60, 0x60, 0x00, 0x7e, 0x66, 0x60, 
   0x60, 0x60, 0x60, 0x00, 0x00, 0x7f, 0x36, 0x36, 0x36, 0x36, 0x36, 0x00, 
   0x7e, 0x66, 0x30, 0x18, 0x30, 0x66, 0x7e, 0x00, 0x00, 0x00, 0x3f, 0x6c, 
   0x6c, 0x6c, 0x38, 0x00, 0x00, 0x33, 0x33, 0x33, 0x33, 0x3e, 0x30, 0x60, 
   0x00, 0x3b, 0x6e, 0x0c, 0x0c, 0x0c, 0x0c, 0x00, 0x7e, 0x18, 0x3c, 0x66, 
   0x66, 0x3c, 0x18, 0x7e, 0x1c, 0x36, 0x63, 0x7f, 0x63, 0x36, 0x1c, 0x00, 
   0x1c, 0x36, 0x63, 0x63, 0x36, 0x36, 0x77, 0x00, 0x0e, 0x18, 0x0c, 0x3e, 
   0x66, 0x66, 0x3c, 0x00, 0x00, 0x00, 0x7e, 0xdb, 0xdb, 0x7e, 0x00, 0x00, 
   0x06, 0x0c, 0x7e, 0xdb, 0xdb, 0x7e, 0x60, 0xc0, 0x1c, 0x30, 0x60, 0x7c, 
   0x60, 0x30, 0x1c, 0x00, 0x3c, 0x66, 0x66, 0x66, 0x66, 0x66, 0x66, 0x00, 
   0x00, 0x7e, 0x00, 0x7e, 0x00, 0x7e, 0x00, 0x00, 0x18, 0x18, 0x7e, 0x18, 
   0x18, 0x00, 0x7e, 0x00, 0x30, 0x18, 0x0c, 0x18, 0x30, 0x00, 0x7e, 0x00, 
   0x0c, 0x18, 0x30, 0x18, 0x0c, 0x00, 0x7e, 0x00, 0x0e, 0x1b, 0x1b, 0x18, 
   0x18, 0x18, 0x18, 0x18, 0x18, 0x18, 0x18, 0x18, 0x18, 0xd8, 0xd8, 0x70, 
   0x18, 0x18, 0x00, 0x7e, 0x00, 0x18, 0x18, 0x00, 0x00, 0x3b, 0x6e, 0x00, 
   0x3b, 0x6e, 0x00, 0x00, 0x1c, 0x36, 0x36, 0x1c, 0x00, 0x00, 0x00, 0x00, 
   0x00, 0x00, 0x00, 0x0c, 0x0c, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
   0x18, 0x00, 0x00, 0x00, 0x0f, 0x0c, 0x0c, 0x0c, 0xec, 0x6c, 0x3c, 0x1c, 
   0x78, 0x6c, 0x6c, 0x6c, 0x6c, 0x00, 0x00, 0x00, 0x70, 0x18, 0x30, 0x60, 
   0x78, 0x00, 0x00, 0x00, 0x00, 0x00, 0x3c, 0x3c, 0x3c, 0x3c, 0x00, 0x00, 
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
};

struct {
    GrFont	  theFont;
    GrFontChrInfo rest[255];
} Std8x8Font = {
    {
	{			/* font header */
	    "pc8x8",		/* font name */
	    "ibmpc",		/* font family name */
	    0,			/* characters have varying width */
	    0,			/* derived from a scalable font */
	    1,			/* font permanently linked into program */
	    GR_FONTCVT_NONE,	/* "tweaked" font (resized, etc..) */
	    8,			/* width (average when proportional) */
	    8,			/* font height */
	    6,			/* baseline pixel pos (from top) */
	    7,			/* underline pixel pos (from top) */
	    1,			/* underline width */
	    0,			/* lowest character code in font */
	    256			/* number of characters in font */
	},
	fnt8x8bits,		/* character bitmap array */
	(void *)0,		/* auxilary bitmap */
	8,			/* width of narrowest character */
	8,			/* width of widest character */
	0,			/* allocated size of auxilary bitmap */
	0,			/* free space in auxilary bitmap */
	{ (void *)0	   },	/* converted character bitmap offsets */
	{ { 8,   8*0	 } }	/* first character info */
    },
    {
	{ 8, 8*  1 }, { 8, 8*  2 }, { 8, 8*  3 }, { 8, 8*  4 }, { 8, 8*  5 },
	{ 8, 8*  6 }, { 8, 8*  7 }, { 8, 8*  8 }, { 8, 8*  9 }, { 8, 8* 10 },
	{ 8, 8* 11 }, { 8, 8* 12 }, { 8, 8* 13 }, { 8, 8* 14 }, { 8, 8* 15 },
	{ 8, 8* 16 }, { 8, 8* 17 }, { 8, 8* 18 }, { 8, 8* 19 }, { 8, 8* 20 },
	{ 8, 8* 21 }, { 8, 8* 22 }, { 8, 8* 23 }, { 8, 8* 24 }, { 8, 8* 25 },
	{ 8, 8* 26 }, { 8, 8* 27 }, { 8, 8* 28 }, { 8, 8* 29 }, { 8, 8* 30 },
	{ 8, 8* 31 }, { 8, 8* 32 }, { 8, 8* 33 }, { 8, 8* 34 }, { 8, 8* 35 },
	{ 8, 8* 36 }, { 8, 8* 37 }, { 8, 8* 38 }, { 8, 8* 39 }, { 8, 8* 40 },
	{ 8, 8* 41 }, { 8, 8* 42 }, { 8, 8* 43 }, { 8, 8* 44 }, { 8, 8* 45 },
	{ 8, 8* 46 }, { 8, 8* 47 }, { 8, 8* 48 }, { 8, 8* 49 }, { 8, 8* 50 },
	{ 8, 8* 51 }, { 8, 8* 52 }, { 8, 8* 53 }, { 8, 8* 54 }, { 8, 8* 55 },
	{ 8, 8* 56 }, { 8, 8* 57 }, { 8, 8* 58 }, { 8, 8* 59 }, { 8, 8* 60 },
	{ 8, 8* 61 }, { 8, 8* 62 }, { 8, 8* 63 }, { 8, 8* 64 }, { 8, 8* 65 },
	{ 8, 8* 66 }, { 8, 8* 67 }, { 8, 8* 68 }, { 8, 8* 69 }, { 8, 8* 70 },
	{ 8, 8* 71 }, { 8, 8* 72 }, { 8, 8* 73 }, { 8, 8* 74 }, { 8, 8* 75 },
	{ 8, 8* 76 }, { 8, 8* 77 }, { 8, 8* 78 }, { 8, 8* 79 }, { 8, 8* 80 },
	{ 8, 8* 81 }, { 8, 8* 82 }, { 8, 8* 83 }, { 8, 8* 84 }, { 8, 8* 85 },
	{ 8, 8* 86 }, { 8, 8* 87 }, { 8, 8* 88 }, { 8, 8* 89 }, { 8, 8* 90 },
	{ 8, 8* 91 }, { 8, 8* 92 }, { 8, 8* 93 }, { 8, 8* 94 }, { 8, 8* 95 },
	{ 8, 8* 96 }, { 8, 8* 97 }, { 8, 8* 98 }, { 8, 8* 99 }, { 8, 8*100 },
	{ 8, 8*101 }, { 8, 8*102 }, { 8, 8*103 }, { 8, 8*104 }, { 8, 8*105 },
	{ 8, 8*106 }, { 8, 8*107 }, { 8, 8*108 }, { 8, 8*109 }, { 8, 8*110 },
	{ 8, 8*111 }, { 8, 8*112 }, { 8, 8*113 }, { 8, 8*114 }, { 8, 8*115 },
	{ 8, 8*116 }, { 8, 8*117 }, { 8, 8*118 }, { 8, 8*119 }, { 8, 8*120 },
	{ 8, 8*121 }, { 8, 8*122 }, { 8, 8*123 }, { 8, 8*124 }, { 8, 8*125 },
	{ 8, 8*126 }, { 8, 8*127 }, { 8, 8*128 }, { 8, 8*129 }, { 8, 8*130 },
	{ 8, 8*131 }, { 8, 8*132 }, { 8, 8*133 }, { 8, 8*134 }, { 8, 8*135 },
	{ 8, 8*136 }, { 8, 8*137 }, { 8, 8*138 }, { 8, 8*139 }, { 8, 8*140 },
	{ 8, 8*141 }, { 8, 8*142 }, { 8, 8*143 }, { 8, 8*144 }, { 8, 8*145 },
	{ 8, 8*146 }, { 8, 8*147 }, { 8, 8*148 }, { 8, 8*149 }, { 8, 8*150 },
	{ 8, 8*151 }, { 8, 8*152 }, { 8, 8*153 }, { 8, 8*154 }, { 8, 8*155 },
	{ 8, 8*156 }, { 8, 8*157 }, { 8, 8*158 }, { 8, 8*159 }, { 8, 8*160 },
	{ 8, 8*161 }, { 8, 8*162 }, { 8, 8*163 }, { 8, 8*164 }, { 8, 8*165 },
	{ 8, 8*166 }, { 8, 8*167 }, { 8, 8*168 }, { 8, 8*169 }, { 8, 8*170 },
	{ 8, 8*171 }, { 8, 8*172 }, { 8, 8*173 }, { 8, 8*174 }, { 8, 8*175 },
	{ 8, 8*176 }, { 8, 8*177 }, { 8, 8*178 }, { 8, 8*179 }, { 8, 8*180 },
	{ 8, 8*181 }, { 8, 8*182 }, { 8, 8*183 }, { 8, 8*184 }, { 8, 8*185 },
	{ 8, 8*186 }, { 8, 8*187 }, { 8, 8*188 }, { 8, 8*189 }, { 8, 8*190 },
	{ 8, 8*191 }, { 8, 8*192 }, { 8, 8*193 }, { 8, 8*194 }, { 8, 8*195 },
	{ 8, 8*196 }, { 8, 8*197 }, { 8, 8*198 }, { 8, 8*199 }, { 8, 8*200 },
	{ 8, 8*201 }, { 8, 8*202 }, { 8, 8*203 }, { 8, 8*204 }, { 8, 8*205 },
	{ 8, 8*206 }, { 8, 8*207 }, { 8, 8*208 }, { 8, 8*209 }, { 8, 8*210 },
	{ 8, 8*211 }, { 8, 8*212 }, { 8, 8*213 }, { 8, 8*214 }, { 8, 8*215 },
	{ 8, 8*216 }, { 8, 8*217 }, { 8, 8*218 }, { 8, 8*219 }, { 8, 8*220 },
	{ 8, 8*221 }, { 8, 8*222 }, { 8, 8*223 }, { 8, 8*224 }, { 8, 8*225 },
	{ 8, 8*226 }, { 8, 8*227 }, { 8, 8*228 }, { 8, 8*229 }, { 8, 8*230 },
	{ 8, 8*231 }, { 8, 8*232 }, { 8, 8*233 }, { 8, 8*234 }, { 8, 8*235 },
	{ 8, 8*236 }, { 8, 8*237 }, { 8, 8*238 }, { 8, 8*239 }, { 8, 8*240 },
	{ 8, 8*241 }, { 8, 8*242 }, { 8, 8*243 }, { 8, 8*244 }, { 8, 8*245 },
	{ 8, 8*246 }, { 8, 8*247 }, { 8, 8*248 }, { 8, 8*249 }, { 8, 8*250 },
	{ 8, 8*251 }, { 8, 8*252 }, { 8, 8*253 }, { 8, 8*254 }, { 8, 8*255 },
    }  
};

GrTextOption stdText = { &Std8x8Font.theFont, {0}, {0}, 0, 0, 0, 0 };

/* Symbolic Constants used */
#define _DEFAULTMODE   0  /* Should be a text mode */
#define _VRES16COLOR   1  /* Should be 640x480x16 */

/* Unused */
#define _GCLEARSCREEN  0

/* how to fill */
#define _GBORDER       0  /* Just draw the border */
#define _GFILLINTERIOR 1  /* Fill in the interior */

short __grcolor = 1;
short __grlocx = 0;
short __grlocy = 0;

inline void _setcolor(short x)      { __grcolor = x; };
inline void _moveto(short x,short y){ __grlocx = x; __grlocy = y; };

#define _setpixel(x,y) GrPlot(x,y,__grcolor)

void _outgtext(const char *s)
{
   
   stdText.txo_fgcolor.v = __grcolor;
   GrDrawString((void *)s, strlen(s), __grlocx, __grlocy, &stdText);
   __grlocx += 8*strlen(s);
};

short _setvideomode(short m)
{
   if (m)
      return(GrSetMode(GR_width_height_color_graphics, 640, 480, 16));
   else
      return(GrSetMode(GR_default_text));
};

void _lineto(short x, short y)
{
   GrLine(__grlocx, __grlocy, x, y, __grcolor);
   __grlocx = x;
   __grlocy = y;
};

void _rectangle(short fill, short x1, short y1, short x2, short y2)
{
   if (fill == _GBORDER)
      GrBox(x1,y1,x2,y2,__grcolor);
   else
      GrFilledBox(x1,y1,x2,y2,__grcolor);
};

#define _clearscreen(x)       ScreenClear();

#endif /* __WATCOMC__ */

/* Definitions */

typedef unsigned char byte;
typedef byte FontTable[256][16];

enum {DIR_WHO_CARES = 0, DIR_UP, DIR_DOWN, DIR_LEFT, DIR_RIGHT};

enum {KEY_HOME = 'G', KEY_UP = 'H', KEY_PGUP = 'I',
      KEY_RIGHT= 'M', KEY_LEFT = 'K',
      KEY_END  = 'O', KEY_DN = 'P', KEY_PGDN = 'Q'};

typedef struct XlateRec {
   short xloc;
   short yloc;
   short next;
   short prev;
   short name;
} XlateRec;

/* Global Values */

FontTable chars;
char filename[256];
XlateRec *xlate;
char buffer[4096];  /* For holding the names */
char gameName[80];

int die(const char *msg)
{
   printf("%s",msg);
   exit(0);
   return(0);
};

void massage(char *s)
{
   short i;

   for(i=0;i<strlen(s);i++)
      while(((s[i]<' ') && (s[i]>0)) || ((s[i]==' ') && (s[i+1]==' ')))
         memmove(s+i,s+i+1,strlen(s)-i);

   while(*s==' ')
      memmove(s,s+1,strlen(s));

   while(s[strlen(s)-1]==' ')
      s[strlen(s)-1]=0;
};

char *getline(char *buf, int maxlen, FILE *f)
{
   while(1)
      {
      if (fgets(buf,maxlen,f))
         {
         massage(buf);
         if (*buf == '#')   continue;
         if (*buf == 0)     continue;
         return(buf);
         }
      else return(NULL);
      };
};

void getCharMap(void)
{
   static char line[4096];
   XlateRec *list;
   short buf_off = 1;
   short size;
   short i;
   short curx, cury;
   short thisx, thisy;
   char *tmp;

   FILE *f = fopen("gredit.ini","r");
   if (f == NULL)
      {
      f = fopen("lib\\xtra\\gredit.ini","r");
      if (f == NULL)
         die("Unable to open graphics map file.\n");
      };

   memset(buffer,0,4096);

   /* Find the [Main] section */
   while(1)
      {
      if (!getline(line,4096,f))
         die("No [Main] section.\n");

      if (strcmp(line,"[Main]")==0)
         break;
      };

   if (!getline(line,4096,f))
      die("Unexpected end of [Main].\n");
   if (*line == '[') die("Unexpected start of next section.\n");
   strcpy(gameName,line);

   if (!getline(line,4096,f))
      die("Unexpected end of [Main].\n");
   if (*line == '[') die("Unexpected start of next section.\n");
   if (sscanf(line,"%hd",&size)!=1) die("Illegal menu count.\n");

   list = (XlateRec *)malloc(sizeof(XlateRec)*(2*size+256));
   if (list == NULL) die("No memory.\n");
   memset(list,0,sizeof(XlateRec)*(2*size+256));
   xlate = list + 2*size;
   curx = 300;
   cury = 150;

   for(i=0;i<size;i++)
      {
      if (!getline(line,4096,f))
         die("Unexpected end of [Main].\n");
      if (*line == '[') die("Unexpected start of next section.\n");
      if (sscanf(line,"(%hd,%hd)",&thisx,&thisy)==2)
         {
         tmp = strchr(line,')')+1;
         while (*tmp == ' ') tmp++;
         curx = thisx;
         cury = thisy+10;
         }
      else
         {
         tmp = line;
         thisx = curx;
         thisy = cury;
         cury += 10;
         };

      xlate[-2*(i+1)].next = (i == (size-1)) ? -2 : (-2*(i+2));
      xlate[-2*(i+1)].prev = (i == 0 ? -2*size : (-2*i));
      xlate[-2*(i+1)].xloc = thisx;
      xlate[-2*(i+1)].yloc = thisy;
      xlate[-2*(i+1)].name = buf_off;
      strcpy(buffer+buf_off,tmp);
      buf_off += strlen(tmp)+1;
      };

   for(i=0;i<size;i++)
      {
      short menu;
      short prev;
      short loc;

      while(1)
         {
         if (*line == '[') break;
         if (!getline(line,4096,f))
            die("Unexpected end of file.\n");
         };

      line[strlen(line)-1]=0;   /* Remove closing ] */
      for(menu = 0;menu < size;menu++)
         if (strcmp(line+1,buffer+xlate[-2*(menu+1)].name)==0)
            break;

      if (menu == size)
         die("Section not found.\n");

      prev = -2*(menu+1)+1;
      curx = 300;
      cury = 150;

      if (!getline(line,4096,f))
         die("Unexpected end of file.\n");
      if (sscanf(line,"(%hd,%hd)",&thisx,&thisy)==2)
         {
         tmp = strchr(line,')')+1;
         while (*tmp == ' ') tmp++;
         curx = thisx;
         cury = thisy+10;
         }
      else
         {
         tmp = line;
         thisx = curx;
         thisy = cury;
         cury += 10;
         };

      xlate[-2*(menu+1)+1].xloc = thisx;
      xlate[-2*(menu+1)+1].yloc = thisy;
      xlate[-2*(menu+1)+1].name = buf_off;
      strcpy(buffer+buf_off,tmp);
      buf_off += strlen(tmp)+1;

      while(1)
         {
         if (!getline(line,4096,f)) break;
         if (*line == '[')          break;

         if (sscanf(line,"%hd (%hd,%hd)",&loc,&thisx,&thisy)==3)
            {
            tmp = strchr(line,')')+1;
            while (*tmp == ' ') tmp++;
            curx = thisx;
            cury = thisy+10;
            }
         else if (sscanf(line,"%hd",&loc)==1)
            {
            tmp = strchr(line,' ');
            while (*tmp == ' ') tmp++;
            thisx = curx;
            thisy = cury;
            cury += 10;
            }
         else break;

         xlate[loc].xloc = thisx;
         xlate[loc].yloc = thisy;
         xlate[loc].prev = prev;
         xlate[prev].next = loc;
         xlate[loc].name = buf_off;
         strcpy(buffer+buf_off,tmp);
         buf_off += strlen(tmp)+1;
         prev = loc;
         };

      xlate[-2*(menu+1)+1].prev = prev;
      xlate[prev].next = -2*(menu+1)+1;
      };

   fclose(f);
};

void prepend(const char *prefix, char *msg)
{
   memmove(msg+strlen(prefix),msg,strlen(msg));
   memcpy(msg,prefix,strlen(prefix));
};

void drawrow(byte bitmap, short color, short x, short y)
{
   short i;

   for(i=7;i>=0;i--)
      {
      _setcolor(bitmap & 0x01 ? color : 0);
      _setpixel(x+i,y);
      bitmap >>= 1;
      };
};

void drawchr(short chr, short color, short x, short y)
{
   short i;

   for(i=0;i<16;i++)
      drawrow(chars[chr][i],color,x,y+i);
};

#define logo()                                        \
   printf("Bitmap Graphics Editor (Version 3.1)\n"   \
          "Written for %s.\n"                         \
          "Copyright 1991-1996 by Mike Marcelais\n"   \
          "Another 4th Bryan Production...\n",gameName);

void putmessage(short colorx)
{
   _setcolor(colorx);
   _moveto(0,0);
   _outgtext("Bitmap Graphics Editor (Version 3.1)");
   _moveto(0,10);
   _outgtext("Written for ");
   _outgtext(gameName);
   _moveto(0,20);
   _outgtext("Copyright 1991-96 Mike Marcelais");
   _moveto(0,30);
   _outgtext("Another 4th Bryan Production...");
};

void shutdown(void)
{
   _setvideomode(_DEFAULTMODE);
   _clearscreen(_GCLEARSCREEN);
   logo();
   exit(0);
};

void initgraphics(short colorx)
{
   if (_setvideomode(_VRES16COLOR))
      {
      _setvideomode(_DEFAULTMODE);
      printf("VGA found\n");
      delay(2000);
      _setvideomode(_VRES16COLOR);
      putmessage(colorx);
      return;
      };

   _clearscreen( _GCLEARSCREEN );
   printf("VGA not found.\n"
          "VGA is required to run this editor and to get graphics support\n"
          "in %s.\n\n",gameName);
   logo();
   exit(1);
};

short fexist(const char *name)
{
   FILE *f = fopen(name, "rb");
   if (f)
      fclose(f);
   return(f!=NULL);
};

void getchars(void)
{
   FILE *f = fopen(filename, "rb");
   fread(chars,4096,1,f);
   fclose(f);
};

void savechars(void)
{
   FILE *f = fopen(filename, "wb");
   fwrite(chars,4096,1,f);
   fclose(f);
};

void importchar(short c, const char *name)
{
   FontTable impfont;
   FILE *f = fopen(name, "rb");
   fread(impfont,4096,1,f);
   fclose(f);
   memcpy(chars+c,impfont+c,16);
};

char idwhich(char *s)
{
   if (stricmp("BORDER",s)==0)
      return('B');
   if (stricmp("CURSOR",s)==0)
      return('C');
   if (stricmp("FILL",s)==0)
      return('F');
   if (stricmp("TEXT",s)==0)
      return('X');
   if (stricmp("TILE",s)==0)
      return('T');
   return(' ');
};

short idcolor(const char *s)
{
   if ((stricmp("BLACK",s)==0) || (strcmp("0",s)==0))
      return(0);
   if ((stricmp("BLUE",s)==0) || (strcmp("1",s)==0))
      return(1);
   if ((stricmp("GREEN",s)==0) || (strcmp("2",s)==0))
      return(2);
   if ((stricmp("CYAN",s)==0) || (strcmp("3",s)==0))
      return(3);
   if ((stricmp("RED",s)==0) || (strcmp("4",s)==0))
      return(4);
   if ((stricmp("MAGENTA",s)==0) || (strcmp("5",s)==0))
      return(5);
   if ((stricmp("BROWN",s)==0) || (strcmp("6",s)==0))
      return(6);
   if ((stricmp("GREY",s)==0) || (stricmp("LIGHT GREY",s)==0) ||
       (stricmp("GRAY",s)==0) || (stricmp("LIGHT GRAY",s)==0) ||
       (stricmp("LIGHTGREY",s)==0) || (stricmp("LIGHTGRAY",s)==0) ||
       (strcmp("7",s)==0))
      return(7);
   if ((stricmp("DARK GREY",s)==0) || (stricmp("DARK GRAY",s)==0) ||
       (stricmp("DARKGREY",s)==0) || (stricmp("DARKGRAY",s)==0) ||
       (strcmp("8",s)==0))
      return(8);
   if ((stricmp("LIGHTBLUE",s)==0) || (stricmp("LIGHT BLUE",s)==0) ||
       (strcmp("9",s)==0))
      return(9);
   if ((stricmp("LIGHTGREEN",s)==0) || (stricmp("LIGHT GREEN",s)==0) ||
       (strcmp("10",s)==0))
      return(10);
   if ((stricmp("LIGHTCYAN",s)==0) || (stricmp("LIGHT CYAN",s)==0) ||
       (strcmp("11",s)==0))
      return(11);
   if ((stricmp("LIGHTRED",s)==0) || (stricmp("LIGHT RED",s)==0) ||
       (strcmp("12",s)==0))
      return(12);
   if ((stricmp("LIGHTMAGENTA",s)==0) || (stricmp("LIGHT MAGENTA",s)==0) ||
       (strcmp("13",s)==0))
      return(13);
   if ((stricmp("YELLOW",s)==0) || (strcmp("14",s)==0))
      return(14);
   if ((stricmp("WHITE",s)==0) || (strcmp("15",s)==0))
      return(15);
   return(-1);
};

void getcolors(short *b, short *c, short *f, short *x, short *t)
{
   FILE *fle;
   char s[256];

   *b = 2;    /* Green */
   *c = 15;   /* White */
   *f = 1;    /* Blue */
   *x = 14;   /* Yellow */
   *t = 4;    /* Red */

   fle = fopen("gredit.prf","r");
   if (!fle)
      fle = fopen("lib\\xtra\\gredit.prf","r");
   if (fle)
      {
      while(getline(s,256,fle))
         {
         short wh, col;
         char *value = strchr(s,'=');
         if (value == NULL)         
            {
            printf("Error in configuration file:\n%s\n",s);
            getch();
            continue;
            };

         *value++ = 0;
         massage(s);
         massage(value);
         wh = idwhich(s);
         col = idcolor(value);

         if ((wh == ' ') || (col == -1))
            {
            printf("Error in configuration file:\n%s\n",s);
            getch();
            };
         switch(wh)
            {
            case 'B':  *b = col; break;
            case 'C':  *c = col; break;
            case 'F':  *f = col; break;
            case 'X':  *x = col; break;
            case 'T':  *t = col; break;
            };
         };
      fclose(fle);
      };
};

void drawstr(const char *s, short color, short x, short y)
{
   while(*s)
      {
      drawchr(*s,color,x,y);
      s++;
      x+=8;
      };
};

void drawcursor(short x, short y, short color)
{
   _setcolor(color);
   _moveto(120+20*x, 80+20*y);
   _lineto(120+20*x,100+20*y);
   _lineto(140+20*x,100+20*y);
   _lineto(140+20*x, 80+20*y);
   _lineto(120+20*x, 80+20*y);
};

void drawgridsq(short x, short y, short fill, short colorb, short colorf)
{
   short i;

   drawcursor(x,y,colorb);
   _setcolor(fill ? colorf : 0);
   for(i=1;i<20;i++)
      {
      _moveto(121+20*x,80+20*y+i);
      _lineto(139+20*x,80+20*y+i);
      };
};

void drawgrid(short ch, short colorb, short colorf)
{
   short i,j,k;

   for(i=0;i<16;i++)
      {
      k = chars[ch][i];
      for(j=7;j>=0;j--)
         {
         drawgridsq(j,i,k&0x01,colorb,colorf);
         k >>= 1;
         };
      };
};

void movecursor(byte dir, short *x, short *y, short colorb, short colorc)
{
   /* dir --> 1 up, 2 down, 3 left, 4 right */
   drawcursor(*x,*y,colorb);
   switch(dir)
      {
      case DIR_UP:    if (*y)            (*y)--;
                      break;
      case DIR_DOWN:  if (*y != 15)      (*y)++;
                      break;
      case DIR_LEFT:  if ((*x) || (!*y)) (*x)--;
                      break;
      case DIR_RIGHT: if (*x != 15)      (*x)++;
                      break;
      };
   drawcursor(*x,*y,colorc);
};

void drawzoom(short ch, short colort, short colorb)
{
   _setcolor(colorb);
   _rectangle(_GBORDER, 28,123,56,175);
   drawchr(ch,colort,30,125);
   drawchr(ch,colort,38,125);
   drawchr(ch,colort,46,125);
   drawchr(ch,colort,30,141);
   drawchr(ch,colort,38,141);
   drawchr(ch,colort,46,141);
   drawchr(ch,colort,30,157);
   drawchr(ch,colort,38,157);
   drawchr(ch,colort,46,157);
};

void drawimage(short ch, short x, short y, short colorc, short colorb,
               short colorf, short colort)
{
   drawgrid(ch,colorb,colorf);
   drawgridsq(0,1,0,colorb,0);
   drawchr(ch,colort,106,84);
   drawzoom(ch,colort,colorb);
   drawcursor(x,y,colorc);
};

void drawbigcursor(short i, short color)
{
   _setcolor(color);
   _moveto(xlate[i].xloc-2,xlate[i].yloc-2);
   _lineto(xlate[i].xloc-2,xlate[i].yloc+8);
   _lineto(xlate[i].xloc+4+8*strlen(buffer+xlate[i].name),xlate[i].yloc+8);
   _lineto(xlate[i].xloc+4+8*strlen(buffer+xlate[i].name),xlate[i].yloc-2);
   _lineto(xlate[i].xloc-2,xlate[i].yloc-2);
};

void printoptions(short colorx, short ch)
{
   _setcolor(0);
   _rectangle(_GFILLINTERIOR,299,19,640,51);
   _setcolor(colorx);
   _moveto(300,20);
   _outgtext("Editing ");
   _outgtext(buffer+xlate[ch].name);
   _moveto(300,30);
   _outgtext("Editing commands are:");
   _moveto(300,40);
   _outgtext("     Space toggles square");
   _moveto(300,50);
   _outgtext("     `B' Blanks character");
   _moveto(300,60);
   _outgtext("     `S' Saves characters to file");
   _moveto(300,70);
   _outgtext("     `C' Character Menu");
   _moveto(300,80);
   _outgtext("     `T' Transfer char from another file");
   _moveto(300,90);
   _outgtext("     ESC quits");
};

void selectchar(short *ch, short x, short y, short colorx, short colort,
                short colorc, short colorb, short colorf)
{
   short i;
   char c;

   do {
      _setcolor(0);
      _rectangle(_GFILLINTERIOR,299,99,640,480);
      _setcolor(colorx);
      _moveto(300,100);
      _outgtext("Use up and down arrows to");
      _moveto(300,110);
      _outgtext("choose a character to edit.");
      _moveto(300,120);
      _outgtext("Press Enter to exit.");
      i = *ch;
      do {
         _moveto(xlate[i].xloc,xlate[i].yloc);
         _outgtext(buffer+xlate[i].name);
         i = xlate[i].next;
         }
      while (i!=*ch);
      do {
         drawbigcursor(i,colorc);
         if (i > 0)
            {
            drawgridsq(-1,0,0,colorb,0);
            drawchr(i,colort,106,84);
            };
         c = getch();
         if (!c)
            c = getch();
         drawbigcursor(i,0);
         switch(c)
            {
            case '8':
            case KEY_UP:   i = xlate[i].prev;
                           break;
            case '2':
            case KEY_DN:   i = xlate[i].next;
                           break;

            case 'S':
            case 's':      savechars();
                           break;

            case 27:       shutdown();
                           break;
            };
         }
      while(c!=13);
      if (i < 0)
         {
         if (i & 1)
            i--;
         else
            i++;
         }
      *ch = i;
      }
   while (i < 0);

   _setcolor(0);
   _rectangle(_GFILLINTERIOR, 299, 99, 640, 480);
   _setcolor(colorx);
   drawimage(*ch,x,y,colorc,colorb,colorf,colort);
   printoptions(colorx,*ch);
   putmessage(colorx);
};

void getstring(short x, short y, char *s, short colorx)
{
   char c;

   *s = 0;
   do {
      switch (c = getch())
         {
         case 8:   if (strlen(s)>1)
                      s[strlen(s)-1]=0;
                   break;
         case 27:  *s = 0;
         case 13:  break;
         default:  s[strlen(s)+1]=0;
                   s[strlen(s)]=c;
                   break;
         };
      _setcolor(0);
      _rectangle(_GFILLINTERIOR, x-1, y-1, 640, y+11);
      _setcolor(colorx);
      _moveto(x,y);
      _outgtext(s);
      }
   while(c!=13);
};

void userinterface(short colorc, short colorb, short colorf, short colort,
                   short colorx, short *ch)
{
   short j,x,y;
   char c;
   char fn[256];

   _setcolor(colorx);
   printoptions(colorx,*ch);
   x = -1;
   y = 0;
   do {
      if ((x<0) || (*ch==0))
         {
         selectchar(ch,x,y,colorx,colort,colorc,colorb,colorf);
         if (x<0)
            movecursor(DIR_RIGHT,&x,&y,colorb,colorc);
         drawchr(*ch,colort,106,84);
         };
      c = getch();
      if (!c)
         c = getch();
      c = toupper(c);
      switch(c)
         {
         case '1':
         case KEY_END:    movecursor(DIR_DOWN,&x,&y,colorb,colorc);
                          movecursor(DIR_LEFT,&x,&y,colorb,colorc);
                          break;
         case '2':
         case KEY_DN:     movecursor(DIR_DOWN,&x,&y,colorb,colorc);
                          break;

         case '3':
         case KEY_PGDN:   movecursor(DIR_RIGHT,&x,&y,colorb,colorc);
                          movecursor(DIR_DOWN,&x,&y,colorb,colorc);
                          break;

         case '4':
         case KEY_LEFT:   movecursor(DIR_LEFT,&x,&y,colorb,colorc);
                          break;

         case '6':
         case KEY_RIGHT:  movecursor(DIR_RIGHT,&x,&y,colorb,colorc);
                          break;
 
         case '7':
         case KEY_HOME:   movecursor(DIR_UP,&x,&y,colorb,colorc);
                          movecursor(DIR_LEFT,&x,&y,colorb,colorc);
                          break;

         case '8':
         case KEY_UP:     movecursor(DIR_UP,&x,&y,colorb,colorc);
                          break;

         case '9':
         case KEY_PGUP:   movecursor(DIR_UP,&x,&y,colorb,colorc);
                          movecursor(DIR_RIGHT,&x,&y,colorb,colorc);
                          break;

         case 27:         break;
         case 'S':        savechars();
                          break;
         case ' ':        chars[*ch][y] ^= (0x80 >> x);
                          drawgridsq(x,y,chars[*ch][y] & (0x80 >> x),colorb,colorf);
                          drawchr(*ch,colort,106,84);
                          drawzoom(*ch,colort,colorb);
                          drawcursor(x,y,colorc);
                          break;

         case 'B':        for(j=0;j<16;j++)
                             chars[*ch][j]=0;
                          drawimage(*ch,x,y,colorc,colorb,colorf,colort);
                          drawchr(*ch,colort,106,84);
                          drawzoom(*ch,colort,colorb);
                          drawcursor(x,y,colorc);
                          break;
         case 'C':        selectchar(ch,x,y,colorx,colort,colorc,colorb,colorf);
                          break;
         case 'T':        _setcolor(colorx);
                          _moveto(300,120);
                          _outgtext("File to import: ");
                          _moveto(300,130);
                          _outgtext("-->");
                          getstring(332,130,fn,colorx);
                          prepend("lib\\xtra\\",fn);
                          if (fexist(fn))
                             {
                             importchar(*ch,fn);
                             drawimage(*ch,x,y,colorc,colorb,colorf,colort);
                             drawchr(*ch,colort,106,84);
                             drawzoom(*ch,colort,colorb);
                             drawcursor(x,y,colorc);
                             _setcolor(colorx);
                             _moveto(300,150);
                             _outgtext("Character imported.");
                             }
                          else
                             {
                             _setcolor(colorx);
                             _moveto(300,150);
                             _outgtext("File not found.");
                             };
                          delay(2000);
                          _setcolor(0);
                          _rectangle(_GFILLINTERIOR,299,119,640,480);
                          break;
         };
      }
   while (c!=27);
};

void help(void)
{
   printf("GrEdit - Bitmap graphic editor\n"
          "usage:   gredit [graphics.fnt]\n\n"
          "If ommited, the file defaults to angband.fnt.\n"
          "The file will always be search for in the lib/xtra directory.\n\n");
   logo();
   exit(0);
};

void main(int argc, char **argv)
{
   short colorb, colorc, colorf, colorx, colort;
   short ch;

   if (argc==2)
      {
      if (strchr(argv[1],'\\')==NULL)
         {
         strcpy(filename,"lib\\xtra\\");
         strcat(filename,argv[1]);
         }
      else
         strcpy(filename,argv[1]);
      }
   else if (argc>2)
      help();
   else
      strcpy(filename,"lib\\xtra\\angband.fnt");

   if (!fexist(filename))
      {
      printf("File %s not found.\n",filename);
      help();
      };

   ch = -2;

   getCharMap();
   _clearscreen( _GCLEARSCREEN );
   getchars();
   getcolors(&colorb,&colorc,&colorf,&colorx,&colort);
   initgraphics(colorx);
   userinterface(colorc,colorb,colorf,colort,colorx,&ch);
   shutdown();
};
