/*
**                      Softwarepraktikum iMaze
**                              1993/94
**                Joerg Czeranski    Hans-Ulrich Kiel
**
** Datei: netzwerk.c
**
** Kommentar:
**  wertet die Pakete aus, die ueber das Netz vom Server kommen und stellt
**  die Antworten zusammen. Dieser Teil ist von der Art des Netzwerks
**  unabhaengig. Das Senden und Empfangen auf unterster Ebene uebernimmt
**  ein anderer Programmteil.
*/


#include <stdio.h>
#include <string.h>

#include "global.h"
#include "fehler.h"
#include "speicher.h"
#include "labyrinth.h"
#include "spieler.h"
#include "signale.h"
#include "ereignisse.h"
#include "protokoll.h"
#include "netzwerk.h"
#include "spiel.h"
#include "system.h"

static char sccsid[] = "@(#)netzwerk.c	1.44 5/18/94";


/* Zeit in sec, nach der die Verbindung zum Server hart beendet wird, auch
   ohne, dass dieser ein Verlassen des Spiels bestaetigt hat */
#ifndef SPIEL_VERLASSEN_TIMEOUT
#define SPIEL_VERLASSEN_TIMEOUT 3
#endif

#define ERG_WAR_DA 1 /* Bit 0 wird gesetzt, wenn ein Ereignis im vorigen
                        Paket mitgeschickt wurde */
#define ERG_IST_DA 2 /* Bit 1 wird gesetzt, wenn das aktuelle Paket dieses
                        Ereignis enthaelt */

/* Feld mit allen Ereignissen, in dem obige Flags gesetzt werden */
static char ereignis_flags[ERG_ANZ];

/* Flag: soll mit dem naechsten Paket dem Server das Verlassen des Spiels
   mitgeteilt werden?
   0 = Nicht verlassen, 1 = Server das Ende mitteilen,
   2 = Ende durch Server bestaetigt */
static int endesignal_senden = 0;


/*
** spielfeld_kopieren
**  speichert die vom Server uebertragenen Labyrinth-Informationen in die
**  interne Darstellung
**
** Parameter:
**  spielfeld: Zeiger auf das Spielfeld (hier werden die Daten abgelegt)
**  feldbreite: Breite des Spielfeldes
**  feldlaenge: Laenge des Spielfeldes
**  daten: Liste mit den Daten
**
** Seiteneffekte:
**  spielfeld wird gesetzt
*/
static void spielfeld_kopieren(spielfeld, feldbreite, feldlaenge, daten)
block ***spielfeld;
int feldbreite, feldlaenge;
u_char *daten;
{
	int x, y; /* Index fuer das Spielfeld */
	int i;    /* Index fuer 4 Himmelsrichtungen */

	/* Feld in passender Groesse anlegen */
	speicher_belegen((void **)spielfeld, feldbreite * sizeof(block *));

	for (x = 0; x < feldbreite; x++)
		speicher_belegen((void **)&(*spielfeld)[x],
			feldlaenge * sizeof(block));

	/* Daten kopieren */
	for (y = 0; y < feldlaenge; y++)
		for (x = 0; x < feldbreite; x++)
			for (i = 0; i < 4; i++)
			{
				struct wand *wand;

				wand = &(*spielfeld)[x][y][i];
				wand->farbe = *daten++;
				/* ob eine Wand als Tuer angezeigt wird, wird an der
				   Farbe entschieden */
				wand->tuer = IST_TUER(wand->farbe);
			}
}


/*
** protokoll_fehler
**  ruft die Fehlerbox mit einer einheitlichen Meldung fuer Verletzungen
**  des iMaze-Protokolls auf (dient nur der Uebersichtlichkeit)
**
** Parameter
**  ursache: String mit Text
*/
static void protokoll_fehler(ursache)
char *ursache;
{
	/* Stringfeld mit dem Meldungstext */
	static char *meldung[] = { "iMaze - Protocol Error", "", NULL, NULL };
	u_char kennung; /* Kennung fuer Mitteilung an den Server */

	/* Mitteilung an den Server schicken */
	kennung = PROL_PAKET_KENNUNG(PAKET_IGNORIERT, PT_ABBRUCH);
	prolog_paket_senden(1, &kennung);

	/* Fehlermeldung ausgeben */
	meldung[2] = ursache;
	uebler_fehler(meldung, NULL);
}


/* bis hier lokaler Teil                       */
/***********************************************/
/* ab hier globaler Teil                       */


/*
** spielparameter_empfangen
**  tauscht mit dem Server einmalig zu uebertragene Informationen ueber
**  den Aufbau der Spielrunde und das Spielfeld aus.
**
** Parameter:
**  programmname: String zur Identifikation des eigenen Programms
**  feldbreite: Zeiger auf die Breite des Spielfeldes
**  feldlaenge: Zeiger auf die Laenge des Spielfeldes
**  spielfeld: Zeiger auf das Spielfeld
**
** Seiteneffekte:
**  *feldbreite, *feldlaenge und *spielfeld werden gefuellt, falls nicht
**  NULL; ereignis_flags werden initialisiert
*/
void spielparameter_empfangen(programmname, feldbreite, feldlaenge,
	spielfeld)
char *programmname;
u_int *feldbreite, *feldlaenge;
block ***spielfeld;
{
	/* Flags: */
	int labyrinth_fehlt;          /* der Client hat das Spielfeld
	                                 noch nicht erhalten */
	int programmname_uebertragen; /* der Client hat sich bereits
	                                 beim Server identifiziert */
	int ignoriert;                /* das aktuelle Paket wird vom Client
	                                 ignoriert (z.B. nicht verstanden) */

	{
		int i;

		/* ereignis_flags initialisieren */
		for (i = 0; i < ERG_ANZ; i++)
			ereignis_flags[i] = 0;
	}

	/* falls feldlaenge oder feldbreite = NULL, soll kein Labyrinth gelesen
	   werden */
	labyrinth_fehlt = feldbreite != NULL || feldlaenge != NULL;
	/* Identifikation-Text wird nicht uebertragen, wenn = NULL */
	programmname_uebertragen = programmname == NULL;
	/* der Client sendet zuerst, er kann also zuerst kein
	   Paket ignorieren */
	ignoriert = PAKET_NICHT_IGNORIERT;

	/* Schleife, bis alle Parameter-Pakete ausgetauscht sind */
	for (;;)
	{
		int bereit;	      /* Flag: der Client hat aus seiner Sicht alle
						     notwendigen Pakete ausgetauscht und ist bereit
		                     fuer den Beginn des Spiels */
		int paket_laenge; /* Laenge eines empfangen Paketes */
		u_char kennung;   /* Paket-Kennung des gesendeten Pakets*/
		u_char *paket;    /* Liste mit einem Paket */

		/* der Client ist bereit fuer den Beginn des Spiel, wenn er sich
		   beim Server identifiziert und das Spielfeld erhalten hat */
		bereit = !labyrinth_fehlt && programmname_uebertragen;

		/* der Client sendet zuerst ein Paket */

		if (bereit)
		{
			/* dem Server die Bereitschaft mitteilen */
			kennung = PROL_PAKET_KENNUNG(ignoriert, PT_BEREIT_FUER_SPIEL);
			prolog_paket_senden(1, &kennung);
		}
		else if (!programmname_uebertragen)
		{
			/* falls Identifikationstext noch nicht uebertragen, wird er
			   in ein Paket dynamischer Laenge getan und an den Server
			   geschickt */
			speicher_belegen((void **)&paket, 1 + strlen(programmname));
			paket[0] = kennung =
				PROL_PAKET_KENNUNG(ignoriert, PT_PROGRAMM_NAME);
			memcpy(&paket[1], programmname, strlen(programmname));
			prolog_paket_senden(1 + strlen(programmname), paket);
			speicher_freigeben((void **)&paket);

			/* das Flag setzen */
			programmname_uebertragen = 1;
		}
		else
		{
			/* falls Spielfeld noch nicht erhalten, vom Server anfordern */
			kennung = PROL_PAKET_KENNUNG(ignoriert, PT_WILL_LABYRINTH);
			prolog_paket_senden(1, &kennung);
		}

		/* jetzt wird ein Paket vom Server empfangen */

		prolog_paket_empfangen(&paket_laenge, (void **)&paket);
		/* das Server-Paket ist vorerst ungekannt */
		ignoriert = PAKET_IGNORIERT;

		/* das Paket muss mindestens ein Zeichen umfassen, sonst ist das
		   ein Protokoll-Fehler */
		if (!paket_laenge)
			protokoll_fehler("Empty prolog packet received.");

		/* das erste Zeichen enthaelt die Paket-Kennung; von ihr haengt die
		   weitere Behandlung ab */
		switch (PROL_PAKET_TYP(paket[0]))
		{
			case PT_LEER: /* leeres Paket */
				break;

			case PT_ABBRUCH: /* Abbruch durch den Server */
				{
					/* hier wird eine Fehlerbox ausgegeben und das Programm
					   beendet */
					static char *meldung[] = { "iMaze - Fatal Error", "",
						"Aborted by server.", NULL };

					kennung = PROL_PAKET_KENNUNG(PAKET_IGNORIERT,
						PT_ABBRUCH);
					prolog_paket_senden(1, &kennung);

					uebler_fehler(meldung, NULL);
				}

			case PT_SPIEL_BEGINNT: /* der Server beginnt das Spiel */
				/* hat der Client noch nicht alle Information, wird mit
				   einem Fehler abgebrochen (nach diesem Paket kann auf
				   dieser Protokollebene nicht weiteruebertragen werden) */
				if (!bereit)
					protokoll_fehler(
						"Game started by server without consent.");

				/* sonst ist dieser Protokollteil beendet */
				speicher_freigeben((void **)&paket);
				return;

			case PT_LABYRINTH: /* der Server schickt das Spielfeld */
				/* wenn feldbreite und feldlaenge = NULL, ist das Spielfeld
				   gar nicht erwuenscht */
				if (feldbreite == NULL && feldlaenge == NULL)
					break;
				/* wenn das Paket nicht einmal die Groessen enthaelt, ist
				   es nicht zu gebrauchen */
				if (paket_laenge < 3)
					break;

				/* feldlaenge und feldbreite auslesen, falls gewuenscht */
				if (feldbreite != NULL)
				{
					*feldbreite = paket[1];
					if (*feldbreite == 0)
						break;
				}
				if (feldlaenge != NULL)
				{
					*feldlaenge = paket[2];
					if (*feldlaenge == 0)
						break;
				}

				/* ist ein Spielfeld nicht erwuenscht, wird abgebrochen */
				if (feldlaenge == NULL || feldlaenge == NULL ||
					spielfeld == NULL)
				{
					labyrinth_fehlt = 0;
					ignoriert = PAKET_NICHT_IGNORIERT;
					break;
				}
				/* stimmt die Laenge des Paketes?
				   (kennung + feldbreite + feldlaenge + feldbreite  *
				   feldlaenge * 4 Waende) */
				if (paket_laenge < 4 * *feldbreite * *feldlaenge + 3)
					break;

				/* Flag setzen und Spielfeld in Datenstrukturen kopieren */
				labyrinth_fehlt = 0;
				ignoriert = PAKET_NICHT_IGNORIERT;
				spielfeld_kopieren(spielfeld, *feldbreite, *feldlaenge,
					&paket[3]);
				break;
		}

		speicher_freigeben((void **)&paket);
	}
}


/*
** spiel_verlassen
**  beendet die Verbindung zum Server; falls der Server nicht mehr
**  ordnungsgemaess antwortet, wird nach Ablauf einer Zeit beendet
**
** Seiteneffekt:
**  endesignal_senden wird gesetzt
*/
void spiel_verlassen()
{
	void *alter_zustand; /* Zwischenspeicher fuer Timer-Zustand */

	alter_zustand = timer_retten();
	/* Zeit fuer maximale Wartezeit setzen */
	timer_starten(SPIEL_VERLASSEN_TIMEOUT * 1000);

	/* in einer Schleife pruefen, ob Bestaetigung durch Server eingetroffen
	   oder Wartezeit abgelaufen */
	for (endesignal_senden = 1; endesignal_senden == 1;)
	{
		int restzeit;

		if ((restzeit = timer_restzeit()) == 0)
			break;

		spiel_paket_erwarten(restzeit);
	}

	/* Timer zuruecksetzen */
	timer_restaurieren(alter_zustand);

	/* Beenden der Verbindung zum Server */
	verbindung_beenden();

	endesignal_senden = 0;
}


/*
** spiel_paket_angekommen
**  wird aufgerufen, wenn ein Paket vom Server ankommt, und
**  wertet dieses aus
**
** Parameter:
**  paket_laenge: Laenge des Pakets
**  paket: Liste mit den Daten
**
** Seiteneffekte:
**  ereignis_flags und endesignal_senden werden veraendert
*/
void spiel_paket_angekommen(paket_laenge, paket)
int paket_laenge;
void *paket;
{
	u_char *daten;      /* Liste mit den Daten eines Pakets */
	int i;
	int *spieler_aktiv; /* Zeiger auf Flag: ist der Spieler selbst am
	                       Leben? */
	int *schuss_aktiv;  /* Zeiger auf Flag: hat der Spieler gerade einen
	                       Schuss im Spiel? */
	int *gegneranz;     /* Zeiger auf die Anzahl der aktuell sichtbaren
	                       Mitspieler */
	int *schussanz;     /* Zeiger auf die Anzahl der Schuesse der
	                       Mitspieler */
	int *abgeschossen_durch; /* Zeiger auf die Farbe des Spielers, von dem
	                            man ggf. abgeschossen wurde */
	char **abschuss_spruch;  /* Zeiger auf String mit dem Spruch des Gegners,
	                            von dem man abgeschossen wurde */
	char *ereignisse;        /* Liste von Ereignissen */
	struct spieler *spieler; /* Zeiger auf Daten ueber den Spieler selbst */
	struct spieler *gegner;  /* Liste mit den Daten der Mitspieler */
	struct schuss *schuss;   /* Zeiger auf Daten ueber den eigenen Schuss */
	struct schuss *schuesse; /* Liste mit den Daten der Gegner-Schuesse */

	/* im Feld ereignis_flags fuer jedes Ereignis das Bit fuer 'Ereignis
	   wurde mituebertragen' loeschen */
	for (i = 0; i < ERG_ANZ; i++)
		ereignis_flags[i] &= ~ERG_IST_DA;

	/* falls dem Server der Ausstieg aus der Spielrunde mitgeteilt werden
	   soll, sind alle Spiel-Daten uninteressant */
	if (endesignal_senden)
	{
		spieler_aktiv = gegneranz = schuss_aktiv = schussanz = NULL;
		spieler = gegner = NULL;
		schuss = schuesse = NULL;
		abgeschossen_durch = NULL;
		abschuss_spruch = NULL;
		ereignisse = NULL;
	}
	else
	/* andernfalls werden die Pufferspeicher fuer die Spieldaten zur
	   Verfuegung gestellt */
		spiel_puffer_anlegen(&spieler_aktiv, &spieler, &gegneranz, &gegner,
			&schuss_aktiv, &schuss, &schussanz, &schuesse,
			&abgeschossen_durch, &abschuss_spruch, &ereignisse);

	/* die Spiel-Daten, die gewuenscht sind, werden auf sinnvolle Werte
	   initialisiert */
	if (spieler_aktiv != NULL)
		*spieler_aktiv = 0;
	if (schuss_aktiv != NULL)
		*schuss_aktiv = 0;
	if (gegneranz != NULL)
		*gegneranz = 0;
	if (schussanz != NULL)
		*schussanz = 0;
	if (abgeschossen_durch != NULL)
		*abgeschossen_durch = -1;
	if (abschuss_spruch != NULL)
		*abschuss_spruch = NULL;
	if (ereignisse != NULL)
		for (i = 0; i < ERG_ANZ; i++)
			ereignisse[i] = 0;

	daten = paket;

	/* Scheife, in der das Paket der Reihe nach ausgewertet wird */
	while (paket_laenge > 0)
	{
		u_char kennung; /* Paket-Kennung des Unterbereichs im Gesamtpaket */
		int laenge;	    /* Laenge der Daten in den Unterpaketen */

		/* Zeichen 1 ist jeweils die Paketkennung */
		kennung = *daten++;
		paket_laenge--;

		/* hat das Paket Daten angehaengt oder besteht es nur aus der
		   Kennung? */
		if (DATENLAENGE_VARIABEL(kennung))
		{
			/* enthaelt das Gesamtpaket noch Daten? */
			if (!paket_laenge)
				break;

			/* Laenge des Unterpakets auslesen */
			laenge = *daten++;
			paket_laenge--;
		}
		else
			/* keine Laenge, falls Unterpaket nur aus der Kennung besteht */
			laenge = 0;

		/* enthaelt das Gesamtpaket soviel Daten, wie angegeben? */
		if (paket_laenge < laenge)
			break;

		/* Auswerten des Unterpakets */
		switch (kennung)
		{
			case BK_ENDE_DURCH_SERVER: /* der Server beendet die Verbindung
			                              von sich aus */
			case BK_ENDE_DURCH_CLIENT: /* der Server beendet die Verbindung
				                          auf Wunsch des Clients */

				/* falls der Timeout laeuft, im Flag vermerken, dass die
				   Bestaetigung von Server vorliegt; dann wird die Verbindung
				   aus der Funktion 'spiel_verlassen' geschlossen */
				if (endesignal_senden)
					endesignal_senden = 2;
				else
				{
					/* sonst (unvorhergesehener Fall) Behandlungsroutine
					   aufrufen */
					netzwerk_spielende();
					return;
				}
				break;

			case BK_SPIELER_POS: /* Daten ueber den eigenen Spieler */

				/* erwartete Laenge pruefen */
				if (laenge != 6)
					break;

				/* Information unerwuenscht? */
				if (spieler_aktiv == NULL)
					break;

				/* der Spieler ist lebendig */
				*spieler_aktiv = 1;

				/* Daten unerwuenscht? */
				if (spieler == NULL)
					break;

				/* Daten in Struktur schreiben */
				spieler->pos.xgrob = daten[0];
				spieler->pos.xfein = daten[1];
				spieler->pos.ygrob = daten[2];
				spieler->pos.yfein = daten[3];
				spieler->farbe = daten[4];
				spieler->blick = daten[5];
				break;

			case BK_GEGNER_POS: /* Daten ueber die Mitspieler */

				/* Daten nicht erwuenscht? */
				if (gegneranz == NULL)
					break;

				/* erwartete Laenge pruefen (je 6 Zeichen pro Spieler
				   und weniger als die maximal zulaessige Spieleranzahl) */
				if (!laenge || laenge / 6 * 6 != laenge ||
					*gegneranz + laenge / 6 > SPIELERANZ)
					break;

				/* falls nur Anzahl, aber nicht Position der Mitspieler
				   erwuenscht */
				if (gegner == NULL)
				{
					*gegneranz += laenge / 6;
					break;
				}

				/* alle Mitspieler in den Datenstrukturen ablegen */
				for (i = 0; i < laenge / 6; i++)
				{
					gegner[*gegneranz].pos.xgrob = daten[6 * i];
					gegner[*gegneranz].pos.xfein = daten[6 * i + 1];
					gegner[*gegneranz].pos.ygrob = daten[6 * i + 2];
					gegner[*gegneranz].pos.yfein = daten[6 * i + 3];
					gegner[*gegneranz].farbe = daten[6 * i + 4];
					gegner[*gegneranz].blick = daten[6 * i + 5];
					(*gegneranz)++;
				}
				break;

			case BK_SPIELER_SCHUSS_POS: /* Daten ueber eigenen Schuss */

				/* erwartete Laenge pruefen */
				if (laenge != 5)
					break;

				/* Information erwuenscht? */
				if (schuss_aktiv == NULL)
					break;

				/* ein Schuss ist unterwegs */
				*schuss_aktiv = 1;

				/* Daten erwuenscht? */
				if (schuss == NULL)
					break;

				/* alle Daten in die Struktur uebernehmen */
				schuss->pos.xgrob = daten[0];
				schuss->pos.xfein = daten[1];
				schuss->pos.ygrob = daten[2];
				schuss->pos.yfein = daten[3];
				schuss->farbe = daten[4];
				break;

			case BK_GEGNER_SCHUSS_POS: /* Daten ueber die Schuesse der
			                              Mitspieler */

				/* Information erwuenscht? */
				if (schussanz == NULL)
					break;

				/* erwartete Laenge pruefen (je 5 Zeichen pro Schuss
				   und weniger als die maximal zulaessige Spieleranzahl) */
				if (!laenge || laenge / 5 * 5 != laenge ||
					*schussanz + laenge / 5 > SPIELERANZ)
					break;

				/* falls nur Anzahl, aber nicht Position der Schuesse
				   erwuenscht */
				if (schuesse == NULL)
				{
					*schussanz += laenge / 5;
					break;
				}

				/* alle Schuesse in den Datenstrukturen ablegen */
				for (i = 0; i < laenge / 5; i++)
				{
					schuesse[*schussanz].pos.xgrob = daten[5 * i];
					schuesse[*schussanz].pos.xfein = daten[5 * i + 1];
					schuesse[*schussanz].pos.ygrob = daten[5 * i + 2];
					schuesse[*schussanz].pos.yfein = daten[5 * i + 3];
					schuesse[*schussanz].farbe = daten[5 * i + 4];
					(*schussanz)++;
				}
				break;

			case BK_ABGESCHOSSEN_DURCH: /* Informationen ueber den
			                               Schuetzen */

				/* Laenge pruefen */
				if (!laenge)
					break;

				/* Farbe des Schuetzen zuweisen, falls erwuenscht */
				if (abgeschossen_durch != NULL)
					*abgeschossen_durch = daten[0];

				/* Abbruch, falls keine weiteren Daten oder Spruch nicht
				   erwuenscht */
				if (laenge < 1 || abschuss_spruch == NULL)
					break;

				/* den Spruch des Schuetzen in internen String uebernehmen */
				speicher_belegen((void *)*abschuss_spruch, laenge);
				memcpy(*abschuss_spruch, &daten[1], laenge - 1);
				(*abschuss_spruch)[laenge - 1] = 0;
				break;

			case BK_EREIGNISSE: /* Liste der aufgetretenen Ereignisse */

				/* Informationen erwuenscht? */
				if (ereignisse == NULL)
					break;

				/* alle Ereignisse in das Feld eintragen */
				for (i = 0; i < laenge; i++)
					if (daten[i] < ERG_ANZ)
					{
						ereignisse[daten[i]] = 1;
						/* doppelte Ereignisse fallen aufeinander */
						ereignis_flags[daten[i]] |= ERG_IST_DA;
					}
				break;

			case BK_ALTE_EREIGNISSE: /* Liste der nachgeschickten Ereignisse
									    (falls im Netz Pakete verloren gehen,
				                        werden die Ereignisse zur Sicherheit
				                        nocheinmal nachgeschickt) */

				/* Informationen erwuenscht? */
				if (ereignisse == NULL)
					break;

				/* alle Ereignisse in das Feld eintragen */
				for (i = 0; i < laenge; i++)
					if (daten[i] < ERG_ANZ)
					{
						/* ist das Ereignis beim letztenmal noch doch richtig
						   uebermittelt worden? */
						if (!(ereignis_flags[daten[i]] & ERG_WAR_DA))
							ereignisse[daten[i]] = 1;
						/* doppelte Ereignisse fallen aufeinander */
						ereignis_flags[daten[i]] |= ERG_IST_DA;
					}
				break;
		}

		/* Gesamtpaket um das jetzt fertig abgearbeitet Unterpaket
		   verkuerzen */
		daten += laenge;
		paket_laenge -= laenge;
	}


	/* jetzt: Senden des Antwort-Pakets */


	/* falls dem Server ein Aussteigen aus dem Spiel mitgeteilt werden soll */
	if (endesignal_senden)
	{
		/* noch keine Bestaetigung vom Server eingetroffen */
		if (endesignal_senden == 1)
		{
			u_char antwort; /* Zeichen mit Paketkennung */

			/* Wunsch nach Ausstieg aus dem Spiel senden */
			antwort = BK_ENDE_SIGNAL;
			spiel_paket_senden(1, &antwort);
		}
	}
	else
	{
		int signalanz;                 /* Anzahl der ausgeloesten
		                                  Eingabesignale */
		char signale[SIGNALANZ];       /* Feld mit Flags fuer alle Signale */
		u_char antwort[SIGNALANZ + 2]; /* Feld mit den zu uebertragenden
		                                  Signalen */

		/* Signale in Eingabeteil abfragen */
		signale_abfragen(signale);

		/* aus dem Feld mit Flags die Nummer des Signals ermitteln und an
		   Antwort-Paket anhaengen */
		for (i = signalanz = 0; i < SIGNALANZ; i++)
			if (signale[i])
				antwort[signalanz +++ 2] = i; /* Uebung: Was macht das? */

		antwort[0] = BK_SIGNALE; /* Paket-Kennung */
		antwort[1] = signalanz;  /* Zeichen 1 enthaelt die Anzahl der
		                            folgenden Signale */

		/* Paket zum Server senden */
		spiel_paket_senden(2 + signalanz, antwort);
	}

	/* in dem Feld der Ereignisse alle neuen Ereignisse als alte Ereignisse
	   fuer naechsten Durchlauf eintragen */
	for (i = 0; i < ERG_ANZ; i++)
		ereignis_flags[i] = ereignis_flags[i] & ERG_IST_DA ? ERG_WAR_DA : 0;
}
