#include <mymalloc.h>
#include <myxlib.h>

#include <string.h>
#if !STDC_HEADERS && HAVE_MEMORY_H
# include <memory.h>
#endif /* not STDC_HEADERS and HAVE_MEMORY_H */

#include <X11/StringDefs.h>

#include "gamesP.h"
#include "reviews.h"

#ifdef    HAVE_NO_MEMCHR_PROTO
extern void *memchr(const void *s, int c, size_t n);
#endif /* HAVE_NO_MEMCHR_PROTO */

#define XTPOINTER_TO_INT(x) ((int)(long) (x))
#define INT_TO_XTPOINTER(x) ((XtPointer) (long) (x))

/* Must be < 0 */
#define REVIEWNONODE  -1
#define REVIEWBADNODE -2

Widget ReviewsButton;
static void DoReview(Widget w, XEvent *evnt, String *str, Cardinal *n);
static XtActionsRec ReviewsActionTable[] = {
    { "doreview",    DoReview   },
};

Game ReviewBase = { &ReviewBase, &ReviewBase };
static Widget ReviewRoot;
static int              ReviewNode = REVIEWNONODE;
static Game            *Reviewed = NULL;
static NumVal          *ReviewGlobal = NULL;
static NumNameListList *ReviewLocal  = NULL;
static int              WantReviews;
static int              DropVariations;
static int              ReviewPos;

extern struct tm UniversalTime;

static int ReviewInsert(Widget w)
{
    return ReviewPos;
}
 
void InitReviews(Widget Toplevel)
{
    Widget ReviewsWidget, ReviewCollect;

    ReviewGlobal = mynew(NumVal);
    ReviewGlobal->Previous = ReviewGlobal->Next  = ReviewGlobal;
    ReviewGlobal->Num   = 0;
    ReviewGlobal->Value = NULL;

    ReviewLocal = mynew(NumNameListList);
    ReviewLocal->Previous = ReviewLocal->Next  = ReviewLocal;
    ReviewLocal->Num = 0;
    ReviewLocal->Value = NULL;

    WantReviews = 0;

    ReviewRoot =
        MyVaCreateManagedWidget("reviews", Toplevel,
                                "insertReview", (XtArgVal) ReviewInsert,
                                NULL);

    ReviewsWidget = XtNameToWidget(ReviewRoot, "*reviewset");
    ReviewCollect = ReviewsWidget;

    XtAppAddActions(AppContext(Toplevel), ReviewsActionTable,
                    XtNumber(ReviewsActionTable));
    XtAddCallback(ReviewsButton, XtNcallback, CallToggleUpDown,
                  (XtPointer) ReviewRoot);
    XtAddCallback(ReviewRoot, XtNpopupCallback, CallToggleOn,
                  (XtPointer) ReviewsButton);
    XtAddCallback(ReviewRoot, XtNpopdownCallback, CallToggleOff,
                  (XtPointer) ReviewsButton);

    XtRealizeWidget(ReviewRoot);
    XtInstallAllAccelerators(ReviewCollect, ReviewCollect);
    DeleteProtocol(ReviewRoot);
    CallToggleUpDown(ReviewsButton, (XtPointer) ReviewRoot, NULL);
}

static void DestroyReview(Game *Entry);
void CleanReviews(void)
{
    if (ReviewGlobal) {
        FreeNumValList(ReviewGlobal);
        ReviewGlobal = NULL;
    }
    if (ReviewLocal) {
        FreeNumNameListList(ReviewLocal);
        ReviewLocal = NULL;
    }
    while (ReviewBase.Next != &ReviewBase) {
        ReviewBase.Next->Widget = NULL;
        DestroyReview(ReviewBase.Next);
    }
    XtDestroyWidget(ReviewRoot);
    Reviewed   = NULL;
    ReviewNode = REVIEWNONODE;
}

static Game *MakeReview(const char *Title)
{
    Game *Ptr;

    Ptr = mynew(Game);
    Ptr->Black     = NULL;
    Ptr->White     = NULL;
    Ptr->Log       = NULL;
    Ptr->Observers = NULL;
    Ptr->ServerId  = REVIEWGAME;
    Ptr->Next      = &ReviewBase;
    Ptr->Previous  =  ReviewBase.Previous;
    Ptr->Previous->Next = Ptr->Next->Previous = Ptr;
    Ptr->Komi          = NULL;
    Ptr->Move          = 0;
    Ptr->Handicap      = 0;
    Ptr->Title         = mystrdup(Title);
    Ptr->Observed      = 0;
    Ptr->WantObserved  = 0;
    Ptr->NrObservers   = 0;
    Ptr->XSize         = -1;
    Ptr->YSize         = -1;
    Ptr->UniversalTime = UniversalTime;
    Ptr->Widget        = 0;
    Ptr->WidgetPlan    = 0;
    Ptr->Pos           = 0;
    Ptr->Mode          = 'U';
    Ptr->Rules         = 'U';
    Ptr->Finished      = OVER;
    Ptr->WhiteCaptures = Ptr->BlackCaptures = 0;
    Ptr->Found         = NEW;
    Ptr->WhiteTime = Ptr->BlackTime = 0;
    Ptr->ByoPeriod     = 0;
    Ptr->BlackByo      = Ptr->WhiteByo = NOBYO;
    Ptr->ToMove        = Empty;
    /* The next lines assume Empty = 0, Black = 1, White = 2 --Ton */
    Ptr->Color = Empty;
    return Ptr;
}

static void DestroyReview(Game *Entry)
{
    if (Entry->Widget) {
        XtDestroyWidget(Entry->Widget);
        Entry->Widget = 0;
    }
    if (Entry->White) {
        FreeDummyPlayer(Entry->White);
        Entry->White = NULL;
    }
    if (Entry->Black) {
        FreeDummyPlayer(Entry->Black);
        Entry->Black = NULL;
    }
    FreeGame(Entry);
}

static void CallReview(Widget w, XtPointer clientdata, XtPointer calldata);
static void DisplayReview(Game *Entry)
{
    const char *Name;
    Game       *Here;
    Widget      New;

    if (Entry->Found == UNCHANGED) return;
    if (Entry->Found == DELETED) {
        if (Entry->Widget) {
            XtDestroyWidget(Entry->Widget);
            Entry->Widget = 0;
        }
        return;
    }

    /* Here we are CHANGED or NEW */

    if (Entry->WantObserved)  Name = "reviewEntryPending";
    else if (Entry->Observed) Name = "reviewEntryDone";
    else                      Name = "reviewEntry";

    ReviewPos = 0;
    for (Here = ReviewBase.Next; Here != &ReviewBase; Here = Here->Next)
        if (Here == Entry) break;
        else if (Here->Widget) ReviewPos++;

    New = MyVaCreateWidget(Name, ReviewRoot,
                           XtNlabel, (String) Entry->Title, NULL);
    XtAddCallback(New, XtNcallback, CallReview, (XtPointer) Entry);
    VaSetManagementChildren(New, 1, Entry->Widget, 0, (Widget) 0);
    if (Entry->Widget) XtDestroyWidget(Entry->Widget);
    Entry->Widget = New;
    Entry->Found = UNCHANGED;
}

static void TryReview(void)
{
    Game *review;

    if (ReviewNode == REVIEWNONODE)
        for (review = ReviewBase.Next; review != &ReviewBase;
             review = review->Next)
            if (review->WantObserved) {
                ReviewNode = 0;
                Reviewed = review;
                SendCommand(NULL, (XtPointer) 1, "review %s", review->Title);
                break;
            }
}

static void WantReview(Game *review)
{
    if (review->Observed || review->WantObserved) return;
    review->WantObserved = 1;
    TryReview();
    review->Found = CHANGED;
    DisplayReview(review);
}

void ReviewList(const NameList *Reviews)
{
    NameList *Review;
    Game     *REntry, *Next;
    int       Length;

    WantReviews = 0;
    for (REntry = ReviewBase.Next; REntry != &ReviewBase;
         REntry = REntry->Next) REntry->Found = DELETED;
    for (Review = Reviews->Next; Review != Reviews; Review = Review->Next) {
        Length = strlen(Review->Name);
        for (REntry = ReviewBase.Next; REntry != &ReviewBase;
             REntry = REntry->Next)
            if (strncmp(REntry->Title, Review->Name, Length) == 0) {
                REntry->Found = UNCHANGED;
                goto next;
            }
        MakeReview(Review->Name);
      next:;
    }
    for (REntry = ReviewBase.Next; REntry != &ReviewBase; REntry = Next) {
        Next = REntry->Next;
        switch(REntry->Found) {
          case DELETED:
            DestroyReview(REntry);
            break;
          case NEW:
            DisplayReview(REntry);
            break;
          default:
            break;
        }
    }
    TryReview();
}

static void ReviewCleanup(void)
{
    NumVal          *Temp1;
    NumNameListList *Temp2;

    if (!Reviewed)
        Raise1(AssertException, "Stop review while not reviewing !?");

    ReviewNode = REVIEWNONODE;
    Reviewed   = NULL;

    if (ReviewGlobal && (Temp1 = ReviewGlobal->Next) != ReviewGlobal) {
        Temp1->Previous = ReviewGlobal->Previous;
        Temp1->Previous->Next = Temp1;
        FreeNumValList(Temp1);
        ReviewGlobal->Previous = ReviewGlobal->Next = ReviewGlobal;
    }
    if (ReviewLocal && (Temp2 = ReviewLocal->Next) != ReviewLocal) {
        Temp2->Previous = ReviewLocal->Previous;
        Temp2->Previous->Next = Temp2;
        FreeNumNameListList(Temp2);
        ReviewLocal->Previous = ReviewLocal->Next = ReviewLocal;
    }
}

void UnReview(Connection conn)
{
    if (Reviewed) {
        DestroyObservers(&Reviewed->Observers);
        if (Reviewed->Log) {
	    FreeGamelog(Reviewed->Log);
            Reviewed->Log = NULL;
        }
        ReviewCleanup();
    }
}

void ReviewNotFound(void)
{
    UnReview(NULL);
    UserSendCommand(NULL, NULL, "revie");
}

void ReviewStart(const char *Name)
{
    Game    *REntry;
    int      Length;
    char    *NewTitle;

    ReviewNode = 0;
    DropVariations = 0;
    LastCommand(NULL, "forward");
    for (REntry = ReviewBase.Next; REntry != &ReviewBase;
         REntry = REntry->Next) {
        Length = strlen(REntry->Title);
        if (strncmp(Name, REntry->Title, Length) == 0) {
            Reviewed = REntry;
            if (Name[Length]) {
                NewTitle = mystrdup(Name);
                myfree(REntry->Title);
                REntry->Title = NewTitle;
                REntry->Found = CHANGED;
            }
            goto found;
        }
    }
    REntry = MakeReview(Name);
    Reviewed = REntry;
  found:
    DisplayReview(REntry);
}

static void SpuriousReview(void)
{
    Warning("Received review information but as far as I know you are "
            "not reviewing a game. Trying to stop spurious review\n");
    ReviewEnd(2);
}

void ReviewEntryBegin(int Nr)
{
    if (ReviewNode >= 0) {
        Nr++;
        if (Nr >= ReviewNode) {
            ReviewNode = Nr+1;
            ReviewLocal->Num = ReviewNode-2;
        } else ReviewEnd(2);
    }
}

#define Opponent(Who) ((Who) == White ? Black :((Who) == Black ? White :Empty))

static void ReviewMove(Game *game, NumNameListList *MoveEntry)
{
    int       Nr, x, y, node, Color;
    Gamelog  *Log;
    char     *move;
    NameList *From, *Here;
    NumNameListList *Ptr;
    StoneList *Stones, *Stone;

    Log = game->Log;
    if (!Log) Raise1(AssertException, "Game does not have a gamelog");
    Nr  = 1+XTPOINTER_TO_INT(FindComment(Log, &MoveFun));
    node = NodeNumber(Log);
    AddComment(Log, &NextMoveFun, INT_TO_XTPOINTER(node+1));
    switch(MoveEntry->Num) {
      case retBLACK:
        Color = Black;
        goto simpleMove;
      case retWHITE:
        Color = White;
      simpleMove:
        move = MoveEntry->Value->Name;
        if (move[0] == 0 || move[1] == 0 || move[2] != 0) {
            GameMessage(game, "----------", "Invalid move %s ignored\n", move);
            goto done;
        }
        x = move[0]-'a';
        y = move[1]-'a';
        if (x == 19 && y == 19 && 19 <= game->XSize && 19 <= game->YSize ||
            x == game->XSize && y == game->YSize) {
            DoPass(Log, Color);
            AddComment(Log, &LastMoveFun, LastFromXY(game, -1, 0));
            GameMessage(game, "..........", "%s passed",
                        Color == Black ? "Black" : "White");
        } else if (0>x || x >= game->XSize || 0>y || y >= game->YSize) {
            GameMessage(game, "----------", "Invalid move %s ignored\n", move);
            goto done;
        } else {
            y = game->YSize-y-1;
            DoMove(Log, x, y, Color);
            AddComment(Log, &LastMoveFun, LastFromXY(game, x, y));
        }
        break;
      case retBLACKSET:
      case retWHITESET:
      case retEMPTYSET:
        Stones = NULL;

        MoveEntry->Previous->Next = NULL;
        for (Ptr = MoveEntry; Ptr; Ptr = Ptr->Next) {
            switch(Ptr->Num) {
              case retBLACKSET: Color = Black; break;
              case retWHITESET: Color = White; break;
              case retEMPTYSET: Color = Empty; break;
              default:
                Raise1(AssertException, "impossibble color type");
                Color = Empty;
                break;
            }
            From = Ptr->Value;
            for (Here = From->Next; Here != From; Here = Here->Next) {
                move = Here->Name;
                if (move[0] == 0 || move[1] == 0 || move[2] != 0) {
                    Warning("Invalid stone %s ignored\n", move);
                    goto done;
                }
                x = move[0]-'a';
                y = move[1]-'a';
                if (0>x || x >= game->XSize || 0>y || y >= game->YSize) {
                    Warning("Invalid stone %s ignored\n", move);
                    goto done;
                }
                Stone = mynew(StoneList);
                Stone->x = x;
                Stone->y = game->YSize-1-y;
                Stone->Color = Color;
                Stone->Next  = Stones;
                Stones = Stone;
            }
        }
        SetStones(Log, Stones);
        FreeStones(Stones);
        AddComment(Log, &LastMoveFun, LastFromXY(game, -1, -1));
        Color = Empty;
        break;
      default:
        Raise1(AssertException, "ReviewwMove called on invalid move type");
        Color = Empty;
    }
    AddComment(Log, &PrevMoveFun, INT_TO_XTPOINTER(node));
    AddComment(Log, &MoveFun,     INT_TO_XTPOINTER(Nr));
    game->Move   = Nr;
    game->ToMove = Opponent(Color);
    ShowObserve(game);
  done:
    FreeNumNameListList(MoveEntry);
}

static void FlushNode(void)
{
    Gamelog         *log;
    NumNameListList *Entry, *MoveEntry, *Next;
    NameList        *Entries, *Here;
    int              i, nodes;
    char           **Text, **Ptr;
    char            *From, *To, *End;

    if (Reviewed) {
        if (ReviewLocal->Next != ReviewLocal) {
            log = Reviewed->Log;
            if (log) {
                nodes = NumberNodes(log)-1;
                for (i=NodeNumber(log); i<nodes; i++) DownGamelog(log);
                MoveEntry = NULL;
                for (Entry = ReviewLocal->Next; Entry != ReviewLocal;
                     Entry = Next) {
                    Next = Entry->Next;
                    switch(Entry->Num) {
                      case retBLACK:
                      case retWHITE:
                        if (MoveEntry) {
                            GameMessage(Reviewed, "----------", "Spurious move"
                                        " entry %s just before node %d\n",
                                        Entry->Value->Name, ReviewLocal->Num);
                            ReviewMove(Reviewed, MoveEntry);
                        }
                        Entry->Previous->Next = Next;
                        Entry->Next->Previous = Entry->Previous;
                        MoveEntry = Entry;
                        MoveEntry->Previous = MoveEntry->Next = MoveEntry;
                        break;
                      case retBLACKSET:
                      case retWHITESET:
                      case retEMPTYSET:
                        if (MoveEntry &&
                            MoveEntry->Num != retBLACKSET &&
                            MoveEntry->Num != retWHITESET &&
                            MoveEntry->Num != retEMPTYSET) {
                            GameMessage(Reviewed, "----------", "Spurious move"
                                        " entry %s just before node %d\n",
                                        Entry->Value->Name, ReviewLocal->Num);
                            ReviewMove(Reviewed, MoveEntry);
                            MoveEntry = NULL;
                        }
                        Entry->Previous->Next = Next;
                        Entry->Next->Previous = Entry->Previous;
                        if (MoveEntry) {
                            Entry->Next = MoveEntry;
                            Entry->Previous = MoveEntry->Previous;
                            Entry->Previous->Next = Entry->Next->Previous =
                                Entry;
                        } else {
                            MoveEntry = Entry;
                            MoveEntry->Previous = MoveEntry->Next = MoveEntry;
                        }
                        break;
                    }
                }
                if (MoveEntry) ReviewMove(Reviewed, MoveEntry);
                while ((Entry = ReviewLocal->Next) != ReviewLocal) {
                    switch(Entry->Num) {
                      case retCOMMENT:
                        End = strchr(Entry->Value->Name, 0);
/* Depends on what the sprintf routines in GameMessage can handle */
#define MAXBUF 1000 
                        if (End > Entry->Value->Name+MAXBUF) {
                            for (From = Entry->Value->Name;
                                 (To = memchr(From, '\n', End-From)) != NULL;
                                 From = To+1) {
                                *To = 0;
                                GameMessage(Reviewed, From, "");
                                *To = '\n';
                            }
                            GameMessage(Reviewed, From, "");
                        } else GameMessage(Reviewed, Entry->Value->Name, "");
                        break;
                      case retBLACKTIME:
                        AddLocalProperty(log, "TimeLeftBlack",
                                         Entry->Value->Name, -1);
                        break;
                      case retWHITETIME:
                        AddLocalProperty(log, "TimeLeftWhite",
                                         Entry->Value->Name, -1);
                        break;
                      case retLETTERS:
                        Entries = Entry->Value;
                        i = 0;
                        for (Here = Entries->Next; Here != Entries;
                             Here = Here->Next) i++;
                        Ptr = Text = mynews(char *, i);
                        WITH_UNWIND {
                            for (Here = Entries->Next; Here != Entries;
                                 Here = Here->Next, Ptr++) *Ptr = Here->Name;
                            AddLocalProperties(log, "PositionLetters", i,
                                               (const char **) Text, NULL);
                        } ON_UNWIND {
                            myfree(Text);
                        } END_UNWIND;
                        break;
                    }
                    FreeNameList(Entry->Value);
                    Entry->Previous->Next = Entry->Next;
                    Entry->Next->Previous = Entry->Previous;
                    myfree(Entry);
                }
            } else {
                GameMessage(Reviewed, "----------",
                            "Throwing away review entries\n");
                while ((Entry = ReviewLocal->Next) != ReviewLocal) {
                    FreeNameList(Entry->Value);
                    Entry->Previous->Next = Entry->Next;
                    Entry->Next->Previous = Entry->Previous;
                    myfree(Entry);
                }
            }
        }
    } else SpuriousReview();
}

void ReviewNewNode(void)
{
    if (!DropVariations) FlushNode();
}

void ReviewOpenVariation(void)
{
    /* Output("Open variation\n"); */
}

void ReviewCloseVariation(void)
{
    if (!Reviewed)
        Raise1(AssertException, "close variation while not reviewing !?");
    if (!DropVariations) {
        FlushNode();
        DropVariations = 1;
        GameMessage(Reviewed, "----------", "Ignoring variations");
        ReviewEnd(2); /* Non zero means stop reviewing */
    }
    /* Output("Close variation\n"); */
}

void ReviewGlobalProperty(int Num, const char *Value)
{
    Gamelog    *Log;
    NumVal     *Entry;
    int         Temp;
    char        Buffer[80];
    char       *ptr;

    if (Reviewed) {
        Log = Reviewed->Log;
        if (Log)
#define AddProp(Name, Value) AddGlobalProperty(Log, Name, Value, -1)
            switch(Num) {
              case retGAME:
                Temp = atoi(Value);
                if (Temp != SGFGO) {
                    GameMessage(Reviewed, "----------", "This turns out not to"
                                "be a game of go, but a game of the "
                                "unsupported type %d\n", Temp);
                    ReviewEnd(2); /* non zero means stop reviewing */
                }
                break;
              case retNODENAME:
                AddProp("NodeName", Value);
                break;
              case retKOMI:
                strtol(Value, &ptr, 10);
                if (*ptr == '.') {
                    strtol(ptr+1, &ptr, 10);
                    if (*--ptr == '0') {
                        do {
                            ptr--;
                        } while (*ptr == '0');
                        Temp = (const char *) ptr - Value + 1;
                        if (Temp < sizeof(Buffer)) {
                            sprintf(Buffer, "%.*s", Temp, Value);
                            Value = Buffer;
                        }
                    }
                }
                CHANGESTRING(Reviewed, Komi, Value,
                             AddGlobalProperty(Log, "Komi", Value, -1);
                             SetKomi(Reviewed->Observers, Value));
                break;
              case retHANDICAP:
                Temp = atoi(Value);
                CHANGEINT(Reviewed, Handicap, Temp, PROPHANDICAP(Temp);
                          SetHandicap(Reviewed->Observers, Temp));
                break;
              case retENTEREDBY:
                AddProp("EnteredBy", Value);
                break; 
              case retCOPYRIGHT:
                AddProp("Copyright", Value);
                break;
              case retPLACE:
                AddProp("Place", Value);
                break; 
              case retDATE:
                AddProp("Date", Value);
                break; 
              case retRESULT:
                AddProp("Result", Value);
                break; 
              case retTOURNAMENT:
                AddProp("Tournament", Value);
                break; 
              case retNAME:
                AddProp("Name", Value);
                break; 
              case retSIZE:
                AddProp("Size", Value);
                break;
              case retWHITESTRENGTH:
                AddProp("WhiteStrength", Value);
                CheckPlayerStrength(Reviewed->White, Value);
                break; 
              case retBLACKSTRENGTH:
                AddProp("BlackStrength", Value);
                CheckPlayerStrength(Reviewed->Black, Value);
                break; 
              case retBLACKNAME:
                AddProp("BlackName", Value);
                RenameDummyPlayer(Reviewed->Black, Value);
                SetObserveDescriptions(Reviewed->Observers, Reviewed);
                break; 
              case retWHITENAME:
                AddProp("WhiteName", Value);
                RenameDummyPlayer(Reviewed->White, Value);
                SetObserveDescriptions(Reviewed->Observers, Reviewed);
                break;
              default:
                sprintf(Buffer, "Invalid global property %d", Num);
                Raise1(AssertException, ExceptionCopy(Buffer));
                break;
            }
#undef AddProp
        else {
            if (Num == retGAME) {
                Temp = atoi(Value);
                if (Temp != SGFGO) {
                    Warning("Review %s is not a game of go, but a game of the"
                            " unsupported type %d\n", Reviewed->Title, Temp);
                    ReviewEnd(2); /* non zero means stop reviewing */
                }
            } else if (Num == retSIZE) {
                Temp = atoi(Value);
                Reviewed->XSize = Reviewed->YSize = Temp;
                Log = AllocGamelog(Reviewed->XSize, Reviewed->YSize);
                AddComment(Log, &MoveFun,     INT_TO_XTPOINTER(-1));
                AddComment(Log, &NextMoveFun, INT_TO_XTPOINTER(1));
                SetStones(Log, NULL);
                AddComment(Log, &LastMoveFun, LastFromXY(Reviewed, -1, -1));
                AddComment(Log, &PrevMoveFun, INT_TO_XTPOINTER(0));
                AddComment(Log, &MoveFun,     INT_TO_XTPOINTER(0));
                Reviewed->Log = Log;
                Reviewed->White = MakeDummyPlayer();
                Reviewed->Black = MakeDummyPlayer();
                while ((Entry = ReviewGlobal->Next) != ReviewGlobal) {
                    ReviewGlobalProperty(Entry->Num, Entry->Value);
                    myfree(Entry->Value);
                    Entry->Previous->Next = Entry->Next;
                    Entry->Next->Previous = Entry->Previous;
                    myfree(Entry);
                }
                ShowObserve(Reviewed);
            } else {
                Entry = mynew(NumVal);
                Entry->Num   = Num;
                Entry->Value = mystrdup(Value);
                Entry->Next  = ReviewGlobal;
                Entry->Previous = ReviewGlobal->Previous;
                Entry->Previous->Next = Entry->Next->Previous = Entry;
            }
        }
    } else SpuriousReview();
}

void ReviewLocalProperty(int Name, const NameList *Value)
{
    NumNameListList *Entry;

    if (DropVariations) return;
    if (ReviewNode >= 0) {
        Entry = mynew(NumNameListList);
        Entry->Num   = Name;
        Entry->Value = NameListDup(Value);
        Entry->Next = ReviewLocal;
        Entry->Previous = ReviewLocal->Previous;
        Entry->Previous->Next = Entry->Next->Previous = Entry;
    }
}

void ReviewEnd(int Last)
{
    Gamelog *log;
    int      i, Nr, Pos;
    char    *Ptr;

    if (Last) {
        ReviewNode = REVIEWBADNODE;
        if (Reviewed) {
            Reviewed->Observed = 1;
            Reviewed->WantObserved = 0;
            Reviewed->Found = CHANGED;
            DisplayReview(Reviewed);
            log = Reviewed->Log;
            if (log) {
                Nr = GetGlobalProperty(log, "Result", -1, 0, NULL, NULL, 0);
                for (i=0; i<Nr; i++) {
                    Pos = 0;
                    Ptr = NULL;
                    GetGlobalProperty(log, "Result", 0, 1, &Pos, &Ptr, 1);
                    GameMessage(Reviewed, "----------", "%s", Ptr);
                }
            }
        }
        UserSendCommand(NULL, NULL, "revie");
    } else if (ReviewNode >= 0) LastCommand(NULL, "forward");
}

void ReviewStop(void)
{
    int Temp;

    if (!Reviewed)
        Raise1(AssertException, "Stop review while not reviewing !?");

    if (Reviewed->WantObserved) {
        Reviewed->WantObserved = 0;
        Reviewed->Found = CHANGED;
    }
    DisplayReview(Reviewed);
    if (Reviewed->Observed) ReviewCleanup();
    else                    UnReview(NULL);
    if (WantReviews) {
        Temp = WantReviews;
        WantReviews = 0;
        ReviewListWanted(NULL, Temp);
    }
    TryReview();
}

void ReviewListWanted(Connection Conn, int Type)
{
    if (Reviewed) {
        if (Type > WantReviews) WantReviews = Type;
    } else if (Type > 1) UserCommand(Conn, "review");
    else                 SendCommand(NULL, NULL, "review");
}

void CheckReview(Connection Conn)
{
    if (Reviewed && ReviewNode != REVIEWBADNODE) {
        /* There should be a review or forward command pending */
        Warning("Something went wrong while reviewing %s. Trying to end it\n",
                Reviewed->Title);
        ReviewEnd(2);
    }
}

static void DoReview(Widget w, XEvent *evnt, String *str, Cardinal *n)
{
    if (*n) XtCallCallbacks(w, XtNcallback, (XtPointer) str[0]);
    else    XtCallCallbacks(w, XtNcallback, (XtPointer) "none");
}

static void CallReview(Widget w, XtPointer clientdata, XtPointer calldata)
{
    Game    *review;
    Observe *observe;
    String   text;

    review = (Game *) clientdata;
    text   = (String) calldata;
    if (strcmp(text, GAMEKEY1) == 0 ||
        strcmp(text, GAMEKEY2) == 0 ||
        strcmp(text, GAMEKEY3) == 0)
        if (review->Log) {
            observe = OpenObserve(review);
            GotoObserve(observe, 1);
        } else WantReview(review);
    else if (strcmp(text, DUMPGAME) == 0) DumpGame(review);
    else WidgetWarning(w, "Unknown argument to doreview");
}

int ReviewP(const Game *game)
{
    return game->ServerId == REVIEWGAME;
}

void ReviewsTime(unsigned long diff)
{
    Game *game;

    for (game = ReviewBase.Next; game != &ReviewBase; game = game->Next)
        if (game->Log)
            ReplayTime(diff, game->Observers, NumberNodes(game->Log)-1);
}
