/*
 * plotps_2D.c - PostScript subroutines for drawing 2D plots
 */

#include <stdio.h>
#include <strings.h>
#include <math.h>
#include "plotps.h"
#include "CNplot.h"

#define EXTREMELY_SMALL 1.0e-99

/* Maximum no of points in an array */
#define MAX_ARR_SIZE  1000

/*
 * FORWARD DECLARATIONS
 */
void   PXplotps2D();
static void drawPSplot2D();
static void axesPS2D();
static void boundaryPS2D();
static void plotPS2D_xticks();
static void plotPS2D_yticks();
static void plotPS2D_Axticks();
static void plotPS2D_Ayticks();
static void plotPS2D_xticklabel();
static void plotPS2D_yticklabel();

static void plotPS2D();
static void plotPS2D_grid();
static void plotPS2D_vectors();
static void plotPS2D_boundary();
static void plotPS2D_trias();
static void plotPS2D_rects();
static void plotPS2D_single_mesh_tria();
static void plotPS2D_single_mesh_rect();
static void plotPS2D_single_fill_tria();
static void plotPS2D_single_fill_rect();
static void plotPS2D_nodes();
static void plotPS2D_dataset_curves();
static void plotPS2D_curve();
static void fillPS2D_curve();
static void plotPS2D_spline_curve();
static void plotPS2D_markers();

static void plotPS2D_pointIDs();
static void plotPS2D_nodeIDs();
static void plotPS2D_triaIDs();
static void plotPS2D_rectIDs();

static void annotatePS2D();
static void plotPS2D_single_annotation();
static void plotPS2D_annot_rect();
static void plotPS2D_annot_line();
static void plotPS2D_annot_arrow();
static void plotPS2D_arrow();
static void plotPS2D_annot_point();
static void plotPS2D_annot_text();
static void clipPS2D_in_xmin();
static void clipPS2D_in_ymin();
static void clipPS2D_in_xmax();
static void clipPS2D_in_ymax();

static void plotPS2D_contlabel();
static void drawPS2D_label();
static void plotPS2D_sidelabels();

static void trn_world_to_PS();
static void trn_world_to_PS_nolog();

/* Plot variables */
static short  grid;
static short  clip;
static short  xflip, yflip;
static short  xabs, yabs, zabs;
static short  xlog, ylog, zlog;
static short  xticks, yticks;
static short  xautorange, yautorange;
static double xyratio;
static double xscale, yscale;
static short  overlay;
static int    plotlabels;
static char   xlabel[MAXCHR];
static char   ylabel[MAXCHR];
static char   toplabel[MAXCHR];
static char   *comment;
static char   *subtitle;

/* Hiddenline - used to mirrow 3D hiddenline plot */
static short  hiddenline;

/* The plotset */
static CNplotsetptr plotdata=NULL;

/* contour steps */
static CNcontstepptr cstephead=NULL;
static CNcontstepptr csteptail=NULL;

/* Contour dptr */
static CNdatasetptr contour_dptr=NULL;

/* The real-world dimensions of the plot */
static double xmin, xmax, ymin, ymax, zmin, zmax;

/* PSplot dimensions */
static double  Pxmin, Pymin, Pxmax, Pymax;

/* Scaling factors */
static double idx, jdy;


/*
 * Draw the plot in PostScript
 */
void PXplotps2D(pdata)
CNplotsetptr pdata;                  /* The plot data is here   */
{
   void PXquery_contours();
   int  PXquery_labels();
   void PXcheck_viewport();
   void PXconvert_viewport_to_log();

   double lxmin, lxmax, lymin, lymax;
   double bxmin, bxmax, bymin, bymax, bzmin, bzmax;
   int    equalscale, fitpage;
   double Pxwid,Pywid,ratio,dx,dy;
   char   label[CN_MAXCHAR];

   /* Error checking */
   if (pdata==NULL) {
      (void) fprintf(stderr,"PXplotps2D(): Error! Cannot plot NULL plotset!\n");
      return;
   }

   /* Copy the variables to the static variables first */
   plotdata  = pdata; 

   /* Initialize some other plot variables */
   grid       = plotdata->plot_pr.grid;
   xflip      = plotdata->plot_pr.xflip;
   yflip      = plotdata->plot_pr.yflip;
   xabs       = plotdata->plot_pr.xabs;
   yabs       = plotdata->plot_pr.yabs;
   zabs       = CN_FALSE;
   xlog       = plotdata->plot_pr.xlog;
   ylog       = plotdata->plot_pr.ylog;
   zlog       = CN_FALSE;
   xticks     = plotdata->plot_pr.xticks;
   yticks     = plotdata->plot_pr.yticks;
   xautorange = plotdata->plot_pr.xautorange;
   yautorange = plotdata->plot_pr.yautorange;
   xyratio    = plotdata->plot_pr.xyratio;
   equalscale = plotdata->plot_pr.equalscale;
   fitpage    = plotdata->plot_pr.fitpage;
   xscale     = plotdata->plot_pr.xscale; if (xscale <= 0) xscale = 1.0;
   yscale     = plotdata->plot_pr.yscale; if (yscale <= 0) yscale = 1.0;
   overlay    = plotdata->plot_pr.overlay;
   clip       = CN_TRUE;
   hiddenline = plotdata->view_pr->hiddenline;
   plotlabels = PXquery_labels(plotdata);

   /* xlabel */
   if (plotdata->plot_pr.xlabel == NULL)
      (void) strcpy(xlabel   , CN_DEF_XLABEL);
   else
      (void) strcpy(xlabel   , plotdata->plot_pr.xlabel);
   if (xscale != 1.0) {
      (void) sprintf(label," (x %g)",xscale);
      (void) strcat(xlabel, label);
   }
 
   /* ylabel */
   if (plotdata->plot_pr.ylabel == NULL)
      (void) strcpy(ylabel   , CN_DEF_YLABEL);
   else
      (void) strcpy(ylabel   , plotdata->plot_pr.ylabel);
   if (yscale != 1.0) {
      (void) sprintf(label," (x %g)",yscale);
      (void) strcat(ylabel, label);
   }
 
   /* Toplabel */
   if (plotdata->plot_pr.toplabel == NULL)
      (void) strcpy(toplabel , CN_DEF_TLABEL);
   else
      (void) strcpy(toplabel , plotdata->plot_pr.toplabel);

    
   /* Comment & subtitle */
   comment = plotdata->plot_pr.comment;
   subtitle= plotdata->plot_pr.subtitle;

   /*
    * The plot boundaries of the plot are given by P->plot_pr->vxmin, etc,
    * which we define as the "viewport" in world coordinates.
    * This is what the user should see, i.e. a plot bounded by vxmin,vxmax.
    *
    * The plotset however has another set of boundaries given by the
    * overlap of all the datasets contained within the plotset.
    * This boundary is used only if the viewport is in error.
    * For example if vxmin=vxmax, then vxmin<=bxmin, vxmax<=bxmax.
    */

   /*
    * Find the plotset boundary from its component datasets
    */
   /* Reset the plot boundaries */
   CNset_plotset_boundaries(plotdata,
                            &bxmin, &bxmax, &bymin, &bymax, &bzmin, &bzmax);

   /* The viewport */
   xmin = plotdata->plot_pr.vxmin;
   xmax = plotdata->plot_pr.vxmax;
   ymin = plotdata->plot_pr.vymin;
   ymax = plotdata->plot_pr.vymax;
   zmin = plotdata->plot_pr.vzmin;
   zmax = plotdata->plot_pr.vzmax;

   zmin = bzmin;
   zmax = bzmax;

   /*
    * Check and fix the viewport
    */
   PXcheck_viewport(&xmin,&xmax,bxmin,bxmax,&xlog,xabs,"x");
   plotdata->plot_pr.vxmax = xmax;
   plotdata->plot_pr.vxmin = xmin;
   plotdata->plot_pr.xlog  = xlog;

   PXcheck_viewport(&ymin,&ymax,bymin,bymax,&ylog,yabs,"y");
   plotdata->plot_pr.vymax = ymax;
   plotdata->plot_pr.vymin = ymin;
   plotdata->plot_pr.ylog  = ylog;

   /*
    * If log axes then convert the boundary values.
    * The min and max boundaries are converted to log-values.
    */
   PXconvert_viewport_to_log(&lxmin, &lxmax, xmin, xmax, xlog, xautorange);
   PXconvert_viewport_to_log(&lymin, &lymax, ymin, ymax, ylog, yautorange);
   xmin = lxmin;
   xmax = lxmax;
   ymin = lymin;
   ymax = lymax;

   /* Save the plot viewport */
   plotdata->plot_pr.pxmin = xmin;
   plotdata->plot_pr.pxmax = xmax;
   plotdata->plot_pr.pymin = ymin;
   plotdata->plot_pr.pymax = ymax;

   /* Equal-scale doen't make sense for a non-linear plot */
   if (((xlog && !ylog) || (!xlog && ylog)) && equalscale) {
      equalscale = CN_FALSE;
      fitpage    = CN_TRUE;
   }

   /* Initialize - Pgxmin, Pgxmax, Pgymin, Pgymax are page dimensions */
   Pxmin = Pgxmin + 1.0*scale*PBDR_DIM;
   Pxmax = Pgxmax - 1.0*scale*PBDR_DIM;
   Pymin = Pgymin + 1.0*scale*PBDR_DIM;
   Pymax = Pgymax - 1.2*scale*PBDR_DIM; /* Leave a bit of room for labels */
   Pxwid = Pxmax - Pxmin;
   Pywid = Pymax - Pymin;
   if (Pxwid < 0) {                     /* Very unlikely to happen... */
      Pxmin = Pgxmin;
      Pxmax = Pgxmax;
      Pxwid = Pxmax - Pxmin;
   }
   if (Pywid < 0) {                     /* Very unlikely to happen... */
      Pymin = Pgymin;
      Pymax = Pgymax;
      Pywid = Pymax - Pymin;
   }

   if (equalscale) {
      /* Fit the plot inside the window */
      dx = xmax - xmin;
      dy = ymax - ymin;
      ratio = dy/dx;
      if (ratio <= (Pywid/Pxwid)) {
         /* decrease Pywid : Pywid  = Pxwid*ratio */
         Pymax = Pymax - 0.5*(Pywid - Pxwid*ratio);
         Pymin = Pymin + 0.5*(Pywid - Pxwid*ratio);
      } else {
         /* decrease xwid : Pxwid  = Pywid/ratio */
         Pxmax = Pxmax - 0.5*(Pxwid - Pywid/ratio);
         Pxmin = Pxmin + 0.5*(Pxwid - Pywid/ratio);
         Pxmax = Pxmin + Pywid/ratio;
      }

   /*EMPTY*/
   } else if (fitpage) {
      /* The plot fills the whole page */
      ;

   } else {
      /* The plot must fit inside the window */
      if (Pxwid*xyratio < Pywid) {
         /* decrease Pywid : Pywid  = Pxwid*xyratio */
         Pymax = Pymax - 0.5*(Pywid - Pxwid*xyratio);
         Pymin = Pymin + 0.5*(Pywid - Pxwid*xyratio);
      } else {
         /* decrease Pxwid : Pxwid  = Pywid/ratio */
         Pxmax = Pxmax - 0.5*(Pxwid - Pywid/xyratio);
         Pxmin = Pxmin + 0.5*(Pxwid - Pywid/xyratio);
         Pxmax = Pxmin + Pywid/xyratio;
      }
   }

   /* initialize the scaling factors */
   dx = xmax - xmin;
   dy = ymax - ymin;
   idx = (Pxmax-Pxmin)/dx;
   jdy = (Pymax-Pymin)/dy; 

   /* 
    * Now do the PostScript plot 
    */
   drawPSplot2D();

}

/*
 * Draw the plot in POSTSCRIPT
 */
static void drawPSplot2D()
{
   int    ltmp=6;

   /* Draw the RAW Postscript plot */

   /* Set contour levels for the contour-type datasets */
   cstephead = NULL;
   csteptail = NULL;
   PXquery_contours(plotdata,&cstephead,&csteptail,&contour_dptr);

   /* Draw the axes */
   axesPS2D();

   /* clip if necessary */
   if (clip) (void) fprintf(ips,"gsave\n");
   if (clip) (void) fprintf(ips,"rectanglepath clip\n");

   /* Draw the plot */
   plotPS2D();

   /* Redraw the boundary */
   boundaryPS2D(&ltmp);

   /* Unclip */
   if (clip) (void) fprintf(ips,"grestore\n");

   /* Draw the annotations */
   annotatePS2D();

   /* Delete contour step-levels */
   CNdelete_contstep_list(&cstephead, &csteptail);
}

/*
 * draw the boundary, axes and labels of the plot and initialize its size
 */
static void axesPS2D()
{
   int    dist, label_len;

   (void) fprintf(ips,"%%------------------------------------------------\n");
   (void) fprintf(ips,"%%--------------Draw the Axes---------------------\n");
   (void) fprintf(ips,"%%------------------------------------------------\n");

   /* define the axes */
   (void) fprintf(ips,"/rectanglepath {\n");
   (void) fprintf(ips,"newpath\n");
   (void) fprintf(ips,"%.2f %.2f moveto\n",Pxmin,Pymin);
   (void) fprintf(ips,"0.0 %.2f rlineto\n",Pymax-Pymin);
   (void) fprintf(ips,"%.2f 0.0 rlineto\n",Pxmax-Pxmin);
   (void) fprintf(ips,"0.0 %.2f rlineto\n",Pymin-Pymax);
   (void) fprintf(ips,"closepath } def\n");

   /* Draw the boundary and tick marks */
   boundaryPS2D(&label_len);

   /* Don't redraw the grid */
   if (grid) grid = CN_FALSE;

   (void) fprintf(ips,"%%------------------------------------------------\n");
   (void) fprintf(ips,"%%--------------Draw the Labels-------------------\n");
   (void) fprintf(ips,"%%------------------------------------------------\n");

   /* Top Label */
   (void) fprintf(ips,"setTopLblFont\n");
   (void) fprintf(ips,"%.2f %.2f moveto ",0.5*(Pxmax+Pxmin),Pymax+30*fscale);
   (void) fprintf(ips,"(%s) UPJ CEJ show\n",PXmodifyPS_string(toplabel));

   /* X-axis Label */
   (void) fprintf(ips,"setSideLblFont\n");
   (void) fprintf(ips,"%.2f %.2f moveto ",0.5*(Pxmax+Pxmin),Pymin-20*fscale);
   (void) fprintf(ips,"(%s) BOJS CEJ show\n",PXmodifyPS_string(xlabel));

   /* Y-axis Label */
   (void) fprintf(ips,"setSideLblFont\n");
   (void) fprintf(ips,"/Ylabel {");
   dist = 10*label_len + 10;
   if (dist > 65) dist = 65;
   (void) fprintf(ips,"%.2f %.2f moveto ",0.5*(Pymax+Pymin),
           dist*fscale-Pxmin);
   (void) fprintf(ips,"(%s) UPJ CEJ show ",PXmodifyPS_string(ylabel));
   (void) fprintf(ips,"} def\n90 rotate Ylabel -90 rotate\n"); 

   /* Comment */
   if (comment != NULL) {
   (void) fprintf(ips,"setDateLblFont\n");
   (void) fprintf(ips,"%.2f %.2f moveto ",
                  Pgxmax-0.3*scale*PBDR_DIM,Pgymax-0.4*scale*PBDR_DIM);
   (void) fprintf(ips,"(%s) UPJ RIJ show\n",PXmodifyPS_string(comment));
   }

   /* Subtitle */
   if (subtitle != NULL) {
   (void) fprintf(ips,"setAxisLblFont\n");
   (void) fprintf(ips,"%.2f %.2f moveto ",0.5*(Pxmax+Pxmin),Pymax+9*fscale);
   (void) fprintf(ips,"(%s) UPJ CEJ show\n",PXmodifyPS_string(subtitle));
   }

   /* Draw the plot labels */
   if (plotlabels) plotPS2D_sidelabels();

   /* Reset the linewidth */
   (void) fprintf(ips,"%.2f setlinewidth\n",0.5*scale);
}

/*
 * Draw the boundary and tick marks of the plot.
 * This is done last 'cos the axes might have been
 * obscured during the course of the plot.
 * The label-length controls the placement of the y-axis label.
 * This routine is called twice - once by axes() and another after plot().
 */
static void boundaryPS2D(label_len)
int  *label_len;
{
   /* 
    * The label-length controls the placement of the y-axis labels 
    */
   (void) fprintf(ips,"%%------------------------------------------------\n");
   (void) fprintf(ips,"%%--------------Draw the TickMarks----------------\n");
   (void) fprintf(ips,"%%------------------------------------------------\n");

   /* Reset color */
   PXsetColorPS(0);

   /* define the axes */
   (void) fprintf(ips,"%g setlinewidth rectanglepath stroke\n",2*scale);

   /* Draw the tick marks and labels */
   (void) fprintf(ips,"%g setlinewidth\n",1*scale);
   *label_len = 6;
   if (!xautorange && !xlog) {
      /* Linear, fixed tickmarks */
      plotPS2D_xticks();
   } else {
      /* If xlog then do lots of tiny tickmarks */
      plotPS2D_Axticks();
   }

   if (!yautorange && !ylog) {
      /* Linear, fixed tickmarks */
      plotPS2D_yticks(label_len);
   } else {
      /* If ylog then do lots of tiny tickmarks */
      plotPS2D_Ayticks(label_len);
   }
}

/*
 * Draw the x-axis tick marks and labels
 * This is NEVER used with log-scale
 */
static void plotPS2D_xticks()
{
   double xdx,xintv;
   double Pxtmp,Pytmp;
   double vallbl;
   int    i;

   /* Useful data */
   xintv = (Pxmax-Pxmin)/(double)xticks;
   xdx   = (xmax-xmin)/(double)xticks;

   /* Tick Marks on the X-axis */
   (void) fprintf(ips,"%% Draw the X-Axis\n");
   (void) fprintf(ips,"setAxisLblFont\n");
   for (i=0; i<=xticks; i++) {
      Pxtmp = Pxmin + xintv*i;
      Pytmp = Pymin;
      (void) fprintf(ips,"%.2f %.2f ",Pxtmp,Pytmp);
      (void) fprintf(ips,"0 %.2f drawline\n",-7*scale);

      vallbl = xmin  + i*xdx;
      if (xflip) vallbl = xmax - i*xdx;
      plotPS2D_xticklabel(vallbl,Pxtmp);

      Pytmp = Pymax;
      (void) fprintf(ips,"%.2f %.2f ",Pxtmp,Pytmp);
      (void) fprintf(ips,"0 %.2f drawline\n",-7*scale);
   }

   /* draw the grid */
   if (grid == CN_TRUE) {
      for (i=1; i<xticks; i++) {
         Pxtmp = Pxmin + xintv*i;
         Pytmp = Pymax;
         (void) fprintf(ips,"[] 1 setdash  ");
         (void) fprintf(ips,"%.2f setlinewidth 0.9 setgray\n",0.5);
         (void) fprintf(ips,"%.2f %.2f ",Pxtmp,Pytmp);
         (void) fprintf(ips,"0 %.2f drawline\n",Pymin-Pymax);
         (void) fprintf(ips,"%.2f setlinewidth 0.0 setgray\n",1.0*scale);
         (void) PXlinetypPS(CN_LN_SOLID,2);
      }
   }
}

/*
 * Draw the y-axis tick marks and labels
 * This is NEVER used with log-scale
 */
static void plotPS2D_yticks(label_len)
int  *label_len;
{
   double ydy,yintv;
   double Pxtmp,Pytmp;
   double vallbl;
   int    i, llen = 0;

   /* Useful data */
   yintv = (Pymax-Pymin)/(double)yticks;
   ydy   = (ymax-ymin)/(double)yticks;

   /* Tick Marks on the Y-axis */
   (void) fprintf(ips,"%% Draw the Y-Axis\n");
   (void) fprintf(ips,"setAxisLblFont\n");
   for (i=0; i<=yticks; i++) {
      Pytmp = Pymin + yintv*i;
      Pxtmp = Pxmin;
      (void) fprintf(ips,"%.2f %.2f ",Pxtmp,Pytmp);
      (void) fprintf(ips,"%.2f 0 drawline\n",-7*scale);

      vallbl = ymin  + i*ydy;
      if (yflip) vallbl = ymax - i*ydy;
      plotPS2D_yticklabel(vallbl,Pytmp,&llen);

      Pxtmp = Pxmax;
      (void) fprintf(ips,"%.2f %.2f ",Pxtmp,Pytmp);
      (void) fprintf(ips,"%.2f 0 drawline\n",-7*scale);
   }

   /* return the label length */
   if (llen < *label_len) *label_len = llen;

   /* draw the grid */
   if (grid == CN_TRUE) {
      for (i=1; i<yticks; i++) {
         Pytmp = Pymin + yintv*i;
         Pxtmp = Pxmax;
         (void) fprintf(ips,"[] 1 setdash  ");
         (void) fprintf(ips,"%.2f setlinewidth 0.9 setgray\n",0.5);
         (void) fprintf(ips,"%.2f %.2f ",Pxtmp,Pytmp);
         (void) fprintf(ips,"%.2f 0 drawline\n",Pxmin-Pxmax);
         (void) fprintf(ips,"%.2f setlinewidth 0.0 setgray\n",1.0*scale);
         (void) PXlinetypPS(CN_LN_SOLID,1);
      }
   }
}


/* 
 * Draw the x-axis tick marks and labels with automatic ranging 
 */
static void plotPS2D_Axticks()
{
   void   PXget_autorange();

   double dmin, dmax, dmin2, dmax2;
   double dtmp, dt, delta;
   double Pxtmp;
   double first_tick, last_tick;   /* To keep track of labels */
   int    i, itick, nticks, tck;

   /*
    * LINEAR SCALE
    * Start with a linear range (e.g. xmin = 0.05  xmax = 50.0)
    * Autoranging              =>     dmin = 0.05  dmax = 50.0
    *                                 dmin2= 0.0   dmax2= 50.0
    *                                 delta= 10.0
    *               => get scales at  x=0,10,20,30,40,50
    *
    * LOG SCALES
    * Start with a linear range (e.g. xmin = 0.05  xmax = 50.0)
    * For log scale, these numbers are converted first to logarithmic values
    * during initialization     (e.g. xmin = -1.3  xmax = 1.7  )
    * The autoranging forces =>       dmin = -1.3  dmax = 1.7
    *                                 dmin2= -2.0  dmax2= 2.0
    *                                 delta=  1.0
    *               => get scales at  x=-2,-1,0,1,2
    * So all I have to do is plot the scales in log10 increments
    */

   /*
    * This does not work too well if (xmax-xmin) << xmin
    * because of floating point errors.
    */

   /* 
    * useful data 
    * X = Pxmin + (x-xmin)*(Pxmax-Pxmin)/(xmax-xmin)
    * Y = Pymin + (y-ymin)*(Pymax-Pymin)/(ymax-ymin)
    */
   
   /* Tick Marks on the X-axis */
   (void) fprintf(ips,"%% Draw the X-Axis\n");
   (void) fprintf(ips,"setAxisLblFont\n");

   /* Get the rounded intervals */
   PXget_autorange(xmin,xmax,
                  &dmin,&dmax,&dmin2,&dmax2,&delta,xlog,xautorange);

   /* If dmin2=dmin then subtract delta from dmin2 so that dmin2 < dmin */
   /* This is used to get the leftmost tick label on the plot           */
   if (dmin2==dmin) dmin2 -= delta;

   /* Initialize first/last tick positions */
   first_tick = Pxmax;   /* Should finally be a bit greater than Pxmin */
   last_tick  = Pxmin;   /* Should finally be a bit less than Pxmax */

   /* Draw the tickmarks from dmin2 to dmax2 - dtmp is distance from dmin2 */
   if (delta <= 0.0) nticks = 0;
   else              nticks = (int)((dmax2 - dmin2)/delta) + 1;
   for (itick=0; itick<nticks; itick++) {
      dtmp = itick*delta;
      for (i=1; i<=10; i++) {
         /* distance from dtmp - actual phy loc is dt+dtmp+dmin2 */
         dt = i*0.1*delta;
         if (xlog) dt = log10((double)i); 
         if (xlog && i==1) continue;  /* Don't do this case */

         /* Translate to plot-coordinates */
         Pxtmp = Pxmin + ((dtmp+(dmin2-dmin))*idx) + (dt*idx);
         if (xflip)
         Pxtmp = Pxmax - ((dtmp+(dmin2-dmin))*idx) - (dt*idx);

         /* Don't draw if this is outside the plot boundaries */
         if (Pxtmp < Pxmin-0.01*scale || Pxtmp > Pxmax+0.01*scale) continue;

         /* Draw the grid */
         if (grid && (Pxtmp > Pxmin+0.01*scale && Pxtmp < Pxmax-0.01*scale)) {
            /*
             * PS major grid is the same as the PS minor grid 
             */
            if (i==10) {
               /* Major grid */
               (void) fprintf(ips,"%.2f setlinewidth 0.9 setgray\n",0.5);
               (void) fprintf(ips,"%.2f %.2f ",Pxtmp,Pymin);
               (void) fprintf(ips,"0 %.2f drawline\n",Pymax-Pymin);
               (void) fprintf(ips,"%.2f setlinewidth 0.0 setgray\n",1.0*scale);
            } else if (scale > 0.4) {
               /* Minor grid - don't draw the minor grid for small plots */
               (void) fprintf(ips,"%.2f setlinewidth 0.9 setgray\n",0.5);
               (void) fprintf(ips,"%.2f %.2f ",Pxtmp,Pymin);
               (void) fprintf(ips,"0 %.2f drawline\n",Pymax-Pymin);
               (void) fprintf(ips,"%.2f setlinewidth 0.0 setgray\n",1.0*scale);
            }
         }

         /* The length of the tick depends on i */
         if      (i==10) tck = 7;  /* major tick */
         else if (i==5 ) tck = 5;  /* major sub-tick */
         else            tck = 3;  /* minor tick */

         /* Draw the ticks */
         (void) fprintf(ips,"%.2f %.2f ",Pxtmp,Pymin);
         (void) fprintf(ips,"0 %.2f drawline\n",-tck*scale);
         (void) fprintf(ips,"%.2f %.2f ",Pxtmp,Pymax);
         (void) fprintf(ips,"0 %.2f drawline\n",-tck*scale);

         /* Draw labels */
         if (i==10) {
            /* If x is very close to 0 print 0 */
            if (fabs((dt+dtmp+dmin2)/delta) < 1e-10)
               plotPS2D_xticklabel(0.0,Pxtmp);
            else
               plotPS2D_xticklabel(dt+dtmp+dmin2,Pxtmp);
            if (Pxtmp < first_tick) first_tick = Pxtmp;
            if (Pxtmp > last_tick ) last_tick  = Pxtmp;
         }
      }
   }

   /* Draw in labels at plot boundaries */
   if ((first_tick - Pxmin) > 20*scale) 
      plotPS2D_xticklabel(((!xflip) ? dmin : dmax),Pxmin);
   if ((Pxmax - last_tick) > 20*scale) 
      plotPS2D_xticklabel(((!xflip) ? dmax : dmin),Pxmax);
}
 

/* 
 * Draw the y-axis tick marks and labels with automatic ranging 
 */
static void plotPS2D_Ayticks(label_len)
int  *label_len;
{
   void   PXget_autorange();

   double dmin, dmax, dmin2, dmax2;
   double dtmp, dt, delta;
   double Pytmp;
   double first_tick, last_tick;   /* To keep track of labels */
   int    i, itick, nticks, tck;
   int    llen=0;

   /*
    * LINEAR SCALE
    * Start with a linear range (e.g. xmin = 0.05  xmax = 50.0)
    * Autoranging              =>     dmin = 0.05  dmax = 50.0
    *                                 dmin2= 0.0   dmax2= 50.0
    *                                 delta= 10.0
    *               => get scales at  x=0,10,20,30,40,50
    *
    * LOG SCALES
    * Start with a linear range (e.g. xmin = 0.05  xmax = 50.0)
    * For log scale, these numbers are converted first to logarithmic values
    * during initialization     (e.g. xmin = -1.3  xmax = 1.7  )
    * The autoranging forces =>       dmin = -1.3  dmax = 1.7
    *                                 dmin2= -2.0  dmax2= 2.0
    *                                 delta=  1.0
    *               => get scales at  x=-2,-1,0,1,2
    * So all I have to do is plot the scales in log10 increments
    */

   /*
    * This does not work too well if (xmax-xmin) << xmin
    * because of floating point errors.
    */

   /* 
    * useful data 
    * X = Pxmin + (x-xmin)*(Pxmax-Pxmin)/(xmax-xmin)
    * Y = Pymin + (y-ymin)*(Pymax-Pymin)/(ymax-ymin)
    */
   
   /* Tick Marks on the Y-axis */
   (void) fprintf(ips,"%% Draw the Y-Axis\n");
   (void) fprintf(ips,"setAxisLblFont\n");

   /* Get the rounded intervals */
   PXget_autorange(ymin,ymax,
                  &dmin,&dmax,&dmin2,&dmax2,&delta,ylog,yautorange);

   /* If dmin2=dmin then subtract delta from dmin2 so that dmin2 < dmin */
   /* This is used to get the leftmost tick label on the plot           */
   if (dmin2==dmin) dmin2 -= delta;

   /* Initialize first/last tick positions */
   first_tick = Pymax;   /* Should finally be a bit greater than Pymin */
   last_tick  = Pymin;   /* Should finally be a bit less than Pymax */

   /* Draw the tickmarks from dmin2 to dmax2 - dtmp is distance from dmin2 */
   if (delta <= 0.0) nticks = 0;
   else              nticks = (int)((dmax2 - dmin2)/delta) + 1;
   for (itick=0; itick<nticks; itick++) {
      dtmp = itick*delta;
      for (i=1; i<=10; i++) {
         /* distance from dtmp - actual phy loc is dt+dtmp+dmin2 */
         dt = i*0.1*delta;
         if (ylog) dt = log10((double)i); 
         if (ylog && i==1) continue;  /* Don't do this case */

         /* Translate to plot-coordinates */
         Pytmp = Pymin + (int)((dtmp+(dmin2-dmin))*jdy) + (int)(dt*jdy);
         if (yflip)
         Pytmp = Pymax - (int)((dtmp+(dmin2-dmin))*jdy) - (int)(dt*jdy);

         /* Don't draw if this is outside the plot boundaries */
         if (Pytmp < Pymin-0.01*scale || Pytmp > Pymax+0.01*scale) continue;

         /* Draw the grid */
         if (grid && (Pytmp > Pymin+0.01*scale && Pytmp < Pymax+0.01*scale)) {
            /*
             * PS major grid is the same as the PS minor grid 
             */
            if (i==10) {
               /* Major grid */
               (void) fprintf(ips,"%.2f setlinewidth 0.9 setgray\n",0.5);
               (void) fprintf(ips,"%.2f %.2f ",Pxmin,Pytmp);
               (void) fprintf(ips,"%.2f 0 drawline\n",Pxmax-Pxmin);
               (void) fprintf(ips,"%.2f setlinewidth 0.0 setgray\n",1.0*scale);
            } else if (scale > 0.4) {
               /* Minor grid - don't draw the minor grid for small plots */
               (void) fprintf(ips,"%.2f setlinewidth 0.9 setgray\n",0.5);
               (void) fprintf(ips,"%.2f %.2f ",Pxmin,Pytmp);
               (void) fprintf(ips,"%.2f 0 drawline\n",Pxmax-Pxmin);
               (void) fprintf(ips,"%.2f setlinewidth 0.0 setgray\n",1.0*scale);
            }
         }

         /* The length of the tick depends on i */
         if      (i==10) tck = 7;  /* major tick */
         else if (i==5 ) tck = 5;  /* major sub-tick */
         else            tck = 3;  /* minor tick */

         /* Draw the ticks */
         (void) fprintf(ips,"%.2f %.2f ",Pxmin,Pytmp);
         (void) fprintf(ips,"%.2f 0 drawline\n",-tck*scale);
         (void) fprintf(ips,"%.2f %.2f ",Pxmax,Pytmp);
         (void) fprintf(ips,"%.2f 0 drawline\n",-tck*scale);

         /* Draw labels */
         if (i==10) {
            /* If x is very close to 0 print 0 */
            if (fabs((dt+dtmp+dmin2)/delta) < 1e-10)
               plotPS2D_yticklabel(0.0,Pytmp,&llen);
            else
               plotPS2D_yticklabel(dt+dtmp+dmin2,Pytmp,&llen);
            if (Pytmp < first_tick) first_tick = Pytmp;
            if (Pytmp > last_tick ) last_tick  = Pytmp;
         }
      }
   }

   /* Draw in labels at plot boundaries */
   if ((first_tick - Pymin) > 20*scale) 
      plotPS2D_yticklabel(((!yflip) ? dmin : dmax),Pymin,&llen);
   if ((Pymax - last_tick) > 20*scale) 
      plotPS2D_yticklabel(((!yflip) ? dmax : dmin),Pymax,&llen);

   /* return the label length */
   if (llen < *label_len) *label_len = llen;
}


/* plot the tick label on the x-axis */
static void plotPS2D_xticklabel(vallbl,Pxtmp)
double vallbl, Pxtmp;
{
   char   label[MAXCHR];

   if (xlog) vallbl = pow(10.0,vallbl);
   vallbl = vallbl/xscale;
   (void) sprintf(label,"%.3g",vallbl);
   (void) fprintf(ips,"%.2f %.2f moveto ",Pxtmp,Pymin-8*scale);
   (void) fprintf(ips,"(%s) BOJA CEJ show\n",label);
}


/* plot the tick label on the y-axis */
static void plotPS2D_yticklabel(vallbl,Pytmp,llen)
double vallbl, Pytmp;
int    *llen;
{
   int    label_len;
   char   label[MAXCHR];

   if (ylog) vallbl = pow(10.0,vallbl);
   vallbl = vallbl/yscale;
   (void) sprintf(label,"%.3g",vallbl);
   label_len = strlen(label);
   if (label_len > *llen) *llen = label_len;
   (void) fprintf(ips,"%.2f %.2f moveto ",Pxmin-8*scale,Pytmp+3*scale);
   (void) fprintf(ips,"(%s) MIJA RIJ show\n",label);
}

/* draw the plot */
static void plotPS2D()
{
   CNdslistptr  DS, ds;
   CNdatasetptr Dptr;
   int          colrinc=0, lineinc=0;
   int          contfill, meshplot;
   int          PARENT_FOUND;

   (void) fprintf(ips,"%%------------------------------------------------\n");
   (void) fprintf(ips,"%%--------------Draw the Plot---------------------\n");
   (void) fprintf(ips,"%%------------------------------------------------\n");

   /*
    * Plot set - draw the grid if that exists
    */
   for (DS=plotdata->datahead; DS!=NULL; DS=DS->next) {
      if (DS->Dptr->grid && DS->Dptr->datatype == CN_GRID4D)
         plotPS2D_grid(DS->Dptr->grid, 
                       DS->Dptr->data_pr.contintrp, 
                       DS->Dptr->data_pr.contclip, 
                       DS->Dptr->data_pr.meshplot);
   }

   /*
    * Plot set - covers multiple plots
    * Draw the colored fills first, because if drawn later
    * the fills will obscure the drawn curves.
    */

   /*
    * Draw the boundary if that is available
    * If this is a PIF-type mesh use the parent's boundary
    */
   for (DS=plotdata->datahead; DS!=NULL; DS=DS->next) {

      if (DS->Dptr->parent != NULL) {
         /* Check to see if the parent is in the list */
         PARENT_FOUND = CN_FALSE;
         for (ds=plotdata->datahead; ds!=NULL && !PARENT_FOUND; ds=ds->next)
           if (ds->Dptr == DS->Dptr->parent) PARENT_FOUND = CN_TRUE;
 
         /* Select the dptr from which to get plot instructions */
         Dptr = PARENT_FOUND ? DS->Dptr->parent : DS->Dptr;
 
         /* Now draw the boundary */
         plotPS2D_boundary(DS->Dptr->parent,
                          (int)Dptr->data_pr.boundary,
                          (int)Dptr->data_pr.fillbnd,
                          (int)Dptr->data_pr.pr_rgID,
                          1,0);
      } else {
         plotPS2D_boundary(DS->Dptr,
                          (int)DS->Dptr->data_pr.boundary,
                          (int)DS->Dptr->data_pr.fillbnd,
                          (int)DS->Dptr->data_pr.pr_rgID,
                          1,0);
      }
   }
 
   /*
    * Draw colored contour fills or hiddenline triangles/rectangles
    */
   for (DS=plotdata->datahead; DS!=NULL; DS=DS->next) {

      /* Draw colored triangles but not the curves */
      contfill = DS->Dptr->data_pr.contstyle == CN_FILLCONT;
      meshplot = DS->Dptr->data_pr.contstyle == CN_PLOTMESH;
      meshplot = meshplot | DS->Dptr->data_pr.meshplot;
      if (contfill || meshplot) {
         plotPS2D_trias(DS->Dptr, contfill, meshplot);
         plotPS2D_rects(DS->Dptr, contfill, meshplot);
      }
   }

   /*
    * Now draw the contour and mesh datasets
    */
   colrinc = 0;
   lineinc = 0;
   for (DS=plotdata->datahead; DS!=NULL; DS=DS->next) {

      /* Now draw the contour curves */
      if (DS->Dptr->datatype == CN_CONTOUR) {
         contfill = DS->Dptr->data_pr.contstyle == CN_FILLCONT;
         meshplot = DS->Dptr->data_pr.contstyle == CN_PLOTMESH;

         /*EMPTY*/
         if (contfill || meshplot) {
            /*
             * Don't draw curves for this dataset
             *
             * Note however that if data_pr.contstyle == CN_LINECONT,
             * and data_pr.meshplot = ON, then both the mesh and the
             * contours will be drawn.
             */
            ;
         } else {
            /* Draw the curve set */
            plotPS2D_dataset_curves(DS->Dptr,colrinc,lineinc);
            if (!overlay) {
               colrinc = 0;
               lineinc = 0;
            } else if (DS->Dptr->data_pr.linetypes == 1) {
               colrinc = colrinc ++;
               lineinc = lineinc ++;
            } else {
               colrinc = colrinc + 3;
               lineinc = lineinc + 3;
            }
         }
      }
   }

   /*
    * Now draw the rest of the datasets
    */
   colrinc = 0;
   lineinc = 0;
   for (DS=plotdata->datahead; DS!=NULL; DS=DS->next) {

      /* Don't draw contours */
      if (DS->Dptr->datatype == CN_CONTOUR) continue;

      /* Draw the curve set */
      plotPS2D_dataset_curves(DS->Dptr, colrinc, lineinc);

      if (DS->Dptr->curvehead != NULL) {
         if (!overlay) {
            colrinc = 0;
            lineinc = 0;
         } else {
            colrinc ++;
            lineinc ++;
         }
      }
   }

   /*
    * Draw the vectors
    */
   for (DS=plotdata->datahead; DS!=NULL; DS=DS->next) {
      if (DS->Dptr->vecbox) 
         plotPS2D_vectors(DS->Dptr->vecbox,
                         (int)    DS->Dptr->data_pr.vlog,
                         (double) DS->Dptr->data_pr.vscale,
                         (double) DS->Dptr->data_pr.vlogscale,
                         (int)    DS->Dptr->data_pr.vhead,
                         (int)    DS->Dptr->data_pr.vtail);
   }

   /*
    * Draw the element ID's if necessary
    * curve-point ID's are plotted together with curves
    */
   for (DS=plotdata->datahead; DS!=NULL; DS=DS->next) {
      /* Set the Dptr */
      Dptr = DS->Dptr;
 
      /* Mesh-based Vectors are treated differently */
      if (DS->Dptr->datatype == CN_VECTOR && DS->Dptr->parent != NULL)
         Dptr = DS->Dptr->parent;

      /* Draw ID's for hierarchical meshes */
      if (DS->Dptr->data_pr.pr_ptID) {
         if (DS->Dptr->parent != NULL)
         plotPS2D_pointIDs(DS->Dptr->parent->pointhead, 
                           DS->Dptr->parent->pointtail, CN_FALSE);
        else
         plotPS2D_pointIDs(Dptr->pointhead, Dptr->pointtail, CN_FALSE);
      }
      if (DS->Dptr->data_pr.pr_ndID)
         plotPS2D_nodeIDs (Dptr->nodehead,  Dptr->nodetail);
      if (DS->Dptr->data_pr.pr_trID)
         plotPS2D_triaIDs (Dptr->triahead,  Dptr->triatail);
      if (DS->Dptr->data_pr.pr_rtID)
         plotPS2D_rectIDs (Dptr->recthead,  Dptr->recttail);
   }

   /*
    * Draw the boundary again without fills
    * If this is a PIF-type mesh use the parent's boundary
    */
   for (DS=plotdata->datahead; DS!=NULL; DS=DS->next) {

      if (DS->Dptr->parent != NULL) {
         /* Check to see if the parent is in the list */
         PARENT_FOUND = CN_FALSE;
         for (ds=plotdata->datahead; ds!=NULL && !PARENT_FOUND; ds=ds->next)
            if (ds->Dptr == DS->Dptr->parent) PARENT_FOUND = CN_TRUE;
 
         /* Select the dptr from which to get plot instructions */
         Dptr = PARENT_FOUND ? DS->Dptr->parent : DS->Dptr;
 
         /* Now draw the boundary */
         plotPS2D_boundary(DS->Dptr->parent,
                          (int)Dptr->data_pr.boundary,
                          (int)Dptr->data_pr.fillbnd,
                          (int)Dptr->data_pr.pr_rgID,
                          0,1);
      } else {
         plotPS2D_boundary(DS->Dptr,
                          (int)DS->Dptr->data_pr.boundary,
                          (int)DS->Dptr->data_pr.fillbnd,
                          (int)DS->Dptr->data_pr.pr_rgID,
                          0,1);
      }
   }
}


/*
 * Plot the grid in POSTSCRIPT
 */
static void plotPS2D_grid(grid, contintrp, contclip, meshplot)
CNgrid4Dptr grid;
short       contintrp;
short       contclip;
short       meshplot;
{
   CNsliceptr slice=NULL;
   CNrectptr  R;
   CNtriaptr  T;
   int        verbose=0;
   double     cxmin, cxmax, cymin, cymax, czmin, czmax, ctmin, ctmax;
   double     tmp;
   short      logx=0, logy=0, logz=0, logt=0;
 
   /* clipping boundaries */
   cxmin = ((xlog) ? pow(10.0,xmin) : xmin);
   cxmax = ((xlog) ? pow(10.0,xmax) : xmax);
   cymin = ((ylog) ? pow(10.0,ymin) : ymin);
   cymax = ((ylog) ? pow(10.0,ymax) : ymax);
   czmin = - CN_LARGE;
   czmax =   CN_LARGE;
   ctmin = czmin;
   ctmax = czmax;
 
   /*
    * Just plot at z=zmax
    */
   tmp = (grid->zmax < zmax) ? grid->zmax : zmax;
   if ((slice = CNslice_grid4D_z(grid,tmp,(int)contintrp,verbose)) != NULL) {
 
      /* Loop through the rectangles */
      for (R=slice->recthead; R!=NULL; R=R->next) {
         /* Quick check to see if the rectangle is in-bounds */
         if (!CNrect_in_bounds(R,
              cxmin,cxmax,cymin,cymax,czmin,czmax,ctmin,ctmax)) continue;
 
         /* Plot the color fills */
         plotPS2D_single_fill_rect(R,
              cxmin,cxmax,cymin,cymax,czmin,czmax,ctmin,ctmax,
              logx, logy, logz, logt, contclip);
 
         /* Plot the mesh */
         if (meshplot)
         plotPS2D_single_mesh_rect(R,
              cxmin,cxmax,cymin,cymax,czmin,czmax,ctmin,ctmax,
              logx, logy, logz, logt, 0);
      }
 
      /* Loop through the triangles */
      for (T=slice->triahead; T!=NULL; T=T->next) {
         /* Quick check to see if the triangle is in-bounds */
         if (!CNtria_in_bounds(T,
              cxmin,cxmax,cymin,cymax,czmin,czmax,ctmin,ctmax)) continue;
 
         /* Plot the color fills */
         plotPS2D_single_fill_tria(T,
              cxmin,cxmax,cymin,cymax,czmin,czmax,ctmin,ctmax,
              logx, logy, logz, logt, contclip);
 
         /* Plot the mesh */
         if (meshplot)
         plotPS2D_single_mesh_tria(T,
              cxmin,cxmax,cymin,cymax,czmin,czmax,ctmin,ctmax,
              logx, logy, logz, logt, 0);
      }

      /* Delete the slice */
      CNdelete_slice(slice);
   }

   /* Reset color */
   PXsetColorPS(0);
}

/*
 * Plot the vectors in POSTSCRIPT
 */
static void plotPS2D_vectors(Vbox, vlog, vscale, vlogscale, vhead, vtail)
CNvecboxptr  Vbox;
int          vlog;
double       vscale, vlogscale;
int          vhead, vtail;
{
   CNvecptr    Vptr;
   double      vx, vy;
   double      x1, y1, x2, y2;
   double      vmin;
   int         p1_clipped=CN_FALSE, p2_clipped=CN_FALSE;
 
   if (Vbox == NULL) return;
 
   if (vlog) {
      vmin = 0.1*Vbox->vlen_min;
      if (vmin == 0.0) vmin = 1.0;
   }

   /* Go thru the vectors */
   for (Vptr=Vbox->vectorhead; Vptr!=NULL; Vptr=Vptr->next) {
 
      /* Check plot flag */
      if (Vptr->noplot) continue;

      /* Scale the vector starting point */
      trn_world_to_PS(Vptr->x,Vptr->y,&x1,&y1);
 
      /* Scale to the dimensions of the plot specified in the Vbox */
      if (vlog) {
         vx   = CNveclog10(Vptr->vx/vmin) * vlogscale;
         vy   = CNveclog10(Vptr->vy/vmin) * vlogscale;
      } else {
         vx   = Vptr->vx * vscale;
         vy   = Vptr->vy * vscale;
      }
 
      /* Find the vector ending point in device coordinates */
      trn_world_to_PS(Vptr->x+vx,Vptr->y+vy,&x2,&y2);
 
      /* Draw the arrow */
      plotPS2D_arrow(x1,y1,x2,y2,
                     p1_clipped, p2_clipped,
                     (char *)NULL,
                     (int)Vbox->linetype,
                     (int)Vbox->linecolor,
                     (int)Vbox->linewidth,
                     (int)( vtail ? Vbox->marktype : CN_MK_NONE ),
                     1,
                     (int)Vbox->markcolor,vhead);
   }
}

/*
 * Plot the boundary in POSTSCRIPT
 */
/*ARGSUSED*/
static void plotPS2D_boundary(Dptr, boundary, fillbnd, pr_rgID, fill, drawlabel)
CNdatasetptr Dptr;
int          boundary, fillbnd, pr_rgID;
int          fill, drawlabel;
{
   CNregionptr R;
   CNpolyptr   P;
   CNnodeptr   node_head=NULL, node_tail=NULL, N;
   double      cxmin, cxmax, cymin, cymax, czmin, czmax, ctmin, ctmax;
   double      midx, midy, x, y;
   char        label[CN_MAXCHAR];
   int         cnt=0;

   if (Dptr->regionhead == NULL) return;

   if (boundary==CN_FALSE && fillbnd==CN_FALSE) return;

   /* Reset the fill - if the boundary is False then don't fill */
   if (fillbnd == CN_FALSE) fill=CN_FALSE;

   /*
    * Clipping Boundary
    */
   cxmin = ((xlog) ? pow(10.0,xmin) : xmin);
   cxmax = ((xlog) ? pow(10.0,xmax) : xmax);
   cymin = ((ylog) ? pow(10.0,ymin) : ymin);
   cymax = ((ylog) ? pow(10.0,ymax) : ymax);
   czmin = - CN_LARGE;
   czmax =   CN_LARGE;
   ctmin = czmin;
   ctmax = czmax;

   /* Set the font */
   (void) fprintf(ips,"setMkrsLblFont\n");

   /* now print out the boundary segments in each region */
   for (R=Dptr->regionhead; R!=NULL; R=R->next) {
      for (P=R->polyhead; P!=NULL; P=P->next) {

         /* Clip the polygon */
         CNclip_poly(P, &node_head, &node_tail,
               cxmin, cxmax, cymin, cymax, czmin, czmax, ctmin, ctmax,
               1, 1, 0, 1, 0);

         /* Plot the nodes */
         plotPS2D_nodes(node_head, node_tail,
                       fill ? CN_FILL_SOLID : CN_FILL_NONE, 
                       PXpolyColorIndexPS(R->color),
                       boundary ? CN_LN_SOLID : CN_LN_NONE, 0, 4);

         /* Print ID/label information */
         if (pr_rgID && drawlabel && node_head!=NULL) {
            /* Get the midpoint - don't count the last point */
            midx = midy = 0.0;
            cnt = 0;
            for (N=node_head; N!=NULL; N=N->next) {
               if (N==node_tail) continue;
               midx += N->coord->x;
               midy += N->coord->y;
               cnt++;
            }
            midx = midx/(double)cnt;
            midy = midy/(double)cnt;
            trn_world_to_PS(midx,midy,&x,&y);

            /* Check to see if this midpoint is actually inside the region */
            if (CNpoint_in_region(Dptr,midx,midy,(int)R->ID,0)) {
               /* Print the region ID */
               (void) sprintf(label,"Region %d",R->ID);
               (void) fprintf(ips,"%.2f %.2f moveto ",x,y-2*scale);
               (void) fprintf(ips,"(%s) BOJM CEJ show\n",
                              PXmodifyPS_string(label));

               /* Print the material type */
               (void) sprintf(label,"Mat \"%s\"",R->matname);
               (void) fprintf(ips,"%.2f %.2f moveto ",x,y+2*scale);
               (void) fprintf(ips,"(%s) UPJ  CEJ show\n",
                              PXmodifyPS_string(label));
            }
         }

         /* Delete the node-list */
         CNremove_node_list(&node_head, &node_tail);

      }
   }
}


/*
 * Plot the triangular mesh in POSTSCRIPT
 */
static void plotPS2D_trias(Dptr,contfill,meshplot)
CNdatasetptr Dptr;
int          contfill;   /* Draw multi-colored triangles */
int          meshplot;   /* Draw only the mesh           */
{
   CNtriaptr   T;
   double      cxmin, cxmax, cymin, cymax, czmin, czmax, ctmin, ctmax;

   if (Dptr->triahead == NULL) return;

   /*
    * Clipping Boundary
    */
   cxmin = ((xlog) ? pow(10.0,xmin) : xmin);
   cxmax = ((xlog) ? pow(10.0,xmax) : xmax);
   cymin = ((ylog) ? pow(10.0,ymin) : ymin);
   cymax = ((ylog) ? pow(10.0,ymax) : ymax);
   czmin = - CN_LARGE;
   czmax =   CN_LARGE;
   ctmin = czmin;
   ctmax = czmax;

   /* Loop through the triangles */
   for (T=Dptr->triahead; T!=NULL; T=T->next) {

      /* Quick check to see if the triangle is in-bounds */
      if (!CNtria_in_bounds(T,
           cxmin,cxmax,cymin,cymax,czmin,czmax,ctmin,ctmax)) continue;

      /* Draw a fill triangle with gradated colors */
      if (contfill) {
          plotPS2D_single_fill_tria(T,
                                   cxmin, cxmax, cymin, cymax,
                                   czmin, czmax, ctmin, ctmax,
                                   Dptr->data_pr.logx,
                                   Dptr->data_pr.logy,
                                   0,
                                   Dptr->data_pr.logz,
                                   Dptr->data_pr.contclip);
      }

      /* Draw a mesh triangle */
      if (meshplot) {
          plotPS2D_single_mesh_tria(T,
                                   cxmin, cxmax, cymin, cymax,
                                   czmin, czmax, ctmin, ctmax,
                                   Dptr->data_pr.logx,
                                   Dptr->data_pr.logy,
                                   0,
                                   Dptr->data_pr.logz, 0);
      }
   }

   /* Reset color */
   PXsetColorPS(0);
}


/*
 * Plot the rectangular mesh in POSTSCRIPT
 */
static void plotPS2D_rects(Dptr,contfill,meshplot)
CNdatasetptr Dptr;
int          contfill;   /* Draw multi-colored rectangles*/
int          meshplot;   /* Draw only the mesh           */
{
   CNrectptr   R;
   double      cxmin, cxmax, cymin, cymax, czmin, czmax, ctmin, ctmax;

   if (Dptr->recthead == NULL) return;
   /*
    * Clipping Boundary
    */
   cxmin = ((xlog) ? pow(10.0,xmin) : xmin);
   cxmax = ((xlog) ? pow(10.0,xmax) : xmax);
   cymin = ((ylog) ? pow(10.0,ymin) : ymin);
   cymax = ((ylog) ? pow(10.0,ymax) : ymax);
   czmin = - CN_LARGE;
   czmax =   CN_LARGE;
   ctmin = czmin;
   ctmax = czmax;

   /* Loop through the rectangles */
   for (R=Dptr->recthead; R!=NULL; R=R->next) {

      /* Quick check to see if the rectangle is in-bounds */
      if (!CNrect_in_bounds(R,
           cxmin,cxmax,cymin,cymax,czmin,czmax,ctmin,ctmax)) continue;

      /* Draw a fill rectangle with gradated colors */
      if (contfill) {
          plotPS2D_single_fill_rect(R,
                                   cxmin, cxmax, cymin, cymax,
                                   czmin, czmax, ctmin, ctmax,
                                   Dptr->data_pr.logx,
                                   Dptr->data_pr.logy,
                                   0,
                                   Dptr->data_pr.logz,
                                   Dptr->data_pr.contclip);
      }

      /* Draw a mesh rectangle */
      if (meshplot) {
          plotPS2D_single_mesh_rect(R,
                                   cxmin, cxmax, cymin, cymax,
                                   czmin, czmax, ctmin, ctmax,
                                   Dptr->data_pr.logx,
                                   Dptr->data_pr.logy,
                                   0,
                                   Dptr->data_pr.logz,
                                   0);
      }
   }

   /* Reset color */
   PXsetColorPS(0);
}


/*
 * Plot a single mesh triangle
 */
static void plotPS2D_single_mesh_tria(T,
                                     cxmin, cxmax, cymin, cymax,
                                     czmin, czmax, ctmin, ctmax,
                                     logx, logy, logz, logt,
                                     fill)
CNtriaptr T;
double    cxmin, cxmax, cymin, cymax, czmin, czmax, ctmin, ctmax;
short     logx, logy, logz, logt;
int       fill;
{
   CNnodeptr   node_head=NULL, node_tail=NULL;
   int         fillcolor, linecolor;

   /* Check the triangle */
   if (T == NULL) return;

   /* Clip the triangle */
   CNclip_tria(T, &node_head, &node_tail,
               cxmin, cxmax, cymin, cymax, czmin, czmax, ctmin, ctmax,
               1, 1, 0, 1, 
               logx, logy, logz, logt, 0);

   /* Plot the nodes */
   fillcolor = PXpolyColorIndexPS(1);   /* yellow */
   linecolor = PXpolyColorIndexPS(4);   /* red    */
   plotPS2D_nodes(node_head, node_tail,
                 ((fill) ? CN_FILL_SOLID : CN_FILL_NONE),
                 fillcolor, CN_LN_SOLID, linecolor, 1);

   /* Delete the node-list */
   CNremove_node_list(&node_head, &node_tail);
}


/*
 * Plot a single mesh rectangle
 */
static void plotPS2D_single_mesh_rect(R,
                                     cxmin, cxmax, cymin, cymax,
                                     czmin, czmax, ctmin, ctmax,
                                     logx, logy, logz, logt,
                                     fill)
CNrectptr R;
double    cxmin, cxmax, cymin, cymax, czmin, czmax, ctmin, ctmax;
short     logx, logy, logz, logt;
int       fill;
{
   CNnodeptr   node_head=NULL, node_tail=NULL;
   int         fillcolor, linecolor;

   /* Check the rectangle */
   if (R == NULL) return;

   /* Clip the rectangle */
   CNclip_rect(R, &node_head, &node_tail,
               cxmin, cxmax, cymin, cymax, czmin, czmax, ctmin, ctmax,
               1, 1, 0, 1,
               logx, logy, logz, logt, 0);

   /* Plot the nodes */
   fillcolor = PXpolyColorIndexPS(1);   /* yellow */
   linecolor = PXpolyColorIndexPS(4);   /* red    */
   plotPS2D_nodes(node_head, node_tail,
                 ((fill) ? CN_FILL_SOLID : CN_FILL_NONE),
                 fillcolor, CN_LN_SOLID, linecolor, 1);

   /* Delete the node-list */
   CNremove_node_list(&node_head, &node_tail);
}


/*
 * Draw fill colors in a triangle
 */
/*ARGSUSED*/
static void plotPS2D_single_fill_tria(T,
                                     cxmin, cxmax, cymin, cymax,
                                     czmin, czmax, ctmin, ctmax,
                                     logx, logy, logz, logt,
                                     contclip)
CNtriaptr T;
double    cxmin, cxmax, cymin, cymax, czmin, czmax, ctmin, ctmax;
short     logx, logy, logz, logt;
short     contclip;
{
   CNcontstepptr C;
   CNnodeptr     node_head=NULL, node_tail=NULL;
   double        tmin, tmax, min, max;
   int           i, colr, nctrs;

   /* Check the triangle */
   if (T == NULL) return;

   /* If the noplot flag is set, skip */
   if (T->noplot) return;

   /* Get the min and max of the triangle */
   CNget_tria_tmaxmin(T,&tmin,&tmax);

   /*
    * The contour steps are calculated outside so that there is
    * a common steps for all the contour datasets
    */
   nctrs = CNcount_contsteps(cstephead, csteptail) - 1;

   /* Loop through the contours - min of 2 contours  */
   i = 0;
   for (C=cstephead; C!=NULL; C=C->next) {
      /*
       * There will be nctrs+1 bands; make sure each of these
       * has a distinct color.
       * Scale the colors from 1 to 32
       */
      colr      = (int)((double)(i*(PX_MAX_FILL_COLORS-1))/(double)nctrs) + 1;

      /* Increment the step number */
      i++;

      /* Set the clipping levels */

      /*
       * The contour steps selected in CNselect_contour_step()
       * do not necessarily cover the whole range [tmin..tmax],
       * i.e. cstephead->value approx cmin, and csteptail->value approx cmax
       * However in PXquery_contours() an additional step corresponding
       * to the the max step size CN_LARGE is automatically added.
       */
      if (C->prev == NULL) {
         max = C->value;
         min = -CN_LARGE;
         if (contclip) continue;
      } else {
         max = C->value;
         min = C->prev->value;
         if (contclip && max==CN_LARGE) continue;
      }

      /* Clip the triangle */
      /*EMPTY*/
      if (tmax < min || tmin > max) {

         /* Don't do anything */
         ;

      } else {

         /* Clip the triangle */
         CNclip_tria(T, &node_head, &node_tail,
                     cxmin, cxmax, cymin, cymax, czmin, czmax, min, max,
                     1, 1, 1, 1,
                     logx, logy, logz, logt, 0);

         /* Plot the nodes */
         plotPS2D_nodes(node_head, node_tail,
                       CN_FILL_SOLID, PXfillColorIndex(colr), CN_LN_NONE,0,1);

         /* Delete the node-list */
         CNremove_node_list(&node_head, &node_tail);
      }
   }
}


/*
 * Draw fill colors in a rectangle
 */
/*ARGSUSED*/
static void plotPS2D_single_fill_rect(R,
                                     cxmin, cxmax, cymin, cymax,
                                     czmin, czmax, ctmin, ctmax,
                                     logx, logy, logz, logt,
                                     contclip)
CNrectptr R;
double    cxmin, cxmax, cymin, cymax, czmin, czmax, ctmin, ctmax;
short     logx, logy, logz, logt;
short     contclip;
{
   CNcontstepptr C;
   CNnodeptr     node_head=NULL, node_tail=NULL;
   double        tmin, tmax, min, max;
   int           i, colr, nctrs;

   /* Check the rectangle */
   if (R == NULL) return;

   /* Get the min and max of the rectangle */
   CNget_rect_tmaxmin(R,&tmin,&tmax);

   /*
    * The contour steps are calculated outside so that there is
    * a common steps for all the contour datasets
    */
   nctrs = CNcount_contsteps(cstephead, csteptail) - 1;

   /* Loop through the contours - min of 2 contours */
   i = 0;
   for (C=cstephead; C!=NULL; C=C->next) {
      /*
       * There will be nctrs+1 bands; make sure each of these
       * has a distinct color.
       * Scale the colors from 1 to 32
       */
      colr      = (int)((double)(i*(PX_MAX_FILL_COLORS-1))/(double)nctrs) + 1;

      /* Increment the step number */
      i++;

      /* Set the clipping levels */

      /*
       * The contour steps selected in CNselect_contour_step()
       * do not necessarily cover the whole range [tmin..tmax],
       * i.e. cstephead->value approx cmin, and csteptail->value approx cmax
       * However in PXquery_contours() an additional step corresponding
       * to the the max step size CN_LARGE is automatically added.
       */
      if (C->prev == NULL) {
         max = C->value;
         min = -CN_LARGE;
         if (contclip) continue;
      } else {
         max = C->value;
         min = C->prev->value;
         if (contclip && max==CN_LARGE) continue;
      }

      /* Clip the rectangle */
      /*EMPTY*/
      if (tmax < min || tmin > max) {

         /* Don't do anything */
         ;
      } else {

         /* Clip the rectangle */
         CNclip_rect(R, &node_head, &node_tail,
                     cxmin, cxmax, cymin, cymax, czmin, czmax, min, max,
                     1, 1, 1, 1, 
                     logx, logy, logz, logt, 0);

         /* Plot the nodes */
         plotPS2D_nodes(node_head, node_tail,
                       CN_FILL_SOLID, PXfillColorIndex(colr), CN_LN_NONE,0,1);

         /* Delete the node-list */
         CNremove_node_list(&node_head, &node_tail);
      }
   }
}


/*
 * Plot a list of nodes
 */
/*ARGSUSED*/
static void plotPS2D_nodes(node_head, node_tail,
                           filltype, fillcolor, linestyle, linecolor, linethick)
CNnodeptr node_head, node_tail;
int       filltype, fillcolor, linestyle, linecolor, linethick;
{
   static int  do_fillpoly5();
   static int  do_fillpoly4();
   static int  do_fillpoly3();
   static int  do_outlinepoly5();
   static int  do_outlinepoly4();
   CNnodeptr   N;
   PSPoint     points[MAX_ARR_SIZE];
   double      x, y;
   int         match = CN_FALSE;
   int         count;

   /* return now if there is nothing to plot */
   if (node_head == NULL) return;

   /* if filltype and linestyle are both NONE (0), return now */
   if ((filltype==CN_FILL_NONE) && (linestyle==CN_LN_NONE)) return;

   /* rescale points, save in array */
   count=0;
   for (N=node_head; N!=NULL && count<MAX_ARR_SIZE; N=N->next) {
      trn_world_to_PS(N->coord->x,N->coord->y,&x,&y);
      points[count].x = x;
      points[count].y = y;
      count++;
   }

   /*
    * The size of the postscript file can be reduced significantly by
    * using triangles/rectangles as necessary, without going to the
    * general n-point polygon drawing routine.
    */
   if ((linestyle==CN_LN_NONE) && (filltype==CN_FILL_SOLID) && (count<=5)) {
      /* Filled triangle/rectangle */
      if (count==5) {
         match = do_fillpoly5(points, count, fillcolor);
      } else if (count==4) {
         match = do_fillpoly4(points, count, fillcolor);
      } else if (count==3) {
         match = do_fillpoly3(points, count, fillcolor);
      }
   } else if ((linestyle!=CN_LN_NONE) && (filltype==CN_FILL_NONE) &&(count<=5)){
      /* Triangle/rectangle outline */
      if (count==5) {
         match = do_outlinepoly5(points, count, 
                                 linestyle, linecolor, linethick);
      } else if (count==4) {
         match = do_outlinepoly4(points, count, 
                                 linestyle, linecolor, linethick);
      }
   } 

   /* Don't draw if a match was found */
   if (match) return;

   /* Fill the polygon */
   PXfillPS_polygon(points, count, 
                    filltype, fillcolor, linestyle, linecolor, linethick);
}


/*
 * Fill a 5-point polygon
 */
static int do_fillpoly5(points, count, fillcolor)
PSPoint *points;
int     count;
int     fillcolor;
{
   static int rightangle();
   int        angle012, angle123, angle234, angle103;
   int        match = CN_FALSE;

   /* Error check */  
   if (points==NULL || count!=5) return(CN_FALSE);

   /*
    * The only time that this returns FALSE is if point[0]!=point[4]
    */

   /* This could be a rectangle with joined ends */
   if ((points[0].x==points[4].x) && (points[0].y==points[4].y)) {
      /* OK, the ends are joined - check for squareness (dot rule) */
      angle103 = rightangle(points[1].x, points[1].y,
                            points[0].x, points[0].y,
                            points[3].x, points[3].y);
      angle012 = rightangle(points[0].x, points[0].y,
                            points[1].x, points[1].y,
                            points[2].x, points[2].y);
      angle123 = rightangle(points[1].x, points[1].y,
                            points[2].x, points[2].y,
                            points[3].x, points[3].y);
      angle234 = rightangle(points[2].x, points[2].y,
                            points[3].x, points[3].y,
                            points[4].x, points[4].y);
      if (angle103 && angle012 && angle123 && angle234) {

         /* Set the color */
         PXsetColorPS(fillcolor);

         /* Print out the rectangle */
         (void) fprintf(ips,"%.2f %.2f %.2f %.2f fillrect\n",
                        points[0].x, points[0].y,
                        points[2].x-points[0].x,
                        points[2].y-points[0].y);

         match = CN_TRUE;
      } else {
         /* 
          * Hmmm - OK, so this is a 4 sided polygon 
          * Assume that the polygon is well behaved, and divide it
          * into 2 triangles
          */

         /* Set the color */
         PXsetColorPS(fillcolor);

         /* Print out the triangles */
         (void) fprintf(ips,"%.2f %.2f %.2f %.2f %.2f %.2f filltria\n",
                        points[0].x, points[0].y,
                        points[1].x, points[1].y,
                        points[2].x, points[2].y);
         (void) fprintf(ips,"%.2f %.2f %.2f %.2f %.2f %.2f filltria\n",
                        points[2].x, points[2].y,
                        points[3].x, points[3].y,
                        points[0].x, points[0].y);

         match = CN_TRUE;
      }
   }
   return(match);
}


/*
 * Fill a 4-point polygon
 */
static int do_fillpoly4(points, count, fillcolor)
PSPoint *points;
int     count;
int     fillcolor;
{
   static int do_fillpoly5();
   PSPoint    newpoints[5];
   int        i;
   int        match = CN_FALSE;

   /* Error check */  
   if (points==NULL || count!=4) return(CN_FALSE);

   /*
    * This routine always returns TRUE 
    */

   /* This could be a triangle with joined ends */
   if ((points[0].x==points[3].x) && (points[0].y==points[3].y)) {
      /* OK, the ends are joined - print triangle */

      /* Set the color */
      PXsetColorPS(fillcolor);
 
      /* Print out the triangle */
      (void) fprintf(ips,"%.2f %.2f %.2f %.2f %.2f %.2f filltria\n",
                     points[0].x, points[0].y,
                     points[1].x, points[1].y,
                     points[2].x, points[2].y);
 
      match = CN_TRUE;

   } else {
      /* 4-sided polygon - check for squareness */

      /* Copy the point list into the new array */
      for (i=0; i<count; i++) {
         newpoints[i].x = points[i].x;
         newpoints[i].y = points[i].y;
      }
      newpoints[count].x = points[0].x;
      newpoints[count].y = points[0].y;

      /* Check 5 sided polygon (count==4) */
      match = do_fillpoly5(newpoints, count+1, fillcolor);

   }
   return(match);
}


/*
 * Fill a 3-point polygon
 */
static int do_fillpoly3(points, count, fillcolor)
PSPoint *points;
int     count;
int     fillcolor;
{
   int        match = CN_FALSE;

   /* Error check */  
   if (points==NULL || count!=3) return(CN_FALSE);

   /*
    * This routine always returns TRUE 
    */

   /* This is could be a line with shared points */
   if ((points[0].x==points[2].x) && (points[0].y==points[2].y)) {
      /* OK, the ends are joined - this is a line, so don;t do a thing */
      match = CN_TRUE;

   } else {
 
      /* Set the color */
      PXsetColorPS(fillcolor);
 
      /* Print out the triangle */
      (void) fprintf(ips,"%.2f %.2f %.2f %.2f %.2f %.2f filltria\n",
                     points[0].x, points[0].y,
                     points[1].x, points[1].y,
                     points[2].x, points[2].y);
 
      match = CN_TRUE;

   } 
   return(match);
}


/*
 * Draw outline of a 5-point polygon
 */
static int do_outlinepoly5(points, count, linestyle, linecolor, linethick)
PSPoint *points;
int     count;
int     linestyle, linecolor, linethick;
{
   static int rightangle();
   int        angle012, angle123, angle234, angle103;
   int        match = CN_FALSE;

   /* Error check */  
   if (points==NULL || count!=5) return(CN_FALSE);

   /*
    * This returns TRUE only if the poly makes up a square 
    */

   /* This could be a rectangle with joined ends */
   if ((points[0].x==points[4].x) && (points[0].y==points[4].y)) {
      /* OK, the ends are joined - check for squareness (dot rule) */
      angle103 = rightangle(points[1].x, points[1].y,
                            points[0].x, points[0].y,
                            points[3].x, points[3].y);
      angle012 = rightangle(points[0].x, points[0].y,
                            points[1].x, points[1].y,
                            points[2].x, points[2].y);
      angle123 = rightangle(points[1].x, points[1].y,
                            points[2].x, points[2].y,
                            points[3].x, points[3].y);
      angle234 = rightangle(points[2].x, points[2].y,
                            points[3].x, points[3].y,
                            points[4].x, points[4].y);
      if (angle103 && angle012 && angle123 && angle234) {

         /* Set the color */
         PXsetColorPS(linecolor);

         /* Set the linetype */
         (void) PXlinetypPS(linestyle,linethick);

         /* Print out the rectangle */
         (void) fprintf(ips,"%.2f %.2f %.2f %.2f drawrect stroke\n",
                        points[0].x, points[0].y,
                        points[2].x-points[0].x,
                        points[2].y-points[0].y);

         match = CN_TRUE;
      } 
   }
   return(match);
}


/*
 * Draw outline of a 4-point polygon
 */
static int do_outlinepoly4(points, count, linestyle, linecolor, linethick)
PSPoint *points;
int     count;
int     linestyle, linecolor, linethick;
{
   int        match = CN_FALSE;

   /* Error check */  
   if (points==NULL || count!=4) return(CN_FALSE);

   /*
    * This routine returns TRUE only if the polygon is closed
    */

   /* This could be a triangle with joined ends */
   if ((points[0].x==points[3].x) && (points[0].y==points[3].y)) {
      /* OK, the ends are joined - print triangle */

      /* Set the color */
      PXsetColorPS(linecolor);
 
      /* Set the linetype */
      (void) PXlinetypPS(linestyle,linethick);
 
      /* Print out the triangle */
      (void) fprintf(ips,"%.2f %.2f %.2f %.2f %.2f %.2f drawtria stroke\n",
                     points[0].x, points[0].y,
                     points[1].x, points[1].y,
                     points[2].x, points[2].y);
 
      match = CN_TRUE;

   } 
   return(match);
}


/*
 * Given 3 points, A, B, C, check the angle ABC using the dot product
 * of the vectors AB and BC
 */
static int rightangle(xa,ya,xb,yb,xc,yc)
double xa,ya,xb,yb,xc,yc;
{
   double dx1, dx2, dy1, dy2;
   double dprod;

   dx1 = xa-xb;
   dy1 = ya-yb;
   dx2 = xc-xb;
   dy2 = yc-yb;
   dprod = dx1*dx2 - dy1*dy2;
   
   /* If 2 of the points are coincident, then return FALSE */
   if ((dx1==0.0 && dy1==0.0) || (dx2==0.0 && dy2==0.0)) 
      return (CN_FALSE);
   else
      return( (fabs(dprod) < CN_SMALL) );
}


/*
 * Draw a set of curves in a dataset
 */
static void plotPS2D_dataset_curves(dptr, colrinc, lineinc)
CNdatasetptr dptr;
int          colrinc, lineinc;
{
   CNcurveptr C;
   int        contour, contlbl, lbloffset=0;
   int        spline, hdnline, applyfill, pr_ptID;

   /* Check the dataset first */
   if (dptr == NULL || dptr->curvehead == NULL) return;

   /* Initialize */
   contour   = dptr->datatype==CN_CONTOUR;
   contlbl   = dptr->data_pr.contlabel;
   spline    = dptr->data_pr.splinetyp;
   hdnline   = hiddenline || dptr->view_pr->hiddenline;
   applyfill = dptr->data_pr.applyfill;
   pr_ptID   = dptr->data_pr.pr_ptID;

   /* Go thru each set of curves */
   for (C=dptr->curvehead; C!=NULL; C=C->next) {

      /*
       * Plot the curve
       */
 
      if (dptr->data_pr.splinetyp == CN_SP_NONE) {
         /* Plot the curve along the given (real) data-points */
         plotPS2D_curve(C,
                        colrinc,lineinc,
                        contour,contlbl,&lbloffset,
                        hdnline,applyfill,pr_ptID);
      } else {
         /* Plot the curve using spline-approximated data-points */
         plotPS2D_spline_curve(C,
                               spline,colrinc,lineinc,
                               contour,contlbl,&lbloffset,
                               hdnline,applyfill,pr_ptID);
      }
   }
}


/*
 * Plot the curve in PS
 */
static void plotPS2D_curve(C, 
                           colrinc, lineinc,
                           contour, contlbl, lbloffset,
                           hiddenline, applyfill, pr_ptID)
CNcurveptr C;
int        colrinc, lineinc;
int        contour, contlbl, *lbloffset;
int        hiddenline, applyfill, pr_ptID;
{
   CNpointptr  pt_head=NULL, pt_tail=NULL, P;
   PSPoint     points[MAX_ARR_SIZE];
   double      x, y;
   int         count, i, linepat; 
   int         linetype, linecolor, marktype, markcolor, fillcolor, filltype;
   int         marksize;
   double      cxmin,cxmax,cymin,cymax,czmin,czmax;

   if (C==NULL || C->pointhead==NULL) return;

   /*
    * Make a copy of the points in the curve
    */
   CNcopy_abslog_pointlist(&pt_head, &pt_tail,
                           C->pointhead, C->pointtail,
                           xabs, yabs, zabs, xlog, ylog, zlog);
   if (pt_head == NULL) return;
 
   /*
    * Add a new point if the filltype is non-zero
    */
   if (C->curv_pr.filltype != CN_FILL_NONE || hiddenline) {
      (void) CNinsert_tailpoint(&pt_head, &pt_tail,
                                pt_head->x,
                                pt_head->y,
                                pt_head->z,
                                pt_head->ID);
   }
 
   /*
    * Clipping against the plot window is not necessary,
    * because in 2D, the domain boundary is always smaller than the
    * plot window.  Note that at least one clipping must be done.
    * If this is NOT done, the Xpoint array will be filled with
    * negative/large numbers corresponding to points outside the
    * drawing area, which slows down the drawing significantly!
    */
 
   /*
    * Clip the curve against the domain boundaries first
    */
 
   /* Domain boundaries */
   cxmin =   xmin;
   cxmax =   xmax;
   cymin =   ymin;
   cymax =   ymax;
   czmin = - CN_LARGE;
   czmax =   CN_LARGE;

   /* Clip - this modifies the temporary point list */
   CNclip_pointlist(&pt_head,&pt_tail,
                    cxmin,cxmax,cymin,cymax,czmin,czmax,1,1,0,0);
 
   if (pt_head == NULL) return;
 
   /* rescale points, save in array */
   for (P=pt_head; P!=NULL; P=P->next) {
      trn_world_to_PS_nolog(P->x,P->y,&x,&y);
      P->x = x;
      P->y = y;
      P->z = 0.0;
   }
 
   /*
    * Set the properties of the curve.
    */

   /* Set the line type */
   linetype = C->curv_pr.linetype;
   if (linetype != 0) linetype += lineinc;

   /* Set the line color */
   linecolor = C->curv_pr.linecolor + colrinc;

   /* Set the marker type */
   marktype = C->curv_pr.marktype;
   if (marktype != 0) marktype += lineinc;

   /* Set the marker type */
   marksize = C->curv_pr.marksize;

   /* Set the marker color */
   markcolor = C->curv_pr.markcolor + colrinc;

   /* Set the filltype - special treatment for hiddenline */
   filltype = C->curv_pr.filltype;
   if (hiddenline && C->curv_pr.filltype==CN_FILL_NONE) 
      filltype = CN_FILL_SOLID;
   if (!applyfill) filltype = CN_FILL_NONE; 

   /* Set the fill color */
   fillcolor = C->curv_pr.fillcolor;

   if (filltype != CN_FILL_NONE) {

      /* Fill the curve first */
      fillPS2D_curve(pt_head,pt_tail,filltype,fillcolor);

      /* Use the specified linecolor */
      PXlineColorPS(linecolor);

      /* Reset the outline color if necessary */
      if (hiddenline) PXlineColorPS(4);

   } else {

      /* Use the specified linecolor */
      PXlineColorPS(linecolor);

   }


   /*
    * Can only plot some 1000-or-so points at a time, so
    * go thru loop until we run out of points.
    * count  = points plotted during each major loop iteration
    */

   /* Set the linepattern and linewidth */
   linepat = PXlinetypPS(linetype,C->curv_pr.linewidth);

   /* First point */
   P = pt_head;

   while (P != NULL && linepat) {

      /* set count to 0 */
      count = 0;

      /* save points in array */
      for ( ; P!=NULL && count<MAX_ARR_SIZE; P=P->next) {
         points[count].x = P->x;
         points[count].y = P->y;
         count++;
      }

      if (count==1) {
         /* Draw the point */
         (void) fprintf(ips,"newpath\n%.2f %.2f moveto\n",
                        points[0].x,points[0].y);
         (void) fprintf(ips,"%.2f %.2f lineto\n",
                        points[0].x,points[0].y);
         (void) fprintf(ips,"stroke\n");
      } else if (count>1) {
         /* Draw the curve */
         (void) fprintf(ips,"newpath\n%.2f %.2f moveto\n",
                        points[0].x,points[0].y);
         for (i=1; i<count; i++) 
            (void) fprintf(ips,"%.2f %.2f lineto\n", points[i].x,points[i].y);
         (void) fprintf(ips,"stroke\n");
      }

      /* if there are more points to come, repeat the last point */
      if (P != NULL) P = P->prev;
   }

   /* Reset the linetype */
   (void) PXlinetypPS(CN_LN_SOLID,1);

   /* Draw the markers - send the original list */
   plotPS2D_markers(C->pointhead, C->pointtail,
                   marktype, marksize, markcolor, pr_ptID, contour);

   /* Put on the contour labels if necessary */
   if (contour && contlbl) 
      plotPS2D_contlabel(pt_head, pt_tail,
                         C->curv_pr.linetype, C->curv_pr.linelabel,
                         (*lbloffset)++);

   /* Delete the point-list */
   CNdelete_point_list(&pt_head, &pt_tail);

   /* Reset foreground color */
   PXsetColorPS(0);
}


/*
 * Fill the curve in PS
 * The pointlist has already been scaled and clipped against
 * the domain and plot boundaries
 */
/*ARGSUSED*/
static void fillPS2D_curve(pt_head, pt_tail, filltype, fillcolor)
CNpointptr pt_head, pt_tail;
int        filltype, fillcolor;
{
   CNpointptr  P;
   PSPoint     points[MAX_ARR_SIZE];
   int         count;

   if (pt_head == NULL) return;

   /* If the curve is not to be filled then get out now */
   if (filltype == 0) return;

   /* rescale points, save in array */
   count = 0;
   for (P=pt_head; P!=NULL && count<MAX_ARR_SIZE; P=P->next) {
      points[count].x = P->x;
      points[count].y = P->y;
      count++;
   }
   
   /* Fill the polygon */
   PXfillPS_polygon(points,count,
                    filltype, PXpolyColorIndexPS(fillcolor), CN_LN_NONE, 0, 1);
}


/*
 * Plot the interpolated curve in PS
 */
static void plotPS2D_spline_curve(C,
                                 splinetype,
                                 colrinc, lineinc,
                                 contour, contlbl, lbloffset,
                                 hiddenline, applyfill, pr_ptID)
CNcurveptr C;
int        splinetype;
int        colrinc, lineinc;
int        contour, contlbl, *lbloffset;
int        hiddenline, applyfill, pr_ptID;
{
   CNpointptr  pt_head=NULL, pt_tail=NULL, P;
   PSPoint     points[MAX_ARR_SIZE];
   double      *xarr, *yarr, *zarr, *xs, *ys, *zs;
   double      x, y;
   double      dist;
   int         npts, nspts, ndiv = 20, closed = 0;
   int         count, i, linepat; 
   int         linetype, linecolor, marktype, markcolor, fillcolor, filltype;
   int         marksize;
   double      cxmin,cxmax,cymin,cymax,czmin,czmax;

   if (C==NULL || C->pointhead==NULL) return;

   /*
    * Make a copy of the points in the curve
    */
   CNcopy_abslog_pointlist(&pt_head, &pt_tail,
                           C->pointhead, C->pointtail,
                           xabs, yabs, zabs, xlog, ylog, zlog);

   /* Count the number of points in the curve */
   npts = CNcount_points(pt_head, pt_tail);

   /*
    * Allocate double-precision arrays to hold x and y values
    * for the original and interpolated data
    */
   xarr = CNcreate_1D_double_array(npts);
   yarr = CNcreate_1D_double_array(npts);
   zarr = CNcreate_1D_double_array(npts);
   xs   = CNcreate_1D_double_array(npts*(ndiv+5));
   ys   = CNcreate_1D_double_array(npts*(ndiv+5));
   zs   = CNcreate_1D_double_array(npts*(ndiv+5));

   /* Copy the data from the curve to the arrays */
   i=0;
   for (P=pt_head; P!=NULL; P=P->next) {
      xarr[i] = P->x;
      yarr[i] = P->y;
      zarr[i] = P->z;
      i++;
   }

   /* Distance between 1st and last point */
   closed = 0;
   dist = (xarr[0] - xarr[npts-1])*(xarr[0] - xarr[npts-1]) +
          (yarr[0] - yarr[npts-1])*(yarr[0] - yarr[npts-1]) +
          (zarr[0] - zarr[npts-1])*(zarr[0] - zarr[npts-1]);
   if (dist < CN_SMALL) closed = 1;
   if (C->curv_pr.filltype != CN_FILL_NONE || hiddenline) closed=1;
   if (closed && dist<CN_SMALL && npts>1) npts--;

   /* Interpolate using splines */
   CNcreate_spline(xarr,npts,xs,&nspts,ndiv,splinetype,closed);
   CNcreate_spline(yarr,npts,ys,&nspts,ndiv,splinetype,closed);
   CNcreate_spline(zarr,npts,zs,&nspts,ndiv,splinetype,closed);

   /* Transfer the arrays back to a pointlist */
   CNdelete_point_list(&pt_head,&pt_tail);
   for (i=0; i<nspts; i++)
      (void) CNinsert_tailpoint(&pt_head,&pt_tail,xs[i],ys[i],zs[i],i);
 
   /* Make sure the closed curve is really closed - this comes from */
   /* a problem with the spline interpolation                       */
   if (closed)
      (void) CNinsert_tailpoint(&pt_head,&pt_tail,xs[0],ys[0],zs[0],i);

   /*
    * Clipping against the plot window is not necessary,
    * because in 2D, the domain boundary is always smaller than the
    * plot window.  Note that at least one clipping must be done.
    * If this is NOT done, the Xpoint array will be filled with
    * negative/large numbers corresponding to points outside the
    * drawing area, which slows down the drawing significantly!
    */
 
   /*
    * Clip the curve against the domain boundaries first
    */
 
   /* Domain boundaries */
   cxmin =   xmin;
   cxmax =   xmax;
   cymin =   ymin;
   cymax =   ymax;
   czmin = - CN_LARGE;
   czmax =   CN_LARGE;

   /* Clip - this modifies on the temporary point list */
   CNclip_pointlist(&pt_head,&pt_tail,
                    cxmin,cxmax,cymin,cymax,czmin,czmax,1,1,0,0);
   if (pt_head == NULL) return;

   /* rescale points, save in array */
   for (P=pt_head; P!=NULL; P=P->next) {
      trn_world_to_PS_nolog(P->x,P->y,&x,&y);
      P->x = x;
      P->y = y;
      P->z = 0.0;
   }

   /*
    * Set the properties of the curve.
    */

   /* Set the line type */
   linetype = C->curv_pr.linetype;
   if (linetype != 0) linetype += lineinc;

   /* Set the line color */
   linecolor = C->curv_pr.linecolor + colrinc;

   /* Set the marker type */
   marktype = C->curv_pr.marktype;
   if (marktype != 0) marktype += lineinc;

   /* Set the marker size */
   marksize = C->curv_pr.marksize;

   /* Set the marker color */
   markcolor = C->curv_pr.markcolor + colrinc;

   /* Set the filltype - special treatment for hiddenline */
   filltype = C->curv_pr.filltype;
   if (hiddenline && C->curv_pr.filltype==CN_FILL_NONE) 
      filltype = CN_FILL_SOLID;
   if (!applyfill) filltype = CN_FILL_NONE; 

   /* Set the fill color */
   fillcolor = C->curv_pr.fillcolor;

   if (filltype != CN_FILL_NONE) {

      /* Fill the curve first */
      fillPS2D_curve(pt_head,pt_tail,filltype,fillcolor);

      /* Use the specified linecolor */
      PXlineColorPS(linecolor);

      /* Reset the outline color if necessary */
      if (hiddenline) PXlineColorPS(4);

   } else {

      /* Use the specified linecolor */
      PXlineColorPS(linecolor);

   }


   /*
    * Can only plot some 1000-or-so points at a time, so
    * go thru loop until we run out of points.
    * P      = running count of points
    * count  = points plotted during each major loop iteration
    */

   /* Set the linepattern and linewidth */
   linepat = PXlinetypPS(linetype,C->curv_pr.linewidth);

   /* First point */
   P = pt_head;

   /* go thru loop until we run out of points */
   while ((P!=NULL) && linepat) {
      /* rescale points, save in array */
      count = 0;
      for ( ; (P!=NULL) && count<MAX_ARR_SIZE; P=P->next) {
         points[count].x = P->x;
         points[count].y = P->y;
         count++;
      }

      if (count==1) {
         /* Draw the point */
         (void) fprintf(ips,"newpath\n%.2f %.2f moveto\n",
                        points[0].x,points[0].y);
         (void) fprintf(ips,"%.2f %.2f lineto\n",
                        points[0].x,points[0].y);
         (void) fprintf(ips,"stroke\n");
      } else if (count>1) {
         /* Draw the curve */
         (void) fprintf(ips,"newpath\n%.2f %.2f moveto\n",
                        points[0].x,points[0].y);
         for (i=1; i<count; i++)
            (void) fprintf(ips,"%.2f %.2f lineto\n", points[i].x,points[i].y);
         (void) fprintf(ips,"stroke\n");
      }

      /* if there are more points to come, repeat the last point */
      if (P != NULL) P=P->prev;
   }

   /* Reset the linetype */
   (void) PXlinetypPS(CN_LN_SOLID,1);

   /* Draw the markers */
   plotPS2D_markers(C->pointhead, C->pointtail,
                    marktype, marksize, markcolor, pr_ptID, contour);

   /* Put on the contour labels if necessary */
   if (contour && contlbl) 
      plotPS2D_contlabel(pt_head, pt_tail,
                         C->curv_pr.linetype, C->curv_pr.linelabel,
                         (*lbloffset)++);

   /* Free the arrays */
   CNfree_1D_double_array(xarr);
   CNfree_1D_double_array(yarr);
   CNfree_1D_double_array(zarr);
   CNfree_1D_double_array(xs);
   CNfree_1D_double_array(ys);
   CNfree_1D_double_array(zs);

   /* Delete the point-list */
   CNdelete_point_list(&pt_head, &pt_tail);

   /* Reset */
   PXsetColorPS(0);
}


/*
 * Plot the curve markers in PS
 */
/*ARGSUSED*/
static void plotPS2D_markers(pointhead,pointtail,
                             marktype,marksize,markcolor,pr_ptID,contour)
CNpointptr pointhead, pointtail;
int        marktype, marksize, markcolor;
int        pr_ptID, contour;
{
   CNpointptr  P, pt_head=NULL, pt_tail=NULL;
   double      x,y;
   double      cxmin, cxmax, cymin, cymax;
   double      pxmin, pxmax, pymin, pymax;

   if (pointhead == NULL) return;

   /*
    * Make a copy of the points in the curve
    */
   CNcopy_abslog_pointlist(&pt_head, &pt_tail,
                           pointhead, pointtail,
                           xabs, yabs, zabs, xlog, ylog, zlog);

   /* Set the linetype and linewidth */
   (void) PXlinetypPS(CN_LN_SOLID,1);
   PXlineColorPS(markcolor);

   /* Domain (Clip) boundaries */
   cxmin = xmin;
   cxmax = xmax;
   cymin = ymin;
   cymax = ymax;

   /* Plot boundaries */
   pxmin = Pgxmin;
   pxmax = Pgxmax;
   pymin = Pgymin;
   pymax = Pgymax;

   /* go thru loop until we run out of points */
   for (P=pt_head; P!=NULL; P=P->next) {

      if (P->x < cxmin || P->x > cxmax) continue;
      if (P->y < cymin || P->y > cymax) continue;

      /* rescale points */
      trn_world_to_PS_nolog(P->x,P->y,&x,&y);

      /* plot the points only when the point is inside the window */
      if (x < pxmin || x > pxmax) continue;
      if (y < pymin || y > pymax) continue;

      /* plot the marker */
      PXmarkerPS(marktype,marksize,x,y);
   }

   /* Draw curve-point ID's */
   if (pr_ptID && !contour)
      plotPS2D_pointIDs(pt_head, pt_tail, CN_TRUE);
 
   /* Delete the point-list */
   CNdelete_point_list(&pt_head, &pt_tail);

   /* Reset */
   PXsetColorPS(0);
}


/*
 * LABELS
 */
 
/*
 * Plot the point ID labels in POSTSCRIPT
 */
/*ARGSUSED*/
static void plotPS2D_pointIDs(pt_head,pt_tail,nolog)
CNpointptr pt_head, pt_tail;
int        nolog;
{
   CNpointptr  P;
   double      x,y;
   char        label[CN_MAXCHAR];
   double      rxmin, rxmax, rymin, rymax;
 
   if (pt_head == NULL) return;
 
   /* Boundary */
   rxmin = Pxmin - scale*10;
   rxmax = Pxmax + scale*10;
   rymin = Pymin - scale*10;
   rymax = Pymax + scale*10;
 
   /* Set the font */
   (void) fprintf(ips,"setMkrsLblFont\n");

   /* Set the linetype, width and color */
   (void) PXlinetypPS(CN_LN_SOLID,1);
   PXlineColorPS(0);
 
   /* Go thru each point */
   for (P=pt_head; P!=NULL; P=P->next) {
      if (nolog)
      trn_world_to_PS_nolog(P->x,P->y,&x,&y);
      else
      trn_world_to_PS(P->x,P->y,&x,&y);
 
      /* Print the label only if the point is in bounds */
      if (x > rxmin && x < rxmax && y > rymin && y < rymax) {
 
         (void) sprintf(label,"P%d",P->ID);
         (void) fprintf(ips,"%.2f %.2f moveto ", x+2.0*scale,y);
         (void) fprintf(ips,"(%s) MIJM LEJ show\n",PXmodifyPS_string(label));
      }
   }
}
 
/*
 * Plot the node ID labels in POSTSCRIPT
 */
/*ARGSUSED*/
static void plotPS2D_nodeIDs(nd_head,nd_tail)
CNnodeptr nd_head, nd_tail;
{
   CNnodeptr   N;
   double      x,y;
   char        label[CN_MAXCHAR];
   double      rxmin, rxmax, rymin, rymax;
 
   if (nd_head == NULL) return;
 
   /* Boundary */
   rxmin = Pxmin - scale*10;
   rxmax = Pxmax + scale*10;
   rymin = Pymin - scale*10;
   rymax = Pymax + scale*10;
 
   /* Set the font */
   (void) fprintf(ips,"setMkrsLblFont\n");

   /* Set the linetype, width and color */
   (void) PXlinetypPS(CN_LN_SOLID,1);
   PXlineColorPS(0);
 
   /*
    * Multiple nodes could share a single point so be smart about this
    */
   /* Reset all the point-flags */
   for (N=nd_head; N!=NULL; N=N->next) N->coord->flag = 0;
 
   /* Go thru each node */
   for (N=nd_head; N!=NULL; N=N->next) {
      trn_world_to_PS(N->coord->x,N->coord->y,&x,&y);
 
      /* Print the label only if the point is in bounds */
      if (x > rxmin && x < rxmax && y > rymin && y < rymax) {
 
         (void) sprintf(label,"N%d",N->ID);
         (void) fprintf(ips,"%.2f %.2f moveto ",
                        x-2.0*scale,y-N->coord->flag*10.0*scale);
         (void) fprintf(ips,"(%s) MIJM RIJ show\n",
                        PXmodifyPS_string(label));

         /* Increment the point-flag */
         (N->coord->flag)++;
      }
   }
 
   /* Reset all the point-flags */
   for (N=nd_head; N!=NULL; N=N->next) N->coord->flag = 0;
}
 
/*
 * Plot the triangle ID labels in POSTSCRIPT
 */
/*ARGSUSED*/
static void plotPS2D_triaIDs(tr_head,tr_tail)
CNtriaptr tr_head, tr_tail;
{
   CNtriaptr   T;
   double      x,y;
   char        label[CN_MAXCHAR];
   double      rxmin, rxmax, rymin, rymax;
   double      midx, midy;
 
   if (tr_head == NULL) return;
 
   /* Boundary */
   rxmin = Pxmin - scale*10;
   rxmax = Pxmax + scale*10;
   rymin = Pymin - scale*10;
   rymax = Pymax + scale*10;
 
   /* Set the font */
   (void) fprintf(ips,"setMkrsLblFont\n");

   /* Set the linetype, width and color */
   (void) PXlinetypPS(CN_LN_SOLID,1);
   PXlineColorPS(0);
 
   /* Go thru each triangle */
   for (T=tr_head; T!=NULL; T=T->next) {
      /* Get the x-y value of the triangle-midpoint */
      midx = T->n1->coord->x + T->n2->coord->x + T->n3->coord->x;
      midy = T->n1->coord->y + T->n2->coord->y + T->n3->coord->y;
      midx = midx/3.0;
      midy = midy/3.0;
      trn_world_to_PS(midx,midy,&x,&y);
 
      /* Print the label only if the point is in bounds */
      if (x > rxmin && x < rxmax && y > rymin && y < rymax) {
 
         (void) sprintf(label,"T%d",T->ID);
         (void) fprintf(ips,"%.2f %.2f moveto ",x,y);
         (void) fprintf(ips,"(%s) MIJM CEJ show\n",
                        PXmodifyPS_string(label));

      }
   }
}
 
/*
 * Plot the rectangle ID labels in POSTSCRIPT
 */
/*ARGSUSED*/
static void plotPS2D_rectIDs(rt_head,rt_tail)
CNrectptr rt_head, rt_tail;
{
   CNrectptr   R;
   double      x,y;
   char        label[CN_MAXCHAR];
   double      rxmin, rxmax, rymin, rymax;
   double      midx, midy;
 
   if (rt_head == NULL) return;
 
   /* Boundary */
   rxmin = Pxmin - scale*10;
   rxmax = Pxmax + scale*10;
   rymin = Pymin - scale*10;
   rymax = Pymax + scale*10;
 
   /* Set the font */
   (void) fprintf(ips,"setMkrsLblFont\n");

   /* Set the linetype, width and color */
   (void) PXlinetypPS(CN_LN_SOLID,1);
   PXlineColorPS(0);
 
   /* Go thru each rectangle */
   for (R=rt_head; R!=NULL; R=R->next) {
      /* Get the x-y value of the rectangle-midpoint */
      midx = R->n1->coord->x + R->n2->coord->x +
             R->n3->coord->x + R->n4->coord->y;
      midy = R->n1->coord->y + R->n2->coord->y +
             R->n3->coord->y + R->n4->coord->y;
      midx = midx/4.0;
      midy = midy/4.0;
      trn_world_to_PS(midx,midy,&x,&y);
 
      /* Print the label only if the point is in bounds */
      if (x > rxmin && x < rxmax && y > rymin && y < rymax) {
 
         (void) sprintf(label,"R%d",R->ID);
         (void) fprintf(ips,"%.2f %.2f moveto ",x,y);
         (void) fprintf(ips,"(%s) MIJM CEJ show\n",
                        PXmodifyPS_string(label));

      }
   }
}


/*
 * ANNOTATIONS
 */

/*
 * Plot annotations
 */
static void annotatePS2D()
{
   CNannotptr  AP;
   CNdslistptr DS;
   int         FOUND=CN_FALSE;

   /* Scan thru and see if there is anything to do here */
   if (plotdata->annothead) FOUND = CN_TRUE; 
   for (DS=plotdata->datahead; DS!=NULL && !FOUND; DS=DS->next)
      if (DS->Dptr->data_pr.plotannot && DS->Dptr->annothead)
         FOUND = CN_TRUE;
   if (!FOUND) return;

   (void) fprintf(ips,"%%------------------------------------------------\n");
   (void) fprintf(ips,"%%--------------Draw the Annotations--------------\n");
   (void) fprintf(ips,"%%------------------------------------------------\n");

   /* Set the font */
   (void) fprintf(ips,"setMkrsLblFont\n");

   /* Plot the annotations in each dataset */
   for (DS=plotdata->datahead; DS!=NULL; DS=DS->next)
      if (DS->Dptr->data_pr.plotannot) {
         for (AP=DS->Dptr->annothead; AP!=NULL; AP=AP->next)
            plotPS2D_single_annotation(AP);
      }

   /* Plot the annotations in the plotset */
   for (AP=plotdata->annothead; AP!=NULL; AP=AP->next)
      plotPS2D_single_annotation(AP);
}

/*
 * Plot a single annotation
 */
static void plotPS2D_single_annotation(AP)
CNannotptr AP;
{
   if (AP == NULL) return;

   switch (AP->type) {
   case CN_AN_RECT : /* Draw a rectangle */
                     plotPS2D_annot_rect(AP);
                     break;
   case CN_AN_LINE : /* Draw a line      */
                     plotPS2D_annot_line(AP);
                     break;
   case CN_AN_ARROW: /* Draw a arrow     */
                     plotPS2D_annot_arrow(AP);
                     break;
   case CN_AN_POINT: /* Draw a point     */
                     plotPS2D_annot_point(AP);
                     break;
   case CN_AN_TEXT : /* Draw a text label */
                     plotPS2D_annot_text(AP);
                     break;
   default         : break;
   }
}


/*
 * Plot an annotation rectangle
 */
static void plotPS2D_annot_rect(AP)
CNannotptr AP;
{
   PSPoint  points[MAX_ARR_SIZE];
   double   x1,y1,x2,y2;
   int      i=0;
   double   rxmin, rxmax, rymin, rymax;
   double   bxmin, bxmax, bymin, bymax;

   /*
    * The annotated object can be specified either
    *  (a) position relative to data coordinates
    *  (b) absolute position in percentage of width/height
    */

   /* Clipping boundary */
   rxmin = Pgxmin;
   rxmax = Pgxmax;
   rymin = Pgymin;
   rymax = Pgymax;
 
   if (AP->property.absolute) {
 
      /* Rescale the points - the values are in percentages */
      x1 = rxmin + AP->pt1.x * (rxmax-rxmin);
      y1 = rymin + AP->pt1.y * (rymax-rymin);
      x2 = rxmin + AP->pt2.x * (rxmax-rxmin);
      y2 = rymin + AP->pt2.y * (rymax-rymin);

   } else {

      /* Clip against the real-world boundary */
      if (AP->property.doclip) {
         rxmin = Pxmin;
         rxmax = Pxmax;
         rymin = Pymin;
         rymax = Pymax;
      }

      /* Rescale the points */
      trn_world_to_PS(AP->pt1.x,AP->pt1.y,&x1,&y1);
      trn_world_to_PS(AP->pt2.x,AP->pt2.y,&x2,&y2);

   }

   /* Check to see if the box is inside the plot area */
   if ((y1 < rymin && y2 < rymin) || (y1 > rymax && y2 > rymax)) return;
   if ((x1 < rxmin && x2 < rxmin) || (x1 > rxmax && x2 > rxmax)) return;

   /* Get the dimensions of the box */
   bxmin = (x1 < x2) ? x1 : x2;
   bxmax = (x1 < x2) ? x2 : x1;
   bymin = (y1 < y2) ? y1 : y2;
   bymax = (y1 < y2) ? y2 : y1;

   /* Clipping boundaries */
   if (bxmin < rxmin) bxmin = rxmin;
   if (bxmax > rxmax) bxmax = rxmax;
   if (bymin < rymin) bymin = rymin;
   if (bymax > rymax) bymax = rymax;

   /* Save points in an array */
   points[i].x = bxmin;   points[i].y = bymin;   i++;
   points[i].x = bxmax;   points[i].y = bymin;   i++;
   points[i].x = bxmax;   points[i].y = bymax;   i++;
   points[i].x = bxmin;   points[i].y = bymax;   i++;
   points[i].x = bxmin;   points[i].y = bymin;   i++;

   /* Fill the polygon */
   PXfillPS_polygon(points, i,
                   (int) AP->property.filltype,
                   PXpolyColorIndexPS( (int) AP->property.fillcolor ),
                   (int) AP->property.linetype,
                   (int) AP->property.linecolor,
                   (int) AP->property.linewidth);

   /* If there is text attached, draw it */
   if (AP->property.linelabel) {
      /* Set the text color */
      if ((AP->property.filltype!=CN_FILL_NONE) && (AP->property.fillcolor==0))
         PXlineColorPS(-1);
      else
         PXlineColorPS(0);

      /* Draw the label centered */
      (void) fprintf(ips,"%.2f %.2f moveto ",
                     0.5*(bxmin+bxmax),0.5*(bymin+bymax));
      (void) fprintf(ips,"(%s) MIJM CEJ show\n",
                     PXmodifyPS_string(AP->property.linelabel));

      /* Reset the line color */
      PXlineColorPS(0);
   }
}

/*
 * Plot an annotation line
 */
static void plotPS2D_annot_line(AP)
CNannotptr AP;
{
   double rxmin,rxmax,rymin,rymax;
   double cxmin,cxmax,cymin,cymax;
   double x1,y1,x2,y2,x,y;
   int    p1_clipped=CN_FALSE, p2_clipped=CN_FALSE;

   /*
    * The annotated object can be specified either
    *  (a) position relative to data coordinates
    *  (b) absolute position in percentage of width/height
    */

   /* Clipping boundary */
   rxmin = Pgxmin;
   rxmax = Pgxmax;
   rymin = Pgymin;
   rymax = Pgymax;
 
   if (AP->property.absolute) {
 
      /* Rescale the points - the values are in percentages */
      x1 = rxmin + AP->pt1.x * (rxmax-rxmin);
      y1 = rymin + AP->pt1.y * (rymax-rymin);
      x2 = rxmin + AP->pt2.x * (rxmax-rxmin);
      y2 = rymin + AP->pt2.y * (rymax-rymin);

   } else {
      /* Get the dimensions of the line */
      x1 = AP->pt1.x;
      y1 = AP->pt1.y;
      x2 = AP->pt2.x;
      y2 = AP->pt2.y;

      /* Do pre-clipping aaginst the real-world boundaries */
      if (AP->property.doclip) {
         /* Clipping boundary */
         cxmin = ((xlog) ? pow(10.0,xmin) : xmin);
         cxmax = ((xlog) ? pow(10.0,xmax) : xmax);
         cymin = ((ylog) ? pow(10.0,ymin) : ymin);
         cymax = ((ylog) ? pow(10.0,ymax) : ymax);

         if ((x1 < cxmin && x2 < cxmin) || (x1 > cxmax && x2 > cxmax)) return;
         if ((y1 < cymin && y2 < cymin) || (y1 > cymax && y2 > cymax)) return;
         clipPS2D_in_xmin(&x1,&y1,&x2,&y2,cxmin,&p1_clipped,&p2_clipped);
         clipPS2D_in_ymin(&x1,&y1,&x2,&y2,cymin,&p1_clipped,&p2_clipped);
         clipPS2D_in_xmax(&x1,&y1,&x2,&y2,cxmax,&p1_clipped,&p2_clipped);
         clipPS2D_in_ymax(&x1,&y1,&x2,&y2,cymax,&p1_clipped,&p2_clipped);
      }

      /* Rescale the points */
      x=x1;  y=y1;
      trn_world_to_PS(x,y,&x1,&y1);
      x=x2;  y=y2;
      trn_world_to_PS(x,y,&x2,&y2);
   }

   /* Check to see if the line is inside the plot area */
   if ((x1 < rxmin && x2 < rxmin) || (x1 > rxmax && x2 > rxmax)) return;
   if ((y1 < rymin && y2 < rymin) || (y1 > rymax && y2 > rymax)) return;

   /* Do another clipping */
   clipPS2D_in_xmin(&x1,&y1,&x2,&y2,rxmin,&p1_clipped,&p2_clipped);
   clipPS2D_in_ymin(&x1,&y1,&x2,&y2,rymin,&p1_clipped,&p2_clipped);
   clipPS2D_in_xmax(&x1,&y1,&x2,&y2,rxmax,&p1_clipped,&p2_clipped);
   clipPS2D_in_ymax(&x1,&y1,&x2,&y2,rymax,&p1_clipped,&p2_clipped);

   /* Draw the line */
   PXlineColorPS((int)AP->property.linecolor);
   PXdrawPS_line(x1,y1,x2,y2,
                 (int)AP->property.linetype, (int)AP->property.linewidth);

   /* Draw the markers */
   if ((AP->property.marktype != CN_MK_NONE) && (!p1_clipped || !p2_clipped)) {
      (void) PXlinetypPS(CN_LN_SOLID,1);
      PXlineColorPS(AP->property.markcolor);
      if (!p1_clipped) PXmarkerPS((int)AP->property.marktype,
                                  (int)AP->property.marksize,x1,y1);
      if (!p2_clipped) PXmarkerPS((int)AP->property.marktype,
                                  (int)AP->property.marksize,x2,y2);
   }

   /* If there is text attached, draw it */
   if (!p1_clipped && AP->property.linelabel) {
      /* Set the label color */
      PXlineColorPS(0);

      /* Text justification is based on comparison of x1, x2 */
      if (x1 > x2) {
         /* Left-justified */
         (void) fprintf(ips,"%.2f %.2f moveto ",x1+6.0*scale,y1);
         (void) fprintf(ips,"(%s) MIJM LEJ show\n",
                        PXmodifyPS_string(AP->property.linelabel));
      } else if (x1 == x2) {
         /* Center-justified */
         if (y1 > y2) {
         (void) fprintf(ips,"%.2f %.2f moveto ",x1,y1+6*scale);
         (void) fprintf(ips,"(%s) UPJ CEJ show\n",
                        PXmodifyPS_string(AP->property.linelabel));
         } else {
         (void) fprintf(ips,"%.2f %.2f moveto ",x1,y1-6*scale);
         (void) fprintf(ips,"(%s) BOJM CEJ show\n",
                        PXmodifyPS_string(AP->property.linelabel));
         }
      } else {
         /* Right-justified */
         (void) fprintf(ips,"%.2f %.2f moveto ",x1-6.0*scale,y1);
         (void) fprintf(ips,"(%s) MIJM RIJ show\n",
                        PXmodifyPS_string(AP->property.linelabel));
      }
   }

   /* Reset */
   PXlineColorPS(0);
}

/*
 * Plot an annotation arrow
 */
static void plotPS2D_annot_arrow(AP)
CNannotptr AP;
{
   double cxmin,cxmax,cymin,cymax;
   double rxmin,rxmax,rymin,rymax;
   double x1,y1,x2,y2,x,y;
   int    p1_clipped=CN_FALSE, p2_clipped=CN_FALSE;

   /*
    * The annotated object can be specified either
    *  (a) position relative to data coordinates
    *  (b) absolute position in percentage of width/height
    */
 
   /* Clipping boundary */
   rxmin = Pgxmin;
   rxmax = Pgxmax;
   rymin = Pgymin;
   rymax = Pgymax;
 
   if (AP->property.absolute) {
 
      /* Rescale the points - the values are in percentages */
      x1 = rxmin + AP->pt1.x * (rxmax-rxmin);
      y1 = rymin + AP->pt1.y * (rymax-rymin);
      x2 = rxmin + AP->pt2.x * (rxmax-rxmin);
      y2 = rymin + AP->pt2.y * (rymax-rymin);

   } else {
      /* Get the dimensions of the line */
      x1 = AP->pt1.x;
      y1 = AP->pt1.y;
      x2 = AP->pt2.x;
      y2 = AP->pt2.y;

      /* Do pre-clipping aaginst the real-world boundaries */
      if (AP->property.doclip) {
         /* Clipping boundary */
         cxmin = ((xlog) ? pow(10.0,xmin) : xmin);
         cxmax = ((xlog) ? pow(10.0,xmax) : xmax);
         cymin = ((ylog) ? pow(10.0,ymin) : ymin);
         cymax = ((ylog) ? pow(10.0,ymax) : ymax);

         if ((x1 < cxmin && x2 < cxmin) || (x1 > cxmax && x2 > cxmax)) return;
         if ((y1 < cymin && y2 < cymin) || (y1 > cymax && y2 > cymax)) return;
         clipPS2D_in_xmin(&x1,&y1,&x2,&y2,cxmin,&p1_clipped,&p2_clipped);
         clipPS2D_in_ymin(&x1,&y1,&x2,&y2,cymin,&p1_clipped,&p2_clipped);
         clipPS2D_in_xmax(&x1,&y1,&x2,&y2,cxmax,&p1_clipped,&p2_clipped);
         clipPS2D_in_ymax(&x1,&y1,&x2,&y2,cymax,&p1_clipped,&p2_clipped);
      }

      /* Rescale the points */
      x=x1;  y=y1;
      trn_world_to_PS(x,y,&x1,&y1);
      x=x2;  y=y2;
      trn_world_to_PS(x,y,&x2,&y2);
   }

   /* Draw the arrow */
   plotPS2D_arrow(x1,y1,x2,y2,
                  p1_clipped, p2_clipped,
                  AP->property.linelabel,
                  (int)AP->property.linetype,
                  (int)AP->property.linecolor,
                  (int)AP->property.linewidth,
                  (int)AP->property.marktype,
                  (int)AP->property.marksize,
                  (int)AP->property.markcolor, 1);
}


/*
 * Draw an arrow
 */
static void plotPS2D_arrow(x1,y1,x2,y2,
                           p1_clipped, p2_clipped,
                           linelabel,
                           linetype,linecolor,linewidth,
                           marktype,marksize,markcolor,
                           arrowhead)
double x1,y1,x2,y2;
int    p1_clipped, p2_clipped;
char   *linelabel;
int    linetype,linecolor,linewidth;
int    marktype,marksize,markcolor;
int    arrowhead;
{
   double rxmin, rxmax, rymin, rymax;
   double vx, vy, vd;
   double xa, ya, xb, yb, xc, yc;
 
   /* Clipping boundary */
   rxmin = Pgxmin;
   rxmax = Pgxmax;
   rymin = Pgymin;
   rymax = Pgymax;

   /* Check to see if the line is inside the plot area */
   if ((x1 < rxmin && x2 < rxmin) || (x1 > rxmax && x2 > rxmax)) return;
   if ((y1 < rymin && y2 < rymin) || (y1 > rymax && y2 > rymax)) return;
 
   /* Clip the arrow against the plot boundaries */
   clipPS2D_in_xmin(&x1,&y1,&x2,&y2,rxmin,&p1_clipped,&p2_clipped);
   clipPS2D_in_ymin(&x1,&y1,&x2,&y2,rymin,&p1_clipped,&p2_clipped);
   clipPS2D_in_xmax(&x1,&y1,&x2,&y2,rxmax,&p1_clipped,&p2_clipped);
   clipPS2D_in_ymax(&x1,&y1,&x2,&y2,rymax,&p1_clipped,&p2_clipped);
 
   /* Draw the arrow origin */
   (void) PXlinetypPS(CN_LN_SOLID,1);
   PXlineColorPS(markcolor);
   if (!p1_clipped) PXmarkerPS(marktype,marksize,x1,y1);
 
   /* Draw the arrow line */
   PXlineColorPS(linecolor);
   PXdrawPS_line(x1,y1,x2,y2, linetype, linewidth);
 
   /* Draw the arrow head */
   if (!p2_clipped && arrowhead) {
      /* Draw the arrow head */
      (void) PXlinetypPS(CN_LN_SOLID,linewidth);
      vx = x2-x1;
      vy = y2-y1;
      vd = sqrt(vx*vx + vy*vy);
      if (vd > 5.0) {
      xa = x2 - 5.0*vx/vd;
      ya = y2 - 5.0*vy/vd;
      xb = xa - 2.0*vy/vd;
      yb = ya + 2.0*vx/vd;
      xc = xa + 2.0*vy/vd;
      yc = ya - 2.0*vx/vd;
      PXdrawPS_line(x2,y2,xb,yb, CN_LN_SOLID, linewidth);
      PXdrawPS_line(x2,y2,xc,yc, CN_LN_SOLID, linewidth);
      }
   }

   /* If there is text attached, draw it */
   if (!p1_clipped && linelabel) {
      /* Set the label color */
      PXlineColorPS(0);

      /* Text justification is based on comparison of x1, x2 */
      if (x1 > x2) {
         /* Left-justified */
         (void) fprintf(ips,"%.2f %.2f moveto ",x1+6.0*scale,y1);
         (void) fprintf(ips,"(%s) MIJM LEJ show\n",
                        PXmodifyPS_string(linelabel));
      } else if (x1 == x2) {
         /* Center-justified */
         if (y1 > y2) {
         (void) fprintf(ips,"%.2f %.2f moveto ",x1,y1+6*scale);
         (void) fprintf(ips,"(%s) UPJ CEJ show\n",
                        PXmodifyPS_string(linelabel));
         } else {
         (void) fprintf(ips,"%.2f %.2f moveto ",x1,y1-6*scale);
         (void) fprintf(ips,"(%s) BOJM CEJ show\n",
                        PXmodifyPS_string(linelabel));
         }
      } else {
         /* Right-justified */
         (void) fprintf(ips,"%.2f %.2f moveto ",x1-6.0*scale,y1);
         (void) fprintf(ips,"(%s) MIJM RIJ show\n",
                        PXmodifyPS_string(linelabel));
      }
   }

 
   /* Reset */
   PXlineColorPS(0);
   (void) PXlinetypPS(CN_LN_SOLID,1);
}


/*
 * Plot an annotation point
 */
static void plotPS2D_annot_point(AP)
CNannotptr AP;
{
   double  x1,y1;
   double  rxmin, rxmax, rymin, rymax;
   double  cxmin, cxmax, cymin, cymax;

   /*
    * The annotated object can be specified either
    *  (a) position relative to data coordinates
    *  (b) absolute position in percentage of width/height
    */
 
   /* Clipping boundary */
   rxmin = Pgxmin;
   rxmax = Pgxmax;
   rymin = Pgymin;
   rymax = Pgymax;
 
   if (AP->property.absolute) {
 
      /* Rescale the points - the values are in percentages */
      x1 = rxmin + AP->pt1.x * (rxmax-rxmin);
      y1 = rymin + AP->pt1.y * (rymax-rymin);

   } else {

      /* Do a pre-clipping */
      if (AP->property.doclip) {
         /* Clipping boundary */
         cxmin = ((xlog) ? pow(10.0,xmin) : xmin);
         cxmax = ((xlog) ? pow(10.0,xmax) : xmax);
         cymin = ((ylog) ? pow(10.0,ymin) : ymin);
         cymax = ((ylog) ? pow(10.0,ymax) : ymax);

         if ((AP->pt1.x < cxmin || AP->pt1.x > cxmax) ||
             (AP->pt1.y < cymin || AP->pt1.y > cymax))
           return;
      }

      /* Rescale the points */
      trn_world_to_PS(AP->pt1.x,AP->pt1.y,&x1,&y1);
   }

   /* Draw the markers */
   if ((rxmin < x1 && x1 < rxmax) && (rymin < y1 && y1 < rymax)) {
      (void) PXlinetypPS(CN_LN_SOLID,1);
      PXlineColorPS(AP->property.markcolor);
      PXmarkerPS((int)AP->property.marktype,
                 (int)AP->property.marksize,x1,y1);

      /* If there is text attached, draw it */
      if (AP->property.linelabel) {

         /* Reset the label color */
         PXlineColorPS(0);

         /* Left-justified, left of the point */
         (void) fprintf(ips,"%.2f %.2f moveto ",x1+6.0,y1);
         (void) fprintf(ips,"(%s) MIJM LEJ show\n",
                     PXmodifyPS_string(AP->property.linelabel));
      }
   }

   /* Reset */
   PXlineColorPS(0);
}

/*
 * Plot an annotation label
 */
static void plotPS2D_annot_text(AP)
CNannotptr AP;
{
   double  x1,y1;
   double  rxmin, rxmax, rymin, rymax;
   double  cxmin, cxmax, cymin, cymax;

   /*
    * The annotated object can be specified either
    *  (a) position relative to data coordinates
    *  (b) absolute position in percentage of width/height
    */
 
   /* Clipping boundary */
   rxmin = Pgxmin;
   rxmax = Pgxmax;
   rymin = Pgymin;
   rymax = Pgymax;
 
   if (AP->property.absolute) {
 
      /* Rescale the points - the values are in percentages */
      x1 = rxmin + AP->pt1.x * (rxmax-rxmin);
      y1 = rymin + AP->pt1.y * (rymax-rymin);

   } else {
      /* Do a pre-clipping */
      if (AP->property.doclip) {
         /* Clipping boundary */
         cxmin = ((xlog) ? pow(10.0,xmin) : xmin);
         cxmax = ((xlog) ? pow(10.0,xmax) : xmax);
         cymin = ((ylog) ? pow(10.0,ymin) : ymin);
         cymax = ((ylog) ? pow(10.0,ymax) : ymax);

         if ((AP->pt1.x < cxmin || AP->pt1.x > cxmax) ||
             (AP->pt1.y < cymin || AP->pt1.y > cymax))
           return;
      }

      /* Rescale the points */
      trn_world_to_PS(AP->pt1.x,AP->pt1.y,&x1,&y1);
   }

   /* Draw the text */
   if ((rxmin < x1 && x1 < rxmax) && (rymin < y1 && y1 < rymax)) {
      if (AP->property.linelabel) {
         (void) PXlinetypPS(CN_LN_SOLID,1);
         PXlineColorPS(0);
         /* Center-justified */
         (void) fprintf(ips,"%.2f %.2f moveto ",x1,y1);
         (void) fprintf(ips,"(%s) MIJM CEJ show\n",
                        PXmodifyPS_string(AP->property.linelabel));
      }
   }
}

/*
 * Clip a line against xmin
 */
static void clipPS2D_in_xmin(x1,y1,x2,y2,rxmin,p1_clipped,p2_clipped)
double *x1, *y1, *x2, *y2, rxmin;
int    *p1_clipped, *p2_clipped;
{
   double t;

   /* Clip the line against rxmin */
   if (*x1 < rxmin && *x2 > rxmin) {
      t = (rxmin - *x1)/(*x2 - *x1);
      *x1 = *x1 + t*(*x2 - *x1);
      *y1 = *y1 + t*(*y2 - *y1);
      if (*x1 < rxmin) *x1 = rxmin;
      *p1_clipped = CN_TRUE;
   } else if (*x2 < rxmin && *x1 > rxmin) {
      t = (rxmin - *x2)/(*x1 - *x2);
      *x2 = *x2 + t*(*x1 - *x2);
      *y2 = *y2 + t*(*y1 - *y2);
      if (*x2 < rxmin) *x2 = rxmin;
      *p2_clipped = CN_TRUE;
   }
}

/*
 * Clip a line against ymin
 */
static void clipPS2D_in_ymin(x1,y1,x2,y2,rymin,p1_clipped,p2_clipped)
double *x1, *y1, *x2, *y2, rymin;
int    *p1_clipped, *p2_clipped;
{
   clipPS2D_in_xmin(y1,x1,y2,x2,rymin,p1_clipped,p2_clipped);
}

/*
 * Clip a line against xmax
 */
static void clipPS2D_in_xmax(x1,y1,x2,y2,rxmax,p1_clipped,p2_clipped)
double *x1, *y1, *x2, *y2, rxmax;
int    *p1_clipped, *p2_clipped;
{
   double t;

   /* Clip the line against rxmax */
   if (*x1 < rxmax && *x2 > rxmax) {
      t = (rxmax - *x2)/(*x1 - *x2);
      *x2 = *x2 + t*(*x1 - *x2);
      *y2 = *y2 + t*(*y1 - *y2);
      if (*x2 > rxmax) *x2 = rxmax;
      *p2_clipped = CN_TRUE;
   } else if (*x2 < rxmax && *x1 > rxmax) {
      t = (rxmax - *x1)/(*x2 - *x1);
      *x1 = *x1 + t*(*x2 - *x1);
      *y1 = *y1 + t*(*y2 - *y1);
      if (*x1 > rxmax) *x1 = rxmax;
      *p1_clipped = CN_TRUE;
   }
}

/*
 * Clip a line against ymax
 */
static void clipPS2D_in_ymax(x1,y1,x2,y2,rymax,p1_clipped,p2_clipped)
double *x1, *y1, *x2, *y2, rymax;
int    *p1_clipped, *p2_clipped;
{
   clipPS2D_in_xmax(y1,x1,y2,x2,rymax,p1_clipped,p2_clipped);
}



/*
 * LABELS
 */

/*
 * Plot the contour labels in PS
 * The pointlist is in plot coordinates
 */
static void plotPS2D_contlabel(pointhead, pointtail, 
                               linetype, linelabel, lbloffset)
CNpointptr pointhead, pointtail;
int        linetype;
char       *linelabel;
int        lbloffset;
{
   CNpointptr   P;
   double       x, y;
   double       dist, last_label_dist;
   double       idl_dist = 150.0*scale;
   int          npts;

   if ((pointhead == NULL) || (linelabel == NULL)) return;

   /* Reset the color */
   PXsetColorPS(0);

   /* label the contours at given points intervals */
   if (linetype == CN_LN_SOLID || linetype == CN_LN_DASHED) {

      /* Count the number of points first */
      npts = CNcount_points(pointhead, pointtail);

      /* 2 or fewer points are treated differently */
      if (npts == 1) {
 
         /* Put the label at the point */
         x = pointhead->x;
         y = pointhead->y;
 
         /* Draw the text */
         (void) drawPS2D_label(x,y,x+1.0,y,linelabel);

      } else if (npts == 2) {
 
         /* Put the label at the midpoint */
         x = 0.5*(pointhead->x + pointtail->x);
         y = 0.5*(pointhead->y + pointtail->y);
 
         /* Draw the text */
         (void) drawPS2D_label(x,y,pointtail->x,pointtail->y,linelabel);
 
      } else {

         /* Place the label at equally-spaced locations on the curve */
         last_label_dist = 0.3*idl_dist + (lbloffset % 5)*0.1*idl_dist;
         for (P=pointhead->next; P!=NULL; P=P->next) {
 
            /* Distance from previous point to current point */
            dist = sqrt( (P->prev->x - P->x)*(P->prev->x - P->x) +
                         (P->prev->y - P->y)*(P->prev->y - P->y));
 
            /* Distance from the last label */
            last_label_dist += dist;
 
            /*
             * If the distance from the last label is greater then
             * the ideal distance, then draw the label
             */
            if (last_label_dist > idl_dist) {
               /* Draw the label at the midpoint of the current segment */
               x = 0.5*(P->prev->x + P->x);
               y = 0.5*(P->prev->y + P->y);
 
               /* Draw the text */
               (void) drawPS2D_label(x,y,P->x,P->y,linelabel);
 
               /* Reset the last label distance */
               last_label_dist = 0.0;
            }
         }
      }
   }
}


/*
 * Put in a label inside the plot
 * Draw the text , but only if the point is inbounds
 */
static void drawPS2D_label( xm, ym, xn, yn, label)
double xm, ym, xn, yn;
char   *label;
{
   double angle;

   /* Calculate the angle of the label */
   if (fabs(xm-xn)< 1.0e-5)
      angle = -90.0;
   else
      angle = 360.0/(2*3.14159)*atan((yn-ym)/(xn-xm));

   /* Put on the label */
   (void) fprintf(ips,"setPlotLblFont\n");
   (void) fprintf(ips,"%.3g %.3g translate ",xm,ym);
   (void) fprintf(ips,"%.3g rotate (%s) plotlabel  %.3g rotate  ",
                  angle,label,-angle);
   (void) fprintf(ips,"%.3g %.3g translate\n",-xm,-ym);
}


/*
 * Plot line labels on the side if necessary
 */
static void plotPS2D_sidelabels()
{
   short  contclip;
   double xoffset;
   int    LABEL_FOUND = CNplotset_has_linelabels(plotdata);
   int    FCONT_FOUND = CNplotset_has_filledcontour(plotdata);
   if (!FCONT_FOUND) FCONT_FOUND = CNplotset_has_grid(plotdata);

   /* Plot linelabels if necessary */
   if (LABEL_FOUND) {
      xoffset = 10*scale;
      if (FCONT_FOUND) xoffset += 45*scale;
      PXplotPS_linelabels(plotdata, hiddenline,
                         Pxmin, Pxmax, Pymin, Pymax, xoffset);
   }

   /* Plot colored scales if necessary */
   if (FCONT_FOUND) {
      xoffset  = 10*scale;
      contclip = CN_FALSE;
      if (contour_dptr) contclip = contour_dptr->data_pr.contclip;
      PXplotPS_contscale(cstephead, csteptail,
                         Pxmin, Pxmax, Pymin, Pymax, xoffset, contclip);
   }
}



/*
 * TRANSLATION ROUTINES
 */

/* 
 * Translate world to PS coordinates and apply log options
 * This is for local use only because it uses local static variables.
 */
static void trn_world_to_PS(xw, yw, xp, yp)
double  xw,  yw;                   /* World coordinates            */
double *xp, *yp;                   /* PS plot coordinates          */
{
   PXtranslate_world_to_PS(xw, yw, xp, yp,
                            Pxmin, Pxmax, Pymin, Pymax,
                            xmin, xmax, ymin, ymax,
                            xlog, ylog, xflip, yflip);
}


/* 
 * Translate world to PS coordinates.
 * Don't apply log options - the data has been previously converted to log
 * This is for local use only because it uses local static variables.
 */
static void trn_world_to_PS_nolog(xw, yw, xp, yp)
double  xw,  yw;                   /* World coordinates            */
double *xp, *yp;                   /* PS plot coordinates          */
{
   PXtranslate_world_to_PS(xw, yw, xp, yp,
                            Pxmin, Pxmax, Pymin, Pymax,
                            xmin, xmax, ymin, ymax,
                            CN_FALSE, CN_FALSE, xflip, yflip);
}

