/* expr.c - expression parsing and evaluation */

/* Written 1994,1995 by Werner Almesberger */


#include <stdlib.h>
#include <string.h>
#include <ctype.h>

#include "common.h"
#include "expr.h"
#include "parse.h"


#define C_ ,
#define UNARY(n,s,t,l) \
  EXPR *n(void) \
  { \
    static const char chars[] = s; \
    static const EXPR_TYPE types[] = { t }; \
    const char *p; \
    \
    if (!non_eof() || !(p = strchr(chars,ch))) { \
      back(); \
      return l(); \
    } \
    return construct(types[p-chars],l(),NULL); \
  }
#define BINARY(n,s,t,l) \
  EXPR *n(void) \
  { \
    EXPR *a; \
    static const char chars[] = s; \
    static const EXPR_TYPE types[] = { t }; \
    char first; \
    int i; \
    \
    a = l(); \
    if (!next()) { \
      back(); \
      return a; \
    } \
    for (i = 0; chars[i]; i += 2) \
      if (ch == chars[i]) break; \
    if (!chars[i]) { \
      back(); \
      return a; \
    } \
    first = ch; \
    (void) non_eof(); \
    for (i = 0; chars[i]; i += 2) \
      if (first == chars[i] && ch == chars[i+1]) break; \
    if (!chars[i]) { \
      if (ch) back(); \
      for (i = 0; chars[i]; i += 2) \
        if (first == chars[i] && !chars[i+1]) break; \
        if (!chars[i]) parse_error("invalid operator \"%c%c\"",first,ch); \
    } \
    return construct(types[i >> 1],a,n()); \
  }


static EXPR *construct(EXPR_TYPE type,EXPR *a,EXPR *b)
{
    EXPR *n;

    n = alloc_t(EXPR);
    n->type = type;
    n->u.op.a = a;
    n->u.op.b = b;
    return n;
}


static EXPR *atom(void)
{
    EXPR *n;
    char *end;

    if (non_eof()) {
	if (ch != '(') parse_error("constant, variable or \"(\" expected");
	n = expr_parse();
	if (non_eof() != ')') parse_error("\")\" expected");
	return n;
    }
    if (quoted_string) parse_error("string constants are not supported");
    n = alloc_t(EXPR);
    if (isdigit(*token)) {
	n->type = et_const;
	n->u.num = strtol(token,&end,0);
	if (*end) parse_error("invalid number \"%s\"",token);
    }
    else {
	char more[2];

	n->type = et_var;
	n->u.var.name = stralloc(token);
	n->u.var.item = NULL;
	more[0] = ':';
	more[1] = 0;
	while (next() == more[0]) {
	    string();
	    if (!(n->u.var.name = realloc((char *) n->u.var.name,(size_t)
	      (strlen(n->u.var.name)+strlen(token)+2)))) die("out of memory");
	    strcat((char *) n->u.var.name,more);
	    strcat((char *) n->u.var.name,token);
	    more[0] = more[0] == '.' ? ':' : '.';
	}
	back();
    }
    return n;
}


static UNARY(op_unary,"!-",et_not C_ et_sign,atom)
static BINARY(op_mult,"*\0/\0%\0",et_mult C_ et_div C_ et_mod,op_unary)
static BINARY(op_add,"+\0-\0",et_plus C_ et_minus,op_mult)
static BINARY(op_rel,"==<=>=!=<\0>\0",et_eq C_ et_le C_ et_ge C_ et_ne C_ et_lt
  C_ et_gt,op_add)
static BINARY(op_land,"&&",et_and,op_rel)
BINARY(expr_parse,"||^^",et_or C_ et_xor,op_land)


int eval(EXPR *this)
{
    switch (this->type) {
	case et_var:
	    {
		const ITEM *item;
		int ind;

		if (!this->u.var.item)
		    if (!(this->u.var.item = lookup(this->u.var.name,&ind,0)))
			return 0;
		item = this->u.var.item;
		switch (item->type) {
		    case it_bool:
			return item->u.bool.on;
		    case it_number:
			return item->u.number.value;
		    case it_choice:
			return ind == 1 ? 1 : ind == item->u.choice.curr;
		    case it_set:
			return ind == -1 ? !!item->u.set.set :
			  (item->u.set.set >> ind) & 1;
		    default:
			return 0;
		}
	    }
	case et_const:
	    return this->u.num;
	case et_plus:
	    return eval(this->u.op.a)+eval(this->u.op.b);
	case et_minus:
	    return eval(this->u.op.a)-eval(this->u.op.b);
	case et_mult:
	    return eval(this->u.op.a)*eval(this->u.op.b);
	case et_div:
	    return eval(this->u.op.a)/eval(this->u.op.b);
	case et_mod:
	    return eval(this->u.op.a) % eval(this->u.op.b);
	case et_and:
	    return eval(this->u.op.a) && eval(this->u.op.b);
	case et_or:
	    return eval(this->u.op.a) || eval(this->u.op.b);
	case et_xor:
	    return !eval(this->u.op.a) != !eval(this->u.op.b);
	case et_eq:
	    return eval(this->u.op.a) == eval(this->u.op.b);
	case et_ne:
	    return eval(this->u.op.a) != eval(this->u.op.b);
	case et_gt:
	    return eval(this->u.op.a) > eval(this->u.op.b);
	case et_ge:
	    return eval(this->u.op.a) >= eval(this->u.op.b);
	case et_lt:
	    return eval(this->u.op.a) < eval(this->u.op.b);
	case et_le:
	    return eval(this->u.op.a) <= eval(this->u.op.b);
	case et_sign:
	    return -eval(this->u.op.a);
	case et_not:
	    return !eval(this->u.op.a);
    }
    return 0; /* for gcc ... */
}
