/* edit.c - full-screen configuration editor */

/* Written 1994,1995 by Werner Almesberger */


#include <stdlib.h>
#include <stdarg.h>
#include <string.h>
#include <ctype.h>
#include <assert.h>

#include "common.h"
#include "cursed.h"
#include "screen.h"
#include "scend.h"
#include "parse.h"
#include "descr.h"
#include "warn.h"


#define TOP_MARGIN	2
#define BOTTOM_MARGIN	4
#define STATUS_LINES	1
#define LEFT_MARGIN	3
#define RIGHT_MARGIN	1
#define SUMMARY_MARGIN	2
#define LEVEL_INDENT	2
#define FIELD_INDENT	4
#define HELP_MARGIN	4


static ITEM *top;
int written = 0;


static int tree_up(ITEM **curr)
{
    ITEM *walk,*next;

    walk = *curr;
    if (next = get_prev(walk->prev)) {
	walk = next;
	while (get_section(walk) && is_visible(walk))
	    for (walk = get_section(walk); next = get(walk->next); walk = next);
    }
    else if (!(walk = walk->parent)) return 0;
	else if (walk->type == it_if)
		if (!tree_up(&walk)) return 0;
    *curr = walk;
    return 1;
}


static int tree_down(ITEM **curr)
{
    ITEM *walk,*next;

    walk = *curr;
    if (get_section(walk) && is_visible(walk)) walk = get_section(walk);
    else if (next = get(walk->next)) walk = next;
	else {
		for (walk = walk->parent; walk; walk = walk->parent)
		    if (next = get(walk->next)) {
			walk = next;
			break;
		    }
		if (!walk) return 0;
	    }
    *curr = walk;
    return 1;
}


static int tree_level(const ITEM *curr)
{
    int level;

    for (level = -1; curr; curr = curr->parent)
	if (curr->type != it_if)level++;
    return level;
}


static void permanent(void)
{
    const char *const title = "Scend 0.5";

    if (strlen(title) > scr_cols()) return;
    scr_move(0,0);
    scr_bold(1);
    scr_printf(" %s ",title);
    scr_bold(0);
    if (!subtitle || strlen(title)+strlen(subtitle)+3 > scr_cols()) return;
    scr_printf(" %s",subtitle);
}


static void status(const char *msg)
{
    static const char *last = NULL;

    if (!msg)
	if (last) msg = last;
	else return;
    else {
	if (last) free((char *) last);
	last = stralloc(msg);
    }
    scr_move(scr_rows()-1,0);
    scr_puts(msg);
    scr_clrtoeol();
}


static int getkey(void)
{
    int key;

    while (1)
	switch (key = getch()) {
	    case 'L'-64:
		scr_redraw();
		break;
	    case 'Z'-64:
		scr_suspend(1);
		break;
	    default:
		return key;
	}
}


static int yesno(const char *msg)
{
    int key;

    scr_move(scr_rows()-1,0);
    scr_printf("%s (Y/N) ",msg);
    scr_clrtoeol();
    scr_sync();
    do key = getkey();
    while (!strchr("YyNn\007",key));
    status("");
    return key == 'y' || key == 'Y';
}


static void set_menu(void *dummy,...)
{
    va_list ap;
    char *item;
    int x,y,i;

    x = y = 0;
    for (i = 0; i < BOTTOM_MARGIN-STATUS_LINES-1; i++) {
	scr_move(scr_rows()-BOTTOM_MARGIN+1+i,0);
	scr_clrtoeol();
    }
    va_start(ap,msg);
    while (item = va_arg(ap,char *)) {
	if (!*item) continue;
	if (strlen(item) > scr_cols()) break;
	if (x+strlen(item) > scr_cols()) {
	    if (++y >= BOTTOM_MARGIN-STATUS_LINES-1) break;
	    x = 0;
	}
	scr_move(scr_rows()-BOTTOM_MARGIN+1+y,x);
	scr_puts(item);
	x += strlen(item)+2;
    }
    va_end(ap);
}


static int newline(int *x,int *y)
{
    char ch;

    *x = HELP_MARGIN;
    if (++(*y) < scr_rows()-BOTTOM_MARGIN) return 1;
    set_menu(NULL,"[Enter] for more","[Q] to stop",NULL);
    scr_sync();
    ch = getkey();
    scr_clear();
    permanent();
    *y = TOP_MARGIN;
    return ch != 'q' && ch != 'Q' && ch != 'H'-64 && ch != 127;
}


static void show_help(const char *txt)
{
    SCR_CTX ctx;
    char tmp[2];
    char *spc,*nl,*end;
    int x,y;

    if (!txt) return;
    tmp[1] = 0;
    scr_save(&ctx);
    scr_clear();
    permanent();
    x = HELP_MARGIN;
    y = TOP_MARGIN;
    while (1) {
	spc = strchr(txt,' ');
	nl = strchr(txt,'\n');
	end = !spc && !nl ? strchr(txt,0) : spc && (nl > spc || !nl) ? spc : nl;
	if (x+end-txt > scr_cols()-HELP_MARGIN)
	    if (!newline(&x,&y)) break;
	scr_move(y,x);
	while (txt < end) {
	    tmp[0] = *txt++;
	    scr_puts(tmp);
	    x++;
	}
	if (*end == ' ') x++;
	else if (*end == '\n') {
		if (x != HELP_MARGIN || y != TOP_MARGIN)
		    if (!newline(&x,&y)) break;
		if (x != HELP_MARGIN || y != TOP_MARGIN)
		    if (!newline(&x,&y)) break;
	    }
	    else {
		set_menu(NULL,"Press [Enter] to return",NULL);
		scr_sync();
		(void) getkey();
		break;
	    }
	txt++;
    }
    scr_restore(ctx);
    scr_sync();
}


static int label_start(const ITEM *curr)
{
    return LEFT_MARGIN+tree_level(curr)*LEVEL_INDENT;
}


static void field_pos(const ITEM *curr,int nth,int *x,int *y)
{
    int left,summary,len,i,col;

    if (!num_fields(curr)) {
	if (x) *x = 0;
	if (y) *y = 0;
	return;
    }
    /* first plan: try to fit everything on the same line */
    left = scr_cols()-RIGHT_MARGIN-strlen(curr->name.long_name)-
      2*SUMMARY_MARGIN-label_start(curr);
    if (!(summary = summarize(curr,NULL,left))) left += SUMMARY_MARGIN;
    else left -= summary;
    len = 0;
    for (i = num_fields(curr)-1; i >= 0; i--) len += field_size(curr,i)+2+!!i;
    if (len <= left) {
	if (y) *y = 0;
	if (!x) return;
	*x = scr_cols()-RIGHT_MARGIN-len-label_start(curr);
	for (i = 0; i < nth; i++) *x += field_size(curr,i)+3;
	return;
    }
    /* second plan: use new lines */
    if (y) *y = 1;
    col = label_start(curr)+FIELD_INDENT;
    for (i = 0; i <= nth; i++) {
	if (field_size(curr,i)+col+2+RIGHT_MARGIN <= scr_cols()) {
	    if (x) *x = col;
	}
	else {
	    col = label_start(curr)+FIELD_INDENT;
	    if (x) *x = col;
	    if (y) (*y)++;
	}
	col += field_size(curr,i)+3;
    }
    if (x) (*x) -= label_start(curr);
}


static int y_size(const ITEM *curr)
{
    int y;

    if (!num_fields(curr)) return 1;
    field_pos(curr,num_fields(curr)-1,NULL,&y);
    return y ? y+1 : 1;
}


static int item_y(ITEM *root,const ITEM *this)
{
    int y;

    y = 0;
    while (root != this) {
	y += y_size(root);
	if (!tree_down(&root)) return -1;
    }
    return y;
}


static int ideal_size_up(ITEM *curr)
{
    ITEM *walk;
    int last;

    walk = curr;
    last = tree_level(curr);
    while (tree_up(&walk))
	if (tree_level(walk) < last) break;
    return item_y(walk,curr);
}


static int ideal_size_down(ITEM *curr)
{
    ITEM *walk;

    walk = curr;
    while (tree_down(&walk))
	if (tree_level(walk) <= tree_level(curr)) return item_y(curr,walk);
    return item_y(curr,walk)+y_size(walk);
}


static void draw_field(const ITEM *curr,int i)
{
    char value[MAX_TOKEN+1];
    const char *frame;
    int j;

    if (curr->flags & FLAG_ABSENT)
	for (j = field_size(curr,i)+2; j; j--) scr_puts("-");
    else {
	frame = field_active(curr,i) ? "[]" : "  ";
	field_content(curr,i,value);
	scr_printf("%c%*s%c",frame[0],field_size(curr,i),value,frame[1]);
    }
}


static void cursor(const ITEM *curr,int on)
{
    int x,y,fx,fy;

    if (on) scr_bold(1);
    x = label_start(curr);
    y = item_y(top,curr)+TOP_MARGIN;
    scr_move(y,x);
    scr_puts(curr->name.long_name);
    if (num_fields(curr) && !(curr->flags & FLAG_ABSENT)) {
	field_pos(curr,curr_field(curr),&fx,&fy);
	scr_move(y+fy,x+fx);
	draw_field(curr,curr_field(curr));
    }
    if (on) scr_bold(0);
}


static void edit_field(ITEM *curr)
{
    char *buffer,tmp[2];
    const char *range;
    int max_size;
    int x,y;
    int done,key,i;

    assert(num_fields(curr) == 1);
    max_size = field_size(curr,curr_field(curr));
    buffer = alloc((size_t) (max_size+1));
    field_content(curr,curr_field(curr),buffer);
    field_pos(curr,curr_field(curr),&x,&y);
    x += label_start(curr);
    y += item_y(top,curr)+TOP_MARGIN;
    set_menu(NULL,"[?] for help","[^H]/[Del] to delete one character",
      "[^U]/[^X] to clear the field","[Enter] to set and exit",
      "[^C]/[^G] to quit without changes",NULL);
    if (range = valid_range(curr)) status(range);
    scr_bold(1);
    scr_move(y,x);
    scr_printf("[%*s]",-max_size,buffer);
    tmp[1] = 0;
    done = 0;
    do {
	scr_move(y,x+(int) strlen(buffer)+1);
	scr_sync();
	key = getkey();
	switch (key) {
	    case 'H'-64:
		/* fall through */
	    case 127:
		if (!*buffer) break;
		scr_move(y,x+(int) strlen(buffer));
		scr_puts(" ");
		buffer[strlen(buffer)-1] = 0;
		break;
	    case 'U'-64:
		/* fall through */
	    case 'X'-64:
		scr_move(y,x+1);
		for (i = 0; i < strlen(buffer); i++) scr_puts(" ");
		*buffer = 0;
		break;
	    case '\n':
		done = set_field(curr,curr_field(curr),buffer);
		break;
	    case 'C'-64:
		/* fall through */
	    case 'G'-64:
		done = 1;
		break;
	    case '?':
		show_help(curr->help);
		break;
	    default:
		if (key < 32 || key > 126) break;
		if (strlen(buffer) == max_size) break;
		tmp[0] = key;
		strcat(buffer,tmp);
		if (check_field(curr,curr_field(curr),buffer) != acc_invalid)
		    scr_puts(tmp);
		else buffer[strlen(buffer)-1] = 0;
		break;
	}
    }
    while (!done);
    scr_bold(0);
    status("");
}


static void make_invisible(ITEM **curr)
{
    ITEM *here,*walk;

    while (!visible(*curr,0) || !get_section(*curr)) {
	here = *curr;
	while (1) {
	    if (!tree_up(curr)) return;
	    for (walk = here->parent; walk; walk = walk->parent)
		if (walk == *curr) break;
	    if (walk) break;
	}
    }
}


static void field_move(ITEM *curr,int dir)
{
    int next,fields;

    if (!(fields = num_fields(curr))) return;
    next = curr_field(curr)+dir;
    field_select(curr,next < 0 ? fields-1 : next == fields ? 0 : next);
}


static void draw_from(ITEM *curr)
{
    char buffer[256]; /* ugly */
    int x,y,i,fx,fy;

    y = item_y(top,curr)+TOP_MARGIN;
    for (i = y; i < scr_rows()-BOTTOM_MARGIN; i++) {
	scr_move(i,0);
	scr_clrtoeol();
    }
    while (y < scr_rows()-BOTTOM_MARGIN) {
	scr_move(y,0);
	scr_puts(is_unseen(curr) ? "*" : " ");
	if (!is_visible(curr) && get_section(curr)) scr_puts("+");
	x = label_start(curr);
	scr_move(y,x);
	scr_puts(curr->name.long_name);
	for (i = 0; i < num_fields(curr); i++) {
	    field_pos(curr,i,&fx,&fy);
	    if (x+fy >= scr_rows()-BOTTOM_MARGIN) return;
	    scr_move(y+fy,x+fx);
	    draw_field(curr,i);
	}
	field_pos(curr,0,&fx,NULL);
	fx += x;
	x += strlen(curr->name.long_name);
	if (summarize(curr,buffer,scr_cols()-RIGHT_MARGIN-2*SUMMARY_MARGIN-x)) {
	    scr_move(y,x+SUMMARY_MARGIN);
	    x += strlen(buffer)+SUMMARY_MARGIN;
	    scr_puts(buffer);
	}
	if (i && !fy)
	    for (i = (x+2) & ~1; i < fx-1; i += 2) {
		scr_move(y,i);
		scr_puts(".");
	    }
	y += y_size(curr);
	if (!tree_down(&curr)) return;
    }
    return;
}


static void write_all(const char *outfile)
{
    WARNING *ctx;
    const char *msg;
    int key;

    if (!edit_group) {
	warn_begin(&ctx);
	while (msg = warn_check(&ctx)) {
	    status(msg);
	    set_menu(NULL,"[?] for help","[Enter] to skip","[I] to ignore",
	      "[^H]/[Del] to stop",NULL);
	    scr_sync();
	    do if ((key = getkey()) == '?') show_help(ctx->help); /* hack */
	    while (!strchr("\r\nIi\010\177\003\007",key));
	    status("");
	    if (strchr("\010\177\003\007",key)) return;
	    if (key == '\r' || key == '\n') warn_skip(&ctx);
	    else warn_ignore(&ctx);
	}
    }
    if (!outfile) status("Ok");
    else if (msg = write_cfg(outfile)) status(msg);
	else {
	    status("Written");
	    changed = 0;
	    written = 1;
	}
}


static void loop(const char *outfile)
{
    static const char in[] = "ABCD";
    static const char out[] = "PNFB";
    ITEM *curr;
    int area,key,size;
    int key_state;

    curr = get(descr);
    while (1) {
	scr_clear();
	permanent();
	status(NULL);
	draw_from(top);
	curr->flags |= FLAG_SEEN;
	set_menu(NULL,"up/down to select item","left/right to select value",
	  "[+]/[-] to display/hide","[?] for help","[Enter] to edit",
	  edit_group ? "[U] unset" : "",outfile ? "[W] to write" : "",
	  "[Q] to quit without saving",NULL);
	cursor(curr,1);
	scr_sync();
	for (key_state = 0; key = getkey(); key_state++)
	    if ((!key_state && key != 27) || (key_state == 1 && key != '[' &&
	      key != 'O') || key_state > 1) break;
	cursor(curr,0);
	if (key_state == 2 && strchr(in,key)) key = out[strchr(in,key)-in]-64;
	switch (islower(key) ? toupper(key) : key) {
	    case ' ':
		if (field_action(curr) == act_edit) edit_field(curr);
		else default_action(curr);
		break;
	    case '+':
		(void) visible(curr,1);
		break;
	    case '-':
		make_invisible(&curr);
		break;
	    case '\n':
		if (field_action(curr) == act_edit) edit_field(curr);
		break;
	    case 'B'-64: /* left */
	    case 'H':
		field_move(curr,-1);
		break;
	    case 'F'-64: /* right */
	    case 'L':
		field_move(curr,1);
		break;
	    case 'P'-64: /* up */
	    case 'K':
		(void) tree_up(&curr);
		break;
	    case 'N'-64: /* down */
	    case 'J':
		(void) tree_down(&curr);
		break;
	    case 'U':
		curr->flags |= FLAG_ABSENT;
		break;
	    case 'W':
		write_all(outfile);
		break;
	    case 'Q':
		if ((written && !changed) || yesno(changed ?
		  "Quit without saving changes ?" : "Really quit ?")) {
		    if (test && curr != get(descr))
			die("leaving test while not on top");
		    return;
		}
		break;
	    case '?':
		show_help(curr->help);
		break;
	    case '!':
		scr_suspend(0);
		break;
	    default:
		if (test) die("unrecognized key in test mode");
	}
        if (!is_active(top))
	    if (!(top = get(descr))) die("empty description");
        if (!is_active(curr)) die("recursive reference");
	if (item_y(top,curr) < 0) {
	    size = ideal_size_up(curr);
	    while (item_y(top,curr) < size)
		if (!tree_up(&top)) break;
	    if (!item_y(top,curr) < 0) abort();
	}
	area = scr_rows()-TOP_MARGIN-BOTTOM_MARGIN;
	size = ideal_size_down(curr);
	while (item_y(top,curr)+size > area)
	    if (top == curr || !tree_down(&top)) break;
	if (item_y(top,curr)+y_size(curr) > area) abort();
    }
}


static int unlock_yn(void)
{
    return yesno("Field is protected. Unlock ?");
}


void edit(const char *outfile)
{
    ask_protected = unlock_yn;
    if (!(top = get(descr))) die("empty description");
    loop(outfile);
}
