use std::cell::RefCell;
use std::rc::Rc;

use super::markup_links;
use super::html_escape;

use markup5ever_rcdom::Node;
use markup5ever_rcdom::NodeData;
use markup5ever_rcdom::RcDom;
use html5ever::tendril::TendrilSink;
use html5ever::tree_builder::Attribute;
use html5ever::tree_builder::TreeBuilderOpts;
use html5ever::{parse_document, ParseOpts};

/// Each block contains the text formatted in pango format
#[derive(Debug, Clone, PartialEq)]
pub enum HtmlBlock {
    Text(String),
    Heading(u32, String),
    UList(Vec<String>),
    OList(Vec<String>),
    Code(String),
    Quote(Rc<Vec<HtmlBlock>>),
}

pub fn markup_html_ignore_tags(s: &str, tags: &[&str]) -> Result<Vec<HtmlBlock>, anyhow::Error> {
    let opts = ParseOpts {
        tree_builder: TreeBuilderOpts {
            drop_doctype: true,
            ..Default::default()
        },
        ..Default::default()
    };
    let dom = parse_document(RcDom::default(), opts)
        .from_utf8()
        .read_from(&mut s.as_bytes())?;

    let document = &dom.document;
    let mut html = document.children.borrow().clone();
    // TODO: When writting `<i>...</i>` in markdown, the first children is of type
    // Comment { contents: "raw HTML omitted" }
    // This should be handled more gracefully
    html.retain(|x| {
        match x.data {
            NodeData::Comment {..} => false,
            _ => true,
        }
    });

    if let Some(h) = &html.get(0) {
        if let Some(node) = &h.children.borrow().get(1) {
            let markup = convert_node(node, tags);
            return Ok(trim_text_blocks(markup))
        }
    }
    Err(anyhow::anyhow!(format!("Could not parse {:?}", html.clone())))
}

pub fn markup_html(s: &str) -> Result<Vec<HtmlBlock>, anyhow::Error> {
    let tags = vec!["body", "mx-reply"];
    markup_html_ignore_tags(s, &tags)
}

fn to_pango(t: &str) -> Option<&'static str> {
    let allowed = [
        "u", "del", "s", "em", "i", "strong", "b", "code", "a"
    ];

    if !allowed.contains(&t) {
        return None;
    }

    match t {
        "a" => Some("a"),
        "em" | "i" => Some("i"),
        "strong" | "b" => Some("b"),
        "del" | "s" => Some("s"),
        "u" => Some("u"),
        "code" => Some("tt"),
        _ => None,
    }
}

fn parse_link(node: &Node, attrs: &RefCell<Vec<Attribute>>) -> String {
    let mut link = "".to_string();
    for attr in attrs.borrow().iter() {
        let s = attr.name.local.to_string();
        match &s[..] {
            "href" => {
                link = attr.value.to_string();
            }
            _ => {}
        }
    }

    format!("<a href=\"{}\">{}</a>", html_escape(&link), get_text_content(node))
}

fn get_text_content(node: &Node) -> String {
    node.children.borrow().iter()
        .map(|node| match node.data {
            NodeData::Text { contents: ref c } => markup_links(&html_escape(&c.borrow())),
            NodeData::Element { name: ref n, attrs: ref a, .. } => {
                let inside = get_text_content(node);

                if &n.local == "a" {
                    return parse_link(node, a);
                }

                match to_pango(&n.local) {
                    Some(t) => format!("<{0}>{1}</{0}>", t, inside),
                    None => inside,
                }
            },
            _ => get_text_content(node)
        })
        .collect::<Vec<String>>()
        .concat()
}

fn get_plain_text_content(node: &Node) -> String {
    node.children.borrow().iter()
        .map(|node| match node.data {
            NodeData::Text { contents: ref c } => c.borrow().to_string(),
            NodeData::Element { name: ref n, attrs: ref _a, .. } => {
                let inside = get_plain_text_content(node);
                match to_pango(&n.local) {
                    Some(t) => {
                        // We don't want tt tags inside codeblocks which
                        // already have <tt>.
                        if t != "tt"  {
                            format!("<{0}>{1}</{0}>", t, inside)
                        } else {
                            inside
                        }
                    },
                    None => inside,
                }
            },
            _ => get_plain_text_content(node)
        })
        .collect::<Vec<String>>()
        .concat()
}

fn get_li_elements(node: &Node) -> Vec<String> {
    node.children.borrow().iter()
        .map(|node| match node.data {
            NodeData::Element { name: ref n, .. } if &n.local == "li" => {
                Some(get_text_content(node))
            },
            _ => None
        })
        .filter(|n| n.is_some())
        .map(|n| n.unwrap())
        .collect::<Vec<String>>()
}

fn join_text_blocks(mut blocks: Vec<HtmlBlock>) -> Vec<HtmlBlock> {
    use HtmlBlock::Text;

    blocks.drain(..)
          .fold(Vec::<HtmlBlock>::new(), |mut v, b| {
              let last = v.last();
              if let (&Text(ref c), Some(Text(a))) = (&b, last) {
                  let t = a.to_string() + &c;
                  v.pop();
                  v.push(Text(t));
              } else {
                  v.push(b);
              }

              v
          })
}

fn html_unescape(s: &str) -> String {
    s.to_string()
        .replace("&amp;", "&")
        .replace("&lt;", "<")
        .replace("&gt;", ">")
        .replace("&quot;", "\"")
}

fn trim_text_blocks(mut blocks: Vec<HtmlBlock>) -> Vec<HtmlBlock> {
    blocks.drain(..)
          .fold(vec!(), |mut v, b| {
              if let HtmlBlock::Text(c) = &b {
                  if !c.trim().is_empty() {
                      v.push(HtmlBlock::Text(c.trim().to_string()));
                  }
              } else {
                  v.push(b);
              }
              v
          })
}

fn convert_node(node: &Node, tags_to_ignore: &[&str]) -> Vec<HtmlBlock> {
    let mut output = vec![];
    match node.data {
        NodeData::Text { contents: ref c } => {
            let s = markup_links(&html_escape(&c.borrow()));
            output.push(HtmlBlock::Text(s));
        }

        NodeData::Element {
            name: ref n,
            attrs: ref a,
            ..
        } => {
            match &n.local as &str {
                // tags to ignore
                tag if tags_to_ignore.contains(&tag) => {
                    for child in node.children.borrow().iter() {
                        for block in convert_node(child, tags_to_ignore) {
                            output.push(block);
                        }
                    }
                }

                h if ["h1", "h2", "h3", "h4", "h5", "h6"].contains(&h) => {
                    let n: u32 = h[1..].parse().unwrap_or(6);
                    let text = get_text_content(node);
                    output.push(HtmlBlock::Heading(n, text));
                }

                "a" => {
                    let link = parse_link(node, a);
                    output.push(HtmlBlock::Text(link));
                }

                "pre" => {
                    let text = get_plain_text_content(node);
                    output.push(HtmlBlock::Code(html_unescape(text.trim())));
                }

                "ul" => {
                    let elements = get_li_elements(node);
                    output.push(HtmlBlock::UList(elements));
                }

                "ol" => {
                    let elements = get_li_elements(node);
                    output.push(HtmlBlock::OList(elements));
                }

                "blockquote" => {
                    let mut content = vec![];
                    for child in node.children.borrow().iter() {
                        for block in convert_node(child, tags_to_ignore) {
                            content.push(block);
                        }
                    }
                    content = trim_text_blocks(join_text_blocks(content));
                    output.push(HtmlBlock::Quote(Rc::new(content)));
                }

                "p" => {
                    let content = get_text_content(node);
                    output.push(HtmlBlock::Text(content));
                    output.push(HtmlBlock::Text("\n".to_string()));
                }

                // TODO: to draw a line we can just use a new widget in the message gtk-box
                // a gtkSeparator for example
                "hr" => {
                    output.push(HtmlBlock::Text(html_escape("<hr />")));
                }

                "br" => {
                    output.push(HtmlBlock::Text("\n".to_string()));
                }

                tag => {
                    let content = get_text_content(node);
                    let block = match to_pango(&tag) {
                        Some(t) => format!("<{0}>{1}</{0}>", t, content),
                        None => String::from(content),
                    };

                    output.push(HtmlBlock::Text(block));
                }
            };
        }
        _ => {}
    }

    join_text_blocks(output)
}

#[cfg(test)]
mod test {
    use super::*;
    use pretty_assertions::assert_eq;

    #[test]
    fn test_html_blocks() {
        let text = "<h1>heading 1 <em>italic</em></h1>\n<h2>heading 2</h2>\n<p>Some text with <em>markup</em> and <strong>other</strong> and more text and some <code>inline code</code>, that's all. And maybe some links http://google.es or <a href=\"http://gnome.org\">GNOME</a>.</p>\n<pre><code>Block text\n</code></pre>\n<ul>\n<li>This is a list</li>\n<li>second element</li>\n</ul>\n<ol>\n<li>another list</li>\n<li>that's all</li>\n</ol>\n";
        let blocks = markup_html(text);
        assert!(blocks.is_ok());
        let blocks = blocks.unwrap();
        assert_eq!(blocks.len(), 6);

        assert_eq!(blocks[0], HtmlBlock::Heading(1, "heading 1 <i>italic</i>".to_string()));
        assert_eq!(blocks[1], HtmlBlock::Heading(2, "heading 2".to_string()));

        assert_eq!(if let HtmlBlock::Text(_s) = &blocks[2] { true } else { false }, true);
        assert_eq!(if let HtmlBlock::Code(_s) = &blocks[3] { true } else { false }, true);
        assert_eq!(if let HtmlBlock::UList(_n) = &blocks[4] { true } else { false }, true);
        assert_eq!(if let HtmlBlock::OList(_n) = &blocks[5] { true } else { false }, true);
    }

    #[test]
    fn test_html_blocks_quote() {
        let text = "
<blockquote>
<h1>heading 1 <em>bold</em></h1>
<h2>heading 2</h2>
<p>Some text with <em>markup</em> and <strong>other</strong> and more things ~~strike~~ and more text and some <code>inline code</code>, that's all. And maybe some links http://google.es or &lt;a href=&quot;http://gnome.org&quot;&gt;GNOME&lt;/a&gt;, <a href=\"http://gnome.org\">GNOME</a>.</p>
<pre><code>`Block text
`
</code></pre>
<ul>
<li>This is a list</li>
<li>second element</li>
</ul>
</blockquote>
<p>quote :D</p>
";

        let blocks = markup_html(text);
        assert!(blocks.is_ok());
        let blocks = blocks.unwrap();
        assert_eq!(blocks.len(), 2);

        if let HtmlBlock::Quote(blks) = &blocks[0] {
            assert_eq!(blks.len(), 5);
        }
    }

    #[test]
    fn test_mxreply() {
        let text = "<mx-reply><blockquote><a href=\"https://matrix.to/#/!hwiGbsdSTZIwSRfybq:matrix.org/$1553513281991555ZdMuB:matrix.org\">In reply to</a> <a href=\"https://matrix.to/#/@afranke:matrix.org\">@afranke:matrix.org</a><br><a href=\"https://matrix.to/#/@gergely:polonkai.eu\">Gergely Polonkai</a>: we have https://gitlab.gnome.org/GNOME/fractal/issues/467 and https://gitlab.gnome.org/GNOME/fractal/issues/347 open, does your issue fit any of these two?</blockquote></mx-reply><p>#467 <em>might</em> be it, let me test a bit<p>";

        let blocks = markup_html(text);
        assert!(blocks.is_ok());
        let blocks = blocks.unwrap();
        assert_eq!(blocks.len(), 2);

        if let HtmlBlock::Text(t) = &blocks[0] {
            assert_eq!(t, "<a href=\"https://matrix.to/#/!hwiGbsdSTZIwSRfybq:matrix.org/$1553513281991555ZdMuB:matrix.org\">In reply to</a> <a href=\"https://matrix.to/#/@afranke:matrix.org\">@afranke:matrix.org</a><a href=\"https://matrix.to/#/@gergely:polonkai.eu\">Gergely Polonkai</a>: we have <a href=\"https://gitlab.gnome.org/GNOME/fractal/issues/467\">https://gitlab.gnome.org/GNOME/fractal/issues/467</a> and <a href=\"https://gitlab.gnome.org/GNOME/fractal/issues/347\">https://gitlab.gnome.org/GNOME/fractal/issues/347</a> open, does your issue fit any of these two?\n#467 <i>might</i> be it, let me test a bit");
        }
    }

    #[test]
    fn test_html_lists() {
        let text = "
<ul>
<li>item 1</li>
<li>item 2</li>
</ul>
";

        let blocks = markup_html(text);
        assert!(blocks.is_ok());
        let blocks = blocks.unwrap();
        assert_eq!(blocks.len(), 1);

        if let HtmlBlock::UList(t) = &blocks[0] {
            assert_eq!(t.len(), 2);
            assert_eq!(t[0], "item 1");
            assert_eq!(t[1], "item 2");
        }
    }

    #[test]
    fn test_html_paragraphs() {
        let text = "
<p>text</p>
<p><i>text2</i></p>
<p><b>text3</b></p>
<ul><li>li<li/></ul>
<p>text4</p>
<p>text5</p>
";
        let blocks = markup_html(text);
        assert!(blocks.is_ok());
        let blocks = blocks.unwrap();
        assert!(!blocks.is_empty());
        assert_eq!(blocks.len(), 3);

        if let HtmlBlock::Text(t) = &blocks[0] {
            assert_eq!(t, "text\n\n<i>text2</i>\n\n<b>text3</b>");
        }
        if let HtmlBlock::Text(t) = &blocks[2] {
            assert_eq!(t, "text4\n\ntext5");
        }
    }

    #[test]
    fn newline_in_blockquote() {
        let text = "<blockquote>html was a mistake\nIndeed</blockquote>";
        let matrix_text = "<blockquote>\n<p>a<br />\nb</p>\n<p>c</p>\n</blockquote>\n";

        let blocks = markup_html(text);
        assert!(blocks.is_ok());
        let blocks = blocks.unwrap();
        assert_eq!(blocks.len(), 1);

        if let HtmlBlock::Quote(blk) = &blocks[0] {
            assert_eq!(blk.len(), 1);
            if let HtmlBlock::Text(h) = &blk[0] {
                assert_eq!(h, "html was a mistake\nIndeed");
            }
        }

        let blocks = markup_html(matrix_text);
        assert!(blocks.is_ok());
        let blocks = blocks.unwrap();
        assert_eq!(blocks.len(), 1);

        if let HtmlBlock::Quote(blk) = &blocks[0] {
            assert_eq!(blk.len(), 1);
            if let HtmlBlock::Text(h) = &blk[0] {
                assert_eq!(h, "a\nb\n\nc");
            }
        }
    }

    #[test]
    fn html_blocks_quote_multiple() {
        let text = "<blockquote>\n<p>Some</p>\n</blockquote>\n<p>text</p>\n<blockquote>\n<p>No</p>\n</blockquote>\n<p><strong>u</strong></p>\n";

        let blocks = markup_html(text);
        assert!(blocks.is_ok());
        let blocks = blocks.unwrap();
        assert_eq!(blocks.len(), 4);

        if let HtmlBlock::Quote(blk) = &blocks[0] {
            assert_eq!(blk.len(), 1);
            if let HtmlBlock::Text(h) = &blk[0] {
                assert_eq!(h, "Some");
            }
        }
        if let HtmlBlock::Text(t) = &blocks[1] {
            assert_eq!(t, "text");
        }
        if let HtmlBlock::Quote(blk) = &blocks[2] {
            assert_eq!(blk.len(), 1);
            if let HtmlBlock::Text(h) = &blk[0] {
                assert_eq!(h, "No");
            }
        }
        if let HtmlBlock::Text(t) = &blocks[3] {
            assert_eq!(t, "<b>u</b>");
        }
    }

    #[test]
    fn html_url_and_text() {
        let text = "<a href=\"https://gnome.org\">GNOME</a>text";

        let blocks = markup_html(text);
        assert!(blocks.is_ok());
        let blocks = blocks.unwrap();
        assert_eq!(blocks.len(), 1);
        if let HtmlBlock::Text(h) = &blocks[0] {
            assert_eq!(h, "<a href=\"https://gnome.org\">GNOME</a>text");
        }
    }

    #[test]
    fn codeblock_empty_whitespace() {
       let text ="
<pre><code>
Block text `a` <i>i</i>

space?
</code></pre>
";

        let blocks = markup_html(text);
        assert!(blocks.is_ok());
        let blocks = blocks.unwrap();
        assert_eq!(blocks.len(), 1);
        if let HtmlBlock::Code(t) = &blocks[0] {
            assert_eq!(t, "Block text `a` <i>i</i>\n\nspace?")
        }
    }

    #[test]
    fn escape_amp() {
        let text = "text: <code>&amp;</code> was not scaped as <code>&amp;amp;</code>";
        let blocks = markup_html(text);
        assert!(blocks.is_ok());
        let blocks = blocks.unwrap();
        assert_eq!(blocks.len(), 1);

        if let HtmlBlock::Text(t) = &blocks[0] {
            assert_eq!(t, "text: <tt>&amp;</tt> was not scaped as <tt>&amp;amp;</tt>");
        }
    }

    #[test]
    fn dont_escape_codeblocks() {
       let text ="
<pre><code>
&
</code></pre>
<code>&<code>
";

        let blocks = markup_html(text);
        assert!(blocks.is_ok());
        let blocks = blocks.unwrap();
        assert_eq!(blocks.len(), 2);
        if let HtmlBlock::Code(t) = &blocks[0] {
            assert_eq!(t, "&")
        }
        if let HtmlBlock::Code(t) = &blocks[1] {
            assert_eq!(t, "&")
        }
    }

    // TODO: Temporal test, this should have a special kind of Block to render a
    // widget showing a line.
    #[test]
    fn horizontal_line() {
       let text ="<hr />\n";

        let blocks = markup_html(text);
        assert!(blocks.is_ok());
        let blocks = blocks.unwrap();
        assert_eq!(blocks.len(), 1);

        if let HtmlBlock::Text(t) = &blocks[0] {
            assert_eq!(t, "&lt;hr /&gt;")
        }
    }

    #[test]
    fn newlines_in_text() {
       let text ="
1
2

3

<p>4</p>
<p>5</p>
";
        let text_matrix = "a<br />\nb</p>\n<p>c<br />\nd";

        let blocks = markup_html(text);
        assert!(blocks.is_ok());
        let blocks = blocks.unwrap();
        assert_eq!(blocks.len(), 1);
        if let HtmlBlock::Text(t) = &blocks[0] {
            assert_eq!(t, "1\n2\n\n3\n\n4\n\n5")
        }

        let blocks = markup_html(text_matrix);
        assert!(blocks.is_ok());
        let blocks = blocks.unwrap();
        assert_eq!(blocks.len(), 1);
        if let HtmlBlock::Text(t) = &blocks[0] {
            assert_eq!(t, "a\n\nb\n\nc\nd")
        }
    }

    #[test]
    fn links_inside_code() {
        let text = "
<pre><code>https://gitlab.gnome.org/World/Fractal/</code></pre>\n
";
        let blocks = markup_html(text);
        assert!(blocks.is_ok());
        let blocks = blocks.unwrap();
        assert_eq!(blocks.len(), 1);

        if let HtmlBlock::Code(t) = &blocks[0] {
            assert_eq!(t, "https://gitlab.gnome.org/World/Fractal/");
        }
    }

    #[test]
    fn ci_links() {
        let text = "
[<a href='https://gitlab.gnome.org/World/Fractal'>World/Fractal</a>]
";
        let blocks = markup_html(text);
        assert!(blocks.is_ok());
        let blocks = blocks.unwrap();
        assert_eq!(blocks.len(), 1);

        if let HtmlBlock::Text(s) = &blocks[0] {
            assert_eq!(s, "[<a href=\"https://gitlab.gnome.org/World/Fractal\">World/Fractal</a>]");
        }
    }

    #[test]
    fn newline_after_quotes() {
        let text = "<mx-reply><blockquote><a href=\"https://matrix.org\">In reply to</a> <a href=\"https://matrix.org\">@okias:matrix.org</a><br>Text</blockquote></mx-reply>F";
        let target = "<a href=\"https://matrix.org\">In reply to</a> <a href=\"https://matrix.org\">@okias:matrix.org</a>\nText";
        let blocks = markup_html(text);
        assert!(blocks.is_ok());
        let blocks = blocks.unwrap();
        assert_eq!(blocks.len(), 2);

        if let HtmlBlock::Quote(blk) = &blocks[0] {
            assert_eq!(blk.len(), 1);
            if let HtmlBlock::Text(h) = &blk[0] {
                assert_eq!(h, target);
            }
        }

        if let HtmlBlock::Text(t) = &blocks[1] {
            assert_eq!(t, "F");
        }
    }
}
