/*****************************************************************************\
  colormatch.cpp : Implimentation for the ColorMatcher class

  Copyright (c) 1996 - 2001, Hewlett-Packard Co.
  All rights reserved.

  Redistribution and use in source and binary forms, with or without
  modification, are permitted provided that the following conditions
  are met:
  1. Redistributions of source code must retain the above copyright
     notice, this list of conditions and the following disclaimer.
  2. Redistributions in binary form must reproduce the above copyright
     notice, this list of conditions and the following disclaimer in the
     documentation and/or other materials provided with the distribution.
  3. Neither the name of Hewlett-Packard nor the names of its
     contributors may be used to endorse or promote products derived
     from this software without specific prior written permission.

  THIS SOFTWARE IS PROVIDED BY THE AUTHOR "AS IS" AND ANY EXPRESS OR IMPLIED
  WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF
  MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.  IN
  NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
  SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED
  TO, PATENT INFRINGEMENT; PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS
  OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON
  ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
  (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF
  THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
\*****************************************************************************/

#include "header.h"
#include "hptypes.h"
#include "colormatch.h"

APDK_BEGIN_NAMESPACE

ColorMatcher::ColorMatcher
(
    SystemServices* pSys,
    ColorMap cm,
    unsigned int DyeCount,
    unsigned int iInputWidth
) : ColorPlaneCount(DyeCount),
    InputWidth(iInputWidth),
    pSS(pSys),
    cmap(cm)
{
    constructor_error = NO_ERROR;
    ASSERT(cmap.ulMap1 != NULL);
    StartPlane = K;       // most common case

    if (ColorPlaneCount == 3)     // CMY pen
    {
        StartPlane = C;
    }

    EndPlane = Y;         // most common case
    if (ColorPlaneCount == 6)
    {
        EndPlane = Mlight;
    }
    if (ColorPlaneCount == 1)
    {
        EndPlane = K;
    }

    Contone = (BYTE*)pSS->AllocMem(InputWidth*ColorPlaneCount);
    if (Contone == NULL)
    {
        goto MemoryError;
    }

    Restart();  // this zeroes buffers and sets nextraster counter

    return;

MemoryError:
    constructor_error=ALLOCMEM_ERROR;

    FreeBuffers();
    return;
} //ColorMatcher

ColorMatcher::~ColorMatcher()
{
    DBG1("destroying ColorMatcher \n");

    FreeBuffers();
} //~ColorMatcher


void ColorMatcher::Restart()
// also reset cache when we have one
{
    memset(Contone, 0, InputWidth*ColorPlaneCount);

    started = FALSE;
}

void ColorMatcher::Flush()
// needed to reset cache
{
    if (!started)
    {
        return;
    }
    Restart();
}

void ColorMatcher::FreeBuffers()
{

   pSS->FreeMemory(Contone);

}


BYTE* ColorMatcher::NextOutputRaster()
{
    if (iRastersReady == 0)
    {
        return NULL;
    }

    iRastersReady--;
    iRastersDelivered++;
    return Contone;
} //NextOutputRaster


unsigned int ColorMatcher::GetOutputWidth()
{
    return InputWidth*ColorPlaneCount;
} //GetOutputWidth


unsigned int ColorMatcher::GetMaxOutputWidth()
{
    return InputWidth*ColorPlaneCount;
} //GetMaxOutPutWidth


//////////////////////////////////////////////////////////////////////////////

void ColorMatcher::ColorMatch
(
    unsigned long width,
    const uint32_t *map,
    unsigned char *rgb,
    unsigned char *kplane,
    unsigned char *cplane,
    unsigned char *mplane,
    unsigned char *yplane
)
{
    HPBool  first = HPTRUE;

    for (unsigned long i = 0; i < width; i++)
    {
        uint32_t r = *rgb++;
        uint32_t g = *rgb++;
        uint32_t b = *rgb++;

        const uint32_t *start;
        start = (const uint32_t *)
            (((r & 0xE0) << 1) + ((r & 0xE0) >> 1) + (r >> 5) +
            ((g & 0xE0) >> 2) + (g >> 5) + (b >> 5) + map);

        Interpolate(start, i, (BYTE)r, (BYTE)g,(BYTE)b, kplane, cplane, mplane, yplane, first);
        first = HPFALSE;
    }

} //ColorMatch


uint32_t Packed(unsigned int k,unsigned int c,unsigned int m,unsigned int y)
{
    uint32_t p = y;
    p = p << 8;
    p += m;
    p = p << 8;
    p += c;
    p = p << 8;
    p += k;
    return p;
} //Packed


DRIVER_ERROR ColorMatcher::MakeGrayMap(const uint32_t *colormap, uint32_t* graymap)
{
    for (unsigned int r = 0; r < 9; r++)
    {
        unsigned long ul_RedMapPtr = r * 9 * 9;
        for (unsigned int g = 0; g < 9; g++)
        {
            unsigned long ul_GreenMapPtr = g * 9;
            for (unsigned int b = 0; b < 9; b++)
            {
                unsigned long mapptr = b + (g * 9) + (r * 9 * 9);       // get address in map
                unsigned long ul_MapPtr = b + ul_GreenMapPtr + ul_RedMapPtr;
                ASSERT(mapptr == ul_MapPtr);
                // put r,g,b in monitor range
                unsigned int oldR = r * 255 >> 3;
                unsigned int oldG = g * 255 >> 3;
                unsigned int oldB = b * 255 >> 3;

                // calculate gray equivalence
                unsigned int gray = ((30 * oldR + 59 * oldG + 11 * oldB + 50) / 100);

                uint32_t *start;
                start = (uint32_t *)
                        ( ((gray & 0xE0) <<1) + ((gray & 0xE0)>>1) + (gray>>5) +
                          ((gray & 0xE0) >>2) + (gray>>5) + (gray>>5) + colormap);

                 BYTE k,c,m,y;
                 Interpolate(start, 0, gray, gray, gray, &k, &c, &m, &y, TRUE);

                // second interpolate if Clight/Mlight

                *(graymap + mapptr) = Packed(k, c, m, y);
            }
        }
    }
    return NO_ERROR;
} //MakeGrayMap


BOOL ColorMatcher::Process(BYTE* pbyInputRGBRaster, unsigned int size)
{
    if (pbyInputRGBRaster == NULL)
    {
        Restart();
        return FALSE;   // no output
    }
    started=TRUE;

    BYTE* buff1 = NULL;
    BYTE* buff2 = NULL;
    BYTE* buff3 = NULL;
    BYTE* buff4 = NULL;

    if (StartPlane==K)
    {
        buff1 = Contone;
        if (EndPlane>K)
        {
            buff2 = buff1 + InputWidth;
            buff3 = buff2 + InputWidth;
            buff4 = buff3 + InputWidth;
        }
    }
    else
    {
        buff2 = Contone;
        buff3 = buff2 + InputWidth;
        buff4 = buff3 + InputWidth;
    }

    // colormatching -- can only handle 4 planes at a time
    ColorMatch( InputWidth, // ASSUMES ALL INPUTWIDTHS EQUAL
        cmap.ulMap1,
        pbyInputRGBRaster,
        buff1,
        buff2,
        buff3,
        buff4
    );

    if (EndPlane > Y)
    {
        BYTE* buff5 = buff4 + InputWidth;
        BYTE* buff6 = buff5 + InputWidth;

        ColorMatch( InputWidth,
            cmap.ulMap2,    // 2nd map is for lighter inks
            pbyInputRGBRaster,
            NULL,           // don't need black again
            buff5,buff6,
            NULL            // don't need yellow again
        );
    }

    iRastersReady = 1;
    iRastersDelivered = 0;
    return TRUE;   // one raster in, one raster out
}

APDK_END_NAMESPACE

