/*
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%                                                                             %
%                                                                             %
%                    M   M   AAA    GGGG  IIIII   CCCC                        %
%                    MM MM  A   A  G        I    C                            %
%                    M M M  AAAAA  G GGG    I    C                            %
%                    M   M  A   A  G   G    I    C                            %
%                    M   M  A   A   GGGG  IIIII   CCCC                        %
%                                                                             %
%                                                                             %
%                     ImageMagick Image Magic Methods                         %
%                                                                             %
%                                                                             %
%                              Software Design                                %
%                              Bob Friesenhahn                                %
%                                 July 2000                                   %
%                                                                             %
%                                                                             %
%  Copyright (C) 2003 ImageMagick Studio, a non-profit organization dedicated %
%  to making software imaging solutions freely available.                     %
%                                                                             %
%  Permission is hereby granted, free of charge, to any person obtaining a    %
%  copy of this software and associated documentation files ("ImageMagick"),  %
%  to deal in ImageMagick without restriction, including without limitation   %
%  the rights to use, copy, modify, merge, publish, distribute, sublicense,   %
%  and/or sell copies of ImageMagick, and to permit persons to whom the       %
%  ImageMagick is furnished to do so, subject to the following conditions:    %
%                                                                             %
%  The above copyright notice and this permission notice shall be included in %
%  all copies or substantial portions of ImageMagick.                         %
%                                                                             %
%  The software is provided "as is", without warranty of any kind, express or %
%  implied, including but not limited to the warranties of merchantability,   %
%  fitness for a particular purpose and noninfringement.  In no event shall   %
%  ImageMagick Studio be liable for any claim, damages or other liability,    %
%  whether in an action of contract, tort or otherwise, arising from, out of  %
%  or in connection with ImageMagick or the use or other dealings in          %
%  ImageMagick.                                                               %
%                                                                             %
%  Except as contained in this notice, the name of the ImageMagick Studio     %
%  shall not be used in advertising or otherwise to promote the sale, use or  %
%  other dealings in ImageMagick without prior written authorization from the %
%  ImageMagick Studio.                                                        %
%                                                                             %
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%
%
*/

/*
  Include declarations.
*/
#include "magick/studio.h"
#include "magick/blob.h"
#include "magick/error.h"
#include "magick/magic.h"
#include "magick/semaphore.h"
#include "magick/utility.h"

/*
  Define declarations.
*/
#define MagicFilename  "magic.mgk"

/*
  Declare color map.
*/
static char
  *MagicMap = (char *)
    "<?xml version=\"1.0\"?>"
    "<magicmap>"
    "  <magic name=\"AVI\" offset=\"0\" target=\"RIFF\" />"
    "  <magic name=\"8BIM\" offset=\"0\" target=\"8BIM\" />"
    "  <magic name=\"8BIMWTEXT\" offset=\"0\" target=\"8\000B\000I\000M\000#\" />"
    "  <magic name=\"8BIMTEXT\" offset=\"0\" target=\"8BIM#\" />"
    "  <magic name=\"BMP\" offset=\"0\" target=\"BA\" />"
    "  <magic name=\"BMP\" offset=\"0\" target=\"BM\" />"
    "  <magic name=\"BMP\" offset=\"0\" target=\"CI\" />"
    "  <magic name=\"BMP\" offset=\"0\" target=\"CP\" />"
    "  <magic name=\"BMP\" offset=\"0\" target=\"IC\" />"
    "  <magic name=\"BMP\" offset=\"0\" target=\"PI\" />"
    "  <magic name=\"CGM\" offset=\"0\" target=\"BEGMF\" />"
    "  <magic name=\"DCM\" offset=\"128\" target=\"DICM\" />"
    "  <magic name=\"DCX\" offset=\"0\" target=\"\261\150\336\72\" />"
    "  <magic name=\"DIB\" offset=\"0\" target=\"\050\000\" />"
    "  <magic name=\"DOT\" offset=\"0\" target=\"digraph\" />"
    "  <magic name=\"EMF\" offset=\"40\" target=\"\040\105\115\106\000\000\001\000\" />"
    "  <magic name=\"EPT\" offset=\"0\" target=\"\305\320\323\306\" />"
    "  <magic name=\"FAX\" offset=\"0\" target=\"DFAX\" />"
    "  <magic name=\"FIG\" offset=\"0\" target=\"#FIG\" />"
    "  <magic name=\"FITS\" offset=\"0\" target=\"IT0\" />"
    "  <magic name=\"FITS\" offset=\"0\" target=\"SIMPLE\" />"
    "  <magic name=\"FPX\" offset=\"0\" target=\"\320\317\021\340\" />"
    "  <magic name=\"GIF\" offset=\"0\" target=\"GIF8\" />"
    "  <magic name=\"GPLT\" offset=\"0\" target=\"#!/usr/local/bin/gnuplot\" />"
    "  <magic name=\"HDF\" offset=\"1\" target=\"HDF\" />"
    "  <magic name=\"HPGL\" offset=\"0\" target=\"IN;\" />"
    "  <magic name=\"HPGL\" offset=\"0\" target=\"\033E\033\" />"
    "  <magic name=\"HTML\" offset=\"1\" target=\"HTML\" />"
    "  <magic name=\"HTML\" offset=\"1\" target=\"html\" />"
    "  <magic name=\"ILBM\" offset=\"8\" target=\"ILBM\" />"
		"  <magic name=\"IPTCWTEXT\" offset=\"0\" target=\"\062\000#\000\060\000=\000\042\000&\000#\000\060\000;\000&\000#\000\062\000;\000\042\000\" />"
    "  <magic name=\"IPTCTEXT\" offset=\"0\" target=\"2#0=\042&#0;&#2;\042\" />"
    "  <magic name=\"IPTC\" offset=\"0\" target=\"\034\002\" />"
    "  <magic name=\"JNG\" offset=\"0\" target=\"\213JNG\r\n\032\n\" />"
    "  <magic name=\"JPEG\" offset=\"0\" target=\"\377\330\377\" />"
    "  <magic name=\"JPC\" offset=\"0\" target=\"\377\117\" />"
    "  <magic name=\"JP2\" offset=\"4\" target=\"\152\120\040\040\015\" />"
    "  <magic name=\"MIFF\" offset=\"0\" target=\"Id=ImageMagick\" />"
    "  <magic name=\"MIFF\" offset=\"0\" target=\"id=ImageMagick\" />"
    "  <magic name=\"MNG\" offset=\"0\" target=\"\212MNG\r\n\032\n\" />"
    "  <magic name=\"MPC\" offset=\"0\" target=\"id=MagickCache\" />"
    "  <magic name=\"MPEG\" offset=\"0\" target=\"\000\000\001\263\" />"
    "  <magic name=\"MVG\" offset=\"0\" target=\"push graphic-context\" />"
    "  <magic name=\"PCD\" offset=\"2048\" target=\"PCD_\" />"
    "  <magic name=\"PCL\" offset=\"0\" target=\"\033E\033\" />"
    "  <magic name=\"PCX\" offset=\"0\" target=\"\012\002\" />"
    "  <magic name=\"PCX\" offset=\"0\" target=\"\012\005\" />"
    "  <magic name=\"PDB\" offset=\"60\" target=\"vIMGView\" />"
    "  <magic name=\"PDF\" offset=\"0\" target=\"%PDF-\" />"
    "  <magic name=\"PFA\" offset=\"0\" target=\"%!PS-AdobeFont-1.0\" />"
    "  <magic name=\"PFB\" offset=\"6\" target=\"%!PS-AdobeFont-1.0\" />"
    "  <magic name=\"PGX\" offset=\"0\" target=\"\050\107\020\115\046\" />"
    "  <magic name=\"PICT\" offset=\"522\" target=\"\000\021\002\377\014\000\" />"
    "  <magic name=\"PICT\" offset=\"522\" target=\"\021\001\" />"
    "  <magic name=\"PNG\" offset=\"0\" target=\"\211PNG\r\n\032\n\" />"
    "  <magic name=\"PNM\" offset=\"0\" target=\"P1\" />"
    "  <magic name=\"PNM\" offset=\"0\" target=\"P2\" />"
    "  <magic name=\"PNM\" offset=\"0\" target=\"P3\" />"
    "  <magic name=\"PNM\" offset=\"0\" target=\"P4\" />"
    "  <magic name=\"PNM\" offset=\"0\" target=\"P5\" />"
    "  <magic name=\"PNM\" offset=\"0\" target=\"P6\" />"
    "  <magic name=\"PNM\" offset=\"0\" target=\"P7\" />"
    "  <magic name=\"PS\" offset=\"0\" target=\"%!\" />"
    "  <magic name=\"PS\" offset=\"0\" target=\"\004%!\" />"
    "  <magic name=\"PS\" offset=\"0\" target=\"\305\320\323\306\" />"
    "  <magic name=\"PSD\" offset=\"0\" target=\"8BPS\" />"
    "  <magic name=\"PWP\" offset=\"0\" target=\"SFW95\" />"
    "  <magic name=\"RAD\" offset=\"0\" target=\"#?RADIANCE\" />"
    "  <magic name=\"RLE\" offset=\"0\" target=\"\122\314\" />"
    "  <magic name=\"SCT\" offset=\"0\" target=\"CT\" />"
    "  <magic name=\"SFW\" offset=\"0\" target=\"SFW94\" />"
    "  <magic name=\"SGI\" offset=\"0\" target=\"\001\332\" />"
    "  <magic name=\"SUN\" offset=\"0\" target=\"\131\246\152\225\" />"
    "  <magic name=\"SVG\" offset=\"1\" target=\"?XML\" />"
    "  <magic name=\"SVG\" offset=\"1\" target=\"?xml\" />"
    "  <magic name=\"TIFF\" offset=\"0\" target=\"\115\115\000\052\" />"
    "  <magic name=\"TIFF\" offset=\"0\" target=\"\111\111\052\000\" />"
    "  <magic name=\"VICAR\" offset=\"0\" target=\"LBLSIZE\" />"
    "  <magic name=\"VICAR\" offset=\"0\" target=\"NJPL1I\" />"
    "  <magic name=\"VIFF\" offset=\"0\" target=\"\253\001\" />"
    "  <magic name=\"WMF\" offset=\"0\" target=\"\327\315\306\232\" />"
    "  <magic name=\"WMF\" offset=\"0\" target=\"\001\000\011\000\" />"
    "  <magic name=\"WPG\" offset=\"0\" target=\"\377WPC\" />"
    "  <magic name=\"XBM\" offset=\"0\" target=\"#define\" />"
    "  <magic name=\"XCF\" offset=\"0\" target=\"gimp xcf\" />"
    "  <magic name=\"XPM\" offset=\"1\" target=\"* XPM *\" />"
    "  <magic name=\"XWD\" offset=\"4\" target=\"\007\000\000\" />"
    "  <magic name=\"XWD\" offset=\"5\" target=\"\000\000\007\" />"
    "</magicmap>";

/*
  Static declarations.
*/
static SemaphoreInfo
  *magic_semaphore = (SemaphoreInfo *) NULL;

static MagicInfo
  *magic_list = (MagicInfo *) NULL;

/*
  Forward declarations.
*/
static unsigned int
  ReadConfigureFile(const char *,const unsigned long,ExceptionInfo *);

/*
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%                                                                             %
%                                                                             %
%                                                                             %
+   D e s t r o y M a g i c I n f o                                           %
%                                                                             %
%                                                                             %
%                                                                             %
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%
%  Method DestroyMagicInfo deallocates memory associated with the magic list.
%
%  The format of the DestroyMagicInfo method is:
%
%      DestroyMagicInfo(void)
%
%
*/
MagickExport void DestroyMagicInfo(void)
{
  MagicInfo
    *magic_info;

  register MagicInfo
    *p;

  AcquireSemaphoreInfo(&magic_semaphore);
  for (p=magic_list; p != (MagicInfo *) NULL; )
  {
    magic_info=p;
    p=p->next;
    if (magic_info->path != (char *) NULL)
      LiberateMemory((void **) &magic_info->path);
    if (magic_info->name != (char *) NULL)
      LiberateMemory((void **) &magic_info->name);
    if (magic_info->target != (char *) NULL)
      LiberateMemory((void **) &magic_info->target);
    if (magic_info->magic != (unsigned char *) NULL)
      LiberateMemory((void **) &magic_info->magic);
    LiberateMemory((void **) &magic_info);
  }
  magic_list=(MagicInfo *) NULL;
  DestroySemaphoreInfo(&magic_semaphore);
}

/*
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%                                                                             %
%                                                                             %
%                                                                             %
+   G e t M a g i c I n f o                                                   %
%                                                                             %
%                                                                             %
%                                                                             %
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%
%  Method GetMagicInfo searches the magic list for the specified name and if
%  found returns attributes for that magic.
%
%  The format of the GetMagicInfo method is:
%
%      const MagicInfo *GetMagicInfo(const unsigned char *magic,
%        const size_t length,ExceptionInfo *exception)
%
%  A description of each parameter follows:
%
%    o magic_info: Method GetMagicInfo searches the magic list for any image
%      format tag that matches the specified image signature and if found
%      returns attributes for that image format.
%
%    o magic: A binary string generally representing the first few characters
%      of the image file or blob.
%
%    o length: The length of the binary signature.
%
%    o exception: Return any errors or warnings in this structure.
%
%
*/
MagickExport const MagicInfo *GetMagicInfo(const unsigned char *magic,
  const size_t length,ExceptionInfo *exception)
{
  register MagicInfo
    *p;

  if (magic_list == (MagicInfo *) NULL)
    {
      AcquireSemaphoreInfo(&magic_semaphore);
      if (magic_list == (MagicInfo *) NULL)
        (void) ReadConfigureFile(MagicFilename,0,exception);
      LiberateSemaphoreInfo(&magic_semaphore);
    }
  if ((magic == (const unsigned char *) NULL) || (length == 0))
    return((const MagicInfo *) magic_list);
  /*
    Search for requested magic.
  */
  AcquireSemaphoreInfo(&magic_semaphore);
  for (p=magic_list; p != (MagicInfo *) NULL; p=p->next)
    if (p->offset+p->length <= length)
      if (memcmp(magic+p->offset,p->magic,p->length) == 0)
        break;
  if (p != (MagicInfo *) NULL)
    if (p != magic_list)
      {
        /*
          Self-adjusting list.
        */
        if (p->previous != (MagicInfo *) NULL)
          p->previous->next=p->next;
        if (p->next != (MagicInfo *) NULL)
          p->next->previous=p->previous;
        p->previous=(MagicInfo *) NULL;
        p->next=magic_list;
        magic_list->previous=p;
        magic_list=p;
      }
  LiberateSemaphoreInfo(&magic_semaphore);
  return((const MagicInfo *) p);
}

/*
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%                                                                             %
%                                                                             %
%                                                                             %
%  L i s t M a g i c I n f o                                                  %
%                                                                             %
%                                                                             %
%                                                                             %
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%
%  Method ListMagicInfo lists the magic info to a file.
%
%  The format of the ListMagicInfo method is:
%
%      unsigned int ListMagicInfo(FILE *file,ExceptionInfo *exception)
%
%  A description of each parameter follows.
%
%    o file:  An pointer to a FILE.
%
%    o exception: Return any errors or warnings in this structure.
%
%
*/
MagickExport unsigned int ListMagicInfo(FILE *file,ExceptionInfo *exception)
{
  register const MagicInfo
    *p;

  register long
    i;

  if (file == (const FILE *) NULL)
    file=stdout;
  (void) GetMagicInfo((unsigned char *) NULL,0,exception);
  AcquireSemaphoreInfo(&magic_semaphore);
  for (p=magic_list; p != (MagicInfo *) NULL; p=p->next)
  {
    if ((p->previous == (MagicInfo *) NULL) ||
        (LocaleCompare(p->path,p->previous->path) != 0))
      {
        if (p->previous != (MagicInfo *) NULL)
          (void) fprintf(file,"\n");
        if (p->path != (char *) NULL)
          (void) fprintf(file,"Path: %.1024s\n\n",p->path);
        (void) fprintf(file,"Name      Offset Target\n");
        (void) fprintf(file,"-------------------------------------------------"
          "------------------------------\n");
      }
    if (p->stealth)
      continue;
    (void) fprintf(file,"%.1024s",p->name);
    for (i=(long) strlen(p->name); i <= 9; i++)
      (void) fprintf(file," ");
    (void) fprintf(file,"%6ld ",p->offset);
    if (p->target != (char *) NULL)
      (void) fprintf(file,"%.1024s",p->target);
    (void) fprintf(file,"\n");
  }
  (void) fflush(file);
  LiberateSemaphoreInfo(&magic_semaphore);
  return(True);
}

/*
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%                                                                             %
%                                                                             %
%                                                                             %
+   R e a d C o n f i g u r e F i l e                                 %
%                                                                             %
%                                                                             %
%                                                                             %
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%
%  Method ReadConfigureFile reads the color configuration file which maps
%  color strings with a particular image format.
%
%  The format of the ReadConfigureFile method is:
%
%      unsigned int ReadConfigureFile(const char *basename,
%        const unsigned long depth,ExceptionInfo *exception)
%
%  A description of each parameter follows:
%
%    o status: Method ReadConfigureFile returns True if at least one color
%      is defined otherwise False.
%
%    o basename:  The color configuration filename.
%
%    o depth: depth of <include /> statements.
%
%    o exception: Return any errors or warnings in this structure.
%
%
*/
static unsigned int ReadConfigureFile(const char *basename,
  const unsigned long depth,ExceptionInfo *exception)
{
  char
    keyword[MaxTextExtent],
    path[MaxTextExtent],
    *q,
    *token,
    *xml;

  size_t
    length;

  /*
    Read the magic configure file.
  */
  (void) strcpy(path,basename);
  if (depth == 0)
    xml=(char *) GetConfigureBlob(basename,path,&length,exception);
  else
    xml=(char *) FileToBlob(basename,&length,exception);
  if (xml == (char *) NULL)
    xml=AllocateString(MagicMap);
  token=AllocateString(xml);
  for (q=xml; *q != '\0'; )
  {
    /*
      Interpret XML.
    */
    GetToken(q,&q,token);
    if (*token == '\0')
      break;
    (void) strncpy(keyword,token,MaxTextExtent-1);
    if (LocaleNCompare(keyword,"<!--",4) == 0)
      {
        /*
          Comment element.
        */
        while ((LocaleNCompare(q,"->",2) != 0) && (*q != '\0'))
          GetToken(q,&q,token);
        continue;
      }
    if (LocaleCompare(keyword,"<include") == 0)
      {
        /*
          Include element.
        */
        while ((*token != '>') && (*q != '\0'))
        {
          (void) strncpy(keyword,token,MaxTextExtent-1);
          GetToken(q,&q,token);
          if (*token != '=')
            continue;
          GetToken(q,&q,token);
          if (LocaleCompare(keyword,"file") == 0)
            {
              if (depth > 200)
                (void) ThrowException(exception,ConfigureError,
                  "IncludeElementNestedTooDeeply",path);
              else
                {
                  char
                    filename[MaxTextExtent];

                  GetPathComponent(path,HeadPath,filename);
                  if (*filename != '\0')
                    (void) strcat(filename,DirectorySeparator);
                  (void) strncat(filename,token,MaxTextExtent-
                    strlen(filename)-1);
                  (void) ReadConfigureFile(filename,depth+1,exception);
                }
              if (magic_list != (MagicInfo *) NULL)
                while (magic_list->next != (MagicInfo *) NULL)
                  magic_list=magic_list->next;
            }
        }
        continue;
      }
    if (LocaleCompare(keyword,"<magic") == 0)
      {
        MagicInfo
          *magic_info;

        /*
          Allocate memory for the magic list.
        */
        magic_info=(MagicInfo *) AcquireMemory(sizeof(MagicInfo));
        if (magic_info == (MagicInfo *) NULL)
          MagickFatalError(ResourceLimitFatalError,"MemoryAllocationFailed",
            "UnableToAllocateMagicInfo");
        (void) memset(magic_info,0,sizeof(MagicInfo));
        magic_info->path=AcquireString(path);
        magic_info->signature=MagickSignature;
        if (magic_list == (MagicInfo *) NULL)
          {
            magic_list=magic_info;
            continue;
          }
        magic_list->next=magic_info;
        magic_info->previous=magic_list;
        magic_list=magic_list->next;
        continue;
      }
    if (magic_list == (MagicInfo *) NULL)
      continue;
    GetToken(q,(char **) NULL,token);
    if (*token != '=')
      continue;
    GetToken(q,&q,token);
    GetToken(q,&q,token);
    switch (*keyword)
    {
      case 'N':
      case 'n':
      {
        if (LocaleCompare((char *) keyword,"name") == 0)
          {
            magic_list->name=AcquireString(token);
            break;
          }
        break;
      }
      case 'O':
      case 'o':
      {
        if (LocaleCompare((char *) keyword,"offset") == 0)
          {
            magic_list->offset=atol(token);
            break;
          }
        break;
      }
      case 'S':
      case 's':
      {
        if (LocaleCompare((char *) keyword,"stealth") == 0)
          {
            magic_list->stealth=LocaleCompare(token,"True") == 0;
            break;
          }
        break;
      }
      case 'T':
      case 't':
      {
        if (LocaleCompare((char *) keyword,"target") == 0)
          {
            const char
              *p;

            register unsigned char
              *q;

            magic_list->target=AcquireString(token);
            magic_list->magic=(unsigned char *) AllocateString(token);
            q=magic_list->magic;
            for (p=magic_list->target; *p != '\0'; )
            {
              if (*p == '\\')
                {
                  p++;
                  if (isdigit((int) *p))
                    {
                      char
                        *end;

                      *q++=(unsigned char) strtol(p,&end,8);
                      p+=(end-p);
                      magic_list->length++;
                      continue;
                    }
                  switch (*p)
                  {
                    case 'b': *q='\b'; break;
                    case 'f': *q='\f'; break;
                    case 'n': *q='\n'; break;
                    case 'r': *q='\r'; break;
                    case 't': *q='\t'; break;
                    case 'v': *q='\v'; break;
                    case 'a': *q='a'; break;
                    case '?': *q='\?'; break;
                    default: *q=(*p); break;
                  }
                  p++;
                  q++;
                  magic_list->length++;
                  continue;
                }
              *q++=(*p++);
              magic_list->length++;
            }
            break;
          }
        break;
      }
      default:
        break;
    }
  }
  LiberateMemory((void **) &token);
  LiberateMemory((void **) &xml);
  if (magic_list == (MagicInfo *) NULL)
    return(False);
  while (magic_list->previous != (MagicInfo *) NULL)
    magic_list=magic_list->previous;
  return(True);
}
