/***********************************
 Update contact list in main window
 (c) 1999 Jeremy Wise
 (c) 2001 Gediminas Paulauskas
 GnomeICU
************************************/

#include "common.h"
#include "gtkfunc.h"
#include "dragdrop.h"
#include "events.h"
#include "gnomecfg.h"
#include "gnomeicu.h"
#include "icons.h"
#include "response.h"
#include "showlist.h"
#include "groups.h"
#include "util.h"

#include <gtk/gtknotebook.h>
#include <string.h>

static gint gnomeicu_tree_total_contacts_in_group (const gchar *group_name);
static gint gnomeicu_tree_online_contacts_in_group (const gchar *group_name);
static gboolean gnomeicu_tree_contact_exists (Contact_Member *contact);
static void gnomeicu_tree_user_add_online (Contact_Member *contact);
static void gnomeicu_tree_user_remove_online (Contact_Member *contact);

void drag_data_get_cb(GtkWidget *widget,
                      GdkDragContext *drag_context,
                      GtkSelectionData *data,
                      guint info,
                      guint time,
                      gpointer user_data)
{
	GtkTreeSelection *sel;
	GtkTreeIter iter;
	GtkTreeModel *model;
	gchar *uid;
	gint gid;
	Contact_Member *contact;
	gboolean cont;
	char *str;

#ifdef TRACE_FUNCTION
	g_print("drag_data_get_cb\n");
#endif

	sel = gtk_tree_view_get_selection (GTK_TREE_VIEW(widget));
	cont = gtk_tree_selection_get_selected (sel, &model, &iter);

	if (cont) {
		gtk_tree_model_get (model, &iter,
		                    GID_COLUMN, &gid,
		                    UID_COLUMN, &uid,
		                    CONTACT_COLUMN, &contact, -1);

		if (uid) {
			str = g_strdup_printf( "%s\n%s", contact->uin,
							   contact->nick );
			gtk_selection_data_set( data,
								data->target,
								8, str, strlen( str ) );
			g_free( str );
		}
		g_free(uid);
	}
}

void gnomeicu_tree_create (void)
{
	GSList *contact, *list;
	GtkWidget *maintreeview, *alltreeview;
	
#ifdef TRACE_FUNCTION
	g_print ("gnomeicu_tree_create\n");
#endif

	for (list = Groups; list != NULL; list = list->next)
	  gnomeicu_tree_add_group(list->data);

	for(contact = Contacts; contact != NULL; contact = contact->next)
	  gnomeicu_tree_user_add(kontakt);

	gnomeicu_tree_update_headers();
	
	
	maintreeview = glade_xml_get_widget(MainData->xml, "maintreeview");
	alltreeview = glade_xml_get_widget(MainData->xml, "alltreeview");
	
	gtk_drag_source_set(GTK_WIDGET(maintreeview), GDK_BUTTON1_MASK,
                      target_table, 1, GDK_ACTION_COPY | GDK_ACTION_MOVE);
	gtk_drag_source_set(GTK_WIDGET(alltreeview), GDK_BUTTON1_MASK,
                      target_table, 1, GDK_ACTION_COPY | GDK_ACTION_MOVE);
	
	
	g_signal_connect (G_OBJECT (maintreeview), "drag_data_get",
	                  G_CALLBACK (drag_data_get_cb), NULL);

	g_signal_connect (G_OBJECT (alltreeview), "drag_data_get",
	                  G_CALLBACK (drag_data_get_cb), NULL);

}

void gnomeicu_tree_user_online (Contact_Member *contact)
{

#ifdef TRACE_FUNCTION
	g_print ("gnomeicu_tree_user_online\n");
#endif

	if (!g_utf8_validate (contact->nick, -1, NULL)) {
		g_free(contact->nick);
		contact->nick = g_strdup(contact->uin);
	}


	gnomeicu_tree_user_add_online(contact);

	gnomeicu_tree_update_headers();
}

void gnomeicu_tree_user_offline (Contact_Member *contact)
{

#ifdef TRACE_FUNCTION
	g_print ("gnomeicu_tree_user_online\n");
#endif

	if (contact->gid == 0 && contact->inlist)
		return;

	gnomeicu_tree_user_remove_online (contact);

	gnomeicu_tree_update_headers();
}

void gnomeicu_tree_user_remove (Contact_Member *contact)
{
  gchar *uid;
  gint gid;
  GtkTreeIter iter, child;
  gboolean valid, found;

#ifdef TRACE_FUNCTION
  g_print ("gnomeicu_tree_user_remove\n");
#endif


  if (contact->gid && contact->inlist) {
    found = FALSE;
    valid = gtk_tree_model_get_iter_first (GTK_TREE_MODEL(MainData->contacts_store),
					   &iter);
    while (valid) {
      gtk_tree_model_get (GTK_TREE_MODEL(MainData->contacts_store), &iter,
			  GID_COLUMN, &gid, -1);
      if ( (gid == contact->gid) || (contact->inlist == FALSE && !gid) ) { 
	/* This is the contact's group */
	valid = gtk_tree_model_iter_children (GTK_TREE_MODEL(MainData->contacts_store),
					      &child, &iter);
	while(valid) {
	  gtk_tree_model_get (GTK_TREE_MODEL (MainData->contacts_store), &child,
			      UID_COLUMN, &uid, -1);
	  if (uid && !strcmp (uid, contact->uin)) {
	    gtk_tree_store_remove (GTK_TREE_STORE (MainData->contacts_store), &child);
	    g_free(uid);
	    found = TRUE;
	    break;
	  }
	  g_free(uid);
	  valid = gtk_tree_model_iter_next (GTK_TREE_MODEL(MainData->contacts_store),
					    &child);
	}
      }
      if (found)
	break;
      valid = gtk_tree_model_iter_next (GTK_TREE_MODEL(MainData->contacts_store),
					&iter);
    }
  }
  gnomeicu_tree_user_remove_online(contact);  

  gnomeicu_tree_update_headers();
}

void gnomeicu_tree_user_remove_online (Contact_Member *contact)
{

  gchar *uid;
  gint gid;
  GtkTreeIter iter, child;
  gboolean valid, child_valid;

#ifdef TRACE_FUNCTION
  g_print ("gnomeicu_tree_user_remove_online\n");
#endif

  valid = gtk_tree_model_get_iter_first (GTK_TREE_MODEL(MainData->visible_store),
					 &iter);
  while (valid) {
    gtk_tree_model_get (GTK_TREE_MODEL(MainData->visible_store), &iter,
			GID_COLUMN, &gid, -1);
    child_valid = gtk_tree_model_iter_children(GTK_TREE_MODEL(MainData->visible_store), 
					     &child, &iter);
    while(child_valid) {
      gtk_tree_model_get (GTK_TREE_MODEL (MainData->visible_store), &child,
			  UID_COLUMN, &uid, -1);
      if (uid && !strcmp (uid, contact->uin)) {
	gtk_tree_store_remove (GTK_TREE_STORE (MainData->visible_store), &child);
	if (!gtk_tree_model_iter_has_child (GTK_TREE_MODEL(MainData->visible_store),
					    &iter))
	  gtk_tree_store_remove (GTK_TREE_STORE (MainData->visible_store), &iter);
	g_free(uid);
	return;
      }
      g_free(uid);
      child_valid = gtk_tree_model_iter_next (GTK_TREE_MODEL(MainData->visible_store), &child);
    }
    valid = gtk_tree_model_iter_next (GTK_TREE_MODEL(MainData->visible_store), &iter);
  }

}


void gnomeicu_tree_user_add (Contact_Member *contact)
{
  gchar *str, *escaped;
  gchar *uid;
  gint gid;
  GtkTreeIter iter, child;
  gboolean valid;

#ifdef TRACE_FUNCTION
  g_print ("gnomeicu_tree_user_add\n");
#endif
  
  if (!contact->nick || !g_utf8_validate (contact->nick, -1, NULL)) {
    g_free(contact->nick);
    contact->nick = g_strdup(contact->uin);
  }

  if (contact->gid > 0) {
    if (gnomeicu_tree_contact_exists (contact))
      return;
    valid = gtk_tree_model_get_iter_first (GTK_TREE_MODEL(MainData->contacts_store),
					   &iter);
    while (valid) {
      gtk_tree_model_get (GTK_TREE_MODEL(MainData->contacts_store), &iter,
			  GID_COLUMN, &gid, UID_COLUMN, &uid, -1);
      if (gid == contact->gid && uid == NULL) {
	escaped = g_markup_escape_text(contact->nick, strlen(contact->nick)+1);
	str = g_strdup_printf("<span foreground=\"%s\">%s</span>",
			      get_foreground_for_status(contact->status),
			      escaped);
	g_free(escaped);
	gtk_tree_store_prepend (GTK_TREE_STORE(MainData->contacts_store),
				&child, &iter);
	gtk_tree_store_set (MainData->contacts_store, &child,
			    ICON_COLUMN, get_pixbuf_for_status(contact->status),
			    NICKNAME_COLUMN, str,
			    GID_COLUMN, contact->gid,
			    UID_COLUMN, g_strdup (contact->uin),
			    STATUS_COLUMN, contact->status,
			    CONTACT_COLUMN, contact,
			    -1);
	g_free(str);
	g_free(uid);
	break;
      }
      g_free(uid);
      valid = gtk_tree_model_iter_next (GTK_TREE_MODEL(MainData->contacts_store),
					&iter);
    }
  }

  gnomeicu_tree_user_add_online(contact);

  gnomeicu_tree_update_headers();
}

void gnomeicu_tree_user_add_online (Contact_Member *contact)
{
  gchar *str, *escaped;
  gchar *uid;
  gint gid;
  GtkTreeIter iter, child;
  gboolean valid;
  GtkTreePath *path;

#ifdef TRACE_FUNCTION
  g_print ("gnomeicu_tree_user_add_online\n");
#endif

  if (contact->inlist && (contact->gid == 0 || contact->status == STATUS_OFFLINE))
    return;
  
  valid = gtk_tree_model_get_iter_first (GTK_TREE_MODEL(MainData->visible_store),
					 &iter);
  while (valid) {
    gtk_tree_model_get (GTK_TREE_MODEL(MainData->visible_store), &iter,
			GID_COLUMN, &gid, UID_COLUMN, &uid, -1);
    if ((gid && (groups_name_by_gid(contact->gid) && groups_name_by_gid(gid) &&
		!strcmp(groups_name_by_gid(contact->gid), groups_name_by_gid(gid)) &&
		uid == NULL)) || (!gid && !contact->inlist)) {
      escaped = g_markup_escape_text(contact->nick, strlen(contact->nick)+1);
      str = g_strdup_printf("<span foreground=\"%s\">%s</span>",
			    get_foreground_for_status(contact->status),
			    escaped);
      g_free(escaped);
      gtk_tree_store_prepend (GTK_TREE_STORE(MainData->visible_store), &child, &iter);
      gtk_tree_store_set (MainData->visible_store, &child,
			  ICON_COLUMN, get_pixbuf_for_status(contact->status),
			  NICKNAME_COLUMN, str,
			  GID_COLUMN, contact->gid,
			  UID_COLUMN, g_strdup (contact->uin),
			  STATUS_COLUMN, contact->status,
			  CONTACT_COLUMN, contact,
			  -1);
      g_free(str);
      g_free(uid);
      break;
    }
    
    g_free(uid);
    valid = gtk_tree_model_iter_next (GTK_TREE_MODEL(MainData->visible_store), &iter);
  }


  if (valid == FALSE) {
    if (contact->inlist) {
      if (groups_name_by_gid(gid))
	escaped = g_markup_escape_text(groups_name_by_gid(gid), strlen(groups_name_by_gid(gid))+1);
      else
	escaped = g_strdup(_("ERROR"));
    
      str = g_strdup_printf ("<b>%s (%d/%d)</b>",
			     escaped,
			     gnomeicu_tree_online_contacts_in_group (groups_name_by_gid(gid)),
			     gnomeicu_tree_total_contacts_in_group (groups_name_by_gid(gid)));
      g_free(escaped);
    }
    else 
      str = g_strdup_printf("<b>%s</b>", _("Not in list"));
    
    gtk_tree_store_prepend (MainData->visible_store, &iter, NULL);
    gtk_tree_store_set (MainData->visible_store, &iter,
			ICON_COLUMN, icon_group_pixbuf,
			NICKNAME_COLUMN, str,
			GID_COLUMN, contact->gid,
			CONTACT_COLUMN, NULL,
			UID_COLUMN, NULL, -1);
    g_free(str);
    
    escaped = g_markup_escape_text(contact->nick, strlen(contact->nick)+1);
    str = g_strdup_printf("<span foreground=\"%s\">%s</span>",
			  get_foreground_for_status(contact->status),
			  escaped);
    g_free(escaped);
    
    gtk_tree_store_append (GTK_TREE_STORE(MainData->visible_store), &child, &iter);
    gtk_tree_store_set (MainData->visible_store, &child,
			ICON_COLUMN, get_pixbuf_for_status(contact->status),
			NICKNAME_COLUMN, str,
			GID_COLUMN, contact->gid,
			UID_COLUMN, g_strdup (contact->uin),
			STATUS_COLUMN, contact->status,
			CONTACT_COLUMN, contact,
			-1);
    
    g_free(str);
    
    path = gtk_tree_model_get_path (GTK_TREE_MODEL (MainData->visible_store), &iter);
    gtk_tree_view_expand_row (GTK_TREE_VIEW (MainData->contacts_tree), path, FALSE);
    gtk_tree_path_free (path);
  }
}

void gnomeicu_tree_set_status (Contact_Member *contact)
{
  gchar *uid;
  gint gid;
  GtkTreeIter iter, child;
  gboolean valid;
  gchar *str, *escaped;

#ifdef TRACE_FUNCTION
  g_print ("gnomeicu_tree_set_status\n");
#endif

  if (!g_utf8_validate (contact->nick, -1, NULL)) {
    g_free(contact->nick);
    contact->nick = g_strdup(contact->uin);
  }

  if (contact->gid == 0 && contact->inlist) 
    return;
	
  valid = gtk_tree_model_get_iter_first (GTK_TREE_MODEL(MainData->visible_store),
					 &iter);
  while (valid) {
    gtk_tree_model_get (GTK_TREE_MODEL(MainData->visible_store), &iter,
			GID_COLUMN, &gid, -1);

    if ( (!gid && !contact->inlist) ||
	 ( gid && contact->status != STATUS_OFFLINE && 
	   groups_name_by_gid (contact->gid) && groups_name_by_gid (gid) && 
	   !strcmp(groups_name_by_gid (gid), groups_name_by_gid (contact->gid)))) {
      /* This is the contact's group */
      valid = gtk_tree_model_iter_children (GTK_TREE_MODEL(MainData->visible_store),
					    &child, &iter);
      while(valid) {
	gtk_tree_model_get (GTK_TREE_MODEL (MainData->visible_store), &child,
			    UID_COLUMN, &uid, -1);
	if (uid && !strcmp (uid, contact->uin)) {
	  escaped = g_markup_escape_text(contact->nick, strlen(contact->nick)+1);

	  str = g_strdup_printf("<span foreground=\"%s\">%s</span>",
				get_foreground_for_status(contact->status),
				escaped);
	  g_free(escaped);
	  gtk_tree_store_set (MainData->visible_store, &child,
			      ICON_COLUMN, get_pixbuf_for_status(contact->status),
			      NICKNAME_COLUMN, str,
			      -1);
	  g_free(str);
	  g_free(uid);
	  break;
	}
	g_free(uid);
	valid = gtk_tree_model_iter_next (GTK_TREE_MODEL(MainData->visible_store),
					  &child);
      }
      break;
    }
    valid = gtk_tree_model_iter_next (GTK_TREE_MODEL(MainData->visible_store), &iter);
  }

  valid = gtk_tree_model_get_iter_first (GTK_TREE_MODEL(MainData->contacts_store),
					 &iter);
  while (valid) {
    gtk_tree_model_get (GTK_TREE_MODEL(MainData->contacts_store), &iter,
			GID_COLUMN, &gid, -1);
    if (gid && gid == contact->gid) { 
      /* This is the contact's group */
      valid = gtk_tree_model_iter_children (GTK_TREE_MODEL(MainData->contacts_store), 
					    &child, &iter);
      while(valid) {
	gtk_tree_model_get (GTK_TREE_MODEL (MainData->contacts_store), &child,
			    UID_COLUMN, &uid, -1);
	if (uid && !strcmp (uid, contact->uin)) {
	  escaped = g_markup_escape_text(contact->nick, strlen(contact->nick)+1);

	  str = g_strdup_printf("<span foreground=\"%s\">%s</span>",
				get_foreground_for_status(contact->status),
				escaped);
	  g_free(escaped);
	  gtk_tree_store_set (MainData->contacts_store, &child,
			      NICKNAME_COLUMN, str,
			      ICON_COLUMN,
			      get_pixbuf_for_status(contact->status),
			      -1);
	  g_free(str);
	  g_free(uid);
	  break;
	}
	g_free(uid);
	valid = gtk_tree_model_iter_next (GTK_TREE_MODEL(MainData->contacts_store),
					  &child);
      }
      break;
    }
    valid = gtk_tree_model_iter_next (GTK_TREE_MODEL(MainData->contacts_store),
				      &iter);
  }
}

void set_contact_icon (Contact_Member *contact, GdkPixbuf *pixbuf)
{
  gchar *uid;
  gint gid;
  GtkTreeIter iter, child;
  gboolean valid;

#ifdef TRACE_FUNCTION
  g_print ("set_contact_icon\n");
#endif

  if (!g_utf8_validate (contact->nick, -1, NULL)) {
    g_free(contact->nick);
    contact->nick = g_strdup(contact->uin);
  }

  if (contact->gid > 0 || !contact->inlist ) {
    valid = gtk_tree_model_get_iter_first (GTK_TREE_MODEL(MainData->visible_store),
					   &iter);
    while (valid) {
      gtk_tree_model_get (GTK_TREE_MODEL(MainData->visible_store), &iter,
			  GID_COLUMN, &gid, -1);
      if ( (!gid && !contact->inlist) ||
	   ( gid && contact->status != STATUS_OFFLINE && 
	     groups_name_by_gid (contact->gid) &&
	     groups_name_by_gid (gid) && 
	     !strcmp(groups_name_by_gid (gid), groups_name_by_gid (contact->gid)))) {

	/* This is the contact's group */
	valid = gtk_tree_model_iter_children (GTK_TREE_MODEL(MainData->visible_store),
					      &child, &iter);
	while(valid) {
	  gtk_tree_model_get (GTK_TREE_MODEL (MainData->visible_store), &child,
			      UID_COLUMN, &uid, -1);
	  if (uid && !strcmp (uid, contact->uin)) {
	    gtk_tree_store_set (MainData->visible_store, &child,
				ICON_COLUMN, pixbuf,
				-1);
	    g_free(uid);
	    break;
	  }
	  g_free(uid);
	  valid = gtk_tree_model_iter_next (GTK_TREE_MODEL(MainData->visible_store),
					    &child);
	}
	break;
      }
      valid = gtk_tree_model_iter_next (GTK_TREE_MODEL(MainData->visible_store),
					&iter);
    }
  }
  if (contact->gid > 0 ) {
    valid = gtk_tree_model_get_iter_first (GTK_TREE_MODEL(MainData->contacts_store),
					   &iter);
    while (valid) {
      gtk_tree_model_get (GTK_TREE_MODEL(MainData->contacts_store), &iter,
			  GID_COLUMN, &gid, -1);
      if (contact->gid == gid) {
	/* This is the contact's group */
	valid = gtk_tree_model_iter_children(GTK_TREE_MODEL(MainData->contacts_store),
					     &child, &iter);
	while(valid) {
	  gtk_tree_model_get (GTK_TREE_MODEL (MainData->contacts_store), &child,
			      UID_COLUMN, &uid, -1);
	  if (uid && !strcmp (uid, contact->uin)) {
	    gtk_tree_store_set (MainData->contacts_store, &child,
				ICON_COLUMN, pixbuf,
				-1);
	    g_free(uid);
	    break;
	  }
	  g_free(uid);
	  valid = gtk_tree_model_iter_next (GTK_TREE_MODEL(MainData->contacts_store),
					    &child);
	}
	break;
      }
      valid = gtk_tree_model_iter_next (GTK_TREE_MODEL(MainData->contacts_store),
					&iter);
    }
  }
}

void gnomeicu_tree_refresh_visuals (void)
{
  GSList *contact;
  gchar *uid;
  GtkTreeIter iter, child;
  gboolean valid;
  gchar *str, *escaped;

#ifdef TRACE_FUNCTION
  g_print ("gnomeicu_tree_refresh_visuals\n");
#endif

  valid = gtk_tree_model_get_iter_first (GTK_TREE_MODEL(MainData->visible_store), 
					 &iter);
  while (valid) {
    valid = gtk_tree_model_iter_children (GTK_TREE_MODEL(MainData->visible_store), 
					  &child, &iter);
    while(valid) {
      gtk_tree_model_get (GTK_TREE_MODEL (MainData->visible_store), &child,
			  UID_COLUMN, &uid, -1);
      contact = Find_User(uid);
      if (contact) {

	if (!g_utf8_validate (kontakt->nick, -1, NULL)) {
	  g_free(kontakt->nick);
	  kontakt->nick = g_strdup(kontakt->uin);
	}
	
	escaped = g_markup_escape_text(kontakt->nick, strlen(kontakt->nick)+1);
		
	str = g_strdup_printf("<span foreground=\"%s\">%s</span>",
			      get_foreground_for_status(kontakt->status),
			      escaped);
	g_free(escaped);
	gtk_tree_store_set (MainData->visible_store, &child,
			    ICON_COLUMN, get_pixbuf_for_status(kontakt->status),
			    NICKNAME_COLUMN, str,
			    -1);
	g_free(str);
      }
      g_free(uid);
      valid = gtk_tree_model_iter_next (GTK_TREE_MODEL(MainData->visible_store), &child);
    }
    valid = gtk_tree_model_iter_next (GTK_TREE_MODEL(MainData->visible_store), &iter);
  }

  valid = gtk_tree_model_get_iter_first (GTK_TREE_MODEL(MainData->contacts_store), &iter);
  while (valid) {
    valid = gtk_tree_model_iter_children (GTK_TREE_MODEL(MainData->contacts_store), &child, &iter);
    while(valid) {
      gtk_tree_model_get (GTK_TREE_MODEL (MainData->contacts_store), &child,
			  UID_COLUMN, &uid, -1);
      contact = Find_User(uid);
      if (contact) {
	escaped = g_markup_escape_text(kontakt->nick, strlen(kontakt->nick)+1);
	str = g_strdup_printf("<span foreground=\"%s\">%s</span>",
			      get_foreground_for_status(kontakt->status),
			      escaped);
	g_free(escaped);
	gtk_tree_store_set (MainData->contacts_store, &child,
			    ICON_COLUMN, get_pixbuf_for_status(kontakt->status),
			    NICKNAME_COLUMN, str,
			    -1);
	g_free(str);
      }
      g_free(uid);
      valid = gtk_tree_model_iter_next (GTK_TREE_MODEL(MainData->contacts_store), &child);
    }
    valid = gtk_tree_model_iter_next (GTK_TREE_MODEL(MainData->contacts_store), &iter);
  }
}

static gint gnomeicu_tree_total_contacts_in_group (const gchar *group_name)
{
        gint gid;
	GtkTreeIter iter;
	gboolean valid;
	gint result=0;

#ifdef TRACE_FUNCTION
	g_print ("gnomeicu_tree_total_contacts_in_group\n");
#endif

	if (!group_name)
	  return 0;

	valid = gtk_tree_model_get_iter_first (GTK_TREE_MODEL(MainData->contacts_store), &iter);
	while (valid) {
		gtk_tree_model_get (GTK_TREE_MODEL(MainData->contacts_store), &iter,
							GID_COLUMN, &gid, -1);
		if (groups_name_by_gid(gid) && !strcmp(groups_name_by_gid(gid),group_name))
		  result += gtk_tree_model_iter_n_children (GTK_TREE_MODEL (MainData->contacts_store), &iter);
		valid = gtk_tree_model_iter_next (GTK_TREE_MODEL(MainData->contacts_store), &iter);
	}

	return result;
}

static gint gnomeicu_tree_online_contacts_in_group (const gchar *group_name)
{
	gint gid;
	GtkTreeIter iter;
	gboolean valid;
	gint result = 0;

#ifdef TRACE_FUNCTION
	g_print ("gnomeicu_tree_online_contacts_in_group\n");
#endif

	if (!group_name)
	  return 0;

	valid = gtk_tree_model_get_iter_first (GTK_TREE_MODEL(MainData->visible_store), &iter);
	while (valid) {
		gtk_tree_model_get (GTK_TREE_MODEL(MainData->visible_store), &iter,
							GID_COLUMN, &gid, -1);
		if (groups_name_by_gid(gid) &&
		    !strcmp(groups_name_by_gid(gid), group_name)) {
			/* This is the group */
			result = gtk_tree_model_iter_n_children (GTK_TREE_MODEL (MainData->visible_store), &iter);
			break;
		}
		valid = gtk_tree_model_iter_next (GTK_TREE_MODEL(MainData->visible_store), &iter);
	}

	return result;
}

gint gnomeicu_tree_online_contacts (void)
{
	gint gid;
	GtkTreeIter iter;
	gboolean valid;
	gint ct = 0;

#ifdef TRACE_FUNCTION
	g_print ("gnomeicu_tree_online_contacts\n");
#endif

	valid = gtk_tree_model_get_iter_first (GTK_TREE_MODEL(MainData->visible_store), &iter);
	while (valid) {
		gtk_tree_model_get (GTK_TREE_MODEL(MainData->visible_store), &iter,
							GID_COLUMN, &gid, -1);

		if (gid)
		  ct += gtk_tree_model_iter_n_children (GTK_TREE_MODEL (MainData->visible_store), &iter);

		valid = gtk_tree_model_iter_next (GTK_TREE_MODEL(MainData->visible_store), &iter);
	}

	return ct;
}

void gnomeicu_tree_update_headers (void)
{
	gchar *str;
	gint gid;
	GtkTreeIter iter;
	gboolean valid;

#ifdef TRACE_FUNCTION
	g_print ("gnomeicu_tree_update_headers\n");
#endif

	valid = gtk_tree_model_get_iter_first (GTK_TREE_MODEL(MainData->visible_store), &iter);
	while (valid) {
		gtk_tree_model_get (GTK_TREE_MODEL(MainData->visible_store), &iter,
							GID_COLUMN, &gid, -1);
		if (gid && groups_name_by_gid(gid)) {
		  str = g_strdup_printf ("<b>%s (%d/%d)</b>",
					 (groups_name_by_gid(gid) ? groups_name_by_gid(gid) : _("ERROR")),
					 gnomeicu_tree_online_contacts_in_group (groups_name_by_gid(gid)),
					 gnomeicu_tree_total_contacts_in_group (groups_name_by_gid(gid)));
		  
		  gtk_tree_store_set (MainData->visible_store, &iter,
				      NICKNAME_COLUMN, str, -1);
		  g_free(str);
		}
		valid = gtk_tree_model_iter_next (GTK_TREE_MODEL(MainData->visible_store), &iter);
	}

	valid = gtk_tree_model_get_iter_first (GTK_TREE_MODEL(MainData->contacts_store), &iter);
	while (valid) {
	  	gtk_tree_model_get (GTK_TREE_MODEL(MainData->contacts_store), &iter,
							GID_COLUMN, &gid, -1);
		str = g_strdup_printf ("<b>%s</b>", (groups_name_by_gid(gid) ? groups_name_by_gid(gid) : _("ERROR")));
		gtk_tree_store_set (MainData->contacts_store, &iter,
		                    NICKNAME_COLUMN, str, -1);
		valid = gtk_tree_model_iter_next (GTK_TREE_MODEL(MainData->contacts_store), &iter);
	}
}

static gboolean gnomeicu_tree_contact_exists (Contact_Member *contact)
{
	gchar *uid;
	GtkTreeIter iter, child;
	gboolean valid, valid_child;

#ifdef TRACE_FUNCTION
	g_print ("gnomeicu_tree_contact_exists\n");
#endif


	valid = gtk_tree_model_get_iter_first (GTK_TREE_MODEL(MainData->contacts_store), &iter);
	while (valid) {

		valid_child = gtk_tree_model_iter_children (GTK_TREE_MODEL(MainData->contacts_store), &child, &iter);
		while(valid_child) {
			gtk_tree_model_get (GTK_TREE_MODEL (MainData->contacts_store), &child,
			                    UID_COLUMN, &uid, -1);
			if (uid && !strcmp (uid, contact->uin)) {
				g_free(uid);
				return TRUE;
			}
			g_free(uid);
			valid_child = gtk_tree_model_iter_next (GTK_TREE_MODEL(MainData->contacts_store), &child);
		}

		valid = gtk_tree_model_iter_next (GTK_TREE_MODEL(MainData->contacts_store), &iter);
	}

	return FALSE;
}

void gnomeicu_tree_add_group (const GroupInfo *ginfo)
{
	gchar *str;
	gint  gid;
	GtkTreeIter iter;
	gboolean valid;
	gboolean exists;

#ifdef TRACE_FUNCTION
	g_print ("gnomeicu_tree_add_group\n");
#endif

	if (!GTK_IS_WIDGET (MainData->contacts_tree))
		return;

	valid = gtk_tree_model_get_iter_first (GTK_TREE_MODEL(MainData->contacts_store), &iter);
	exists = FALSE;

	while (valid) {
		gtk_tree_model_get (GTK_TREE_MODEL(MainData->contacts_store), &iter,
		                    GID_COLUMN, &gid, -1);

		if (gid && ginfo->gid == gid) {
			exists = TRUE;
			break;
		}
		valid = gtk_tree_model_iter_next (GTK_TREE_MODEL(MainData->contacts_store), &iter);
	}

	if (!exists) {
		gtk_tree_store_append (MainData->contacts_store, &iter, NULL);
		str = g_strdup_printf("<b>%s</b>", ginfo->name);
		gtk_tree_store_set (MainData->contacts_store, &iter,
		                    ICON_COLUMN, icon_group_pixbuf, NICKNAME_COLUMN,
		                    str, GID_COLUMN, ginfo->gid,
		                    CONTACT_COLUMN, NULL,
		                    UID_COLUMN, NULL, -1);
		g_free(str);
	}
}

void gnomeicu_tree_rename_group (const GroupInfo *ginfo)
{
  GtkTreeIter iter, child;
  gint gid;
  gboolean valid, valid_child;
  gchar *str;
  Contact_Member *contact;
  
#ifdef TRACE_FUNCTION
  g_print ("gnomeicu_tree_rename_group\n");
#endif

  valid = gtk_tree_model_get_iter_first (GTK_TREE_MODEL(MainData->contacts_store), &iter);

  while (valid) {
    gtk_tree_model_get (GTK_TREE_MODEL(MainData->contacts_store), &iter,
			GID_COLUMN, &gid, -1);
    if (gid == ginfo->gid) {
      str = g_strdup_printf("<b>%s</b>", ginfo->name);
      gtk_tree_store_set (MainData->contacts_store, &iter,
		                    NICKNAME_COLUMN, str, -1);
      g_free(str);
      valid_child = gtk_tree_model_iter_children (GTK_TREE_MODEL(MainData->contacts_store), &child, &iter);
      while(valid_child) {
	gtk_tree_model_get (GTK_TREE_MODEL(MainData->contacts_store), &child,
			    CONTACT_COLUMN, &contact, -1);
	gnomeicu_tree_user_remove_online (contact);
	gnomeicu_tree_user_add_online (contact);
	valid_child = gtk_tree_model_iter_next (GTK_TREE_MODEL(MainData->contacts_store), &child);
      }
    }
    valid =  gtk_tree_model_iter_next (GTK_TREE_MODEL(MainData->contacts_store), &iter);
  }

  gnomeicu_tree_update_headers();

}

void gnomeicu_tree_remove_group (const GroupInfo *ginfo)
{
	gint gid;
	GtkTreeIter iter;
	gboolean valid;

#ifdef TRACE_FUNCTION
	g_print ("gnomeicu_tree_remove_group\n");
#endif

	if (!GTK_IS_WIDGET (MainData->contacts_tree))
		return;

	valid = gtk_tree_model_get_iter_first (GTK_TREE_MODEL(MainData->contacts_store), &iter);

	while (valid) {
		gtk_tree_model_get (GTK_TREE_MODEL(MainData->contacts_store), &iter,
		                    GID_COLUMN, &gid, -1);
		if (ginfo->gid == gid) {
			gtk_tree_store_remove(MainData->contacts_store, &iter);
		}
		valid = gtk_tree_model_iter_next (GTK_TREE_MODEL(MainData->contacts_store), &iter);
	}
}

