/* ****************************************************************************
  This file is part of KBabel

  Copyright (C) 1999-2000 by Matthias Kiefer
                            <matthias.kiefer@gmx.de>
                2001      by Stanislav Visnovsky <visnovsky@kde.org>

  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

**************************************************************************** */
#ifndef CATMANLISTITEM_H
#define CATMANLISTITEM_H

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <qstringlist.h>
#include <qlistview.h>
#include <qdatetime.h>
#include <qfileinfo.h>

class CatalogManagerView;
class QPixmap;
struct PoInfo;

class CatManListItem : public QListViewItem
{
public:
   /** the type of this item */
   enum Type{File,Dir};

   CatManListItem(CatalogManagerView *view, QListViewItem* parent,QString fullPath
				   ,QString fullPotPath,QString package);

   /** creates the toplevel root item with package name "/" */
   CatManListItem(CatalogManagerView *view, QListView* parent,QString fullPath, QString fullPotPath);

   /** 
	* returns the package names (including relative path) of the 
	* children of this item 
	*/
   QStringList contentsList(bool onlyFiles=false) const;
   /**
   * returns the package names of all children of this item 
   * (including all subdirectries)
   * @param onlyFiles flag, if only the names of files should be returned
   * @see CatManListItem::contentsList
   */
   QStringList allChildrenList(bool onlyFiles=false) const;

   void setMarked(bool on);
   bool marked() const;
   /**
   * checks if the file on the disc has changed,
   * reads information about the file and displays it
   * @param noParents flag, if the update has to include the parent
   * of the item, if the status has changed. Since at the first build of 
   * the tree, the status of every item changes, this is not useful then.
   */
   void checkUpdate(bool noParents=false);
   void forceUpdate();


   /** return the absolute filename of the po-File */
   QString poFile() const;
   /** return the absolute filename of the pot-File */
   QString potFile() const;
   /** returns the package name (inlcuding relative path to base-directory) */
   QString package(bool rootSlash=true) const;

   /** returns the package name (without path) */
   QString name() const;

   /**
   * returns the type of this item
   * @see CatManListItem::Type
   */
   Type type() const{return _type;}
   bool isDir() const;
   bool isFile() const;
   /** returns true, if the po-file exists*/
   bool hasPo() const;
   /** returns true, if the pot-file exists*/
   bool hasPot() const;
   /** 
	* @return the number of fuzzy messages in the po-file, 
	* 0 if no po-file exists
	*/
   int fuzzy() const;
   /** 
	* @return the number of untranslated messages in the po-file, 
	* @ref total if no po-file exists
	*/
   int untranslated() const;
   /** @return number of messages in the po- or pot-file */
   int total() const;
   /** 
	* @return true, if there are untranslated or fuzzy items.
	* If this item is a directory, it returns true, if a subitem
	* contains untranslated or fuzzy items
	*/
   bool needsWork() const;
   /** 
	* @return true, if there were errors while parsing the file
	*/
   bool hasErrors() const {return _hasErrors;}

   virtual QString key(int col,bool) const;
   virtual void setOpen(bool);

   /** paints the marking, if this package has no template */
   QPixmap paintExclamation(QPixmap*);
   
   void updateAfterSave( PoInfo &po);
   
   QStringList &wordList() { return _wordList; }
   bool wordsUpdated() { return _wordListUpdated; }

   /** These are not in Qt, so we need to implement it ourselves*/
   QListViewItem *previousSibling();
   QListViewItem *lastChild();
   
private:
   void init(const QString& fullPath, const QString& fullPotPath,const QString& package);
   /**
   * updates the item
   * @param showPoInfo if true, reads information about the 
   * file using @ref Catalog::info
   * ( slow for big files )
   * @param includeChildren flag, if possible children should be updated,too
   * @param noParents flag, if parents should be updated, when state 
   * of the item has changed
   */
   void update(bool showPoInfo=true,bool includeChildren=false
				   , bool noParents=false );
   void updateParents();

private:
   /**
   * holds the date and the time this item was
   * last updated. This is used to check, if the file
   * on the disc has changed since last update.
   */
   QDateTime _lastUpdated;

   /** the po-file */
   QFileInfo _primary;
   /** the pot-file */
   QFileInfo _template;
   /**
   * The package name, includes the relative path beginning 
   * at the base directory.
   * The package name begins with "/" and if this is a directory it end with "/"
   * The root item has the package name "/"
   * @see CatManListItem::CatManListItem
   */
   QString _package;
   Type _type;
   bool _marked;

   /** flag, to detect if file has been deleted or is new */
   bool _hasPo;
   /** flag, to detect if file has been deleted or is new */
   bool _hasPot;

   /** flag, if the PO-file has a syntax error */
   bool _hasErrors;
   
   /** parent view for this item, used for stopping the activity */
   CatalogManagerView *_view;
   
   /** index of words, but it does not contain any useful information as values */
   QStringList _wordList;
   bool _wordListUpdated;
};

#endif // CATMANLISTITEM_H
