/*
 * power.cpp
 *
 * Copyright (c) 1999 Paul Campbell <paul@taniwha.com>
 * Copyright (c) 2002 Marc Mutz <mutz@kde.org>
 *
 * Requires the Qt widget libraries, available at no cost at
 * http://www.troll.no/
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include "power.h"
#include "portable.h"
#include "version.h"

#include <klocale.h>
#include <kconfig.h>
#include <knumvalidator.h>
#include <kdialog.h>
#include <kapplication.h>

#include <qlayout.h>
#include <qvbuttongroup.h>
#include <qspinbox.h>
#include <qradiobutton.h>
#include <qhbox.h>

PowerConfig::PowerConfig (QWidget * parent, const char *name)
  : KCModule(parent, name),
    nopowerBox(0),
    nopowerStandby(0),
    nopowerSuspend(0),
    nopowerOff(0),
    nopowerHibernate(0),

    powerBox(0),
    powerStandby(0),
    powerSuspend(0),
    powerOff(0),
    powerHibernate(0),

    noeditwait(0),
    editwait(0)
{
  apm = laptop_portable::has_power_management();
  config =  new KConfig("kcmlaptoprc");

  if (!apm) {
    QVBoxLayout *top_layout = new QVBoxLayout(this, KDialog::marginHint(),
					      KDialog::spacingHint());

    QLabel* explain = laptop_portable::no_power_management_explanation(this);
    top_layout->addWidget(explain);

    top_layout->addStretch(1);
  } else {
    int can_standby = laptop_portable::has_standby();
    int can_suspend = laptop_portable::has_suspend();
    int can_hibernate = laptop_portable::has_hibernation();

    if (!can_standby && !can_suspend && !can_hibernate)
      apm = 0;
    if (!apm) {
      QVBoxLayout *top_layout = new QVBoxLayout(this, KDialog::marginHint(),
						KDialog::spacingHint());

      QLabel* explain = laptop_portable::how_to_do_suspend_resume(this);
      top_layout->addWidget(explain);

      top_layout->addStretch(1);
    } else {
      QVBoxLayout *top_layout = new QVBoxLayout(this, KDialog::marginHint(),
						KDialog::spacingHint());
      QHBoxLayout *hlay = new QHBoxLayout( top_layout );

      nopowerBox = new QVButtonGroup(i18n("Not Powered:"), this);
      nopowerBox->layout()->setSpacing( KDialog::spacingHint() );
      hlay->addWidget( nopowerBox );

      if (can_suspend)
	nopowerSuspend = new QRadioButton(i18n("&Suspend"), nopowerBox);
      if (can_standby)
	nopowerStandby = new QRadioButton(i18n("Standb&y"), nopowerBox);
      if (can_hibernate)
	nopowerHibernate = new QRadioButton(i18n("H&ibernate"), nopowerBox);

      nopowerOff = new QRadioButton(i18n("&Off"), nopowerBox);
      connect(nopowerBox, SIGNAL(clicked(int)), this, SLOT(configChanged()));

      QHBox *hbox = new QHBox( nopowerBox );
      QLabel* noedlabel = new QLabel(i18n("&Wait for:"), hbox);
      noeditwait = new QSpinBox( 1, 60*24 /*1 day*/, 1, hbox );
      noeditwait->setSuffix( i18n("keep short, unit in spinbox", "min") );
      noedlabel->setBuddy( noeditwait );
      hbox->setStretchFactor( noeditwait, 1 );
      connect( noeditwait, SIGNAL(valueChanged(int)),
	       this, SLOT(configChanged()));

      powerBox = new QVButtonGroup(i18n("Powered:"), this);
      powerBox->layout()->setSpacing( KDialog::spacingHint() );
      hlay->addWidget( powerBox );

      if (can_suspend)
	powerSuspend = new QRadioButton(i18n("S&uspend"), powerBox);
      if (can_standby)
	powerStandby = new QRadioButton(i18n("Sta&ndby"), powerBox);
      if (can_hibernate)
	powerHibernate = new QRadioButton(i18n("Hi&bernate"), powerBox);

      powerOff = new QRadioButton(i18n("O&ff"), powerBox);
      connect(powerBox, SIGNAL(clicked(int)), this, SLOT(configChanged()));

      hbox = new QHBox( powerBox );
      QLabel* edlabel = new QLabel(i18n("Wai&t for:"), hbox);
      editwait = new QSpinBox( 1, 60*24 /*1 day*/, 1, hbox );
      editwait->setSuffix( i18n("keep short, unit in spinbox", "min") );
      edlabel->setBuddy( editwait );
      hbox->setStretchFactor( editwait, 1 );
      connect( editwait, SIGNAL(valueChanged(int)),
	       this, SLOT(configChanged()));

      hlay->addStretch(1);

      QLabel* explain = new QLabel(i18n("This panel configures the behavior of the automatic power-down feature - it works as a sort of extreme\nscreensaver. You can configure different timeouts\nand types of behavior depending on whether or not\nyour laptop is plugged in to the mains supply."), this);
      top_layout->addWidget(explain, 0, Qt::AlignLeft);

      if (can_standby) {
	QLabel* explain3 = new QLabel(i18n("Different laptops may respond to 'standby' in\ndifferent ways - in many it's only a temporary\nstate and may not be useful for you."), this);
	top_layout->addWidget(explain3, 0, Qt::AlignLeft);
      }

      top_layout->addStretch(1);

      top_layout->addWidget( new QLabel( i18n("Version: %1").arg(LAPTOP_VERSION), this ), 0, Qt::AlignRight );
    }
  }

  load();
}


void PowerConfig::save()
{
  if (editwait) {
      	power = getPower();
      	nopower = getNoPower();
	edit_wait = editwait->value();
	noedit_wait = noeditwait->value();
  }

  config->setGroup("LaptopPower");
  config->writeEntry("NoPowerSuspend", nopower);
  config->writeEntry("PowerSuspend", power);
  config->writeEntry("PowerWait", edit_wait);
  config->writeEntry("NoPowerWait", noedit_wait);
  config->sync();
  changed(false);
  kapp->startServiceByDesktopName("klaptopdaemon");
}

void PowerConfig::load()
{
  config->setGroup("LaptopPower");
  nopower = config->readNumEntry("NoPowerSuspend", (nopowerStandby?1:nopowerSuspend?2:0));
  power = config->readNumEntry("PowerSuspend", 0);
  edit_wait = config->readNumEntry("PowerWait", 20);
  noedit_wait = config->readNumEntry("NoPowerWait", 5);

  // the GUI should reflect the real values
  if (editwait) {
	editwait->setValue(edit_wait);
	noeditwait->setValue(noedit_wait);
   	setPower(power, nopower);
  }
  changed(false);
}

void PowerConfig::defaults()
{
  setPower(1, 0);
  edit_wait = 20;
  noedit_wait = 5;
  if (editwait) {
	editwait->setValue(edit_wait);
        noeditwait->setValue(noedit_wait);
  }
}

int  PowerConfig::getNoPower()
{
  if (!apm)
	return(nopower);
  if (nopowerHibernate && nopowerHibernate->isChecked())
    return 3;
  if (nopowerStandby && nopowerStandby->isChecked())
    return 1;
  if (nopowerSuspend && nopowerSuspend->isChecked())
    return 2;
  return(0);
}

int  PowerConfig::getPower()
{
  if (!apm || !powerOff)
	return(power);
  if (powerHibernate && powerHibernate->isChecked())
    return 3;
  if (powerStandby && powerStandby->isChecked())
    return 1;
  if (powerSuspend && powerSuspend->isChecked())
    return 2;
  return(0);
}

void PowerConfig::setPower(int p, int np)
{
  if (!apm || nopowerOff == 0)
	return;
  if (nopowerSuspend) {
	nopowerSuspend->setChecked(FALSE);
  } else {
	if (np == 2) np = 0;
  }
  if (nopowerStandby) {
	nopowerStandby->setChecked(FALSE);
  } else {
	if (np == 1) np = 0;
  }
  if (nopowerHibernate) {
	nopowerHibernate->setChecked(FALSE);
  } else {
	if (np == 3) np = 0;
  }
  nopowerOff->setChecked(FALSE);
  switch (np) {
  case 0: nopowerOff->setChecked(TRUE);break;
  case 1: nopowerStandby->setChecked(TRUE);break;
  case 2: nopowerSuspend->setChecked(TRUE);break;
  case 3: nopowerHibernate->setChecked(TRUE);break;
  }
  if (powerSuspend) {
	powerSuspend->setChecked(FALSE);
  } else {
	if (p == 2) np = 0;
  }
  if (powerStandby) {
	powerStandby->setChecked(FALSE);
  } else {
	if (p == 1) np = 0;
  }
  if (powerHibernate) {
	powerHibernate->setChecked(FALSE);
  } else {
	if (p == 3) np = 0;
  }
  powerOff->setChecked(FALSE);
  switch (p) {
  case 0: powerOff->setChecked(TRUE);break;
  case 1: powerStandby->setChecked(TRUE);break;
  case 2: powerSuspend->setChecked(TRUE);break;
  case 3: powerHibernate->setChecked(TRUE);break;
  }
}



void PowerConfig::configChanged()
{
  emit changed(true);
}


QString PowerConfig::quickHelp() const
{
  return i18n("<h1>Laptop Power Control</h1>This module allows you to "
	"control the power settings of your laptop");

}

#include "power.moc"
