#include "board.h"
#include "board.moc"

#include "base/defines.h"
#include "base/piece.h"
#include "misc_ui.h"
#include "ai.h"


Board::Board(bool graphic, GiftPool *gp, QWidget *parent)
: BaseBoard(graphic, parent), _giftPool(gp), aiEngine(0)
{}

Board::~Board()
{
	delete aiEngine;
}

void Board::setType(bool _ai)
{
	Q_ASSERT( graphic() );
	if (_ai) {
		if ( aiEngine==0 ) aiEngine = Factory::self()->createAI();
	} else {
		delete aiEngine;
		aiEngine = 0;
	}
}

void Board::init(const GTInitData &data)
{
	BaseBoard::init(data);
	randomGarbage.setSeed(data.seed);
    _giftPool->reset();
}

void Board::stop()
{
    BaseBoard::stop();
	if (aiEngine) aiEngine->stop();
}

void Board::showBoard(bool show)
{
    BaseBoard::showBoard(show);
	showCanvas(_next, show);
}

void Board::unpause()
{
    BaseBoard::unpause();
	if (aiEngine) aiEngine->start(); // eventually restart thinking
}

void Board::updateLevel(uint newLevel)
{
	BaseBoard::updateLevel(newLevel);
	emit updateLevelSignal(newLevel);
	if ( graphic() ) startTimer();
}

/*****************************************************************************/
void Board::pMoveLeft()
{
	if ( state==Normal ) {
		moveLeft();
		main->update();
	}
}

void Board::pMoveRight()
{
	if ( state==Normal ) {
		moveRight();
		main->update();
	}
}

void Board::pOneLineDown()
{
	if ( state==Normal ) {
		oneLineDown();
		main->update();
	}
}

void Board::pRotateLeft()
{
	if ( state==Normal ) {
		rotateLeft();
		main->update();
	}
}

void Board::pRotateRight()
{
	if ( state==Normal ) {
		rotateRight();
		main->update();
	}
}

void Board::pDropDown()
{
	if ( state!=Normal ) return;
	if ( !graphic() ) dropDown();
	else {
		_dropHeight = 0;
		oneLineDown();
		if ( state!=Normal ) return;
		state = DropDown;
		startTimer();
	}
}

void Board::pieceDropped(uint dropHeight)
{
	if ( state==DropDown ) state = Normal;
	else _dropHeight = dropHeight;
	_beforeGlue(true);
}

void Board::_beforeGlue(bool first)
{
	if ( graphic() ) {
		state = (beforeGlue(_dropHeight>=1, first) ? BeforeGlue : Normal);
		if ( state==BeforeGlue ) {
			startTimer();
			return;
		}
	}
	gluePiece();
}

void Board::gluePiece()
{
	BaseBoard::gluePiece();
	_afterGlue(true);
}

void Board::_afterGlue(bool first)
{
	bool b = afterGlue(!graphic(), first);
	if ( graphic() ) {
		state = (b ? AfterGlue : Normal);
		if ( state==AfterGlue ) {
			startTimer();
			return;
		}
	}

	updateScore(score() + _dropHeight);
	if ( needRemoving() ) {
		_beforeRemove(true);
		return;
	}

    checkGift();
}

void Board::checkGift()
{
    if ( graphic() && _giftPool->pending() ) {
        if ( putGift(_giftPool->take()) ) {
            computeInfos();
            _afterGift(true);
        } else gameOver();
        return;
    }
    newPiece();
}

void Board::_afterGift(bool first)
{
	Q_ASSERT( graphic() );
    state = (afterGift(first) ? AfterGift : Normal);
    if ( state==AfterGift ) startTimer();
    else checkGift();
}

void Board::newPiece()
{
    Q_ASSERT( !graphic() || state==Normal );
    BaseBoard::newPiece();
    if ( graphic() ) {
        if ( state==GameOver ) return;
        main->update();
        _next->update();
        startTimer();
        if (aiEngine) aiEngine->launch(this);
        // else : a human player can think by himself ...
    }
}

bool Board::timeout()
{
    if ( BaseBoard::timeout() ) return true;

	switch (state) {
        case DropDown:   _dropHeight++;
        case Normal:     oneLineDown();      break;
        case BeforeGlue: _beforeGlue(false); break;
        case AfterGlue:  _afterGlue(false);  break;
        case AfterGift:  _afterGift(false);  break;
        default:         return false;
	}
	main->update();
    return true;
}

bool Board::startTimer()
{
    if ( BaseBoard::startTimer() ) return true;

    const BoardInfo &bi = Factory::self()->boardInfo();
	switch (state) {
        case Normal:
            timer.start(bi.baseTime / (1 + level()));
            break;
        case DropDown:
            timer.start(bi.dropDownTime);
            break;
        case BeforeGlue:
            timer.start(bi.beforeGlueTime, true);
            break;
        case AfterGlue:
            timer.start(bi.afterGlueTime, true);
            break;
        case AfterGift:
            timer.start(bi.afterGiftTime, true);
            break;
        default:
            return false;
	}
    return true;
}


//-----------------------------------------------------------------------------
bool Board::beforeGlue(bool bump, bool first)
{
	if ( !bump ) return false;
	if (first) {
        loop = 0;
        return true;
    } else loop++;

	float dec = blockSize() * (loop+1) * -0.1;
	if (_animations) bumpCurrentPiece((int)dec);

	return ( loop!=Factory::self()->boardInfo().nbBumpStages );
}
