/*****************************************************************************/

/*
 *      diag.c  --  P3D Telemetry Decoder Application, diagnostic part.
 *
 *      Copyright (C) 2000
 *        Thomas Sailer (sailer@ife.ee.ethz.ch)
 *
 *      This program is free software; you can redistribute it and/or modify
 *      it under the terms of the GNU General Public License as published by
 *      the Free Software Foundation; either version 2 of the License, or
 *      (at your option) any later version.
 *
 *      This program is distributed in the hope that it will be useful,
 *      but WITHOUT ANY WARRANTY; without even the implied warranty of
 *      MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *      GNU General Public License for more details.
 *
 *      You should have received a copy of the GNU General Public License
 *      along with this program; if not, write to the Free Software
 *      Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

/*****************************************************************************/

#define _GNU_SOURCE
#define _REENTRANT

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

/* AIX requires this to be the first thing in the file.  */
#ifndef __GNUC__
# if HAVE_ALLOCA_H
#  include <alloca.h>
# else
#  ifdef _AIX
#pragma alloca
#  else
#   ifndef alloca /* predefined by HP cc +Olibcalls */
char *alloca ();
#   endif
#  endif
# endif
#endif

#include "p3dapp.h"
#include "modem.h"
#include "audioio.h"

#include <gtk/gtk.h>

#include "interface.h"
#include "support.h"
#include "callbacks.h"

#include "spectrum.h"
#include "scope.h"

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <pthread.h>
#include <unistd.h>

#ifdef WIN32

#include <windows.h>
#include <winsock.h>

#else

#include <sys/uio.h>
#include <netinet/in.h>
#include <sys/socket.h>
#include <arpa/inet.h>

#endif

/* ---------------------------------------------------------------------- */

//#ifdef WIN32
#define TXTERMNOCANCEL
//#endif


#define DIAGFLG_MODEM           1
#define DIAGFLG_SCOPE           2
#define DIAGFLG_SPECTRUM        4
#define DIAGFLG_WINDOWMASK      (DIAGFLG_SCOPE|DIAGFLG_SPECTRUM)

static struct {
	unsigned int flags;
	struct modulator *modch;
	struct demodulator *demodch;
	guint timeoutid;
	unsigned int samplerate;
	int stpsock;
	struct {
		unsigned int newstate;
		unsigned int state;
		unsigned int carrierfreq;
		unsigned int newpacket;
		u_int16_t crc;
		unsigned char packet[512+2];
	} p3d;
        unsigned int traceupd;
        void *modstate, *demodstate;
	unsigned int bitrate;
	unsigned int filein;
        pthread_t rxthread, txthread;
        pthread_cond_t txcond;
        pthread_mutex_t txmutex;
	struct pttio pttio;
	struct audioio *audioio;
#ifdef TXTERMNOCANCEL
        unsigned int txterminate;
#endif
} diagstate = {
	0, &p3dmodulator, &p3ddemodulator, 0,
	0, -1, 
	{ 0, 0, 0, 0, },
        0,
        NULL, NULL, 0, 0,
        { }, { }, PTHREAD_COND_INITIALIZER, PTHREAD_MUTEX_INITIALIZER
};

/* ---------------------------------------------------------------------- */

/* win32 does not support writev */

#if WIN32

typedef char *caddr_t;

struct iovec {
        caddr_t iov_base;
        size_t iov_len;
};

int writev(int fd, const struct iovec *iov, int niov)
{
	size_t len = 0;
	unsigned int i;
	unsigned char *buf, *bp;

	if (niov < 0) {
		errno = EINVAL;
		return -1;
	}
	for (i = 0; i < niov; i++) {
		if (iov[i].iov_len < 0) {
			errno = EINVAL;
			return -1;
		}
		len += iov[i].iov_len;
	}
	if (!len)
		return 0;
	bp = buf = alloca(len);
	for (i = 0; i < niov; i++) {
		if (iov[i].iov_len > 0) {
			memcpy(bp, iov[i].iov_base, iov[i].iov_len);
			bp += iov[i].iov_len;
		}
	}
	return send(fd, buf, len, 0);
}

#endif

/* ---------------------------------------------------------------------- */

const struct modemparams stpparams[] = {
        { "udpport", "STP UDP Port", "UDP port number for Satellite Telemetry Protocol broadcasts; -1 disables STP",
          "-1", MODEMPAR_NUMERIC, { n: { -1, 65535, 100, 1000 } } },
        { "udpaddr", "STP UDP Address", "IP Address for UDP STP broadcasts", "127.0.0.1", MODEMPAR_STRING }, 
        { NULL }
};

static void stp_close(void)
{
	int fd = diagstate.stpsock;

	if (fd == -1)
		return;
	diagstate.stpsock = -1;
	close(fd);
}

static int parseip(struct in_addr *ipaddr, const char *cp)
{
        unsigned int b1, b2, b3, b4;
        
        if (!cp || !*cp)
                return 0;
        if (sscanf(cp, "%u.%u.%u.%u", &b1, &b2, &b3, &b4) == 4 &&
            b1 < 256 && b2 < 256 && b3 < 256 && b4 < 256) {
                ipaddr->s_addr = htonl((b1 << 24) | (b2 << 16) | (b3 << 8) | b4);
                return 1;
        }
        ipaddr->s_addr = 0;
        logprintf(MLOG_ERROR, "stp: invalid IP address \"%s\"\n", cp);
        return 0;
}

static int stp_init(const char *params[])
{
	struct sockaddr_in sin;
	int fd, port;

	stp_close();
        if (!params[0])
                return -1;
        port = strtol(params[0], NULL, 0);
        if (port < 0 || port > 65535)
                return -1;
	fd = socket(PF_INET, SOCK_DGRAM, 0);
	if (fd == -1)
		return -1;
	sin.sin_family = AF_INET;
        if (!parseip(&sin.sin_addr, params[1]))
                sin.sin_addr.s_addr = ntohl(INADDR_LOOPBACK);
	sin.sin_port = htons(port);
	if (connect(fd, &sin, sizeof(sin))) {
		close(fd);
		return -1;
	}
	diagstate.stpsock = fd;
	return 0;
}

static int stp_send(const unsigned char *pkt)
{
	static const unsigned char syncvec[4] = { 0x39, 0x15, 0xed, 0x30 };
	char hdr[256];
	int hdlen;
	char htime[128];
	struct iovec iov[3];
	struct tm *tm;
	time_t t;
	
	if (diagstate.stpsock == -1)
		return -1;
	time(&t);
	tm = gmtime(&t);
	strftime(htime, sizeof(htime), "Date: %a, %d %b %Y %H:%M:%S UTC\r\n", tm);
	hdlen = snprintf(hdr, sizeof(hdr), 
			"Source: amsat.ao-40.ihu.standard\r\n"
			"%s"
			"Length: 4144\r\n\r\n", htime);
	if (hdlen >= sizeof(hdr) || hdlen <= 0)
		return -1;
	iov[0].iov_base = hdr;
	iov[0].iov_len = hdlen;
	iov[1].iov_base = &syncvec[0];
	iov[1].iov_len = 4;
	iov[2].iov_base = pkt;
	iov[2].iov_len = 514;
	if (diagstate.stpsock == -1)
		return -1;
	if (writev(diagstate.stpsock, iov, 3) == -1) {
		fprintf(stderr, "STP: write error\n");
		return -1;
	}
	return 0;
}

/* ---------------------------------------------------------------------- */

void p3dreceive(struct modemchannel *chan, const unsigned char *pkt, u_int16_t crc)
{
	if (!crc)
		stp_send(pkt);
	memcpy(diagstate.p3d.packet, pkt, 512+2);
	diagstate.p3d.crc = crc;
	diagstate.p3d.newpacket = 1;
}

void p3drxstate(struct modemchannel *chan, unsigned int synced, unsigned int carrierfreq)
{
	diagstate.p3d.state = synced;
	diagstate.p3d.carrierfreq = carrierfreq;
	diagstate.p3d.newstate = 1;
}

/* ---------------------------------------------------------------------- */

void audiowrite(struct modemchannel *chan, const int16_t *samples, unsigned int nr)
{
	if (!diagstate.audioio->write)
		return;
        diagstate.audioio->write(diagstate.audioio, samples, nr);
}

void audioread(struct modemchannel *chan, int16_t *samples, unsigned int nr, u_int16_t tim)
{
	if (!diagstate.audioio->read) {
		pthread_exit(NULL);
		return;
	}
        diagstate.audioio->read(diagstate.audioio, samples, nr, tim);
}

u_int16_t audiocurtime(struct modemchannel *chan)
{
	if (!diagstate.audioio->curtime)
		return 0;
        return diagstate.audioio->curtime(diagstate.audioio);
}

/* ---------------------------------------------------------------------- */

static void *transmitter(void *dummy)
{
        pthread_mutex_lock(&diagstate.txmutex);
        for (;;) {
#ifdef TXTERMNOCANCEL
                if (diagstate.txterminate) {
                        pthread_mutex_unlock(&diagstate.txmutex);
                        return NULL;
                }
#endif
                if (!diagstate.modch->modulate || !diagstate.audioio->write) {
                        pthread_cond_wait(&diagstate.txcond, &diagstate.txmutex);
                        continue;
                }
                pttsetptt(&diagstate.pttio, 1);
		if (diagstate.audioio->transmitstart)
			diagstate.audioio->transmitstart(diagstate.audioio);
                diagstate.modch->modulate(diagstate.modstate, 0);
		if (diagstate.audioio->transmitstop)
			diagstate.audioio->transmitstop(diagstate.audioio);
                pttsetptt(&diagstate.pttio, 0);
        }
}

static void *receiver(void *dummy)
{
        if (diagstate.demodch->demodulate)
                diagstate.demodch->demodulate(diagstate.demodstate);
        return NULL;
}

/* ---------------------------------------------------------------------- */

static void insert_rxtext(const char *text, unsigned int len)
{
	GtkText *txt;
        
        txt = GTK_TEXT(gtk_object_get_data(GTK_OBJECT(receivewindow), "packetcooked"));
        gtk_text_freeze(txt);
        gtk_text_set_point(txt, gtk_text_get_length(txt));
        gtk_text_insert(txt, NULL, NULL, NULL, text, len);
        len = gtk_text_get_length(txt);
        if (len > 65536) {
                len -= 65536;
                gtk_text_set_point(txt, 0);
                gtk_text_forward_delete(txt, len);
                gtk_text_set_point(txt, gtk_text_get_length(txt));
        }
        gtk_text_thaw(txt);
}

static void update_rxtext(void)
{
	char buf[4096];
        unsigned char pkt[512], *bp;
        unsigned int i;
        
        memcpy(pkt, diagstate.p3d.packet, 512);
        for (bp = pkt, i = 0; i < 512; i++, bp++)
                *bp &= 0x7f;  /* remove highlighting */
        if (pkt[0] >= 'K' && pkt[0] <= 'N') {
                insert_rxtext(buf, snprintf(buf, sizeof(buf), "%.64s\n%.64s\n%.64s\n%.64s\n%.64s\n%.64s\n%.64s\n%.64s\n\n",
                                            &pkt[0], &pkt[64], &pkt[128], &pkt[192], &pkt[256], &pkt[320], &pkt[384], &pkt[448]));
                return;
        }
        if (pkt[0] == 'A' || pkt[0] == 'E') {
                insert_rxtext(buf, snprintf(buf, sizeof(buf), "%.64s\n%.64s\n%.64s\n%.64s\n\n",
                                            &pkt[0], &pkt[64], &pkt[128], &pkt[192]));
                return;
        }
}

static void update_display(void)
{
	GtkText *txt;
        GtkWidget *w;
	unsigned int i, j;
	char buf[4096], *cp;

	if (diagstate.p3d.newstate) {
		diagstate.p3d.newstate = 0;
		snprintf(buf, sizeof(buf), "%s", diagstate.p3d.state == 2 ? "RECEIVE (SYNCSCAN)" : diagstate.p3d.state ? "RECEIVE" : "SYNC HUNT");
		w = GTK_WIDGET(gtk_object_get_data(GTK_OBJECT(receivewindow), "rxstatus"));
		gtk_entry_set_text(GTK_ENTRY(w), buf);
		snprintf(buf, sizeof(buf), "%u", diagstate.p3d.carrierfreq);
		w = GTK_WIDGET(gtk_object_get_data(GTK_OBJECT(receivewindow), "carrierfreq"));
		gtk_entry_set_text(GTK_ENTRY(w), buf);
	}
	if (diagstate.p3d.newpacket) {
		diagstate.p3d.newpacket = 0;
		cp = buf;
		cp += sprintf(cp, "P3D Telemetry Packet: CRC 0x%04x", diagstate.p3d.crc);
		for (i = 0; i < 512; i += 0x10) {
			cp += sprintf(cp, "\n%04x:", i);
			for (j = 0; j < 0x10; j++)
				cp += sprintf(cp, " %02x", diagstate.p3d.packet[i+j]);
			cp += sprintf(cp, "  ");
			for (j = 0; j < 0x10; j++) {
				if (diagstate.p3d.packet[i+j] < ' ' || diagstate.p3d.packet[i+j] >= 0x80)
					cp += sprintf(cp, ".");
				else
					cp += sprintf(cp, "%c", diagstate.p3d.packet[i+j]);
			}
		}
		cp += sprintf(cp, "\n%04x: %02x %02x%44s", 512, diagstate.p3d.packet[512], diagstate.p3d.packet[514], " ");
		for (i = 512; i < 514; i++) {
			if (diagstate.p3d.packet[i] < ' ' || diagstate.p3d.packet[i] >= 0x80)
				cp += sprintf(cp, ".");
			else
				cp += sprintf(cp, "%c", diagstate.p3d.packet[i]);
		}
		cp += sprintf(cp, "\n\n");
		txt = GTK_TEXT(gtk_object_get_data(GTK_OBJECT(receivewindow), "packetraw"));
		gtk_text_freeze(txt);
		gtk_text_set_point(txt, gtk_text_get_length(txt));
		gtk_text_insert(txt, NULL, NULL, NULL, buf, cp-buf);
		i = gtk_text_get_length(txt);
		if (i > 16384) {
			i -= 16384;
			gtk_text_set_point(txt, 0);
			gtk_text_forward_delete(txt, i);
			gtk_text_set_point(txt, gtk_text_get_length(txt));
		}
		gtk_text_thaw(txt);
		/* decode cooked packet if CRC ok or passall selected */
		if (!diagstate.p3d.crc ||
		    gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(gtk_object_get_data(GTK_OBJECT(receivewindow), "buttonpassall")))) {
			update_rxtext();
		}
	}
}

static gint periodictasks(gpointer user_data)
{
        int16_t samp[SPECTRUM_NUMSAMPLES];

        update_display();        
        diagstate.traceupd++;
        if (diagstate.traceupd > 1)
                diagstate.traceupd = 0;
        if (diagstate.traceupd || !(diagstate.flags & (DIAGFLG_SCOPE|DIAGFLG_SPECTRUM)) ||
	    !diagstate.audioio->read || !diagstate.audioio->curtime)
                return TRUE; /* repeat */
        diagstate.audioio->read(diagstate.audioio, samp, SPECTRUM_NUMSAMPLES, diagstate.audioio->curtime(diagstate.audioio)-SPECTRUM_NUMSAMPLES);
	if (diagstate.flags & DIAGFLG_SCOPE) {
                Scope *scope = SCOPE(gtk_object_get_data(GTK_OBJECT(scopewindow), "scope"));
                scope_setdata(scope, &samp[SPECTRUM_NUMSAMPLES-SCOPE_NUMSAMPLES]);
        }
	if (diagstate.flags & DIAGFLG_SPECTRUM) {
                Spectrum *spec = SPECTRUM(gtk_object_get_data(GTK_OBJECT(specwindow), "spec"));
                spectrum_setdata(spec, samp);
	}
	return TRUE; /* repeat */
}

/* ---------------------------------------------------------------------- */

void diag_stop(void)
{
        if (!(diagstate.flags & DIAGFLG_MODEM)) {
                diagstate.flags = 0;
                return;
        }
        gtk_timeout_remove(diagstate.timeoutid);
#ifdef TXTERMNOCANCEL
        diagstate.txterminate = 1;
        pthread_cond_broadcast(&diagstate.txcond);
#else
        pthread_cancel(diagstate.txthread);
#endif
#if 0
        pthread_cancel(diagstate.rxthread);
#else
	if (diagstate.audioio->terminateread)
		diagstate.audioio->terminateread(diagstate.audioio);
#endif
        g_printerr("Joining TxThread\n");
        pthread_join(diagstate.txthread, NULL);
        g_printerr("Joining RxThread\n");
        pthread_join(diagstate.rxthread, NULL);
        g_printerr("Releasing IO\n");
        diagstate.audioio->release(diagstate.audioio);
        if (diagstate.modch->free)
                diagstate.modch->free(diagstate.modstate);
        if (diagstate.demodch->free)
                diagstate.demodch->free(diagstate.demodstate);
        pttsetptt(&diagstate.pttio, 0);
        pttrelease(&diagstate.pttio);
        diagstate.flags = 0;
	stp_close();
}

#define MAX_PAR 10

int diag_start(unsigned int filein)
{
	const struct modemparams *par;
        char params[MAX_PAR][64];
        const char *parptr[MAX_PAR];
	unsigned int i;

        if (diagstate.flags & DIAGFLG_MODEM)
                return 0;
	diagstate.filein = filein;
        pthread_mutex_init(&diagstate.txmutex, NULL);
        pthread_cond_init(&diagstate.txcond, NULL);
#ifdef TXTERMNOCANCEL
        diagstate.txterminate = 0;
#endif
        g_print("Modulator: %s Demodulator: %s\n", diagstate.modch->name, diagstate.demodch->name);
        /* prepare modulator/demodulator and find minimum sampling rate */
        diagstate.samplerate = 5000;
        if (diagstate.modch->params && diagstate.modch->config) {
                memset(parptr, 0, sizeof(parptr));
                par = diagstate.modch->params;
                for (i = 0; i < MAX_PAR && par->name; i++, par++) {
                        if (xml_getprop("mod", par->name, params[i], sizeof(params[i])) > 0)
                                parptr[i] = params[i];
                        g_print("Modulator: parameter %s value %s\n", par->name, parptr[i] ? : "(null)");
                }
                i = diagstate.samplerate;
                diagstate.modstate = diagstate.modch->config(NULL, &i, parptr);
                if (i > diagstate.samplerate)
                        diagstate.samplerate = i;
        }
        if (diagstate.demodch->params && diagstate.demodch->config) {
                memset(parptr, 0, sizeof(parptr));
                par = diagstate.demodch->params;
                for (i = 0; i < MAX_PAR && par->name; i++, par++) {
                        if (xml_getprop("demod", par->name, params[i], sizeof(params[i])) > 0)
                                parptr[i] = params[i];
                        g_print("Demodulator: parameter %s value %s\n", par->name, parptr[i] ? : "(null)");
                }
                i = diagstate.samplerate;
                diagstate.demodstate = diagstate.demodch->config(NULL, &i, parptr);
                if (i > diagstate.samplerate)
                        diagstate.samplerate = i;
        }
        g_print("Minimum sampling rate: %u\n", diagstate.samplerate);
        /* start Audio */
        memset(parptr, 0, sizeof(parptr));
        par = filein ? ioparams_filein : ioparams_soundcard;
        for (i = 0; i < 10 && par->name; i++, par++)
                if (xml_getprop("audio", par->name, params[i], sizeof(params[i])) > 0)
                        parptr[i] = params[i];
	if (filein)
		diagstate.audioio = ioopen_filein(&diagstate.samplerate, IO_RDONLY, parptr);
	else
		diagstate.audioio = ioopen_soundcard(&diagstate.samplerate, diagstate.modch->modulate ? IO_RDWR : IO_RDONLY, parptr);
        if (!diagstate.audioio) {
                if (diagstate.modch->free)
                        diagstate.modch->free(diagstate.modstate);
                if (diagstate.demodch->free)
                        diagstate.demodch->free(diagstate.demodstate);
                error_dialog("Cannot start audio IO\n");
                return -1;
        }
	/* start stp */
        memset(parptr, 0, sizeof(parptr));
        par = stpparams;
        for (i = 0; i < 10 && par->name; i++, par++)
                if (xml_getprop("stp", par->name, params[i], sizeof(params[i])) > 0)
                        parptr[i] = params[i];        
	/* if (!filein) */
	if (stp_init(parptr))
		g_printerr("STP output disabled\n");
        /* start modems */
        g_print("Real sampling rate: %u\n", diagstate.samplerate);
        if (diagstate.modch->init)
                diagstate.modch->init(diagstate.modstate, diagstate.samplerate);
        if (diagstate.demodch->init)
                diagstate.demodch->init(diagstate.demodstate, diagstate.samplerate, &diagstate.bitrate);
        /* start PTT */
        memset(parptr, 0, sizeof(parptr));
        par = pttparams;
        for (i = 0; i < 10 && par->name; i++, par++)
                if (xml_getprop("ptt", par->name, params[i], sizeof(params[i])) > 0)
                        parptr[i] = params[i];
        if (pttinit(&diagstate.pttio, parptr))
                g_printerr("cannot start PTT output\n");
        /* periodic start */
        diagstate.timeoutid = gtk_timeout_add(100, periodictasks, NULL);
        diagstate.flags |= DIAGFLG_MODEM;
        if (pthread_create(&diagstate.rxthread, NULL, receiver, NULL))
                logerr(MLOG_FATAL, "pthread_create");
        if (pthread_create(&diagstate.txthread, NULL, transmitter, NULL))
                logerr(MLOG_FATAL, "pthread_create");        
        return 0;
}

/* ---------------------------------------------------------------------- */

void on_diagscope_activate(GtkMenuItem *menuitem, gpointer user_data)
{
	if (diag_start(diagstate.filein))
                return;
        diagstate.flags |= DIAGFLG_SCOPE;
	gtk_widget_show(scopewindow);
}

void on_diagspectrum_activate(GtkMenuItem *menuitem, gpointer user_data)
{
	if (diag_start(diagstate.filein))
		return;
        diagstate.flags |= DIAGFLG_SPECTRUM;
	gtk_widget_show(specwindow);
}

/* ---------------------------------------------------------------------- */

gboolean on_spec_motion_event(GtkWidget *widget, GdkEventMotion *event, gpointer user_data)
{
        Spectrum *spec;
        GtkEntry *entry;
        char buf[16];

        snprintf(buf, sizeof(buf), "%d Hz", (int)(event->x * diagstate.samplerate * (1.0 / SPECTRUM_NUMSAMPLES)));
        entry = GTK_ENTRY(gtk_object_get_data(GTK_OBJECT(specwindow), "specfreqpointer"));
        gtk_entry_set_text(entry, buf);
        spec = SPECTRUM(gtk_object_get_data(GTK_OBJECT(specwindow), "spec"));
        spectrum_setmarker(spec, event->x);
	return FALSE;
}

gboolean on_specwindow_delete_event(GtkWidget *widget, GdkEvent *event, gpointer user_data)
{
        diagstate.flags &= ~DIAGFLG_SPECTRUM;
	gtk_widget_hide(specwindow);
	return TRUE;
}

gboolean on_scopewindow_delete_event(GtkWidget *widget, GdkEvent *event, gpointer user_data)
{
        diagstate.flags &= ~DIAGFLG_SCOPE;
	gtk_widget_hide(scopewindow);
	return TRUE;
}
