/*
 * Constants for DV codec
 * Copyright (c) 2002 Fabrice Bellard.
 *
 * This file is part of FFmpeg.
 *
 * FFmpeg is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * FFmpeg is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with FFmpeg; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA
 */

/**
 * @file dvdata.h
 * Constants for DV codec.
 */

#ifndef FFMPEG_DVDATA_H
#define FFMPEG_DVDATA_H

#include "avcodec.h"
#include "rational.h"

/*
 * DVprofile is used to express the differences between various
 * DV flavors. For now it's primarily used for differentiating
 * 525/60 and 625/50, but the plans are to use it for various
 * DV specs as well (e.g. SMPTE314M vs. IEC 61834).
 */
typedef struct DVprofile {
    int              dsf;                 /* value of the dsf in the DV header */
    int              frame_size;          /* total size of one frame in bytes */
    int              difseg_size;         /* number of DIF segments per DIF channel */
    int              n_difchan;           /* number of DIF channels per frame */
    int              frame_rate;
    int              frame_rate_base;
    int              ltc_divisor;         /* FPS from the LTS standpoint */
    int              height;              /* picture height in pixels */
    int              width;               /* picture width in pixels */
    AVRational       sar[2];              /* sample aspect ratios for 4:3 and 16:9 */
    const uint16_t  *video_place;         /* positions of all DV macro blocks */
    enum PixelFormat pix_fmt;             /* picture pixel format */

    int              audio_stride;        /* size of audio_shuffle table */
    int              audio_min_samples[3];/* min ammount of audio samples */
                                          /* for 48Khz, 44.1Khz and 32Khz */
    int              audio_samples_dist[5];/* how many samples are supposed to be */
                                         /* in each frame in a 5 frames window */
    const uint8_t  (*audio_shuffle)[9];  /* PCM shuffling table */
} DVprofile;

#define NB_DV_VLC 409

/*
 * There's a catch about the following three tables: the mapping they establish
 * between (run, level) and vlc is not 1-1. So you have to watch out for that
 * when building misc. tables. E.g. (1, 0) can be either 0x7cf or 0x1f82.
 */
static const uint16_t dv_vlc_bits[409] = {
 0x0000, 0x0002, 0x0007, 0x0008, 0x0009, 0x0014, 0x0015, 0x0016,
 0x0017, 0x0030, 0x0031, 0x0032, 0x0033, 0x0068, 0x0069, 0x006a,
 0x006b, 0x006c, 0x006d, 0x006e, 0x006f, 0x00e0, 0x00e1, 0x00e2,
 0x00e3, 0x00e4, 0x00e5, 0x00e6, 0x00e7, 0x00e8, 0x00e9, 0x00ea,
 0x00eb, 0x00ec, 0x00ed, 0x00ee, 0x00ef, 0x01e0, 0x01e1, 0x01e2,
 0x01e3, 0x01e4, 0x01e5, 0x01e6, 0x01e7, 0x01e8, 0x01e9, 0x01ea,
 0x01eb, 0x01ec, 0x01ed, 0x01ee, 0x01ef, 0x03e0, 0x03e1, 0x03e2,
 0x03e3, 0x03e4, 0x03e5, 0x03e6, 0x07ce, 0x07cf, 0x07d0, 0x07d1,
 0x07d2, 0x07d3, 0x07d4, 0x07d5, 0x0fac, 0x0fad, 0x0fae, 0x0faf,
 0x0fb0, 0x0fb1, 0x0fb2, 0x0fb3, 0x0fb4, 0x0fb5, 0x0fb6, 0x0fb7,
 0x0fb8, 0x0fb9, 0x0fba, 0x0fbb, 0x0fbc, 0x0fbd, 0x0fbe, 0x0fbf,
 0x1f80, 0x1f81, 0x1f82, 0x1f83, 0x1f84, 0x1f85, 0x1f86, 0x1f87,
 0x1f88, 0x1f89, 0x1f8a, 0x1f8b, 0x1f8c, 0x1f8d, 0x1f8e, 0x1f8f,
 0x1f90, 0x1f91, 0x1f92, 0x1f93, 0x1f94, 0x1f95, 0x1f96, 0x1f97,
 0x1f98, 0x1f99, 0x1f9a, 0x1f9b, 0x1f9c, 0x1f9d, 0x1f9e, 0x1f9f,
 0x1fa0, 0x1fa1, 0x1fa2, 0x1fa3, 0x1fa4, 0x1fa5, 0x1fa6, 0x1fa7,
 0x1fa8, 0x1fa9, 0x1faa, 0x1fab, 0x1fac, 0x1fad, 0x1fae, 0x1faf,
 0x1fb0, 0x1fb1, 0x1fb2, 0x1fb3, 0x1fb4, 0x1fb5, 0x1fb6, 0x1fb7,
 0x1fb8, 0x1fb9, 0x1fba, 0x1fbb, 0x1fbc, 0x1fbd, 0x1fbe, 0x1fbf,
 0x7f00, 0x7f01, 0x7f02, 0x7f03, 0x7f04, 0x7f05, 0x7f06, 0x7f07,
 0x7f08, 0x7f09, 0x7f0a, 0x7f0b, 0x7f0c, 0x7f0d, 0x7f0e, 0x7f0f,
 0x7f10, 0x7f11, 0x7f12, 0x7f13, 0x7f14, 0x7f15, 0x7f16, 0x7f17,
 0x7f18, 0x7f19, 0x7f1a, 0x7f1b, 0x7f1c, 0x7f1d, 0x7f1e, 0x7f1f,
 0x7f20, 0x7f21, 0x7f22, 0x7f23, 0x7f24, 0x7f25, 0x7f26, 0x7f27,
 0x7f28, 0x7f29, 0x7f2a, 0x7f2b, 0x7f2c, 0x7f2d, 0x7f2e, 0x7f2f,
 0x7f30, 0x7f31, 0x7f32, 0x7f33, 0x7f34, 0x7f35, 0x7f36, 0x7f37,
 0x7f38, 0x7f39, 0x7f3a, 0x7f3b, 0x7f3c, 0x7f3d, 0x7f3e, 0x7f3f,
 0x7f40, 0x7f41, 0x7f42, 0x7f43, 0x7f44, 0x7f45, 0x7f46, 0x7f47,
 0x7f48, 0x7f49, 0x7f4a, 0x7f4b, 0x7f4c, 0x7f4d, 0x7f4e, 0x7f4f,
 0x7f50, 0x7f51, 0x7f52, 0x7f53, 0x7f54, 0x7f55, 0x7f56, 0x7f57,
 0x7f58, 0x7f59, 0x7f5a, 0x7f5b, 0x7f5c, 0x7f5d, 0x7f5e, 0x7f5f,
 0x7f60, 0x7f61, 0x7f62, 0x7f63, 0x7f64, 0x7f65, 0x7f66, 0x7f67,
 0x7f68, 0x7f69, 0x7f6a, 0x7f6b, 0x7f6c, 0x7f6d, 0x7f6e, 0x7f6f,
 0x7f70, 0x7f71, 0x7f72, 0x7f73, 0x7f74, 0x7f75, 0x7f76, 0x7f77,
 0x7f78, 0x7f79, 0x7f7a, 0x7f7b, 0x7f7c, 0x7f7d, 0x7f7e, 0x7f7f,
 0x7f80, 0x7f81, 0x7f82, 0x7f83, 0x7f84, 0x7f85, 0x7f86, 0x7f87,
 0x7f88, 0x7f89, 0x7f8a, 0x7f8b, 0x7f8c, 0x7f8d, 0x7f8e, 0x7f8f,
 0x7f90, 0x7f91, 0x7f92, 0x7f93, 0x7f94, 0x7f95, 0x7f96, 0x7f97,
 0x7f98, 0x7f99, 0x7f9a, 0x7f9b, 0x7f9c, 0x7f9d, 0x7f9e, 0x7f9f,
 0x7fa0, 0x7fa1, 0x7fa2, 0x7fa3, 0x7fa4, 0x7fa5, 0x7fa6, 0x7fa7,
 0x7fa8, 0x7fa9, 0x7faa, 0x7fab, 0x7fac, 0x7fad, 0x7fae, 0x7faf,
 0x7fb0, 0x7fb1, 0x7fb2, 0x7fb3, 0x7fb4, 0x7fb5, 0x7fb6, 0x7fb7,
 0x7fb8, 0x7fb9, 0x7fba, 0x7fbb, 0x7fbc, 0x7fbd, 0x7fbe, 0x7fbf,
 0x7fc0, 0x7fc1, 0x7fc2, 0x7fc3, 0x7fc4, 0x7fc5, 0x7fc6, 0x7fc7,
 0x7fc8, 0x7fc9, 0x7fca, 0x7fcb, 0x7fcc, 0x7fcd, 0x7fce, 0x7fcf,
 0x7fd0, 0x7fd1, 0x7fd2, 0x7fd3, 0x7fd4, 0x7fd5, 0x7fd6, 0x7fd7,
 0x7fd8, 0x7fd9, 0x7fda, 0x7fdb, 0x7fdc, 0x7fdd, 0x7fde, 0x7fdf,
 0x7fe0, 0x7fe1, 0x7fe2, 0x7fe3, 0x7fe4, 0x7fe5, 0x7fe6, 0x7fe7,
 0x7fe8, 0x7fe9, 0x7fea, 0x7feb, 0x7fec, 0x7fed, 0x7fee, 0x7fef,
 0x7ff0, 0x7ff1, 0x7ff2, 0x7ff3, 0x7ff4, 0x7ff5, 0x7ff6, 0x7ff7,
 0x7ff8, 0x7ff9, 0x7ffa, 0x7ffb, 0x7ffc, 0x7ffd, 0x7ffe, 0x7fff,
 0x0006,
};

static const uint8_t dv_vlc_len[409] = {
  2,  3,  4,  4,  4,  5,  5,  5,
  5,  6,  6,  6,  6,  7,  7,  7,
  7,  7,  7,  7,  7,  8,  8,  8,
  8,  8,  8,  8,  8,  8,  8,  8,
  8,  8,  8,  8,  8,  9,  9,  9,
  9,  9,  9,  9,  9,  9,  9,  9,
  9,  9,  9,  9,  9, 10, 10, 10,
 10, 10, 10, 10, 11, 11, 11, 11,
 11, 11, 11, 11, 12, 12, 12, 12,
 12, 12, 12, 12, 12, 12, 12, 12,
 12, 12, 12, 12, 12, 12, 12, 12,
 13, 13, 13, 13, 13, 13, 13, 13,
 13, 13, 13, 13, 13, 13, 13, 13,
 13, 13, 13, 13, 13, 13, 13, 13,
 13, 13, 13, 13, 13, 13, 13, 13,
 13, 13, 13, 13, 13, 13, 13, 13,
 13, 13, 13, 13, 13, 13, 13, 13,
 13, 13, 13, 13, 13, 13, 13, 13,
 13, 13, 13, 13, 13, 13, 13, 13,
 15, 15, 15, 15, 15, 15, 15, 15,
 15, 15, 15, 15, 15, 15, 15, 15,
 15, 15, 15, 15, 15, 15, 15, 15,
 15, 15, 15, 15, 15, 15, 15, 15,
 15, 15, 15, 15, 15, 15, 15, 15,
 15, 15, 15, 15, 15, 15, 15, 15,
 15, 15, 15, 15, 15, 15, 15, 15,
 15, 15, 15, 15, 15, 15, 15, 15,
 15, 15, 15, 15, 15, 15, 15, 15,
 15, 15, 15, 15, 15, 15, 15, 15,
 15, 15, 15, 15, 15, 15, 15, 15,
 15, 15, 15, 15, 15, 15, 15, 15,
 15, 15, 15, 15, 15, 15, 15, 15,
 15, 15, 15, 15, 15, 15, 15, 15,
 15, 15, 15, 15, 15, 15, 15, 15,
 15, 15, 15, 15, 15, 15, 15, 15,
 15, 15, 15, 15, 15, 15, 15, 15,
 15, 15, 15, 15, 15, 15, 15, 15,
 15, 15, 15, 15, 15, 15, 15, 15,
 15, 15, 15, 15, 15, 15, 15, 15,
 15, 15, 15, 15, 15, 15, 15, 15,
 15, 15, 15, 15, 15, 15, 15, 15,
 15, 15, 15, 15, 15, 15, 15, 15,
 15, 15, 15, 15, 15, 15, 15, 15,
 15, 15, 15, 15, 15, 15, 15, 15,
 15, 15, 15, 15, 15, 15, 15, 15,
 15, 15, 15, 15, 15, 15, 15, 15,
 15, 15, 15, 15, 15, 15, 15, 15,
 15, 15, 15, 15, 15, 15, 15, 15,
 15, 15, 15, 15, 15, 15, 15, 15,
 15, 15, 15, 15, 15, 15, 15, 15,
 15, 15, 15, 15, 15, 15, 15, 15,
  4,
};

static const uint8_t dv_vlc_run[409] = {
  0,  0,  1,  0,  0,  2,  1,  0,
  0,  3,  4,  0,  0,  5,  6,  2,
  1,  1,  0,  0,  0,  7,  8,  9,
 10,  3,  4,  2,  1,  1,  1,  0,
  0,  0,  0,  0,  0, 11, 12, 13,
 14,  5,  6,  3,  4,  2,  2,  1,
  0,  0,  0,  0,  0,  5,  3,  3,
  2,  1,  1,  1,  0,  1,  6,  4,
  3,  1,  1,  1,  2,  3,  4,  5,
  7,  8,  9, 10,  7,  8,  4,  3,
  2,  2,  2,  2,  2,  1,  1,  1,
  0,  1,  2,  3,  4,  5,  6,  7,
  8,  9, 10, 11, 12, 13, 14, 15,
 16, 17, 18, 19, 20, 21, 22, 23,
 24, 25, 26, 27, 28, 29, 30, 31,
 32, 33, 34, 35, 36, 37, 38, 39,
 40, 41, 42, 43, 44, 45, 46, 47,
 48, 49, 50, 51, 52, 53, 54, 55,
 56, 57, 58, 59, 60, 61, 62, 63,
  0,  0,  0,  0,  0,  0,  0,  0,
  0,  0,  0,  0,  0,  0,  0,  0,
  0,  0,  0,  0,  0,  0,  0,  0,
  0,  0,  0,  0,  0,  0,  0,  0,
  0,  0,  0,  0,  0,  0,  0,  0,
  0,  0,  0,  0,  0,  0,  0,  0,
  0,  0,  0,  0,  0,  0,  0,  0,
  0,  0,  0,  0,  0,  0,  0,  0,
  0,  0,  0,  0,  0,  0,  0,  0,
  0,  0,  0,  0,  0,  0,  0,  0,
  0,  0,  0,  0,  0,  0,  0,  0,
  0,  0,  0,  0,  0,  0,  0,  0,
  0,  0,  0,  0,  0,  0,  0,  0,
  0,  0,  0,  0,  0,  0,  0,  0,
  0,  0,  0,  0,  0,  0,  0,  0,
  0,  0,  0,  0,  0,  0,  0,  0,
  0,  0,  0,  0,  0,  0,  0,  0,
  0,  0,  0,  0,  0,  0,  0,  0,
  0,  0,  0,  0,  0,  0,  0,  0,
  0,  0,  0,  0,  0,  0,  0,  0,
  0,  0,  0,  0,  0,  0,  0,  0,
  0,  0,  0,  0,  0,  0,  0,  0,
  0,  0,  0,  0,  0,  0,  0,  0,
  0,  0,  0,  0,  0,  0,  0,  0,
  0,  0,  0,  0,  0,  0,  0,  0,
  0,  0,  0,  0,  0,  0,  0,  0,
  0,  0,  0,  0,  0,  0,  0,  0,
  0,  0,  0,  0,  0,  0,  0,  0,
  0,  0,  0,  0,  0,  0,  0,  0,
  0,  0,  0,  0,  0,  0,  0,  0,
  0,  0,  0,  0,  0,  0,  0,  0,
  0,  0,  0,  0,  0,  0,  0,  0,
127,
};

static const uint8_t dv_vlc_level[409] = {
  1,  2,  1,  3,  4,  1,  2,  5,
  6,  1,  1,  7,  8,  1,  1,  2,
  3,  4,  9, 10, 11,  1,  1,  1,
  1,  2,  2,  3,  5,  6,  7, 12,
 13, 14, 15, 16, 17,  1,  1,  1,
  1,  2,  2,  3,  3,  4,  5,  8,
 18, 19, 20, 21, 22,  3,  4,  5,
  6,  9, 10, 11,  0,  0,  3,  4,
  6, 12, 13, 14,  0,  0,  0,  0,
  2,  2,  2,  2,  3,  3,  5,  7,
  7,  8,  9, 10, 11, 15, 16, 17,
  0,  0,  0,  0,  0,  0,  0,  0,
  0,  0,  0,  0,  0,  0,  0,  0,
  0,  0,  0,  0,  0,  0,  0,  0,
  0,  0,  0,  0,  0,  0,  0,  0,
  0,  0,  0,  0,  0,  0,  0,  0,
  0,  0,  0,  0,  0,  0,  0,  0,
  0,  0,  0,  0,  0,  0,  0,  0,
  0,  0,  0,  0,  0,  0,  0,  0,
  0,  1,  2,  3,  4,  5,  6,  7,
  8,  9, 10, 11, 12, 13, 14, 15,
 16, 17, 18, 19, 20, 21, 22, 23,
 24, 25, 26, 27, 28, 29, 30, 31,
 32, 33, 34, 35, 36, 37, 38, 39,
 40, 41, 42, 43, 44, 45, 46, 47,
 48, 49, 50, 51, 52, 53, 54, 55,
 56, 57, 58, 59, 60, 61, 62, 63,
 64, 65, 66, 67, 68, 69, 70, 71,
 72, 73, 74, 75, 76, 77, 78, 79,
 80, 81, 82, 83, 84, 85, 86, 87,
 88, 89, 90, 91, 92, 93, 94, 95,
 96, 97, 98, 99, 100, 101, 102, 103,
 104, 105, 106, 107, 108, 109, 110, 111,
 112, 113, 114, 115, 116, 117, 118, 119,
 120, 121, 122, 123, 124, 125, 126, 127,
 128, 129, 130, 131, 132, 133, 134, 135,
 136, 137, 138, 139, 140, 141, 142, 143,
 144, 145, 146, 147, 148, 149, 150, 151,
 152, 153, 154, 155, 156, 157, 158, 159,
 160, 161, 162, 163, 164, 165, 166, 167,
 168, 169, 170, 171, 172, 173, 174, 175,
 176, 177, 178, 179, 180, 181, 182, 183,
 184, 185, 186, 187, 188, 189, 190, 191,
 192, 193, 194, 195, 196, 197, 198, 199,
 200, 201, 202, 203, 204, 205, 206, 207,
 208, 209, 210, 211, 212, 213, 214, 215,
 216, 217, 218, 219, 220, 221, 222, 223,
 224, 225, 226, 227, 228, 229, 230, 231,
 232, 233, 234, 235, 236, 237, 238, 239,
 240, 241, 242, 243, 244, 245, 246, 247,
 248, 249, 250, 251, 252, 253, 254, 255,
  0,
};

/* unquant tables (not used directly) */
static const uint8_t dv_88_areas[64] = {
    0,0,0,1,1,1,2,2,
    0,0,1,1,1,2,2,2,
    0,1,1,1,2,2,2,3,
    1,1,1,2,2,2,3,3,
    1,1,2,2,2,3,3,3,
    1,2,2,2,3,3,3,3,
    2,2,2,3,3,3,3,3,
    2,2,3,3,3,3,3,3,
};

static const uint8_t dv_248_areas[64] = {
    0,0,1,1,1,2,2,3,
    0,0,1,1,2,2,2,3,
    0,1,1,2,2,2,3,3,
    0,1,1,2,2,2,3,3,
    1,1,2,2,2,3,3,3,
    1,1,2,2,2,3,3,3,
    1,2,2,2,3,3,3,3,
    1,2,2,3,3,3,3,3,
};

static const uint8_t dv_quant_shifts[22][4] = {
  { 3,3,4,4 },
  { 3,3,4,4 },
  { 2,3,3,4 },
  { 2,3,3,4 },
  { 2,2,3,3 },
  { 2,2,3,3 },
  { 1,2,2,3 },
  { 1,2,2,3 },
  { 1,1,2,2 },
  { 1,1,2,2 },
  { 0,1,1,2 },
  { 0,1,1,2 },
  { 0,0,1,1 },
  { 0,0,1,1 },
  { 0,0,0,1 },
  { 0,0,0,0 },
  { 0,0,0,0 },
  { 0,0,0,0 },
  { 0,0,0,0 },
  { 0,0,0,0 },
  { 0,0,0,0 },
  { 0,0,0,0 },
};

static const uint8_t dv_quant_offset[4] = { 6, 3, 0, 1 };

/* NOTE: I prefer hardcoding the positioning of dv blocks, it is
   simpler :-) */

static const uint16_t dv_place_420[1620] = {
 0x0c24, 0x2412, 0x3036, 0x0000, 0x1848,
 0x0e24, 0x2612, 0x3236, 0x0200, 0x1a48,
 0x1024, 0x2812, 0x3436, 0x0400, 0x1c48,
 0x1026, 0x2814, 0x3438, 0x0402, 0x1c4a,
 0x0e26, 0x2614, 0x3238, 0x0202, 0x1a4a,
 0x0c26, 0x2414, 0x3038, 0x0002, 0x184a,
 0x0c28, 0x2416, 0x303a, 0x0004, 0x184c,
 0x0e28, 0x2616, 0x323a, 0x0204, 0x1a4c,
 0x1028, 0x2816, 0x343a, 0x0404, 0x1c4c,
 0x102a, 0x2818, 0x343c, 0x0406, 0x1c4e,
 0x0e2a, 0x2618, 0x323c, 0x0206, 0x1a4e,
 0x0c2a, 0x2418, 0x303c, 0x0006, 0x184e,
 0x0c2c, 0x241a, 0x303e, 0x0008, 0x1850,
 0x0e2c, 0x261a, 0x323e, 0x0208, 0x1a50,
 0x102c, 0x281a, 0x343e, 0x0408, 0x1c50,
 0x102e, 0x281c, 0x3440, 0x040a, 0x1c52,
 0x0e2e, 0x261c, 0x3240, 0x020a, 0x1a52,
 0x0c2e, 0x241c, 0x3040, 0x000a, 0x1852,
 0x0c30, 0x241e, 0x3042, 0x000c, 0x1854,
 0x0e30, 0x261e, 0x3242, 0x020c, 0x1a54,
 0x1030, 0x281e, 0x3442, 0x040c, 0x1c54,
 0x1032, 0x2820, 0x3444, 0x040e, 0x1c56,
 0x0e32, 0x2620, 0x3244, 0x020e, 0x1a56,
 0x0c32, 0x2420, 0x3044, 0x000e, 0x1856,
 0x0c34, 0x2422, 0x3046, 0x0010, 0x1858,
 0x0e34, 0x2622, 0x3246, 0x0210, 0x1a58,
 0x1034, 0x2822, 0x3446, 0x0410, 0x1c58,
 0x1224, 0x2a12, 0x3636, 0x0600, 0x1e48,
 0x1424, 0x2c12, 0x3836, 0x0800, 0x2048,
 0x1624, 0x2e12, 0x3a36, 0x0a00, 0x2248,
 0x1626, 0x2e14, 0x3a38, 0x0a02, 0x224a,
 0x1426, 0x2c14, 0x3838, 0x0802, 0x204a,
 0x1226, 0x2a14, 0x3638, 0x0602, 0x1e4a,
 0x1228, 0x2a16, 0x363a, 0x0604, 0x1e4c,
 0x1428, 0x2c16, 0x383a, 0x0804, 0x204c,
 0x1628, 0x2e16, 0x3a3a, 0x0a04, 0x224c,
 0x162a, 0x2e18, 0x3a3c, 0x0a06, 0x224e,
 0x142a, 0x2c18, 0x383c, 0x0806, 0x204e,
 0x122a, 0x2a18, 0x363c, 0x0606, 0x1e4e,
 0x122c, 0x2a1a, 0x363e, 0x0608, 0x1e50,
 0x142c, 0x2c1a, 0x383e, 0x0808, 0x2050,
 0x162c, 0x2e1a, 0x3a3e, 0x0a08, 0x2250,
 0x162e, 0x2e1c, 0x3a40, 0x0a0a, 0x2252,
 0x142e, 0x2c1c, 0x3840, 0x080a, 0x2052,
 0x122e, 0x2a1c, 0x3640, 0x060a, 0x1e52,
 0x1230, 0x2a1e, 0x3642, 0x060c, 0x1e54,
 0x1430, 0x2c1e, 0x3842, 0x080c, 0x2054,
 0x1630, 0x2e1e, 0x3a42, 0x0a0c, 0x2254,
 0x1632, 0x2e20, 0x3a44, 0x0a0e, 0x2256,
 0x1432, 0x2c20, 0x3844, 0x080e, 0x2056,
 0x1232, 0x2a20, 0x3644, 0x060e, 0x1e56,
 0x1234, 0x2a22, 0x3646, 0x0610, 0x1e58,
 0x1434, 0x2c22, 0x3846, 0x0810, 0x2058,
 0x1634, 0x2e22, 0x3a46, 0x0a10, 0x2258,
 0x1824, 0x3012, 0x3c36, 0x0c00, 0x2448,
 0x1a24, 0x3212, 0x3e36, 0x0e00, 0x2648,
 0x1c24, 0x3412, 0x4036, 0x1000, 0x2848,
 0x1c26, 0x3414, 0x4038, 0x1002, 0x284a,
 0x1a26, 0x3214, 0x3e38, 0x0e02, 0x264a,
 0x1826, 0x3014, 0x3c38, 0x0c02, 0x244a,
 0x1828, 0x3016, 0x3c3a, 0x0c04, 0x244c,
 0x1a28, 0x3216, 0x3e3a, 0x0e04, 0x264c,
 0x1c28, 0x3416, 0x403a, 0x1004, 0x284c,
 0x1c2a, 0x3418, 0x403c, 0x1006, 0x284e,
 0x1a2a, 0x3218, 0x3e3c, 0x0e06, 0x264e,
 0x182a, 0x3018, 0x3c3c, 0x0c06, 0x244e,
 0x182c, 0x301a, 0x3c3e, 0x0c08, 0x2450,
 0x1a2c, 0x321a, 0x3e3e, 0x0e08, 0x2650,
 0x1c2c, 0x341a, 0x403e, 0x1008, 0x2850,
 0x1c2e, 0x341c, 0x4040, 0x100a, 0x2852,
 0x1a2e, 0x321c, 0x3e40, 0x0e0a, 0x2652,
 0x182e, 0x301c, 0x3c40, 0x0c0a, 0x2452,
 0x1830, 0x301e, 0x3c42, 0x0c0c, 0x2454,
 0x1a30, 0x321e, 0x3e42, 0x0e0c, 0x2654,
 0x1c30, 0x341e, 0x4042, 0x100c, 0x2854,
 0x1c32, 0x3420, 0x4044, 0x100e, 0x2856,
 0x1a32, 0x3220, 0x3e44, 0x0e0e, 0x2656,
 0x1832, 0x3020, 0x3c44, 0x0c0e, 0x2456,
 0x1834, 0x3022, 0x3c46, 0x0c10, 0x2458,
 0x1a34, 0x3222, 0x3e46, 0x0e10, 0x2658,
 0x1c34, 0x3422, 0x4046, 0x1010, 0x2858,
 0x1e24, 0x3612, 0x4236, 0x1200, 0x2a48,
 0x2024, 0x3812, 0x4436, 0x1400, 0x2c48,
 0x2224, 0x3a12, 0x4636, 0x1600, 0x2e48,
 0x2226, 0x3a14, 0x4638, 0x1602, 0x2e4a,
 0x2026, 0x3814, 0x4438, 0x1402, 0x2c4a,
 0x1e26, 0x3614, 0x4238, 0x1202, 0x2a4a,
 0x1e28, 0x3616, 0x423a, 0x1204, 0x2a4c,
 0x2028, 0x3816, 0x443a, 0x1404, 0x2c4c,
 0x2228, 0x3a16, 0x463a, 0x1604, 0x2e4c,
 0x222a, 0x3a18, 0x463c, 0x1606, 0x2e4e,
 0x202a, 0x3818, 0x443c, 0x1406, 0x2c4e,
 0x1e2a, 0x3618, 0x423c, 0x1206, 0x2a4e,
 0x1e2c, 0x361a, 0x423e, 0x1208, 0x2a50,
 0x202c, 0x381a, 0x443e, 0x1408, 0x2c50,
 0x222c, 0x3a1a, 0x463e, 0x1608, 0x2e50,
 0x222e, 0x3a1c, 0x4640, 0x160a, 0x2e52,
 0x202e, 0x381c, 0x4440, 0x140a, 0x2c52,
 0x1e2e, 0x361c, 0x4240, 0x120a, 0x2a52,
 0x1e30, 0x361e, 0x4242, 0x120c, 0x2a54,
 0x2030, 0x381e, 0x4442, 0x140c, 0x2c54,
 0x2230, 0x3a1e, 0x4642, 0x160c, 0x2e54,
 0x2232, 0x3a20, 0x4644, 0x160e, 0x2e56,
 0x2032, 0x3820, 0x4444, 0x140e, 0x2c56,
 0x1e32, 0x3620, 0x4244, 0x120e, 0x2a56,
 0x1e34, 0x3622, 0x4246, 0x1210, 0x2a58,
 0x2034, 0x3822, 0x4446, 0x1410, 0x2c58,
 0x2234, 0x3a22, 0x4646, 0x1610, 0x2e58,
 0x2424, 0x3c12, 0x0036, 0x1800, 0x3048,
 0x2624, 0x3e12, 0x0236, 0x1a00, 0x3248,
 0x2824, 0x4012, 0x0436, 0x1c00, 0x3448,
 0x2826, 0x4014, 0x0438, 0x1c02, 0x344a,
 0x2626, 0x3e14, 0x0238, 0x1a02, 0x324a,
 0x2426, 0x3c14, 0x0038, 0x1802, 0x304a,
 0x2428, 0x3c16, 0x003a, 0x1804, 0x304c,
 0x2628, 0x3e16, 0x023a, 0x1a04, 0x324c,
 0x2828, 0x4016, 0x043a, 0x1c04, 0x344c,
 0x282a, 0x4018, 0x043c, 0x1c06, 0x344e,
 0x262a, 0x3e18, 0x023c, 0x1a06, 0x324e,
 0x242a, 0x3c18, 0x003c, 0x1806, 0x304e,
 0x242c, 0x3c1a, 0x003e, 0x1808, 0x3050,
 0x262c, 0x3e1a, 0x023e, 0x1a08, 0x3250,
 0x282c, 0x401a, 0x043e, 0x1c08, 0x3450,
 0x282e, 0x401c, 0x0440, 0x1c0a, 0x3452,
 0x262e, 0x3e1c, 0x0240, 0x1a0a, 0x3252,
 0x242e, 0x3c1c, 0x0040, 0x180a, 0x3052,
 0x2430, 0x3c1e, 0x0042, 0x180c, 0x3054,
 0x2630, 0x3e1e, 0x0242, 0x1a0c, 0x3254,
 0x2830, 0x401e, 0x0442, 0x1c0c, 0x3454,
 0x2832, 0x4020, 0x0444, 0x1c0e, 0x3456,
 0x2632, 0x3e20, 0x0244, 0x1a0e, 0x3256,
 0x2432, 0x3c20, 0x0044, 0x180e, 0x3056,
 0x2434, 0x3c22, 0x0046, 0x1810, 0x3058,
 0x2634, 0x3e22, 0x0246, 0x1a10, 0x3258,
 0x2834, 0x4022, 0x0446, 0x1c10, 0x3458,
 0x2a24, 0x4212, 0x0636, 0x1e00, 0x3648,
 0x2c24, 0x4412, 0x0836, 0x2000, 0x3848,
 0x2e24, 0x4612, 0x0a36, 0x2200, 0x3a48,
 0x2e26, 0x4614, 0x0a38, 0x2202, 0x3a4a,
 0x2c26, 0x4414, 0x0838, 0x2002, 0x384a,
 0x2a26, 0x4214, 0x0638, 0x1e02, 0x364a,
 0x2a28, 0x4216, 0x063a, 0x1e04, 0x364c,
 0x2c28, 0x4416, 0x083a, 0x2004, 0x384c,
 0x2e28, 0x4616, 0x0a3a, 0x2204, 0x3a4c,
 0x2e2a, 0x4618, 0x0a3c, 0x2206, 0x3a4e,
 0x2c2a, 0x4418, 0x083c, 0x2006, 0x384e,
 0x2a2a, 0x4218, 0x063c, 0x1e06, 0x364e,
 0x2a2c, 0x421a, 0x063e, 0x1e08, 0x3650,
 0x2c2c, 0x441a, 0x083e, 0x2008, 0x3850,
 0x2e2c, 0x461a, 0x0a3e, 0x2208, 0x3a50,
 0x2e2e, 0x461c, 0x0a40, 0x220a, 0x3a52,
 0x2c2e, 0x441c, 0x0840, 0x200a, 0x3852,
 0x2a2e, 0x421c, 0x0640, 0x1e0a, 0x3652,
 0x2a30, 0x421e, 0x0642, 0x1e0c, 0x3654,
 0x2c30, 0x441e, 0x0842, 0x200c, 0x3854,
 0x2e30, 0x461e, 0x0a42, 0x220c, 0x3a54,
 0x2e32, 0x4620, 0x0a44, 0x220e, 0x3a56,
 0x2c32, 0x4420, 0x0844, 0x200e, 0x3856,
 0x2a32, 0x4220, 0x0644, 0x1e0e, 0x3656,
 0x2a34, 0x4222, 0x0646, 0x1e10, 0x3658,
 0x2c34, 0x4422, 0x0846, 0x2010, 0x3858,
 0x2e34, 0x4622, 0x0a46, 0x2210, 0x3a58,
 0x3024, 0x0012, 0x0c36, 0x2400, 0x3c48,
 0x3224, 0x0212, 0x0e36, 0x2600, 0x3e48,
 0x3424, 0x0412, 0x1036, 0x2800, 0x4048,
 0x3426, 0x0414, 0x1038, 0x2802, 0x404a,
 0x3226, 0x0214, 0x0e38, 0x2602, 0x3e4a,
 0x3026, 0x0014, 0x0c38, 0x2402, 0x3c4a,
 0x3028, 0x0016, 0x0c3a, 0x2404, 0x3c4c,
 0x3228, 0x0216, 0x0e3a, 0x2604, 0x3e4c,
 0x3428, 0x0416, 0x103a, 0x2804, 0x404c,
 0x342a, 0x0418, 0x103c, 0x2806, 0x404e,
 0x322a, 0x0218, 0x0e3c, 0x2606, 0x3e4e,
 0x302a, 0x0018, 0x0c3c, 0x2406, 0x3c4e,
 0x302c, 0x001a, 0x0c3e, 0x2408, 0x3c50,
 0x322c, 0x021a, 0x0e3e, 0x2608, 0x3e50,
 0x342c, 0x041a, 0x103e, 0x2808, 0x4050,
 0x342e, 0x041c, 0x1040, 0x280a, 0x4052,
 0x322e, 0x021c, 0x0e40, 0x260a, 0x3e52,
 0x302e, 0x001c, 0x0c40, 0x240a, 0x3c52,
 0x3030, 0x001e, 0x0c42, 0x240c, 0x3c54,
 0x3230, 0x021e, 0x0e42, 0x260c, 0x3e54,
 0x3430, 0x041e, 0x1042, 0x280c, 0x4054,
 0x3432, 0x0420, 0x1044, 0x280e, 0x4056,
 0x3232, 0x0220, 0x0e44, 0x260e, 0x3e56,
 0x3032, 0x0020, 0x0c44, 0x240e, 0x3c56,
 0x3034, 0x0022, 0x0c46, 0x2410, 0x3c58,
 0x3234, 0x0222, 0x0e46, 0x2610, 0x3e58,
 0x3434, 0x0422, 0x1046, 0x2810, 0x4058,
 0x3624, 0x0612, 0x1236, 0x2a00, 0x4248,
 0x3824, 0x0812, 0x1436, 0x2c00, 0x4448,
 0x3a24, 0x0a12, 0x1636, 0x2e00, 0x4648,
 0x3a26, 0x0a14, 0x1638, 0x2e02, 0x464a,
 0x3826, 0x0814, 0x1438, 0x2c02, 0x444a,
 0x3626, 0x0614, 0x1238, 0x2a02, 0x424a,
 0x3628, 0x0616, 0x123a, 0x2a04, 0x424c,
 0x3828, 0x0816, 0x143a, 0x2c04, 0x444c,
 0x3a28, 0x0a16, 0x163a, 0x2e04, 0x464c,
 0x3a2a, 0x0a18, 0x163c, 0x2e06, 0x464e,
 0x382a, 0x0818, 0x143c, 0x2c06, 0x444e,
 0x362a, 0x0618, 0x123c, 0x2a06, 0x424e,
 0x362c, 0x061a, 0x123e, 0x2a08, 0x4250,
 0x382c, 0x081a, 0x143e, 0x2c08, 0x4450,
 0x3a2c, 0x0a1a, 0x163e, 0x2e08, 0x4650,
 0x3a2e, 0x0a1c, 0x1640, 0x2e0a, 0x4652,
 0x382e, 0x081c, 0x1440, 0x2c0a, 0x4452,
 0x362e, 0x061c, 0x1240, 0x2a0a, 0x4252,
 0x3630, 0x061e, 0x1242, 0x2a0c, 0x4254,
 0x3830, 0x081e, 0x1442, 0x2c0c, 0x4454,
 0x3a30, 0x0a1e, 0x1642, 0x2e0c, 0x4654,
 0x3a32, 0x0a20, 0x1644, 0x2e0e, 0x4656,
 0x3832, 0x0820, 0x1444, 0x2c0e, 0x4456,
 0x3632, 0x0620, 0x1244, 0x2a0e, 0x4256,
 0x3634, 0x0622, 0x1246, 0x2a10, 0x4258,
 0x3834, 0x0822, 0x1446, 0x2c10, 0x4458,
 0x3a34, 0x0a22, 0x1646, 0x2e10, 0x4658,
 0x3c24, 0x0c12, 0x1836, 0x3000, 0x0048,
 0x3e24, 0x0e12, 0x1a36, 0x3200, 0x0248,
 0x4024, 0x1012, 0x1c36, 0x3400, 0x0448,
 0x4026, 0x1014, 0x1c38, 0x3402, 0x044a,
 0x3e26, 0x0e14, 0x1a38, 0x3202, 0x024a,
 0x3c26, 0x0c14, 0x1838, 0x3002, 0x004a,
 0x3c28, 0x0c16, 0x183a, 0x3004, 0x004c,
 0x3e28, 0x0e16, 0x1a3a, 0x3204, 0x024c,
 0x4028, 0x1016, 0x1c3a, 0x3404, 0x044c,
 0x402a, 0x1018, 0x1c3c, 0x3406, 0x044e,
 0x3e2a, 0x0e18, 0x1a3c, 0x3206, 0x024e,
 0x3c2a, 0x0c18, 0x183c, 0x3006, 0x004e,
 0x3c2c, 0x0c1a, 0x183e, 0x3008, 0x0050,
 0x3e2c, 0x0e1a, 0x1a3e, 0x3208, 0x0250,
 0x402c, 0x101a, 0x1c3e, 0x3408, 0x0450,
 0x402e, 0x101c, 0x1c40, 0x340a, 0x0452,
 0x3e2e, 0x0e1c, 0x1a40, 0x320a, 0x0252,
 0x3c2e, 0x0c1c, 0x1840, 0x300a, 0x0052,
 0x3c30, 0x0c1e, 0x1842, 0x300c, 0x0054,
 0x3e30, 0x0e1e, 0x1a42, 0x320c, 0x0254,
 0x4030, 0x101e, 0x1c42, 0x340c, 0x0454,
 0x4032, 0x1020, 0x1c44, 0x340e, 0x0456,
 0x3e32, 0x0e20, 0x1a44, 0x320e, 0x0256,
 0x3c32, 0x0c20, 0x1844, 0x300e, 0x0056,
 0x3c34, 0x0c22, 0x1846, 0x3010, 0x0058,
 0x3e34, 0x0e22, 0x1a46, 0x3210, 0x0258,
 0x4034, 0x1022, 0x1c46, 0x3410, 0x0458,
 0x4224, 0x1212, 0x1e36, 0x3600, 0x0648,
 0x4424, 0x1412, 0x2036, 0x3800, 0x0848,
 0x4624, 0x1612, 0x2236, 0x3a00, 0x0a48,
 0x4626, 0x1614, 0x2238, 0x3a02, 0x0a4a,
 0x4426, 0x1414, 0x2038, 0x3802, 0x084a,
 0x4226, 0x1214, 0x1e38, 0x3602, 0x064a,
 0x4228, 0x1216, 0x1e3a, 0x3604, 0x064c,
 0x4428, 0x1416, 0x203a, 0x3804, 0x084c,
 0x4628, 0x1616, 0x223a, 0x3a04, 0x0a4c,
 0x462a, 0x1618, 0x223c, 0x3a06, 0x0a4e,
 0x442a, 0x1418, 0x203c, 0x3806, 0x084e,
 0x422a, 0x1218, 0x1e3c, 0x3606, 0x064e,
 0x422c, 0x121a, 0x1e3e, 0x3608, 0x0650,
 0x442c, 0x141a, 0x203e, 0x3808, 0x0850,
 0x462c, 0x161a, 0x223e, 0x3a08, 0x0a50,
 0x462e, 0x161c, 0x2240, 0x3a0a, 0x0a52,
 0x442e, 0x141c, 0x2040, 0x380a, 0x0852,
 0x422e, 0x121c, 0x1e40, 0x360a, 0x0652,
 0x4230, 0x121e, 0x1e42, 0x360c, 0x0654,
 0x4430, 0x141e, 0x2042, 0x380c, 0x0854,
 0x4630, 0x161e, 0x2242, 0x3a0c, 0x0a54,
 0x4632, 0x1620, 0x2244, 0x3a0e, 0x0a56,
 0x4432, 0x1420, 0x2044, 0x380e, 0x0856,
 0x4232, 0x1220, 0x1e44, 0x360e, 0x0656,
 0x4234, 0x1222, 0x1e46, 0x3610, 0x0658,
 0x4434, 0x1422, 0x2046, 0x3810, 0x0858,
 0x4634, 0x1622, 0x2246, 0x3a10, 0x0a58,
 0x0024, 0x1812, 0x2436, 0x3c00, 0x0c48,
 0x0224, 0x1a12, 0x2636, 0x3e00, 0x0e48,
 0x0424, 0x1c12, 0x2836, 0x4000, 0x1048,
 0x0426, 0x1c14, 0x2838, 0x4002, 0x104a,
 0x0226, 0x1a14, 0x2638, 0x3e02, 0x0e4a,
 0x0026, 0x1814, 0x2438, 0x3c02, 0x0c4a,
 0x0028, 0x1816, 0x243a, 0x3c04, 0x0c4c,
 0x0228, 0x1a16, 0x263a, 0x3e04, 0x0e4c,
 0x0428, 0x1c16, 0x283a, 0x4004, 0x104c,
 0x042a, 0x1c18, 0x283c, 0x4006, 0x104e,
 0x022a, 0x1a18, 0x263c, 0x3e06, 0x0e4e,
 0x002a, 0x1818, 0x243c, 0x3c06, 0x0c4e,
 0x002c, 0x181a, 0x243e, 0x3c08, 0x0c50,
 0x022c, 0x1a1a, 0x263e, 0x3e08, 0x0e50,
 0x042c, 0x1c1a, 0x283e, 0x4008, 0x1050,
 0x042e, 0x1c1c, 0x2840, 0x400a, 0x1052,
 0x022e, 0x1a1c, 0x2640, 0x3e0a, 0x0e52,
 0x002e, 0x181c, 0x2440, 0x3c0a, 0x0c52,
 0x0030, 0x181e, 0x2442, 0x3c0c, 0x0c54,
 0x0230, 0x1a1e, 0x2642, 0x3e0c, 0x0e54,
 0x0430, 0x1c1e, 0x2842, 0x400c, 0x1054,
 0x0432, 0x1c20, 0x2844, 0x400e, 0x1056,
 0x0232, 0x1a20, 0x2644, 0x3e0e, 0x0e56,
 0x0032, 0x1820, 0x2444, 0x3c0e, 0x0c56,
 0x0034, 0x1822, 0x2446, 0x3c10, 0x0c58,
 0x0234, 0x1a22, 0x2646, 0x3e10, 0x0e58,
 0x0434, 0x1c22, 0x2846, 0x4010, 0x1058,
 0x0624, 0x1e12, 0x2a36, 0x4200, 0x1248,
 0x0824, 0x2012, 0x2c36, 0x4400, 0x1448,
 0x0a24, 0x2212, 0x2e36, 0x4600, 0x1648,
 0x0a26, 0x2214, 0x2e38, 0x4602, 0x164a,
 0x0826, 0x2014, 0x2c38, 0x4402, 0x144a,
 0x0626, 0x1e14, 0x2a38, 0x4202, 0x124a,
 0x0628, 0x1e16, 0x2a3a, 0x4204, 0x124c,
 0x0828, 0x2016, 0x2c3a, 0x4404, 0x144c,
 0x0a28, 0x2216, 0x2e3a, 0x4604, 0x164c,
 0x0a2a, 0x2218, 0x2e3c, 0x4606, 0x164e,
 0x082a, 0x2018, 0x2c3c, 0x4406, 0x144e,
 0x062a, 0x1e18, 0x2a3c, 0x4206, 0x124e,
 0x062c, 0x1e1a, 0x2a3e, 0x4208, 0x1250,
 0x082c, 0x201a, 0x2c3e, 0x4408, 0x1450,
 0x0a2c, 0x221a, 0x2e3e, 0x4608, 0x1650,
 0x0a2e, 0x221c, 0x2e40, 0x460a, 0x1652,
 0x082e, 0x201c, 0x2c40, 0x440a, 0x1452,
 0x062e, 0x1e1c, 0x2a40, 0x420a, 0x1252,
 0x0630, 0x1e1e, 0x2a42, 0x420c, 0x1254,
 0x0830, 0x201e, 0x2c42, 0x440c, 0x1454,
 0x0a30, 0x221e, 0x2e42, 0x460c, 0x1654,
 0x0a32, 0x2220, 0x2e44, 0x460e, 0x1656,
 0x0832, 0x2020, 0x2c44, 0x440e, 0x1456,
 0x0632, 0x1e20, 0x2a44, 0x420e, 0x1256,
 0x0634, 0x1e22, 0x2a46, 0x4210, 0x1258,
 0x0834, 0x2022, 0x2c46, 0x4410, 0x1458,
 0x0a34, 0x2222, 0x2e46, 0x4610, 0x1658,
};

static const uint16_t dv_place_411P[1620] = {
 0x0c24, 0x2710, 0x3334, 0x0000, 0x1848,
 0x0d24, 0x2810, 0x3434, 0x0100, 0x1948,
 0x0e24, 0x2910, 0x3534, 0x0200, 0x1a48,
 0x0f24, 0x2914, 0x3538, 0x0300, 0x1b48,
 0x1024, 0x2814, 0x3438, 0x0400, 0x1c48,
 0x1124, 0x2714, 0x3338, 0x0500, 0x1d48,
 0x1128, 0x2614, 0x3238, 0x0504, 0x1d4c,
 0x1028, 0x2514, 0x3138, 0x0404, 0x1c4c,
 0x0f28, 0x2414, 0x3038, 0x0304, 0x1b4c,
 0x0e28, 0x2418, 0x303c, 0x0204, 0x1a4c,
 0x0d28, 0x2518, 0x313c, 0x0104, 0x194c,
 0x0c28, 0x2618, 0x323c, 0x0004, 0x184c,
 0x0c2c, 0x2718, 0x333c, 0x0008, 0x1850,
 0x0d2c, 0x2818, 0x343c, 0x0108, 0x1950,
 0x0e2c, 0x2918, 0x353c, 0x0208, 0x1a50,
 0x0f2c, 0x291c, 0x3540, 0x0308, 0x1b50,
 0x102c, 0x281c, 0x3440, 0x0408, 0x1c50,
 0x112c, 0x271c, 0x3340, 0x0508, 0x1d50,
 0x1130, 0x261c, 0x3240, 0x050c, 0x1d54,
 0x1030, 0x251c, 0x3140, 0x040c, 0x1c54,
 0x0f30, 0x241c, 0x3040, 0x030c, 0x1b54,
 0x0e30, 0x2420, 0x3044, 0x020c, 0x1a54,
 0x0d30, 0x2520, 0x3144, 0x010c, 0x1954,
 0x0c30, 0x2620, 0x3244, 0x000c, 0x1854,
 0x0c34, 0x2720, 0x3344, 0x0010, 0x1858,
 0x0d34, 0x2820, 0x3444, 0x0110, 0x1a58,
 0x0e34, 0x2920, 0x3544, 0x0210, 0x1c58,
 0x1224, 0x2d10, 0x3934, 0x0600, 0x1e48,
 0x1324, 0x2e10, 0x3a34, 0x0700, 0x1f48,
 0x1424, 0x2f10, 0x3b34, 0x0800, 0x2048,
 0x1524, 0x2f14, 0x3b38, 0x0900, 0x2148,
 0x1624, 0x2e14, 0x3a38, 0x0a00, 0x2248,
 0x1724, 0x2d14, 0x3938, 0x0b00, 0x2348,
 0x1728, 0x2c14, 0x3838, 0x0b04, 0x234c,
 0x1628, 0x2b14, 0x3738, 0x0a04, 0x224c,
 0x1528, 0x2a14, 0x3638, 0x0904, 0x214c,
 0x1428, 0x2a18, 0x363c, 0x0804, 0x204c,
 0x1328, 0x2b18, 0x373c, 0x0704, 0x1f4c,
 0x1228, 0x2c18, 0x383c, 0x0604, 0x1e4c,
 0x122c, 0x2d18, 0x393c, 0x0608, 0x1e50,
 0x132c, 0x2e18, 0x3a3c, 0x0708, 0x1f50,
 0x142c, 0x2f18, 0x3b3c, 0x0808, 0x2050,
 0x152c, 0x2f1c, 0x3b40, 0x0908, 0x2150,
 0x162c, 0x2e1c, 0x3a40, 0x0a08, 0x2250,
 0x172c, 0x2d1c, 0x3940, 0x0b08, 0x2350,
 0x1730, 0x2c1c, 0x3840, 0x0b0c, 0x2354,
 0x1630, 0x2b1c, 0x3740, 0x0a0c, 0x2254,
 0x1530, 0x2a1c, 0x3640, 0x090c, 0x2154,
 0x1430, 0x2a20, 0x3644, 0x080c, 0x2054,
 0x1330, 0x2b20, 0x3744, 0x070c, 0x1f54,
 0x1230, 0x2c20, 0x3844, 0x060c, 0x1e54,
 0x1234, 0x2d20, 0x3944, 0x0610, 0x1e58,
 0x1334, 0x2e20, 0x3a44, 0x0710, 0x2058,
 0x1434, 0x2f20, 0x3b44, 0x0810, 0x2258,
 0x1824, 0x3310, 0x3f34, 0x0c00, 0x2448,
 0x1924, 0x3410, 0x4034, 0x0d00, 0x2548,
 0x1a24, 0x3510, 0x4134, 0x0e00, 0x2648,
 0x1b24, 0x3514, 0x4138, 0x0f00, 0x2748,
 0x1c24, 0x3414, 0x4038, 0x1000, 0x2848,
 0x1d24, 0x3314, 0x3f38, 0x1100, 0x2948,
 0x1d28, 0x3214, 0x3e38, 0x1104, 0x294c,
 0x1c28, 0x3114, 0x3d38, 0x1004, 0x284c,
 0x1b28, 0x3014, 0x3c38, 0x0f04, 0x274c,
 0x1a28, 0x3018, 0x3c3c, 0x0e04, 0x264c,
 0x1928, 0x3118, 0x3d3c, 0x0d04, 0x254c,
 0x1828, 0x3218, 0x3e3c, 0x0c04, 0x244c,
 0x182c, 0x3318, 0x3f3c, 0x0c08, 0x2450,
 0x192c, 0x3418, 0x403c, 0x0d08, 0x2550,
 0x1a2c, 0x3518, 0x413c, 0x0e08, 0x2650,
 0x1b2c, 0x351c, 0x4140, 0x0f08, 0x2750,
 0x1c2c, 0x341c, 0x4040, 0x1008, 0x2850,
 0x1d2c, 0x331c, 0x3f40, 0x1108, 0x2950,
 0x1d30, 0x321c, 0x3e40, 0x110c, 0x2954,
 0x1c30, 0x311c, 0x3d40, 0x100c, 0x2854,
 0x1b30, 0x301c, 0x3c40, 0x0f0c, 0x2754,
 0x1a30, 0x3020, 0x3c44, 0x0e0c, 0x2654,
 0x1930, 0x3120, 0x3d44, 0x0d0c, 0x2554,
 0x1830, 0x3220, 0x3e44, 0x0c0c, 0x2454,
 0x1834, 0x3320, 0x3f44, 0x0c10, 0x2458,
 0x1934, 0x3420, 0x4044, 0x0d10, 0x2658,
 0x1a34, 0x3520, 0x4144, 0x0e10, 0x2858,
 0x1e24, 0x3910, 0x4534, 0x1200, 0x2a48,
 0x1f24, 0x3a10, 0x4634, 0x1300, 0x2b48,
 0x2024, 0x3b10, 0x4734, 0x1400, 0x2c48,
 0x2124, 0x3b14, 0x4738, 0x1500, 0x2d48,
 0x2224, 0x3a14, 0x4638, 0x1600, 0x2e48,
 0x2324, 0x3914, 0x4538, 0x1700, 0x2f48,
 0x2328, 0x3814, 0x4438, 0x1704, 0x2f4c,
 0x2228, 0x3714, 0x4338, 0x1604, 0x2e4c,
 0x2128, 0x3614, 0x4238, 0x1504, 0x2d4c,
 0x2028, 0x3618, 0x423c, 0x1404, 0x2c4c,
 0x1f28, 0x3718, 0x433c, 0x1304, 0x2b4c,
 0x1e28, 0x3818, 0x443c, 0x1204, 0x2a4c,
 0x1e2c, 0x3918, 0x453c, 0x1208, 0x2a50,
 0x1f2c, 0x3a18, 0x463c, 0x1308, 0x2b50,
 0x202c, 0x3b18, 0x473c, 0x1408, 0x2c50,
 0x212c, 0x3b1c, 0x4740, 0x1508, 0x2d50,
 0x222c, 0x3a1c, 0x4640, 0x1608, 0x2e50,
 0x232c, 0x391c, 0x4540, 0x1708, 0x2f50,
 0x2330, 0x381c, 0x4440, 0x170c, 0x2f54,
 0x2230, 0x371c, 0x4340, 0x160c, 0x2e54,
 0x2130, 0x361c, 0x4240, 0x150c, 0x2d54,
 0x2030, 0x3620, 0x4244, 0x140c, 0x2c54,
 0x1f30, 0x3720, 0x4344, 0x130c, 0x2b54,
 0x1e30, 0x3820, 0x4444, 0x120c, 0x2a54,
 0x1e34, 0x3920, 0x4544, 0x1210, 0x2a58,
 0x1f34, 0x3a20, 0x4644, 0x1310, 0x2c58,
 0x2034, 0x3b20, 0x4744, 0x1410, 0x2e58,
 0x2424, 0x3f10, 0x0334, 0x1800, 0x3048,
 0x2524, 0x4010, 0x0434, 0x1900, 0x3148,
 0x2624, 0x4110, 0x0534, 0x1a00, 0x3248,
 0x2724, 0x4114, 0x0538, 0x1b00, 0x3348,
 0x2824, 0x4014, 0x0438, 0x1c00, 0x3448,
 0x2924, 0x3f14, 0x0338, 0x1d00, 0x3548,
 0x2928, 0x3e14, 0x0238, 0x1d04, 0x354c,
 0x2828, 0x3d14, 0x0138, 0x1c04, 0x344c,
 0x2728, 0x3c14, 0x0038, 0x1b04, 0x334c,
 0x2628, 0x3c18, 0x003c, 0x1a04, 0x324c,
 0x2528, 0x3d18, 0x013c, 0x1904, 0x314c,
 0x2428, 0x3e18, 0x023c, 0x1804, 0x304c,
 0x242c, 0x3f18, 0x033c, 0x1808, 0x3050,
 0x252c, 0x4018, 0x043c, 0x1908, 0x3150,
 0x262c, 0x4118, 0x053c, 0x1a08, 0x3250,
 0x272c, 0x411c, 0x0540, 0x1b08, 0x3350,
 0x282c, 0x401c, 0x0440, 0x1c08, 0x3450,
 0x292c, 0x3f1c, 0x0340, 0x1d08, 0x3550,
 0x2930, 0x3e1c, 0x0240, 0x1d0c, 0x3554,
 0x2830, 0x3d1c, 0x0140, 0x1c0c, 0x3454,
 0x2730, 0x3c1c, 0x0040, 0x1b0c, 0x3354,
 0x2630, 0x3c20, 0x0044, 0x1a0c, 0x3254,
 0x2530, 0x3d20, 0x0144, 0x190c, 0x3154,
 0x2430, 0x3e20, 0x0244, 0x180c, 0x3054,
 0x2434, 0x3f20, 0x0344, 0x1810, 0x3058,
 0x2534, 0x4020, 0x0444, 0x1910, 0x3258,
 0x2634, 0x4120, 0x0544, 0x1a10, 0x3458,
 0x2a24, 0x4510, 0x0934, 0x1e00, 0x3648,
 0x2b24, 0x4610, 0x0a34, 0x1f00, 0x3748,
 0x2c24, 0x4710, 0x0b34, 0x2000, 0x3848,
 0x2d24, 0x4714, 0x0b38, 0x2100, 0x3948,
 0x2e24, 0x4614, 0x0a38, 0x2200, 0x3a48,
 0x2f24, 0x4514, 0x0938, 0x2300, 0x3b48,
 0x2f28, 0x4414, 0x0838, 0x2304, 0x3b4c,
 0x2e28, 0x4314, 0x0738, 0x2204, 0x3a4c,
 0x2d28, 0x4214, 0x0638, 0x2104, 0x394c,
 0x2c28, 0x4218, 0x063c, 0x2004, 0x384c,
 0x2b28, 0x4318, 0x073c, 0x1f04, 0x374c,
 0x2a28, 0x4418, 0x083c, 0x1e04, 0x364c,
 0x2a2c, 0x4518, 0x093c, 0x1e08, 0x3650,
 0x2b2c, 0x4618, 0x0a3c, 0x1f08, 0x3750,
 0x2c2c, 0x4718, 0x0b3c, 0x2008, 0x3850,
 0x2d2c, 0x471c, 0x0b40, 0x2108, 0x3950,
 0x2e2c, 0x461c, 0x0a40, 0x2208, 0x3a50,
 0x2f2c, 0x451c, 0x0940, 0x2308, 0x3b50,
 0x2f30, 0x441c, 0x0840, 0x230c, 0x3b54,
 0x2e30, 0x431c, 0x0740, 0x220c, 0x3a54,
 0x2d30, 0x421c, 0x0640, 0x210c, 0x3954,
 0x2c30, 0x4220, 0x0644, 0x200c, 0x3854,
 0x2b30, 0x4320, 0x0744, 0x1f0c, 0x3754,
 0x2a30, 0x4420, 0x0844, 0x1e0c, 0x3654,
 0x2a34, 0x4520, 0x0944, 0x1e10, 0x3658,
 0x2b34, 0x4620, 0x0a44, 0x1f10, 0x3858,
 0x2c34, 0x4720, 0x0b44, 0x2010, 0x3a58,
 0x3024, 0x0310, 0x0f34, 0x2400, 0x3c48,
 0x3124, 0x0410, 0x1034, 0x2500, 0x3d48,
 0x3224, 0x0510, 0x1134, 0x2600, 0x3e48,
 0x3324, 0x0514, 0x1138, 0x2700, 0x3f48,
 0x3424, 0x0414, 0x1038, 0x2800, 0x4048,
 0x3524, 0x0314, 0x0f38, 0x2900, 0x4148,
 0x3528, 0x0214, 0x0e38, 0x2904, 0x414c,
 0x3428, 0x0114, 0x0d38, 0x2804, 0x404c,
 0x3328, 0x0014, 0x0c38, 0x2704, 0x3f4c,
 0x3228, 0x0018, 0x0c3c, 0x2604, 0x3e4c,
 0x3128, 0x0118, 0x0d3c, 0x2504, 0x3d4c,
 0x3028, 0x0218, 0x0e3c, 0x2404, 0x3c4c,
 0x302c, 0x0318, 0x0f3c, 0x2408, 0x3c50,
 0x312c, 0x0418, 0x103c, 0x2508, 0x3d50,
 0x322c, 0x0518, 0x113c, 0x2608, 0x3e50,
 0x332c, 0x051c, 0x1140, 0x2708, 0x3f50,
 0x342c, 0x041c, 0x1040, 0x2808, 0x4050,
 0x352c, 0x031c, 0x0f40, 0x2908, 0x4150,
 0x3530, 0x021c, 0x0e40, 0x290c, 0x4154,
 0x3430, 0x011c, 0x0d40, 0x280c, 0x4054,
 0x3330, 0x001c, 0x0c40, 0x270c, 0x3f54,
 0x3230, 0x0020, 0x0c44, 0x260c, 0x3e54,
 0x3130, 0x0120, 0x0d44, 0x250c, 0x3d54,
 0x3030, 0x0220, 0x0e44, 0x240c, 0x3c54,
 0x3034, 0x0320, 0x0f44, 0x2410, 0x3c58,
 0x3134, 0x0420, 0x1044, 0x2510, 0x3e58,
 0x3234, 0x0520, 0x1144, 0x2610, 0x4058,
 0x3624, 0x0910, 0x1534, 0x2a00, 0x4248,
 0x3724, 0x0a10, 0x1634, 0x2b00, 0x4348,
 0x3824, 0x0b10, 0x1734, 0x2c00, 0x4448,
 0x3924, 0x0b14, 0x1738, 0x2d00, 0x4548,
 0x3a24, 0x0a14, 0x1638, 0x2e00, 0x4648,
 0x3b24, 0x0914, 0x1538, 0x2f00, 0x4748,
 0x3b28, 0x0814, 0x1438, 0x2f04, 0x474c,
 0x3a28, 0x0714, 0x1338, 0x2e04, 0x464c,
 0x3928, 0x0614, 0x1238, 0x2d04, 0x454c,
 0x3828, 0x0618, 0x123c, 0x2c04, 0x444c,
 0x3728, 0x0718, 0x133c, 0x2b04, 0x434c,
 0x3628, 0x0818, 0x143c, 0x2a04, 0x424c,
 0x362c, 0x0918, 0x153c, 0x2a08, 0x4250,
 0x372c, 0x0a18, 0x163c, 0x2b08, 0x4350,
 0x382c, 0x0b18, 0x173c, 0x2c08, 0x4450,
 0x392c, 0x0b1c, 0x1740, 0x2d08, 0x4550,
 0x3a2c, 0x0a1c, 0x1640, 0x2e08, 0x4650,
 0x3b2c, 0x091c, 0x1540, 0x2f08, 0x4750,
 0x3b30, 0x081c, 0x1440, 0x2f0c, 0x4754,
 0x3a30, 0x071c, 0x1340, 0x2e0c, 0x4654,
 0x3930, 0x061c, 0x1240, 0x2d0c, 0x4554,
 0x3830, 0x0620, 0x1244, 0x2c0c, 0x4454,
 0x3730, 0x0720, 0x1344, 0x2b0c, 0x4354,
 0x3630, 0x0820, 0x1444, 0x2a0c, 0x4254,
 0x3634, 0x0920, 0x1544, 0x2a10, 0x4258,
 0x3734, 0x0a20, 0x1644, 0x2b10, 0x4458,
 0x3834, 0x0b20, 0x1744, 0x2c10, 0x4658,
 0x3c24, 0x0f10, 0x1b34, 0x3000, 0x0048,
 0x3d24, 0x1010, 0x1c34, 0x3100, 0x0148,
 0x3e24, 0x1110, 0x1d34, 0x3200, 0x0248,
 0x3f24, 0x1114, 0x1d38, 0x3300, 0x0348,
 0x4024, 0x1014, 0x1c38, 0x3400, 0x0448,
 0x4124, 0x0f14, 0x1b38, 0x3500, 0x0548,
 0x4128, 0x0e14, 0x1a38, 0x3504, 0x054c,
 0x4028, 0x0d14, 0x1938, 0x3404, 0x044c,
 0x3f28, 0x0c14, 0x1838, 0x3304, 0x034c,
 0x3e28, 0x0c18, 0x183c, 0x3204, 0x024c,
 0x3d28, 0x0d18, 0x193c, 0x3104, 0x014c,
 0x3c28, 0x0e18, 0x1a3c, 0x3004, 0x004c,
 0x3c2c, 0x0f18, 0x1b3c, 0x3008, 0x0050,
 0x3d2c, 0x1018, 0x1c3c, 0x3108, 0x0150,
 0x3e2c, 0x1118, 0x1d3c, 0x3208, 0x0250,
 0x3f2c, 0x111c, 0x1d40, 0x3308, 0x0350,
 0x402c, 0x101c, 0x1c40, 0x3408, 0x0450,
 0x412c, 0x0f1c, 0x1b40, 0x3508, 0x0550,
 0x4130, 0x0e1c, 0x1a40, 0x350c, 0x0554,
 0x4030, 0x0d1c, 0x1940, 0x340c, 0x0454,
 0x3f30, 0x0c1c, 0x1840, 0x330c, 0x0354,
 0x3e30, 0x0c20, 0x1844, 0x320c, 0x0254,
 0x3d30, 0x0d20, 0x1944, 0x310c, 0x0154,
 0x3c30, 0x0e20, 0x1a44, 0x300c, 0x0054,
 0x3c34, 0x0f20, 0x1b44, 0x3010, 0x0058,
 0x3d34, 0x1020, 0x1c44, 0x3110, 0x0258,
 0x3e34, 0x1120, 0x1d44, 0x3210, 0x0458,
 0x4224, 0x1510, 0x2134, 0x3600, 0x0648,
 0x4324, 0x1610, 0x2234, 0x3700, 0x0748,
 0x4424, 0x1710, 0x2334, 0x3800, 0x0848,
 0x4524, 0x1714, 0x2338, 0x3900, 0x0948,
 0x4624, 0x1614, 0x2238, 0x3a00, 0x0a48,
 0x4724, 0x1514, 0x2138, 0x3b00, 0x0b48,
 0x4728, 0x1414, 0x2038, 0x3b04, 0x0b4c,
 0x4628, 0x1314, 0x1f38, 0x3a04, 0x0a4c,
 0x4528, 0x1214, 0x1e38, 0x3904, 0x094c,
 0x4428, 0x1218, 0x1e3c, 0x3804, 0x084c,
 0x4328, 0x1318, 0x1f3c, 0x3704, 0x074c,
 0x4228, 0x1418, 0x203c, 0x3604, 0x064c,
 0x422c, 0x1518, 0x213c, 0x3608, 0x0650,
 0x432c, 0x1618, 0x223c, 0x3708, 0x0750,
 0x442c, 0x1718, 0x233c, 0x3808, 0x0850,
 0x452c, 0x171c, 0x2340, 0x3908, 0x0950,
 0x462c, 0x161c, 0x2240, 0x3a08, 0x0a50,
 0x472c, 0x151c, 0x2140, 0x3b08, 0x0b50,
 0x4730, 0x141c, 0x2040, 0x3b0c, 0x0b54,
 0x4630, 0x131c, 0x1f40, 0x3a0c, 0x0a54,
 0x4530, 0x121c, 0x1e40, 0x390c, 0x0954,
 0x4430, 0x1220, 0x1e44, 0x380c, 0x0854,
 0x4330, 0x1320, 0x1f44, 0x370c, 0x0754,
 0x4230, 0x1420, 0x2044, 0x360c, 0x0654,
 0x4234, 0x1520, 0x2144, 0x3610, 0x0658,
 0x4334, 0x1620, 0x2244, 0x3710, 0x0858,
 0x4434, 0x1720, 0x2344, 0x3810, 0x0a58,
 0x0024, 0x1b10, 0x2734, 0x3c00, 0x0c48,
 0x0124, 0x1c10, 0x2834, 0x3d00, 0x0d48,
 0x0224, 0x1d10, 0x2934, 0x3e00, 0x0e48,
 0x0324, 0x1d14, 0x2938, 0x3f00, 0x0f48,
 0x0424, 0x1c14, 0x2838, 0x4000, 0x1048,
 0x0524, 0x1b14, 0x2738, 0x4100, 0x1148,
 0x0528, 0x1a14, 0x2638, 0x4104, 0x114c,
 0x0428, 0x1914, 0x2538, 0x4004, 0x104c,
 0x0328, 0x1814, 0x2438, 0x3f04, 0x0f4c,
 0x0228, 0x1818, 0x243c, 0x3e04, 0x0e4c,
 0x0128, 0x1918, 0x253c, 0x3d04, 0x0d4c,
 0x0028, 0x1a18, 0x263c, 0x3c04, 0x0c4c,
 0x002c, 0x1b18, 0x273c, 0x3c08, 0x0c50,
 0x012c, 0x1c18, 0x283c, 0x3d08, 0x0d50,
 0x022c, 0x1d18, 0x293c, 0x3e08, 0x0e50,
 0x032c, 0x1d1c, 0x2940, 0x3f08, 0x0f50,
 0x042c, 0x1c1c, 0x2840, 0x4008, 0x1050,
 0x052c, 0x1b1c, 0x2740, 0x4108, 0x1150,
 0x0530, 0x1a1c, 0x2640, 0x410c, 0x1154,
 0x0430, 0x191c, 0x2540, 0x400c, 0x1054,
 0x0330, 0x181c, 0x2440, 0x3f0c, 0x0f54,
 0x0230, 0x1820, 0x2444, 0x3e0c, 0x0e54,
 0x0130, 0x1920, 0x2544, 0x3d0c, 0x0d54,
 0x0030, 0x1a20, 0x2644, 0x3c0c, 0x0c54,
 0x0034, 0x1b20, 0x2744, 0x3c10, 0x0c58,
 0x0134, 0x1c20, 0x2844, 0x3d10, 0x0e58,
 0x0234, 0x1d20, 0x2944, 0x3e10, 0x1058,
 0x0624, 0x2110, 0x2d34, 0x4200, 0x1248,
 0x0724, 0x2210, 0x2e34, 0x4300, 0x1348,
 0x0824, 0x2310, 0x2f34, 0x4400, 0x1448,
 0x0924, 0x2314, 0x2f38, 0x4500, 0x1548,
 0x0a24, 0x2214, 0x2e38, 0x4600, 0x1648,
 0x0b24, 0x2114, 0x2d38, 0x4700, 0x1748,
 0x0b28, 0x2014, 0x2c38, 0x4704, 0x174c,
 0x0a28, 0x1f14, 0x2b38, 0x4604, 0x164c,
 0x0928, 0x1e14, 0x2a38, 0x4504, 0x154c,
 0x0828, 0x1e18, 0x2a3c, 0x4404, 0x144c,
 0x0728, 0x1f18, 0x2b3c, 0x4304, 0x134c,
 0x0628, 0x2018, 0x2c3c, 0x4204, 0x124c,
 0x062c, 0x2118, 0x2d3c, 0x4208, 0x1250,
 0x072c, 0x2218, 0x2e3c, 0x4308, 0x1350,
 0x082c, 0x2318, 0x2f3c, 0x4408, 0x1450,
 0x092c, 0x231c, 0x2f40, 0x4508, 0x1550,
 0x0a2c, 0x221c, 0x2e40, 0x4608, 0x1650,
 0x0b2c, 0x211c, 0x2d40, 0x4708, 0x1750,
 0x0b30, 0x201c, 0x2c40, 0x470c, 0x1754,
 0x0a30, 0x1f1c, 0x2b40, 0x460c, 0x1654,
 0x0930, 0x1e1c, 0x2a40, 0x450c, 0x1554,
 0x0830, 0x1e20, 0x2a44, 0x440c, 0x1454,
 0x0730, 0x1f20, 0x2b44, 0x430c, 0x1354,
 0x0630, 0x2020, 0x2c44, 0x420c, 0x1254,
 0x0634, 0x2120, 0x2d44, 0x4210, 0x1258,
 0x0734, 0x2220, 0x2e44, 0x4310, 0x1458,
 0x0834, 0x2320, 0x2f44, 0x4410, 0x1658,
};

static const uint16_t dv_place_411[1350] = {
 0x0c24, 0x2710, 0x3334, 0x0000, 0x1848,
 0x0d24, 0x2810, 0x3434, 0x0100, 0x1948,
 0x0e24, 0x2910, 0x3534, 0x0200, 0x1a48,
 0x0f24, 0x2914, 0x3538, 0x0300, 0x1b48,
 0x1024, 0x2814, 0x3438, 0x0400, 0x1c48,
 0x1124, 0x2714, 0x3338, 0x0500, 0x1d48,
 0x1128, 0x2614, 0x3238, 0x0504, 0x1d4c,
 0x1028, 0x2514, 0x3138, 0x0404, 0x1c4c,
 0x0f28, 0x2414, 0x3038, 0x0304, 0x1b4c,
 0x0e28, 0x2418, 0x303c, 0x0204, 0x1a4c,
 0x0d28, 0x2518, 0x313c, 0x0104, 0x194c,
 0x0c28, 0x2618, 0x323c, 0x0004, 0x184c,
 0x0c2c, 0x2718, 0x333c, 0x0008, 0x1850,
 0x0d2c, 0x2818, 0x343c, 0x0108, 0x1950,
 0x0e2c, 0x2918, 0x353c, 0x0208, 0x1a50,
 0x0f2c, 0x291c, 0x3540, 0x0308, 0x1b50,
 0x102c, 0x281c, 0x3440, 0x0408, 0x1c50,
 0x112c, 0x271c, 0x3340, 0x0508, 0x1d50,
 0x1130, 0x261c, 0x3240, 0x050c, 0x1d54,
 0x1030, 0x251c, 0x3140, 0x040c, 0x1c54,
 0x0f30, 0x241c, 0x3040, 0x030c, 0x1b54,
 0x0e30, 0x2420, 0x3044, 0x020c, 0x1a54,
 0x0d30, 0x2520, 0x3144, 0x010c, 0x1954,
 0x0c30, 0x2620, 0x3244, 0x000c, 0x1854,
 0x0c34, 0x2720, 0x3344, 0x0010, 0x1858,
 0x0d34, 0x2820, 0x3444, 0x0110, 0x1a58,
 0x0e34, 0x2920, 0x3544, 0x0210, 0x1c58,
 0x1224, 0x2d10, 0x3934, 0x0600, 0x1e48,
 0x1324, 0x2e10, 0x3a34, 0x0700, 0x1f48,
 0x1424, 0x2f10, 0x3b34, 0x0800, 0x2048,
 0x1524, 0x2f14, 0x3b38, 0x0900, 0x2148,
 0x1624, 0x2e14, 0x3a38, 0x0a00, 0x2248,
 0x1724, 0x2d14, 0x3938, 0x0b00, 0x2348,
 0x1728, 0x2c14, 0x3838, 0x0b04, 0x234c,
 0x1628, 0x2b14, 0x3738, 0x0a04, 0x224c,
 0x1528, 0x2a14, 0x3638, 0x0904, 0x214c,
 0x1428, 0x2a18, 0x363c, 0x0804, 0x204c,
 0x1328, 0x2b18, 0x373c, 0x0704, 0x1f4c,
 0x1228, 0x2c18, 0x383c, 0x0604, 0x1e4c,
 0x122c, 0x2d18, 0x393c, 0x0608, 0x1e50,
 0x132c, 0x2e18, 0x3a3c, 0x0708, 0x1f50,
 0x142c, 0x2f18, 0x3b3c, 0x0808, 0x2050,
 0x152c, 0x2f1c, 0x3b40, 0x0908, 0x2150,
 0x162c, 0x2e1c, 0x3a40, 0x0a08, 0x2250,
 0x172c, 0x2d1c, 0x3940, 0x0b08, 0x2350,
 0x1730, 0x2c1c, 0x3840, 0x0b0c, 0x2354,
 0x1630, 0x2b1c, 0x3740, 0x0a0c, 0x2254,
 0x1530, 0x2a1c, 0x3640, 0x090c, 0x2154,
 0x1430, 0x2a20, 0x3644, 0x080c, 0x2054,
 0x1330, 0x2b20, 0x3744, 0x070c, 0x1f54,
 0x1230, 0x2c20, 0x3844, 0x060c, 0x1e54,
 0x1234, 0x2d20, 0x3944, 0x0610, 0x1e58,
 0x1334, 0x2e20, 0x3a44, 0x0710, 0x2058,
 0x1434, 0x2f20, 0x3b44, 0x0810, 0x2258,
 0x1824, 0x3310, 0x0334, 0x0c00, 0x2448,
 0x1924, 0x3410, 0x0434, 0x0d00, 0x2548,
 0x1a24, 0x3510, 0x0534, 0x0e00, 0x2648,
 0x1b24, 0x3514, 0x0538, 0x0f00, 0x2748,
 0x1c24, 0x3414, 0x0438, 0x1000, 0x2848,
 0x1d24, 0x3314, 0x0338, 0x1100, 0x2948,
 0x1d28, 0x3214, 0x0238, 0x1104, 0x294c,
 0x1c28, 0x3114, 0x0138, 0x1004, 0x284c,
 0x1b28, 0x3014, 0x0038, 0x0f04, 0x274c,
 0x1a28, 0x3018, 0x003c, 0x0e04, 0x264c,
 0x1928, 0x3118, 0x013c, 0x0d04, 0x254c,
 0x1828, 0x3218, 0x023c, 0x0c04, 0x244c,
 0x182c, 0x3318, 0x033c, 0x0c08, 0x2450,
 0x192c, 0x3418, 0x043c, 0x0d08, 0x2550,
 0x1a2c, 0x3518, 0x053c, 0x0e08, 0x2650,
 0x1b2c, 0x351c, 0x0540, 0x0f08, 0x2750,
 0x1c2c, 0x341c, 0x0440, 0x1008, 0x2850,
 0x1d2c, 0x331c, 0x0340, 0x1108, 0x2950,
 0x1d30, 0x321c, 0x0240, 0x110c, 0x2954,
 0x1c30, 0x311c, 0x0140, 0x100c, 0x2854,
 0x1b30, 0x301c, 0x0040, 0x0f0c, 0x2754,
 0x1a30, 0x3020, 0x0044, 0x0e0c, 0x2654,
 0x1930, 0x3120, 0x0144, 0x0d0c, 0x2554,
 0x1830, 0x3220, 0x0244, 0x0c0c, 0x2454,
 0x1834, 0x3320, 0x0344, 0x0c10, 0x2458,
 0x1934, 0x3420, 0x0444, 0x0d10, 0x2658,
 0x1a34, 0x3520, 0x0544, 0x0e10, 0x2858,
 0x1e24, 0x3910, 0x0934, 0x1200, 0x2a48,
 0x1f24, 0x3a10, 0x0a34, 0x1300, 0x2b48,
 0x2024, 0x3b10, 0x0b34, 0x1400, 0x2c48,
 0x2124, 0x3b14, 0x0b38, 0x1500, 0x2d48,
 0x2224, 0x3a14, 0x0a38, 0x1600, 0x2e48,
 0x2324, 0x3914, 0x0938, 0x1700, 0x2f48,
 0x2328, 0x3814, 0x0838, 0x1704, 0x2f4c,
 0x2228, 0x3714, 0x0738, 0x1604, 0x2e4c,
 0x2128, 0x3614, 0x0638, 0x1504, 0x2d4c,
 0x2028, 0x3618, 0x063c, 0x1404, 0x2c4c,
 0x1f28, 0x3718, 0x073c, 0x1304, 0x2b4c,
 0x1e28, 0x3818, 0x083c, 0x1204, 0x2a4c,
 0x1e2c, 0x3918, 0x093c, 0x1208, 0x2a50,
 0x1f2c, 0x3a18, 0x0a3c, 0x1308, 0x2b50,
 0x202c, 0x3b18, 0x0b3c, 0x1408, 0x2c50,
 0x212c, 0x3b1c, 0x0b40, 0x1508, 0x2d50,
 0x222c, 0x3a1c, 0x0a40, 0x1608, 0x2e50,
 0x232c, 0x391c, 0x0940, 0x1708, 0x2f50,
 0x2330, 0x381c, 0x0840, 0x170c, 0x2f54,
 0x2230, 0x371c, 0x0740, 0x160c, 0x2e54,
 0x2130, 0x361c, 0x0640, 0x150c, 0x2d54,
 0x2030, 0x3620, 0x0644, 0x140c, 0x2c54,
 0x1f30, 0x3720, 0x0744, 0x130c, 0x2b54,
 0x1e30, 0x3820, 0x0844, 0x120c, 0x2a54,
 0x1e34, 0x3920, 0x0944, 0x1210, 0x2a58,
 0x1f34, 0x3a20, 0x0a44, 0x1310, 0x2c58,
 0x2034, 0x3b20, 0x0b44, 0x1410, 0x2e58,
 0x2424, 0x0310, 0x0f34, 0x1800, 0x3048,
 0x2524, 0x0410, 0x1034, 0x1900, 0x3148,
 0x2624, 0x0510, 0x1134, 0x1a00, 0x3248,
 0x2724, 0x0514, 0x1138, 0x1b00, 0x3348,
 0x2824, 0x0414, 0x1038, 0x1c00, 0x3448,
 0x2924, 0x0314, 0x0f38, 0x1d00, 0x3548,
 0x2928, 0x0214, 0x0e38, 0x1d04, 0x354c,
 0x2828, 0x0114, 0x0d38, 0x1c04, 0x344c,
 0x2728, 0x0014, 0x0c38, 0x1b04, 0x334c,
 0x2628, 0x0018, 0x0c3c, 0x1a04, 0x324c,
 0x2528, 0x0118, 0x0d3c, 0x1904, 0x314c,
 0x2428, 0x0218, 0x0e3c, 0x1804, 0x304c,
 0x242c, 0x0318, 0x0f3c, 0x1808, 0x3050,
 0x252c, 0x0418, 0x103c, 0x1908, 0x3150,
 0x262c, 0x0518, 0x113c, 0x1a08, 0x3250,
 0x272c, 0x051c, 0x1140, 0x1b08, 0x3350,
 0x282c, 0x041c, 0x1040, 0x1c08, 0x3450,
 0x292c, 0x031c, 0x0f40, 0x1d08, 0x3550,
 0x2930, 0x021c, 0x0e40, 0x1d0c, 0x3554,
 0x2830, 0x011c, 0x0d40, 0x1c0c, 0x3454,
 0x2730, 0x001c, 0x0c40, 0x1b0c, 0x3354,
 0x2630, 0x0020, 0x0c44, 0x1a0c, 0x3254,
 0x2530, 0x0120, 0x0d44, 0x190c, 0x3154,
 0x2430, 0x0220, 0x0e44, 0x180c, 0x3054,
 0x2434, 0x0320, 0x0f44, 0x1810, 0x3058,
 0x2534, 0x0420, 0x1044, 0x1910, 0x3258,
 0x2634, 0x0520, 0x1144, 0x1a10, 0x3458,
 0x2a24, 0x0910, 0x1534, 0x1e00, 0x3648,
 0x2b24, 0x0a10, 0x1634, 0x1f00, 0x3748,
 0x2c24, 0x0b10, 0x1734, 0x2000, 0x3848,
 0x2d24, 0x0b14, 0x1738, 0x2100, 0x3948,
 0x2e24, 0x0a14, 0x1638, 0x2200, 0x3a48,
 0x2f24, 0x0914, 0x1538, 0x2300, 0x3b48,
 0x2f28, 0x0814, 0x1438, 0x2304, 0x3b4c,
 0x2e28, 0x0714, 0x1338, 0x2204, 0x3a4c,
 0x2d28, 0x0614, 0x1238, 0x2104, 0x394c,
 0x2c28, 0x0618, 0x123c, 0x2004, 0x384c,
 0x2b28, 0x0718, 0x133c, 0x1f04, 0x374c,
 0x2a28, 0x0818, 0x143c, 0x1e04, 0x364c,
 0x2a2c, 0x0918, 0x153c, 0x1e08, 0x3650,
 0x2b2c, 0x0a18, 0x163c, 0x1f08, 0x3750,
 0x2c2c, 0x0b18, 0x173c, 0x2008, 0x3850,
 0x2d2c, 0x0b1c, 0x1740, 0x2108, 0x3950,
 0x2e2c, 0x0a1c, 0x1640, 0x2208, 0x3a50,
 0x2f2c, 0x091c, 0x1540, 0x2308, 0x3b50,
 0x2f30, 0x081c, 0x1440, 0x230c, 0x3b54,
 0x2e30, 0x071c, 0x1340, 0x220c, 0x3a54,
 0x2d30, 0x061c, 0x1240, 0x210c, 0x3954,
 0x2c30, 0x0620, 0x1244, 0x200c, 0x3854,
 0x2b30, 0x0720, 0x1344, 0x1f0c, 0x3754,
 0x2a30, 0x0820, 0x1444, 0x1e0c, 0x3654,
 0x2a34, 0x0920, 0x1544, 0x1e10, 0x3658,
 0x2b34, 0x0a20, 0x1644, 0x1f10, 0x3858,
 0x2c34, 0x0b20, 0x1744, 0x2010, 0x3a58,
 0x3024, 0x0f10, 0x1b34, 0x2400, 0x0048,
 0x3124, 0x1010, 0x1c34, 0x2500, 0x0148,
 0x3224, 0x1110, 0x1d34, 0x2600, 0x0248,
 0x3324, 0x1114, 0x1d38, 0x2700, 0x0348,
 0x3424, 0x1014, 0x1c38, 0x2800, 0x0448,
 0x3524, 0x0f14, 0x1b38, 0x2900, 0x0548,
 0x3528, 0x0e14, 0x1a38, 0x2904, 0x054c,
 0x3428, 0x0d14, 0x1938, 0x2804, 0x044c,
 0x3328, 0x0c14, 0x1838, 0x2704, 0x034c,
 0x3228, 0x0c18, 0x183c, 0x2604, 0x024c,
 0x3128, 0x0d18, 0x193c, 0x2504, 0x014c,
 0x3028, 0x0e18, 0x1a3c, 0x2404, 0x004c,
 0x302c, 0x0f18, 0x1b3c, 0x2408, 0x0050,
 0x312c, 0x1018, 0x1c3c, 0x2508, 0x0150,
 0x322c, 0x1118, 0x1d3c, 0x2608, 0x0250,
 0x332c, 0x111c, 0x1d40, 0x2708, 0x0350,
 0x342c, 0x101c, 0x1c40, 0x2808, 0x0450,
 0x352c, 0x0f1c, 0x1b40, 0x2908, 0x0550,
 0x3530, 0x0e1c, 0x1a40, 0x290c, 0x0554,
 0x3430, 0x0d1c, 0x1940, 0x280c, 0x0454,
 0x3330, 0x0c1c, 0x1840, 0x270c, 0x0354,
 0x3230, 0x0c20, 0x1844, 0x260c, 0x0254,
 0x3130, 0x0d20, 0x1944, 0x250c, 0x0154,
 0x3030, 0x0e20, 0x1a44, 0x240c, 0x0054,
 0x3034, 0x0f20, 0x1b44, 0x2410, 0x0058,
 0x3134, 0x1020, 0x1c44, 0x2510, 0x0258,
 0x3234, 0x1120, 0x1d44, 0x2610, 0x0458,
 0x3624, 0x1510, 0x2134, 0x2a00, 0x0648,
 0x3724, 0x1610, 0x2234, 0x2b00, 0x0748,
 0x3824, 0x1710, 0x2334, 0x2c00, 0x0848,
 0x3924, 0x1714, 0x2338, 0x2d00, 0x0948,
 0x3a24, 0x1614, 0x2238, 0x2e00, 0x0a48,
 0x3b24, 0x1514, 0x2138, 0x2f00, 0x0b48,
 0x3b28, 0x1414, 0x2038, 0x2f04, 0x0b4c,
 0x3a28, 0x1314, 0x1f38, 0x2e04, 0x0a4c,
 0x3928, 0x1214, 0x1e38, 0x2d04, 0x094c,
 0x3828, 0x1218, 0x1e3c, 0x2c04, 0x084c,
 0x3728, 0x1318, 0x1f3c, 0x2b04, 0x074c,
 0x3628, 0x1418, 0x203c, 0x2a04, 0x064c,
 0x362c, 0x1518, 0x213c, 0x2a08, 0x0650,
 0x372c, 0x1618, 0x223c, 0x2b08, 0x0750,
 0x382c, 0x1718, 0x233c, 0x2c08, 0x0850,
 0x392c, 0x171c, 0x2340, 0x2d08, 0x0950,
 0x3a2c, 0x161c, 0x2240, 0x2e08, 0x0a50,
 0x3b2c, 0x151c, 0x2140, 0x2f08, 0x0b50,
 0x3b30, 0x141c, 0x2040, 0x2f0c, 0x0b54,
 0x3a30, 0x131c, 0x1f40, 0x2e0c, 0x0a54,
 0x3930, 0x121c, 0x1e40, 0x2d0c, 0x0954,
 0x3830, 0x1220, 0x1e44, 0x2c0c, 0x0854,
 0x3730, 0x1320, 0x1f44, 0x2b0c, 0x0754,
 0x3630, 0x1420, 0x2044, 0x2a0c, 0x0654,
 0x3634, 0x1520, 0x2144, 0x2a10, 0x0658,
 0x3734, 0x1620, 0x2244, 0x2b10, 0x0858,
 0x3834, 0x1720, 0x2344, 0x2c10, 0x0a58,
 0x0024, 0x1b10, 0x2734, 0x3000, 0x0c48,
 0x0124, 0x1c10, 0x2834, 0x3100, 0x0d48,
 0x0224, 0x1d10, 0x2934, 0x3200, 0x0e48,
 0x0324, 0x1d14, 0x2938, 0x3300, 0x0f48,
 0x0424, 0x1c14, 0x2838, 0x3400, 0x1048,
 0x0524, 0x1b14, 0x2738, 0x3500, 0x1148,
 0x0528, 0x1a14, 0x2638, 0x3504, 0x114c,
 0x0428, 0x1914, 0x2538, 0x3404, 0x104c,
 0x0328, 0x1814, 0x2438, 0x3304, 0x0f4c,
 0x0228, 0x1818, 0x243c, 0x3204, 0x0e4c,
 0x0128, 0x1918, 0x253c, 0x3104, 0x0d4c,
 0x0028, 0x1a18, 0x263c, 0x3004, 0x0c4c,
 0x002c, 0x1b18, 0x273c, 0x3008, 0x0c50,
 0x012c, 0x1c18, 0x283c, 0x3108, 0x0d50,
 0x022c, 0x1d18, 0x293c, 0x3208, 0x0e50,
 0x032c, 0x1d1c, 0x2940, 0x3308, 0x0f50,
 0x042c, 0x1c1c, 0x2840, 0x3408, 0x1050,
 0x052c, 0x1b1c, 0x2740, 0x3508, 0x1150,
 0x0530, 0x1a1c, 0x2640, 0x350c, 0x1154,
 0x0430, 0x191c, 0x2540, 0x340c, 0x1054,
 0x0330, 0x181c, 0x2440, 0x330c, 0x0f54,
 0x0230, 0x1820, 0x2444, 0x320c, 0x0e54,
 0x0130, 0x1920, 0x2544, 0x310c, 0x0d54,
 0x0030, 0x1a20, 0x2644, 0x300c, 0x0c54,
 0x0034, 0x1b20, 0x2744, 0x3010, 0x0c58,
 0x0134, 0x1c20, 0x2844, 0x3110, 0x0e58,
 0x0234, 0x1d20, 0x2944, 0x3210, 0x1058,
 0x0624, 0x2110, 0x2d34, 0x3600, 0x1248,
 0x0724, 0x2210, 0x2e34, 0x3700, 0x1348,
 0x0824, 0x2310, 0x2f34, 0x3800, 0x1448,
 0x0924, 0x2314, 0x2f38, 0x3900, 0x1548,
 0x0a24, 0x2214, 0x2e38, 0x3a00, 0x1648,
 0x0b24, 0x2114, 0x2d38, 0x3b00, 0x1748,
 0x0b28, 0x2014, 0x2c38, 0x3b04, 0x174c,
 0x0a28, 0x1f14, 0x2b38, 0x3a04, 0x164c,
 0x0928, 0x1e14, 0x2a38, 0x3904, 0x154c,
 0x0828, 0x1e18, 0x2a3c, 0x3804, 0x144c,
 0x0728, 0x1f18, 0x2b3c, 0x3704, 0x134c,
 0x0628, 0x2018, 0x2c3c, 0x3604, 0x124c,
 0x062c, 0x2118, 0x2d3c, 0x3608, 0x1250,
 0x072c, 0x2218, 0x2e3c, 0x3708, 0x1350,
 0x082c, 0x2318, 0x2f3c, 0x3808, 0x1450,
 0x092c, 0x231c, 0x2f40, 0x3908, 0x1550,
 0x0a2c, 0x221c, 0x2e40, 0x3a08, 0x1650,
 0x0b2c, 0x211c, 0x2d40, 0x3b08, 0x1750,
 0x0b30, 0x201c, 0x2c40, 0x3b0c, 0x1754,
 0x0a30, 0x1f1c, 0x2b40, 0x3a0c, 0x1654,
 0x0930, 0x1e1c, 0x2a40, 0x390c, 0x1554,
 0x0830, 0x1e20, 0x2a44, 0x380c, 0x1454,
 0x0730, 0x1f20, 0x2b44, 0x370c, 0x1354,
 0x0630, 0x2020, 0x2c44, 0x360c, 0x1254,
 0x0634, 0x2120, 0x2d44, 0x3610, 0x1258,
 0x0734, 0x2220, 0x2e44, 0x3710, 0x1458,
 0x0834, 0x2320, 0x2f44, 0x3810, 0x1658,
};

/* 4:2:2 macroblock placement tables created by dvtables.py */

/* 2 channels per frame, 10 DIF sequences per channel,
   27 video segments per DIF sequence, 5 macroblocks per video segment */
static const uint16_t dv_place_422_525[2*10*27*5] = {
 0x0c48, 0x2424, 0x306c, 0x0000, 0x1890,
 0x0d48, 0x2524, 0x316c, 0x0100, 0x1990,
 0x0e48, 0x2624, 0x326c, 0x0200, 0x1a90,
 0x0e4c, 0x2628, 0x3270, 0x0204, 0x1a94,
 0x0d4c, 0x2528, 0x3170, 0x0104, 0x1994,
 0x0c4c, 0x2428, 0x3070, 0x0004, 0x1894,
 0x0c50, 0x242c, 0x3074, 0x0008, 0x1898,
 0x0d50, 0x252c, 0x3174, 0x0108, 0x1998,
 0x0e50, 0x262c, 0x3274, 0x0208, 0x1a98,
 0x0e54, 0x2630, 0x3278, 0x020c, 0x1a9c,
 0x0d54, 0x2530, 0x3178, 0x010c, 0x199c,
 0x0c54, 0x2430, 0x3078, 0x000c, 0x189c,
 0x0c58, 0x2434, 0x307c, 0x0010, 0x18a0,
 0x0d58, 0x2534, 0x317c, 0x0110, 0x19a0,
 0x0e58, 0x2634, 0x327c, 0x0210, 0x1aa0,
 0x0e5c, 0x2638, 0x3280, 0x0214, 0x1aa4,
 0x0d5c, 0x2538, 0x3180, 0x0114, 0x19a4,
 0x0c5c, 0x2438, 0x3080, 0x0014, 0x18a4,
 0x0c60, 0x243c, 0x3084, 0x0018, 0x18a8,
 0x0d60, 0x253c, 0x3184, 0x0118, 0x19a8,
 0x0e60, 0x263c, 0x3284, 0x0218, 0x1aa8,
 0x0e64, 0x2640, 0x3288, 0x021c, 0x1aac,
 0x0d64, 0x2540, 0x3188, 0x011c, 0x19ac,
 0x0c64, 0x2440, 0x3088, 0x001c, 0x18ac,
 0x0c68, 0x2444, 0x308c, 0x0020, 0x18b0,
 0x0d68, 0x2544, 0x318c, 0x0120, 0x19b0,
 0x0e68, 0x2644, 0x328c, 0x0220, 0x1ab0,
 0x1248, 0x2a24, 0x366c, 0x0600, 0x1e90,
 0x1348, 0x2b24, 0x376c, 0x0700, 0x1f90,
 0x1448, 0x2c24, 0x386c, 0x0800, 0x2090,
 0x144c, 0x2c28, 0x3870, 0x0804, 0x2094,
 0x134c, 0x2b28, 0x3770, 0x0704, 0x1f94,
 0x124c, 0x2a28, 0x3670, 0x0604, 0x1e94,
 0x1250, 0x2a2c, 0x3674, 0x0608, 0x1e98,
 0x1350, 0x2b2c, 0x3774, 0x0708, 0x1f98,
 0x1450, 0x2c2c, 0x3874, 0x0808, 0x2098,
 0x1454, 0x2c30, 0x3878, 0x080c, 0x209c,
 0x1354, 0x2b30, 0x3778, 0x070c, 0x1f9c,
 0x1254, 0x2a30, 0x3678, 0x060c, 0x1e9c,
 0x1258, 0x2a34, 0x367c, 0x0610, 0x1ea0,
 0x1358, 0x2b34, 0x377c, 0x0710, 0x1fa0,
 0x1458, 0x2c34, 0x387c, 0x0810, 0x20a0,
 0x145c, 0x2c38, 0x3880, 0x0814, 0x20a4,
 0x135c, 0x2b38, 0x3780, 0x0714, 0x1fa4,
 0x125c, 0x2a38, 0x3680, 0x0614, 0x1ea4,
 0x1260, 0x2a3c, 0x3684, 0x0618, 0x1ea8,
 0x1360, 0x2b3c, 0x3784, 0x0718, 0x1fa8,
 0x1460, 0x2c3c, 0x3884, 0x0818, 0x20a8,
 0x1464, 0x2c40, 0x3888, 0x081c, 0x20ac,
 0x1364, 0x2b40, 0x3788, 0x071c, 0x1fac,
 0x1264, 0x2a40, 0x3688, 0x061c, 0x1eac,
 0x1268, 0x2a44, 0x368c, 0x0620, 0x1eb0,
 0x1368, 0x2b44, 0x378c, 0x0720, 0x1fb0,
 0x1468, 0x2c44, 0x388c, 0x0820, 0x20b0,
 0x1848, 0x3024, 0x006c, 0x0c00, 0x2490,
 0x1948, 0x3124, 0x016c, 0x0d00, 0x2590,
 0x1a48, 0x3224, 0x026c, 0x0e00, 0x2690,
 0x1a4c, 0x3228, 0x0270, 0x0e04, 0x2694,
 0x194c, 0x3128, 0x0170, 0x0d04, 0x2594,
 0x184c, 0x3028, 0x0070, 0x0c04, 0x2494,
 0x1850, 0x302c, 0x0074, 0x0c08, 0x2498,
 0x1950, 0x312c, 0x0174, 0x0d08, 0x2598,
 0x1a50, 0x322c, 0x0274, 0x0e08, 0x2698,
 0x1a54, 0x3230, 0x0278, 0x0e0c, 0x269c,
 0x1954, 0x3130, 0x0178, 0x0d0c, 0x259c,
 0x1854, 0x3030, 0x0078, 0x0c0c, 0x249c,
 0x1858, 0x3034, 0x007c, 0x0c10, 0x24a0,
 0x1958, 0x3134, 0x017c, 0x0d10, 0x25a0,
 0x1a58, 0x3234, 0x027c, 0x0e10, 0x26a0,
 0x1a5c, 0x3238, 0x0280, 0x0e14, 0x26a4,
 0x195c, 0x3138, 0x0180, 0x0d14, 0x25a4,
 0x185c, 0x3038, 0x0080, 0x0c14, 0x24a4,
 0x1860, 0x303c, 0x0084, 0x0c18, 0x24a8,
 0x1960, 0x313c, 0x0184, 0x0d18, 0x25a8,
 0x1a60, 0x323c, 0x0284, 0x0e18, 0x26a8,
 0x1a64, 0x3240, 0x0288, 0x0e1c, 0x26ac,
 0x1964, 0x3140, 0x0188, 0x0d1c, 0x25ac,
 0x1864, 0x3040, 0x0088, 0x0c1c, 0x24ac,
 0x1868, 0x3044, 0x008c, 0x0c20, 0x24b0,
 0x1968, 0x3144, 0x018c, 0x0d20, 0x25b0,
 0x1a68, 0x3244, 0x028c, 0x0e20, 0x26b0,
 0x1e48, 0x3624, 0x066c, 0x1200, 0x2a90,
 0x1f48, 0x3724, 0x076c, 0x1300, 0x2b90,
 0x2048, 0x3824, 0x086c, 0x1400, 0x2c90,
 0x204c, 0x3828, 0x0870, 0x1404, 0x2c94,
 0x1f4c, 0x3728, 0x0770, 0x1304, 0x2b94,
 0x1e4c, 0x3628, 0x0670, 0x1204, 0x2a94,
 0x1e50, 0x362c, 0x0674, 0x1208, 0x2a98,
 0x1f50, 0x372c, 0x0774, 0x1308, 0x2b98,
 0x2050, 0x382c, 0x0874, 0x1408, 0x2c98,
 0x2054, 0x3830, 0x0878, 0x140c, 0x2c9c,
 0x1f54, 0x3730, 0x0778, 0x130c, 0x2b9c,
 0x1e54, 0x3630, 0x0678, 0x120c, 0x2a9c,
 0x1e58, 0x3634, 0x067c, 0x1210, 0x2aa0,
 0x1f58, 0x3734, 0x077c, 0x1310, 0x2ba0,
 0x2058, 0x3834, 0x087c, 0x1410, 0x2ca0,
 0x205c, 0x3838, 0x0880, 0x1414, 0x2ca4,
 0x1f5c, 0x3738, 0x0780, 0x1314, 0x2ba4,
 0x1e5c, 0x3638, 0x0680, 0x1214, 0x2aa4,
 0x1e60, 0x363c, 0x0684, 0x1218, 0x2aa8,
 0x1f60, 0x373c, 0x0784, 0x1318, 0x2ba8,
 0x2060, 0x383c, 0x0884, 0x1418, 0x2ca8,
 0x2064, 0x3840, 0x0888, 0x141c, 0x2cac,
 0x1f64, 0x3740, 0x0788, 0x131c, 0x2bac,
 0x1e64, 0x3640, 0x0688, 0x121c, 0x2aac,
 0x1e68, 0x3644, 0x068c, 0x1220, 0x2ab0,
 0x1f68, 0x3744, 0x078c, 0x1320, 0x2bb0,
 0x2068, 0x3844, 0x088c, 0x1420, 0x2cb0,
 0x2448, 0x0024, 0x0c6c, 0x1800, 0x3090,
 0x2548, 0x0124, 0x0d6c, 0x1900, 0x3190,
 0x2648, 0x0224, 0x0e6c, 0x1a00, 0x3290,
 0x264c, 0x0228, 0x0e70, 0x1a04, 0x3294,
 0x254c, 0x0128, 0x0d70, 0x1904, 0x3194,
 0x244c, 0x0028, 0x0c70, 0x1804, 0x3094,
 0x2450, 0x002c, 0x0c74, 0x1808, 0x3098,
 0x2550, 0x012c, 0x0d74, 0x1908, 0x3198,
 0x2650, 0x022c, 0x0e74, 0x1a08, 0x3298,
 0x2654, 0x0230, 0x0e78, 0x1a0c, 0x329c,
 0x2554, 0x0130, 0x0d78, 0x190c, 0x319c,
 0x2454, 0x0030, 0x0c78, 0x180c, 0x309c,
 0x2458, 0x0034, 0x0c7c, 0x1810, 0x30a0,
 0x2558, 0x0134, 0x0d7c, 0x1910, 0x31a0,
 0x2658, 0x0234, 0x0e7c, 0x1a10, 0x32a0,
 0x265c, 0x0238, 0x0e80, 0x1a14, 0x32a4,
 0x255c, 0x0138, 0x0d80, 0x1914, 0x31a4,
 0x245c, 0x0038, 0x0c80, 0x1814, 0x30a4,
 0x2460, 0x003c, 0x0c84, 0x1818, 0x30a8,
 0x2560, 0x013c, 0x0d84, 0x1918, 0x31a8,
 0x2660, 0x023c, 0x0e84, 0x1a18, 0x32a8,
 0x2664, 0x0240, 0x0e88, 0x1a1c, 0x32ac,
 0x2564, 0x0140, 0x0d88, 0x191c, 0x31ac,
 0x2464, 0x0040, 0x0c88, 0x181c, 0x30ac,
 0x2468, 0x0044, 0x0c8c, 0x1820, 0x30b0,
 0x2568, 0x0144, 0x0d8c, 0x1920, 0x31b0,
 0x2668, 0x0244, 0x0e8c, 0x1a20, 0x32b0,
 0x2a48, 0x0624, 0x126c, 0x1e00, 0x3690,
 0x2b48, 0x0724, 0x136c, 0x1f00, 0x3790,
 0x2c48, 0x0824, 0x146c, 0x2000, 0x3890,
 0x2c4c, 0x0828, 0x1470, 0x2004, 0x3894,
 0x2b4c, 0x0728, 0x1370, 0x1f04, 0x3794,
 0x2a4c, 0x0628, 0x1270, 0x1e04, 0x3694,
 0x2a50, 0x062c, 0x1274, 0x1e08, 0x3698,
 0x2b50, 0x072c, 0x1374, 0x1f08, 0x3798,
 0x2c50, 0x082c, 0x1474, 0x2008, 0x3898,
 0x2c54, 0x0830, 0x1478, 0x200c, 0x389c,
 0x2b54, 0x0730, 0x1378, 0x1f0c, 0x379c,
 0x2a54, 0x0630, 0x1278, 0x1e0c, 0x369c,
 0x2a58, 0x0634, 0x127c, 0x1e10, 0x36a0,
 0x2b58, 0x0734, 0x137c, 0x1f10, 0x37a0,
 0x2c58, 0x0834, 0x147c, 0x2010, 0x38a0,
 0x2c5c, 0x0838, 0x1480, 0x2014, 0x38a4,
 0x2b5c, 0x0738, 0x1380, 0x1f14, 0x37a4,
 0x2a5c, 0x0638, 0x1280, 0x1e14, 0x36a4,
 0x2a60, 0x063c, 0x1284, 0x1e18, 0x36a8,
 0x2b60, 0x073c, 0x1384, 0x1f18, 0x37a8,
 0x2c60, 0x083c, 0x1484, 0x2018, 0x38a8,
 0x2c64, 0x0840, 0x1488, 0x201c, 0x38ac,
 0x2b64, 0x0740, 0x1388, 0x1f1c, 0x37ac,
 0x2a64, 0x0640, 0x1288, 0x1e1c, 0x36ac,
 0x2a68, 0x0644, 0x128c, 0x1e20, 0x36b0,
 0x2b68, 0x0744, 0x138c, 0x1f20, 0x37b0,
 0x2c68, 0x0844, 0x148c, 0x2020, 0x38b0,
 0x3048, 0x0c24, 0x186c, 0x2400, 0x0090,
 0x3148, 0x0d24, 0x196c, 0x2500, 0x0190,
 0x3248, 0x0e24, 0x1a6c, 0x2600, 0x0290,
 0x324c, 0x0e28, 0x1a70, 0x2604, 0x0294,
 0x314c, 0x0d28, 0x1970, 0x2504, 0x0194,
 0x304c, 0x0c28, 0x1870, 0x2404, 0x0094,
 0x3050, 0x0c2c, 0x1874, 0x2408, 0x0098,
 0x3150, 0x0d2c, 0x1974, 0x2508, 0x0198,
 0x3250, 0x0e2c, 0x1a74, 0x2608, 0x0298,
 0x3254, 0x0e30, 0x1a78, 0x260c, 0x029c,
 0x3154, 0x0d30, 0x1978, 0x250c, 0x019c,
 0x3054, 0x0c30, 0x1878, 0x240c, 0x009c,
 0x3058, 0x0c34, 0x187c, 0x2410, 0x00a0,
 0x3158, 0x0d34, 0x197c, 0x2510, 0x01a0,
 0x3258, 0x0e34, 0x1a7c, 0x2610, 0x02a0,
 0x325c, 0x0e38, 0x1a80, 0x2614, 0x02a4,
 0x315c, 0x0d38, 0x1980, 0x2514, 0x01a4,
 0x305c, 0x0c38, 0x1880, 0x2414, 0x00a4,
 0x3060, 0x0c3c, 0x1884, 0x2418, 0x00a8,
 0x3160, 0x0d3c, 0x1984, 0x2518, 0x01a8,
 0x3260, 0x0e3c, 0x1a84, 0x2618, 0x02a8,
 0x3264, 0x0e40, 0x1a88, 0x261c, 0x02ac,
 0x3164, 0x0d40, 0x1988, 0x251c, 0x01ac,
 0x3064, 0x0c40, 0x1888, 0x241c, 0x00ac,
 0x3068, 0x0c44, 0x188c, 0x2420, 0x00b0,
 0x3168, 0x0d44, 0x198c, 0x2520, 0x01b0,
 0x3268, 0x0e44, 0x1a8c, 0x2620, 0x02b0,
 0x3648, 0x1224, 0x1e6c, 0x2a00, 0x0690,
 0x3748, 0x1324, 0x1f6c, 0x2b00, 0x0790,
 0x3848, 0x1424, 0x206c, 0x2c00, 0x0890,
 0x384c, 0x1428, 0x2070, 0x2c04, 0x0894,
 0x374c, 0x1328, 0x1f70, 0x2b04, 0x0794,
 0x364c, 0x1228, 0x1e70, 0x2a04, 0x0694,
 0x3650, 0x122c, 0x1e74, 0x2a08, 0x0698,
 0x3750, 0x132c, 0x1f74, 0x2b08, 0x0798,
 0x3850, 0x142c, 0x2074, 0x2c08, 0x0898,
 0x3854, 0x1430, 0x2078, 0x2c0c, 0x089c,
 0x3754, 0x1330, 0x1f78, 0x2b0c, 0x079c,
 0x3654, 0x1230, 0x1e78, 0x2a0c, 0x069c,
 0x3658, 0x1234, 0x1e7c, 0x2a10, 0x06a0,
 0x3758, 0x1334, 0x1f7c, 0x2b10, 0x07a0,
 0x3858, 0x1434, 0x207c, 0x2c10, 0x08a0,
 0x385c, 0x1438, 0x2080, 0x2c14, 0x08a4,
 0x375c, 0x1338, 0x1f80, 0x2b14, 0x07a4,
 0x365c, 0x1238, 0x1e80, 0x2a14, 0x06a4,
 0x3660, 0x123c, 0x1e84, 0x2a18, 0x06a8,
 0x3760, 0x133c, 0x1f84, 0x2b18, 0x07a8,
 0x3860, 0x143c, 0x2084, 0x2c18, 0x08a8,
 0x3864, 0x1440, 0x2088, 0x2c1c, 0x08ac,
 0x3764, 0x1340, 0x1f88, 0x2b1c, 0x07ac,
 0x3664, 0x1240, 0x1e88, 0x2a1c, 0x06ac,
 0x3668, 0x1244, 0x1e8c, 0x2a20, 0x06b0,
 0x3768, 0x1344, 0x1f8c, 0x2b20, 0x07b0,
 0x3868, 0x1444, 0x208c, 0x2c20, 0x08b0,
 0x0048, 0x1824, 0x246c, 0x3000, 0x0c90,
 0x0148, 0x1924, 0x256c, 0x3100, 0x0d90,
 0x0248, 0x1a24, 0x266c, 0x3200, 0x0e90,
 0x024c, 0x1a28, 0x2670, 0x3204, 0x0e94,
 0x014c, 0x1928, 0x2570, 0x3104, 0x0d94,
 0x004c, 0x1828, 0x2470, 0x3004, 0x0c94,
 0x0050, 0x182c, 0x2474, 0x3008, 0x0c98,
 0x0150, 0x192c, 0x2574, 0x3108, 0x0d98,
 0x0250, 0x1a2c, 0x2674, 0x3208, 0x0e98,
 0x0254, 0x1a30, 0x2678, 0x320c, 0x0e9c,
 0x0154, 0x1930, 0x2578, 0x310c, 0x0d9c,
 0x0054, 0x1830, 0x2478, 0x300c, 0x0c9c,
 0x0058, 0x1834, 0x247c, 0x3010, 0x0ca0,
 0x0158, 0x1934, 0x257c, 0x3110, 0x0da0,
 0x0258, 0x1a34, 0x267c, 0x3210, 0x0ea0,
 0x025c, 0x1a38, 0x2680, 0x3214, 0x0ea4,
 0x015c, 0x1938, 0x2580, 0x3114, 0x0da4,
 0x005c, 0x1838, 0x2480, 0x3014, 0x0ca4,
 0x0060, 0x183c, 0x2484, 0x3018, 0x0ca8,
 0x0160, 0x193c, 0x2584, 0x3118, 0x0da8,
 0x0260, 0x1a3c, 0x2684, 0x3218, 0x0ea8,
 0x0264, 0x1a40, 0x2688, 0x321c, 0x0eac,
 0x0164, 0x1940, 0x2588, 0x311c, 0x0dac,
 0x0064, 0x1840, 0x2488, 0x301c, 0x0cac,
 0x0068, 0x1844, 0x248c, 0x3020, 0x0cb0,
 0x0168, 0x1944, 0x258c, 0x3120, 0x0db0,
 0x0268, 0x1a44, 0x268c, 0x3220, 0x0eb0,
 0x0648, 0x1e24, 0x2a6c, 0x3600, 0x1290,
 0x0748, 0x1f24, 0x2b6c, 0x3700, 0x1390,
 0x0848, 0x2024, 0x2c6c, 0x3800, 0x1490,
 0x084c, 0x2028, 0x2c70, 0x3804, 0x1494,
 0x074c, 0x1f28, 0x2b70, 0x3704, 0x1394,
 0x064c, 0x1e28, 0x2a70, 0x3604, 0x1294,
 0x0650, 0x1e2c, 0x2a74, 0x3608, 0x1298,
 0x0750, 0x1f2c, 0x2b74, 0x3708, 0x1398,
 0x0850, 0x202c, 0x2c74, 0x3808, 0x1498,
 0x0854, 0x2030, 0x2c78, 0x380c, 0x149c,
 0x0754, 0x1f30, 0x2b78, 0x370c, 0x139c,
 0x0654, 0x1e30, 0x2a78, 0x360c, 0x129c,
 0x0658, 0x1e34, 0x2a7c, 0x3610, 0x12a0,
 0x0758, 0x1f34, 0x2b7c, 0x3710, 0x13a0,
 0x0858, 0x2034, 0x2c7c, 0x3810, 0x14a0,
 0x085c, 0x2038, 0x2c80, 0x3814, 0x14a4,
 0x075c, 0x1f38, 0x2b80, 0x3714, 0x13a4,
 0x065c, 0x1e38, 0x2a80, 0x3614, 0x12a4,
 0x0660, 0x1e3c, 0x2a84, 0x3618, 0x12a8,
 0x0760, 0x1f3c, 0x2b84, 0x3718, 0x13a8,
 0x0860, 0x203c, 0x2c84, 0x3818, 0x14a8,
 0x0864, 0x2040, 0x2c88, 0x381c, 0x14ac,
 0x0764, 0x1f40, 0x2b88, 0x371c, 0x13ac,
 0x0664, 0x1e40, 0x2a88, 0x361c, 0x12ac,
 0x0668, 0x1e44, 0x2a8c, 0x3620, 0x12b0,
 0x0768, 0x1f44, 0x2b8c, 0x3720, 0x13b0,
 0x0868, 0x2044, 0x2c8c, 0x3820, 0x14b0,
 0x0f48, 0x2724, 0x336c, 0x0300, 0x1b90,
 0x1048, 0x2824, 0x346c, 0x0400, 0x1c90,
 0x1148, 0x2924, 0x356c, 0x0500, 0x1d90,
 0x114c, 0x2928, 0x3570, 0x0504, 0x1d94,
 0x104c, 0x2828, 0x3470, 0x0404, 0x1c94,
 0x0f4c, 0x2728, 0x3370, 0x0304, 0x1b94,
 0x0f50, 0x272c, 0x3374, 0x0308, 0x1b98,
 0x1050, 0x282c, 0x3474, 0x0408, 0x1c98,
 0x1150, 0x292c, 0x3574, 0x0508, 0x1d98,
 0x1154, 0x2930, 0x3578, 0x050c, 0x1d9c,
 0x1054, 0x2830, 0x3478, 0x040c, 0x1c9c,
 0x0f54, 0x2730, 0x3378, 0x030c, 0x1b9c,
 0x0f58, 0x2734, 0x337c, 0x0310, 0x1ba0,
 0x1058, 0x2834, 0x347c, 0x0410, 0x1ca0,
 0x1158, 0x2934, 0x357c, 0x0510, 0x1da0,
 0x115c, 0x2938, 0x3580, 0x0514, 0x1da4,
 0x105c, 0x2838, 0x3480, 0x0414, 0x1ca4,
 0x0f5c, 0x2738, 0x3380, 0x0314, 0x1ba4,
 0x0f60, 0x273c, 0x3384, 0x0318, 0x1ba8,
 0x1060, 0x283c, 0x3484, 0x0418, 0x1ca8,
 0x1160, 0x293c, 0x3584, 0x0518, 0x1da8,
 0x1164, 0x2940, 0x3588, 0x051c, 0x1dac,
 0x1064, 0x2840, 0x3488, 0x041c, 0x1cac,
 0x0f64, 0x2740, 0x3388, 0x031c, 0x1bac,
 0x0f68, 0x2744, 0x338c, 0x0320, 0x1bb0,
 0x1068, 0x2844, 0x348c, 0x0420, 0x1cb0,
 0x1168, 0x2944, 0x358c, 0x0520, 0x1db0,
 0x1548, 0x2d24, 0x396c, 0x0900, 0x2190,
 0x1648, 0x2e24, 0x3a6c, 0x0a00, 0x2290,
 0x1748, 0x2f24, 0x3b6c, 0x0b00, 0x2390,
 0x174c, 0x2f28, 0x3b70, 0x0b04, 0x2394,
 0x164c, 0x2e28, 0x3a70, 0x0a04, 0x2294,
 0x154c, 0x2d28, 0x3970, 0x0904, 0x2194,
 0x1550, 0x2d2c, 0x3974, 0x0908, 0x2198,
 0x1650, 0x2e2c, 0x3a74, 0x0a08, 0x2298,
 0x1750, 0x2f2c, 0x3b74, 0x0b08, 0x2398,
 0x1754, 0x2f30, 0x3b78, 0x0b0c, 0x239c,
 0x1654, 0x2e30, 0x3a78, 0x0a0c, 0x229c,
 0x1554, 0x2d30, 0x3978, 0x090c, 0x219c,
 0x1558, 0x2d34, 0x397c, 0x0910, 0x21a0,
 0x1658, 0x2e34, 0x3a7c, 0x0a10, 0x22a0,
 0x1758, 0x2f34, 0x3b7c, 0x0b10, 0x23a0,
 0x175c, 0x2f38, 0x3b80, 0x0b14, 0x23a4,
 0x165c, 0x2e38, 0x3a80, 0x0a14, 0x22a4,
 0x155c, 0x2d38, 0x3980, 0x0914, 0x21a4,
 0x1560, 0x2d3c, 0x3984, 0x0918, 0x21a8,
 0x1660, 0x2e3c, 0x3a84, 0x0a18, 0x22a8,
 0x1760, 0x2f3c, 0x3b84, 0x0b18, 0x23a8,
 0x1764, 0x2f40, 0x3b88, 0x0b1c, 0x23ac,
 0x1664, 0x2e40, 0x3a88, 0x0a1c, 0x22ac,
 0x1564, 0x2d40, 0x3988, 0x091c, 0x21ac,
 0x1568, 0x2d44, 0x398c, 0x0920, 0x21b0,
 0x1668, 0x2e44, 0x3a8c, 0x0a20, 0x22b0,
 0x1768, 0x2f44, 0x3b8c, 0x0b20, 0x23b0,
 0x1b48, 0x3324, 0x036c, 0x0f00, 0x2790,
 0x1c48, 0x3424, 0x046c, 0x1000, 0x2890,
 0x1d48, 0x3524, 0x056c, 0x1100, 0x2990,
 0x1d4c, 0x3528, 0x0570, 0x1104, 0x2994,
 0x1c4c, 0x3428, 0x0470, 0x1004, 0x2894,
 0x1b4c, 0x3328, 0x0370, 0x0f04, 0x2794,
 0x1b50, 0x332c, 0x0374, 0x0f08, 0x2798,
 0x1c50, 0x342c, 0x0474, 0x1008, 0x2898,
 0x1d50, 0x352c, 0x0574, 0x1108, 0x2998,
 0x1d54, 0x3530, 0x0578, 0x110c, 0x299c,
 0x1c54, 0x3430, 0x0478, 0x100c, 0x289c,
 0x1b54, 0x3330, 0x0378, 0x0f0c, 0x279c,
 0x1b58, 0x3334, 0x037c, 0x0f10, 0x27a0,
 0x1c58, 0x3434, 0x047c, 0x1010, 0x28a0,
 0x1d58, 0x3534, 0x057c, 0x1110, 0x29a0,
 0x1d5c, 0x3538, 0x0580, 0x1114, 0x29a4,
 0x1c5c, 0x3438, 0x0480, 0x1014, 0x28a4,
 0x1b5c, 0x3338, 0x0380, 0x0f14, 0x27a4,
 0x1b60, 0x333c, 0x0384, 0x0f18, 0x27a8,
 0x1c60, 0x343c, 0x0484, 0x1018, 0x28a8,
 0x1d60, 0x353c, 0x0584, 0x1118, 0x29a8,
 0x1d64, 0x3540, 0x0588, 0x111c, 0x29ac,
 0x1c64, 0x3440, 0x0488, 0x101c, 0x28ac,
 0x1b64, 0x3340, 0x0388, 0x0f1c, 0x27ac,
 0x1b68, 0x3344, 0x038c, 0x0f20, 0x27b0,
 0x1c68, 0x3444, 0x048c, 0x1020, 0x28b0,
 0x1d68, 0x3544, 0x058c, 0x1120, 0x29b0,
 0x2148, 0x3924, 0x096c, 0x1500, 0x2d90,
 0x2248, 0x3a24, 0x0a6c, 0x1600, 0x2e90,
 0x2348, 0x3b24, 0x0b6c, 0x1700, 0x2f90,
 0x234c, 0x3b28, 0x0b70, 0x1704, 0x2f94,
 0x224c, 0x3a28, 0x0a70, 0x1604, 0x2e94,
 0x214c, 0x3928, 0x0970, 0x1504, 0x2d94,
 0x2150, 0x392c, 0x0974, 0x1508, 0x2d98,
 0x2250, 0x3a2c, 0x0a74, 0x1608, 0x2e98,
 0x2350, 0x3b2c, 0x0b74, 0x1708, 0x2f98,
 0x2354, 0x3b30, 0x0b78, 0x170c, 0x2f9c,
 0x2254, 0x3a30, 0x0a78, 0x160c, 0x2e9c,
 0x2154, 0x3930, 0x0978, 0x150c, 0x2d9c,
 0x2158, 0x3934, 0x097c, 0x1510, 0x2da0,
 0x2258, 0x3a34, 0x0a7c, 0x1610, 0x2ea0,
 0x2358, 0x3b34, 0x0b7c, 0x1710, 0x2fa0,
 0x235c, 0x3b38, 0x0b80, 0x1714, 0x2fa4,
 0x225c, 0x3a38, 0x0a80, 0x1614, 0x2ea4,
 0x215c, 0x3938, 0x0980, 0x1514, 0x2da4,
 0x2160, 0x393c, 0x0984, 0x1518, 0x2da8,
 0x2260, 0x3a3c, 0x0a84, 0x1618, 0x2ea8,
 0x2360, 0x3b3c, 0x0b84, 0x1718, 0x2fa8,
 0x2364, 0x3b40, 0x0b88, 0x171c, 0x2fac,
 0x2264, 0x3a40, 0x0a88, 0x161c, 0x2eac,
 0x2164, 0x3940, 0x0988, 0x151c, 0x2dac,
 0x2168, 0x3944, 0x098c, 0x1520, 0x2db0,
 0x2268, 0x3a44, 0x0a8c, 0x1620, 0x2eb0,
 0x2368, 0x3b44, 0x0b8c, 0x1720, 0x2fb0,
 0x2748, 0x0324, 0x0f6c, 0x1b00, 0x3390,
 0x2848, 0x0424, 0x106c, 0x1c00, 0x3490,
 0x2948, 0x0524, 0x116c, 0x1d00, 0x3590,
 0x294c, 0x0528, 0x1170, 0x1d04, 0x3594,
 0x284c, 0x0428, 0x1070, 0x1c04, 0x3494,
 0x274c, 0x0328, 0x0f70, 0x1b04, 0x3394,
 0x2750, 0x032c, 0x0f74, 0x1b08, 0x3398,
 0x2850, 0x042c, 0x1074, 0x1c08, 0x3498,
 0x2950, 0x052c, 0x1174, 0x1d08, 0x3598,
 0x2954, 0x0530, 0x1178, 0x1d0c, 0x359c,
 0x2854, 0x0430, 0x1078, 0x1c0c, 0x349c,
 0x2754, 0x0330, 0x0f78, 0x1b0c, 0x339c,
 0x2758, 0x0334, 0x0f7c, 0x1b10, 0x33a0,
 0x2858, 0x0434, 0x107c, 0x1c10, 0x34a0,
 0x2958, 0x0534, 0x117c, 0x1d10, 0x35a0,
 0x295c, 0x0538, 0x1180, 0x1d14, 0x35a4,
 0x285c, 0x0438, 0x1080, 0x1c14, 0x34a4,
 0x275c, 0x0338, 0x0f80, 0x1b14, 0x33a4,
 0x2760, 0x033c, 0x0f84, 0x1b18, 0x33a8,
 0x2860, 0x043c, 0x1084, 0x1c18, 0x34a8,
 0x2960, 0x053c, 0x1184, 0x1d18, 0x35a8,
 0x2964, 0x0540, 0x1188, 0x1d1c, 0x35ac,
 0x2864, 0x0440, 0x1088, 0x1c1c, 0x34ac,
 0x2764, 0x0340, 0x0f88, 0x1b1c, 0x33ac,
 0x2768, 0x0344, 0x0f8c, 0x1b20, 0x33b0,
 0x2868, 0x0444, 0x108c, 0x1c20, 0x34b0,
 0x2968, 0x0544, 0x118c, 0x1d20, 0x35b0,
 0x2d48, 0x0924, 0x156c, 0x2100, 0x3990,
 0x2e48, 0x0a24, 0x166c, 0x2200, 0x3a90,
 0x2f48, 0x0b24, 0x176c, 0x2300, 0x3b90,
 0x2f4c, 0x0b28, 0x1770, 0x2304, 0x3b94,
 0x2e4c, 0x0a28, 0x1670, 0x2204, 0x3a94,
 0x2d4c, 0x0928, 0x1570, 0x2104, 0x3994,
 0x2d50, 0x092c, 0x1574, 0x2108, 0x3998,
 0x2e50, 0x0a2c, 0x1674, 0x2208, 0x3a98,
 0x2f50, 0x0b2c, 0x1774, 0x2308, 0x3b98,
 0x2f54, 0x0b30, 0x1778, 0x230c, 0x3b9c,
 0x2e54, 0x0a30, 0x1678, 0x220c, 0x3a9c,
 0x2d54, 0x0930, 0x1578, 0x210c, 0x399c,
 0x2d58, 0x0934, 0x157c, 0x2110, 0x39a0,
 0x2e58, 0x0a34, 0x167c, 0x2210, 0x3aa0,
 0x2f58, 0x0b34, 0x177c, 0x2310, 0x3ba0,
 0x2f5c, 0x0b38, 0x1780, 0x2314, 0x3ba4,
 0x2e5c, 0x0a38, 0x1680, 0x2214, 0x3aa4,
 0x2d5c, 0x0938, 0x1580, 0x2114, 0x39a4,
 0x2d60, 0x093c, 0x1584, 0x2118, 0x39a8,
 0x2e60, 0x0a3c, 0x1684, 0x2218, 0x3aa8,
 0x2f60, 0x0b3c, 0x1784, 0x2318, 0x3ba8,
 0x2f64, 0x0b40, 0x1788, 0x231c, 0x3bac,
 0x2e64, 0x0a40, 0x1688, 0x221c, 0x3aac,
 0x2d64, 0x0940, 0x1588, 0x211c, 0x39ac,
 0x2d68, 0x0944, 0x158c, 0x2120, 0x39b0,
 0x2e68, 0x0a44, 0x168c, 0x2220, 0x3ab0,
 0x2f68, 0x0b44, 0x178c, 0x2320, 0x3bb0,
 0x3348, 0x0f24, 0x1b6c, 0x2700, 0x0390,
 0x3448, 0x1024, 0x1c6c, 0x2800, 0x0490,
 0x3548, 0x1124, 0x1d6c, 0x2900, 0x0590,
 0x354c, 0x1128, 0x1d70, 0x2904, 0x0594,
 0x344c, 0x1028, 0x1c70, 0x2804, 0x0494,
 0x334c, 0x0f28, 0x1b70, 0x2704, 0x0394,
 0x3350, 0x0f2c, 0x1b74, 0x2708, 0x0398,
 0x3450, 0x102c, 0x1c74, 0x2808, 0x0498,
 0x3550, 0x112c, 0x1d74, 0x2908, 0x0598,
 0x3554, 0x1130, 0x1d78, 0x290c, 0x059c,
 0x3454, 0x1030, 0x1c78, 0x280c, 0x049c,
 0x3354, 0x0f30, 0x1b78, 0x270c, 0x039c,
 0x3358, 0x0f34, 0x1b7c, 0x2710, 0x03a0,
 0x3458, 0x1034, 0x1c7c, 0x2810, 0x04a0,
 0x3558, 0x1134, 0x1d7c, 0x2910, 0x05a0,
 0x355c, 0x1138, 0x1d80, 0x2914, 0x05a4,
 0x345c, 0x1038, 0x1c80, 0x2814, 0x04a4,
 0x335c, 0x0f38, 0x1b80, 0x2714, 0x03a4,
 0x3360, 0x0f3c, 0x1b84, 0x2718, 0x03a8,
 0x3460, 0x103c, 0x1c84, 0x2818, 0x04a8,
 0x3560, 0x113c, 0x1d84, 0x2918, 0x05a8,
 0x3564, 0x1140, 0x1d88, 0x291c, 0x05ac,
 0x3464, 0x1040, 0x1c88, 0x281c, 0x04ac,
 0x3364, 0x0f40, 0x1b88, 0x271c, 0x03ac,
 0x3368, 0x0f44, 0x1b8c, 0x2720, 0x03b0,
 0x3468, 0x1044, 0x1c8c, 0x2820, 0x04b0,
 0x3568, 0x1144, 0x1d8c, 0x2920, 0x05b0,
 0x3948, 0x1524, 0x216c, 0x2d00, 0x0990,
 0x3a48, 0x1624, 0x226c, 0x2e00, 0x0a90,
 0x3b48, 0x1724, 0x236c, 0x2f00, 0x0b90,
 0x3b4c, 0x1728, 0x2370, 0x2f04, 0x0b94,
 0x3a4c, 0x1628, 0x2270, 0x2e04, 0x0a94,
 0x394c, 0x1528, 0x2170, 0x2d04, 0x0994,
 0x3950, 0x152c, 0x2174, 0x2d08, 0x0998,
 0x3a50, 0x162c, 0x2274, 0x2e08, 0x0a98,
 0x3b50, 0x172c, 0x2374, 0x2f08, 0x0b98,
 0x3b54, 0x1730, 0x2378, 0x2f0c, 0x0b9c,
 0x3a54, 0x1630, 0x2278, 0x2e0c, 0x0a9c,
 0x3954, 0x1530, 0x2178, 0x2d0c, 0x099c,
 0x3958, 0x1534, 0x217c, 0x2d10, 0x09a0,
 0x3a58, 0x1634, 0x227c, 0x2e10, 0x0aa0,
 0x3b58, 0x1734, 0x237c, 0x2f10, 0x0ba0,
 0x3b5c, 0x1738, 0x2380, 0x2f14, 0x0ba4,
 0x3a5c, 0x1638, 0x2280, 0x2e14, 0x0aa4,
 0x395c, 0x1538, 0x2180, 0x2d14, 0x09a4,
 0x3960, 0x153c, 0x2184, 0x2d18, 0x09a8,
 0x3a60, 0x163c, 0x2284, 0x2e18, 0x0aa8,
 0x3b60, 0x173c, 0x2384, 0x2f18, 0x0ba8,
 0x3b64, 0x1740, 0x2388, 0x2f1c, 0x0bac,
 0x3a64, 0x1640, 0x2288, 0x2e1c, 0x0aac,
 0x3964, 0x1540, 0x2188, 0x2d1c, 0x09ac,
 0x3968, 0x1544, 0x218c, 0x2d20, 0x09b0,
 0x3a68, 0x1644, 0x228c, 0x2e20, 0x0ab0,
 0x3b68, 0x1744, 0x238c, 0x2f20, 0x0bb0,
 0x0348, 0x1b24, 0x276c, 0x3300, 0x0f90,
 0x0448, 0x1c24, 0x286c, 0x3400, 0x1090,
 0x0548, 0x1d24, 0x296c, 0x3500, 0x1190,
 0x054c, 0x1d28, 0x2970, 0x3504, 0x1194,
 0x044c, 0x1c28, 0x2870, 0x3404, 0x1094,
 0x034c, 0x1b28, 0x2770, 0x3304, 0x0f94,
 0x0350, 0x1b2c, 0x2774, 0x3308, 0x0f98,
 0x0450, 0x1c2c, 0x2874, 0x3408, 0x1098,
 0x0550, 0x1d2c, 0x2974, 0x3508, 0x1198,
 0x0554, 0x1d30, 0x2978, 0x350c, 0x119c,
 0x0454, 0x1c30, 0x2878, 0x340c, 0x109c,
 0x0354, 0x1b30, 0x2778, 0x330c, 0x0f9c,
 0x0358, 0x1b34, 0x277c, 0x3310, 0x0fa0,
 0x0458, 0x1c34, 0x287c, 0x3410, 0x10a0,
 0x0558, 0x1d34, 0x297c, 0x3510, 0x11a0,
 0x055c, 0x1d38, 0x2980, 0x3514, 0x11a4,
 0x045c, 0x1c38, 0x2880, 0x3414, 0x10a4,
 0x035c, 0x1b38, 0x2780, 0x3314, 0x0fa4,
 0x0360, 0x1b3c, 0x2784, 0x3318, 0x0fa8,
 0x0460, 0x1c3c, 0x2884, 0x3418, 0x10a8,
 0x0560, 0x1d3c, 0x2984, 0x3518, 0x11a8,
 0x0564, 0x1d40, 0x2988, 0x351c, 0x11ac,
 0x0464, 0x1c40, 0x2888, 0x341c, 0x10ac,
 0x0364, 0x1b40, 0x2788, 0x331c, 0x0fac,
 0x0368, 0x1b44, 0x278c, 0x3320, 0x0fb0,
 0x0468, 0x1c44, 0x288c, 0x3420, 0x10b0,
 0x0568, 0x1d44, 0x298c, 0x3520, 0x11b0,
 0x0948, 0x2124, 0x2d6c, 0x3900, 0x1590,
 0x0a48, 0x2224, 0x2e6c, 0x3a00, 0x1690,
 0x0b48, 0x2324, 0x2f6c, 0x3b00, 0x1790,
 0x0b4c, 0x2328, 0x2f70, 0x3b04, 0x1794,
 0x0a4c, 0x2228, 0x2e70, 0x3a04, 0x1694,
 0x094c, 0x2128, 0x2d70, 0x3904, 0x1594,
 0x0950, 0x212c, 0x2d74, 0x3908, 0x1598,
 0x0a50, 0x222c, 0x2e74, 0x3a08, 0x1698,
 0x0b50, 0x232c, 0x2f74, 0x3b08, 0x1798,
 0x0b54, 0x2330, 0x2f78, 0x3b0c, 0x179c,
 0x0a54, 0x2230, 0x2e78, 0x3a0c, 0x169c,
 0x0954, 0x2130, 0x2d78, 0x390c, 0x159c,
 0x0958, 0x2134, 0x2d7c, 0x3910, 0x15a0,
 0x0a58, 0x2234, 0x2e7c, 0x3a10, 0x16a0,
 0x0b58, 0x2334, 0x2f7c, 0x3b10, 0x17a0,
 0x0b5c, 0x2338, 0x2f80, 0x3b14, 0x17a4,
 0x0a5c, 0x2238, 0x2e80, 0x3a14, 0x16a4,
 0x095c, 0x2138, 0x2d80, 0x3914, 0x15a4,
 0x0960, 0x213c, 0x2d84, 0x3918, 0x15a8,
 0x0a60, 0x223c, 0x2e84, 0x3a18, 0x16a8,
 0x0b60, 0x233c, 0x2f84, 0x3b18, 0x17a8,
 0x0b64, 0x2340, 0x2f88, 0x3b1c, 0x17ac,
 0x0a64, 0x2240, 0x2e88, 0x3a1c, 0x16ac,
 0x0964, 0x2140, 0x2d88, 0x391c, 0x15ac,
 0x0968, 0x2144, 0x2d8c, 0x3920, 0x15b0,
 0x0a68, 0x2244, 0x2e8c, 0x3a20, 0x16b0,
 0x0b68, 0x2344, 0x2f8c, 0x3b20, 0x17b0,
};

/* 2 channels per frame, 12 DIF sequences per channel,
   27 video segments per DIF sequence, 5 macroblocks per video segment */
static const uint16_t dv_place_422_625[2*12*27*5] = {
 0x0c48, 0x2424, 0x306c, 0x0000, 0x1890,
 0x0d48, 0x2524, 0x316c, 0x0100, 0x1990,
 0x0e48, 0x2624, 0x326c, 0x0200, 0x1a90,
 0x0e4c, 0x2628, 0x3270, 0x0204, 0x1a94,
 0x0d4c, 0x2528, 0x3170, 0x0104, 0x1994,
 0x0c4c, 0x2428, 0x3070, 0x0004, 0x1894,
 0x0c50, 0x242c, 0x3074, 0x0008, 0x1898,
 0x0d50, 0x252c, 0x3174, 0x0108, 0x1998,
 0x0e50, 0x262c, 0x3274, 0x0208, 0x1a98,
 0x0e54, 0x2630, 0x3278, 0x020c, 0x1a9c,
 0x0d54, 0x2530, 0x3178, 0x010c, 0x199c,
 0x0c54, 0x2430, 0x3078, 0x000c, 0x189c,
 0x0c58, 0x2434, 0x307c, 0x0010, 0x18a0,
 0x0d58, 0x2534, 0x317c, 0x0110, 0x19a0,
 0x0e58, 0x2634, 0x327c, 0x0210, 0x1aa0,
 0x0e5c, 0x2638, 0x3280, 0x0214, 0x1aa4,
 0x0d5c, 0x2538, 0x3180, 0x0114, 0x19a4,
 0x0c5c, 0x2438, 0x3080, 0x0014, 0x18a4,
 0x0c60, 0x243c, 0x3084, 0x0018, 0x18a8,
 0x0d60, 0x253c, 0x3184, 0x0118, 0x19a8,
 0x0e60, 0x263c, 0x3284, 0x0218, 0x1aa8,
 0x0e64, 0x2640, 0x3288, 0x021c, 0x1aac,
 0x0d64, 0x2540, 0x3188, 0x011c, 0x19ac,
 0x0c64, 0x2440, 0x3088, 0x001c, 0x18ac,
 0x0c68, 0x2444, 0x308c, 0x0020, 0x18b0,
 0x0d68, 0x2544, 0x318c, 0x0120, 0x19b0,
 0x0e68, 0x2644, 0x328c, 0x0220, 0x1ab0,
 0x1248, 0x2a24, 0x366c, 0x0600, 0x1e90,
 0x1348, 0x2b24, 0x376c, 0x0700, 0x1f90,
 0x1448, 0x2c24, 0x386c, 0x0800, 0x2090,
 0x144c, 0x2c28, 0x3870, 0x0804, 0x2094,
 0x134c, 0x2b28, 0x3770, 0x0704, 0x1f94,
 0x124c, 0x2a28, 0x3670, 0x0604, 0x1e94,
 0x1250, 0x2a2c, 0x3674, 0x0608, 0x1e98,
 0x1350, 0x2b2c, 0x3774, 0x0708, 0x1f98,
 0x1450, 0x2c2c, 0x3874, 0x0808, 0x2098,
 0x1454, 0x2c30, 0x3878, 0x080c, 0x209c,
 0x1354, 0x2b30, 0x3778, 0x070c, 0x1f9c,
 0x1254, 0x2a30, 0x3678, 0x060c, 0x1e9c,
 0x1258, 0x2a34, 0x367c, 0x0610, 0x1ea0,
 0x1358, 0x2b34, 0x377c, 0x0710, 0x1fa0,
 0x1458, 0x2c34, 0x387c, 0x0810, 0x20a0,
 0x145c, 0x2c38, 0x3880, 0x0814, 0x20a4,
 0x135c, 0x2b38, 0x3780, 0x0714, 0x1fa4,
 0x125c, 0x2a38, 0x3680, 0x0614, 0x1ea4,
 0x1260, 0x2a3c, 0x3684, 0x0618, 0x1ea8,
 0x1360, 0x2b3c, 0x3784, 0x0718, 0x1fa8,
 0x1460, 0x2c3c, 0x3884, 0x0818, 0x20a8,
 0x1464, 0x2c40, 0x3888, 0x081c, 0x20ac,
 0x1364, 0x2b40, 0x3788, 0x071c, 0x1fac,
 0x1264, 0x2a40, 0x3688, 0x061c, 0x1eac,
 0x1268, 0x2a44, 0x368c, 0x0620, 0x1eb0,
 0x1368, 0x2b44, 0x378c, 0x0720, 0x1fb0,
 0x1468, 0x2c44, 0x388c, 0x0820, 0x20b0,
 0x1848, 0x3024, 0x3c6c, 0x0c00, 0x2490,
 0x1948, 0x3124, 0x3d6c, 0x0d00, 0x2590,
 0x1a48, 0x3224, 0x3e6c, 0x0e00, 0x2690,
 0x1a4c, 0x3228, 0x3e70, 0x0e04, 0x2694,
 0x194c, 0x3128, 0x3d70, 0x0d04, 0x2594,
 0x184c, 0x3028, 0x3c70, 0x0c04, 0x2494,
 0x1850, 0x302c, 0x3c74, 0x0c08, 0x2498,
 0x1950, 0x312c, 0x3d74, 0x0d08, 0x2598,
 0x1a50, 0x322c, 0x3e74, 0x0e08, 0x2698,
 0x1a54, 0x3230, 0x3e78, 0x0e0c, 0x269c,
 0x1954, 0x3130, 0x3d78, 0x0d0c, 0x259c,
 0x1854, 0x3030, 0x3c78, 0x0c0c, 0x249c,
 0x1858, 0x3034, 0x3c7c, 0x0c10, 0x24a0,
 0x1958, 0x3134, 0x3d7c, 0x0d10, 0x25a0,
 0x1a58, 0x3234, 0x3e7c, 0x0e10, 0x26a0,
 0x1a5c, 0x3238, 0x3e80, 0x0e14, 0x26a4,
 0x195c, 0x3138, 0x3d80, 0x0d14, 0x25a4,
 0x185c, 0x3038, 0x3c80, 0x0c14, 0x24a4,
 0x1860, 0x303c, 0x3c84, 0x0c18, 0x24a8,
 0x1960, 0x313c, 0x3d84, 0x0d18, 0x25a8,
 0x1a60, 0x323c, 0x3e84, 0x0e18, 0x26a8,
 0x1a64, 0x3240, 0x3e88, 0x0e1c, 0x26ac,
 0x1964, 0x3140, 0x3d88, 0x0d1c, 0x25ac,
 0x1864, 0x3040, 0x3c88, 0x0c1c, 0x24ac,
 0x1868, 0x3044, 0x3c8c, 0x0c20, 0x24b0,
 0x1968, 0x3144, 0x3d8c, 0x0d20, 0x25b0,
 0x1a68, 0x3244, 0x3e8c, 0x0e20, 0x26b0,
 0x1e48, 0x3624, 0x426c, 0x1200, 0x2a90,
 0x1f48, 0x3724, 0x436c, 0x1300, 0x2b90,
 0x2048, 0x3824, 0x446c, 0x1400, 0x2c90,
 0x204c, 0x3828, 0x4470, 0x1404, 0x2c94,
 0x1f4c, 0x3728, 0x4370, 0x1304, 0x2b94,
 0x1e4c, 0x3628, 0x4270, 0x1204, 0x2a94,
 0x1e50, 0x362c, 0x4274, 0x1208, 0x2a98,
 0x1f50, 0x372c, 0x4374, 0x1308, 0x2b98,
 0x2050, 0x382c, 0x4474, 0x1408, 0x2c98,
 0x2054, 0x3830, 0x4478, 0x140c, 0x2c9c,
 0x1f54, 0x3730, 0x4378, 0x130c, 0x2b9c,
 0x1e54, 0x3630, 0x4278, 0x120c, 0x2a9c,
 0x1e58, 0x3634, 0x427c, 0x1210, 0x2aa0,
 0x1f58, 0x3734, 0x437c, 0x1310, 0x2ba0,
 0x2058, 0x3834, 0x447c, 0x1410, 0x2ca0,
 0x205c, 0x3838, 0x4480, 0x1414, 0x2ca4,
 0x1f5c, 0x3738, 0x4380, 0x1314, 0x2ba4,
 0x1e5c, 0x3638, 0x4280, 0x1214, 0x2aa4,
 0x1e60, 0x363c, 0x4284, 0x1218, 0x2aa8,
 0x1f60, 0x373c, 0x4384, 0x1318, 0x2ba8,
 0x2060, 0x383c, 0x4484, 0x1418, 0x2ca8,
 0x2064, 0x3840, 0x4488, 0x141c, 0x2cac,
 0x1f64, 0x3740, 0x4388, 0x131c, 0x2bac,
 0x1e64, 0x3640, 0x4288, 0x121c, 0x2aac,
 0x1e68, 0x3644, 0x428c, 0x1220, 0x2ab0,
 0x1f68, 0x3744, 0x438c, 0x1320, 0x2bb0,
 0x2068, 0x3844, 0x448c, 0x1420, 0x2cb0,
 0x2448, 0x3c24, 0x006c, 0x1800, 0x3090,
 0x2548, 0x3d24, 0x016c, 0x1900, 0x3190,
 0x2648, 0x3e24, 0x026c, 0x1a00, 0x3290,
 0x264c, 0x3e28, 0x0270, 0x1a04, 0x3294,
 0x254c, 0x3d28, 0x0170, 0x1904, 0x3194,
 0x244c, 0x3c28, 0x0070, 0x1804, 0x3094,
 0x2450, 0x3c2c, 0x0074, 0x1808, 0x3098,
 0x2550, 0x3d2c, 0x0174, 0x1908, 0x3198,
 0x2650, 0x3e2c, 0x0274, 0x1a08, 0x3298,
 0x2654, 0x3e30, 0x0278, 0x1a0c, 0x329c,
 0x2554, 0x3d30, 0x0178, 0x190c, 0x319c,
 0x2454, 0x3c30, 0x0078, 0x180c, 0x309c,
 0x2458, 0x3c34, 0x007c, 0x1810, 0x30a0,
 0x2558, 0x3d34, 0x017c, 0x1910, 0x31a0,
 0x2658, 0x3e34, 0x027c, 0x1a10, 0x32a0,
 0x265c, 0x3e38, 0x0280, 0x1a14, 0x32a4,
 0x255c, 0x3d38, 0x0180, 0x1914, 0x31a4,
 0x245c, 0x3c38, 0x0080, 0x1814, 0x30a4,
 0x2460, 0x3c3c, 0x0084, 0x1818, 0x30a8,
 0x2560, 0x3d3c, 0x0184, 0x1918, 0x31a8,
 0x2660, 0x3e3c, 0x0284, 0x1a18, 0x32a8,
 0x2664, 0x3e40, 0x0288, 0x1a1c, 0x32ac,
 0x2564, 0x3d40, 0x0188, 0x191c, 0x31ac,
 0x2464, 0x3c40, 0x0088, 0x181c, 0x30ac,
 0x2468, 0x3c44, 0x008c, 0x1820, 0x30b0,
 0x2568, 0x3d44, 0x018c, 0x1920, 0x31b0,
 0x2668, 0x3e44, 0x028c, 0x1a20, 0x32b0,
 0x2a48, 0x4224, 0x066c, 0x1e00, 0x3690,
 0x2b48, 0x4324, 0x076c, 0x1f00, 0x3790,
 0x2c48, 0x4424, 0x086c, 0x2000, 0x3890,
 0x2c4c, 0x4428, 0x0870, 0x2004, 0x3894,
 0x2b4c, 0x4328, 0x0770, 0x1f04, 0x3794,
 0x2a4c, 0x4228, 0x0670, 0x1e04, 0x3694,
 0x2a50, 0x422c, 0x0674, 0x1e08, 0x3698,
 0x2b50, 0x432c, 0x0774, 0x1f08, 0x3798,
 0x2c50, 0x442c, 0x0874, 0x2008, 0x3898,
 0x2c54, 0x4430, 0x0878, 0x200c, 0x389c,
 0x2b54, 0x4330, 0x0778, 0x1f0c, 0x379c,
 0x2a54, 0x4230, 0x0678, 0x1e0c, 0x369c,
 0x2a58, 0x4234, 0x067c, 0x1e10, 0x36a0,
 0x2b58, 0x4334, 0x077c, 0x1f10, 0x37a0,
 0x2c58, 0x4434, 0x087c, 0x2010, 0x38a0,
 0x2c5c, 0x4438, 0x0880, 0x2014, 0x38a4,
 0x2b5c, 0x4338, 0x0780, 0x1f14, 0x37a4,
 0x2a5c, 0x4238, 0x0680, 0x1e14, 0x36a4,
 0x2a60, 0x423c, 0x0684, 0x1e18, 0x36a8,
 0x2b60, 0x433c, 0x0784, 0x1f18, 0x37a8,
 0x2c60, 0x443c, 0x0884, 0x2018, 0x38a8,
 0x2c64, 0x4440, 0x0888, 0x201c, 0x38ac,
 0x2b64, 0x4340, 0x0788, 0x1f1c, 0x37ac,
 0x2a64, 0x4240, 0x0688, 0x1e1c, 0x36ac,
 0x2a68, 0x4244, 0x068c, 0x1e20, 0x36b0,
 0x2b68, 0x4344, 0x078c, 0x1f20, 0x37b0,
 0x2c68, 0x4444, 0x088c, 0x2020, 0x38b0,
 0x3048, 0x0024, 0x0c6c, 0x2400, 0x3c90,
 0x3148, 0x0124, 0x0d6c, 0x2500, 0x3d90,
 0x3248, 0x0224, 0x0e6c, 0x2600, 0x3e90,
 0x324c, 0x0228, 0x0e70, 0x2604, 0x3e94,
 0x314c, 0x0128, 0x0d70, 0x2504, 0x3d94,
 0x304c, 0x0028, 0x0c70, 0x2404, 0x3c94,
 0x3050, 0x002c, 0x0c74, 0x2408, 0x3c98,
 0x3150, 0x012c, 0x0d74, 0x2508, 0x3d98,
 0x3250, 0x022c, 0x0e74, 0x2608, 0x3e98,
 0x3254, 0x0230, 0x0e78, 0x260c, 0x3e9c,
 0x3154, 0x0130, 0x0d78, 0x250c, 0x3d9c,
 0x3054, 0x0030, 0x0c78, 0x240c, 0x3c9c,
 0x3058, 0x0034, 0x0c7c, 0x2410, 0x3ca0,
 0x3158, 0x0134, 0x0d7c, 0x2510, 0x3da0,
 0x3258, 0x0234, 0x0e7c, 0x2610, 0x3ea0,
 0x325c, 0x0238, 0x0e80, 0x2614, 0x3ea4,
 0x315c, 0x0138, 0x0d80, 0x2514, 0x3da4,
 0x305c, 0x0038, 0x0c80, 0x2414, 0x3ca4,
 0x3060, 0x003c, 0x0c84, 0x2418, 0x3ca8,
 0x3160, 0x013c, 0x0d84, 0x2518, 0x3da8,
 0x3260, 0x023c, 0x0e84, 0x2618, 0x3ea8,
 0x3264, 0x0240, 0x0e88, 0x261c, 0x3eac,
 0x3164, 0x0140, 0x0d88, 0x251c, 0x3dac,
 0x3064, 0x0040, 0x0c88, 0x241c, 0x3cac,
 0x3068, 0x0044, 0x0c8c, 0x2420, 0x3cb0,
 0x3168, 0x0144, 0x0d8c, 0x2520, 0x3db0,
 0x3268, 0x0244, 0x0e8c, 0x2620, 0x3eb0,
 0x3648, 0x0624, 0x126c, 0x2a00, 0x4290,
 0x3748, 0x0724, 0x136c, 0x2b00, 0x4390,
 0x3848, 0x0824, 0x146c, 0x2c00, 0x4490,
 0x384c, 0x0828, 0x1470, 0x2c04, 0x4494,
 0x374c, 0x0728, 0x1370, 0x2b04, 0x4394,
 0x364c, 0x0628, 0x1270, 0x2a04, 0x4294,
 0x3650, 0x062c, 0x1274, 0x2a08, 0x4298,
 0x3750, 0x072c, 0x1374, 0x2b08, 0x4398,
 0x3850, 0x082c, 0x1474, 0x2c08, 0x4498,
 0x3854, 0x0830, 0x1478, 0x2c0c, 0x449c,
 0x3754, 0x0730, 0x1378, 0x2b0c, 0x439c,
 0x3654, 0x0630, 0x1278, 0x2a0c, 0x429c,
 0x3658, 0x0634, 0x127c, 0x2a10, 0x42a0,
 0x3758, 0x0734, 0x137c, 0x2b10, 0x43a0,
 0x3858, 0x0834, 0x147c, 0x2c10, 0x44a0,
 0x385c, 0x0838, 0x1480, 0x2c14, 0x44a4,
 0x375c, 0x0738, 0x1380, 0x2b14, 0x43a4,
 0x365c, 0x0638, 0x1280, 0x2a14, 0x42a4,
 0x3660, 0x063c, 0x1284, 0x2a18, 0x42a8,
 0x3760, 0x073c, 0x1384, 0x2b18, 0x43a8,
 0x3860, 0x083c, 0x1484, 0x2c18, 0x44a8,
 0x3864, 0x0840, 0x1488, 0x2c1c, 0x44ac,
 0x3764, 0x0740, 0x1388, 0x2b1c, 0x43ac,
 0x3664, 0x0640, 0x1288, 0x2a1c, 0x42ac,
 0x3668, 0x0644, 0x128c, 0x2a20, 0x42b0,
 0x3768, 0x0744, 0x138c, 0x2b20, 0x43b0,
 0x3868, 0x0844, 0x148c, 0x2c20, 0x44b0,
 0x3c48, 0x0c24, 0x186c, 0x3000, 0x0090,
 0x3d48, 0x0d24, 0x196c, 0x3100, 0x0190,
 0x3e48, 0x0e24, 0x1a6c, 0x3200, 0x0290,
 0x3e4c, 0x0e28, 0x1a70, 0x3204, 0x0294,
 0x3d4c, 0x0d28, 0x1970, 0x3104, 0x0194,
 0x3c4c, 0x0c28, 0x1870, 0x3004, 0x0094,
 0x3c50, 0x0c2c, 0x1874, 0x3008, 0x0098,
 0x3d50, 0x0d2c, 0x1974, 0x3108, 0x0198,
 0x3e50, 0x0e2c, 0x1a74, 0x3208, 0x0298,
 0x3e54, 0x0e30, 0x1a78, 0x320c, 0x029c,
 0x3d54, 0x0d30, 0x1978, 0x310c, 0x019c,
 0x3c54, 0x0c30, 0x1878, 0x300c, 0x009c,
 0x3c58, 0x0c34, 0x187c, 0x3010, 0x00a0,
 0x3d58, 0x0d34, 0x197c, 0x3110, 0x01a0,
 0x3e58, 0x0e34, 0x1a7c, 0x3210, 0x02a0,
 0x3e5c, 0x0e38, 0x1a80, 0x3214, 0x02a4,
 0x3d5c, 0x0d38, 0x1980, 0x3114, 0x01a4,
 0x3c5c, 0x0c38, 0x1880, 0x3014, 0x00a4,
 0x3c60, 0x0c3c, 0x1884, 0x3018, 0x00a8,
 0x3d60, 0x0d3c, 0x1984, 0x3118, 0x01a8,
 0x3e60, 0x0e3c, 0x1a84, 0x3218, 0x02a8,
 0x3e64, 0x0e40, 0x1a88, 0x321c, 0x02ac,
 0x3d64, 0x0d40, 0x1988, 0x311c, 0x01ac,
 0x3c64, 0x0c40, 0x1888, 0x301c, 0x00ac,
 0x3c68, 0x0c44, 0x188c, 0x3020, 0x00b0,
 0x3d68, 0x0d44, 0x198c, 0x3120, 0x01b0,
 0x3e68, 0x0e44, 0x1a8c, 0x3220, 0x02b0,
 0x4248, 0x1224, 0x1e6c, 0x3600, 0x0690,
 0x4348, 0x1324, 0x1f6c, 0x3700, 0x0790,
 0x4448, 0x1424, 0x206c, 0x3800, 0x0890,
 0x444c, 0x1428, 0x2070, 0x3804, 0x0894,
 0x434c, 0x1328, 0x1f70, 0x3704, 0x0794,
 0x424c, 0x1228, 0x1e70, 0x3604, 0x0694,
 0x4250, 0x122c, 0x1e74, 0x3608, 0x0698,
 0x4350, 0x132c, 0x1f74, 0x3708, 0x0798,
 0x4450, 0x142c, 0x2074, 0x3808, 0x0898,
 0x4454, 0x1430, 0x2078, 0x380c, 0x089c,
 0x4354, 0x1330, 0x1f78, 0x370c, 0x079c,
 0x4254, 0x1230, 0x1e78, 0x360c, 0x069c,
 0x4258, 0x1234, 0x1e7c, 0x3610, 0x06a0,
 0x4358, 0x1334, 0x1f7c, 0x3710, 0x07a0,
 0x4458, 0x1434, 0x207c, 0x3810, 0x08a0,
 0x445c, 0x1438, 0x2080, 0x3814, 0x08a4,
 0x435c, 0x1338, 0x1f80, 0x3714, 0x07a4,
 0x425c, 0x1238, 0x1e80, 0x3614, 0x06a4,
 0x4260, 0x123c, 0x1e84, 0x3618, 0x06a8,
 0x4360, 0x133c, 0x1f84, 0x3718, 0x07a8,
 0x4460, 0x143c, 0x2084, 0x3818, 0x08a8,
 0x4464, 0x1440, 0x2088, 0x381c, 0x08ac,
 0x4364, 0x1340, 0x1f88, 0x371c, 0x07ac,
 0x4264, 0x1240, 0x1e88, 0x361c, 0x06ac,
 0x4268, 0x1244, 0x1e8c, 0x3620, 0x06b0,
 0x4368, 0x1344, 0x1f8c, 0x3720, 0x07b0,
 0x4468, 0x1444, 0x208c, 0x3820, 0x08b0,
 0x0048, 0x1824, 0x246c, 0x3c00, 0x0c90,
 0x0148, 0x1924, 0x256c, 0x3d00, 0x0d90,
 0x0248, 0x1a24, 0x266c, 0x3e00, 0x0e90,
 0x024c, 0x1a28, 0x2670, 0x3e04, 0x0e94,
 0x014c, 0x1928, 0x2570, 0x3d04, 0x0d94,
 0x004c, 0x1828, 0x2470, 0x3c04, 0x0c94,
 0x0050, 0x182c, 0x2474, 0x3c08, 0x0c98,
 0x0150, 0x192c, 0x2574, 0x3d08, 0x0d98,
 0x0250, 0x1a2c, 0x2674, 0x3e08, 0x0e98,
 0x0254, 0x1a30, 0x2678, 0x3e0c, 0x0e9c,
 0x0154, 0x1930, 0x2578, 0x3d0c, 0x0d9c,
 0x0054, 0x1830, 0x2478, 0x3c0c, 0x0c9c,
 0x0058, 0x1834, 0x247c, 0x3c10, 0x0ca0,
 0x0158, 0x1934, 0x257c, 0x3d10, 0x0da0,
 0x0258, 0x1a34, 0x267c, 0x3e10, 0x0ea0,
 0x025c, 0x1a38, 0x2680, 0x3e14, 0x0ea4,
 0x015c, 0x1938, 0x2580, 0x3d14, 0x0da4,
 0x005c, 0x1838, 0x2480, 0x3c14, 0x0ca4,
 0x0060, 0x183c, 0x2484, 0x3c18, 0x0ca8,
 0x0160, 0x193c, 0x2584, 0x3d18, 0x0da8,
 0x0260, 0x1a3c, 0x2684, 0x3e18, 0x0ea8,
 0x0264, 0x1a40, 0x2688, 0x3e1c, 0x0eac,
 0x0164, 0x1940, 0x2588, 0x3d1c, 0x0dac,
 0x0064, 0x1840, 0x2488, 0x3c1c, 0x0cac,
 0x0068, 0x1844, 0x248c, 0x3c20, 0x0cb0,
 0x0168, 0x1944, 0x258c, 0x3d20, 0x0db0,
 0x0268, 0x1a44, 0x268c, 0x3e20, 0x0eb0,
 0x0648, 0x1e24, 0x2a6c, 0x4200, 0x1290,
 0x0748, 0x1f24, 0x2b6c, 0x4300, 0x1390,
 0x0848, 0x2024, 0x2c6c, 0x4400, 0x1490,
 0x084c, 0x2028, 0x2c70, 0x4404, 0x1494,
 0x074c, 0x1f28, 0x2b70, 0x4304, 0x1394,
 0x064c, 0x1e28, 0x2a70, 0x4204, 0x1294,
 0x0650, 0x1e2c, 0x2a74, 0x4208, 0x1298,
 0x0750, 0x1f2c, 0x2b74, 0x4308, 0x1398,
 0x0850, 0x202c, 0x2c74, 0x4408, 0x1498,
 0x0854, 0x2030, 0x2c78, 0x440c, 0x149c,
 0x0754, 0x1f30, 0x2b78, 0x430c, 0x139c,
 0x0654, 0x1e30, 0x2a78, 0x420c, 0x129c,
 0x0658, 0x1e34, 0x2a7c, 0x4210, 0x12a0,
 0x0758, 0x1f34, 0x2b7c, 0x4310, 0x13a0,
 0x0858, 0x2034, 0x2c7c, 0x4410, 0x14a0,
 0x085c, 0x2038, 0x2c80, 0x4414, 0x14a4,
 0x075c, 0x1f38, 0x2b80, 0x4314, 0x13a4,
 0x065c, 0x1e38, 0x2a80, 0x4214, 0x12a4,
 0x0660, 0x1e3c, 0x2a84, 0x4218, 0x12a8,
 0x0760, 0x1f3c, 0x2b84, 0x4318, 0x13a8,
 0x0860, 0x203c, 0x2c84, 0x4418, 0x14a8,
 0x0864, 0x2040, 0x2c88, 0x441c, 0x14ac,
 0x0764, 0x1f40, 0x2b88, 0x431c, 0x13ac,
 0x0664, 0x1e40, 0x2a88, 0x421c, 0x12ac,
 0x0668, 0x1e44, 0x2a8c, 0x4220, 0x12b0,
 0x0768, 0x1f44, 0x2b8c, 0x4320, 0x13b0,
 0x0868, 0x2044, 0x2c8c, 0x4420, 0x14b0,
 0x0f48, 0x2724, 0x336c, 0x0300, 0x1b90,
 0x1048, 0x2824, 0x346c, 0x0400, 0x1c90,
 0x1148, 0x2924, 0x356c, 0x0500, 0x1d90,
 0x114c, 0x2928, 0x3570, 0x0504, 0x1d94,
 0x104c, 0x2828, 0x3470, 0x0404, 0x1c94,
 0x0f4c, 0x2728, 0x3370, 0x0304, 0x1b94,
 0x0f50, 0x272c, 0x3374, 0x0308, 0x1b98,
 0x1050, 0x282c, 0x3474, 0x0408, 0x1c98,
 0x1150, 0x292c, 0x3574, 0x0508, 0x1d98,
 0x1154, 0x2930, 0x3578, 0x050c, 0x1d9c,
 0x1054, 0x2830, 0x3478, 0x040c, 0x1c9c,
 0x0f54, 0x2730, 0x3378, 0x030c, 0x1b9c,
 0x0f58, 0x2734, 0x337c, 0x0310, 0x1ba0,
 0x1058, 0x2834, 0x347c, 0x0410, 0x1ca0,
 0x1158, 0x2934, 0x357c, 0x0510, 0x1da0,
 0x115c, 0x2938, 0x3580, 0x0514, 0x1da4,
 0x105c, 0x2838, 0x3480, 0x0414, 0x1ca4,
 0x0f5c, 0x2738, 0x3380, 0x0314, 0x1ba4,
 0x0f60, 0x273c, 0x3384, 0x0318, 0x1ba8,
 0x1060, 0x283c, 0x3484, 0x0418, 0x1ca8,
 0x1160, 0x293c, 0x3584, 0x0518, 0x1da8,
 0x1164, 0x2940, 0x3588, 0x051c, 0x1dac,
 0x1064, 0x2840, 0x3488, 0x041c, 0x1cac,
 0x0f64, 0x2740, 0x3388, 0x031c, 0x1bac,
 0x0f68, 0x2744, 0x338c, 0x0320, 0x1bb0,
 0x1068, 0x2844, 0x348c, 0x0420, 0x1cb0,
 0x1168, 0x2944, 0x358c, 0x0520, 0x1db0,
 0x1548, 0x2d24, 0x396c, 0x0900, 0x2190,
 0x1648, 0x2e24, 0x3a6c, 0x0a00, 0x2290,
 0x1748, 0x2f24, 0x3b6c, 0x0b00, 0x2390,
 0x174c, 0x2f28, 0x3b70, 0x0b04, 0x2394,
 0x164c, 0x2e28, 0x3a70, 0x0a04, 0x2294,
 0x154c, 0x2d28, 0x3970, 0x0904, 0x2194,
 0x1550, 0x2d2c, 0x3974, 0x0908, 0x2198,
 0x1650, 0x2e2c, 0x3a74, 0x0a08, 0x2298,
 0x1750, 0x2f2c, 0x3b74, 0x0b08, 0x2398,
 0x1754, 0x2f30, 0x3b78, 0x0b0c, 0x239c,
 0x1654, 0x2e30, 0x3a78, 0x0a0c, 0x229c,
 0x1554, 0x2d30, 0x3978, 0x090c, 0x219c,
 0x1558, 0x2d34, 0x397c, 0x0910, 0x21a0,
 0x1658, 0x2e34, 0x3a7c, 0x0a10, 0x22a0,
 0x1758, 0x2f34, 0x3b7c, 0x0b10, 0x23a0,
 0x175c, 0x2f38, 0x3b80, 0x0b14, 0x23a4,
 0x165c, 0x2e38, 0x3a80, 0x0a14, 0x22a4,
 0x155c, 0x2d38, 0x3980, 0x0914, 0x21a4,
 0x1560, 0x2d3c, 0x3984, 0x0918, 0x21a8,
 0x1660, 0x2e3c, 0x3a84, 0x0a18, 0x22a8,
 0x1760, 0x2f3c, 0x3b84, 0x0b18, 0x23a8,
 0x1764, 0x2f40, 0x3b88, 0x0b1c, 0x23ac,
 0x1664, 0x2e40, 0x3a88, 0x0a1c, 0x22ac,
 0x1564, 0x2d40, 0x3988, 0x091c, 0x21ac,
 0x1568, 0x2d44, 0x398c, 0x0920, 0x21b0,
 0x1668, 0x2e44, 0x3a8c, 0x0a20, 0x22b0,
 0x1768, 0x2f44, 0x3b8c, 0x0b20, 0x23b0,
 0x1b48, 0x3324, 0x3f6c, 0x0f00, 0x2790,
 0x1c48, 0x3424, 0x406c, 0x1000, 0x2890,
 0x1d48, 0x3524, 0x416c, 0x1100, 0x2990,
 0x1d4c, 0x3528, 0x4170, 0x1104, 0x2994,
 0x1c4c, 0x3428, 0x4070, 0x1004, 0x2894,
 0x1b4c, 0x3328, 0x3f70, 0x0f04, 0x2794,
 0x1b50, 0x332c, 0x3f74, 0x0f08, 0x2798,
 0x1c50, 0x342c, 0x4074, 0x1008, 0x2898,
 0x1d50, 0x352c, 0x4174, 0x1108, 0x2998,
 0x1d54, 0x3530, 0x4178, 0x110c, 0x299c,
 0x1c54, 0x3430, 0x4078, 0x100c, 0x289c,
 0x1b54, 0x3330, 0x3f78, 0x0f0c, 0x279c,
 0x1b58, 0x3334, 0x3f7c, 0x0f10, 0x27a0,
 0x1c58, 0x3434, 0x407c, 0x1010, 0x28a0,
 0x1d58, 0x3534, 0x417c, 0x1110, 0x29a0,
 0x1d5c, 0x3538, 0x4180, 0x1114, 0x29a4,
 0x1c5c, 0x3438, 0x4080, 0x1014, 0x28a4,
 0x1b5c, 0x3338, 0x3f80, 0x0f14, 0x27a4,
 0x1b60, 0x333c, 0x3f84, 0x0f18, 0x27a8,
 0x1c60, 0x343c, 0x4084, 0x1018, 0x28a8,
 0x1d60, 0x353c, 0x4184, 0x1118, 0x29a8,
 0x1d64, 0x3540, 0x4188, 0x111c, 0x29ac,
 0x1c64, 0x3440, 0x4088, 0x101c, 0x28ac,
 0x1b64, 0x3340, 0x3f88, 0x0f1c, 0x27ac,
 0x1b68, 0x3344, 0x3f8c, 0x0f20, 0x27b0,
 0x1c68, 0x3444, 0x408c, 0x1020, 0x28b0,
 0x1d68, 0x3544, 0x418c, 0x1120, 0x29b0,
 0x2148, 0x3924, 0x456c, 0x1500, 0x2d90,
 0x2248, 0x3a24, 0x466c, 0x1600, 0x2e90,
 0x2348, 0x3b24, 0x476c, 0x1700, 0x2f90,
 0x234c, 0x3b28, 0x4770, 0x1704, 0x2f94,
 0x224c, 0x3a28, 0x4670, 0x1604, 0x2e94,
 0x214c, 0x3928, 0x4570, 0x1504, 0x2d94,
 0x2150, 0x392c, 0x4574, 0x1508, 0x2d98,
 0x2250, 0x3a2c, 0x4674, 0x1608, 0x2e98,
 0x2350, 0x3b2c, 0x4774, 0x1708, 0x2f98,
 0x2354, 0x3b30, 0x4778, 0x170c, 0x2f9c,
 0x2254, 0x3a30, 0x4678, 0x160c, 0x2e9c,
 0x2154, 0x3930, 0x4578, 0x150c, 0x2d9c,
 0x2158, 0x3934, 0x457c, 0x1510, 0x2da0,
 0x2258, 0x3a34, 0x467c, 0x1610, 0x2ea0,
 0x2358, 0x3b34, 0x477c, 0x1710, 0x2fa0,
 0x235c, 0x3b38, 0x4780, 0x1714, 0x2fa4,
 0x225c, 0x3a38, 0x4680, 0x1614, 0x2ea4,
 0x215c, 0x3938, 0x4580, 0x1514, 0x2da4,
 0x2160, 0x393c, 0x4584, 0x1518, 0x2da8,
 0x2260, 0x3a3c, 0x4684, 0x1618, 0x2ea8,
 0x2360, 0x3b3c, 0x4784, 0x1718, 0x2fa8,
 0x2364, 0x3b40, 0x4788, 0x171c, 0x2fac,
 0x2264, 0x3a40, 0x4688, 0x161c, 0x2eac,
 0x2164, 0x3940, 0x4588, 0x151c, 0x2dac,
 0x2168, 0x3944, 0x458c, 0x1520, 0x2db0,
 0x2268, 0x3a44, 0x468c, 0x1620, 0x2eb0,
 0x2368, 0x3b44, 0x478c, 0x1720, 0x2fb0,
 0x2748, 0x3f24, 0x036c, 0x1b00, 0x3390,
 0x2848, 0x4024, 0x046c, 0x1c00, 0x3490,
 0x2948, 0x4124, 0x056c, 0x1d00, 0x3590,
 0x294c, 0x4128, 0x0570, 0x1d04, 0x3594,
 0x284c, 0x4028, 0x0470, 0x1c04, 0x3494,
 0x274c, 0x3f28, 0x0370, 0x1b04, 0x3394,
 0x2750, 0x3f2c, 0x0374, 0x1b08, 0x3398,
 0x2850, 0x402c, 0x0474, 0x1c08, 0x3498,
 0x2950, 0x412c, 0x0574, 0x1d08, 0x3598,
 0x2954, 0x4130, 0x0578, 0x1d0c, 0x359c,
 0x2854, 0x4030, 0x0478, 0x1c0c, 0x349c,
 0x2754, 0x3f30, 0x0378, 0x1b0c, 0x339c,
 0x2758, 0x3f34, 0x037c, 0x1b10, 0x33a0,
 0x2858, 0x4034, 0x047c, 0x1c10, 0x34a0,
 0x2958, 0x4134, 0x057c, 0x1d10, 0x35a0,
 0x295c, 0x4138, 0x0580, 0x1d14, 0x35a4,
 0x285c, 0x4038, 0x0480, 0x1c14, 0x34a4,
 0x275c, 0x3f38, 0x0380, 0x1b14, 0x33a4,
 0x2760, 0x3f3c, 0x0384, 0x1b18, 0x33a8,
 0x2860, 0x403c, 0x0484, 0x1c18, 0x34a8,
 0x2960, 0x413c, 0x0584, 0x1d18, 0x35a8,
 0x2964, 0x4140, 0x0588, 0x1d1c, 0x35ac,
 0x2864, 0x4040, 0x0488, 0x1c1c, 0x34ac,
 0x2764, 0x3f40, 0x0388, 0x1b1c, 0x33ac,
 0x2768, 0x3f44, 0x038c, 0x1b20, 0x33b0,
 0x2868, 0x4044, 0x048c, 0x1c20, 0x34b0,
 0x2968, 0x4144, 0x058c, 0x1d20, 0x35b0,
 0x2d48, 0x4524, 0x096c, 0x2100, 0x3990,
 0x2e48, 0x4624, 0x0a6c, 0x2200, 0x3a90,
 0x2f48, 0x4724, 0x0b6c, 0x2300, 0x3b90,
 0x2f4c, 0x4728, 0x0b70, 0x2304, 0x3b94,
 0x2e4c, 0x4628, 0x0a70, 0x2204, 0x3a94,
 0x2d4c, 0x4528, 0x0970, 0x2104, 0x3994,
 0x2d50, 0x452c, 0x0974, 0x2108, 0x3998,
 0x2e50, 0x462c, 0x0a74, 0x2208, 0x3a98,
 0x2f50, 0x472c, 0x0b74, 0x2308, 0x3b98,
 0x2f54, 0x4730, 0x0b78, 0x230c, 0x3b9c,
 0x2e54, 0x4630, 0x0a78, 0x220c, 0x3a9c,
 0x2d54, 0x4530, 0x0978, 0x210c, 0x399c,
 0x2d58, 0x4534, 0x097c, 0x2110, 0x39a0,
 0x2e58, 0x4634, 0x0a7c, 0x2210, 0x3aa0,
 0x2f58, 0x4734, 0x0b7c, 0x2310, 0x3ba0,
 0x2f5c, 0x4738, 0x0b80, 0x2314, 0x3ba4,
 0x2e5c, 0x4638, 0x0a80, 0x2214, 0x3aa4,
 0x2d5c, 0x4538, 0x0980, 0x2114, 0x39a4,
 0x2d60, 0x453c, 0x0984, 0x2118, 0x39a8,
 0x2e60, 0x463c, 0x0a84, 0x2218, 0x3aa8,
 0x2f60, 0x473c, 0x0b84, 0x2318, 0x3ba8,
 0x2f64, 0x4740, 0x0b88, 0x231c, 0x3bac,
 0x2e64, 0x4640, 0x0a88, 0x221c, 0x3aac,
 0x2d64, 0x4540, 0x0988, 0x211c, 0x39ac,
 0x2d68, 0x4544, 0x098c, 0x2120, 0x39b0,
 0x2e68, 0x4644, 0x0a8c, 0x2220, 0x3ab0,
 0x2f68, 0x4744, 0x0b8c, 0x2320, 0x3bb0,
 0x3348, 0x0324, 0x0f6c, 0x2700, 0x3f90,
 0x3448, 0x0424, 0x106c, 0x2800, 0x4090,
 0x3548, 0x0524, 0x116c, 0x2900, 0x4190,
 0x354c, 0x0528, 0x1170, 0x2904, 0x4194,
 0x344c, 0x0428, 0x1070, 0x2804, 0x4094,
 0x334c, 0x0328, 0x0f70, 0x2704, 0x3f94,
 0x3350, 0x032c, 0x0f74, 0x2708, 0x3f98,
 0x3450, 0x042c, 0x1074, 0x2808, 0x4098,
 0x3550, 0x052c, 0x1174, 0x2908, 0x4198,
 0x3554, 0x0530, 0x1178, 0x290c, 0x419c,
 0x3454, 0x0430, 0x1078, 0x280c, 0x409c,
 0x3354, 0x0330, 0x0f78, 0x270c, 0x3f9c,
 0x3358, 0x0334, 0x0f7c, 0x2710, 0x3fa0,
 0x3458, 0x0434, 0x107c, 0x2810, 0x40a0,
 0x3558, 0x0534, 0x117c, 0x2910, 0x41a0,
 0x355c, 0x0538, 0x1180, 0x2914, 0x41a4,
 0x345c, 0x0438, 0x1080, 0x2814, 0x40a4,
 0x335c, 0x0338, 0x0f80, 0x2714, 0x3fa4,
 0x3360, 0x033c, 0x0f84, 0x2718, 0x3fa8,
 0x3460, 0x043c, 0x1084, 0x2818, 0x40a8,
 0x3560, 0x053c, 0x1184, 0x2918, 0x41a8,
 0x3564, 0x0540, 0x1188, 0x291c, 0x41ac,
 0x3464, 0x0440, 0x1088, 0x281c, 0x40ac,
 0x3364, 0x0340, 0x0f88, 0x271c, 0x3fac,
 0x3368, 0x0344, 0x0f8c, 0x2720, 0x3fb0,
 0x3468, 0x0444, 0x108c, 0x2820, 0x40b0,
 0x3568, 0x0544, 0x118c, 0x2920, 0x41b0,
 0x3948, 0x0924, 0x156c, 0x2d00, 0x4590,
 0x3a48, 0x0a24, 0x166c, 0x2e00, 0x4690,
 0x3b48, 0x0b24, 0x176c, 0x2f00, 0x4790,
 0x3b4c, 0x0b28, 0x1770, 0x2f04, 0x4794,
 0x3a4c, 0x0a28, 0x1670, 0x2e04, 0x4694,
 0x394c, 0x0928, 0x1570, 0x2d04, 0x4594,
 0x3950, 0x092c, 0x1574, 0x2d08, 0x4598,
 0x3a50, 0x0a2c, 0x1674, 0x2e08, 0x4698,
 0x3b50, 0x0b2c, 0x1774, 0x2f08, 0x4798,
 0x3b54, 0x0b30, 0x1778, 0x2f0c, 0x479c,
 0x3a54, 0x0a30, 0x1678, 0x2e0c, 0x469c,
 0x3954, 0x0930, 0x1578, 0x2d0c, 0x459c,
 0x3958, 0x0934, 0x157c, 0x2d10, 0x45a0,
 0x3a58, 0x0a34, 0x167c, 0x2e10, 0x46a0,
 0x3b58, 0x0b34, 0x177c, 0x2f10, 0x47a0,
 0x3b5c, 0x0b38, 0x1780, 0x2f14, 0x47a4,
 0x3a5c, 0x0a38, 0x1680, 0x2e14, 0x46a4,
 0x395c, 0x0938, 0x1580, 0x2d14, 0x45a4,
 0x3960, 0x093c, 0x1584, 0x2d18, 0x45a8,
 0x3a60, 0x0a3c, 0x1684, 0x2e18, 0x46a8,
 0x3b60, 0x0b3c, 0x1784, 0x2f18, 0x47a8,
 0x3b64, 0x0b40, 0x1788, 0x2f1c, 0x47ac,
 0x3a64, 0x0a40, 0x1688, 0x2e1c, 0x46ac,
 0x3964, 0x0940, 0x1588, 0x2d1c, 0x45ac,
 0x3968, 0x0944, 0x158c, 0x2d20, 0x45b0,
 0x3a68, 0x0a44, 0x168c, 0x2e20, 0x46b0,
 0x3b68, 0x0b44, 0x178c, 0x2f20, 0x47b0,
 0x3f48, 0x0f24, 0x1b6c, 0x3300, 0x0390,
 0x4048, 0x1024, 0x1c6c, 0x3400, 0x0490,
 0x4148, 0x1124, 0x1d6c, 0x3500, 0x0590,
 0x414c, 0x1128, 0x1d70, 0x3504, 0x0594,
 0x404c, 0x1028, 0x1c70, 0x3404, 0x0494,
 0x3f4c, 0x0f28, 0x1b70, 0x3304, 0x0394,
 0x3f50, 0x0f2c, 0x1b74, 0x3308, 0x0398,
 0x4050, 0x102c, 0x1c74, 0x3408, 0x0498,
 0x4150, 0x112c, 0x1d74, 0x3508, 0x0598,
 0x4154, 0x1130, 0x1d78, 0x350c, 0x059c,
 0x4054, 0x1030, 0x1c78, 0x340c, 0x049c,
 0x3f54, 0x0f30, 0x1b78, 0x330c, 0x039c,
 0x3f58, 0x0f34, 0x1b7c, 0x3310, 0x03a0,
 0x4058, 0x1034, 0x1c7c, 0x3410, 0x04a0,
 0x4158, 0x1134, 0x1d7c, 0x3510, 0x05a0,
 0x415c, 0x1138, 0x1d80, 0x3514, 0x05a4,
 0x405c, 0x1038, 0x1c80, 0x3414, 0x04a4,
 0x3f5c, 0x0f38, 0x1b80, 0x3314, 0x03a4,
 0x3f60, 0x0f3c, 0x1b84, 0x3318, 0x03a8,
 0x4060, 0x103c, 0x1c84, 0x3418, 0x04a8,
 0x4160, 0x113c, 0x1d84, 0x3518, 0x05a8,
 0x4164, 0x1140, 0x1d88, 0x351c, 0x05ac,
 0x4064, 0x1040, 0x1c88, 0x341c, 0x04ac,
 0x3f64, 0x0f40, 0x1b88, 0x331c, 0x03ac,
 0x3f68, 0x0f44, 0x1b8c, 0x3320, 0x03b0,
 0x4068, 0x1044, 0x1c8c, 0x3420, 0x04b0,
 0x4168, 0x1144, 0x1d8c, 0x3520, 0x05b0,
 0x4548, 0x1524, 0x216c, 0x3900, 0x0990,
 0x4648, 0x1624, 0x226c, 0x3a00, 0x0a90,
 0x4748, 0x1724, 0x236c, 0x3b00, 0x0b90,
 0x474c, 0x1728, 0x2370, 0x3b04, 0x0b94,
 0x464c, 0x1628, 0x2270, 0x3a04, 0x0a94,
 0x454c, 0x1528, 0x2170, 0x3904, 0x0994,
 0x4550, 0x152c, 0x2174, 0x3908, 0x0998,
 0x4650, 0x162c, 0x2274, 0x3a08, 0x0a98,
 0x4750, 0x172c, 0x2374, 0x3b08, 0x0b98,
 0x4754, 0x1730, 0x2378, 0x3b0c, 0x0b9c,
 0x4654, 0x1630, 0x2278, 0x3a0c, 0x0a9c,
 0x4554, 0x1530, 0x2178, 0x390c, 0x099c,
 0x4558, 0x1534, 0x217c, 0x3910, 0x09a0,
 0x4658, 0x1634, 0x227c, 0x3a10, 0x0aa0,
 0x4758, 0x1734, 0x237c, 0x3b10, 0x0ba0,
 0x475c, 0x1738, 0x2380, 0x3b14, 0x0ba4,
 0x465c, 0x1638, 0x2280, 0x3a14, 0x0aa4,
 0x455c, 0x1538, 0x2180, 0x3914, 0x09a4,
 0x4560, 0x153c, 0x2184, 0x3918, 0x09a8,
 0x4660, 0x163c, 0x2284, 0x3a18, 0x0aa8,
 0x4760, 0x173c, 0x2384, 0x3b18, 0x0ba8,
 0x4764, 0x1740, 0x2388, 0x3b1c, 0x0bac,
 0x4664, 0x1640, 0x2288, 0x3a1c, 0x0aac,
 0x4564, 0x1540, 0x2188, 0x391c, 0x09ac,
 0x4568, 0x1544, 0x218c, 0x3920, 0x09b0,
 0x4668, 0x1644, 0x228c, 0x3a20, 0x0ab0,
 0x4768, 0x1744, 0x238c, 0x3b20, 0x0bb0,
 0x0348, 0x1b24, 0x276c, 0x3f00, 0x0f90,
 0x0448, 0x1c24, 0x286c, 0x4000, 0x1090,
 0x0548, 0x1d24, 0x296c, 0x4100, 0x1190,
 0x054c, 0x1d28, 0x2970, 0x4104, 0x1194,
 0x044c, 0x1c28, 0x2870, 0x4004, 0x1094,
 0x034c, 0x1b28, 0x2770, 0x3f04, 0x0f94,
 0x0350, 0x1b2c, 0x2774, 0x3f08, 0x0f98,
 0x0450, 0x1c2c, 0x2874, 0x4008, 0x1098,
 0x0550, 0x1d2c, 0x2974, 0x4108, 0x1198,
 0x0554, 0x1d30, 0x2978, 0x410c, 0x119c,
 0x0454, 0x1c30, 0x2878, 0x400c, 0x109c,
 0x0354, 0x1b30, 0x2778, 0x3f0c, 0x0f9c,
 0x0358, 0x1b34, 0x277c, 0x3f10, 0x0fa0,
 0x0458, 0x1c34, 0x287c, 0x4010, 0x10a0,
 0x0558, 0x1d34, 0x297c, 0x4110, 0x11a0,
 0x055c, 0x1d38, 0x2980, 0x4114, 0x11a4,
 0x045c, 0x1c38, 0x2880, 0x4014, 0x10a4,
 0x035c, 0x1b38, 0x2780, 0x3f14, 0x0fa4,
 0x0360, 0x1b3c, 0x2784, 0x3f18, 0x0fa8,
 0x0460, 0x1c3c, 0x2884, 0x4018, 0x10a8,
 0x0560, 0x1d3c, 0x2984, 0x4118, 0x11a8,
 0x0564, 0x1d40, 0x2988, 0x411c, 0x11ac,
 0x0464, 0x1c40, 0x2888, 0x401c, 0x10ac,
 0x0364, 0x1b40, 0x2788, 0x3f1c, 0x0fac,
 0x0368, 0x1b44, 0x278c, 0x3f20, 0x0fb0,
 0x0468, 0x1c44, 0x288c, 0x4020, 0x10b0,
 0x0568, 0x1d44, 0x298c, 0x4120, 0x11b0,
 0x0948, 0x2124, 0x2d6c, 0x4500, 0x1590,
 0x0a48, 0x2224, 0x2e6c, 0x4600, 0x1690,
 0x0b48, 0x2324, 0x2f6c, 0x4700, 0x1790,
 0x0b4c, 0x2328, 0x2f70, 0x4704, 0x1794,
 0x0a4c, 0x2228, 0x2e70, 0x4604, 0x1694,
 0x094c, 0x2128, 0x2d70, 0x4504, 0x1594,
 0x0950, 0x212c, 0x2d74, 0x4508, 0x1598,
 0x0a50, 0x222c, 0x2e74, 0x4608, 0x1698,
 0x0b50, 0x232c, 0x2f74, 0x4708, 0x1798,
 0x0b54, 0x2330, 0x2f78, 0x470c, 0x179c,
 0x0a54, 0x2230, 0x2e78, 0x460c, 0x169c,
 0x0954, 0x2130, 0x2d78, 0x450c, 0x159c,
 0x0958, 0x2134, 0x2d7c, 0x4510, 0x15a0,
 0x0a58, 0x2234, 0x2e7c, 0x4610, 0x16a0,
 0x0b58, 0x2334, 0x2f7c, 0x4710, 0x17a0,
 0x0b5c, 0x2338, 0x2f80, 0x4714, 0x17a4,
 0x0a5c, 0x2238, 0x2e80, 0x4614, 0x16a4,
 0x095c, 0x2138, 0x2d80, 0x4514, 0x15a4,
 0x0960, 0x213c, 0x2d84, 0x4518, 0x15a8,
 0x0a60, 0x223c, 0x2e84, 0x4618, 0x16a8,
 0x0b60, 0x233c, 0x2f84, 0x4718, 0x17a8,
 0x0b64, 0x2340, 0x2f88, 0x471c, 0x17ac,
 0x0a64, 0x2240, 0x2e88, 0x461c, 0x16ac,
 0x0964, 0x2140, 0x2d88, 0x451c, 0x15ac,
 0x0968, 0x2144, 0x2d8c, 0x4520, 0x15b0,
 0x0a68, 0x2244, 0x2e8c, 0x4620, 0x16b0,
 0x0b68, 0x2344, 0x2f8c, 0x4720, 0x17b0,
};

/* DV25/50 DCT coefficient weights and inverse weights */
/* created by dvtables.py */
static const int dv_weight_bits = 18;
static const int dv_weight_88[64] = {
 131072, 257107, 257107, 242189, 252167, 242189, 235923, 237536,
 237536, 235923, 229376, 231390, 223754, 231390, 229376, 222935,
 224969, 217965, 217965, 224969, 222935, 200636, 218652, 211916,
 212325, 211916, 218652, 200636, 188995, 196781, 205965, 206433,
 206433, 205965, 196781, 188995, 185364, 185364, 200636, 200704,
 200636, 185364, 185364, 174609, 180568, 195068, 195068, 180568,
 174609, 170091, 175557, 189591, 175557, 170091, 165371, 170627,
 170627, 165371, 160727, 153560, 160727, 144651, 144651, 136258,
};
static const int dv_weight_248[64] = {
 131072, 242189, 257107, 237536, 229376, 200636, 242189, 223754,
 224969, 196781, 262144, 242189, 229376, 200636, 257107, 237536,
 211916, 185364, 235923, 217965, 229376, 211916, 206433, 180568,
 242189, 223754, 224969, 196781, 211916, 185364, 235923, 217965,
 200704, 175557, 222935, 205965, 200636, 185364, 195068, 170627,
 229376, 211916, 206433, 180568, 200704, 175557, 222935, 205965,
 175557, 153560, 188995, 174609, 165371, 144651, 200636, 185364,
 195068, 170627, 175557, 153560, 188995, 174609, 165371, 144651,
};
static const int dv_iweight_bits = 14;
static const int dv_iweight_88[64] = {
 32768, 16710, 16710, 17735, 17015, 17735, 18197, 18079,
 18079, 18197, 18725, 18559, 19196, 18559, 18725, 19284,
 19108, 19692, 19692, 19108, 19284, 21400, 19645, 20262,
 20214, 20262, 19645, 21400, 22733, 21845, 20867, 20815,
 20815, 20867, 21845, 22733, 23173, 23173, 21400, 21400,
 21400, 23173, 23173, 24600, 23764, 22017, 22017, 23764,
 24600, 25267, 24457, 22672, 24457, 25267, 25971, 25191,
 25191, 25971, 26715, 27962, 26715, 29642, 29642, 31536,
};
static const int dv_iweight_248[64] = {
 32768, 17735, 16710, 18079, 18725, 21400, 17735, 19196,
 19108, 21845, 16384, 17735, 18725, 21400, 16710, 18079,
 20262, 23173, 18197, 19692, 18725, 20262, 20815, 23764,
 17735, 19196, 19108, 21845, 20262, 23173, 18197, 19692,
 21400, 24457, 19284, 20867, 21400, 23173, 22017, 25191,
 18725, 20262, 20815, 23764, 21400, 24457, 19284, 20867,
 24457, 27962, 22733, 24600, 25971, 29642, 21400, 23173,
 22017, 25191, 24457, 27962, 22733, 24600, 25971, 29642,
};

static const uint8_t dv_audio_shuffle525[10][9] = {
  {  0, 30, 60, 20, 50, 80, 10, 40, 70 }, /* 1st channel */
  {  6, 36, 66, 26, 56, 86, 16, 46, 76 },
  { 12, 42, 72,  2, 32, 62, 22, 52, 82 },
  { 18, 48, 78,  8, 38, 68, 28, 58, 88 },
  { 24, 54, 84, 14, 44, 74,  4, 34, 64 },

  {  1, 31, 61, 21, 51, 81, 11, 41, 71 }, /* 2nd channel */
  {  7, 37, 67, 27, 57, 87, 17, 47, 77 },
  { 13, 43, 73,  3, 33, 63, 23, 53, 83 },
  { 19, 49, 79,  9, 39, 69, 29, 59, 89 },
  { 25, 55, 85, 15, 45, 75,  5, 35, 65 },
};

static const uint8_t dv_audio_shuffle625[12][9] = {
  {   0,  36,  72,  26,  62,  98,  16,  52,  88}, /* 1st channel */
  {   6,  42,  78,  32,  68, 104,  22,  58,  94},
  {  12,  48,  84,   2,  38,  74,  28,  64, 100},
  {  18,  54,  90,   8,  44,  80,  34,  70, 106},
  {  24,  60,  96,  14,  50,  86,   4,  40,  76},
  {  30,  66, 102,  20,  56,  92,  10,  46,  82},

  {   1,  37,  73,  27,  63,  99,  17,  53,  89}, /* 2nd channel */
  {   7,  43,  79,  33,  69, 105,  23,  59,  95},
  {  13,  49,  85,   3,  39,  75,  29,  65, 101},
  {  19,  55,  91,   9,  45,  81,  35,  71, 107},
  {  25,  61,  97,  15,  51,  87,   5,  41,  77},
  {  31,  67, 103,  21,  57,  93,  11,  47,  83},
};

static const av_unused int dv_audio_frequency[3] = {
    48000, 44100, 32000,
};

static const DVprofile dv_profiles[] = {
    { .dsf = 0,
      .frame_size = 120000,        /* IEC 61834, SMPTE-314M - 525/60 (NTSC) */
      .difseg_size = 10,
      .n_difchan = 1,
      .frame_rate = 30000,
      .ltc_divisor = 30,
      .frame_rate_base = 1001,
      .height = 480,
      .width = 720,
      .sar = {{10, 11}, {40, 33}},
      .video_place = dv_place_411,
      .pix_fmt = PIX_FMT_YUV411P,
      .audio_stride = 90,
      .audio_min_samples = { 1580, 1452, 1053 }, /* for 48, 44.1 and 32Khz */
      .audio_samples_dist = { 1600, 1602, 1602, 1602, 1602 }, /* per SMPTE-314M */
      .audio_shuffle = dv_audio_shuffle525,
    },
    { .dsf = 1,
      .frame_size = 144000,        /* IEC 61834 - 625/50 (PAL) */
      .difseg_size = 12,
      .n_difchan = 1,
      .frame_rate = 25,
      .frame_rate_base = 1,
      .ltc_divisor = 25,
      .height = 576,
      .width = 720,
      .sar = {{59, 54}, {118, 81}},
      .video_place = dv_place_420,
      .pix_fmt = PIX_FMT_YUV420P,
      .audio_stride = 108,
      .audio_min_samples = { 1896, 1742, 1264 }, /* for 48, 44.1 and 32Khz */
      .audio_samples_dist = { 1920, 1920, 1920, 1920, 1920 },
      .audio_shuffle = dv_audio_shuffle625,
    },
    { .dsf = 1,
      .frame_size = 144000,        /* SMPTE-314M - 625/50 (PAL) */
      .difseg_size = 12,
      .n_difchan = 1,
      .frame_rate = 25,
      .frame_rate_base = 1,
      .ltc_divisor = 25,
      .height = 576,
      .width = 720,
      .sar = {{59, 54}, {118, 81}},
      .video_place = dv_place_411P,
      .pix_fmt = PIX_FMT_YUV411P,
      .audio_stride = 108,
      .audio_min_samples = { 1896, 1742, 1264 }, /* for 48, 44.1 and 32Khz */
      .audio_samples_dist = { 1920, 1920, 1920, 1920, 1920 },
      .audio_shuffle = dv_audio_shuffle625,
    },
    { .dsf = 0,
      .frame_size = 240000,        /* SMPTE-314M - 525/60 (NTSC) 50 Mbps */
      .difseg_size = 10,           /* also known as "DVCPRO50" */
      .n_difchan = 2,
      .frame_rate = 30000,
      .ltc_divisor = 30,
      .frame_rate_base = 1001,
      .height = 480,
      .width = 720,
      .sar = {{10, 11}, {40, 33}},
      .video_place = dv_place_422_525,
      .pix_fmt = PIX_FMT_YUV422P,
      .audio_stride = 90,
      .audio_min_samples = { 1580, 1452, 1053 }, /* for 48, 44.1 and 32Khz */
      .audio_samples_dist = { 1600, 1602, 1602, 1602, 1602 }, /* per SMPTE-314M */
      .audio_shuffle = dv_audio_shuffle525,
    },
    { .dsf = 1,
      .frame_size = 288000,        /* SMPTE-314M - 625/50 (PAL) 50 Mbps */
      .difseg_size = 12,           /* also known as "DVCPRO50" */
      .n_difchan = 2,
      .frame_rate = 25,
      .frame_rate_base = 1,
      .ltc_divisor = 25,
      .height = 576,
      .width = 720,
      .sar = {{59, 54}, {118, 81}},
      .video_place = dv_place_422_625,
      .pix_fmt = PIX_FMT_YUV422P,
      .audio_stride = 108,
      .audio_min_samples = { 1896, 1742, 1264 }, /* for 48, 44.1 and 32Khz */
      .audio_samples_dist = { 1920, 1920, 1920, 1920, 1920 },
      .audio_shuffle = dv_audio_shuffle625,
    }
};

enum dv_section_type {
     dv_sect_header  = 0x1f,
     dv_sect_subcode = 0x3f,
     dv_sect_vaux    = 0x56,
     dv_sect_audio   = 0x76,
     dv_sect_video   = 0x96,
};

enum dv_pack_type {
     dv_header525     = 0x3f, /* see dv_write_pack for important details on */
     dv_header625     = 0xbf, /* these two packs */
     dv_timecode      = 0x13,
     dv_audio_source  = 0x50,
     dv_audio_control = 0x51,
     dv_audio_recdate = 0x52,
     dv_audio_rectime = 0x53,
     dv_video_source  = 0x60,
     dv_video_control = 0x61,
     dv_video_recdate = 0x62,
     dv_video_rectime = 0x63,
     dv_unknown_pack  = 0xff,
};

/* minimum number of bytes to read from a DV stream in order to determine the profile */
#define DV_PROFILE_BYTES (6*80) /* 6 DIF blocks */

/* largest possible DV frame, in bytes (PAL 50Mbps) */
#define DV_MAX_FRAME_SIZE 288000

static inline const DVprofile* dv_frame_profile(const uint8_t* frame)
{
    if ((frame[3] & 0x80) == 0) {      /* DSF flag */
        /* it's an NTSC format */
        if ((frame[80*5 + 48 + 3] & 0x4) && (frame[80*5 + 48] == dv_video_source)) { /* 4:2:2 sampling */
            return &dv_profiles[3]; /* NTSC 50Mbps */
        } else { /* 4:1:1 sampling */
            return &dv_profiles[0]; /* NTSC 25Mbps */
        }
    } else {
        /* it's a PAL format */
        if ((frame[80*5 + 48 + 3] & 0x4) && (frame[80*5 + 48] == dv_video_source)) { /* 4:2:2 sampling */
            return &dv_profiles[4]; /* PAL 50Mbps */
        } else if ((frame[5] & 0x07) == 0) { /* APT flag */
            return &dv_profiles[1]; /* PAL 25Mbps 4:2:0 */
        } else
            return &dv_profiles[2]; /* PAL 25Mbps 4:1:1 */
    }
}

static inline const DVprofile* dv_codec_profile(AVCodecContext* codec)
{
    int i;

    if (codec->width != 720)
        return NULL;

    for (i=0; i<sizeof(dv_profiles)/sizeof(DVprofile); i++)
       if (codec->height == dv_profiles[i].height && codec->pix_fmt == dv_profiles[i].pix_fmt)
           return &dv_profiles[i];

    return NULL;
}

static inline int dv_write_dif_id(enum dv_section_type t, uint8_t chan_num, uint8_t seq_num,
                                  uint8_t dif_num, uint8_t* buf)
{
    buf[0] = (uint8_t)t;    /* Section type */
    buf[1] = (seq_num<<4) | /* DIF seq number 0-9 for 525/60; 0-11 for 625/50 */
             (chan_num << 3) | /* FSC: for 50Mb/s 0 - first channel; 1 - second */
             7;             /* reserved -- always 1 */
    buf[2] = dif_num;       /* DIF block number Video: 0-134, Audio: 0-8 */
    return 3;
}


static inline int dv_write_ssyb_id(uint8_t syb_num, uint8_t fr, uint8_t* buf)
{
    if (syb_num == 0 || syb_num == 6) {
        buf[0] = (fr<<7) | /* FR ID 1 - first half of each channel; 0 - second */
                 (0<<4)  | /* AP3 (Subcode application ID) */
                 0x0f;     /* reserved -- always 1 */
    }
    else if (syb_num == 11) {
        buf[0] = (fr<<7) | /* FR ID 1 - first half of each channel; 0 - second */
                 0x7f;     /* reserved -- always 1 */
    }
    else {
        buf[0] = (fr<<7) | /* FR ID 1 - first half of each channel; 0 - second */
                 (0<<4)  | /* APT (Track application ID) */
                 0x0f;     /* reserved -- always 1 */
    }
    buf[1] = 0xf0 |            /* reserved -- always 1 */
             (syb_num & 0x0f); /* SSYB number 0 - 11   */
    buf[2] = 0xff;             /* reserved -- always 1 */
    return 3;
}

#endif /* FFMPEG_DVDATA_H */
