/** 
 * \file hotplug.c
 * Example program to create hotplug scripts.
 *
 * Copyright (C) 2005-2007 Linus Walleij <triad@df.lth.se>
 * Copyright (C) 2006 Marcus Meissner <marcus@jet.franken.de>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 */
#include "common.h"
#include <unistd.h>
#include <stdlib.h>
#include <stdio.h>
#include <string.h>

static void usage(void)
{
  fprintf(stderr, "usage: hotplug [-h -u -a\"ACTION\"]\n");
  fprintf(stderr, "       -u:  use udev syntax\n");
  fprintf(stderr, "       -H:  use hal syntax\n");
  fprintf(stderr, "       -a\"ACTION\": perform udev action ACTION on attachment\n");
  exit(1);
}

enum style {
	style_usbmap,
	style_udev,
	style_hal
};

int main (int argc, char **argv)
{
  LIBMTP_device_entry_t *entries;
  int numentries;
  int i;
  int ret;
  enum style style = style_usbmap;
  int opt;
  extern int optind;
  extern char *optarg;
  char *udev_action = NULL;
  char default_udev_action[] = "SYMLINK+=\"libmtp-%k\", MODE=\"666\"";

  while ( (opt = getopt(argc, argv, "uHa:")) != -1 ) {
    switch (opt) {
    case 'a':
      udev_action = strdup(optarg);
    case 'u':
      style = style_udev;
      break;
    case 'H':
      style = style_hal;
      break;
    default:
      usage();
    }
  }

  LIBMTP_Init();
  ret = LIBMTP_Get_Supported_Devices_List(&entries, &numentries);
  if (ret == 0) {
    switch (style) {
    case style_udev:
      printf("# UDEV-style hotplug map for libmtp\n");
      printf("# Put this file in /etc/udev/rules.d\n\n");
      printf("SUBSYSTEM!=\"usb_device\", ACTION!=\"add\", GOTO=\"libmtp_rules_end\"\n\n");
      break;
    case style_usbmap:
      printf("# This usermap will call the script \"libmtp.sh\" whenever a known MTP device is attached.\n\n");
      break;
    case style_hal:
      printf("<?xml version=\"1.0\" encoding=\"ISO-8859-1\"?> <!-- -*- SGML -*- -->\n");
      printf("<!-- This file was generated by %s - - fdi -->\n", argv[0]);
      printf("<deviceinfo version=\"0.2\">\n");
      printf(" <device>\n");
      printf("  <match key=\"info.bus\" string=\"usb\">\n");
      break;
    }

    for (i = 0; i < numentries; i++) {
      LIBMTP_device_entry_t * entry = &entries[i];
      
      switch (style) {
      case style_udev: {
	char *action;

        printf("# %s\n", entry->name);
	if (udev_action != NULL) {
	  action = udev_action;
	} else {
	  action = default_udev_action;
	}
	printf("SYSFS{idVendor}==\"%04x\", SYSFS{idProduct}==\"%04x\", %s\n", entry->vendor_id, entry->product_id, action);
	break;
      }
      case style_usbmap:
        printf("# %s\n", entry->name);
	printf("libmtp.sh    0x0003  0x%04x  0x%04x  0x0000  0x0000  0x00    0x00    0x00    0x00    0x00    0x00    0x00000000\n", entry->vendor_id, entry->product_id);
	break;
      case style_hal:
	printf("   <match key=\"usb.vendor_id\" int=\"%d\">\n", entry->vendor_id);
	printf("    <match key=\"usb.product_id\" int=\"%d\">\n", entry->product_id);
	printf("     <merge key=\"info.category\" type=\"string\">portable_audio_player</merge>\n");
	printf("     <append key=\"info.capabilities\" type=\"strlist\">portable_audio_player</append>\n");
	printf("     <merge key=\"portable_audio_player.access_method\" type=\"string\">user</merge>\n");
	printf("     <merge key=\"portable_audio_player.type\" type=\"string\">mtp</merge>\n");

	/* FIXME: needs true list of formats ... But all of them can do MP3 */
	printf("     <append key=\"portable_audio_player.output_formats\" type=\"strlist\">audio/mpeg</append>\n");
	printf("    </match>\n");
	printf("   </match>\n");
	break;
      }
    }
  } else {
    printf("Error.\n");
    exit(1);
  }

  switch (style) {
  case style_usbmap:
    break;
  case style_udev:
    printf("\nLABEL=\"libmtp_rules_end\"\n");
    break;
  case style_hal:
    printf("  </match>\n");
    printf(" </device>\n");
    printf("</deviceinfo>\n");
    break;
  }

  exit (0);
}
