#include <stdio.h>
#include <stdarg.h>
#include <string.h>
#include <unistd.h>
#include <math.h>

#include <glib.h>
#include <glib-object.h>

#include <gdk-pixbuf/gdk-pixdata.h>

#include <libxfcegui4/dialogs.h>
#include <libxfcegui4/libxfcegui4.h>
#include <libxfcegui4/netk-screen.h>
#include <libxfcegui4/netk-window.h>
#include <libxfce4util/libxfce4util.h>

#include "ls_iconboxwin.h"

/* keep in sync with panel */
#define HANDLE_WIDTH 10
#define SNAP_WIDTH   20
#define OPAQUE       0xffffffff
#define USE_OPACITY  1

struct _LSIconBoxWinPrivate
{
    LSIconBox *pIconBox;
    GtkVBox *pVBox;
    GtkHBox *pHBox;
    GtkFrame *pFrame;
    GtkWidget *pHandleRight;
    GtkWidget *pHandleLeft;
    GtkWidget *pMenu;
    GdkPixbuf *pIcon;
    gboolean bFirstOrientationChange;
    gboolean bHideOnEmpty;

    glong lOrientation;

    struct _LSIconBoxWinPrivateOffset
    {
	gint nX;
	gint nY;
    } offset;

    struct _LSIconBoxWinPrivetPosition
    {
	glong horizontal;
	glong vertical;
    } position;

    struct _LSIconBoxWinPrivateSignals
    {
	gulong button_release_event[2];
	gulong size_allocate;
	gulong event[2];
    } signals;
};

static void ls_iconboxwin_init (LSIconBoxWin * plsibw);
static void ls_iconboxwin_class_init (LSIconBoxWinClass * plsibwc);
static void ls_iconboxwin_finalize (GObject * pgo);
GType ls_iconboxwin_get_type ();

/* CALLBACKS									*/
static void cb_lsibw_handle_move_end (GtkWidget * pWidget, gpointer pUserData);
static gboolean cb_lsibw_size_allocate (GtkWidget * pWidget, 
                                        GtkAllocation * pAlloc,
				        gpointer pData);
static gboolean cb_lsibw_handle_button2_pressed (GtkWidget * pWidget,
					         GdkEvent * pEvent,
					         gpointer * pUserData);
static void cb_lsibw_iconbox_hide (GtkWidget * pWidget, gpointer pUserData);
static void cb_lsibw_iconbox_show (GtkWidget * pWidget, gpointer pUserData);

#if USE_OPACITY
static void set_window_opacity (GdkWindow *window, gboolean translucent);

static gboolean cb_lsibw_enter (GtkWidget *pWidget, 
                                GdkEventCrossing *event, 
                                gpointer data);

static gboolean cb_lsibw_leave (GtkWidget *pWidget, 
                                GdkEventCrossing *event, 
                                gpointer data);
#endif

static guint transparency = 0xc0000000;

static gpointer parent_class = NULL;

/********************************************************************************/
/* GLIB OBJECT FUNCTIONS							*/
/********************************************************************************/
GType
ls_iconboxwin_get_type ()
{
    static GType object_type = 0;

    if (!object_type)
    {
	static const GTypeInfo object_info = {
	    sizeof (LSIconBoxWinClass),
	    (GBaseInitFunc) NULL,
	    (GBaseFinalizeFunc) NULL,
	    (GClassInitFunc) ls_iconboxwin_class_init,
	    NULL,		/* class_finalize */
	    NULL,		/* class_data */
	    sizeof (LSIconBoxWin),
	    0,			/* n_preallocs */
	    (GInstanceInitFunc) ls_iconboxwin_init
	};

	object_type = g_type_register_static (	/* GTK_TYPE_BUTTON */
						 GTK_TYPE_WINDOW,
						 "LSIconBoxWin", &object_info,
						 0);
    }

    return object_type;
}

static void
ls_iconboxwin_init (LSIconBoxWin * plsibw)
{
    return;
}

static void
ls_iconboxwin_class_init (LSIconBoxWinClass * plsic)
{
    GObjectClass *object_class = G_OBJECT_CLASS (plsic);

    parent_class = g_type_class_peek_parent (plsic);
    object_class->finalize = ls_iconboxwin_finalize;


    return;
}

static void
ls_iconboxwin_finalize (GObject * pgo)
{
    LSIconBoxWin *plsibw;

    g_return_if_fail (LS_IS_ICONBOXWIN (pgo));
    plsibw = LS_ICONBOXWIN (pgo);

    g_object_unref (plsibw->priv->pIcon);

    /* TODO do we really need to remove the handles for objects we destroy?       */
    /* i think not!                                                               */
    /*
       g_signal_handler_disconnect(G_OBJECT(plsibw->priv->pHandleLeft), 
       plsibw->priv->signals.button_release_event[0]);
       g_signal_handler_disconnect(G_OBJECT(plsibw->priv->pHandleRight),
       plsibw->priv->signals.button_release_event[1]);
       g_signal_handler_disconnect(G_OBJECT(plsibw), 
       plsibw->priv->signals.size_allocate);
       g_signal_handler_disconnect(G_OBJECT(plsibw->priv->pHandleLeft),
       plsibw->priv->signals.event[0]);
       g_signal_handler_disconnect(G_OBJECT(plsibw->priv->pHandleRight),
       plsibw->priv->signals.event[1]);
     */

    /*
       g_object_unref(plsibw->priv->pIconBox);
       g_object_unref(plsibw->priv->pFrame);
       g_object_unref(plsibw->priv->pHandleRight);
       g_object_unref(plsibw->priv->pHandleLeft);
       g_object_unref(plsibw->priv->pMenu);
     */

    if (plsibw->priv->lOrientation == LS_ICONBOX_HORIZONTAL)
    {
	g_object_unref (plsibw->priv->pVBox);
    }
    else
    {
	g_object_unref (plsibw->priv->pHBox);
    }

    G_OBJECT_CLASS (parent_class)->finalize (pgo);

    return;
}

/********************************************************************************/
/* CONSTRUCTORS									*/
/********************************************************************************/
LSIconBoxWin *
ls_iconboxwin_new (NetkScreen * pScreen)
{
    LSIconBoxWin *plsibw;

    /* INIT */
    plsibw = 
        LS_ICONBOXWIN (g_object_new (LS_TYPE_ICONBOXWIN,
			             "type", GTK_WINDOW_TOPLEVEL, 
                                     "type-hint", GDK_WINDOW_TYPE_HINT_DOCK,
                                     NULL));
    plsibw->priv = g_new (LSIconBoxWinPrivate, 1);
    plsibw->priv->lOrientation = LS_ICONBOX_HORIZONTAL;
    plsibw->priv->pIconBox = LS_ICONBOX (ls_iconbox_new (pScreen,
							 plsibw->priv->
							 lOrientation));
    plsibw->priv->pHBox = GTK_HBOX (gtk_hbox_new (FALSE, 0));
    plsibw->priv->pVBox = GTK_VBOX (gtk_vbox_new (FALSE, 0));
    plsibw->priv->pFrame = GTK_FRAME (gtk_frame_new (NULL));
    plsibw->priv->bHideOnEmpty = TRUE;
    plsibw->priv->pHandleLeft = xfce_movehandler_new (GTK_WIDGET (plsibw));
    plsibw->priv->pHandleRight = xfce_movehandler_new (GTK_WIDGET (plsibw));
    plsibw->priv->pIcon = xfce_themed_icon_load ("xfce4-iconbox", 48);

    gtk_widget_set_size_request (plsibw->priv->pHandleLeft, HANDLE_WIDTH,
	    			 HANDLE_WIDTH);
    gtk_widget_set_size_request (plsibw->priv->pHandleRight, HANDLE_WIDTH,
	    			 HANDLE_WIDTH);
    
    g_signal_connect_object (G_OBJECT (plsibw->priv->pIconBox), "hide",
			     G_CALLBACK (cb_lsibw_iconbox_hide), plsibw, 0);
    g_signal_connect_object (G_OBJECT (plsibw->priv->pIconBox), "show",
			     G_CALLBACK (cb_lsibw_iconbox_show), plsibw, 0);

    ls_iconboxwin_set_offset (plsibw, 0, 0);
    ls_iconboxwin_set_vertical_position (plsibw, LS_ICONBOX_BOTTOM);
    ls_iconboxwin_set_horizontal_position (plsibw, LS_ICONBOX_RIGHT);

    gtk_window_set_icon (GTK_WINDOW (plsibw), plsibw->priv->pIcon);
    gtk_frame_set_shadow_type (GTK_FRAME (plsibw->priv->pFrame),
			       GTK_SHADOW_OUT);
    gtk_container_set_border_width (GTK_CONTAINER (plsibw->priv->pFrame), 0);

    gtk_container_set_border_width (GTK_CONTAINER (plsibw->priv->pVBox), 0);
    gtk_container_set_border_width (GTK_CONTAINER (plsibw->priv->pHBox), 0);
    gtk_container_set_border_width (GTK_CONTAINER (plsibw), 0);

#if 0
    gtk_window_set_skip_taskbar_hint (GTK_WINDOW (plsibw), TRUE);
    gtk_window_set_skip_pager_hint (GTK_WINDOW (plsibw), TRUE);
#endif
    gtk_window_stick (GTK_WINDOW (plsibw));
    gtk_window_set_decorated (GTK_WINDOW (plsibw), FALSE);
    gtk_window_set_resizable (GTK_WINDOW (plsibw), FALSE);
    gtk_window_set_title (GTK_WINDOW (plsibw), "Lil' Star - IconBoxWindow");

    plsibw->priv->signals.button_release_event[0] =
	g_signal_connect (G_OBJECT (plsibw->priv->pHandleLeft),
			  "move-end",
			  G_CALLBACK (cb_lsibw_handle_move_end),
			  (gpointer) plsibw);
    plsibw->priv->signals.button_release_event[1] =
	g_signal_connect (G_OBJECT (plsibw->priv->pHandleRight),
			  "move-end",
			  G_CALLBACK (cb_lsibw_handle_move_end),
			  (gpointer) plsibw);
    plsibw->priv->signals.size_allocate =
	g_signal_connect (G_OBJECT (plsibw), "size_allocate",
			  G_CALLBACK (cb_lsibw_size_allocate),
			  (gpointer) plsibw);

#if USE_OPACITY
    g_signal_connect (plsibw, "enter-notify-event",
		      G_CALLBACK (cb_lsibw_enter), NULL);

    g_signal_connect (plsibw, "leave-notify-event",
		      G_CALLBACK (cb_lsibw_leave), NULL);
#endif

    gtk_container_add (GTK_CONTAINER (plsibw),
		       GTK_WIDGET (plsibw->priv->pFrame));

    gtk_widget_show (GTK_WIDGET (plsibw->priv->pFrame));
    gtk_widget_show (GTK_WIDGET (plsibw->priv->pHandleLeft));
    gtk_widget_show (GTK_WIDGET (plsibw->priv->pHandleRight));
    gtk_widget_show (GTK_WIDGET (plsibw->priv->pIconBox));

    if (plsibw->priv->lOrientation == LS_ICONBOX_HORIZONTAL)
    {
	gtk_box_pack_start (GTK_BOX (plsibw->priv->pHBox),
			    plsibw->priv->pHandleLeft, FALSE, FALSE, 0);
	gtk_box_pack_start (GTK_BOX (plsibw->priv->pHBox),
			    GTK_WIDGET (plsibw->priv->pIconBox), FALSE, FALSE,
			    0);
	gtk_box_pack_start (GTK_BOX (plsibw->priv->pHBox),
			    plsibw->priv->pHandleRight, FALSE, FALSE, 0);

	gtk_container_add (GTK_CONTAINER (plsibw->priv->pFrame),
			   GTK_WIDGET (plsibw->priv->pHBox));
	gtk_widget_show (GTK_WIDGET (plsibw->priv->pHBox));

	g_object_ref (plsibw->priv->pVBox);
    }
    else
    {
	gtk_box_pack_start (GTK_BOX (plsibw->priv->pVBox),
			    plsibw->priv->pHandleLeft, FALSE, FALSE, 0);
	gtk_box_pack_start (GTK_BOX (plsibw->priv->pVBox),
			    GTK_WIDGET (plsibw->priv->pIconBox), FALSE, FALSE,
			    0);
	gtk_box_pack_start (GTK_BOX (plsibw->priv->pVBox),
			    plsibw->priv->pHandleRight, FALSE, FALSE, 0);

	gtk_container_add (GTK_CONTAINER (plsibw->priv->pFrame),
			   GTK_WIDGET (plsibw->priv->pVBox));
	gtk_widget_show (GTK_WIDGET (plsibw->priv->pVBox));

	g_object_ref (plsibw->priv->pHBox);
    }

    /* POPUP MENU                                                                 */
    ls_iconboxwin_create_menu (plsibw);

    plsibw->priv->signals.event[0] =
	g_signal_connect_object (G_OBJECT (plsibw->priv->pHandleLeft),
				 "event",
				 G_CALLBACK (cb_lsibw_handle_button2_pressed),
				 (gpointer) plsibw, 0);
    plsibw->priv->signals.event[0] =
	g_signal_connect_object (G_OBJECT (plsibw->priv->pHandleRight),
				 "event",
				 G_CALLBACK (cb_lsibw_handle_button2_pressed),
				 (gpointer) plsibw, 0);

    return plsibw;
}

/********************************************************************************/
/* FUNCTIONS                                                                    */
/********************************************************************************/
void
ls_iconboxwin_set_hide_on_empty (LSIconBoxWin * plsibw, gboolean bHide)
{
    gboolean bVisible;

    if (plsibw->priv->bHideOnEmpty == bHide)
	return;

    ls_iconbox_set_hide_on_empty (plsibw->priv->pIconBox, bHide);

    if (plsibw->priv->bHideOnEmpty == TRUE && bHide == FALSE)
    {
	/* simply show them */
	gtk_widget_show (GTK_WIDGET (plsibw->priv->pHandleRight));
	gtk_widget_show (GTK_WIDGET (plsibw->priv->pHandleLeft));
	plsibw->priv->bHideOnEmpty = FALSE;
    }

    if (plsibw->priv->bHideOnEmpty == FALSE && bHide == TRUE)
    {
	g_object_get (G_OBJECT (plsibw->priv->pIconBox), "visible", &bVisible,
		      NULL);
	if (bVisible == FALSE)
	{
	    gtk_widget_hide (GTK_WIDGET (plsibw->priv->pHandleRight));
	    gtk_widget_hide (GTK_WIDGET (plsibw->priv->pHandleLeft));
	}
	plsibw->priv->bHideOnEmpty = TRUE;
    }

    return;
}

LSIconBox *
ls_iconboxwin_get_iconbox (LSIconBoxWin * plsibw)
{
    return plsibw->priv->pIconBox;
}

void
ls_iconboxwin_set_horizontal_position (LSIconBoxWin * plsibw, glong lAlign)
{
    g_return_if_fail (LS_IS_ICONBOXWIN (plsibw));

    if (lAlign != LS_ICONBOX_LEFT && lAlign != LS_ICONBOX_RIGHT)
	return;

    plsibw->priv->position.horizontal = lAlign;
    ls_iconboxwin_re_evaluate_position (plsibw);
    return;
}

void
ls_iconboxwin_set_vertical_position (LSIconBoxWin * plsibw, glong lAlign)
{
    g_return_if_fail (LS_IS_ICONBOXWIN (plsibw));

    if (lAlign != LS_ICONBOX_TOP && lAlign != LS_ICONBOX_BOTTOM)
	return;

    plsibw->priv->position.vertical = lAlign;
    ls_iconboxwin_re_evaluate_position (plsibw);
    return;
}

void
ls_iconboxwin_set_offset (LSIconBoxWin * plsibw, gint nX, gint nY)
{
    g_return_if_fail (LS_IS_ICONBOXWIN (plsibw));

    DBG (" X:%d Y:%d", nX, nY);

    if (nY >= 0)
    {
	plsibw->priv->offset.nY = nY;
    }
    if (nX >= 0)
    {
	plsibw->priv->offset.nX = nX;
    }

    ls_iconboxwin_re_evaluate_position (plsibw);
    return;
}

void
ls_iconboxwin_get_offset (LSIconBoxWin * plsibw, gint * nX, gint * nY)
{
    g_return_if_fail (LS_IS_ICONBOXWIN (plsibw));

    *nY = plsibw->priv->offset.nY;
    *nX = plsibw->priv->offset.nX;

    return;
}

void
ls_iconboxwin_show (LSIconBoxWin * plsibw)
{
    gtk_widget_show (GTK_WIDGET (plsibw));
#if USE_OPACITY
    set_window_opacity (GTK_WIDGET (plsibw)->window, TRUE);
#endif
    return;
}

void
ls_iconboxwin_hide (LSIconBoxWin * plsibw)
{
    gtk_widget_hide (GTK_WIDGET (plsibw));

    return;
}

void
ls_iconboxwin_set_orientation (LSIconBoxWin * plsibw, glong lOrientation)
{
    GtkContainer *pSrc;
    GtkContainer *pDst;

    if (plsibw->priv->lOrientation == lOrientation)
	return;

    plsibw->priv->lOrientation = lOrientation;

    if (plsibw->priv->lOrientation == LS_ICONBOX_HORIZONTAL)
    {
	pSrc = GTK_CONTAINER (plsibw->priv->pVBox);
	pDst = GTK_CONTAINER (plsibw->priv->pHBox);
    }
    else
    {
	pSrc = GTK_CONTAINER (plsibw->priv->pHBox);
	pDst = GTK_CONTAINER (plsibw->priv->pVBox);
    }

    gtk_widget_ref (GTK_WIDGET (plsibw->priv->pHandleLeft));
    gtk_container_remove (pSrc, plsibw->priv->pHandleLeft);
    gtk_box_pack_start (GTK_BOX (pDst), plsibw->priv->pHandleLeft, FALSE,
			FALSE, 0);
    gtk_widget_unref (GTK_WIDGET (plsibw->priv->pHandleLeft));

    gtk_widget_ref (GTK_WIDGET (plsibw->priv->pIconBox));
    gtk_container_remove (pSrc, GTK_WIDGET (plsibw->priv->pIconBox));
    gtk_box_pack_start (GTK_BOX (pDst), GTK_WIDGET (plsibw->priv->pIconBox),
			FALSE, FALSE, 0);
    gtk_widget_unref (GTK_WIDGET (plsibw->priv->pIconBox));

    gtk_widget_ref (GTK_WIDGET (plsibw->priv->pHandleRight));
    gtk_container_remove (pSrc, plsibw->priv->pHandleRight);
    gtk_box_pack_start (GTK_BOX (pDst), plsibw->priv->pHandleRight, FALSE,
			FALSE, 0);
    gtk_widget_unref (GTK_WIDGET (plsibw->priv->pHandleRight));

    g_object_ref (pSrc);
    gtk_container_remove (GTK_CONTAINER (plsibw->priv->pFrame),
			  GTK_WIDGET (pSrc));
    gtk_container_add (GTK_CONTAINER (plsibw->priv->pFrame),
		       GTK_WIDGET (pDst));
    g_object_unref (pDst);

    gtk_widget_show (GTK_WIDGET (pDst));
    gtk_widget_hide (GTK_WIDGET (pSrc));

    ls_iconbox_set_orientation (ls_iconboxwin_get_iconbox (plsibw),
				lOrientation);

    ls_iconboxwin_re_evaluate_position (plsibw);

    return;
}

void
ls_iconboxwin_re_evaluate_position (LSIconBoxWin * plsibw)
{
    gint nX;
    gint nY;
    gint nWidth;
    gint nHeight;
    gint nActX;
    gint nActY;
    gint nScreenWidth;
    gint nScreenHeight;
    gint monitor;
    GdkScreen *screen;
    GdkRectangle r;

    gtk_window_get_size (GTK_WINDOW (plsibw), &nWidth, &nHeight);
    gtk_window_get_position (GTK_WINDOW (plsibw), &nActX, &nActY);

    screen = gtk_widget_get_screen (GTK_WIDGET (plsibw));
    monitor = gdk_screen_get_monitor_at_point (screen, 
                                               nActX +  nWidth / 2,
                                               nActY + nHeight / 2);
    gdk_screen_get_monitor_geometry (screen, monitor, &r);
    nScreenWidth = gdk_screen_get_width (screen);
    nScreenHeight = gdk_screen_get_height (screen);
    
    if (ls_iconboxwin_get_horizontal_position (plsibw) == LS_ICONBOX_RIGHT)
    {
	nX =  nScreenWidth - nWidth - plsibw->priv->offset.nX;
    }
    else
    {
	nX = plsibw->priv->offset.nX;
    }

    if (ls_iconboxwin_get_vertical_position (plsibw) == LS_ICONBOX_BOTTOM)
    {
	nY = nScreenHeight - nHeight - plsibw->priv->offset.nY;
    }
    else
    {
	nY = plsibw->priv->offset.nY;
    }

    if (nX + nWidth > r.x + r.width - SNAP_WIDTH)
        nX = r.x + r.width - nWidth;
    
    if (nX <= r.x + SNAP_WIDTH)
        nX = r.x;

    if (nY + nHeight > r.y + r.height - SNAP_WIDTH)
        nY = r.y + r.height - nHeight;
    
    if (nY <= r.y + SNAP_WIDTH)
        nY = r.y;

    if (nX != nActX || nY != nActY)
    {
	gtk_window_move (GTK_WINDOW (plsibw), nX, nY);
    }

    return;
}

glong
ls_iconboxwin_get_horizontal_position (LSIconBoxWin * plsib)
{
    return plsib->priv->position.horizontal;
}

glong
ls_iconboxwin_get_vertical_position (LSIconBoxWin * plsib)
{
    return plsib->priv->position.vertical;
}

/*******************************************************************************/
/* CALLBACKS                                                                   */
/*******************************************************************************/
static gboolean
cb_lsibw_size_allocate (GtkWidget * pWidget,
			GtkAllocation * pAlloc, gpointer pData)
{
    g_return_val_if_fail (LS_IS_ICONBOXWIN (pWidget), FALSE);

    ls_iconboxwin_re_evaluate_position (LS_ICONBOXWIN (pWidget));

    gtk_widget_queue_draw (pWidget);

    return TRUE;
}

static void
cb_lsibw_handle_move_end (GtkWidget * pWidget, gpointer pUserData)
{
    LSIconBoxWin *plsibw = LS_ICONBOXWIN (pUserData);
    gint lXOff, lYOff;
    gint nWidth;
    gint nHeight;
    gint nX;
    gint nY;
    gint nScreenWidth;
    gint nScreenHeight;
    GdkScreen *screen;

    gtk_window_get_size (GTK_WINDOW (plsibw), &nWidth, &nHeight);
    gtk_window_get_position (GTK_WINDOW (plsibw), &nX, &nY);

    screen = gtk_widget_get_screen (GTK_WIDGET (plsibw));
    nScreenWidth = gdk_screen_get_width (screen);
    nScreenHeight = gdk_screen_get_height (screen);

    if (ls_iconboxwin_get_horizontal_position (plsibw) == LS_ICONBOX_RIGHT)
    {
        lXOff = nScreenWidth - nWidth - nX;
    }
    else
    {
        lXOff = nX;
    }

    if (ls_iconboxwin_get_vertical_position (plsibw) == LS_ICONBOX_BOTTOM)
    {
        lYOff = nScreenHeight - nHeight - nY;
    }
    else
    {
        lYOff = nY;
    }

    plsibw->priv->offset.nX = lXOff;
    plsibw->priv->offset.nY = lYOff;
    
    ls_iconboxwin_re_evaluate_position (plsibw);
}

static gboolean
cb_lsibw_handle_button2_pressed (GtkWidget * pWidget,
				 GdkEvent * pEvent, gpointer * pUserData)
{
    if (pEvent->type != GDK_BUTTON_PRESS)
	return FALSE;
    if (((GdkEventButton *) pEvent)->button != 3)
	return FALSE;

    gtk_menu_popup (GTK_MENU (LS_ICONBOXWIN (pUserData)->priv->pMenu), NULL,
		    NULL, NULL, NULL, ((GdkEventButton *) pEvent)->button,
		    ((GdkEventButton *) pEvent)->time);

    return TRUE;
}

static void
cb_lsibw_iconbox_hide (GtkWidget * pWidget, gpointer pUserData)
{
    g_return_if_fail (LS_IS_ICONBOXWIN (pUserData));

    if (LS_ICONBOXWIN (pUserData)->priv->bHideOnEmpty == TRUE)
    {
	gtk_widget_hide (GTK_WIDGET
			 (LS_ICONBOXWIN (pUserData)->priv->pHandleRight));
	gtk_widget_hide (GTK_WIDGET
			 (LS_ICONBOXWIN (pUserData)->priv->pHandleLeft));
	gtk_widget_hide (GTK_WIDGET
			 (LS_ICONBOXWIN (pUserData)->priv->pFrame));
    }

    return;
}

static void
cb_lsibw_iconbox_show (GtkWidget * pWidget, gpointer pUserData)
{
    g_return_if_fail (LS_IS_ICONBOXWIN (pUserData));

    gtk_widget_show (GTK_WIDGET
		     (LS_ICONBOXWIN (pUserData)->priv->pHandleRight));
    gtk_widget_show (GTK_WIDGET
		     (LS_ICONBOXWIN (pUserData)->priv->pHandleLeft));
    gtk_widget_show (GTK_WIDGET (LS_ICONBOXWIN (pUserData)->priv->pFrame));

    return;
}

#if USE_OPACITY
static void
get_opacity_setting (void)
{
    char *file = 
        xfce_resource_lookup (XFCE_RESOURCE_CONFIG,
                              "xfce4" G_DIR_SEPARATOR_S "transparency");

    if (file)
    {
        FILE *fp;
        
        if ((fp = fopen (file, "r")) != NULL)
        {
            char line[255];
            guint trans;

            while (fgets(line, sizeof (line), fp))
            {
                if (sscanf (line, "iconbox = %u", &trans) > 0)
                {
                    trans = CLAMP (trans, 0, 80);

                    transparency = OPAQUE - rint ((double)trans * OPAQUE / 100);

                    DBG ("transparency: %u\n", trans);
                    DBG ("opacity: 0x%x\n", transparency);

                    break;
                }
            }

            fclose (fp);
        }
        
        g_free (file);
    }
}


    
static void
set_window_opacity (GdkWindow *window, gboolean translucent)
{
    static gboolean initialized = FALSE;
    guint opacity;

    if (!window)
        return;
    
    if (G_UNLIKELY (!initialized))
    {
        get_opacity_setting ();
        
        initialized = TRUE;
    }
    
    if (transparency == OPAQUE)
        return;
    
    opacity = (translucent ? transparency : OPAQUE);
    gdk_error_trap_push ();

    gdk_property_change (window,
			 gdk_atom_intern ("_NET_WM_WINDOW_OPACITY", FALSE),
			 gdk_atom_intern ("CARDINAL", FALSE), 32,
			 GDK_PROP_MODE_REPLACE, (guchar *) & opacity, 1L);

    gdk_error_trap_pop ();
}

static gboolean 
cb_lsibw_enter (GtkWidget *pWidget, GdkEventCrossing *event, gpointer data)
{
    if (pWidget->window && event->detail != GDK_NOTIFY_INFERIOR)
        set_window_opacity (pWidget->window, FALSE);
    
    return FALSE;
}

static gboolean 
cb_lsibw_leave (GtkWidget *pWidget, GdkEventCrossing *event, gpointer data)
{
    if (pWidget->window && event->detail != GDK_NOTIFY_INFERIOR)
        set_window_opacity (pWidget->window, TRUE);
    
    return FALSE;
}
#endif


/********************************************************************************/
/* MENU										*/
/********************************************************************************/
static void cb_lsibwm_close_iconboxwin (gpointer pUserData, gint nAction,
					GtkWidget * pWidget);
static void cb_lsibwm_save_pos (gpointer pUserData, gint nAction,
				GtkWidget * pWidget);
static void cb_lsibwm_open_iconbox_config (gpointer pUserData, gint nAction,
					   GtkWidget * pWidget);

static GtkItemFactoryEntry lsIconBoxWinMenu[] = {
    {N_("/Close"), NULL, cb_lsibwm_close_iconboxwin, 0, "<Item>"},
    {N_("/Save Position"), NULL, cb_lsibwm_save_pos, 0, "<Item>"},
    {"/Sep0", NULL, NULL, 0, "<Separator>"},
    {N_("/Properties..."), NULL, cb_lsibwm_open_iconbox_config, 0, "<Item>"}
};

static gint nLsIconBoxWinMenu =
    sizeof (lsIconBoxWinMenu) / sizeof (lsIconBoxWinMenu[0]);

static gchar *
translator (const gchar * path, gpointer data)
{
    return ((gchar*)dgettext (GETTEXT_PACKAGE, path));
}

static void
menu_deactivated (GtkWidget *menu, GtkWidget *plsibw)
{
    if (gdk_window_at_pointer (NULL, NULL) != plsibw->window)
    {
        set_window_opacity (plsibw->window, TRUE);
    }
}

void
ls_iconboxwin_create_menu (LSIconBoxWin * plsibw)
{
    GtkItemFactory *pItemFactory;
    GtkAccelGroup *pAccelGroup;

    pAccelGroup = gtk_accel_group_new ();
    pItemFactory =
	gtk_item_factory_new (GTK_TYPE_MENU, "<main>", pAccelGroup);
    gtk_item_factory_set_translate_func (pItemFactory, translator, NULL,
					 NULL);
    gtk_item_factory_create_items (pItemFactory, nLsIconBoxWinMenu,
				   lsIconBoxWinMenu, (gpointer) plsibw);

    plsibw->priv->pMenu =
	gtk_item_factory_get_widget (pItemFactory, "<main>");

    g_signal_connect (plsibw->priv->pMenu, "deactivate", 
                      G_CALLBACK (menu_deactivated), plsibw);
}

/********************************************************************************/
/* MENU CALLBACKS								*/
/********************************************************************************/
static void
cb_lsibwm_open_iconbox_config (gpointer pUserData,
			       gint nAction, GtkWidget * pWidget)
{
    GError *pgError = NULL;
    gboolean bError;

    bError = g_spawn_command_line_async ("xfce-setting-show xfibx", &pgError);
    if (bError == FALSE)
    {
	DBG ("g_spawn_command_line_async(\"xfce-setting-show xfibx\") "
	     "failed! error: %s", pgError->message);
	xfce_err ("%s", pgError->message);
    }

    return;
}

static void
cb_lsibwm_close_iconboxwin (gpointer pUserData,
			    gint nAction, GtkWidget * pWidget)
{
    g_return_if_fail (LS_IS_ICONBOXWIN (pUserData));
    gtk_widget_destroy (GTK_WIDGET (pUserData));

    return;
}

extern void save_offset (gint nXOff, gint nYOff);

static void
cb_lsibwm_save_pos (gpointer pUserData, gint nAction, GtkWidget * pWidget)
{
    gint nXOff, nYOff;

    g_return_if_fail (LS_IS_ICONBOXWIN (pUserData));

    ls_iconboxwin_get_offset (LS_ICONBOXWIN (pUserData), &nXOff, &nYOff);
    save_offset (nXOff, nYOff);

    return;
}
