// $Id: kpaint.cpp,v 1.118.2.1 2004/02/11 19:29:41 coolo Exp $

#include <qclipboard.h>
#include <qlayout.h>
#include <qsignalmapper.h>
#include <qscrollview.h>

#include <config.h>

#include <kaction.h>
#include <kapplication.h>
#include <kcolorbutton.h>
#include <kcolorcombo.h>
#include <kcolordialog.h>
#include <kconfig.h>
#include <kdebug.h>
#include <kedittoolbar.h>
#include <kfiledialog.h>
#include <kimageio.h>
#include <kinputdialog.h>
#include <kio/netaccess.h>
#include <klocale.h>
#include <kkeydialog.h>
#include <kmessagebox.h>
#include <kmainwindow.h>
#include <kstdaction.h>
#include <ktoolbar.h>
#include <kurl.h>
#include <ktempfile.h>
#include <kstatusbar.h>
#include <kstdguiitem.h>

#include "canvas.h"
#include "version.h"
#include "kpaint.h"
#include "manager.h"
#include "canvassize.h"
#include "palettedialog.h"
#include "infodialog.h"
#include "depthdialog.h"
#include "mainview.h"
#include "navwidget.h"

KPaint::KPaint(const KURL & _url) : KMainWindow(0)
{
    tmpFile = 0;
    zoom= 100;

    m_mainView = new MainView(this);
    m_view = m_mainView->getViewport();
    m_canvas = m_mainView->getCanvas();

    m_view->resize(m_canvas->size());
    setCentralWidget( m_mainView );

    man= new Manager(m_canvas);
    connect(m_canvas, SIGNAL(startDrawing()), man, SLOT(updateProperties()));

    initMenus();
    initSideBar();
    initStatus();
    readOptions();

    enableEditPaste();
    connect( kapp->clipboard(), SIGNAL( dataChanged() ), SLOT( enableEditPaste() ) );
    resize(640,480);
    show();
    m_canvas->clearModified();

    if ( !_url.isEmpty() )
        loadFile(_url);
    else
        fileNew();

    allowEditPalette = m_canvas->pixmap()->depth() == 8;
    updateCommands();
    connect(m_canvas, SIGNAL(modified()), this, SLOT(updateCommands()));
}

KPaint::~KPaint()
{
    delete man;
    delete tmpFile;
}


void KPaint::setPixmap(QPixmap *p)
{
    url = KURL();
    m_canvas->setPixmap(p);
}


bool KPaint::queryClose()
{
  if (!m_canvas->isModified())
    return true;

  QString msg = i18n(""
        "The image has been modified.\n"
        "Would you like to save it?" );

  switch( KMessageBox::warningYesNoCancel( this, msg ) )
  {
     case KMessageBox::Yes: // Save, then exit
          if( url.isEmpty())
          {
             if (!fileSaveAs())
                 return false;
          }
          else
          {
             if (!fileSave())
                 return false;
          }
          return true;

     case KMessageBox::No: // Don't save but exit.
          return true;

     case KMessageBox::Cancel: // Don't save and don't exit.
     default:
	  break; // Don't exit...
  }
  return false;

}

void KPaint::updateCommands()
{
    revertAction->setEnabled(m_canvas->isModified());
    paletteAction->setEnabled(allowEditPalette);
    setCaption(filename, m_canvas->isModified());
}

void KPaint::readOptions()
{
    setAutoSaveSettings("MainWindowSettings");
}

void KPaint::enableEditPaste()
{
    QClipboard *cb = kapp->clipboard();

    bool e;
    if ( cb->pixmap().isNull() )
	e = false;
    else
	e = true;

    kdDebug() << "kpaint: clipboard has image: " << e << endl;
    pasteAction->setEnabled( e );
    pasteImageAction->setEnabled( e );
}


void KPaint::canvasSizeChanged()
{
    QString size(i18n( "%1 x %2" ));
    statusbar->changeItem(size.arg( m_canvas->width() )
        .arg( m_canvas->height() ), ID_FILE_SIZE);
    m_mainView->updateLayout();
}


void KPaint::initStatus()
{
    statusbar= statusBar();

    /* write the image size */
    QString size( "%1 x %2" );
    statusbar->insertItem( size.arg( m_canvas->width() )
        .arg( m_canvas->height() ), ID_FILE_SIZE );

    /* write the color depth */
    size = i18n( " %1 bpp" ).arg(m_canvas->depth());
    statusbar->insertItem(size, ID_COLOR_DEPTH);

    /* write the zoomfactor */
    statusbar->insertItem("100%", ID_ZOOM_FACTOR);

    /* write the toolname */
    /* get the max. font length of the toolnames */
    ToolList tl = man->toolList();
    QFontMetrics fm = statusbar->fontMetrics();
    Tool *maxt = 0;
    uint maxtlen= 0, tmp;

    for ( Tool *t= tl.first(); t; t= tl.next() )
    {
        tmp = fm.width(t->name());
        /* check the tool with the max. name len */
        if ( maxtlen < tmp )
        {
            maxtlen = tmp;
            maxt = t;
        }
    }
    /* write the maxlen toolname */
    if (maxt)
      size = maxt->name();
    else
      size = "XXXX";
    statusbar->insertItem(size, ID_TOOL_NAME);

    /* write the filename */
    statusbar->insertItem(url.path(), ID_FILE_NAME);

    /* update to the current toolname */
    statusbar->changeItem(man->currentTool().name(), ID_TOOL_NAME);
    man->setStatusBar(statusbar);
}

/*
    this sidebar is actually a toolbar with custom widgets inserted
    instead of the usual buttons.  It can be docked to either side,
    top or bottom of window but starts on right side
*/
bool KPaint::initSideBar()
{
    int iw = 66;
    int ih = 66;

    // the toolbar itself
    sideToolbar = toolBar("sideToolBar");
    sideToolbar->setText( i18n( "&Sidebar" ) );

    // the preview/navigation widget with container

    QWidget *navWidget = new QWidget(sideToolbar);

    navWidget->setMinimumWidth(iw);
    navWidget->setMinimumHeight(ih);

    QBoxLayout *ml = new QVBoxLayout(navWidget, 10);

    nav = new NavWidget(navWidget, "navwidget");
    nav->setMinimumSize(120, 80);
    nav->setPixmap(m_canvas->pixmap());

    connect(m_canvas, SIGNAL(pixmapChanged(QPixmap *)),
	    nav, SLOT(setPixmap(QPixmap *)));
    connect(m_canvas, SIGNAL(modified()),
        nav, SLOT(pixmapChanged()));
    ml->addWidget(nav);

    QWidget *mouseWidget = new QWidget(sideToolbar);
    QBoxLayout *vl = new QVBoxLayout(mouseWidget);

    // The colour selector widgets with container widget
    QWidget *colorsWidget = new QWidget(mouseWidget);
    colorsWidget->setMinimumWidth(32);
    colorsWidget->setMinimumHeight(32);

    ml = new QHBoxLayout(colorsWidget, 10);
    lmbColourBtn = new KColorButton(red, colorsWidget, "lmbColour");
    rmbColourBtn = new KColorButton(green, colorsWidget, "rmbColour");

    lmbColourBtn->setFixedSize(30, 30);
    rmbColourBtn->setFixedSize(30, 30);

    man->setLMColorButton(lmbColourBtn);
    man->setRMColorButton(rmbColourBtn);

    connect(lmbColourBtn, SIGNAL(changed(const QColor &)),
	    man, SLOT(setLMBcolour(const QColor &)));
    connect(rmbColourBtn, SIGNAL(changed(const QColor &)),
	    man, SLOT(setRMBcolour(const QColor &)));

    ml->addWidget(lmbColourBtn);
    ml->addWidget(rmbColourBtn);
    vl->addWidget(colorsWidget);

    KPaletteTable *paletteTable = new KPaletteTable( sideToolbar , 120 , 8 );

    connect(paletteTable, SIGNAL(colorSelected(const QColor &, const QString &)),
	  man, SLOT(setLMBcolour(const QColor &)));

    // insert containter widgets so they show up as toolbar items
    sideToolbar->insertWidget(ID_NAV_WIDGET, iw, navWidget);
    sideToolbar->insertWidget(ID_COLORS_WIDGET, iw, mouseWidget);
    sideToolbar->insertWidget(ID_COLOR_BAR, iw, paletteTable);

    return true;
}


void KPaint::initMenus()
{
    // file menu
    (void) KStdAction::openNew(this, SLOT(fileNew()), actionCollection());
    (void) KStdAction::open(this, SLOT(fileOpen()), actionCollection());
    (void) KStdAction::close(this, SLOT(fileClose()), actionCollection());
    revertAction = KStdAction::revert(this, SLOT(fileRevert()), actionCollection());
    saveAction = KStdAction::save(this, SLOT(fileSave()), actionCollection());
    saveAsAction = KStdAction::saveAs(this, SLOT(fileSaveAs()), actionCollection());
    (void) KStdAction::quit(this, SLOT(close()), actionCollection());

    // edit menu
    copyAction = KStdAction::copy(this, SLOT(editCopy()), actionCollection());
    cutAction = KStdAction::cut(this, SLOT(editCut()), actionCollection());
    pasteAction = KStdAction::paste(this, SLOT(editPaste()), actionCollection());
    pasteImageAction = new KAction(i18n("Paste &Image"), KShortcut(), this,
                       SLOT(editPasteImage()), actionCollection(),
                       "edit_paste_image");
    (void) new KAction(KStdGuiItem::clear().text(), KShortcut(), this, SLOT(editClear()), actionCollection(), "edit_image_clear");

    // view menu
    zoomInAction = KStdAction::zoomIn(this, SLOT(editZoomIn()), actionCollection());
    zoomOutAction = KStdAction::zoomOut(this, SLOT(editZoomOut()), actionCollection());
    zoomNormalAction = KStdAction::actualSize(this, SLOT(editZoomNormal()), actionCollection());
    zoomAction = new KSelectAction( i18n( "Zoom" ), KShortcut(), actionCollection(), "view_zoom" );
    connect( zoomAction, SIGNAL( activated( const QString & ) ),
             this, SLOT( viewZoom( const QString & ) ) );
    QStringList lst;
    lst << "10%";
    lst << "20%";
    lst << "30%";
    lst << "40%";
    lst << "50%";
    lst << "60%";
    lst << "70%";
    lst << "80%";
    lst << "90%";
    lst << "100%";
    lst << "200%";
    lst << "300%";
    lst << "400%";
    lst << "500%";
    lst << "600%";
    lst << "700%";
    lst << "800%";
    lst << "900%";
    lst << "1000%";
    zoomAction ->setItems( lst );
    zoomAction ->setCurrentItem(9);
    // image menu
    (void) new KAction(i18n("&Information"), KShortcut(), this,
                       SLOT(imageInfo()), actionCollection(),
                       "image_info");
    (void) new KAction(i18n("&Resize..."), KShortcut(), this,
                       SLOT(imageResize()), actionCollection(),
                       "image_resize");

	(void) new KAction(i18n("&Mirror"), KShortcut(), this,
                       SLOT(imageMirror()), actionCollection(),
                       "image_mirror");

    (void) new KAction(i18n("Rotate &90°"), 0, this,
        SLOT(imageRotate90()), actionCollection(),
        "image_rotate90");
    (void) new KAction(i18n("Rotate &180°"), 0, this,
        SLOT(imageRotate180()), actionCollection(),
        "image_rotate180");
    (void) new KAction(i18n("Rotate &270°"), 0, this,
        SLOT(imageRotate270()), actionCollection(),
        "image_rotate270");
    (void) new KAction(i18n("Free Rotate"), 0, this,
        SLOT(imageRotateFree()), actionCollection(),
        "image_rotatefree");


    paletteAction = new KAction(i18n("&Edit Palette..."), KShortcut(), this,
                       SLOT(imageEditPalette()), actionCollection(),
                       "image_palette");
    (void) new KAction(i18n("&Change Color Depth..."), KShortcut(), this,
                       SLOT(imageChangeDepth()), actionCollection(),
                       "image_color");

    // settings menu
    createStandardStatusBarAction();
    setStandardToolBarMenuEnabled(true);

    KStdAction::keyBindings(this, SLOT(editKeyBindings()), actionCollection());

    // tool menu
    (void) new KAction(i18n("&Tool Properties"), "configure", KShortcut(), this,
                   SLOT(toolProperties()), actionCollection(),
                   "tool_config");

    QSignalMapper *mapper = new QSignalMapper(this);
    connect( mapper, SIGNAL( mapped(int) ),
	     this, SLOT( setTool(int) ) );

    int id = 0;
    ToolList tl = man->toolList();
    for ( Tool *t= tl.first(); t; t= tl.next() )
    {
      if ( !t->isSpecial() ) {
	KToggleAction *action = new KToggleAction(t->name(), t->pixmap(),
      KShortcut(), 0, 0, actionCollection(),
      QCString().sprintf("tool_%d", id));
	action->setExclusiveGroup( "tools" );

	mapper->setMapping( action, id++ );
	connect( action, SIGNAL( activated() ),
		 mapper, SLOT( map() ) );
      }
    }

    KStdAction::configureToolbars(this, SLOT(editToolbars()), actionCollection());
    createGUI();
}

void KPaint::editToolbars()
{
    saveMainWindowSettings(KGlobal::config(), autoSaveGroup());
    KEditToolbar dlg(actionCollection());
    connect(&dlg, SIGNAL(newToolbarConfig()), SLOT(newToolbarConfig()));
    dlg.exec();
}

void KPaint::newToolbarConfig()
{
    // OK/Apply pressed in the toolbar editor
    createGUI();
    // Our sidebar isn't XMLGUI-compatible. Recreate it...
    initSideBar();
    applyMainWindowSettings(KGlobal::config(), autoSaveGroup());
}

void KPaint::editKeyBindings()
{
    KKeyDialog::configure(actionCollection());
    actionCollection()->readShortcutSettings();
}

bool KPaint::fileOpen()
{
    bool result = false;
    if (!url.isEmpty() || m_canvas->isModified())
    {
        KPaint *kp = new KPaint();
        result = kp->fileOpen();
        if (result)
           kp->show();
        else
           delete kp;
        return result;
    }

    kdDebug(4400) << "fileOpen()" << endl;

    KURL url = KFileDialog::getImageOpenURL(QString::null, this);

    if (url.isEmpty())
        return false;

	return loadFile(url);
}


bool KPaint::loadFile(const KURL& _url)
{
    bool result = false;
    KTempFile *newTmpFile = 0;

    if (_url.isLocalFile())
    {
       result = m_canvas->load(_url.path());
       if (result)
          filename = _url.path();
    }
    else
    {
       newTmpFile = new KTempFile();
       newTmpFile->setAutoDelete(true);
       newTmpFile->close();
       QString tmpname = newTmpFile->name();
       if (KIO::NetAccess::download(_url, tmpname, 0))
       {
          result =  m_canvas->load(tmpname);
       }
       if (result)
          filename = tmpname;
       else {
          delete newTmpFile;
          newTmpFile = 0L;
       }
    }

    if (!result)
    {
       QString msg
                = i18n("KPaint could not open %1 \n").arg(_url.prettyURL());
       KMessageBox::sorry(this, msg);
       return false;
    }

    delete tmpFile;
    tmpFile = newTmpFile;
    url = _url;
    format = KImageIO::type(url.fileName());
    statusbar->changeItem(url.prettyURL(), ID_FILE_NAME);
    statusbar->changeItem( i18n( "%1 bpp" ).arg( m_canvas->depth() ),
        ID_COLOR_DEPTH );
    orig_width = m_canvas->width();
    orig_height = m_canvas->height();
    canvasSizeChanged();
    m_canvas->clearModified();
    updateCommands();
    return true;
}


bool KPaint::fileSave()
{
    kdDebug(4400) << "fileSave()\n" << endl;

    if(url.path().isEmpty()) {
        kdDebug(4400) << "Path '" << url.path() << "' is empty\n" << endl;

        if (!fileSaveAs()) {
           kdDebug(4400) << "Passing to save as\n" << endl;
           return false;
	}
    }
    else if (!saveFile(filename, format, url))
      return false;

    return true;
}


bool KPaint::saveFile(QString filename_, QString format_, KURL &url_)
{
    kdDebug(4400) << "saveFile reached, calling canvas save...\n" << endl;

    if (!m_canvas->save(filename_, format_.ascii()))
        return false;

    filename = filename_;
    format = format_;

    if (!url_.isLocalFile())
    {
        if (!KIO::NetAccess::upload(filename_, url_, 0))
	    return false;
    }
    filename = filename_;
    format = format_;
    url = url_;

    statusbar->changeItem(filename, ID_FILE_NAME);
    m_canvas->clearModified();
    updateCommands();
    return true;
}


bool KPaint::fileSaveAs()
{
    kdDebug(4400) << "fileSaveAsCommand" << endl;

    KURL file = KFileDialog::getSaveURL(QString::null, KImageIO::pattern(KImageIO::Writing), this);

    if(file.isEmpty())
        return false;

    QString tmpname;

    KTempFile *newTmpFile = 0;
    if (file.isLocalFile())
    {
        tmpname = file.path();
    }
    else
    {
        newTmpFile = new KTempFile();
        newTmpFile->setAutoDelete(true);
        newTmpFile->close();
        tmpname = newTmpFile->name();
    }

    if (!saveFile(tmpname, KImageIO::type(file.path()).ascii(), file))
    {
        delete newTmpFile;
        return false;
    }
    delete tmpFile;
    tmpFile = newTmpFile;

    m_canvas->clearModified();
    return true;
}

void KPaint::fileClose()
{
    delete tmpFile;
    tmpFile = 0;
    url = KURL();
    m_canvas->clearModified();
    fileNew();
}

void KPaint::fileRevert()
{
    if (!url.isEmpty())
    {
       m_canvas->load(filename);
    }
    else
    {
       QPixmap p;
       p.resize(orig_width, orig_height);
       p.fill(QColor("white"));
       m_canvas->setPixmap(&p);
    }
    canvasSizeChanged();
    m_canvas->clearModified();
    updateCommands();
}

void KPaint::fileNew()
{
    kdDebug(4400) << "File New" << endl;

    if (!url.isEmpty() || m_canvas->isModified())
    {
        KPaint *kp = new KPaint();
        kp->show();
        kp->fileNew();
        return;
    }

    m_canvas->clearModified();
    QPixmap p;
    p.resize(320, 200);
    p.fill(QColor("white"));
    m_canvas->setPixmap(&p);
    man->setCurrentTool(0);
    format = "PNG";
    filename = i18n("untitled") + ".png";
    url=KURL();
    delete tmpFile;
    tmpFile = 0;

    statusbar->changeItem(filename, ID_FILE_NAME);
    orig_width = m_canvas->width();
    orig_height = m_canvas->height();
    canvasSizeChanged();
    repaint(0);

//      canvasSizeDialog sz(this, "canvassizedlg",m_canvas->width(), m_canvas->height());
//      if ( sz.exec() ) {
//          p.resize( sz.getWidth(), sz.getHeight() );
//          p.fill( QColor("white") );
//          m_canvas->setPixmap(&p);
//          canvasSizeChanged();
//          repaint(0);
//      }
}

// Edit
void KPaint::editCopy()
{
    kdDebug(4400) << "editCopy()\n" << endl;
    m_canvas->copy();
}

void KPaint::editCut()
{
    kdDebug(4400) << "editCut()\n" << endl;
    m_canvas->cut();
}

/*
    paste is a tool now. It's in brush.cpp and brush.h,
    and can later be extended for painting with clip
    pixmap as a brush as well as pasting down
*/
void KPaint::editPaste()
{
    kdDebug(4400) << "editPaste()\n" << endl;

    QClipboard *cb = kapp->clipboard();
    QPixmap clipPix = cb->pixmap();
    if ( !clipPix.isNull() )
      man->setCurrentTool(9);

    m_canvas->markModified();
}

/* This is intended to copy clipboard to a
   file and open it in a new window - */

void KPaint::editPasteImage()
{
    kdDebug(4400) << "editPasteImage()" << endl;

    QClipboard *cb = kapp->clipboard();
    QPixmap clipPix = cb->pixmap();
    if ( !clipPix.isNull() ) {
        QPixmap *p = new QPixmap( clipPix );
        KPaint *kp = new KPaint();
        kp->setPixmap(p);
	kp->m_canvas->markModified();
        kp->show();
    }
}

void KPaint::editClear()
{
    kdDebug(4400) << "editClear()" << endl;
    m_canvas->clear();
}

void KPaint::editZoomNormal()
{
    kdDebug(4400) << "editZoomIn()" << endl;

    int zoom = 100;
    m_canvas->setZoom( zoom );
    QString zoomstr;
    zoomstr.setNum(zoom);
    zoomstr += '%';

    statusbar->changeItem( zoomstr, ID_ZOOM_FACTOR );
    QStringList list=zoomAction ->items();
    zoomAction->setCurrentItem(list.findIndex(zoomstr));
    zoomInAction->setEnabled( true );
    zoomOutAction->setEnabled( true );
}


void KPaint::editZoomIn()
{
    kdDebug(4400) << "editZoomIn()" << endl;

    if (zoom >= 100)
    {
        zoom += 100;
        if (zoom == 1000)
        {
            zoomInAction->setEnabled(false);
            zoomOutAction->setEnabled(true);
        }
    }
    else
    {
        zoom += 10;
        zoomInAction->setEnabled(true);
        zoomOutAction->setEnabled(true);
    }

    m_canvas->setZoom(zoom);

    QString zoomstr;
    zoomstr.setNum(zoom);
    zoomstr += '%';

    statusbar->changeItem(zoomstr, ID_ZOOM_FACTOR);
    QStringList list=zoomAction ->items();
    zoomAction->setCurrentItem(list.findIndex(zoomstr));
}

void KPaint::viewZoom( const QString &_strZoom )
{
    QString z( _strZoom );
    z = z.replace( "%", "" );
    z = z.simplifyWhiteSpace();
    int new_zoom = z.toInt();

    if(new_zoom==10)
    {
        zoomOutAction->setEnabled(false);
        zoomInAction->setEnabled(true);
    }
    else if(new_zoom>=1000)
    {
        zoomInAction->setEnabled(false);
        zoomOutAction->setEnabled(true);
    }
    else
    {
        zoomInAction->setEnabled(true);
        zoomOutAction->setEnabled(true);
    }

    if(zoom!=new_zoom)
    {
        m_canvas->setZoom(new_zoom);
        zoom=new_zoom;
        statusbar->changeItem(_strZoom, ID_ZOOM_FACTOR);
    }
}

void KPaint::editZoomOut()
{
    kdDebug(4400) << "editZoomOut()" << endl;

    if (zoom > 100)
    {
        zoom -= 100;
        zoomOutAction->setEnabled(true);
        zoomInAction->setEnabled(true);
    }
    else
    {
        zoom -= 10;
        if (zoom <= 10)
        {
            zoomOutAction->setEnabled(false);
            zoomInAction->setEnabled(true);
        }
    }

    m_canvas->setZoom(zoom);

    QString zoomstr;
    zoomstr.setNum(zoom);
    zoomstr += '%';
    QStringList list=zoomAction ->items();
    zoomAction->setCurrentItem(list.findIndex(zoomstr));

    statusbar->changeItem(zoomstr, ID_ZOOM_FACTOR);
}


void KPaint::editOptions()
{
    kdDebug(4400) << "editOptions()" << endl;
    /* obsolet (jha)
    KKeyDialog::configureKeys(keys); */
}


void KPaint::imageInfo()
{
  imageInfoDialog info(m_canvas, 0, "Image Information");
  kdDebug(4400) << "imageInfo()" << endl;
  info.exec();
}


void KPaint::imageResize()
{
    kdDebug(4400) << "imageResize()" << endl;

    canvasSizeDialog sz(this,"imageResize",m_canvas->imageWidth(), m_canvas->imageHeight());
    if (sz.exec())
    {
        int w= sz.getWidth();
        int h= sz.getHeight();
        if( w!=0 && h!=0)
        {
            m_canvas->resizeImage(w, h);
            canvasSizeChanged();
        }
    }
}

void KPaint::imageEditPalette()
{
    kdDebug(4400) << "imageEditPalette()" << endl;
    paletteDialog pal(m_canvas->pixmap());

    if (pal.exec())
    {
        m_canvas->setPixmap(pal.pixmap());
        m_canvas->repaint(0);
    }
}

void KPaint::imageChangeDepth()
{
    depthDialog d(m_canvas);

    kdDebug(4400) << "imageChangeDepth()" << endl;

    if (d.exec())
    {
        switch (d.depthBox->currentItem())
        {

    case ID_COLOR_1:
      kdDebug(4400) << "setDepth to 1" << endl;
      allowEditPalette= false;
      m_canvas->setDepth(1);
      break;

    case ID_COLOR_8:
      kdDebug(4400) << "setDepth to 8" << endl;
      allowEditPalette= true;
      m_canvas->setDepth(8);
      break;

    case ID_COLOR_32:
      kdDebug(4400) << "setDepth to 32" << endl;
      allowEditPalette= false;
      m_canvas->setDepth(32);
      break;

    default:
      break;

        }
    }
}

// Tool
void KPaint::setTool(int t)
{
    man->setCurrentTool(t);
}

void KPaint::toolProperties()
{
    kdDebug(4400) << "toolProperties()" << endl;
    man->showPropertiesDialog();
}


void KPaint::imageRotate90()
{
  kdDebug(4400) << "imageRotate90()" << endl;

  m_canvas->rotateImage(90);
  // width will become height,
  // height will become width,
  // so size is changed
  canvasSizeChanged();
}

void KPaint::imageRotate180()
{
  kdDebug(4400) << "imageRotate180()" << endl;

  m_canvas->rotateImage(180);
}

void KPaint::imageRotate270()
{
  kdDebug(4400) << "imageRotate270()" << endl;

  m_canvas->rotateImage(270);
  // width will become height,
  // height will become width,
  // so size is changed
  canvasSizeChanged();
}

void KPaint::imageRotateFree()
{
  kdDebug(4400) << "imageRotateFree()" << endl;

  bool ok;
  int degree = KInputDialog::getInteger(
	  i18n("Free Rotation"), i18n("Specify rotation in degrees:"),
	   180, 0, 360, 1, &ok, this);
  if (ok)
    m_canvas->rotateImage(degree);

  // size is probably changed
  canvasSizeChanged();
}

void KPaint::imageMirror()
{
  kdDebug(4400) << "imageMirror()" << endl;

  m_canvas->mirrorImage();
}

#include "kpaint.moc"


