/*
 * xlog - GTK+ logging program for amateur radio operators
 * Copyright (C) 2001-2003 Joop Stakenborg <pa4tu@amsat.org>
 *
 * This program is free oftware; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU Library General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

/* callbacks.c
 *
 * Callbacks for the preferences dialog and the dialogs called from here. 
 * For callbacks related to the widgets in the main window, see the files
 * called callbacks_<widgetname>.[c|h].
 */


#include <stdlib.h>
#include <libgen.h>
#include <gtk/gtk.h>
#include <string.h>

#include "callbacks.h"
#include "interface.h"
#include "support.h"
#include "types.h"
#include "log.h"
#include "utils.h"

#if WANT_HAMLIB
#	include <string.h>
#	include <hamlib-utils.h>
#	include <hamlib/rig.h>
#endif

extern GtkWidget *mainwindow;
extern preferencestype preferences;
extern statetype state;
extern gchar **bandsplit;
extern gchar **modesplit;
extern GtkWidget *preferencesdialog;
extern gint clocktimer, savetimer;
extern GList *logwindowlist;

#if WANT_HAMLIB
extern GList *riglist;
extern gint hamlibtimer;
extern RIG *myrig;
#endif


/* PREFERENCES DIALOG */

#if WANT_HAMLIB
/* return value for setting widgets */
static gint
whichhamlibwidgets (gboolean frequency, gboolean smeter)
{
  if (frequency && smeter)
    return (4);
  else if (frequency && (!smeter))
    return (2);
  else if ((!frequency) && smeter)
    return (3);
  else
    return (1);
}
#endif

/* check save value, if autosave value larger than 0 start timer */
static void
set_autosave (gint value, gint saving)
{
  if (saving != 1 && preferences.saving == 1)
  {
	  gdk_input_remove (savetimer);
	  savetimer = -1;
  }
  else if (saving == 1 && preferences.saving != 1)
  {
    savetimer =
      gtk_timeout_add (value * 60 * 1000, (GtkFunction) autosave, NULL);
  }
  else if (value != preferences.autosave)
  {     /* new value, stop and start timer */
    gdk_input_remove (savetimer);
    savetimer =
      gtk_timeout_add (value * 60 * 1000, (GtkFunction) autosave, NULL);
  }
  preferences.autosave = value;
  preferences.saving = saving;
}

/* set the theme if changed */
static void
set_theme (gchar * color, gchar * font, gboolean alsoforlog)
{
  GtkWidget *clocklabel, *qsoframe, *dxccframe;
  gint i, row;
  logtype *logwindow;
  gint logcolor;
  gboolean logcolordone = FALSE;
  GtkStyle *cellstyle;
#if WANT_HAMLIB
  GtkWidget *frequencylabel;
#endif

  logcolor = alsoforlog ? 1 : 0;

  if (g_strcasecmp (color, preferences.themecolor) ||
      g_strcasecmp (font, preferences.themefont))
    {
      clocklabel = lookup_widget (mainwindow, "clocklabel");
      setlabelstyle (clocklabel, color, font);
#if WANT_HAMLIB
      frequencylabel = lookup_widget (mainwindow, "frequencylabel");
      setlabelstyle (frequencylabel, color, font);
#endif
      qsoframe = lookup_widget (mainwindow, "qsoframe");
      dxccframe = lookup_widget (mainwindow, "dxccframe");
      setframestyle (qsoframe, color, font);
      setframestyle (dxccframe, color, font);
      for (i = 0; i < g_list_length (logwindowlist); i++)
	{
	  logwindow = g_list_nth_data (logwindowlist, i);
	  setlabelstyle (logwindow->label, color, font);
	}
      if (alsoforlog)		/* apply to the log if needed */
	{
	  for (i = 0; i < g_list_length (logwindowlist); i++)
	    {
	      logwindow = g_list_nth_data (logwindowlist, i);
	      gtk_clist_freeze (GTK_CLIST (logwindow->clist));
	      for (row = 0; row < logwindow->qsos; row++)
		setcallstyle (logwindow->clist, row, color, font);
	      gtk_clist_thaw (GTK_CLIST (logwindow->clist));
	    }
	  logcolordone = TRUE;
	}
      preferences.themecolor = color;
      preferences.themefont = g_strdup (font);
    }

  /* does the log need color? */
  if (logcolor != preferences.logcolor)
    {
      if ((logcolor == 1) && (!logcolordone))
	{
	  for (i = 0; i < g_list_length (logwindowlist); i++)
	    {
	      logwindow = g_list_nth_data (logwindowlist, i);
	      gtk_clist_freeze (GTK_CLIST (logwindow->clist));
	      for (row = 0; row < logwindow->qsos; row++)
		setcallstyle (logwindow->clist, row,
			      preferences.themecolor, preferences.themefont);
	      gtk_clist_thaw (GTK_CLIST (logwindow->clist));
	    }
	}
      if (logcolor == 0)
	{
	  for (i = 0; i < g_list_length (logwindowlist); i++)
	    {
	      logwindow = g_list_nth_data (logwindowlist, i);
	      gtk_clist_freeze (GTK_CLIST (logwindow->clist));
	      cellstyle =
		gtk_clist_get_cell_style (GTK_CLIST (logwindow->clist), 0, 0);
	      for (row = 0; row < logwindow->qsos; row++)
		gtk_clist_set_cell_style (GTK_CLIST (logwindow->clist), row,
					  CALL, cellstyle);
	      gtk_clist_thaw (GTK_CLIST (logwindow->clist));
	    }
	}
      preferences.logcolor = logcolor;
    }
}

/* set savepath if changed */
static void
set_path (gchar * pathstr)
{
  gchar *str, *fpathstr;
  GString *path;

  if (strlen (pathstr) > 0)
    {
      path = g_string_new (pathstr);
      str = g_strdup (path->str);
      /* remove trailing slash */
      g_strreverse (str);
      if (!g_strncasecmp (str, "/", 1))
	{
	  path = g_string_erase (path, (path->len) - 1, 1);
	}
      else			/* in case a file is selected */
	{
	  fpathstr = g_strdup (path->str);
	  path = g_string_new (dirname (fpathstr));
	  g_free (fpathstr);
	}
      g_free (str);
      if (g_strcasecmp (preferences.savedir, path->str))
	preferences.savedir = g_strdup (path->str);
      g_string_free (path, TRUE);
    }
}

/* set logs to load if changed */
static void
set_logstoload (gchar * logs)
{
  gchar **logsplit;

  if (g_strcasecmp (logs, preferences.logstoload) != 0)
    {
      if (strlen (logs) == 0)
	logs = g_strdup ("*");
      logsplit = g_strsplit (logs, ",", -1);	/* delete spaces */
      deletespaces (logsplit);
      preferences.logstoload = g_strjoinv (",", logsplit);
      g_strfreev (logsplit);
    }
}

/* set QTH locator */
static void
set_qthlocator (gchar * locator)
{
  if (g_strcasecmp (locator, preferences.locator) != 0)
    {
      if (strlen (locator) == 0)
	locator = g_strdup ("AA00AA");
      preferences.locator = g_strdup (locator);
    }
}

/* set callsign */
static void
set_callsign (gchar * callsign)
{
  if (g_strcasecmp (callsign, preferences.callsign) != 0)
    {
      if (strlen (callsign) == 0)
	callsign = g_strdup ("N0CALL");
      preferences.callsign = g_strdup (callsign);
    }
}

/* enable/disable clock */
static void
set_clock (gboolean on)
{
  GtkWidget *clocklabel, *clockpixmap, *clockhbox, *clockhandlebox;

  clocklabel = lookup_widget (mainwindow, "clocklabel");
  clockpixmap = lookup_widget (mainwindow, "clockpixmap");
  clockhbox = lookup_widget (mainwindow, "clockhbox");
  clockhandlebox = lookup_widget (mainwindow, "clockhandlebox");
  if (on)
    {				/* yes, we want clock */
      if (preferences.clock == 0)
	clocktimer = gtk_timeout_add (1000, (GtkFunction) updateclock, NULL);
      gtk_widget_show (clocklabel);
      gtk_widget_show (clockpixmap);
      gtk_widget_show (clockhbox);
      gtk_widget_show (clockhandlebox);
      preferences.clock = 1;
    }
  else
    {				/* no clock */
      if (preferences.clock > 0)
	gdk_input_remove (clocktimer);
      clocktimer = -1;
      gtk_widget_hide (clocklabel);
      gtk_widget_hide (clockpixmap);
      gtk_widget_hide (clockhbox);
      gtk_widget_hide (clockhandlebox);
      preferences.clock = 0;
    }
}

/* ok button clicked */
void
on_preferencesokbutton_clicked (GtkButton * button, gpointer user_data)
{
  GtkWidget *latentry, *longentry, *NScomboentry, *EWcomboentry,
    *unitscomboentry, *bandsentry, *bandoptionmenu, *modesentry,
    *modeoptionmenu, *bandsradiobutton1, *modesradiobutton1,
    *clockcheckbutton, *examplelabel, *autosavecomboentry,
    *themecheckbutton, *pathentry, *logsentry, *qthlocatorentry,
    *callsignentry, *fontentry, *bandentry, *modeentry, *savecheckbutton,
    *autosavecheckbutton, *typeaheadfindcheckbutton, *remoteradiobutton1;
  gchar *temp, *bandcopy, *font;
  gboolean toggletrue, logcoloryesno;
  gint bandindex = 0, modeindex = 0, autosavevalue, saving;
#if WANT_HAMLIB
  GtkWidget *hamlibcheckbutton, *radiocomboentry, *devicecomboentry,
    *frequencycheckbutton, *digitscomboentry, *smetercheckbutton,
    *drawingarea;
  gchar *radio = NULL, *device = NULL, *digits = NULL;
  gboolean hamlibyes, frequencyyesno, smeteryesno;
  gint rigid, retcode;
  gint hamlibwidgets = 0;
#endif

  /* check autosave value */
  autosavecheckbutton = 
    lookup_widget (preferencesdialog, "autosavecheckbutton");
  savecheckbutton =
    lookup_widget (preferencesdialog, "savecheckbutton");
  if (gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(autosavecheckbutton)))
    saving = 1;
  else if (gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(savecheckbutton)))
    saving = 2;
  else saving = 0;
  autosavecomboentry =
    lookup_widget (preferencesdialog, "autosavecomboentry");
  temp = gtk_editable_get_chars (GTK_EDITABLE (autosavecomboentry), 0, -1);
  autosavevalue = atoi (temp);
  set_autosave (autosavevalue, saving);
  
  /* check type ahead find */
  typeaheadfindcheckbutton = lookup_widget 
    (preferencesdialog, "typeaheadfindcheckbutton");
  if (gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON 
      (typeaheadfindcheckbutton)))
    preferences.typeaheadfind = 1;
  else
    preferences.typeaheadfind = 0;

  /* remote logging */
  remoteradiobutton1 = lookup_widget (preferencesdialog, "remoteradiobutton1");
  if (gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (remoteradiobutton1)))
    preferences.remoteadding = 1;
  else
    preferences.remoteadding = 2;
    
  /* apply theme color */
  fontentry = lookup_widget (preferencesdialog, "fontentry");
  examplelabel = lookup_widget (preferencesdialog, "examplelabel");
  themecheckbutton = lookup_widget (preferencesdialog, "themecheckbutton");
  logcoloryesno = gtk_toggle_button_get_active
    (GTK_TOGGLE_BUTTON (themecheckbutton));
  temp = getlabelcolor (examplelabel);
  font = gtk_editable_get_chars (GTK_EDITABLE (fontentry), 0, -1);
  set_theme (temp, font, logcoloryesno);
  g_free (font);

  /* get path for the logs */
  pathentry = lookup_widget (preferencesdialog, "pathentry");
  temp = gtk_editable_get_chars (GTK_EDITABLE (pathentry), 0, -1);
  set_path (temp);

  /* get logs to load */
  logsentry = lookup_widget (preferencesdialog, "logsentry");
  temp = gtk_editable_get_chars (GTK_EDITABLE (logsentry), 0, -1);
  set_logstoload (temp);

  /* locator */
  qthlocatorentry = lookup_widget (preferencesdialog, "qthlocatorentry");
  temp = gtk_editable_get_chars (GTK_EDITABLE (qthlocatorentry), 0, -1);
  set_qthlocator (temp);

  /* callsign */
  callsignentry = lookup_widget (preferencesdialog, "callsignentry");
  temp = gtk_editable_get_chars (GTK_EDITABLE (callsignentry), 0, -1);
  set_callsign (temp);

  /* set latitude, longitude and units */
  latentry = lookup_widget (preferencesdialog, "latentry");
  longentry = lookup_widget (preferencesdialog, "longentry");
  NScomboentry = lookup_widget (preferencesdialog, "NScomboentry");
  EWcomboentry = lookup_widget (preferencesdialog, "EWcomboentry");
  unitscomboentry = lookup_widget (preferencesdialog, "unitscomboentry");
  temp = gtk_editable_get_chars (GTK_EDITABLE (latentry), 0, -1);
  g_strstrip (temp);
  if (strlen (temp) == 0)
    preferences.latitude = 0.0;
  else
    preferences.latitude = atof (temp);
  temp = gtk_editable_get_chars (GTK_EDITABLE (longentry), 0, -1);
  g_strstrip (temp);
  if (strlen (temp) == 0)
    preferences.longitude = 0.0;
  else
    preferences.longitude = atof (temp);
  temp = gtk_editable_get_chars (GTK_EDITABLE (NScomboentry), 0, 1);
  if (g_strncasecmp (temp, _("N"), 1) == 0)
    preferences.NS = 1;
  else
    preferences.NS = 2;
  temp = gtk_editable_get_chars (GTK_EDITABLE (EWcomboentry), 0, 1);
  if (g_strncasecmp (temp, _("E"), 1) == 0)
    preferences.EW = 1;
  else
    preferences.EW = 2;
  temp = gtk_editable_get_chars (GTK_EDITABLE (unitscomboentry), 0, -1);
  if (g_strncasecmp (temp, "K", 1) == 0)
    preferences.units = 1;
  else
    preferences.units = 0;
  state.mylocation = setlocation (preferences.latitude, preferences.NS,
				  preferences.longitude, preferences.EW);
  /* get the comma seperated list of modes and assign to the optionmenu */
  modesentry = lookup_widget (preferencesdialog, "modesentry");
  modeoptionmenu = lookup_widget (mainwindow, "modeoptionmenu");
  temp = gtk_editable_get_chars (GTK_EDITABLE (modesentry), 0, -1);
  modesplit = g_strsplit (temp, ",", 0);
  deletespaces (modesplit);
  updateoptionmenu (modeoptionmenu, modesplit);

  /* set the optionmenu for modes */
  modeentry = lookup_widget (mainwindow, "modeentry");
  temp = gtk_editable_get_chars (GTK_EDITABLE (modeentry), 0, -1);
  if (strlen (temp) > 0)
    {
      for (;;)
	{
	  if (g_strcasecmp (temp, modesplit[modeindex]) == 0)
	    {
	      gtk_option_menu_set_history (GTK_OPTION_MENU (modeoptionmenu),
					   modeindex);
	      break;
	    }
	  modeindex++;
	}
    }

  /* get the comma seperated list of bands and assign to the optionmenu */
  bandsentry = lookup_widget (preferencesdialog, "bandsentry");
  bandoptionmenu = lookup_widget (mainwindow, "bandoptionmenu");
  temp = gtk_editable_get_chars (GTK_EDITABLE (bandsentry), 0, -1);
  bandsplit = g_strsplit (temp, ",", 0);
  deletespaces (bandsplit);
  updateoptionmenu (bandoptionmenu, bandsplit);

  /* set the optionmenu for bands in the main window */
  bandentry = lookup_widget (mainwindow, "bandentry");
  temp = gtk_editable_get_chars (GTK_EDITABLE (bandentry), 0, -1);
  if (strlen (temp) > 0)
    {
      if (strlen (temp) <= 2)
	bandcopy = g_strdup (temp);
      else
	bandcopy = finddot (temp);
      if (atoi (bandcopy) < 29)
	{			/* only support HF bands for now */
	  for (;;)
	    {
	      if (g_strcasecmp (bandcopy, bandsplit[bandindex]) == 0)
		{
		  gtk_option_menu_set_history (GTK_OPTION_MENU
					       (bandoptionmenu), bandindex);
		  break;
		}
	      bandindex++;
	    }
	}
      g_free (bandcopy);
    }

  preferences.modes = g_strjoinv (",", modesplit);
  preferences.bands = g_strjoinv (",", bandsplit);

  /* set visibility of widgets in QSO frame depending on state of toggle buttons */
  bandsradiobutton1 = lookup_widget (preferencesdialog, "bandsradiobutton1");
  toggletrue =
    gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (bandsradiobutton1));
  if (toggletrue)
    {
      preferences.bandseditbox = 1;
      gtk_widget_hide (bandoptionmenu);
      gtk_widget_show (bandentry);
    }
  else
    {
      preferences.bandseditbox = 0;
      gtk_widget_hide (bandentry);
      gtk_widget_show (bandoptionmenu);
    }
  modesradiobutton1 = lookup_widget (preferencesdialog, "modesradiobutton1");
  toggletrue =
    gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (modesradiobutton1));
  if (toggletrue)
    {
      preferences.modeseditbox = 1;
      gtk_widget_hide (modeoptionmenu);
      gtk_widget_show (modeentry);
    }
  else
    {
      preferences.modeseditbox = 0;
      gtk_widget_hide (modeentry);
      gtk_widget_show (modeoptionmenu);
    }

#if WANT_HAMLIB
  hamlibcheckbutton = lookup_widget (preferencesdialog, "hamlibcheckbutton");
  frequencycheckbutton =
    lookup_widget (preferencesdialog, "frequencycheckbutton");
  smetercheckbutton = lookup_widget (preferencesdialog, "smetercheckbutton");
  radiocomboentry = lookup_widget (preferencesdialog, "radiocomboentry");
  devicecomboentry = lookup_widget (preferencesdialog, "devicecomboentry");
  digitscomboentry = lookup_widget (preferencesdialog, "digitscomboentry");
  hamlibyes = gtk_toggle_button_get_active
    (GTK_TOGGLE_BUTTON (hamlibcheckbutton));
  frequencyyesno = gtk_toggle_button_get_active
    (GTK_TOGGLE_BUTTON (frequencycheckbutton));
  smeteryesno = gtk_toggle_button_get_active
    (GTK_TOGGLE_BUTTON (smetercheckbutton));
  radio = gtk_editable_get_chars (GTK_EDITABLE (radiocomboentry), 0, -1);
  device = gtk_editable_get_chars (GTK_EDITABLE (devicecomboentry), 0, -1);
  digits = gtk_editable_get_chars (GTK_EDITABLE (digitscomboentry), 0, -1);

  if ((preferences.hamlib == 0) && hamlibyes)
    {
      /* first use of hamlib, open radio */
      rigid = get_rigid (radio);
      if (rigid != -1)
	{
	  myrig = rig_init (rigid);
	  strncpy (myrig->state.rigport.pathname, device, FILPATHLEN);
	  retcode = rig_open (myrig);
	  if (retcode != RIG_OK)
	    {
	      temp =
		g_strdup_printf (_
				 ("An error occured while opening port %s: %s"),
				 myrig->state.rigport.pathname,
				 rigerror (retcode));
	      update_statusbar (temp);
	    }
	  else
	    {
	      hamlibtimer =
		gtk_timeout_add (200, (GtkFunction) get_riginfo, NULL);
	    }
	  preferences.radio = g_strdup (radio);
	  preferences.device = g_strdup (device);
	  preferences.round = atoi (digits);
	  hamlibwidgets = whichhamlibwidgets (frequencyyesno, smeteryesno);
	}
      else
	{
	  update_statusbar (_("Hamlib error: get_rigid failed"));
	}
    }
  else if ((preferences.hamlib > 0) && hamlibyes)
    {
      /* see if we have selected another radio or port */
      if ((g_strcasecmp (radio, preferences.radio) != 0)
	  || (g_strcasecmp (device, preferences.device) != 0))
	{
	  gdk_input_remove (hamlibtimer);
	  rig_close ((RIG *) myrig);
	  rig_cleanup ((RIG *) myrig);
	  rigid = get_rigid (radio);
	  myrig = rig_init (rigid);
	  strncpy (myrig->state.rigport.pathname, device, FILPATHLEN);
	  retcode = rig_open (myrig);
	  if (retcode != RIG_OK)
	    {
	      temp =
		g_strdup_printf ("An error occured while opening port %s: %s",
				 myrig->state.rigport.pathname,
				 rigerror (retcode));
	      update_statusbar (temp);
	    }
	  else
	    {
	      hamlibtimer =
		gtk_timeout_add (200, (GtkFunction) get_riginfo, NULL);
	    }
	}
      preferences.radio = g_strdup (radio);
      preferences.device = g_strdup (device);
      preferences.round = atoi (digits);
      hamlibwidgets = whichhamlibwidgets (frequencyyesno, smeteryesno);
    }
  else if ((preferences.hamlib > 0) && !hamlibyes)
    {
      /* hamlib disabled, close rig */
      hamlibwidgets = 0;
      gdk_input_remove (hamlibtimer);
      hamlibtimer = -1;
      rig_close ((RIG *) myrig);
      rig_cleanup ((RIG *) myrig);
      preferences.radio = g_strdup ("Dummy");
      preferences.device = g_strdup ("/dev/ttyS1");
      preferences.round = 0;
    }
  else if ((preferences.hamlib = 0) && !hamlibyes)
    {
      preferences.radio = g_strdup ("Dummy");
      preferences.device = g_strdup ("/dev/ttyS1");
      preferences.round = 0;
    }

  /* set widgets if they are changed */
  if (hamlibwidgets != preferences.hamlib)
    {
      sethamlibwidgets (hamlibwidgets);
      /* pixmaps need to be loaded if s-meter is enabled and was disabled before */
      if (((hamlibwidgets == 3) || (hamlibwidgets == 4))
	  && ((preferences.hamlib == 0) || (preferences.hamlib == 1)
	      || (preferences.hamlib == 2)))
	{
	  drawingarea = lookup_widget (mainwindow, "smeterdrawingarea");
	  loadsmeter (drawingarea);
	}
      preferences.hamlib = hamlibwidgets;
    }
  g_free (radio);
  g_free (device);
  g_free (digits);
#endif

  /* clock */
  clockcheckbutton = lookup_widget (preferencesdialog, "clockcheckbutton");
  toggletrue =
    gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (clockcheckbutton));
  set_clock (toggletrue);

  g_free (temp);
  gtk_widget_set_sensitive (mainwindow, 1);
  gtk_widget_destroy (preferencesdialog);
}

/* toggle hamlib widgets in the preferences dialog */
void
on_hamlibcheckbutton_toggled (GtkToggleButton * togglebutton,
			      gpointer user_data)
{
  GtkWidget *hamlibframe;
  gboolean toggletrue;

  hamlibframe = lookup_widget (preferencesdialog, "hamlibframe");
  toggletrue = gtk_toggle_button_get_active (togglebutton);
  if (toggletrue)
    gtk_widget_set_sensitive (hamlibframe, 1);
  else
    gtk_widget_set_sensitive (hamlibframe, 0);
}

/* cancel the preferences dialog */
void
on_preferencescancelbutton_clicked (GtkButton * button, gpointer user_data)
{
  gtk_widget_set_sensitive (mainwindow, 1);
  gtk_widget_destroy (gtk_widget_get_toplevel (GTK_WIDGET (button)));
}

/* the preferences dialog is deleted */
gboolean
on_preferencesdialog_delete_event (GtkWidget * widget, GdkEvent * event,
				   gpointer user_data)
{
  gtk_widget_set_sensitive (mainwindow, 1);
  return FALSE;
}

/* toggle sensistivity of band widgets in preferences dialog */
void
on_bandsradiobutton_toggled (GtkToggleButton * togglebutton,
			     gpointer user_data)
{
  GtkWidget *bandslabel, *bandsentry;

  bandslabel = lookup_widget (preferencesdialog, "bandslabel");
  bandsentry = lookup_widget (preferencesdialog, "bandsentry");
  switch (atoi (user_data))
    {
    case 1:
      gtk_widget_set_sensitive (bandslabel, 0);
      gtk_widget_set_sensitive (bandsentry, 0);
      break;
    case 2:
      gtk_widget_set_sensitive (bandslabel, 1);
      gtk_widget_set_sensitive (bandsentry, 1);
      break;
    }
}

/* toggle sensistivity of modes widgets in preferences dialog */
void
on_modesradiobutton_toggled (GtkToggleButton * togglebutton,
			     gpointer user_data)
{
  GtkWidget *modeslabel, *modesentry;

  modeslabel = lookup_widget (preferencesdialog, "modeslabel");
  modesentry = lookup_widget (preferencesdialog, "modesentry");
  switch (atoi (user_data))
    {
    case 1:
      gtk_widget_set_sensitive (modeslabel, 0);
      gtk_widget_set_sensitive (modesentry, 0);
      break;
    case 2:
      gtk_widget_set_sensitive (modeslabel, 1);
      gtk_widget_set_sensitive (modesentry, 1);
      break;
    }

}

/* show color selection dialog and apply the theme color */
void
on_themebutton_clicked (GtkButton * button, gpointer user_data)
{
  GtkWidget *colorselectiondialog;
  gdouble rgb[3];
  gchar *themecolor;
  GdkColor color;

  colorselectiondialog = create_colorselectiondialog ();
  gtk_widget_destroy (GTK_COLOR_SELECTION_DIALOG (colorselectiondialog)->
		      help_button);

  themecolor = color_parse (preferences.themecolor);
  gdk_color_parse (themecolor, &color);
  g_free (themecolor);
  rgb[0] = color.red / 65535.0;
  rgb[1] = color.green / 65535.0;
  rgb[2] = color.blue / 65535.0;
  gtk_color_selection_set_color (GTK_COLOR_SELECTION
				 (GTK_COLOR_SELECTION_DIALOG
				  (colorselectiondialog)->colorsel), rgb);

  gtk_widget_set_sensitive (preferencesdialog, 0);
  gtk_widget_show (colorselectiondialog);
}

/* change theme font */
void
on_fontbutton_clicked (GtkButton * button, gpointer user_data)
{
  GtkWidget *fontselectiondialog;

  fontselectiondialog = create_fontselectiondialog ();
  gtk_widget_destroy (GTK_FONT_SELECTION_DIALOG (fontselectiondialog)->
		      apply_button);
  gtk_font_selection_dialog_set_preview_text (GTK_FONT_SELECTION_DIALOG
					      (fontselectiondialog),
					      _("How about this font?"));
  gtk_font_selection_dialog_set_font_name (GTK_FONT_SELECTION_DIALOG
					   (fontselectiondialog),
					   preferences.themefont);
  gtk_widget_set_sensitive (preferencesdialog, 0);
  gtk_widget_show (fontselectiondialog);
}

/* change path for the logs */
void
on_pathbutton_clicked (GtkButton * button, gpointer user_data)
{
  GtkWidget *pathselection;

  pathselection = create_pathselection ();
  gtk_entry_set_editable (GTK_ENTRY
			  (GTK_FILE_SELECTION (pathselection)->
			   selection_entry), FALSE);
  gtk_widget_set_sensitive (preferencesdialog, 0);
  gtk_widget_show (pathselection);
}

/* check autosave */
void
on_autosavecheckbutton_toggled (GtkToggleButton *togglebutton,
                                    gpointer user_data)
{
  gboolean state;
  GtkWidget *savecheckbutton, *autosaveframe;
  
  state = gtk_toggle_button_get_active(togglebutton);
  savecheckbutton = lookup_widget(preferencesdialog, "savecheckbutton");
  if (gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(savecheckbutton)))
    gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(savecheckbutton), !state);
  autosaveframe = lookup_widget(preferencesdialog, "autosaveframe");
  gtk_widget_set_sensitive(autosaveframe, state);
}

/* check save */
void
on_savecheckbutton_toggled (GtkToggleButton *togglebutton,
                                    gpointer user_data)
{
  gboolean state;
  GtkWidget *autosavecheckbutton, *autosaveframe;
  
  state = gtk_toggle_button_get_active(togglebutton);
  autosavecheckbutton = lookup_widget(preferencesdialog, "autosavecheckbutton");
  if (gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(autosavecheckbutton)))
    gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(autosavecheckbutton), !state);
  autosaveframe = lookup_widget(preferencesdialog, "autosaveframe");
  gtk_widget_set_sensitive(autosaveframe, !state);
}


/* COLOR SELECTION DIALOG */
void
on_colorokbutton_clicked (GtkButton * button, gpointer user_data)
{
  GtkWidget *colorselectiondialog, *examplelabel, *exampleframe;
  gdouble rgb[3];
  gchar *colorred, *colorgreen, *colorblue, *color;

  colorselectiondialog = gtk_widget_get_toplevel (GTK_WIDGET (button));
  gtk_color_selection_get_color (GTK_COLOR_SELECTION
				 (GTK_COLOR_SELECTION_DIALOG
				  (colorselectiondialog)->colorsel), rgb);
  colorred = g_strdup_printf ("%04X", (gint) (rgb[0] * 65535));
  colorgreen = g_strdup_printf ("%04X", (gint) (rgb[1] * 65535));
  colorblue = g_strdup_printf ("%04X", (gint) (rgb[2] * 65535));
  color = g_strconcat (colorred, ",", colorgreen, ",", colorblue, NULL);
  exampleframe = lookup_widget (preferencesdialog, "exampleframe");
  setframestyle (exampleframe, color, preferences.themefont);
  examplelabel = lookup_widget (preferencesdialog, "examplelabel");
  setlabelstyle (examplelabel, color, preferences.themefont);
  g_free (colorblue);
  g_free (colorgreen);
  g_free (colorred);
  g_free (color);
  gtk_widget_set_sensitive (preferencesdialog, 1);
  gtk_widget_destroy (gtk_widget_get_toplevel (GTK_WIDGET (button)));
}


/* cancel color selection */
void
on_colorcancelbutton_clicked (GtkButton * button, gpointer user_data)
{
  gtk_widget_set_sensitive (preferencesdialog, 1);
  gtk_widget_destroy (gtk_widget_get_toplevel (GTK_WIDGET (button)));
}

/* color selection dialog is deleted */
gboolean
on_colorselectiondialog_delete_event (GtkWidget * widget, GdkEvent * event,
				      gpointer user_data)
{
  gtk_widget_set_sensitive (preferencesdialog, 1);
  return FALSE;
}

/* PATH SELECTION DIALOG */
gboolean
on_pathselection_delete_event (GtkWidget * widget, GdkEvent * event,
			       gpointer user_data)
{
  gtk_widget_set_sensitive (preferencesdialog, 1);
  return FALSE;
}


void
on_pathselection_ok_button_clicked (GtkButton * button, gpointer user_data)
{
  GtkWidget *pathselection, *pathentry;
  gchar *selpath;

  pathselection = gtk_widget_get_toplevel (GTK_WIDGET (button));
  selpath =
    gtk_file_selection_get_filename (GTK_FILE_SELECTION (pathselection));
  pathentry = lookup_widget (preferencesdialog, "pathentry");
  gtk_entry_set_text (GTK_ENTRY (pathentry), selpath);
  gtk_widget_destroy (pathselection);
  gtk_widget_set_sensitive (preferencesdialog, 1);
}


void
on_pathselection_cancel_button_clicked (GtkButton * button,
					gpointer user_data)
{
  gtk_widget_destroy (gtk_widget_get_toplevel (GTK_WIDGET (button)));
  gtk_widget_set_sensitive (preferencesdialog, 1);
}

/* FONT SELECTION DIALOG */
void
on_fontokbutton_clicked (GtkButton * button, gpointer user_data)
{
  GtkWidget *fontselectiondialog, *fontentry, *exampleframe, *examplelabel;
  gchar *font;

  fontselectiondialog = gtk_widget_get_toplevel (GTK_WIDGET (button));
  font = gtk_font_selection_dialog_get_font_name
    (GTK_FONT_SELECTION_DIALOG (fontselectiondialog));
  fontentry = lookup_widget (preferencesdialog, "fontentry");
  gtk_entry_set_text (GTK_ENTRY (fontentry), font);
  exampleframe = lookup_widget (preferencesdialog, "exampleframe");
  setframestyle (exampleframe, preferences.themecolor, font);
  examplelabel = lookup_widget (preferencesdialog, "examplelabel");
  setlabelstyle (examplelabel, preferences.themecolor, font);
  gtk_widget_set_sensitive (preferencesdialog, 1);
  gtk_widget_destroy (fontselectiondialog);
}


/* delete font selection dialog */
gboolean
on_fontselectiondialog_delete_event (GtkWidget * widget, GdkEvent * event,
				     gpointer user_data)
{
  gtk_widget_set_sensitive (preferencesdialog, 1);
  return FALSE;
}

/* cancel font selection */
void
on_fontcancelbutton_clicked (GtkButton * button, gpointer user_data)
{
  gtk_widget_set_sensitive (preferencesdialog, 1);
  gtk_widget_destroy (gtk_widget_get_toplevel (GTK_WIDGET (button)));
}

/* end of file */
