/*
  This file is part of TALER
  Copyright (C) 2025 Taler Systems SA

  TALER is free software; you can redistribute it and/or modify it under the
  terms of the GNU General Public License as published by the Free Software
  Foundation; either version 3, or (at your option) any later version.

  TALER is distributed in the hope that it will be useful, but WITHOUT ANY
  WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR
  A PARTICULAR PURPOSE.  See the GNU General Public License for more details.

  You should have received a copy of the GNU General Public License along with
  TALER; see the file COPYING.  If not, see
  <http://www.gnu.org/licenses/>
*/
/**
 * @file lib/exchange_api_blinding_prepare.c
 * @brief Implementation of /blinding-prepare requests
 * @author Özgür Kesim
 */

#include "taler/platform.h"
#include <gnunet/gnunet_common.h>
#include <jansson.h>
#include <microhttpd.h> /* just for HTTP status codes */
#include <gnunet/gnunet_util_lib.h>
#include <gnunet/gnunet_json_lib.h>
#include <gnunet/gnunet_curl_lib.h>
#include <sys/wait.h>
#include "taler/taler_curl_lib.h"
#include "taler/taler_error_codes.h"
#include "taler/taler_json_lib.h"
#include "taler/taler_exchange_service.h"
#include "exchange_api_common.h"
#include "exchange_api_handle.h"
#include "taler/taler_signatures.h"
#include "exchange_api_curl_defaults.h"
#include "taler/taler_util.h"

/**
 * A /blinding-prepare request-handle
 */
struct TALER_EXCHANGE_BlindingPrepareHandle
{
  /**
   * number of elements to prepare
   */
  size_t num;

  /**
   * True, if this operation is for melting (or withdraw otherwise).
   */
  bool for_melt;

  /**
   * The seed for the batch of nonces.
   */
  const struct TALER_BlindingMasterSeedP *seed;

  /**
   * The url for this request.
   */
  char *url;

  /**
   * Context for curl.
   */
  struct GNUNET_CURL_Context *curl_ctx;

  /**
   * CURL handle for the request job.
   */
  struct GNUNET_CURL_Job *job;

  /**
   * Post Context
   */
  struct TALER_CURL_PostContext post_ctx;

  /**
   * Function to call with withdraw response results.
   */
  TALER_EXCHANGE_BlindingPrepareCallback callback;

  /**
   * Closure for @e callback
   */
  void *callback_cls;

};


/**
 * We got a 200 OK response for the /blinding-prepare operation.
 * Extract the r_pub values and return them to the caller via the callback
 *
 * @param handle operation handle
 * @param response response details
 * @return #GNUNET_OK on success, #GNUNET_SYSERR on errors
 */
static enum GNUNET_GenericReturnValue
blinding_prepare_ok (struct TALER_EXCHANGE_BlindingPrepareHandle *handle,
                     struct TALER_EXCHANGE_BlindingPrepareResponse *response)
{
  const json_t *j_r_pubs;
  const char *cipher;
  struct GNUNET_JSON_Specification spec[] = {
    GNUNET_JSON_spec_string ("cipher",
                             &cipher),
    GNUNET_JSON_spec_array_const ("r_pubs",
                                  &j_r_pubs),
    GNUNET_JSON_spec_end ()
  };

  if (GNUNET_OK !=
      GNUNET_JSON_parse (response->hr.reply,
                         spec,
                         NULL, NULL))
  {
    GNUNET_break_op (0);
    return GNUNET_SYSERR;
  }

  if (strcmp ("CS", cipher))
  {
    GNUNET_break_op (0);
    return GNUNET_SYSERR;
  }

  if (json_array_size (j_r_pubs)
      != handle->num)
  {
    GNUNET_break_op (0);
    return GNUNET_SYSERR;
  }

  {
    size_t num = handle->num;
    const json_t *j_pair;
    size_t idx;
    struct TALER_ExchangeBlindingValues blinding_values[GNUNET_NZL (num)];

    memset (blinding_values,
            0,
            sizeof(blinding_values));

    json_array_foreach (j_r_pubs, idx, j_pair) {
      struct GNUNET_CRYPTO_BlindingInputValues *bi =
        GNUNET_new (struct GNUNET_CRYPTO_BlindingInputValues);
      struct GNUNET_CRYPTO_CSPublicRPairP *csv = &bi->details.cs_values;
      struct GNUNET_JSON_Specification tuple[] =  {
        GNUNET_JSON_spec_fixed (NULL,
                                &csv->r_pub[0],
                                sizeof(csv->r_pub[0])),
        GNUNET_JSON_spec_fixed (NULL,
                                &csv->r_pub[1],
                                sizeof(csv->r_pub[1])),
        GNUNET_JSON_spec_end ()
      };
      struct GNUNET_JSON_Specification jspec[] = {
        TALER_JSON_spec_tuple_of (NULL, tuple),
        GNUNET_JSON_spec_end ()
      };
      const char *err_msg;
      unsigned int err_line;

      if (GNUNET_OK !=
          GNUNET_JSON_parse (j_pair,
                             jspec,
                             &err_msg,
                             &err_line))
      {
        GNUNET_break_op (0);
        GNUNET_free (bi);
        for (size_t i=0; i < idx; i++)
          TALER_denom_ewv_free (&blinding_values[i]);
        GNUNET_log (GNUNET_ERROR_TYPE_ERROR,
                    "Error while parsing response: in line %d: %s",
                    err_line,
                    err_msg);
        return GNUNET_SYSERR;
      }

      bi->cipher = GNUNET_CRYPTO_BSA_CS;
      bi->rc = 1;
      blinding_values[idx].blinding_inputs = bi;
    }

    response->details.ok.blinding_values = blinding_values;
    response->details.ok.num_blinding_values = num;

    handle->callback (
      handle->callback_cls,
      response);

    for (size_t i = 0; i < num; i++)
      TALER_denom_ewv_free (&blinding_values[i]);
  }
  return GNUNET_OK;
}


/**
 * Function called when we're done processing the HTTP /blinding-prepare request.
 *
 * @param cls the `struct TALER_EXCHANGE_BlindingPrepareHandle`
 * @param response_code HTTP response code, 0 on error
 * @param response parsed JSON result, NULL on error
 */
static void
handle_blinding_prepare_finished (void *cls,
                                  long response_code,
                                  const void *response)
{
  struct TALER_EXCHANGE_BlindingPrepareHandle *handle = cls;
  const json_t *j_response = response;
  struct TALER_EXCHANGE_BlindingPrepareResponse bpr = {
    .hr = {
      .reply = j_response,
      .http_status = (unsigned int) response_code
    },
  };

  handle->job = NULL;

  switch (response_code)
  {
  case 0:
    bpr.hr.ec = TALER_EC_GENERIC_INVALID_RESPONSE;
    break;

  case MHD_HTTP_OK:
    {
      if (GNUNET_OK !=
          blinding_prepare_ok (handle,
                               &bpr))
      {
        GNUNET_break_op (0);
        bpr.hr.http_status = 0;
        bpr.hr.ec = TALER_EC_GENERIC_REPLY_MALFORMED;
        break;
      }
    }
    TALER_EXCHANGE_blinding_prepare_cancel (handle);
    return;

  case MHD_HTTP_BAD_REQUEST:
    /* This should never happen, either us or the exchange is buggy
       (or API version conflict); just pass JSON reply to the application */
    bpr.hr.ec = TALER_JSON_get_error_code (j_response);
    bpr.hr.hint = TALER_JSON_get_error_hint (j_response);
    break;

  case MHD_HTTP_NOT_FOUND:
    /* Nothing really to verify, the exchange basically just says
       that it doesn't know the /csr endpoint or denomination.
       Can happen if the exchange doesn't support Clause Schnorr.
       We should simply pass the JSON reply to the application. */
    bpr.hr.ec = TALER_JSON_get_error_code (j_response);
    bpr.hr.hint = TALER_JSON_get_error_hint (j_response);
    break;

  case MHD_HTTP_GONE:
    /* could happen if denomination was revoked */
    /* Note: one might want to check /keys for revocation
       signature here, alas tricky in case our /keys
       is outdated => left to clients */
    bpr.hr.ec = TALER_JSON_get_error_code (j_response);
    bpr.hr.hint = TALER_JSON_get_error_hint (j_response);
    break;

  case MHD_HTTP_INTERNAL_SERVER_ERROR:
    /* Server had an internal issue; we should retry, but this API
       leaves this to the application */
    bpr.hr.ec = TALER_JSON_get_error_code (j_response);
    bpr.hr.hint = TALER_JSON_get_error_hint (j_response);
    break;

  default:
    /* unexpected response code */
    GNUNET_break_op (0);
    bpr.hr.ec = TALER_JSON_get_error_code (j_response);
    bpr.hr.hint = TALER_JSON_get_error_hint (j_response);
    GNUNET_log (GNUNET_ERROR_TYPE_ERROR,
                "Unexpected response code %u/%d for the blinding-prepare request\n",
                (unsigned int) response_code,
                (int) bpr.hr.ec);
    break;

  }

  handle->callback (handle->callback_cls,
                    &bpr);
  handle->callback = NULL;
  TALER_EXCHANGE_blinding_prepare_cancel (handle);
}


struct TALER_EXCHANGE_BlindingPrepareHandle *
TALER_EXCHANGE_blinding_prepare (
  struct GNUNET_CURL_Context *curl_ctx,
  const char *exchange_url,
  const struct TALER_BlindingMasterSeedP *seed,
  bool for_melt,
  size_t num,
  const struct TALER_EXCHANGE_NonceKey nonce_keys[static num],
  TALER_EXCHANGE_BlindingPrepareCallback callback,
  void *callback_cls)
{
  struct TALER_EXCHANGE_BlindingPrepareHandle *bph;

  if (0 == num)
  {
    GNUNET_break (0);
    return NULL;
  }
  for (unsigned int i = 0; i<num; i++)
    if (GNUNET_CRYPTO_BSA_CS !=
        nonce_keys[i].pk->key.bsign_pub_key->cipher)
    {
      GNUNET_break (0);
      return NULL;
    }
  bph = GNUNET_new (struct TALER_EXCHANGE_BlindingPrepareHandle);
  bph->num = num;
  bph->callback = callback;
  bph->for_melt = for_melt;
  bph->callback_cls = callback_cls;
  bph->url = TALER_url_join (exchange_url,
                             "blinding-prepare",
                             NULL);
  if (NULL == bph->url)
  {
    GNUNET_break (0);
    GNUNET_free (bph);
    return NULL;
  }
  {
    CURL *eh;
    json_t *j_nks;
    json_t *j_request = GNUNET_JSON_PACK (
      GNUNET_JSON_pack_string ("cipher",
                               "CS"),
      GNUNET_JSON_pack_string ("operation",
                               for_melt ? "melt" : "withdraw"),
      GNUNET_JSON_pack_data_auto ("seed",
                                  seed));
    GNUNET_assert (NULL != j_request);

    j_nks = json_array ();
    GNUNET_assert (NULL != j_nks);

    for (size_t i = 0; i<num; i++)
    {
      const struct TALER_EXCHANGE_NonceKey *nk = &nonce_keys[i];
      json_t *j_entry = GNUNET_JSON_PACK (
        GNUNET_JSON_pack_uint64 ("coin_offset",
                                 nk->cnc_num),
        GNUNET_JSON_pack_data_auto ("denom_pub_hash",
                                    &nk->pk->h_key));

      GNUNET_assert (NULL != j_entry);
      GNUNET_assert (0 ==
                     json_array_append_new (j_nks,
                                            j_entry));
    }
    GNUNET_assert (0 ==
                   json_object_set_new (j_request,
                                        "nks",
                                        j_nks));
    eh = TALER_EXCHANGE_curl_easy_get_ (bph->url);
    if ( (NULL == eh) ||
         (GNUNET_OK !=
          TALER_curl_easy_post (&bph->post_ctx,
                                eh,
                                j_request)))
    {
      GNUNET_break (0);
      if (NULL != eh)
        curl_easy_cleanup (eh);
      json_decref (j_request);
      GNUNET_free (bph->url);
      GNUNET_free (bph);
      return NULL;
    }

    json_decref (j_request);
    bph->job = GNUNET_CURL_job_add2 (curl_ctx,
                                     eh,
                                     bph->post_ctx.headers,
                                     &handle_blinding_prepare_finished,
                                     bph);
    if (NULL == bph->job)
    {
      GNUNET_break (0);
      TALER_EXCHANGE_blinding_prepare_cancel (bph);
      return NULL;
    }
  }
  return bph;
}


void
TALER_EXCHANGE_blinding_prepare_cancel (
  struct TALER_EXCHANGE_BlindingPrepareHandle *bph)
{
  if (NULL == bph)
    return;
  if (NULL != bph->job)
  {
    GNUNET_CURL_job_cancel (bph->job);
    bph->job = NULL;
  }
  GNUNET_free (bph->url);
  TALER_curl_easy_post_finished (&bph->post_ctx);
  GNUNET_free (bph);
}


/* end of lib/exchange_api_blinding_prepare.c */
