/*  gnutrition - a nutrition and diet analysis program.
 *  Copyright (C) 2000 Edgar Denny (e.denny@ic.ac.uk)
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */
#ifdef HAVE_CONFIG_H
#	include <config.h>
#endif

#include <gnome.h>
#include <glade/glade.h>
#include <gnome-xml/tree.h>
#include <gnome-xml/parser.h>
#include <ctype.h>

#include "text_util.h"
#include "food.h"
#include "recipe.h"
#include "nutrgoal_io.h"
#include "recipe_win.h"
#include "support.h"
#include "nutr_goal_dlg.h"
#include "nutr_comp_dlg.h"
#include "search_dlg.h"
#include "nutr_io.h"

GList *food_glist = NULL;
static GladeXML *xml = NULL;
static Recipe_t *recipe = NULL;

/* Load the database files, and update progress bar. */
gint 
on_idle_in_main( gpointer dummy)
{
	GtkWidget *progressbar = NULL;
	static gboolean loaded_xml = FALSE;
	static gboolean loaded_foods = FALSE;

	/* load the glade interface. */
	if ( !loaded_xml)
	{
		xml = glade_xml_new( GNUTRITION_GLADEDIR "/gnutrition.glade", NULL);

		if ( xml != NULL)
		{
			GtkWidget *progress_bar_win = NULL;

			glade_xml_signal_autoconnect( xml);

			progress_bar_win = glade_xml_get_widget( xml, "pb_window");
			gtk_widget_show( progress_bar_win);
		}
		else
		{
			g_log ("Gnutrition", G_LOG_LEVEL_ERROR,
					"on_idle_in_main: cannot load glade file\n");
		}
		loaded_xml = TRUE;

		return( TRUE);
	}

	progressbar = glade_xml_get_widget( xml, "pb_progressbar");

	/* load the list of foods. */
	if ( !loaded_foods)
	{
		food_glist = gnutr_create_food_glist();
		gtk_progress_set_value( GTK_PROGRESS( progressbar), 95.0);
		loaded_foods = TRUE;
		return( TRUE);
	}

	/* hide the progress bar, allocate recipe, and
	 * show the recipie window. */
	{
		GtkWidget *m_app_win = glade_xml_get_widget( xml, "mw_app");
		GtkWidget *progress_bar_win = glade_xml_get_widget( xml, "pb_window");

		recipe = ( Recipe_t *)g_malloc( sizeof( Recipe_t));
		recipe->nutr = ( Nutrition_t *)g_malloc( sizeof( Nutrition_t));
		recipe->ingredient_clist = (GtkCList *)
				glade_xml_get_widget( xml, "mw_fd_clist");
		recipe->no_ingredients = 0;
		recipe->file_name = NULL;
		recipe->selected_row = -1;
		recipe->name = NULL;

		/* read in nutrient goals. */
		gnutr_read_nutr_goal( &(recipe->nutr));

		gtk_widget_hide( progress_bar_win);
		gtk_widget_show( m_app_win);
	}
	return( FALSE);
}

Recipe_t *
gnutr_ret_recipe()
{
	return recipe; 
}

GladeXML *
gnutr_ret_xml()
{
	return xml;
}

GList *
gnutr_ret_food_glist()
{
	return food_glist;
}

/* Exit the application when "exit" is selected from the recipe
 * window menu bar. */
void
on_exit1_activate( GtkMenuItem *menuitem,
                   gpointer     user_data)
{

	gnutr_free_recipe();
	gnutr_free_food_glist( &food_glist);
	gtk_main_quit();
}

/* Open a file dialog when "new recipe" is selected from the
 * recipe window menu bar. */
void
on_new_file1_activate( GtkMenuItem *menuitem,
                       gpointer     user_data)
{
	GtkButton *dummy = NULL;
	on_mw_new_button_released( dummy, user_data);
}

/* Open the search dialog when "add food" is selected from the
 * recipe window menu bar. */
void
on_add_food1_activate( GtkMenuItem *menuitem,
                       gpointer     user_data)
{
	GtkButton *dummy = NULL;
	on_mw_add_fd_released( dummy, user_data);
}


/* Open the search dialog when "edit food" is selected from the
 * recipe window menu bar. */
void
on_edit_food1_activate( GtkMenuItem *menuitem,
                        gpointer     user_data)
{
	GtkButton *dummy = NULL;
	on_mw_edit_released( dummy, user_data);
}

/* Delete the selected ingredient when "delete food" is selected from the
 * recipe window menu bar. */
void
on_delete_food1_activate( GtkMenuItem *menuitem,
                          gpointer     user_data)
{
	GtkButton *dummy = NULL;
	on_mw_delete_released( dummy, user_data);
}

/* Open the nutrient goal dialog when "nutrient goal" is selected from the
 * recipe window menu bar. */
void
on_pref1_activate( GtkMenuItem *menuitem,
                   gpointer     user_data)
{
	GtkButton *dummy = NULL;
	on_mw_nutr_goal_released( dummy, user_data);
}

/* The about dialog. */
void
on_about1_activate( GtkMenuItem *menuitem,
                    gpointer     user_data)
{
	GtkWidget *about_dlg = glade_xml_get_widget( xml, "about1");
	gnome_dialog_close_hides( GNOME_DIALOG( about_dlg), TRUE);
	gtk_widget_show( about_dlg); 
}

/* removes an ingredient from the recipe when the delete button is
 * released in the AppBar on the recipe window. */
void
on_mw_delete_released                  (GtkButton       *button,
                                        gpointer         user_data)
{
	/* check that an ingredient has been selected. */
	if ( recipe->selected_row == -1)
	{
		GtkWidget *message = glade_xml_get_widget( xml, "nis_msgbox");
		gnome_dialog_close_hides( GNOME_DIALOG( message), TRUE);
	 	gtk_widget_show( message);
		return;
	}

	gtk_clist_remove( recipe->ingredient_clist, recipe->selected_row);
	recipe->no_ingredients--;
	recipe->selected_row = -1;
}

/* get the row that has been selected the the ingredient list. */
void
on_mw_fd_clist_select_row( GtkCList *clist,
                           gint      row,
                           gint      column,
                           GdkEvent *event,
                           gpointer  user_data)
{
	recipe->selected_row = row;
}


/* Just here for the future. */
void
on_manual1_activate( GtkMenuItem *menuitem,
                     gpointer     user_data)
{

}

/* reset all the recipe text entries to null and free
 * any existing memory. */
void
gnutr_free_recipe()
{		
	GtkWidget *recipe_name_entry = 
			glade_xml_get_widget( xml, "mw_recipe_entry");
	GtkWidget *no_serv_entry = 
			glade_xml_get_widget( xml, "mw_no_serv_entry");

	recipe->ingredient_clist = (GtkCList *)
			glade_xml_get_widget( xml, "mw_fd_clist");

	gtk_clist_clear( recipe->ingredient_clist);

	recipe->no_ingredients = 0;

	g_free( recipe->file_name); 
	recipe->file_name = NULL;

	recipe->name = NULL;
	gtk_entry_set_text( GTK_ENTRY( recipe_name_entry), "");

	recipe->no_serv = NULL;
	gtk_entry_set_text( GTK_ENTRY( no_serv_entry), "");
}


/* When the OK button is clicked on: "the number of servings is not
 * specified" message box. */
void
on_nserv_msgbox_clicked( GnomeDialog *gnomedialog,
                         gint         button,
                         gpointer     user_data)
{
	gtk_widget_hide( GTK_WIDGET( gnomedialog));
}

/* update the status bar on entering the new recipe button in the 
 * recipe window. */
void
on_mw_new_button_enter( GtkButton *button,
                        gpointer   user_data)
{
	GtkWidget *widget = glade_xml_get_widget( xml, "mw_status_msg");
	gnome_appbar_push( GNOME_APPBAR( widget), 
			"Create a New Recipe.");
}

/* update the status bar on entering the open recipe file button in the 
 * recipe window. */
void
on_mw_open_button_enter( GtkButton *button,
                         gpointer   user_data)
{
	GtkWidget *widget = glade_xml_get_widget( xml, "mw_status_msg");
	gnome_appbar_push( GNOME_APPBAR( widget), 
			"Open an Existing Recipe File.");
}

/* update the status bar on entering the save recipe to file button in the 
 * recipe window. */
void
on_mw_save_button_enter( GtkButton *button,
                         gpointer   user_data)
{
	GtkWidget *widget = glade_xml_get_widget( xml, "mw_status_msg");
	gnome_appbar_push( GNOME_APPBAR( widget), 
			"Save the Recipe to File.");
}

/* update the status bar on entering the add ingredient to recipe button
 * in the recipe window. */
void
on_add_button_enter( GtkButton *button,
                     gpointer   user_data)
{
	GtkWidget *widget = glade_xml_get_widget( xml, "mw_status_msg");
	gnome_appbar_push( GNOME_APPBAR( widget), 
			"Add an Ingredient to the Recipe.");
}

/* update the status bar on entering the delete ingredient from recipe 
 * button in the recipe window. */
void
on_delete_button_enter( GtkButton *button,
                        gpointer   user_data)
{
	GtkWidget *widget = glade_xml_get_widget( xml, "mw_status_msg");
	gnome_appbar_push( GNOME_APPBAR( widget), 
			"Delete the Selected Ingredient from the Recipe.");
}

/* update the status bar on entering the edit ingredient 
 * button in the recipe window. */
void
on_edit_button_enter( GtkButton *button,
                      gpointer   user_data)
{
	GtkWidget *widget = glade_xml_get_widget( xml, "mw_status_msg");
	gnome_appbar_push( GNOME_APPBAR( widget), 
			"Edit the Selected Ingredient.");
}

/* update the status bar on entering the show nutrient composition dialog
 * button in the recipe window. */
void
on_nutr_comp_button_enter( GtkButton *button,
                           gpointer   user_data)
{
	GtkWidget *widget = glade_xml_get_widget( xml, "mw_status_msg");
	gnome_appbar_push( GNOME_APPBAR( widget), 
			"Open the Recipe Nutrient Composition Dialog.");
}

/* update the status bar on entering the show nutrient goal dialog
 * button in the recipe window. */
void
on_nutr_goal_button_enter( GtkButton *button,
                           gpointer   user_data)
{
	GtkWidget *widget = glade_xml_get_widget( xml, "mw_status_msg");
	gnome_appbar_push( GNOME_APPBAR( widget), 
			"Open the Nutrient Goal Dialog.");
}

/* on pointer leaving new recipe button in recipe window. */
void
on_mw_new_button_leave( GtkButton *button,
                        gpointer   user_data)
{
	GtkWidget *widget = glade_xml_get_widget( xml, "mw_status_msg");
	gnome_appbar_clear_stack( GNOME_APPBAR( widget));
}

/* on pointer leaving open recipe button in recipe window. */
void
on_mw_open_button_leave( GtkButton *button,
                         gpointer   user_data)
{
	GtkWidget *widget = glade_xml_get_widget( xml, "mw_status_msg");
	gnome_appbar_clear_stack( GNOME_APPBAR( widget));
}

/* on pointer leaving save recipe button in recipe window. */
void
on_mw_save_button_leave( GtkButton *button,
                         gpointer   user_data)
{
	GtkWidget *widget = glade_xml_get_widget( xml, "mw_status_msg");
	gnome_appbar_clear_stack( GNOME_APPBAR( widget));
}

/* on pointer leaving add ingredient button in recipe window. */
void
on_add_button_leave( GtkButton *button,
                     gpointer   user_data)
{
	GtkWidget *widget = glade_xml_get_widget( xml, "mw_status_msg");
	gnome_appbar_clear_stack( GNOME_APPBAR( widget));
}

/* on pointer leaving delete ingredient button in recipe window. */
void
on_delete_button_leave( GtkButton *button,
                        gpointer   user_data)
{
	GtkWidget *widget = glade_xml_get_widget( xml, "mw_status_msg");
	gnome_appbar_clear_stack( GNOME_APPBAR( widget));
}

/* on pointer leaving edit ingredient button in recipe window. */
void
on_edit_button_leave( GtkButton *button,
                      gpointer   user_data)
{
	GtkWidget *widget = glade_xml_get_widget( xml, "mw_status_msg");
	gnome_appbar_clear_stack( GNOME_APPBAR( widget));
}

/* on pointer leaving nutrient composition  button in recipe window. */
void
on_nutr_comp_button_leave( GtkButton *button,
                           gpointer   user_data)
{
	GtkWidget *widget = glade_xml_get_widget( xml, "mw_status_msg");
	gnome_appbar_clear_stack( GNOME_APPBAR( widget));
}

/* on pointer leaving nutrition goal button in recipe window. */
void
on_nutr_goal_button_leave( GtkButton *button,
                           gpointer   user_data)
{
	GtkWidget *widget = glade_xml_get_widget( xml, "mw_status_msg");
	gnome_appbar_clear_stack( GNOME_APPBAR( widget));
}
