/* Miscellaneous tools common to all types of queries.
   Copyright (C) 1994, 95, 96, 1997 Free Software Foundation, Inc.
   Contributed by Brendan Kehoe (brendan@cygnus.com).

This file is part of GNU GNATS.

GNU GNATS is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation; either version 2, or (at your option)
any later version.

GNU GNATS is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with GNU GNATS; see the file COPYING.  If not, write to the Free
Software Foundation, 59 Temple Place - Suite 330, Boston, MA 02111, USA.  */

#include "config.h"
#include "gnats.h"
#include "query.h"
#include "pcodes.h"

char *
disbar (str)
     char *str;
{
  char *q;
  
  while ((q = strchr (str, '|')))
    *q = '!';

  return str;
}

/* Return the numeric equivalent of this state; 0 if not available. */
int
enum_numeric (text, field)
     char *text;
     int   field;
{
  char *values;
  char *idx, *last_idx;
  int count, slen, llen;

  /* Init everybody and perform trivial checks. */
  count = 0;
  if ((text == NULL) || (text[0] == '\0'))
    return 0;
  /* else */
  values = pr[field].enum_values; /* gets us a '|'-separated list. */
  if ((values == NULL) || (values[0] == '\0'))
    return 0; /* don't enforce format of values here */
  /* else */
  slen = strlen (text);
  llen = strlen (values);

  for (last_idx = idx = values; idx != NULL; idx = strchr (idx, '|'))
    {
      /* Inch past the bar and its trailing space; but don't do this
         if we're still on the first value. */
      if (count > 0)
        idx += 2;

      /* Return 0 if match impossible. */
      llen -= (idx - last_idx);
      if (slen > llen)
        return 0;

      if ((strncmp (idx, text, slen) == 0)
          && ((idx[slen] == ' ') || (idx[slen] == '\0')))
        return (count + 1); /* no o-b-o-e playing here! */
      else
        count++;

      last_idx = idx;
    }

  return 0;
}

int
sql_types (p, type)
     char *p;
     Sql_Types type;
{
  switch (type)
    {
    case Severity:
      return enum_numeric (p, SEVERITY);
      break;
    case Priority:
      return enum_numeric (p, PRIORITY);
      break;
    case State:
      return enum_numeric (p, STATE);
      break;
    case Class:
      return enum_numeric (p, CLASS);
      break;
    }

  return 0;
}

char *
sql_time (s)
     char *s;
{
  time_t t;
  struct tm *ar_time;
  /* Note: we deliberately use 21 here, since that is what the width of
     this time string will end up looking like.  In the case where we
     use things relative to timezone and such, it'd grow---but not here.  */
  char *buf = (char *) xmalloc (21);

  t = get_date (s, NULL);
  ar_time = (struct tm *) localtime (&t);
  strftime (buf, 21, "%Y-%m-%d %H:%M:%S", ar_time);

  return buf;
}

char *
sql_time_from_index (s)
     char *s;
{
  time_t t;
  struct tm *ar_time;
  /* Note: we deliberately use 21 here, since that is what the width of
     this time string will end up looking like.  In the case where we
     use things relative to timezone and such, it'd grow---but not here.  */
  char *buf = (char *) xmalloc (21);

  t = (time_t) atol (s);
  ar_time = (struct tm *) localtime (&t);
  strftime (buf, 21, "%Y-%m-%d %H:%M:%S", ar_time);

  return buf;
}

char *
make_path (c, n)
     char *c, *n;
{
  char *path = (char *) xmalloc (PATH_MAX);

  sprintf (path, "%s/%s/%s", gnats_root, c, n);
  return path;
}

int
numeric (p)
     char *p;
{
  int i;
  int l = strlen(p);
  for (i=0; i<l; i++)
    {
      if (!isdigit (p[i]))
	return 0;
    }
  return 1;
}

static char case_fold[256];

/* Return 0 if matched, 1 otherwise. */
int
regcmp (pat, match)
     char *pat, *match;
{
  struct re_pattern_buffer buf;
  union {
    const char *c;
    int i;
  } r;

  memset ((void *) &buf, 0, sizeof (buf));

  if (case_fold[1] == 0)
    {
      int i;
      for (i=0; i<256; i++)
	case_fold[i] = tolower(i);
    }
  buf.translate = case_fold;
#ifdef USE_RX
  buf.syntax_parens = (char *)1;
#endif
  
  r.c = re_compile_pattern (pat, strlen (pat), &buf);
  if (r.c)
    {
      fprintf (stderr, "%s: re_compile_pattern: %s\n", program_name, r.c);
      exit (2);
    }
  r.i = re_match (&buf, match, strlen (match), 0, 0);
  buf.translate = NULL;
  regfree (&buf);
  switch (r.i)
    {
    case -2:
      fprintf (stderr, "%s: warning: re_match died with pattern %s and string $s\n",
	       program_name, pat, match);
      /*FALLTHRU*/
    case -1:
      return 1;
    default:
      return 0;
    }
}
      
/* Return 0 if found, 1 otherwise. */
int
regfind (pat, match)
     char *pat, *match;
{
  struct re_pattern_buffer buf;
  union {
    const char *c;
    int i;
  } r;

  memset ((void *) &buf, 0, sizeof (buf));

  if (case_fold[1] == 0)
    {
      int i;
      for (i=0; i<256; i++)
	case_fold[i] = tolower(i);
    }
  buf.translate = case_fold;
  buf.fastmap = xmalloc (256);
#ifdef USE_RX
  buf.syntax_parens = (char *)1;
#endif
  
  r.c = re_compile_pattern (pat, strlen (pat), &buf);
  if (r.c)
    {
      fprintf (stderr, "%s: re_compile_pattern: %s\n", program_name, r.c);
      exit (2);
    }
  r.i = strlen (match);
  r.i = re_search (&buf, match, r.i, 0, r.i, 0);
  buf.translate = NULL;
  regfree (&buf);
  switch (r.i)
    {
    case -2:
      fprintf (stderr, "%s: warning: re_match died with pattern %s and string $s\n",
	       program_name, pat, match);
      /*FALLTHRU*/
    case -1:
      return 1;
    default:
      return 0;
    }
}

/* Should we try to convert the date given to us?  */
#define NOCONVERT	0
#define DOCONVERT	1

int
date_compare (tst, sdate, type, convert_p)
     time_t tst;
     char *sdate;
     int type;
     int convert_p;
{
  time_t val;

  /* If it's still a flat date string, we need to use get_date instead of just
     doing an atol on the pre-converted value.  We assume that get_date can
     handle the value, since we should be getting one that's already made
     it through the system.  */
  if (convert_p)
    val = get_date (sdate, NULL);
  else
    val = atol (sdate);

  /* 0 means the date is unset, so nothing should match it */
  if (! val)
    return 1;

  if (type == 0)
    return val > tst; /* ! val < tst */
  else if (type == 1)
    return val < tst; /* ! val > tst */
  else
    abort (); /* Should never happen.  */
}

int
pr_matches (s, i, opened)
     Index *s;
     Index *i;
     int   *opened;
{
  int status;
  
  *opened = 0;

  if (skip_closed && check_state_type (i->state, "closed"))
    return 0;

  if (!s || !searching)
    return 1;

  if (!(
       (!s->category     || (regcmp  (s->category,     i->category)     == 0))
    && (!s->submitter    || (regcmp  (s->submitter,    i->submitter)    == 0))
    && (!s->responsible  || (regcmp  (s->responsible,  i->responsible)  == 0))
    && (!s->state        || (regcmp  (s->state,        i->state)        == 0))
    && (!s->confidential || (regcmp  (s->confidential, i->confidential) == 0))
    && (!s->severity     || (regcmp  (s->severity,     i->severity)     == 0))
    && (!s->priority     || (regcmp  (s->priority,     i->priority)     == 0))
    && (!s->originator   || (regfind (s->originator,   i->originator)   == 0))
    && (!s->class        || (regfind (s->class,        i->class)        == 0))
    && (!s->synopsis     || (regfind (s->synopsis,     i->synopsis)     == 0))
    && (!s->release      || (regfind (s->release,      i->release)      == 0))
#ifdef GNATS_RELEASE_BASED
    && (!s->quarter      || (regfind (s->quarter,      i->quarter)      == 0))
    && (!s->keywords     || (regfind (s->keywords,     i->keywords)     == 0))
    && (!required_before || (date_compare (required_before, i->date_required, 0, NOCONVERT) == 0))
    && (!required_after  || (date_compare (required_after,  i->date_required, 1, NOCONVERT) == 0))
#endif
    && (!arrived_before  || (date_compare (arrived_before,  i->arrival_date,  0, NOCONVERT) == 0))
    && (!arrived_after   || (date_compare (arrived_after,   i->arrival_date,  1, NOCONVERT) == 0))
    && (!modified_before || (date_compare (modified_before, i->last_modified, 0, NOCONVERT) == 0))
    && (!modified_after  || (date_compare (modified_after,  i->last_modified, 1, NOCONVERT) == 0))
    && (!closed_before   || (date_compare (closed_before,   i->closed_date,   0, NOCONVERT) == 0))
    && (!closed_after    || (date_compare (closed_after,    i->closed_date,   1, NOCONVERT) == 0))
    && (!text_search     || (check_text (s, i) == 1))))
       return 0;

  if (m_text_search)
    {
      int    val = 0;
      char *path = make_path (i->category, i->number);
      
      if (val = get_pr (path, i->number, quiet))
         *opened = 1;
      xfree (path);

      if (!val || !check_mtext())
        return 0;
    }
  
  return 1;
}

int
check_text (s, i)
     Index *s;
     Index *i;
{
  if (text_search)
    {
  return (
       /* Note that text_search uses regfind, but the individual query options use regcmp */
       (regfind (text_search, i->category)     == 0)
    || (regfind (text_search, i->submitter)    == 0)
    || (regfind (text_search, i->responsible)  == 0)
    || (regfind (text_search, i->state)        == 0)
    || (regfind (text_search, i->confidential) == 0)
    || (regfind (text_search, i->severity)     == 0)
    || (regfind (text_search, i->priority)     == 0)
    || (regfind (text_search, i->originator)   == 0)
    || (regfind (text_search, i->class)        == 0)
    || (regfind (text_search, i->synopsis)     == 0)
    || (regfind (text_search, i->release)      == 0)
#ifdef GNATS_RELEASE_BASED
    || (regfind (text_search, i->quarter)      == 0)
    || (regfind (text_search, i->keywords)     == 0)
#endif
       );
    }
  return 1;
}

int
check_mtext ()
{
  PR_Name i;

  if (m_text_search)
    {
      for (i = NUMBER; i < NUM_PR_ITEMS; i++)
	{
	  if (pr[i].datatype != MultiText)
	    continue;
	  
	  if (pr[i].value && regfind (m_text_search, pr[i].value) == 0)
	    return 1;
	}
      return 0;
    }
  return 1;
}

/* Get the PR at PATH.  If EXISTP is non-nil, only say if it existed
   or not.  If it's nil, read in the PR.  If SILENT is non-nil, we
   don't emit an error message.  */
int
get_pr (path, p, silent)
     char *path, *p;
     int silent;
{
  FILE *fp = fopen (path, "r");
     
  if (fp == (FILE *)NULL)
    {
      if (! silent)
	fprintf (stderr, "%s: couldn't read PR %s\n", program_name, p);
      return 0;
    }

  if (read_header (fp) < 0)
    {
      if (! silent)
	fprintf (stderr, "%s: error reading PR %s\n", program_name, p);
      return 0;
    }

  read_pr (fp, (query_format & ~FORMAT_FULL) && !m_text_search);

  fclose (fp);

  return 1;
}

int
do_pr (i, opened)
     Index *i;
     int   opened;
{
  int val = 0;
  char *path = make_path (i->category, i->number);

  val = print_pr (path, i->number, opened, i);
  
  xfree (path);

  return val;
}

/* Only say `PRs follow' once.  */
static int started = 0;

void
reset_started ()
{
  started = 0;
}

int
print_pr (path, p, opened, i)
     char *path, *p;
     int opened;
     Index *i;
{
  /* Note: don't use P in any real way, cuz it could be "1234" or
     "category/1234".  */
  if ((query_format & FORMAT_FULL) && (! opened && ! (get_pr (path, p, quiet))))
    return 0;

  if (is_daemon && !started)
    {
      printf ("%d PRs follow.\r\n", CODE_PR_READY);
      started = 1;
    }

  if (query_format & FORMAT_FULL)
    {
      write_header (outfile, NUM_HEADER_ITEMS);
      fprintf (outfile, "\n");
      write_pr (outfile, NUM_PR_ITEMS);
      return 1;
    }

  if (!i)
    return 0;
  
  if (query_format & FORMAT_SQL)
    {
      char *t, *q, *tempstr;

      tempstr = (char *) xmalloc (strlen (i->synopsis) + 1);
      strcpy (tempstr, i->synopsis);
      fprintf (outfile, "%-8.8s|%-16.16s|%-128.128s|%-3.3s|",
	       i->number, i->category, disbar (tempstr), i->confidential);
      xfree (tempstr);

      /* Trim `foo (Full Foo)' to just `foo'.  */
      tempstr = (char *) xmalloc (strlen (i->responsible) + 1);
      strcpy (tempstr, i->responsible);
      q = (char *) strchr (tempstr, ' ');
      if (q != NULL)
	*q = '\0';

      fprintf (outfile, "%1.1d|%1.1d|%-16.16s|%1.1d|%1.1d|%-16.16s|",
	       sql_types (i->severity, Severity), sql_types (i->priority, Priority),
	       tempstr,
	       sql_types (i->state, State), sql_types (i->class, Class),
	       i->submitter);
      xfree (tempstr);
      
      if (strlen (i->arrival_date) == 0 || !strcmp (i->arrival_date, "0")) {
        fprintf (outfile, "%-21.21s|", "");
      } else {
        t = sql_time_from_index (i->arrival_date);
        fprintf (outfile, "%-21.21s|", t);
        xfree (t);
      }
      
      fprintf (outfile, "%-64.64s|%-64.64s|", disbar (i->originator), i->release);
      
      if (strlen (i->last_modified) == 0 || !strcmp (i->last_modified, "0")) {
        fprintf (outfile, "%-21.21s|", "");
      } else {
        t = sql_time_from_index (i->last_modified);
        fprintf (outfile, "%-21.21s|", t);
        xfree (t);
      }
      
      if (strlen (i->closed_date) == 0 || !strcmp (i->closed_date, "0")) {
        fprintf (outfile, "%-21.21s|", "");
      } else {
        t = sql_time_from_index (i->closed_date);
        fprintf (outfile, "%-21.21s|", t);
        xfree (t);
      }
#ifdef GNATS_RELEASE_BASED
      fprintf (outfile, "%-64.64s|%-64.64s|", i->quarter, i->keywords);
      if (strlen (i->date_required) == 0 || !strcmp (i->date_required, "0")) {
        fprintf (outfile, "%-21.21s|", "");
      } else {
        t = sql_time_from_index (i->date_required);
        fprintf (outfile, "%-21.21s|", t);
        xfree (t);
      }
#endif
    }  
  else if (query_format & FORMAT_SQL2)
    {
      char *t, *q, *tempstr;

      tempstr = (char *) xmalloc (strlen (i->synopsis) + 1);
      strcpy (tempstr, i->synopsis);
      fprintf (outfile, "%s|%s|%s|%s|",
	       i->number, i->category, disbar (tempstr), i->confidential);
      xfree (tempstr);

      /* Trim `foo (Full Foo)' to just `foo'.  */
      tempstr = (char *) xmalloc (strlen (i->responsible) + 1);
      strcpy (tempstr, i->responsible);
      q = (char *) strchr (tempstr, ' ');
      if (q != NULL)
	*q = '\0';

      fprintf (outfile, "%d|%d|%s|%d|%d|%s|",
	       sql_types (i->severity, Severity), sql_types (i->priority, Priority),
	       tempstr,
	       sql_types (i->state, State), sql_types (i->class, Class),
	       i->submitter);
      xfree (tempstr);
      
      if (strlen (i->arrival_date) == 0 || !strcmp (i->arrival_date, "0")) {
        fprintf (outfile, "%s|", "");
      } else {
        t = sql_time_from_index (i->arrival_date);
        fprintf (outfile, "%s|", t);
        xfree (t);
      }
      
      fprintf (outfile, "%s|%s|", disbar (i->originator), i->release);
      
      if (strlen (i->last_modified) == 0 || !strcmp (i->last_modified, "0")) {
        fprintf (outfile, "%s|", "");
      } else {
        t = sql_time_from_index (i->last_modified);
        fprintf (outfile, "%s|", t);
        xfree (t);
      }
      
      if (strlen (i->closed_date) == 0 || !strcmp (i->closed_date, "0")) {
        fprintf (outfile, "%s|", "");
      } else {
        t = sql_time_from_index (i->closed_date);
        fprintf (outfile, "%s|", t);
        xfree (t);
      }
#ifdef GNATS_RELEASE_BASED
      fprintf (outfile, "%s|%s|", i->quarter, i->keywords);
      if (strlen (i->date_required) == 0 || !strcmp (i->date_required, "0")) {
        fprintf (outfile, "%s|", "");
      } else {
        t = sql_time_from_index (i->date_required);
        fprintf (outfile, "%s|", t);
        xfree (t);
      }
#endif
    }  
  else if (query_format & FORMAT_SUMM)
    {
      char *q, *tempstr;

      /* Trim `foo (Full Foo)' to just `foo'.  */
      tempstr = (char *) xmalloc (strlen (i->responsible) + 1);
      strcpy (tempstr, i->responsible);
      q = (char *) strchr (tempstr, ' ');
      if (q != NULL)
	*q = '\0';

      fprintf (outfile,
	       "%8s %-8.8s %-8.8s %-9.9s %-9.9s %-8.8s %-10.10s %s",
	       i->number,   i->responsible, i->category, i->state, i->severity,
	       i->priority, i->submitter,   i->synopsis);
      xfree (tempstr);
    }
  else
    {
      /* Print this out for emacs when people do `M-x query-pr' and want
	 to be able to use the next-error function on the buffer.  */
      if (print_path)
	fprintf (outfile, "%s:0:\n", path);

      write_pr_from_index (outfile, NUMBER,        i->number);
      write_pr_from_index (outfile, CATEGORY,      i->category);
      write_pr_from_index (outfile, SYNOPSIS,      i->synopsis);
      write_pr_from_index (outfile, CONFIDENTIAL,  i->confidential);
      write_pr_from_index (outfile, SEVERITY,      i->severity);
      write_pr_from_index (outfile, PRIORITY,      i->priority);
      write_pr_from_index (outfile, RESPONSIBLE,   i->responsible);
      write_pr_from_index (outfile, STATE,         i->state);
      write_pr_from_index (outfile, CLASS,         i->class);
#ifdef GNATS_RELEASE_BASED
      write_pr_from_index (outfile, QUARTER,       i->quarter);
      write_pr_from_index (outfile, KEYWORDS,      i->keywords);
      write_pr_from_index (outfile, DATE_REQUIRED, i->date_required);
#endif
      write_pr_from_index (outfile, SUBMITTER,     i->submitter);
      write_pr_from_index (outfile, ARRIVAL_DATE,  i->arrival_date);
      write_pr_from_index (outfile, CLOSED_DATE,   i->closed_date);
      write_pr_from_index (outfile, LAST_MODIFIED, i->last_modified);
      write_pr_from_index (outfile, ORIGINATOR,    i->originator);
      write_pr_from_index (outfile, RELEASE,       i->release);
    }
  return 1;
}
