/*
 *  Ball And Paddle
 *
 *  Copyright (C) 2007 by Eric Hutchins
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 3 of the License, or
 *  (at your option) any later version.
 *  
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *  
 *  You should have received a copy of the GNU General Public License
 *  along with this program.  If not, see <http://www.gnu.org/licenses/>.
 **/

#include "highscoremenu.h"
#include "mainmenu.h"

HighScoreMenu
  HighScoreMenu::m_instance;

void
HighScoreMenu::init ()
{
  enter = false;
  m_shift = false;
  menuItems.push_back ("Back");
  curr_levelset_index = 0;
  spacing = 40;
  string location = IMAGEPATH + "/menuback.png";
  SDL_Surface *temp = IMG_Load (location.c_str ());
  if (temp != NULL)
    imgBackground = SDL_DisplayFormat (temp);
  SDL_FreeSurface (temp);
  cursor = 0;
#ifndef SINGLEDIR
  location = string (PREFIX) + "/share/ballandpaddle/levelsets";
#endif
#ifdef SINGLEDIR
  location = string (PREFIX) + "/levelsets";
#endif
  ifstream file;
  file.open (location.c_str (), ifstream::in);
  file >> levelset_count;
  // buffer to consume rest of line
  char buffer[256];
  // consume the rest of the line with the number of levelsets
  file.getline (buffer, 256);
  for (int i = 0; i < levelset_count; i++)
    {
      char strLevelsetName[256];
      file.getline (strLevelsetName, 256);
      levelsetNames.push_back (string (strLevelsetName));
      char strLevelsetFilename[256];
      file.getline (strLevelsetFilename, 256);
      levelsetFilenames.push_back (string (strLevelsetFilename));
    }
  file.close ();
  loadHighScores ();
}

void
HighScoreMenu::cleanup ()
{
  while (!menuItems.empty ())
    menuItems.pop_back ();
  SDL_FreeSurface (imgBackground);
}

void
HighScoreMenu::loadHighScores (string filename)
{
  for (int i = 0; i < levelset_count; i++)
    {
      if (levelsetFilenames[i].compare (filename) == 0)
	{
	  curr_levelset_index = i;
	  break;
	}
    }
  loadHighScores ();
}

void
HighScoreMenu::loadHighScores ()
{
  scores = vector < int >();
  names = vector < string > ();
  ifstream file;
#ifndef SINGLEDIR
  string location =
    string (getenv ("HOME")) + "/.ballandpaddle/" +
    levelsetFilenames[curr_levelset_index] + "_scores";
#endif
#ifdef SINGLEDIR
  string location = levelsetFilenames[curr_levelset_index] + "_scores";
#endif
  file.open (location.c_str (), ifstream::in);
  if (!file.fail ())
    {
      for (int i = 0; i < 5; i++)
	{
	  int buffer;
	  file >> buffer;
	  scores.push_back (buffer);
	}
      char buffer[50];
      // get garbage data on end of line (should be just a newline character)
      file.getline (buffer, 50);
      for (int i = 0; i < 5; i++)
	{
	  char strName[20];
	  file.getline (strName, 20);
	  names.push_back (string (strName));
	}
      file.close ();
    }
  else
    {
      for (int i = 0; i < 10; i++)
	scores.push_back (0);
      for (int i = 0; i < 10; i++)
	names.push_back ("0");
    }
}

void
HighScoreMenu::saveHighScores ()
{
  if (scores.size () < 5)
    {
      int size = scores.size ();
      for (int i = 0; i < 5 - size; i++)
	scores.push_back (0);
    }
  ofstream file;
#ifndef SINGLEDIR
  string location =
    string (getenv ("HOME")) + "/.ballandpaddle/" +
    levelsetFilenames[curr_levelset_index] + "_scores";
#endif
#ifdef SINGLEDIR
  string location = levelsetFilenames[curr_levelset_index] + "_scores";
#endif
  file.open (location.c_str ());
  for (int i = 0; i < 5; i++)
    {
      file << scores[i] << endl;
    }
  for (int i = 0; i < 5; i++)
    {
      file << names[i] << endl;
    }
  file.close ();
}

void
HighScoreMenu::pause ()
{
}

void
HighScoreMenu::resume ()
{
}

void
HighScoreMenu::draw (GameStateManager * gameStateManager)
{
  int scoreIndex = gameStateManager->getScoreIndex ();

  SDL_BlitSurface (imgBackground, NULL, gameStateManager->screen, NULL);
  SDL_Rect r = { 100, 180, 440, 40 };
  SDL_FillRect (gameStateManager->screen, &r,
		SDL_MapRGB (gameStateManager->screen->format, 64, 64, 64));
  gameStateManager->drawString (FONT_ACTIVE, "<", 100, 180);
  gameStateManager->drawString (FONT_ACTIVE, ">", 524, 180);
  gameStateManager->drawStringCentered (FONT_ACTIVE,
					levelsetNames[curr_levelset_index],
					320, 180);
  SDL_Rect r2 = { 160, 220, 320, 240 };
  SDL_FillRect (gameStateManager->screen, &r2,
		SDL_MapRGB (gameStateManager->screen->format, 64, 64, 64));
  for (int i = 0; i < 5; i++)
    {
      if (enter && i == scoreIndex)
	gameStateManager->drawString (FONT_ACTIVE, names[i], 164,
				      240 + i * 40);
      else
	gameStateManager->drawString (FONT_INACTIVE, names[i], 164,
				      240 + i * 40);
      stringstream ss (stringstream::in | stringstream::out);
      ss << scores[i];
      string line = ss.str ();
      for (int j = 0; j < 8 - ss.str ().length (); j++)
	line = string (" ") + line;
      if (enter && i == scoreIndex)
	gameStateManager->drawString (FONT_ACTIVE, line, 340, 240 + i * 40);
      else
	gameStateManager->drawString (FONT_INACTIVE, line, 340, 240 + i * 40);
    }
  if (enter)
    {
      if (SDL_GetTicks () % 1000 < 500)
	{
	  r.x = 164 + names[scoreIndex].length () * 16;
	  r.y = 240 + scoreIndex * 40;
	  r.w = 1;
	  r.h = 29;
	  SDL_FillRect (gameStateManager->screen, &r,
			SDL_MapRGB (gameStateManager->screen->format, 0, 0,
				    0));
	}
    }

  for (int i = 0; i < menuItems.size (); i++)
    {
      if (cursor == i)
	gameStateManager->drawStringCentered (FONT_ACTIVE, menuItems[i], 320,
					      430 + i * spacing);
      else
	gameStateManager->drawStringCentered (FONT_INACTIVE, menuItems[i],
					      320, 430 + i * spacing);
    }
}

void
HighScoreMenu::handleEvents (GameStateManager * gameStateManager)
{
  int scoreIndex = gameStateManager->getScoreIndex ();

  int mouseX;
  int mouseY;

  SDL_Event event;

  if (enter)
    {
      while (SDL_PollEvent (&event))
	{
	  switch (event.type)
	    {
	    case SDL_QUIT:
	      gameStateManager->quit ();
	      break;
	    case SDL_KEYDOWN:
	      switch (event.key.keysym.sym)
		{
		case SDLK_LSHIFT:
		  m_shift = true;
		  break;
		case SDLK_RSHIFT:
		  m_shift = true;
		  break;
		case SDLK_RETURN:
		  gameStateManager->setEnteringHighScore (false);
		  enter = false;
		  saveHighScores ();
		  break;
		case SDLK_a:
		  names[scoreIndex] += ((m_shift) ? ('A') : ('a'));
		  break;
		case SDLK_b:
		  names[scoreIndex] += ((m_shift) ? ('B') : ('b'));
		  break;
		case SDLK_c:
		  names[scoreIndex] += ((m_shift) ? ('C') : ('c'));
		  break;
		case SDLK_d:
		  names[scoreIndex] += ((m_shift) ? ('D') : ('d'));
		  break;
		case SDLK_e:
		  names[scoreIndex] += ((m_shift) ? ('E') : ('e'));
		  break;
		case SDLK_f:
		  names[scoreIndex] += ((m_shift) ? ('F') : ('f'));
		  break;
		case SDLK_g:
		  names[scoreIndex] += ((m_shift) ? ('G') : ('g'));
		  break;
		case SDLK_h:
		  names[scoreIndex] += ((m_shift) ? ('H') : ('h'));
		  break;
		case SDLK_i:
		  names[scoreIndex] += ((m_shift) ? ('I') : ('i'));
		  break;
		case SDLK_j:
		  names[scoreIndex] += ((m_shift) ? ('J') : ('j'));
		  break;
		case SDLK_k:
		  names[scoreIndex] += ((m_shift) ? ('K') : ('k'));
		  break;
		case SDLK_l:
		  names[scoreIndex] += ((m_shift) ? ('L') : ('l'));
		  break;
		case SDLK_m:
		  names[scoreIndex] += ((m_shift) ? ('M') : ('m'));
		  break;
		case SDLK_n:
		  names[scoreIndex] += ((m_shift) ? ('N') : ('n'));
		  break;
		case SDLK_o:
		  names[scoreIndex] += ((m_shift) ? ('O') : ('o'));
		  break;
		case SDLK_p:
		  names[scoreIndex] += ((m_shift) ? ('P') : ('p'));
		  break;
		case SDLK_q:
		  names[scoreIndex] += ((m_shift) ? ('Q') : ('q'));
		  break;
		case SDLK_r:
		  names[scoreIndex] += ((m_shift) ? ('R') : ('r'));
		  break;
		case SDLK_s:
		  names[scoreIndex] += ((m_shift) ? ('S') : ('s'));
		  break;
		case SDLK_t:
		  names[scoreIndex] += ((m_shift) ? ('T') : ('t'));
		  break;
		case SDLK_u:
		  names[scoreIndex] += ((m_shift) ? ('U') : ('u'));
		  break;
		case SDLK_v:
		  names[scoreIndex] += ((m_shift) ? ('V') : ('v'));
		  break;
		case SDLK_w:
		  names[scoreIndex] += ((m_shift) ? ('W') : ('w'));
		  break;
		case SDLK_x:
		  names[scoreIndex] += ((m_shift) ? ('X') : ('x'));
		  break;
		case SDLK_y:
		  names[scoreIndex] += ((m_shift) ? ('Y') : ('y'));
		  break;
		case SDLK_z:
		  names[scoreIndex] += ((m_shift) ? ('Z') : ('z'));
		  break;
		case SDLK_SPACE:
		  names[scoreIndex] += ' ';
		  break;
		case SDLK_BACKSPACE:
		  names[scoreIndex] =
		    names[scoreIndex].substr (0,
					      names[scoreIndex].length () -
					      1);
		  break;
		}
	      break;
	    case SDL_KEYUP:
	      switch (event.key.keysym.sym)
		{
		case SDLK_LSHIFT:
		  m_shift = false;
		  break;
		case SDLK_RSHIFT:
		  m_shift = false;
		  break;
		}
	    }
	}
      return;
    }

  while (SDL_PollEvent (&event))
    {
      switch (event.type)
	{
	case SDL_QUIT:
	  gameStateManager->quit ();
	  break;
	case SDL_KEYDOWN:
	  switch (event.key.keysym.sym)
	    {
	    case SDLK_RETURN:
	      handleItem (gameStateManager);
	      break;
	    case SDLK_UP:
	      cursor--;
	      if (cursor < 0)
		cursor = menuItems.size () - 1;
	      break;
	    case SDLK_DOWN:
	      cursor++;
	      if (cursor >= menuItems.size ())
		cursor = 0;
	      break;
	    case SDLK_LEFT:
	      curr_levelset_index--;
	      if (curr_levelset_index < 0)
		curr_levelset_index = 0;
	      else
		loadHighScores ();
	      break;
	    case SDLK_RIGHT:
	      curr_levelset_index++;
	      if (curr_levelset_index >= levelset_count)
		curr_levelset_index = levelset_count - 1;
	      else
		loadHighScores ();
	      break;
	    }
	  break;
	case SDL_MOUSEMOTION:
	  mouseX = event.motion.x;
	  mouseY = event.motion.y;

	  int x1;
	  int x2;
	  for (int i = 0; i < menuItems.size (); i++)
	    {
	      x1 =
		320 -
		menuItems[i].length () *
		gameStateManager->getCharWidth (FONT_INACTIVE) / 2;
	      x2 =
		x1 +
		menuItems[i].length () *
		gameStateManager->getCharWidth (FONT_INACTIVE);
	      if (mouseX >= x1 && mouseX <= x2 && mouseY >= 250 + i * spacing
		  && mouseY <=
		  250 + gameStateManager->getCharHeight (FONT_INACTIVE) +
		  i * spacing)
		cursor = i;
	    }
	  break;
	case SDL_MOUSEBUTTONDOWN:
	  handleItem (gameStateManager);
	  break;
	}
    }

}

void
HighScoreMenu::update (GameStateManager * gameStateManager)
{
  if (gameStateManager->isEnteringHighScore ())
    enter = true;
}

void
HighScoreMenu::handleItem (GameStateManager * gameStateManager)
{
  switch (cursor)
    {
    case 0:
      gameStateManager->changeState (MainMenu::instance ());
      break;
    }
}
