/*
 This file is part of GNU Taler
 (C) 2022 Taler Systems S.A.

 GNU Taler is free software; you can redistribute it and/or modify it under the
 terms of the GNU General Public License as published by the Free Software
 Foundation; either version 3, or (at your option) any later version.

 GNU Taler is distributed in the hope that it will be useful, but WITHOUT ANY
 WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR
 A PARTICULAR PURPOSE.  See the GNU General Public License for more details.

 You should have received a copy of the GNU General Public License along with
 GNU Taler; see the file COPYING.  If not, see <http://www.gnu.org/licenses/>
 */

import { Amounts, stringifyPaytoUri } from "@gnu-taler/taler-util";
import { WalletApiOperation } from "@gnu-taler/taler-wallet-core";
import { alertFromError, useAlertContext } from "../../context/alert.js";
import { useBackendContext } from "../../context/backend.js";
import { useTranslationContext } from "@gnu-taler/web-util/browser";
import { useAsyncAsHook } from "../../hooks/useAsyncAsHook.js";
import { Props, State } from "./index.js";
import { useState } from "preact/hooks";

export function useComponentState({
  account,
  scope,
  cancel,
  onSuccess,
}: Props): State {
  const api = useBackendContext();
  const { pushAlertOnError } = useAlertContext();
  const [amount, setAmount] = useState(Amounts.zeroOfCurrency(scope.currency))
  const amountStr = Amounts.stringify(amount);

  const hook = useAsyncAsHook(async () => {
    const deposit = await api.wallet.call(WalletApiOperation.PrepareDeposit, {
      amount: amountStr,
      depositPaytoUri: stringifyPaytoUri(account),
    });
    return deposit;
  }, [amountStr]);
  // const { i18n } = useTranslationContext();

  // if (!hook) return { status: "loading", error: undefined };
  // if (hook.hasError) {
  //   return {
  //     status: "error",
  //     error: alertFromError(
  //       i18n,
  //       i18n.str`Could not load the status of deposit`,
  //       hook,
  //     ),
  //   };
  // }

  const debitAmount = !hook || hook.hasError ? Amounts.zeroOfCurrency(scope.currency) : Amounts.parseOrThrow(hook.response.effectiveDepositAmount);
  const toBeReceived = !hook || hook.hasError ? Amounts.zeroOfCurrency(scope.currency) : Amounts.parseOrThrow(hook.response.totalDepositCost);
  // const { deposit, uri, amount } = hook.response;
  async function doDeposit(): Promise<void> {
    const resp = await api.wallet.call(WalletApiOperation.CreateDepositGroup, {
      amount: amountStr,
      depositPaytoUri: stringifyPaytoUri(account),
    });
    onSuccess(resp.transactionId);
  }

  return {
    status: "ready",
    error: undefined,
    account,
    amount: {
      value: amount,
      onInput: pushAlertOnError(async (e) => setAmount(e)),
      error: Amounts.isZero(amount) ? "Can't be zero" : undefined,
    },
    confirm: {
      onClick: pushAlertOnError(doDeposit),
    },
    fee: Amounts.sub(toBeReceived, debitAmount)
      .amount,
    cost: Amounts.parseOrThrow(toBeReceived),
    effective: Amounts.parseOrThrow(debitAmount),
    cancel,
  };
}
