--
-- This file is part of TALER
-- Copyright (C) 2014--2023 Taler Systems SA
--
-- TALER is free software; you can redistribute it and/or modify it under the
-- terms of the GNU General Public License as published by the Free Software
-- Foundation; either version 3, or (at your option) any later version.
--
-- TALER is distributed in the hope that it will be useful, but WITHOUT ANY
-- WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR
-- A PARTICULAR PURPOSE.  See the GNU General Public License for more details.
--
-- You should have received a copy of the GNU General Public License along with
-- TALER; see the file COPYING.  If not, see <http://www.gnu.org/licenses/>
--

CREATE FUNCTION create_table_reserves_close(
  IN partition_suffix TEXT DEFAULT NULL
)
RETURNS VOID
LANGUAGE plpgsql
AS $$
DECLARE
  table_name TEXT default 'reserves_close';
BEGIN
  PERFORM create_partitioned_table(
    'CREATE TABLE %I'
      '(close_uuid BIGINT GENERATED BY DEFAULT AS IDENTITY'
      ',reserve_pub BYTEA NOT NULL'
      ',execution_date INT8 NOT NULL'
      ',wtid BYTEA NOT NULL CHECK (LENGTH(wtid)=32)'
      ',wire_target_h_payto BYTEA CHECK (LENGTH(wire_target_h_payto)=32)'
      ',amount taler_amount NOT NULL'
      ',closing_fee taler_amount NOT NULL'
      ',close_request_row INT8 NOT NULL DEFAULT(0)'
    ') %s ;'
    ,table_name
    ,'PARTITION BY HASH (reserve_pub)'
    ,partition_suffix
  );
  PERFORM comment_partitioned_table(
     'wire transfers executed by the reserve to close reserves'
    ,table_name
    ,partition_suffix
  );
  PERFORM comment_partitioned_column(
     'Identifies the credited bank account (and KYC status). Note that closing does not depend on KYC.'
    ,'wire_target_h_payto'
    ,table_name
    ,partition_suffix
  );
END
$$;


CREATE FUNCTION constrain_table_reserves_close(
  IN partition_suffix TEXT
)
RETURNS void
LANGUAGE plpgsql
AS $$
DECLARE
  table_name TEXT default 'reserves_close';
BEGIN
  table_name = concat_ws('_', table_name, partition_suffix);
  EXECUTE FORMAT (
    'ALTER TABLE ' || table_name ||
    ' ADD CONSTRAINT ' || table_name || '_close_uuid_pkey'
    ' PRIMARY KEY (close_uuid)'
  );
  EXECUTE FORMAT (
    'CREATE INDEX ' || table_name || '_by_reserve_pub_index '
    'ON ' || table_name || ' (reserve_pub);'
  );
END $$;


CREATE FUNCTION foreign_table_reserves_close()
RETURNS void
LANGUAGE plpgsql
AS $$
DECLARE
  table_name TEXT default 'reserves_close';
BEGIN
  EXECUTE FORMAT (
    'ALTER TABLE ' || table_name ||
    ' ADD CONSTRAINT ' || table_name || '_foreign_reserve_pub'
    ' FOREIGN KEY (reserve_pub)'
    ' REFERENCES reserves (reserve_pub) ON DELETE CASCADE'
  );
END $$;


CREATE OR REPLACE FUNCTION reserves_close_insert_trigger()
  RETURNS trigger
  LANGUAGE plpgsql
  AS $$
BEGIN
  INSERT INTO reserve_history
    (reserve_pub
    ,table_name
    ,serial_id)
  VALUES
    (NEW.reserve_pub
    ,'reserves_close'
    ,NEW.close_uuid);
  RETURN NEW;
END $$;
COMMENT ON FUNCTION reserves_close_insert_trigger()
  IS 'Automatically generate reserve history entry.';


CREATE FUNCTION master_table_reserves_close()
RETURNS VOID
LANGUAGE plpgsql
AS $$
BEGIN
  CREATE TRIGGER reserves_close_on_insert
    AFTER INSERT
     ON reserves_close
     FOR EACH ROW EXECUTE FUNCTION reserves_close_insert_trigger();
END $$;


INSERT INTO exchange_tables
    (name
    ,version
    ,action
    ,partitioned
    ,by_range)
  VALUES
    ('reserves_close'
    ,'exchange-0002'
    ,'create'
    ,TRUE
    ,FALSE),
    ('reserves_close'
    ,'exchange-0002'
    ,'constrain'
    ,TRUE
    ,FALSE),
    ('reserves_close'
    ,'exchange-0002'
    ,'foreign'
    ,TRUE
    ,FALSE),
    ('reserves_close'
    ,'exchange-0002'
    ,'master'
    ,TRUE
    ,FALSE);
