#!/bin/sh

# Copyright (C) 2000-2005 The Free Software Foundation, Inc.

# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2, or (at your option)
# any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.

#
# This program is intended to take a check.log file generated by a failed run of
# sanity.sh as input and run expr line by line on it.  It seems a much easier
# way of spotting a single failed line in a 100 line test result.
#

#
# Contributed by Derek R. Price <derek.price@openavenue.com>
#



usage ()
{
	echo "\
usage: $0 [-afh] [file...]

       -a          process alternate pattern
       -f          process first pattern (default)
       -h          print this text

     file          files to process (default = check.log)"
}

# Do a line by line match with expr
#
# INPUTS
#    $1 = text file name
#    $2 = pattern file name
expr_line_by_line ()
{
	dcl_line=0
	dcl_wrong=
	# We are assuming a newline at the end of the file.  The way sanity.sh
	# uses echo to create the log message guarantees this newline and since
	# expr ignores the last newline when the anchor is present anyhow, no
	# information is being lost in the transition
	while test $dcl_line -lt `wc -l <$1` -a $dcl_line -lt `wc -l <$2`; do
		dcl_line=`expr $dcl_line + 1`
		if test `sed -ne${dcl_line}p <$1 |wc -c` -eq 1 \
				-a `sed -ne${dcl_line}p <$2 |wc -c` -eq 1; then
			# This is a workaround for what I am calling a bug in GNU
			# expr - it won't match the empty string to the empty
			# string.  In this case the assumption is that a single
			# character is always a newline.  Since we already checked
			# for the end of the file, we know sed will echo the
			# newline.
			: 
		elif expr "`sed -ne${dcl_line}p <$1`" : \
				"`sed -ne${dcl_line}p <$2`\$" >/dev/null; then
			:
		else
			echo "$dcl_line: `sed -ne${dcl_line}p <$1`"
			echo "$dcl_line: `sed -ne${dcl_line}p <$2`\$"
			dcl_wrong="$dcl_wrong $dcl_line"
		fi
	done
	if test `wc -l <$1` -ne `wc -l <$2`; then
		echo "output & pattern contain differing number of lines"
	elif test -z "$dcl_wrong"; then
		echo "no mismatched lines"
	else
		echo "mismatched lines: $dcl_wrong"
	fi
}

# Process a single check.log file
#
# INPUTS
#    $1 = filename
process_check_log ()
{
	# abort if we can't find any expressions
	if grep '^\*\* got: $' <$1 >/dev/null; then
		:
	else
		echo "WARNING:  No expressions in file: $1" >&2
		echo "          Either not a check.log or sanity.sh exited for some other reason," >&2
		echo "          like bad exit status.  Try tail." >&2
		return
	fi

	dcl_exprfiles=""
	if grep '^\*\* or: $' <$1 >/dev/null; then
		# file contains a second regex
		if test $dcl_dofirst -eq 1; then
			# get the first pattern
			sed -ne '/^\*\* expected: $/,/^\*\* or: $/p' <$1 >/tmp/dcle$$
			dcl_exprfiles="$dcl_exprfiles /tmp/dcle$$"
		fi
		if test $dcl_doalternate -eq 1; then
			# get the alternate pattern
			sed -ne '/^\*\* or: $/,/^\*\* got: $/p' <$1 >/tmp/dclo$$
			dcl_exprfiles="$dcl_exprfiles /tmp/dclo$$"
		else
			echo "WARNING:  Ignoring alternate pattern in file: $1" >&2
		fi
	else
		# file doesn't contain a second regex
		if test $dcl_dofirst = 1; then
			# get the only pattern
			sed -ne '/^\*\* expected: $/,/^\*\* got: $/p' <$1 >/tmp/dcle$$
			dcl_exprfiles="$dcl_exprfiles /tmp/dcle$$"
		fi
		if test $dcl_doalternate -eq 1; then
			echo "WARNING:  No alternate pattern in file:  $1" >&2
		fi
	fi

	# and get the actual output
	sed -ne '/^\*\* got: $/,$p' <$1 >/tmp/dclg$$
	sed -ne '1D
$D
p' </tmp/dclg$$ >/tmp/dclh$$
	mv /tmp/dclh$$ /tmp/dclg$$

	# compare the output against each pattern requested
	for dcl_f in $dcl_exprfiles; do
		sed -ne '1D
$D
p' <$dcl_f >/tmp/dclp$$
		mv /tmp/dclp$$ $dcl_f

		case $dcl_f in
			/tmp/dcle*)
				echo "********** $1 : Primary **********"
				;;
			/tmp/dclo*)
				echo "********** $1 : Alternate **********"
				;;
		esac

		expr_line_by_line /tmp/dclg$$ $dcl_f

		rm $dcl_f
	done

	rm /tmp/dclg$$
}

###
### MAIN
###

# set up defaults
dcl_doalternate=0
dcl_dofirst=0

# process options
while getopts afh arg; do
	case $arg in
		a)
			dcl_doalternate=1
			;;
		f)
			dcl_dofirst=1
			;;
		\?|h)
			usage
			exit 1
			;;
	esac
done

# dispose of processed args
shift `expr $OPTIND - 1`
OPTIND=1

# set the default mode
if test $dcl_doalternate -eq 0; then
	dcl_dofirst=1
fi

# set default arg
if test $# -eq 0; then
	if test -f src/check.log && test -r src/check.log; then
		set src/check.log
	else
		set check.log
	fi
fi

for file in "$@"; do
	process_check_log $file;
done

exit 0
