/* -*- Mode: c++ -*- */
/*
 * Copyright 2001 Free Software Foundation, Inc.
 * 
 * This file is part of GNU Radio
 * 
 * GNU Radio is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2, or (at your option)
 * any later version.
 * 
 * GNU Radio is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with GNU Radio; see the file COPYING.  If not, write to
 * the Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 */
/*
 *  Copyright 1997 Massachusetts Institute of Technology
 * 
 *  Permission to use, copy, modify, distribute, and sell this software and its
 *  documentation for any purpose is hereby granted without fee, provided that
 *  the above copyright notice appear in all copies and that both that
 *  copyright notice and this permission notice appear in supporting
 *  documentation, and that the name of M.I.T. not be used in advertising or
 *  publicity pertaining to distribution of the software without specific,
 *  written prior permission.  M.I.T. makes no representations about the
 *  suitability of this software for any purpose.  It is provided "as is"
 *  without express or implied warranty.
 * 
 */

#ifndef _VRCYCLECOUNT_H_
#define _VRCYCLECOUNT_H_

extern "C" {
#include <stdio.h>
#ifdef THREADS
#include<pthread.h>
#endif
}

//brettv 6/24/99 I junked the old user-level perfmon stuff.
//PERMON now requires the kernel driver to operation

#ifdef PERFMON
#if defined (__i386__)
#include <libperfctr.h>
#define PERF1_CODE p6_events[P6_DCU_MISS_OUTSTANDING]
//#define PERF0_NAME "DCU miss cycles"
#define PERF2_CODE p6_events[P6_RESOURCE_STALLS]
//#define PERF1_NAME "Resource stalls"

#define PERF1_NAME PERF1_CODE.desc
#define PERF2_NAME PERF2_CODE.desc

#define START_COUNTS() start_counters(PERF1_CODE.id,PERF2_CODE.id)
#define STOP_COUNTS() perfctr_disable()
#else
//not supported
#error PERMON only supported on x86
/*alpha  #define CYCLE_COUNT() ({ volatile unsigned long time; \
	     __asm__("rpcc %0" :"=r" (time)); \
	     (unsigned long)((time >> 32) + (time & 0x00000000ffffffff)); })*/
#endif //x86

//Global counts:
//-------------

//Per module counts:
//-----------------
typedef struct {
  unsigned long long counter[3];
} pmcs;

class VrCycleCount {
private:
  long totalSamples;
  pmcs total;

#ifdef THREADS
  pthread_key_t mystart; 
  pthread_mutex_t mutex;
#else
  pmcs start;
#endif  
  //*** the rest of this stuff is Pspectra internal performance stuff
public:
  //By occurrence:
  //--------------
  //no one computed the data ahead of mine, and I can skip it
  unsigned long long skippedData; //number of units skipped

  //couldn't mark data because there wasn't enough room in the downstream
  //buffer
  unsigned long long bufferFullOnWrite; //# of occurrences

  //data is computed (ready) but other threads won't know
  //  because we're using an oversimplified linked list.
  //(i.e. out of order completion of writing threads.)
  //if both this and blockOnMarkedData are high
  //more work should be done here to fix our writeLL data
  //structure
  unsigned long long WPdelayedUpdate; //#occurrences

  //no one has marked the data ahead of mine yet so I have to wait!
  //(i.e. I couldnt' skip data)
  unsigned long long blockOnUnmarkedData; //#occurrences

  //# of times work() didn't finish all its work
  unsigned long long uncompleteWork; //#occurrences

  //By duration:
  //--------------
  //cycles are kind of bogus right now -- should use perfctr stuff
  //FIX

  //the previous thread didn't finish data I need!
  unsigned long long blockOnMarkedData; //#cycles spent waiting

  //blocked in sync() 
  unsigned long long blockOnSync; //#cycles spent waiting

public:
  /*** Performance monitoring procedures ***/
  void startCount() {

#ifdef THREADS
    pmcs *p=(pmcs *) pthread_getspecific(mystart);
    if(p == 0) {
      //init thread specific data
      p=new pmcs;
      pthread_setspecific(mystart,(void *) p);

      if (0){	// FIXME -eb
	pthread_mutex_lock(&mutex);
	if(START_COUNTS()<0) {
	  perror("VrCycleCount: could not start counts");
	  fprintf(stderr,"Must be running on a system with the /dev/perfctr driver installed.\n");
	  exit(-1);
	}
	pthread_mutex_unlock(&mutex);
      }

    } 
    else if (PARANOID && p->counter[0]!=0) {
      fprintf(stderr, "VrCycleCount(%p): counter already started! \n", this);
      abort ();
    }

#else

    pmcs *p = &start;
    if(PARANOID && start.counter[0]) {
      fprintf(stderr, "VrCycleCount(%p): counter already started! \n", this);
      abort ();
    }

#endif

    struct perfctr_state state;
    if( perfctr_read(&state) < 0 ) {
      perror("perfctr_read");
      exit(1);
    }
    p->counter[0] = state.counters.tsc.u64;
    p->counter[1] = state.counters.pmc[0].u64;
    p->counter[2] = state.counters.pmc[1].u64;
}

  void stopCount() {
#ifdef THREADS
    pmcs *p=(pmcs *) pthread_getspecific(mystart);
    if(PARANOID && (p == 0 || !p->counter[0])) {
      fprintf(stderr, "VrCycleCount(%p): counter not running! \n", this);
      exit(-1);
    }
#else
    pmcs *p = &start;
    if(PARANOID && !start.counter[0]) {
      fprintf(stderr, "VrCycleCount: counter not running! \n");
      exit(-1);
    }
#endif

    struct perfctr_state state;
    if (perfctr_read(&state) < 0 ) {
      perror("perfctr_read");
      exit(1);
    }

    MUTEX_LOCK(&mutex);

    if (PARANOID){
      if(state.counters.tsc.u64 < p->counter[0] ||
	 state.counters.pmc[0].u64 < p->counter[1] ||
	 state.counters.pmc[1].u64 < p->counter[2]) {
	fprintf(stderr, "Counters running backwards!\n");
	abort ();
      }
    }

    total.counter[0] += (state.counters.tsc.u64 - p->counter[0]);
    total.counter[1] += (state.counters.pmc[0].u64 - p->counter[1]);
    total.counter[2] += (state.counters.pmc[1].u64 - p->counter[2]);

    MUTEX_UNLOCK(&mutex);

#ifdef THREADS
    p->counter[0]=0;
#else
    start.counter[0] = 0;
#endif
  }

  void resetCount() {
    for(int i=0;i<3;i++) {
      total.counter[i]=0;
    }    
    skippedData = bufferFullOnWrite = WPdelayedUpdate =
      blockOnUnmarkedData = uncompleteWork =
      blockOnMarkedData = blockOnSync = 0;
  }

  long long getTotalCycles() {return total.counter[0];}

  long long getTotalCycles(int measure) {
    if(measure < 3)
      return total.counter[measure];
    else
      return 0;
  }

  static char *measurementName(int n) {
    switch(n) {
    case 0:
      return "cycles";
    case 1:
      return PERF1_NAME;
    case 2:
      return PERF2_NAME;
    default:
      return NULL;
    }
  }

  void updateSamples(int n) {totalSamples += n;}
  long getTotalSamples() {return totalSamples;}

  void print_stats();
  VrCycleCount();
};

#endif //PERFMON
#endif
