"""This module defines a detector for variable type.

This program is free software: you can redistribute it and/or modify it under
the terms of the GNU General Public License as published by the Free Software
Foundation, either version 3 of the License, or (at your option) any later
version.
This program is distributed in the hope that it will be useful, but WITHOUT
ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
FOR A PARTICULAR PURPOSE. See the GNU General Public License for more details.
You should have received a copy of the GNU General Public License along with
this program. If not, see <http://www.gnu.org/licenses/>.
"""
import numpy as np
import copy
from scipy.stats import kstest, ks_2samp, norm, multinomial
from scipy.stats import chisquare  # Only needed if the chisquare test is used (self.used_multinomial_test == 'Chi')
import os
import logging
import sys

from aminer import AMinerConfig
from aminer.AMinerConfig import STAT_LEVEL, STAT_LOG_NAME
from aminer.AnalysisChild import AnalysisContext
from aminer.input import AtomHandlerInterface
from aminer.util import TimeTriggeredComponentInterface
from aminer.util import PersistenceUtil
from aminer.analysis import CONFIG_KEY_LOG_LINE_PREFIX


class VariableTypeDetector(AtomHandlerInterface, TimeTriggeredComponentInterface):
    """
    This class tests each variable of the event_types for the implemented variable types.
    This module needs to run after the EventTypeDetector is initialized
    """

    def __init__(self, aminer_config, anomaly_event_handlers, event_type_detector, persistence_id='Default', path_list=None, ks_alpha=0.05,
                 s_ks_alpha=0.05, s_ks_bt_alpha=0.05, d_alpha=0.1, d_bt_alpha=0.1, div_thres=0.3, sim_thres=0.1, indicator_thres=0.4,
                 num_init=100, num_update=50, num_update_unq=200, num_s_ks_values=50, num_s_ks_bt=30, num_d_bt=30, num_pause_discrete=5,
                 num_pause_others=2, test_ks_int=True, update_var_type_bool=True, num_stop_update=False,
                 silence_output_without_confidence=False, silence_output_except_indicator=True, num_var_type_hist_ref=10,
                 num_update_var_type_hist_ref=10,  num_var_type_considered_ind=10, num_stat_stop_update=200,
                 num_updates_until_var_reduction=20, var_reduction_thres=0.6, num_skipped_ind_for_weights=1, num_ind_for_weights=100,
                 used_multinomial_test='Chi', use_empiric_distr=True, save_statistics=True, output_log_line=True, ignore_list=None,
                 constraint_list=None):
        """Initialize the detector. This will also trigger reading or creation of persistence storage location."""
        self.next_persist_time = None
        self.anomaly_event_handlers = anomaly_event_handlers
        self.aminer_config = aminer_config

        # General options
        # Used to track the indicators and changed variable types
        self.save_statistics = save_statistics
        # States if empiric distributions of the variables should be used if no continuous distribution is detected
        self.use_empiric_distr = use_empiric_distr

        # Input parameters
        # significance niveau for p-value for the distribution test of the initialization
        # Recomended values are the implemented values of crit_val_ini and crit_val_upd
        self.ks_alpha = ks_alpha
        # significance niveau for p-value for the sliding KS-test in the update step
        # Recomended values are the implemented values of crit_val_upd
        self.s_ks_alpha = s_ks_alpha
        # significance niveau for the binomialtest of the testresults of the sKS-test
        self.s_ks_bt_alpha = s_ks_bt_alpha
        # significance niveau for the binomialtest of the single discrete variables
        # If used_multinomial_test == 'Approx' then faster runtime for values in the p list of bt_min_succ_data
        self.d_alpha = d_alpha
        # significance niveau for the binomialtest of the testresults of the discrete tests
        self.d_bt_alpha = d_bt_alpha
        # threshold for diversity of the values of a variable (the higher the more values have to be distinct to be considered to be
        # continuous distributed)
        self.div_thres = div_thres
        # threshold for similarity of the values of a variable (the higher the more values have to be common to be considered discrete)
        self.sim_thres = sim_thres
        # threshold for the variable indicators to be used in the event indicator
        self.indicator_thres = indicator_thres
        # Number of read in lines before detecting the variable types
        # Recomended values are the implemented values of crit_val_ini and crit_val_upd
        self.num_init = num_init
        # Number of values for which the variableType is updated
        # If used_multinomial_test == 'Approx' then faster runtime for values in the p list of bt_min_succ_data
        self.num_update = num_update
        # Number of values for which the values of type unq is unique (the last num_update + num_update_unq values are unique)
        self.num_update_unq = num_update_unq
        # Number of values which are tested in the sKS-test. The value has to be <= num_init, >= num_update.
        # Recomended values are the implemented values of crit_val_upd
        self.num_s_ks_values = num_s_ks_values
        # Number of tested sKS-Tests for the binomialtest of the testresults of the sKS tests
        self.num_s_ks_bt = num_s_ks_bt
        # Number of tested discrete samples for the binomialtest of the testresults of the discrete tests
        self.num_d_bt = num_d_bt
        # Number of paused updates, before the discrete var type is adapted
        self.num_pause_discrete = num_pause_discrete
        # Number of paused update runs, before trying to find a new var_type
        self.num_pause_others = num_pause_others
        # States if integer number should be tested for the continuous variable type
        self.test_ks_int = test_ks_int
        # States, if the the found variable types are updated if a test fails
        self.update_var_type_bool = update_var_type_bool
        # Stops updating the found variableTypes after num_stop_update processed lines. If False the updating of lines will not be stopped
        self.num_stop_update = num_stop_update
        # Silences the all messages without a confidence-entry
        self.silence_output_without_confidence = silence_output_without_confidence
        # Silences the all messages which are not related with the calculated indicator
        self.silence_output_except_indicator = silence_output_except_indicator
        self.output_log_line = output_log_line
        # States how long the reference for the varTypeHist is. The reference is used in the evaluation.
        self.num_var_type_hist_ref = num_var_type_hist_ref
        # Number of updatesteps before the varTypeHistRef is being updated
        self.num_update_var_type_hist_ref = num_update_var_type_hist_ref
        # This attribute states how many variable types of the history are used as the recent history in the calculation of the
        # indicator. False if no output of the indicator should be generated
        self.num_var_type_considered_ind = num_var_type_considered_ind
        # Number of static values of a variable, to stop tracking the variable type and read in in eventTypeD. False if not wanted.
        self.num_stat_stop_update = num_stat_stop_update
        # Number of updatesteps until the variables are tested if they are suitable for an indicator. If not suitable, they are removed
        # from the tracking of EvTypeD (reduce checked variables). Equals False if disabled.
        self.num_updates_until_var_reduction = num_updates_until_var_reduction
        # Threshold for the reduction of variable types. The most likely none others var type must have a higher relative appearance
        # for the variable to be further checked.
        self.var_reduction_thres = var_reduction_thres
        # Number of the skipped indicators for the calculation of the indicator weights
        self.num_skipped_ind_for_weights = num_skipped_ind_for_weights
        # Number of indicators used in the calculation of the indicator weights
        self.num_ind_for_weights = num_ind_for_weights
        # States the used multinomial test. the value can be of the list ['MT', 'Approx', 'Chi'], where 'MT' means original MT, 'Approx'
        # is the approximation with single BTs and 'Chi' is the ChisquareTest
        self.used_multinomial_test = used_multinomial_test
        # List of paths, which variables are being tested for a type. All other paths will not get a type assigned. If None all paths are
        # being tested.
        self.path_list = path_list

        # Initialization of variables, which are no input parameters
        # Saves the minimal number of successes for the BT for the sKS-test
        self.s_ks_bt_min_success = self.bt_min_successes(self.num_s_ks_bt, 1 - self.s_ks_alpha, self.s_ks_bt_alpha)
        # Saves the minimal number of successes for the BT for discrete values
        self.d_bt_min_success = self.bt_min_successes(self.num_d_bt, 1 - self.d_alpha, self.d_bt_alpha)
        # Number of eventTypes
        self.num_events = 0
        # event_type_detector. Used to get the eventNumbers and values of the variables, etc.
        self.event_type_detector = event_type_detector
        # Add the varTypeDetector to the list of the modules, which use the event_type_detector.
        self.event_type_detector.add_following_modules(self)
        if self.event_type_detector.min_num_vals < max(num_init, num_update):
            msg = 'Changed the parameter min_num_vals of the ETD from %s to %s to prevent errors in the execution of the VTD' % (
                    self.event_type_detector.min_num_vals, max(num_init, num_update))
            logging.getLogger(AMinerConfig.DEBUG_LOG_NAME).warning(msg)
            print('WARNING: ' + msg, file=sys.stderr)
            self.event_type_detector.min_num_vals = max(num_init, num_update)
        if self.event_type_detector.max_num_vals <= max(num_init, num_update) + 500:
            msg = 'Changed the parameter max_num_vals of the ETD from %s to %s to prevent errors in the execution of the VTD' % (
                    self.event_type_detector.max_num_vals, max(num_init, num_update) + 500)
            logging.getLogger(AMinerConfig.DEBUG_LOG_NAME).warning(msg)
            print('WARNING: ' + msg, file=sys.stderr)
            self.event_type_detector.max_num_vals = max(num_init, num_update) + 500
        # List of the numbers of variables of the eventTypes
        self.length = []
        # Used to keep track of the indices of the variables if the path_list is not empty
        self.variable_path_num = []
        # List of the found vartypes
        self.var_type = []
        # Stores the alternative distribution types of continuous variables
        self.alternative_distribution_types = []
        # Stores the values the betam and special distributions. The values are needed in the sKS test
        self.distr_val = []
        # List of the successes of the binomialtest for the rejection in the sKS or variables of discrete type
        self.bt_results = []
        # List of the history of varTypes of the single variables. The lists to the variables take the form
        # [others, static, [discrete, number of appended steps], asc, desc, unique, EV of continuous distributions]
        self.var_type_history_list = []
        # Reference of a var_type_history_list. Used in the calculation of the indicator.
        self.var_type_history_list_reference = []
        # Order of the var_type_history_list [others, static, [discrete, number of appended steps], asc, desc, unique, EV of continuous
        # distributions]
        self.var_type_history_list_order = ['others', 'stat', 'd', 'asc', 'desc', 'unq', 'contEV']
        # List of the distributions for which the sKS test is implemented
        self.distr_list = ['nor', 'uni', 'spec', 'beta', 'betam', 'emp']
        # List of the numbers of log lines of this eventType, when an indicator failed
        self.failed_indicators = []
        # Stores the standardised values of all tested distributions for better performance. The list is hardcoded below
        self.distribution_data = {}
        # Stores the number of minimal successes for the BT for selected samplesize and probabilities.
        self.bt_min_succ_data = {}

        self.log_success = 0
        self.log_total = 0
        self.log_new_learned = 0
        self.log_new_learned_values = []
        self.log_updated = 0

        # Initialize lists used for the tracking of the indicator
        if self.save_statistics:
            self.statistics_file_name = AMinerConfig.build_persistence_file_name(aminer_config, self.__class__.__name__, 'statistics')
            # List of the numbers of total parsed log lines, when an indicator failed. Only used for the statistics
            self.failed_indicators_total = []
            # List of the confidences of the indicators
            self.failed_indicators_values = []
            # List of the paths of the indicators
            self.failed_indicators_paths = []
            # List of the numbers of total parsed log lines, when a variable changed its type. Only used for the statistics
            self.changed_var_types = []

        # Stores the standardised values of all tested distributions for better performance.
        self.distribution_data = {
            'spec': np.array([-1.8273443302401238,-1.7593797798616286,-1.709951015949996,-1.6790580385052258,-1.6543436565494094,-1.6296292745935932,-1.6110934881267307,-1.5925577016598689,-1.5740219151930066,-1.561664724215098,-1.5493075332371908,-1.5369503422592823,-1.5245931512813744,-1.512235960303466,-1.5060573648145121,-1.4937001738366036,-1.4875215783476496,-1.4751643873697418,-1.4689857918807874,-1.4628071963918334,-1.4566286009028795,-1.444271409924971,-1.438092814436017,-1.431914218947063,-1.4257356234581091,-1.4195570279691547,-1.4133784324802008,-1.4071998369912468,-1.401021241502293,-1.3948426460133383,-1.3886640505243844,-1.3824854550354304,-1.3763068595464765,-1.373217561801999,-1.370128264057522,-1.363949668568568,-1.357771073079614,-1.35159247759066,-1.3485031798461826,-1.3454138821017056,-1.3392352866127517,-1.3330566911237978,-1.3299673933793208,-1.3268780956348438,-1.3206995001458892,-1.3176102024014122,-1.3145209046569353,-1.3083423091679813,-1.3052530114235044,-1.3021637136790274,-1.295985118190073,-1.292895820445596,-1.289806522701119,-1.2836279272121651,-1.2805386294676881,-1.2774493317232112,-1.2743600339787335,-1.2712707362342566,-1.2650921407453026,-1.2620028430008257,-1.2589135452563487,-1.2558242475118717,-1.2527349497673947,-1.2496456520229173,-1.2465563542784404,-1.2434670565339634,-1.2403777587894864,-1.2341991633005323,-1.2311098655560553,-1.2280205678115783,-1.2249312700671011,-1.2218419723226241,-1.2187526745781472,-1.2156633768336702,-1.2125740790891932,-1.209484781344716,-1.206395483600239,-1.203306185855762,-1.2002168881112847,-1.1971275903668077,-1.1940382926223307,-1.1909489948778538,-1.1878596971333768,-1.1847703993888996,-1.1827108675592481,-1.1806513357295971,-1.1785918038999457,-1.175502506155469,-1.172413208410992,-1.169323910666515,-1.1662346129220378,-1.1631453151775606,-1.1600560174330836,-1.1569667196886066,-1.1538774219441297,-1.1518178901144782,-1.1497583582848272,-1.1476988264551757,-1.1446095287106988,-1.1415202309662216,-1.1384309332217442,-1.1353416354772672,-1.133282103647616,-1.1312225718179647,-1.1291630399883132,-1.1260737422438363,-1.1229844444993593,-1.120924912669708,-1.1188653808400566,-1.1168058490104051,-1.113716551265928,-1.110627253521451,-1.107537955776974,-1.104448658032497,-1.1023891262028456,-1.1003295943731946,-1.098270062543543,-1.0951807647990661,-1.092091467054589,-1.0900319352249375,-1.087972403395286,-1.0859128715656345,-1.0828235738211576,-1.0797342760766806,-1.0776747442470294,-1.0756152124173781,-1.0735556805877267,-1.0704663828432497,-1.0673770850987725,-1.065317553269121,-1.0632580214394698,-1.0611984896098183,-1.0591389577801669,-1.0570794259505158,-1.0550198941208644,-1.0519305963763874,-1.0488412986319104,-1.046781766802259,-1.0447222349726075,-1.0426627031429563,-1.0395734053984789,-1.036484107654002,-1.0344245758243507,-1.0323650439946994,-1.030305512165048,-1.0282459803353967,-1.0261864485057455,-1.024126916676094,-1.021037618931617,-1.0179483211871398,-1.0158887893574884,-1.0138292575278371,-1.0117697256981857,-1.0097101938685342,-1.0076506620388832,-1.0055911302092317,-1.0025018324647548,-0.9994125347202778,-0.9973530028906263,-0.995293471060975,-0.9932339392313235,-0.9911744074016722,-0.9891148755720207,-0.9870553437423694,-0.9839660459978924,-0.9808767482534154,-0.978817216423764,-0.9767576845941128,-0.9746981527644615,-0.97263862093481,-0.9705790891051587,-0.9685195572755072,-0.96543025953103,-0.962340961786553,-0.9602814299569016,-0.9582218981272504,-0.9561623662975991,-0.9541028344679476,-0.9520433026382965,-0.9499837708086452,-0.9468944730641682,-0.9438051753196909,-0.9417456434900395,-0.9396861116603881,-0.9376265798307367,-0.9355670480010853,-0.9335075161714341,-0.9314479843417828,-0.9283586865973058,-0.9252693888528288,-0.9232098570231774,-0.921150325193526,-0.9190907933638746,-0.9170312615342232,-0.9149717297045717,-0.9129121978749204,-0.9098229001304434,-0.9067336023859665,-0.904674070556315,-0.9026145387266639,-0.9005550068970124,-0.898495475067361,-0.8964359432377096,-0.8943764114080582,-0.891287113663581,-0.888197815919104,-0.8861382840894528,-0.8840787522598015,-0.8820192204301504,-0.878929922685673,-0.8758406249411961,-0.8737810931115447,-0.8717215612818936,-0.8696620294522421,-0.8676024976225908,-0.8655429657929393,-0.863483433963288,-0.860394136218811,-0.857304838474334,-0.8552453066446826,-0.8531857748150312,-0.8511262429853798,-0.8490667111557284,-0.8470071793260773,-0.8449476474964258,-0.8418583497519488,-0.8387690520074716,-0.8367095201778202,-0.8346499883481691,-0.8325904565185177,-0.8305309246888662,-0.8284713928592149,-0.8264118610295634,-0.8233225632850865,-0.8202332655406095,-0.8181737337109581,-0.8161142018813067,-0.8140546700516553,-0.8119951382220039,-0.8099356063923527,-0.8078760745627014,-0.8047867768182241,-0.8016974790737471,-0.7996379472440958,-0.7975784154144446,-0.7955188835847932,-0.7934593517551418,-0.7913998199254904,-0.789340288095839,-0.786250990351362,-0.7831616926068851,-0.7811021607772336,-0.7790426289475822,-0.7769830971179308,-0.7749235652882794,-0.7728640334586283,-0.7708045016289768,-0.7677152038844999,-0.7646259061400227,-0.7625663743103712,-0.7605068424807201,-0.7584473106510687,-0.7553580129065914,-0.7522687151621145,-0.7502091833324631,-0.748149651502812,-0.7460901196731605,-0.7440305878435092,-0.7419710560138577,-0.7399115241842064,-0.7368222264397294,-0.7337329286952524,-0.731673396865601,-0.7296138650359496,-0.7275543332062981,-0.7254948013766468,-0.7234352695469957,-0.7213757377173442,-0.7182864399728672,-0.71519714222839,-0.7131376103987386,-0.7110780785690874,-0.7090185467394361,-0.7069590149097846,-0.7048994830801333,-0.7028399512504818,-0.6997506535060048,-0.6966613557615279,-0.6946018239318765,-0.6925422921022254,-0.6904827602725739,-0.687393462528097,-0.6843041647836198,-0.6822446329539683,-0.6801851011243172,-0.6781255692946658,-0.6760660374650144,-0.674006505635363,-0.6719469738057116,-0.6688576760612346,-0.6657683783167576,-0.6637088464871063,-0.6616493146574548,-0.6595897828278035,-0.6565004850833265,-0.6534111873388495,-0.651351655509198,-0.6492921236795467,-0.6472325918498952,-0.645173060020244,-0.6431135281905925,-0.6410539963609413,-0.6379646986164642,-0.6348754008719871,-0.6328158690423358,-0.6307563372126844,-0.6286968053830332,-0.625607507638556,-0.6225182098940789,-0.6204586780644277,-0.6183991462347762,-0.616339614405125,-0.6142800825754736,-0.6122205507458222,-0.6101610189161708,-0.6070717211716938,-0.6039824234272169,-0.6019228915975654,-0.599863359767914,-0.5978038279382626,-0.5947145301937856,-0.5916252324493086,-0.5895657006196572,-0.5875061687900058,-0.5854466369603544,-0.5833871051307031,-0.5813275733010518,-0.5792680414714004,-0.5761787437269233,-0.5730894459824463,-0.571029914152795,-0.5689703823231436,-0.5669108504934923,-0.5638215527490152,-0.560732255004538,-0.5586727231748868,-0.5566131913452355,-0.5545536595155841,-0.551464361771107,-0.5483750640266299,-0.5463155321969787,-0.5442560003673272,-0.542196468537676,-0.5401369367080245,-0.5380774048783732,-0.5360178730487217,-0.5329285753042448,-0.5298392775597678,-0.5277797457301164,-0.525720213900465,-0.5236606820708136,-0.5205713843263367,-0.5174820865818597,-0.5154225547522082,-0.5133630229225569,-0.5113034910929054,-0.5082141933484284,-0.5051248956039515,-0.5030653637743001,-0.5010058319446486,-0.4989463001149973,-0.49585700237052033,-0.49276770462604347,-0.49070817279639206,-0.4886486409667408,-0.4865891091370894,-0.48349981139261244,-0.4804105136481353,-0.4783509818184839,-0.47629144998883266,-0.47423191815918125,-0.47217238632952985,-0.47011285449987855,-0.46805332267022715,-0.46496402492575006,-0.4618747271812731,-0.4598151953516217,-0.4577556635219704,-0.455696131692319,-0.4526068339478419,-0.44951753620336493,-0.4474580043737135,-0.44539847254406223,-0.4433389407144108,-0.4402496429699337,-0.4371603452254567,-0.4351008133958053,-0.43304128156615407,-0.43098174973650266,-0.4278924519920255,-0.42480315424754855,-0.42274362241789715,-0.4206840905882459,-0.4186245587585945,-0.41553526101411736,-0.4124459632696404,-0.410386431439989,-0.40832689961033775,-0.40626736778068634,-0.4031780700362092,-0.40008877229173223,-0.3980292404620808,-0.3959697086324296,-0.3939101768027782,-0.39082087905830104,-0.38773158131382407,-0.38567204948417266,-0.3836125176545214,-0.38155298582487,-0.3784636880803929,-0.3753743903359159,-0.37228509259143894,-0.36919579484696186,-0.36713626301731045,-0.36507673118765915,-0.36301719935800775,-0.3599279016135308,-0.3568386038690537,-0.3547790720394023,-0.352719540209751,-0.3506600083800996,-0.3475707106356226,-0.34448141289114553,-0.3424218810614941,-0.34036234923184283,-0.3383028174021914,-0.33521351965771445,-0.33212422191323737,-0.33006469008358597,-0.32800515825393467,-0.32594562642428326,-0.3228563286798063,-0.3197670309353292,-0.3177074991056778,-0.3156479672760265,-0.3135884354463751,-0.31049913770189813,-0.30740983995742105,-0.3043205422129441,-0.3012312444684671,-0.2991717126388157,-0.2971121808091644,-0.295052648979513,-0.29196335123503603,-0.2888740534905589,-0.2868145216609075,-0.28475498983125624,-0.28269545800160484,-0.27960616025712787,-0.27651686251265073,-0.27342756476817365,-0.2703382670236967,-0.26827873519404527,-0.266219203364394,-0.26415967153474257,-0.2610703737902655,-0.2579810760457885,-0.2559215442161371,-0.2538620123864858,-0.2518024805568344,-0.24871318281235733,-0.24562388506788035,-0.24253458732340338,-0.23944528957892627,-0.23738575774927487,-0.2353262259196236,-0.2332666940899722,-0.23017739634549522,-0.2270880986010181,-0.223998800856541,-0.22090950311206403,-0.21884997128241263,-0.21679043945276136,-0.21473090762310995,-0.21164160987863284,-0.20855231213415587,-0.20649278030450446,-0.20443324847485317,-0.20237371664520176,-0.19928441890072465,-0.19619512115624768,-0.1931058234117707,-0.1900165256672936,-0.1879569938376422,-0.18589746200799093,-0.18383793017833952,-0.18074863243386255,-0.17765933468938544,-0.17457003694490833,-0.17148073920043136,-0.16942120737077995,-0.16736167554112868,-0.16530214371147728,-0.1622128459670003,-0.15912354822252334,-0.15603425047804623,-0.15294495273356926,-0.15088542090391785,-0.14882588907426658,-0.14676635724461518,-0.14367705950013807,-0.1405877617556611,-0.13749846401118412,-0.13440916626670701,-0.1313198685222299,-0.12823057077775293,-0.12617103894810153,-0.12411150711845026,-0.12205197528879885,-0.11896267754432174,-0.11587337979984477,-0.1127840820553678,-0.10969478431089069,-0.10763525248123929,-0.10557572065158802,-0.10351618882193661,-0.10042689107745964,-0.09733759333298253,-0.09424829558850542,-0.09115899784402844,-0.08806970009955147,-0.08498040235507436,-0.08292087052542295,-0.08086133869577168,-0.07880180686612027,-0.0757125091216433,-0.0726232113771662,-0.06953391363268908,-0.06644461588821211,-0.06335531814373514,-0.06026602039925803,-0.058206488569606626,-0.05614695673995536,-0.05408742491030395,-0.05099812716582691,-0.04790882942134987,-0.04481953167687283,-0.04173023393239579,-0.03864093618791875,-0.03555163844344171,-0.03246234069896467,-0.029373042954487626,-0.02731351112483629,-0.025253979295184883,-0.023194447465533546,-0.020105149721056502,-0.01701585197657946,-0.013926554232102421,-0.010837256487625381,-0.00774795874314834,-0.0046586609986712995,-0.0015693632541942586,0.0015199344902827824,0.00357946631993412,0.005638998149585526,0.007698529979236864,0.010787827723713905,0.013877125468190945,0.016966423212667985,0.020055720957145026,0.02314501870162207,0.02623431644609911,0.02932361419057615,0.03241291193505319,0.03550220967953016,0.03859150742400713,0.04168080516848417,0.04477010291296122,0.04682963474261256,0.04888916657226396,0.0509486984019153,0.05403799614639234,0.05712729389086938,0.06021659163534642,0.06330588937982347,0.06639518712430051,0.06948448486877755,0.07257378261325459,0.07566308035773163,0.07875237810220867,0.08184167584668571,0.08493097359116275,0.08802027133563979,0.09110956908011683,0.09419886682459387,0.09625839865424521,0.09831793048389662,0.10037746231354795,0.10346676005802499,0.10655605780250203,0.10964535554697907,0.11273465329145611,0.11582395103593315,0.1189132487804102,0.12200254652488723,0.1250918442693643,0.12818114201384126,0.13127043975831829,0.13435973750279534,0.13744903524727237,0.14053833299174942,0.14362763073622645,0.1467169284807035,0.14980622622518053,0.15289552396965758,0.1559848217141346,0.15907411945861166,0.1621634172030887,0.16525271494756574,0.16834201269204277,0.17143131043651982,0.17452060818099685,0.1776099059254739,0.18069920366995093,0.18378850141442798,0.18687779915890504,0.18996709690338207,0.19305639464785912,0.19614569239233615,0.1992349901368132,0.20232428788129023,0.2054135856257672,0.20850288337024425,0.21159218111472128,0.21468147885919833,0.21777077660367536,0.2208600743481524,0.22394937209262944,0.2270386698371065,0.23012796758158352,0.23321726532606057,0.2363065630705376,0.23939586081501465,0.24248515855949168,0.24557445630396874,0.24866375404844576,0.2517530517929228,0.25484234953739987,0.2579316472818769,0.2610209450263539,0.264110242770831,0.26719954051530803,0.27028883825978506,0.2733781360042621,0.27646743374873917,0.2795567314932162,0.2826460292376932,0.28573532698217025,0.29191392247112435,0.2950032202156014,0.2980925179600784,0.30118181570455543,0.30427111344903246,0.3073604111935095,0.3104497089379865,0.31353900668246354,0.3166283044269406,0.31971760217141765,0.3228068999158947,0.3258961976603717,0.3289854954048488,0.3320747931493258,0.33516409089380284,0.34134268638275694,0.34443198412723397,0.34752128187171094,0.3506105796161881,0.35369987736066505,0.35678917510514213,0.3598784728496191,0.36296777059409624,0.3660570683385732,0.3691463660830503,0.37223566382752726,0.37841425931648137,0.3815035570609584,0.3845928548054354,0.38768215254991245,0.39077145029438953,0.39386074803886656,0.3969500457833436,0.4000393435278206,0.4031286412722977,0.40930723676125175,0.4123965345057288,0.41548583225020586,0.4185751299946829,0.4216644277391599,0.424753725483637,0.427843023228114,0.43402161871706807,0.43711091646154504,0.4402002142060222,0.44328951195049915,0.44637880969497623,0.4494681074394532,0.45255740518393034,0.45873600067288434,0.46182529841736136,0.46491459616183844,0.46800389390631547,0.4710931916507925,0.4741824893952695,0.4772717871397466,0.48345038262870066,0.4865396803731777,0.48962897811765477,0.4927182758621318,0.4958075736066088,0.5019861690955629,0.50507546684004,0.508164764584517,0.5112540623289941,0.514343360073471,0.5205219555624251,0.5236112533069022,0.5267005510513791,0.5297898487958562,0.5328791465403333,0.5390577420292874,0.5421470397737643,0.5452363375182414,0.5483256352627185,0.5514149330071955,0.5575935284961496,0.5606828262406266,0.5637721239851037,0.5668614217295807,0.5699507194740577,0.5761293149630118,0.5792186127074889,0.582307910451966,0.58848650594092,0.5915758036853971,0.594665101429874,0.5977543991743511,0.6008436969188281,0.6070222924077823,0.6101115901522592,0.6132008878967362,0.6193794833856904,0.6224687811301673,0.6255580788746444,0.6317366743635986,0.6348259721080756,0.6379152698525525,0.6440938653415067,0.6471831630859837,0.6502724608304606,0.6533617585749378,0.6564510563194148,0.6626296518083689,0.6657189495528459,0.6688082472973229,0.674986842786277,0.6780761405307542,0.6811654382752311,0.6873440337641852,0.6904333315086623,0.6935226292531392,0.6997012247420933,0.7058798202310475,0.7089691179755245,0.7120584157200015,0.7182370112089556,0.7213263089534326,0.7244156066979096,0.7305942021868638,0.7336834999313409,0.7367727976758178,0.7429513931647719,0.746040690909249,0.7491299886537259,0.7553085841426801,0.7614871796316341,0.7645764773761111,0.7676657751205882,0.7738443706095423,0.7769336683540192,0.7800229660984964,0.7862015615874505,0.7923801570764044,0.7954694548208815,0.7985587525653586,0.8047373480543126,0.8078266457987897,0.8109159435432668,0.8170945390322207,0.823273134521175,0.826362432265652,0.829451730010129,0.835630325499083,0.8418089209880371,0.844898218732514,0.8479875164769911,0.8541661119659453,0.8603447074548993,0.8634340051993764,0.8665233029438534,0.8727018984328074,0.8788804939217616,0.8850590894107156,0.8881483871551927,0.8912376848996697,0.8974162803886238,0.9035948758775779,0.9066841736220549,0.909773471366532,0.915952066855486,0.9221306623444401,0.9283092578333942,0.9313985555778712,0.9344878533223482,0.9406664488113023,0.9468450443002563,0.9530236397892106,0.9561129375336875,0.9592022352781645,0.9653808307671187,0.9715594262560726,0.9777380217450269,0.9839166172339808,0.9870059149784579,0.990095212722935,0.9962738082118889,1.002452403700843,1.008630999189797,1.0148095946787512,1.0209881901677054,1.0271667856566593,1.0302560834011365,1.0333453811456135,1.0395239766345676,1.0457025721235216,1.0518811676124757,1.0580597631014297,1.0642383585903838,1.070416954079338,1.076595549568292,1.0796848473127691,1.0827741450572461,1.0889527405462003,1.0951313360351542,1.1013099315241082,1.1074885270130623,1.1136671225020165,1.1198457179909704,1.1260243134799246,1.1322029089688785,1.1383815044578327,1.1445600999467869,1.1507386954357408,1.156917290924695,1.1630958864136491,1.169274481902603,1.1754530773915572,1.1816316728805112,1.1878102683694653,1.1939888638584195,1.2001674593473735,1.2063460548363276,1.2125246503252818,1.2187032458142357,1.2248818413031899,1.2310604367921438,1.237239032281098,1.243417627770052,1.249596223259006,1.2557748187479605,1.2619534142369144,1.2681320097258684,1.2743106052148225,1.2804892007037767,1.2866677961927306,1.2990249871706385,1.3052035826595927,1.311382178148547,1.317560773637501,1.323739369126455,1.3299179646154091,1.3360965601043633,1.3484537510822712,1.3546323465712253,1.3608109420601797,1.3669895375491337,1.3731681330380876,1.3793467285270418,1.3917039195049499,1.3978825149939038,1.404061110482858,1.4102397059718124,1.4225968969497202,1.4287754924386744,1.4349540879276284,1.4411326834165825,1.4534898743944904,1.4596684698834448,1.465847065372399,1.4782042563503068,1.484382851839261,1.4905614473282152,1.502918638306123,1.5090972337950774,1.5152758292840316,1.5276330202619395,1.5338116157508936,1.5461688067288017,1.5523474022177557,1.5647045931956642,1.5708831886846182,1.5832403796625263,1.5894189751514805,1.6017761661293883,1.6079547616183427,1.6203119525962508,1.6264905480852048,1.6388477390631127,1.645026334552067,1.657383525529975,1.6697407165078835,1.6759193119968374,1.6882765029747453,1.6944550984636997,1.7068122894416076,1.7191694804195161,1.731526671397424,1.737705266886378,1.7500624578642865,1.7624196488421944,1.7747768398201027,1.7871340307980106,1.7994912217759191,1.805669817264873,1.8180270082427814,1.8303841992206893,1.8427413901985972,1.8550985811765057,1.8674557721544136,1.879812963132322,1.8983487495991844,1.9107059405770923,1.9230631315550006,1.9354203225329085,1.947777513510817,1.966313299977679,1.9786704909555872,1.991027681933495,2.0095634684003576,2.0219206593782655,2.0404564458451278,2.052813636823036,2.071349423289898,2.0898852097567606,2.102242400734669,2.1207781872015308,2.139313973668393,2.1578497601352553,2.176385546602118,2.19492133306898,2.213457119535842,2.2319929060027044,2.2567072879585206,2.2752430744253833,2.293778860892245,2.3184932428480614,2.3432076248038776,2.3679220067596938,2.3926363887155104,2.4173507706713266,2.442065152627143,2.4729581300719135,2.4976725120277297,2.5285654894725,2.55945846691727,2.590351444362041,2.627423017295765,2.66449459022949,2.701566163163214,2.744816331585892,2.788066500008571,2.8374952639202036,2.8869240278318364,2.942531387232423,3.004317342121964,3.072281892500459,3.1464250383679073,3.2391039707022187,3.344140094014438,3.473890599282474,3.646891272973188]),  # skipcq: FLK-E231, FLK-E501
            'betam1': np.array([0.0,0.00025,0.0005,0.00075,0.001,0.0012000000000000001,0.0014,0.0015999999999999999,0.0018,0.002,0.0022,0.0024000000000000002,0.0026,0.0028,0.003,0.0032,0.0034000000000000002,0.0036,0.0038,0.004,0.0042,0.0044,0.0046,0.0048000000000000004,0.005,0.0052,0.0054,0.0056,0.0058000000000000005,0.006,0.0062,0.0064,0.0066,0.0068000000000000005,0.007,0.0072,0.0074,0.0076,0.0078000000000000005,0.008,0.0082,0.0084,0.0086,0.008799999999999999,0.009,0.00925,0.0095,0.00975,0.01,0.0102,0.0104,0.0106,0.010799999999999999,0.011,0.0112,0.0114,0.0116,0.0118,0.012,0.0122,0.0124,0.0126,0.012799999999999999,0.013,0.0132,0.0134,0.0136,0.0138,0.014,0.01425,0.014499999999999999,0.01475,0.015,0.0152,0.0154,0.0156,0.0158,0.016,0.0162,0.0164,0.0166,0.016800000000000002,0.017,0.01725,0.0175,0.01775,0.018,0.018199999999999997,0.0184,0.0186,0.0188,0.019,0.0192,0.0194,0.0196,0.0198,0.02,0.02025,0.0205,0.02075,0.021,0.0212,0.0214,0.0216,0.0218,0.022,0.02225,0.0225,0.02275,0.023,0.0232,0.0234,0.0236,0.0238,0.024,0.02425,0.0245,0.02475,0.025,0.0252,0.0254,0.0256,0.0258,0.026,0.02625,0.0265,0.02675,0.027,0.0272,0.0274,0.0276,0.027800000000000002,0.028,0.02825,0.0285,0.02875,0.029,0.0292,0.0294,0.0296,0.0298,0.03,0.03025,0.0305,0.03075,0.031,0.0312,0.0314,0.0316,0.0318,0.032,0.03225,0.0325,0.03275,0.033,0.03325,0.0335,0.03375,0.034,0.0342,0.0344,0.034600000000000006,0.034800000000000005,0.035,0.035250000000000004,0.035500000000000004,0.03575,0.036,0.03625,0.0365,0.03675,0.037,0.0372,0.037399999999999996,0.0376,0.0378,0.038,0.03825,0.0385,0.03875,0.039,0.03925,0.0395,0.03975,0.04,0.04025,0.0405,0.04075,0.041,0.04125,0.0415,0.04175,0.042,0.0422,0.0424,0.0426,0.0428,0.043,0.04325,0.0435,0.04375,0.044,0.04425,0.0445,0.04475,0.045,0.04525,0.0455,0.04575,0.046,0.04625,0.0465,0.04675,0.047,0.04725,0.0475,0.04775,0.048,0.0482,0.0484,0.048600000000000004,0.0488,0.049,0.04925,0.0495,0.04975,0.05,0.05025,0.0505,0.050749999999999997,0.051,0.05125,0.0515,0.05175,0.052,0.05225,0.0525,0.05275,0.053,0.05325,0.0535,0.05375,0.054,0.05425,0.0545,0.05475,0.055,0.05525,0.0555,0.05575,0.056,0.05625,0.0565,0.05675,0.057,0.05725,0.0575,0.05775,0.058,0.05825,0.058499999999999996,0.05875,0.059,0.05933333333333333,0.059666666666666666,0.06,0.06025,0.0605,0.06075,0.061,0.06125,0.0615,0.06175,0.062,0.06225,0.0625,0.06275,0.063,0.06325,0.0635,0.06375,0.064,0.06425,0.0645,0.06475,0.065,0.06525,0.0655,0.06575,0.066,0.06633333333333334,0.06666666666666667,0.067,0.06725,0.0675,0.06775,0.068,0.06825,0.0685,0.06875,0.069,0.06925,0.0695,0.06975,0.07,0.07033333333333333,0.07066666666666667,0.071,0.07125,0.0715,0.07175,0.072,0.07225,0.0725,0.07275,0.073,0.07325,0.0735,0.07375,0.074,0.07433333333333333,0.07466666666666666,0.075,0.07525,0.0755,0.07575,0.076,0.07633333333333334,0.07666666666666666,0.077,0.07725,0.0775,0.07775,0.078,0.07825,0.0785,0.07875,0.079,0.07933333333333334,0.07966666666666666,0.08,0.08025,0.0805,0.08075,0.081,0.08133333333333334,0.08166666666666667,0.082,0.08225,0.0825,0.08275,0.083,0.08333333333333334,0.08366666666666667,0.084,0.08425,0.0845,0.08475,0.085,0.08533333333333333,0.08566666666666667,0.086,0.08625,0.0865,0.08675,0.087,0.08733333333333333,0.08766666666666666,0.088,0.08825,0.0885,0.08875,0.089,0.08933333333333333,0.08966666666666666,0.09,0.09025,0.0905,0.09075,0.091,0.09133333333333334,0.09166666666666666,0.092,0.09225,0.0925,0.09275,0.093,0.09333333333333334,0.09366666666666666,0.094,0.09433333333333334,0.09466666666666666,0.095,0.09525,0.0955,0.09575,0.096,0.09633333333333334,0.09666666666666666,0.097,0.09733333333333334,0.09766666666666667,0.098,0.09825,0.0985,0.09875,0.099,0.09933333333333334,0.09966666666666667,0.1,0.10033333333333334,0.10066666666666667,0.101,0.10133333333333333,0.10166666666666667,0.102,0.10225,0.1025,0.10275,0.103,0.10333333333333333,0.10366666666666666,0.104,0.10433333333333333,0.10466666666666666,0.105,0.10533333333333333,0.10566666666666666,0.106,0.10633333333333334,0.10666666666666666,0.107,0.10725,0.1075,0.10775,0.108,0.10833333333333334,0.10866666666666666,0.109,0.10933333333333334,0.10966666666666666,0.11,0.11033333333333334,0.11066666666666666,0.111,0.11133333333333334,0.11166666666666666,0.112,0.11233333333333334,0.11266666666666666,0.113,0.11333333333333334,0.11366666666666667,0.114,0.11433333333333334,0.11466666666666667,0.115,0.11533333333333334,0.11566666666666667,0.116,0.11625,0.1165,0.11675,0.117,0.11733333333333333,0.11766666666666667,0.118,0.11833333333333333,0.11866666666666666,0.119,0.11933333333333333,0.11966666666666666,0.12,0.12033333333333333,0.12066666666666666,0.121,0.12133333333333333,0.12166666666666666,0.122,0.12233333333333334,0.12266666666666666,0.123,0.1235,0.124,0.12433333333333334,0.12466666666666666,0.125,0.12533333333333332,0.12566666666666668,0.126,0.12633333333333333,0.12666666666666668,0.127,0.12733333333333333,0.12766666666666668,0.128,0.12833333333333333,0.12866666666666668,0.129,0.12933333333333333,0.12966666666666668,0.13,0.13033333333333333,0.13066666666666668,0.131,0.13133333333333333,0.13166666666666668,0.132,0.1325,0.133,0.13333333333333333,0.13366666666666668,0.134,0.13433333333333333,0.13466666666666668,0.135,0.13533333333333333,0.13566666666666669,0.136,0.13633333333333333,0.1366666666666667,0.137,0.1375,0.138,0.13833333333333334,0.1386666666666667,0.139,0.13933333333333334,0.1396666666666667,0.14,0.14033333333333334,0.14066666666666666,0.141,0.1415,0.142,0.1423333333333333,0.14266666666666666,0.143,0.1433333333333333,0.14366666666666666,0.144,0.1445,0.145,0.14533333333333331,0.14566666666666667,0.146,0.14633333333333332,0.14666666666666667,0.147,0.1475,0.148,0.14833333333333332,0.14866666666666667,0.149,0.14933333333333332,0.14966666666666667,0.15,0.1505,0.151,0.15133333333333332,0.15166666666666667,0.152,0.1525,0.153,0.15333333333333332,0.15366666666666667,0.154,0.1545,0.155,0.15533333333333332,0.15566666666666668,0.156,0.15633333333333332,0.15666666666666668,0.157,0.1575,0.158,0.15833333333333333,0.15866666666666668,0.159,0.1595,0.16,0.16033333333333333,0.16066666666666668,0.161,0.1615,0.162,0.16233333333333333,0.16266666666666668,0.163,0.1635,0.164,0.1645,0.165,0.16533333333333333,0.16566666666666668,0.166,0.1665,0.167,0.16733333333333333,0.16766666666666669,0.168,0.1685,0.169,0.1695,0.17,0.17033333333333334,0.1706666666666667,0.171,0.17149999999999999,0.172,0.1725,0.173,0.1733333333333333,0.17366666666666666,0.174,0.1745,0.175,0.1755,0.176,0.17633333333333331,0.17666666666666667,0.177,0.1775,0.178,0.1785,0.179,0.17933333333333332,0.17966666666666667,0.18,0.1805,0.181,0.1815,0.182,0.1825,0.183,0.18333333333333332,0.18366666666666667,0.184,0.1845,0.185,0.1855,0.186,0.1865,0.187,0.1875,0.188,0.1885,0.189,0.18933333333333333,0.18966666666666668,0.19,0.1905,0.191,0.1915,0.192,0.1925,0.193,0.1935,0.194,0.1945,0.195,0.1955,0.196,0.1965,0.197,0.1975,0.198,0.19833333333333333,0.19866666666666669,0.199,0.1995,0.2,0.2005,0.201,0.2015,0.202,0.2025,0.203,0.20350000000000001,0.204,0.2045,0.205,0.2055,0.206,0.2065,0.207,0.2075,0.208,0.2085,0.209,0.2095,0.21,0.2105,0.211,0.212,0.2125,0.213,0.2135,0.214,0.2145,0.215,0.2155,0.216,0.2165,0.217,0.2175,0.218,0.2185,0.219,0.2195,0.22,0.221,0.2215,0.222,0.2225,0.223,0.2235,0.224,0.2245,0.225,0.2255,0.226,0.227,0.2275,0.228,0.2285,0.229,0.2295,0.23,0.231,0.2315,0.232,0.2325,0.233,0.2335,0.234,0.235,0.2355,0.236,0.2365,0.237,0.238,0.2385,0.239,0.2395,0.24,0.241,0.2415,0.242,0.2425,0.243,0.244,0.2445,0.245,0.2455,0.246,0.247,0.2475,0.248,0.249,0.2495,0.25,0.2505,0.251,0.252,0.2525,0.253,0.254,0.2545,0.255,0.256,0.2565,0.257,0.258,0.2585,0.259,0.26,0.2605,0.261,0.262,0.2625,0.263,0.264,0.2645,0.265,0.266,0.2665,0.267,0.268,0.2685,0.269,0.27,0.271,0.2715,0.272,0.273,0.2735,0.274,0.275,0.276,0.2765,0.277,0.278,0.279,0.2795,0.28,0.281,0.282,0.2825,0.283,0.284,0.285,0.2855,0.286,0.287,0.288,0.2885,0.289,0.29,0.291,0.292,0.2925,0.293,0.294,0.295,0.296,0.2965,0.297,0.298,0.299,0.3,0.301,0.3015,0.302,0.303,0.304,0.305,0.306,0.307,0.3075,0.308,0.309,0.31,0.311,0.312,0.313,0.314,0.3145,0.315,0.316,0.317,0.318,0.319,0.32,0.321,0.322,0.323,0.324,0.325,0.326,0.327,0.328,0.329,0.33,0.331,0.332,0.333,0.334,0.335,0.336,0.337,0.338,0.339,0.34,0.341,0.342,0.343,0.344,0.345,0.346,0.347,0.349,0.35,0.351,0.352,0.353,0.354,0.355,0.357,0.358,0.359,0.36,0.361,0.363,0.364,0.365,0.366,0.367,0.369,0.37,0.371,0.373,0.374,0.375,0.376,0.378,0.379,0.381,0.382,0.383,0.385,0.386,0.387,0.389,0.39,0.392,0.393,0.395,0.396,0.398,0.399,0.401,0.402,0.404,0.406,0.407,0.409,0.411,0.412,0.414,0.416,0.417,0.419,0.421,0.423,0.424,0.426,0.428,0.43,0.432,0.434,0.436,0.438,0.44,0.442,0.444,0.446,0.448,0.45,0.453,0.455,0.457,0.46,0.462,0.464,0.467,0.469,0.472,0.474,0.477,0.48,0.483,0.485,0.488,0.491,0.494,0.497,0.501,0.504,0.507,0.511,0.514,0.518,0.522,0.525,0.529,0.534,0.538,0.542,0.547,0.552,0.557,0.562,0.568,0.574,0.58,0.587,0.594,0.602,0.61,0.619,0.629,0.64,0.653,0.668,0.687,0.711,0.749,0.999]),  # skipcq: FLK-E231, FLK-E501
            'betam2': np.array([0.251,0.288,0.313,0.331,0.346,0.359,0.371,0.381,0.39,0.398,0.406,0.413,0.419,0.426,0.432,0.437,0.442,0.448,0.452,0.457,0.462,0.466,0.47,0.474,0.478,0.482,0.485,0.489,0.492,0.496,0.499,0.502,0.505,0.508,0.511,0.514,0.517,0.52,0.522,0.525,0.528,0.53,0.533,0.535,0.538,0.54,0.542,0.545,0.547,0.549,0.551,0.553,0.555,0.558,0.56,0.562,0.564,0.566,0.567,0.569,0.571,0.573,0.575,0.577,0.579,0.58,0.582,0.584,0.586,0.587,0.589,0.591,0.592,0.594,0.595,0.597,0.599,0.6,0.602,0.603,0.605,0.606,0.608,0.609,0.61,0.612,0.613,0.615,0.616,0.617,0.619,0.62,0.622,0.623,0.624,0.626,0.627,0.628,0.629,0.631,0.632,0.633,0.634,0.636,0.637,0.638,0.639,0.64,0.642,0.643,0.644,0.645,0.646,0.647,0.649,0.65,0.651,0.652,0.653,0.654,0.655,0.656,0.657,0.658,0.659,0.66,0.662,0.663,0.664,0.665,0.666,0.667,0.668,0.669,0.67,0.671,0.672,0.673,0.674,0.675,0.676,0.6765000000000001,0.677,0.678,0.679,0.68,0.681,0.682,0.683,0.684,0.685,0.686,0.687,0.688,0.6884999999999999,0.689,0.69,0.691,0.692,0.693,0.694,0.695,0.6955,0.696,0.697,0.698,0.699,0.7,0.7004999999999999,0.701,0.702,0.703,0.704,0.705,0.7055,0.706,0.707,0.708,0.709,0.7095,0.71,0.711,0.712,0.7124999999999999,0.713,0.714,0.715,0.716,0.7164999999999999,0.717,0.718,0.719,0.7195,0.72,0.721,0.7215,0.722,0.723,0.724,0.7244999999999999,0.725,0.726,0.727,0.7275,0.728,0.729,0.7295,0.73,0.731,0.732,0.7324999999999999,0.733,0.734,0.7344999999999999,0.735,0.736,0.7364999999999999,0.737,0.738,0.7384999999999999,0.739,0.74,0.7404999999999999,0.741,0.742,0.7424999999999999,0.743,0.744,0.7444999999999999,0.745,0.746,0.7464999999999999,0.747,0.748,0.7484999999999999,0.749,0.7495,0.75,0.751,0.7515000000000001,0.752,0.753,0.7535000000000001,0.754,0.7545,0.755,0.756,0.7565,0.757,0.7575000000000001,0.758,0.759,0.7595000000000001,0.76,0.7605,0.761,0.762,0.7625,0.763,0.7635000000000001,0.764,0.765,0.7655000000000001,0.766,0.7665,0.767,0.768,0.7685,0.769,0.7695000000000001,0.77,0.7705,0.771,0.7715000000000001,0.772,0.773,0.7735000000000001,0.774,0.7745,0.775,0.7755000000000001,0.776,0.7765,0.777,0.778,0.7785,0.779,0.7795000000000001,0.78,0.7805,0.781,0.7815000000000001,0.782,0.7825,0.783,0.784,0.7845,0.785,0.7855000000000001,0.786,0.7865,0.787,0.7875000000000001,0.788,0.7885,0.789,0.7895000000000001,0.79,0.7905,0.791,0.7915000000000001,0.792,0.7925,0.793,0.7935000000000001,0.794,0.7945,0.795,0.7955000000000001,0.796,0.7965,0.797,0.7975000000000001,0.798,0.7985,0.799,0.7995000000000001,0.8,0.8005,0.801,0.8015000000000001,0.802,0.8025,0.803,0.8035000000000001,0.804,0.8045,0.805,0.8055000000000001,0.806,0.8063333333333333,0.8066666666666668,0.807,0.8075000000000001,0.808,0.8085,0.809,0.8095000000000001,0.81,0.8105,0.811,0.8115000000000001,0.812,0.8125,0.813,0.8133333333333332,0.8136666666666666,0.814,0.8145,0.815,0.8154999999999999,0.816,0.8165,0.817,0.8173333333333332,0.8176666666666667,0.818,0.8185,0.819,0.8194999999999999,0.82,0.8205,0.821,0.8213333333333332,0.8216666666666667,0.822,0.8225,0.823,0.8234999999999999,0.824,0.8245,0.825,0.8253333333333333,0.8256666666666667,0.826,0.8265,0.827,0.8274999999999999,0.828,0.8283333333333333,0.8286666666666667,0.829,0.8294999999999999,0.83,0.8303333333333333,0.8306666666666667,0.831,0.8314999999999999,0.832,0.8325,0.833,0.8333333333333333,0.8336666666666667,0.834,0.8345,0.835,0.8353333333333333,0.8356666666666667,0.836,0.8365,0.837,0.8373333333333333,0.8376666666666667,0.838,0.8385,0.839,0.8393333333333333,0.8396666666666667,0.84,0.8405,0.841,0.8413333333333333,0.8416666666666667,0.842,0.8425,0.843,0.8433333333333333,0.8436666666666667,0.844,0.8445,0.845,0.8453333333333333,0.8456666666666667,0.846,0.8465,0.847,0.8473333333333333,0.8476666666666667,0.848,0.8485,0.849,0.8493333333333333,0.8496666666666667,0.85,0.8503333333333333,0.8506666666666667,0.851,0.8514999999999999,0.852,0.8523333333333333,0.8526666666666667,0.853,0.8533333333333333,0.8536666666666667,0.854,0.8545,0.855,0.8553333333333333,0.8556666666666667,0.856,0.8563333333333333,0.8566666666666667,0.857,0.8574999999999999,0.858,0.8583333333333333,0.8586666666666667,0.859,0.8593333333333333,0.8596666666666667,0.86,0.8605,0.861,0.8613333333333333,0.8616666666666667,0.862,0.8623333333333333,0.8626666666666667,0.863,0.8633333333333333,0.8636666666666667,0.864,0.8643333333333333,0.8646666666666667,0.865,0.8654999999999999,0.866,0.8663333333333333,0.8666666666666667,0.867,0.8673333333333333,0.8676666666666667,0.868,0.8683333333333333,0.8686666666666667,0.869,0.8693333333333333,0.8696666666666667,0.87,0.8703333333333333,0.8706666666666667,0.871,0.8714999999999999,0.872,0.8723333333333333,0.8726666666666667,0.873,0.8733333333333333,0.8736666666666667,0.874,0.8743333333333333,0.8746666666666667,0.875,0.8753333333333333,0.8756666666666667,0.876,0.8763333333333333,0.8766666666666667,0.877,0.8773333333333333,0.8776666666666667,0.878,0.8783333333333333,0.8786666666666667,0.879,0.8793333333333333,0.8796666666666667,0.88,0.8803333333333333,0.8806666666666667,0.881,0.8813333333333333,0.8816666666666667,0.882,0.8823333333333333,0.8826666666666667,0.883,0.8833333333333333,0.8836666666666667,0.884,0.8843333333333333,0.8846666666666667,0.885,0.8853333333333333,0.8856666666666667,0.886,0.8863333333333333,0.8866666666666667,0.887,0.8873333333333333,0.8876666666666667,0.888,0.88825,0.8885000000000001,0.88875,0.889,0.8893333333333333,0.8896666666666667,0.89,0.8903333333333333,0.8906666666666667,0.891,0.8913333333333333,0.8916666666666667,0.892,0.8923333333333333,0.8926666666666667,0.893,0.8933333333333333,0.8936666666666667,0.894,0.89425,0.8945000000000001,0.89475,0.895,0.8953333333333333,0.8956666666666667,0.896,0.8963333333333333,0.8966666666666667,0.897,0.8973333333333333,0.8976666666666667,0.898,0.89825,0.8985000000000001,0.89875,0.899,0.8993333333333333,0.8996666666666667,0.9,0.9003333333333333,0.9006666666666667,0.901,0.9013333333333333,0.9016666666666667,0.902,0.90225,0.9025000000000001,0.90275,0.903,0.9033333333333333,0.9036666666666667,0.904,0.9043333333333333,0.9046666666666667,0.905,0.90525,0.9055,0.90575,0.906,0.9063333333333333,0.9066666666666667,0.907,0.90725,0.9075,0.9077500000000001,0.908,0.9083333333333333,0.9086666666666667,0.909,0.9093333333333333,0.9096666666666667,0.91,0.91025,0.9105000000000001,0.9107500000000001,0.911,0.9113333333333333,0.9116666666666667,0.912,0.91225,0.9125000000000001,0.9127500000000001,0.913,0.9133333333333333,0.9136666666666667,0.914,0.91425,0.9145000000000001,0.9147500000000001,0.915,0.9153333333333333,0.9156666666666667,0.916,0.91625,0.9165000000000001,0.9167500000000001,0.917,0.9173333333333333,0.9176666666666667,0.918,0.91825,0.9185000000000001,0.9187500000000001,0.919,0.9193333333333333,0.9196666666666667,0.92,0.92025,0.9205000000000001,0.9207500000000001,0.921,0.92125,0.9215,0.9217500000000001,0.922,0.9223333333333333,0.9226666666666667,0.923,0.92325,0.9235,0.9237500000000001,0.924,0.92425,0.9245000000000001,0.9247500000000001,0.925,0.9253333333333333,0.9256666666666667,0.926,0.92625,0.9265000000000001,0.9267500000000001,0.927,0.92725,0.9275,0.9277500000000001,0.928,0.9283333333333333,0.9286666666666668,0.929,0.92925,0.9295,0.9297500000000001,0.93,0.93025,0.9305000000000001,0.9307500000000001,0.931,0.93125,0.9315,0.9317500000000001,0.932,0.9323333333333333,0.9326666666666668,0.933,0.93325,0.9335,0.9337500000000001,0.934,0.93425,0.9345000000000001,0.9347500000000001,0.935,0.93525,0.9355,0.9357500000000001,0.936,0.93625,0.9365000000000001,0.9367500000000001,0.937,0.93725,0.9375,0.93775,0.938,0.9383333333333332,0.9386666666666666,0.939,0.9392499999999999,0.9395,0.93975,0.94,0.9402499999999999,0.9404999999999999,0.94075,0.941,0.9412499999999999,0.9415,0.94175,0.942,0.9422499999999999,0.9424999999999999,0.94275,0.943,0.9432499999999999,0.9435,0.94375,0.944,0.9442499999999999,0.9444999999999999,0.94475,0.945,0.9452499999999999,0.9455,0.94575,0.946,0.9462499999999999,0.9464999999999999,0.94675,0.947,0.9472499999999999,0.9475,0.94775,0.948,0.9482499999999999,0.9484999999999999,0.94875,0.949,0.9492499999999999,0.9495,0.94975,0.95,0.9502499999999999,0.9504999999999999,0.95075,0.951,0.9512499999999999,0.9515,0.95175,0.952,0.9522499999999999,0.9524999999999999,0.95275,0.953,0.9531999999999999,0.9533999999999999,0.9536,0.9538,0.954,0.9542499999999999,0.9544999999999999,0.95475,0.955,0.9552499999999999,0.9555,0.95575,0.956,0.9562499999999999,0.9564999999999999,0.95675,0.957,0.9572499999999999,0.9575,0.95775,0.958,0.9581999999999999,0.9583999999999999,0.9586,0.9588,0.959,0.9592499999999999,0.9595,0.95975,0.96,0.9602499999999999,0.9604999999999999,0.96075,0.961,0.9612499999999999,0.9615,0.96175,0.962,0.9621999999999999,0.9623999999999999,0.9626,0.9628,0.963,0.9632499999999999,0.9635,0.96375,0.964,0.9642499999999999,0.9644999999999999,0.96475,0.965,0.9652,0.9653999999999999,0.9656,0.9658,0.966,0.9662499999999999,0.9664999999999999,0.96675,0.967,0.9672499999999999,0.9675,0.96775,0.968,0.9682,0.9683999999999999,0.9686,0.9688,0.969,0.96925,0.9695,0.96975,0.97,0.9702,0.9703999999999999,0.9706,0.9708,0.971,0.97125,0.9715,0.97175,0.972,0.9722,0.9723999999999999,0.9726,0.9728,0.973,0.97325,0.9735,0.97375,0.974,0.9742,0.9743999999999999,0.9746,0.9748,0.975,0.97525,0.9755,0.97575,0.976,0.9762,0.9763999999999999,0.9766,0.9768,0.977,0.97725,0.9775,0.97775,0.978,0.9782,0.9783999999999999,0.9786,0.9788,0.979,0.97925,0.9795,0.97975,0.98,0.9802,0.9803999999999999,0.9806,0.9808,0.981,0.9812,0.9813999999999999,0.9816,0.9818,0.982,0.98225,0.9824999999999999,0.98275,0.983,0.9832,0.9833999999999999,0.9836,0.9838,0.984,0.9842,0.9843999999999999,0.9846,0.9848,0.985,0.98525,0.9855,0.98575,0.986,0.9862,0.9863999999999999,0.9866,0.9868,0.987,0.9872,0.9873999999999999,0.9876,0.9878,0.988,0.9882,0.9884,0.9886,0.9888,0.989,0.98925,0.9895,0.98975,0.99,0.9902,0.9904,0.9906,0.9908,0.991,0.9912,0.9914,0.9916,0.9918,0.992,0.9922,0.9924,0.9926,0.9928,0.993,0.9932,0.9934,0.9936,0.9938,0.994,0.9942,0.9944,0.9946,0.9948,0.995,0.9952,0.9954,0.9956,0.9958,0.996,0.9962,0.9964,0.9966,0.9968,0.997,0.9972,0.9974,0.9976,0.9978,0.998,0.9982,0.9984,0.9986,0.9988,0.999,0.999,0.999,0.999,0.999])  # skipcq: FLK-E231, FLK-E501
        }

        # List of the maximal distances to the significance niveau 'ks_alpha', the samplesize 'num_init' and the single
        # distributions in the initial KS-tests
        self.crit_val_ini = {
            0.01: {
                20: {
                    'uni': 0.348124887139, 'nor': 0.224499609111, 'beta1': 0.345319670312, 'beta2': 0.216795152755,
                    'beta4': 0.25
                },
                50: {
                    'uni': 0.225420589564, 'nor': 0.145294870373, 'beta1': 0.225192815141, 'beta2': 0.140053781955,
                    'beta4': 0.177034995439
                },
                100: {
                    'uni': 0.160728194817, 'nor': 0.103707395283, 'beta1': 0.160715202747, 'beta2': 0.100262074938,
                    'beta4': 0.132857001726
                },
                150: {
                    'uni': 0.131283109305, 'nor': 0.0850209323211, 'beta1': 0.131520176978, 'beta2': 0.0821266317247,
                    'beta4': 0.112496947432
                },
                200: {
                    'uni': 0.11414159345, 'nor': 0.073814797172, 'beta1': 0.114167391557, 'beta2': 0.0712715308668,
                    'beta4': 0.0999214494489
                }
            },
            0.05: {
                20: {
                    'uni': 0.291328932926, 'nor': 0.193089576397, 'beta1': 0.288470457906, 'beta2': 0.186764430369,
                    'beta4': 0.21131117329
                },
                50: {
                    'uni': 0.187857939084, 'nor': 0.124976281159, 'beta1': 0.187579279621, 'beta2': 0.120739023025,
                    'beta4': 0.1472078114
                },
                100: {
                    'uni': 0.133759516365, 'nor': 0.0890663949534, 'beta1': 0.13405821848, 'beta2': 0.0861800886195,
                    'beta4': 0.11
                },
                150: {
                    'uni': 0.109587712896, 'nor': 0.0731082119117, 'beta1': 0.10960917714, 'beta2': 0.0706864579564,
                    'beta4': 0.0926528616077
                },
                200: {
                    'uni': 0.0950292895371, 'nor': 0.0633952199486, 'beta1': 0.0951133976321, 'beta2': 0.0614202560412,
                    'beta4': 0.0811635018839
                }
            },
            0.10: {
                20: {
                    'uni': 0.263127886669, 'nor': 0.177538180228, 'beta1': 0.259791341766, 'beta2': 0.171861758546,
                    'beta4': 0.198152215369
                },
                50: {
                    'uni': 0.169183855849, 'nor': 0.114591510429, 'beta1': 0.168787738005, 'beta2': 0.11112278739,
                    'beta4': 0.13404097074
                },
                100: {
                    'uni': 0.120542327047, 'nor': 0.0818166095908, 'beta1': 0.120484063551, 'beta2': 0.0793243637592,
                    'beta4': 0.0992270269286
                },
                150: {
                    'uni': 0.09867439782, 'nor': 0.067089238925, 'beta1': 0.0988457759044, 'beta2': 0.0650761056877,
                    'beta4': 0.0824041716017
                },
                200: {
                    'uni': 0.0856220256896, 'nor': 0.0582374637319, 'beta1': 0.0857116539328, 'beta2': 0.0564867252362,
                    'beta4': 0.0724720056026
                }
            }
        }

        # List of the maximal distances to the significance niveau 'ks_alpha', the samplesize 'num_init' in the initialization and
        # the samplesize 'num_update' in the update step and the single distributions in the sKS-tests in the update steps
        self.crit_val_upd = {
            0.01: {
                50: {
                    10: {
                        'uni': 0.496888970511, 'nor': 0.519945086131, 'beta1': 0.492182170145, 'beta2': 0.516332672128,
                        'beta4': 0.518350685953
                    },
                    20: {
                        'uni': 0.360946908918, 'nor': 0.391875037468, 'beta1': 0.353740242897, 'beta2': 0.38462471075,
                        'beta4': 0.394761192346
                    },
                    50: {
                        'uni': 0.2364048355, 'nor': 0.282412331796, 'beta1': 0.225833184975, 'beta2': 0.274133465745,
                        'beta4': 0.286667829985
                    },
                    100: {
                        'uni': 0.175089180032, 'nor': 0.231964002245, 'beta1': 0.1632997257, 'beta2': 0.224136111709,
                        'beta4': 0.241499060808
                    },
                    150: {
                        'uni': 0.150314420227, 'nor': 0.212013857479, 'beta1': 0.136292970627, 'beta2': 0.203810621947,
                        'beta4': 0.226666666667
                    },
                    200: {
                        'uni': 0.136640128419, 'nor': 0.199769030128, 'beta1': 0.122886040376, 'beta2': 0.191341205468,
                        'beta4': 0.218970709501
                    }
                },
                100: {
                    10: {
                        'uni': 0.494283760444, 'nor': 0.502628314811, 'beta1': 0.488366024959, 'beta2': 0.500628015326,
                        'beta4': 0.504621453745
                    },
                    20: {
                        'uni': 0.354323780354, 'nor': 0.373653207036, 'beta1': 0.352462020844, 'beta2': 0.370571598957,
                        'beta4': 0.374628062376
                    },
                    50: {
                        'uni': 0.228910672752, 'nor': 0.255661039734, 'beta1': 0.225847047913, 'beta2': 0.251053816314,
                        'beta4': 0.255917463721
                    },
                    100: {
                        'uni': 0.164921260704, 'nor': 0.200123588294, 'beta1': 0.161417596329, 'beta2': 0.194940701779,
                        'beta4': 0.20027749479
                    },
                    150: {
                        'uni': 0.135482022179, 'nor': 0.175830889639, 'beta1': 0.13187575212, 'beta2': 0.171195176268,
                        'beta4': 0.18
                    },
                    200: {
                        'uni': 0.118615255649, 'nor': 0.1648901313, 'beta1': 0.114207850996, 'beta2': 0.157770516553,
                        'beta4': 0.169664613602
                    }
                },
                150: {
                    10: {
                        'uni': 0.489737315312, 'nor': 0.497864891192, 'beta1': 0.487921854614, 'beta2': 0.496850168022,
                        'beta4': 0.501240972478
                    },
                    20: {
                        'uni': 0.352961510556, 'nor': 0.364288580679, 'beta1': 0.352975930751, 'beta2': 0.363710121748,
                        'beta4': 0.365776045005
                    },
                    50: {
                        'uni': 0.227413865337, 'nor': 0.245324202262, 'beta1': 0.225426171779, 'beta2': 0.243072223435,
                        'beta4': 0.245558958332
                    },
                    100: {
                        'uni': 0.162227837985, 'nor': 0.188379049798, 'beta1': 0.161446580977, 'beta2': 0.183909494947,
                        'beta4': 0.186801912433
                    },
                    150: {
                        'uni': 0.133445640931, 'nor': 0.162891939681, 'beta1': 0.131600793498, 'beta2': 0.158415541981,
                        'beta4': 0.163596208405
                    },
                    200: {
                        'uni': 0.116193337566, 'nor': 0.148195272135, 'beta1': 0.114087518538, 'beta2': 0.145471430429,
                        'beta4': 0.150435363804
                    }
                },
                200: {
                    10: {
                        'uni': 0.492355247252, 'nor': 0.498372062428, 'beta1': 0.486917037362, 'beta2': 0.493913157262,
                        'beta4': 0.496995766487
                    },
                    20: {
                        'uni': 0.354473713682, 'nor': 0.362440493911, 'beta1': 0.352121204197, 'beta2': 0.361472141933,
                        'beta4': 0.362170596303
                    },
                    50: {
                        'uni': 0.227198657223, 'nor': 0.24184159736, 'beta1': 0.227023913235, 'beta2': 0.238737887084,
                        'beta4': 0.240854680058
                    },
                    100: {
                        'uni': 0.161979040041, 'nor': 0.180972404209, 'beta1': 0.16046944814, 'beta2': 0.178827368024,
                        'beta4': 0.179752395924
                    },
                    150: {
                        'uni': 0.133232825981, 'nor': 0.155687208155, 'beta1': 0.131355221131, 'beta2': 0.152493163614,
                        'beta4': 0.155514618341
                    },
                    200: {
                        'uni': 0.115296060947, 'nor': 0.141389175515, 'beta1': 0.114384144801, 'beta2': 0.137880217687,
                        'beta4': 0.14190052815
                    }
                }
            },
            0.05: {
                50: {
                    10: {
                        'uni': 0.417265610972, 'nor': 0.431670351832, 'beta1': 0.410096257626, 'beta2': 0.42909033207,
                        'beta4': 0.432398289202
                    },
                    20: {
                        'uni': 0.301399033188, 'nor': 0.324303748137, 'beta1': 0.295002069354, 'beta2': 0.320702711766,
                        'beta4': 0.325252508886
                    },
                    50: {
                        'uni': 0.197020955951, 'nor': 0.231775298752, 'beta1': 0.189187959134, 'beta2': 0.226318739974,
                        'beta4': 0.233993573908
                    },
                    100: {
                        'uni': 0.144987170156, 'nor': 0.188834523524, 'beta1': 0.136002700168, 'beta2': 0.182342752946,
                        'beta4': 0.193070870167
                    },
                    150: {
                        'uni': 0.123232457933, 'nor': 0.171484419894, 'beta1': 0.113333333333, 'beta2': 0.164671037041,
                        'beta4': 0.177783072759
                    },
                    200: {
                        'uni': 0.110677102373, 'nor': 0.162135212071, 'beta1': 0.1, 'beta2': 0.154588679086,
                        'beta4': 0.169910282609
                    }
                },
                100: {
                    10: {
                        'uni': 0.411773476839, 'nor': 0.420548778373, 'beta1': 0.409442048149, 'beta2': 0.418581674681,
                        'beta4': 0.420435420789
                    },
                    20: {
                        'uni': 0.296812598919, 'nor': 0.309359422063, 'beta1': 0.294159359623, 'beta2': 0.307368472814,
                        'beta4': 0.308938197085
                    },
                    50: {
                        'uni': 0.190726544783, 'nor': 0.211109731822, 'beta1': 0.188363406074, 'beta2': 0.20792565631,
                        'beta4': 0.210517767028
                    },
                    100: {
                        'uni': 0.136758284745, 'nor': 0.163870028597, 'beta1': 0.13412979671, 'beta2': 0.15996873838,
                        'beta4': 0.164270534936
                    },
                    150: {
                        'uni': 0.112858597869, 'nor': 0.144364730479, 'beta1': 0.109805011673, 'beta2': 0.140013403237,
                        'beta4': 0.145801697487
                    },
                    200: {
                        'uni': 0.0987067080902, 'nor': 0.133352281188, 'beta1': 0.0954050158312, 'beta2': 0.128573220133,
                        'beta4': 0.135275914246
                    }
                },
                150: {
                    10: {
                        'uni': 0.411015760689, 'nor': 0.417270596603, 'beta1': 0.408901188338, 'beta2': 0.415843756711,
                        'beta4': 0.416965036073
                    },
                    20: {
                        'uni': 0.295673211556, 'nor': 0.304346624842, 'beta1': 0.294348558971, 'beta2': 0.302656891266,
                        'beta4': 0.303901823221
                    },
                    50: {
                        'uni': 0.189770465497, 'nor': 0.204019231966, 'beta1': 0.188360071778, 'beta2': 0.201782670729,
                        'beta4': 0.202846913244
                    },
                    100: {
                        'uni': 0.135429763631, 'nor': 0.154726140402, 'beta1': 0.133894140157, 'beta2': 0.152014350301,
                        'beta4': 0.15395515534
                    },
                    150: {
                        'uni': 0.1111456267, 'nor': 0.134128253323, 'beta1': 0.109716923326, 'beta2': 0.130869695547,
                        'beta4': 0.133523280381
                    },
                    200: {
                        'uni': 0.0966690896542, 'nor': 0.122268164736, 'beta1': 0.09517152783, 'beta2': 0.118675140868,
                        'beta4': 0.122243741774
                    }
                },
                200: {
                    10: {
                        'uni': 0.410835132966, 'nor': 0.414898738295, 'beta1': 0.409362652164, 'beta2': 0.414348695514,
                        'beta4': 0.415373856218
                    },
                    20: {
                        'uni': 0.295201058546, 'nor': 0.302089827989, 'beta1': 0.293929961808, 'beta2': 0.300712032791,
                        'beta4': 0.301361211791
                    },
                    50: {
                        'uni': 0.189378345582, 'nor': 0.200280681315, 'beta1': 0.188462703846, 'beta2': 0.198324329152,
                        'beta4': 0.199258535998
                    },
                    100: {
                        'uni': 0.134817549428, 'nor': 0.149891213491, 'beta1': 0.134022229898, 'beta2': 0.147676069254,
                        'beta4': 0.148929976922
                    },
                    150: {
                        'uni': 0.110563850619, 'nor': 0.128436220554, 'beta1': 0.109709133843, 'beta2': 0.125945549651,
                        'beta4': 0.127627426039
                    },
                    200: {
                        'uni': 0.096008472884, 'nor': 0.116217345787, 'beta1': 0.0950905899576, 'beta2': 0.113333585524,
                        'beta4': 0.115641208528
                    }
                }
            },
            0.10: {
                50: {
                    10: {
                        'uni': 0.374838186425, 'nor': 0.386801876299, 'beta1': 0.369224239134, 'beta2': 0.385908277508,
                        'beta4': 0.38919265426
                    },
                    20: {
                        'uni': 0.271467001084, 'nor': 0.290670049669, 'beta1': 0.264320473686, 'beta2': 0.286509131322,
                        'beta4': 0.290185851682
                    },
                    50: {
                        'uni': 0.17702431228, 'nor': 0.206222270729, 'beta1': 0.170702702926, 'beta2': 0.201471793298,
                        'beta4': 0.206819817388
                    },
                    100: {
                        'uni': 0.130266866062, 'nor': 0.167183296866, 'beta1': 0.122073041045, 'beta2': 0.161982907329,
                        'beta4': 0.169062524072
                    },
                    150: {
                        'uni': 0.11018576154, 'nor': 0.150823833173, 'beta1': 0.101113554161, 'beta2': 0.145657965857,
                        'beta4': 0.153751436399
                    },
                    200: {
                        'uni': 0.0988392246677, 'nor': 0.142325125502, 'beta1': 0.09, 'beta2': 0.136339391548,
                        'beta4': 0.146637696044
                    }
                },
                100: {
                    10: {
                        'uni': 0.371769867941, 'nor': 0.377999695631, 'beta1': 0.369299067324, 'beta2': 0.377502642334,
                        'beta4': 0.377350959828
                    },
                    20: {
                        'uni': 0.26684439939, 'nor': 0.277945445898, 'beta1': 0.265157992022, 'beta2': 0.276690417492,
                        'beta4': 0.277570874567
                    },
                    50: {
                        'uni': 0.171978780727, 'nor': 0.188942000353, 'beta1': 0.169966250686, 'beta2': 0.186114896133,
                        'beta4': 0.188561426673
                    },
                    100: {
                        'uni': 0.123317841722, 'nor': 0.14625624923, 'beta1': 0.120845393242, 'beta2': 0.142784651961,
                        'beta4': 0.145704274295
                    },
                    150: {
                        'uni': 0.101600218484, 'nor': 0.128804271305, 'beta1': 0.0987047909149, 'beta2': 0.124823356929,
                        'beta4': 0.127694497817
                    },
                    200: {
                        'uni': 0.0888591626073, 'nor': 0.118356927498, 'beta1': 0.0857472570331, 'beta2': 0.114462392351,
                        'beta4': 0.118389894735
                    }
                },
                150: {
                    10: {
                        'uni': 0.371216168791, 'nor': 0.374977020662, 'beta1': 0.368730563415, 'beta2': 0.374740261911,
                        'beta4': 0.375582769688
                    },
                    20: {
                        'uni': 0.265589930198, 'nor': 0.273636798101, 'beta1': 0.264439542245, 'beta2': 0.272883403835,
                        'beta4': 0.272433806756
                    },
                    50: {
                        'uni': 0.171061444036, 'nor': 0.182837108803, 'beta1': 0.170151501072, 'beta2': 0.181172627133,
                        'beta4': 0.181783889017
                    },
                    100: {
                        'uni': 0.122009070635, 'nor': 0.138246116327, 'beta1': 0.120643173596, 'beta2': 0.135926632896,
                        'beta4': 0.137469938512
                    },
                    150: {
                        'uni': 0.100012854643, 'nor': 0.119512882075, 'beta1': 0.0987883309699, 'beta2': 0.116547214583,
                        'beta4': 0.118848473315
                    },
                    200: {
                        'uni': 0.0872086083578, 'nor': 0.108740026506, 'beta1': 0.0855473193062, 'beta2': 0.105631016153,
                        'beta4': 0.108266583603
                    }
                },
                200: {
                    10: {
                        'uni': 0.369162613138, 'nor': 0.372992763332, 'beta1': 0.369163620401, 'beta2': 0.372758549076,
                        'beta4': 0.373274988478
                    },
                    20: {
                        'uni': 0.265070098045, 'nor': 0.271135977648, 'beta1': 0.265723150778, 'beta2': 0.269960601419,
                        'beta4': 0.27120912702
                    },
                    50: {
                        'uni': 0.170313150875, 'nor': 0.179618130632, 'beta1': 0.169535942528, 'beta2': 0.177993053878,
                        'beta4': 0.178246123576
                    },
                    100: {
                        'uni': 0.121440577395, 'nor': 0.134167830597, 'beta1': 0.121062406807, 'beta2': 0.132320882949,
                        'beta4': 0.132812154196
                    },
                    150: {
                        'uni': 0.0994403210074, 'nor': 0.115095458468, 'beta1': 0.0988127353944, 'beta2': 0.11238752212,
                        'beta4': 0.11389714621
                    },
                    200: {
                        'uni': 0.0865395713527, 'nor': 0.103728842217, 'beta1': 0.0859945910524, 'beta2': 0.101105432659,
                        'beta4': 0.102541639283
                    }
                }
            }
        }

        if self.used_multinomial_test == 'Approx':
            # Stores the number of minimal successes for the BT with the stated samplesizes and probabilities
            self.bt_min_succ_data = {
                'num_bt = 1000, alpha = 0.1': [0.00010537719726562501, 0.00053192138671875, 0.0011025695800781247, 0.001745849609375, 0.002434478759765625, 0.0031548461914062496, 0.0038988952636718746, 0.004661621093749999, 0.0054394836425781246, 0.00622998046875, 0.007031219482421876, 0.007841796875000001, 0.008660491943359376, 0.009486450195312503, 0.010318756103515628, 0.011156921386718754, 0.01200039672851563, 0.012848632812500007, 0.013701263427734384, 0.014557983398437509, 0.015418487548828134, 0.01628247070312501, 0.017149749755859388, 0.018020080566406263, 0.01889334106445314, 0.019769226074218764, 0.02064773559570314, 0.021528625488281265, 0.022411773681640636, 0.02329711914062501, 0.024184600830078132, 0.025073974609375008, 0.025965240478515637, 0.026858337402343765, 0.027753143310546888, 0.028649597167968763, 0.029547637939453135, 0.030447204589843763, 0.031348175048828135, 0.03225061035156251, 0.03315444946289064, 0.03405950927734376, 0.03496591186523438, 0.035873535156250004, 0.03678231811523438, 0.037692199707031256, 0.03860324096679688, 0.03951538085937501, 0.04042855834960939, 0.041342712402343766, 0.04225790405273439, 0.043174011230468774, 0.04409109497070315, 0.04500903320312503, 0.04592788696289065, 0.046847595214843774, 0.04776815795898439, 0.04868951416015627, 0.0496116638183594, 0.050534606933593774, 0.05145828247070316, 0.05238275146484379, 0.05330795288085941, 0.054233825683593785, 0.05516036987304691, 0.05608764648437504, 0.05701559448242192, 0.0579441528320313, 0.058873382568359424, 0.05980322265625005, 0.060733673095703176, 0.0616647338867188, 0.06259634399414066, 0.06352856445312505, 0.06446139526367192, 0.06539471435546879, 0.06632864379882816, 0.06726306152343753, 0.06819802856445314, 0.06913348388671878, 0.07006948852539066, 0.07100598144531253, 0.07194296264648442, 0.07288037109375006, 0.07381832885742196, 0.07475677490234384, 0.07569564819335947, 0.07663494873046883, 0.07757473754882821, 0.07851495361328134, 0.0794555969238282, 0.08039666748046884, 0.0813381652832032, 0.08228002929687508, 0.08322238159179696, 0.08416510009765632, 0.08510824584960947, 0.0860517578125001, 0.08699563598632823, 0.08793994140625011, 0.08888455200195325, 0.08982958984375011, 0.0907749938964845, 0.09172076416015637, 0.0926668395996095, 0.09361334228515636, 0.09456015014648447, 0.0955073242187501, 0.09645480346679697, 0.09740264892578135, 0.09835079956054699, 0.09929931640625012, 0.10024807739257824, 0.10119726562500012, 0.10214669799804699, 0.10309643554687511, 0.10404653930664076, 0.10499694824218764, 0.10594760131835951, 0.10689855957031263, 0.10784988403320328, 0.10880145263671892, 0.10975332641601579, 0.11070544433593765, 0.11165786743164077, 0.11261059570312515, 0.11356356811523452, 0.11451684570312515, 0.11547036743164077, 0.11642419433593765, 0.11737826538085952, 0.11833258056640639, 0.11928720092773451, 0.12024206542968766, 0.1211971740722658, 0.12215252685546893, 0.12310818481445332, 0.12406402587890647, 0.12502017211914085, 0.12597650146484402, 0.12693313598632844, 0.1278899536132816, 0.128847076416016, 0.12980438232421915, 0.13076193237304728, 0.13171972656250042, 0.13267776489257854, 0.13363598632812546, 0.13459445190429736, 0.13555316162109426, 0.13651211547851616, 0.1374712524414068, 0.1384306335449224, 0.13939019775390682, 0.14035000610351622, 0.14130999755859436, 0.1422702331542975, 0.14323065185546935, 0.1441913146972662, 0.14515216064453185, 0.14611318969726622, 0.14707446289062565, 0.1480359191894538, 0.14899761962890695, 0.14995944213867257, 0.15092150878906319, 0.1518837585449226, 0.15284625244140698, 0.15380886840820385, 0.1547717285156257, 0.1557347717285163, 0.15669799804687568, 0.15766140747070378, 0.15862500000000063, 0.15958877563476626, 0.16055273437500062, 0.16151693725586003, 0.1624812622070319, 0.16344577026367252, 0.16441046142578192, 0.16537533569336005, 0.1663403930664069, 0.1673056335449225, 0.1682710571289069, 0.16923660278320374, 0.17020239257812564, 0.17116830444336, 0.17213439941406317, 0.17310067749023506, 0.17406707763671947, 0.17503372192382888, 0.17600048828125076, 0.17696737670898516, 0.17793450927734455, 0.17890176391601642, 0.17986920166015702, 0.18083676147461014, 0.18180456542968826, 0.18277243041992264, 0.18374053955078207, 0.18470877075195397, 0.1856771240234384, 0.18664566040039154, 0.18761437988281343, 0.18858322143554784, 0.189552246093751, 0.1905213928222666, 0.191490722656251, 0.19246017456054787, 0.19342980957031353, 0.19439956665039165, 0.19536950683593857, 0.196339569091798, 0.1973097534179699, 0.19828012084961055, 0.19925061035156372, 0.2002212829589856, 0.20119207763672003, 0.20216299438476693, 0.20313409423828255, 0.2041053161621107, 0.20507672119140757, 0.20604818725586072, 0.2070198364257826, 0.2079916687011732, 0.20896356201172006, 0.20993563842773566, 0.21090783691406378, 0.21188021850586064, 0.21285266113281376, 0.21382528686523566, 0.21479803466797004, 0.21577090454101694, 0.21674395751953257, 0.21771707153320446, 0.2186903686523451, 0.21966378784179824, 0.22063732910156386, 0.221610992431642, 0.22258483886718888, 0.22355874633789202, 0.2245328369140639, 0.2255070495605483, 0.22648138427734515, 0.22745584106445454, 0.2284304199218764, 0.2294051208496108, 0.23037994384765764, 0.23135494995117323, 0.23233001708984508, 0.23330520629882945, 0.23428057861328255, 0.23525601196289192, 0.23623162841797002, 0.23720730590820438, 0.23818316650390753, 0.23915908813476694, 0.24013519287109508, 0.2411113586425795, 0.24208764648437636, 0.24306411743164197, 0.24404064941406384, 0.24501730346679823, 0.24599414062500136, 0.24697103881836074, 0.2479480590820326, 0.24892520141601698, 0.24990246582031383, 0.25087985229492316, 0.251857360839845, 0.2528349304199231, 0.25381268310547, 0.25479049682617305, 0.2557684936523449, 0.256746551513673, 0.2577247314453136, 0.25870303344726675, 0.2596814575195324, 0.26065994262695436, 0.26163861083984497, 0.26261734008789184, 0.26359619140625123, 0.26457516479492316, 0.2655542602539076, 0.2665334167480483, 0.2675127563476577, 0.26849215698242335, 0.2694716796875015, 0.2704513244628922, 0.2714310302734392, 0.2724109191894548, 0.2733908691406267, 0.27437094116211114, 0.2753510742187518, 0.2763313293457049, 0.2773117675781268, 0.2782922058105487, 0.2792728271484394, 0.2802535095214863, 0.28123431396484566, 0.28221524047851754, 0.28319628906250194, 0.2841773986816426, 0.2851586303710957, 0.286139923095705, 0.28712139892578314, 0.28810293579101753, 0.28908459472656445, 0.2900663146972675, 0.2910481567382831, 0.2920301208496112, 0.2930121459960956, 0.2939942932128925, 0.2949765625000019, 0.2959589538574239, 0.296941406250002, 0.29792391967773635, 0.2989066162109395, 0.2998893737792989, 0.3008721923828146, 0.30185519409179906, 0.3028382568359398, 0.3038213806152368, 0.3048046264648462, 0.3057879943847681, 0.30677148437500257, 0.3077550354003933, 0.30873864746094026, 0.3097224426269559, 0.3107062377929716, 0.311690216064456, 0.3126742553710967, 0.3136583557128937, 0.31464263916015933, 0.315626922607425, 0.3166113891601594, 0.3175959167480501, 0.31858050537109706, 0.3195652160644564, 0.3205500488281283, 0.3215349426269565, 0.32251995849609716, 0.3235050354003941, 0.32449023437500346, 0.3254754943847691, 0.3264608764648472, 0.3274463806152379, 0.32843194580078483, 0.3294175720214879, 0.3304033813476598, 0.33138919067383166, 0.33237512207031605, 0.333361175537113, 0.33434729003906616, 0.33533352661133187, 0.33631982421875384, 0.3373062438964882, 0.33829272460937887, 0.33927932739258204, 0.3402659912109415, 0.34125277709961344, 0.34223962402344166, 0.3432265930175824, 0.3442136230468794, 0.34520077514648884, 0.3461879882812545, 0.34717532348633273, 0.3481627197265672, 0.34915017700195794, 0.35013781738281735, 0.35112545776367676, 0.3521132202148487, 0.35310110473633316, 0.3540890502929739, 0.35507705688477087, 0.35606518554688027, 0.3570534362793022, 0.3580417480468804, 0.35903012084961483, 0.3600186157226618, 0.36100717163086493, 0.3619957885742243, 0.3629845886230525, 0.36397338867188067, 0.36496231079102137, 0.3659513549804745, 0.3669403991699276, 0.3679296264648495, 0.3689188537597714, 0.3699082641601621, 0.3708976745605528, 0.3718872070312559, 0.3728768615722715, 0.3738665771484434, 0.37485635375977155, 0.37584625244141223, 0.37683621215820917, 0.37782623291016226, 0.3788163757324279, 0.379806640625006, 0.38079690551758416, 0.3817873535156311, 0.382777801513678, 0.38376837158203736, 0.3847590637207092, 0.3857497558593811, 0.38674063110352175, 0.3877315063476624, 0.38872250366211547, 0.38971362304688106, 0.39070474243164666, 0.39169604492188104, 0.3926873474121154, 0.3936787719726623, 0.39467031860352164, 0.39566186523438096, 0.3966535339355528, 0.3976453247070372, 0.3986371765136778, 0.3996290893554747, 0.4006211242675841, 0.4016132202148498, 0.40260537719727174, 0.4035976562500061, 0.4045899963378967, 0.40558245849609986, 0.40657498168945927, 0.40756756591797494, 0.40856027221680313, 0.4095530395507876, 0.4105458679199283, 0.41153881835938144, 0.41253182983399084, 0.4135249023437565, 0.4145180969238347, 0.4155113525390691, 0.4165047302246161, 0.4174981689453192, 0.4184916687011786, 0.4194852905273505, 0.4204789733886787, 0.4214727172851631, 0.42246658325196007, 0.4234605102539132, 0.42445449829102255, 0.42544860839844445, 0.4264427795410226, 0.42743701171875703, 0.428431365966804, 0.4294257812500071, 0.4304203186035227, 0.4314149169921946, 0.43240957641602273, 0.43340429687500714, 0.4343991394043041, 0.43539404296875717, 0.4363890686035228, 0.43738415527344465, 0.4383793029785228, 0.43937457275391345, 0.4403698425293041, 0.44136529541016345, 0.4423607482910228, 0.44335632324219465, 0.4443519592285228, 0.4453477172851634, 0.44634353637696034, 0.4473394165039134, 0.448335418701179, 0.44933148193360084, 0.45032760620117895, 0.4513238525390696, 0.4523201599121165, 0.45331652832031955, 0.45431295776367886, 0.4553095092773507, 0.45630618286133506, 0.45730285644531943, 0.4582996520996163, 0.45929650878906936, 0.4602934875488349, 0.46129052734375675, 0.46228762817383484, 0.46328485107422546, 0.46428213500977233, 0.46527947998047536, 0.46627688598633465, 0.46727441406250647, 0.46827200317383455, 0.46926971435547515, 0.470267486572272, 0.47126531982422515, 0.4722632751464908, 0.47326123046875646, 0.4742593688964908, 0.4752575073242251, 0.47625576782227197, 0.4772540893554751, 0.47825247192383447, 0.47925097656250637, 0.48024954223633454, 0.48124822998047523, 0.4822469787597722, 0.4832457885742253, 0.48424465942383466, 0.48524365234375655, 0.4862427062988347, 0.4872418212890691, 0.48824105834961606, 0.48924035644531916, 0.4902397155761785, 0.4912391967773504, 0.49223873901367854, 0.4932384033203192, 0.4942380676269599, 0.49523785400391296, 0.49623776245117857, 0.4972376708984442, 0.4982377014160223, 0.49923785400391296, 0.5002380065918036, 0.5012382812500068, 0.5022386779785225, 0.5032390747070382, 0.5042395935058663, 0.505240234375007, 0.5062408752441475, 0.5072416381836006, 0.5082425231933662, 0.5092434082031317, 0.5102444152832099, 0.5112454833984442, 0.5122466735839911, 0.5132479248046943, 0.5142492370605537, 0.5152506713867255, 0.5162521667480537, 0.5172537231445381, 0.5182554016113349, 0.5192571411132879, 0.5202589416503972, 0.521260864257819, 0.5222628479003971, 0.5232648925781315, 0.5242670593261783, 0.5252692871093815, 0.5262715759277409, 0.5272739868164127, 0.5282764587402409, 0.5292790527343815, 0.530281646728522, 0.531284362792975, 0.5322872009277405, 0.5332901000976623, 0.5342930603027404, 0.5352960815429747, 0.5362992248535216, 0.5373024291992247, 0.5383057556152403, 0.5393091430664122, 0.5403125915527404, 0.541316162109381, 0.542319793701178, 0.5433234863281312, 0.544327301025397, 0.5453311767578188, 0.546335113525397, 0.5473391723632876, 0.5483432922363344, 0.5493475341796937, 0.5503517761230531, 0.5513562011718812, 0.5523606262207094, 0.55336517333985, 0.5543698425293032, 0.5553745117187564, 0.5563793029785221, 0.5573842163086004, 0.5583891906738349, 0.5593942260742256, 0.5603993225097726, 0.5614045410156322, 0.5624098815918042, 0.563415222167976, 0.5644207458496165, 0.5654262695312572, 0.5664319152832104, 0.5674376220703198, 0.5684434509277417, 0.56944934082032, 0.5704552917480544, 0.5714613647461014, 0.5724674987793047, 0.5734737548828205, 0.5744800720214924, 0.5754864501953205, 0.576492950439461, 0.5774995727539142, 0.5785061950683673, 0.579512939453133, 0.5805198059082112, 0.5815267333984456, 0.5825337219238363, 0.5835408325195396, 0.5845480041503991, 0.5855552368164149, 0.586562591552743, 0.5875700073242273, 0.5885775451660241, 0.5895851440429772, 0.5905928649902429, 0.5916006469726648, 0.5926085510253992, 0.5936165161132899, 0.5946245422363369, 0.5956326904296964, 0.5966408996582121, 0.5976492309570403, 0.5986576232910245, 0.599666076660165, 0.6006746520996181, 0.6016833496093837, 0.6026921081543056, 0.6037009277343838, 0.6047098693847744, 0.6057188720703214, 0.6067279968261808, 0.6077371826171966, 0.6087464904785247, 0.6097558593750089, 0.6107653503418057, 0.6117749023437588, 0.6127845764160244, 0.6137943115234462, 0.6148041076660243, 0.615814025878915, 0.6168240661621182, 0.6178341674804776, 0.6188443298339933, 0.6198546142578214, 0.6208650207519618, 0.6218754882812586, 0.6228860168457117, 0.6238966674804772, 0.6249074401855553, 0.6259182739257897, 0.6269291687011803, 0.6279401855468835, 0.6289513244628991, 0.629962524414071, 0.6309738464355553, 0.6319852294921958, 0.6329966735839926, 0.6340083007812581, 0.6350199279785237, 0.636031738281258, 0.6370436096191486, 0.6380555419921955, 0.6390675964355549, 0.6400797119140705, 0.6410919494628986, 0.6421043090820392, 0.6431167297363359, 0.6441292724609452, 0.6451418762207107, 0.6461546020507888, 0.6471674499511794, 0.6481803588867262, 0.6491933288574293, 0.6502064208984449, 0.651219635009773, 0.6522329711914135, 0.6532463684082103, 0.6542598266601632, 0.6552734069824288, 0.6562871093750068, 0.6573009338378973, 0.6583148193359442, 0.6593288269043035, 0.6603428955078191, 0.6613570861816471, 0.6623713378906314, 0.6633857727050844, 0.6644002075195373, 0.6654148254394591, 0.6664295043945371, 0.6674443054199276, 0.6684591674804744, 0.6694741516113337, 0.6704892578125056, 0.6715044860839899, 0.6725197753906306, 0.6735351867675837, 0.6745506591796931, 0.675566253662115, 0.6765819702148493, 0.6775978088378961, 0.6786137084960991, 0.6796297302246147, 0.6806458740234428, 0.6816621398925834, 0.6826784667968803, 0.6836949157714897, 0.6847114257812553, 0.6857281188964898, 0.6867448730468805, 0.6877617492675837, 0.6887786865234432, 0.6897958068847714, 0.6908129882812556, 0.6918302917480524, 0.6928476562500056, 0.6938652038574274, 0.6948828125000056, 0.6959005432128963, 0.6969183959960995, 0.6979363708496152, 0.6989544067382871, 0.6999726257324276, 0.7009909057617244, 0.7020093078613338, 0.7030278320312556, 0.7040464172363338, 0.7050651855468807, 0.7060840148925839, 0.7071029663085996, 0.7081220397949277, 0.7091412353515684, 0.7101605529785214, 0.711179992675787, 0.7121994934082089, 0.7132191772460995, 0.7142389221191464, 0.7152587890625058, 0.7162787780761777, 0.717298889160162, 0.7183191833496151, 0.7193395385742244, 0.72035995483399, 0.7213805541992243, 0.7224012756347712, 0.7234221191406306, 0.7244430847168024, 0.7254641723632868, 0.7264853210449272, 0.7275066528320365, 0.7285281066894583, 0.7295496826171927, 0.7305713195800833, 0.7315931396484426, 0.7326150817871144, 0.7336371459960987, 0.7346593322753955, 0.7356816406250047, 0.7367040710449265, 0.7377266235351608, 0.7387492980957077, 0.7397721557617233, 0.740795074462895, 0.7418181762695356, 0.7428413391113323, 0.7438646850585978, 0.7448881530761758, 0.7459117431640664, 0.7469355163574257, 0.7479593505859413, 0.7489833679199255, 0.7500074462890659, 0.7510317077636751, 0.7520561523437531, 0.7530806579589874, 0.7541053466796904, 0.755130157470706, 0.756155090332034, 0.7571801452636745, 0.7582053833007837, 0.7592307434082054, 0.7602562255859396, 0.7612818908691427, 0.762307617187502, 0.7633335876464861, 0.7643596191406264, 0.7653858337402356, 0.7664121704101573, 0.7674386901855478, 0.7684653320312508, 0.7694920959472663, 0.7705190429687506, 0.7715461120605475, 0.7725733642578131, 0.7736007385253911, 0.7746282348632817, 0.7756559143066409, 0.776683776855469, 0.7777117614746095, 0.7787398681640626, 0.7797681579589845, 0.7807966308593751, 0.7818252258300783, 0.7828540039062502, 0.7838829040527345, 0.7849119873046876, 0.7859411926269532, 0.7869705810546875, 0.7880001525878907, 0.7890299072265625, 0.7900597839355469, 0.7910897827148436, 0.7921200256347655, 0.7931503906249998, 0.794180938720703, 0.7952116699218748, 0.7962425231933591, 0.7972735595703121, 0.7983047790527339, 0.7993361816406245, 0.8003677673339838, 0.8013994750976556, 0.8024314270019525, 0.8034635009765617, 0.8044957580566398, 0.8055281982421866, 0.8065608215332022, 0.8075936279296864, 0.8086266174316393, 0.8096597900390611, 0.8106932067871079, 0.8117267456054672, 0.8127604675292952, 0.813794372558592, 0.8148285217285137, 0.815862792968748, 0.8168973083496073, 0.8179320068359354, 0.8189668884277322, 0.8200019531249979, 0.8210372619628884, 0.8220726928710914, 0.8231084289550757, 0.8241442871093726, 0.8251803894042945, 0.8262166748046852, 0.8272532043457009, 0.8282899169921852, 0.8293268127441383, 0.8303639526367165, 0.8314012756347634, 0.8324388427734353, 0.8334766540527322, 0.8345146484374978, 0.8355528869628884, 0.8365913085937479, 0.8376299743652321, 0.8386688842773414, 0.8397080383300758, 0.8407473754882789, 0.841786956787107, 0.8428267822265602, 0.8438668518066382, 0.8449071655273414, 0.8459476623535132, 0.8469884643554663, 0.8480295104980443, 0.8490707397460912, 0.8501122741699192, 0.8511540527343723, 0.8521960754394503, 0.8532384033203096, 0.8542809143066378, 0.8553237304687472, 0.8563667907714815, 0.8574101562499971, 0.8584537658691377, 0.8594976196289034, 0.8605417785644504, 0.8615862426757785, 0.8626309509277316, 0.863675964355466, 0.8647212219238255, 0.8657667846679661, 0.866812652587888, 0.8678588256835912, 0.8689053039550755, 0.8699520874023411, 0.8709991149902317, 0.8720465087890599, 0.8730942077636693, 0.8741422119140599, 0.8751905212402318, 0.8762391967773412, 0.8772881774902318, 0.8783374633789036, 0.879387115478513, 0.8804371337890599, 0.881487457275388, 0.8825381469726538, 0.8835892028808568, 0.8846405639648413, 0.8856922912597632, 0.886744445800779, 0.887796905517576, 0.8888497924804667, 0.8899030456542949, 0.8909566650390605, 0.89201071166992, 0.8930651245117168, 0.8941199645996075, 0.8951751708984357, 0.8962308654785137, 0.8972869262695294, 0.8983434143066388, 0.899400329589842, 0.900457733154295, 0.9015155639648419, 0.9025738220214824, 0.903632568359373, 0.9046917419433572, 0.9057514648437478, 0.906811614990232, 0.9078722534179663, 0.9089334411621067, 0.9099951171874973, 0.9110572814941379, 0.9121199951171847, 0.9131832580566378, 0.9142470703124973, 0.9153114318847629, 0.9163763427734348, 0.9174418640136691, 0.9185079345703097, 0.9195746765136691, 0.9206419677734347, 0.9217098693847627, 0.9227784423828097, 0.9238476867675752, 0.9249175415039033, 0.9259881286621064, 0.9270593872070282, 0.9281313781738249, 0.9292041015624966, 0.9302774963378871, 0.9313517456054652, 0.9324267272949184, 0.9335025024414026, 0.9345790710449183, 0.9356565551757777, 0.9367348327636683, 0.9378140258789026, 0.9388941345214806, 0.9399751586914024, 0.941057098388668, 0.9421400756835899, 0.9432240905761681, 0.9443091430664025, 0.9453952331542931, 0.9464824218749962, 0.9475708312988242, 0.9486603393554648, 0.9497511291503867, 0.95084320068359, 0.9519365539550743, 0.9530312499999961, 0.9541274108886679, 0.9552249755859336, 0.9563241271972618, 0.9574248046874961, 0.9585271301269492, 0.9596312255859335, 0.9607370910644492, 0.9618448486328085, 0.9629546203613241, 0.964066406249996, 0.9651803894042928, 0.9662966308593709, 0.9674153137206989, 0.9685365600585895, 0.969660491943355, 0.9707873535156205, 0.9719172058105424, 0.9730503540039018, 0.9741870422363237, 0.9753275146484331, 0.976472015380855, 0.9776209716796831, 0.9787746887206988, 0.9799337158203082, 0.9810985412597611, 0.9822698364257768, 0.9834483337402299, 0.9846350097656206, 0.9858310241699174, 0.987037719726558, 0.9882571716308547, 0.9894917602539016, 0.9907451477050735, 0.9920223388671828, 0.9933315124511671, 0.9946865234374952, 0.9961158752441358, 0.9977000732421827],  # skipcq: FLK-E501
                'num_bt = 1000, alpha = 0.05,': [5.1300048828125e-05, 0.00035546875, 0.0008181457519531249, 0.001367431640625, 0.0019721374511718747, 0.002616149902343749, 0.0032897644042968735, 0.003986877441406248, 0.0047030334472656235, 0.005435119628906249, 0.006180877685546875, 0.006938415527343749, 0.007706207275390624, 0.008483215332031249, 0.009268402099609375, 0.010061035156249998, 0.010860321044921874, 0.011665710449218752, 0.012476776123046877, 0.013293029785156254, 0.01411404418945313, 0.014939575195312506, 0.015769256591796878, 0.016602905273437503, 0.017440155029296876, 0.0182808837890625, 0.019124908447265623, 0.01997198486328125, 0.020821990966796877, 0.0216748046875, 0.022530242919921874, 0.02338818359375, 0.024248565673828125, 0.0251112060546875, 0.02597610473632813, 0.02684307861328126, 0.02771206665039063, 0.028583007812500005, 0.029455841064453134, 0.03033050537109376, 0.03120687866210939, 0.03208489990234376, 0.03296456909179689, 0.03384582519531252, 0.0347286071777344, 0.03561279296875002, 0.03649844360351565, 0.03738549804687503, 0.038273895263671906, 0.03916351318359378, 0.04005447387695316, 0.04094659423828129, 0.04183999633789067, 0.04273449707031255, 0.04363015747070318, 0.04452685546875006, 0.04542471313476568, 0.046323608398437556, 0.04722348022460943, 0.048124389648437564, 0.04902621459960944, 0.04992907714843756, 0.0508327941894532, 0.05173748779296882, 0.052643035888671946, 0.053549438476562565, 0.05445669555664069, 0.05536480712890631, 0.05627371215820319, 0.05718347167968756, 0.05809396362304694, 0.05900524902343756, 0.05991732788085944, 0.06083007812500007, 0.06174356079101569, 0.06265783691406254, 0.06357272338867193, 0.06448834228515629, 0.06540463256835943, 0.0663215942382813, 0.06723916625976567, 0.0681574096679688, 0.06907632446289066, 0.06999578857421879, 0.07091592407226566, 0.0718366088867188, 0.07275790405273444, 0.07367980957031256, 0.07460226440429693, 0.07552526855468755, 0.07644888305664069, 0.07737298583984381, 0.07829763793945319, 0.07922283935546881, 0.08014852905273445, 0.08107476806640634, 0.08200155639648449, 0.08292877197265636, 0.08385653686523448, 0.08478479003906261, 0.08571347045898448, 0.08664270019531259, 0.08757235717773446, 0.08850244140625008, 0.08943307495117195, 0.09036407470703131, 0.09129556274414069, 0.09222747802734382, 0.09315982055664071, 0.09409265136718759, 0.09502584838867195, 0.09595947265625007, 0.09689352416992195, 0.09782800292968757, 0.0987628479003907, 0.09969812011718757, 0.1006338195800782, 0.10156982421875008, 0.10250631713867195, 0.10344311523437508, 0.10438034057617196, 0.10531793212890633, 0.10625588989257823, 0.10719415283203138, 0.10813284301757825, 0.10907189941406262, 0.1100113220214845, 0.11095104980468765, 0.11189114379882828, 0.1128316040039064, 0.11377243041992202, 0.11471350097656266, 0.11565499877929702, 0.11659680175781265, 0.11753890991210952, 0.11848138427734389, 0.11942416381835952, 0.1203672485351564, 0.12131063842773454, 0.12225439453125017, 0.12319839477539082, 0.12414276123046897, 0.1250874328613284, 0.12603234863281282, 0.1269776306152347, 0.12792315673828158, 0.12886904907226598, 0.12981518554687538, 0.13076162719726603, 0.13170831298828167, 0.13265536499023484, 0.133602661132813, 0.13455020141601615, 0.13549804687500056, 0.13644619750976617, 0.13739459228515677, 0.13834323120117237, 0.13929223632812548, 0.1402414245605474, 0.14119091796875055, 0.1421406555175787, 0.14309063720703186, 0.14404092407226626, 0.14499145507812566, 0.14594223022461006, 0.14689324951171945, 0.14784451293945383, 0.14879608154296942, 0.1497478332519538, 0.15069989013671942, 0.15165219116211004, 0.15260467529296945, 0.15355746459961006, 0.15451043701171946, 0.15546371459961006, 0.15641717529296945, 0.15737088012695383, 0.1583248291015632, 0.15927902221679757, 0.160233459472657, 0.16118807983398514, 0.1621429443359383, 0.16309805297851643, 0.16405340576171956, 0.16500894165039143, 0.1659647216796883, 0.16692068481445393, 0.16787689208984458, 0.1688333435058602, 0.16978997802734463, 0.1707467956542978, 0.1717039184570322, 0.17266116333007908, 0.17361865234375096, 0.17457638549804783, 0.17553430175781348, 0.17649240112304787, 0.1774507446289073, 0.17840927124023548, 0.17936804199218864, 0.18032699584961054, 0.18128613281250117, 0.18224545288086058, 0.183205017089845, 0.18416476440429813, 0.18512469482422, 0.1860848083496106, 0.1870451660156262, 0.1880057067871106, 0.1889664306640637, 0.18992733764648562, 0.19088842773437625, 0.19184976196289188, 0.19281121826172004, 0.1937729187011732, 0.1947347412109388, 0.19569680786132942, 0.19665905761718883, 0.19762149047851696, 0.19858404541015756, 0.19954684448242316, 0.20050982666015754, 0.20147299194336066, 0.20243627929687624, 0.20339981079101682, 0.20436346435546993, 0.20532736206054802, 0.20629138183593865, 0.20725558471679806, 0.20822003173828246, 0.20918460083007934, 0.21014929199218874, 0.21111422729492313, 0.21207928466797, 0.21304458618164185, 0.21401000976562623, 0.21497561645507934, 0.21594134521484498, 0.21690731811523561, 0.21787341308593877, 0.21883969116211066, 0.21980609130859508, 0.22077267456054822, 0.2217394409179701, 0.2227063903808607, 0.22367346191406384, 0.2246407165527357, 0.2256081542968763, 0.22657571411132943, 0.22754345703125134, 0.22851138305664198, 0.22947943115234515, 0.23044766235351705, 0.23141601562500147, 0.23238455200195463, 0.2333532714843765, 0.23432211303711092, 0.2352910766601578, 0.23626028442382968, 0.23722955322265782, 0.23819906616211095, 0.23916864013672035, 0.24013845825195473, 0.24110833740234539, 0.24207846069336103, 0.24304864501953294, 0.24401901245117358, 0.244989562988283, 0.2459602355957049, 0.24693109130859553, 0.24790206909179868, 0.2488731689453143, 0.24984445190429871, 0.25081585693359565, 0.25178744506836126, 0.2527591552734394, 0.25373098754883006, 0.2547030029296894, 0.25567514038086125, 0.2566474609375019, 0.2576198425292988, 0.25859246826172066, 0.2595651550292988, 0.2605380249023457, 0.261511016845705, 0.2624841918945331, 0.26345742797851746, 0.2644308471679705, 0.2654044494628923, 0.2663781738281267, 0.2673519592285173, 0.26832598876953284, 0.26930007934570466, 0.27027435302734526, 0.2712487487792984, 0.27222326660156404, 0.27319796752929837, 0.27417272949218896, 0.27514767456054834, 0.27612274169922024, 0.27709799194336093, 0.2780733032226579, 0.2790487976074235, 0.28002441406250167, 0.28100015258789235, 0.2819760742187517, 0.2829520568847673, 0.2839282226562517, 0.28490451049804866, 0.285880920410158, 0.28685745239257987, 0.28783410644531426, 0.28881094360351733, 0.28978784179687667, 0.2907649230957048, 0.29174212646484543, 0.2927194519042985, 0.29369689941406407, 0.2946744689941422, 0.2956521606445328, 0.29662997436523586, 0.2976079711914077, 0.2985860290527358, 0.2995642700195327, 0.3005426330566421, 0.301521118164064, 0.3024996643066421, 0.303478393554689, 0.30445724487304837, 0.3054362182617203, 0.30641531372070474, 0.3073945312500017, 0.3083738708496111, 0.30935339355468927, 0.3103329772949237, 0.31131268310547067, 0.31229251098633004, 0.3132725219726582, 0.3142525939941426, 0.3152327880859396, 0.31621310424804894, 0.3171936035156271, 0.3181741638183615, 0.31915484619140844, 0.3201356506347678, 0.3211166381835959, 0.3220976867675803, 0.32307885742187725, 0.3240601501464866, 0.32504156494140846, 0.32602310180664285, 0.32700476074218976, 0.3279865417480491, 0.32896844482422094, 0.3299504699707053, 0.3309326171875022, 0.33191488647461165, 0.33289721679687734, 0.3338797302246117, 0.3348623657226586, 0.33584506225586175, 0.3368278808593774, 0.3378108825683619, 0.3387939453125026, 0.33977713012695576, 0.34076043701172143, 0.3417438659667996, 0.34272741699219034, 0.34371109008789347, 0.34469482421875286, 0.34567874145508104, 0.3466627197265655, 0.34764682006836245, 0.3486311035156281, 0.34961544799805, 0.35059991455078443, 0.3515844421386751, 0.3525691528320345, 0.3535539855957064, 0.35453887939453455, 0.3555238952636752, 0.3565090332031283, 0.35749429321289394, 0.3584796752929721, 0.35946517944336276, 0.3604507446289097, 0.3614364929199253, 0.36242230224609717, 0.36340823364258157, 0.3643942871093785, 0.36538046264648794, 0.36636669921875364, 0.36735311889648803, 0.3683395996093787, 0.36932620239258185, 0.37031292724609755, 0.3712997131347695, 0.37228668212891025, 0.37327371215820726, 0.3742608642578167, 0.37524813842773863, 0.3762355346679731, 0.37722299194336384, 0.37821063232422325, 0.3791983337402389, 0.3801861572265671, 0.38117410278320785, 0.38216210937500483, 0.38315023803711423, 0.3841385498046924, 0.3851268615722706, 0.3861153564453176, 0.38710397338867697, 0.3880926513671926, 0.3890814514160208, 0.3900703735351615, 0.39105935668945846, 0.3920485229492241, 0.393037750244146, 0.3940270996093804, 0.3950165100097711, 0.3960061035156305, 0.3969957580566461, 0.39798553466797426, 0.39897543334961494, 0.3999653930664119, 0.40095547485352123, 0.4019456787109431, 0.4029360046386775, 0.40392645263672444, 0.4049169616699275, 0.4059075927734431, 0.40689834594727126, 0.40788916015625565, 0.4088801574707087, 0.40987121582031805, 0.41086233520508364, 0.411853637695318, 0.41284500122070866, 0.4138364868164117, 0.4148280944824273, 0.41581976318359914, 0.4168115539550835, 0.4178034667968804, 0.4187955017089898, 0.4197875976562555, 0.42077987670898986, 0.4217721557617242, 0.42276461791992737, 0.4237571411132868, 0.4247497863769587, 0.42574255371094316, 0.42673544311524014, 0.4277283935546933, 0.42872146606445893, 0.4297146606445371, 0.43070791625977156, 0.43170129394531853, 0.4326947937011779, 0.43368835449219356, 0.434682098388678, 0.4356759033203187, 0.43666976928711565, 0.4376638183593813, 0.4386579284668032, 0.4396521606445376, 0.4406464538574283, 0.44164093017578765, 0.4426354675293033, 0.44363006591797516, 0.44462484741211583, 0.44561968994141277, 0.4466146545410221, 0.4476096801757877, 0.44860482788086586, 0.4496000976562565, 0.4505954895019596, 0.45159094238281894, 0.4525865173339908, 0.4535822143554752, 0.4545780334472721, 0.45557391357422516, 0.45656991577149075, 0.4575659790039126, 0.45856222534180324, 0.45955853271485014, 0.4605549011230532, 0.46155145263672503, 0.46254806518555314, 0.46354479980469376, 0.46454159545899065, 0.4655385742187562, 0.4665355529785218, 0.46753271484375614, 0.46852993774414675, 0.4695272827148498, 0.47052474975586533, 0.4715223388671934, 0.47251998901367775, 0.4735177612304746, 0.47451559448242775, 0.4755135498046934, 0.47651162719727147, 0.47750982666016206, 0.4785080871582089, 0.4795064697265683, 0.4805049743652402, 0.48150354003906837, 0.4825022888183653, 0.4835010375976623, 0.4844999694824279, 0.4854989624023498, 0.4864980773925842, 0.48749731445313116, 0.48849661254883425, 0.48949603271484987, 0.490495574951178, 0.4914951782226624, 0.4924949645996155, 0.4934947509765686, 0.49449472045899046, 0.4954947509765686, 0.49649490356445924, 0.4974951782226623, 0.49849551391602165, 0.49949603271484977, 0.5004965515136779, 0.5014972534179748, 0.5024980163574279, 0.5034989013671934, 0.5044999084472716, 0.5055009765625059, 0.5065021667480528, 0.5075034790039122, 0.5085048522949279, 0.5095064086914122, 0.5105080261230527, 0.5115097045898495, 0.512511566162115, 0.5135134887695368, 0.5145154724121149, 0.5155176391601618, 0.5165198669433649, 0.5175222167968806, 0.5185246887207087, 0.519527221679693, 0.5205298767089898, 0.521532653808599, 0.5225355529785208, 0.5235385131835989, 0.5245415954589895, 0.5255447998046926, 0.526548065185552, 0.5275514526367239, 0.5285549621582083, 0.5295585937500052, 0.5305622863769585, 0.5315661010742242, 0.5325700378418023, 0.533574096679693, 0.5345782165527398, 0.5355824584960992, 0.5365868225097711, 0.5375913085937555, 0.5385958557128961, 0.5396005249023493, 0.540605316162115, 0.541610168457037, 0.5426152038574278, 0.5436203002929748, 0.5446255187988344, 0.5456307983398502, 0.5466361999511784, 0.5476417236328189, 0.5486473693847721, 0.5496531372070378, 0.5506589660644597, 0.5516649169921942, 0.5526709899902412, 0.5536771850586008, 0.5546834411621165, 0.5556898193359447, 0.5566963195800853, 0.5577029418945384, 0.5587096252441478, 0.559716491699226, 0.5607234191894604, 0.5617304077148512, 0.5627375793457107, 0.5637448120117264, 0.5647522277832108, 0.5657597045898514, 0.5667672424316482, 0.5677749633789139, 0.5687827453613358, 0.5697906494140702, 0.5707986755371172, 0.5718068237304766, 0.5728150329589924, 0.5738234252929767, 0.5748318786621173, 0.5758404541015704, 0.576849151611336, 0.5778579101562579, 0.5788668518066485, 0.5798758544921955, 0.5808849792480549, 0.5818942260742268, 0.5829035339355548, 0.5839130249023515, 0.5849225769043046, 0.5859322509765702, 0.5869420471191483, 0.587951965332039, 0.5889620056152421, 0.5899721679687578, 0.5909823913574297, 0.5919927368164138, 0.5930032043457106, 0.59401379394532, 0.5950245056152418, 0.5960353393554761, 0.5970462341308668, 0.5980573120117262, 0.5990684509277419, 0.60007971191407, 0.6010910949707106, 0.6021026000976636, 0.6031142272949291, 0.604125915527351, 0.6051377868652416, 0.6061497192382884, 0.6071618347168041, 0.6081740112304759, 0.6091863098144602, 0.610198730468757, 0.6112112731933662, 0.612223937988288, 0.6132366638183661, 0.6142495727539129, 0.6152626037597723, 0.6162756958007879, 0.6172889709472721, 0.6183023071289125, 0.6193157653808655, 0.6203294067382873, 0.6213431091308653, 0.6223569335937559, 0.623370880126959, 0.6243849487304746, 0.6253991394043026, 0.6264134521484431, 0.6274278869628961, 0.6284424438476616, 0.6294571228027396, 0.6304719238281301, 0.6314867858886769, 0.6325018310546925, 0.6335169982910206, 0.6345322875976612, 0.6355476989746144, 0.6365631713867238, 0.6375788269043019, 0.6385946044921925, 0.6396105041503956, 0.6406265258789111, 0.6416426086425829, 0.6426588745117235, 0.6436752624511767, 0.6446917724609423, 0.6457084045410205, 0.6467251586914112, 0.6477420349121145, 0.6487590942382864, 0.6497762145996144, 0.6507934570312549, 0.6518108825683643, 0.6528283691406299, 0.6538459777832081, 0.654863769531255, 0.6558816833496145, 0.6568997192382864, 0.6579178161621144, 0.6589361572265674, 0.6599545593261767, 0.6609730834960986, 0.661991729736333, 0.6630105590820362, 0.6640294494628957, 0.6650485229492238, 0.6660677185058644, 0.6670870361328175, 0.6681064758300831, 0.6691260986328175, 0.6701457824707081, 0.6711656494140675, 0.6721856384277394, 0.6732057495117237, 0.6742259826660205, 0.6752463989257861, 0.6762669372558642, 0.6772875366210985, 0.6783083801269579, 0.6793292846679736, 0.6803503112793017, 0.6813715209960985, 0.6823928527832078, 0.6834143676757859, 0.6844359436035202, 0.6854577026367233, 0.686479583740239, 0.6875016479492232, 0.6885237731933637, 0.6895460815429729, 0.690568572998051, 0.6915911254882853, 0.6926138610839884, 0.693636718750004, 0.6946597595214882, 0.6956829223632849, 0.6967062072753941, 0.6977296142578159, 0.6987532043457064, 0.6997769775390658, 0.7008008117675814, 0.7018248291015655, 0.7028490295410185, 0.7038733520507839, 0.7048977966308619, 0.7059224243164087, 0.706947174072268, 0.7079720458984398, 0.7089971008300803, 0.7100223388671895, 0.7110476989746112, 0.7120731811523454, 0.7130988464355484, 0.714124633789064, 0.7151506042480481, 0.7161766967773447, 0.7172029724121102, 0.7182293701171881, 0.7192559509277349, 0.7202827148437504, 0.7213096008300784, 0.722336608886719, 0.7233638000488284, 0.7243911743164064, 0.725418670654297, 0.7264463500976562, 0.7274742126464843, 0.7285021972656248, 0.7295303649902342, 0.730558654785156, 0.7315871276855467, 0.7326157836914061, 0.733644561767578, 0.7346735229492186, 0.7357026672363279, 0.736731994628906, 0.7377614440917967, 0.7387910766601561, 0.7398208923339842, 0.7408508300781248, 0.7418809509277341, 0.7429112548828122, 0.743941741943359, 0.7449724121093747, 0.7460032043457028, 0.7470341796874996, 0.7480653381347652, 0.7490966796874995, 0.7501282043457026, 0.7511599121093744, 0.752191802978515, 0.7532238159179679, 0.754256072998046, 0.7552884521484365, 0.7563210144042959, 0.7573538208007801, 0.7583867492675768, 0.7594198608398423, 0.7604532165527328, 0.7614866943359359, 0.7625203552246076, 0.7635542602539044, 0.7645882873535136, 0.7656225585937478, 0.7666569519042946, 0.7676915893554664, 0.7687264099121071, 0.7697614135742165, 0.7707966613769507, 0.7718320312499974, 0.7728676452636692, 0.7739034423828098, 0.7749394226074192, 0.7759755859374974, 0.7770119934082006, 0.7780485839843725, 0.7790853576660132, 0.7801223754882788, 0.7811595153808569, 0.7821969604492164, 0.7832345275878883, 0.7842723999023414, 0.785310394287107, 0.7863486328124977, 0.7873870544433571, 0.7884257202148414, 0.7894646301269507, 0.7905037231445288, 0.7915429992675757, 0.7925825195312475, 0.7936222839355443, 0.7946622314453099, 0.7957024230957005, 0.7967428588867161, 0.7977834777832004, 0.7988243408203096, 0.7998654479980439, 0.800906738281247, 0.8019482727050751, 0.8029901123046844, 0.8040320739746062, 0.8050743408203093, 0.8061168518066374, 0.8071595458984343, 0.8082025451660124, 0.8092457275390593, 0.8102892150878874, 0.8113328857421843, 0.8123768615722625, 0.8134210205078095, 0.8144654846191377, 0.8155101928710907, 0.8165551452636689, 0.8176003417968721, 0.8186458435058565, 0.8196915283203096, 0.820737518310544, 0.8217838134765597, 0.822830291748044, 0.8238770751953096, 0.8249241638183565, 0.8259714965820284, 0.8270190734863253, 0.8280669555664034, 0.8291151428222628, 0.8301635742187471, 0.8312123107910127, 0.8322612915039033, 0.8333106384277313, 0.8343602294921845, 0.8354100646972625, 0.8364602661132782, 0.8375107727050751, 0.838561523437497, 0.8396125793457002, 0.8406640014648409, 0.8417156677246066, 0.8427677001953099, 0.8438200378417943, 0.84487268066406, 0.845925628662107, 0.8469788818359352, 0.8480325012207007, 0.8490864257812477, 0.850140716552732, 0.8511953124999977, 0.8522502746582009, 0.8533055419921852, 0.8543611755371071, 0.8554171752929666, 0.8564734802246071, 0.8575302124023415, 0.8585872497558572, 0.8596446533203103, 0.8607024230957009, 0.8617606201171851, 0.8628191223144507, 0.8638780517578101, 0.8649373474121068, 0.8659970092773411, 0.8670570983886692, 0.8681175537109348, 0.8691784362792941, 0.8702396850585908, 0.8713013610839814, 0.8723634643554657, 0.8734259948730438, 0.8744889526367157, 0.8755523376464812, 0.8766161499023407, 0.8776803894042938, 0.8787450561523408, 0.8798102111816376, 0.8808758544921845, 0.8819419250488251, 0.8830084838867157, 0.8840754699707, 0.8851430053710906, 0.8862109680175748, 0.8872794799804654, 0.8883484802246059, 0.8894179687499965, 0.8904880065917932, 0.89155853271484, 0.8926296081542929, 0.8937012329101522, 0.8947733459472614, 0.8958460693359331, 0.8969194030761675, 0.8979932250976518, 0.8990676574706985, 0.9001427001953078, 0.9012183532714797, 0.9022945556640579, 0.9033714294433547, 0.9044489135742141, 0.9055270690917921, 0.9066058349609326, 0.9076853332519482, 0.9087654418945262, 0.9098462829589792, 0.910927795410151, 0.9120100402831978, 0.9130929565429634, 0.9141766662597602, 0.9152611694335884, 0.9163464050292917, 0.9174323730468698, 0.9185191955566353, 0.9196068725585883, 0.9206953430175725, 0.9217846679687444, 0.9228748474121037, 0.9239659423828069, 0.9250579528808539, 0.9261508789062446, 0.927244720458979, 0.9283395996093696, 0.9294354553222602, 0.9305323486328071, 0.9316302795410102, 0.9327292480468697, 0.933829376220698, 0.9349306030273384, 0.9360329895019478, 0.9371365966796822, 0.9382414245605416, 0.9393475341796821, 0.9404549255371039, 0.9415635986328069, 0.9426737365722601, 0.943785278320307, 0.9448982849121038, 0.946012817382807, 0.9471288757324162, 0.9482465820312442, 0.9493659973144474, 0.9504871215820254, 0.9516100769042911, 0.9527348632812441, 0.9538616638183534, 0.954990478515619, 0.9561213684081972, 0.9572545166015566, 0.958389923095697, 0.9595277709960878, 0.9606681213378846, 0.9618111572265564, 0.9629569396972596, 0.9641056518554628, 0.9652574768066347, 0.966412597656244, 0.9675711364746035, 0.9687333984374941, 0.9698995056152283, 0.9710698852539001, 0.9722447204589783, 0.9734243774414001, 0.9746092224121033, 0.9757997436523377, 0.9769964294433534, 0.9781998901367127, 0.9794107360839782, 0.9806298217773376, 0.9818581237792906, 0.9830968017578062, 0.9843474426269468, 0.9856117553710874, 0.9868923034667906, 0.988192199707025, 0.9895158996581969, 0.9908700561523373, 0.9922647399902279, 0.9937177124023372, 0.9952649841308528, 0.9970087280273371],  # skipcq: FLK-E501
                'num_bt = 1000, alpha = 0.025': [2.5299072265625002e-05, 0.00024230957031250004, 0.0006191101074218749, 0.0010908813476562499, 0.0016253967285156249, 0.0022049560546874994, 0.002818878173828124, 0.0034599609374999984, 0.004123382568359373, 0.004805480957031248, 0.005503570556640623, 0.006215515136718748, 0.006939605712890622, 0.007674499511718747, 0.008419036865234372, 0.009172302246093746, 0.00993350219726562, 0.010701965332031247, 0.011477020263671874, 0.01225823974609375, 0.013045196533203126, 0.013837402343750003, 0.014634552001953127, 0.015436401367187505, 0.01624252319335938, 0.01705279541015626, 0.017866912841796888, 0.018684692382812514, 0.01950595092773439, 0.020330505371093766, 0.021158172607421893, 0.02198883056640627, 0.022822357177734398, 0.023658630371093774, 0.0244975280761719, 0.025338928222656272, 0.0261827087402344, 0.02702880859375003, 0.027877166748046904, 0.028727600097656278, 0.029580169677734407, 0.03043469238281253, 0.03129116821289066, 0.03214947509765629, 0.03300961303710941, 0.033871520996093786, 0.03473507690429691, 0.03560028076171879, 0.03646707153320316, 0.03733538818359378, 0.038205230712890656, 0.039076538085937536, 0.03994924926757815, 0.04082336425781253, 0.0416988220214844, 0.04257556152343753, 0.0434535827636719, 0.04433288574218753, 0.04521334838867191, 0.046095031738281285, 0.04697787475585941, 0.04786187744140629, 0.048746917724609415, 0.04963311767578129, 0.05052029418945316, 0.051408569335937536, 0.052297821044921915, 0.05318811035156253, 0.05407931518554691, 0.054971496582031286, 0.055864654541015656, 0.05675866699218753, 0.057653594970703166, 0.05854943847656254, 0.05944613647460942, 0.06034368896484379, 0.06124203491210941, 0.06214123535156253, 0.06304122924804689, 0.06394195556640625, 0.06484353637695311, 0.06574584960937499, 0.06664895629882814, 0.067552734375, 0.06845730590820315, 0.06936254882812501, 0.07026846313476565, 0.07117510986328127, 0.0720824890136719, 0.07299047851562504, 0.07389913940429693, 0.07480847167968757, 0.0757184143066407, 0.07662896728515634, 0.0775401916503907, 0.07845202636718757, 0.07936447143554692, 0.0802775268554688, 0.08119113159179692, 0.08210534667968755, 0.08302011108398444, 0.08393548583984381, 0.08485134887695317, 0.08576782226562504, 0.08668484497070317, 0.08760235595703128, 0.08852041625976564, 0.08943902587890629, 0.09035812377929692, 0.0912777709960938, 0.09219790649414067, 0.09311853027343756, 0.09403964233398443, 0.09496130371093756, 0.09588339233398444, 0.0968059692382813, 0.09772897338867192, 0.0986525268554688, 0.09957650756835942, 0.1005009155273438, 0.10142581176757817, 0.1023511352539063, 0.10327688598632817, 0.10420312500000006, 0.10512979125976571, 0.10605682373046885, 0.10698434448242197, 0.10791223144531262, 0.10884054565429699, 0.10976928710937511, 0.11069845581054699, 0.11162799072265636, 0.11255795288085949, 0.11348828125000013, 0.1144190368652345, 0.11535015869140638, 0.11628170776367203, 0.1172135620117189, 0.11814584350585955, 0.11907849121093769, 0.12001150512695333, 0.12094488525390645, 0.1218786315917971, 0.12281274414062524, 0.12374722290039086, 0.12468206787109398, 0.1256172180175784, 0.1265527343750003, 0.12748861694335972, 0.12842480468750034, 0.12936135864257847, 0.13029827880859407, 0.13123544311523466, 0.13217303466796904, 0.13311093139648467, 0.13404913330078155, 0.1349876403808597, 0.1359265136718753, 0.13686569213867217, 0.1378051757812503, 0.13874496459960967, 0.1396851196289065, 0.1406255187988284, 0.1415662841796878, 0.1425072937011722, 0.14344866943359408, 0.14439028930664094, 0.14533227539062532, 0.1462745056152347, 0.14721704101562533, 0.14815982055664095, 0.1491029663085941, 0.15004635620117224, 0.15099005126953163, 0.15193405151367223, 0.15287829589843788, 0.15382284545898478, 0.15476763916015668, 0.15571273803710983, 0.15665808105468798, 0.1576037292480474, 0.158549682617188, 0.1594958190917974, 0.16044226074218804, 0.16138900756835994, 0.16233599853515684, 0.16328323364257874, 0.16423071289062563, 0.16517849731445378, 0.16612652587890692, 0.16707479858398505, 0.16802331542968818, 0.1689720764160163, 0.16992114257812568, 0.17087045288086006, 0.17181994628906322, 0.17276974487304764, 0.17371978759765705, 0.17467007446289146, 0.1756205444335946, 0.176571319580079, 0.17752233886718838, 0.17847360229492276, 0.17942504882812588, 0.180376739501954, 0.18132873535156335, 0.18228091430664145, 0.1832333374023446, 0.18418594360351648, 0.1851388549804696, 0.18609194946289148, 0.18704528808593834, 0.1879988708496102, 0.1889526367187508, 0.18990664672851643, 0.19086090087890706, 0.19181533813476648, 0.1927700195312509, 0.1937249450683603, 0.19468005371093844, 0.1956353454589853, 0.19659094238281344, 0.1975467224121103, 0.19850268554687595, 0.1994588928222666, 0.20041528320312602, 0.20137191772461044, 0.2023287353515636, 0.20328579711914174, 0.20424304199218862, 0.20520046997070424, 0.20615814208984484, 0.20711599731445424, 0.20807409667968862, 0.2090323791503918, 0.2099908447265637, 0.21094949340820435, 0.21190838623046998, 0.2128674621582044, 0.21382678222656382, 0.21478628540039196, 0.21574591064453258, 0.21670584106445445, 0.21766589355468885, 0.21862612915039198, 0.2195866088867201, 0.22054727172851696, 0.2215081176757826, 0.22246914672851698, 0.2234304199218764, 0.2243918151855483, 0.22535345458984518, 0.2263152770996108, 0.22727722167968895, 0.22823941040039208, 0.22920178222656395, 0.23016433715820456, 0.23112707519531395, 0.23208999633789207, 0.23305310058593892, 0.2340163879394545, 0.23497985839843888, 0.23594351196289198, 0.23690734863281387, 0.2378713684082045, 0.2388355712890639, 0.23979995727539205, 0.24076452636718892, 0.24172921752929832, 0.24269415283203272, 0.24365921020507958, 0.24462451171875144, 0.24558993530273582, 0.24655554199218893, 0.24752133178711083, 0.24848730468750146, 0.24945346069336088, 0.2504197387695327, 0.25138626098632966, 0.252352905273439, 0.25331973266601715, 0.2542866821289078, 0.2552538757324234, 0.25622119140625155, 0.25718869018554846, 0.25815637207031406, 0.25912417602539217, 0.26009222412109523, 0.2610603942871108, 0.2620286865234389, 0.26299722290039207, 0.26396588134765775, 0.2649347229003921, 0.265903686523439, 0.2668728942871109, 0.2678422241210954, 0.26881167602539235, 0.269781311035158, 0.27075112915039246, 0.2717211303710956, 0.2726912536621112, 0.27366156005859565, 0.2746319885253926, 0.27560260009765825, 0.27657339477539267, 0.2775443115234396, 0.27851541137695524, 0.2794866333007834, 0.2804580383300803, 0.28142956542968967, 0.2824012756347678, 0.2833731689453147, 0.28434518432617406, 0.2853173828125022, 0.28628970336914283, 0.28726220703125216, 0.288234832763674, 0.28920764160156465, 0.29018063354492396, 0.29115368652343954, 0.29212698364258016, 0.2931004028320332, 0.29407394409179877, 0.2950476684570331, 0.29602151489258, 0.29699554443359555, 0.2979696960449236, 0.2989440307617205, 0.2999184875488299, 0.3008930664062518, 0.3018678283691424, 0.30284277343750177, 0.3038177795410174, 0.3047930297851581, 0.30576834106445505, 0.3067438354492207, 0.3077195129394551, 0.30869531250000204, 0.3096712341308614, 0.31064733886718954, 0.3116235656738302, 0.3125999145507833, 0.31357644653320516, 0.31455310058593955, 0.3155299377441426, 0.3165068969726582, 0.31748397827148633, 0.31846124267578324, 0.3194385681152363, 0.3204161376953144, 0.3213937683105488, 0.3223715820312518, 0.32334957885742366, 0.32432763671875176, 0.32530587768554864, 0.32628424072265805, 0.32726278686523613, 0.32824145507812674, 0.3292202453613299, 0.33019915771484554, 0.33117825317383, 0.33215747070312696, 0.33313681030273634, 0.3341163330078145, 0.3350959777832052, 0.3360757446289083, 0.33705563354492396, 0.3380357055664084, 0.33901589965820533, 0.3399962158203147, 0.34097671508789285, 0.34195727539062726, 0.34293801879883035, 0.34391888427734596, 0.34489993286133036, 0.345881042480471, 0.34686233520508036, 0.3478437500000022, 0.3488253479003929, 0.3498070068359398, 0.3507888488769554, 0.3517708129882835, 0.35275289916992414, 0.3537351074218773, 0.35471749877929926, 0.3557000122070336, 0.3566826477050805, 0.35766540527343993, 0.35864828491211187, 0.3596313476562526, 0.36061453247070574, 0.3615978393554714, 0.3625812683105496, 0.3635648193359403, 0.36454849243164344, 0.36553234863281536, 0.3665163269042998, 0.3675004272460968, 0.36848464965820615, 0.36946899414062806, 0.3704534606933625, 0.3714381103515656, 0.37242288208008123, 0.3734077148437531, 0.3743927307128938, 0.37537792968750316, 0.3763631896972688, 0.37734857177734693, 0.37833413696289386, 0.37931976318359695, 0.3803055725097688, 0.3812915039062532, 0.38227755737305014, 0.3832637329101596, 0.3842500915527377, 0.3852365112304721, 0.38622311401367526, 0.3872097778320347, 0.3881966247558629, 0.38918359375000366, 0.3901706848144568, 0.3911578979492225, 0.3921452331543007, 0.39313269042969146, 0.39412033081055087, 0.39510803222656654, 0.396095916748051, 0.397083923339848, 0.3980719909668011, 0.39906024169922305, 0.4000486145019575, 0.4010371093750045, 0.40202572631836386, 0.4030144653320358, 0.4040033874511765, 0.40499237060547344, 0.4059815368652391, 0.406970764160161, 0.40796017456055167, 0.4089496459960986, 0.40993930053711425, 0.4109290771484424, 0.4119189758300831, 0.41290899658203617, 0.4138991394043018, 0.4148894042968799, 0.4158797912597706, 0.4168703002929737, 0.4178609313964893, 0.4188516845703174, 0.4198426208496143, 0.4208336181640674, 0.42182479858398925, 0.42281604003906736, 0.42380746459961427, 0.4247989501953173, 0.42579061889648917, 0.42678240966797354, 0.42777432250977043, 0.4287662963867236, 0.42975845336914553, 0.4307507324218799, 0.43174313354492677, 0.4327356567382862, 0.4337283020019581, 0.43472106933594257, 0.43571395874023955, 0.43670697021484894, 0.4377001647949271, 0.43869342041016157, 0.43968679809570854, 0.4406802978515679, 0.4416739807128961, 0.44266772460938053, 0.44366165161133364, 0.444655639648443, 0.4456498107910212, 0.4466440429687556, 0.4476384582519587, 0.44863293457031805, 0.4496275939941462, 0.4506223144531306, 0.4516172180175837, 0.4526122436523493, 0.4536073303222712, 0.45460260009766185, 0.4555979919433649, 0.4565935058593805, 0.45758914184570865, 0.4585848999023493, 0.4595807189941462, 0.4605767211914118, 0.4615728454589899, 0.46256909179688055, 0.4635654602050836, 0.4645619506835992, 0.4655585632324273, 0.4665552978515679, 0.4675522155761772, 0.4685491943359428, 0.46954629516602087, 0.4705435180664115, 0.4715408630371145, 0.47253833007813006, 0.4735359802246144, 0.474533691406255, 0.47553152465820814, 0.47652954101563005, 0.4775276184082081, 0.4785258178710987, 0.4795242004394581, 0.48052264404297373, 0.48152127075195816, 0.48251995849609886, 0.4835188293457082, 0.48451776123047385, 0.48551687622070827, 0.4865161132812552, 0.4875154113769583, 0.4885148925781302, 0.4895144958496146, 0.49051416015625526, 0.4915140075683646, 0.4925139770507865, 0.4935140686035209, 0.4945142822265678, 0.49551455688477086, 0.4965150146484427, 0.4975155944824271, 0.498516296386724, 0.49951712036133333, 0.5005180664062553, 0.5015191955566459, 0.5025203857421928, 0.5035216979980522, 0.5045231323242241, 0.5055246887207084, 0.5065264282226615, 0.5075282287597708, 0.5085302124023489, 0.5095322570800832, 0.5105344238281301, 0.5115367736816457, 0.5125392456054738, 0.5135417785644582, 0.5145444946289112, 0.5155473327636767, 0.5165502319335985, 0.5175533142089891, 0.5185565185546922, 0.5195598449707078, 0.520563293457036, 0.5215668640136767, 0.5225705566406299, 0.5235744323730518, 0.5245783691406299, 0.5255824279785205, 0.5265866699218799, 0.5275909729003955, 0.5285954589843799, 0.5296000061035205, 0.53060473632813, 0.531609588623052, 0.5326145019531302, 0.5336195983886772, 0.5346248168945368, 0.5356301574707087, 0.536635620117193, 0.5376412658691462, 0.5386469726562556, 0.5396528015136776, 0.5406588134765683, 0.5416648864746153, 0.5426711425781311, 0.5436775207519593, 0.5446839599609435, 0.5456905822753967, 0.5466973266601624, 0.5477042541503968, 0.5487112426757875, 0.5497183532714908, 0.5507256469726627, 0.5517330017089908, 0.5527405395507875, 0.5537481994628969, 0.5547559204101625, 0.555763824462897, 0.5567719116211002, 0.5577800598144597, 0.5587883300781316, 0.559796783447272, 0.5608052978515689, 0.5618139953613345, 0.5628228149414126, 0.5638317565918033, 0.5648408203125065, 0.5658500671386784, 0.5668593750000064, 0.5678688659668031, 0.5688784179687562, 0.569888153076178, 0.5708980102539124, 0.5719080505371156, 0.572918151855475, 0.5739284362793031, 0.5749387817382873, 0.5759493103027403, 0.5769599609375059, 0.5779707946777403, 0.5789816894531309, 0.5799927673339903, 0.581003906250006, 0.5820152282714902, 0.583026672363287, 0.5840382995605525, 0.5850499877929743, 0.586061859130865, 0.5870738525390681, 0.5880859680175837, 0.5890982055664118, 0.5901106262207086, 0.5911231689453179, 0.5921358337402397, 0.593148620605474, 0.5941615905761771, 0.5951746215820365, 0.5961878356933645, 0.5972011718750051, 0.5982146911621143, 0.5992282714843798, 0.6002420349121141, 0.601255920410161, 0.6022699890136766, 0.6032841186523483, 0.6042984313964889, 0.6053128662109418, 0.6063274841308636, 0.6073422241210978, 0.6083570861816446, 0.609372070312504, 0.6103872375488321, 0.6114024658203164, 0.6124179382324259, 0.6134334716796915, 0.6144491882324258, 0.6154650268554726, 0.6164809875488318, 0.6174971313476599, 0.6185133972168005, 0.6195297851562537, 0.6205463562011756, 0.62156304931641, 0.622579864501957, 0.6235968627929728, 0.624613983154301, 0.6256312866210977, 0.6266486511230508, 0.627666259765629, 0.6286839294433634, 0.6297017822265666, 0.6307197570800823, 0.6317379150390666, 0.6327561950683634, 0.6337745971679728, 0.6347931823730509, 0.6358118896484416, 0.636830780029301, 0.6378497924804729, 0.6388689880371136, 0.6398882446289104, 0.6409077453613322, 0.6419273681640666, 0.6429471130371135, 0.6439670410156292, 0.6449870910644573, 0.6460072631835979, 0.6470276184082072, 0.6480481567382853, 0.6490688171386759, 0.6500895996093791, 0.6511105651855509, 0.6521317138671915, 0.6531529846191445, 0.65417437744141, 0.6551959533691444, 0.6562177124023475, 0.6572395935058631, 0.6582615966796912, 0.6592838439941443, 0.6603061523437536, 0.6613287048339879, 0.6623513183593784, 0.6633741760253941, 0.6643971557617221, 0.6654202575683626, 0.6664435424804719, 0.6674670104980499, 0.6684906005859405, 0.6695143737792998, 0.6705383300781278, 0.6715624084472683, 0.6725866699218775, 0.6736110534667993, 0.6746356201171898, 0.6756603698730491, 0.6766853027343771, 0.6777103576660176, 0.6787355957031268, 0.6797609558105485, 0.680786499023439, 0.6818122253417983, 0.6828381347656263, 0.6838641662597668, 0.6848903808593759, 0.6859167785644539, 0.6869432983398444, 0.6879700622558599, 0.6889969482421879, 0.6900239562988285, 0.6910512084960941, 0.6920785827636722, 0.6931062011718753, 0.6941339416503908, 0.6951618652343751, 0.6961899108886719, 0.6972182006835937, 0.6982466125488281, 0.6992752075195313, 0.7003039855957032, 0.7013329467773438, 0.7023620910644531, 0.7033914184570312, 0.7044208679199219, 0.7054505615234375, 0.7064804382324218, 0.7075104370117186, 0.7085406188964841, 0.7095710449218747, 0.7106015930175779, 0.7116323242187497, 0.7126632995605465, 0.7136943969726558, 0.7147256774902339, 0.7157571411132807, 0.7167888488769525, 0.7178206787109368, 0.718852752685546, 0.7198849487304678, 0.7209173889160146, 0.7219500122070303, 0.7229828186035144, 0.7240158081054674, 0.7250489807128891, 0.7260823364257797, 0.727115875244139, 0.7281496582031232, 0.7291836242675761, 0.7302177734374978, 0.7312521057128883, 0.7322866210937476, 0.733321380615232, 0.7343563232421851, 0.735391448974607, 0.7364268188476538, 0.7374623107910131, 0.7384980468749974, 0.7395340270996068, 0.740570190429685, 0.7416065368652319, 0.7426430664062477, 0.7436798400878882, 0.7447167968749976, 0.745753997802732, 0.7467913818359352, 0.7478290100097633, 0.7488668212890602, 0.7499048156738258, 0.7509430541992165, 0.7519815368652322, 0.7530202026367164, 0.7540591125488257, 0.7550982055664038, 0.756137542724607, 0.7571770629882788, 0.7582168273925757, 0.7592568359374975, 0.7602970275878881, 0.7613374633789037, 0.7623781433105442, 0.7634190063476535, 0.7644601135253878, 0.7655014648437471, 0.7665430603027313, 0.7675848388671843, 0.7686269226074186, 0.7696691894531217, 0.7707117004394498, 0.7717543945312467, 0.7727973937988248, 0.773840637207028, 0.7748840637206998, 0.775927795410153, 0.7769717712402312, 0.7780159301757782, 0.7790603942871064, 0.7801050415039031, 0.7811499938964813, 0.7821951904296846, 0.7832406311035127, 0.7842863159179657, 0.7853322448730439, 0.7863784790039033, 0.7874248962402315, 0.7884716186523409, 0.7895186462402316, 0.790565856933591, 0.7916133728027316, 0.7926611328124972, 0.7937091979980441, 0.7947575073242159, 0.7958060607910128, 0.7968549194335909, 0.7979040832519503, 0.7989534912109345, 0.8000032043457, 0.8010531616210906, 0.8021033630371063, 0.8031539306640595, 0.8042047424316376, 0.8052558593749971, 0.8063072814941378, 0.8073589477539035, 0.8084109191894505, 0.8094631958007787, 0.8105157775878881, 0.8115686645507788, 0.8126218566894508, 0.8136753540039038, 0.8147291564941382, 0.815783264160154, 0.8168376770019509, 0.817892395019529, 0.8189474182128885, 0.8200028076171854, 0.8210585021972635, 0.822114501953123, 0.8231708679199198, 0.8242275390624979, 0.8252845153808573, 0.8263418579101541, 0.8273995666503885, 0.8284575805664042, 0.8295158996582009, 0.8305746459960915, 0.8316336975097633, 0.8326930541992165, 0.8337528381347633, 0.8348129882812475, 0.835873443603513, 0.8369342651367161, 0.8379955139160128, 0.8390570678710908, 0.8401190490722626, 0.8411813964843721, 0.8422441101074188, 0.8433071899414032, 0.8443706970214814, 0.8454345703124971, 0.8464988098144502, 0.8475634765624972, 0.8486285705566378, 0.8496940917968722, 0.850759979248044, 0.8518262939453097, 0.8528930358886692, 0.8539602050781222, 0.8550277404785128, 0.8560957641601533, 0.8571642150878876, 0.8582331542968719, 0.8593024597167936, 0.8603722534179654, 0.8614425354003873, 0.8625132446289029, 0.8635844421386684, 0.8646560668945277, 0.8657282409667931, 0.8668008422851524, 0.8678739318847616, 0.8689475097656209, 0.8700216369628865, 0.871096252441402, 0.8721713562011676, 0.8732470092773394, 0.8743231506347614, 0.8753998413085895, 0.876477081298824, 0.8775548706054644, 0.878633148193355, 0.879712036132808, 0.8807915344238236, 0.8818715209960891, 0.8829521789550733, 0.8840333862304638, 0.8851151428222607, 0.8861975708007762, 0.8872806091308544, 0.888364257812495, 0.8894485168456981, 0.89053344726562, 0.8916189880371043, 0.8927052612304637, 0.8937921447753855, 0.8948797607421822, 0.8959679870605416, 0.8970570068359321, 0.8981466979980416, 0.899237121582026, 0.9003282775878855, 0.9014202270507762, 0.9025129089355418, 0.9036063842773386, 0.9047005920410105, 0.90579565429687, 0.9068915710449169, 0.907988281249995, 0.9090858459472607, 0.9101842651367139, 0.9112835388183546, 0.9123837890624952, 0.9134848937988234, 0.9145869750976514, 0.9156899719238232, 0.9167939453124949, 0.917898956298823, 0.9190049438476511, 0.9201119689941354, 0.9212200317382758, 0.9223292541503852, 0.9234395141601509, 0.9245509338378853, 0.9256635131835885, 0.9267772521972604, 0.9278922729492135, 0.9290085144042916, 0.9301259765624946, 0.9312448425292913, 0.9323649902343693, 0.933486541748041, 0.9346094970703066, 0.9357339172363223, 0.936859863281244, 0.9379873962402282, 0.9391164550781187, 0.940247161865228, 0.9413795776367123, 0.942513763427728, 0.9436497192382748, 0.9447875671386655, 0.9459273071288999, 0.9470690612792906, 0.9482128906249937, 0.9493588562011657, 0.9505070800781187, 0.9516576232910093, 0.9528105468749936, 0.9539660339355404, 0.9551240844726497, 0.9562849426269465, 0.9574486083984307, 0.9586152648925711, 0.9597850952148367, 0.960958282470696, 0.9621348876953054, 0.9633151550292897, 0.9644993896484303, 0.9656876525878834, 0.9668803100585867, 0.968077667236321, 0.9692799682617116, 0.9704875793456958, 0.9717009887695238, 0.9729205017089769, 0.9741467285156175, 0.9753802795410081, 0.9766218261718677, 0.9778722229003832, 0.9791323242187425, 0.9804033508300707, 0.9816867675781177, 0.982984222412102, 0.9842979736328051, 0.9856308288574146, 0.9869865722656176, 0.9883705139160083, 0.989790344238274, 0.9912579650878833, 0.9927941894531177, 0.9944410705566334, 0.996317932128899],  # skipcq: FLK-E501
                'num_bt = 200, alpha = 0.1': [0.0005266952514648437, 0.0026622009277343754, 0.005522804260253907, 0.008751449584960939, 0.012211112976074221, 0.015833129882812498, 0.01957775115966797, 0.023419113159179696, 0.02733966827392579, 0.031326751708984385, 0.03537067413330079, 0.03946403503417971, 0.04360103607177738, 0.047776947021484405, 0.05198780059814457, 0.05623039245605472, 0.060501899719238306, 0.06480010986328127, 0.06912288665771488, 0.07346855163574223, 0.07783550262451176, 0.08222236633300783, 0.08662792205810552, 0.09105125427246098, 0.09549121856689458, 0.09994697570800787, 0.10441783905029303, 0.10890296936035163, 0.11340183258056649, 0.11791374206542979, 0.12243816375732434, 0.12697463989257826, 0.13152271270751967, 0.1360819244384767, 0.14065181732177745, 0.14523216247558604, 0.14982250213623055, 0.15442260742187514, 0.15903209686279313, 0.1636506652832033, 0.168278160095215, 0.17291435241699235, 0.17755886077880872, 0.18221160888671892, 0.1868722915649416, 0.19154083251953144, 0.19621692657470724, 0.20090049743652366, 0.20559139251709008, 0.21028930664062526, 0.21499423980712914, 0.2197060394287112, 0.2244245529174807, 0.2291496276855471, 0.23388118743896508, 0.2386191558837893, 0.24336330413818386, 0.2481136322021487, 0.2528699874877932, 0.25763236999511746, 0.26240062713623075, 0.26717460632324247, 0.27195438385009796, 0.27673973083496123, 0.2815306472778324, 0.2863271331787113, 0.2911289596557621, 0.2959362792968754, 0.30074878692626994, 0.30556663513183635, 0.31038967132568407, 0.3152178192138677, 0.32005115509033255, 0.32488945007324277, 0.3297328567504889, 0.33458114624023505, 0.33943439483642646, 0.3442926025390632, 0.34915561676025453, 0.35402343750000065, 0.35889614105224676, 0.36377349853515695, 0.36865566253662174, 0.37354255676269593, 0.37843410491943424, 0.38333030700683657, 0.3882311630249029, 0.39313659667968803, 0.39804668426513723, 0.40296134948730516, 0.4078805923461919, 0.41280441284179736, 0.4177328109741217, 0.42266563415527403, 0.4276031112670905, 0.4325450134277351, 0.4374914169311531, 0.4424423980712899, 0.44739788055420004, 0.45235778808593835, 0.45732227325439534, 0.4622911834716805, 0.46726467132568444, 0.47224258422851645, 0.4772250747680672, 0.48221199035644613, 0.48720348358154386, 0.492199478149415, 0.49720005035400483, 0.5022051239013681, 0.5072147750854501, 0.5122290039062509, 0.5172478103637705, 0.5222712707519543, 0.5272993087768567, 0.5323320007324233, 0.5373693466186538, 0.5424113464355484, 0.5474580764770525, 0.5525096130371112, 0.5575658798217793, 0.5626268768310567, 0.567692756652834, 0.5727635192871114, 0.5778391647338887, 0.5829196929931661, 0.5880052566528342, 0.5930957794189475, 0.5981914138793967, 0.6032921600341818, 0.6083980178833028, 0.6135090637207052, 0.6186254501342794, 0.6237471008300801, 0.6288740921020528, 0.634006576538088, 0.6391444778442403, 0.6442880249023457, 0.6494372177124044, 0.6545920562744161, 0.6597527694702168, 0.6649192810058613, 0.6700918197631853, 0.6752703857421893, 0.6804551315307634, 0.6856461334228532, 0.690843467712404, 0.6960472869873062, 0.7012576675415054, 0.7064747619628922, 0.711698722839357, 0.7169297027587906, 0.7221677780151383, 0.7274131011962908, 0.7326659011840839, 0.7379263305664081, 0.7431945419311543, 0.7484706878662131, 0.7537550735473655, 0.7590478515625023, 0.7643492507934593, 0.7696595764160179, 0.7749789810180687, 0.7803078460693382, 0.7856464004516623, 0.7909950256347678, 0.7963540267944358, 0.8017237854003927, 0.80710460662842, 0.8124971008300801, 0.8179015731811543, 0.823318634033205, 0.8287487411499042, 0.8341925048828143, 0.8396506118774432, 0.845123825073244, 0.8506128311157246, 0.8561184692382833, 0.8616418838500997, 0.8671839904785177, 0.8727460861206076, 0.8783294677734395, 0.8839356613159199, 0.8895665740966816, 0.8952240371704121, 0.9009105682373066, 0.9066288375854512, 0.9123821258544944, 0.9181743240356468, 0.9240102386474631, 0.9298956680297872, 0.9358380889892599, 0.9418471908569359, 0.9479356384277366, 0.9541212844848657, 0.9604301452636743, 0.9669025039672876, 0.9736086273193385, 0.9806913375854518, 0.9885530853271509],  # skipcq: FLK-E501
                'num_bt = 200, alpha = 0.05': [0.00025646209716796877, 0.0017797088623046878, 0.004100608825683594, 0.0068597412109375, 0.009901237487792966, 0.013143997192382809, 0.016539649963378908, 0.0200567626953125, 0.023673591613769536, 0.027374267578125014, 0.031146888732910168, 0.03498214721679687, 0.03887271881103515, 0.04281265258789061, 0.046796989440917945, 0.0508216857910156, 0.054883308410644496, 0.05897872924804685, 0.06310550689697261, 0.06726142883300776, 0.07144443511962884, 0.0756529235839843, 0.07988521575927726, 0.08414009094238271, 0.08841632843017569, 0.09271270751953116, 0.09702823638916008, 0.101362075805664, 0.10571338653564448, 0.11008140563964841, 0.11446544647216796, 0.11886489868164063, 0.1232790756225586, 0.12770744323730468, 0.13214962005615233, 0.13660499572753904, 0.14107326507568357, 0.14555389404296878, 0.15004657745361333, 0.154551010131836, 0.15906673431396487, 0.16359352111816405, 0.16813098907470705, 0.17267898559570313, 0.1772372055053711, 0.1818053436279297, 0.18638324737548825, 0.19097068786621088, 0.19556743621826167, 0.20017326354980466, 0.2047880172729492, 0.20941154479980467, 0.21404361724853516, 0.2186841583251953, 0.2233330154418945, 0.22798995971679686, 0.23265491485595705, 0.23732772827148438, 0.2420083236694336, 0.24669654846191408, 0.25139225006103516, 0.25609542846679695, 0.2608059310913086, 0.26552360534667974, 0.2702483749389649, 0.27498023986816417, 0.27971904754638677, 0.28446472167968756, 0.28921718597412116, 0.29397636413574224, 0.29874225616455086, 0.30351470947265635, 0.3082937240600586, 0.3130792236328125, 0.3178710556030273, 0.3226693725585936, 0.3274739456176757, 0.3322847747802734, 0.33710178375244143, 0.341925048828125, 0.3467544174194337, 0.3515898895263674, 0.356431465148926, 0.3612790679931643, 0.36613262176513695, 0.37099212646484403, 0.37585765838623075, 0.3807290649414065, 0.38560634613037126, 0.3904895782470705, 0.3953786087036135, 0.4002734375000002, 0.40517414093017595, 0.4100806427001955, 0.4149929428100588, 0.41991104125976586, 0.42483486175537133, 0.4297644805908206, 0.4346998977661136, 0.43964103698730506, 0.44458789825439493, 0.4495405578613286, 0.45449901580810603, 0.4594631958007818, 0.4644331741333013, 0.4694088745117193, 0.4743904495239264, 0.4793777465820319, 0.48437084197998115, 0.4893698120117194, 0.4943745803833015, 0.49938522338867264, 0.5044017410278328, 0.5094241333007821, 0.5144524765014657, 0.5194866943359384, 0.5245269393920908, 0.5295732116699228, 0.5346254348754893, 0.5396837615966807, 0.5447482681274425, 0.549818801879884, 0.5548955917358409, 0.5599785614013684, 0.5650677871704115, 0.5701633453369155, 0.5752653121948257, 0.5803736877441421, 0.5854884719848648, 0.5906098937988296, 0.595737800598146, 0.6008724212646499, 0.6060138320922865, 0.6111619567871107, 0.6163170242309584, 0.6214789581298841, 0.626647987365724, 0.6318241882324231, 0.6370075607299817, 0.6421982574462901, 0.6473963546752941, 0.652602005004884, 0.6578152847290049, 0.6630363464355478, 0.6682652664184578, 0.6735021209716805, 0.678747215270997, 0.6840005493164072, 0.6892622756958018, 0.6945326232910167, 0.6998117446899426, 0.7050997924804701, 0.7103969955444349, 0.7157034301757828, 0.7210194778442398, 0.7263452148437517, 0.7316809463500995, 0.7370269012451192, 0.7423833084106466, 0.7477504730224629, 0.7531287002563496, 0.7585182952880878, 0.7639195632934588, 0.7693328857421893, 0.7747586441040055, 0.7801972198486344, 0.785649147033693, 0.7911147308349626, 0.796594657897951, 0.8020893096923847, 0.8075993728637716, 0.813125457763674, 0.8186683273315452, 0.824228668212893, 0.8298072433471704, 0.8354051208496118, 0.8410232925415064, 0.8466627502441431, 0.8523247909545921, 0.858010711669924, 0.8637222671508811, 0.8694609832763693, 0.875228843688967, 0.8810282135009788, 0.8868616104125999, 0.8927320098876976, 0.8986429977417016, 0.9045986175537135, 0.9106038284301785, 0.9166648864746121, 0.922789039611819, 0.9289858245849636, 0.9352674484252956, 0.9416502380371119, 0.9481566238403347, 0.9548195648193389, 0.961690254211429, 0.9688574218750032, 0.9765015029907258, 0.9851329803466828],  # skipcq: FLK-E501
                'num_bt = 200, alpha = 0.025': [0.00012660980224609377, 0.0012133789062500006, 0.0031041336059570313, 0.00547554016113281, 0.00816616058349609, 0.011087417602539059, 0.014185523986816401, 0.017424774169921872, 0.020780448913574223, 0.024234161376953132, 0.027772254943847663, 0.031383972167968746, 0.03506069183349609, 0.03879554748535156, 0.042582817077636705, 0.046417617797851536, 0.0502959060668945, 0.05421424865722652, 0.058169593811035106, 0.06215934753417964, 0.06618122100830073, 0.07023315429687496, 0.07431339263916009, 0.0784202575683593, 0.08255237579345695, 0.08670837402343741, 0.09088710784912102, 0.09508750915527336, 0.09930858612060539, 0.10354949951171868, 0.10780941009521477, 0.11208747863769523, 0.11638309478759756, 0.12069564819335926, 0.12502437591552723, 0.12936889648437494, 0.13372867584228512, 0.13810310363769526, 0.14249187469482413, 0.14689445495605458, 0.1513105392456054, 0.15573974609374996, 0.1601816940307617, 0.16463607788085938, 0.1691025161743164, 0.17358085632324222, 0.1780707168579102, 0.1825718688964844, 0.18708415985107427, 0.19160713195800783, 0.1961407852172852, 0.20068489074707038, 0.20523906707763678, 0.20980331420898446, 0.21437740325927743, 0.21896118164062506, 0.2235544204711915, 0.2281569671630861, 0.23276882171630878, 0.2373896789550783, 0.2420194625854494, 0.2466580200195314, 0.25130535125732434, 0.25596115112304696, 0.26062549591064466, 0.26529823303222666, 0.26997913360595704, 0.27466827392578125, 0.27936550140380856, 0.28407066345214843, 0.28878376007080075, 0.2935047149658203, 0.2982334518432618, 0.3029698181152345, 0.3077138137817384, 0.31246543884277356, 0.3172245407104493, 0.3219910430908204, 0.32676502227783205, 0.3315462493896485, 0.3363348007202148, 0.3411306762695312, 0.3459336471557617, 0.35074386596679696, 0.35556118011474624, 0.3603855895996096, 0.36521709442138695, 0.37005554199218776, 0.3749010848999026, 0.3797535705566409, 0.3846130752563479, 0.3894794464111331, 0.3943527603149417, 0.3992329406738284, 0.4041200637817386, 0.4090139770507816, 0.4139148330688479, 0.4188224792480471, 0.42373706817626977, 0.4286584472656253, 0.433586692810059, 0.43852172851562543, 0.44346363067627004, 0.4484123992919927, 0.45336803436279355, 0.45833053588867245, 0.46329982757568416, 0.4682761383056646, 0.4732592391967779, 0.47824928283691465, 0.4832463455200201, 0.4882502746582037, 0.49326122283935603, 0.4982791900634771, 0.5033041000366216, 0.5083361816406258, 0.5133752822875985, 0.5184215545654305, 0.523474998474122, 0.5285356903076182, 0.5336035537719737, 0.5386787414550791, 0.5437613296508799, 0.5488512420654308, 0.5539486312866222, 0.5590534973144541, 0.5641659927368174, 0.5692860412597667, 0.5744137954711925, 0.5795494079589856, 0.584692726135255, 0.5898440551757824, 0.5950033950805675, 0.6001707458496106, 0.605346260070802, 0.6105300903320323, 0.6157223129272471, 0.6209229278564463, 0.6261321640014659, 0.6313500976562509, 0.6365768814086923, 0.6418125152587899, 0.6470573043823251, 0.6523112487792979, 0.657574577331544, 0.6628474426269542, 0.6681299209594738, 0.6734222412109387, 0.6787246322631848, 0.6840371704101575, 0.6893601608276381, 0.6946936798095718, 0.7000381088256851, 0.7053936004638688, 0.7107603836059586, 0.7161387634277359, 0.7215289688110367, 0.7269313049316422, 0.7323460769653336, 0.7377736663818375, 0.7432143020629898, 0.7486685180664077, 0.7541365432739273, 0.7596189117431658, 0.7651160049438495, 0.770628433227541, 0.7761565017700216, 0.7817009735107443, 0.7872623825073264, 0.7928412628173851, 0.7984385299682641, 0.8040547180175805, 0.8096908187866236, 0.8153476715087917, 0.8210262680053737, 0.8267276763916043, 0.8324530410766631, 0.8382038116455108, 0.843981285095218, 0.8497872161865265, 0.8556233596801788, 0.8614917755126983, 0.8673948287963897, 0.8733351135253937, 0.8793157577514681, 0.8853402709960971, 0.8914127731323276, 0.8975381469726595, 0.9037223434448278, 0.909972457885746, 0.9162973403930701, 0.922708053588871, 0.9292190170288124, 0.9358493804931678, 0.9426256179809609, 0.949586410522465, 0.9567917251586955, 0.9643453216552775, 0.9724580764770547, 0.981724624633793],  # skipcq: FLK-E501
                'num_bt = 100, alpha = 0.1': [0.0010530471801757815, 0.005330657958984376, 0.011070747375488285, 0.01755897521972656, 0.024520378112792965, 0.03181716918945312, 0.039369010925292956, 0.0471240997314453, 0.0550467300415039, 0.06311126708984374, 0.07129817962646484, 0.07959266662597655, 0.08798267364501955, 0.0964585876464844, 0.10501262664794928, 0.11363807678222662, 0.12232944488525396, 0.13108200073242193, 0.13989154815673835, 0.14875457763671884, 0.15766803741455088, 0.16662918090820322, 0.1756356430053712, 0.18468521118164077, 0.19377597808837904, 0.20290626525878921, 0.21207454681396504, 0.22127944946289085, 0.23051967620849634, 0.23979415893554712, 0.24910182952880888, 0.25844184875488313, 0.2678132247924808, 0.2772153472900394, 0.28664737701416043, 0.29610877990722684, 0.30559902191162136, 0.31511749267578154, 0.32466373443603547, 0.3342373657226566, 0.3438380813598637, 0.3534654235839848, 0.363119163513184, 0.37279899597168015, 0.3825047683715824, 0.39223625183105515, 0.4019932937622075, 0.4117758178710942, 0.4215836715698247, 0.4314168548583989, 0.4412752151489262, 0.45115890502929734, 0.4610678482055669, 0.47100219726562553, 0.48096195220947324, 0.4909472656250006, 0.5009583663940438, 0.5109952545166023, 0.5210583877563484, 0.5311478424072273, 0.5412639999389655, 0.5514071655273443, 0.5615777206420904, 0.5717761230468756, 0.5820027542114266, 0.5922581481933602, 0.6025429153442392, 0.612857666015626, 0.6232031631469737, 0.6335800170898447, 0.6439892196655284, 0.654431610107423, 0.6649082565307629, 0.6754202270507825, 0.6859688186645521, 0.6965554809570325, 0.7071815872192395, 0.7178489685058607, 0.7285595321655287, 0.739315338134767, 0.7501188278198256, 0.7609728240966811, 0.7718803024292007, 0.7828448486328139, 0.7938705825805679, 0.8049623107910171, 0.8161255264282241, 0.827366867065431, 0.8386942672729505, 0.8501174163818372, 0.8616481399536144, 0.8733016204833997, 0.8850971603393565, 0.8970608520507823, 0.9092285537719738, 0.9216524505615247, 0.9344142532348645, 0.9476547241210949, 0.9616604995727549, 0.9772372436523449],  # skipcq: FLK-E501
                'num_bt = 100, alpha = 0.05': [0.0005128097534179688, 0.0035651397705078114, 0.008225822448730467, 0.01377662658691406, 0.019905586242675778, 0.026449737548828123, 0.03331188201904297, 0.040428848266601564, 0.0477566146850586, 0.055263214111328125, 0.06292453765869141, 0.07072181701660157, 0.07864017486572264, 0.0866675567626953, 0.09479404449462892, 0.10301116943359379, 0.11131214141845708, 0.1196907806396485, 0.12814197540283206, 0.13666130065917975, 0.14524478912353522, 0.15388900756835944, 0.16259090423583994, 0.17134765625000012, 0.18015705108642593, 0.18901679992675796, 0.19792491912841811, 0.2068797302246095, 0.21587963104248062, 0.22492324829101581, 0.23400913238525406, 0.24313629150390642, 0.2523035049438478, 0.26150985717773456, 0.2707544326782229, 0.2800365447998049, 0.2893553543090823, 0.29871025085449254, 0.30810070037841836, 0.31752601623535204, 0.32698581695556694, 0.33647972106933655, 0.34600734710693426, 0.3555682373046881, 0.36516231536865296, 0.37478912353515687, 0.3844486618041999, 0.39414070129394607, 0.4038650131225594, 0.41362174987793054, 0.42341068267822346, 0.4332318878173837, 0.443085365295411, 0.4529711914062508, 0.46288959503173915, 0.4728406524658212, 0.48282451629638756, 0.4928414916992196, 0.5028918075561533, 0.5129758453369149, 0.5230939102172858, 0.5332465362548835, 0.5434341049194344, 0.5536571502685554, 0.5639163589477547, 0.5742122650146493, 0.5845457077026375, 0.5949174499511725, 0.605328330993653, 0.6157794189453132, 0.6262716293334969, 0.6368063354492195, 0.6473846817016609, 0.6580081939697273, 0.6686784744262704, 0.6793972015380867, 0.6901663589477547, 0.7009882354736336, 0.7118651199340829, 0.7227997589111337, 0.7337952804565437, 0.7448551177978522, 0.7559832382202156, 0.767184143066407, 0.778463096618653, 0.7898260498046881, 0.8012801742553717, 0.812833862304688, 0.8244971847534186, 0.836282348632813, 0.8482045364379889, 0.8602828979492192, 0.8725419998168951, 0.8850147247314459, 0.8977466201782232, 0.9108037567138676, 0.9242892074584965, 0.9383808135986333, 0.9534402084350593, 0.9704869842529305],  # skipcq: FLK-E501
                'num_bt = 100, alpha = 0.025': [0.00025318145751953127, 0.0024313354492187496, 0.00622997283935547, 0.011004486083984376, 0.016431846618652345, 0.022334899902343747, 0.02860530853271484, 0.03517158508300782, 0.04198360443115235, 0.04900466918945313, 0.056207008361816406, 0.06356887817382811, 0.07107303619384765, 0.07870536804199217, 0.0864543533325195, 0.09431030273437495, 0.10226490020751948, 0.11031120300292963, 0.11844318389892572, 0.12665557861328114, 0.13494373321533193, 0.14330360412597648, 0.15173160552978507, 0.1602246093749999, 0.16877971649169915, 0.1773944091796874, 0.18606639862060537, 0.19479362487792956, 0.20357418060302723, 0.21240638732910144, 0.22128879547119132, 0.23021987915039055, 0.23919857025146474, 0.24822349548339834, 0.25729381561279285, 0.26640838623046864, 0.2755665969848631, 0.2847674560546874, 0.2940104293823241, 0.3032947540283201, 0.31261997222900373, 0.32198554992675765, 0.33139102935791, 0.3408360290527342, 0.350320243835449, 0.35984336853027327, 0.369405174255371, 0.3790055084228515, 0.38864414215087884, 0.3983211517333983, 0.4080363082885741, 0.4177897644042967, 0.42758152008056627, 0.43741157531738273, 0.44728015899658197, 0.45718749999999997, 0.46713375091552733, 0.4771192169189453, 0.48714420318603513, 0.49720916748046873, 0.5073144912719727, 0.517460708618164, 0.5276483535766601, 0.5378781127929686, 0.5481506729125976, 0.5584667205810545, 0.5688272476196286, 0.5792331695556638, 0.589685478210449, 0.600185317993164, 0.6107340621948241, 0.6213330078124999, 0.631983757019043, 0.6426879119873048, 0.6534475326538087, 0.6642645263671876, 0.6751412582397462, 0.6860803222656253, 0.6970846176147463, 0.7081573486328125, 0.7193020248413087, 0.7305229187011719, 0.7418246078491211, 0.7532124328613279, 0.7646924972534177, 0.7762720489501949, 0.7879593276977535, 0.7997643280029292, 0.8116988754272456, 0.8237773895263665, 0.8360177230834953, 0.8484423828124994, 0.8610802841186516, 0.8739700317382804, 0.8871651077270499, 0.9007428741455068, 0.9148239517211905, 0.9296160888671865, 0.9455405807495105, 0.9637833404541005],  # skipcq: FLK-E501
                'num_bt = 50, alpha = 0.1': [0.002104988098144531, 0.010686798095703127, 0.022243995666503903, 0.03534767150878906, 0.04944561004638671, 0.06426040649414061, 0.07962970733642577, 0.09544906616210938, 0.11164653778076172, 0.12817016601562498, 0.14498111724853516, 0.1620494842529297, 0.1793517684936524, 0.19686920166015628, 0.21458644866943363, 0.23249114990234374, 0.25057315826416016, 0.2688238525390625, 0.28723644256591796, 0.30580520629882824, 0.32452541351318376, 0.34339347839355494, 0.36240657806396515, 0.38156257629394563, 0.4008601760864261, 0.42029869079589877, 0.439878196716309, 0.45959922790527385, 0.4794631576538091, 0.4994721984863286, 0.5196290206909184, 0.5399374389648441, 0.5604021072387699, 0.5810285949707035, 0.601823997497559, 0.6227967071533207, 0.6439570236206058, 0.6653173065185549, 0.6868929672241213, 0.7087026977539064, 0.7307703018188476, 0.7531256103515627, 0.7758077621459962, 0.7988690185546876, 0.8223818588256835, 0.8464518737792968, 0.8712435531616209, 0.897040786743164, 0.9244193649291994, 0.9549925994873047],  # skipcq: FLK-E501
                'num_bt = 50, alpha = 0.05': [0.0010253524780273436, 0.007153701782226562, 0.016551856994628904, 0.027787704467773428, 0.040236625671386715, 0.053571395874023424, 0.06759670257568356, 0.08218505859375, 0.09724811553955079, 0.11272163391113282, 0.1285573959350586, 0.14471817016601565, 0.1611745834350586, 0.1779032135009766, 0.19488491058349616, 0.2121041107177735, 0.22954799652099614, 0.24720588684082034, 0.2650691604614259, 0.28313056945800796, 0.3013843917846682, 0.3198258972167971, 0.33845157623291044, 0.3572587585449221, 0.376245918273926, 0.395412063598633, 0.4147572708129885, 0.4342823791503908, 0.45398914337158225, 0.4738802337646486, 0.4939592361450197, 0.5142308807373048, 0.5347008895874026, 0.5553767395019535, 0.5762670516967778, 0.5973825836181648, 0.6187364578247077, 0.6403443145751957, 0.6622255325317388, 0.6844039154052739, 0.7069094467163091, 0.7297798919677738, 0.7530647659301761, 0.7768299865722659, 0.8011670303344729, 0.826208877563477, 0.852162818908692, 0.8793858337402352, 0.9086018753051766, 0.9418449401855475],  # skipcq: FLK-E501
                'num_bt = 50, alpha = 0.025': [0.0005062484741210939, 0.004881439208984375, 0.012548561096191409, 0.022227935791015625, 0.03327510833740233, 0.045335311889648414, 0.058191719055175756, 0.07170074462890623, 0.08576206207275389, 0.10030220031738282, 0.11526584625244143, 0.1306098937988281, 0.1463006210327148, 0.1623106384277343, 0.17861782073974602, 0.1952041625976562, 0.21205471038818355, 0.22915710449218749, 0.24650104522705074, 0.26407836914062494, 0.28188220977783196, 0.29990722656249996, 0.31814914703369135, 0.3366050720214844, 0.3552730178833008, 0.37415191650390633, 0.3932419967651368, 0.4125440979003908, 0.4320604324340822, 0.4517940521240237, 0.47174915313720733, 0.4919313812255862, 0.5123475265502933, 0.5330061340332035, 0.5539176559448247, 0.5750946807861332, 0.5965523147583014, 0.6183092498779303, 0.6403881454467779, 0.6628169250488287, 0.685630607604981, 0.7088736724853519, 0.7326040267944338, 0.7568986511230471, 0.7818646621704104, 0.8076572418212895, 0.8345180892944339, 0.8628623962402346, 0.8935304641723636, 0.9288782501220707],  # skipcq: FLK-E501
                'num_bt = 30, alpha = 0.1': [0.003505865732828776, 0.0178689956665039, 0.03730777104695638, 0.05944360097249348, 0.08335453669230143, 0.10857747395833334, 0.13484245936075845, 0.16197719573974612, 0.18986501693725588, 0.21842352549235028, 0.24759359359741212, 0.2773321787516274, 0.3076083819071449, 0.33840077718098927, 0.3696960131327308, 0.40148760477701795, 0.4337754885355629, 0.4665662765502927, 0.4998735745747882, 0.5337188084920246, 0.5681330680847166, 0.6031595865885414, 0.6388580004374184, 0.6753109614054361, 0.7126363436381019, 0.7510082880655921, 0.7907005627950029, 0.8321870803833004, 0.8764300346374508, 0.926118723551432],  # skipcq: FLK-E501
                'num_bt = 30, alpha = 0.05': [0.0017083168029785158, 0.011975797017415368, 0.02781553268432617, 0.04685484568277994, 0.06805556615193684, 0.09087403615315756, 0.11498689651489258, 0.14018510182698568, 0.1663259824117025, 0.19330844879150394, 0.22105944951375328, 0.2495258967081706, 0.2786695798238119, 0.30846405029296875, 0.3388926506042481, 0.3699475606282553, 0.4016289710998535, 0.4339452743530273, 0.46691370010375977, 0.5005613327026367, 0.5349272727966307, 0.5700660705566404, 0.6060525576273598, 0.6429908752441403, 0.6810288429260251, 0.7203848520914709, 0.7614021619160968, 0.8046739578247067, 0.851403903961181, 0.9049661636352534],  # skipcq: FLK-E501
                'num_bt = 30, alpha = 0.025': [0.0008435885111490885, 0.008178138732910154, 0.021117115020751948, 0.03755346934000651, 0.056421693166097, 0.07713553110758462, 0.09933786392211913, 0.12279478708902995, 0.1473451932271322, 0.1728741963704427, 0.19929863611857096, 0.22655766805013022, 0.2546075503031413, 0.283418083190918, 0.3129702568054199, 0.3432552337646484, 0.3742734591166178, 0.4060349146525065, 0.4385598182678224, 0.4718799591064454, 0.5060409863789878, 0.5411063512166342, 0.5771634737650556, 0.6143334706624354, 0.6527883211771653, 0.6927816390991218, 0.7347115516662605, 0.7792645772298182, 0.827830537160238, 0.8842966715494796],  # skipcq: FLK-E501
                'num_bt = 20, alpha = 0.1': [0.005254220962524414, 0.026914119720458988, 0.05641789436340332, 0.0902134895324707, 0.12692608833312985, 0.16587238311767571, 0.20666403770446767, 0.24906482696533194, 0.29292883872985825, 0.33817090988159165, 0.3847514629364012, 0.4326707839965818, 0.4819692134857176, 0.5327330589294432, 0.5851095676422118, 0.6393381118774413, 0.6958132266998289, 0.7552347183227537, 0.8190390110015868, 0.8912508964538572],  # skipcq: FLK-E501
                'num_bt = 20, alpha = 0.05': [0.002561426162719727, 0.01806516647338867, 0.04216942787170409, 0.07135391235351564, 0.1040808200836182, 0.13955373764038093, 0.17731089591979982, 0.21706857681274416, 0.25865063667297367, 0.30195388793945327, 0.3469314098358156, 0.39358491897583026, 0.4419655323028567, 0.49218158721923855, 0.5444176197052006, 0.5989718437194829, 0.6563361644744876, 0.7173814773559575, 0.7838938236236577, 0.8608916282653813],  # skipcq: FLK-E501
                'num_bt = 20, alpha = 0.025': [0.0012650966644287111, 0.012348556518554692, 0.032070970535278326, 0.05733404159545899, 0.08657145500183107, 0.11893157958984377, 0.1539091587066651, 0.19119005203247075, 0.2305778980255127, 0.27195787429809565, 0.31527810096740716, 0.36054258346557605, 0.4078114986419677, 0.4572108268737792, 0.5089540958404539, 0.5633859634399412, 0.6210731983184814, 0.6830172538757324, 0.7512671947479248, 0.8315665245056152]  # skipcq: FLK-E501
                }

        self.ignore_list = ignore_list
        if self.ignore_list is None:
            self.ignore_list = []
        self.constraint_list = constraint_list
        if self.constraint_list is None:
            self.constraint_list = []

        # Loads the persistence
        self.persistence_id = persistence_id
        PersistenceUtil.add_persistable_component(self)
        self.persistence_file_name = AMinerConfig.build_persistence_file_name(aminer_config, self.__class__.__name__, persistence_id)
        persistence_data = PersistenceUtil.load_json(self.persistence_file_name)

        # Imports the persistence
        if persistence_data is not None:
            self.load_persistence_data(persistence_data)
            logging.getLogger(AMinerConfig.DEBUG_LOG_NAME).debug('%s loaded persistence data.', self.__class__.__name__)

    def receive_atom(self, log_atom):
        """
        Receive an parsed atom and the information about the parser match. Initializes Variables for new eventTypes.
        @param log_atom the parsed log atom
        @return True if this handler was really able to handle and process the match.
        """
        event_index = self.event_type_detector.current_index
        if event_index == -1:
            return False

        self.log_total += 1

        parser_match = log_atom.parser_match
        # Skip paths from ignore_list.
        for ignore_path in self.ignore_list:
            if ignore_path in parser_match.get_match_dictionary().keys():
                return False

        if self.path_list is None or len(self.path_list) == 0:
            constraint_path_flag = False
            for constraint_path in self.constraint_list:
                if parser_match.get_match_dictionary().get(constraint_path) is not None:
                    constraint_path_flag = True
                    break
            if not constraint_path_flag and self.constraint_list != []:
                return False

        # Initialize new entries in lists for a new eventType if necessary
        if len(self.length) < event_index + 1 or self.var_type[event_index] == []:
            for _ in range(event_index + 1 - len(self.length)):
                self.length.append(0)
                self.variable_path_num.append([])
                self.var_type.append([])
                self.alternative_distribution_types.append([])
                self.distr_val.append([])
                self.bt_results.append([])

            # Number of variables
            self.length[event_index] = len(self.event_type_detector.variable_key_list[event_index])
            # List of the found vartypes
            self.var_type[event_index] = [[] for i in range(self.length[event_index])]
            # Stores the alternative distributions of the variable
            self.alternative_distribution_types[event_index] = [[] for i in range(self.length[event_index])]
            # Stores the values the distribution, which are needed for the sKS
            self.distr_val[event_index] = [[] for i in range(self.length[event_index])]
            # List of the successes of the binomialtest for the rejection in the sKS or variables of discrete type
            self.bt_results[event_index] = [[] for i in range(self.length[event_index])]

            # Adds the variable indices to the variable_path_num-list if the path_list is not empty
            if self.path_list is not None:
                for var_index in range(self.length[event_index]):
                    if self.event_type_detector.variable_key_list[event_index][var_index] in self.path_list:
                        self.variable_path_num[event_index].append(var_index)
            if self.num_events < event_index + 1:
                self.num_events = event_index + 1

        # Processes the current log-line by testing and updating
        self.process_ll(event_index, log_atom)
        return True

    def get_time_trigger_class(self):
        """Get the trigger class this component can be registered for. This detector only needs persisteny triggers in real time."""
        return AnalysisContext.TIME_TRIGGER_CLASS_REALTIME

    def do_timer(self, trigger_time):
        """Check if current ruleset should be persisted."""
        if self.next_persist_time is None:
            return self.aminer_config.config_properties.get(AMinerConfig.KEY_PERSISTENCE_PERIOD, AMinerConfig.DEFAULT_PERSISTENCE_PERIOD)

        delta = self.next_persist_time - trigger_time
        if delta < 0:
            self.do_persist()
            delta = self.aminer_config.config_properties.get(AMinerConfig.KEY_PERSISTENCE_PERIOD, AMinerConfig.DEFAULT_PERSISTENCE_PERIOD)
        return delta

    def do_persist(self):
        """Immediately write persistence data to storage."""
        tmp_list = [self.var_type, self.alternative_distribution_types, self.var_type_history_list,
                    self.var_type_history_list_reference, self.failed_indicators, [[self.distr_val[event_index][var_index] if (
                        len(self.distr_val[event_index][var_index]) > 0 and self.var_type[event_index][var_index][0] == 'emp') else [] for
                            var_index in range(len(self.distr_val[event_index]))] for event_index in range(len(self.distr_val))]]
        PersistenceUtil.store_json(self.persistence_file_name, tmp_list)
        self.next_persist_time = None

        if self.save_statistics:
            PersistenceUtil.store_json(self.statistics_file_name, [
                self.failed_indicators_total, self.failed_indicators_values, self.failed_indicators_paths, self.failed_indicators])
        logging.getLogger(AMinerConfig.DEBUG_LOG_NAME).debug('%s persisted data.', self.__class__.__name__)

    def load_persistence_data(self, persistence_data):
        """Extract the persistence data and appends various lists to create a consistent state."""
        # Import the lists of the persistence
        self.var_type = persistence_data[0]
        self.alternative_distribution_types = persistence_data[1]
        self.var_type_history_list = persistence_data[2]
        self.var_type_history_list_reference = persistence_data[3]
        self.failed_indicators = persistence_data[4]
        self.distr_val = persistence_data[5]
        self.num_events = len(self.var_type)

        # Create the initial lists which derive from the persistence
        # Number of variables of the single events
        self.length = [len(self.event_type_detector.variable_key_list[event_index]) for event_index in range(self.num_events)]
        self.variable_path_num = [[] for _ in range(self.num_events)]
        # List of the successes of the binomialtest for the rejection in the sKS or variables of discrete type
        self.bt_results = [[[] for var_index in range(self.length[event_index])] for event_index in range(self.num_events)]

        # Updates the lists for each eventType individually
        for event_index in range(self.num_events):
            # Adds the variable indices to the variable_path_num-list if the path_list is not empty
            if self.path_list is not None:
                for var_index in range(self.length[event_index]):
                    if self.event_type_detector.variable_key_list[event_index][var_index] in self.path_list:
                        self.variable_path_num[event_index].append(var_index)

            # Initializes the lists for the discrete distribution, or continuous distribution
            for var_index in range(len(self.var_type[event_index])):
                if len(self.var_type[event_index][var_index]) > 0:

                    if self.var_type[event_index][var_index][0] in self.distr_list:
                        self.bt_results[event_index][var_index] = [1] * self.num_s_ks_bt
                        if self.var_type[event_index][var_index][0] == 'betam' or self.var_type[event_index][var_index][0] == 'spec':
                            self.s_ks_get_quantiles(event_index, var_index)
                    elif self.var_type[event_index][var_index][0] == 'd':
                        self.d_init_bt(event_index, var_index)

    def process_ll(self, event_index, log_atom):
        """Process the log line. Extracts and appends the values of the log line to the values-list."""
        # Return if no variable is tracked in the VTD
        if len(self.event_type_detector.variable_key_list[event_index]) == 0 or (
                self.path_list is not None and self.variable_path_num[event_index] == []):
            return

        # Initial detection of varTypes
        if self.event_type_detector.num_eventlines[event_index] == self.num_init and self.var_type[event_index][0] == []:
            # Test all variables

            logging.getLogger(AMinerConfig.DEBUG_LOG_NAME).debug('%s started initial detection of var types.', self.__class__.__name__)
            if self.path_list is None:
                for var_index in range(self.length[event_index]):
                    tmp_var_type = self.detect_var_type(event_index, var_index)

                    # VarType is empiric distribution
                    if tmp_var_type[0] == 'emp':
                        self.var_type[event_index][var_index] = tmp_var_type
                        self.bt_results[event_index][var_index] = [1] * self.num_s_ks_bt
                        self.s_ks_get_quantiles(event_index, var_index)

                    # VarType is a continuous distribution
                    elif tmp_var_type[0] in self.distr_list:
                        self.var_type[event_index][var_index] = tmp_var_type[:-1]
                        self.alternative_distribution_types[event_index][var_index] = tmp_var_type[-1]
                        self.bt_results[event_index][var_index] = [1] * self.num_s_ks_bt
                        if self.var_type[event_index][var_index][0] == 'betam' or self.var_type[event_index][var_index][0] == 'spec':
                            self.s_ks_get_quantiles(event_index, var_index)

                    # Initializes the binomialtest for the discrete type
                    elif tmp_var_type[0] == 'd':
                        self.var_type[event_index][var_index] = tmp_var_type
                        self.d_init_bt(event_index, var_index)

                    # mMrk the variables, which could be static parts of the parser model
                    elif tmp_var_type[0] == 'stat':
                        self.var_type[event_index][var_index] = tmp_var_type
                        self.var_type[event_index][var_index][2] = True

                    else:
                        self.var_type[event_index][var_index] = tmp_var_type

            # Test only the variables with paths in the path_list
            else:
                for var_index in self.variable_path_num[event_index]:
                    tmp_var_type = self.detect_var_type(event_index, var_index)

                    # VarType is empiric distribution
                    if tmp_var_type[0] == 'emp':
                        self.var_type[event_index][var_index] = tmp_var_type
                        self.bt_results[event_index][var_index] = [1] * self.num_s_ks_bt
                        self.s_ks_get_quantiles(event_index, var_index)

                    # VarType is a continuous distribution
                    elif tmp_var_type[0] in self.distr_list:
                        self.var_type[event_index][var_index] = tmp_var_type[:-1]
                        self.alternative_distribution_types[event_index][var_index] = tmp_var_type[-1]
                        self.bt_results[event_index][var_index] = [1] * self.num_s_ks_bt
                        if self.var_type[event_index][var_index][0] == 'betam' or self.var_type[event_index][var_index][0] == 'spec':
                            self.s_ks_get_quantiles(event_index, var_index)

                    # Initializes the binomialtest for the discrete type
                    elif tmp_var_type[0] == 'd':
                        self.var_type[event_index][var_index] = tmp_var_type
                        self.d_init_bt(event_index, var_index)

                    # mMrk the variables, which could be static parts of the parser model
                    elif tmp_var_type[0] == 'stat':
                        self.var_type[event_index][var_index] = tmp_var_type
                        self.var_type[event_index][var_index][2] = True

                    else:
                        self.var_type[event_index][var_index] = tmp_var_type

            self.init_var_type_history_list(event_index)
            self.print_initial_var_type(event_index, log_atom)
            self.log_new_learned += len(self.var_type[event_index])
            self.log_new_learned_values.append(self.var_type[event_index])

        # Update varTypes
        elif self.event_type_detector.num_eventlines[event_index] > self.num_init and (
                self.event_type_detector.num_eventlines[event_index] - self.num_init) % self.num_update == 0:

            logging.getLogger(AMinerConfig.DEBUG_LOG_NAME).debug('%s started update phase of var types.', self.__class__.__name__)
            # Checks if the updates of the varTypes should be stopped
            if self.update_var_type_bool and (not isinstance(self.num_stop_update, bool)) and (
                    self.event_type_detector.total_records >= self.num_stop_update):
                self.update_var_type_bool = False

            # Get the index_list for the variables which should be updated
            index_list = None
            if self.path_list is None:
                index_list = range(self.length[event_index])
            else:
                index_list = self.variable_path_num[event_index]
            self.log_updated += len(index_list)

            # Update the variableTypes
            for var_index in index_list:
                # Skips the variable if checkVariable is False
                if not self.event_type_detector.check_variables[event_index][var_index]:
                    continue

                # update varTypes
                self.update_var_type(event_index, var_index, log_atom)

                # This section updates the historyList of the variableTypes
                if self.var_type[event_index][var_index][0] in self.var_type_history_list_order:
                    # Index of the variableType in the list  # [others, static, [discrete, number of appended steps],
                    # asc, desc, unique, EV of continuous distributions]
                    type_index = self.var_type_history_list_order.index(self.var_type[event_index][var_index][0])
                else:
                    type_index = -1

                for tmp_type_index in range(len(self.var_type_history_list[event_index][var_index])):
                    if tmp_type_index == type_index:
                        if len(self.var_type_history_list[event_index][var_index][type_index]) >= 1 and isinstance(
                                self.var_type_history_list[event_index][var_index][type_index][0], list):
                            self.var_type_history_list[event_index][var_index][type_index][0].append(1)
                            for i in range(1, len(self.var_type_history_list[event_index][var_index][type_index])):
                                self.var_type_history_list[event_index][var_index][type_index][i].append(0)
                        else:
                            self.var_type_history_list[event_index][var_index][type_index].append(1)
                    else:
                        if len(self.var_type_history_list[event_index][var_index][tmp_type_index]) >= 1 and isinstance(
                                self.var_type_history_list[event_index][var_index][tmp_type_index][0], list):
                            for i in range(len(self.var_type_history_list[event_index][var_index][tmp_type_index])):
                                self.var_type_history_list[event_index][var_index][tmp_type_index][i].append(0)
                        else:
                            self.var_type_history_list[event_index][var_index][tmp_type_index].append(0)

                if type_index == -1:
                    # Continuously distributed variable type. Index 6 in the history list
                    if self.var_type[event_index][var_index][0] == 'uni':
                        self.var_type_history_list[event_index][var_index][6][0][-1] = (
                            self.var_type[event_index][var_index][1] + self.var_type[event_index][var_index][2]) / 2
                        self.var_type_history_list[event_index][var_index][6][1][-1] = (
                            self.var_type[event_index][var_index][2] - self.var_type[event_index][var_index][1]) / np.sqrt(12)
                    else:
                        self.var_type_history_list[event_index][var_index][6][0][-1] = self.var_type[event_index][var_index][1]
                        self.var_type_history_list[event_index][var_index][6][1][-1] = self.var_type[event_index][var_index][2]

            # Reduce the number of variables, which are tracked
            if (not (isinstance(self.num_updates_until_var_reduction, bool)) and (
                    self.event_type_detector.num_eventlines[event_index] - self.num_init) / self.num_update ==
                    self.num_updates_until_var_reduction - 1):

                for var_index in range(len(self.var_type_history_list[event_index])):
                    # Skips the variable if it is already not being checked
                    if not self.event_type_detector.check_variables[event_index][var_index]:
                        continue

                    tmp_max = 1
                    exceeded_thresh = False
                    for type_index in range(1, len(self.var_type_history_list[event_index][var_index])):
                        # Continuous Distribution
                        if type_index == 6:  # continuously distributed variable type
                            num_app = len([1 for x in self.var_type_history_list[event_index][var_index][type_index][1] if x != 0])
                            if num_app / self.num_updates_until_var_reduction >= self.var_reduction_thres:
                                exceeded_thresh = True
                                break
                            if num_app > tmp_max:
                                tmp_max = num_app
                        # Distributions which are not continuous
                        else:
                            if len(self.var_type_history_list[event_index][var_index][type_index]) >= 1 and isinstance(
                                    self.var_type_history_list[event_index][var_index][type_index][0], list):
                                num_app = sum(self.var_type_history_list[event_index][var_index][type_index][0])
                                if num_app / self.num_updates_until_var_reduction >= self.var_reduction_thres:
                                    exceeded_thresh = True
                                    break
                                if num_app > tmp_max:
                                    tmp_max = num_app
                            else:
                                num_app = sum(self.var_type_history_list[event_index][var_index][type_index])
                                if num_app / self.num_updates_until_var_reduction >= self.var_reduction_thres:
                                    exceeded_thresh = True
                                    break
                                if num_app > tmp_max:
                                    tmp_max = num_app

                    # Remove the variable if it did not exceed the threshold
                    if not exceeded_thresh:
                        self.event_type_detector.check_variables[event_index][var_index] = False
                        self.event_type_detector.values[event_index][var_index] = []
                        self.var_type[event_index][var_index] = []
                        self.var_type_history_list[event_index][var_index] = []
                        if len(self.var_type_history_list_reference) > event_index and len(
                                self.var_type_history_list_reference[event_index]) > var_index:
                            self.var_type_history_list_reference[event_index][var_index] = []
                        affected_path = self.event_type_detector.variable_key_list[event_index][var_index]
                        self.print(
                            'Stopped tracking the variable with Path:\n%s\nbecause of irregular variable types.' %
                            affected_path, log_atom, affected_path, confidence=1 / (1 + np.exp(-4 / tmp_max)) / 0.9820137900379085)
                        # 1 / (1 + np.exp(-4 / tmp_max)) / 0.9820137900379085 is the scaled sigmoidfunction.
                        # 1 / (1 + np.exp(-4)) = 0.9820137900379085

            # Saves the initial reference state of the var_type_history_list for the calculation of the indicator
            if ((isinstance(self.num_updates_until_var_reduction, bool) and self.num_updates_until_var_reduction is False) or (
                    self.event_type_detector.num_eventlines[event_index] - self.num_init) / self.num_update >=
                    self.num_updates_until_var_reduction - 1) and (not isinstance(self.num_var_type_hist_ref, bool)) and (
                    (len(self.var_type_history_list_reference) < event_index + 1) or
                    self.var_type_history_list_reference[event_index] == []) and (
                    (self.event_type_detector.num_eventlines[event_index] - self.num_init) / self.num_update >=
                    self.num_var_type_hist_ref - 1):

                if len(self.var_type_history_list_reference) < event_index + 1:
                    for i in range(event_index + 1 - len(self.var_type_history_list_reference)):
                        self.var_type_history_list_reference.append([])

                for var_index in range(len(self.var_type_history_list[event_index])):
                    self.var_type_history_list_reference[event_index].append([])
                    for type_index in range(len(self.var_type_history_list[event_index][var_index])):
                        if len(self.var_type_history_list[event_index][var_index][type_index]) >= 1 and isinstance(
                                self.var_type_history_list[event_index][var_index][type_index][0], list):
                            if type_index == 6:  # continuously distributed variable type
                                self.var_type_history_list_reference[event_index][var_index].append([sum(
                                        self.var_type_history_list[event_index][var_index][type_index][0][-self.num_var_type_hist_ref:]) /
                                        max(len([1 for x in self.var_type_history_list[event_index][var_index][type_index][0][
                                            -self.num_var_type_hist_ref:] if x != 0]), 1), sum(
                                        self.var_type_history_list[event_index][var_index][type_index][1][-self.num_var_type_hist_ref:]) /
                                        max(len([1 for x in self.var_type_history_list[event_index][var_index][type_index][1][
                                            -self.num_var_type_hist_ref:] if x != 0]), 1)])
                            else:
                                self.var_type_history_list_reference[event_index][var_index].append([sum(x[
                                    -self.num_var_type_hist_ref:]) for x in self.var_type_history_list[event_index][var_index][type_index]])
                        else:
                            self.var_type_history_list_reference[event_index][var_index].append(
                                    sum(self.var_type_history_list[event_index][var_index][type_index][-self.num_var_type_hist_ref:]))

            # Checks the indicator for the varTypes of the Event and generates an output, if it fails
            else:
                if ((isinstance(self.num_updates_until_var_reduction, bool) and self.num_updates_until_var_reduction is False) or (
                        self.event_type_detector.num_eventlines[event_index] - self.num_init) /
                        self.num_update >= self.num_updates_until_var_reduction - 1) and (not isinstance(
                        self.num_var_type_considered_ind, bool)) and (not isinstance(self.num_var_type_hist_ref, bool)) and len(
                    self.var_type_history_list_reference) > event_index and (self.var_type_history_list_reference[event_index] != []) and (
                        ((self.event_type_detector.num_eventlines[event_index] - self.num_init) / self.num_update
                            - self.num_var_type_hist_ref) % self.num_var_type_considered_ind) == 0:

                    # Shorten the var_type_history_list
                    if len(self.var_type_history_list[event_index]) > 0 and len(self.var_type_history_list[event_index][0]) > 0 and len(
                        self.var_type_history_list[event_index][0][0]) > max(
                            self.num_var_type_considered_ind, self.num_var_type_hist_ref):
                        for var_index in range(len(self.var_type_history_list[event_index])):
                            for type_index in range(len(self.var_type_history_list[event_index][var_index])):
                                # Differentiation between the entries, which are lists (e.g. discrete) and values
                                if type(self.var_type_history_list[event_index][var_index][type_index][i]) is list:
                                    for i in range(len(self.var_type_history_list[event_index][var_index][type_index])):
                                        self.var_type_history_list[event_index][var_index][type_index][i] = \
                                            self.var_type_history_list[event_index][var_index][type_index][i][
                                            -max(self.num_var_type_considered_ind, self.num_var_type_hist_ref):]
                                else:
                                    self.var_type_history_list[event_index][var_index][type_index] = \
                                        self.var_type_history_list[event_index][var_index][type_index][
                                        -max(self.num_var_type_considered_ind, self.num_var_type_hist_ref):]

                    indicator_list = self.get_indicator(event_index)

                    indicator = max(0, max(indicator_list))
                    if indicator >= self.indicator_thres:

                        # Update the list of the failed indicators, which is used for the weights of the indicator
                        if len(self.failed_indicators) < event_index + 1:  # Extend the lists if necessary
                            tmp_len = len(self.failed_indicators)
                            for i in range(event_index + 1 - tmp_len):
                                self.failed_indicators.append([[] for _ in range(len(self.var_type[tmp_len + i]))])

                        # indices of the variables, which would have failed the indicator
                        indices_failed_tests = []
                        for var_index in range(len(self.var_type[event_index])):
                            if indicator_list[var_index] >= self.indicator_thres:
                                indices_failed_tests.append(var_index)
                                self.failed_indicators[event_index][var_index].append(self.event_type_detector.num_eventlines[event_index])

                        # Multiply the single values of the indicator with their corresponding weights
                        # Number of the log line which corresponds to the first indicator, which is taken into account
                        first_line_num = self.event_type_detector.num_eventlines[event_index] - self.num_update *\
                            self.num_var_type_considered_ind * (self.num_ind_for_weights + self.num_skipped_ind_for_weights)
                        # Number of the log line which corresponds to the last indicator, which is taken into account
                        last_line_num = self.event_type_detector.num_eventlines[event_index] - self.num_update *\
                            self.num_var_type_considered_ind * self.num_skipped_ind_for_weights

                        for var_index in indices_failed_tests:
                            lower_ind = False  # Index of the lower bound of the considered values of the failed_indicator list
                            upper_ind = False  # Index of the upper bound of the considered values of the failed_indicator list

                            for i in range(len(self.failed_indicators[event_index][var_index])):
                                if self.failed_indicators[event_index][var_index][i] >= first_line_num:
                                    lower_ind = i
                                    break

                            if isinstance(lower_ind, bool):
                                lower_ind = 0
                                upper_ind = 0
                            else:
                                for i in range(lower_ind, len(self.failed_indicators[event_index][var_index])):
                                    if self.failed_indicators[event_index][var_index][i] >= last_line_num:
                                        upper_ind = i
                                        break
                                if isinstance(upper_ind, bool):
                                    upper_ind = len(self.failed_indicators[event_index][var_index])

                            # Calculating the weight for the indicator
                            indicator_weight = 1 / (1 + upper_ind - lower_ind)
                            indicator_list[var_index] = indicator_list[var_index] * indicator_weight

                            # Reduce the list of the failed indicators
                            self.failed_indicators[event_index][var_index] = self.failed_indicators[event_index][var_index][lower_ind:]

                        # calculate and print the confidence of the failed indicator
                        indicator = sum(indicator_list[var_index] for var_index in indices_failed_tests)

                        if self.save_statistics:
                            self.failed_indicators_total.append(self.event_type_detector.total_records)
                            self.failed_indicators_values.append(np.arctan(2 * indicator) / np.pi * 2)
                            self.failed_indicators_paths.append(self.event_type_detector.longest_path[event_index])

                        tmp_string = ''
                        affected_paths = [self.event_type_detector.variable_key_list[event_index][var_index] for var_index in
                                          indices_failed_tests]
                        if self.var_type_history_list:
                            tmp_string += 'Indicator of a change in system behaviour: %s. Paths to the corresponding variables: %s' % (
                                np.arctan(2 * indicator) / np.pi * 2, affected_paths)

                        self.print(tmp_string, log_atom, affected_paths, np.arctan(2 * indicator) / np.pi * 2, indicator=True)

                # Update the var_type_history_list_reference
                if self.update_var_type_bool and (not isinstance(self.num_var_type_hist_ref, bool)) and (
                        not isinstance(self.num_update_var_type_hist_ref, bool)) and len(
                        self.var_type_history_list_reference) >= event_index + 1 and \
                        self.var_type_history_list_reference[event_index] != [] and (((
                        self.event_type_detector.num_eventlines[event_index] - self.num_init) / self.num_update
                        - self.num_var_type_hist_ref) % self.num_update_var_type_hist_ref == 0):

                    for var_index in range(len(self.var_type_history_list[event_index])):
                        self.var_type_history_list_reference[event_index][var_index] = []
                        for type_index in range(len(self.var_type_history_list[event_index][var_index])):
                            if len(self.var_type_history_list[event_index][var_index][type_index]) >= 1 and isinstance(
                                    self.var_type_history_list[event_index][var_index][type_index][0], list):
                                if type_index == 6:  # continuously distributed variable type
                                    self.var_type_history_list_reference[event_index][var_index].append([sum(
                                            self.var_type_history_list[event_index][var_index][type_index][0][
                                                -self.num_var_type_hist_ref:]) / max(len([1 for x in self.var_type_history_list[
                                                    event_index][var_index][type_index][0][-self.num_var_type_hist_ref:] if x != 0]), 1),
                                            sum(self.var_type_history_list[event_index][var_index][type_index][1][
                                                -self.num_var_type_hist_ref:]) / max(len([1 for x in self.var_type_history_list[
                                                    event_index][var_index][type_index][1][-self.num_var_type_hist_ref:] if x != 0]), 1)])
                                else:
                                    self.var_type_history_list_reference[event_index][var_index].append(
                                            [sum(x[-self.num_var_type_hist_ref:]) for x in
                                                self.var_type_history_list[event_index][var_index][type_index]])
                            else:
                                self.var_type_history_list_reference[event_index][var_index].append(sum(
                                        self.var_type_history_list[event_index][var_index][type_index][-self.num_var_type_hist_ref:]))

    def detect_var_type(self, event_index, var_index):
        """Give back the assumed variableType of the variable with the in self.event_type_detector stored values."""
        # Values which are being tested
        values = self.event_type_detector.values[event_index][var_index][-self.num_init:]
        # Unique values
        values_set = set(values)
        # Number of unique values
        num_diff_vals = len(values_set)

        if num_diff_vals == 1:
            return ['stat', list(values_set), False]

        # List of floats or False
        float_values = convert_to_floats(values)
        is_int = False
        if len(float_values) > 0:
            is_int = consists_of_ints(float_values)

        # Values are integer numbers
        if len(float_values) > 0:
            previous_val = float_values[0]
            asc = True
            desc = True
            # Test for ascending
            for v in float_values[1:]:
                if previous_val > v:
                    asc = False
                    break
                previous_val = v

            previous_val = float_values[0]
            # Test for descending
            for v in float_values[1:]:
                if previous_val < v:
                    desc = False
                    break
                previous_val = v
            if asc:
                if is_int:
                    return ['asc', 'int']
                return ['asc', 'float']
            if desc:
                if is_int:
                    return ['desc', 'int']
                return ['desc', 'float']

        # Checking if no integers should be tested and if the values are integers
        if not self.test_ks_int and is_int:
            float_values = []

        if len(float_values) > 0 and (num_diff_vals > self.div_thres * self.num_init):
            # test for a continuous distribution. If none fits, the function will return ['d']
            var_type = self.detect_continuous_shape(float_values)
        else:
            # discrete var type
            var_type = ['d']

        # Test for discrete, unique and others
        if var_type == ['d']:
            if self.num_init == num_diff_vals and (len(float_values) == 0 or is_int):
                # unique var type
                return ['unq', values]
            if num_diff_vals >= self.num_init * (1 - self.sim_thres):
                # values do not follow a specific pattern, the second entry is the number of update runs without a new type.
                return ['others', 0]
            # Initialize the discrete type
            values_set = list(values_set)
            values_app = [0 for _ in range(num_diff_vals)]
            for value in values:
                values_app[values_set.index(value)] += 1

            values_app = [x / len(values) for x in values_app]
            # discrete var type
            return ['d', values_set, values_app, len(values)]
        return var_type

    def detect_continuous_shape(self, values):
        """
        Detect if the sample follows one of the checked continuous distribution and returns the found type in a fitting format.
        ['d'] if none fit.
        """
        # List of the p-values of the distributions
        significance = []
        # List of the tested distributions
        distribution = []

        # Converts the floats/integer to an array for faster manipulations and tests
        values = np.array(values)

        # Test for uniform distribution
        Min = min(values)
        Max = max(values)
        if self.ks_alpha in self.crit_val_ini and self.num_init in self.crit_val_ini[self.ks_alpha]:
            significance.append(self.ks_alpha * self.crit_val_ini[self.ks_alpha][self.num_init][
                'uni'] / kstest(values, 'uniform', args=(Min, Max - Min))[0])
            distribution.append(['uni', Min, Max])
        else:
            significance.append(kstest(values, 'uniform', args=(Min, Max - Min))[1])
            distribution.append(['uni', Min, Max])

        # Test for normal distribution
        # Getting the expected value and sigma
        [EV, sigma] = norm.fit(values)

        # KS-test of the standardised values and the distribution
        if self.ks_alpha in self.crit_val_ini and self.num_init in self.crit_val_ini[self.ks_alpha]:
            significance.append(self.ks_alpha * self.crit_val_ini[self.ks_alpha][self.num_init][
                'nor'] / kstest((values - EV) / sigma, 'norm')[0])
            distribution.append(['nor', EV, sigma, Min, Max])
        else:
            significance.append(kstest((values - EV) / sigma, 'norm')[1])
            distribution.append(['nor', EV, sigma, Min, Max])

        # Test for beta distribution
        # (0.5*0.5/((0.5+0.5+1)(0.5+0.5)^2))^(1/2) = 2.82842712
        EV_tmp = (Min + Max) / 2
        sigma_tmp = (Max - Min) / 2.82842712

        if self.ks_alpha in self.crit_val_ini and self.num_init in self.crit_val_ini[self.ks_alpha]:
            significance.append(self.ks_alpha * self.crit_val_ini[self.ks_alpha][self.num_init][
                'beta1'] / kstest((values-Min)/(Max-Min), 'beta', args=(0.5, 0.5))[0])
            distribution.append(['beta', EV_tmp, sigma_tmp, Min, Max, 1])
        else:
            significance.append(kstest((values-Min)/(Max-Min), 'beta', args=(0.5, 0.5))[1])
            distribution.append(['beta', EV_tmp, sigma_tmp, Min, Max, 1])

        # KS-test of the standardised values and the distribution
        if self.ks_alpha in self.crit_val_ini and self.num_init in self.crit_val_ini[self.ks_alpha]:
            # Beta 2
            significance.append(self.ks_alpha * self.crit_val_ini[self.ks_alpha][self.num_init][
                'beta2'] / kstest((values-EV)/sigma*pow(5*2/(5+2+1), 1/2)/(5+2)+5/(5+2), 'beta', args=(5, 2))[0])
            distribution.append(['beta', EV, sigma, Min, Max, 2])

            # Beta 3
            significance.append(self.ks_alpha * self.crit_val_ini[self.ks_alpha][self.num_init][
                'beta2'] / kstest((values-EV)/sigma*pow(5*2/(5+2+1), 1/2)/(5+2)+2/(5+2), 'beta', args=(2, 5))[0])
            distribution.append(['beta', EV, sigma, Min, Max, 3])

            # Beta 4
            significance.append(self.ks_alpha * self.crit_val_ini[self.ks_alpha][self.num_init][
                'beta4'] / kstest((values-EV)/sigma*pow(1*5/(1+5+1), 1/2)/(1+5)+1/(1+5), 'beta', args=(1, 5))[0])
            distribution.append(['beta', EV, sigma, Min, Max, 4])

            # Beta 5
            significance.append(self.ks_alpha * self.crit_val_ini[self.ks_alpha][self.num_init][
                'beta4'] / kstest((values-EV)/sigma*pow(1*5/(1+5+1), 1/2)/(1+5)+5/(1+5), 'beta', args=(5, 1))[0])
            distribution.append(['beta', EV, sigma, Min, Max, 5])
        else:
            # Beta 2
            significance.append(kstest((values-EV)/sigma*pow(5*2/(5+2+1), 1/2)/(5+2)+5/(5+2), 'beta', args=(5, 2))[1])
            distribution.append(['beta', EV, sigma, Min, Max, 2])

            # Beta 3
            significance.append(kstest((values-EV)/sigma*pow(5*2/(5+2+1), 1/2)/(5+2)+2/(5+2), 'beta', args=(2, 5))[1])
            distribution.append(['beta', EV, sigma, Min, Max, 3])

            # Beta 4
            significance.append(kstest((values-EV)/sigma*pow(1*5/(1+5+1), 1/2)/(1+5)+1/(1+5), 'beta', args=(1, 5))[1])
            distribution.append(['beta', EV, sigma, Min, Max, 4])

            # Beta 5
            significance.append(kstest((values-EV)/sigma*pow(1*5/(1+5+1), 1/2)/(1+5)+5/(1+5), 'beta', args=(5, 1))[1])
            distribution.append(['beta', EV, sigma, Min, Max, 5])

        # Maximal distance for the self generated or mixed distributions
        max_val = pow(-np.log(self.ks_alpha) * 3 / self.num_init / 4, 1 / 2)
        est_penalty = 1.4  # Estimated penalty for the adapted EV and SD

        # Test for the mixed beta distribution
        # EV/Sigma of Beta 4: EV=1/(1+5)   Sigma=pow(1*5/(1+5+1),1/5)/(1+5)
        # sigma in [sigmaBetam1,sigmaBetam2]
        if 1 / 6 < (EV - Min) / (Max - Min) < 5 / 6:
            # Interpolate the expected distribution functions threw the sigma in the interval
            proportion = ((EV - Min) / (Max - Min) - 5 / 6) / (-4 / 6)
            tmp_index = [int(round(i / proportion)) for i in range(int(round(1000 * proportion)))]

            if self.ks_alpha in self.crit_val_ini and self.num_init in self.crit_val_ini[self.ks_alpha]:
                significance.append(ks_2samp([self.distribution_data['betam1'][i] for i in tmp_index] + [self.distribution_data['betam2'][
                    i] for i in range(1000) if i not in tmp_index], (values - Min) / (Max - Min))[0] / max_val * est_penalty)
                distribution.append(['betam', Min, Max - Min, Min, Max, proportion])
            else:
                significance.append(ks_2samp([self.distribution_data['betam1'][i] for i in tmp_index] + [self.distribution_data['betam2'][
                    i] for i in range(1000) if i not in tmp_index], (values - Min) / (Max - Min))[1])
                distribution.append(['betam', Min, Max - Min, Min, Max, proportion])

        # Test for alternative distribution
        # KS-test of the standardised values and the distribution
        if self.ks_alpha in self.crit_val_ini and self.num_init in self.crit_val_ini[self.ks_alpha]:
            significance.append(ks_2samp(self.distribution_data['spec'], (values - EV) / sigma)[0] / max_val * est_penalty)
            distribution.append(['spec', EV, sigma, Min, Max, 0])

            significance.append(
              ks_2samp(self.distribution_data['spec'], -(values - EV) / sigma)[0] / max_val * est_penalty)
            distribution.append(['spec', EV, sigma, Min, Max, 1])
        else:
            significance.append(ks_2samp(self.distribution_data['spec'], (values - EV) / sigma)[1])
            distribution.append(['spec', EV, sigma, Min, Max, 0])

            significance.append(ks_2samp(self.distribution_data['spec'], -(values - EV) / sigma)[1])
            distribution.append(['spec', EV, sigma, Min, Max, 1])

        # Checks if one of the above tested continuous distribution fits
        if max(significance) >= self.ks_alpha:
            sort_indices = np.argsort(significance)
            sort_list = []
            for i in range(len(sort_indices) - 2, -1, -1):
                if significance[sort_indices[i]] >= self.ks_alpha:
                    sort_list.append(distribution[sort_indices[i]])
            return distribution[sort_indices[-1]] + [sort_list]

        if self.use_empiric_distr:
            return ['emp', EV, sigma]
        # discrete if no distribution fits
        return ['d']

    def update_var_type(self, event_index, var_index, log_atom):
        """Test if the new num_update values fit the detected var type and updates the var type if the test fails."""
        # Getting the new values and saving the old distribution for printing-purposes if the test fails
        new_values = self.event_type_detector.values[event_index][var_index][-self.num_update:]
        VT_old = self.var_type[event_index][var_index]

        # Test and update for continuous distribution
        if self.var_type[event_index][var_index][0] in self.distr_list:
            if not consists_of_floats(new_values):
                # A value is not a float or integer, so the new assigned type is others
                # values do not follow a specific pattern
                self.var_type[event_index][var_index] = ['others', 0]
                self.distr_val[event_index][var_index] = []
                self.bt_results[event_index][var_index] = []
                self.print_changed_var_type(event_index, VT_old, ['others'], var_index, log_atom, 1.0)
                return

            # Used to test the current distribution with the BT and to discard the alternative distributions if they fail the sKS-test once
            first_distr = True
            s_ks_result = self.s_ks_test(event_index, var_index, first_distr)
            # Calculate the confidence as the stretched sigmaoid function of the maximal distance of the step fct
            # 1 / (1 + np.exp(-2)) = 1.1353352832366128
            confidence = 1 / (1 + np.exp(-2 * s_ks_result[1])) * 1.1353352832366128
            while not s_ks_result[0]:  # If the test fails a new shape is searched for in the alternative distributions
                self.bt_results[event_index][var_index] = self.bt_results[event_index][var_index][1:] + [0]  # Update the results of the BT
                first_distr = False

                # Checks if the BT is applicable and if it holds
                if first_distr and (sum(self.bt_results[event_index][var_index]) >= self.s_ks_bt_min_success):
                    return

                if not self.update_var_type_bool:  # Do not update variableType
                    self.bt_results[event_index][var_index] = [1] * self.num_s_ks_bt
                    self.print_reject_var_type(event_index, self.var_type[event_index][var_index], var_index, log_atom)
                    self.var_type_history_list[event_index][var_index][0][-1] = 1
                    return

                if len(self.alternative_distribution_types[event_index][var_index]) != 0:  # There is at least one alternative distribution
                    # Initializes the distributionvalues and bucketnumbers
                    self.var_type[event_index][var_index] = self.alternative_distribution_types[event_index][var_index][0]
                    self.alternative_distribution_types[event_index][var_index] = self.alternative_distribution_types[event_index][
                        var_index][1:]
                    self.bt_results[event_index][var_index] = [1] * self.num_s_ks_bt
                    if self.var_type[event_index][var_index][0] == 'betam' or self.var_type[event_index][var_index][0] == 'spec':
                        self.s_ks_get_quantiles(event_index, var_index)

                    s_ks_result = self.s_ks_test(event_index, var_index, first_distr)

                # There is no alternative distribution. The var type is set to others
                else:
                    # values do not follow a specific pattern
                    self.var_type[event_index][var_index] = ['others', 0]
                    self.distr_val[event_index][var_index] = []
                    self.bt_results[event_index][var_index] = []
                    self.print_changed_var_type(event_index, VT_old, ['others'], var_index, log_atom, confidence)
                    return

            # Checks if the s_ks_test was successful and remarks the success
            if first_distr:
                self.bt_results[event_index][var_index] = self.bt_results[event_index][var_index][1:] + [1]

            # Print a message if the vartype has changed
            if VT_old != self.var_type[event_index][var_index]:
                self.print_changed_var_type(event_index, VT_old, self.var_type[event_index][var_index], var_index, log_atom, confidence)

        # Test and update for ascending values
        elif self.var_type[event_index][var_index][0] == 'asc':
            # Searches for a not ascending sequence in the values
            for j in range(-self.num_update, 0):
                if self.event_type_detector.values[event_index][var_index][j - 1] >\
                        self.event_type_detector.values[event_index][var_index][j]:
                    # Do not update variableType
                    if not self.update_var_type_bool:
                        self.print_reject_var_type(event_index, self.var_type[event_index][var_index], var_index, log_atom)
                        self.var_type_history_list[event_index][var_index][0][-1] = 1
                        return

                    self.var_type[event_index][var_index] = ['others', 0]  # values do not follow a specific pattern
                    self.print_changed_var_type(event_index, VT_old, ['others'], var_index, log_atom)
                    return

        elif self.var_type[event_index][var_index][0] == 'desc':  # Test and update for descending values
            for j in range(-self.num_update, 0):  # Searches for a not ascending sequence in the values
                if self.event_type_detector.values[event_index][var_index][j - 1] <\
                        self.event_type_detector.values[event_index][var_index][j]:
                    if not self.update_var_type_bool:  # Do not update variableType
                        self.print_reject_var_type(event_index, self.var_type[event_index][var_index], var_index, log_atom)
                        self.var_type_history_list[event_index][var_index][0][-1] = 1
                        return
                    self.var_type[event_index][var_index] = ['others', 0]  # values do not follow a specific pattern
                    self.print_changed_var_type(event_index, VT_old, ['others'], var_index, log_atom)
                    return

        elif self.var_type[event_index][var_index][0] == 'd':  # Test and update for values of the discrete type
            # Checks if new values have appeared
            if len(set(new_values + self.var_type[event_index][var_index][1])) > len(self.var_type[event_index][var_index][1]):
                # New values have appeared
                # Test if vartype others
                if len(set(new_values + self.var_type[event_index][var_index][1])) >= (
                        self.num_update + self.var_type[event_index][var_index][3]) * (1 - self.sim_thres):
                    # Do not update variableType
                    if not self.update_var_type_bool:
                        self.print_reject_var_type(event_index, self.var_type[event_index][var_index], var_index, log_atom)
                        self.var_type_history_list[event_index][var_index][0][-1] = 1
                        return
                    self.var_type[event_index][var_index] = ['others', 0]  # values do not follow a specific pattern
                    self.print_changed_var_type(event_index, VT_old, ['others'], var_index, log_atom)
                    return

                # Do not update variableType
                if not self.update_var_type_bool:
                    self.print_reject_var_type(event_index, self.var_type[event_index][var_index], var_index, log_atom)
                    self.var_type_history_list[event_index][var_index][2][1][-1] = 1
                    return

                # Create the new value-set and expands the occurrence-list for the new values
                new_values_set = list(set(self.event_type_detector.values[event_index][var_index][-self.num_init:]))
                for val in new_values_set:
                    if val not in self.var_type[event_index][var_index][1]:
                        self.var_type[event_index][var_index][1].append(val)
                        self.var_type[event_index][var_index][2].append(0)

                # update the occurrences
                # List for the appearances of the new values
                values_app = list([0] * len(self.var_type[event_index][var_index][1]))
                for i in range(-self.num_init, 0):
                    values_app[self.var_type[event_index][var_index][1].index(
                        self.event_type_detector.values[event_index][var_index][i])] += 1

                tmp_number = self.var_type[event_index][var_index][3] / (
                        self.num_update + self.var_type[event_index][var_index][3])
                # Updates the appearance-list in the var type of the discrete variable
                for j in range(len(self.var_type[event_index][var_index][2])):
                    self.var_type[event_index][var_index][2][j] = \
                        self.var_type[event_index][var_index][2][j] * tmp_number + values_app[j] / (
                            self.num_update + self.var_type[event_index][var_index][3])

                self.var_type[event_index][var_index][3] = self.num_update + self.var_type[event_index][var_index][3]

                self.d_init_bt(event_index, var_index)
                self.print_changed_var_type(event_index, VT_old, self.var_type[event_index][var_index], var_index, log_atom)
                self.var_type_history_list[event_index][var_index][2][1][-1] = 1
                return

            # No new values have appeared, so the normal test for discrete variables is used
            self.d_test(event_index, var_index)

            # Check if the values should be considered others or if the BT failed
            if (len(set(new_values + self.var_type[event_index][var_index][1])) >= (
                    self.num_update + self.var_type[event_index][var_index][3]) * (1 - self.sim_thres)) or (sum(
                        self.bt_results[event_index][var_index][0]) < self.d_bt_min_success):

                # Do not update variableType
                if not self.update_var_type_bool:
                    self.print_reject_var_type(event_index, self.var_type[event_index][var_index], var_index, log_atom)
                    self.bt_results[event_index][var_index][0] = [1] * self.num_d_bt
                    self.var_type_history_list[event_index][var_index][0][-1] = 1
                    return

                # values do not follow a specific pattern
                self.var_type[event_index][var_index] = ['others', 0]
                self.print_changed_var_type(event_index, VT_old, ['others'], var_index, log_atom)
                return

            # Update the probabilities of the discrete values
            if self.update_var_type_bool and self.bt_results[event_index][var_index][0][-1]:
                # List for the number of appearance of the values
                values_app = [0 for x in range(len(self.var_type[event_index][var_index][1]))]
                for val in new_values:
                    values_app[self.var_type[event_index][var_index][1].index(val)] += 1

                tmp_number = self.var_type[event_index][var_index][3] / (
                    self.num_update + self.var_type[event_index][var_index][3])
                # Updates the appearance-list in the var type of the discrete variable
                for j in range(len(self.var_type[event_index][var_index][2])):
                    self.var_type[event_index][var_index][2][j] = \
                        self.var_type[event_index][var_index][2][j] * tmp_number + values_app[j] / (
                            self.num_update + self.var_type[event_index][var_index][3])

                self.var_type[event_index][var_index][3] = self.num_update + self.var_type[event_index][var_index][3]

                # Check if the discrete distribution has to be updated
                if ((self.var_type[event_index][var_index][3] - self.num_init) % self.num_pause_discrete) == 0:
                    self.d_init_bt(event_index, var_index)
                return

        # Test and update for static variables
        elif self.var_type[event_index][var_index][0] == 'stat':
            # Check if still static
            if all(new_values[i] == self.event_type_detector.values[event_index][var_index][0] for i in range(self.num_update)):
                if self.var_type[event_index][var_index][2] and self.num_stat_stop_update is True and \
                        self.event_type_detector.num_eventlines[event_index] >= self.num_stat_stop_update:
                    self.event_type_detector.check_variables[event_index][var_index] = False
                    self.event_type_detector.values[event_index][var_index] = []
                    self.var_type[event_index][var_index] = []
                    self.var_type_history_list[event_index][var_index] = []
                    if len(self.var_type_history_list_reference) > event_index and len(self.var_type_history_list_reference[event_index]) >\
                            var_index:
                        self.var_type_history_list_reference[event_index][var_index] = []

                    affected_path = self.event_type_detector.variable_key_list[event_index][var_index]
                    self.print('Stopped tracking the variable with Path:\n%s\nbecause of its static values.' %
                               affected_path, log_atom, affected_path, confidence=1 - 1 / self.num_stat_stop_update)
                return

            # Do not update variableType
            if not self.update_var_type_bool:
                self.print_reject_var_type(event_index, self.var_type[event_index][var_index], var_index, log_atom)
                self.var_type_history_list[event_index][var_index][0][-1] = 1
                return

            # Check if new values appear to be of type others
            if len(set(new_values)) >= self.num_update * (1 - self.sim_thres) and self.num_update >= 3:
                self.var_type[event_index][var_index] = ['others', 0]  # values do not follow a specific pattern
                self.print_changed_var_type(event_index, VT_old, ['others'], var_index, log_atom)
                return
            # Change the var type from static to discrete

            # list of the values
            values_set = list(set(self.event_type_detector.values[event_index][var_index][-self.num_init:]))
            values_app = [0 for _ in range(len(values_set))]  # List to store the appearance of the values

            for j in range(-self.num_init, 0):
                values_app[values_set.index(self.event_type_detector.values[event_index][var_index][j])] += 1
            values_app = [x / self.num_init for x in values_app]

            # values follow a discrete pattern
            self.var_type[event_index][var_index] = ['d', values_set, values_app, self.num_init]
            self.d_init_bt(event_index, var_index)
            self.print_changed_var_type(event_index, VT_old, self.var_type[event_index][var_index], var_index, log_atom)
            return

        # Test and update for unique values
        elif self.var_type[event_index][var_index][0] == 'unq':
            # Checks if the new values are not unique
            if len(set(self.event_type_detector.values[event_index][var_index][-self.num_update:])) != self.num_update:
                if not self.update_var_type_bool:  # Do not update variableType
                    self.print_reject_var_type(event_index, self.var_type[event_index][var_index], var_index, log_atom)
                    self.var_type_history_list[event_index][var_index][0][-1] = 1
                    return

                self.var_type[event_index][var_index] = ['others', 0]
                self.print_changed_var_type(event_index, VT_old, ['others'], var_index, log_atom)
                return

            # Check if one of the new values has appeared in the last self.num_update_unq values
            for j in self.event_type_detector.values[event_index][var_index][-self.num_update:]:
                if j in self.event_type_detector.values[event_index][var_index][
                        -self.num_update_unq - self.num_update:-self.num_update]:
                    # Do not update variableType
                    if not self.update_var_type_bool:
                        self.print_reject_var_type(event_index, self.var_type[event_index][var_index], var_index, log_atom)
                        self.var_type_history_list[event_index][var_index][0][-1] = 1
                        return

                    self.var_type[event_index][var_index] = ['others', 0]
                    self.print_changed_var_type(event_index, VT_old, ['others'], var_index, log_atom)
                    return
            return

        # Update for var type others
        elif self.var_type[event_index][var_index][0] == 'others':
            # Do not update variableType
            if not self.update_var_type_bool:
                return

            # Check if it has passed enough time, to check if the values have a new var_type
            if (self.var_type[event_index][var_index][1] + 1) % (self.num_pause_others + 1) == 0:
                # Added a exponential waiting time to avoid redundant tests
                if not consists_of_ints([np.log2((self.var_type[event_index][var_index][1] + 1) / (self.num_pause_others + 1))]):
                    self.var_type[event_index][var_index][1] += 1
                    return

                # Checking for a new var_type
                vt_new = self.detect_var_type(event_index, var_index)
                # Only increase the number of skipped update-cycles
                if vt_new[0] == 'others':
                    self.var_type[event_index][var_index][1] += 1
                    return

                # The variable gets assigned a new var_type
                # VarType is empiric distribution
                if vt_new[0] == 'emp':
                    self.var_type[event_index][var_index] = vt_new
                    self.bt_results[event_index][var_index] = [1] * self.num_s_ks_bt
                    self.s_ks_get_quantiles(event_index, var_index)

                # VarType is a continuous distribution
                elif vt_new[0] in self.distr_list:
                    self.var_type[event_index][var_index] = vt_new[:-1]
                    self.alternative_distribution_types[event_index][var_index] = vt_new[-1]
                    self.bt_results[event_index][var_index] = [1] * self.num_s_ks_bt
                    if self.var_type[event_index][var_index][0] == 'betam' or self.var_type[event_index][var_index][0] == 'spec':
                        self.s_ks_get_quantiles(event_index, var_index)

                # VarType is discrete
                elif vt_new[0] == 'd':
                    self.var_type[event_index][var_index] = vt_new
                    self.d_init_bt(event_index, var_index)

                else:
                    self.var_type[event_index][var_index] = vt_new
                self.print_changed_var_type(event_index, ['others'], vt_new, var_index, log_atom)
            else:
                self.var_type[event_index][var_index][1] += 1
            return

    def s_ks_get_quantiles(self, event_index, var_index):
        """Generate the needed quantiles of the distribution for the sliding KS-test."""
        if self.var_type[event_index][var_index][0] == 'emp':
            # Get a list of almost equidistant indices
            indices = [int(i) for i in [self.num_init * j / (2 * self.num_s_ks_values) for j in
                       range(2 * self.num_s_ks_values)]]

            # Get the list of values and sort them
            sorted_values = copy.copy(self.event_type_detector.values[event_index][var_index][-self.num_init:])
            sorted_values.sort()

            # Generate the list of distribution values
            distr_val = []
            for index in indices:
                distr_val.append(sorted_values[index])

            self.distr_val[event_index][var_index] = distr_val
            return

        if self.var_type[event_index][var_index][0] == 'spec':  # Calculate the quantiles of the special distribution
            EV = self.var_type[event_index][var_index][1]
            sigma = self.var_type[event_index][var_index][2]

            indices = 0 + np.array(range(2 * self.num_s_ks_values)) / (2 * self.num_s_ks_values - 1) * (
                    1000 - 1)
            indices = indices.astype(int)

            # Generate the quantiles for the var type with the standardised quantiles
            self.distr_val[event_index][var_index] = self.distribution_data['spec'][indices] * sigma + EV
            return

        # Calculate the quantiles of the mixed beta distribution
        if self.var_type[event_index][var_index][0] == 'betam':
            Min = self.var_type[event_index][var_index][1]
            scale = self.var_type[event_index][var_index][2]
            proportion = self.var_type[event_index][var_index][5]

            indices1 = [int(round(i / proportion)) for i in range(int(round(1000 * proportion)))]
            indices2 = [i for i in range(1000) if i not in indices1]

            # Generate the quantiles for the var type with the standardised quantiles
            self.distr_val[event_index][var_index] = self.distribution_data['betam1'][indices1] * scale + Min
            self.distr_val[event_index][var_index] = np.append(
                self.distr_val[event_index][var_index], self.distribution_data['betam2'][indices2] * scale + Min)
            self.distr_val[event_index][var_index].sort()
            return

    def s_ks_test(self, event_index, var_index, first_distr):
        """
        Make a KS-test solely with the bucketlist.
        @return a list with the first entry True/False and as the second entry the maximal distance of the step functions
        """
        num_distr_val = 2 * self.num_s_ks_values

        # Calculate the critical distance for the KS-test
        # The parameters are in the list of the critical distances
        distribution = self.var_type[event_index][var_index][0]
        if distribution == 'beta':
            distribution += str(self.var_type[event_index][var_index][-1])
        if self.s_ks_alpha in self.crit_val_upd and self.num_init in self.crit_val_upd[self.s_ks_alpha] \
                and self.num_s_ks_values in self.crit_val_upd[self.s_ks_alpha][self.num_init] \
                and distribution in self.crit_val_upd[self.s_ks_alpha][self.num_init][self.num_s_ks_values]:
            crit_valance = \
                self.crit_val_upd[self.s_ks_alpha][self.num_init][self.num_s_ks_values][distribution]
        else:
            crit_valance = ((num_distr_val + self.num_s_ks_values) * (np.log(2 / self.s_ks_alpha)) / (
                2 * num_distr_val * self.num_s_ks_values)) ** (1 / 2)

        # scipy KS-test for uniformal distribution
        if self.var_type[event_index][var_index][0] == 'uni':
            max_val = kstest(
                self.event_type_detector.values[event_index][var_index][-self.num_s_ks_values:], 'uniform',
                args=(self.var_type[event_index][var_index][1], self.var_type[event_index][var_index][2]-self.var_type[event_index][
                    var_index][1]))[0]
            if first_distr:
                if max_val > crit_valance:
                    return [False, max_val]
                return [True, max_val]
            if max_val > crit_valance:
                return [False, 1.0]
            return [True, 0.0]

        # scipy KS-test for normal distribution
        if self.var_type[event_index][var_index][0] == 'nor':
            max_val = kstest(self.event_type_detector.values[event_index][var_index][-self.num_s_ks_values:], 'norm', args=(
                self.var_type[event_index][var_index][1], self.var_type[event_index][var_index][2]))[0]
            if first_distr:
                if max_val > crit_valance:
                    return [False, max_val]
                return [True, max_val]
            if max_val > crit_valance:
                return [False, 1.0]
            return [True, 0.0]

        # scipy KS-test for beta distributions
        if self.var_type[event_index][var_index][0] == 'beta':
            if self.var_type[event_index][var_index][5] == 1:
                max_val = kstest(self.event_type_detector.values[event_index][var_index][-self.num_s_ks_values:], 'beta', args=(
                    0.5, 0.5, self.var_type[event_index][var_index][3], self.var_type[event_index][var_index][4] - self.var_type[
                        event_index][var_index][3]))[0]
                if first_distr:
                    if max_val > crit_valance:
                        return [False, max_val]
                    return [True, max_val]
                if max_val > crit_valance:
                    return [False, 1.0]
                return [True, 0.0]
            if self.var_type[event_index][var_index][5] == 2:
                # Mu and sigma of the desired distribution
                [mu, sigma] = [5 / (5 + 2), pow(5 * 2 / (5 + 2 + 1), 1 / 2) / (5 + 2)]
                max_val = kstest(self.event_type_detector.values[event_index][var_index][-self.num_s_ks_values:], 'beta', args=(
                        5, 2, self.var_type[event_index][var_index][1] - mu * self.var_type[event_index][var_index][2] / sigma,
                        self.var_type[event_index][var_index][2] / sigma))[0]
                if first_distr:
                    if max_val > crit_valance:
                        return [False, max_val]
                    return [True, max_val]
                if max_val > crit_valance:
                    return [False, 1.0]
                return [True, 0.0]
            if self.var_type[event_index][var_index][5] == 3:
                # Mu and sigma of the desired distribution
                [mu, sigma] = [2 / (5 + 2), pow(5 * 2 / (5 + 2 + 1), 1 / 2) / (5 + 2)]
                max_val = kstest(self.event_type_detector.values[event_index][var_index][-self.num_s_ks_values:], 'beta', args=(
                    2, 5, self.var_type[event_index][var_index][1] - mu * self.var_type[event_index][var_index][2] / sigma,
                    self.var_type[event_index][var_index][2] / sigma))[0]
                if first_distr:
                    if max_val > crit_valance:
                        return [False, max_val]
                    return [True, max_val]
                if max_val > crit_valance:
                    return [False, 1.0]
                return [True, 0.0]
            if self.var_type[event_index][var_index][5] == 4:
                # Mu and sigma of the desired distribution
                [mu, sigma] = [1 / (5 + 1), pow(5 * 1 / (5 + 1 + 1), 1 / 2) / (5 + 1)]
                max_val = kstest(self.event_type_detector.values[event_index][var_index][-self.num_s_ks_values:], 'beta', args=(
                    1, 5, self.var_type[event_index][var_index][1] - mu * self.var_type[event_index][var_index][2] / sigma,
                    self.var_type[event_index][var_index][2] / sigma))[0]
                if first_distr:
                    if max_val > crit_valance:
                        return [False, max_val]
                    return [True, max_val]
                if max_val > crit_valance:
                    return [False, 1.0]
                return [True, 0.0]
            if self.var_type[event_index][var_index][5] == 5:
                # Mu and sigma of the desired distribution
                [mu, sigma] = [5 / (5 + 1), pow(5 * 1 / (5 + 1 + 1), 1 / 2) / (5 + 1)]
                max_val = kstest(self.event_type_detector.values[event_index][var_index][-self.num_s_ks_values:], 'beta', args=(
                    5, 1, self.var_type[event_index][var_index][1] - mu * self.var_type[event_index][var_index][2] / sigma,
                    self.var_type[event_index][var_index][2] / sigma))[0]
                if first_distr:
                    if max_val > crit_valance:
                        return [False, max_val]
                    return [True, max_val]
                if max_val > crit_valance:
                    return [False, 1.0]
                return [True, 0.0]

        # sKS-test for the empirical, mixed beta and special distribution
        else:
            max_val = ks_2samp(self.distr_val[event_index][var_index], self.event_type_detector.values[event_index][var_index][
                    -self.num_s_ks_values:])[0]

        if first_distr:
            if max_val > crit_valance:
                return [False, max_val]
            return [True, max_val]
        if max_val > crit_valance:
            return [False, 1.0]
        return [True, 0.0]

    def d_test(self, event_index, var_index):
        """Make a test if the new variables follow the discrete distribution and appends the result to the BT."""
        if self.used_multinomial_test == 'MT':
            # Count the appearance of the values
            values_app = [0] * len(self.var_type[event_index][var_index][1])
            for v in self.event_type_detector.values[event_index][var_index][-self.num_update:]:
                values_app[self.var_type[event_index][var_index][1].index(v)] += 1

            # probability of the values or the test sample
            prob_of_sample = self.bt_results[event_index][var_index][1].pmf(values_app)
            # Sum of the probabilities, which are smaller than the probability of the values
            smaller_prob_sum = 0
            if len(self.var_type[event_index][var_index][1]) <= 5:
                for i in range(self.num_update + 1):
                    if len(self.var_type[event_index][var_index][1]) == 2:
                        tmp_prob = self.bt_results[event_index][var_index][1].pmf([i, self.num_update - i])
                        if tmp_prob <= prob_of_sample:
                            smaller_prob_sum += tmp_prob
                    else:
                        for j in range(self.num_update - i + 1):
                            if len(self.var_type[event_index][var_index][1]) == 3:
                                tmp_prob = self.bt_results[event_index][var_index][1].pmf([i, j, self.num_update - (i + j)])
                                if tmp_prob <= prob_of_sample:
                                    smaller_prob_sum += tmp_prob
                            else:
                                for k in range(self.num_update - (i + j) + 1):
                                    if len(self.var_type[event_index][var_index][1]) == 4:
                                        tmp_prob = self.bt_results[event_index][var_index][1].pmf(
                                            [i, j, k, self.num_update - (i + j + k)])
                                        if tmp_prob <= prob_of_sample:
                                            smaller_prob_sum += tmp_prob
                                    else:
                                        for l in range(self.num_update - (i + j + k) + 1):
                                            tmp_prob = self.bt_results[event_index][var_index][1].pmf(
                                                [i, j, k, l, self.num_update - (i + j + k + l)])
                                            if tmp_prob <= prob_of_sample:
                                                smaller_prob_sum += tmp_prob

            # Make a multinomial test
            if smaller_prob_sum < self.d_alpha:
                self.bt_results[event_index][var_index][0] = self.bt_results[event_index][var_index][0][1:] + [0]
                return
            self.bt_results[event_index][var_index][0] = self.bt_results[event_index][var_index][0][1:] + [1]
            return

        if self.used_multinomial_test == 'Chi':
            # Count the appearance of the values
            values_app = [0] * len(self.var_type[event_index][var_index][1])
            for v in self.event_type_detector.values[event_index][var_index][-self.num_update:]:
                values_app[self.var_type[event_index][var_index][1].index(v)] += 1

            # Make a chisquare test
            if chisquare(values_app, f_exp=[i * self.num_update for i in self.var_type[event_index][var_index][2]])[1] < \
                    self.d_alpha:
                self.bt_results[event_index][var_index][0] = self.bt_results[event_index][var_index][0][1:] + [0]
                return
            self.bt_results[event_index][var_index][0] = self.bt_results[event_index][var_index][0][1:] + [1]
            return

        # Make an approximated multinomial test which consists of binomial tests
        if self.used_multinomial_test == 'Approx':
            # Count the appearance of the values
            values_app = [0] * len(self.var_type[event_index][var_index][1])
            for v in self.event_type_detector.values[event_index][var_index][-self.num_update:]:
                values_app[self.var_type[event_index][var_index][1].index(v)] += 1

            # Makes for each value a twosided BT. If one fails the d-test fails
            for i, value in enumerate(values_app):
                if value < self.bt_results[event_index][var_index][1][i] or value > self.bt_results[event_index][var_index][2][i]:
                    self.bt_results[event_index][var_index][0] = self.bt_results[event_index][var_index][0][1:] + [0]
                    return

            self.bt_results[event_index][var_index][0] = self.bt_results[event_index][var_index][0][1:] + [1]
            return

    def d_init_bt(self, event_index, var_index):
        """Initialize the BT for discrete variables."""
        if self.used_multinomial_test == 'MT':
            # Initialize the list for the results and the multinomialtest
            self.bt_results[event_index][var_index] = [
                [1] * self.num_d_bt, multinomial(self.num_update, self.var_type[event_index][var_index][2])]

        elif self.used_multinomial_test == 'Approx':
            # Generates a list of the lower bounds of the individual BTs of the single values
            lower_bound_list = self.num_update - self.bt_min_successes_multi_p(
                self.num_update, 1 - np.array(self.var_type[event_index][var_index][2]), self.d_alpha / 2,
                event_index, var_index)

            # Generates a list of the upper bounds of the individual BTs of the single values
            upper_bound_list = self.bt_min_successes_multi_p(
                self.num_update, self.var_type[event_index][var_index][2],  self.d_alpha / 2, event_index, var_index)

            # Initialize the list for the results
            self.bt_results[event_index][var_index] = [[1] * self.num_d_bt, lower_bound_list, upper_bound_list]

        else:
            # Initialize the list for the results
            self.bt_results[event_index][var_index] = [[1] * self.num_d_bt]

    def init_var_type_history_list(self, event_index):
        """Initialize the history of the variabletypes of the eventType."""
        if len(self.var_type_history_list) < event_index + 1 or self.var_type_history_list[event_index] == []:
            for i in range(event_index + 1 - len(self.var_type_history_list)):
                self.var_type_history_list.append([])

            # [others, static, [discrete, number of appended steps], asc, desc, unique, EV of continuous distributions]
            if not self.var_type_history_list[event_index]:
                self.var_type_history_list[event_index] = [[[], [], [[], []], [], [], [], [[], []]] for _ in range(len(
                    self.var_type[event_index]))]

            # Append the first entries to the historyList
            # Test only the variables with paths in the path_list
            if self.path_list is None:
                index_list = range(self.length[event_index])
            # Test all variables
            else:
                index_list = self.variable_path_num[event_index]

            for var_index in index_list:
                # This section updates the historyList of the variableTypes
                if self.var_type[event_index][var_index][0] in self.var_type_history_list_order:
                    # Index of the variableType in the list  # [others, static, [discrete, number of appended steps],
                    # asc, desc, unique, EV of continuous distributions]
                    type_index = self.var_type_history_list_order.index(self.var_type[event_index][var_index][0])
                else:
                    type_index = -1

                for tmp_type_index in range(len(self.var_type_history_list[event_index][var_index])):
                    if tmp_type_index == type_index:
                        if len(self.var_type_history_list[event_index][var_index][type_index]) >= 1 and isinstance(
                                self.var_type_history_list[event_index][var_index][type_index][0], list):
                            self.var_type_history_list[event_index][var_index][type_index][0].append(1)
                            for i in range(1, len(self.var_type_history_list[event_index][var_index][type_index])):
                                self.var_type_history_list[event_index][var_index][type_index][i].append(0)
                        else:
                            self.var_type_history_list[event_index][var_index][type_index].append(1)
                    else:
                        if len(self.var_type_history_list[event_index][var_index][tmp_type_index]) >= 1 and isinstance(
                                self.var_type_history_list[event_index][var_index][tmp_type_index][0], list):
                            for i in range(len(self.var_type_history_list[event_index][var_index][tmp_type_index])):
                                self.var_type_history_list[event_index][var_index][tmp_type_index][i].append(0)
                        else:
                            self.var_type_history_list[event_index][var_index][tmp_type_index].append(0)

                if type_index == -1:
                    # Continuously distributed variable type. Index 6 in the history list
                    if self.var_type[event_index][var_index][0] == 'uni':
                        self.var_type_history_list[event_index][var_index][6][0][-1] = (
                            self.var_type[event_index][var_index][1] + self.var_type[event_index][var_index][2]) / 2
                        self.var_type_history_list[event_index][var_index][6][1][-1] = (
                            self.var_type[event_index][var_index][2] - self.var_type[event_index][var_index][1]) / np.sqrt(12)
                    else:
                        self.var_type_history_list[event_index][var_index][6][0][-1] = self.var_type[event_index][var_index][1]
                        self.var_type_history_list[event_index][var_index][6][1][-1] = self.var_type[event_index][var_index][2]

    def get_indicator(self, event_index):
        """Calculate and returns a indicator for a change in the system behaviour based on the analysis of VarTypeD."""
        # List which stores the single indicators for the variables
        indicator_list = []

        for var_index in range(len(self.var_type_history_list[event_index])):
            if not self.event_type_detector.check_variables[event_index][var_index]:
                indicator_list.append(0)
                continue

            # List, which stores the differences of probabilities of the types, where the current history is higher than the reference.
            diff_list = []
            # Length of the reference
            len_ref = self.num_var_type_hist_ref
            # Length of the current historylist
            len_cur = self.num_var_type_considered_ind

            # Appends the positive differnces of the probabilities to diff_list
            for type_index in range(len(self.var_type_history_list[event_index][var_index])):
                if self.var_type_history_list_reference[event_index][var_index][1] == len_ref and sum(
                        self.var_type_history_list[event_index][var_index][1]) < len_cur:
                    diff_list.append(1)
                    break
                # Differentiation of the entries, which are lists (e.g. discrete, continuously distributed)
                if type_index in [2, 6]:
                    if type_index == 6:  # continuously distributed variable type
                        if self.var_type_history_list_reference[event_index][var_index][type_index] == 0:
                            diff_list.append(len([
                                    1 for x in self.var_type_history_list[event_index][var_index][type_index][1][
                                        -self.num_var_type_considered_ind:] if x != 0]) / len_cur)
                        else:
                            var_type_ev = sum(self.var_type_history_list[event_index][var_index][type_index][0][
                                    -self.num_var_type_considered_ind:]) / max(len([
                                        1 for x in self.var_type_history_list[event_index][var_index][type_index][0][
                                            -self.num_var_type_considered_ind:] if x != 0]), 1)
                            var_type_sd = sum(self.var_type_history_list[event_index][var_index][type_index][1][
                                    -self.num_var_type_considered_ind:]) / max(len([
                                        1 for x in self.var_type_history_list[event_index][var_index][type_index][1][
                                            -self.num_var_type_considered_ind:] if x != 0]), 1)

                            # Formula to include the impact of the mean, standard deviation and changes of the distribution
                            if max(self.var_type_history_list_reference[event_index][var_index][type_index][1], var_type_sd) > 0:
                                diff_list.append((min(1, abs((self.var_type_history_list_reference[event_index][var_index][
                                    type_index][0] - var_type_ev) / max(abs(self.var_type_history_list_reference[event_index][var_index][
                                        type_index][0]), abs(var_type_ev))) / 3 + abs((self.var_type_history_list_reference[event_index][
                                            var_index][type_index][1] - var_type_sd) / max(abs(self.var_type_history_list_reference[
                                                event_index][var_index][type_index][1]), abs(var_type_sd))) / 3 + 1 / 3) * len([
                                                    x for x in self.var_type_history_list[event_index][var_index][type_index][1][
                                                        -self.num_var_type_considered_ind:] if x != 0])) / len_cur)
                            else:
                                diff_list.append(0)
                    else:
                        tmp_max = 0
                        for j in range(len(self.var_type_history_list[event_index][var_index][type_index])):
                            if j == 0 and self.var_type_history_list_reference[event_index][var_index][type_index][j] == 0:
                                tmp_max = max(tmp_max, (sum(self.var_type_history_list[event_index][var_index][type_index][j][
                                        -self.num_var_type_considered_ind:]) /
                                        len_cur - self.var_type_history_list_reference[event_index][var_index][type_index][j] / len_ref))
                            else:
                                tmp_max = max(tmp_max, (sum(self.var_type_history_list[event_index][var_index][type_index][j][
                                        -self.num_var_type_considered_ind:]) /
                                        len_cur - self.var_type_history_list_reference[event_index][var_index][type_index][j] / len_ref) /
                                        2)
                        diff_list.append(tmp_max)

                else:
                    if self.var_type_history_list_reference[event_index][var_index][type_index] == 0:
                        diff_list.append(sum(self.var_type_history_list[event_index][var_index][type_index][
                            -self.num_var_type_considered_ind:]) / len_cur)
                    else:
                        diff_list.append(max(0, (sum(self.var_type_history_list[event_index][var_index][type_index][
                            -self.num_var_type_considered_ind:]) / len_cur - self.var_type_history_list_reference[event_index][var_index][
                                type_index] / len_ref)) / 2)

            if len(diff_list) == 0:
                indicator_list.append(0)
            else:
                indicator_list.append(sum(diff_list))
        return indicator_list

    def bt_min_successes(self, num_bt, p, alpha):  # skipcq: PYL-R0201
        """
        Calculate the minimal number of successes for the BT with significance alpha.
        p is the probability of success and num_bt is the number of observed tests.
        """
        tmp_sum = 0.0
        max_observations_factorial = np.math.factorial(num_bt)
        i_factorial = 1
        for i in range(num_bt + 1):
            i_factorial = i_factorial * max(i, 1)
            tmp_sum = tmp_sum + max_observations_factorial / (i_factorial * np.math.factorial(num_bt - i)) * ((1 - p) ** i) * (
                p ** (num_bt - i))
            if tmp_sum > alpha:
                return num_bt - i
        return 0

    def bt_min_successes_multi_p(self, num_bt, p_list, alpha, event_index, var_index):
        """
        Calculate the minimal number of successes for the BT with significance alpha.
        p_list is a list of probabilities of successes and num_bt is the number of observed tests.
        """
        if 'num_bt = %s, alpha = %s' % (num_bt, alpha) in self.bt_min_succ_data:
            # Here the min_successes are not being generated, but instead the right Indices are searched for in the bt_min_succ_data-list
            return np.searchsorted(self.bt_min_succ_data['num_bt = %s, alpha = %s' % (num_bt, alpha)], p_list, side='left', sorter=None)

        # Calculate the min_successes normally for each value one by one
        tmp_list = []
        for i in range(len(self.var_type[event_index][var_index][1])):
            tmp_list.append(self.bt_min_successes(num_bt, p_list[i], alpha))
        tmp_list = np.array(tmp_list)
        return tmp_list

    def print_initial_var_type(self, event_index, log_atom):
        """Print the initial variable types."""
        if self.silence_output_without_confidence or self.silence_output_except_indicator:
            return
        message = 'Initial detection of varTypes in lines like %s:' % repr(log_atom.raw_data)
        tmp_string = ''
        type_info = {}

        for var_index in range(self.length[event_index]):
            if self.var_type[event_index][var_index]:
                if self.var_type[event_index][var_index][0] == 'unq':
                    tmp_string += "  Path '%s': ['unq']\n" % (self.event_type_detector.variable_key_list[event_index][var_index])
                    type_info[self.event_type_detector.variable_key_list[event_index][var_index]] = ['unq']
                elif self.var_type[event_index][var_index][0] == 'd':
                    tmp_string += "  Path '%s': ['d']\n" % (self.event_type_detector.variable_key_list[event_index][var_index])
                    type_info[self.event_type_detector.variable_key_list[event_index][var_index]] = ['d']
                else:
                    tmp_string += "  Path '%s': %s\n" % (
                        self.event_type_detector.variable_key_list[event_index][var_index], self.var_type[event_index][var_index])
                    type_info[self.event_type_detector.variable_key_list[event_index][var_index]] = self.var_type[event_index][var_index]
        tmp_string = tmp_string.lstrip('  ')

        original_log_line_prefix = self.aminer_config.config_properties.get(CONFIG_KEY_LOG_LINE_PREFIX)
        if original_log_line_prefix is None:
            original_log_line_prefix = ''
        if self.output_log_line:
            sorted_log_lines = [tmp_string + original_log_line_prefix + log_atom.raw_data.decode()]
            analysis_component = {'AffectedLogAtomPaths': list(log_atom.parser_match.get_match_dictionary().keys())}
        else:
            sorted_log_lines = [tmp_string + log_atom.raw_data.decode()]
            analysis_component = {'AffectedLogAtomPaths': [self.event_type_detector.variable_key_list[event_index][var_index]]}

        event_data = {'AnalysisComponent': analysis_component, 'TotalRecords': self.event_type_detector.total_records,
                      'TypeInfo': type_info}
        for listener in self.anomaly_event_handlers:
            listener.receive_event(
                'Analysis.%s' % self.__class__.__name__, message, sorted_log_lines, event_data, log_atom, self)

    def print_changed_var_type(self, event_index, vt_old, vt_new, var_index, log_atom, confidence=None):
        """Print the changed variable types."""
        if self.save_statistics and ((not (isinstance(self.num_updates_until_var_reduction, bool)) and (
                self.event_type_detector.num_eventlines[event_index] - self.num_init) / self.num_update >=
                self.num_updates_until_var_reduction - 1)):
            self.changed_var_types.append(self.event_type_detector.num_eventlines[event_index])

        if (self.silence_output_without_confidence and confidence is None) or self.silence_output_except_indicator:
            return

        original_log_line_prefix = self.aminer_config.config_properties.get(CONFIG_KEY_LOG_LINE_PREFIX)
        if original_log_line_prefix is None:
            original_log_line_prefix = ''
        if self.output_log_line:
            tmp_str = ''
            for x in list(log_atom.parser_match.get_match_dictionary().keys()):
                tmp_str += '  ' + x + os.linesep
            tmp_str = tmp_str.lstrip('  ')
            sorted_log_lines = [tmp_str + original_log_line_prefix + log_atom.raw_data.decode()]
            analysis_component = {'AffectedLogAtomPaths': list(log_atom.parser_match.get_match_dictionary().keys())}
        else:
            sorted_log_lines = [
                '  ' + self.event_type_detector.variable_key_list[event_index][var_index] + os.linesep + log_atom.raw_data.decode()]
            analysis_component = {'AffectedLogAtomPaths': [self.event_type_detector.variable_key_list[event_index][var_index]]}

        event_data = {'AnalysisComponent': analysis_component, 'TotalRecords': self.event_type_detector.total_records,
                      'TypeInfo': {'from': vt_old[0], 'to': vt_new[0], 'lines': self.event_type_detector.num_eventlines[event_index]}}
        for listener in self.anomaly_event_handlers:
            listener.receive_event(
                'Analysis.%s' % self.__class__.__name__,
                "VariableType of path '%s' changed from '%s' to '%s' after the %s-th analysed line" % (
                    self.event_type_detector.variable_key_list[event_index][var_index], vt_old[0], vt_new[0],
                    self.event_type_detector.num_eventlines[event_index]), sorted_log_lines, event_data, log_atom, self)

    def print_reject_var_type(self, event_index, vt, var_index, log_atom):
        """Print the changed variable types."""
        if self.silence_output_without_confidence or self.silence_output_except_indicator:
            return

        original_log_line_prefix = self.aminer_config.config_properties.get(CONFIG_KEY_LOG_LINE_PREFIX)
        if original_log_line_prefix is None:
            original_log_line_prefix = ''
        if self.output_log_line:
            tmp_str = ''
            for x in list(log_atom.parser_match.get_match_dictionary().keys()):
                tmp_str += '  ' + x + os.linesep
            tmp_str = tmp_str.lstrip('  ')
            sorted_log_lines = [tmp_str + original_log_line_prefix + log_atom.raw_data.decode()]
            analysis_component = {'AffectedLogAtomPaths': list(log_atom.parser_match.get_match_dictionary().keys())}
        else:
            sorted_log_lines = [
                '  ' + self.event_type_detector.variable_key_list[event_index][var_index] + os.linesep + log_atom.raw_data.decode()]
            analysis_component = {'AffectedLogAtomPaths': [self.event_type_detector.variable_key_list[event_index][var_index]]}

        event_data = {'AnalysisComponent': analysis_component, 'TotalRecords': self.event_type_detector.total_records,
                      'TypeInfo': {'reject': vt[0], 'lines': self.event_type_detector.num_eventlines[event_index]}}
        for listener in self.anomaly_event_handlers:
            listener.receive_event(
                'Analysis.%s' % self.__class__.__name__,
                "VariableType of path '%s' would reject the type '%s' after the %s-th analysed line" % (
                    self.event_type_detector.variable_key_list[event_index][var_index], vt[0], self.event_type_detector.num_eventlines[
                        event_index]), sorted_log_lines, event_data, log_atom, self)

    def print(self, message, log_atom, affected_path, confidence=None, indicator=None):
        """Print the message."""
        if isinstance(affected_path, str):
            affected_path = [affected_path]
        if (self.silence_output_without_confidence and confidence is None) or (
                self.silence_output_except_indicator and indicator is None):
            return

        original_log_line_prefix = self.aminer_config.config_properties.get(CONFIG_KEY_LOG_LINE_PREFIX)
        if original_log_line_prefix is None:
            original_log_line_prefix = ''
        if self.output_log_line:
            tmp_str = ''
            for x in list(log_atom.parser_match.get_match_dictionary().keys()):
                tmp_str += '  ' + x + os.linesep
            tmp_str = tmp_str.lstrip('  ')
            sorted_log_lines = [tmp_str + original_log_line_prefix + log_atom.raw_data.decode()]
            analysis_component = {'AffectedLogAtomPaths': list(log_atom.parser_match.get_match_dictionary().keys())}
        else:
            tmp_str = ''
            for x in affected_path:
                tmp_str += '  ' + x + os.linesep
            tmp_str = tmp_str.lstrip('  ')
            sorted_log_lines = [tmp_str + log_atom.raw_data.decode()]
            analysis_component = {'AffectedLogAtomPaths': affected_path}

        event_data = {'AnalysisComponent': analysis_component, 'TotalRecords': self.event_type_detector.total_records,
                      'TypeInfo': {'Confidence': confidence, 'Indicator': indicator}}
        for listener in self.anomaly_event_handlers:
            listener.receive_event('Analysis.%s' % self.__class__.__name__, message, sorted_log_lines, event_data, log_atom, self)

    def log_statistics(self, component_name):
        """
        Log statistics of an AtomHandler. Override this method for more sophisticated statistics output of the AtomHandler.
        @param component_name the name of the component which is printed in the log line.
        """
        if STAT_LEVEL == 1:
            logging.getLogger(STAT_LOG_NAME).info(
                "'%s' processed %d out of %d log atoms successfully and learned %d new variable types and updated %d variable types "
                "in the last 60 minutes.", component_name, self.log_success, self.log_total, self.log_new_learned, self.log_updated)
        elif STAT_LEVEL == 2:
            logging.getLogger(STAT_LOG_NAME).info(
                "'%s' processed %d out of %d log atoms successfully and learned %d new variable types and updated %d variable types "
                "in the last 60 minutes. Following new variable types were learned: %s", component_name, self.log_success, self.log_total,
                self.log_new_learned, self.log_updated, self.log_new_learned_values)
        self.log_success = 0
        self.log_total = 0
        self.log_new_learned = 0
        self.log_new_learned_values = []
        self.log_updated = 0


def convert_to_floats(list_in):
    """Give back false if one entry of the list is no float and returns the list of floats otherwise."""
    num_list = []
    for item in list_in:
        try:
            num_list.append(float(item))
        except:  # skipcq: FLK-E722
            return []
    return num_list


def consists_of_floats(list_in):
    """Give back false if one entry of the list is no float or integer. True otherwise."""
    return all(isinstance(x, (float, int)) for x in list_in)


def consists_of_ints(list_in):
    """Give back True if all entries are integers an False otherwise."""
    for item in list_in:
        if item != int(item):
            return False
    return True
