/*******************************************************************
 * This file is part of the Emulex Linux Device Driver for         *
 * Enterprise Fibre Channel Host Bus Adapters.                     *
 * Refer to the README file included with this package for         *
 * driver version and adapter support.                             *
 * Copyright (C) 2003 Emulex Corporation.                          *
 * www.emulex.com                                                  *
 *                                                                 *
 * This program is free software; you can redistribute it and/or   *
 * modify it under the terms of the GNU General Public License     *
 * as published by the Free Software Foundation; either version 2  *
 * of the License, or (at your option) any later version.          *
 *                                                                 *
 * This program is distributed in the hope that it will be useful, *
 * but WITHOUT ANY WARRANTY; without even the implied warranty of  *
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the   *
 * GNU General Public License for more details, a copy of which    *
 * can be found in the file COPYING included with this package.    *
 *******************************************************************/

#include "fc_os.h"

#include "fc_hw.h"
#include "fc.h"

#include "fcdiag.h"
#include "hbaapi.h"
#include "fcfgparm.h"
#include "fcmsg.h"
#include "fc_crtn.h"
#include "fc_ertn.h"   /* Environment - external routine definitions */

extern fc_dd_ctl_t      DD_CTL;
extern iCfgParam        icfgparam[];
extern char             *lpfc_release_version;
extern int              fc_max_els_sent;

/* Routine Declaration - Local */
_local_ int   fc_chksparm(FC_BRD_INFO *binfo,volatile SERV_PARM *sp, uint32 cls);
_local_ int   fc_els_retry(FC_BRD_INFO *binfo, RING *rp, IOCBQ *iocb, uint32 cmd,
              NODELIST *nlp);
_local_ int   fc_status_action(FC_BRD_INFO *binfo, IOCBQ *iocb, uint32 cmd,
              NODELIST *nlp);
/* End Routine Declaration - Local */

/******************************************************/
/**  handle_els_event                                **/
/**                                                  **/
/**  Description: Process an ELS Response Ring cmpl. **/
/**                                                  **/
/******************************************************/
_static_ int
handle_els_event(
fc_dev_ctl_t *p_dev_ctl,
RING         *rp,
IOCBQ        *temp)
{
   IOCB          * cmd;
   FC_BRD_INFO   * binfo;
   IOCBQ         * xmitiq;
   volatile SERV_PARM   * sp;
   uint32        * lp0, * lp1;
   MATCHMAP      * mp,  * rmp;
   DMATCHMAP     * drmp;
   NODELIST      * ndlp;
   MAILBOXQ      * mb;
   ELS_PKT       * ep;
   iCfgParam     * clp;
   ADISC         * ap;
   void          * ioa;
   int           rc;
   uint32        command;
   uint32        did, bumpcnt;
   volatile uint32 ha_copy;

   /* Called from host_interrupt() to process ELS R0ATT */
   rc = 0;
   ndlp = 0;
   binfo = &BINFO;
   cmd = &temp->iocb;

   /* look up xmit complete by IoTag */
   if ((xmitiq = fc_ringtxp_get(rp, cmd->ulpIoTag)) == 0) {
      /* completion with missing xmit command */
      FCSTATCTR.elsStrayXmitCmpl++;

      /* Stray ELS completion */
      fc_log_printf_msg_vargs( binfo->fc_brd_no,
             &fc_msgBlk0102,                   /* ptr to msg structure */
              fc_mes0102,                      /* ptr to msg */
               fc_msgBlk0102.msgPreambleStr,   /* begin varargs */
                cmd->ulpCommand,
                 cmd->ulpIoTag);               /* end varargs */

      return(EIO);
   }
   temp->retry = xmitiq->retry;

   if(binfo->fc_ffstate < FC_READY) {
      /* If we are in discovery, and a Link Event is pending, abandon 
       * discovery, clean up pending actions, and take the Link Event.
       */
     ioa = FC_MAP_IO(&binfo->fc_iomap_io);  /* map in io registers */
     /* Read host attention register to determine interrupt source */
     ha_copy = READ_CSR_REG(binfo, FC_HA_REG(binfo, ioa));
     FC_UNMAP_MEMIO(ioa);

     if(ha_copy & HA_LATT) {
        /* Pending Link Event during Discovery */
        fc_log_printf_msg_vargs( binfo->fc_brd_no,
               &fc_msgBlk0250,                   /* ptr to msg structure */
                fc_mes0250,                      /* ptr to msg */
                 fc_msgBlk0250.msgPreambleStr,   /* begin varargs */
                  (uint32)cmd->ulpCommand,
                   (uint32)cmd->ulpIoTag,
                    (uint32)cmd->ulpStatus,
                     cmd->un.ulpWord[4]);        /* end varargs */
        fc_abort_discovery(p_dev_ctl);
        temp->retry = 0xff;
     }
  }

   /* Check for aborted ELS command */
   if(temp->retry == 0xff) {

      /* Aborted ELS IOCB */
      fc_log_printf_msg_vargs( binfo->fc_brd_no,
             &fc_msgBlk0104,                   /* ptr to msg structure */
              fc_mes0104,                      /* ptr to msg */
               fc_msgBlk0104.msgPreambleStr,   /* begin varargs */
                cmd->ulpCommand,
                 cmd->ulpIoTag);               /* end varargs */
      switch (cmd->ulpCommand) {
      case CMD_ELS_REQUEST_CR:
      case CMD_ELS_REQUEST64_CR:
      case CMD_ELS_REQUEST_CX:
      case CMD_ELS_REQUEST64_CX:
         mp = (MATCHMAP * )xmitiq->bp;
         rmp = (MATCHMAP * )xmitiq->info;
         lp0 = (uint32 * )mp->virt;
         ndlp = 0;
         command = *lp0;
         switch (command) {
         case ELS_CMD_PLOGI:
         case ELS_CMD_LOGO:
         case ELS_CMD_PRLI:
         case ELS_CMD_PDISC:
         case ELS_CMD_ADISC:
            rmp->fc_mptr = (uchar *)0;
            break;
         }
         break;
      case CMD_XMIT_ELS_RSP_CX: /* Normal ELS response completion */
      case CMD_XMIT_ELS_RSP64_CX: /* Normal ELS response completion */
         mp = (MATCHMAP * )xmitiq->bp;
         ndlp = (NODELIST * )xmitiq->ndlp;
         break;
      case CMD_GEN_REQUEST64_CX:
      case CMD_GEN_REQUEST64_CR:
         if(xmitiq->bpl == 0) {
            /* User initiated request */
            drmp = (DMATCHMAP * )xmitiq->info;
            drmp->dfc_flag = -1;
            fc_mem_put(binfo, MEM_IOCB, (uchar * )xmitiq);
         }
         else {
            /* Driver initiated request */
            /* Just free resources and let timer timeout */
            fc_mem_put(binfo, MEM_BPL, (uchar * )xmitiq->bpl);
            if(xmitiq->bp) {
               fc_mem_put(binfo, MEM_BUF, (uchar * )xmitiq->bp);
            }
            if(xmitiq->info)
               fc_free_ct_rsp(p_dev_ctl, (MATCHMAP *)xmitiq->info);
         fc_mem_put(binfo, MEM_IOCB, (uchar * )xmitiq);
         }
         return(0);
      }
      goto out2;
   }

   /* ELS completion */
   fc_log_printf_msg_vargs( binfo->fc_brd_no,
          &fc_msgBlk0105,                   /* ptr to msg structure */
           fc_mes0105,                      /* ptr to msg */
            fc_msgBlk0105.msgPreambleStr,   /* begin varargs */
             (uint32)cmd->ulpCommand,
              (uint32)cmd->ulpIoTag,
               (uint32)cmd->ulpStatus,
                cmd->un.ulpWord[4]);        /* end varargs */

   switch (cmd->ulpCommand) {

   case CMD_GEN_REQUEST64_CR:
      if(xmitiq->bpl == 0) {
         /* User initiated request */
         drmp = (DMATCHMAP * )xmitiq->info;
         drmp->dfc_flag = -2;
      }
      else {
         /* Driver initiated request */
         /* Just free resources and let timer timeout */
         fc_mem_put(binfo, MEM_BPL, (uchar * )xmitiq->bpl);
         if(xmitiq->bp) {
            fc_mem_put(binfo, MEM_BUF, (uchar * )xmitiq->bp);
         }
         if(xmitiq->info)
            fc_free_ct_rsp(p_dev_ctl, (MATCHMAP *)xmitiq->info);
      }
      
      break;

   case CMD_ELS_REQUEST_CR: /* Local error in ELS command */
   case CMD_ELS_REQUEST64_CR: /* Local error in ELS command */

      FCSTATCTR.elsXmitErr++;

      /* Find out which command failed */
      mp = (MATCHMAP * )xmitiq->bp;
      rmp = (MATCHMAP * )xmitiq->info;
      ndlp = (NODELIST *)rmp->fc_mptr;
      rmp->fc_mptr = (uchar *)0;

      lp0 = (uint32 * )mp->virt;
      command = *lp0;

      if (fc_els_retry(binfo, rp, temp, command, ndlp) == 0) {
         /* retry of ELS command failed */
         switch (command) {
         case ELS_CMD_FLOGI:       /* Fabric login */
            if (ndlp)
               ndlp->nlp_flag &= ~NLP_REQ_SND;
            fc_freenode_did(binfo, Fabric_DID, 1);
            if (binfo->fc_ffstate == FC_FLOGI) {
               binfo->fc_flag &= ~(FC_FABRIC | FC_PUBLIC_LOOP);
               if (binfo->fc_topology == TOPOLOGY_LOOP) {
                  binfo->fc_edtov = FF_DEF_EDTOV;
                  binfo->fc_ratov = FF_DEF_RATOV;
                  if ((mb=(MAILBOXQ * )fc_mem_get(binfo, MEM_MBOX | MEM_PRI))) {
                     fc_config_link(p_dev_ctl, (MAILBOX * )mb);
                     if (issue_mb_cmd(binfo, (MAILBOX * )mb, MBX_NOWAIT) != MBX_BUSY) {
                        fc_mem_put(binfo, MEM_MBOX, (uchar * )mb);
                     }
                  }
                  binfo->fc_flag |= FC_DELAY_DISC;
               } else {
                  /* Device Discovery completion error */
                  fc_log_printf_msg_vargs( binfo->fc_brd_no,
                         &fc_msgBlk0206,                   /* ptr to msg structure */
                          fc_mes0206,                      /* ptr to msg */
                           fc_msgBlk0206.msgPreambleStr,   /* begin varargs */
                             cmd->ulpStatus,
                              cmd->un.ulpWord[4],
                               cmd->un.ulpWord[5]);        /* end varargs */
                  binfo->fc_ffstate = FC_ERROR;
                  if ((mb=(MAILBOXQ * )fc_mem_get(binfo, MEM_MBOX | MEM_PRI))) {
                     fc_clear_la(binfo, (MAILBOX * )mb);
                     if (issue_mb_cmd(binfo, (MAILBOX * )mb, MBX_NOWAIT) != MBX_BUSY) {
                        fc_mem_put(binfo, MEM_MBOX, (uchar * )mb);
                     }
                  }
               }
            }
            break;
   
         case ELS_CMD_PLOGI:       /* NPort login */
            if ((ndlp == 0) && ((ndlp = fc_findnode_odid(binfo, NLP_SEARCH_ALL, (uint32)cmd->un.elsreq.remoteID)) == 0)) {
               break;
            }

            /* If we are in the middle of Discovery */
            if (ndlp->nlp_action & (NLP_DO_ADDR_AUTH | NLP_DO_DISC_START | NLP_DO_RSCN)) {
               /* Goto next entry */
               fc_nextnode(p_dev_ctl, ndlp);
            }

            ndlp->nlp_action &= ~NLP_DO_RNID;
            ndlp->nlp_flag &= ~NLP_REQ_SND;

            if((ndlp->nlp_type & (NLP_AUTOMAP | NLP_SEED_MASK)) == 0) {
               fc_freenode(binfo, ndlp, 1);
            }
            else {
               fc_freenode(binfo, ndlp, 0);
               ndlp->nlp_state = NLP_LIMBO;
               fc_nlp_bind(binfo, ndlp);
            }
            break;

         case ELS_CMD_PRLI:        /* Process Log In */
            if ((ndlp == 0) && ((ndlp = fc_findnode_odid(binfo, NLP_SEARCH_ALL, (uint32)cmd->un.elsreq.remoteID)) == 0)) {
               break;
            }
   
            /* If we are in the middle of Discovery */
            if (ndlp->nlp_action & (NLP_DO_DISC_START | NLP_DO_RSCN)) {
               /* Goto next entry */
               fc_nextnode(p_dev_ctl, ndlp);
            }
            ndlp->nlp_flag &= ~NLP_REQ_SND;
            ndlp->nlp_state = NLP_LOGIN;
            fc_nlp_unmap(binfo, ndlp);
            break;

         case ELS_CMD_PDISC:       /* Pdisc */
         case ELS_CMD_ADISC:       /* Adisc */
            if ((ndlp == 0) && ((ndlp = fc_findnode_odid(binfo, NLP_SEARCH_ALL,
               (uint32)cmd->un.elsreq.remoteID)) == 0)) {
               break;
            }

            fc_freenode(binfo, ndlp, 0);
            ndlp->nlp_state = NLP_LIMBO;
            fc_nlp_bind(binfo, ndlp);

            ndlp->nlp_action |= NLP_DO_DISC_START;
            binfo->fc_nlp_cnt++;
            fc_els_cmd(binfo, ELS_CMD_PLOGI, (void *)((ulong)cmd->un.elsreq.remoteID), (uint32)0, (ushort)0, ndlp);
            break;
   
         case ELS_CMD_LOGO:        /* Logout */
            if ((ndlp = fc_findnode_odid(binfo, NLP_SEARCH_ALL, (uint32)cmd->un.elsreq.remoteID)) == 0) {
               break;
            }
   
            /* If we are in the middle of Discovery */
            if (ndlp->nlp_action & (NLP_DO_ADDR_AUTH | NLP_DO_DISC_START | NLP_DO_RSCN)) {
               /* Goto next entry */
               fc_nextnode(p_dev_ctl, ndlp);
            }
   
            fc_freenode(binfo, ndlp, 0);
            ndlp->nlp_state = NLP_LIMBO;
            fc_nlp_bind(binfo, ndlp);
            break;
   
         case ELS_CMD_FARPR:       /* Farp-res */
            ep = (ELS_PKT * )lp0;
            if((ndlp = fc_findnode_wwpn(binfo, NLP_SEARCH_ALL,
               &ep->un.farp.RportName)) == 0)
               break;
   
            /* If we are in the middle of Discovery */
            if (ndlp->nlp_action & NLP_DO_DISC_START) {
               /* Goto next entry */
               fc_nextnode(p_dev_ctl, ndlp);
            }
   
            fc_freenode(binfo, ndlp, 0);
            ndlp->nlp_state = NLP_LIMBO;
            fc_nlp_bind(binfo, ndlp);
            break;

         case ELS_CMD_FARP:        /* Farp-req */
            ep = (ELS_PKT * )lp0;
   
            if((ndlp = fc_findnode_wwpn(binfo, NLP_SEARCH_ALL,
               &ep->un.farp.RportName)) == 0)
               break;
   
            ndlp->nlp_flag &= ~NLP_FARP_SND;
            ndlp->nlp_action &= ~NLP_DO_ADDR_AUTH;
            break;
   
         case ELS_CMD_SCR: /* State Change Registration */
            break;
   
         case ELS_CMD_RNID: /* Receive Node Identification */
            break;
   
         default:
            /* Unknown ELS command <elsCmd> */
            fc_log_printf_msg_vargs( binfo->fc_brd_no,
                   &fc_msgBlk0106,                   /* ptr to msg structure */
                    fc_mes0106,                      /* ptr to msg */
                     fc_msgBlk0106.msgPreambleStr,   /* begin varargs */
                      command);                      /* end varargs */
            FCSTATCTR.elsCmdPktInval++;
            break;
         }
         /* ELS command completion error */
         fc_log_printf_msg_vargs( binfo->fc_brd_no,
                &fc_msgBlk0107,                   /* ptr to msg structure */
                 fc_mes0107,                      /* ptr to msg */
                  fc_msgBlk0107.msgPreambleStr,   /* begin varargs */
                   cmd->ulpCommand,
                    cmd->ulpStatus,
                     cmd->un.ulpWord[4],
                      cmd->un.ulpWord[5]);        /* end varargs */
      }
      else {
         /* Retry in progress */
         if ((command == ELS_CMD_PLOGI) &&
            ((cmd->un.ulpWord[4] & 0xff) == IOERR_LOOP_OPEN_FAILURE)) {
            if (ndlp->nlp_action & (NLP_DO_ADDR_AUTH | NLP_DO_DISC_START | NLP_DO_RSCN)) {
               /* Goto next entry */
               fc_nextnode(p_dev_ctl, ndlp);
            }
         }
      }

      if (xmitiq->bpl) {
         fc_mem_put(binfo, MEM_BPL, (uchar * )xmitiq->bpl);
      }
      fc_mem_put(binfo, MEM_BUF, (uchar * )mp);
      fc_mem_put(binfo, MEM_BUF, (uchar * )xmitiq->info);
      break;

   case CMD_XMIT_ELS_RSP_CX: /* Normal ELS response completion */
   case CMD_XMIT_ELS_RSP64_CX: /* Normal ELS response completion */

      ndlp = (NODELIST * )xmitiq->ndlp;
      did = 0;
      bumpcnt = 0;
      if ((ndlp) && (ndlp->nlp_flag & NLP_SND_PLOGI)) {
         ndlp->nlp_flag &= ~NLP_SND_PLOGI;
         did = ndlp->nlp_DID;
         if((binfo->fc_flag & FC_RSCN_MODE) ||
            (binfo->fc_ffstate < FC_READY)) {
            binfo->fc_nlp_cnt++;
            bumpcnt = 1;
         }
      }
      mp = (MATCHMAP * )xmitiq->bp;
      lp0 = (uint32 * )mp->virt;
      /* get command that errored */
      command = *lp0++;
      sp = (volatile SERV_PARM * )lp0;
      if (cmd->ulpStatus) {
         /* Error occurred sending ELS response */
         /* check to see if we should retry */
         /* ELS response completion error */
         fc_log_printf_msg_vargs( binfo->fc_brd_no,
                &fc_msgBlk0108,                   /* ptr to msg structure */
                 fc_mes0108,                      /* ptr to msg */
                  fc_msgBlk0108.msgPreambleStr,   /* begin varargs */
                   cmd->ulpCommand,
                    cmd->ulpStatus,
                     cmd->un.ulpWord[4],
                      cmd->un.ulpWord[5]);        /* end varargs */
         FCSTATCTR.elsXmitErr++;

         if (fc_els_retry(binfo, rp, temp, command, ndlp) == 0) {
            /* No retries */
            if ((ndlp) && (ndlp->nlp_flag & NLP_RM_ENTRY) && 
                !(ndlp->nlp_flag & NLP_REQ_SND)) {
               if (ndlp->nlp_type & NLP_FCP_TARGET) {
                  ndlp->nlp_flag &= ~NLP_RM_ENTRY;
                  ndlp->nlp_action &= ~NLP_DO_ADDR_AUTH;
                  if(binfo->fc_ffstate == FC_READY) {
                     if(!(binfo->fc_flag & FC_RSCN_MODE)) {
                        binfo->fc_flag |= FC_RSCN_MODE;
                        ndlp->nlp_action |= NLP_DO_RSCN;
                        ndlp->nlp_flag &= ~NLP_NODEV_TMO;
                        fc_nextrscn(p_dev_ctl, 1);
                     }
                  }
                  else {
                     ndlp->nlp_action |= NLP_DO_DISC_START;
                     fc_nextdisc(p_dev_ctl, 1);
                  }
               } else {
                  fc_freenode(binfo, ndlp, 0);
                  ndlp->nlp_state = NLP_LIMBO;
                  fc_nlp_bind(binfo, ndlp);
               }
            }
            else {
               if (ndlp) {
                  if(!(ndlp->nlp_flag & NLP_REQ_SND)) {
                     fc_freenode(binfo, ndlp, 0);
                     ndlp->nlp_state = NLP_LIMBO;
                     fc_nlp_bind(binfo, ndlp);
                  }
               }
            }
         }
      } else {
         FCSTATCTR.elsXmitCmpl++;
         if(ndlp) {
            /* ELS response completion */
            fc_log_printf_msg_vargs( binfo->fc_brd_no,
                   &fc_msgBlk0109,                   /* ptr to msg structure */
                    fc_mes0109,                      /* ptr to msg */
                     fc_msgBlk0109.msgPreambleStr,   /* begin varargs */
                      ndlp->nlp_DID,
                       ndlp->nlp_type,
                        ndlp->nlp_flag,
                         ndlp->nlp_state);           /* end varargs */
            if ((ndlp->nlp_flag & NLP_REG_INP)) {
               uint32          size;
               MATCHMAP      * bmp;
               ULP_BDE64     * bpl;

               bmp = (MATCHMAP *)(xmitiq->bpl);
               bpl = (ULP_BDE64 * )bmp->virt;
               bpl->tus.w = PCIMEM_LONG(bpl->tus.w);
               size = (uint32)bpl->tus.f.bdeSize;
               if(size == (sizeof(SERV_PARM) + sizeof(uint32))) {
                  fc_process_reglogin(p_dev_ctl, ndlp);
               }
            }

            if ((ndlp->nlp_flag & NLP_RM_ENTRY) && 
                !(ndlp->nlp_flag & NLP_REQ_SND)) {
               if (ndlp->nlp_type & NLP_FCP_TARGET) {
                  ndlp->nlp_flag &= ~NLP_RM_ENTRY;
                  ndlp->nlp_action &= ~NLP_DO_ADDR_AUTH;
                  if(binfo->fc_ffstate == FC_READY) {
                     fc_freenode(binfo, ndlp, 0);
                     ndlp->nlp_state = NLP_LIMBO;
                     fc_nlp_bind(binfo, ndlp);
                     if(!(binfo->fc_flag & FC_RSCN_MODE)) {
                        did = 0;
                        if(bumpcnt)
                           binfo->fc_nlp_cnt--;

                        binfo->fc_flag |= FC_RSCN_MODE;
                        ndlp->nlp_action |= NLP_DO_RSCN;
                        ndlp->nlp_flag &= ~NLP_NODEV_TMO;
                        fc_nextrscn(p_dev_ctl, 1);
                     }
                  }
                  else {
                     did = 0;
                     if(bumpcnt)
                        binfo->fc_nlp_cnt--;

                     ndlp->nlp_action |= NLP_DO_DISC_START;
                     fc_nextdisc(p_dev_ctl, 1);
                  }
               } else {
                  if (ndlp->nlp_type & NLP_IP_NODE) {
                     fc_freenode(binfo, ndlp, 0);
                     ndlp->nlp_state = NLP_LIMBO;
                     fc_nlp_bind(binfo, ndlp);
                  }
                  else {
                     fc_freenode(binfo, ndlp, 1);
                  }
               }
            }
         }
      }

      if (xmitiq->bpl) {
         fc_mem_put(binfo, MEM_BPL, (uchar * )xmitiq->bpl);
      }
      fc_mem_put(binfo, MEM_BUF, (uchar * )mp);
      if(did && (!(ndlp->nlp_flag & NLP_NS_REMOVED))) {
         fc_els_cmd(binfo, ELS_CMD_PLOGI, (void *)((ulong)did),
          (uint32)0, (ushort)0, ndlp);
      }
      break;

   case CMD_GEN_REQUEST64_CX:
      if(xmitiq->bpl == 0) {
         /* User initiated request */
         drmp = (DMATCHMAP * )xmitiq->info;
         fc_mpdata_sync(drmp->dfc.dma_handle, 0, 0, DDI_DMA_SYNC_FORKERNEL);

         if (cmd->ulpStatus) {
            /* Error occurred sending ELS command */
            if ((cmd->un.ulpWord[4] & 0xff) == IOERR_SEQUENCE_TIMEOUT)
               drmp->dfc_flag = -1;
            else
               drmp->dfc_flag = -2;
         }
         else {
            drmp->dfc_flag = (int)(cmd->un.genreq64.bdl.bdeSize);
         }
      }
      else {
         /* Driver initiated request */
         if (cmd->ulpStatus == 0) {
            mp = (MATCHMAP * )xmitiq->bp;
            ndlp = (NODELIST *)mp->fc_mptr;
            if(ndlp && (ndlp->nlp_DID == NameServer_DID)) {
               fc_ns_rsp(p_dev_ctl, (NODELIST *)mp->fc_mptr,
                  (MATCHMAP *)xmitiq->info,
                  (uint32)(cmd->un.genreq64.bdl.bdeSize));
            }
            /* FDMI */
            if(ndlp && (ndlp->nlp_DID == FDMI_DID)) {
              fc_fdmi_rsp(p_dev_ctl, (MATCHMAP *)mp, (MATCHMAP *)xmitiq->info);
            }
         }
         if(xmitiq->info)
            fc_free_ct_rsp(p_dev_ctl, (MATCHMAP *)xmitiq->info);
         fc_mem_put(binfo, MEM_BPL, (uchar * )xmitiq->bpl);
         if(xmitiq->bp) {
            fc_mem_put(binfo, MEM_BUF, (uchar * )xmitiq->bp);
         }
      }
      break;


   case CMD_ELS_REQUEST_CX: /* Normal ELS command completion */
   case CMD_ELS_REQUEST64_CX: /* Normal ELS command completion */

      /* get command that was accepted */
      mp = (MATCHMAP * )xmitiq->bp;
      lp0 = (uint32 * )mp->virt;
      command = *lp0;

      /* ELS command successful, get ptr to service params */
      rmp = (MATCHMAP * )xmitiq->info;
      ndlp = (NODELIST *)rmp->fc_mptr;
      rmp->fc_mptr = (uchar *)0;

      lp1 = (uint32 * )rmp->virt;
      fc_mpdata_sync(rmp->dma_handle, 0, 0, DDI_DMA_SYNC_FORKERNEL);

      sp = (volatile SERV_PARM * )((char *)lp1 + sizeof(uint32));

      if (cmd->ulpStatus) {
         /* Error occurred sending ELS command */
         FCSTATCTR.elsXmitErr++;
         /* ELS command completion error */
         fc_log_printf_msg_vargs( binfo->fc_brd_no,
                &fc_msgBlk0110,                   /* ptr to msg structure */
                 fc_mes0110,                      /* ptr to msg */
                  fc_msgBlk0110.msgPreambleStr,   /* begin varargs */
                   command,
                    cmd->ulpStatus,
                     cmd->un.ulpWord[4],
                      cmd->un.ulpWord[5]);        /* end varargs */
         if ((command == ELS_CMD_FARP) || 
             (command == ELS_CMD_FARPR)) {
            ep = (ELS_PKT * )lp0;
            if((ndlp=fc_findnode_wwpn(binfo, NLP_SEARCH_ALL, &ep->un.farp.RportName))) {
               ndlp->nlp_flag &= ~NLP_FARP_SND;
               ndlp->nlp_action &= ~NLP_DO_ADDR_AUTH;
            }
         }

         /* If error occurred on ADISC/PDISC, check to see if address
             * still needs to be authenticated.
             */
         if ((command == ELS_CMD_ADISC) || (command == ELS_CMD_PDISC)) {
            if(ndlp == 0) {
               ndlp = fc_findnode_odid(binfo, NLP_SEARCH_ALL,
                   (uint32)cmd->un.elsreq.remoteID);
            }
         }
         else {
            if (command == ELS_CMD_PLOGI) {
               if(ndlp == 0) {
                  ndlp = fc_findnode_odid(binfo, NLP_SEARCH_ALL,
                      (uint32)cmd->un.elsreq.remoteID);
               }
            }
         }

         /* check to see if we should retry */
         if (fc_els_retry(binfo, rp, temp, command, ndlp) == 0) {
            /* retry of ELS command failed */
            switch (command) {
            case ELS_CMD_FLOGI:
               if (ndlp)
                  ndlp->nlp_flag &= ~NLP_REQ_SND;
               if (binfo->fc_ffstate == FC_FLOGI) {
                  binfo->fc_flag &= ~(FC_FABRIC | FC_PUBLIC_LOOP);
                  fc_freenode_did(binfo, Fabric_DID, 1);
                  if (binfo->fc_topology == TOPOLOGY_LOOP) {
                     binfo->fc_edtov = FF_DEF_EDTOV;
                     binfo->fc_ratov = FF_DEF_RATOV;
                     if ((mb = (MAILBOXQ * )fc_mem_get(binfo, MEM_MBOX | MEM_PRI))) {
                        fc_config_link(p_dev_ctl, (MAILBOX * )mb);
                        if (issue_mb_cmd(binfo, (MAILBOX * )mb, MBX_NOWAIT)
                           != MBX_BUSY) {
                           fc_mem_put(binfo, MEM_MBOX, (uchar * )mb);
                        }
                     }
                     binfo->fc_flag |= FC_DELAY_DISC;
                  } else {
                     /* Device Discovery completion error */
                     fc_log_printf_msg_vargs( binfo->fc_brd_no,
                            &fc_msgBlk0207,                   /* ptr to msg structure */
                             fc_mes0207,                      /* ptr to msg */
                              fc_msgBlk0207.msgPreambleStr);  /* begin & end varargs */
                     binfo->fc_ffstate = FC_ERROR;
                     if ((mb = (MAILBOXQ * )fc_mem_get(binfo,
                         MEM_MBOX | MEM_PRI))) {
                        fc_clear_la(binfo, (MAILBOX * )mb);
                        if (issue_mb_cmd(binfo, (MAILBOX * )mb, MBX_NOWAIT)
                           != MBX_BUSY) {
                           fc_mem_put(binfo, MEM_MBOX, (uchar * )mb);
                        }
                     }
                  }
               }
               break;

            case ELS_CMD_PLOGI:
               /* Cache entry in case we are in a
                * LOGI collision.
                */
               if ((ndlp == 0) && ((ndlp = fc_findnode_odid(binfo, NLP_SEARCH_ALL,
                  (uint32)cmd->un.elsreq.remoteID)) == 0)) {
                  break;
               }

               /* If we are in the middle of Discovery */
               if (ndlp->nlp_action & (NLP_DO_ADDR_AUTH | NLP_DO_DISC_START | NLP_DO_RSCN)) {
                  if((ndlp->nlp_state < NLP_LOGIN) &&
                      !(ndlp->nlp_flag & NLP_REG_INP)) {
                  /* Goto next entry */
                  fc_nextnode(p_dev_ctl, ndlp);
               }
               }

               ndlp->nlp_action &= ~NLP_DO_RNID;
               ndlp->nlp_flag &= ~NLP_REQ_SND;

               if((ndlp->nlp_type & (NLP_AUTOMAP | NLP_SEED_MASK)) == 0) {
                  fc_freenode(binfo, ndlp, 1);
               }
               else {
                  fc_freenode(binfo, ndlp, 0);
                  ndlp->nlp_state = NLP_LIMBO;
                  fc_nlp_bind(binfo, ndlp);
               }
               break;

            case ELS_CMD_PRLI:
               if ((ndlp == 0) && ((ndlp = fc_findnode_odid(binfo, NLP_SEARCH_ALL,
                  (uint32)cmd->un.elsreq.remoteID)) == 0)) {
                  break;
               }

               /* If we are in the middle of Discovery */
               if (ndlp->nlp_action & (NLP_DO_DISC_START | NLP_DO_RSCN)) {
                  /* Goto next entry */
                  fc_nextnode(p_dev_ctl, ndlp);
               }
               ndlp->nlp_flag &= ~NLP_REQ_SND;
               ndlp->nlp_state = NLP_LOGIN;
               fc_nlp_unmap(binfo, ndlp);
               break;

            case ELS_CMD_PDISC:
            case ELS_CMD_ADISC:
               if ((ndlp == 0) && ((ndlp = fc_findnode_odid(binfo, NLP_SEARCH_ALL,
                  (uint32)cmd->un.elsreq.remoteID)) == 0)) {
                  break;
               }

               fc_freenode(binfo, ndlp, 0);
               ndlp->nlp_state = NLP_LIMBO;
               fc_nlp_bind(binfo, ndlp);

               ndlp->nlp_action |= NLP_DO_DISC_START;
               binfo->fc_nlp_cnt++;
               fc_els_cmd(binfo, ELS_CMD_PLOGI,
                  (void *)((ulong)cmd->un.elsreq.remoteID),
                  (uint32)0, (ushort)0, ndlp);
               break;

            case ELS_CMD_LOGO:
               if ((ndlp == 0) && ((ndlp = fc_findnode_odid(binfo, NLP_SEARCH_ALL,
                  (uint32)cmd->un.elsreq.remoteID)) == 0)) {
                  break;
               }

               /* If we are in the middle of Discovery */
               if (ndlp->nlp_action & (NLP_DO_ADDR_AUTH | NLP_DO_DISC_START | NLP_DO_RSCN)) {
                  /* Goto next entry */
                  fc_nextnode(p_dev_ctl, ndlp);
               }

               fc_freenode(binfo, ndlp, 0);
               ndlp->nlp_state = NLP_LIMBO;
               fc_nlp_bind(binfo, ndlp);
               break;

            case ELS_CMD_FARP:  /* Farp-req */
               if (ndlp == 0)
                  break;

               ndlp->nlp_flag &= ~NLP_FARP_SND;
               ndlp->nlp_action &= ~NLP_DO_ADDR_AUTH;
               break;

            case ELS_CMD_FARPR: /* Farp-res */
               if (ndlp == 0)
                  break;

               ndlp->nlp_flag &= ~NLP_FARP_SND;
               ndlp->nlp_action &= ~NLP_DO_ADDR_AUTH;
               fc_freenode(binfo, ndlp, 0);
               ndlp->nlp_state = NLP_LIMBO;
               fc_nlp_bind(binfo, ndlp);
               break;

            case ELS_CMD_SCR:   /* State Change Registration */
               break;

            case ELS_CMD_RNID:   /* Node Identification */
               break;

            default:
               FCSTATCTR.elsCmdPktInval++;

               /* Unknown ELS command <elsCmd> */
               fc_log_printf_msg_vargs( binfo->fc_brd_no,
                      &fc_msgBlk0111,                   /* ptr to msg structure */
                       fc_mes0111,                      /* ptr to msg */
                        fc_msgBlk0111.msgPreambleStr,   /* begin varargs */
                         command);                      /* end varargs */
               break;
            }
         }
         else {
            /* Retry in progress */
            if ((command == ELS_CMD_PLOGI) &&
               ((cmd->un.ulpWord[4] & 0xff) == IOERR_LOOP_OPEN_FAILURE)) {
               if (ndlp->nlp_action & (NLP_DO_ADDR_AUTH | NLP_DO_DISC_START | NLP_DO_RSCN)) {
                  /* Goto next entry */
                  fc_nextnode(p_dev_ctl, ndlp);
               }
            }
         }
      } else {
         FCSTATCTR.elsXmitCmpl++;

         /* Process successful command completion */
         switch (command) {
         case ELS_CMD_FLOGI:
            /* FLOGI completes successfully */
            fc_log_printf_msg_vargs( binfo->fc_brd_no,
                   &fc_msgBlk0208,                   /* ptr to msg structure */
                    fc_mes0208,                      /* ptr to msg */
                     fc_msgBlk0208.msgPreambleStr,   /* begin varargs */
                      cmd->un.ulpWord[4],
                       sp->cmn.e_d_tov,
                        sp->cmn.w2.r_a_tov,
                         sp->cmn.edtovResolution);   /* end varargs */
            if (ndlp)
               ndlp->nlp_flag &= ~NLP_REQ_SND;

            /* register the login, REG_LOGIN */
            if (binfo->fc_ffstate == FC_FLOGI) {
               /* If Common Service Parameters indicate Nport
                * we are point to point, if Fport we are Fabric.
                */
               if (sp->cmn.fPort) {
                  binfo->fc_flag |= FC_FABRIC;
                  if (sp->cmn.edtovResolution) {
                     /* E_D_TOV ticks are in nanoseconds */
                     binfo->fc_edtov = (SWAP_DATA(sp->cmn.e_d_tov) + 999999) / 1000000;
                  } else {
                     /* E_D_TOV ticks are in milliseconds */
                     binfo->fc_edtov = SWAP_DATA(sp->cmn.e_d_tov);
                  }
                  binfo->fc_ratov = (SWAP_DATA(sp->cmn.w2.r_a_tov) + 999) / 1000;

                  if (binfo->fc_topology == TOPOLOGY_LOOP) {
                     binfo->fc_flag |= FC_PUBLIC_LOOP;
                  } else {
                     /* If we are a N-port connected to a Fabric, 
                      * fixup sparam's so logins to devices on
                      * remote loops work.
                      */
                     binfo->fc_sparam.cmn.altBbCredit = 1;
                  }

                  binfo->fc_myDID = cmd->un.ulpWord[4] & Mask_DID;

                  if ((ndlp == 0) && ((ndlp = fc_findnode_odid(binfo, NLP_SEARCH_ALL,
                     (uint32)cmd->un.elsreq.remoteID)) == 0)) {
                     break;
                  }
                  ndlp->nlp_type |= NLP_FABRIC;
                  ndlp->nlp_flag &= ~NLP_REQ_SND;

                  fc_nlp_logi(binfo, ndlp,
                     (NAME_TYPE *)&sp->portName, (NAME_TYPE *)&sp->nodeName);

                  if ((mb=(MAILBOXQ * )fc_mem_get(binfo, MEM_MBOX | MEM_PRI))) {
                     fc_config_link(p_dev_ctl, (MAILBOX * )mb);
                     if (issue_mb_cmd(binfo, (MAILBOX * )mb, MBX_NOWAIT)
                        != MBX_BUSY) {
                        fc_mem_put(binfo, MEM_MBOX, (uchar * )mb);
                     }
                  }

                  /* register the login with adapter */
                  if (ndlp->nlp_Rpi == 0) {
                     fc_bcopy((void *)sp, (void *) & binfo->fc_fabparam, 
                         sizeof(SERV_PARM));

                     if ((mb = (MAILBOXQ * )fc_mem_get(binfo,
                        MEM_MBOX | MEM_PRI))) {
                        fc_reg_login(binfo, cmd->un.elsreq.remoteID,
                           (uchar * )sp, (MAILBOX * )mb, 0);
                        if (issue_mb_cmd(binfo, (MAILBOX * )mb, MBX_NOWAIT)
                           != MBX_BUSY) {
                           fc_mem_put(binfo, MEM_MBOX, (uchar * )mb);
                        }
                     }
                  }

                  binfo->fc_flag |= FC_DELAY_DISC;
               } else {
                  binfo->fc_flag &= ~(FC_FABRIC | FC_PUBLIC_LOOP);
                  binfo->fc_edtov = FF_DEF_EDTOV;
                  binfo->fc_ratov = FF_DEF_RATOV;
                  if ((rc = fc_geportname((NAME_TYPE * ) & binfo->fc_portname,
                      (NAME_TYPE * ) & sp->portName))) {
                     /* This side will initiate the PLOGI */
                     binfo->fc_flag |= FC_PT2PT_PLOGI;

                     /* N_Port ID cannot be 0, set our to LocalID the 
                      * other side will be RemoteID.
                      */
                     fc_freenode_did(binfo, 0, 1);

                     /* not equal */
                     if (rc == 1)
                        binfo->fc_myDID = PT2PT_LocalID;
                     rc = 0;

                     if ((mb = (MAILBOXQ * )fc_mem_get(binfo,
                         MEM_MBOX | MEM_PRI))) {
                        fc_config_link(p_dev_ctl, (MAILBOX * )mb);
                        if (issue_mb_cmd(binfo, (MAILBOX * )mb, MBX_NOWAIT)
                           != MBX_BUSY) {
                           fc_mem_put(binfo, MEM_MBOX, (uchar * )mb);
                        }
                     }
                     if ((ndlp = fc_findnode_odid(binfo, NLP_SEARCH_ALL, (uint32)binfo->fc_myDID)) == 0) {
                        if((ndlp = (NODELIST *)fc_mem_get(binfo, MEM_NLP))) {
                           fc_bzero((void *)ndlp, sizeof(NODELIST));
                           ndlp->sync = binfo->fc_sync;
                           ndlp->capabilities = binfo->fc_capabilities;
                           ndlp->nlp_DID = binfo->fc_myDID;
                           ndlp->nlp_state = NLP_LIMBO;
                           fc_nlp_bind(binfo, ndlp);
                        }
                        else
                           break;
                     }
                     ndlp->nlp_DID = binfo->fc_myDID;
                     fc_nlp_logi(binfo, ndlp,
                        (NAME_TYPE *)&sp->portName, (NAME_TYPE *)&sp->nodeName);
                  } else {
                     if ((mb = (MAILBOXQ * )fc_mem_get(binfo,
                         MEM_MBOX | MEM_PRI))) {
                        fc_config_link(p_dev_ctl, (MAILBOX * )mb);
                        if (issue_mb_cmd(binfo, (MAILBOX * )mb, MBX_NOWAIT)
                           != MBX_BUSY) {
                           fc_mem_put(binfo, MEM_MBOX, (uchar * )mb);
                        }
                     }
                  }
                  binfo->fc_flag |= FC_PT2PT;
                  /* Use Fabric timer as pt2pt link up timer */
                  binfo->fc_fabrictmo = (2 * binfo->fc_ratov) +
                     ((4 * binfo->fc_edtov) / 1000) + 1;
                  if(FABRICTMO) {
                     fc_clk_res(p_dev_ctl, binfo->fc_fabrictmo, FABRICTMO);
                  }
                  else {
                     FABRICTMO = fc_clk_set(p_dev_ctl, binfo->fc_fabrictmo,
                            fc_fabric_timeout, 0, 0);
                  }
                  fc_freenode_did(binfo, Fabric_DID, 1);

                  /* This is Login at init, clear la */
                  if ((mb=(MAILBOXQ * )fc_mem_get(binfo, MEM_MBOX | MEM_PRI))) {
                     binfo->fc_ffstate = FC_CLEAR_LA;
                     fc_clear_la(binfo, (MAILBOX * )mb);
                     if (issue_mb_cmd(binfo, (MAILBOX * )mb, MBX_NOWAIT)
                        != MBX_BUSY) {
                        fc_mem_put(binfo, MEM_MBOX, (uchar * )mb);
                     }
                  }
               }
            } else {
               FCSTATCTR.elsRcvDrop++;
               fc_freenode_did(binfo, Fabric_DID, 1);
            }
            break;

         case ELS_CMD_PLOGI:
            /* PLOGI completes successfully */
            fc_log_printf_msg_vargs( binfo->fc_brd_no,
                   &fc_msgBlk0112,                   /* ptr to msg structure */
                    fc_mes0112,                      /* ptr to msg */
                     fc_msgBlk0112.msgPreambleStr,   /* begin varargs */
                      cmd->un.elsreq.remoteID,
                       cmd->un.ulpWord[4],
                        cmd->un.ulpWord[5],
                         binfo->fc_ffstate);         /* end varargs */
            if ((ndlp == 0) && ((ndlp = fc_findnode_odid(binfo, NLP_SEARCH_ALL,
              (uint32)cmd->un.elsreq.remoteID)) == 0)) {
               break;
            }
            fc_nlp_logi(binfo, ndlp,
               (NAME_TYPE *)&sp->portName, (NAME_TYPE *)&sp->nodeName);

            if (ndlp->nlp_DID != NameServer_DID)
               ndlp->nlp_action &= ~NLP_DO_ADDR_AUTH;


            ndlp->nlp_action &= ~NLP_DO_RNID;

            if (binfo->fc_flag & FC_PT2PT) {
               /* Device Discovery completes */
               fc_log_printf_msg_vargs( binfo->fc_brd_no,
                      &fc_msgBlk0209,                   /* ptr to msg structure */
                       fc_mes0209,                      /* ptr to msg */
                        fc_msgBlk0209.msgPreambleStr);  /* begin & end varargs */
               /* Fix up any changed RPIs in FCP IOCBs queued up a txq */
               fc_fcp_fix_txq(p_dev_ctl);

               binfo->fc_ffstate = FC_READY;

               binfo->fc_firstopen = 0;
               if(FABRICTMO) {
                  fc_clk_can(p_dev_ctl, FABRICTMO);
                  FABRICTMO = 0;
               }
               ndlp->nlp_Rpi = 0; /* Keep the same rpi */
            }

            if (ndlp->nlp_Rpi) {
               /* must explicitly unregister the login, UREG_LOGIN */
               /* This is so pending I/Os get returned with NO_RPI */
               if ((mb = (MAILBOXQ * )fc_mem_get(binfo, MEM_MBOX | MEM_PRI))) {
                  fc_unreg_login(binfo, ndlp->nlp_Rpi, (MAILBOX * )mb);
                  if (issue_mb_cmd(binfo,(MAILBOX * )mb,MBX_NOWAIT) != MBX_BUSY) {
                     fc_mem_put(binfo, MEM_MBOX, (uchar * )mb);
                  }
               }
               binfo->fc_nlplookup[ndlp->nlp_Rpi] = 0;
               ndlp->nlp_Rpi = 0;
            }

            /* register the login, REG_LOGIN */
            if ((mb = (MAILBOXQ * )fc_mem_get(binfo, MEM_MBOX | MEM_PRI))) {
               fc_reg_login(binfo, cmd->un.elsreq.remoteID, (uchar * )sp, 
                   (MAILBOX * )mb, 0);
               if (issue_mb_cmd(binfo,(MAILBOX * )mb,MBX_NOWAIT) != MBX_BUSY) {
                  fc_mem_put(binfo, MEM_MBOX, (uchar * )mb);
               }
            }

            /* Fill in the FCP/IP class */
            {
                  clp = DD_CTL.p_config[binfo->fc_brd_no];
                  if ( (clp[CFG_FCP_CLASS].a_current == CLASS2) &&
                       (sp->cls2.classValid) ) {
                     ndlp->id.nlp_fcp_info |= CLASS2;
                  } else {
                     ndlp->id.nlp_fcp_info |= CLASS3;
                  }

                  if ( (clp[CFG_IP_CLASS].a_current == CLASS2) &&
                       (sp->cls2.classValid) ) {
                     ndlp->id.nlp_ip_info = CLASS2;
                  } else {
                     ndlp->id.nlp_ip_info = CLASS3;
                  }
            }

            /* REG_LOGIN cmpl will goto nextnode */
            ndlp->nlp_flag &= ~NLP_REQ_SND;
            ndlp->nlp_flag |= NLP_REG_INP;
            break;

         case ELS_CMD_PRLI:
            /* PRLI completes successfully */
            fc_log_printf_msg_vargs( binfo->fc_brd_no,
                   &fc_msgBlk0113,                   /* ptr to msg structure */
                    fc_mes0113,                      /* ptr to msg */
                     fc_msgBlk0113.msgPreambleStr,   /* begin varargs */
                      cmd->un.elsreq.remoteID,
                       cmd->un.ulpWord[4],
                        cmd->un.ulpWord[5],
                         binfo->fc_ffstate);         /* end varargs */
            if ((ndlp == 0) && ((ndlp = fc_findnode_odid(binfo, NLP_SEARCH_ALL,
               cmd->un.elsreq.remoteID)) == 0))
               break;

            ndlp->nlp_flag &= ~NLP_REQ_SND;

            /* If we are in the middle of Discovery or in pt2pt mode */
            if ((ndlp->nlp_action & (NLP_DO_DISC_START | NLP_DO_RSCN)) ||
               (binfo->fc_flag & FC_PT2PT)) {
               int     index;
               node_t  * node_ptr;
               PRLI    * npr;

               npr = (PRLI * )sp;
               clp = DD_CTL.p_config[binfo->fc_brd_no];
               if ((npr->acceptRspCode == PRLI_REQ_EXECUTED) && 
                   (npr->prliType == PRLI_FCP_TYPE) && 
                   (npr->targetFunc == 1)) {

                  if(clp[CFG_FCP_ON].a_current) {
                     if (!(fc_assign_scsid(p_dev_ctl, ndlp))) {
                        /* No more SCSI ids available */
                        fc_nextnode(p_dev_ctl, ndlp);
                        ndlp->nlp_state = NLP_PRLI;
                        fc_nlp_unmap(binfo, ndlp);
                        ndlp->nlp_action &= ~NLP_DO_SCSICMD;
                        break;
                     }

                     if ((node_ptr = (node_t * )ndlp->nlp_targetp) == NULL) {
                        index = INDEX(ndlp->id.nlp_pan, ndlp->id.nlp_sid);
                        node_ptr =  binfo->device_queue_hash[index].node_ptr;
                     }
                     /* PRLI target assigned */
                     fc_log_printf_msg_vargs( binfo->fc_brd_no,
                            &fc_msgBlk0210,                   /* ptr to msg structure */
                             fc_mes0210,                      /* ptr to msg */
                              fc_msgBlk0210.msgPreambleStr,   /* begin varargs */
                               cmd->un.ulpWord[5],            /* did */
                                ndlp->id.nlp_pan,
                                 ndlp->id.nlp_sid);           /* end varargs */
                     /* Now check for FCP-2 support */
                     if(node_ptr) {
                        if(npr->Retry && npr->TaskRetryIdReq)
                           node_ptr->flags |= FC_FCP2_RECOVERY;
                        else
                           node_ptr->flags &= ~FC_FCP2_RECOVERY;
                     }

                  }
                  else {
                     goto prlierr;
                  }

                  /* If PRLI is successful, we have a FCP target device */
                  if (((PRLI * )sp)->Retry == 1) {
                     ndlp->id.nlp_fcp_info |= NLP_FCP_2_DEVICE;
                  } 
                  ndlp->nlp_type |= NLP_FCP_TARGET;
                  if((ndlp->nlp_type & NLP_SEED_MASK) == 0) {
                     switch(p_dev_ctl->fcp_mapping) {
                     case FCP_SEED_DID:
                        ndlp->nlp_type |= NLP_SEED_DID;
                        break;
                     case FCP_SEED_WWPN:
                        ndlp->nlp_type |= NLP_SEED_WWPN;
                        break;
                     case FCP_SEED_WWNN:
                     default:
                        ndlp->nlp_type |= NLP_SEED_WWNN;
                        break;
                     }
                     if(clp[CFG_AUTOMAP].a_current)
                        ndlp->nlp_type |= NLP_AUTOMAP;
                  }
                  ndlp->nlp_state = NLP_ALLOC;
                  fc_nlp_map(binfo, ndlp);

                  /* Fix up any changed RPIs in FCP IOCBs queued up a txq */
                  fc_fcp_fix_txq(p_dev_ctl);

                  fc_nextnode(p_dev_ctl, ndlp);
               } else {
prlierr:
                   ndlp->nlp_state = NLP_LOGIN;
                   fc_nlp_unmap(binfo, ndlp);
                   fc_nextnode(p_dev_ctl, ndlp);
               }
            }
            else {
               PRLI    * npr;

               npr = (PRLI * )sp;
               if ((npr->acceptRspCode == PRLI_REQ_EXECUTED) && 
                   (npr->prliType == PRLI_FCP_TYPE) && 
                   (npr->targetFunc == 1)) {
                   if(ndlp->nlp_type & NLP_FCP_TARGET) {
                      ndlp->nlp_state = NLP_ALLOC;
                      fc_nlp_map(binfo, ndlp);
                   }
                   else {
                       ndlp->nlp_state = NLP_PRLI;
                       fc_nlp_unmap(binfo, ndlp);
                   }
               }
               else {
                   ndlp->nlp_state = NLP_LOGIN;
                   fc_nlp_unmap(binfo, ndlp);
               }
            }

            ndlp->nlp_action &= ~NLP_DO_SCSICMD;
            break;

         case ELS_CMD_PRLO:
            /* PRLO completes successfully */
            fc_log_printf_msg_vargs( binfo->fc_brd_no,
                   &fc_msgBlk0114,                   /* ptr to msg structure */
                    fc_mes0114,                      /* ptr to msg */
                     fc_msgBlk0114.msgPreambleStr,   /* begin varargs */
                      cmd->un.elsreq.remoteID,
                       cmd->un.ulpWord[4],
                        cmd->un.ulpWord[5],
                         binfo->fc_ffstate);         /* end varargs */
            break;

         case ELS_CMD_LOGO:
            /* LOGO completes successfully */
            fc_log_printf_msg_vargs( binfo->fc_brd_no,
                   &fc_msgBlk0115,                   /* ptr to msg structure */
                    fc_mes0115,                      /* ptr to msg */
                     fc_msgBlk0115.msgPreambleStr,   /* begin varargs */
                      cmd->un.elsreq.remoteID,
                       cmd->un.ulpWord[4],
                        cmd->un.ulpWord[5],
                         binfo->fc_ffstate);         /* end varargs */
            if ((ndlp == 0) && ((ndlp = fc_findnode_odid(binfo, NLP_SEARCH_ALL,
               (uint32)cmd->un.elsreq.remoteID)) == 0)) {
               break;
            }

            /* If we are in the middle of Discovery */
            if (ndlp->nlp_action & (NLP_DO_ADDR_AUTH | NLP_DO_DISC_START | NLP_DO_RSCN)) {
               /* Goto next entry */
               fc_nextnode(p_dev_ctl, ndlp);
            }

            fc_freenode(binfo, ndlp, 0);
            ndlp->nlp_state = NLP_LIMBO;
            fc_nlp_bind(binfo, ndlp);
            break;

         case ELS_CMD_PDISC:
            /* PDISC completes successfully */
            fc_log_printf_msg_vargs( binfo->fc_brd_no,
                   &fc_msgBlk0116,                   /* ptr to msg structure */
                    fc_mes0116,                      /* ptr to msg */
                     fc_msgBlk0116.msgPreambleStr,   /* begin varargs */
                      cmd->un.elsreq.remoteID,
                       cmd->un.ulpWord[4],
                        cmd->un.ulpWord[5],
                         binfo->fc_ffstate);         /* end varargs */
            if ((ndlp == 0) && ((ndlp = fc_findnode_odid(binfo, NLP_SEARCH_ALL,
               (uint32)cmd->un.elsreq.remoteID)) == 0)) {
               break;
            }

            /* If we are in the middle of Address Authentication */
            if (ndlp->nlp_action & (NLP_DO_ADDR_AUTH)) {
               if (fc_chkpadisc(binfo, ndlp, &sp->nodeName,
                   &sp->portName) == 0) {
                  fc_freenode(binfo, ndlp, 0);
                  ndlp->nlp_state = NLP_LIMBO;
                  fc_nlp_bind(binfo, ndlp);
                  ndlp->nlp_action |= NLP_DO_DISC_START;
               }
               /* Goto next entry */
               fc_nextnode(p_dev_ctl, ndlp);
            }
            ndlp->nlp_flag &= ~NLP_REQ_SND_PDISC;
            break;

         case ELS_CMD_ADISC:
            /* ADISC completes successfully */
            fc_log_printf_msg_vargs( binfo->fc_brd_no,
                   &fc_msgBlk0117,                   /* ptr to msg structure */
                    fc_mes0117,                      /* ptr to msg */
                     fc_msgBlk0117.msgPreambleStr,   /* begin varargs */
                      cmd->un.elsreq.remoteID,
                       cmd->un.ulpWord[4],
                        cmd->un.ulpWord[5],
                         binfo->fc_ffstate);         /* end varargs */
            if ((ndlp == 0) && ((ndlp = fc_findnode_odid(binfo, NLP_SEARCH_ALL,
               (uint32)cmd->un.elsreq.remoteID)) == 0)) {
               break;
            }
            ndlp->nlp_flag &= ~NLP_REQ_SND_ADISC;

            /* If we are in the middle of Address Authentication */
            if (ndlp->nlp_action & (NLP_DO_ADDR_AUTH | NLP_DO_RSCN)) {

               ap = (ADISC * )sp;
               if(fc_chkpadisc(binfo, ndlp, &ap->nodeName,&ap->portName) == 0) {
                  ndlp->nlp_action &= ~(NLP_DO_ADDR_AUTH | NLP_DO_DISC_START);
                  fc_freenode(binfo, ndlp, 0);
                  ndlp->nlp_state = NLP_LIMBO;
                  fc_nlp_bind(binfo, ndlp);


                  if((binfo->fc_flag & FC_RSCN_MODE) ||
                     (binfo->fc_ffstate < FC_READY)) {
                     ndlp->nlp_DID = (uint32)cmd->un.elsreq.remoteID;


                     if(binfo->fc_flag & FC_RSCN_MODE) {
                        ndlp->nlp_action |= NLP_DO_RSCN;
                        binfo->fc_nlp_cnt--;
                        if (binfo->fc_nlp_cnt <= 0) {
                           binfo->fc_nlp_cnt = 0;
                           fc_nextrscn(p_dev_ctl, fc_max_els_sent);
                        }
                     }
                     else {
                        ndlp->nlp_action |= NLP_DO_DISC_START;
                        binfo->fc_nlp_cnt--;
                        if (binfo->fc_nlp_cnt <= 0) {
                           binfo->fc_nlp_cnt = 0;
                           fc_nextdisc(p_dev_ctl, fc_max_els_sent);
                        }
                     }
                  }
               }
               else {
                  fc_nextnode(p_dev_ctl, ndlp);
               }
            }
            break;

         case ELS_CMD_FARP:
         case ELS_CMD_FARPR:
            /* FARP completes successfully */
            fc_log_printf_msg_vargs( binfo->fc_brd_no,
                   &fc_msgBlk0118,                   /* ptr to msg structure */
                    fc_mes0118,                      /* ptr to msg */
                     fc_msgBlk0118.msgPreambleStr,   /* begin varargs */
                      cmd->un.elsreq.remoteID,
                       cmd->un.ulpWord[4],
                        cmd->un.ulpWord[5],
                         command );                  /* end varargs */
            ep = (ELS_PKT * )lp0;
            if((ndlp = fc_findnode_wwpn(binfo, NLP_SEARCH_ALL,
                  &ep->un.farp.RportName)) == 0)
               break;

            ndlp->nlp_flag &= ~NLP_FARP_SND;
            ndlp->nlp_action &= ~NLP_DO_ADDR_AUTH;
            break;

         case ELS_CMD_SCR:      /* State Change Registration */
            /* SCR completes successfully */
            fc_log_printf_msg_vargs( binfo->fc_brd_no,
                   &fc_msgBlk0119,                   /* ptr to msg structure */
                    fc_mes0119,                      /* ptr to msg */
                     fc_msgBlk0119.msgPreambleStr,   /* begin varargs */
                      cmd->un.elsreq.remoteID,
                       cmd->un.ulpWord[4],
                        cmd->un.ulpWord[5],
                         binfo->fc_ffstate);         /* end varargs */
            break;

         case ELS_CMD_RNID:   /* Node Identification */
            /* RNID completes successfully */
            fc_log_printf_msg_vargs( binfo->fc_brd_no,
                   &fc_msgBlk0120,                   /* ptr to msg structure */
                    fc_mes0120,                      /* ptr to msg */
                     fc_msgBlk0120.msgPreambleStr,   /* begin varargs */
                      cmd->un.elsreq.remoteID,
                       cmd->un.ulpWord[4],
                        cmd->un.ulpWord[5],
                         binfo->fc_ffstate);         /* end varargs */
            break;

         default:
            FCSTATCTR.elsCmdPktInval++;

            /* Unknown ELS command <elsCmd> completed */
            fc_log_printf_msg_vargs( binfo->fc_brd_no,
                   &fc_msgBlk0121,                   /* ptr to msg structure */
                    fc_mes0121,                      /* ptr to msg */
                     fc_msgBlk0121.msgPreambleStr,   /* begin varargs */
                      command);                      /* end varargs */
            break;
         }
      }
      if (xmitiq->bpl) {
         fc_mem_put(binfo, MEM_BPL, (uchar * )xmitiq->bpl);
      }
      fc_mem_put(binfo, MEM_BUF, (uchar * )mp);
      fc_mem_put(binfo, MEM_BUF, (uchar * )rmp);
      break;

   default:
      FCSTATCTR.elsCmdIocbInval++;

      /* Unknown ELS IOCB */
      fc_log_printf_msg_vargs( binfo->fc_brd_no,
             &fc_msgBlk0122,                   /* ptr to msg structure */
              fc_mes0122,                      /* ptr to msg */
               fc_msgBlk0122.msgPreambleStr,   /* begin varargs */
                cmd->ulpCommand );             /* end varargs */

out2:
      rc = EINVAL;

      if(xmitiq->bp) {
            fc_mem_put(binfo, MEM_BUF, (uchar * )xmitiq->bp);
      }
      if(xmitiq->info) {
            fc_mem_put(binfo, MEM_BUF, (uchar * )xmitiq->info);
      }
      if(xmitiq->bpl) {
            fc_mem_put(binfo, MEM_BPL, (uchar * )xmitiq->bpl);
      }

      break;
   } /* switch(cmd->ulpCommand) */

   fc_mem_put(binfo, MEM_IOCB, (uchar * )xmitiq);

   return(rc);
}       /* End handle_els_event */


/**********************************************/
/**  handle_rcv_els_req                      **/
/**                                          **/
/**  Process an incoming ELS request         **/
/**********************************************/
_static_ int
handle_rcv_els_req(
fc_dev_ctl_t *p_dev_ctl,
RING         *rp,
IOCBQ        *temp)
{
   IOCB         * iocb;
   FC_BRD_INFO  * binfo;
   uint32       * lp;
   NODELIST     * ndlp;
   volatile SERV_PARM   * sp;
   NAME_TYPE    * np;
   ELS_PKT      * ep;
   MAILBOXQ     * mb;
   MATCHMAP     * mp;
   IOCBQ        * saveq;
   IOCBQ        * iocbq;
   uchar        * bp;
   uchar        * bdeAddr;
   iCfgParam    * clp;
   int          i, cnt;
   uint32       cmd;
   uint32       did;
   LS_RJT       stat;
   REG_WD30     wd30;

   iocb = &temp->iocb;
   binfo = &BINFO;
   clp = DD_CTL.p_config[binfo->fc_brd_no];

   if (binfo->fc_flag & FC_SLI2) {
      /* type of ELS cmd is first 32bit word in packet */
      mp = fc_getvaddr(p_dev_ctl, rp,
         (uchar * )getPaddr(iocb->un.cont64[0].addrHigh,
                            iocb->un.cont64[0].addrLow));
   } else {
      /* type of ELS cmd is first 32bit word in packet */
      mp = fc_getvaddr(p_dev_ctl, rp, (uchar * )((ulong)iocb->un.cont[0].bdeAddress));
   }

   if (mp == 0) {

      if (binfo->fc_flag & FC_SLI2) {
         bdeAddr = (uchar *)getPaddr(iocb->un.cont64[0].addrHigh,
                            iocb->un.cont64[0].addrLow);
      }
      else {
         bdeAddr = (uchar *)((ulong)iocb->un.cont[0].bdeAddress);
      }
      FCSTATCTR.elsRcvDrop++;

      goto out;
   }

   bp = mp->virt;
   lp = (uint32 * )bp;
   cmd = *lp++;

      /* Received ELS command <elsCmd> */
      fc_log_printf_msg_vargs( binfo->fc_brd_no,
             &fc_msgBlk0123,                   /* ptr to msg structure */
              fc_mes0123,                      /* ptr to msg */
               fc_msgBlk0123.msgPreambleStr,   /* begin varargs */
                cmd,
                 iocb->un.ulpWord[5],
                  iocb->ulpStatus,
                   binfo->fc_ffstate);         /* end varargs */
   if ((iocb->ulpStatus) || 
       ((binfo->fc_ffstate <= FC_FLOGI) && 
       ((cmd != ELS_CMD_FLOGI) && (cmd != ELS_CMD_ADISC) && 
       (cmd != ELS_CMD_FAN)))) {
      if ((iocb->ulpStatus == 0) && (cmd == ELS_CMD_PLOGI)) {
         /* Do this for pt2pt as well, testing with miniport driver */

         /* Reject this request because we are in process of discovery */
         stat.un.b.lsRjtRsvd0 = 0;
         stat.un.b.lsRjtRsnCode = LSRJT_UNABLE_TPC;
         stat.un.b.lsRjtRsnCodeExp = LSEXP_NOTHING_MORE;
         stat.un.b.vendorUnique = 0;
         fc_els_rsp(binfo, ELS_CMD_LS_RJT, (uint32)iocb->ulpContext,
             (uint32)iocb->ulpClass, (void *)0, (uint32)stat.un.lsRjtError, 0);
      }

      fc_mem_put(binfo, MEM_BUF, (uchar * )mp);

      i = 1;
      /* free resources associated with iocb and repost the ring buffers */
      if (!(binfo->fc_flag & FC_SLI2)) {
         for (i = 1; i < (int)iocb->ulpBdeCount; i++) {
         mp = fc_getvaddr(p_dev_ctl, rp, (uchar * )((ulong)iocb->un.cont[i].bdeAddress));
            if (mp) {
               fc_mem_put(binfo, MEM_BUF, (uchar * )mp);
            }
         }
      }
      fc_post_buffer(p_dev_ctl, rp, i);
      /* Drop frame if there is an error */
      FCSTATCTR.elsRcvDrop++;
      return(0);
   }

   /* Special case RSCN cause 2 byte payload length field is variable */
   if ((cmd & ELS_CMD_MASK) == ELS_CMD_RSCN) {
      /* Received RSCN command */
      fc_log_printf_msg_vargs( binfo->fc_brd_no,
             &fc_msgBlk0211,                   /* ptr to msg structure */
              fc_mes0211,                      /* ptr to msg */
               fc_msgBlk0211.msgPreambleStr,   /* begin varargs */
                binfo->fc_flag, 
                 binfo->fc_defer_rscn.q_cnt,
                  binfo->fc_rscn.q_cnt,
                   binfo->fc_mbox_active );    /* end varargs */
      /* ACCEPT the rscn request */
      fc_els_rsp(binfo, ELS_CMD_ACC, (uint32)iocb->ulpContext,
          (uint32)iocb->ulpClass, (void *)0, (uint32)(sizeof(uint32)), 0);

      if ((binfo->fc_flag & FC_RSCN_DISCOVERY) ||
         ((binfo->fc_flag & FC_RSCN_MODE) && !(binfo->fc_flag & FC_NSLOGI_TMR)) ||
         (binfo->fc_ffstate != FC_READY)) {
         if(binfo->fc_defer_rscn.q_cnt > FC_MAX_HOLD_RSCN) {
            binfo->fc_flag |= (FC_RSCN_DISCOVERY | FC_RSCN_MODE);
            fc_flush_rscn_defer(p_dev_ctl);
            goto dropit;
         }
         if(binfo->fc_flag & FC_RSCN_DISCOVERY) {
            goto dropit;
         }
         else {
            /* get an iocb buffer to copy entry into */
            if ((saveq = (IOCBQ * )fc_mem_get(binfo, MEM_IOCB | MEM_PRI)) != NULL) {
               fc_bcopy((uchar *)temp, (uchar *)saveq, sizeof(IOCBQ));
               if (binfo->fc_defer_rscn.q_first) {
                  /* queue command to end of list */
                  ((IOCBQ * )binfo->fc_defer_rscn.q_last)->q = (uchar * )saveq;
                  binfo->fc_defer_rscn.q_last = (uchar * )saveq;
               } else {
                  /* add command to empty list */
                  binfo->fc_defer_rscn.q_first = (uchar * )saveq;
                  binfo->fc_defer_rscn.q_last = (uchar * )saveq;
               }
               saveq->q = NULL;
               *((MATCHMAP **)&saveq->iocb) = mp;
               binfo->fc_defer_rscn.q_cnt++;
               binfo->fc_flag |= FC_RSCN_MODE;
               if (!(binfo->fc_flag & FC_SLI2)) {
                  i = (int)iocb->ulpBdeCount;
               }
               else {
                  i = 1;
               }
               fc_post_buffer(p_dev_ctl, rp, i);
               return(0);
            }
            else
               goto dropit;
         }
      }

      /* Make sure all outstanding Mailbox cmds (reg/unreg login) are processed
       * before processing RSCN.
       */
      if (binfo->fc_mbox_active) {
         /* get an iocb buffer to copy entry into */
         if ((saveq = (IOCBQ * )fc_mem_get(binfo, MEM_IOCB | MEM_PRI)) != NULL) {
            fc_bcopy((uchar *)temp, (uchar *)saveq, sizeof(IOCBQ));
            binfo->fc_flag |= (FC_DELAY_RSCN | FC_RSCN_MODE);
            if (binfo->fc_rscn.q_first) {
               /* queue command to end of list */
               ((IOCBQ * )binfo->fc_rscn.q_last)->q = (uchar * )saveq;
               binfo->fc_rscn.q_last = (uchar * )saveq;
            } else {
               /* add command to empty list */
               binfo->fc_rscn.q_first = (uchar * )saveq;
               binfo->fc_rscn.q_last = (uchar * )saveq;
            }

            saveq->q = NULL;
            *((MATCHMAP **)&saveq->iocb) = mp;
            binfo->fc_rscn.q_cnt++;
            if (!(binfo->fc_flag & FC_SLI2)) {
               i = (int)iocb->ulpBdeCount;
            }
            else {
               i = 1;
            }
            fc_post_buffer(p_dev_ctl, rp, i);
            return(0);
         }
         else
            goto dropit;
      }
      cmd &= ELS_CMD_MASK;
   }

   switch (cmd) {
   case ELS_CMD_PLOGI:
   case ELS_CMD_FLOGI:
      sp = (volatile SERV_PARM * )lp;
      did = iocb->un.elsreq.remoteID;
      if (cmd == ELS_CMD_FLOGI) {
         FCSTATCTR.elsRcvFLOGI++;
         if (binfo->fc_topology == TOPOLOGY_LOOP) {
            /* We should never recieve a FLOGI in loop mode, ignore it */
            /* An FLOGI ELS command <elsCmd> was received from DID <did> in Loop Mode */
            fc_log_printf_msg_vargs( binfo->fc_brd_no,
                   &fc_msgBlk0124,                   /* ptr to msg structure */
                    fc_mes0124,                      /* ptr to msg */
                     fc_msgBlk0124.msgPreambleStr,   /* begin varargs */
                      cmd,
                       did);                         /* end varargs */
            break;
         }
         did = Fabric_DID;
         if (fc_chksparm(binfo, sp, CLASS3)) {
            /* For a FLOGI we accept, then if our portname is greater
             * then the remote portname we initiate Nport login. 
             */
            int rc;
	    MAILBOX *tmpmb;

            rc = fc_geportname((NAME_TYPE * ) & binfo->fc_portname,
                (NAME_TYPE * ) & sp->portName);

	    if (rc == 2) {    /* ourselves */
               /* It's ourselves, so we will just reset link */
               if ((tmpmb = (MAILBOX * )fc_mem_get(binfo, MEM_MBOX | MEM_PRI)) == NULL) {
                  binfo->fc_ffstate = FC_ERROR;
                  return(1);
               }

               binfo->fc_flag |= FC_SCSI_RLIP;
               clp = DD_CTL.p_config[binfo->fc_brd_no];

               /* Setup and issue mailbox INITIALIZE LINK command */
               fc_linkdown(p_dev_ctl);
               fc_init_link(binfo, (MAILBOX * )tmpmb, clp[CFG_TOPOLOGY].a_current, clp[CFG_LINK_SPEED].a_current);
               tmpmb->un.varInitLnk.lipsr_AL_PA = 0;
               if (issue_mb_cmd(binfo, (MAILBOX * )tmpmb, MBX_NOWAIT) != MBX_BUSY)
                  fc_mem_put(binfo, MEM_MBOX, (uchar * )tmpmb);
               break;
            }

            if (p_dev_ctl->fc_waitflogi) {
               if (p_dev_ctl->fc_waitflogi != (FCCLOCK *)1)
                  fc_clk_can(p_dev_ctl, p_dev_ctl->fc_waitflogi);
               p_dev_ctl->fc_waitflogi = 0;
	       p_dev_ctl->power_up = 1; 
               fc_snd_flogi(p_dev_ctl, 0, 0);
            }

            fc_els_rsp(binfo, ELS_CMD_ACC, (uint32)iocb->ulpContext,
                (uint32)iocb->ulpClass, (void *)0, (uint32)sizeof(SERV_PARM), 0);
            if (rc == 1) {    /* greater than */
               binfo->fc_flag |= FC_PT2PT_PLOGI;
            }
            binfo->fc_flag |= FC_PT2PT;
            /* Use Fabric timer as pt2pt link up timer */
            binfo->fc_fabrictmo = (2 * binfo->fc_ratov) +
               ((4 * binfo->fc_edtov) / 1000) + 1;
            if(FABRICTMO) {
               fc_clk_res(p_dev_ctl, binfo->fc_fabrictmo, FABRICTMO);
            }
            else {
               FABRICTMO = fc_clk_set(p_dev_ctl, binfo->fc_fabrictmo,
                            fc_fabric_timeout, 0, 0);
            }
            binfo->fc_flag &= ~(FC_FABRIC | FC_PUBLIC_LOOP);
         } else {
            /* Reject this request because invalid parameters */
            stat.un.b.lsRjtRsvd0 = 0;
            stat.un.b.lsRjtRsnCode = LSRJT_UNABLE_TPC;
            stat.un.b.lsRjtRsnCodeExp = LSEXP_SPARM_OPTIONS;
            stat.un.b.vendorUnique = 0;
            fc_els_rsp(binfo, ELS_CMD_LS_RJT, (uint32)iocb->ulpContext,
               (uint32)iocb->ulpClass, (void *)0, (uint32)stat.un.lsRjtError,
               0);
         }
         break;
      }
      FCSTATCTR.elsRcvPLOGI++;

      if (!(binfo->fc_flag & FC_PT2PT) && (binfo->fc_ffstate <= FC_FLOGI)) {
         /* Reject this PLOGI because we are in rediscovery */
         stat.un.b.lsRjtRsvd0 = 0;
         stat.un.b.lsRjtRsnCode = LSRJT_UNABLE_TPC;
         stat.un.b.lsRjtRsnCodeExp = LSEXP_NOTHING_MORE;
         stat.un.b.vendorUnique = 0;
         fc_els_rsp(binfo, ELS_CMD_LS_RJT, (uint32)iocb->ulpContext,
            (uint32)iocb->ulpClass, (void *)0, (uint32)stat.un.lsRjtError, 0);
         break;
      }

      if(did == NameServer_DID)
         break;

      if((did & Fabric_DID_MASK) != Fabric_DID_MASK) {
         /* Check to see if an existing cached entry is bad */
         ndlp = fc_findnode_wwpn(binfo, NLP_SEARCH_ALL, (NAME_TYPE *)&sp->portName);
         if (ndlp && ndlp->nlp_DID && (ndlp->nlp_DID != did)) {
            /* Check for a FARP generated nlplist entry */
            if (ndlp->nlp_DID == Bcast_DID)
               ndlp->nlp_DID = did;
            else {
               fc_freenode(binfo, ndlp, 0);
               ndlp->nlp_state = NLP_LIMBO;
               fc_nlp_bind(binfo, ndlp);
            }
         }
      }

      if ((ndlp = fc_findnode_odid(binfo, NLP_SEARCH_ALL, did)) == 0) {
         /* This is a new node so allocate an nlplist entry and accept
          * the LOGI request.
          */
         if((ndlp = (NODELIST *)fc_mem_get(binfo, MEM_NLP))) {
            fc_bzero((void *)ndlp, sizeof(NODELIST));
            ndlp->sync = binfo->fc_sync;
            ndlp->capabilities = binfo->fc_capabilities;
            ndlp->nlp_DID = did;
            ndlp->nlp_state = NLP_LIMBO;
            fc_nlp_bind(binfo, ndlp);
            fc_nlp_logi(binfo, ndlp,
               (NAME_TYPE *)&sp->portName, (NAME_TYPE *)&sp->nodeName);
         }
         else
            break;
      }
      /* Received PLOGI command */
      fc_log_printf_msg_vargs( binfo->fc_brd_no,
             &fc_msgBlk0125,                   /* ptr to msg structure */
              fc_mes0125,                      /* ptr to msg */
               fc_msgBlk0125.msgPreambleStr,   /* begin varargs */
                ndlp->nlp_DID,
                 ndlp->nlp_state,
                  ndlp->nlp_flag,
                   ndlp->nlp_Rpi);             /* end varargs */
      /* If we are pt2pt and this is the first PLOGI rcv'ed */
      if ((binfo->fc_flag & FC_PT2PT) && (binfo->fc_myDID == 0)) {
         if(!(fc_chksparm(binfo, sp, CLASS3))) {
            /* Reject this request because invalid parameters */
            stat.un.b.lsRjtRsvd0 = 0;
            stat.un.b.lsRjtRsnCode = LSRJT_UNABLE_TPC;
            stat.un.b.lsRjtRsnCodeExp = LSEXP_SPARM_OPTIONS;
            stat.un.b.vendorUnique = 0;
            fc_els_rsp(binfo, ELS_CMD_LS_RJT, (uint32)iocb->ulpContext,
                (uint32)iocb->ulpClass, (void *)0, (uint32)stat.un.lsRjtError,
                ndlp);
            break;
         }
         wd30.word = 0;
         wd30.f.xri   = iocb->ulpContext;
         wd30.f.class = iocb->ulpClass;

         fc_freenode_did(binfo, 0, 1);
         binfo->fc_myDID = iocb->un.ulpWord[4] & Mask_DID;

         if ((mb = (MAILBOXQ * )fc_mem_get(binfo, MEM_MBOX | MEM_PRI))) {
            fc_config_link(p_dev_ctl, (MAILBOX * )mb);
            if (issue_mb_cmd(binfo, (MAILBOX * )mb, MBX_NOWAIT)
               != MBX_BUSY) {
               fc_mem_put(binfo, MEM_MBOX, (uchar * )mb);
            }
         }
         if ((ndlp = fc_findnode_odid(binfo, NLP_SEARCH_ALL, binfo->fc_myDID)) == 0) {
            if((ndlp = (NODELIST *)fc_mem_get(binfo, MEM_NLP))) {
               fc_bzero((void *)ndlp, sizeof(NODELIST));
               ndlp->sync = binfo->fc_sync;
               ndlp->capabilities = binfo->fc_capabilities;
               ndlp->nlp_DID = binfo->fc_myDID;
               ndlp->nlp_state = NLP_LIMBO;
               fc_nlp_bind(binfo, ndlp);
            }
         }
         if(ndlp) {
            ndlp->nlp_DID = binfo->fc_myDID;
            fc_nlp_logi(binfo, ndlp, &binfo->fc_portname, &binfo->fc_nodename);
            if ((mb=(MAILBOXQ * )fc_mem_get(binfo, MEM_MBOX | MEM_PRI))) {
               fc_reg_login(binfo, binfo->fc_myDID,
                  (uchar * ) & binfo->fc_sparam, (MAILBOX * )mb, 0);
               if (issue_mb_cmd(binfo, (MAILBOX * )mb, MBX_NOWAIT)
                  != MBX_BUSY) {
                  fc_mem_put(binfo, MEM_MBOX, (uchar * )mb);
               }
            }
         }
         /* Device Discovery completes */
         fc_log_printf_msg_vargs( binfo->fc_brd_no,
                &fc_msgBlk0212,                   /* ptr to msg structure */
                 fc_mes0212,                      /* ptr to msg */
                  fc_msgBlk0212.msgPreambleStr);  /* begin & end varargs */
         binfo->fc_ffstate = FC_READY;

         binfo->fc_firstopen = 0;
         if(FABRICTMO) {
            fc_clk_can(p_dev_ctl, FABRICTMO);
            FABRICTMO = 0;
         }

         /* issue mailbox command to register login with the adapter */
         if ((mb = (MAILBOXQ * )fc_mem_get(binfo, MEM_MBOX | MEM_PRI))) {
            fc_reg_login(binfo,did,(uchar * )sp, (MAILBOX * )mb, wd30.word);
            if (issue_mb_cmd(binfo,(MAILBOX * )mb,MBX_NOWAIT) != MBX_BUSY) {
               fc_mem_put(binfo, MEM_MBOX, (uchar * )mb);
            }
         }
         break;
      }

      cnt = 1;
      switch(ndlp->nlp_state) {
      case NLP_PLOGI:
         cnt = 0;
         break;

      case NLP_LIMBO:
         if (ndlp->nlp_flag & NLP_REQ_SND) {
            cnt = 0;
            break;
         }

      case NLP_LOGOUT:
         fc_nlp_logi(binfo, ndlp,
            (NAME_TYPE *)&sp->portName, (NAME_TYPE *)&sp->nodeName);

      case NLP_LOGIN:
      case NLP_PRLI:
      case NLP_ALLOC:
         ndlp->nlp_flag &= ~NLP_FARP_SND;
         ndlp->nlp_action &= ~NLP_DO_ADDR_AUTH;
         /* Keep the rpi we have and send ACC / LS_RJT */
         if (fc_chksparm(binfo, sp, CLASS3)) {

            if (ndlp->nlp_Rpi) {
               fc_els_rsp(binfo, ELS_CMD_ACC, (uint32)iocb->ulpContext,
                  (uint32)iocb->ulpClass, (void *)0, (uint32)(sizeof(SERV_PARM)), ndlp);
               break;
            }
            /*  no rpi so we must reglogin */
            ndlp->nlp_flag |= NLP_RCV_PLOGI;
            wd30.word = 0;
            wd30.f.xri   = iocb->ulpContext;
            wd30.f.class = iocb->ulpClass;
            /* issue mailbox command to register login with the adapter */
            if ((mb = (MAILBOXQ * )fc_mem_get(binfo, MEM_MBOX | MEM_PRI))) {
               fc_reg_login(binfo,did,(uchar * )sp, (MAILBOX * )mb, wd30.word);
               if (issue_mb_cmd(binfo,(MAILBOX * )mb,MBX_NOWAIT) != MBX_BUSY) {
                  fc_mem_put(binfo, MEM_MBOX, (uchar * )mb);
               }
            }
         } else {
            /* Reject this request because invalid parameters */
            stat.un.b.lsRjtRsvd0 = 0;
            stat.un.b.lsRjtRsnCode = LSRJT_UNABLE_TPC;
            stat.un.b.lsRjtRsnCodeExp = LSEXP_SPARM_OPTIONS;
            stat.un.b.vendorUnique = 0;
            fc_els_rsp(binfo, ELS_CMD_LS_RJT, (uint32)iocb->ulpContext,
                (uint32)iocb->ulpClass, (void *)0, (uint32)stat.un.lsRjtError,
                ndlp);

            if ((ndlp->nlp_state == NLP_ALLOC) && (binfo->fc_ffstate == FC_READY)) {
               /* unregister existing login first */
               ndlp->nlp_flag |= NLP_UNREG_LOGO;
               fc_freenode(binfo, ndlp, 0);
               ndlp->nlp_state = NLP_LIMBO;
               fc_nlp_bind(binfo, ndlp);
            }
         }
         break;
      }
 
      if(cnt)
         break;


      did = iocb->un.elsreq.remoteID;

      /* If a nlplist entry already exists, we potentially have
       * a PLOGI collision.
       */

      if (!(ndlp->nlp_flag & NLP_REQ_SND)) {
         /* In this case we are already logged in */
         ndlp->nlp_flag &= ~NLP_FARP_SND;
         ndlp->nlp_action &= ~NLP_DO_ADDR_AUTH;
         goto chkparm;
      }

      FCSTATCTR.elsLogiCol++;

      /* For a PLOGI, we only accept if our portname is less
       * than the remote portname. 
       */
      if (!(fc_geportname((NAME_TYPE * ) & binfo->fc_portname,
          (NAME_TYPE * ) & sp->portName))) {
chkparm:
         fc_nlp_logi(binfo, ndlp,
            (NAME_TYPE *)&sp->portName, (NAME_TYPE *)&sp->nodeName);
         if (fc_chksparm(binfo, sp, CLASS3)) {
            /* PLOGI chkparm OK */
            fc_log_printf_msg_vargs( binfo->fc_brd_no,
                   &fc_msgBlk0126,                   /* ptr to msg structure */
                    fc_mes0126,                      /* ptr to msg */
                     fc_msgBlk0126.msgPreambleStr,   /* begin varargs */
                      ndlp->nlp_DID,
                       ndlp->nlp_state,
                        ndlp->nlp_flag,
                         ndlp->nlp_Rpi );            /* end varargs */
            if (ndlp->nlp_Rpi == 0) {
               if (ndlp->nlp_flag & NLP_REQ_SND) {
                  /* Abort the current outstanding PLOGI */
                  unsigned long iflag;

                  iflag = lpfc_q_disable_lock(p_dev_ctl);
                  iocbq = (IOCBQ * )(rp->fc_txp.q_first);
                  while (iocbq) {
                     if(iocbq->iocb.un.elsreq.remoteID == ndlp->nlp_DID) {
                        iocbq->retry = 0xff;
                     }
                     iocbq = (IOCBQ * )iocbq->q;
                  }
                  lpfc_q_unlock_enable(p_dev_ctl, iflag);
                  /* In case its on fc_delay_timeout list */
                  fc_abort_delay_els_cmd(p_dev_ctl, ndlp->nlp_DID);

                  ndlp->nlp_flag &= ~NLP_REQ_SND;
                  /* The following reg_login acts as if original PLOGI cmpl */
               }
               else
                  ndlp->nlp_flag |= NLP_RCV_PLOGI;

               wd30.word = 0;
               wd30.f.xri   = iocb->ulpContext;
               wd30.f.class = iocb->ulpClass;
               ndlp->nlp_flag |= NLP_REG_INP;

               /* issue mailbox command to register login with the adapter */
               if ((mb = (MAILBOXQ * )fc_mem_get(binfo, MEM_MBOX | MEM_PRI))) {
                  fc_reg_login(binfo, did, (uchar * )sp, (MAILBOX * )mb,
                     wd30.word);
                  if (issue_mb_cmd(binfo, (MAILBOX * )mb, MBX_NOWAIT)
                     != MBX_BUSY) {
                     fc_mem_put(binfo, MEM_MBOX, (uchar * )mb);
                  }
               }
            } else {
               fc_els_rsp(binfo, ELS_CMD_ACC, (uint32)iocb->ulpContext,
                  (uint32)iocb->ulpClass, (void *)0, (uint32)(sizeof(SERV_PARM)), ndlp);
            }
         } else {
            /* Reject this request because invalid parameters */
            stat.un.b.lsRjtRsvd0 = 0;
            stat.un.b.lsRjtRsnCode = LSRJT_UNABLE_TPC;
            stat.un.b.lsRjtRsnCodeExp = LSEXP_SPARM_OPTIONS;
            stat.un.b.vendorUnique = 0;
            fc_els_rsp(binfo, ELS_CMD_LS_RJT, (uint32)iocb->ulpContext,
               (uint32)iocb->ulpClass, (void *)0, (uint32)stat.un.lsRjtError,
               ndlp);

            if (binfo->fc_ffstate == FC_READY) {
               /* unregister existing login first */
               ndlp->nlp_flag |= NLP_UNREG_LOGO;
               fc_freenode(binfo, ndlp, 0);
               ndlp->nlp_state = NLP_LIMBO;
               fc_nlp_bind(binfo, ndlp);
            }
         }
      } else {
         /* Reject this request because the remote node will accept ours */
         stat.un.b.lsRjtRsvd0 = 0;
         stat.un.b.lsRjtRsnCode = LSRJT_UNABLE_TPC;
         stat.un.b.lsRjtRsnCodeExp = LSEXP_CMD_IN_PROGRESS;
         stat.un.b.vendorUnique = 0;
         fc_els_rsp(binfo, ELS_CMD_LS_RJT, (uint32)iocb->ulpContext,
            (uint32)iocb->ulpClass, (void *)0, (uint32)stat.un.lsRjtError, ndlp);
      }
      break;

   case ELS_CMD_LOGO:
      FCSTATCTR.elsRcvLOGO++;
      goto skip1;
   case ELS_CMD_PRLO:
      FCSTATCTR.elsRcvPRLO++;
skip1:
      lp++;  /* lp now points to portname */
      np = (NAME_TYPE * )lp;
      did = iocb->un.elsreq.remoteID;

      if ((ndlp = fc_findnode_odid(binfo, NLP_SEARCH_ALL, did)) == 0) {
         if(((ndlp = fc_findnode_wwpn(binfo, NLP_SEARCH_ALL, np)) == 0) ||
             (ndlp->nlp_DID == 0))
            /* ACCEPT the logout request */
            fc_els_rsp(binfo, ELS_CMD_ACC, (uint32)iocb->ulpContext,
               (uint32)iocb->ulpClass, (void *)0, (uint32)(sizeof(uint32)), 0);
            break;
      }


      if (ndlp) {
         if((ndlp->nlp_state >= NLP_LOGIN) ||
          ((!(ndlp->nlp_flag & (NLP_FARP_SND | NLP_RM_ENTRY))) &&
            (!(ndlp->nlp_action & (NLP_DO_ADDR_AUTH | NLP_DO_DISC_START | NLP_DO_RSCN))))) {
            /* ACCEPT the logout request */
            unsigned long iflag;

            fc_els_rsp(binfo, ELS_CMD_ACC, (uint32)iocb->ulpContext,
               (uint32)iocb->ulpClass, (void *)0, (uint32)(sizeof(uint32)), ndlp);
            ndlp->nlp_flag &= ~NLP_REQ_SND;
            ndlp->nlp_flag |= NLP_RM_ENTRY;

            iflag = lpfc_q_disable_lock(p_dev_ctl);
            iocbq = (IOCBQ * )(rp->fc_txp.q_first);
            while (iocbq) {
               if(iocbq->iocb.un.elsreq.remoteID == ndlp->nlp_DID) {
                  if(ndlp->nlp_action & (NLP_DO_ADDR_AUTH | NLP_DO_DISC_START | NLP_DO_RSCN)) {
                     ndlp->nlp_flag &= ~(NLP_REQ_SND_ADISC | NLP_REQ_SND_PDISC | NLP_REQ_SND);
                  }
                  iocbq->retry = 0xff;
                  if((binfo->fc_flag & FC_RSCN_MODE) ||
                     (binfo->fc_ffstate < FC_READY)) {
                     if((ndlp->nlp_state >= NLP_PLOGI) &&
                        (ndlp->nlp_state <= NLP_ALLOC)) {
                        binfo->fc_nlp_cnt--;
                     }
                     if (binfo->fc_nlp_cnt <= 0) {
                         binfo->fc_nlp_cnt = 0;
                     }
                  }
               }
               iocbq = (IOCBQ * )iocbq->q;
            }
            lpfc_q_unlock_enable(p_dev_ctl, iflag);
            /* In case its on fc_delay_timeout list */
            fc_abort_delay_els_cmd(p_dev_ctl, ndlp->nlp_DID);

            if ((ndlp->nlp_type & NLP_FCP_TARGET) &&
                (ndlp->nlp_state >= NLP_LOGIN)) {
               ndlp->nlp_flag |= NLP_SND_PLOGI;
            }
            if (ndlp->nlp_action & (NLP_DO_ADDR_AUTH | NLP_DO_DISC_START | NLP_DO_RSCN)) {
            fc_nextnode(p_dev_ctl, ndlp);
            }
         }
         else {
            /* ACCEPT the logout request */
            fc_els_rsp(binfo, ELS_CMD_ACC, (uint32)iocb->ulpContext,
               (uint32)iocb->ulpClass, (void *)0, (uint32)(sizeof(uint32)), 0);
         }
      }
      else {
         /* ACCEPT the logout request */
         fc_els_rsp(binfo, ELS_CMD_ACC, (uint32)iocb->ulpContext,
            (uint32)iocb->ulpClass, (void *)0, (uint32)(sizeof(uint32)), 0);
      }
      break;

   case ELS_CMD_FAN:
      FCSTATCTR.elsRcvFAN++;
      /* FAN received */
      fc_log_printf_msg_vargs( binfo->fc_brd_no,
             &fc_msgBlk0213,                   /* ptr to msg structure */
              fc_mes0213,                      /* ptr to msg */
               fc_msgBlk0213.msgPreambleStr,   /* begin varargs */
                iocb->un.ulpWord[4],
                 binfo->fc_ffstate);           /* end varargs */
      /* Check to see if we were waiting for FAN */
      if ((binfo->fc_ffstate != FC_FLOGI) || 
          (binfo->fc_topology != TOPOLOGY_LOOP) || 
          (!(binfo->fc_flag & FC_PUBLIC_LOOP)))
         break;

      ep = (ELS_PKT * )bp;

      /* Check to make sure we haven't switched fabrics */
      if ((fc_geportname((NAME_TYPE * ) & ep->un.fan.FportName,
          (NAME_TYPE * ) & binfo->fc_fabparam.portName) != 2) || 
          (fc_geportname((NAME_TYPE * ) & ep->un.fan.FnodeName,
          (NAME_TYPE * ) & binfo->fc_fabparam.nodeName) != 2)) {
         /* We switched, so we need to FLOGI again after timeout */
         break;
      }

      if(FABRICTMO) {
         fc_clk_can(p_dev_ctl, FABRICTMO);
         FABRICTMO = 0;
      }

      binfo->fc_myDID = iocb->un.ulpWord[4] & Mask_DID;

      if ((ndlp = fc_findnode_odid(binfo, NLP_SEARCH_ALL, Fabric_DID)) == 0) {
         if ((ndlp = fc_findnode_odid(binfo, NLP_SEARCH_ALL, (uint32)iocb->un.elsreq.remoteID)) == 0) {
            break;
         }
         fc_nlp_logi(binfo, ndlp, &ep->un.fan.FportName, &ep->un.fan.FnodeName);
      }
      ndlp->nlp_type |= NLP_FABRIC;

      if ((mb = (MAILBOXQ * )fc_mem_get(binfo, MEM_MBOX | MEM_PRI))) {
         fc_config_link(p_dev_ctl, (MAILBOX * )mb);
         if (issue_mb_cmd(binfo, (MAILBOX * )mb, MBX_NOWAIT) != MBX_BUSY) {
            fc_mem_put(binfo, MEM_MBOX, (uchar * )mb);
         }
      }

      /* register the login with adapter */
      if (ndlp->nlp_Rpi == 0) {
         if ((mb = (MAILBOXQ * )fc_mem_get(binfo, MEM_MBOX | MEM_PRI))) {
            fc_reg_login(binfo, iocb->un.elsreq.remoteID, 
               (uchar * ) & binfo->fc_fabparam, (MAILBOX * )mb, 0);
            if (issue_mb_cmd(binfo, (MAILBOX * )mb, MBX_NOWAIT) != MBX_BUSY) {
               fc_mem_put(binfo, MEM_MBOX, (uchar * )mb);
            }
         }
      }

      /* Since this is a FAN, we don't need to do any discovery stuff */
      fc_fanovery(p_dev_ctl);
      break;

   case ELS_CMD_RSCN:
      FCSTATCTR.elsRcvRSCN++;
      fc_process_rscn(p_dev_ctl, temp, mp);
      break;

   case ELS_CMD_ADISC:
      FCSTATCTR.elsRcvADISC++;
      ep = (ELS_PKT * )bp;
      did = iocb->un.elsreq.remoteID;
      if (((ndlp = fc_findnode_odid(binfo, NLP_SEARCH_ALL, did)) != 0) &&
         (ndlp->nlp_state >= NLP_LOGIN)) {
         if (fc_chkpadisc(binfo, ndlp, &ep->un.adisc.nodeName,
            &ep->un.adisc.portName)) {
            fc_els_rsp(binfo, ELS_CMD_ADISC, (uint32)iocb->ulpContext,
               (uint32)iocb->ulpClass, (void *)0, (uint32)sizeof(SERV_PARM),
               ndlp);
         } else {
            /* Reject this request because invalid parameters */
            stat.un.b.lsRjtRsvd0 = 0;
            stat.un.b.lsRjtRsnCode = LSRJT_UNABLE_TPC;
            stat.un.b.lsRjtRsnCodeExp = LSEXP_SPARM_OPTIONS;
            stat.un.b.vendorUnique = 0;
            fc_els_rsp(binfo, ELS_CMD_LS_RJT, (uint32)iocb->ulpContext,
               (uint32)iocb->ulpClass, (void *)0, (uint32)stat.un.lsRjtError,
               ndlp);
            if (!(ndlp->nlp_flag & NLP_REQ_SND)) {
               ndlp->nlp_flag |= NLP_UNREG_LOGO;
               fc_freenode_did(binfo, did, 0);
            }
         }
      } else {
         /* Reject this request because not logged in */
         stat.un.b.lsRjtRsvd0 = 0;
         stat.un.b.lsRjtRsnCode = LSRJT_UNABLE_TPC;
         stat.un.b.lsRjtRsnCodeExp = LSEXP_CANT_GIVE_DATA;
         stat.un.b.vendorUnique = 0;
         fc_els_rsp(binfo, ELS_CMD_LS_RJT, (uint32)iocb->ulpContext,
            (uint32)iocb->ulpClass, (void *)0, (uint32)stat.un.lsRjtError, ndlp);
         if ((ndlp == 0) || (!(ndlp->nlp_flag & NLP_REQ_SND)))
            fc_freenode_did(binfo, did, 0);
      }
      break;

   case ELS_CMD_PDISC:
      FCSTATCTR.elsRcvPDISC++;
      sp = (volatile SERV_PARM * )lp;
      did = iocb->un.elsreq.remoteID;
      if (((ndlp = fc_findnode_odid(binfo, NLP_SEARCH_ALL, did)) != 0) &&
         (ndlp->nlp_state >= NLP_LOGIN)) {
         if (fc_chkpadisc(binfo, ndlp, &sp->nodeName, &sp->portName)) {
            fc_els_rsp(binfo, ELS_CMD_ACC, (uint32)iocb->ulpContext,
               (uint32)iocb->ulpClass, (void *)0, (uint32)sizeof(SERV_PARM),
               ndlp);
         } else {
            /* Reject this request because invalid parameters */
            stat.un.b.lsRjtRsvd0 = 0;
            stat.un.b.lsRjtRsnCode = LSRJT_UNABLE_TPC;
            stat.un.b.lsRjtRsnCodeExp = LSEXP_SPARM_OPTIONS;
            stat.un.b.vendorUnique = 0;
            fc_els_rsp(binfo, ELS_CMD_LS_RJT, (uint32)iocb->ulpContext,
               (uint32)iocb->ulpClass, (void *)0, (uint32)stat.un.lsRjtError,
               ndlp);
            if (!(ndlp->nlp_flag & NLP_REQ_SND)) {
               ndlp->nlp_flag |= NLP_UNREG_LOGO;
               fc_freenode_did(binfo, did, 0);
            }
         }
      } else {
         /* Reject this request because not logged in */
         stat.un.b.lsRjtRsvd0 = 0;
         stat.un.b.lsRjtRsnCode = LSRJT_UNABLE_TPC;
         stat.un.b.lsRjtRsnCodeExp = LSEXP_CANT_GIVE_DATA;
         stat.un.b.vendorUnique = 0;
         fc_els_rsp(binfo, ELS_CMD_LS_RJT, (uint32)iocb->ulpContext,
            (uint32)iocb->ulpClass, (void *)0, (uint32)stat.un.lsRjtError, ndlp);
         if ((ndlp == 0) || (!(ndlp->nlp_flag & NLP_REQ_SND)))
            fc_freenode_did(binfo, did, 0);
      }
      break;

   case ELS_CMD_FARPR:
      FCSTATCTR.elsRcvFARPR++;
      ep = (ELS_PKT * )bp;
      did = iocb->un.elsreq.remoteID;
      /* FARP-RSP received from DID <did> */
      fc_log_printf_msg_vargs( binfo->fc_brd_no,
             &fc_msgBlk0600,                   /* ptr to msg structure */
              fc_mes0600,                      /* ptr to msg */
               fc_msgBlk0600.msgPreambleStr,   /* begin varargs */
                did );                         /* end varargs */
      /* ACCEPT the Farp resp request */
      fc_els_rsp(binfo, ELS_CMD_ACC, (uint32)iocb->ulpContext,
          (uint32)iocb->ulpClass, (void *)0, (uint32)(sizeof(uint32)), 0);
      break;

   case ELS_CMD_FARP:
      FCSTATCTR.elsRcvFARP++;
      ep = (ELS_PKT * )bp;
      did = iocb->un.elsreq.remoteID;
      /* FARP-REQ received from DID <did> */
      fc_log_printf_msg_vargs( binfo->fc_brd_no,
             &fc_msgBlk0601,                   /* ptr to msg structure */
              fc_mes0601,                      /* ptr to msg */
               fc_msgBlk0601.msgPreambleStr,   /* begin varargs */
                did );                         /* end varargs */
      if ((ndlp = fc_findnode_odid(binfo, NLP_SEARCH_ALL, did)) == 0) {
         if((ndlp = (NODELIST *)fc_mem_get(binfo, MEM_NLP))) {
            fc_bzero((void *)ndlp, sizeof(NODELIST));
            ndlp->sync = binfo->fc_sync;
            ndlp->capabilities = binfo->fc_capabilities;
            ndlp->nlp_DID = did;
            fc_nlp_logi(binfo,ndlp, &ep->un.farp.OportName, &ep->un.farp.OnodeName);
            ndlp->nlp_state = NLP_LIMBO;
         }
         else
            break;
      }

      /* We will only support match on WWPN or WWNN */
      if (ep->un.farp.Mflags & ~(FARP_MATCH_NODE | FARP_MATCH_PORT))
         break;

      cnt = 0;
      /* If this FARP command is searching for my portname */
      if (ep->un.farp.Mflags & FARP_MATCH_PORT) { 
         if (fc_geportname(&ep->un.farp.RportName, &binfo->fc_portname) == 2)
            cnt = 1;
         else
            cnt = 0;
      }

      /* If this FARP command is searching for my nodename */
      if (ep->un.farp.Mflags & FARP_MATCH_NODE) {
         if (fc_geportname(&ep->un.farp.RnodeName, &binfo->fc_nodename) == 2)
            cnt = 1;
         else
            cnt = 0;
      }

      if (cnt) {
         if (!(binfo->fc_flag & FC_LNK_DOWN) && 
             (binfo->fc_ffstate >= rp->fc_xmitstate) && 
             !(ndlp->nlp_flag & NLP_REQ_SND) &&
             !(ndlp->nlp_action & NLP_DO_ADDR_AUTH)) {
            /* We need to re-login to that node */
            if ((ep->un.farp.Rflags & FARP_REQUEST_PLOGI) &&
               !(ndlp->nlp_flag & NLP_REQ_SND)) {
               fc_freenode(binfo, ndlp, 0);
               ndlp->nlp_state = NLP_LIMBO;
               fc_nlp_bind(binfo, ndlp);
               fc_els_cmd(binfo, ELS_CMD_PLOGI, (void *)((ulong)ndlp->nlp_DID),
                   (uint32)0, (ushort)0, ndlp);
            }

            /* We need to send FARP response to that node */
            if (ep->un.farp.Rflags & FARP_REQUEST_FARPR) {
               fc_els_cmd(binfo, ELS_CMD_FARPR, (void *)((ulong)ndlp->nlp_DID),
                   (uint32)0, (ushort)0, ndlp);
            }
         }
      }
      break;

   case ELS_CMD_RRQ:
      FCSTATCTR.elsRcvRRQ++;
      ep = (ELS_PKT * )bp;
      /* Get oxid / rxid from payload and internally abort it */
      if ((ep->un.rrq.SID == SWAP_DATA(binfo->fc_myDID))) {
         fc_abort_ixri_cx(binfo, ep->un.rrq.Oxid, CMD_CLOSE_XRI_CX, rp);
      } else {
         fc_abort_ixri_cx(binfo, ep->un.rrq.Rxid, CMD_CLOSE_XRI_CX, rp);
      }
      /* ACCEPT the rrq request */
      fc_els_rsp(binfo, ELS_CMD_ACC, (uint32)iocb->ulpContext,
         (uint32)iocb->ulpClass, (void *)0, (uint32)(sizeof(uint32)), 0);
      break;

   case ELS_CMD_PRLI:
      /* ACCEPT the prli request */
      did = iocb->un.elsreq.remoteID;
      if ((ndlp = fc_findnode_odid(binfo, NLP_SEARCH_ALL, did))) {
         fc_els_rsp(binfo, ELS_CMD_PRLI, (uint32)iocb->ulpContext,
            (uint32)iocb->ulpClass, (void *)0, (uint32)(sizeof(uint32)), ndlp);
      }
      else {
         stat.un.b.lsRjtRsvd0 = 0;
         stat.un.b.lsRjtRsnCode = LSRJT_UNABLE_TPC;
         stat.un.b.lsRjtRsnCodeExp = LSEXP_CANT_GIVE_DATA;
         stat.un.b.vendorUnique = 0;
         fc_els_rsp(binfo, ELS_CMD_LS_RJT, (uint32)iocb->ulpContext,
            (uint32)iocb->ulpClass, (void *)0, (uint32)stat.un.lsRjtError, 0);
      }
      break;

   case ELS_CMD_RNID:
      did = iocb->un.elsreq.remoteID;
      ep = (ELS_PKT * )bp;
      switch(ep->un.rnid.Format) {
      case 0:
      case RNID_TOPOLOGY_DISC:
         fc_els_rsp(binfo, ELS_CMD_RNID, (uint32)iocb->ulpContext,
            (uint32)iocb->ulpClass, (void *)0, (uint32)ep->un.rnid.Format, 0);
         break;
      default:
         /* Reject this request because format not supported */
         stat.un.b.lsRjtRsvd0 = 0;
         stat.un.b.lsRjtRsnCode = LSRJT_UNABLE_TPC;
         stat.un.b.lsRjtRsnCodeExp = LSEXP_CANT_GIVE_DATA;
         stat.un.b.vendorUnique = 0;
         fc_els_rsp(binfo, ELS_CMD_LS_RJT, (uint32)iocb->ulpContext,
            (uint32)iocb->ulpClass, (void *)0, (uint32)stat.un.lsRjtError, 0);
      }
      break;

   default:
      /* Unsupported ELS command, reject */
      stat.un.b.lsRjtRsvd0 = 0;
      stat.un.b.lsRjtRsnCode = LSRJT_CMD_UNSUPPORTED;
      stat.un.b.lsRjtRsnCodeExp = LSEXP_NOTHING_MORE;
      stat.un.b.vendorUnique = 0;
      fc_els_rsp(binfo, ELS_CMD_LS_RJT, (uint32)iocb->ulpContext,
         (uint32)iocb->ulpClass, (void *)0, (uint32)stat.un.lsRjtError, 0);
      FCSTATCTR.elsCmdPktInval++;

      did = iocb->un.elsreq.remoteID;
      /* Unknown ELS command <elsCmd> received from NPORT <did> */
      fc_log_printf_msg_vargs( binfo->fc_brd_no,
             &fc_msgBlk0127,                   /* ptr to msg structure */
              fc_mes0127,                      /* ptr to msg */
               fc_msgBlk0127.msgPreambleStr,   /* begin varargs */
                cmd,
                 did);                         /* end varargs */
      break;
   }

dropit:

   FCSTATCTR.elsRcvFrame++;
   fc_mem_put(binfo, MEM_BUF, (uchar * )mp);

out:

   i = 1;
   /* free resources associated with this iocb and repost the ring buffers */
   if (!(binfo->fc_flag & FC_SLI2)) {
      for (i = 1; i < (int)iocb->ulpBdeCount; i++) {
         mp = fc_getvaddr(p_dev_ctl, rp, (uchar * )((ulong)iocb->un.cont[i].bdeAddress));
         if (mp) {
            fc_mem_put(binfo, MEM_BUF, (uchar * )mp);
         }
      }
   }

   fc_post_buffer(p_dev_ctl, rp, i);

   return(1);
}       /* End handle_rcv_els_req */


/***************************************/
/**  fc_process_rscn  Process ELS     **/
/**                   RSCN command    **/
/***************************************/
_static_ int
fc_process_rscn(
fc_dev_ctl_t *p_dev_ctl,
IOCBQ *temp,
MATCHMAP *mp)
{
   FC_BRD_INFO  * binfo;
   IOCB         * iocb;
   uchar        * bp;
   uint32       * lp;
   D_ID         rdid;
   uint32       cmd;
   int          i, j, cnt;

   binfo = &BINFO;
   iocb = &temp->iocb;
   bp = mp->virt;
   lp = (uint32 * )bp;
   cmd = *lp++;
   i = SWAP_DATA(cmd) & 0xffff;   /* payload length */
   i -= sizeof(uint32);  /* take off word 0 */
   cmd &= ELS_CMD_MASK;

   /* RSCN received */
   fc_log_printf_msg_vargs( binfo->fc_brd_no,
          &fc_msgBlk0214,                   /* ptr to msg structure */
           fc_mes0214,                      /* ptr to msg */
            fc_msgBlk0214.msgPreambleStr,   /* begin varargs */
             binfo->fc_flag,
              i,
               *lp,
                binfo->fc_rscn_id_cnt);     /* end varargs */
   cnt = 0; /* cnt will determine if we need to access NameServer */

   /* Loop through all DIDs in the payload */
   binfo->fc_flag |= FC_RSCN_MODE;

   while (i) {
      rdid.un.word = *lp++;
      rdid.un.word = SWAP_DATA(rdid.un.word);
      if(binfo->fc_rscn_id_cnt < FC_MAX_HOLD_RSCN) {
         for(j=0;j<binfo->fc_rscn_id_cnt;j++) {
            if(binfo->fc_rscn_id_list[j] == rdid.un.word) {
               goto skip_id;
            }
         }
         binfo->fc_rscn_id_list[binfo->fc_rscn_id_cnt++] = rdid.un.word;
      }
      else {
         binfo->fc_flag |= FC_RSCN_DISCOVERY;
         fc_flush_rscn_defer(p_dev_ctl);
         cnt = 0;
         break;
      }
skip_id:
      cnt += (fc_handle_rscn(p_dev_ctl, &rdid));
      i -= sizeof(uint32);
   }

   /* RSCN processed */
   fc_log_printf_msg_vargs( binfo->fc_brd_no,
          &fc_msgBlk0215,                   /* ptr to msg structure */
           fc_mes0215,                      /* ptr to msg */
            fc_msgBlk0215.msgPreambleStr,   /* begin varargs */
             binfo->fc_flag,
              cnt,
               binfo->fc_rscn_id_cnt,
                binfo->fc_ffstate );        /* end varargs */
   if (cnt == 0) {
      /* no need for nameserver login */
      fc_nextrscn(p_dev_ctl, fc_max_els_sent);
   }
   else {
      if(!(binfo->fc_flag & FC_NSLOGI_TMR))
         fc_clk_set(p_dev_ctl, 1, fc_issue_ns_query, 0, 0);
      binfo->fc_flag |= FC_NSLOGI_TMR;
   }
   return(0);
}


/***************************************/
/**  fc_handle_rscn   Handle ELS      **/
/**                   RSCN command    **/
/***************************************/
_static_ int
fc_handle_rscn(
fc_dev_ctl_t *p_dev_ctl,
D_ID         *didp)
{
   FC_BRD_INFO * binfo;
   NODELIST    * ndlp;
   NODELIST    * new_ndlp;
   NODELIST    * callnextnode;
   iCfgParam   * clp;
   D_ID        did;
   int         change;
   int         numchange;
   int         ns;

   binfo = &BINFO;
   clp = DD_CTL.p_config[binfo->fc_brd_no];
   callnextnode = 0;

   dfc_hba_put_event(p_dev_ctl, HBA_EVENT_RSCN, binfo->fc_myDID, didp->un.word, 0, 0);
   dfc_put_event(p_dev_ctl, FC_REG_RSCN_EVENT, didp->un.word, 0, 0);

   /* Is this an RSCN for me? */
   if (didp->un.word == binfo->fc_myDID)
      return(0);

   /* Always query nameserver on RSCN (zoning) if CFG_ZONE_RSCN it set */
   ns = (int)clp[CFG_ZONE_RSCN].a_current;
   numchange = 0;

   ndlp = binfo->fc_nlpbind_start;
   if(ndlp == (NODELIST *)&binfo->fc_nlpbind_start)
     ndlp = binfo->fc_nlpunmap_start;
   if(ndlp == (NODELIST *)&binfo->fc_nlpunmap_start)
      ndlp = binfo->fc_nlpmap_start;
   new_ndlp = 0;
   while(ndlp != (NODELIST *)&binfo->fc_nlpmap_start) {
      new_ndlp = (NODELIST *)ndlp->nlp_listp_next;

      /* Skip over FABRIC nodes and myself */
      if ((ndlp->nlp_DID == binfo->fc_myDID) || 
          (ndlp->nlp_type & NLP_FABRIC)) {

         ndlp = new_ndlp;
         if(ndlp == (NODELIST *)&binfo->fc_nlpunmap_start)
            ndlp = binfo->fc_nlpmap_start;
         continue;
      }

      did.un.word = ndlp->nlp_DID;
      change = 0;

      switch (didp->un.b.resv) {
      case 0:   /* Single N_Port ID effected */
         if (did.un.word == didp->un.word) {
            change = 1;
         }
         break;

      case 1:   /* Whole N_Port Area effected */
         if ((did.un.b.domain == didp->un.b.domain) && 
             (did.un.b.area   == didp->un.b.area)) {
            ns = 1;
            change = 1;
         }
         break;

      case 2:   /* Whole N_Port Domain effected */
         if (did.un.b.domain == didp->un.b.domain) {
            ns = 1;
            change = 1;
         }
         break;

      case 3:   /* Whole Fabric effected */
         binfo->fc_flag |= FC_RSCN_DISCOVERY;
         fc_flush_rscn_defer(p_dev_ctl);
         return(0);

      default:
         /* Unknown Identifier in RSCN payload */
         fc_log_printf_msg_vargs( binfo->fc_brd_no,
                &fc_msgBlk0216,                   /* ptr to msg structure */
                 fc_mes0216,                      /* ptr to msg */
                  fc_msgBlk0216.msgPreambleStr,   /* begin varargs */
                   didp->un.word );               /* end varargs */
         break;

      }

      if (change) {
         numchange++;
         if((ndlp->nlp_state == NLP_ALLOC) ||
            (ndlp->nlp_state == NLP_LOGIN)) {

            if (ndlp->nlp_flag & NLP_REQ_SND) {
               RING       * rp;
               IOCBQ      * iocbq;
               unsigned long iflag;

               /* Look through ELS ring and remove any ELS cmds in progress */
               iflag = lpfc_q_disable_lock(p_dev_ctl);
               rp = &binfo->fc_ring[FC_ELS_RING];
               iocbq = (IOCBQ * )(rp->fc_txp.q_first);
               while (iocbq) {
                  if (iocbq->iocb.un.elsreq.remoteID == ndlp->nlp_DID) {
                     iocbq->retry = 0xff;  /* Mark for abort */
                  }
                  iocbq = (IOCBQ * )iocbq->q;
               }
               lpfc_q_unlock_enable(p_dev_ctl, iflag);
               /* In case its on fc_delay_timeout list */
               fc_abort_delay_els_cmd(p_dev_ctl, ndlp->nlp_DID);

               ndlp->nlp_flag &= ~NLP_REQ_SND;
            }

            /* We are always using ADISC for RSCN validation */
            /* IF we are using ADISC, leave ndlp on mapped or unmapped q */

            ndlp->nlp_flag &= ~NLP_NODEV_TMO;

            /* Mark node for authentication */
            ndlp->nlp_action |= NLP_DO_RSCN;

         } else {

            if (ndlp->nlp_flag & NLP_REQ_SND) {
               if((callnextnode == 0) && (ndlp->nlp_action & NLP_DO_RSCN))
                  callnextnode = ndlp;
            }
            fc_freenode(binfo, ndlp, 0);
            ndlp->nlp_flag &= ~NLP_NODEV_TMO;
            ndlp->nlp_state = NLP_LIMBO;
            fc_nlp_bind(binfo, ndlp);

            /* Mark node for authentication */
            ndlp->nlp_action |= NLP_DO_RSCN;
         }
      }
      ndlp = new_ndlp;
      if(ndlp == (NODELIST *)&binfo->fc_nlpbind_start)
        ndlp = binfo->fc_nlpunmap_start;
      if(ndlp == (NODELIST *)&binfo->fc_nlpunmap_start)
         ndlp = binfo->fc_nlpmap_start;
   }

   /* If nothing in our node table is effected, 
    * we need to goto the Nameserver.
    */
   if (numchange == 0) {
      /* Is this a single N_Port  that wasn't in our table */
      if (didp->un.b.resv == 0) {
         if ((ndlp = fc_findnode_odid(binfo, NLP_SEARCH_ALL, didp->un.word)) == 0) {
            if((ndlp = (NODELIST *)fc_mem_get(binfo, MEM_NLP))) {
               fc_bzero((void *)ndlp, sizeof(NODELIST));
               ndlp->sync = binfo->fc_sync;
               ndlp->capabilities = binfo->fc_capabilities;
               ndlp->nlp_DID = didp->un.word;
            }
            else
               return(ns);
         }
         ndlp->nlp_action |= NLP_DO_RSCN;
         ndlp->nlp_flag &= ~NLP_NODEV_TMO;
         ndlp->nlp_state = NLP_LIMBO;
         fc_nlp_bind(binfo, ndlp);
      }
      else {
         ns = 1;
      }
   }

   /* Is this an area or domain N_Port */
   if (didp->un.b.resv != 0) {
      ns = 1;
   }

   if((ns == 0) && (callnextnode))
      fc_nextnode(p_dev_ctl, callnextnode);

   /* Tell calling routine if NameServer access is required
    * and return number of nodes presently being authenticated.
    */
   return(ns);
}       /* End fc_handle_rscn */


/*************************************************/
/**  fc_chksparm    Check service parameters    **/
/*************************************************/
_local_ int
fc_chksparm(
FC_BRD_INFO          *binfo,
volatile SERV_PARM   *sp,
uint32               class)
{
   volatile SERV_PARM   *hsp;

   hsp = &binfo->fc_sparam;
   /* First check for supported version */

   /* Next check for class validity */
   if (sp->cls1.classValid) {
      if (sp->cls1.rcvDataSizeMsb > hsp->cls1.rcvDataSizeMsb)
         sp->cls1.rcvDataSizeMsb = hsp->cls1.rcvDataSizeMsb;
      if (sp->cls1.rcvDataSizeLsb > hsp->cls1.rcvDataSizeLsb)
         sp->cls1.rcvDataSizeLsb = hsp->cls1.rcvDataSizeLsb;
   } else if (class == CLASS1) {
      return(0);
   }

   if (sp->cls2.classValid) {
      if (sp->cls2.rcvDataSizeMsb > hsp->cls2.rcvDataSizeMsb)
         sp->cls2.rcvDataSizeMsb = hsp->cls2.rcvDataSizeMsb;
      if (sp->cls2.rcvDataSizeLsb > hsp->cls2.rcvDataSizeLsb)
         sp->cls2.rcvDataSizeLsb = hsp->cls2.rcvDataSizeLsb;
   } else if (class == CLASS2) {
      return(0);
   }

   if (sp->cls3.classValid) {
      if (sp->cls3.rcvDataSizeMsb > hsp->cls3.rcvDataSizeMsb)
         sp->cls3.rcvDataSizeMsb = hsp->cls3.rcvDataSizeMsb;
      if (sp->cls3.rcvDataSizeLsb > hsp->cls3.rcvDataSizeLsb)
         sp->cls3.rcvDataSizeLsb = hsp->cls3.rcvDataSizeLsb;
   } else if (class == CLASS3) {
      return(0);
   }

   if (sp->cmn.bbRcvSizeMsb > hsp->cmn.bbRcvSizeMsb)
      sp->cmn.bbRcvSizeMsb = hsp->cmn.bbRcvSizeMsb;
   if (sp->cmn.bbRcvSizeLsb > hsp->cmn.bbRcvSizeLsb)
      sp->cmn.bbRcvSizeLsb = hsp->cmn.bbRcvSizeLsb;

   return(1);
}       /* End fc_chksparm */


/***************************************/
/**  fc_chkpadisc Check               **/
/**               P/ADISC parameters  **/
/***************************************/
_static_ int
fc_chkpadisc(
FC_BRD_INFO *binfo,
NODELIST    *ndlp,
volatile NAME_TYPE   *nn,
volatile NAME_TYPE   *pn)
{
   if (fc_geportname((NAME_TYPE * )nn, &ndlp->nlp_nodename) != 2) {
      return(0);
   }

   if (fc_geportname((NAME_TYPE * )pn, &ndlp->nlp_portname) != 2) {
      return(0);
   }

   return(1);
}       /* End fc_chkpadisc */


/***************************************/
/**  fc_els_cmd    Issue an           **/
/**                ELS command        **/
/***************************************/
_static_ int
fc_els_cmd(
FC_BRD_INFO *binfo,
uint32      elscmd,
void        *arg,
uint32      retry,
ushort      iotag,
NODELIST    *ndlp)
{
   IOCB          * icmd;
   IOCBQ         * temp;
   RING          * rp;
   uchar         * bp;
   ULP_BDE64     * bpl;
   MATCHMAP      * mp, * rmp, * bmp;
   MAILBOXQ      * mb;
   iCfgParam     * clp;
   union {
      SERV_PARM  * sp;
      ADISC      * ap;
      FARP       * fp;
      fc_vpd_t   * vpd;
      PRLI       * npr;
   } un;
   uint32        * lp;
   ushort        size;
   ulong         setdelay;
   fc_dev_ctl_t  * p_dev_ctl;

   clp = DD_CTL.p_config[binfo->fc_brd_no];
   rp = &binfo->fc_ring[FC_ELS_RING];
   p_dev_ctl = (fc_dev_ctl_t *)(binfo->fc_p_dev_ctl);

   if ((elscmd == ELS_CMD_LOGO) && (iotag == 0)) {
      /* First do unreglogin for did before sending ELS LOGO request */
      if ((ndlp = fc_findnode_odid(binfo, NLP_SEARCH_ALL, (uint32)((ulong)arg))) && ndlp->nlp_Rpi) {
         /* If we are in the middle of Discovery */
         if (ndlp->nlp_action & (NLP_DO_ADDR_AUTH | NLP_DO_DISC_START | NLP_DO_RSCN)) {
            /* Goto next entry */
            fc_nextnode(p_dev_ctl, ndlp);
         }
         ndlp->nlp_flag |= NLP_UNREG_LOGO;
         fc_freenode(binfo, ndlp, 0);
         ndlp->nlp_state = NLP_LIMBO;
         fc_nlp_bind(binfo, ndlp);
         return(0);
      }
   } 
   /* Allocate buffer for  command iocb */
   if ((temp = (IOCBQ * )fc_mem_get(binfo, MEM_IOCB | MEM_PRI)) == 0) {
      return(1);
   }
   fc_bzero((void *)temp, sizeof(IOCBQ));
   icmd = &temp->iocb;
   setdelay = 0;

   /* fill in BDEs for command */
   /* Allocate buffer for command payload */
   if ((mp = (MATCHMAP * )fc_mem_get(binfo, MEM_BUF | MEM_PRI)) == 0) {
      fc_mem_put(binfo, MEM_IOCB, (uchar * )temp);
      return(1);
   }

   /* Allocate buffer for response payload */
   if ((rmp = (MATCHMAP * )fc_mem_get(binfo, MEM_BUF | MEM_PRI)) == 0) {
      fc_mem_put(binfo, MEM_IOCB, (uchar * )temp);
      fc_mem_put(binfo, MEM_BUF, (uchar * )mp);
      return(1);
   }
   fc_bzero((void *)rmp->virt, sizeof(ELS_PKT));

   if (binfo->fc_flag & FC_SLI2) {
      /* Allocate buffer for Buffer ptr list */
      if ((bmp = (MATCHMAP * )fc_mem_get(binfo, MEM_BPL | MEM_PRI)) == 0) {
         fc_mem_put(binfo, MEM_IOCB, (uchar * )temp);
         fc_mem_put(binfo, MEM_BUF, (uchar * )mp);
         fc_mem_put(binfo, MEM_BUF, (uchar * )rmp);
         return(1);
      }
      bpl = (ULP_BDE64 * )bmp->virt;
      bpl->addrLow = PCIMEM_LONG(putPaddrLow((ulong)mp->phys));
      bpl->addrHigh = PCIMEM_LONG(putPaddrHigh((ulong)mp->phys));
      bpl->tus.f.bdeFlags = 0;
      bpl++;
      bpl->addrLow = PCIMEM_LONG(putPaddrLow((ulong)rmp->phys));
      bpl->addrHigh = PCIMEM_LONG(putPaddrHigh((ulong)rmp->phys));
      bpl->tus.f.bdeSize = FCELSSIZE;
      bpl->tus.f.bdeFlags = BUFF_USE_RCV;
      bpl->tus.w = PCIMEM_LONG(bpl->tus.w);

      bpl--; /* so we can fill in size later */

      icmd->un.elsreq64.bdl.ulpIoTag32 = (uint32)0;
      icmd->un.elsreq64.bdl.addrHigh = (uint32)putPaddrHigh(bmp->phys);
      icmd->un.elsreq64.bdl.addrLow = (uint32)putPaddrLow(bmp->phys);
      icmd->un.elsreq64.bdl.bdeSize = (2 * sizeof(ULP_BDE64));
      icmd->un.elsreq64.bdl.bdeFlags = BUFF_TYPE_BDL;
      temp->bpl = (uchar *)bmp;
   } else {
      bpl = 0;
      bmp = 0;
      icmd->un.cont[0].bdeAddress = (uint32)putPaddrLow(mp->phys);
      icmd->un.cont[1].bdeAddress = (uint32)putPaddrLow(rmp->phys);
      icmd->un.cont[1].bdeSize = FCELSSIZE;
      temp->bpl = 0;
   }

   bp = mp->virt;
   /* Save for completion so we can release these resources */
   temp->bp = (uchar * )mp;
   temp->info = (uchar * )rmp;

   /* Fill in command field in payload */
   *((uint32 * )(bp)) = elscmd;           /* FLOGI, PLOGI or LOGO */
   bp += sizeof(uint32);

   switch (elscmd) {
   case ELS_CMD_PLOGI:  /* NPort login */
   case ELS_CMD_PDISC:  /* exchange parameters */
      if(ndlp && (ndlp->nlp_DID == 0)) {
         ndlp->nlp_DID = (uint32)((ulong)arg);
      }
   case ELS_CMD_FLOGI:  /* Fabric login */
      /* For LOGI request, remainder of payload is service parameters */
      fc_bcopy((void *) & binfo->fc_sparam, (void *)bp, sizeof(SERV_PARM));
      un.sp = (SERV_PARM * )bp;

      if (elscmd == ELS_CMD_FLOGI) {
         un.sp->cmn.e_d_tov = 0;
         un.sp->cmn.w2.r_a_tov = 0;
         un.sp->cls1.classValid = 0;
         un.sp->cls2.seqDelivery = 1;
         un.sp->cls3.seqDelivery = 1;
         if (un.sp->cmn.fcphLow < FC_PH3)
            un.sp->cmn.fcphLow = FC_PH3;
         if(FABRICTMO) {
            fc_clk_res(p_dev_ctl, binfo->fc_fabrictmo, FABRICTMO);
         }
      } else {
         /* Seagate drives can't handle FC_PH3 value! */
         if (un.sp->cmn.fcphLow < FC_PH_4_3)
            un.sp->cmn.fcphLow = FC_PH_4_3;
      }

      if (un.sp->cmn.fcphHigh < FC_PH3)
         un.sp->cmn.fcphHigh = FC_PH3;

      icmd->un.elsreq.remoteID = (uint32)((ulong)arg);   /* DID */
      size = (sizeof(uint32) + sizeof(SERV_PARM));

      if (elscmd != ELS_CMD_PDISC) {
         /* Allocate a nlplist entry, ELS cmpl will fill it in */
         if ((ndlp == 0) && 
            ((ndlp = fc_findnode_odid(binfo, NLP_SEARCH_ALL, (uint32)((ulong)arg))) == 0)) {
            if((ndlp = (NODELIST *)fc_mem_get(binfo, MEM_NLP))) {
               fc_bzero((void *)ndlp, sizeof(NODELIST));
               ndlp->sync = binfo->fc_sync;
               ndlp->capabilities = binfo->fc_capabilities;
               ndlp->nlp_DID = (uint32)((ulong)arg);
            }
            else {
               fc_mem_put(binfo, MEM_IOCB, (uchar * )temp);
               fc_mem_put(binfo, MEM_BUF, (uchar * )mp);
               fc_mem_put(binfo, MEM_BUF, (uchar * )rmp);
               if (bmp) {
                  fc_mem_put(binfo, MEM_BPL, (uchar * )bmp);
               }
               return(1);
            }
            ndlp->nlp_state = NLP_LIMBO;
            fc_nlp_bind(binfo, ndlp);
         }
         ndlp->nlp_flag &= ~NLP_RM_ENTRY;
         ndlp->nlp_flag |= NLP_REQ_SND;

         if (elscmd == ELS_CMD_PLOGI) {

            ndlp->nlp_flag &= ~NLP_SND_PLOGI;
            if (ndlp->nlp_Rpi) {
               /* must explicitly unregister the login, UREG_LOGIN */
               if ((mb = (MAILBOXQ * )fc_mem_get(binfo, MEM_MBOX | MEM_PRI))) {
                  fc_unreg_login(binfo, ndlp->nlp_Rpi, (MAILBOX * )mb);
                  if (issue_mb_cmd(binfo,(MAILBOX * )mb,MBX_NOWAIT) != MBX_BUSY) {
                     fc_mem_put(binfo, MEM_MBOX, (uchar * )mb);
                  }
               }
               binfo->fc_nlplookup[ndlp->nlp_Rpi] = 0;
               ndlp->nlp_Rpi = 0;
            }

            /* For PLOGI requests, must make sure all outstanding Mailbox
             * commands have been processed. This is to ensure UNREG_LOGINs
             * complete before we try to login.
             */
            if (binfo->fc_mbox_active) {
               fc_mem_put(binfo, MEM_BUF, (uchar * )mp);
               fc_mem_put(binfo, MEM_BUF, (uchar * )rmp);
               if (bmp) {
                  fc_mem_put(binfo, MEM_BPL, (uchar * )bmp);
               }
               temp->info = (uchar *)0;
               temp->bp = (uchar *)0;
               temp->bpl = (uchar *)0;
               fc_plogi_put(binfo, temp);
               return(1);
            }

            if ((ulong)arg == NameServer_DID) {
               if (binfo->fc_ffstate == FC_READY) {
                  if(binfo->fc_flag & FC_RSCN_MODE)
                     ndlp->nlp_action |= NLP_DO_RSCN;
                  else
                     ndlp->nlp_action |= NLP_DO_ADDR_AUTH;
               }
               else
                  ndlp->nlp_action |= NLP_DO_ADDR_AUTH;
            }
         }
      }
      break;

   case ELS_CMD_LOGO:   /* Logout */
      icmd->un.elsreq.remoteID = (uint32)((ulong)arg);   /* DID */

      *((uint32 * )(bp)) = SWAP_DATA(binfo->fc_myDID);
      bp += sizeof(uint32);

      /* Last field in payload is our portname */
      fc_bcopy((void *) & binfo->fc_portname, (void *)bp, sizeof(NAME_TYPE));
      size = sizeof(uint32) + sizeof(uint32) + sizeof(NAME_TYPE);
      break;

   case ELS_CMD_ADISC:
      icmd->un.elsreq.remoteID = (uint32)((ulong)arg);        /* DID */

      if ((ndlp == 0) && 
         ((ndlp = fc_findnode_odid(binfo, NLP_SEARCH_ALL, (uint32)((ulong)arg))) == 0)) {
         fc_mem_put(binfo, MEM_IOCB, (uchar * )temp);
         fc_mem_put(binfo, MEM_BUF, (uchar * )mp);
         fc_mem_put(binfo, MEM_BUF, (uchar * )rmp);
         if (bmp) {
            fc_mem_put(binfo, MEM_BPL, (uchar * )bmp);
         }
         return(1);
      }
      ndlp->nlp_DID = (uint32)((ulong)arg);
      ndlp->nlp_flag |= NLP_REQ_SND_ADISC;
      un.ap = (ADISC * )(bp);
      un.ap->hardAL_PA = binfo->fc_pref_ALPA;
      fc_bcopy((void *) & binfo->fc_portname, (void *) & un.ap->portName,
          sizeof(NAME_TYPE));
      fc_bcopy((void *) & binfo->fc_nodename, (void *) & un.ap->nodeName,
          sizeof(NAME_TYPE));
      un.ap->DID = SWAP_DATA(binfo->fc_myDID);

      size = sizeof(uint32) + sizeof(ADISC);
      break;

   case ELS_CMD_PRLI:   /* Process login */
      icmd->un.elsreq.remoteID = (uint32)((ulong)arg);   /* DID */
      if ((ndlp == 0) && 
         ((ndlp = fc_findnode_odid(binfo, NLP_SEARCH_ALL, (uint32)((ulong)arg))) == 0)) {
         fc_mem_put(binfo, MEM_IOCB, (uchar * )temp);
         fc_mem_put(binfo, MEM_BUF, (uchar * )mp);
         fc_mem_put(binfo, MEM_BUF, (uchar * )rmp);
         if (bmp) {
            fc_mem_put(binfo, MEM_BPL, (uchar * )bmp);
         }
         return(1);
      }
      ndlp->nlp_flag |= NLP_REQ_SND;

      /* For PRLI, remainder of payload is PRLI parameter page */
      fc_bzero((void *)bp, sizeof(PRLI));
      un.npr = (PRLI *)bp;

      /*
       * If our firmware version is 3.20 or later, 
       * set the following bits for FC-TAPE support.
       */
      if ( p_dev_ctl->vpd.rev.feaLevelHigh >= 0x02 ) {
         un.npr->ConfmComplAllowed = 1;
         un.npr->Retry = 1;
         un.npr->TaskRetryIdReq = 1;
      }

      un.npr->estabImagePair = 1;
      un.npr->readXferRdyDis = 1;
      if(clp[CFG_FCP_ON].a_current) {
         un.npr->prliType = PRLI_FCP_TYPE;
         un.npr->initiatorFunc = 1;
      }

      size = sizeof(uint32) + sizeof(PRLI);
      break;

   case ELS_CMD_PRLO:   /* Process logout */
      /* For PRLO, remainder of payload is PRLO parameter page */
      fc_bzero((void *)bp, sizeof(PRLO));

      icmd->un.elsreq.remoteID = (uint32)((ulong)arg);   /* DID */
      size = sizeof(uint32) + sizeof(PRLO);
      break;

   case ELS_CMD_SCR:            /* State Change Registration */
      /* For SCR, remainder of payload is SCR parameter page */
      fc_bzero((void *)bp, sizeof(SCR));
      ((SCR * )bp)->Function = SCR_FUNC_FULL;

      icmd->un.elsreq.remoteID = (uint32)((ulong)arg);   /* DID */
      size = sizeof(uint32) + sizeof(SCR);
      break;

   case ELS_CMD_RNID:            /* Node Identification */
      fc_bzero((void *)bp, sizeof(RNID));
      ((RNID * )bp)->Format = 0;

      icmd->un.elsreq.remoteID = (uint32)((ulong)arg);   /* DID */
      size = sizeof(uint32) + sizeof(uint32);
      break;

   case ELS_CMD_FARP:   /* Farp */
      {
         un.fp = (FARP * )(bp);
         fc_bzero((void *)un.fp, sizeof(FARP));
         lp = (uint32 *)bp;
         *lp++ = SWAP_DATA(binfo->fc_myDID);
         un.fp->Mflags = FARP_MATCH_PORT;
         un.fp->Rflags = FARP_REQUEST_PLOGI;
         fc_bcopy((void *) & binfo->fc_portname, (void *) & un.fp->OportName,
             sizeof(NAME_TYPE));
         fc_bcopy((void *) & binfo->fc_nodename, (void *) & un.fp->OnodeName,
             sizeof(NAME_TYPE));
         switch(retry) {
         case 0:
            un.fp->Mflags = FARP_MATCH_PORT;
            un.fp->RportName.nameType = NAME_IEEE;      /* IEEE name */
            un.fp->RportName.IEEEextMsn = 0;
            un.fp->RportName.IEEEextLsb = 0;
            fc_bcopy(arg, (void *)un.fp->RportName.IEEE, 6);
            un.fp->RnodeName.nameType = NAME_IEEE;      /* IEEE name */
            un.fp->RnodeName.IEEEextMsn = 0;
            un.fp->RnodeName.IEEEextLsb = 0;
            fc_bcopy(arg, (void *)un.fp->RnodeName.IEEE, 6);
            break;
         case 1:
            un.fp->Mflags = FARP_MATCH_PORT;
            fc_bcopy(arg, (void *)&un.fp->RportName, sizeof(NAME_TYPE));
            retry = 0;
            break;
         case 2:
            un.fp->Mflags = FARP_MATCH_NODE;
            fc_bcopy(arg, (void *)&un.fp->RnodeName, sizeof(NAME_TYPE));
            retry = 0;
            break;
         }

         if((ndlp = fc_findnode_wwpn(binfo, NLP_SEARCH_ALL, &un.fp->RportName))) {
            ndlp->nlp_flag |= NLP_FARP_SND;
            ndlp->nlp_flag &= ~NLP_RM_ENTRY;
         }
         size = sizeof(uint32) + sizeof(FARP);
         iotag = 0;
      }
      break;

   case ELS_CMD_FARPR:  /* Farp response */
      {
         icmd->un.elsreq.remoteID = (uint32)((ulong)arg);        /* DID */
         un.fp = (FARP * )(bp);
         lp = (uint32 *)bp;
         *lp++ = SWAP_DATA((uint32)((ulong)arg));
         *lp++ = SWAP_DATA(binfo->fc_myDID);
         un.fp->Rflags = 0;
         un.fp->Mflags = (FARP_MATCH_PORT | FARP_MATCH_NODE);

         fc_bcopy((void *) & binfo->fc_portname, (void *) & un.fp->RportName,
             sizeof(NAME_TYPE));
         fc_bcopy((void *) & binfo->fc_nodename, (void *) & un.fp->RnodeName,
             sizeof(NAME_TYPE));
         if ((ndlp = fc_findnode_odid(binfo, NLP_SEARCH_ALL, (uint32)((ulong)arg)))) {
            fc_bcopy((void *) & ndlp->nlp_portname, (void *) & un.fp->OportName,
                sizeof(NAME_TYPE));
            fc_bcopy((void *) & ndlp->nlp_nodename, (void *) & un.fp->OnodeName,
                sizeof(NAME_TYPE));
         }

         size = sizeof(uint32) + sizeof(FARP);
         iotag = 0;
      }
      break;

   default:
      fc_mem_put(binfo, MEM_IOCB, (uchar * )temp);
      fc_mem_put(binfo, MEM_BUF, (uchar * )mp);
      fc_mem_put(binfo, MEM_BUF, (uchar * )rmp);
      if (bmp) {
         fc_mem_put(binfo, MEM_BPL, (uchar * )bmp);
      }
      /* Xmit unknown ELS command <elsCmd> */
      fc_log_printf_msg_vargs( binfo->fc_brd_no,
             &fc_msgBlk0128,                   /* ptr to msg structure */
              fc_mes0128,                      /* ptr to msg */
               fc_msgBlk0128.msgPreambleStr,   /* begin varargs */
                elscmd);                       /* end varargs */
      return(1);
   }

   if (binfo->fc_flag & FC_SLI2) {
      icmd->ulpCommand = CMD_ELS_REQUEST64_CR;
      bpl->tus.f.bdeSize = size;
      bpl->tus.w = PCIMEM_LONG(bpl->tus.w);

      fc_mpdata_sync(bmp->dma_handle, 0, 0, DDI_DMA_SYNC_FORDEV);
   } else {
      icmd->ulpCommand = CMD_ELS_REQUEST_CR;
      icmd->un.cont[0].bdeSize = size;
   }

   fc_mpdata_sync(mp->dma_handle, 0, 0, DDI_DMA_SYNC_FORDEV);

   if (iotag) {
      icmd->ulpIoTag = iotag;
   }
   icmd->ulpIoTag0 = (unsigned)rp->fc_iotag++;
   if ((rp->fc_iotag & 0x3fff) == 0) {
      rp->fc_iotag = 1;
   }

   /* Fill in rest of iocb */
   icmd->ulpBdeCount = 1;
   icmd->ulpLe = 1;
   icmd->ulpClass = CLASS3;
   icmd->ulpOwner = OWN_CHIP;
   temp->retry = (uchar)retry;  /* retry = uint32 */
   rmp->fc_mptr = (uchar *)ndlp;
   /* Xmit ELS command <elsCmd> to remote NPORT <did> */
   fc_log_printf_msg_vargs( binfo->fc_brd_no,
          &fc_msgBlk0129,                   /* ptr to msg structure */
           fc_mes0129,                      /* ptr to msg */
            fc_msgBlk0129.msgPreambleStr,   /* begin varargs */
             elscmd,
              icmd->un.ulpWord[5],          /* did */
               icmd->ulpIoTag,
                binfo->fc_ffstate);         /* end varargs */
   /*
    * For handleing Dump command when system panic, 
    * the FC_BUS_RESET needs to be checked. If FC_BUS_RESET is set, 
    * there is no delay for issuing ELS command.
    * FC_BUS_RESET is set by the lpfc_scsi_reset().
    */
   if(icmd->ulpDelayXmit)
   {
      if(icmd->ulpDelayXmit == 2) {
         /* Delay issue of iocb 2048 interrupt latencies */
         if(binfo->fc_delayxmit) {
            IOCBQ *iop;
            iop = binfo->fc_delayxmit;
            while(iop->q)
               iop = (IOCBQ *)iop->q;
            iop->q = (uchar *)temp;
         }
         else {
            binfo->fc_delayxmit = temp;
         }
         temp->q = 0;
         temp->rsvd2 = 2048;
      }
      else {
         /* Delay issue of iocb for 1 to 2 seconds */
         temp->q = 0;

         setdelay = 1; /* seconds */
         fc_clk_set(p_dev_ctl, setdelay, fc_delay_timeout, (void *)temp, ndlp);
      }
   }
   else {
      issue_iocb_cmd(binfo, rp, temp);
   }

   FCSTATCTR.elsXmitFrame++;
   return(0);
}       /* End fc_els_cmd */


/***************************************/
/**  fc_els_rsp    Issue an           **/
/**                ELS response       **/
/***************************************/
_static_ int
fc_els_rsp(
FC_BRD_INFO *binfo,
uint32      elscmd,
uint32      Xri,
uint32      class,
void        *iocbp,
uint32      flag,
NODELIST    *ndlp)
{
   IOCB          * icmd;
   IOCBQ         * temp;
   RING          * rp;
   uchar         * bp;
   MATCHMAP      * mp, * bmp;
   ULP_BDE64     * bpl;
   ADISC         * ap;
   RNID          * rn;
   fc_vpd_t      * vpd;
   PRLI          * npr;
   iCfgParam     * clp;
   fc_dev_ctl_t  * p_dev_ctl;
   ushort        size;

   rp = &binfo->fc_ring[FC_ELS_RING];
   clp = DD_CTL.p_config[binfo->fc_brd_no];
   p_dev_ctl = (fc_dev_ctl_t *)(binfo->fc_p_dev_ctl);

   /* Allocate buffer for command iocb */
   if ((temp = (IOCBQ * )fc_mem_get(binfo, MEM_IOCB)) == 0) {
      return(1);
   }
   fc_bzero((void *)temp, sizeof(IOCBQ));
   icmd = &temp->iocb;

   /* fill in BDEs for command */
   /* Allocate buffer for response payload */
   if ((mp = (MATCHMAP * )fc_mem_get(binfo, MEM_BUF)) == 0) {
      fc_mem_put(binfo, MEM_IOCB, (uchar * )temp);
      return(1);
   }

   if (binfo->fc_flag & FC_SLI2) {
      /* Allocate buffer for Buffer ptr list */
      if ((bmp = (MATCHMAP * )fc_mem_get(binfo, MEM_BPL)) == 0) {
         fc_mem_put(binfo, MEM_IOCB, (uchar * )temp);
         fc_mem_put(binfo, MEM_BUF, (uchar * )mp);
         return(1);
      }
      bpl = (ULP_BDE64 * )bmp->virt;
      bpl->addrLow = PCIMEM_LONG(putPaddrLow((ulong)mp->phys));
      bpl->addrHigh = PCIMEM_LONG(putPaddrHigh((ulong)mp->phys));
      bpl->tus.f.bdeFlags = 0;

      icmd->un.elsreq64.bdl.ulpIoTag32 = (uint32)0;
      icmd->un.elsreq64.bdl.addrHigh = (uint32)putPaddrHigh(bmp->phys);
      icmd->un.elsreq64.bdl.addrLow = (uint32)putPaddrLow(bmp->phys);
      icmd->un.elsreq64.bdl.bdeSize = sizeof(ULP_BDE64);
      icmd->un.elsreq64.bdl.bdeFlags = BUFF_TYPE_BDL;
      temp->bpl = (uchar *)bmp;
   } else {
      bpl = 0;
      bmp = 0;
      icmd->un.cont[0].bdeAddress = (uint32)putPaddrLow(mp->phys);
      temp->bpl = 0;
   }

   bp = mp->virt;

   /* Save for completion so we can release these resources */
   temp->bp = (uchar * )mp;
   temp->ndlp = (uchar * )ndlp;

   /* Fill in command field in payload */
   *((uint32 * )(bp)) = elscmd;           /* ACC or LS_RJT */

   switch (elscmd) {
   case ELS_CMD_ACC:    /* Accept Response */
      /* ACCEPT will optionally contain service parameters, 
       * depending on flag.
       */
      bp += sizeof(uint32);
      if (flag >= sizeof(SERV_PARM)) {
         fc_bcopy((void *) & binfo->fc_sparam, (void *)bp, sizeof(SERV_PARM));
         size = (sizeof(SERV_PARM) + sizeof(uint32));
      } else {
         size = sizeof(uint32);
      }
      break;

   case ELS_CMD_LS_RJT: /* reject response */
      bp += sizeof(uint32);
      *((uint32 * )(bp)) = flag;                /* fill in error code */
      size = sizeof(uint32) + sizeof(uint32);
      break;

   case ELS_CMD_ADISC:
      *((uint32 * )(bp)) = ELS_CMD_ACC;
      bp += sizeof(uint32);
      if(ndlp)
         icmd->un.elsreq.remoteID = ndlp->nlp_DID;       /* DID */

      ap = (ADISC * )(bp);
      ap->hardAL_PA = binfo->fc_pref_ALPA;
      fc_bcopy((void *) & binfo->fc_portname, (void *) & ap->portName,
          sizeof(NAME_TYPE));
      fc_bcopy((void *) & binfo->fc_nodename, (void *) & ap->nodeName,
          sizeof(NAME_TYPE));
      ap->DID = SWAP_DATA(binfo->fc_myDID);

      size = sizeof(uint32) + sizeof(ADISC);
      break;

   case ELS_CMD_PRLI:
      *((uint32 * )(bp)) = (ELS_CMD_ACC | (ELS_CMD_PRLI & ~ELS_RSP_MASK));
      bp += sizeof(uint32);
      npr = (PRLI *)bp;
      if(ndlp)
         icmd->un.elsreq.remoteID = ndlp->nlp_DID;       /* DID */

      /* For PRLI, remainder of payload is PRLI parameter page */
      fc_bzero((void *)bp, sizeof(PRLI));

      vpd = &p_dev_ctl->vpd;
      /*
       * If our firmware version is 3.20 or later, 
       * set the following bits for FC-TAPE support.
       */
      if ( vpd->rev.feaLevelHigh >= 0x02 ) {
         npr->ConfmComplAllowed = 1;
         npr->Retry = 1;
         npr->TaskRetryIdReq = 1;
      }

      npr->acceptRspCode = PRLI_REQ_EXECUTED;
      npr->estabImagePair = 1;
      npr->readXferRdyDis = 1;
      npr->ConfmComplAllowed = 1;
      if(clp[CFG_FCP_ON].a_current) {
         npr->prliType = PRLI_FCP_TYPE;
         npr->initiatorFunc = 1;
      }

      size = sizeof(uint32) + sizeof(PRLI);
      break;

   case ELS_CMD_RNID:
      *((uint32 * )(bp)) = ELS_CMD_ACC;
      bp += sizeof(uint32);

      rn = (RNID * )(bp);
      fc_bzero((void *)bp, sizeof(RNID));
      rn->Format = (uchar)flag;
      rn->CommonLen = (2 * sizeof(NAME_TYPE));
      fc_bcopy((void *) & binfo->fc_portname, (void *) & rn->portName,
          sizeof(NAME_TYPE));
      fc_bcopy((void *) & binfo->fc_nodename, (void *) & rn->nodeName,
          sizeof(NAME_TYPE));
      switch(flag) {
      case 0:
         rn->SpecificLen = 0;
         break;
      case RNID_TOPOLOGY_DISC:
         rn->SpecificLen = sizeof(RNID_TOP_DISC);
         fc_bcopy((void *) & binfo->fc_portname,
            (void *) & rn->un.topologyDisc.portName, sizeof(NAME_TYPE));
         rn->un.topologyDisc.unitType = RNID_HBA;
         rn->un.topologyDisc.physPort = 0;
         rn->un.topologyDisc.attachedNodes = 0;
         if(clp[CFG_NETWORK_ON].a_current) {
            rn->un.topologyDisc.ipVersion = binfo->ipVersion;
            rn->un.topologyDisc.UDPport = binfo->UDPport;
            fc_bcopy((void *) & binfo->ipAddr[0],
               (void *) & rn->un.topologyDisc.ipAddr[0], 16);
         }
         break;
      default:
         rn->CommonLen = 0;
         rn->SpecificLen = 0;
         break;
      }
      size = sizeof(uint32) + sizeof(uint32) + rn->CommonLen + rn->SpecificLen;
      break;

   default:
      fc_mem_put(binfo, MEM_IOCB, (uchar * )temp);
      fc_mem_put(binfo, MEM_BUF, (uchar * )mp);
      if (bmp) {
         fc_mem_put(binfo, MEM_BPL, (uchar * )bmp);
      }
      /* Xmit unknown ELS response (elsCmd> */
      fc_log_printf_msg_vargs( binfo->fc_brd_no,
             &fc_msgBlk0130,                   /* ptr to msg structure */
              fc_mes0130,                      /* ptr to msg */
               fc_msgBlk0130.msgPreambleStr,   /* begin varargs */
                elscmd );                      /* end varargs */
      return(1);
   }

   if (binfo->fc_flag & FC_SLI2) {
      icmd->ulpCommand = CMD_XMIT_ELS_RSP64_CX;
      bpl->tus.f.bdeSize = size;
      bpl->tus.w = PCIMEM_LONG(bpl->tus.w);

      fc_mpdata_sync(bmp->dma_handle, 0, 0, DDI_DMA_SYNC_FORDEV);
   } else {
      icmd->ulpCommand = CMD_XMIT_ELS_RSP_CX;
      icmd->un.cont[0].bdeSize = size;
   }

   fc_mpdata_sync(mp->dma_handle, 0, 0, DDI_DMA_SYNC_FORDEV);

   /* If iotag is zero, assign one from global counter for board */
   if (iocbp == 0) {
      temp->retry = 0;
   } else {
      icmd->ulpIoTag = ((IOCB *)iocbp)->ulpIoTag;
      temp->retry = ((IOCBQ *)iocbp)->retry;
   }
   icmd->ulpIoTag0 = (unsigned)rp->fc_iotag++;
   if ((rp->fc_iotag & 0x3fff) == 0) {
      rp->fc_iotag = 1;
   }

   /* fill in rest of iocb */
   icmd->ulpContext = (volatile ushort)Xri;
   icmd->ulpBdeCount = 1;
   icmd->ulpLe = 1;
   icmd->ulpClass = class;
   icmd->ulpOwner = OWN_CHIP;
   /* Xmit ELS response <elsCmd> to remote NPORT <did> */
   fc_log_printf_msg_vargs( binfo->fc_brd_no,
          &fc_msgBlk0131,                   /* ptr to msg structure */
           fc_mes0131,                      /* ptr to msg */
            fc_msgBlk0131.msgPreambleStr,   /* begin varargs */
             elscmd,
              icmd->un.ulpWord[5],          /* did */
               icmd->ulpIoTag,
                size);                      /* end varargs */
   issue_iocb_cmd(binfo, rp, temp);

   FCSTATCTR.elsXmitFrame++;
   return(0);
}       /* End fc_els_rsp */


/* Retries the appropriate ELS command if necessary */
_local_ int
fc_els_retry(
FC_BRD_INFO     *binfo,
RING            *rp,
IOCBQ           *iocbq,
uint32          cmd,
NODELIST        *ndlp)
{
   IOCB *iocb;
   MATCHMAP     *bmp;

   if (((binfo->fc_flag & FC_RSCN_MODE) && (binfo->fc_ffstate == FC_READY)) ||
      (binfo->fc_ffstate == FC_LOOP_DISC) ||
      (binfo->fc_ffstate == FC_NODE_DISC)) {
      binfo->fc_fabrictmo = (2 * binfo->fc_ratov) +
         ((4 * binfo->fc_edtov) / 1000) + 1;
      if(FABRICTMO) {
         fc_clk_res((fc_dev_ctl_t *)(binfo->fc_p_dev_ctl),
            binfo->fc_fabrictmo, FABRICTMO);
      }
      else {
         FABRICTMO = fc_clk_set((fc_dev_ctl_t *)(binfo->fc_p_dev_ctl),
            binfo->fc_fabrictmo, fc_fabric_timeout, 0, 0);
      }
   }

   iocb = &iocbq->iocb;
   /* Do not retry FARP/ADISC/PDISC */
   if ((cmd == ELS_CMD_FARP) || 
       (cmd == ELS_CMD_FARPR) || 
       (cmd == ELS_CMD_ADISC) || 
       (cmd == ELS_CMD_PDISC)) {
      goto out;
   }

   if (fc_status_action(binfo, iocbq, cmd, ndlp)) {
      /* Indicates iocb should be retried */
      /* Retry ELS response/command */
      FCSTATCTR.elsXmitRetry++;
      switch (iocb->ulpCommand) {
      case CMD_ELS_REQUEST_CR:
      case CMD_ELS_REQUEST64_CR:
      case CMD_ELS_REQUEST_CX:
      case CMD_ELS_REQUEST64_CX:
         fc_els_cmd(binfo, cmd, (void *)((ulong)iocb->un.elsreq.remoteID),
             (uint32)iocbq->retry, (ushort)iocb->ulpIoTag, ndlp);
         break;
      case CMD_XMIT_ELS_RSP_CX:
         fc_els_rsp(binfo,cmd,(uint32)iocb->ulpContext, (uint32)iocb->ulpClass,
             (void *)iocbq, (uint32)iocb->un.cont[0].bdeSize, ndlp);
         break;
      case CMD_XMIT_ELS_RSP64_CX:
         bmp = (MATCHMAP *)iocbq->bpl;
         if(bmp && bmp->virt) {
            fc_els_rsp(binfo,cmd,(uint32)iocb->ulpContext, 
               (uint32)iocb->ulpClass, (void *)iocbq,
               (uint32)(((ULP_BDE64 * )bmp->virt)->tus.f.bdeSize), ndlp);
         }
         break;
      default:
         goto out;
      }
      return(1);
   }

out:
   /* ELS Retry failed */
   fc_log_printf_msg_vargs( binfo->fc_brd_no,
          &fc_msgBlk0132,                   /* ptr to msg structure */
           fc_mes0132,                      /* ptr to msg */
            fc_msgBlk0132.msgPreambleStr,   /* begin varargs */
             cmd,
              iocb->un.ulpWord[4] );        /* end varargs */
   return(0);
}       /* End fc_els_retry */


/* Determines what action to take as result of the status 
 * field in the iocb. If the status indicates a retry, the iocb
 * will be setup for retry and a 1 will be returned. If the status
 * indicates error with no action, a 0 will be returned.
 * The retry count is kept in the ls byte of the iotag.
 */
_local_ int
fc_status_action(
FC_BRD_INFO *binfo,
IOCBQ       *iocbq,
uint32      cmd,
NODELIST    *ndlp)
{
   uint32       class;
   uchar        tag;
   int          maxretry;
   LS_RJT       stat;
   IOCB         *iocb;

   maxretry = FC_MAXRETRY;
   iocb = &iocbq->iocb;
   iocb->ulpDelayXmit = 0;

   if(ndlp) {
      if(ndlp->nlp_action & NLP_DO_RNID)
         return(0);
      if((ndlp->nlp_DID == 0) && (ndlp->nlp_type == 0))
         return(0);
   }

   switch (iocb->ulpStatus) {
   case IOSTAT_FCP_RSP_ERROR:
   case IOSTAT_REMOTE_STOP:
      break;

   case IOSTAT_LOCAL_REJECT:
      if ((iocb->un.ulpWord[4] & 0xff) == IOERR_LINK_DOWN)
         return(0);

      if ((iocb->un.ulpWord[4] & 0xff) == IOERR_LOOP_OPEN_FAILURE) {
         if(cmd == ELS_CMD_PLOGI) { 
            if (iocbq->retry == 0)
               iocb->ulpDelayXmit = 2;
         }
         goto elsretry;
      }
      if ((iocb->un.ulpWord[4] & 0xff) == IOERR_SEQUENCE_TIMEOUT) {
         goto elsretry;
      }
      if ((iocb->un.ulpWord[4] & 0xff) == IOERR_NO_RESOURCES) {
         if(cmd == ELS_CMD_PLOGI) 
            iocb->ulpDelayXmit = 1;
         goto elsretry;
      }
      if ((iocb->un.ulpWord[4] & 0xff) == IOERR_INVALID_RPI) {
         goto elsretry;
      }
      break;

   case IOSTAT_NPORT_RJT:
   case IOSTAT_FABRIC_RJT:
      /* iotag is retry count */
      if ((tag = (iocbq->retry + 1)) >= maxretry) {
         FCSTATCTR.elsRetryExceeded++;
         break;
      }

      iocbq->retry = tag;
      if (iocb->un.ulpWord[4] & RJT_UNAVAIL_TEMP) {
         /* not avail temporary */
         /* Retry ELS command */
         return(1);
      }
      if (iocb->un.ulpWord[4] & RJT_UNSUP_CLASS) {
         /* class not supported */
         if (cmd == ELS_CMD_FARP)
            return(0);
         if (binfo->fc_topology == TOPOLOGY_LOOP) {
            /* for FC-AL retry logic goes class 3 - 2 - 1 */
            if (iocb->ulpClass == CLASS3) {
               class = CLASS2;
            } else {
               break;
            }
         } else {
            /* for non FC-AL retry logic goes class 1 - 2  */
            if (iocb->ulpClass == CLASS1) {
               class = CLASS2;
            } else {
               break;
            }
         }
         iocb->ulpClass = class;
         /* Retry ELS command */
         return(1);
      }
      break;

   case IOSTAT_NPORT_BSY:
   case IOSTAT_FABRIC_BSY:
elsretry:
      tag = (iocbq->retry + 1);
      /* iotag is retry count */
      if(ndlp) {
         if(cmd == ELS_CMD_PLOGI) { 
            if((ndlp->nlp_state >= NLP_LOGIN) ||
               (ndlp->nlp_flag & NLP_REG_INP)) {
               return(0);  /* Don't retry */
            }
         }
         if(ndlp->nlp_flag & NLP_NODEV_TMO) {
            iocbq->retry = tag;
            /* Retry ELS command */
            return(1);
         }
      }
      if(tag >= maxretry) {
         FCSTATCTR.elsRetryExceeded++;
         break;
      }
      iocbq->retry = tag;
      /* Retry ELS command */
      return(1);

   case IOSTAT_LS_RJT:
      stat.un.lsRjtError = SWAP_DATA(iocb->un.ulpWord[4]);
      switch(stat.un.b.lsRjtRsnCode) {
      case LSRJT_UNABLE_TPC:
         if(stat.un.b.lsRjtRsnCodeExp == LSEXP_CMD_IN_PROGRESS) {
            if(cmd == ELS_CMD_PLOGI) {
               iocb->ulpDelayXmit = 1;
               maxretry = 48;
            }
            goto elsretry;
         }
         if(cmd == ELS_CMD_PLOGI) {
            iocb->ulpDelayXmit = 1;

            /* allow for 1sec FLOGI delay */
            maxretry = FC_MAXRETRY + 1;
            goto elsretry;
         }
         break;

      case LSRJT_LOGICAL_BSY:
         if(cmd == ELS_CMD_PLOGI) {
            iocb->ulpDelayXmit = 1;
            maxretry = 48;
         }
         goto elsretry;
      }

      break;

   case IOSTAT_INTERMED_RSP:
   case IOSTAT_BA_RJT:
      break;

   default:
      break;
   }

   if((cmd == ELS_CMD_FLOGI) && (binfo->fc_topology != TOPOLOGY_LOOP)) {
      iocb->ulpDelayXmit = 1;
      maxretry = 48;
      if ((tag = (iocbq->retry + 1)) >= maxretry)
         return(0);
      iocbq->retry = tag;
      return(1);
   }
   return(0);
}       /* End fc_status_action */


_static_ void
fc_snd_flogi(
fc_dev_ctl_t * p_dev_ctl,
void *p1,
void *p2)
{
   FC_BRD_INFO  * binfo;
   RING         * rp;

   binfo = &BINFO;
   /* Stop the link down watchdog timer */
   rp = &binfo->fc_ring[FC_FCP_RING];
   if(RINGTMO) {
      fc_clk_can(p_dev_ctl, RINGTMO);
      RINGTMO = 0;
   }
   binfo->fc_flag &= ~(FC_LD_TIMEOUT | FC_LD_TIMER);

   /* We are either private or public loop topology */
   /* We are either Fabric or point-to-point topology */
   /* Now build FLOGI payload and issue ELS command to find out */
   fc_els_cmd(binfo, ELS_CMD_FLOGI, (void *)Fabric_DID,
       (uint32)0, (ushort)0, (NODELIST *)0);

   /*
    * Cancel the establish reset timer
    * If we come to this point, we don't need tht timer to
    * clear the FC_ESTABLISH_LINK flag.
    */
   if (p_dev_ctl->fc_estabtmo) {
      fc_clk_can(p_dev_ctl, p_dev_ctl->fc_estabtmo);
      p_dev_ctl->fc_estabtmo = 0;
   }
   return;
}

/* Wait < a second before sending intial FLOGI to start discovery */
int
fc_initial_flogi(
fc_dev_ctl_t * p_dev_ctl)       /* point to dev_ctl area */
{
   if((p_dev_ctl->fc_waitflogi = fc_clk_set(p_dev_ctl, 0, fc_snd_flogi, 0, 0)) == 0)
      fc_snd_flogi(p_dev_ctl, 0, 0);
   return(0);
}

/***************************************/
/**  fc_issue_ct_rsp   Issue an       **/
/**                    CT rsp         **/
/***************************************/
_static_ int
fc_issue_ct_rsp(
FC_BRD_INFO * binfo,
uint32        tag,
MATCHMAP    * bmp,
DMATCHMAP   * inp)
{
   IOCB          * icmd;
   IOCBQ         * temp;
   RING          * rp;
   fc_dev_ctl_t  * p_dev_ctl;
   uint32        num_entry;

   rp = &binfo->fc_ring[FC_ELS_RING];
   num_entry = (uint32)inp->dfc_flag;
   inp->dfc_flag = 0;

   /* Allocate buffer for  command iocb */
   if ((temp = (IOCBQ * )fc_mem_get(binfo, MEM_IOCB)) == 0) {
      return(1);
   }
   fc_bzero((void *)temp, sizeof(IOCBQ));
   icmd = &temp->iocb;

   icmd->un.xseq64.bdl.ulpIoTag32 = (uint32)0;
   icmd->un.xseq64.bdl.addrHigh = (uint32)putPaddrHigh(bmp->phys);
   icmd->un.xseq64.bdl.addrLow = (uint32)putPaddrLow(bmp->phys);
   icmd->un.xseq64.bdl.bdeFlags = BUFF_TYPE_BDL;
   icmd->un.xseq64.bdl.bdeSize = (num_entry * sizeof(ULP_BDE64));

   /* Save for completion so we can release these resources */
   temp->bp = (uchar * )inp;

   icmd->un.xseq64.w5.hcsw.Fctl = (LS | LA);
   icmd->un.xseq64.w5.hcsw.Dfctl = 0;
   icmd->un.xseq64.w5.hcsw.Rctl = FC_SOL_CTL;
   icmd->un.xseq64.w5.hcsw.Type = FC_COMMON_TRANSPORT_ULP;

   p_dev_ctl = (fc_dev_ctl_t *)(binfo->fc_p_dev_ctl);
   fc_mpdata_sync(bmp->dma_handle, 0, 0, DDI_DMA_SYNC_FORDEV);

   /* If iotag is zero, assign one from global counter for board */
   icmd->ulpIoTag0 = (unsigned)rp->fc_iotag++;
   if ((rp->fc_iotag & 0x3fff) == 0) {
      rp->fc_iotag = 1;
   }

   /* Fill in rest of iocb */
   icmd->ulpCommand = CMD_XMIT_SEQUENCE64_CX;
   icmd->ulpBdeCount = 1;
   icmd->ulpLe = 1;
   icmd->ulpClass = CLASS3;
   icmd->ulpContext = (ushort)tag;
   icmd->ulpOwner = OWN_CHIP;
   /* Xmit CT response on exchange <xid> */
   fc_log_printf_msg_vargs( binfo->fc_brd_no,
          &fc_msgBlk0133,                   /* ptr to msg structure */
           fc_mes0133,                      /* ptr to msg */
            fc_msgBlk0133.msgPreambleStr,   /* begin varargs */
             icmd->ulpContext,              /* xid */
              icmd->ulpIoTag,
               binfo->fc_ffstate );         /* end varargs */
   issue_iocb_cmd(binfo, rp, temp);
   return(0);
} /* fc_issue_ct_rsp */

/***************************************/
/**  fc_gen_req    Issue an           **/
/**                GEN_REQUEST cmd    **/
/***************************************/
_static_ int
fc_gen_req(
FC_BRD_INFO * binfo,
MATCHMAP    * bmp,
MATCHMAP    * inp,
MATCHMAP    * outp,
uint32        rpi,
uint32        usr_flg,
uint32        num_entry,
uint32        tmo)
{
   IOCB          * icmd;
   IOCBQ         * temp;
   RING          * rp;
   fc_dev_ctl_t  * p_dev_ctl;


   rp = &binfo->fc_ring[FC_ELS_RING];

   /* Allocate buffer for  command iocb */
   if ((temp = (IOCBQ * )fc_mem_get(binfo, MEM_IOCB | MEM_PRI)) == 0) {
      return(1);
   }
   fc_bzero((void *)temp, sizeof(IOCBQ));
   icmd = &temp->iocb;

   icmd->un.genreq64.bdl.ulpIoTag32 = (uint32)0;
   icmd->un.genreq64.bdl.addrHigh = (uint32)putPaddrHigh(bmp->phys);
   icmd->un.genreq64.bdl.addrLow = (uint32)putPaddrLow(bmp->phys);
   icmd->un.genreq64.bdl.bdeFlags = BUFF_TYPE_BDL;
   icmd->un.genreq64.bdl.bdeSize = (num_entry * sizeof(ULP_BDE64));

   if(usr_flg)
      temp->bpl = 0;
   else
      temp->bpl = (uchar *)bmp;

   /* Save for completion so we can release these resources */
   temp->bp = (uchar * )inp;
   temp->info = (uchar * )outp;

   /* Fill in payload, bp points to frame payload */
   icmd->ulpCommand = CMD_GEN_REQUEST64_CR;

   p_dev_ctl = (fc_dev_ctl_t *)(binfo->fc_p_dev_ctl);
   fc_mpdata_sync(bmp->dma_handle, 0, 0, DDI_DMA_SYNC_FORDEV);

   /* If iotag is zero, assign one from global counter for board */
   icmd->ulpIoTag0 = (unsigned)rp->fc_iotag++;
   if ((rp->fc_iotag & 0x3fff) == 0) {
      rp->fc_iotag = 1;
   }

   /* Fill in rest of iocb */
   icmd->un.genreq64.w5.hcsw.Fctl = (SI | LA);
   icmd->un.genreq64.w5.hcsw.Dfctl = 0;
   icmd->un.genreq64.w5.hcsw.Rctl = FC_UNSOL_CTL;
   icmd->un.genreq64.w5.hcsw.Type = FC_COMMON_TRANSPORT_ULP;

   if(tmo == 0)
      tmo = (2 * binfo->fc_ratov);
   icmd->ulpTimeout = tmo;
   icmd->ulpBdeCount = 1;
   icmd->ulpLe = 1;
   icmd->ulpClass = CLASS3;
   icmd->ulpContext = (volatile ushort)rpi;
   icmd->ulpOwner = OWN_CHIP;
   /* Issue GEN REQ IOCB for NPORT <did> */
   fc_log_printf_msg_vargs( binfo->fc_brd_no,
          &fc_msgBlk0134,                   /* ptr to msg structure */
           fc_mes0134,                      /* ptr to msg */
            fc_msgBlk0134.msgPreambleStr,   /* begin varargs */
             icmd->un.ulpWord[5],           /* did */
              icmd->ulpIoTag,
               binfo->fc_ffstate );         /* end varargs */
   issue_iocb_cmd(binfo, rp, temp);

   FCSTATCTR.elsXmitFrame++;
   return(0);
}       /* End fc_gen_req */


/***************************************/
/**  fc_rnid_req   Issue an           **/
/**                RNID REQUEST cmd   **/
/***************************************/
_static_ int
fc_rnid_req(
FC_BRD_INFO * binfo,
DMATCHMAP    * inp,
DMATCHMAP    * outp,
MATCHMAP   ** bmpp,
uint32        rpi)
{
   IOCB          * icmd;
   IOCBQ         * temp;
   RING          * rp;
   ULP_BDE64     * bpl;
   MATCHMAP      * bmp;
   fc_dev_ctl_t  * p_dev_ctl;


   rp = &binfo->fc_ring[FC_ELS_RING];

   /* Allocate buffer for  command iocb */
   if ((temp = (IOCBQ * )fc_mem_get(binfo, MEM_IOCB | MEM_PRI)) == 0) {
      return(1);
   }
   fc_bzero((void *)temp, sizeof(IOCBQ));
   icmd = &temp->iocb;

   if (binfo->fc_flag & FC_SLI2) {
      /* Allocate buffer for Buffer ptr list */
      if ((bmp = (MATCHMAP * )fc_mem_get(binfo, MEM_BPL | MEM_PRI)) == 0) {
         fc_mem_put(binfo, MEM_IOCB, (uchar * )temp);
         return(1);
      }
      *bmpp = bmp;  /* to free BPL on compl */
      bpl = (ULP_BDE64 * )bmp->virt;
      bpl->addrLow = PCIMEM_LONG(putPaddrLow((ulong)inp->dfc.phys));
      bpl->addrHigh = PCIMEM_LONG(putPaddrHigh((ulong)inp->dfc.phys));
      bpl->tus.f.bdeFlags = 0;
      bpl++;
      bpl->addrLow = PCIMEM_LONG(putPaddrLow((ulong)outp->dfc.phys));
      bpl->addrHigh = PCIMEM_LONG(putPaddrHigh((ulong)outp->dfc.phys));
      bpl->tus.f.bdeSize = (ushort)((ulong)(outp->dfc.fc_mptr));
      bpl->tus.f.bdeFlags = BUFF_USE_RCV;
      bpl->tus.w = PCIMEM_LONG(bpl->tus.w);

      bpl--; /* so we can fill in size later */

      icmd->un.genreq64.bdl.ulpIoTag32 = (uint32)0;
      icmd->un.genreq64.bdl.addrHigh = (uint32)putPaddrHigh(bmp->phys);
      icmd->un.genreq64.bdl.addrLow = (uint32)putPaddrLow(bmp->phys);
      icmd->un.genreq64.bdl.bdeSize = (2 * sizeof(ULP_BDE64));
      icmd->un.genreq64.bdl.bdeFlags = BUFF_TYPE_BDL;
      temp->bpl = 0;
   } else {
      fc_mem_put(binfo, MEM_IOCB, (uchar * )temp);
      return(1);
   }

   /* Save for completion so we can release these resources */
   temp->info = (uchar * )outp;

   /* Fill in payload, bp points to frame payload */
   icmd->ulpCommand = CMD_GEN_REQUEST64_CR;
   bpl->tus.f.bdeSize = (ushort)((ulong)(inp->dfc.fc_mptr));
   bpl->tus.w = PCIMEM_LONG(bpl->tus.w);

   p_dev_ctl = (fc_dev_ctl_t *)(binfo->fc_p_dev_ctl);
   fc_mpdata_sync(bmp->dma_handle, 0, 0, DDI_DMA_SYNC_FORDEV);
   fc_mpdata_sync(inp->dfc.dma_handle, 0, 0, DDI_DMA_SYNC_FORDEV);

   /* If iotag is zero, assign one from global counter for board */
   icmd->ulpIoTag0 = (unsigned)rp->fc_iotag++;
   if ((rp->fc_iotag & 0x3fff) == 0) {
      rp->fc_iotag = 1;
   }

   /* Fill in rest of iocb */
   icmd->un.genreq64.w5.hcsw.Fctl = (SI | LA);
   icmd->un.genreq64.w5.hcsw.Dfctl = 0;
   icmd->un.genreq64.w5.hcsw.Rctl = FC_ELS_REQ;
   icmd->un.genreq64.w5.hcsw.Type = FC_ELS_DATA;

   icmd->ulpBdeCount = 1;
   icmd->ulpLe = 1;
   icmd->ulpClass = CLASS3;
   icmd->ulpTimeout = (uchar)(rp->fc_ringtmo - 2);
   icmd->ulpContext = (volatile ushort)rpi;
   icmd->ulpOwner = OWN_CHIP;
   /* Issue GEN REQ IOCB for RNID */
   fc_log_printf_msg_vargs( binfo->fc_brd_no,
          &fc_msgBlk0135,                   /* ptr to msg structure */
           fc_mes0135,                      /* ptr to msg */
            fc_msgBlk0135.msgPreambleStr,   /* begin varargs */
             icmd->un.ulpWord[5],           /* did */
              icmd->ulpIoTag,
               binfo->fc_ffstate );         /* end varargs */
   issue_iocb_cmd(binfo, rp, temp);
   outp->dfc.fc_mptr = 0;

   FCSTATCTR.elsXmitFrame++;
   return(0);
}       /* End fc_rnid_req */


/***************************************/
/**  fc_issue_ct_req    Issue a       **/
/**       CT request to nameserver    **/
/***************************************/
_static_ int
fc_issue_ct_req(
FC_BRD_INFO * binfo,
uint32        portid,
MATCHMAP     * bmp,
DMATCHMAP    * inmp,
DMATCHMAP    * outmp,
uint32         tmo)
{
   uint32 size;
   NODELIST     * ndlp;

   size = (uint32)outmp->dfc_flag;
   /* Find nameserver entry */
   if((((ndlp = fc_findnode_odid(binfo, NLP_SEARCH_ALL, portid))) == 0) ||
      (ndlp->nlp_Rpi == 0) || 
      (binfo->fc_flag & FC_RSCN_MODE)) {

      if ((binfo->fc_flag & FC_FABRIC) && (binfo->fc_ffstate == FC_READY)) {
         if ((ndlp == 0) || ((ndlp->nlp_state < NLP_PLOGI) && !(ndlp->nlp_flag & NLP_NS_REMOVED))) {
            /* We can LOGIN to the port first */
            fc_els_cmd(binfo, ELS_CMD_PLOGI, (void *)((ulong)portid),
               (uint32)0, (ushort)0, ndlp);
         }
         return(ENODEV);
      }
      return(EACCES);
   }

   if((fc_gen_req(binfo, bmp, (MATCHMAP *)inmp, (MATCHMAP *)outmp,
      ndlp->nlp_Rpi, 1, (inmp->dfc_flag + outmp->dfc_flag), tmo)))
      return(ENOMEM);

   outmp->dfc_flag = 0;
   return(0);
}

/**************************************************/
/**                                              **/
/**  Free any deferred RSCNs                     **/
/**                                              **/
/**************************************************/
_static_ int
fc_flush_rscn_defer(
fc_dev_ctl_t *p_dev_ctl)
{
   FC_BRD_INFO   * binfo;
   RING     * rp;
   IOCBQ    * xmitiq;
   IOCB     * iocb;
   MATCHMAP * mp;
   int  i;

   binfo = &BINFO;
   rp = &binfo->fc_ring[FC_ELS_RING];
   while (binfo->fc_defer_rscn.q_first) {
      xmitiq = (IOCBQ * )binfo->fc_defer_rscn.q_first;
      if ((binfo->fc_defer_rscn.q_first = xmitiq->q) == 0) {
         binfo->fc_defer_rscn.q_last = 0;
      }
      binfo->fc_defer_rscn.q_cnt--;
      iocb = &xmitiq->iocb;
      mp = *((MATCHMAP **)iocb);
      *((MATCHMAP **)iocb) = 0;
      xmitiq->q = NULL;
      fc_mem_put(binfo, MEM_BUF, (uchar * )mp);

      i = 1;
      /* free resources associated with this iocb and repost the ring buffers */
      if (!(binfo->fc_flag & FC_SLI2)) {
         for (i = 1; i < (int)iocb->ulpBdeCount; i++) {
            mp = fc_getvaddr(p_dev_ctl, rp, (uchar * )((ulong)iocb->un.cont[i].bdeAddress));
            if (mp) {
               fc_mem_put(binfo, MEM_BUF, (uchar * )mp);
            }
         }
      }
      fc_mem_put(binfo, MEM_IOCB, (uchar * )xmitiq);
   }
   return(0);
}

/**************************************************/
/**                                              **/
/** Issue a NameServer query for RSCN processing **/
/**                                              **/
/**************************************************/
_static_ void
fc_issue_ns_query(
fc_dev_ctl_t *p_dev_ctl,
void *a1,
void *a2)
{
   FC_BRD_INFO   * binfo;
   NODELIST      * ndlp;

   binfo = &BINFO;
   binfo->fc_flag &= ~FC_NSLOGI_TMR;
   /* Now check with NameServer */
   if ((ndlp = fc_findnode_odid(binfo, NLP_SEARCH_ALL, NameServer_DID)) == 0) {
      /* We can LOGIN to the NameServer now */
      fc_els_cmd(binfo, ELS_CMD_PLOGI, (void *)NameServer_DID,
          (uint32)0, (ushort)0, ndlp);
   }
   else {
      /* Issue GID_FT to Nameserver */
      if (fc_ns_cmd(p_dev_ctl, ndlp, SLI_CTNS_GID_FT)) {
         /* error so start discovery */
         /* Done with NameServer for now, but keep logged in */
         ndlp->nlp_action &= ~NLP_DO_RSCN;
  
         /* Fire out PLOGIs on nodes marked for discovery */
         if ((binfo->fc_nlp_cnt <= 0) && 
             !(binfo->fc_flag & FC_NLP_MORE)) {
             binfo->fc_nlp_cnt = 0;
             fc_nextrscn(p_dev_ctl, fc_max_els_sent);
         }
         else {
             fc_nextnode(p_dev_ctl, ndlp);
         }
         ndlp->nlp_action &= ~NLP_DO_ADDR_AUTH;
      }
   }
   return;
}

_static_ int
fc_abort_discovery(
fc_dev_ctl_t *p_dev_ctl)
{
   FC_BRD_INFO * binfo;
   MAILBOXQ * mb;

   binfo = &BINFO;

   fc_linkdown(p_dev_ctl);

   /* This should turn off DELAYED ABTS for ELS timeouts */
   if ((mb = (MAILBOXQ * )fc_mem_get(binfo, MEM_MBOX))) {
      fc_set_slim(binfo, (MAILBOX * )mb, 0x052198, 0);
      if (issue_mb_cmd(binfo, (MAILBOX * )mb, MBX_NOWAIT) != MBX_BUSY) {
         fc_mem_put(binfo, MEM_MBOX, (uchar * )mb);
      }
   }

   /* This is at init, clear la */
   if ((mb = (MAILBOXQ * )fc_mem_get(binfo, MEM_MBOX | MEM_PRI))) {
      fc_clear_la(binfo, (MAILBOX * )mb);
      if (issue_mb_cmd(binfo, (MAILBOX * )mb, MBX_NOWAIT) != MBX_BUSY) {
         fc_mem_put(binfo, MEM_MBOX, (uchar * )mb);
      }
   } else {
      binfo->fc_ffstate = FC_ERROR;
      /* Device Discovery completion error */
      fc_log_printf_msg_vargs( binfo->fc_brd_no,
             &fc_msgBlk0217,                   /* ptr to msg structure */
              fc_mes0217,                      /* ptr to msg */
               fc_msgBlk0217.msgPreambleStr);  /* begin & end varargs */
   }
   if(FABRICTMO) {
      fc_clk_can(p_dev_ctl, FABRICTMO);
      FABRICTMO = 0;
   }
   return(0);
}

#define FOURBYTES   4

/**************************************************/
/**  fc_fdmi_cmd                                 **/
/**                                              **/
/**  Description:                                **/
/**  Issue Cmd to HBA Management Server          **/
/**     SLI_MGMT_RHBA                            **/
/**     SLI_MGMT_RPRT                            **/
/**     SLI_MGMT_DHBA                            **/
/**     SLI_MGMT_DPRT                            **/
/**                                              **/
/**     Accept Payload for those 4 commands      **/
/**     is 0                                     **/
/**                                              **/
/**    Returns:                                  **/
/**                                              **/
/**************************************************/
_static_ int
fc_fdmi_cmd(
fc_dev_ctl_t *p_dev_ctl,
NODELIST        *ndlp,
int cmdcode)
{
   FC_BRD_INFO       * binfo;
   MATCHMAP          * mp, *bmp;
   SLI_CT_REQUEST    * CtReq;
   ULP_BDE64         * bpl;
   u32bit          size;
   PREG_HBA            rh;
   PPORT_ENTRY         pe;
   PREG_PORT_ATTRIBUTE pab;  
   PATTRIBUTE_BLOCK    ab;
   PATTRIBUTE_ENTRY    ae;
   uint32          id;

   binfo = &BINFO;

   /* fill in BDEs for command */
   /* Allocate buffer for command payload */
   if ((mp = (MATCHMAP * )fc_mem_get(binfo, MEM_BUF)) == 0) {
      return(1);
   }

   bmp = 0;

   /* Allocate buffer for Buffer ptr list */
   if ((bmp = (MATCHMAP * )fc_mem_get(binfo, MEM_BPL)) == 0) {
      fc_mem_put(binfo, MEM_BUF, (uchar * )mp);
      return(1);
   }
   /* FDMI Req */
   fc_log_printf_msg_vargs( binfo->fc_brd_no,
          &fc_msgBlk0218,                   /* ptr to msg structure */
           fc_mes0218,                      /* ptr to msg */
            fc_msgBlk0218.msgPreambleStr,   /* begin varargs */
             cmdcode,
               binfo->fc_flag );            /* end varargs */
   CtReq = (SLI_CT_REQUEST * )mp->virt;
   /* 
    * Initialize mp, 1024 bytes 
    */
   fc_bzero((void *)CtReq, FCELSSIZE);   

   CtReq->RevisionId.bits.Revision = SLI_CT_REVISION;
   CtReq->RevisionId.bits.InId = 0;

   CtReq->FsType = SLI_CT_MANAGEMENT_SERVICE;
   CtReq->FsSubType = SLI_CT_FDMI_Subtypes;
   size = 0;

   switch (cmdcode) {
     case SLI_MGMT_RHBA :                          
       {
       fc_vpd_t * vp;
       char     * str;
       uint32     i, j, incr;
       uchar      HWrev[8];
  
       vp = &VPD;
  
       CtReq->CommandResponse.bits.CmdRsp = SWAP_DATA16(SLI_MGMT_RHBA);
       CtReq->CommandResponse.bits.Size = 0; 
       rh = (PREG_HBA)&CtReq->un.PortID;
       fc_bcopy((uchar * )&binfo->fc_sparam.portName, (uchar * )&rh->hi.PortName,
         sizeof(NAME_TYPE));
       rh->rpl.EntryCnt = SWAP_DATA(1); /* One entry (port) per adapter */
       fc_bcopy((uchar * )&binfo->fc_sparam.portName, (uchar * )&rh->rpl.pe,
         sizeof(NAME_TYPE));
     
       /* point to the HBA attribute block */
       size = sizeof(NAME_TYPE) + FOURBYTES + sizeof(NAME_TYPE);
       ab = (PATTRIBUTE_BLOCK)((uchar *)rh + size);
       ab->EntryCnt = 0;

       /* Point to the begin of the first HBA attribute entry */
       /* #1 HBA attribute entry */
       size += FOURBYTES;
       ae = (PATTRIBUTE_ENTRY)((uchar *)rh + size);
       ae->ad.bits.AttrType = SWAP_DATA16(NODE_NAME);
       ae->ad.bits.AttrLen = SWAP_DATA16(sizeof(NAME_TYPE));
       fc_bcopy((uchar * )&binfo->fc_sparam.nodeName, (uchar * )&ae->un.NodeName,
         sizeof(NAME_TYPE));
       ab->EntryCnt++;
       size += FOURBYTES + sizeof(NAME_TYPE);

       /* #2 HBA attribute entry */
       ae = (PATTRIBUTE_ENTRY)((uchar *)rh + size);
       ae->ad.bits.AttrType = SWAP_DATA16(MANUFACTURER);
       ae->ad.bits.AttrLen = SWAP_DATA16(24);
       fc_bcopy("Emulex Network Systems", ae->un.Manufacturer, 22);        
       ab->EntryCnt++;
       size += FOURBYTES + 24;

       /* #3 HBA attribute entry */
       ae = (PATTRIBUTE_ENTRY)((uchar *)rh + size);
       ae->ad.bits.AttrType = SWAP_DATA16(SERIAL_NUMBER);
       ae->ad.bits.AttrLen = SWAP_DATA16(32);
       fc_bcopy(binfo->fc_SerialNumber, ae->un.SerialNumber, 32);
       ab->EntryCnt++;
       size += FOURBYTES + 32;

       /* #4 HBA attribute entry */
       id = fc_rdpci_32(p_dev_ctl, PCI_VENDOR_ID_REGISTER);
       switch((id >> 16) & 0xffff) {                          
         case PCI_DEVICE_ID_SUPERFLY:
           if((vp->rev.biuRev == 1) || (vp->rev.biuRev == 2) || 
             (vp->rev.biuRev == 3)) {
             ae = (PATTRIBUTE_ENTRY)((uchar *)rh + size);
             ae->ad.bits.AttrType = SWAP_DATA16(MODEL);
             ae->ad.bits.AttrLen = SWAP_DATA16(8);
             fc_bcopy("LP7000", ae->un.Model, 6);
             ab->EntryCnt++;
             size += FOURBYTES + 8;
           
             /* #5 HBA attribute entry */
             ae = (PATTRIBUTE_ENTRY)((uchar *)rh + size);
             ae->ad.bits.AttrType = SWAP_DATA16(MODEL_DESCRIPTION);
             ae->ad.bits.AttrLen = SWAP_DATA16(64);
             fc_bcopy("Emulex LightPulse LP7000 1 Gigabit PCI Fibre Channel Adapter",
               ae->un.ModelDescription, 62);
             ab->EntryCnt++;
             size += FOURBYTES + 64;
           } else {
             ae = (PATTRIBUTE_ENTRY)((uchar *)rh + size);
             ae->ad.bits.AttrType = SWAP_DATA16(MODEL);
             ae->ad.bits.AttrLen = SWAP_DATA16(8);
             fc_bcopy("LP7000E", ae->un.Model, 7);
             ab->EntryCnt++;
             size += FOURBYTES + 8;
           
             /* #5 HBA attribute entry */
             ae = (PATTRIBUTE_ENTRY)((uchar *)rh + size);
             ae->ad.bits.AttrType = SWAP_DATA16(MODEL_DESCRIPTION);
             ae->ad.bits.AttrLen = SWAP_DATA16(64);
             fc_bcopy("Emulex LightPulse LP7000E 1 Gigabit PCI Fibre Channel Adapter",
               ae->un.ModelDescription, 62);
             ab->EntryCnt++;
             size += FOURBYTES + 64;
           }
           break;
         case PCI_DEVICE_ID_DRAGONFLY:
           ae = (PATTRIBUTE_ENTRY)((uchar *)rh + size);
           ae->ad.bits.AttrType = SWAP_DATA16(MODEL);
           ae->ad.bits.AttrLen = SWAP_DATA16(8);
           fc_bcopy("LP8000", ae->un.Model, 6);
           ab->EntryCnt++;
           size += FOURBYTES + 8;
           
           /* #5 HBA attribute entry */
           ae = (PATTRIBUTE_ENTRY)((uchar *)rh + size);
           ae->ad.bits.AttrType = SWAP_DATA16(MODEL_DESCRIPTION);
           ae->ad.bits.AttrLen = SWAP_DATA16(64);
           fc_bcopy("Emulex LightPulse LP8000 1 Gigabit PCI Fibre Channel Adapter",
             ae->un.ModelDescription, 62);
           ab->EntryCnt++;
           size += FOURBYTES + 64;
           break;
         case PCI_DEVICE_ID_CENTAUR:
           if(FC_JEDEC_ID(vp->rev.biuRev) == CENTAUR_2G_JEDEC_ID) {
             ae = (PATTRIBUTE_ENTRY)((uchar *)rh + size);
             ae->ad.bits.AttrType = SWAP_DATA16(MODEL);
             ae->ad.bits.AttrLen = SWAP_DATA16(8);
             fc_bcopy("LP9002", ae->un.Model, 6);
             ab->EntryCnt++;
             size += FOURBYTES + 8;
           
             /* #5 HBA attribute entry */
             ae = (PATTRIBUTE_ENTRY)((uchar *)rh + size);
             ae->ad.bits.AttrType = SWAP_DATA16(MODEL_DESCRIPTION);
             ae->ad.bits.AttrLen = SWAP_DATA16(64);
             fc_bcopy("Emulex LightPulse LP9002 2 Gigabit PCI Fibre Channel Adapter",
               ae->un.ModelDescription, 62);
             ab->EntryCnt++;
             size += FOURBYTES + 64;
           } else {
             ae = (PATTRIBUTE_ENTRY)((uchar *)rh + size);
             ae->ad.bits.AttrType = SWAP_DATA16(MODEL);
             ae->ad.bits.AttrLen = SWAP_DATA16(8);
             fc_bcopy("LP9000", ae->un.Model, 6);
             ab->EntryCnt++;
             size += FOURBYTES + 8;
           
             /* #5 HBA attribute entry */
             ae = (PATTRIBUTE_ENTRY)((uchar *)rh + size);
             ae->ad.bits.AttrType = SWAP_DATA16(MODEL_DESCRIPTION);
             ae->ad.bits.AttrLen = SWAP_DATA16(64);
             fc_bcopy("Emulex LightPulse LP9000 1 Gigabit PCI Fibre Channel Adapter",
               ae->un.ModelDescription, 62);
             ab->EntryCnt++;
             size += FOURBYTES + 64;
           }
           break;
         case PCI_DEVICE_ID_PEGASUS:
           ae = (PATTRIBUTE_ENTRY)((uchar *)rh + size);
           ae->ad.bits.AttrType = SWAP_DATA16(MODEL);
           ae->ad.bits.AttrLen = SWAP_DATA16(8);
           fc_bcopy("LP9802", ae->un.Model, 6);
           ab->EntryCnt++;
           size += FOURBYTES + 8;
           
           /* #5 HBA attribute entry */
           ae = (PATTRIBUTE_ENTRY)((uchar *)rh + size);
           ae->ad.bits.AttrType = SWAP_DATA16(MODEL_DESCRIPTION);
           ae->ad.bits.AttrLen = SWAP_DATA16(64);
           fc_bcopy("Emulex LightPulse LP9802 2 Gigabit PCI Fibre Channel Adapter",
             ae->un.ModelDescription, 62);
           ab->EntryCnt++;
           size += FOURBYTES + 64;
           break;
         case PCI_DEVICE_ID_PFLY:
           ae = (PATTRIBUTE_ENTRY)((uchar *)rh + size);
           ae->ad.bits.AttrType = SWAP_DATA16(MODEL);
           ae->ad.bits.AttrLen = SWAP_DATA16(8);
           fc_bcopy("LP982", ae->un.Model, 5);
           ab->EntryCnt++;
           size += FOURBYTES + 8;
           
           /* #5 HBA attribute entry */
           ae = (PATTRIBUTE_ENTRY)((uchar *)rh + size);
           ae->ad.bits.AttrType = SWAP_DATA16(MODEL_DESCRIPTION);
           ae->ad.bits.AttrLen = SWAP_DATA16(64);
           fc_bcopy("Emulex LightPulse LP982 2 Gigabit PCI Fibre Channel Adapter",
             ae->un.ModelDescription, 62);
           ab->EntryCnt++;
           size += FOURBYTES + 64;
           break;
       }
           
       /* #6 HBA attribute entry */
       ae = (PATTRIBUTE_ENTRY)((uchar *)rh + size);
       ae->ad.bits.AttrType = SWAP_DATA16(HARDWARE_VERSION);
       ae->ad.bits.AttrLen = SWAP_DATA16(8);
       /* Convert JEDEC ID to ascii for hardware version */
       incr = vp->rev.biuRev;
       for(i=0;i<8;i++) {
         j = (incr & 0xf);
         if(j <= 9)
            HWrev[7-i] = (char)((uchar)0x30 + (uchar)j);
         else
            HWrev[7-i] = (char)((uchar)0x61 + (uchar)(j-10));
         incr = (incr >> 4);
       }
       fc_bcopy((uchar *)HWrev, ae->un.HardwareVersion, 8);
       ab->EntryCnt++;
       size += FOURBYTES + 8;
       
       /* #7 HBA attribute entry */
       ae = (PATTRIBUTE_ENTRY)((uchar *)rh + size);
       ae->ad.bits.AttrType = SWAP_DATA16(DRIVER_VERSION);
       ae->ad.bits.AttrLen = SWAP_DATA16(16);
       for (i=0; lpfc_release_version[i]; i++);
       fc_bcopy((uchar *)lpfc_release_version, ae->un.DriverVersion, i);
       ab->EntryCnt++;
       size += FOURBYTES + 16;

       /* #8 HBA attribute entry */
       ae = (PATTRIBUTE_ENTRY)((uchar *)rh + size);
       ae->ad.bits.AttrType = SWAP_DATA16(OPTION_ROM_VERSION);
       ae->ad.bits.AttrLen = SWAP_DATA16(32);
       fc_bcopy(binfo->fc_OptionROMVersion, ae->un.OptionROMVersion, 32);
       ab->EntryCnt++;
       size += FOURBYTES + 32;

       /* #9 HBA attribute entry */
       ae = (PATTRIBUTE_ENTRY)((uchar *)rh + size);
       ae->ad.bits.AttrType = SWAP_DATA16(FIRMWARE_VERSION);
       ae->ad.bits.AttrLen = SWAP_DATA16(32);
       str = decode_firmware_rev(binfo, vp);
       fc_bcopy((uchar *)str, ae->un.FirmwareVersion, 32);
       ab->EntryCnt++;
       size += FOURBYTES + 32;

       /* #10 HBA attribute entry */
       ae = (PATTRIBUTE_ENTRY)((uchar *)rh + size);
       ae->ad.bits.AttrType = SWAP_DATA16(VENDOR_SPECIFIC);
       ae->ad.bits.AttrLen = SWAP_DATA16(4);
       id = SWAP_LONG(id);
       id = (((SWAP_ALWAYS16(id >> 16)) << 16) | SWAP_ALWAYS16(id));
       ae->un.VendorSpecific = id;
       ab->EntryCnt++;
       size += FOURBYTES + 4;

       /* #11 HBA attribute entry */
       ae = (PATTRIBUTE_ENTRY)((uchar *)rh + size);
       ae->ad.bits.AttrType = SWAP_DATA16(DRIVER_NAME);
       ae->ad.bits.AttrLen = SWAP_DATA16(4);
       fc_bcopy("lpfc", ae->un.DriverName, 4);
       ab->EntryCnt++;
       size += FOURBYTES + 4;



       ab->EntryCnt = SWAP_DATA(ab->EntryCnt);
       /* Total size */
       size = GID_REQUEST_SZ - 4 + size;
       }
       break;

     case SLI_MGMT_RPRT :                          
       {
       fc_vpd_t   * vp;
       SERV_PARM  * hsp;    
  
       vp = &VPD;

       CtReq->CommandResponse.bits.CmdRsp = SWAP_DATA16(SLI_MGMT_RPRT);
       CtReq->CommandResponse.bits.Size = 0;
       pab = (PREG_PORT_ATTRIBUTE)&CtReq->un.PortID;
       size = sizeof(NAME_TYPE) + sizeof(NAME_TYPE) + FOURBYTES;
       fc_bcopy((uchar * )&binfo->fc_sparam.portName, (uchar * )&pab->HBA_PortName,
         sizeof(NAME_TYPE));
       fc_bcopy((uchar * )&binfo->fc_sparam.portName, (uchar * )&pab->PortName,
         sizeof(NAME_TYPE));
       pab->ab.EntryCnt = 0;

       /* #1 Port attribute entry */
       ae = (PATTRIBUTE_ENTRY)((uchar *)pab + size);
       ae->ad.bits.AttrType = SWAP_DATA16(SUPPORTED_FC4_TYPES);
       ae->ad.bits.AttrLen = SWAP_DATA16(8);
       ae->un.SupportFC4Types[4]   = 1; 
       pab->ab.EntryCnt++;
       size += FOURBYTES + 8;

       /* #2 Port attribute entry */
       ae = (PATTRIBUTE_ENTRY)((uchar *)pab + size);
       ae->ad.bits.AttrType = SWAP_DATA16(SUPPORTED_SPEED);
       ae->ad.bits.AttrLen = SWAP_DATA16(4);
       if(FC_JEDEC_ID(vp->rev.biuRev) == CENTAUR_2G_JEDEC_ID)
         ae->un.SupportSpeed = HBA_PORTSPEED_2GBIT;
       else
         ae->un.SupportSpeed = HBA_PORTSPEED_1GBIT;
       pab->ab.EntryCnt++;
       size += FOURBYTES + 4;

       /* #3 Port attribute entry */
       ae = (PATTRIBUTE_ENTRY)((uchar *)pab + size);
       ae->ad.bits.AttrType = SWAP_DATA16(PORT_SPEED);
       ae->ad.bits.AttrLen = SWAP_DATA16(4);
       if( binfo->fc_linkspeed == LA_2GHZ_LINK)
         ae->un.PortSpeed = HBA_PORTSPEED_2GBIT;
       else
         ae->un.PortSpeed = HBA_PORTSPEED_1GBIT;
       pab->ab.EntryCnt++;
       size += FOURBYTES + 4;

       /* #4 Port attribute entry */
       ae = (PATTRIBUTE_ENTRY)((uchar *)pab + size);
       ae->ad.bits.AttrType = SWAP_DATA16(MAX_FRAME_SIZE);
       ae->ad.bits.AttrLen = SWAP_DATA16(4);
       hsp = (SERV_PARM *)&binfo->fc_sparam;
       ae->un.MaxFrameSize = (((uint32)hsp->cmn.bbRcvSizeMsb) << 8) |
         (uint32)hsp->cmn.bbRcvSizeLsb;
       pab->ab.EntryCnt++;
       size += FOURBYTES + 4;

       /* #5 Port attribute entry */
       ae = (PATTRIBUTE_ENTRY)((uchar *)pab + size);
       ae->ad.bits.AttrType = SWAP_DATA16(OS_DEVICE_NAME);
       ae->ad.bits.AttrLen = SWAP_DATA16(4);
       fc_bcopy("lpfc", (uchar * )&ae->un.DriverName, 4);
       pab->ab.EntryCnt++;
       size += FOURBYTES + 4;

       pab->ab.EntryCnt = SWAP_DATA(pab->ab.EntryCnt);
       /* Total size */
       size = GID_REQUEST_SZ - 4 + size;
       } 
       break;

     case SLI_MGMT_DHBA :                         
       CtReq->CommandResponse.bits.CmdRsp = SWAP_DATA16(SLI_MGMT_DHBA);
       CtReq->CommandResponse.bits.Size = 0;
       pe = (PPORT_ENTRY)&CtReq->un.PortID;
       fc_bcopy((uchar * )&binfo->fc_sparam.portName, (uchar * )&pe->PortName,
         sizeof(NAME_TYPE));
       size = GID_REQUEST_SZ - 4 + sizeof(NAME_TYPE);
       break;

     case SLI_MGMT_DPRT :                          
       CtReq->CommandResponse.bits.CmdRsp = SWAP_DATA16(SLI_MGMT_DPRT);
       CtReq->CommandResponse.bits.Size = 0;
       pe = (PPORT_ENTRY)&CtReq->un.PortID;
       fc_bcopy((uchar * )&binfo->fc_sparam.portName, (uchar * )&pe->PortName,
         sizeof(NAME_TYPE));
       size = GID_REQUEST_SZ - 4 + sizeof(NAME_TYPE);
       break;
   }

   bpl = (ULP_BDE64 * )bmp->virt;
   bpl->addrHigh = PCIMEM_LONG((uint32)putPaddrHigh(mp->phys));
   bpl->addrLow = PCIMEM_LONG((uint32)putPaddrLow(mp->phys));
   bpl->tus.f.bdeFlags = 0;
   bpl->tus.f.bdeSize = size;
   bpl->tus.w = PCIMEM_LONG(bpl->tus.w);


   if(fc_ct_cmd(p_dev_ctl, mp, bmp, ndlp)) {
      fc_mem_put(binfo, MEM_BUF, (uchar * )mp);
      fc_mem_put(binfo, MEM_BPL, (uchar * )bmp);
   }
   return(0);
}       /* End fc_ns_cmd */

/**************************************************/
/**  fc_fdmi_rsp                                 **/
/**                                              **/
/**  Description:                                **/
/**  Process Rsp from HBA Management Server      **/
/**     SLI_MGMT_RHBA                            **/
/**     SLI_MGMT_RPRT                            **/
/**     SLI_MGMT_DHBA                            **/
/**     SLI_MGMT_DPRT                            **/
/**                                              **/
/**    Returns:                                  **/
/**                                              **/
/**************************************************/
_static_ void
fc_fdmi_rsp(
fc_dev_ctl_t   *p_dev_ctl,
MATCHMAP       *mp,
MATCHMAP       *rsp_mp)

{
   FC_BRD_INFO       * binfo;
   SLI_CT_REQUEST  * Cmd;
   SLI_CT_REQUEST  * Rsp;
   NODELIST        * ndlp;
   ushort        fdmi_cmd;
   ushort        fdmi_rsp;
   int           rc;

   binfo = &BINFO;

   ndlp = (NODELIST *)mp->fc_mptr;
   Cmd = (SLI_CT_REQUEST *)mp->virt;
   Rsp = (SLI_CT_REQUEST *)rsp_mp->virt;

   fdmi_rsp = Rsp->CommandResponse.bits.CmdRsp;

   fdmi_cmd = Cmd->CommandResponse.bits.CmdRsp;
   rc = 1;

   switch (SWAP_DATA16(fdmi_cmd)) {
     case SLI_MGMT_RHBA :
       rc = fc_fdmi_cmd(p_dev_ctl, ndlp, SLI_MGMT_RPRT);
       break;

     case SLI_MGMT_RPRT :
       break;

     case SLI_MGMT_DHBA :
       rc = fc_fdmi_cmd(p_dev_ctl, ndlp, SLI_MGMT_RHBA);
       break;

     case SLI_MGMT_DPRT :
       rc = fc_fdmi_cmd(p_dev_ctl, ndlp, SLI_MGMT_DHBA);
       break;
   }

   if (rc) {
      /* FDMI rsp failed */
      fc_log_printf_msg_vargs( binfo->fc_brd_no,
             &fc_msgBlk0251,                   /* ptr to msg structure */
              fc_mes0251,                      /* ptr to msg */
               fc_msgBlk0251.msgPreambleStr,   /* begin varargs */
                SWAP_DATA16(fdmi_cmd) );       /* end varargs */
   }
} /* fc_fdmi_rsp */
   

/*****************************************************************************/
/*
 * NAME:     fc_plogi_put
 *
 * FUNCTION: put iocb cmd onto the iocb plogi queue.
 *
 * EXECUTION ENVIRONMENT: process and interrupt level.
 *
 * NOTES:
 *
 * CALLED FROM:
 *      issue_els_cmd
 *
 * INPUT:
 *      binfo           - pointer to the device info area
 *      iocbq           - pointer to iocb queue entry
 *
 * RETURNS:  
 *      NULL - command queued
 */
/*****************************************************************************/
_static_ void
fc_plogi_put(
FC_BRD_INFO *binfo,
IOCBQ       *iocbq)               /* pointer to iocbq entry */
{
   if (binfo->fc_plogi.q_first) {
      /* queue command to end of list */
      ((IOCBQ * )binfo->fc_plogi.q_last)->q = (uchar * )iocbq;
      binfo->fc_plogi.q_last = (uchar * )iocbq;
   } else {
      /* add command to empty list */
      binfo->fc_plogi.q_first = (uchar * )iocbq;
      binfo->fc_plogi.q_last = (uchar * )iocbq;
   }

   iocbq->q = NULL;
   binfo->fc_plogi.q_cnt++;
   binfo->fc_flag |= FC_DELAY_NSLOGI;
   return;

}       /* End fc_plogi_put */


/*****************************************************************************/
/*
 * NAME:     fc_plogi_get
 *
 * FUNCTION: get a iocb command from iocb plogi command queue
 *
 * EXECUTION ENVIRONMENT: interrupt level.
 *
 * NOTES:
 *
 * CALLED FROM:
 *      handle_mb_event
 *
 * INPUT:
 *      binfo       - pointer to the device info area
 *
 * RETURNS:  
 *      NULL - no match found
 *      iocb pointer - pointer to a iocb command
 */
/*****************************************************************************/
_static_ IOCBQ *
fc_plogi_get(
FC_BRD_INFO *binfo)
{
   IOCBQ * p_first = NULL;

   if (binfo->fc_plogi.q_first) {
      p_first = (IOCBQ * )binfo->fc_plogi.q_first;
      if ((binfo->fc_plogi.q_first = p_first->q) == 0) {
         binfo->fc_plogi.q_last = 0;
         binfo->fc_flag &= ~FC_DELAY_NSLOGI;
      }
      p_first->q = NULL;
      binfo->fc_plogi.q_cnt--;
   }
   return(p_first);

}       /* End fc_plogi_get */

