// This file is part of OpenCV project.
// It is subject to the license terms in the LICENSE file found in the top-level directory
// of this distribution and at http://opencv.org/license.html.

namespace {

/**
  * Dictionaries are stored as a list of bytes in its four rotations.
  * On each rotation, the marker is divided in bytes assuming a row-major order.
  * This format allows a faster marker identification.
  * For a dictionary composed by M markers of NxN bits, the structure dimensions should be:
  * const char name[nMarkers][4rotations][nBytes], or more specifically:
  * const char name[M][4][ceil(NxN/8)]
  * The element [i][j][k] represents the k-th byte of the i-th marker in the dictionary
  * in its j-th rotation.
  * Each rotation implies a 90 degree rotation of the marker in anticlockwise direction.
  */
static unsigned char DICT_APRILTAG_16h5_BYTES[][4][2] =
{
    { {216, 196},
      {128, 190},
      {35, 27},
      {125, 1}
    },
    { {165, 116},
      {106, 120},
      {46, 165},
      {30, 86}
    },
    { {86, 44},
      {134, 209},
      {52, 106},
      {139, 97}
    },
    { {157, 162},
      {195, 78},
      {69, 185},
      {114, 195}
    },
    { {101, 158},
      {105, 211},
      {121, 166},
      {203, 150}
    },
    { {214, 254},
      {167, 251},
      {127, 107},
      {223, 229}
    },
    { {26, 205},
      {148, 55},
      {179, 88},
      {236, 41}
    },
    { {162, 231},
      {31, 58},
      {231, 69},
      {92, 248}
    },
    { {154, 127},
      {183, 61},
      {254, 89},
      {188, 237}
    },
    { {182, 168},
      {142, 75},
      {21, 109},
      {210, 113}
    },
    { {208, 28},
      {160, 153},
      {56, 11},
      {153, 5}
    },
    { {213, 15},
      {209, 217},
      {240, 171},
      {155, 139}
    },
    { {33, 176},
      {106, 2},
      {13, 132},
      {64, 86}
    },
    { {108, 226},
      {11, 230},
      {71, 54},
      {103, 208}
    },
    { {78, 49},
      {54, 196},
      {140, 114},
      {35, 108}
    },
    { {8, 245},
      {50, 54},
      {175, 16},
      {108, 76}
    },
    { {60, 144},
      {168, 70},
      {9, 60},
      {98, 21}
    },
    { {45, 201},
      {88, 103},
      {147, 180},
      {230, 26}
    },
    { {192, 165},
      {18, 154},
      {165, 3},
      {89, 72}
    },
    { {241, 98},
      {203, 168},
      {70, 143},
      {21, 211}
    },
    { {236, 135},
      {25, 222},
      {225, 55},
      {123, 152}
    },
    { {169, 234},
      {75, 47},
      {87, 149},
      {244, 210}
    },
    { {66, 251},
      {55, 163},
      {223, 66},
      {197, 236}
    },
    { {184, 56},
      {170, 13},
      {28, 29},
      {176, 85}
    },
    { {59, 151},
      {253, 22},
      {233, 220},
      {104, 191}
    },
    { {181, 206},
      {201, 123},
      {115, 173},
      {222, 147}
    },
    { {250, 181},
      {190, 158},
      {173, 95},
      {121, 125}
    },
    { {12, 171},
      {19, 71},
      {213, 48},
      {226, 200}
    },
    { {83, 224},
      {198, 162},
      {7, 202},
      {69, 99}
    },
    { {116, 245},
      {186, 242},
      {175, 46},
      {79, 93}
    }
};



static unsigned char DICT_APRILTAG_25h9_BYTES[][4][4] =
{
    { {143, 211, 170, 1},
      {234, 146, 237, 1},
      {170, 229, 248, 1},
      {219, 164, 171, 1}
    },
    { {109, 139, 39, 1},
      {170, 251, 21, 0},
      {242, 104, 219, 0},
      {84, 111, 170, 1}
    },
    { {22, 208, 222, 1},
      {94, 6, 244, 1},
      {189, 133, 180, 0},
      {151, 176, 61, 0}
    },
    { {179, 147, 87, 1},
      {62, 118, 217, 0},
      {245, 100, 230, 1},
      {77, 183, 62, 0}
    },
    { {115, 40, 116, 1},
      {28, 189, 146, 0},
      {151, 10, 103, 0},
      {36, 222, 156, 0}
    },
    { {221, 18, 172, 0},
      {164, 145, 124, 1},
      {26, 164, 93, 1},
      {159, 68, 146, 1}
    },
    { {103, 234, 238, 0},
      {98, 191, 182, 1},
      {59, 171, 243, 0},
      {182, 254, 163, 0}
    },
    { {213, 186, 96, 1},
      {46, 157, 78, 0},
      {131, 46, 213, 1},
      {57, 92, 186, 0}
    },
    { {231, 69, 93, 1},
      {89, 117, 157, 1},
      {221, 81, 115, 1},
      {220, 215, 77, 0}
    },
    { {17, 101, 123, 0},
      {85, 214, 3, 1},
      {111, 83, 68, 0},
      {224, 53, 213, 0}
    },
    { {228, 155, 129, 0},
      {34, 105, 109, 0},
      {64, 236, 147, 1},
      {91, 75, 34, 0}
    },
    { {130, 77, 138, 0},
      {65, 10, 169, 1},
      {40, 217, 32, 1},
      {202, 168, 65, 0}
    },
    { {117, 19, 219, 0},
      {52, 119, 101, 1},
      {109, 228, 87, 0},
      {211, 119, 22, 0}
    },
    { {152, 112, 167, 1},
      {204, 194, 122, 0},
      {242, 135, 12, 1},
      {47, 33, 153, 1}
    },
    { {75, 152, 26, 1},
      {154, 27, 192, 1},
      {172, 12, 233, 0},
      {129, 236, 44, 1}
    },
    { {65, 65, 182, 1},
      {88, 147, 49, 0},
      {182, 193, 65, 0},
      {70, 100, 141, 0}
    },
    { {172, 7, 198, 1},
      {169, 38, 61, 0},
      {177, 240, 26, 1},
      {94, 50, 74, 1}
    },
    { {129, 217, 185, 0},
      {82, 216, 105, 1},
      {78, 205, 192, 1},
      {203, 13, 165, 0}
    },
    { {35, 119, 206, 1},
      {105, 54, 243, 1},
      {185, 247, 98, 0},
      {231, 182, 75, 0}
    },
    { {144, 92, 117, 0},
      {85, 204, 88, 0},
      {87, 29, 4, 1},
      {13, 25, 213, 0}
    },
    { {138, 62, 250, 0},
      {177, 142, 234, 1},
      {47, 190, 40, 1},
      {171, 184, 198, 1}
    },
    { {63, 236, 133, 1},
      {207, 120, 182, 0},
      {208, 155, 254, 0},
      {54, 143, 121, 1}
    },
    { {70, 169, 59, 0},
      {18, 203, 135, 1},
      {110, 74, 177, 0},
      {240, 233, 164, 0}
    },
    { {162, 116, 1, 1},
      {73, 96, 202, 0},
      {192, 23, 34, 1},
      {41, 131, 73, 0}
    },
    { {52, 42, 195, 1},
      {44, 110, 38, 0},
      {225, 170, 22, 0},
      {50, 59, 26, 0}
    },
    { {37, 240, 225, 0},
      {66, 244, 102, 0},
      {67, 135, 210, 0},
      {51, 23, 161, 0}
    },
    { {144, 226, 137, 0},
      {102, 64, 42, 1},
      {72, 163, 132, 1},
      {170, 1, 51, 0}
    },
    { {15, 33, 102, 1},
      {136, 150, 151, 0},
      {179, 66, 120, 0},
      {116, 180, 136, 1}
    },
    { {165, 79, 19, 0},
      {113, 122, 13, 0},
      {100, 121, 82, 1},
      {88, 47, 71, 0}
    },
    { {221, 124, 193, 0},
      {197, 93, 110, 0},
      {65, 159, 93, 1},
      {59, 93, 81, 1}
    },
    { {254, 142, 154, 0},
      {183, 43, 172, 1},
      {44, 184, 191, 1},
      {154, 234, 118, 1}
    },
    { {10, 11, 60, 1},
      {184, 136, 145, 1},
      {158, 104, 40, 0},
      {196, 136, 142, 1}
    },
    { {139, 80, 95, 0},
      {208, 86, 216, 1},
      {125, 5, 104, 1},
      {141, 181, 5, 1}
    },
    { {13, 198, 237, 1},
      {235, 212, 52, 1},
      {219, 177, 216, 0},
      {150, 21, 235, 1}
    },
    { {115, 137, 157, 0},
      {22, 121, 177, 1},
      {92, 200, 231, 0},
      {198, 207, 52, 0}
    }
};



static unsigned char DICT_APRILTAG_36h10_BYTES[][4][5] =
{
    { {225, 101, 73, 83, 8},
      {49, 6, 165, 238, 1},
      {28, 169, 42, 104, 7},
      {135, 122, 86, 8, 12}
    },
    { {50, 53, 132, 160, 4},
      {65, 139, 98, 33, 2},
      {32, 82, 26, 196, 12},
      {72, 68, 109, 24, 2}
    },
    { {107, 237, 252, 94, 4},
      {43, 167, 244, 249, 13},
      {39, 163, 251, 125, 6},
      {185, 242, 254, 93, 4}
    },
    { {217, 189, 115, 45, 4},
      {123, 106, 82, 246, 12},
      {43, 76, 235, 217, 11},
      {54, 244, 165, 109, 14}
    },
    { {87, 115, 222, 38, 12},
      {233, 207, 79, 209, 1},
      {54, 71, 188, 238, 10},
      {136, 191, 47, 57, 7}
    },
    { {245, 43, 80, 249, 12},
      {161, 42, 235, 222, 10},
      {57, 240, 173, 74, 15},
      {87, 189, 117, 72, 5}
    },
    { {46, 91, 153, 155, 12},
      {210, 166, 237, 21, 11},
      {61, 153, 157, 167, 4},
      {218, 139, 118, 84, 11}
    },
    { {127, 167, 237, 114, 2},
      {179, 155, 190, 233, 5},
      {68, 235, 126, 95, 14},
      {169, 119, 221, 156, 13}
    },
    { {72, 191, 207, 28, 10},
      {123, 243, 157, 160, 8},
      {83, 143, 63, 209, 2},
      {16, 91, 156, 253, 14}
    },
    { {235, 64, 193, 206, 10},
      {58, 180, 33, 203, 3},
      {87, 56, 48, 45, 7},
      {205, 56, 66, 213, 12}
    },
    { {215, 152, 253, 186, 6},
      {240, 185, 214, 211, 15},
      {101, 219, 241, 158, 11},
      {252, 182, 185, 208, 15}
    },
    { {92, 212, 54, 176, 14},
      {194, 93, 211, 176, 6},
      {112, 214, 194, 179, 10},
      {96, 220, 187, 164, 3}
    },
    { {114, 66, 21, 239, 14},
      {24, 61, 107, 157, 3},
      {127, 122, 132, 36, 14},
      {203, 157, 107, 193, 8}
    },
    { {200, 18, 236, 76, 1},
      {110, 161, 12, 138, 4},
      {131, 35, 116, 129, 3},
      {37, 19, 8, 87, 6}
    },
    { {27, 178, 99, 54, 1},
      {127, 72, 154, 65, 5},
      {134, 204, 100, 221, 8},
      {168, 37, 145, 47, 14}
    },
    { {185, 202, 170, 229, 1},
      {14, 204, 62, 78, 14},
      {138, 117, 85, 57, 13},
      {119, 39, 195, 55, 0}
    },
    { {55, 38, 64, 234, 9},
      {165, 40, 43, 105, 3},
      {149, 112, 38, 78, 12},
      {201, 109, 65, 74, 5}
    },
    { {141, 86, 137, 137, 9},
      {214, 164, 13, 102, 2},
      {153, 25, 22, 171, 1},
      {70, 107, 2, 86, 11}
    },
    { {220, 110, 245, 96, 5},
      {183, 141, 74, 186, 12},
      {160, 106, 247, 99, 11},
      {53, 213, 43, 30, 13}
    },
    { {81, 225, 58, 109, 5},
      {13, 110, 86, 220, 4},
      {171, 101, 200, 120, 10},
      {35, 182, 167, 107, 0}
    },
    { {40, 201, 94, 212, 13},
      {46, 71, 245, 24, 10},
      {178, 183, 169, 49, 4},
      {81, 138, 254, 39, 4}
    },
    { {155, 25, 144, 102, 13},
      {78, 138, 67, 91, 9},
      {182, 96, 153, 141, 9},
      {157, 172, 37, 23, 2}
    },
    { {198, 37, 228, 219, 13},
      {165, 163, 193, 175, 7},
      {189, 178, 122, 70, 3},
      {239, 88, 60, 90, 5}
    },
    { {238, 221, 198, 163, 3},
      {230, 215, 50, 167, 11},
      {204, 86, 59, 183, 7},
      {222, 84, 206, 182, 7}
    },
    { {168, 123, 189, 36, 7},
      {95, 151, 110, 18, 12},
      {226, 75, 221, 225, 5},
      {52, 135, 110, 159, 10}
    },
    { {38, 247, 86, 43, 7},
      {229, 119, 122, 53, 1},
      {237, 70, 174, 246, 4},
      {138, 197, 238, 234, 7}
    },
    { {119, 95, 17, 146, 15},
      {212, 30, 233, 241, 11},
      {244, 152, 143, 174, 14},
      {216, 249, 119, 130, 11}
    },
    { {227, 232, 42, 118, 0},
      {9, 68, 182, 203, 13},
      {6, 229, 65, 124, 7},
      {189, 54, 210, 41, 0}
    },
    { {219, 116, 9, 26, 8},
      {83, 44, 133, 227, 1},
      {21, 137, 2, 237, 11},
      {140, 122, 19, 76, 10}
    },
    { {166, 92, 117, 179, 8},
      {240, 5, 163, 55, 15},
      {28, 218, 227, 166, 5},
      {254, 204, 90, 0, 15}
    },
    { {247, 162, 51, 66, 4},
      {145, 72, 120, 219, 5},
      {36, 44, 196, 94, 15},
      {173, 177, 225, 40, 9}
    },
    { {45, 210, 218, 190, 4},
      {234, 228, 254, 80, 3},
      {39, 213, 180, 187, 4},
      {192, 167, 242, 117, 7}
    },
    { {158, 138, 87, 237, 4},
      {186, 105, 90, 31, 10},
      {43, 126, 165, 23, 9},
      {95, 133, 169, 101, 13}
    },
    { {192, 186, 16, 180, 12},
      {73, 0, 219, 146, 10},
      {50, 208, 133, 208, 3},
      {84, 157, 176, 9, 2}
    },
    { {19, 70, 233, 230, 12},
      {56, 140, 79, 105, 7},
      {54, 121, 118, 44, 8},
      {233, 111, 35, 17, 12}
    },
    { {177, 22, 100, 69, 12},
      {104, 9, 105, 110, 4},
      {58, 34, 102, 136, 13},
      {39, 105, 105, 1, 6}
    },
    { {74, 118, 173, 59, 12},
      {83, 165, 207, 165, 5},
      {61, 203, 86, 229, 2},
      {170, 95, 58, 92, 10}
    },
    { {133, 254, 70, 49, 2},
      {225, 85, 154, 102, 8},
      {72, 198, 39, 250, 1},
      {22, 101, 154, 168, 7}
    },
    { {173, 105, 121, 110, 10},
      {187, 54, 39, 90, 13},
      {87, 105, 233, 107, 5},
      {181, 174, 70, 205, 13}
    },
    { {32, 21, 210, 100, 6},
      {104, 210, 98, 56, 0},
      {98, 100, 186, 128, 4},
      {1, 196, 100, 177, 6}
    },
    { {147, 181, 91, 6, 6},
      {121, 90, 84, 115, 1},
      {102, 13, 170, 220, 9},
      {140, 226, 165, 169, 14}
    },
    { {202, 157, 31, 171, 6},
      {82, 115, 86, 183, 11},
      {109, 95, 139, 149, 3},
      {222, 214, 172, 228, 10}
    },
    { {187, 35, 104, 210, 14},
      {35, 26, 237, 75, 7},
      {116, 177, 108, 77, 13},
      {237, 43, 117, 140, 4}
    },
    { {52, 107, 195, 95, 14},
      {185, 254, 233, 12, 9},
      {127, 172, 61, 98, 12},
      {147, 9, 119, 249, 13}
    },
    { {160, 207, 113, 248, 9},
      {52, 38, 187, 58, 14},
      {145, 248, 239, 48, 5},
      {117, 205, 214, 66, 12}
    },
    { {83, 31, 184, 90, 9},
      {68, 170, 141, 249, 13},
      {149, 161, 223, 140, 10},
      {185, 251, 21, 82, 2}
    },
    { {241, 127, 53, 41, 9},
      {85, 47, 43, 246, 12},
      {153, 74, 207, 232, 15},
      {54, 253, 79, 74, 10}
    },
    { {152, 80, 211, 144, 5},
      {118, 204, 192, 18, 2},
      {160, 156, 176, 161, 9},
      {68, 128, 51, 54, 14}
    },
    { {123, 240, 90, 34, 5},
      {103, 76, 118, 209, 1},
      {164, 69, 160, 253, 14},
      {136, 182, 227, 46, 6}
    },
    { {197, 168, 151, 126, 5},
      {157, 225, 210, 218, 9},
      {167, 238, 145, 90, 3},
      {149, 180, 184, 123, 9}
    },
    { {22, 216, 30, 157, 5},
      {204, 109, 212, 21, 10},
      {171, 151, 129, 182, 8},
      {90, 130, 187, 99, 3}
    },
    { {180, 132, 224, 47, 5},
      {172, 168, 114, 38, 5},
      {175, 64, 114, 18, 13},
      {166, 68, 225, 83, 5}
    },
    { {237, 52, 164, 150, 13},
      {207, 129, 225, 226, 7},
      {182, 146, 82, 203, 7},
      {228, 120, 120, 31, 3}
    },
    { {9, 146, 15, 177, 3},
      {86, 81, 158, 68, 2},
      {200, 223, 4, 153, 0},
      {66, 39, 152, 166, 10}
    },
    { {251, 90, 185, 188, 11},
      {94, 188, 175, 211, 14},
      {211, 217, 213, 173, 15},
      {124, 191, 83, 215, 10}
    },
    { {37, 6, 52, 30, 11},
      {140, 49, 169, 112, 5},
      {215, 130, 198, 10, 4},
      {160, 233, 88, 195, 1}
    },
    { {224, 145, 168, 72, 15},
      {68, 178, 117, 138, 4},
      {241, 33, 88, 144, 7},
      {37, 26, 228, 210, 2}
    },
    { {194, 115, 147, 11, 8},
      {81, 230, 9, 151, 1},
      {29, 12, 156, 228, 3},
      {142, 153, 6, 120, 10}
    },
    { {16, 43, 26, 208, 4},
      {1, 74, 204, 24, 10},
      {32, 181, 141, 64, 8},
      {81, 131, 53, 40, 0}
    },
    { {56, 215, 41, 175, 4},
      {90, 46, 126, 36, 7},
      {47, 89, 78, 177, 12},
      {226, 71, 231, 69, 10}
    },
    { {214, 63, 130, 165, 12},
      {201, 202, 75, 167, 10},
      {58, 84, 31, 198, 11},
      {94, 93, 37, 57, 3}
    },
    { {93, 176, 247, 42, 2},
      {243, 249, 18, 208, 5},
      {69, 78, 240, 219, 10},
      {160, 180, 137, 252, 15}
    },
    { {115, 36, 212, 252, 10},
      {41, 185, 163, 249, 2},
      {83, 242, 178, 76, 14},
      {73, 252, 89, 217, 4}
    },
    { {71, 220, 230, 148, 6},
      {232, 213, 208, 225, 14},
      {98, 150, 115, 190, 2},
      {120, 112, 186, 177, 7}
    },
    { {221, 26, 76, 114, 14},
      {226, 25, 207, 202, 9},
      {116, 227, 37, 139, 11},
      {149, 63, 57, 132, 7}
    },
    { {154, 222, 49, 181, 1},
      {94, 12, 154, 55, 14},
      {138, 216, 199, 181, 9},
      {126, 197, 147, 7, 10}
    },
    { {76, 25, 234, 75, 9},
      {230, 226, 5, 140, 13},
      {157, 37, 121, 131, 2},
      {179, 26, 4, 118, 7}
    },
    { {129, 149, 32, 65, 5},
      {68, 2, 80, 110, 4},
      {168, 32, 74, 152, 1},
      {39, 96, 164, 2, 2}
    },
    { {208, 173, 68, 239, 5},
      {45, 43, 82, 174, 11},
      {175, 114, 43, 80, 11},
      {215, 84, 173, 75, 4}
    },
    { {169, 3, 2, 54, 13},
      {14, 66, 235, 66, 1},
      {182, 196, 12, 9, 5},
      {132, 45, 116, 39, 0}
    },
    { {90, 99, 243, 101, 13},
      {63, 206, 75, 157, 4},
      {186, 108, 252, 101, 10},
      {43, 157, 39, 63, 12}
    },
    { {167, 56, 140, 168, 15},
      {197, 177, 103, 67, 10},
      {241, 83, 17, 206, 5},
      {92, 46, 104, 218, 3}
    },
    { {193, 252, 209, 46, 0},
      {121, 164, 18, 242, 9},
      {7, 72, 179, 248, 3},
      {148, 244, 130, 89, 14}
    },
    { {18, 162, 88, 245, 0},
      {41, 8, 158, 29, 2},
      {10, 241, 164, 84, 8},
      {75, 135, 145, 9, 4}
    },
    { {176, 210, 149, 151, 0},
      {88, 141, 184, 22, 3},
      {14, 154, 148, 176, 13},
      {198, 129, 219, 17, 10}
    },
    { {172, 158, 132, 125, 4},
      {202, 161, 250, 46, 8},
      {43, 226, 23, 147, 5},
      {23, 69, 248, 85, 3}
    },
    { {112, 105, 55, 71, 12},
      {25, 79, 97, 156, 13},
      {62, 46, 201, 96, 14},
      {179, 152, 111, 41, 8}
    },
    { {68, 181, 20, 179, 2},
      {193, 19, 146, 180, 3},
      {76, 210, 138, 210, 2},
      {194, 212, 156, 136, 3}
    },
    { {231, 77, 237, 224, 10},
      {176, 151, 39, 235, 14},
      {80, 123, 123, 46, 7},
      {125, 126, 78, 144, 13}
    },
    { {53, 29, 98, 66, 10},
      {224, 90, 33, 104, 13},
      {84, 36, 107, 138, 12},
      {177, 104, 69, 160, 7}
    },
    { {142, 125, 171, 62, 10},
      {219, 246, 135, 35, 13},
      {87, 205, 91, 231, 1},
      {188, 78, 22, 253, 11}
    },
    { {82, 219, 5, 133, 6},
      {88, 31, 88, 133, 10},
      {106, 26, 13, 180, 10},
      {90, 17, 175, 129, 10}
    },
    { {240, 135, 244, 59, 6},
      {32, 187, 250, 182, 5},
      {109, 194, 254, 16, 15},
      {166, 213, 253, 208, 4}
    },
    { {78, 144, 152, 110, 1},
      {206, 160, 22, 153, 1},
      {135, 97, 144, 151, 2},
      {137, 150, 128, 87, 3}
    },
    { {210, 100, 39, 89, 9},
      {21, 109, 129, 175, 4},
      {153, 174, 66, 100, 11},
      {47, 88, 27, 106, 8}
    },
    { {36, 194, 245, 253, 5},
      {188, 165, 250, 28, 6},
      {171, 250, 244, 50, 4},
      {99, 133, 250, 83, 13}
    },
    { {50, 142, 228, 41, 3},
      {36, 185, 58, 37, 12},
      {201, 66, 119, 20, 12},
      {58, 69, 201, 210, 4}
    },
    { {107, 62, 162, 144, 11},
      {71, 208, 169, 225, 14},
      {208, 148, 87, 205, 6},
      {120, 121, 80, 190, 2}
    },
    { {217, 65, 43, 194, 11},
      {22, 94, 5, 202, 7},
      {212, 61, 72, 41, 11},
      {229, 58, 7, 166, 8}
    },
    { {156, 165, 116, 197, 7},
      {175, 27, 80, 62, 6},
      {234, 50, 234, 83, 9},
      {103, 192, 173, 143, 5}
    },
    { {17, 109, 219, 74, 15},
      {53, 254, 69, 120, 9},
      {245, 45, 187, 104, 8},
      {145, 234, 39, 250, 12}
    },
    { {174, 87, 71, 134, 8},
      {250, 71, 41, 35, 3},
      {22, 30, 46, 167, 5},
      {204, 73, 78, 37, 15}
    },
    { {114, 224, 117, 97, 4},
      {49, 13, 114, 157, 4},
      {40, 106, 224, 116, 14},
      {43, 148, 235, 8, 12}
    },
    { {198, 150, 180, 86, 6},
      {200, 145, 216, 187, 5},
      {102, 162, 214, 150, 3},
      {173, 209, 184, 145, 3}
    },
    { {168, 197, 195, 205, 1},
      {62, 230, 48, 46, 2},
      {139, 60, 58, 49, 5},
      {71, 64, 198, 119, 12}
    },
    { {91, 21, 74, 119, 1},
      {110, 74, 134, 237, 1},
      {142, 229, 42, 141, 10},
      {139, 118, 21, 39, 6}
    },
    { {205, 163, 184, 200, 5},
      {135, 162, 92, 218, 6},
      {161, 49, 220, 91, 3},
      {101, 179, 164, 94, 1}
    },
    { {74, 204, 221, 214, 7},
      {62, 149, 212, 185, 11},
      {230, 187, 179, 53, 2},
      {217, 210, 186, 151, 12}
    },
    { {102, 82, 239, 170, 15},
      {244, 245, 111, 129, 7},
      {245, 95, 116, 166, 6},
      {232, 31, 106, 242, 15}
    },
    { {173, 218, 37, 160, 0},
      {210, 5, 58, 66, 14},
      {0, 90, 69, 187, 5},
      {116, 37, 202, 4, 11}
    },
    { {252, 230, 67, 94, 0},
      {187, 108, 184, 170, 1},
      {7, 172, 38, 115, 15},
      {133, 81, 211, 109, 13}
    },
    { {187, 145, 56, 91, 8},
      {66, 42, 181, 95, 5},
      {29, 161, 200, 157, 13},
      {175, 170, 213, 68, 2}
    },
    { {52, 217, 147, 145, 4},
      {208, 206, 240, 20, 10},
      {40, 156, 153, 178, 12},
      {82, 128, 247, 48, 11}
    },
    { {160, 29, 41, 117, 12},
      {88, 2, 231, 46, 12},
      {58, 233, 75, 128, 5},
      {55, 78, 116, 1, 10}
    },
    { {197, 91, 119, 176, 10},
      {240, 87, 139, 210, 14},
      {80, 222, 237, 170, 3},
      {116, 189, 30, 160, 15}
    },
    { {180, 103, 48, 193, 10},
      {129, 30, 41, 62, 6},
      {88, 48, 206, 98, 13},
      {103, 201, 71, 136, 1}
    },
    { {79, 215, 217, 189, 10},
      {250, 182, 159, 245, 2},
      {91, 217, 190, 191, 2},
      {74, 255, 150, 213, 15}
    },
    { {62, 239, 157, 76, 6},
      {155, 191, 124, 57, 8},
      {99, 43, 159, 119, 12},
      {25, 195, 239, 221, 9}
    },
    { {211, 16, 230, 71, 6},
      {104, 217, 64, 207, 5},
      {110, 38, 112, 140, 11},
      {175, 48, 41, 177, 6}
    },
    { {150, 212, 131, 130, 1},
      {212, 204, 16, 35, 3},
      {132, 28, 18, 182, 9},
      {204, 64, 131, 50, 11}
    },
    { {190, 66, 191, 212, 9},
      {158, 205, 173, 27, 6},
      {146, 191, 212, 39, 13},
      {109, 139, 91, 55, 9}
    },
    { {66, 18, 49, 102, 9},
      {92, 0, 11, 153, 5},
      {150, 104, 196, 132, 2},
      {169, 157, 0, 3, 10}
    },
    { {51, 202, 85, 219, 9},
      {52, 45, 185, 93, 11},
      {157, 186, 165, 60, 12},
      {219, 169, 219, 66, 12}
    },
    { {165, 14, 174, 0, 13},
      {132, 193, 109, 98, 12},
      {176, 7, 87, 10, 5},
      {52, 107, 104, 50, 1}
    },
    { {86, 110, 32, 220, 13},
      {141, 44, 201, 169, 14},
      {179, 176, 71, 102, 10},
      {121, 89, 51, 75, 1}
    },
    { {126, 177, 2, 164, 3},
      {207, 90, 50, 129, 2},
      {194, 84, 8, 215, 14},
      {72, 20, 197, 175, 3}
    },
    { {175, 51, 166, 149, 7},
      {207, 211, 232, 71, 6},
      {234, 150, 92, 207, 5},
      {110, 33, 124, 191, 3}
    },
    { {144, 39, 133, 201, 15},
      {21, 187, 73, 46, 2},
      {249, 58, 30, 64, 9},
      {71, 73, 45, 218, 8}
    },
    { {26, 175, 114, 194, 0},
      {35, 74, 24, 57, 15},
      {4, 52, 239, 85, 8},
      {249, 193, 133, 44, 4}
    },
    { {140, 72, 88, 22, 8},
      {170, 4, 133, 18, 9},
      {22, 129, 161, 35, 1},
      {148, 138, 18, 5, 5}
    },
    { {221, 184, 28, 187, 8},
      {195, 41, 151, 214, 11},
      {29, 211, 129, 219, 11},
      {214, 190, 153, 76, 3}
    },
    { {41, 92, 192, 64, 12},
      {98, 132, 97, 104, 8},
      {48, 32, 51, 169, 4},
      {17, 104, 98, 20, 6}
    },
    { {204, 21, 5, 90, 14},
      {210, 51, 193, 170, 1},
      {117, 170, 10, 131, 3},
      {133, 88, 60, 196, 11}
    },
    { {218, 107, 24, 244, 9},
      {15, 14, 143, 155, 10},
      {146, 241, 141, 101, 11},
      {93, 159, 23, 15, 0}
    },
    { {167, 135, 108, 37, 9},
      {172, 3, 63, 103, 4},
      {154, 67, 110, 30, 5},
      {46, 111, 204, 3, 5}
    },
    { {246, 183, 42, 152, 5},
      {197, 106, 252, 163, 6},
      {161, 149, 78, 214, 15},
      {108, 83, 245, 106, 3}
    },
    { {176, 8, 245, 30, 13},
      {60, 169, 225, 18, 13},
      {183, 138, 241, 0, 13},
      {180, 136, 121, 83, 12}
    },
    { {75, 168, 124, 205, 13},
      {47, 33, 85, 221, 14},
      {187, 51, 225, 93, 2},
      {123, 186, 168, 79, 4}
    },
    { {132, 228, 215, 70, 3},
      {189, 213, 16, 58, 1},
      {198, 46, 178, 114, 1},
      {133, 192, 138, 187, 13}
    },
    { {108, 145, 86, 98, 0},
      {226, 67, 50, 152, 1},
      {4, 102, 168, 147, 6},
      {129, 148, 204, 36, 7}
    },
    { {161, 89, 232, 111, 0},
      {104, 166, 38, 78, 13},
      {15, 97, 121, 168, 5},
      {183, 38, 70, 81, 6}
    },
    { {240, 101, 172, 214, 8},
      {9, 143, 165, 170, 7},
      {22, 179, 90, 96, 15},
      {229, 90, 95, 25, 0}
    },
    { {151, 176, 34, 145, 10},
      {193, 88, 145, 71, 6},
      {88, 148, 64, 222, 9},
      {110, 40, 145, 168, 3}
    },
    { {191, 36, 1, 206, 6},
      {155, 56, 96, 107, 3},
      {103, 56, 2, 79, 13},
      {205, 96, 97, 205, 9}
    },
    { {164, 202, 31, 226, 1},
      {148, 69, 62, 26, 11},
      {132, 127, 133, 50, 5},
      {213, 135, 202, 34, 9}
    },
    { {17, 222, 135, 135, 9},
      {92, 205, 25, 100, 11},
      {158, 30, 23, 184, 8},
      {210, 105, 139, 51, 10}
    },
    { {57, 73, 184, 51, 5},
      {6, 142, 230, 84, 13},
      {172, 193, 217, 41, 12},
      {178, 166, 119, 22, 0}
    },
    { {134, 25, 53, 96, 13},
      {212, 3, 67, 27, 12},
      {176, 106, 201, 134, 1},
      {61, 140, 44, 2, 11}
    },
    { {200, 75, 66, 10, 11},
      {38, 118, 9, 130, 9},
      {213, 4, 45, 33, 3},
      {148, 25, 6, 230, 4}
    },
    { {81, 136, 22, 43, 15},
      {4, 121, 83, 212, 9},
      {253, 70, 129, 24, 10},
      {146, 188, 169, 226, 0}
    },
    { {198, 116, 204, 159, 0},
      {233, 165, 132, 167, 3},
      {15, 147, 50, 230, 3},
      {206, 82, 26, 89, 7}
    },
    { {168, 22, 147, 66, 12},
      {82, 192, 105, 58, 1},
      {52, 44, 150, 129, 5},
      {133, 201, 96, 52, 10}
    },
    { {91, 182, 26, 190, 12},
      {75, 104, 223, 241, 3},
      {55, 213, 134, 221, 10},
      {200, 255, 177, 109, 2}
    },
    { {64, 57, 11, 138, 10},
      {81, 114, 5, 128, 11},
      {85, 29, 9, 192, 2},
      {208, 26, 4, 232, 10}
    },
    { {49, 229, 119, 99, 10},
      {49, 95, 51, 124, 5},
      {92, 110, 234, 120, 12},
      {163, 236, 207, 168, 12}
    },
    { {138, 181, 190, 128, 6},
      {67, 211, 84, 51, 6},
      {96, 23, 218, 213, 1},
      {108, 194, 172, 188, 2}
    },
    { {232, 127, 242, 172, 9},
      {111, 230, 43, 178, 14},
      {147, 84, 255, 225, 7},
      {116, 221, 70, 127, 6}
    },
    { {133, 144, 118, 85, 9},
      {236, 65, 145, 94, 4},
      {154, 166, 224, 154, 1},
      {39, 168, 152, 35, 7}
    },
    { {173, 4, 85, 41, 5},
      {182, 33, 98, 118, 0},
      {169, 74, 162, 11, 5},
      {6, 228, 104, 70, 13}
    },
    { {94, 100, 156, 243, 5},
      {135, 141, 198, 189, 3},
      {172, 243, 146, 103, 10},
      {203, 214, 59, 30, 1}
    },
    { {113, 188, 174, 157, 13},
      {77, 233, 245, 228, 14},
      {187, 151, 83, 216, 14},
      {114, 122, 249, 123, 2}
    },
    { {202, 194, 32, 47, 13},
      {14, 36, 91, 135, 5},
      {191, 64, 68, 53, 3},
      {174, 29, 162, 71, 0}
    },
    { {237, 109, 12, 225, 0},
      {131, 7, 38, 238, 10},
      {8, 115, 11, 107, 7},
      {87, 118, 78, 12, 1}
    },
    { {211, 179, 54, 142, 8},
      {73, 107, 25, 211, 7},
      {23, 22, 204, 220, 11},
      {236, 185, 141, 105, 2}
    },
    { {88, 251, 152, 132, 4},
      {75, 142, 92, 144, 10},
      {34, 17, 157, 241, 10},
      {80, 147, 167, 29, 2}
    },
    { {66, 176, 73, 175, 12},
      {121, 32, 87, 133, 3},
      {63, 89, 32, 212, 2},
      {202, 30, 160, 73, 14}
    },
    { {165, 172, 222, 241, 10},
      {161, 209, 183, 126, 10},
      {88, 247, 179, 90, 5},
      {87, 238, 216, 184, 5}
    },
    { {126, 74, 98, 45, 6},
      {170, 124, 106, 133, 12},
      {107, 68, 101, 39, 14},
      {58, 21, 99, 229, 5}
    },
    { {93, 233, 95, 78, 9},
      {191, 111, 21, 216, 9},
      {151, 47, 169, 123, 10},
      {145, 186, 143, 111, 13}
    },
    { {34, 197, 24, 23, 9},
      {12, 6, 181, 53, 1},
      {158, 129, 138, 52, 4},
      {138, 202, 214, 3, 0}
    },
    { {115, 117, 108, 186, 5},
      {101, 47, 230, 225, 7},
      {165, 211, 106, 236, 14},
      {232, 118, 127, 74, 6}
    },
    { {80, 47, 87, 216, 11},
      {53, 123, 137, 184, 10},
      {209, 190, 175, 64, 10},
      {81, 217, 29, 234, 12}
    },
    { {154, 160, 25, 211, 11},
      {23, 24, 149, 31, 3},
      {220, 185, 128, 85, 9},
      {207, 138, 145, 142, 8}
    },
    { {61, 180, 129, 166, 15},
      {223, 152, 115, 96, 3},
      {246, 88, 18, 219, 12},
      {192, 108, 225, 159, 11}
    },
    { {39, 218, 159, 156, 8},
      {216, 229, 189, 81, 10},
      {19, 159, 149, 190, 4},
      {88, 171, 218, 113, 11}
    },
    { {58, 103, 134, 237, 6},
      {11, 255, 106, 45, 2},
      {107, 118, 30, 101, 12},
      {75, 69, 111, 253, 0}
    },
    { {132, 55, 15, 143, 6},
      {217, 115, 76, 38, 3},
      {111, 31, 14, 194, 1},
      {198, 67, 44, 233, 11}
    },
    { {53, 66, 74, 6, 5},
      {172, 76, 108, 64, 1},
      {166, 5, 36, 42, 12},
      {128, 35, 99, 35, 5}
    },
    { {131, 37, 139, 194, 7},
      {21, 210, 68, 107, 3},
      {228, 61, 26, 76, 1},
      {205, 98, 36, 186, 8}
    },
    { {6, 72, 246, 57, 4},
      {160, 229, 194, 21, 12},
      {41, 198, 241, 38, 0},
      {58, 132, 58, 112, 5}
    },
    { {46, 148, 213, 97, 12},
      {242, 129, 115, 61, 0},
      {56, 106, 178, 151, 4},
      {11, 204, 232, 20, 15}
    },
    { {72, 50, 160, 103, 2},
      {75, 144, 10, 140, 5},
      {78, 96, 84, 193, 2},
      {163, 21, 0, 157, 2}
    },
    { {228, 117, 33, 147, 1},
      {213, 6, 160, 166, 7},
      {140, 152, 74, 226, 7},
      {230, 80, 86, 10, 11}
    },
    { {110, 253, 140, 158, 9},
      {207, 167, 181, 161, 11},
      {151, 147, 27, 247, 6},
      {216, 90, 222, 95, 3}
    },
    { {0, 211, 242, 119, 9},
      {108, 198, 155, 28, 5},
      {158, 228, 252, 176, 0},
      {163, 141, 150, 51, 6}
    },
    { {99, 216, 68, 68, 11},
      {108, 21, 49, 201, 8},
      {210, 34, 33, 188, 6},
      {25, 56, 202, 131, 6}
    },
    { {249, 28, 183, 98, 7},
      {86, 217, 98, 250, 13},
      {228, 110, 211, 137, 15},
      {181, 244, 105, 182, 10}
    },
    { {148, 34, 208, 61, 7},
      {173, 184, 202, 22, 0},
      {235, 192, 180, 66, 9},
      {6, 133, 49, 219, 5}
    },
    { {187, 200, 20, 31, 6},
      {10, 61, 240, 87, 9},
      {111, 130, 129, 61, 13},
      {158, 160, 251, 197, 0}
    },
    { {69, 152, 237, 76, 14},
      {248, 177, 85, 200, 12},
      {115, 43, 113, 154, 2},
      {49, 58, 168, 209, 15}
    },
    { {83, 226, 244, 224, 9},
      {37, 141, 27, 217, 6},
      {144, 114, 244, 124, 10},
      {105, 189, 139, 26, 4}
    },
    { {241, 178, 125, 130, 9},
      {117, 9, 61, 210, 7},
      {148, 27, 228, 216, 15},
      {228, 187, 201, 10, 14}
    },
    { {42, 202, 178, 62, 9},
      {14, 228, 187, 17, 13},
      {151, 196, 213, 53, 4},
      {184, 141, 210, 119, 0}
    },
    { {123, 122, 214, 143, 9},
      {111, 237, 41, 213, 11},
      {159, 22, 181, 237, 14},
      {218, 185, 75, 127, 6}
    },
    { {130, 225, 42, 209, 13},
      {5, 70, 213, 15, 6},
      {184, 181, 72, 116, 1},
      {111, 10, 182, 42, 0}
    },
    { {37, 245, 188, 164, 11},
      {205, 151, 55, 112, 6},
      {210, 83, 218, 250, 4},
      {96, 238, 206, 155, 3}
    },
    { {150, 173, 51, 250, 11},
      {149, 122, 147, 59, 15},
      {213, 252, 203, 86, 9},
      {253, 204, 149, 234, 9}
    },
    { {116, 221, 218, 89, 11},
      {228, 254, 181, 188, 8},
      {217, 165, 187, 178, 14},
      {19, 218, 215, 242, 7}
    },
    { {190, 51, 16, 82, 7},
      {199, 26, 232, 27, 1},
      {228, 160, 140, 199, 13},
      {141, 129, 117, 142, 3}
    },
    { {175, 106, 142, 227, 12},
      {131, 197, 111, 79, 11},
      {60, 119, 21, 111, 5},
      {223, 47, 106, 60, 1}
    },
    { {80, 157, 15, 85, 14},
      {88, 91, 213, 172, 8},
      {122, 175, 11, 144, 10},
      {19, 90, 189, 161, 10}
    },
    { {90, 168, 135, 202, 13},
      {23, 233, 81, 137, 11},
      {181, 62, 17, 85, 10},
      {217, 24, 169, 126, 8}
    },
    { {248, 216, 14, 113, 13},
      {70, 77, 247, 142, 8},
      {184, 231, 1, 177, 15},
      {23, 30, 251, 38, 2}
    },
    { {18, 41, 212, 58, 0},
      {33, 171, 130, 17, 9},
      {5, 194, 185, 68, 8},
      {152, 132, 29, 88, 4}
    },
    { {58, 213, 230, 98, 8},
      {98, 207, 51, 41, 5},
      {20, 102, 122, 181, 12},
      {169, 76, 207, 52, 6}
    },
    { {68, 63, 156, 94, 2},
      {201, 179, 140, 184, 9},
      {71, 163, 159, 194, 2},
      {145, 211, 28, 217, 3}
    },
    { {53, 224, 26, 247, 2},
      {137, 92, 182, 92, 3},
      {78, 245, 128, 122, 12},
      {195, 166, 211, 169, 1}
    },
    { {82, 84, 77, 114, 6},
      {112, 29, 198, 169, 1},
      {100, 235, 34, 164, 10},
      {137, 86, 59, 128, 14}
    },
    { {8, 222, 33, 65, 9},
      {86, 4, 25, 44, 12},
      {152, 40, 71, 177, 0},
      {51, 73, 130, 6, 10}
    },
    { {151, 233, 242, 229, 5},
      {173, 206, 82, 95, 14},
      {170, 116, 249, 126, 9},
      {127, 164, 167, 59, 5}
    },
    { {117, 185, 123, 135, 5},
      {253, 74, 116, 212, 15},
      {174, 29, 233, 218, 14},
      {242, 178, 229, 43, 15}
    },
    { {28, 149, 63, 106, 13},
      {214, 107, 87, 56, 5},
      {181, 111, 202, 147, 8},
      {161, 206, 173, 102, 11}
    },
    { {72, 136, 208, 174, 15},
      {46, 176, 83, 144, 11},
      {247, 80, 177, 17, 2},
      {208, 156, 160, 215, 4}
    },
    { {35, 82, 133, 125, 8},
      {88, 165, 171, 77, 0},
      {27, 234, 20, 172, 4},
      {11, 45, 90, 81, 10}
    },
    { {145, 10, 12, 159, 8},
      {8, 41, 141, 70, 11},
      {31, 147, 5, 8, 9},
      {214, 43, 25, 65, 0}
    },
    { {114, 106, 253, 204, 4},
      {57, 173, 108, 153, 14},
      {35, 59, 245, 100, 14},
      {121, 147, 107, 89, 12}
    },
    { {70, 182, 223, 184, 12},
      {241, 225, 223, 177, 2},
      {49, 223, 182, 214, 2},
      {72, 223, 184, 120, 15}
    },
    { {55, 30, 152, 201, 12},
      {192, 168, 109, 125, 10},
      {57, 49, 151, 142, 12},
      {91, 235, 97, 80, 3}
    },
    { {242, 133, 64, 180, 10},
      {40, 26, 179, 163, 2},
      {82, 208, 42, 20, 15},
      {76, 92, 213, 129, 4}
    },
    { {155, 53, 4, 69, 10},
      {75, 27, 1, 111, 0},
      {90, 34, 10, 205, 9},
      {15, 104, 13, 141, 2}
    },
    { {138, 236, 145, 248, 13},
      {23, 164, 211, 59, 10},
      {177, 248, 147, 117, 1},
      {93, 204, 178, 94, 8}
    },
    { {173, 85, 173, 77, 15},
      {222, 183, 101, 110, 4},
      {251, 43, 90, 171, 5},
      {39, 106, 110, 215, 11}
    },
    { {42, 24, 202, 208, 2},
      {98, 208, 164, 9, 10},
      {64, 181, 49, 133, 4},
      {89, 2, 80, 180, 6}
    },
    { {237, 2, 60, 165, 10},
      {138, 17, 47, 214, 6},
      {90, 83, 196, 11, 7},
      {102, 191, 72, 133, 1}
    },
    { {190, 137, 112, 155, 1},
      {166, 42, 176, 23, 15},
      {141, 144, 233, 23, 13},
      {254, 128, 213, 70, 5}
    },
    { {66, 233, 185, 200, 9},
      {21, 166, 21, 153, 14},
      {145, 57, 217, 116, 2},
      {121, 154, 134, 90, 8}
    },
    { {224, 185, 52, 114, 9},
      {69, 3, 179, 154, 13},
      {148, 226, 201, 208, 7},
      {181, 156, 220, 10, 2}
    },
    { {135, 32, 220, 54, 11},
      {173, 145, 135, 83, 1},
      {214, 195, 176, 78, 1},
      {140, 174, 24, 155, 5}
    },
    { {93, 132, 96, 137, 7},
      {166, 56, 80, 228, 6},
      {233, 16, 98, 27, 10},
      {98, 112, 161, 198, 5}
    },
    { {144, 204, 205, 2, 15},
      {52, 157, 85, 34, 9},
      {244, 11, 51, 48, 9},
      {148, 74, 171, 146, 12}
    },
    { {140, 134, 212, 168, 8},
      {162, 161, 27, 50, 2},
      {17, 82, 182, 19, 1},
      {68, 205, 136, 84, 5}
    },
    { {120, 169, 76, 36, 12},
      {43, 11, 119, 128, 8},
      {50, 67, 41, 81, 14},
      {16, 30, 237, 13, 4}
    },
    { {182, 101, 251, 203, 12},
      {177, 238, 101, 63, 7},
      {61, 61, 250, 102, 13},
      {239, 202, 103, 120, 13}
    },
    { {39, 167, 32, 26, 6},
      {129, 50, 248, 97, 5},
      {101, 128, 78, 94, 4},
      {168, 97, 244, 200, 1}
    },
    { {193, 202, 69, 13, 5},
      {60, 37, 88, 198, 8},
      {171, 10, 37, 56, 3},
      {22, 49, 170, 67, 12}
    },
    { {75, 38, 242, 6, 13},
      {47, 192, 73, 241, 5},
      {182, 4, 246, 77, 2},
      {168, 249, 32, 63, 4}
    },
    { {146, 37, 46, 39, 11},
      {13, 91, 7, 39, 5},
      {222, 71, 74, 68, 9},
      {174, 78, 13, 171, 0}
    },
    { {240, 175, 98, 24, 8},
      {33, 106, 185, 162, 12},
      {17, 132, 111, 80, 15},
      {52, 89, 213, 104, 4}
    },
    { {50, 161, 124, 11, 6},
      {33, 59, 116, 21, 5},
      {109, 3, 232, 84, 12},
      {170, 130, 237, 200, 4}
    },
    { {107, 9, 58, 162, 14},
      {2, 82, 103, 209, 15},
      {116, 85, 201, 13, 6},
      {248, 190, 100, 164, 0}
    },
    { {134, 231, 189, 205, 5},
      {157, 167, 92, 63, 6},
      {171, 59, 222, 118, 1},
      {111, 195, 174, 91, 9}
    },
    { {70, 76, 128, 181, 11},
      {140, 148, 131, 165, 10},
      {218, 208, 19, 38, 2},
      {90, 92, 18, 147, 1}
    },
    { {110, 146, 188, 233, 7},
      {198, 177, 126, 157, 6},
      {233, 115, 212, 151, 6},
      {107, 151, 232, 214, 3}
    },
    { {10, 141, 210, 173, 4},
      {42, 226, 82, 53, 10},
      {43, 84, 187, 21, 0},
      {90, 196, 164, 117, 4}
    },
    { {133, 26, 254, 234, 9},
      {228, 225, 15, 90, 15},
      {149, 119, 245, 138, 1},
      {245, 175, 8, 114, 7}
    },
    { {102, 122, 112, 73, 9},
      {229, 36, 41, 157, 12},
      {153, 32, 229, 230, 6},
      {59, 153, 66, 74, 7}
    },
    { {113, 49, 97, 102, 13},
      {125, 10, 99, 200, 5},
      {182, 104, 104, 200, 14},
      {161, 60, 101, 11, 14}
    },
    { {197, 119, 160, 82, 15},
      {197, 150, 201, 234, 5},
      {244, 160, 94, 234, 3},
      {165, 121, 54, 154, 3}
    },
    { {224, 70, 76, 187, 12},
      {32, 37, 239, 166, 3},
      {61, 211, 38, 32, 7},
      {198, 95, 122, 64, 4}
    },
    { {244, 9, 85, 141, 10},
      {184, 59, 33, 150, 10},
      {91, 26, 169, 2, 15},
      {86, 152, 77, 193, 13}
    },
    { {34, 79, 148, 169, 9},
      {4, 167, 43, 53, 10},
      {153, 82, 159, 36, 4},
      {90, 205, 78, 82, 0}
    },
    { {229, 88, 200, 244, 13},
      {236, 132, 231, 202, 10},
      {178, 241, 49, 170, 7},
      {85, 62, 114, 19, 7}
    },
    { {243, 18, 209, 192, 11},
      {116, 152, 41, 219, 2},
      {208, 56, 180, 140, 15},
      {77, 185, 65, 146, 14}
    },
    { {41, 114, 88, 124, 11},
      {111, 52, 175, 88, 0},
      {211, 225, 164, 233, 4},
      {1, 175, 82, 207, 6}
    },
    { {65, 179, 38, 98, 4},
      {65, 67, 90, 200, 5},
      {36, 102, 76, 216, 2},
      {161, 53, 172, 40, 2}
    },
    { {218, 87, 244, 214, 12},
      {106, 143, 201, 187, 7},
      {54, 178, 254, 165, 11},
      {237, 217, 63, 21, 6}
    },
    { {186, 1, 109, 186, 9},
      {54, 43, 167, 3, 7},
      {149, 219, 104, 5, 13},
      {236, 14, 93, 70, 12}
    },
    { {53, 137, 198, 176, 5},
      {164, 203, 242, 64, 10},
      {160, 214, 57, 26, 12},
      {80, 36, 253, 50, 5}
    },
    { {223, 69, 8, 189, 5},
      {142, 46, 198, 231, 2},
      {171, 209, 10, 47, 11},
      {78, 118, 55, 71, 1}
    },
    { {50, 82, 34, 250, 8},
      {64, 108, 171, 9, 7},
      {21, 244, 68, 164, 12},
      {233, 13, 83, 96, 2}
    },
    { {87, 54, 121, 125, 4},
      {249, 40, 206, 253, 4},
      {43, 233, 230, 206, 10},
      {43, 247, 49, 73, 15}
    },
    { {100, 77, 74, 190, 10},
      {168, 118, 167, 160, 11},
      {87, 213, 43, 34, 6},
      {208, 94, 86, 225, 5}
    },
    { {13, 189, 14, 79, 10},
      {203, 115, 21, 108, 9},
      {95, 39, 11, 219, 0},
      {147, 106, 140, 237, 3}
    },
    { {20, 18, 155, 246, 13},
      {220, 200, 207, 24, 3},
      {182, 253, 148, 130, 8},
      {193, 143, 49, 51, 11}
    },
    { {160, 137, 141, 158, 1},
      {28, 163, 180, 2, 11},
      {135, 155, 25, 16, 5},
      {212, 2, 220, 83, 8}
    },
    { {116, 80, 182, 226, 11},
      {196, 221, 35, 152, 7},
      {212, 118, 208, 162, 14},
      {225, 156, 75, 178, 3}
    },
    { {224, 228, 28, 54, 7},
      {13, 21, 246, 178, 1},
      {230, 195, 130, 112, 7},
      {132, 214, 250, 139, 0}
    },
    { {105, 149, 101, 253, 11},
      {126, 51, 179, 236, 6},
      {219, 250, 106, 153, 6},
      {99, 124, 220, 199, 14}
    },
    { {103, 160, 233, 22, 8},
      {185, 128, 181, 193, 5},
      {22, 137, 112, 94, 6},
      {168, 58, 208, 25, 13}
    },
    { {25, 252, 180, 77, 12},
      {75, 173, 81, 124, 12},
      {59, 34, 211, 249, 8},
      {51, 232, 171, 93, 2}
    },
    { {240, 254, 74, 108, 6},
      {105, 124, 126, 170, 8},
      {99, 101, 39, 240, 15},
      {21, 87, 227, 233, 6}
    },
    { {78, 229, 220, 11, 14},
      {163, 183, 85, 181, 1},
      {125, 3, 186, 119, 2},
      {138, 218, 174, 220, 5}
    },
    { {178, 90, 177, 231, 7},
      {92, 156, 106, 31, 15},
      {238, 120, 213, 164, 13},
      {255, 133, 99, 147, 10}
    },
    { {84, 197, 166, 143, 13},
      {140, 239, 81, 164, 7},
      {191, 22, 90, 50, 10},
      {226, 88, 175, 115, 1}
    },
    { {247, 21, 47, 212, 3},
      {220, 91, 164, 235, 6},
      {194, 191, 74, 142, 15},
      {109, 114, 93, 163, 11}
    },
    { {79, 145, 125, 232, 10},
      {242, 51, 23, 217, 6},
      {81, 123, 232, 159, 2},
      {105, 190, 140, 196, 15}
    },
    { {9, 53, 24, 238, 6},
      {75, 50, 70, 120, 3},
      {103, 113, 138, 201, 0},
      {193, 230, 36, 205, 2}
    },
    { {170, 77, 229, 141, 6},
      {58, 183, 96, 39, 14},
      {107, 26, 123, 37, 5},
      {126, 64, 110, 213, 12}
    },
    { {251, 189, 163, 100, 14},
      {91, 218, 115, 235, 12},
      {114, 108, 91, 221, 15},
      {61, 124, 229, 189, 10}
    },
    { {194, 140, 79, 157, 13},
      {60, 97, 213, 167, 10},
      {187, 159, 35, 20, 3},
      {94, 90, 184, 99, 12}
    },
    { {221, 221, 90, 236, 8},
      {234, 110, 23, 250, 10},
      {19, 117, 171, 187, 11},
      {85, 254, 135, 101, 7}
    },
    { {154, 123, 45, 105, 4},
      {83, 47, 78, 15, 12},
      {41, 107, 77, 229, 9},
      {63, 7, 47, 76, 10}
    },
    { {218, 46, 220, 145, 1},
      {39, 137, 140, 183, 10},
      {136, 147, 183, 69, 11},
      {94, 211, 25, 30, 4}
    },
    { {18, 21, 66, 25, 5},
      {100, 106, 192, 37, 0},
      {169, 132, 42, 132, 8},
      {10, 64, 53, 98, 6}
    },
    { {58, 253, 126, 65, 13},
      {103, 79, 117, 61, 12},
      {184, 39, 235, 245, 12},
      {59, 202, 239, 46, 6}
    },
    { {210, 145, 48, 33, 12},
      {64, 10, 83, 151, 4},
      {56, 64, 200, 148, 11},
      {46, 156, 165, 0, 2}
    },
    { {136, 91, 123, 237, 6},
      {122, 118, 78, 30, 14},
      {107, 125, 237, 161, 1},
      {119, 135, 38, 229, 14}
    },
    { {255, 26, 148, 3, 13},
      {198, 137, 105, 215, 9},
      {188, 2, 149, 143, 15},
      {158, 185, 105, 22, 3}
    },
    { {138, 210, 5, 218, 10},
      {82, 53, 153, 11, 3},
      {85, 186, 4, 181, 1},
      {205, 9, 154, 196, 10}
    },
    { {124, 254, 229, 86, 14},
      {251, 157, 249, 168, 13},
      {118, 170, 119, 243, 14},
      {177, 89, 251, 157, 15}
    },
    { {15, 29, 188, 45, 9},
      {206, 163, 7, 117, 12},
      {155, 67, 219, 143, 0},
      {58, 238, 12, 87, 3}
    },
    { {173, 125, 51, 119, 9},
      {223, 70, 163, 126, 13},
      {158, 236, 203, 235, 5},
      {183, 236, 86, 47, 11}
    },
    { {94, 35, 218, 148, 5},
      {175, 202, 204, 145, 2},
      {162, 149, 188, 71, 10},
      {72, 147, 53, 63, 5}
    },
    { {202, 135, 100, 114, 13},
      {38, 3, 219, 171, 5},
      {180, 226, 110, 21, 3},
      {173, 93, 188, 6, 4}
    },
    { {42, 52, 86, 226, 7},
      {103, 81, 98, 57, 3},
      {228, 118, 162, 197, 4},
      {201, 196, 104, 174, 6}
    },
    { {179, 128, 104, 125, 6},
      {40, 56, 246, 79, 4},
      {107, 225, 96, 28, 13},
      {47, 38, 241, 193, 4}
    },
    { {73, 224, 165, 31, 6},
      {27, 181, 208, 196, 5},
      {111, 138, 80, 121, 2},
      {162, 48, 186, 221, 8}
    },
    { {144, 226, 83, 62, 9},
      {61, 108, 155, 18, 1},
      {151, 204, 164, 112, 9},
      {132, 141, 147, 107, 12}
    },
    { {221, 1, 6, 99, 13},
      {134, 75, 67, 206, 1},
      {188, 102, 8, 11, 11},
      {135, 60, 45, 38, 1}
    },
    { {118, 160, 92, 144, 8},
      {161, 9, 181, 145, 2},
      {16, 147, 160, 86, 14},
      {72, 154, 217, 8, 5}
    },
    { {14, 217, 3, 223, 6},
      {218, 118, 208, 13, 11},
      {111, 188, 9, 183, 0},
      {219, 0, 182, 229, 11}
    },
    { {43, 112, 183, 113, 13},
      {87, 197, 227, 93, 4},
      {184, 238, 208, 237, 4},
      {43, 172, 122, 62, 10}
    },
    { {78, 140, 226, 119, 3},
      {174, 208, 146, 173, 13},
      {206, 228, 115, 23, 2},
      {187, 84, 144, 183, 5}
    },
    { {164, 109, 4, 140, 4},
      {137, 39, 96, 34, 10},
      {35, 18, 11, 98, 5},
      {84, 64, 110, 73, 1}
    },
    { {174, 72, 136, 91, 10},
      {130, 180, 165, 15, 9},
      {93, 161, 17, 39, 5},
      {159, 10, 82, 212, 1}
    },
    { {10, 163, 70, 143, 3},
      {47, 115, 24, 5, 3},
      {207, 22, 44, 85, 0},
      {202, 1, 140, 239, 4}
    },
    { {106, 225, 111, 19, 7},
      {55, 87, 244, 133, 5},
      {236, 143, 104, 117, 6},
      {170, 18, 254, 174, 12}
    },
    { {149, 35, 13, 186, 13},
      {149, 43, 207, 66, 3},
      {181, 219, 12, 74, 9},
      {196, 47, 61, 74, 9}
    },
    { {228, 136, 181, 77, 4},
      {152, 161, 112, 158, 12},
      {43, 42, 209, 18, 7},
      {55, 144, 232, 81, 9}
    },
    { {121, 10, 75, 108, 10},
      {58, 120, 47, 200, 8},
      {83, 109, 37, 9, 14},
      {17, 63, 65, 229, 12}
    },
    { {64, 35, 127, 147, 9},
      {53, 67, 141, 148, 7},
      {156, 159, 236, 64, 2},
      {226, 155, 28, 42, 12}
    },
    { {45, 191, 41, 200, 3},
      {215, 50, 60, 104, 14},
      {193, 57, 79, 219, 4},
      {113, 99, 196, 206, 11}
    },
    { {202, 214, 76, 54, 0},
      {106, 5, 158, 163, 1},
      {6, 195, 38, 181, 3},
      {140, 87, 154, 5, 6}
    },
    { {211, 175, 89, 151, 10},
      {57, 26, 157, 247, 11},
      {94, 153, 175, 92, 11},
      {222, 251, 149, 137, 12}
    },
    { {157, 46, 167, 143, 5},
      {159, 233, 72, 102, 15},
      {175, 30, 87, 75, 9},
      {246, 97, 41, 127, 9}
    },
    { {34, 211, 98, 6, 7},
      {108, 86, 120, 1, 5},
      {230, 4, 108, 180, 4},
      {168, 1, 230, 163, 6}
    },
    { {204, 169, 43, 134, 6},
      {155, 82, 84, 130, 15},
      {102, 29, 73, 83, 3},
      {244, 18, 164, 173, 9}
    },
    { {85, 43, 159, 167, 1},
      {157, 203, 14, 212, 11},
      {142, 95, 157, 74, 10},
      {210, 183, 13, 59, 9}
    },
    { {203, 178, 173, 205, 11},
      {95, 177, 29, 207, 6},
      {219, 59, 84, 221, 3},
      {111, 59, 136, 223, 10}
    },
    { {68, 250, 70, 198, 7},
      {237, 85, 88, 136, 11},
      {230, 54, 37, 242, 2},
      {209, 17, 170, 171, 7}
    },
    { {231, 38, 143, 185, 7},
      {149, 241, 238, 231, 2},
      {233, 223, 22, 78, 7},
      {78, 119, 120, 250, 9}
    },
    { {36, 175, 23, 170, 2},
      {145, 115, 58, 48, 11},
      {69, 94, 143, 82, 4},
      {208, 197, 204, 232, 9}
    },
    { {67, 232, 194, 173, 10},
      {41, 244, 19, 197, 10},
      {91, 84, 49, 124, 2},
      {90, 60, 130, 249, 4}
    },
    { {145, 205, 26, 71, 13},
      {12, 78, 85, 126, 9},
      {190, 37, 139, 56, 9},
      {151, 234, 167, 35, 0}
    },
    { {104, 5, 2, 194, 10},
      {2, 82, 33, 168, 3},
      {84, 52, 10, 1, 6},
      {193, 88, 68, 164, 0}
    },
    { {5, 213, 122, 29, 10},
      {232, 118, 149, 116, 4},
      {91, 133, 234, 186, 0},
      {34, 234, 150, 225, 7}
    },
    { {88, 51, 178, 20, 8},
      {75, 202, 137, 144, 4},
      {18, 132, 220, 193, 10},
      {32, 153, 21, 61, 2}
    },
    { {207, 23, 24, 178, 4},
      {194, 2, 206, 243, 3},
      {36, 209, 142, 143, 3},
      {204, 247, 52, 4, 3}
    },
    { {86, 20, 124, 139, 2},
      {224, 57, 4, 181, 7},
      {77, 19, 226, 134, 10},
      {234, 210, 9, 192, 7}
    },
    { {254, 81, 189, 45, 1},
      {222, 175, 38, 151, 4},
      {139, 75, 216, 167, 15},
      {46, 150, 79, 87, 11}
    },
    { {48, 238, 165, 168, 4},
      {17, 173, 122, 32, 14},
      {33, 90, 87, 112, 12},
      {112, 69, 235, 88, 8}
    },
    { {94, 27, 78, 166, 2},
      {234, 91, 14, 129, 11},
      {70, 87, 45, 135, 10},
      {216, 23, 13, 165, 7}
    },
    { {37, 57, 142, 52, 3},
      {205, 211, 166, 64, 8},
      {194, 199, 25, 202, 4},
      {16, 38, 92, 187, 3}
    },
    { {71, 238, 53, 187, 2},
      {145, 53, 154, 245, 15},
      {77, 218, 199, 126, 2},
      {250, 245, 154, 200, 9}
    },
    { {105, 226, 147, 152, 11},
      {23, 244, 185, 208, 2},
      {209, 156, 148, 121, 6},
      {64, 185, 210, 254, 8}
    },
    { {203, 235, 3, 139, 4},
      {19, 102, 88, 199, 11},
      {45, 28, 13, 125, 3},
      {222, 49, 166, 108, 8}
    },
    { {151, 33, 193, 117, 9},
      {189, 138, 131, 79, 0},
      {154, 232, 56, 78, 9},
      {15, 44, 21, 27, 13}
    },
    { {117, 81, 72, 151, 9},
      {236, 14, 165, 196, 3},
      {158, 145, 40, 170, 14},
      {194, 58, 87, 3, 7}
    },
    { {87, 243, 99, 229, 3},
      {253, 94, 26, 205, 6},
      {202, 124, 108, 254, 10},
      {107, 53, 135, 171, 15}
    },
    { {174, 100, 20, 104, 0},
      {131, 37, 34, 59, 0},
      {1, 98, 130, 103, 5},
      {13, 196, 74, 76, 1}
    },
    { {126, 147, 161, 155, 5},
      {214, 170, 248, 133, 7},
      {173, 152, 92, 151, 14},
      {234, 17, 245, 86, 11}
    },
    { {169, 131, 209, 245, 2},
      {58, 146, 186, 94, 2},
      {74, 248, 188, 25, 5},
      {71, 165, 212, 149, 12}
    },
    { {100, 91, 1, 175, 9},
      {220, 38, 43, 132, 11},
      {159, 88, 13, 162, 6},
      {210, 29, 70, 67, 11}
    },
    { {13, 167, 121, 86, 5},
      {191, 2, 220, 120, 5},
      {166, 169, 238, 91, 0},
      {161, 227, 180, 15, 13}
    },
    { {114, 16, 159, 147, 13},
      {84, 201, 229, 149, 3},
      {188, 159, 144, 132, 14},
      {202, 154, 121, 50, 10}
    },
    { {141, 12, 71, 198, 11},
      {190, 81, 1, 106, 11},
      {214, 62, 35, 11, 1},
      {213, 104, 8, 167, 13}
    },
    { {109, 229, 76, 66, 3},
      {167, 23, 52, 232, 1},
      {196, 35, 42, 123, 6},
      {129, 114, 206, 142, 5}
    },
    { {241, 35, 191, 102, 11},
      {29, 219, 47, 218, 5},
      {214, 111, 220, 72, 15},
      {165, 191, 77, 187, 8}
    },
    { {180, 231, 234, 99, 7},
      {165, 222, 126, 46, 5},
      {236, 101, 126, 114, 13},
      {167, 71, 231, 186, 5}
    },
    { {229, 224, 226, 108, 5},
      {173, 228, 114, 202, 4},
      {163, 100, 112, 122, 7},
      {37, 52, 226, 123, 5}
    },
    { {90, 49, 4, 158, 15},
      {79, 59, 193, 129, 3},
      {247, 146, 8, 197, 10},
      {200, 24, 61, 207, 2}
    },
    { {137, 248, 81, 23, 11},
      {127, 20, 145, 86, 9},
      {222, 136, 161, 249, 1},
      {150, 168, 146, 143, 14}
    },
    { {111, 187, 159, 46, 10},
      {219, 243, 63, 209, 9},
      {87, 79, 157, 223, 6},
      {152, 191, 204, 253, 11}
    },
    { {80, 175, 172, 250, 6},
      {1, 187, 222, 168, 15},
      {101, 243, 95, 80, 10},
      {241, 87, 189, 216, 0}
    },
    { {181, 113, 7, 57, 6},
      {209, 127, 226, 70, 0},
      {105, 206, 8, 234, 13},
      {6, 36, 127, 232, 11}
    },
    { {121, 119, 47, 9, 2},
      {83, 127, 44, 228, 4},
      {73, 15, 78, 233, 14},
      {34, 115, 79, 236, 10}
    },
    { {177, 29, 65, 73, 3},
      {116, 58, 32, 110, 8},
      {201, 40, 43, 136, 13},
      {23, 96, 69, 194, 14}
    },
    { {29, 227, 35, 74, 14},
      {147, 126, 89, 72, 5},
      {117, 44, 76, 123, 8},
      {161, 41, 167, 236, 9}
    },
    { {179, 148, 34, 108, 13},
      {76, 104, 115, 107, 4},
      {179, 100, 66, 156, 13},
      {45, 108, 225, 99, 2}
    },
    { {44, 126, 92, 173, 1},
      {239, 37, 46, 52, 10},
      {139, 83, 167, 227, 4},
      {82, 199, 74, 79, 7}
    },
    { {95, 115, 48, 151, 13},
      {207, 14, 201, 213, 7},
      {190, 144, 204, 239, 10},
      {234, 185, 55, 15, 3}
    },
    { {69, 251, 224, 160, 13},
      {229, 134, 91, 192, 14},
      {176, 80, 125, 250, 2},
      {112, 61, 166, 26, 7}
    },
    { {9, 122, 74, 176, 0},
      {99, 68, 142, 64, 10},
      {0, 213, 37, 233, 0},
      {80, 39, 18, 44, 6}
    },
    { {179, 119, 146, 163, 10},
      {65, 222, 43, 119, 3},
      {92, 84, 158, 236, 13},
      {206, 237, 71, 184, 2}
    },
    { {98, 35, 175, 220, 7},
      {29, 243, 236, 137, 6},
      {227, 191, 92, 68, 6},
      {105, 19, 124, 251, 8}
    },
    { {108, 155, 99, 108, 9},
      {254, 98, 59, 136, 12},
      {147, 108, 109, 147, 6},
      {49, 29, 196, 103, 15}
    },
    { {155, 78, 142, 46, 9},
      {14, 237, 15, 99, 9},
      {151, 71, 23, 45, 9},
      {156, 111, 11, 119, 0}
    },
    { {249, 195, 194, 21, 3},
      {46, 222, 184, 198, 0},
      {202, 132, 60, 57, 15},
      {6, 49, 215, 183, 4}
    },
    { {119, 43, 15, 234, 11},
      {149, 123, 47, 201, 11},
      {213, 127, 13, 78, 14},
      {217, 63, 77, 234, 9}
    },
    { {102, 250, 154, 171, 4},
      {193, 228, 126, 149, 11},
      {45, 85, 149, 246, 6},
      {218, 151, 226, 120, 3}
    },
    { {116, 98, 77, 223, 13},
      {189, 45, 237, 140, 3},
      {191, 187, 36, 98, 14},
      {195, 27, 123, 75, 13}
    },
    { {25, 222, 78, 102, 4},
      {106, 77, 94, 104, 9},
      {38, 103, 39, 185, 8},
      {145, 103, 171, 37, 6}
    },
    { {148, 41, 230, 53, 14},
      {169, 219, 195, 6, 12},
      {122, 198, 121, 66, 9},
      {54, 12, 61, 185, 5}
    },
    { {76, 236, 163, 246, 4},
      {155, 196, 210, 168, 15},
      {38, 252, 83, 115, 2},
      {241, 84, 178, 61, 9}
    },
    { {164, 180, 6, 106, 6},
      {193, 113, 114, 42, 1},
      {101, 102, 2, 210, 5},
      {133, 68, 232, 232, 3}
    },
    { {30, 86, 88, 215, 4},
      {234, 12, 204, 61, 3},
      {46, 177, 166, 167, 8},
      {203, 195, 51, 5, 7}
    },
    { {219, 73, 192, 161, 2},
      {34, 158, 2, 199, 10},
      {72, 80, 57, 45, 11},
      {94, 52, 7, 148, 4}
    },
    { {79, 81, 144, 218, 7},
      {198, 182, 192, 217, 3},
      {229, 176, 152, 175, 2},
      {201, 176, 54, 214, 3}
    },
    { {1, 67, 44, 173, 5},
      {12, 39, 78, 68, 6},
      {171, 83, 76, 40, 0},
      {98, 39, 46, 67, 0}
    },
    { {82, 252, 40, 115, 9},
      {69, 12, 151, 173, 13},
      {156, 225, 67, 244, 10},
      {187, 94, 147, 10, 2}
    },
    { {67, 147, 62, 210, 15},
      {68, 83, 221, 217, 7},
      {244, 183, 204, 156, 2},
      {233, 187, 188, 162, 2}
    },
    { {107, 7, 29, 172, 0},
      {26, 35, 46, 241, 2},
      {3, 91, 142, 13, 6},
      {72, 247, 76, 69, 8}
    },
    { {81, 13, 171, 142, 1},
      {28, 234, 4, 224, 15},
      {135, 29, 91, 8, 10},
      {240, 114, 5, 115, 8}
    },
    { {80, 201, 26, 157, 10},
      {8, 126, 149, 148, 10},
      {91, 149, 137, 48, 10},
      {82, 154, 151, 225, 0}
    },
    { {95, 229, 148, 45, 0},
      {139, 175, 18, 245, 0},
      {11, 66, 154, 127, 10},
      {10, 244, 143, 93, 1}
    },
    { {88, 194, 177, 178, 10},
      {18, 156, 155, 144, 7},
      {84, 216, 212, 49, 10},
      {224, 157, 147, 148, 8}
    },
    { {137, 65, 77, 139, 14},
      {50, 55, 69, 70, 3},
      {125, 27, 40, 41, 1},
      {198, 42, 46, 196, 12}
    },
    { {132, 10, 38, 120, 5},
      {132, 97, 202, 10, 12},
      {161, 230, 69, 2, 1},
      {53, 5, 56, 98, 1}
    },
    { {185, 96, 91, 1, 13},
      {55, 76, 101, 86, 0},
      {184, 13, 160, 105, 13},
      {6, 170, 99, 46, 12}
    },
    { {20, 186, 220, 247, 7},
      {237, 153, 222, 28, 11},
      {238, 243, 181, 210, 8},
      {211, 135, 185, 155, 7}
    },
    { {246, 219, 203, 198, 4},
      {248, 206, 124, 139, 11},
      {38, 61, 61, 182, 15},
      {221, 19, 231, 49, 15}
    },
    { {192, 171, 73, 169, 0},
      {49, 34, 30, 134, 10},
      {9, 89, 45, 80, 3},
      {86, 23, 132, 72, 12}
    },
    { {153, 38, 27, 157, 2},
      {27, 120, 140, 118, 2},
      {75, 157, 134, 73, 9},
      {70, 227, 17, 237, 8}
    },
    { {97, 25, 251, 19, 6},
      {112, 210, 228, 212, 13},
      {108, 141, 249, 136, 6},
      {178, 178, 116, 176, 14}
    },
    { {136, 247, 2, 236, 5},
      {79, 102, 90, 42, 2},
      {163, 116, 14, 241, 1},
      {69, 69, 166, 111, 2}
    },
    { {59, 109, 198, 146, 3},
      {39, 223, 160, 97, 11},
      {196, 150, 59, 109, 12},
      {216, 96, 95, 190, 4}
    },
    { {78, 98, 149, 182, 0},
      {155, 133, 138, 145, 3},
      {6, 218, 148, 103, 2},
      {200, 149, 26, 29, 9}
    },
    { {217, 253, 38, 220, 10},
      {75, 127, 145, 234, 14},
      {83, 182, 75, 249, 11},
      {117, 120, 159, 237, 2}
    },
    { {185, 170, 215, 180, 13},
      {63, 201, 251, 82, 10},
      {178, 222, 181, 89, 13},
      {84, 173, 249, 63, 12}
    },
    { {77, 237, 86, 136, 15},
      {167, 119, 81, 240, 10},
      {241, 22, 171, 123, 2},
      {80, 248, 174, 238, 5}
    },
    { {32, 85, 135, 53, 1},
      {92, 199, 162, 36, 0},
      {138, 206, 26, 160, 4},
      {2, 68, 94, 51, 10}
    },
    { {229, 194, 49, 87, 1},
      {156, 4, 184, 222, 5},
      {142, 168, 196, 58, 7},
      {167, 177, 210, 3, 9}
    },
    { {203, 185, 136, 37, 6},
      {75, 146, 86, 199, 8},
      {106, 65, 25, 221, 3},
      {30, 54, 164, 157, 2}
    },
    { {43, 23, 42, 185, 9},
      {70, 98, 175, 101, 6},
      {153, 213, 78, 141, 4},
      {106, 111, 84, 102, 2}
    },
    { {171, 204, 25, 105, 3},
      {22, 52, 54, 127, 8},
      {201, 105, 131, 61, 5},
      {31, 230, 194, 198, 8}
    },
    { {136, 158, 50, 170, 15},
      {70, 112, 91, 50, 15},
      {245, 84, 199, 145, 1},
      {244, 205, 160, 230, 2}
    },
    { {13, 103, 151, 181, 12},
      {155, 199, 203, 116, 2},
      {58, 222, 158, 107, 0},
      {66, 237, 62, 61, 9}
    },
    { {110, 236, 61, 135, 6},
      {155, 21, 116, 181, 15},
      {110, 27, 195, 119, 6},
      {250, 210, 234, 141, 9}
    },
    { {163, 18, 150, 137, 14},
      {64, 241, 105, 87, 2},
      {121, 22, 148, 140, 5},
      {78, 169, 104, 240, 2}
    },
    { {49, 39, 126, 125, 11},
      {45, 123, 175, 124, 4},
      {219, 231, 238, 72, 12},
      {35, 239, 93, 235, 4}
    },
    { {197, 59, 33, 41, 14},
      {209, 50, 75, 198, 12},
      {121, 72, 77, 202, 3},
      {54, 61, 36, 200, 11}
    },
    { {14, 92, 116, 153, 15},
      {230, 53, 193, 53, 14},
      {249, 146, 227, 167, 0},
      {122, 200, 58, 198, 7}
    },
    { {206, 216, 31, 33, 1},
      {214, 69, 22, 151, 8},
      {136, 79, 129, 183, 3},
      {30, 150, 138, 38, 11}
    },
    { {17, 190, 238, 22, 3},
      {109, 217, 156, 96, 13},
      {198, 135, 119, 216, 8},
      {176, 99, 153, 187, 6}
    },
    { {91, 246, 119, 192, 6},
      {115, 93, 88, 249, 6},
      {96, 62, 230, 253, 10},
      {105, 241, 171, 172, 14}
    },
    { {205, 49, 157, 20, 14},
      {219, 147, 197, 210, 0},
      {114, 139, 152, 203, 3},
      {4, 186, 60, 157, 11}
    },
    { {33, 104, 14, 72, 13},
      {5, 101, 101, 72, 8},
      {177, 39, 1, 104, 4},
      {17, 42, 106, 106, 0}
    },
    { {40, 39, 132, 250, 11},
      {7, 179, 171, 40, 3},
      {213, 242, 30, 65, 4},
      {193, 77, 92, 222, 0}
    },
    { {168, 244, 166, 26, 0},
      {67, 229, 176, 34, 5},
      {5, 134, 82, 241, 5},
      {164, 64, 218, 124, 2}
    },
    { {49, 118, 81, 59, 4},
      {113, 44, 234, 116, 1},
      {45, 200, 166, 232, 12},
      {130, 229, 115, 72, 14}
    },
    { {247, 105, 206, 110, 2},
      {169, 255, 38, 203, 9},
      {71, 103, 57, 110, 15},
      {157, 54, 79, 249, 5}
    },
    { {83, 146, 163, 177, 13},
      {84, 200, 219, 197, 6},
      {184, 220, 84, 156, 10},
      {106, 61, 177, 50, 10}
    },
    { {88, 101, 173, 108, 15},
      {31, 191, 71, 168, 4},
      {243, 107, 90, 97, 10},
      {33, 94, 47, 223, 8}
    },
    { {190, 115, 121, 11, 0},
      {243, 46, 44, 23, 5},
      {13, 9, 236, 231, 13},
      {174, 131, 71, 76, 15}
    },
    { {90, 232, 238, 91, 11},
      {39, 253, 149, 141, 13},
      {221, 167, 113, 117, 10},
      {187, 26, 155, 254, 4}
    },
    { {139, 106, 57, 122, 0},
      {19, 36, 142, 91, 13},
      {5, 233, 197, 109, 1},
      {189, 167, 18, 76, 8}
    },
    { {163, 233, 94, 124, 14},
      {41, 119, 247, 91, 8},
      {115, 231, 169, 124, 5},
      {29, 174, 254, 233, 4}
    },
    { {91, 159, 159, 132, 13},
      {94, 203, 93, 241, 10},
      {178, 31, 159, 157, 10},
      {88, 251, 173, 55, 10}
    },
    { {47, 75, 147, 199, 11},
      {158, 214, 41, 93, 11},
      {222, 60, 157, 47, 4},
      {219, 169, 70, 183, 9}
    },
    { {76, 117, 32, 169, 6},
      {195, 54, 66, 164, 6},
      {105, 80, 74, 227, 2},
      {98, 84, 38, 204, 3}
    },
    { {98, 163, 2, 254, 14},
      {9, 114, 251, 137, 3},
      {119, 244, 12, 84, 6},
      {201, 29, 244, 233, 0}
    },
    { {103, 190, 132, 139, 7},
      {197, 177, 120, 229, 11},
      {237, 18, 23, 222, 6},
      {218, 113, 232, 218, 3}
    },
    { {183, 81, 31, 133, 12},
      {216, 79, 101, 87, 2},
      {58, 31, 136, 174, 13},
      {78, 170, 111, 33, 11}
    },
    { {24, 89, 92, 70, 7},
      {110, 31, 68, 24, 9},
      {230, 35, 169, 161, 8},
      {145, 130, 47, 135, 6}
    },
    { {21, 230, 55, 117, 0},
      {153, 77, 154, 124, 4},
      {10, 238, 198, 122, 8},
      {35, 229, 155, 41, 9}
    },
    { {152, 85, 193, 87, 13},
      {126, 142, 193, 46, 1},
      {190, 168, 58, 161, 9},
      {135, 72, 55, 23, 14}
    },
    { {221, 31, 87, 116, 9},
      {254, 75, 139, 250, 8},
      {146, 238, 175, 139, 11},
      {21, 253, 29, 39, 15}
    },
    { {116, 139, 16, 186, 5},
      {132, 42, 250, 144, 11},
      {165, 208, 141, 18, 14},
      {208, 149, 245, 66, 1}
    },
    { {28, 253, 149, 152, 1},
      {215, 175, 144, 48, 10},
      {129, 154, 155, 243, 8},
      {80, 192, 159, 94, 11}
    },
    { {99, 111, 43, 206, 2},
      {25, 118, 44, 233, 15},
      {71, 61, 79, 108, 6},
      {249, 115, 70, 233, 8}
    },
    { {175, 140, 36, 152, 13},
      {134, 33, 241, 99, 14},
      {177, 146, 67, 31, 5},
      {124, 104, 248, 70, 1}
    },
    { {33, 45, 73, 54, 5},
      {61, 2, 230, 96, 9},
      {166, 201, 43, 72, 4},
      {144, 102, 116, 11, 12}
    },
    { {173, 161, 22, 32, 14},
      {131, 83, 115, 82, 0},
      {112, 70, 136, 91, 5},
      {4, 172, 236, 172, 1}
    },
    { {239, 49, 13, 89, 13},
      {215, 35, 229, 207, 0},
      {185, 171, 8, 207, 7},
      {15, 58, 124, 78, 11}
    },
    { {155, 50, 155, 65, 14},
      {83, 216, 77, 95, 0},
      {120, 45, 148, 205, 9},
      {15, 171, 33, 188, 10}
    },
    { {95, 76, 1, 168, 15},
      {150, 60, 67, 225, 10},
      {241, 88, 3, 47, 10},
      {88, 124, 35, 198, 9}
    },
    { {110, 151, 175, 238, 0},
      {218, 227, 62, 169, 7},
      {7, 127, 94, 151, 6},
      {233, 87, 204, 117, 11}
    },
    { {129, 68, 91, 180, 8},
      {56, 68, 135, 114, 2},
      {18, 221, 162, 40, 1},
      {68, 238, 18, 33, 12}
    },
    { {196, 142, 130, 78, 7},
      {140, 240, 88, 170, 9},
      {231, 36, 23, 18, 3},
      {149, 81, 160, 243, 1}
    },
    { {225, 39, 182, 160, 12},
      {1, 195, 107, 242, 6},
      {48, 86, 222, 72, 7},
      {100, 253, 108, 56, 0}
    },
    { {88, 201, 151, 188, 7},
      {30, 255, 210, 144, 10},
      {227, 222, 153, 49, 10},
      {80, 148, 191, 247, 8}
    },
    { {37, 121, 224, 205, 7},
      {237, 182, 96, 76, 14},
      {235, 48, 121, 234, 4},
      {115, 32, 102, 219, 7}
    },
    { {107, 93, 4, 34, 7},
      {70, 23, 98, 225, 9},
      {228, 66, 11, 173, 6},
      {152, 116, 110, 134, 2}
    },
    { {109, 39, 67, 156, 2},
      {187, 114, 168, 224, 2},
      {67, 156, 46, 75, 6},
      {64, 113, 84, 237, 13}
    },
    { {168, 101, 37, 190, 2},
      {27, 55, 162, 34, 7},
      {71, 218, 74, 97, 5},
      {228, 68, 94, 205, 8}
    },
    { {197, 187, 243, 235, 0},
      {241, 226, 26, 222, 15},
      {13, 124, 253, 218, 3},
      {247, 181, 132, 120, 15}
    },
    { {224, 138, 63, 86, 10},
      {24, 81, 189, 154, 13},
      {86, 175, 197, 16, 7},
      {181, 155, 216, 161, 8}
    },
    { {209, 250, 49, 233, 7},
      {85, 60, 90, 222, 14},
      {233, 120, 197, 248, 11},
      {119, 181, 163, 202, 10}
    },
    { {102, 72, 123, 117, 10},
      {184, 84, 167, 157, 12},
      {90, 237, 225, 38, 6},
      {59, 158, 82, 161, 13}
    },
    { {152, 233, 237, 143, 5},
      {63, 175, 84, 6, 15},
      {175, 27, 121, 113, 9},
      {246, 2, 175, 95, 12}
    },
    { {156, 76, 216, 209, 15},
      {166, 156, 197, 62, 10},
      {248, 177, 179, 35, 9},
      {87, 202, 51, 150, 5}
    },
    { {219, 162, 17, 122, 14},
      {19, 56, 219, 219, 1},
      {117, 232, 132, 93, 11},
      {141, 189, 177, 204, 8}
    },
    { {252, 157, 45, 143, 13},
      {222, 43, 117, 166, 15},
      {191, 27, 75, 147, 15},
      {246, 90, 237, 71, 11}
    },
    { {42, 216, 44, 171, 15},
      {70, 53, 119, 5, 15},
      {253, 83, 65, 181, 4},
      {250, 14, 234, 198, 2}
    },
    { {203, 77, 206, 196, 5},
      {46, 199, 68, 235, 10},
      {162, 55, 59, 45, 3},
      {93, 114, 46, 55, 4}
    },
    { {212, 230, 126, 105, 10},
      {161, 125, 31, 190, 4},
      {89, 103, 230, 114, 11},
      {39, 223, 139, 232, 5}
    },
    { {182, 118, 99, 52, 5},
      {253, 76, 234, 35, 4},
      {162, 204, 102, 230, 13},
      {44, 69, 115, 43, 15}
    },
    { {129, 161, 64, 232, 13},
      {37, 34, 83, 74, 2},
      {177, 112, 40, 88, 1},
      {69, 44, 164, 74, 4}
    },
    { {36, 248, 181, 42, 13},
      {213, 165, 115, 16, 13},
      {181, 74, 209, 242, 4},
      {176, 140, 234, 90, 11}
    },
    { {33, 212, 75, 137, 7},
      {116, 116, 116, 100, 2},
      {233, 29, 34, 184, 4},
      {66, 98, 226, 226, 14}
    },
    { {143, 130, 58, 23, 11},
      {142, 80, 157, 87, 5},
      {222, 133, 196, 31, 1},
      {174, 171, 144, 167, 1}
    },
    { {246, 113, 26, 188, 10},
      {201, 126, 167, 147, 2},
      {83, 213, 136, 230, 15},
      {76, 158, 87, 233, 3}
    },
    { {63, 94, 2, 121, 7},
      {198, 124, 234, 109, 8},
      {233, 228, 7, 175, 12},
      {27, 101, 115, 230, 3}
    },
    { {205, 212, 228, 99, 15},
      {230, 149, 83, 238, 5},
      {252, 98, 114, 187, 3},
      {167, 124, 170, 150, 7}
    },
    { {89, 70, 5, 49, 5},
      {22, 13, 202, 228, 0},
      {168, 202, 6, 41, 10},
      {2, 117, 59, 6, 8}
    },
    { {32, 186, 102, 97, 13},
      {101, 65, 123, 12, 12},
      {184, 102, 101, 208, 4},
      {51, 13, 232, 42, 6}
    },
    { {151, 191, 26, 90, 0},
      {193, 106, 156, 123, 9},
      {5, 165, 143, 222, 9},
      {157, 227, 149, 104, 3}
    },
    { {126, 66, 218, 161, 8},
      {162, 204, 47, 149, 2},
      {24, 85, 180, 39, 14},
      {74, 159, 67, 52, 5}
    },
    { {98, 148, 45, 213, 5},
      {92, 1, 244, 173, 6},
      {170, 187, 66, 148, 6},
      {107, 82, 248, 3, 10}
    },
    { {177, 229, 50, 4, 7},
      {13, 94, 112, 114, 4},
      {226, 4, 202, 120, 13},
      {36, 224, 231, 171, 0}
    },
    { {222, 100, 233, 20, 12},
      {187, 140, 197, 163, 4},
      {50, 137, 114, 103, 11},
      {44, 90, 51, 29, 13}
    },
    { {71, 94, 177, 229, 0},
      {216, 132, 10, 253, 14},
      {10, 120, 215, 174, 2},
      {123, 245, 2, 17, 11}
    },
    { {25, 165, 108, 166, 9},
      {47, 11, 23, 96, 7},
      {150, 83, 106, 89, 8},
      {224, 110, 141, 15, 4}
    },
    { {149, 5, 117, 79, 13},
      {188, 43, 65, 126, 5},
      {191, 42, 234, 10, 9},
      {167, 232, 45, 67, 13}
    },
    { {54, 171, 61, 147, 0},
      {145, 11, 188, 21, 15},
      {12, 155, 205, 86, 12},
      {250, 131, 221, 8, 9}
    },
    { {87, 132, 157, 120, 1},
      {148, 169, 150, 249, 0},
      {129, 235, 146, 30, 10},
      {9, 246, 153, 82, 9}
    },
    { {164, 135, 244, 206, 4},
      {168, 163, 120, 58, 7},
      {39, 50, 254, 18, 5},
      {229, 193, 236, 81, 5}
    },
    { {177, 1, 150, 215, 4},
      {8, 203, 224, 94, 3},
      {46, 182, 152, 8, 13},
      {199, 160, 125, 49, 0}
    },
    { {202, 139, 220, 248, 6},
      {34, 179, 222, 155, 10},
      {97, 243, 189, 21, 3},
      {93, 151, 188, 212, 4}
    },
    { {241, 104, 15, 187, 1},
      {21, 109, 166, 198, 11},
      {141, 223, 1, 104, 15},
      {214, 54, 91, 106, 8}
    },
    { {119, 235, 139, 132, 9},
      {157, 206, 61, 193, 10},
      {146, 29, 29, 126, 14},
      {88, 59, 199, 59, 9}
    },
    { {2, 83, 18, 189, 4},
      {72, 102, 202, 21, 2},
      {43, 212, 140, 164, 0},
      {74, 133, 54, 97, 2}
    },
    { {29, 42, 141, 92, 1},
      {159, 169, 140, 72, 8},
      {131, 171, 21, 75, 8},
      {17, 35, 25, 95, 9}
    },
    { {39, 117, 108, 47, 10},
      {233, 55, 39, 101, 5},
      {95, 67, 106, 238, 4},
      {170, 110, 78, 201, 7}
    },
    { {61, 80, 83, 188, 9},
      {254, 108, 163, 80, 2},
      {147, 220, 160, 171, 12},
      {64, 172, 83, 103, 15}
    },
    { {120, 181, 174, 182, 4},
      {75, 203, 246, 160, 7},
      {38, 215, 90, 209, 14},
      {224, 86, 253, 61, 2}
    },
    { {42, 93, 133, 78, 1},
      {94, 167, 32, 41, 9},
      {135, 42, 27, 165, 4},
      {153, 64, 78, 87, 10}
    },
    { {0, 239, 149, 206, 12},
      {25, 167, 89, 56, 11},
      {55, 58, 159, 112, 0},
      {209, 201, 174, 89, 8}
    },
    { {198, 248, 201, 176, 10},
      {241, 148, 151, 131, 10},
      {80, 217, 49, 246, 3},
      {92, 30, 146, 152, 15}
    },
    { {52, 138, 244, 180, 8},
      {168, 137, 187, 16, 14},
      {18, 210, 245, 18, 12},
      {112, 141, 217, 17, 5}
    },
    { {154, 221, 191, 118, 3},
      {94, 223, 150, 59, 13},
      {198, 239, 219, 181, 9},
      {189, 198, 159, 183, 10}
    },
    { {68, 183, 153, 113, 1},
      {213, 130, 158, 188, 0},
      {136, 233, 158, 210, 2},
      {3, 215, 148, 26, 11}
    },
    { {53, 31, 237, 192, 9},
      {244, 139, 45, 104, 14},
      {144, 59, 127, 138, 12},
      {113, 107, 77, 18, 15}
    },
    { {240, 132, 133, 185, 5},
      {20, 169, 242, 166, 2},
      {169, 218, 18, 16, 15},
      {70, 84, 249, 82, 8}
    },
    { {130, 188, 93, 116, 5},
      {125, 1, 214, 59, 8},
      {162, 235, 163, 212, 1},
      {29, 198, 184, 11, 14}
    },
    { {58, 134, 179, 151, 15},
      {30, 216, 249, 53, 7},
      {254, 156, 214, 21, 12},
      {234, 201, 241, 183, 8}
    },
    { {140, 232, 137, 29, 15},
      {159, 180, 213, 6, 8},
      {251, 137, 17, 115, 1},
      {22, 10, 178, 223, 9}
    },
    { {63, 123, 205, 83, 8},
      {243, 143, 173, 77, 9},
      {28, 171, 61, 239, 12},
      {155, 43, 95, 28, 15}
    },
    { {180, 13, 145, 70, 7},
      {156, 154, 96, 58, 9},
      {230, 40, 155, 2, 13},
      {149, 192, 101, 147, 9}
    },
    { {109, 172, 86, 182, 6},
      {171, 81, 242, 240, 11},
      {102, 214, 163, 91, 6},
      {208, 244, 248, 173, 5}
    },
    { {88, 149, 105, 0, 13},
      {118, 10, 85, 160, 4},
      {176, 9, 106, 145, 10},
      {32, 90, 165, 6, 14}
    },
    { {76, 163, 63, 89, 3},
      {151, 115, 156, 156, 4},
      {201, 175, 204, 83, 2},
      {35, 147, 156, 238, 9}
    },
    { {53, 167, 215, 80, 1},
      {181, 203, 184, 120, 0},
      {128, 174, 190, 90, 12},
      {1, 225, 221, 58, 13}
    },
    { {22, 233, 116, 244, 11},
      {173, 31, 147, 25, 14},
      {210, 242, 233, 118, 8},
      {121, 140, 159, 139, 5}
    },
    { {106, 219, 126, 99, 3},
      {102, 87, 62, 157, 13},
      {204, 103, 237, 181, 6},
      {187, 151, 206, 166, 6}
    },
    { {172, 254, 199, 138, 5},
      {247, 229, 120, 34, 11},
      {165, 30, 55, 243, 5},
      {212, 65, 234, 126, 15}
    },
    { {195, 109, 112, 53, 6},
      {41, 22, 194, 247, 12},
      {106, 192, 235, 108, 3},
      {62, 244, 54, 137, 4}
    },
    { {133, 82, 84, 245, 6},
      {232, 21, 202, 94, 2},
      {106, 242, 164, 170, 1},
      {71, 165, 58, 129, 7}
    },
    { {211, 112, 148, 110, 7},
      {77, 189, 66, 219, 1},
      {231, 98, 144, 236, 11},
      {141, 180, 43, 219, 2}
    },
    { {157, 177, 206, 126, 12},
      {235, 235, 215, 74, 1},
      {55, 231, 56, 219, 9},
      {133, 46, 189, 125, 7}
    },
    { {163, 136, 217, 180, 14},
      {56, 144, 247, 83, 10},
      {114, 217, 177, 28, 5},
      {92, 174, 240, 145, 12}
    },
    { {98, 252, 134, 118, 14},
      {73, 213, 243, 169, 9},
      {118, 230, 19, 244, 6},
      {153, 92, 250, 185, 2}
    },
    { {209, 228, 206, 15, 2},
      {41, 253, 20, 230, 1},
      {79, 7, 50, 120, 11},
      {134, 114, 139, 249, 4}
    },
    { {224, 197, 119, 135, 0},
      {56, 71, 48, 182, 7},
      {14, 30, 234, 48, 7},
      {230, 208, 206, 33, 12}
    },
    { {136, 137, 241, 121, 7},
      {54, 178, 210, 30, 12},
      {233, 232, 249, 17, 1},
      {55, 132, 180, 214, 12}
    },
    { {175, 103, 124, 158, 0},
      {171, 39, 172, 115, 7},
      {7, 147, 238, 111, 5},
      {236, 227, 94, 77, 5}
    },
    { {250, 135, 32, 206, 8},
      {10, 42, 57, 171, 7},
      {23, 48, 78, 21, 15},
      {237, 89, 197, 69, 0}
    },
    { {32, 136, 108, 197, 14},
      {40, 17, 117, 12, 14},
      {122, 51, 97, 16, 4},
      {115, 10, 232, 129, 4}
    },
    { {26, 153, 204, 140, 14},
      {106, 187, 85, 1, 10},
      {115, 19, 57, 149, 8},
      {88, 10, 173, 213, 6}
    },
    { {136, 165, 101, 91, 1},
      {55, 35, 144, 46, 5},
      {141, 170, 106, 81, 1},
      {167, 64, 156, 78, 12}
    },
    { {214, 247, 236, 2, 5},
      {229, 143, 92, 163, 5},
      {164, 3, 126, 246, 11},
      {172, 83, 175, 26, 7}
    },
    { {244, 246, 145, 8, 4},
      {209, 172, 120, 178, 0},
      {33, 8, 150, 242, 15},
      {4, 209, 227, 88, 11}
    },
    { {76, 10, 202, 63, 5},
      {174, 224, 206, 132, 9},
      {175, 197, 53, 3, 2},
      {146, 23, 48, 119, 5}
    },
    { {31, 5, 164, 12, 7},
      {142, 187, 64, 97, 4},
      {227, 2, 90, 15, 8},
      {40, 96, 45, 215, 1}
    },
    { {250, 93, 132, 124, 6},
      {74, 191, 226, 171, 8},
      {99, 226, 27, 165, 15},
      {29, 84, 127, 213, 2}
    },
    { {19, 29, 173, 215, 12},
      {88, 139, 197, 109, 15},
      {62, 187, 91, 140, 8},
      {251, 106, 61, 17, 10}
    },
    { {195, 96, 214, 57, 15},
      {37, 245, 195, 215, 0},
      {249, 198, 176, 108, 3},
      {14, 188, 58, 250, 4}
    },
    { {126, 199, 122, 149, 3},
      {174, 94, 188, 181, 6},
      {202, 149, 238, 55, 14},
      {106, 211, 215, 167, 5}
    },
    { {45, 212, 161, 248, 1},
      {214, 164, 178, 104, 6},
      {129, 248, 82, 187, 4},
      {97, 100, 210, 86, 11}
    },
    { {196, 27, 169, 22, 11},
      {220, 146, 141, 130, 13},
      {214, 137, 93, 130, 3},
      {180, 27, 20, 147, 11}
    },
    { {174, 83, 48, 32, 14},
      {194, 22, 107, 19, 4},
      {112, 64, 204, 167, 5},
      {44, 141, 102, 132, 3}
    },
    { {137, 195, 118, 107, 2},
      {34, 119, 26, 94, 5},
      {77, 102, 236, 57, 1},
      {167, 165, 142, 228, 4}
    },
    { {17, 167, 104, 69, 2},
      {41, 26, 28, 108, 4},
      {74, 33, 110, 88, 8},
      {35, 99, 133, 137, 4}
    },
    { {115, 24, 54, 184, 10},
      {64, 121, 163, 209, 14},
      {81, 214, 193, 140, 14},
      {120, 188, 89, 224, 2}
    },
    { {152, 195, 136, 238, 2},
      {10, 190, 30, 10, 3},
      {71, 113, 28, 49, 9},
      {197, 7, 135, 213, 0}
    },
    { {50, 158, 254, 117, 6},
      {104, 217, 254, 61, 12},
      {106, 231, 247, 148, 12},
      {59, 199, 249, 177, 6}
    },
    { {235, 41, 103, 213, 2},
      {59, 83, 160, 207, 14},
      {74, 190, 105, 77, 7},
      {127, 48, 92, 173, 12}
    },
    { {118, 246, 246, 147, 13},
      {229, 205, 249, 181, 7},
      {188, 150, 246, 246, 14},
      {234, 217, 251, 58, 7}
    },
    { {75, 54, 229, 94, 10},
      {123, 177, 137, 233, 5},
      {87, 170, 118, 205, 2},
      {169, 121, 24, 221, 14}
    },
    { {18, 100, 241, 41, 15},
      {53, 188, 67, 53, 4},
      {249, 72, 242, 100, 8},
      {42, 204, 35, 218, 12}
    },
    { {195, 216, 179, 216, 12},
      {80, 228, 209, 219, 14},
      {49, 188, 209, 188, 3},
      {125, 184, 178, 112, 10}
    },
    { {184, 252, 222, 50, 9},
      {103, 205, 183, 50, 9},
      {148, 199, 179, 241, 13},
      {148, 206, 219, 62, 6}
    },
    { {53, 97, 247, 116, 12},
      {185, 207, 227, 88, 4},
      {50, 238, 248, 106, 12},
      {33, 172, 127, 57, 13}
    },
    { {116, 255, 31, 22, 6},
      {217, 95, 252, 176, 9},
      {102, 143, 143, 242, 14},
      {144, 211, 255, 169, 11}
    },
    { {29, 80, 232, 123, 6},
      {226, 188, 198, 76, 5},
      {109, 225, 112, 171, 8},
      {163, 38, 51, 212, 7}
    },
    { {71, 201, 216, 44, 9},
      {172, 166, 23, 209, 8},
      {147, 65, 185, 62, 2},
      {24, 190, 134, 83, 5}
    },
    { {198, 131, 135, 46, 9},
      {156, 227, 27, 131, 1},
      {151, 78, 28, 22, 3},
      {140, 29, 140, 115, 9}
    },
    { {182, 245, 65, 140, 14},
      {249, 62, 113, 35, 2},
      {115, 24, 42, 246, 13},
      {76, 72, 231, 201, 15}
    },
    { {209, 45, 250, 27, 10},
      {33, 250, 133, 246, 13},
      {93, 133, 251, 72, 11},
      {182, 250, 21, 248, 4}
    },
    { {123, 222, 164, 209, 14},
      {66, 157, 249, 237, 14},
      {120, 178, 87, 189, 14},
      {123, 121, 251, 148, 2}
    },
    { {148, 215, 11, 206, 8},
      {216, 110, 29, 42, 3},
      {23, 61, 14, 178, 9},
      {197, 75, 135, 97, 11}
    },
    { {139, 161, 235, 181, 10},
      {59, 210, 151, 71, 6},
      {90, 221, 120, 93, 1},
      {110, 46, 148, 189, 12}
    },
    { {2, 76, 23, 111, 5},
      {28, 101, 66, 61, 9},
      {175, 110, 131, 36, 0},
      {155, 196, 42, 99, 8}
    },
    { {54, 104, 158, 115, 11},
      {133, 221, 167, 29, 9},
      {220, 231, 145, 102, 12},
      {155, 142, 91, 186, 1}
    },
    { {208, 33, 104, 228, 4},
      {41, 10, 70, 138, 6},
      {34, 113, 104, 64, 11},
      {101, 22, 37, 9, 4}
    },
    { {155, 143, 16, 45, 14},
      {10, 58, 91, 119, 8},
      {123, 64, 143, 29, 9},
      {30, 237, 165, 197, 0}
    },
    { {126, 132, 92, 39, 11},
      {174, 25, 55, 181, 1},
      {222, 67, 162, 23, 14},
      {138, 222, 201, 135, 5}
    },
    { {118, 223, 8, 118, 12},
      {200, 14, 255, 169, 9},
      {54, 225, 15, 182, 14},
      {153, 95, 247, 1, 3}
    },
    { {251, 188, 21, 149, 7},
      {95, 25, 240, 247, 10},
      {234, 154, 131, 221, 15},
      {94, 240, 249, 143, 10}
    },
    { {70, 128, 175, 59, 2},
      {144, 241, 150, 133, 5},
      {77, 207, 80, 22, 2},
      {170, 22, 152, 240, 9}
    },
    { {209, 94, 13, 85, 9},
      {92, 13, 141, 238, 8},
      {154, 171, 7, 168, 11},
      {23, 123, 27, 3, 10}
    },
    { {140, 34, 108, 129, 4},
      {163, 1, 76, 6, 6},
      {40, 19, 100, 67, 1},
      {102, 3, 40, 12, 5}
    },
    { {118, 43, 180, 156, 1},
      {141, 171, 168, 145, 14},
      {131, 146, 221, 70, 14},
      {120, 145, 93, 91, 1}
    },
    { {115, 86, 122, 41, 0},
      {96, 108, 46, 245, 4},
      {9, 69, 230, 172, 14},
      {42, 247, 67, 96, 6}
    },
    { {65, 92, 222, 11, 1},
      {100, 229, 4, 244, 9},
      {141, 7, 179, 168, 2},
      {146, 242, 10, 114, 6}
    },
    { {9, 152, 244, 128, 11},
      {102, 145, 17, 80, 14},
      {208, 18, 241, 153, 0},
      {112, 168, 136, 150, 6}
    },
    { {227, 53, 25, 14, 13},
      {93, 34, 101, 243, 1},
      {183, 9, 138, 204, 7},
      {140, 250, 100, 75, 10}
    },
    { {54, 133, 243, 244, 3},
      {188, 218, 178, 57, 6},
      {194, 252, 250, 22, 12},
      {105, 196, 213, 179, 13}
    },
    { {149, 140, 227, 231, 8},
      {184, 200, 19, 110, 15},
      {30, 124, 115, 26, 9},
      {247, 108, 129, 49, 13}
    },
    { {119, 22, 87, 104, 15},
      {244, 121, 107, 249, 0},
      {241, 110, 166, 142, 14},
      {9, 253, 105, 226, 15}
    },
    { {192, 211, 150, 95, 4},
      {72, 231, 216, 158, 1},
      {47, 166, 156, 176, 3},
      {135, 145, 190, 113, 2}
    },
    { {212, 159, 135, 115, 2},
      {208, 219, 154, 174, 9},
      {76, 238, 31, 146, 11},
      {151, 85, 157, 176, 11}
    },
    { {69, 191, 48, 71, 8},
      {201, 2, 25, 252, 13},
      {30, 32, 207, 218, 2},
      {179, 249, 132, 9, 3}
    },
    { {178, 247, 139, 151, 9},
      {93, 206, 189, 39, 3},
      {158, 157, 30, 244, 13},
      {206, 75, 215, 59, 10}
    },
    { {182, 81, 172, 193, 7},
      {196, 159, 100, 15, 6},
      {232, 51, 88, 166, 13},
      {111, 2, 111, 146, 3}
    },
    { {18, 188, 225, 89, 10},
      {113, 184, 145, 45, 12},
      {89, 168, 115, 212, 8},
      {59, 72, 145, 216, 14}
    },
    { {142, 109, 66, 22, 5},
      {175, 70, 192, 35, 9},
      {166, 132, 43, 103, 1},
      {156, 64, 54, 47, 5}
    },
    { {247, 219, 218, 17, 6},
      {224, 222, 252, 215, 8},
      {104, 133, 189, 190, 15},
      {30, 179, 247, 176, 7}
    },
    { {95, 40, 2, 124, 14},
      {139, 120, 195, 201, 8},
      {115, 228, 1, 79, 10},
      {25, 60, 49, 237, 1}
    },
    { {2, 18, 169, 157, 14},
      {88, 176, 205, 5, 6},
      {123, 153, 84, 132, 0},
      {106, 11, 48, 209, 10}
    },
    { {191, 254, 168, 18, 10},
      {195, 156, 189, 99, 13},
      {84, 129, 87, 255, 13},
      {188, 107, 211, 156, 3}
    },
    { {186, 49, 92, 161, 1},
      {103, 11, 38, 23, 2},
      {136, 83, 168, 197, 13},
      {78, 134, 77, 14, 6}
    },
    { {7, 242, 225, 29, 7},
      {253, 180, 216, 69, 4},
      {235, 136, 116, 254, 0},
      {42, 33, 178, 219, 15}
    },
    { {234, 79, 43, 164, 13},
      {30, 70, 111, 163, 14},
      {178, 93, 79, 37, 7},
      {124, 95, 102, 39, 8}
    },
    { {165, 190, 34, 214, 12},
      {201, 64, 249, 106, 15},
      {54, 180, 71, 218, 5},
      {245, 105, 240, 41, 3}
    },
    { {241, 169, 225, 184, 14},
      {49, 186, 243, 194, 14},
      {113, 216, 121, 88, 15},
      {116, 60, 245, 216, 12}
    },
    { {202, 168, 155, 183, 7},
      {31, 208, 214, 151, 11},
      {238, 221, 145, 85, 3},
      {222, 150, 176, 191, 8}
    },
    { {84, 134, 196, 245, 2},
      {168, 153, 154, 172, 2},
      {74, 242, 54, 18, 10},
      {67, 85, 153, 145, 5}
    },
    { {53, 121, 76, 30, 4},
      {233, 47, 228, 64, 9},
      {39, 131, 41, 234, 12},
      {144, 34, 127, 73, 7}
    },
    { {110, 216, 72, 137, 8},
      {226, 36, 53, 133, 10},
      {25, 17, 33, 183, 6},
      {90, 26, 194, 68, 7}
    },
    { {186, 232, 137, 103, 6},
      {27, 156, 118, 15, 9},
      {110, 105, 17, 117, 13},
      {159, 6, 227, 157, 8}
    },
    { {148, 192, 40, 189, 0},
      {136, 44, 150, 6, 6},
      {11, 209, 64, 50, 9},
      {102, 6, 147, 65, 1}
    },
    { {115, 76, 162, 183, 6},
      {8, 220, 226, 229, 15},
      {110, 212, 83, 44, 14},
      {250, 116, 115, 177, 0}
    },
    { {47, 241, 60, 122, 11},
      {199, 55, 183, 89, 5},
      {213, 227, 200, 255, 4},
      {169, 174, 222, 206, 3}
    },
    { {204, 2, 99, 100, 7},
      {190, 80, 74, 138, 4},
      {226, 108, 100, 3, 3},
      {37, 21, 32, 167, 13}
    },
    { {222, 144, 69, 41, 2},
      {242, 57, 18, 135, 0},
      {73, 74, 32, 151, 11},
      {14, 20, 137, 196, 15}
    },
    { {2, 43, 108, 122, 15},
      {37, 51, 207, 9, 13},
      {245, 227, 109, 68, 0},
      {185, 15, 60, 202, 4}
    },
    { {64, 75, 24, 30, 13},
      {12, 38, 205, 144, 9},
      {183, 129, 141, 32, 2},
      {144, 155, 54, 67, 0}
    },
    { {44, 49, 42, 198, 5},
      {207, 66, 100, 8, 7},
      {166, 53, 72, 195, 4},
      {225, 2, 100, 47, 3}
    },
    { {191, 189, 15, 65, 6},
      {211, 91, 116, 111, 8},
      {104, 47, 11, 223, 13},
      {31, 98, 237, 172, 11}
    },
    { {232, 58, 99, 203, 12},
      {115, 96, 105, 142, 15},
      {61, 60, 101, 193, 7},
      {247, 25, 96, 108, 14}
    },
    { {231, 89, 156, 35, 2},
      {192, 151, 38, 215, 9},
      {76, 67, 153, 174, 7},
      {158, 182, 78, 144, 3}
    },
    { {255, 230, 113, 141, 6},
      {187, 60, 120, 247, 6},
      {107, 24, 230, 127, 15},
      {110, 241, 227, 205, 13}
    },
    { {197, 136, 65, 238, 2},
      {184, 48, 18, 202, 11},
      {71, 120, 33, 26, 3},
      {213, 52, 128, 193, 13}
    },
    { {171, 180, 217, 16, 2},
      {115, 144, 180, 115, 0},
      {64, 137, 178, 221, 5},
      {12, 226, 208, 156, 14}
    },
    { {10, 23, 123, 120, 4},
      {114, 98, 206, 57, 4},
      {33, 237, 238, 133, 0},
      {41, 199, 52, 100, 14}
    },
    { {78, 123, 115, 156, 14},
      {251, 118, 201, 145, 14},
      {115, 156, 237, 231, 2},
      {120, 153, 54, 237, 15}
    },
    { {34, 227, 251, 26, 1},
      {53, 230, 188, 17, 5},
      {133, 141, 252, 116, 4},
      {168, 131, 214, 122, 12}
    },
    { {54, 93, 186, 61, 2},
      {200, 254, 166, 53, 12},
      {75, 197, 219, 166, 12},
      {58, 198, 87, 241, 3}
    },
    { {59, 220, 189, 56, 4},
      {82, 173, 246, 113, 12},
      {33, 203, 211, 189, 12},
      {56, 230, 251, 84, 10}
    },
    { {200, 199, 188, 19, 5},
      {6, 135, 220, 182, 5},
      {172, 131, 222, 49, 3},
      {166, 211, 190, 22, 0}
    },
    { {91, 16, 250, 200, 6},
      {98, 248, 68, 217, 6},
      {97, 53, 240, 141, 10},
      {105, 178, 33, 244, 6}
    },
    { {208, 249, 92, 92, 11},
      {109, 63, 149, 154, 8},
      {211, 163, 169, 240, 11},
      {21, 154, 159, 203, 6}
    },
    { {153, 6, 142, 77, 7},
      {14, 249, 76, 110, 0},
      {235, 39, 22, 9, 9},
      {7, 99, 41, 247, 0}
    },
    { {200, 39, 17, 208, 2},
      {19, 18, 136, 186, 2},
      {64, 184, 142, 65, 3},
      {69, 209, 20, 140, 8}
    },
    { {164, 84, 34, 73, 9},
      {196, 100, 33, 46, 4},
      {153, 36, 66, 162, 5},
      {39, 72, 66, 98, 3}
    },
    { {169, 103, 185, 198, 3},
      {31, 150, 44, 122, 7},
      {198, 57, 222, 105, 5},
      {229, 227, 70, 159, 8}
    },
    { {227, 227, 61, 193, 6},
      {17, 23, 124, 223, 6},
      {104, 59, 204, 124, 7},
      {111, 179, 238, 136, 8}
    },
    { {242, 227, 250, 36, 6},
      {41, 222, 126, 147, 4},
      {98, 69, 252, 116, 15},
      {44, 151, 231, 185, 4}
    },
    { {171, 214, 110, 194, 10},
      {98, 85, 61, 107, 7},
      {84, 55, 102, 189, 5},
      {237, 107, 202, 164, 6}
    },
    { {150, 166, 172, 174, 12},
      {137, 169, 95, 35, 7},
      {55, 83, 86, 86, 9},
      {236, 79, 169, 89, 1}
    },
    { {17, 94, 218, 37, 15},
      {108, 220, 79, 116, 8},
      {250, 69, 183, 168, 8},
      {18, 239, 35, 179, 6}
    },
    { {224, 72, 217, 79, 9},
      {60, 164, 37, 158, 9},
      {159, 41, 177, 32, 7},
      {151, 154, 66, 83, 12}
    },
    { {62, 162, 162, 69, 13},
      {143, 200, 121, 13, 4},
      {186, 36, 84, 87, 12},
      {43, 9, 225, 63, 1}
    },
    { {147, 122, 113, 157, 10},
      {121, 60, 137, 87, 14},
      {91, 152, 229, 236, 9},
      {126, 169, 19, 201, 14}
    },
    { {109, 101, 53, 32, 4},
      {147, 7, 98, 240, 4},
      {32, 74, 202, 107, 6},
      {32, 244, 110, 12, 9}
    },
    { {218, 6, 151, 74, 11},
      {22, 249, 9, 187, 1},
      {213, 46, 150, 5, 11},
      {141, 217, 9, 246, 8}
    },
    { {20, 176, 37, 32, 10},
      {209, 25, 19, 0, 4},
      {80, 74, 64, 210, 8},
      {32, 12, 137, 136, 11}
    },
    { {122, 150, 135, 109, 5},
      {94, 233, 122, 173, 0},
      {171, 110, 22, 149, 14},
      {11, 85, 233, 119, 10}
    },
    { {221, 120, 18, 100, 5},
      {207, 76, 66, 218, 8},
      {162, 100, 129, 235, 11},
      {21, 180, 35, 47, 3}
    },
    { {130, 195, 122, 162, 9},
      {36, 70, 31, 19, 7},
      {148, 85, 236, 52, 1},
      {236, 143, 134, 34, 4}
    },
    { {151, 25, 250, 15, 4},
      {232, 234, 68, 87, 13},
      {47, 5, 249, 142, 9},
      {190, 162, 37, 113, 7}
    },
    { {58, 108, 193, 181, 7},
      {63, 156, 226, 37, 10},
      {234, 216, 51, 101, 12},
      {90, 68, 115, 159, 12}
    },
    { {97, 33, 177, 73, 13},
      {21, 162, 97, 220, 4},
      {185, 40, 216, 72, 6},
      {35, 184, 100, 90, 8}
    },
    { {208, 123, 6, 29, 14},
      {73, 127, 201, 134, 8},
      {123, 134, 13, 224, 11},
      {22, 25, 63, 233, 2}
    },
    { {69, 190, 247, 38, 9},
      {253, 193, 27, 240, 13},
      {150, 78, 247, 218, 2},
      {176, 253, 136, 59, 15}
    },
    { {157, 70, 148, 66, 4},
      {130, 141, 72, 122, 1},
      {36, 34, 150, 43, 9},
      {133, 225, 43, 20, 1}
    },
    { {77, 251, 186, 241, 4},
      {195, 198, 222, 220, 14},
      {40, 245, 221, 251, 2},
      {115, 183, 182, 60, 3}
    },
    { {200, 236, 99, 163, 1},
      {55, 68, 18, 166, 15},
      {140, 92, 99, 113, 3},
      {246, 84, 130, 46, 12}
    },
    { {41, 38, 49, 96, 9},
      {23, 0, 43, 120, 4},
      {144, 104, 198, 73, 4},
      {33, 237, 64, 14, 8}
    },
    { {34, 234, 181, 80, 10},
      {17, 149, 185, 25, 12},
      {80, 170, 213, 116, 4},
      {57, 137, 218, 152, 8}
    },
    { {12, 83, 62, 124, 9},
      {206, 103, 143, 24, 4},
      {147, 231, 204, 163, 0},
      {33, 143, 30, 103, 3}
    },
    { {241, 209, 122, 65, 9},
      {100, 78, 53, 222, 4},
      {152, 37, 232, 184, 15},
      {39, 186, 199, 34, 6}
    },
    { {190, 41, 45, 21, 12},
      {155, 11, 229, 7, 12},
      {58, 139, 73, 71, 13},
      {62, 10, 125, 13, 9}
    },
    { {173, 236, 230, 5, 15},
      {175, 213, 113, 102, 12},
      {250, 6, 115, 123, 5},
      {54, 104, 234, 191, 5}
    },
    { {23, 198, 147, 157, 3},
      {156, 252, 152, 117, 2},
      {203, 156, 150, 62, 8},
      {74, 225, 147, 243, 9}
    },
    { {76, 195, 212, 88, 9},
      {166, 167, 153, 152, 0},
      {145, 162, 188, 51, 2},
      {1, 153, 158, 86, 5}
    },
    { {154, 214, 102, 33, 4},
      {98, 77, 90, 39, 4},
      {40, 70, 102, 181, 9},
      {46, 69, 171, 36, 6}
    },
    { {19, 133, 65, 215, 3},
      {60, 26, 144, 109, 3},
      {206, 184, 42, 28, 8},
      {203, 96, 149, 131, 12}
    },
    { {86, 15, 151, 244, 5},
      {156, 203, 202, 185, 10},
      {162, 254, 159, 6, 10},
      {89, 213, 61, 51, 9}
    },
    { {166, 192, 165, 246, 0},
      {152, 133, 178, 11, 7},
      {6, 250, 80, 54, 5},
      {237, 4, 218, 17, 9}
    },
    { {95, 226, 229, 89, 8},
      {179, 173, 153, 205, 4},
      {25, 170, 116, 127, 10},
      {43, 57, 155, 92, 13}
    },
    { {67, 190, 136, 118, 5},
      {77, 128, 222, 233, 9},
      {166, 225, 23, 220, 2},
      {153, 119, 176, 27, 2}
    },
    { {8, 144, 129, 205, 4},
      {90, 160, 80, 12, 2},
      {43, 56, 16, 145, 0},
      {67, 0, 160, 85, 10}
    },
    { {77, 234, 23, 239, 15},
      {159, 117, 91, 220, 11},
      {255, 126, 133, 123, 2},
      {211, 189, 170, 239, 9}
    },
    { {124, 58, 173, 18, 0},
      {211, 137, 172, 128, 13},
      {4, 139, 85, 195, 14},
      {176, 19, 89, 28, 11}
    },
    { {67, 86, 162, 174, 9},
      {76, 228, 11, 225, 7},
      {151, 84, 86, 172, 2},
      {232, 125, 2, 115, 2}
    },
    { {236, 183, 0, 75, 5},
      {199, 34, 120, 174, 1},
      {173, 32, 14, 211, 7},
      {135, 81, 228, 78, 3}
    },
    { {247, 97, 178, 178, 7},
      {133, 222, 226, 211, 7},
      {228, 212, 216, 110, 15},
      {236, 180, 119, 186, 1}
    },
    { {137, 36, 99, 124, 13},
      {63, 96, 195, 106, 4},
      {179, 236, 98, 73, 1},
      {37, 108, 48, 111, 12}
    },
    { {114, 58, 98, 179, 1},
      {101, 72, 170, 133, 15},
      {140, 212, 101, 196, 14},
      {250, 21, 81, 42, 6}
    },
    { {41, 180, 125, 69, 1},
      {127, 1, 52, 124, 4},
      {138, 43, 226, 217, 4},
      {35, 226, 200, 15, 14}
    },
    { {18, 107, 239, 149, 10},
      {57, 223, 141, 5, 14},
      {90, 159, 125, 100, 8},
      {122, 11, 31, 185, 12}
    },
    { {161, 94, 96, 121, 10},
      {96, 52, 171, 110, 12},
      {89, 224, 103, 168, 5},
      {55, 109, 82, 192, 6}
    },
    { {126, 76, 172, 58, 13},
      {134, 173, 231, 161, 13},
      {181, 195, 83, 39, 14},
      {184, 94, 123, 86, 1}
    },
    { {114, 142, 21, 50, 3},
      {20, 25, 186, 177, 9},
      {196, 202, 135, 20, 14},
      {152, 213, 217, 130, 8}
    },
    { {50, 5, 228, 151, 0},
      {40, 139, 160, 37, 7},
      {14, 146, 122, 4, 12},
      {234, 64, 93, 17, 4}
    },
    { {20, 99, 87, 142, 15},
      {189, 127, 73, 16, 3},
      {247, 30, 172, 98, 8},
      {192, 137, 47, 235, 13}
    },
    { {70, 114, 172, 144, 8},
      {193, 133, 141, 129, 6},
      {16, 147, 84, 230, 2},
      {104, 27, 26, 24, 3}
    },
    { {218, 88, 222, 192, 8},
      {98, 205, 5, 155, 10},
      {16, 55, 177, 165, 11},
      {93, 154, 11, 52, 6}
    },
    { {201, 31, 228, 54, 4},
      {106, 131, 202, 226, 13},
      {38, 194, 127, 137, 3},
      {180, 117, 60, 21, 6}
    },
    { {161, 197, 254, 154, 12},
      {32, 231, 245, 114, 7},
      {53, 151, 250, 56, 5},
      {228, 234, 254, 112, 4}
    },
    { {247, 70, 210, 71, 7},
      {172, 220, 104, 255, 1},
      {238, 36, 182, 46, 15},
      {143, 241, 99, 179, 5}
    },
    { {56, 73, 248, 205, 2},
      {42, 190, 36, 28, 14},
      {75, 49, 249, 33, 12},
      {115, 130, 71, 213, 4}
    },
    { {186, 114, 36, 10, 6},
      {67, 61, 104, 3, 5},
      {101, 2, 68, 229, 13},
      {172, 1, 107, 204, 2}
    },
    { {100, 162, 99, 127, 3},
      {189, 112, 186, 140, 5},
      {207, 236, 100, 82, 6},
      {163, 21, 208, 235, 13}
    },
    { {132, 80, 221, 43, 3},
      {244, 181, 6, 22, 1},
      {205, 75, 176, 162, 1},
      {134, 134, 10, 210, 15}
    },
    { {233, 134, 163, 18, 6},
      {18, 208, 248, 226, 5},
      {100, 140, 86, 25, 7},
      {164, 113, 240, 180, 8}
    },
    { {31, 106, 10, 155, 13},
      {135, 108, 205, 69, 11},
      {189, 149, 5, 111, 8},
      {218, 43, 51, 110, 1}
    },
    { {193, 131, 19, 193, 9},
      {20, 66, 25, 222, 2},
      {152, 60, 140, 24, 3},
      {71, 185, 132, 34, 8}
    },
    { {68, 244, 102, 59, 11},
      {229, 117, 147, 164, 5},
      {221, 198, 98, 242, 2},
      {162, 92, 154, 234, 7}
    },
    { {1, 157, 216, 198, 10},
      {104, 146, 21, 120, 11},
      {86, 49, 187, 152, 0},
      {209, 234, 132, 145, 6}
    },
    { {207, 106, 252, 4, 8},
      {171, 133, 13, 211, 12},
      {18, 3, 245, 111, 3},
      {60, 187, 10, 29, 5}
    },
    { {146, 174, 195, 5, 6},
      {57, 216, 88, 39, 8},
      {106, 12, 55, 84, 9},
      {30, 65, 161, 185, 12}
    },
    { {37, 21, 93, 154, 5},
      {244, 35, 228, 112, 3},
      {165, 155, 170, 138, 4},
      {192, 226, 124, 66, 15}
    },
    { {22, 10, 92, 69, 6},
      {168, 25, 76, 29, 8},
      {106, 35, 165, 6, 8},
      {27, 131, 41, 129, 5}
    },
    { {42, 195, 144, 107, 5},
      {6, 166, 122, 29, 1},
      {173, 96, 156, 53, 4},
      {139, 133, 230, 86, 0}
    },
    { {46, 122, 182, 246, 15},
      {207, 213, 235, 25, 15},
      {246, 246, 213, 231, 4},
      {249, 141, 122, 191, 3}
    },
    { {158, 250, 147, 214, 12},
      {219, 204, 217, 27, 11},
      {54, 188, 149, 247, 9},
      {221, 137, 179, 61, 11}
    },
    { {70, 52, 147, 36, 2},
      {217, 208, 2, 177, 0},
      {66, 76, 146, 198, 2},
      {8, 212, 0, 185, 11}
    },
    { {10, 253, 84, 150, 12},
      {107, 7, 209, 49, 11},
      {54, 146, 171, 245, 0},
      {216, 200, 190, 13, 6}
    },
    { {176, 98, 61, 37, 0},
      {25, 13, 46, 22, 4},
      {10, 75, 196, 96, 13},
      {38, 135, 75, 9, 8}
    },
    { {145, 210, 194, 45, 8},
      {104, 236, 27, 70, 0},
      {27, 68, 52, 184, 9},
      {6, 45, 131, 113, 6}
    },
    { {180, 12, 140, 215, 2},
      {136, 153, 164, 46, 11},
      {78, 179, 19, 2, 13},
      {215, 66, 89, 145, 1}
    },
    { {159, 137, 21, 197, 9},
      {158, 11, 17, 95, 10},
      {154, 58, 137, 31, 9},
      {95, 168, 141, 7, 9}
    },
    { {231, 171, 31, 92, 9},
      {157, 99, 189, 219, 8},
      {147, 175, 141, 94, 7},
      {29, 187, 220, 107, 9}
    },
    { {64, 95, 208, 111, 7},
      {108, 182, 74, 188, 9},
      {239, 96, 191, 160, 2},
      {147, 213, 38, 211, 6}
    },
    { {107, 30, 197, 224, 6},
      {114, 145, 106, 233, 10},
      {96, 122, 55, 141, 6},
      {89, 117, 104, 148, 14}
    },
    { {172, 27, 229, 165, 10},
      {250, 147, 43, 6, 14},
      {90, 90, 125, 131, 5},
      {118, 13, 76, 149, 15}
    },
    { {230, 47, 3, 108, 10},
      {153, 114, 43, 171, 8},
      {83, 108, 15, 70, 7},
      {29, 93, 68, 233, 9}
    },
    { {127, 229, 102, 248, 8},
      {163, 111, 179, 233, 6},
      {17, 246, 106, 127, 14},
      {105, 124, 223, 108, 5}
    },
    { {163, 194, 62, 50, 14},
      {0, 85, 255, 83, 5},
      {116, 199, 196, 60, 5},
      {172, 175, 250, 160, 0}
    },
    { {148, 62, 117, 118, 6},
      {249, 25, 202, 58, 13},
      {102, 234, 231, 194, 9},
      {181, 197, 57, 137, 15}
    },
    { {52, 186, 30, 90, 15},
      {197, 121, 253, 24, 9},
      {245, 167, 133, 210, 12},
      {145, 139, 249, 234, 3}
    },
    { {3, 165, 18, 156, 11},
      {13, 114, 145, 113, 2},
      {211, 148, 138, 92, 0},
      {72, 232, 148, 235, 0}
    },
    { {37, 82, 144, 131, 7},
      {196, 148, 104, 84, 3},
      {236, 16, 148, 170, 4},
      {194, 161, 98, 146, 3}
    },
    { {152, 56, 107, 83, 15},
      {119, 88, 197, 14, 13},
      {252, 173, 97, 193, 9},
      {183, 10, 49, 174, 14}
    },
    { {81, 8, 229, 57, 2},
      {48, 185, 130, 196, 12},
      {73, 202, 113, 8, 10},
      {50, 52, 25, 208, 12}
    },
    { {109, 47, 236, 46, 14},
      {171, 179, 111, 224, 13},
      {119, 67, 127, 75, 6},
      {176, 127, 108, 221, 5}
    },
    { {82, 188, 16, 202, 10},
      {65, 56, 17, 185, 11},
      {85, 48, 131, 212, 10},
      {217, 216, 129, 200, 2}
    },
    { {179, 40, 54, 67, 10},
      {1, 89, 33, 95, 13},
      {92, 38, 193, 76, 13},
      {191, 168, 73, 168, 0}
    },
    { {119, 53, 50, 215, 6},
      {201, 90, 224, 253, 7},
      {110, 180, 202, 206, 14},
      {235, 240, 117, 169, 3}
    },
    { {60, 107, 123, 81, 7},
      {183, 94, 236, 28, 12},
      {232, 173, 237, 99, 12},
      {51, 131, 119, 174, 13}
    },
    { {58, 210, 85, 76, 8},
      {122, 45, 57, 25, 0},
      {19, 42, 164, 181, 12},
      {9, 137, 203, 69, 14}
    },
    { {121, 133, 192, 110, 10},
      {42, 186, 51, 232, 1},
      {87, 96, 58, 25, 14},
      {129, 124, 197, 213, 4}
    },
    { {197, 187, 154, 199, 7},
      {205, 210, 92, 222, 11},
      {238, 53, 157, 218, 3},
      {215, 179, 164, 187, 3}
    },
    { {12, 197, 122, 240, 11},
      {166, 86, 151, 56, 6},
      {208, 245, 234, 51, 0},
      {97, 206, 150, 166, 5}
    },
    { {248, 208, 240, 214, 2},
      {106, 156, 176, 154, 7},
      {70, 176, 240, 177, 15},
      {229, 144, 211, 149, 6}
    },
    { {129, 63, 211, 226, 11},
      {117, 210, 11, 122, 11},
      {212, 124, 191, 200, 1},
      {213, 237, 4, 186, 14}
    },
    { {13, 236, 216, 183, 12},
      {171, 132, 215, 116, 11},
      {62, 209, 179, 123, 0},
      {210, 238, 178, 29, 5}
    },
    { {27, 74, 227, 19, 2},
      {50, 220, 136, 69, 13},
      {76, 140, 117, 45, 8},
      {186, 33, 19, 180, 12}
    },
    { {210, 32, 64, 67, 5},
      {37, 8, 64, 143, 1},
      {172, 32, 32, 68, 11},
      {143, 16, 33, 10, 4}
    },
    { {20, 134, 143, 224, 4},
      {144, 201, 94, 40, 2},
      {32, 127, 22, 18, 8},
      {65, 71, 169, 48, 9}
    },
    { {129, 156, 58, 252, 10},
      {72, 112, 151, 122, 14},
      {83, 245, 195, 152, 1},
      {117, 238, 144, 225, 2}
    },
    { {69, 204, 88, 112, 14},
      {160, 20, 215, 248, 8},
      {112, 225, 163, 58, 2},
      {17, 254, 178, 128, 5}
    },
    { {85, 209, 44, 27, 7},
      {196, 63, 212, 196, 5},
      {237, 131, 72, 186, 10},
      {162, 50, 191, 194, 3}
    },
    { {124, 82, 159, 25, 14},
      {210, 253, 237, 148, 0},
      {121, 143, 148, 163, 14},
      {2, 155, 123, 244, 11}
    },
    { {112, 145, 54, 145, 1},
      {68, 75, 176, 148, 6},
      {136, 150, 200, 144, 14},
      {98, 144, 221, 34, 2}
    },
    { {252, 90, 66, 132, 10},
      {234, 92, 41, 130, 10},
      {82, 20, 37, 163, 15},
      {84, 25, 67, 165, 7}
    },
    { {87, 151, 148, 193, 3},
      {196, 155, 24, 253, 2},
      {200, 50, 158, 158, 10},
      {75, 241, 141, 146, 3}
    },
    { {165, 209, 196, 248, 2},
      {224, 183, 178, 74, 2},
      {65, 242, 56, 186, 5},
      {69, 36, 222, 208, 7}
    },
    { {129, 60, 188, 144, 8},
      {65, 129, 133, 114, 14},
      {16, 147, 211, 200, 1},
      {116, 234, 24, 24, 2}
    },
    { {9, 113, 70, 213, 3},
      {111, 87, 128, 76, 2},
      {202, 182, 40, 233, 0},
      {67, 32, 30, 175, 6}
    },
    { {106, 207, 210, 216, 5},
      {38, 230, 248, 185, 10},
      {161, 180, 191, 53, 6},
      {89, 209, 246, 118, 4}
    },
    { {115, 244, 119, 116, 9},
      {125, 77, 179, 249, 4},
      {146, 238, 226, 252, 14},
      {41, 252, 219, 43, 14}
    },
    { {245, 157, 172, 185, 10},
      {192, 187, 183, 230, 14},
      {89, 211, 91, 154, 15},
      {118, 126, 221, 208, 3}
    },
    { {202, 249, 118, 232, 4},
      {99, 103, 82, 155, 14},
      {33, 118, 233, 245, 3},
      {125, 148, 174, 108, 6}
    },
    { {145, 223, 213, 147, 14},
      {112, 159, 217, 118, 11},
      {124, 154, 191, 184, 9},
      {214, 233, 191, 144, 14}
    },
    { {151, 224, 139, 96, 13},
      {149, 204, 87, 75, 0},
      {176, 109, 16, 126, 9},
      {13, 46, 163, 58, 9}
    },
    { {171, 24, 195, 41, 7},
      {118, 240, 98, 71, 8},
      {233, 76, 49, 141, 5},
      {30, 36, 96, 246, 14}
    },
    { {12, 117, 233, 207, 4},
      {251, 166, 68, 44, 7},
      {47, 57, 122, 227, 0},
      {227, 66, 38, 93, 15}
    },
    { {18, 88, 90, 126, 11},
      {108, 124, 135, 25, 9},
      {215, 229, 161, 164, 8},
      {153, 142, 19, 227, 6}
    },
    { {226, 61, 82, 49, 9},
      {101, 66, 163, 183, 8},
      {152, 196, 171, 196, 7},
      {30, 220, 84, 42, 6}
    },
    { {148, 62, 206, 46, 10},
      {233, 249, 15, 34, 9},
      {87, 71, 55, 194, 9},
      {148, 79, 9, 249, 7}
    },
    { {77, 128, 42, 254, 11},
      {142, 112, 151, 200, 7},
      {215, 245, 64, 27, 2},
      {225, 62, 144, 231, 1}
    },
    { {99, 222, 29, 74, 1},
      {84, 37, 60, 249, 9},
      {133, 43, 135, 188, 6},
      {153, 243, 202, 66, 10}
    },
    { {234, 149, 240, 24, 8},
      {98, 162, 177, 179, 4},
      {17, 128, 250, 149, 7},
      {44, 216, 212, 84, 6}
    },
    { {20, 204, 99, 128, 10},
      {176, 92, 17, 32, 14},
      {80, 28, 99, 50, 8},
      {112, 72, 131, 160, 13}
    },
    { {200, 176, 49, 75, 15},
      {87, 48, 81, 158, 5},
      {253, 40, 192, 209, 3},
      {167, 152, 160, 206, 10}
    },
    { {158, 5, 82, 110, 15},
      {174, 122, 67, 59, 1},
      {247, 100, 170, 7, 9},
      {141, 204, 37, 231, 5}
    },
    { {90, 49, 246, 81, 3},
      {103, 219, 128, 157, 4},
      {200, 166, 248, 197, 10},
      {43, 144, 29, 190, 6}
    },
    { {2, 64, 141, 174, 9},
      {28, 165, 7, 1, 3},
      {151, 91, 16, 36, 0},
      {200, 14, 10, 83, 8}
    },
    { {3, 39, 204, 189, 6},
      {41, 179, 206, 101, 2},
      {107, 211, 62, 76, 0},
      {74, 103, 60, 217, 4}
    },
    { {57, 199, 200, 136, 5},
      {38, 174, 124, 96, 2},
      {161, 17, 62, 57, 12},
      {64, 99, 231, 86, 4}
    },
    { {29, 241, 95, 61, 5},
      {255, 111, 214, 84, 0},
      {171, 207, 168, 251, 8},
      {2, 166, 191, 111, 15}
    },
    { {53, 248, 87, 146, 8},
      {241, 77, 177, 80, 11},
      {20, 158, 161, 250, 12},
      {208, 168, 219, 40, 15}
    },
    { {246, 170, 194, 38, 15},
      {173, 216, 123, 131, 9},
      {246, 68, 53, 86, 15},
      {156, 29, 225, 187, 5}
    },
    { {231, 224, 91, 154, 9},
      {181, 100, 181, 211, 3},
      {149, 157, 160, 126, 7},
      {204, 186, 210, 106, 13}
    },
    { {35, 84, 88, 78, 2},
      {104, 52, 36, 121, 1},
      {71, 33, 162, 172, 4},
      {137, 226, 66, 193, 6}
    },
    { {168, 240, 123, 208, 12},
      {115, 68, 245, 26, 6},
      {48, 189, 224, 241, 5},
      {101, 138, 242, 44, 14}
    },
    { {31, 123, 242, 62, 3},
      {239, 254, 138, 81, 13},
      {199, 196, 253, 239, 8},
      {184, 165, 23, 255, 7}
    },
    { {166, 34, 215, 254, 13},
      {189, 225, 235, 27, 3},
      {183, 254, 180, 70, 5},
      {205, 141, 120, 123, 13}
    },
    { {60, 224, 174, 102, 10},
      {139, 221, 55, 8, 5},
      {86, 103, 80, 115, 12},
      {161, 14, 203, 189, 1}
    },
    { {157, 234, 252, 246, 9},
      {175, 141, 159, 90, 15},
      {150, 243, 245, 123, 9},
      {245, 175, 155, 31, 5}
    },
    { {118, 28, 15, 110, 1},
      {220, 105, 38, 169, 9},
      {135, 111, 3, 134, 14},
      {153, 86, 73, 99, 11}
    },
    { {23, 229, 15, 201, 12},
      {145, 111, 85, 109, 2},
      {57, 63, 10, 126, 8},
      {75, 106, 175, 104, 9}
    },
    { {107, 215, 2, 212, 12},
      {74, 70, 249, 233, 2},
      {50, 180, 14, 189, 6},
      {73, 121, 246, 37, 2}
    },
    { {86, 44, 136, 80, 8},
      {129, 136, 133, 169, 8},
      {16, 161, 19, 70, 10},
      {25, 90, 17, 24, 1}
    },
    { {117, 18, 248, 29, 15},
      {236, 184, 237, 212, 4},
      {251, 129, 244, 138, 14},
      {34, 187, 113, 211, 7}
    },
    { {126, 225, 73, 21, 0},
      {187, 14, 180, 133, 0},
      {10, 137, 40, 119, 14},
      {10, 18, 215, 13, 13}
    },
    { {16, 118, 198, 246, 1},
      {109, 205, 138, 40, 3},
      {134, 246, 54, 224, 8},
      {193, 69, 27, 59, 6}
    },
    { {212, 235, 35, 223, 2},
      {153, 126, 152, 142, 15},
      {79, 188, 77, 114, 11},
      {247, 17, 151, 233, 9}
    },
    { {40, 207, 171, 134, 10},
      {26, 214, 61, 32, 15},
      {86, 29, 95, 49, 4},
      {240, 75, 198, 181, 8}
    },
    { {172, 20, 113, 133, 3},
      {254, 16, 32, 54, 6},
      {202, 24, 226, 131, 5},
      {102, 192, 64, 135, 15}
    },
    { {175, 152, 175, 103, 14},
      {218, 209, 119, 79, 13},
      {126, 111, 81, 159, 5},
      {191, 46, 232, 181, 11}
    },
    { {193, 158, 114, 157, 1},
      {108, 96, 152, 246, 14},
      {139, 148, 231, 152, 3},
      {118, 241, 144, 99, 6}
    },
    { {207, 95, 181, 152, 6},
      {210, 183, 200, 243, 14},
      {97, 154, 223, 175, 3},
      {124, 241, 62, 212, 11}
    },
    { {213, 221, 187, 147, 2},
      {208, 222, 148, 246, 15},
      {76, 157, 219, 186, 11},
      {246, 242, 151, 176, 11}
    },
    { {79, 134, 37, 109, 3},
      {158, 49, 26, 237, 4},
      {203, 106, 70, 31, 2},
      {43, 117, 136, 199, 9}
    },
    { {106, 36, 71, 111, 0},
      {59, 97, 34, 173, 1},
      {15, 110, 34, 69, 6},
      {139, 84, 72, 109, 12}
    },
    { {128, 45, 147, 26, 1},
      {21, 226, 128, 50, 9},
      {133, 140, 155, 64, 1},
      {148, 192, 20, 122, 8}
    },
    { {29, 205, 131, 180, 9},
      {158, 206, 147, 96, 10},
      {146, 220, 27, 59, 8},
      {80, 108, 151, 55, 9}
    },
    { {218, 134, 193, 229, 12},
      {58, 136, 91, 175, 2},
      {58, 120, 54, 21, 11},
      {79, 93, 161, 21, 12}
    },
    { {230, 246, 109, 46, 3},
      {253, 53, 62, 163, 5},
      {199, 75, 102, 246, 7},
      {172, 87, 202, 203, 15}
    },
    { {61, 137, 16, 149, 7},
      {142, 26, 240, 84, 10},
      {234, 144, 137, 27, 12},
      {82, 160, 245, 135, 1}
    },
    { {210, 121, 65, 88, 12},
      {113, 46, 193, 139, 8},
      {49, 168, 41, 228, 11},
      {29, 24, 55, 72, 14}
    },
    { {160, 175, 28, 189, 14},
      {9, 51, 255, 54, 10},
      {123, 211, 143, 80, 5},
      {86, 207, 252, 201, 0}
    },
    { {1, 87, 110, 34, 13},
      {100, 71, 79, 96, 5},
      {180, 71, 110, 168, 0},
      {160, 111, 46, 34, 6}
    },
    { {87, 248, 236, 245, 8},
      {233, 141, 151, 205, 14},
      {26, 243, 113, 254, 10},
      {123, 62, 155, 25, 7}
    },
    { {106, 255, 41, 18, 1},
      {87, 6, 188, 161, 13},
      {132, 137, 79, 245, 6},
      {184, 83, 214, 14, 10}
    },
    { {251, 33, 174, 202, 0},
      {3, 235, 36, 203, 7},
      {5, 55, 88, 77, 15},
      {237, 50, 77, 124, 0}
    },
    { {11, 139, 189, 114, 11},
      {22, 147, 159, 89, 13},
      {212, 235, 221, 29, 0},
      {185, 175, 156, 150, 8}
    },
    { {212, 152, 51, 194, 13},
      {212, 72, 81, 154, 15},
      {180, 60, 193, 146, 11},
      {245, 152, 161, 34, 11}
    },
    { {225, 75, 0, 230, 14},
      {8, 22, 107, 202, 11},
      {118, 112, 13, 40, 7},
      {213, 61, 102, 129, 0}
    },
    { {26, 95, 155, 85, 5},
      {94, 206, 204, 61, 8},
      {170, 173, 159, 165, 8},
      {27, 195, 55, 55, 10}
    },
    { {240, 142, 136, 165, 15},
      {12, 152, 127, 166, 10},
      {250, 81, 23, 16, 15},
      {86, 95, 225, 147, 0}
    },
    { {167, 153, 98, 167, 6},
      {232, 82, 114, 71, 15},
      {110, 84, 105, 158, 5},
      {254, 36, 228, 161, 7}
    },
    { {117, 137, 204, 235, 6},
      {160, 187, 118, 204, 11},
      {109, 115, 57, 26, 14},
      {211, 54, 237, 208, 5}
    },
    { {146, 223, 143, 67, 12},
      {80, 207, 93, 47, 9},
      {60, 47, 31, 180, 9},
      {159, 75, 175, 48, 10}
    },
    { {174, 86, 155, 223, 2},
      {218, 244, 172, 63, 3},
      {79, 189, 150, 167, 5},
      {207, 195, 82, 245, 11}
    },
    { {210, 12, 103, 74, 4},
      {48, 105, 64, 171, 13},
      {37, 46, 99, 4, 11},
      {189, 80, 41, 96, 12}
    },
    { {40, 70, 65, 78, 7},
      {62, 52, 104, 40, 1},
      {231, 40, 38, 33, 4},
      {129, 65, 98, 199, 12}
    },
    { {108, 201, 54, 127, 1},
      {142, 103, 178, 156, 13},
      {143, 230, 201, 51, 6},
      {179, 148, 222, 103, 1}
    },
    { {194, 238, 156, 100, 12},
      {9, 133, 95, 187, 8},
      {50, 99, 151, 116, 3},
      {29, 223, 170, 25, 0}
    },
    { {36, 163, 220, 7, 0},
      {169, 131, 60, 20, 1},
      {14, 3, 188, 82, 4},
      {130, 131, 204, 25, 5}
    },
    { {41, 169, 53, 183, 11},
      {31, 19, 179, 84, 15},
      {222, 218, 201, 89, 4},
      {242, 172, 220, 143, 8}
    },
    { {105, 9, 143, 242, 0},
      {18, 195, 166, 200, 11},
      {4, 255, 25, 9, 6},
      {209, 54, 92, 52, 8}
    },
    { {221, 179, 45, 144, 2},
      {211, 27, 156, 194, 6},
      {64, 155, 76, 219, 11},
      {100, 51, 157, 140, 11}
    },
    { {17, 24, 73, 168, 13},
      {116, 40, 71, 64, 10},
      {177, 89, 33, 136, 8},
      {80, 46, 33, 66, 14}
    },
    { {182, 179, 217, 164, 5},
      {253, 138, 126, 19, 2},
      {162, 89, 188, 214, 13},
      {76, 135, 229, 27, 15}
    },
    { {160, 207, 230, 60, 2},
      {40, 247, 186, 34, 12},
      {67, 198, 127, 48, 5},
      {52, 69, 222, 241, 4}
    },
    { {55, 59, 166, 59, 8},
      {193, 235, 171, 69, 13},
      {29, 198, 93, 206, 12},
      {186, 45, 93, 120, 3}
    },
    { {161, 242, 208, 206, 8},
      {105, 164, 57, 90, 3},
      {23, 48, 180, 248, 5},
      {197, 169, 194, 89, 6}
    },
    { {86, 143, 18, 81, 15},
      {132, 90, 217, 189, 8},
      {248, 164, 143, 22, 10},
      {27, 217, 181, 162, 1}
    },
    { {194, 231, 12, 94, 10},
      {9, 55, 157, 171, 1},
      {87, 163, 14, 116, 3},
      {141, 91, 158, 201, 0}
    },
    { {156, 25, 191, 90, 0},
      {210, 235, 132, 26, 13},
      {5, 175, 217, 131, 9},
      {181, 130, 29, 116, 11}
    },
    { {117, 207, 171, 65, 6},
      {144, 222, 124, 236, 12},
      {104, 45, 95, 58, 14},
      {51, 115, 231, 176, 9}
    },
    { {126, 204, 41, 109, 0},
      {154, 44, 54, 173, 12},
      {11, 105, 67, 55, 14},
      {59, 86, 195, 69, 9}
    },
    { {243, 91, 227, 26, 14},
      {112, 254, 233, 195, 13},
      {117, 140, 125, 172, 15},
      {188, 57, 119, 240, 14}
    },
    { {190, 117, 133, 241, 8},
      {211, 143, 163, 47, 2},
      {24, 250, 26, 231, 13},
      {79, 76, 95, 28, 11}
    },
    { {33, 154, 217, 93, 10},
      {120, 176, 189, 92, 8},
      {91, 169, 181, 152, 4},
      {19, 171, 208, 209, 14}
    },
    { {187, 162, 184, 116, 12},
      {11, 136, 255, 91, 4},
      {50, 225, 212, 93, 13},
      {45, 175, 241, 29, 0}
    },
    { {24, 170, 15, 78, 6},
      {27, 121, 92, 8, 9},
      {103, 47, 5, 81, 8},
      {145, 3, 169, 237, 8}
    },
    { {141, 163, 255, 65, 12},
      {179, 195, 93, 94, 4},
      {56, 47, 252, 91, 1},
      {39, 171, 172, 60, 13}
    },
    { {204, 241, 13, 15, 1},
      {223, 39, 20, 134, 1},
      {143, 11, 8, 243, 3},
      {134, 18, 142, 79, 11}
    },
    { {15, 249, 178, 82, 13},
      {199, 198, 209, 89, 13},
      {180, 164, 217, 255, 0},
      {185, 168, 182, 62, 3}
    },
    { {1, 159, 5, 220, 10},
      {88, 51, 153, 104, 10},
      {83, 186, 15, 152, 0},
      {81, 105, 156, 193, 10}
    },
    { {223, 138, 224, 53, 4},
      {170, 136, 218, 199, 12},
      {42, 192, 117, 31, 11},
      {62, 53, 177, 21, 5}
    },
    { {206, 123, 160, 13, 5},
      {207, 166, 72, 135, 12},
      {171, 0, 93, 231, 3},
      {62, 17, 38, 95, 3}
    },
    { {217, 125, 237, 88, 6},
      {115, 191, 196, 234, 12},
      {97, 171, 123, 233, 11},
      {53, 114, 63, 220, 14}
    },
    { {181, 147, 30, 12, 6},
      {200, 123, 124, 82, 0},
      {99, 7, 140, 154, 13},
      {4, 163, 237, 225, 3}
    },
    { {18, 139, 154, 110, 4},
      {8, 234, 94, 25, 9},
      {39, 101, 157, 20, 8},
      {153, 135, 165, 113, 0}
    },
    { {242, 22, 60, 166, 3},
      {76, 25, 46, 179, 7},
      {198, 83, 198, 132, 15},
      {236, 215, 73, 131, 2}
    },
    { {232, 10, 7, 115, 7},
      {22, 81, 234, 142, 9},
      {236, 238, 5, 1, 7},
      {151, 21, 120, 166, 8}
    },
    { {122, 115, 30, 194, 4},
      {67, 79, 108, 153, 3},
      {36, 55, 140, 229, 14},
      {201, 147, 111, 44, 2}
    },
    { {53, 218, 126, 240, 11},
      {228, 93, 191, 88, 14},
      {208, 247, 229, 186, 12},
      {113, 175, 219, 162, 7}
    },
    { {153, 130, 253, 45, 1},
      {62, 169, 30, 86, 4},
      {139, 75, 244, 25, 9},
      {38, 167, 137, 87, 12}
    },
    { {223, 169, 37, 94, 13},
      {159, 43, 209, 203, 13},
      {183, 170, 73, 95, 11},
      {189, 56, 189, 79, 9}
    },
    { {165, 30, 214, 188, 15},
      {236, 241, 235, 114, 10},
      {243, 214, 183, 138, 5},
      {84, 237, 120, 243, 7}
    },
    { {36, 8, 213, 209, 9},
      {180, 129, 161, 28, 10},
      {152, 186, 177, 2, 4},
      {83, 136, 88, 18, 13}
    },
    { {181, 248, 233, 45, 6},
      {249, 188, 118, 70, 12},
      {107, 73, 113, 250, 13},
      {54, 38, 227, 217, 15}
    },
    { {145, 54, 188, 78, 0},
      {73, 169, 12, 122, 5},
      {7, 35, 214, 200, 9},
      {165, 227, 9, 89, 2}
    },
    { {174, 88, 224, 232, 5},
      {230, 164, 98, 11, 14},
      {161, 112, 113, 167, 5},
      {125, 4, 98, 86, 7}
    },
    { {47, 128, 197, 170, 11},
      {182, 177, 51, 65, 3},
      {213, 90, 48, 31, 4},
      {200, 44, 200, 214, 13}
    },
    { {218, 73, 123, 17, 9},
      {54, 78, 133, 151, 12},
      {152, 141, 233, 37, 11},
      {62, 154, 23, 38, 12}
    },
    { {162, 246, 228, 83, 14},
      {97, 149, 249, 47, 5},
      {124, 162, 118, 244, 5},
      {175, 73, 250, 152, 6}
    },
    { {33, 39, 109, 132, 12},
      {57, 3, 109, 96, 6},
      {50, 27, 110, 72, 4},
      {96, 107, 108, 9, 12}
    },
    { {107, 121, 40, 104, 12},
      {67, 38, 103, 201, 12},
      {49, 97, 73, 237, 6},
      {57, 62, 102, 76, 2}
    },
    { {157, 218, 93, 177, 13},
      {246, 13, 223, 86, 10},
      {184, 219, 165, 187, 9},
      {86, 175, 187, 6, 15}
    },
    { {47, 207, 178, 181, 5},
      {142, 198, 250, 117, 14},
      {170, 212, 223, 63, 4},
      {122, 229, 246, 55, 1}
    },
    { {175, 32, 71, 30, 8},
      {187, 97, 161, 67, 1},
      {23, 142, 32, 79, 5},
      {140, 40, 88, 109, 13}
    },
    { {234, 199, 75, 58, 9},
      {54, 102, 191, 163, 1},
      {149, 205, 46, 53, 7},
      {140, 95, 214, 102, 12}
    },
    { {204, 229, 29, 252, 14},
      {155, 55, 215, 186, 2},
      {115, 251, 138, 115, 3},
      {69, 222, 190, 205, 9}
    },
    { {97, 48, 133, 50, 1},
      {85, 129, 162, 192, 1},
      {132, 202, 16, 200, 6},
      {128, 52, 88, 26, 10}
    },
    { {180, 146, 25, 41, 1},
      {212, 40, 62, 22, 0},
      {137, 73, 132, 146, 13},
      {6, 135, 193, 66, 11}
    },
    { {193, 207, 189, 157, 10},
      {24, 183, 157, 246, 14},
      {91, 155, 223, 56, 3},
      {118, 251, 158, 209, 8}
    },
    { {8, 221, 26, 54, 6},
      {74, 86, 214, 48, 9},
      {102, 197, 139, 177, 0},
      {144, 198, 182, 165, 2}
    },
    { {120, 67, 92, 168, 15},
      {38, 63, 111, 144, 2},
      {241, 83, 172, 33, 14},
      {64, 159, 111, 198, 4}
    },
    { {32, 220, 186, 83, 14},
      {64, 212, 245, 60, 13},
      {124, 165, 211, 176, 4},
      {179, 202, 242, 176, 2}
    },
    { {227, 132, 165, 236, 6},
      {24, 177, 114, 235, 6},
      {99, 122, 82, 28, 7},
      {109, 116, 232, 209, 8}
    },
    { {131, 145, 171, 122, 13},
      {84, 226, 215, 75, 5},
      {181, 237, 88, 156, 1},
      {173, 46, 180, 114, 10}
    },
    { {114, 38, 60, 80, 11},
      {5, 25, 173, 185, 4},
      {208, 163, 198, 68, 14},
      {41, 219, 89, 138, 0}
    },
    { {242, 174, 11, 171, 12},
      {17, 104, 127, 167, 11},
      {61, 93, 7, 84, 15},
      {222, 95, 225, 104, 8}
    },
    { {46, 136, 9, 214, 3},
      {158, 16, 180, 9, 11},
      {198, 185, 1, 23, 4},
      {217, 2, 208, 135, 9}
    },
    { {183, 203, 246, 151, 12},
      {168, 207, 249, 87, 15},
      {62, 150, 253, 62, 13},
      {254, 169, 255, 49, 5}
    },
    { {49, 251, 244, 57, 14},
      {97, 191, 251, 84, 12},
      {121, 194, 253, 248, 12},
      {50, 173, 255, 216, 6}
    },
    { {142, 199, 82, 154, 6},
      {162, 118, 216, 51, 3},
      {101, 148, 174, 55, 1},
      {204, 193, 182, 228, 5}
    },
    { {94, 235, 242, 10, 12},
      {163, 238, 89, 145, 13},
      {53, 4, 253, 119, 10},
      {184, 153, 167, 124, 5}
    },
    { {227, 59, 129, 212, 12},
      {89, 130, 233, 203, 10},
      {50, 184, 29, 204, 7},
      {93, 57, 116, 25, 10}
    },
    { {172, 124, 242, 222, 10},
      {235, 244, 161, 58, 15},
      {87, 180, 243, 227, 5},
      {245, 200, 82, 253, 7}
    },
    { {78, 1, 1, 147, 2},
      {146, 18, 128, 133, 3},
      {76, 152, 8, 7, 2},
      {202, 16, 20, 132, 9}
    },
    { {201, 144, 131, 96, 5},
      {86, 192, 82, 202, 0},
      {160, 108, 16, 153, 3},
      {5, 52, 160, 54, 10}
    },
    { {245, 102, 105, 160, 13},
      {181, 12, 111, 226, 6},
      {176, 89, 102, 106, 15},
      {100, 127, 99, 10, 13}
    },
    { {44, 3, 150, 212, 14},
      {138, 211, 233, 24, 2},
      {114, 182, 156, 3, 4},
      {65, 137, 124, 181, 1}
    },
    { {118, 193, 117, 204, 7},
      {188, 63, 112, 153, 6},
      {227, 58, 232, 54, 14},
      {105, 144, 239, 195, 13}
    },
    { {176, 251, 83, 107, 5},
      {117, 110, 122, 30, 9},
      {173, 108, 173, 240, 13},
      {151, 133, 231, 106, 14}
    },
    { {135, 53, 206, 169, 8},
      {225, 227, 7, 103, 2},
      {25, 87, 58, 206, 1},
      {78, 110, 12, 120, 7}
    },
    { {231, 139, 244, 106, 11},
      {164, 179, 59, 219, 13},
      {213, 98, 253, 30, 7},
      {189, 189, 204, 210, 5}
    },
    { {185, 178, 1, 226, 1},
      {87, 8, 58, 74, 3},
      {132, 120, 4, 217, 13},
      {197, 37, 193, 14, 10}
    },
    { {44, 230, 244, 199, 9},
      {175, 133, 57, 60, 7},
      {158, 50, 246, 115, 4},
      {227, 201, 202, 31, 5}
    },
    { {240, 26, 58, 172, 4},
      {72, 104, 110, 146, 14},
      {35, 85, 197, 128, 15},
      {116, 151, 97, 97, 2}
    },
    { {107, 29, 238, 143, 4},
      {106, 227, 100, 229, 15},
      {47, 23, 123, 141, 6},
      {250, 114, 108, 117, 6}
    },
    { {10, 99, 89, 22, 11},
      {63, 22, 141, 17, 1},
      {214, 137, 172, 101, 0},
      {136, 139, 22, 143, 12}
    },
    { {61, 80, 237, 142, 0},
      {250, 173, 36, 64, 7},
      {7, 27, 112, 171, 12},
      {224, 34, 75, 85, 15}
    },
    { {12, 234, 137, 42, 4},
      {147, 164, 94, 0, 9},
      {37, 73, 21, 115, 0},
      {144, 7, 162, 92, 9}
    },
    { {146, 159, 44, 154, 3},
      {68, 59, 156, 35, 15},
      {197, 147, 79, 148, 9},
      {252, 67, 157, 194, 2}
    },
    { {144, 185, 171, 44, 8},
      {89, 234, 23, 2, 12},
      {19, 77, 89, 208, 9},
      {52, 14, 133, 121, 10}
    },
    { {96, 223, 140, 227, 4},
      {64, 135, 126, 172, 11},
      {44, 115, 31, 176, 6},
      {211, 87, 238, 16, 2}
    },
    { {211, 163, 99, 9, 11},
      {53, 122, 25, 199, 4},
      {217, 12, 108, 92, 11},
      {46, 57, 133, 234, 12}
    },
    { {37, 61, 159, 195, 3},
      {213, 211, 36, 124, 11},
      {204, 63, 155, 202, 4},
      {211, 226, 76, 186, 11}
    },
    { {48, 25, 30, 15, 13},
      {76, 107, 101, 20, 9},
      {191, 7, 137, 128, 12},
      {146, 138, 109, 99, 2}
    },
    { {90, 151, 28, 16, 1},
      {70, 11, 156, 177, 0},
      {128, 131, 142, 149, 10},
      {8, 211, 157, 6, 2}
    },
    { {42, 15, 107, 51, 10},
      {50, 82, 175, 37, 13},
      {92, 205, 111, 5, 4},
      {186, 79, 84, 164, 12}
    },
    { {106, 117, 209, 236, 0},
      {123, 166, 34, 185, 2},
      {3, 120, 186, 229, 6},
      {73, 212, 70, 93, 14}
    },
    { {30, 157, 187, 139, 1},
      {214, 234, 20, 53, 15},
      {141, 29, 219, 151, 8},
      {250, 194, 133, 118, 11}
    },
    { {47, 58, 225, 220, 9},
      {255, 160, 169, 73, 14},
      {147, 184, 117, 207, 4},
      {121, 41, 80, 95, 15}
    },
    { {99, 208, 151, 52, 7},
      {92, 213, 242, 209, 0},
      {226, 206, 144, 188, 6},
      {8, 180, 250, 179, 10}
    },
    { {216, 157, 137, 67, 7},
      {86, 154, 84, 174, 9},
      {236, 41, 27, 145, 11},
      {151, 82, 165, 150, 10}
    },
    { {131, 189, 136, 80, 11},
      {69, 146, 149, 107, 8},
      {208, 161, 27, 220, 1},
      {29, 106, 148, 154, 2}
    },
    { {34, 78, 125, 138, 14},
      {48, 53, 109, 49, 15},
      {117, 27, 231, 36, 4},
      {248, 203, 106, 192, 12}
    },
    { {121, 75, 63, 158, 10},
      {26, 127, 173, 208, 15},
      {87, 159, 205, 41, 14},
      {240, 187, 95, 229, 8}
    },
    { {141, 51, 167, 110, 15},
      {223, 243, 75, 74, 5},
      {247, 110, 92, 203, 1},
      {165, 45, 44, 255, 11}
    },
    { {33, 133, 188, 205, 7},
      {12, 179, 116, 124, 6},
      {235, 51, 218, 24, 4},
      {99, 226, 236, 211, 0}
    },
    { {240, 218, 227, 80, 3},
      {116, 220, 184, 138, 12},
      {192, 172, 117, 176, 15},
      {53, 17, 211, 178, 14}
    },
    { {204, 223, 42, 42, 8},
      {194, 102, 31, 162, 13},
      {21, 69, 79, 179, 3},
      {180, 95, 134, 100, 3}
    },
    { {56, 130, 84, 157, 1},
      {46, 41, 184, 20, 2},
      {139, 146, 164, 17, 12},
      {66, 129, 217, 71, 4}
    },
    { {110, 45, 44, 91, 14},
      {131, 51, 229, 173, 13},
      {125, 163, 75, 71, 6},
      {187, 90, 124, 204, 1}
    },
    { {122, 43, 200, 47, 2},
      {43, 186, 46, 133, 9},
      {79, 65, 61, 69, 14},
      {154, 23, 69, 221, 4}
    },
    { {87, 219, 131, 112, 8},
      {208, 206, 155, 201, 8},
      {16, 236, 29, 190, 10},
      {25, 61, 151, 48, 11}
    },
    { {201, 217, 206, 10, 6},
      {98, 247, 84, 194, 9},
      {101, 7, 57, 185, 3},
      {148, 50, 174, 244, 6}
    },
    { {140, 81, 7, 168, 12},
      {210, 103, 67, 2, 2},
      {49, 94, 8, 163, 1},
      {68, 12, 46, 100, 11}
    },
    { {189, 143, 171, 242, 11},
      {150, 218, 191, 106, 15},
      {212, 253, 95, 27, 13},
      {245, 111, 213, 182, 9}
    },
    { {100, 133, 26, 10, 13},
      {132, 98, 117, 176, 1},
      {181, 5, 138, 18, 6},
      {128, 218, 228, 98, 1}
    },
    { {220, 109, 58, 176, 4},
      {131, 78, 198, 178, 14},
      {32, 213, 203, 99, 11},
      {116, 214, 55, 44, 1}
    },
    { {231, 251, 25, 168, 3},
      {213, 54, 62, 211, 10},
      {193, 89, 141, 254, 7},
      {92, 183, 198, 202, 11}
    },
    { {177, 204, 84, 80, 5},
      {36, 13, 240, 122, 8},
      {160, 162, 163, 56, 13},
      {21, 224, 251, 2, 4}
    },
    { {97, 110, 245, 148, 7},
      {61, 149, 232, 240, 14},
      {226, 154, 247, 104, 6},
      {112, 241, 122, 155, 12}
    },
    { {147, 83, 11, 252, 1},
      {92, 110, 142, 75, 2},
      {131, 253, 12, 172, 9},
      {77, 39, 23, 99, 10}
    },
    { {27, 41, 1, 82, 8},
      {19, 10, 129, 73, 9},
      {20, 168, 9, 77, 8},
      {153, 40, 21, 12, 8}
    },
    { {197, 245, 7, 104, 1},
      {213, 103, 18, 234, 0},
      {129, 110, 10, 250, 3},
      {5, 116, 142, 106, 11}
    },
    { {52, 186, 247, 153, 2},
      {241, 249, 184, 20, 14},
      {73, 158, 245, 210, 12},
      {114, 129, 217, 248, 15}
    },
    { {169, 176, 189, 187, 13},
      {87, 161, 247, 86, 7},
      {189, 219, 208, 217, 5},
      {230, 174, 248, 94, 10}
    },
    { {149, 57, 193, 12, 5},
      {253, 170, 64, 66, 8},
      {163, 8, 57, 202, 9},
      {20, 32, 37, 91, 15}
    },
    { {50, 223, 204, 191, 2},
      {104, 191, 190, 37, 11},
      {79, 211, 63, 180, 12},
      {218, 71, 223, 209, 6}
    },
    { {237, 234, 172, 111, 7},
      {143, 181, 126, 206, 13},
      {239, 99, 85, 123, 7},
      {183, 55, 234, 223, 1}
    },
    { {77, 86, 15, 172, 10},
      {218, 117, 15, 224, 2},
      {83, 95, 6, 171, 2},
      {64, 127, 10, 229, 11}
    },
    { {116, 172, 190, 43, 0},
      {129, 233, 54, 180, 13},
      {13, 71, 211, 82, 14},
      {178, 214, 201, 120, 1}
    },
    { {35, 182, 120, 236, 5},
      {109, 32, 126, 121, 6},
      {163, 113, 230, 220, 4},
      {105, 231, 224, 75, 6}
    },
    { {232, 210, 137, 53, 1},
      {94, 132, 190, 134, 0},
      {138, 201, 20, 177, 7},
      {6, 23, 210, 23, 10}
    },
    { {235, 150, 185, 93, 6},
      {90, 176, 252, 255, 4},
      {107, 169, 214, 157, 7},
      {47, 243, 240, 213, 10}
    },
    { {85, 145, 91, 105, 0},
      {240, 106, 22, 220, 0},
      {9, 109, 168, 154, 10},
      {3, 182, 133, 96, 15}
    },
    { {40, 186, 189, 218, 6},
      {83, 177, 252, 24, 15},
      {101, 187, 213, 209, 4},
      {241, 131, 248, 220, 10}
    },
    { {35, 163, 75, 130, 11},
      {53, 82, 61, 65, 3},
      {212, 29, 44, 92, 4},
      {200, 43, 196, 170, 12}
    },
    { {193, 172, 20, 84, 10},
      {9, 17, 145, 250, 8},
      {82, 162, 131, 88, 3},
      {21, 248, 152, 137, 0}
    },
    { {100, 182, 175, 131, 2},
      {209, 209, 60, 164, 7},
      {76, 31, 86, 210, 6},
      {226, 83, 200, 184, 11}
    },
    { {82, 25, 155, 54, 1},
      {92, 202, 134, 145, 9},
      {134, 205, 153, 132, 10},
      {152, 150, 21, 51, 10}
    },
    { {99, 183, 61, 54, 4},
      {89, 3, 254, 241, 5},
      {38, 203, 206, 220, 6},
      {168, 247, 252, 9, 10}
    },
    { {192, 85, 112, 85, 8},
      {104, 6, 129, 190, 4},
      {26, 160, 234, 160, 3},
      {39, 216, 22, 1, 6}
    },
    { {52, 181, 171, 169, 13},
      {213, 234, 119, 36, 6},
      {185, 93, 90, 210, 12},
      {98, 78, 229, 122, 11}
    },
    { {4, 214, 173, 116, 7},
      {220, 149, 222, 40, 4},
      {226, 235, 86, 178, 0},
      {33, 71, 186, 147, 11}
    },
    { {58, 185, 185, 47, 15},
      {95, 186, 119, 21, 13},
      {255, 73, 217, 213, 12},
      {186, 142, 229, 223, 10}
    },
    { {30, 193, 204, 115, 3},
      {166, 159, 150, 13, 1},
      {204, 227, 56, 55, 8},
      {139, 6, 159, 150, 5}
    },
    { {227, 95, 151, 12, 12},
      {88, 231, 105, 243, 8},
      {51, 14, 159, 172, 7},
      {28, 249, 110, 113, 10}
    },
    { {82, 233, 202, 244, 7},
      {45, 222, 214, 137, 10},
      {226, 245, 57, 116, 10},
      {89, 22, 183, 187, 4}
    },
    { {171, 99, 139, 22, 6},
      {27, 214, 236, 67, 1},
      {102, 141, 28, 109, 5},
      {140, 35, 118, 189, 8}
    },
    { {138, 229, 208, 83, 7},
      {39, 150, 208, 63, 1},
      {236, 160, 186, 117, 1},
      {143, 192, 182, 158, 4}
    },
    { {103, 173, 81, 43, 1},
      {181, 34, 50, 245, 9},
      {141, 72, 171, 94, 6},
      {154, 244, 196, 74, 13}
    },
    { {179, 24, 122, 4, 11},
      {108, 88, 37, 83, 12},
      {210, 5, 225, 140, 13},
      {60, 170, 65, 163, 6}
    },
    { {60, 198, 137, 195, 8},
      {146, 140, 61, 44, 3},
      {28, 57, 22, 51, 12},
      {195, 75, 195, 20, 9}
    },
    { {250, 116, 11, 175, 7},
      {95, 124, 102, 167, 3},
      {239, 93, 2, 229, 15},
      {206, 86, 99, 239, 10}
    },
    { {4, 250, 221, 72, 11},
      {245, 181, 29, 24, 8},
      {209, 43, 181, 242, 0},
      {17, 139, 138, 218, 15}
    },
    { {153, 138, 132, 196, 12},
      {10, 137, 89, 74, 10},
      {50, 50, 21, 25, 9},
      {85, 41, 169, 21, 0}
    },
    { {63, 252, 218, 241, 1},
      {231, 204, 182, 125, 10},
      {136, 245, 179, 255, 12},
      {91, 230, 211, 62, 7}
    },
    { {148, 250, 34, 140, 9},
      {205, 108, 25, 2, 14},
      {147, 20, 69, 242, 9},
      {116, 9, 131, 107, 3}
    },
    { {228, 237, 60, 145, 13},
      {133, 7, 245, 182, 14},
      {184, 147, 203, 114, 7},
      {118, 218, 254, 10, 1}
    },
    { {3, 55, 138, 20, 12},
      {73, 194, 205, 97, 0},
      {50, 133, 30, 204, 0},
      {8, 107, 52, 57, 2}
    },
    { {216, 27, 34, 208, 1},
      {70, 74, 136, 138, 14},
      {128, 180, 77, 129, 11},
      {117, 17, 21, 38, 2}
    },
    { {25, 91, 18, 85, 10},
      {74, 94, 137, 92, 8},
      {90, 164, 141, 169, 8},
      {19, 169, 23, 165, 2}
    },
    { {33, 72, 120, 182, 2},
      {40, 20, 166, 80, 15},
      {70, 209, 225, 40, 4},
      {240, 166, 82, 129, 4}
    },
    { {40, 93, 236, 19, 2},
      {98, 151, 164, 36, 13},
      {76, 131, 123, 161, 4},
      {178, 66, 94, 148, 6}
    },
    { {123, 100, 52, 77, 7},
      {15, 61, 96, 253, 4},
      {235, 34, 194, 109, 14},
      {43, 240, 107, 207, 0}
    },
    { {98, 75, 151, 138, 7},
      {20, 247, 104, 145, 11},
      {229, 30, 157, 36, 6},
      {216, 145, 110, 242, 8}
    },
    { {103, 119, 188, 11, 1},
      {197, 167, 44, 245, 5},
      {141, 3, 222, 238, 6},
      {170, 243, 78, 90, 3}
    },
    { {144, 144, 113, 245, 10},
      {120, 24, 147, 30, 6},
      {90, 248, 224, 144, 9},
      {103, 140, 145, 129, 14}
    },
    { {37, 206, 233, 123, 13},
      {180, 164, 255, 108, 13},
      {189, 233, 119, 58, 4},
      {179, 111, 242, 82, 13}
    },
    { {253, 83, 136, 111, 13},
      {206, 174, 111, 206, 1},
      {191, 97, 28, 171, 15},
      {135, 63, 103, 87, 3}
    },
    { {231, 105, 53, 36, 11},
      {157, 23, 35, 211, 12},
      {210, 74, 201, 110, 7},
      {60, 188, 78, 139, 9}
    },
    { {24, 201, 137, 127, 9},
      {30, 174, 151, 12, 9},
      {159, 233, 25, 49, 8},
      {147, 14, 151, 87, 8}
    },
    { {101, 96, 64, 176, 10},
      {161, 20, 163, 192, 2},
      {80, 208, 32, 106, 6},
      {64, 60, 82, 136, 5}
    },
    { {56, 46, 157, 194, 13},
      {23, 137, 109, 56, 11},
      {180, 59, 151, 65, 12},
      {209, 203, 105, 30, 8}
    },
    { {205, 242, 177, 37, 13},
      {223, 132, 91, 214, 4},
      {186, 72, 212, 251, 3},
      {38, 189, 162, 31, 11}
    },
    { {9, 78, 179, 61, 6},
      {26, 244, 202, 116, 12},
      {107, 204, 215, 41, 0},
      {50, 229, 50, 245, 8}
    },
    { {57, 223, 214, 61, 1},
      {110, 239, 186, 116, 8},
      {139, 198, 191, 185, 12},
      {18, 229, 223, 119, 6}
    },
    { {249, 54, 192, 90, 7},
      {103, 184, 232, 234, 1},
      {229, 160, 54, 201, 15},
      {133, 113, 113, 222, 6}
    },
    { {119, 243, 69, 1, 5},
      {245, 15, 120, 197, 0},
      {168, 10, 44, 254, 14},
      {10, 49, 239, 10, 15}
    },
    { {229, 170, 108, 152, 3},
      {165, 49, 188, 194, 14},
      {193, 147, 101, 90, 7},
      {116, 51, 216, 202, 5}
    },
    { {176, 232, 167, 19, 14},
      {17, 221, 241, 6, 13},
      {124, 142, 81, 112, 13},
      {182, 8, 251, 184, 8}
    },
    { {221, 219, 120, 40, 6},
      {226, 62, 94, 210, 12},
      {97, 65, 237, 187, 11},
      {52, 183, 167, 196, 7}
    },
    { {59, 9, 157, 107, 2},
      {18, 187, 38, 93, 9},
      {77, 107, 153, 13, 12},
      {155, 166, 77, 212, 8}
    },
    { {64, 234, 217, 35, 3},
      {53, 148, 30, 148, 9},
      {204, 73, 181, 112, 2},
      {146, 151, 130, 154, 12}
    },
    { {26, 247, 252, 44, 4},
      {107, 175, 94, 49, 4},
      {35, 67, 254, 245, 8},
      {40, 199, 175, 93, 6}
    },
    { {60, 154, 211, 10, 11},
      {246, 248, 57, 16, 9},
      {213, 12, 181, 147, 12},
      {144, 137, 193, 246, 15}
    },
    { {179, 6, 81, 36, 2},
      {56, 24, 42, 115, 0},
      {66, 72, 166, 12, 13},
      {12, 229, 65, 129, 12}
    },
    { {50, 189, 243, 128, 9},
      {117, 202, 49, 49, 14},
      {144, 28, 251, 212, 12},
      {120, 200, 197, 58, 14}
    },
    { {108, 225, 50, 82, 6},
      {131, 86, 240, 152, 5},
      {100, 164, 200, 115, 6},
      {161, 144, 246, 172, 1}
    },
    { {87, 93, 59, 77, 8},
      {216, 110, 5, 253, 12},
      {27, 45, 203, 174, 10},
      {59, 250, 7, 97, 11}
    },
    { {50, 250, 120, 150, 11},
      {109, 28, 189, 17, 15},
      {214, 145, 229, 244, 12},
      {248, 139, 211, 139, 6}
    },
    { {201, 37, 74, 207, 11},
      {47, 114, 5, 238, 3},
      {223, 53, 42, 73, 3},
      {199, 122, 4, 239, 4}
    },
    { {91, 227, 88, 153, 8},
      {35, 46, 157, 213, 2},
      {25, 145, 172, 125, 10},
      {74, 187, 151, 76, 4}
    },
    { {162, 249, 225, 35, 2},
      {113, 150, 50, 7, 13},
      {76, 72, 121, 244, 5},
      {190, 4, 198, 152, 14}
    },
    { {119, 181, 82, 205, 9},
      {237, 106, 49, 253, 2},
      {155, 52, 170, 222, 14},
      {75, 248, 197, 107, 7}
    },
    { {176, 219, 205, 212, 10},
      {120, 159, 189, 10, 10},
      {82, 187, 61, 176, 13},
      {85, 11, 223, 145, 14}
    },
    { {203, 147, 21, 157, 14},
      {90, 51, 217, 215, 2},
      {123, 154, 140, 157, 3},
      {78, 185, 188, 197, 10}
    },
    { {126, 39, 103, 62, 11},
      {191, 123, 171, 161, 5},
      {215, 206, 110, 71, 14},
      {168, 93, 93, 239, 13}
    },
    { {27, 68, 157, 3, 7},
      {22, 157, 68, 117, 1},
      {236, 11, 146, 45, 8},
      {138, 226, 43, 150, 8}
    },
    { {214, 213, 57, 224, 1},
      {212, 14, 22, 187, 6},
      {128, 121, 202, 182, 11},
      {109, 214, 135, 2, 11}
    },
    { {153, 17, 59, 93, 5},
      {94, 106, 196, 94, 4},
      {171, 173, 200, 137, 9},
      {39, 162, 53, 103, 10}
    },
    { {237, 201, 222, 175, 0},
      {170, 231, 54, 214, 11},
      {15, 87, 185, 59, 7},
      {214, 182, 206, 117, 5}
    },
    { {152, 79, 114, 58, 0},
      {34, 110, 138, 50, 13},
      {5, 196, 239, 33, 9},
      {180, 197, 23, 100, 4}
    },
    { {146, 177, 194, 94, 11},
      {109, 250, 145, 11, 1},
      {215, 164, 56, 212, 9},
      {141, 8, 149, 251, 6}
    },
    { {233, 83, 55, 170, 15},
      {86, 119, 107, 210, 7},
      {245, 94, 204, 169, 7},
      {228, 189, 110, 230, 10}
    },
    { {184, 249, 145, 224, 11},
      {87, 158, 51, 26, 10},
      {208, 120, 153, 241, 13},
      {85, 140, 199, 158, 10}
    },
    { {54, 190, 41, 6, 4},
      {217, 8, 124, 33, 13},
      {38, 9, 71, 214, 12},
      {184, 67, 225, 9, 11}
    },
    { {109, 88, 75, 223, 15},
      {254, 116, 229, 204, 11},
      {255, 189, 33, 171, 6},
      {211, 58, 114, 231, 15}
    },
    { {124, 73, 255, 32, 11},
      {182, 223, 39, 144, 12},
      {208, 79, 249, 35, 14},
      {48, 158, 79, 182, 13}
    },
    { {204, 4, 237, 38, 9},
      {190, 129, 7, 162, 5},
      {150, 75, 114, 3, 3},
      {164, 94, 8, 23, 13}
    },
    { {126, 191, 195, 88, 12},
      {243, 234, 249, 169, 8},
      {49, 172, 63, 215, 14},
      {25, 89, 245, 124, 15}
    },
    { {35, 71, 74, 52, 11},
      {44, 86, 175, 97, 0},
      {210, 197, 46, 44, 4},
      {8, 111, 86, 163, 4}
    },
    { {247, 2, 101, 153, 14},
      {176, 57, 233, 199, 6},
      {121, 154, 100, 14, 15},
      {110, 57, 121, 192, 13}
    },
    { {183, 134, 11, 177, 9},
      {148, 72, 191, 103, 2},
      {152, 221, 6, 30, 13},
      {78, 111, 209, 34, 9}
    },
    { {29, 237, 84, 211, 10},
      {163, 31, 145, 124, 11},
      {92, 178, 171, 123, 8},
      {211, 232, 159, 140, 5}
    },
    { {19, 129, 28, 244, 14},
      {8, 27, 215, 89, 2},
      {114, 243, 136, 28, 8},
      {73, 174, 189, 129, 0}
    },
    { {38, 136, 183, 166, 7},
      {156, 209, 114, 17, 15},
      {230, 94, 209, 22, 4},
      {248, 132, 232, 179, 9}
    },
    { {218, 189, 5, 242, 12},
      {83, 11, 211, 171, 11},
      {52, 250, 11, 213, 11},
      {221, 92, 189, 12, 10}
    },
    { {179, 196, 25, 242, 6},
      {16, 28, 246, 123, 3},
      {100, 249, 130, 60, 13},
      {205, 230, 243, 128, 8}
    },
    { {3, 88, 126, 158, 12},
      {104, 101, 197, 81, 15},
      {55, 151, 225, 172, 0},
      {248, 170, 58, 97, 6}
    },
    { {138, 226, 81, 223, 4},
      {59, 36, 216, 31, 3},
      {47, 184, 164, 117, 1},
      {207, 129, 178, 77, 12}
    },
    { {116, 44, 226, 207, 6},
      {169, 248, 96, 172, 15},
      {111, 52, 115, 66, 14},
      {243, 80, 97, 249, 5}
    },
    { {80, 87, 235, 180, 4},
      {120, 206, 206, 160, 6},
      {34, 221, 126, 160, 10},
      {96, 87, 55, 49, 14}
    },
    { {234, 122, 109, 152, 13},
      {119, 37, 237, 131, 14},
      {177, 155, 101, 229, 7},
      {124, 27, 122, 78, 14}
    },
    { {81, 124, 23, 242, 2},
      {81, 93, 130, 248, 11},
      {68, 254, 131, 232, 10},
      {209, 244, 27, 168, 10}
    },
    { {116, 206, 76, 238, 0},
      {168, 45, 62, 168, 11},
      {7, 115, 39, 50, 14},
      {209, 87, 203, 65, 5}
    },
    { {235, 112, 71, 178, 6},
      {115, 85, 226, 195, 3},
      {100, 222, 32, 237, 7},
      {204, 52, 122, 172, 14}
    },
    { {44, 21, 121, 54, 0},
      {250, 2, 166, 48, 5},
      {6, 201, 234, 131, 4},
      {160, 198, 84, 5, 15}
    },
    { {10, 102, 96, 115, 1},
      {39, 4, 138, 45, 5},
      {140, 224, 102, 101, 0},
      {171, 69, 18, 14, 4}
    },
    { {15, 250, 22, 84, 3},
      {207, 85, 152, 89, 8},
      {194, 166, 133, 255, 0},
      {25, 161, 154, 175, 3}
    },
    { {248, 13, 152, 242, 6},
      {2, 154, 230, 186, 11},
      {100, 241, 155, 1, 15},
      {213, 214, 117, 148, 0}
    },
    { {17, 81, 213, 137, 2},
      {112, 191, 0, 84, 2},
      {73, 26, 184, 168, 8},
      {66, 160, 15, 208, 14}
    },
    { {90, 127, 72, 72, 2},
      {99, 62, 12, 169, 8},
      {65, 33, 47, 229, 10},
      {25, 83, 7, 204, 6}
    },
    { {220, 246, 137, 159, 6},
      {219, 188, 220, 166, 3},
      {111, 153, 22, 243, 11},
      {198, 83, 179, 221, 11}
    },
    { {142, 116, 239, 41, 6},
      {243, 245, 70, 39, 4},
      {105, 79, 114, 231, 1},
      {46, 70, 42, 252, 15}
    },
    { {122, 205, 165, 235, 5},
      {22, 175, 114, 173, 15},
      {173, 122, 91, 53, 14},
      {251, 84, 239, 86, 8}
    },
    { {73, 74, 158, 70, 15},
      {14, 213, 77, 216, 9},
      {246, 39, 149, 41, 2},
      {145, 187, 42, 183, 0}
    },
    { {92, 188, 59, 49, 3},
      {215, 88, 150, 180, 12},
      {200, 205, 195, 211, 10},
      {50, 214, 145, 174, 11}
    },
    { {24, 172, 214, 61, 15},
      {47, 249, 211, 52, 8},
      {251, 198, 179, 81, 8},
      {18, 204, 185, 255, 4}
    },
    { {128, 152, 198, 222, 6},
      {104, 241, 208, 10, 11},
      {103, 182, 49, 144, 1},
      {213, 0, 184, 241, 6}
    },
    { {171, 158, 93, 173, 10},
      {122, 49, 63, 119, 10},
      {91, 91, 167, 157, 5},
      {94, 239, 200, 197, 14}
    },
    { {187, 215, 181, 242, 15},
      {86, 159, 251, 123, 7},
      {244, 250, 222, 189, 13},
      {237, 237, 255, 150, 10}
    },
    { {231, 64, 19, 227, 5},
      {148, 68, 98, 223, 3},
      {172, 124, 128, 46, 7},
      {207, 180, 98, 34, 9}
    },
    { {185, 180, 142, 161, 12},
      {67, 201, 119, 102, 2},
      {56, 87, 18, 217, 13},
      {70, 110, 233, 60, 2}
    },
    { {45, 174, 203, 210, 6},
      {179, 208, 252, 104, 11},
      {100, 189, 55, 91, 4},
      {209, 99, 240, 188, 13}
    },
    { {14, 251, 136, 89, 3},
      {199, 182, 156, 13, 8},
      {201, 161, 29, 247, 0},
      {27, 3, 150, 222, 3}
    },
    { {140, 253, 170, 193, 1},
      {199, 198, 20, 46, 14},
      {136, 53, 91, 243, 1},
      {119, 66, 134, 62, 3}
    },
    { {80, 221, 255, 96, 6},
      {112, 223, 86, 184, 12},
      {96, 111, 251, 176, 10},
      {49, 214, 175, 176, 14}
    },
    { {171, 67, 201, 91, 13},
      {54, 166, 237, 79, 1},
      {189, 169, 60, 45, 5},
      {143, 43, 118, 86, 12}
    },
    { {80, 242, 159, 163, 14},
      {81, 221, 95, 148, 3},
      {124, 95, 148, 240, 10},
      {194, 159, 171, 184, 10}
    },
    { {128, 166, 71, 246, 14},
      {57, 81, 219, 42, 3},
      {118, 254, 38, 80, 1},
      {197, 77, 184, 169, 12}
    },
    { {60, 168, 157, 27, 10},
      {147, 185, 181, 20, 9},
      {93, 139, 145, 83, 12},
      {146, 138, 217, 220, 9}
    },
    { {24, 77, 221, 97, 1},
      {54, 143, 6, 60, 8},
      {136, 107, 187, 33, 8},
      {19, 198, 15, 22, 12}
    },
    { {83, 127, 71, 115, 5},
      {117, 79, 202, 237, 9},
      {172, 238, 47, 236, 10},
      {155, 117, 63, 42, 14}
    },
    { {164, 149, 152, 181, 4},
      {200, 130, 246, 54, 2},
      {42, 209, 154, 146, 5},
      {70, 198, 244, 17, 3}
    },
    { {58, 161, 233, 220, 1},
      {63, 170, 180, 9, 6},
      {131, 185, 120, 85, 12},
      {105, 2, 213, 95, 12}
    },
    { {72, 193, 6, 108, 11},
      {14, 119, 19, 136, 0},
      {211, 102, 8, 49, 2},
      {1, 28, 142, 231, 0}
    },
    { {227, 219, 6, 53, 1},
      {76, 71, 186, 199, 8},
      {138, 198, 13, 188, 7},
      {30, 53, 222, 35, 2}
    },
    { {26, 241, 58, 17, 0},
      {67, 78, 148, 21, 4},
      {8, 133, 200, 245, 8},
      {42, 130, 151, 44, 2}
    },
    { {137, 31, 189, 213, 1},
      {94, 131, 140, 126, 14},
      {138, 187, 223, 137, 1},
      {119, 227, 28, 23, 10}
    },
    { {142, 68, 6, 133, 7},
      {142, 85, 64, 39, 2},
      {234, 22, 2, 39, 1},
      {78, 64, 42, 167, 1}
    },
    { {135, 127, 42, 107, 5},
      {197, 102, 78, 111, 13},
      {173, 101, 79, 238, 1},
      {191, 103, 38, 106, 3}
    },
    { {122, 152, 134, 35, 6},
      {66, 217, 114, 133, 9},
      {108, 70, 17, 149, 14},
      {154, 20, 233, 180, 2}
    },
    { {228, 197, 141, 119, 10},
      {152, 151, 183, 174, 1},
      {94, 235, 26, 50, 7},
      {135, 94, 222, 145, 9}
    },
    { {70, 156, 46, 177, 8},
      {192, 65, 151, 165, 14},
      {24, 215, 67, 150, 2},
      {122, 94, 152, 32, 3}
    },
    { {165, 170, 62, 63, 8},
      {137, 97, 191, 86, 13},
      {31, 199, 197, 90, 5},
      {182, 175, 216, 105, 1}
    },
    { {14, 105, 46, 93, 5},
      {143, 103, 196, 13, 12},
      {171, 167, 73, 103, 0},
      {59, 2, 62, 111, 1}
    },
    { {6, 57, 165, 147, 9},
      {213, 131, 129, 5, 15},
      {156, 154, 89, 198, 0},
      {250, 8, 28, 26, 11}
    },
    { {96, 63, 93, 167, 6},
      {121, 19, 110, 180, 11},
      {110, 91, 175, 192, 6},
      {210, 215, 108, 137, 14}
    },
    { {72, 219, 238, 150, 9},
      {110, 199, 157, 128, 15},
      {150, 151, 125, 177, 2},
      {240, 27, 158, 55, 6}
    },
    { {95, 215, 109, 233, 5},
      {246, 47, 94, 237, 6},
      {169, 123, 110, 191, 10},
      {107, 119, 175, 70, 15}
    },
    { {101, 185, 180, 188, 6},
      {201, 179, 242, 208, 14},
      {99, 210, 217, 218, 6},
      {112, 180, 252, 217, 3}
    },
    { {6, 179, 0, 213, 15},
      {205, 18, 217, 13, 2},
      {250, 176, 12, 214, 0},
      {75, 9, 180, 139, 3}
    },
    { {76, 98, 251, 222, 11},
      {191, 244, 141, 152, 7},
      {215, 189, 244, 99, 2},
      {225, 155, 18, 255, 13}
    },
    { {154, 177, 225, 193, 15},
      {119, 154, 81, 15, 6},
      {248, 56, 120, 213, 9},
      {111, 8, 165, 158, 14}
    },
    { {116, 98, 52, 56, 13},
      {133, 45, 235, 144, 4},
      {177, 194, 196, 98, 14},
      {32, 157, 123, 74, 1}
    },
    { {229, 34, 138, 114, 8},
      {129, 192, 175, 202, 1},
      {20, 229, 20, 74, 7},
      {133, 63, 80, 56, 1}
    },
    { {20, 217, 9, 82, 15},
      {212, 30, 213, 8, 9},
      {244, 169, 9, 178, 8},
      {145, 10, 183, 130, 11}
    },
    { {218, 24, 89, 233, 7},
      {118, 56, 70, 159, 10},
      {233, 121, 161, 133, 11},
      {95, 150, 33, 198, 14}
    },
    { {38, 80, 144, 16, 9},
      {196, 132, 161, 17, 0},
      {144, 128, 144, 166, 4},
      {8, 136, 82, 18, 3}
    },
    { {25, 235, 51, 121, 9},
      {23, 110, 155, 92, 12},
      {153, 236, 205, 121, 8},
      {51, 173, 151, 110, 8}
    },
    { {105, 80, 30, 18, 12},
      {66, 69, 229, 208, 1},
      {52, 135, 128, 169, 6},
      {128, 186, 122, 36, 2}
    },
    { {150, 139, 94, 59, 15},
      {164, 123, 223, 23, 9},
      {253, 199, 173, 22, 9},
      {158, 143, 189, 226, 5}
    },
    { {218, 116, 48, 6, 5},
      {79, 12, 64, 179, 5},
      {166, 0, 194, 229, 11},
      {172, 208, 35, 15, 2}
    },
    { {25, 179, 22, 144, 13},
      {71, 75, 217, 80, 2},
      {176, 150, 140, 217, 8},
      {64, 169, 189, 46, 2}
    },
    { {25, 3, 122, 37, 8},
      {42, 74, 15, 84, 4},
      {26, 69, 236, 9, 8},
      {34, 175, 5, 37, 4}
    },
    { {45, 131, 24, 137, 9},
      {134, 34, 61, 84, 2},
      {153, 17, 140, 27, 4},
      {66, 171, 196, 70, 1}
    },
    { {90, 179, 165, 245, 1},
      {95, 139, 154, 141, 6},
      {138, 250, 92, 213, 10},
      {107, 21, 157, 31, 10}
    },
    { {122, 166, 126, 219, 8},
      {35, 105, 189, 189, 7},
      {29, 183, 230, 85, 14},
      {235, 219, 217, 108, 4}
    },
    { {74, 39, 236, 111, 9},
      {47, 163, 15, 173, 5},
      {159, 99, 126, 69, 2},
      {171, 95, 12, 95, 4}
    },
    { {199, 32, 161, 106, 9},
      {149, 160, 3, 203, 5},
      {149, 104, 80, 78, 3},
      {173, 60, 0, 90, 9}
    },
    { {70, 101, 77, 162, 0},
      {177, 7, 6, 161, 3},
      {4, 91, 42, 102, 2},
      {200, 86, 14, 8, 13}
    },
    { {69, 170, 40, 84, 15},
      {141, 16, 221, 200, 12},
      {242, 161, 69, 90, 2},
      {49, 59, 176, 139, 1}
    },
    { {174, 121, 83, 131, 6},
      {243, 86, 96, 23, 11},
      {108, 28, 169, 231, 5},
      {222, 128, 102, 172, 15}
    },
    { {156, 98, 82, 80, 12},
      {163, 76, 201, 26, 0},
      {48, 164, 164, 99, 9},
      {5, 137, 51, 44, 5}
    },
    { {12, 88, 52, 54, 3},
      {206, 21, 130, 16, 13},
      {198, 194, 193, 163, 0},
      {176, 132, 26, 135, 3}
    },
    { {74, 45, 0, 174, 1},
      {15, 34, 2, 161, 11},
      {135, 80, 11, 69, 2},
      {216, 84, 4, 79, 0}
    },
    { {116, 221, 28, 199, 10},
      {200, 31, 53, 188, 11},
      {94, 51, 139, 178, 14},
      {211, 218, 207, 129, 3}
    },
    { {255, 184, 139, 185, 11},
      {215, 248, 183, 199, 10},
      {217, 221, 17, 223, 15},
      {94, 62, 209, 254, 11}
    },
    { {72, 11, 173, 35, 1},
      {22, 131, 14, 132, 13},
      {140, 75, 93, 1, 2},
      {178, 23, 12, 22, 8}
    },
    { {7, 41, 136, 152, 1},
      {133, 162, 132, 65, 10},
      {129, 145, 25, 78, 0},
      {88, 34, 20, 90, 1}
    },
    { {230, 29, 184, 139, 11},
      {196, 178, 37, 183, 15},
      {221, 17, 219, 134, 7},
      {254, 218, 68, 210, 3}
    },
    { {115, 138, 131, 230, 3},
      {28, 216, 58, 201, 11},
      {198, 124, 21, 28, 14},
      {217, 53, 193, 179, 8}
    },
    { {39, 0, 88, 211, 10},
      {160, 16, 165, 93, 3},
      {92, 177, 160, 14, 4},
      {203, 170, 80, 128, 5}
    },
    { {99, 133, 237, 25, 0},
      {48, 163, 180, 229, 4},
      {9, 139, 122, 28, 6},
      {42, 114, 220, 80, 12}
    },
    { {43, 58, 207, 150, 12},
      {123, 193, 237, 65, 11},
      {54, 159, 53, 205, 4},
      {216, 43, 120, 61, 14}
    },
    { {219, 208, 152, 69, 3},
      {78, 156, 20, 223, 0},
      {202, 33, 144, 189, 11},
      {15, 178, 131, 151, 2}
    },
    { {157, 205, 210, 135, 6},
      {170, 222, 80, 118, 11},
      {110, 20, 187, 59, 9},
      {214, 224, 167, 181, 5}
    },
    { {49, 244, 150, 171, 1},
      {69, 237, 50, 116, 3},
      {141, 86, 146, 248, 12},
      {194, 228, 203, 122, 2}
    },
    { {31, 47, 77, 37, 3},
      {191, 27, 14, 101, 8},
      {202, 75, 47, 79, 8},
      {26, 103, 13, 143, 13}
    },
    { {238, 22, 15, 225, 14},
      {210, 81, 111, 175, 2},
      {120, 127, 6, 135, 7},
      {79, 95, 104, 164, 11}
    },
    { {10, 161, 221, 53, 1},
      {63, 131, 150, 21, 0},
      {138, 203, 184, 85, 0},
      {10, 134, 156, 31, 12}
    },
    { {86, 173, 47, 172, 10},
      {153, 123, 23, 161, 14},
      {83, 95, 75, 86, 10},
      {120, 94, 141, 233, 9}
    },
    { {211, 126, 245, 50, 15},
      {117, 157, 203, 243, 13},
      {244, 202, 247, 236, 11},
      {188, 253, 59, 154, 14}
    },
    { {26, 77, 168, 98, 11},
      {6, 158, 7, 41, 13},
      {212, 97, 91, 37, 8},
      {185, 78, 7, 150, 0}
    },
    { {14, 14, 83, 226, 12},
      {178, 64, 75, 57, 11},
      {52, 124, 167, 7, 0},
      {217, 205, 32, 36, 13}
    },
    { {186, 19, 25, 227, 12},
      {82, 10, 111, 31, 3},
      {60, 121, 140, 133, 13},
      {207, 143, 101, 4, 10}
    },
    { {166, 38, 10, 6, 2},
      {137, 80, 44, 35, 1},
      {70, 5, 6, 70, 5},
      {140, 67, 64, 169, 1}
    },
    { {59, 165, 11, 14, 1},
      {31, 106, 52, 97, 1},
      {135, 13, 10, 93, 12},
      {136, 98, 197, 111, 8}
    },
    { {38, 36, 208, 138, 6},
      {161, 176, 96, 49, 3},
      {101, 16, 178, 70, 4},
      {200, 192, 96, 216, 5}
    },
    { {34, 170, 93, 184, 11},
      {53, 49, 191, 17, 10},
      {209, 219, 165, 84, 4},
      {88, 143, 216, 202, 12}
    },
    { {204, 89, 30, 234, 11},
      {198, 119, 7, 154, 11},
      {213, 119, 137, 163, 3},
      {213, 158, 14, 230, 3}
    },
    { {93, 163, 232, 36, 7},
      {175, 154, 94, 192, 4},
      {226, 65, 124, 91, 10},
      {32, 55, 165, 159, 5}
    },
    { {49, 70, 31, 250, 13},
      {20, 109, 239, 120, 3},
      {181, 255, 134, 40, 12},
      {193, 239, 123, 98, 8}
    },
    { {225, 228, 77, 220, 5},
      {61, 37, 244, 234, 2},
      {163, 187, 34, 120, 7},
      {69, 114, 250, 75, 12}
    },
    { {44, 242, 109, 104, 5},
      {247, 37, 126, 8, 4},
      {161, 107, 100, 243, 4},
      {33, 7, 234, 78, 15}
    },
    { {188, 184, 23, 132, 2},
      {219, 89, 48, 18, 10},
      {66, 30, 129, 211, 13},
      {84, 128, 201, 173, 11}
    },
    { {229, 105, 162, 218, 12},
      {129, 230, 225, 202, 15},
      {53, 180, 89, 106, 7},
      {245, 56, 118, 120, 1}
    },
    { {126, 187, 69, 235, 1},
      {247, 43, 58, 141, 11},
      {141, 122, 45, 215, 14},
      {219, 21, 205, 78, 15}
    },
    { {95, 193, 180, 208, 10},
      {130, 159, 145, 217, 6},
      {80, 178, 216, 63, 10},
      {105, 184, 159, 148, 1}
    },
    { {67, 231, 225, 87, 9},
      {61, 134, 153, 237, 5},
      {158, 168, 126, 124, 2},
      {171, 121, 150, 27, 12}
    },
    { {74, 231, 170, 10, 11},
      {7, 246, 29, 161, 5},
      {213, 5, 94, 117, 2},
      {168, 91, 134, 254, 0}
    },
    { {165, 185, 85, 105, 15},
      {245, 51, 115, 94, 8},
      {249, 106, 169, 218, 5},
      {23, 172, 236, 202, 15}
    },
    { {224, 84, 118, 144, 13},
      {100, 69, 225, 178, 6},
      {176, 150, 226, 160, 7},
      {100, 216, 122, 34, 6}
    },
    { {210, 45, 14, 153, 2},
      {1, 123, 132, 167, 10},
      {73, 151, 11, 68, 11},
      {94, 82, 29, 232, 0}
    },
    { {24, 168, 223, 230, 1},
      {63, 201, 22, 24, 11},
      {134, 127, 177, 81, 8},
      {209, 134, 137, 63, 12}
    },
    { {69, 219, 175, 11, 10},
      {208, 247, 29, 196, 13},
      {93, 15, 93, 186, 2},
      {178, 59, 142, 240, 11}
    },
    { {120, 139, 137, 141, 8},
      {26, 170, 61, 132, 10},
      {27, 25, 29, 17, 14},
      {82, 27, 197, 85, 8}
    },
    { {241, 221, 156, 23, 1},
      {76, 143, 180, 246, 9},
      {142, 131, 155, 184, 15},
      {150, 242, 223, 19, 2}
    },
    { {168, 83, 82, 179, 8},
      {98, 70, 171, 22, 3},
      {28, 212, 172, 161, 5},
      {198, 141, 86, 36, 6}
    },
    { {168, 106, 55, 142, 9},
      {31, 101, 41, 18, 15},
      {151, 30, 197, 97, 5},
      {244, 137, 74, 111, 8}
    },
    { {189, 126, 84, 235, 14},
      {227, 61, 107, 126, 11},
      {125, 114, 167, 235, 13},
      {215, 237, 107, 204, 7}
    },
    { {223, 220, 164, 141, 1},
      {206, 173, 16, 231, 14},
      {139, 18, 83, 191, 11},
      {126, 112, 139, 87, 3}
    },
    { {164, 184, 123, 136, 7},
      {245, 112, 116, 18, 14},
      {225, 29, 225, 210, 5},
      {116, 130, 224, 234, 15}
    },
    { {156, 22, 106, 229, 0},
      {234, 72, 14, 46, 6},
      {10, 117, 102, 131, 9},
      {103, 71, 1, 37, 7}
    },
    { {7, 66, 172, 242, 15},
      {132, 149, 207, 73, 7},
      {244, 243, 84, 46, 0},
      {233, 47, 58, 146, 1}
    },
    { {76, 120, 158, 87, 8},
      {203, 197, 133, 156, 9},
      {30, 167, 145, 227, 2},
      {147, 154, 26, 61, 3}
    },
    { {154, 239, 33, 84, 12},
      {27, 14, 217, 43, 12},
      {50, 168, 79, 117, 9},
      {61, 73, 183, 13, 8}
    },
    { {114, 33, 143, 98, 6},
      {17, 219, 102, 137, 1},
      {100, 111, 24, 68, 14},
      {137, 22, 109, 184, 8}
    },
    { {144, 22, 79, 84, 2},
      {120, 89, 140, 42, 0},
      {66, 175, 38, 128, 9},
      {5, 67, 25, 161, 14}
    },
    { {255, 38, 45, 47, 8},
      {155, 41, 47, 231, 5},
      {31, 75, 70, 79, 15},
      {174, 127, 73, 77, 9}
    },
    { {155, 75, 33, 101, 11},
      {30, 30, 11, 79, 12},
      {218, 104, 77, 45, 9},
      {63, 45, 7, 135, 8}
    },
    { {157, 250, 242, 161, 11},
      {231, 220, 27, 86, 14},
      {216, 84, 245, 251, 9},
      {118, 173, 131, 190, 7}
    },
    { {88, 151, 203, 43, 2},
      {114, 250, 30, 164, 1},
      {77, 77, 62, 145, 10},
      {130, 87, 133, 244, 14}
    },
    { {71, 47, 124, 245, 10},
      {169, 19, 143, 253, 14},
      {90, 243, 239, 78, 2},
      {123, 255, 28, 137, 5}
    },
    { {231, 58, 247, 133, 1},
      {253, 193, 40, 215, 14},
      {138, 30, 245, 206, 7},
      {126, 177, 72, 59, 15}
    },
    { {207, 215, 242, 172, 4},
      {234, 230, 90, 243, 6},
      {35, 84, 254, 191, 3},
      {108, 245, 166, 117, 7}
    },
    { {34, 145, 90, 60, 13},
      {108, 98, 247, 17, 0},
      {179, 197, 168, 148, 4},
      {8, 142, 244, 99, 6}
    },
    { {34, 133, 70, 209, 9},
      {36, 67, 177, 45, 2},
      {152, 182, 42, 20, 4},
      {75, 72, 220, 34, 4}
    },
    { {114, 208, 93, 189, 9},
      {124, 45, 183, 149, 2},
      {155, 219, 160, 180, 14},
      {74, 158, 219, 67, 14}
    },
    { {66, 217, 149, 82, 0},
      {80, 135, 144, 153, 9},
      {4, 170, 153, 180, 2},
      {153, 144, 158, 16, 10}
    },
    { {138, 133, 121, 144, 1},
      {54, 2, 148, 51, 6},
      {128, 153, 234, 21, 1},
      {108, 194, 148, 6, 12}
    },
    { {197, 198, 42, 24, 4},
      {128, 100, 220, 226, 4},
      {33, 133, 70, 58, 3},
      {36, 115, 178, 96, 1}
    },
    { {233, 151, 88, 80, 13},
      {102, 2, 253, 250, 0},
      {176, 161, 174, 153, 7},
      {5, 251, 244, 6, 6}
    },
    { {14, 234, 177, 70, 7},
      {159, 148, 88, 25, 13},
      {230, 40, 213, 119, 0},
      {185, 129, 162, 159, 9}
    },
    { {145, 53, 173, 2, 13},
      {85, 139, 69, 98, 5},
      {180, 11, 90, 200, 9},
      {164, 106, 45, 26, 10}
    },
    { {190, 141, 200, 78, 4},
      {170, 170, 116, 43, 9},
      {39, 33, 59, 23, 13},
      {157, 66, 229, 85, 5}
    },
    { {173, 173, 184, 75, 2},
      {131, 178, 52, 126, 13},
      {77, 33, 219, 91, 5},
      {183, 226, 196, 220, 1}
    },
    { {239, 18, 224, 176, 8},
      {226, 128, 171, 195, 6},
      {16, 208, 116, 143, 7},
      {108, 61, 80, 20, 7}
    },
    { {110, 30, 54, 228, 4},
      {202, 65, 106, 185, 14},
      {34, 118, 199, 135, 6},
      {121, 213, 104, 37, 3}
    },
    { {182, 167, 125, 25, 13},
      {181, 43, 253, 55, 4},
      {185, 139, 238, 86, 13},
      {46, 203, 253, 74, 13}
    },
    { {23, 86, 83, 139, 13},
      {244, 108, 73, 117, 3},
      {189, 28, 166, 174, 8},
      {202, 233, 35, 98, 15}
    },
    { {95, 18, 213, 36, 1},
      {254, 137, 10, 209, 0},
      {130, 74, 180, 143, 10},
      {8, 181, 9, 23, 15}
    },
    { {70, 117, 172, 246, 3},
      {205, 151, 134, 169, 7},
      {198, 243, 90, 230, 2},
      {233, 86, 30, 155, 3}
    },
    { {60, 106, 94, 42, 12},
      {163, 109, 111, 16, 9},
      {53, 71, 165, 99, 12},
      {144, 143, 107, 108, 5}
    },
    { {36, 27, 3, 248, 15},
      {212, 114, 235, 8, 10},
      {241, 252, 13, 130, 4},
      {81, 13, 116, 226, 11}
    },
    { {174, 183, 170, 241, 10},
      {195, 210, 191, 47, 6},
      {88, 245, 94, 215, 5},
      {111, 79, 212, 188, 3}
    },
    { {128, 123, 11, 149, 5},
      {93, 70, 204, 6, 10},
      {170, 157, 13, 224, 1},
      {86, 3, 54, 43, 10}
    },
    { {6, 123, 185, 142, 2},
      {217, 182, 12, 17, 15},
      {71, 25, 221, 230, 0},
      {248, 131, 6, 217, 11}
    },
    { {187, 205, 97, 202, 11},
      {54, 62, 49, 107, 15},
      {213, 56, 107, 61, 13},
      {253, 104, 199, 198, 12}
    },
    { {102, 111, 155, 27, 2},
      {145, 246, 172, 181, 9},
      {77, 141, 159, 102, 6},
      {154, 211, 86, 248, 9}
    },
    { {220, 128, 108, 119, 5},
      {174, 9, 214, 142, 5},
      {174, 227, 96, 19, 11},
      {167, 22, 185, 7, 5}
    },
    { {244, 76, 119, 37, 13},
      {188, 77, 99, 182, 12},
      {186, 78, 227, 34, 15},
      {54, 220, 107, 35, 13}
    },
    { {114, 219, 75, 25, 1},
      {116, 110, 188, 133, 8},
      {137, 141, 45, 180, 14},
      {26, 19, 215, 98, 14}
    },
    { {16, 18, 108, 127, 9},
      {108, 41, 143, 12, 5},
      {159, 227, 100, 128, 8},
      {163, 15, 25, 67, 6}
    },
    { {162, 1, 124, 141, 13},
      {44, 35, 101, 23, 6},
      {187, 19, 232, 4, 5},
      {110, 138, 108, 67, 4}
    },
    { {92, 43, 98, 199, 13},
      {175, 74, 73, 140, 15},
      {190, 52, 109, 67, 10},
      {243, 25, 37, 47, 5}
    },
    { {246, 17, 249, 50, 13},
      {244, 138, 231, 147, 5},
      {180, 201, 248, 134, 15},
      {172, 158, 117, 18, 15}
    },
    { {29, 202, 246, 89, 3},
      {166, 253, 152, 92, 12},
      {201, 166, 245, 59, 8},
      {51, 161, 155, 246, 5}
    },
    { {246, 222, 90, 183, 0},
      {232, 76, 190, 183, 11},
      {14, 213, 167, 182, 15},
      {222, 215, 211, 33, 7}
    },
    { {2, 227, 95, 133, 4},
      {57, 71, 92, 21, 2},
      {42, 31, 172, 116, 0},
      {74, 131, 174, 41, 12}
    },
    { {154, 82, 200, 49, 13},
      {102, 140, 207, 7, 0},
      {184, 193, 52, 165, 9},
      {14, 15, 51, 22, 6}
    },
    { {201, 156, 238, 248, 13},
      {102, 225, 215, 234, 14},
      {177, 247, 115, 153, 3},
      {117, 126, 184, 118, 6}
    },
    { {164, 0, 185, 178, 3},
      {148, 144, 166, 18, 7},
      {196, 217, 208, 2, 5},
      {228, 134, 80, 146, 9}
    },
    { {83, 189, 160, 149, 3},
      {77, 154, 144, 229, 14},
      {202, 144, 91, 220, 10},
      {122, 112, 149, 155, 2}
    },
    { {229, 156, 80, 168, 12},
      {224, 32, 115, 242, 10},
      {49, 80, 163, 154, 7},
      {84, 252, 224, 64, 7}
    },
    { {171, 95, 148, 193, 7},
      {70, 151, 104, 127, 10},
      {232, 50, 159, 173, 5},
      {95, 225, 110, 150, 2}
    },
    { {154, 1, 159, 43, 13},
      {22, 235, 71, 23, 1},
      {189, 79, 152, 5, 9},
      {142, 142, 45, 118, 8}
    },
    { {179, 111, 58, 61, 12},
      {9, 110, 239, 119, 12},
      {59, 197, 207, 108, 13},
      {62, 239, 119, 105, 0}
    },
    { {192, 187, 244, 25, 3},
      {101, 179, 152, 150, 12},
      {201, 130, 253, 208, 3},
      {54, 145, 156, 218, 6}
    },
    { {20, 123, 78, 197, 0},
      {233, 79, 12, 12, 10},
      {10, 55, 45, 226, 8},
      {83, 3, 15, 41, 7}
    },
    { {90, 199, 71, 252, 14},
      {58, 127, 219, 169, 2},
      {115, 254, 46, 53, 10},
      {73, 93, 191, 229, 12}
    },
    { {55, 33, 111, 157, 0},
      {185, 107, 164, 69, 6},
      {11, 159, 104, 78, 12},
      {106, 34, 93, 105, 13}
    },
    { {1, 111, 252, 42, 3},
      {37, 183, 14, 112, 13},
      {197, 67, 255, 104, 0},
      {176, 231, 14, 218, 4}
    },
    { {89, 15, 175, 61, 8},
      {26, 235, 143, 228, 12},
      {27, 207, 95, 9, 10},
      {50, 127, 29, 117, 8}
    },
    { {153, 1, 84, 216, 10},
      {34, 59, 129, 90, 2},
      {81, 178, 168, 9, 9},
      {69, 168, 29, 196, 4}
    },
    { {67, 238, 76, 167, 7},
      {45, 21, 94, 229, 11},
      {238, 83, 39, 124, 2},
      {218, 119, 170, 139, 4}
    },
    { {54, 90, 193, 205, 0},
      {248, 172, 40, 13, 10},
      {11, 56, 53, 166, 12},
      {91, 1, 67, 81, 15}
    },
    { {131, 166, 230, 183, 5},
      {45, 193, 218, 103, 7},
      {174, 214, 118, 92, 1},
      {238, 101, 184, 59, 4}
    },
    { {19, 38, 191, 52, 9},
      {29, 201, 143, 113, 4},
      {146, 207, 214, 76, 8},
      {40, 239, 25, 59, 8}
    },
    { {75, 112, 99, 4, 11},
      {127, 84, 1, 193, 4},
      {210, 12, 96, 237, 2},
      {40, 56, 2, 175, 14}
    },
    { {138, 249, 189, 185, 10},
      {83, 183, 151, 23, 14},
      {89, 219, 217, 245, 1},
      {126, 142, 158, 220, 10}
    },
    { {210, 150, 211, 86, 13},
      {124, 200, 217, 187, 1},
      {182, 172, 182, 148, 11},
      {141, 217, 177, 51, 14}
    },
    { {191, 171, 59, 228, 2},
      {155, 90, 62, 91, 14},
      {66, 125, 205, 95, 13},
      {125, 167, 197, 173, 9}
    },
    { {174, 86, 204, 98, 6},
      {226, 149, 110, 43, 1},
      {100, 99, 54, 167, 5},
      {141, 71, 106, 148, 7}
    },
    { {178, 87, 177, 0, 1},
      {84, 142, 40, 51, 4},
      {128, 8, 222, 164, 13},
      {44, 193, 71, 18, 10}
    },
    { {62, 59, 212, 245, 0},
      {235, 139, 170, 29, 10},
      {10, 242, 189, 199, 12},
      {91, 133, 93, 29, 7}
    },
    { {81, 226, 98, 81, 13},
      {37, 76, 217, 204, 4},
      {184, 164, 100, 120, 10},
      {35, 57, 179, 42, 4}
    },
    { {109, 172, 163, 64, 13},
      {151, 192, 113, 232, 12},
      {176, 44, 83, 91, 6},
      {49, 120, 224, 62, 9}
    },
    { {140, 97, 186, 38, 3},
      {143, 214, 6, 18, 5},
      {198, 69, 216, 99, 1},
      {164, 134, 6, 191, 1}
    },
    { {188, 245, 45, 34, 0},
      {211, 15, 54, 34, 5},
      {4, 75, 74, 243, 13},
      {164, 70, 207, 12, 11}
    },
    { {211, 103, 251, 121, 3},
      {53, 254, 142, 255, 4},
      {201, 237, 254, 108, 11},
      {47, 247, 23, 250, 12}
    },
    { {84, 87, 19, 84, 12},
      {216, 78, 201, 184, 0},
      {50, 172, 142, 162, 10},
      {1, 217, 55, 33, 11}
    },
    { {252, 69, 102, 73, 14},
      {162, 127, 97, 174, 4},
      {121, 38, 106, 35, 15},
      {39, 88, 111, 228, 5}
    },
    { {201, 116, 9, 180, 3},
      {95, 20, 134, 226, 2},
      {194, 217, 2, 233, 3},
      {68, 118, 18, 143, 10}
    },
    { {219, 223, 58, 117, 6},
      {74, 94, 222, 255, 12},
      {106, 229, 207, 189, 11},
      {63, 247, 183, 165, 2}
    },
    { {124, 198, 54, 99, 4},
      {130, 77, 122, 188, 5},
      {44, 102, 198, 51, 14},
      {163, 213, 235, 36, 1}
    },
    { {199, 37, 21, 237, 7},
      {157, 51, 66, 255, 2},
      {235, 122, 138, 78, 3},
      {79, 244, 44, 203, 9}
    },
    { {243, 129, 78, 141, 3},
      {44, 123, 52, 199, 2},
      {203, 23, 40, 28, 15},
      {78, 50, 205, 227, 4}
    },
    { {1, 97, 236, 83, 14},
      {33, 151, 197, 76, 5},
      {124, 163, 120, 104, 0},
      {163, 42, 62, 152, 4}
    },
    { {185, 10, 129, 53, 14},
      {26, 152, 235, 70, 8},
      {122, 200, 21, 9, 13},
      {22, 45, 113, 149, 8}
    },
    { {206, 10, 5, 154, 5},
      {150, 33, 200, 131, 11},
      {165, 154, 5, 7, 3},
      {220, 17, 56, 70, 9}
    },
    { {177, 143, 157, 57, 3},
      {20, 187, 190, 118, 8},
      {201, 203, 159, 24, 13},
      {22, 231, 221, 210, 8}
    },
    { {148, 176, 8, 30, 13},
      {205, 40, 213, 2, 1},
      {183, 129, 0, 210, 9},
      {132, 10, 177, 75, 3}
    },
    { {80, 66, 146, 213, 15},
      {12, 220, 201, 156, 2},
      {250, 180, 148, 32, 10},
      {67, 153, 51, 179, 0}
    },
    { {35, 150, 114, 80, 11},
      {100, 80, 185, 121, 4},
      {208, 164, 230, 156, 4},
      {41, 233, 208, 162, 6}
    },
    { {68, 239, 204, 54, 11},
      {173, 151, 159, 160, 9},
      {214, 195, 63, 114, 2},
      {144, 95, 158, 155, 5}
    },
    { {31, 253, 132, 177, 6},
      {195, 159, 210, 101, 10},
      {104, 210, 27, 255, 8},
      {90, 100, 191, 156, 3}
    },
    { {197, 49, 181, 191, 8},
      {217, 163, 131, 214, 7},
      {31, 218, 216, 202, 3},
      {230, 188, 28, 89, 11}
    },
    { {22, 150, 178, 17, 4},
      {192, 200, 216, 53, 4},
      {40, 132, 214, 150, 8},
      {42, 193, 177, 48, 3}
    },
    { {38, 79, 248, 144, 13},
      {164, 134, 237, 49, 14},
      {176, 145, 255, 38, 4},
      {120, 203, 118, 18, 5}
    },
    { {246, 196, 223, 173, 7},
      {188, 253, 118, 183, 2},
      {235, 95, 178, 54, 15},
      {78, 214, 235, 243, 13}
    },
    { {156, 24, 84, 188, 5},
      {238, 41, 194, 18, 10},
      {163, 210, 161, 131, 9},
      {84, 132, 57, 71, 7}
    },
    { {19, 208, 219, 228, 10},
      {120, 220, 23, 89, 2},
      {82, 125, 176, 188, 8},
      {73, 174, 131, 177, 14}
    },
    { {249, 179, 22, 59, 4},
      {67, 107, 250, 214, 1},
      {45, 198, 140, 217, 15},
      {134, 181, 253, 108, 2}
    },
    { {81, 149, 114, 252, 7},
      {108, 122, 210, 248, 6},
      {227, 244, 234, 152, 10},
      {97, 244, 181, 227, 6}
    },
    { {242, 39, 249, 70, 5},
      {61, 138, 108, 187, 5},
      {166, 41, 254, 68, 15},
      {173, 211, 101, 27, 12}
    },
    { {102, 190, 189, 206, 13},
      {221, 161, 125, 185, 15},
      {183, 59, 215, 214, 6},
      {249, 219, 232, 91, 11}
    },
    { {214, 222, 191, 213, 4},
      {216, 205, 220, 191, 14},
      {42, 191, 215, 182, 11},
      {127, 211, 187, 49, 11}
    },
    { {42, 6, 160, 45, 8},
      {10, 160, 43, 37, 4},
      {27, 64, 86, 5, 4},
      {42, 77, 64, 85, 0}
    },
    { {20, 53, 95, 118, 11},
      {253, 91, 135, 56, 1},
      {214, 239, 170, 194, 8},
      {129, 206, 29, 171, 15}
    },
    { {10, 147, 60, 139, 0},
      {66, 35, 28, 21, 7},
      {13, 19, 204, 149, 0},
      {234, 131, 140, 68, 2}
    },
    { {41, 181, 114, 31, 7},
      {111, 114, 240, 116, 5},
      {239, 132, 234, 217, 4},
      {162, 224, 244, 239, 6}
    },
    { {10, 176, 119, 109, 10},
      {123, 113, 19, 29, 4},
      {91, 110, 224, 213, 0},
      {43, 140, 136, 237, 14}
    },
    { {73, 99, 125, 47, 0},
      {59, 39, 14, 212, 5},
      {15, 75, 236, 105, 2},
      {162, 183, 14, 77, 12}
    },
    { {129, 17, 157, 135, 11},
      {92, 147, 5, 86, 3},
      {222, 27, 152, 136, 1},
      {198, 170, 12, 147, 10}
    },
    { {140, 179, 51, 58, 10},
      {211, 114, 155, 18, 5},
      {85, 204, 204, 211, 1},
      {164, 141, 148, 236, 11}
    },
    { {178, 29, 155, 97, 0},
      {80, 202, 38, 63, 8},
      {8, 109, 155, 132, 13},
      {31, 198, 69, 48, 10}
    },
    { {188, 118, 206, 213, 10},
      {235, 221, 173, 46, 2},
      {90, 183, 54, 227, 13},
      {71, 75, 91, 189, 7}
    },
    { {160, 15, 75, 230, 8},
      {56, 66, 47, 42, 11},
      {22, 125, 47, 0, 5},
      {213, 79, 68, 33, 12}
    },
    { {68, 168, 183, 24, 8},
      {145, 225, 145, 144, 12},
      {17, 142, 209, 82, 2},
      {48, 152, 152, 120, 9}
    },
    { {192, 147, 189, 209, 12},
      {80, 131, 221, 158, 6},
      {56, 187, 220, 144, 3},
      {103, 155, 188, 16, 10}
    },
    { {67, 88, 203, 57, 12},
      {112, 228, 199, 197, 8},
      {57, 205, 49, 172, 2},
      {26, 62, 50, 112, 14}
    },
    { {138, 61, 137, 8, 12},
      {83, 162, 69, 35, 8},
      {49, 9, 27, 197, 1},
      {28, 74, 36, 92, 10}
    },
    { {252, 190, 21, 102, 11},
      {223, 25, 59, 186, 9},
      {214, 106, 135, 211, 15},
      {149, 221, 201, 143, 11}
    },
    { {203, 103, 181, 191, 5},
      {31, 167, 202, 247, 7},
      {175, 218, 222, 109, 3},
      {238, 245, 62, 95, 8}
    },
    { {233, 7, 248, 189, 4},
      {42, 162, 238, 246, 6},
      {43, 209, 254, 9, 7},
      {102, 247, 116, 85, 4}
    },
    { {75, 186, 42, 243, 10},
      {67, 80, 159, 205, 15},
      {92, 245, 69, 221, 2},
      {251, 63, 144, 172, 2}
    },
    { {113, 234, 245, 7, 10},
      {57, 157, 57, 212, 13},
      {94, 10, 245, 120, 14},
      {178, 185, 203, 153, 12}
    },
    { {147, 43, 182, 127, 6},
      {9, 251, 202, 95, 13},
      {111, 230, 221, 76, 9},
      {191, 165, 61, 249, 0}
    },
    { {83, 241, 61, 70, 1},
      {93, 15, 20, 217, 5},
      {134, 43, 200, 252, 10},
      {169, 178, 143, 11, 10}
    },
    { {99, 45, 7, 61, 3},
      {29, 115, 162, 229, 8},
      {203, 206, 11, 76, 6},
      {26, 116, 92, 235, 8}
    },
    { {187, 21, 254, 209, 0},
      {98, 203, 164, 127, 6},
      {8, 183, 250, 141, 13},
      {111, 226, 93, 52, 6}
    },
    { {249, 117, 16, 139, 12},
      {67, 46, 97, 246, 3},
      {61, 16, 138, 233, 15},
      {198, 248, 103, 76, 2}
    },
    { {82, 137, 180, 183, 15},
      {12, 155, 211, 149, 15},
      {254, 210, 217, 20, 10},
      {250, 156, 189, 147, 0}
    },
    { {27, 210, 12, 213, 13},
      {78, 13, 221, 77, 2},
      {186, 179, 4, 189, 8},
      {75, 43, 187, 7, 2}
    },
    { {177, 242, 231, 215, 11},
      {125, 221, 185, 78, 7},
      {222, 190, 116, 248, 13},
      {231, 41, 219, 187, 14}
    },
    { {59, 199, 234, 23, 8},
      {42, 206, 189, 101, 5},
      {30, 133, 126, 61, 12},
      {170, 107, 215, 53, 4}
    },
    { {129, 160, 44, 14, 10},
      {9, 49, 21, 66, 5},
      {87, 3, 64, 88, 1},
      {164, 42, 136, 201, 0}
    },
    { {175, 47, 221, 40, 8},
      {179, 163, 47, 115, 8},
      {17, 75, 191, 79, 5},
      {28, 239, 76, 92, 13}
    },
    { {102, 160, 63, 60, 5},
      {157, 97, 246, 145, 4},
      {163, 207, 192, 86, 6},
      {40, 150, 248, 107, 9}
    },
    { {97, 107, 54, 64, 2},
      {1, 87, 40, 216, 12},
      {64, 38, 205, 104, 6},
      {49, 177, 78, 168, 0}
    },
    { {179, 110, 223, 102, 4},
      {57, 205, 110, 123, 9},
      {38, 111, 183, 108, 13},
      {157, 231, 107, 57, 12}
    },
    { {221, 206, 234, 142, 13},
      {174, 236, 93, 226, 15},
      {183, 21, 119, 59, 11},
      {244, 123, 163, 119, 5}
    },
    { {191, 161, 228, 191, 4},
      {171, 171, 242, 71, 7},
      {47, 210, 120, 95, 13},
      {238, 36, 253, 93, 5}
    },
    { {181, 50, 205, 129, 2},
      {241, 153, 44, 70, 2},
      {72, 27, 52, 202, 13},
      {70, 35, 73, 152, 15}
    },
    { {93, 75, 26, 56, 8},
      {130, 110, 143, 208, 8},
      {17, 197, 141, 43, 10},
      {16, 191, 23, 100, 1}
    },
    { {178, 246, 24, 165, 12},
      {73, 12, 127, 55, 2},
      {58, 81, 134, 244, 13},
      {78, 207, 227, 9, 2}
    },
    { {56, 237, 90, 165, 11},
      {47, 94, 55, 52, 10},
      {218, 85, 171, 113, 12},
      {82, 206, 199, 175, 4}
    },
    { {73, 219, 225, 19, 5},
      {118, 134, 216, 196, 13},
      {172, 136, 125, 185, 2},
      {178, 49, 182, 22, 14}
    },
    { {73, 170, 182, 189, 9},
      {15, 225, 155, 212, 14},
      {155, 214, 213, 89, 2},
      {114, 189, 152, 127, 0}
    },
    { {84, 139, 234, 26, 2},
      {160, 250, 156, 128, 13},
      {69, 133, 125, 18, 10},
      {176, 19, 149, 240, 5}
    },
    { {157, 15, 70, 19, 1},
      {166, 75, 136, 102, 9},
      {140, 134, 47, 11, 9},
      {150, 97, 29, 38, 5}
    },
    { {102, 97, 19, 14, 1},
      {157, 102, 32, 145, 1},
      {135, 12, 136, 102, 6},
      {136, 144, 70, 107, 9}
    },
    { {248, 9, 47, 37, 2},
      {26, 91, 38, 134, 12},
      {74, 79, 73, 1, 15},
      {54, 22, 77, 165, 8}
    },
    { {11, 201, 149, 46, 14},
      {26, 183, 83, 81, 9},
      {119, 74, 153, 61, 0},
      {152, 172, 174, 213, 8}
    },
    { {184, 19, 104, 249, 3},
      {102, 58, 174, 14, 6},
      {201, 241, 108, 129, 13},
      {103, 7, 85, 198, 6}
    },
    { {247, 151, 168, 45, 7},
      {204, 186, 126, 231, 4},
      {235, 65, 94, 158, 15},
      {46, 119, 229, 211, 3}
    },
    { {64, 198, 85, 224, 12},
      {48, 5, 91, 184, 2},
      {48, 122, 166, 48, 2},
      {65, 221, 170, 0, 12}
    },
    { {119, 132, 246, 36, 5},
      {172, 201, 114, 241, 4},
      {162, 70, 242, 30, 14},
      {40, 244, 233, 51, 5}
    },
    { {67, 149, 31, 37, 8},
      {88, 67, 23, 245, 0},
      {26, 79, 138, 156, 2},
      {10, 254, 140, 33, 10}
    },
    { {45, 159, 144, 243, 0},
      {194, 130, 186, 124, 11},
      {12, 240, 159, 155, 4},
      {211, 229, 212, 20, 3}
    },
    { {146, 221, 176, 205, 11},
      {76, 190, 17, 63, 14},
      {219, 48, 219, 180, 9},
      {127, 200, 135, 211, 2}
    },
    { {246, 199, 216, 26, 12},
      {160, 174, 253, 179, 1},
      {53, 129, 190, 54, 15},
      {140, 219, 247, 80, 5}
    },
    { {24, 194, 89, 137, 6},
      {50, 60, 92, 20, 2},
      {105, 25, 164, 49, 8},
      {66, 131, 163, 196, 12}
    },
    { {191, 94, 230, 119, 1},
      {238, 205, 170, 111, 13},
      {142, 230, 119, 175, 13},
      {191, 101, 91, 55, 7}
    },
    { {154, 185, 210, 41, 10},
      {99, 250, 19, 23, 8},
      {89, 68, 185, 213, 9},
      {30, 140, 133, 252, 6}
    },
    { {85, 134, 127, 171, 1},
      {180, 105, 30, 244, 7},
      {141, 95, 230, 26, 10},
      {226, 247, 137, 98, 13}
    },
    { {184, 210, 232, 62, 14},
      {106, 188, 255, 2, 5},
      {119, 193, 116, 177, 13},
      {164, 15, 243, 213, 6}
    },
    { {8, 95, 31, 243, 10},
      {82, 87, 143, 60, 11},
      {92, 255, 143, 161, 0},
      {211, 207, 30, 164, 10}
    },
    { {18, 244, 23, 56, 3},
      {85, 125, 146, 49, 0},
      {193, 206, 130, 244, 8},
      {8, 196, 155, 234, 10}
    },
    { {166, 113, 141, 223, 14},
      {217, 183, 229, 15, 3},
      {127, 187, 24, 230, 5},
      {207, 10, 126, 217, 11}
    },
    { {192, 142, 206, 109, 0},
      {40, 225, 30, 174, 8},
      {11, 103, 55, 16, 3},
      {23, 87, 136, 113, 4}
    },
    { {109, 149, 12, 180, 7},
      {206, 19, 246, 224, 2},
      {226, 211, 10, 155, 6},
      {64, 118, 252, 135, 3}
    },
    { {150, 47, 248, 18, 6},
      {161, 154, 204, 51, 13},
      {100, 129, 255, 70, 9},
      {188, 195, 53, 152, 5}
    },
    { {100, 178, 210, 193, 10},
      {225, 208, 57, 156, 2},
      {88, 52, 180, 210, 6},
      {67, 153, 192, 184, 7}
    },
    { {171, 232, 165, 221, 12},
      {27, 165, 241, 79, 14},
      {59, 186, 81, 125, 5},
      {127, 40, 250, 93, 8}
    },
    { {1, 1, 118, 119, 7},
      {44, 83, 194, 92, 5},
      {238, 230, 232, 8, 0},
      {163, 164, 60, 163, 4}
    },
    { {122, 87, 242, 127, 10},
      {106, 254, 171, 189, 5},
      {95, 228, 254, 165, 14},
      {171, 221, 87, 245, 6}
    },
    { {49, 50, 151, 33, 7},
      {85, 217, 106, 84, 0},
      {232, 78, 148, 200, 12},
      {2, 165, 105, 186, 10}
    },
    { {84, 80, 181, 69, 13},
      {220, 141, 65, 156, 4},
      {186, 42, 208, 162, 10},
      {35, 152, 43, 19, 11}
    },
    { {11, 87, 84, 254, 5},
      {110, 39, 202, 121, 3},
      {167, 242, 174, 173, 0},
      {201, 229, 62, 71, 6}
    },
    { {182, 45, 55, 60, 6},
      {153, 123, 226, 51, 12},
      {99, 206, 203, 70, 13},
      {60, 196, 125, 233, 9}
    },
    { {148, 78, 188, 41, 1},
      {132, 173, 14, 54, 12},
      {137, 67, 215, 34, 9},
      {54, 199, 11, 82, 1}
    },
    { {212, 209, 225, 203, 8},
      {240, 174, 17, 142, 7},
      {29, 56, 120, 178, 11},
      {231, 24, 135, 80, 15}
    },
    { {41, 127, 114, 212, 15},
      {111, 86, 233, 120, 14},
      {242, 180, 239, 233, 4},
      {113, 233, 118, 175, 6}
    },
    { {185, 41, 143, 89, 11},
      {23, 251, 165, 78, 8},
      {217, 175, 25, 73, 13},
      {23, 42, 93, 254, 8}
    },
    { {156, 178, 65, 173, 13},
      {255, 40, 91, 6, 2},
      {187, 88, 36, 211, 9},
      {70, 13, 161, 79, 15}
    },
    { {9, 10, 241, 177, 13},
      {54, 128, 203, 84, 14},
      {184, 216, 245, 9, 0},
      {114, 173, 48, 22, 12}
    },
    { {16, 185, 135, 171, 7},
      {85, 251, 82, 4, 11},
      {237, 94, 25, 208, 8},
      {210, 4, 173, 250, 10}
    },
    { {126, 194, 225, 64, 6},
      {178, 156, 120, 137, 4},
      {96, 40, 116, 55, 14},
      {41, 17, 227, 148, 13}
    },
    { {67, 167, 86, 127, 1},
      {45, 99, 154, 253, 1},
      {143, 230, 174, 92, 2},
      {139, 245, 156, 107, 4}
    },
    { {26, 3, 90, 19, 6},
      {34, 90, 204, 21, 1},
      {108, 133, 172, 5, 8},
      {138, 131, 53, 164, 4}
    },
    { {29, 135, 214, 9, 13},
      {166, 235, 89, 116, 0},
      {185, 6, 190, 27, 8},
      {2, 233, 173, 118, 5}
    },
    { {166, 252, 66, 185, 15},
      {229, 116, 243, 39, 10},
      {249, 212, 35, 246, 5},
      {94, 76, 242, 234, 7}
    },
    { {107, 129, 183, 247, 6},
      {26, 211, 242, 221, 7},
      {110, 254, 216, 29, 6},
      {235, 180, 252, 181, 8}
    },
    { {69, 254, 109, 129, 14},
      {241, 21, 93, 228, 14},
      {120, 27, 103, 250, 2},
      {114, 123, 170, 136, 15}
    },
    { {53, 236, 43, 184, 7},
      {149, 124, 246, 96, 14},
      {225, 221, 67, 122, 12},
      {112, 102, 243, 234, 9}
    },
    { {49, 88, 4, 149, 14},
      {72, 29, 225, 68, 10},
      {122, 146, 1, 168, 12},
      {82, 40, 123, 129, 2}
    },
    { {152, 19, 34, 45, 15},
      {78, 122, 75, 6, 4},
      {251, 68, 76, 129, 9},
      {38, 13, 37, 231, 2}
    },
    { {154, 159, 90, 231, 3},
      {110, 90, 30, 63, 11},
      {206, 117, 175, 149, 9},
      {223, 199, 133, 167, 6}
    },
    { {212, 231, 111, 182, 1},
      {189, 79, 158, 162, 7},
      {134, 223, 110, 114, 11},
      {228, 87, 159, 43, 13}
    },
    { {229, 90, 150, 106, 14},
      {192, 245, 107, 218, 9},
      {117, 102, 149, 170, 7},
      {149, 189, 106, 240, 3}
    },
    { {79, 187, 212, 222, 13},
      {239, 163, 217, 217, 11},
      {183, 178, 189, 223, 2},
      {217, 185, 188, 95, 7}
    },
    { {121, 157, 133, 152, 15},
      {86, 187, 241, 224, 10},
      {241, 154, 27, 153, 14},
      {80, 120, 253, 214, 10}
    },
    { {9, 112, 158, 165, 9},
      {79, 197, 7, 84, 2},
      {154, 87, 144, 233, 0},
      {66, 174, 10, 63, 2}
    },
    { {3, 87, 205, 203, 8},
      {112, 167, 13, 109, 3},
      {29, 59, 62, 172, 0},
      {203, 107, 14, 80, 14}
    },
    { {212, 185, 186, 58, 5},
      {197, 234, 214, 146, 13},
      {165, 197, 217, 210, 11},
      {180, 150, 181, 122, 3}
    },
    { {24, 64, 161, 74, 8},
      {18, 172, 1, 8, 5},
      {21, 40, 80, 33, 8},
      {161, 8, 3, 84, 8}
    },
    { {118, 93, 35, 164, 0},
      {216, 78, 34, 161, 14},
      {2, 92, 75, 166, 14},
      {120, 84, 71, 33, 11}
    },
    { {215, 210, 250, 82, 11},
      {228, 220, 157, 219, 5},
      {212, 165, 244, 190, 11},
      {173, 187, 147, 178, 7}
    },
    { {112, 131, 115, 144, 14},
      {48, 90, 249, 144, 6},
      {112, 156, 236, 16, 14},
      {96, 153, 245, 160, 12}
    },
    { {54, 219, 219, 178, 9},
      {244, 206, 191, 17, 11},
      {148, 221, 189, 182, 12},
      {216, 143, 215, 50, 15}
    },
    { {238, 65, 132, 139, 9},
      {134, 167, 33, 135, 3},
      {157, 18, 24, 39, 7},
      {206, 24, 78, 86, 1}
    },
    { {82, 197, 48, 152, 6},
      {0, 62, 208, 177, 6},
      {97, 144, 202, 52, 10},
      {104, 208, 183, 192, 0}
    },
    { {252, 86, 183, 22, 2},
      {218, 221, 168, 178, 5},
      {70, 142, 214, 163, 15},
      {164, 209, 91, 181, 11}
    },
    { {116, 165, 189, 128, 1},
      {149, 139, 52, 176, 6},
      {128, 27, 218, 82, 14},
      {96, 210, 205, 26, 9}
    },
    { {149, 143, 205, 134, 4},
      {184, 139, 92, 98, 11},
      {38, 27, 63, 26, 9},
      {212, 99, 173, 17, 13}
    },
    { {217, 178, 146, 205, 0},
      {75, 232, 24, 222, 2},
      {11, 52, 148, 217, 11},
      {71, 177, 129, 125, 2}
    },
    { {167, 235, 71, 193, 10},
      {177, 87, 57, 79, 10},
      {88, 62, 45, 126, 5},
      {95, 41, 206, 168, 13}
    },
    { {79, 1, 94, 2, 9},
      {166, 67, 5, 209, 1},
      {148, 7, 168, 15, 2},
      {136, 186, 12, 38, 5}
    },
    { {106, 36, 9, 113, 10},
      {19, 16, 167, 173, 0},
      {88, 233, 2, 69, 6},
      {11, 94, 80, 140, 8}
    },
    { {206, 68, 49, 8, 9},
      {150, 36, 1, 179, 4},
      {145, 8, 194, 39, 3},
      {44, 216, 2, 70, 9}
    },
    { {49, 255, 30, 233, 10},
      {65, 127, 63, 124, 10},
      {89, 119, 143, 248, 12},
      {83, 239, 207, 232, 2}
    },
    { {20, 10, 7, 54, 11},
      {156, 89, 139, 0, 9},
      {214, 206, 5, 2, 8},
      {144, 13, 25, 163, 9}
    },
    { {70, 59, 86, 20, 0},
      {233, 67, 136, 145, 8},
      {2, 134, 173, 198, 2},
      {24, 145, 28, 41, 7}
    },
    { {59, 6, 12, 183, 14},
      {10, 25, 239, 101, 3},
      {126, 211, 6, 13, 12},
      {202, 111, 121, 133, 0}
    },
    { {183, 192, 126, 222, 10},
      {168, 125, 181, 91, 7},
      {87, 183, 224, 62, 13},
      {237, 170, 219, 225, 5}
    },
    { {131, 167, 59, 199, 12},
      {25, 66, 93, 127, 7},
      {62, 61, 206, 92, 1},
      {239, 235, 164, 41, 8}
    },
    { {169, 68, 61, 125, 8},
      {26, 37, 167, 126, 4},
      {27, 235, 194, 41, 5},
      {39, 238, 90, 69, 8}
    },
    { {89, 62, 133, 40, 0},
      {83, 169, 10, 224, 8},
      {1, 74, 23, 201, 10},
      {16, 117, 9, 92, 10}
    },
    { {91, 223, 9, 190, 3},
      {94, 62, 158, 225, 11},
      {199, 217, 15, 189, 10},
      {216, 119, 151, 199, 10}
    },
    { {90, 199, 81, 168, 1},
      {54, 46, 26, 177, 2},
      {129, 88, 174, 53, 10},
      {72, 213, 135, 70, 12}
    },
    { {172, 197, 14, 185, 6},
      {130, 119, 246, 38, 2},
      {105, 215, 10, 51, 5},
      {70, 70, 254, 228, 1}
    },
    { {140, 158, 159, 124, 11},
      {222, 241, 159, 58, 8},
      {211, 239, 151, 147, 1},
      {21, 207, 152, 247, 11}
    },
    { {94, 223, 124, 158, 10},
      {234, 63, 157, 177, 15},
      {87, 147, 239, 183, 10},
      {248, 219, 159, 197, 7}
    },
    { {194, 247, 145, 150, 7},
      {93, 150, 216, 179, 3},
      {230, 152, 158, 244, 3},
      {204, 209, 182, 155, 10}
    },
    { {104, 128, 250, 139, 3},
      {38, 240, 52, 148, 7},
      {205, 21, 240, 17, 6},
      {226, 146, 192, 246, 4}
    },
    { {164, 118, 61, 185, 15},
      {213, 53, 239, 54, 6},
      {249, 219, 198, 226, 5},
      {102, 207, 122, 202, 11}
    },
    { {232, 136, 149, 37, 11},
      {30, 145, 51, 150, 8},
      {218, 74, 145, 17, 7},
      {22, 156, 200, 151, 8}
    },
    { {94, 137, 11, 42, 0},
      {146, 106, 22, 129, 9},
      {5, 77, 9, 23, 10},
      {152, 22, 133, 100, 9}
    },
    { {168, 252, 52, 240, 6},
      {67, 21, 242, 58, 14},
      {96, 242, 195, 241, 5},
      {117, 196, 250, 140, 2}
    },
    { {102, 80, 220, 156, 14},
      {232, 181, 229, 145, 2},
      {115, 147, 176, 166, 6},
      {72, 154, 122, 209, 7}
    },
    { {217, 235, 112, 198, 6},
      {43, 30, 88, 218, 15},
      {102, 48, 237, 121, 11},
      {245, 177, 167, 141, 4}
    },
    { {6, 57, 26, 27, 7},
      {197, 114, 196, 21, 9},
      {237, 133, 137, 198, 0},
      {154, 130, 52, 234, 3}
    },
    { {100, 213, 236, 219, 14},
      {224, 183, 245, 172, 7},
      {125, 179, 122, 178, 6},
      {227, 90, 254, 208, 7}
    },
    { {80, 23, 140, 98, 3},
      {68, 155, 14, 168, 1},
      {196, 99, 30, 128, 10},
      {129, 87, 13, 146, 2}
    },
    { {128, 175, 91, 93, 8},
      {57, 98, 157, 62, 8},
      {27, 173, 175, 80, 1},
      {23, 203, 148, 105, 12}
    },
    { {178, 110, 247, 189, 9},
      {61, 237, 171, 55, 14},
      {155, 222, 247, 100, 13},
      {126, 205, 91, 123, 12}
    },
    { {210, 217, 41, 11, 11},
      {84, 62, 21, 135, 13},
      {221, 9, 73, 180, 11},
      {190, 26, 135, 194, 10}
    },
    { {142, 230, 136, 246, 1},
      {143, 132, 158, 43, 3},
      {134, 241, 22, 119, 1},
      {205, 71, 146, 31, 1}
    },
    { {144, 133, 126, 179, 5},
      {36, 75, 214, 54, 7},
      {172, 215, 234, 16, 9},
      {230, 198, 189, 34, 4}
    },
    { {214, 244, 185, 51, 14},
      {209, 156, 215, 183, 5},
      {124, 201, 210, 246, 11},
      {174, 222, 179, 152, 11}
    },
    { {68, 17, 154, 175, 12},
      {200, 226, 71, 148, 3},
      {63, 85, 152, 130, 2},
      {194, 158, 36, 113, 3}
    },
    { {48, 177, 90, 93, 2},
      {105, 122, 180, 28, 0},
      {75, 165, 168, 208, 12},
      {3, 130, 213, 233, 6}
    },
    { {66, 57, 254, 118, 6},
      {105, 211, 198, 153, 13},
      {102, 231, 249, 196, 2},
      {185, 150, 60, 185, 6}
    },
    { {108, 223, 228, 140, 12},
      {234, 167, 121, 160, 14},
      {51, 18, 127, 179, 6},
      {112, 89, 238, 85, 7}
    },
    { {110, 159, 250, 19, 0},
      {226, 194, 188, 181, 13},
      {12, 133, 255, 151, 6},
      {186, 211, 212, 52, 7}
    },
    { {251, 180, 205, 219, 9},
      {119, 169, 181, 239, 3},
      {157, 187, 50, 221, 15},
      {207, 122, 217, 94, 14}
    },
    { {106, 94, 54, 11, 12},
      {66, 101, 105, 181, 13},
      {61, 6, 199, 165, 6},
      {186, 217, 106, 100, 2}
    },
    { {187, 128, 110, 3, 5},
      {38, 73, 116, 71, 5},
      {172, 7, 96, 29, 13},
      {174, 34, 233, 38, 4}
    },
    { {151, 235, 197, 191, 10},
      {185, 191, 155, 71, 11},
      {95, 218, 61, 126, 9},
      {222, 45, 159, 217, 13}
    },
    { {15, 117, 226, 80, 2},
      {227, 214, 128, 105, 4},
      {64, 164, 122, 239, 0},
      {41, 96, 22, 188, 7}
    },
    { {11, 49, 73, 156, 8},
      {123, 34, 133, 65, 2},
      {19, 153, 40, 205, 0},
      {72, 42, 20, 77, 14}
    },
    { {215, 174, 213, 193, 8},
      {177, 137, 25, 255, 10},
      {24, 58, 183, 94, 11},
      {95, 249, 137, 24, 13}
    },
    { {210, 248, 182, 187, 9},
      {69, 237, 147, 151, 15},
      {157, 214, 209, 244, 11},
      {254, 156, 155, 122, 2}
    },
    { {34, 55, 245, 42, 9},
      {117, 163, 43, 49, 5},
      {149, 74, 254, 196, 4},
      {168, 205, 76, 90, 14}
    },
    { {68, 127, 202, 153, 4},
      {225, 230, 204, 164, 10},
      {41, 149, 63, 226, 2},
      {82, 83, 54, 120, 7}
    },
    { {141, 146, 126, 41, 15},
      {230, 113, 95, 86, 4},
      {249, 71, 228, 155, 1},
      {38, 175, 168, 230, 7}
    },
    { {13, 54, 100, 46, 5},
      {239, 33, 74, 96, 5},
      {167, 66, 102, 203, 0},
      {160, 101, 40, 79, 7}
    },
    { {213, 191, 237, 221, 12},
      {249, 171, 221, 238, 14},
      {59, 187, 127, 218, 11},
      {119, 123, 189, 89, 15}
    },
    { {139, 198, 212, 229, 1},
      {46, 133, 26, 127, 2},
      {138, 114, 182, 61, 1},
      {79, 229, 138, 23, 4}
    },
    { {221, 244, 124, 110, 1},
      {239, 45, 22, 250, 5},
      {135, 99, 226, 251, 11},
      {165, 246, 139, 79, 7}
    },
    { {69, 159, 45, 147, 1},
      {212, 3, 156, 228, 15},
      {140, 155, 79, 154, 2},
      {242, 115, 156, 2, 11}
    },
    { {134, 21, 175, 172, 5},
      {220, 227, 70, 35, 6},
      {163, 95, 90, 134, 1},
      {108, 70, 44, 115, 11}
    },
    { {52, 174, 42, 121, 9},
      {133, 104, 191, 44, 12},
      {153, 229, 71, 82, 12},
      {51, 79, 209, 106, 1}
    },
    { {246, 135, 29, 191, 4},
      {152, 43, 254, 183, 3},
      {47, 219, 142, 22, 15},
      {206, 215, 253, 65, 9}
    },
    { {22, 183, 193, 11, 4},
      {241, 170, 88, 37, 1},
      {45, 8, 62, 214, 8},
      {138, 65, 165, 88, 15}
    },
    { {122, 54, 203, 160, 5},
      {119, 200, 110, 161, 2},
      {160, 93, 54, 197, 14},
      {72, 87, 97, 62, 14}
    },
    { {127, 66, 16, 98, 10},
      {130, 28, 43, 217, 1},
      {84, 96, 132, 47, 14},
      {137, 189, 67, 132, 1}
    },
    { {103, 112, 117, 206, 8},
      {249, 37, 33, 217, 7},
      {23, 58, 224, 238, 6},
      {233, 184, 74, 73, 15}
    },
    { {167, 70, 60, 209, 12},
      {128, 5, 237, 127, 6},
      {56, 179, 198, 46, 5},
      {111, 235, 122, 0, 1}
    },
    { {153, 117, 131, 110, 2},
      {91, 254, 2, 106, 1},
      {71, 108, 26, 233, 9},
      {133, 100, 7, 253, 10}
    },
    { {81, 145, 211, 21, 9},
      {124, 202, 145, 212, 0},
      {154, 140, 184, 152, 10},
      {2, 184, 149, 51, 14}
    },
    { {53, 76, 2, 185, 8},
      {128, 108, 163, 100, 10},
      {25, 212, 3, 42, 12},
      {82, 108, 83, 96, 1}
    },
    { {208, 27, 128, 57, 5},
      {68, 170, 202, 134, 8},
      {169, 192, 29, 128, 11},
      {22, 21, 53, 82, 2}
    },
    { {123, 39, 171, 229, 3},
      {31, 218, 46, 237, 6},
      {202, 125, 94, 77, 14},
      {107, 119, 69, 191, 8}
    },
    { {168, 140, 171, 169, 4},
      {18, 224, 118, 38, 14},
      {41, 93, 83, 17, 5},
      {118, 70, 224, 116, 8}
    },
    { {138, 112, 46, 100, 13},
      {79, 69, 71, 11, 4},
      {178, 103, 64, 229, 1},
      {45, 14, 42, 47, 2}
    },
    { {17, 144, 216, 79, 13},
      {108, 168, 85, 92, 1},
      {191, 33, 176, 152, 8},
      {131, 170, 161, 83, 6}
    },
    { {168, 56, 156, 230, 8},
      {75, 129, 39, 26, 11},
      {22, 115, 145, 193, 5},
      {213, 142, 72, 29, 2}
    },
    { {99, 172, 245, 8, 11},
      {53, 177, 49, 241, 12},
      {209, 10, 243, 92, 6},
      {56, 248, 200, 218, 12}
    },
    { {153, 224, 174, 41, 11},
      {7, 253, 23, 70, 4},
      {217, 71, 80, 121, 9},
      {38, 46, 139, 254, 0}
    },
    { {232, 139, 97, 157, 3},
      {62, 50, 184, 134, 14},
      {203, 152, 109, 17, 7},
      {118, 17, 212, 199, 12}
    },
    { {17, 175, 187, 184, 0},
      {17, 234, 158, 112, 14},
      {1, 221, 223, 88, 8},
      {112, 231, 149, 120, 8}
    },
    { {98, 250, 27, 120, 14},
      {81, 116, 255, 153, 8},
      {113, 237, 133, 244, 6},
      {25, 159, 242, 232, 10}
    },
    { {241, 171, 236, 42, 5},
      {37, 171, 126, 194, 13},
      {165, 67, 125, 88, 15},
      {180, 55, 237, 90, 4}
    },
    { {164, 120, 185, 64, 2},
      {209, 148, 36, 26, 12},
      {64, 41, 209, 226, 5},
      {53, 130, 66, 152, 11}
    },
    { {78, 147, 112, 26, 3},
      {230, 50, 152, 145, 5},
      {197, 128, 236, 151, 2},
      {168, 145, 148, 198, 7}
    },
    { {93, 31, 190, 33, 6},
      {194, 219, 78, 244, 12},
      {104, 71, 223, 139, 10},
      {50, 247, 45, 180, 3}
    },
    { {222, 153, 178, 148, 7},
      {206, 218, 208, 147, 14},
      {226, 148, 217, 151, 11},
      {124, 144, 181, 183, 3}
    },
    { {119, 8, 209, 116, 7},
      {188, 152, 226, 217, 8},
      {226, 232, 177, 14, 14},
      {25, 180, 113, 147, 13}
    },
    { {187, 192, 140, 154, 8},
      {2, 173, 181, 67, 3},
      {21, 147, 16, 61, 13},
      {204, 42, 219, 84, 0}
    },
    { {234, 28, 8, 68, 4},
      {74, 0, 100, 171, 8},
      {34, 33, 3, 133, 7},
      {29, 82, 96, 5, 2}
    },
    { {184, 154, 172, 7, 8},
      {74, 137, 61, 6, 13},
      {30, 3, 85, 145, 13},
      {182, 11, 201, 21, 2}
    },
    { {151, 79, 121, 58, 13},
      {180, 46, 207, 115, 13},
      {181, 201, 239, 46, 9},
      {188, 239, 55, 66, 13}
    },
    { {124, 55, 168, 31, 9},
      {207, 170, 173, 164, 5},
      {159, 129, 94, 195, 14},
      {162, 91, 85, 95, 3}
    },
    { {132, 5, 69, 184, 11},
      {180, 51, 131, 34, 2},
      {209, 218, 42, 2, 1},
      {68, 76, 28, 194, 13}
    },
    { {137, 136, 232, 242, 10},
      {34, 144, 151, 74, 15},
      {84, 241, 113, 25, 1},
      {245, 46, 144, 148, 4}
    },
    { {122, 5, 47, 138, 7},
      {22, 123, 100, 161, 7},
      {229, 31, 74, 5, 14},
      {232, 82, 109, 230, 8}
    },
    { {192, 107, 231, 168, 3},
      {53, 247, 10, 130, 14},
      {193, 94, 125, 96, 3},
      {116, 21, 14, 250, 12}
    },
    { {225, 214, 143, 229, 13},
      {92, 197, 127, 238, 2},
      {186, 127, 22, 184, 7},
      {71, 127, 234, 51, 10}
    },
    { {135, 193, 94, 208, 1},
      {164, 71, 148, 91, 2},
      {128, 183, 168, 62, 1},
      {77, 162, 158, 34, 5}
    },
    { {177, 141, 71, 4, 14},
      {56, 91, 113, 98, 8},
      {114, 14, 43, 24, 13},
      {20, 104, 237, 161, 12}
    },
    { {185, 96, 48, 55, 10},
      {11, 28, 163, 86, 5},
      {94, 192, 192, 105, 13},
      {166, 172, 83, 141, 0}
    },
    { {57, 22, 249, 242, 5},
      {118, 136, 238, 120, 7},
      {164, 249, 246, 137, 12},
      {225, 231, 113, 22, 14}
    },
    { {91, 245, 56, 225, 10},
      {67, 30, 23, 253, 6},
      {88, 113, 202, 253, 10},
      {107, 254, 135, 140, 2}
    },
    { {69, 2, 140, 137, 3},
      {132, 177, 12, 196, 2},
      {201, 19, 20, 10, 2},
      {66, 51, 8, 210, 1}
    },
    { {139, 247, 79, 21, 14},
      {123, 87, 221, 103, 0},
      {122, 143, 46, 253, 1},
      {14, 107, 190, 173, 14}
    },
    { {52, 169, 38, 14, 14},
      {137, 123, 113, 0, 13},
      {119, 6, 73, 82, 12},
      {176, 8, 237, 233, 1}
    },
    { {254, 161, 180, 87, 2},
      {139, 155, 176, 159, 5},
      {78, 162, 216, 87, 15},
      {175, 144, 221, 157, 1}
    },
    { {170, 172, 74, 33, 2},
      {35, 80, 54, 39, 8},
      {72, 69, 35, 85, 5},
      {30, 70, 192, 172, 4}
    },
    { {199, 149, 142, 146, 10},
      {192, 211, 149, 227, 3},
      {84, 151, 26, 158, 3},
      {204, 122, 156, 176, 3}
    },
    { {77, 114, 150, 13, 7},
      {207, 245, 72, 212, 0},
      {235, 6, 148, 235, 2},
      {2, 177, 42, 255, 3}
    },
    { {198, 113, 60, 138, 13},
      {197, 39, 69, 147, 7},
      {181, 19, 200, 230, 3},
      {236, 154, 46, 74, 3}
    },
    { {26, 219, 8, 147, 10},
      {66, 30, 157, 5, 11},
      {92, 145, 13, 181, 8},
      {218, 11, 151, 132, 2}
    },
    { {235, 178, 94, 21, 3},
      {111, 81, 188, 215, 0},
      {202, 135, 164, 221, 7},
      {14, 179, 216, 175, 6}
    },
    { {26, 140, 101, 49, 13},
      {54, 9, 211, 37, 12},
      {184, 202, 99, 21, 8},
      {58, 76, 185, 6, 12}
    },
    { {184, 194, 144, 210, 13},
      {6, 140, 249, 26, 3},
      {180, 176, 148, 49, 13},
      {197, 137, 243, 22, 0}
    },
    { {237, 10, 185, 59, 12},
      {146, 160, 239, 214, 13},
      {61, 201, 213, 11, 7},
      {182, 191, 112, 84, 9}
    },
    { {53, 64, 254, 171, 14},
      {160, 253, 103, 84, 7},
      {125, 87, 240, 42, 12},
      {226, 174, 107, 240, 5}
    },
    { {77, 141, 169, 163, 10},
      {146, 146, 23, 228, 15},
      {92, 89, 91, 27, 2},
      {242, 126, 132, 148, 9}
    },
    { {25, 111, 83, 7, 9},
      {63, 78, 9, 116, 9},
      {158, 12, 175, 105, 8},
      {146, 233, 7, 47, 12}
    },
    { {170, 161, 45, 108, 8},
      {27, 35, 55, 11, 4},
      {19, 107, 72, 85, 5},
      {45, 14, 204, 77, 8}
    },
    { {212, 51, 219, 225, 6},
      {241, 218, 78, 158, 2},
      {104, 125, 188, 194, 11},
      {71, 151, 37, 184, 15}
    },
    { {41, 68, 117, 210, 10},
      {50, 21, 161, 120, 7},
      {84, 186, 226, 41, 4},
      {225, 232, 90, 132, 12}
    },
    { {222, 189, 217, 145, 6},
      {243, 154, 212, 183, 10},
      {104, 153, 187, 215, 11},
      {94, 210, 181, 156, 15}
    },
    { {123, 98, 241, 8, 1},
      {55, 172, 40, 209, 4},
      {129, 8, 244, 109, 14},
      {40, 177, 67, 94, 12}
    },
    { {16, 243, 23, 219, 8},
      {81, 111, 153, 28, 3},
      {29, 190, 140, 240, 8},
      {195, 137, 159, 104, 10}
    },
    { {184, 117, 135, 64, 5},
      {87, 207, 96, 42, 0},
      {160, 46, 26, 225, 13},
      {5, 64, 111, 62, 10}
    },
    { {154, 182, 79, 175, 0},
      {123, 105, 30, 39, 3},
      {15, 95, 38, 213, 9},
      {206, 71, 137, 109, 14}
    },
    { {212, 4, 243, 160, 1},
      {180, 200, 2, 178, 6},
      {128, 92, 242, 2, 11},
      {100, 212, 1, 50, 13}
    },
    { {225, 106, 123, 127, 5},
      {61, 100, 238, 222, 13},
      {175, 237, 229, 104, 7},
      {183, 183, 114, 107, 12}
    },
    { {12, 218, 188, 26, 8},
      {194, 165, 157, 16, 13},
      {21, 131, 213, 179, 0},
      {176, 139, 154, 84, 3}
    },
    { {4, 96, 183, 149, 11},
      {157, 213, 129, 20, 6},
      {218, 158, 208, 98, 0},
      {98, 136, 26, 187, 9}
    },
    { {28, 206, 196, 143, 3},
      {174, 189, 24, 36, 11},
      {207, 18, 55, 51, 8},
      {210, 65, 139, 215, 5}
    },
    { {35, 222, 229, 165, 12},
      {120, 133, 123, 101, 14},
      {58, 90, 119, 188, 4},
      {122, 109, 234, 17, 14}
    },
    { {99, 5, 255, 232, 5},
      {52, 227, 102, 249, 6},
      {161, 127, 250, 12, 6},
      {105, 246, 108, 114, 12}
    },
    { {225, 195, 39, 185, 4},
      {16, 103, 250, 198, 6},
      {41, 222, 76, 56, 7},
      {102, 53, 254, 96, 8}
    },
    { {140, 43, 164, 30, 0},
      {139, 163, 136, 2, 13},
      {7, 130, 93, 67, 1},
      {180, 1, 28, 93, 1}
    },
    { {28, 217, 222, 180, 0},
      {234, 207, 150, 16, 10},
      {2, 215, 185, 179, 8},
      {80, 134, 159, 53, 7}
    },
    { {85, 100, 240, 183, 1},
      {173, 140, 130, 244, 7},
      {142, 208, 242, 106, 10},
      {226, 244, 19, 27, 5}
    },
    { {176, 199, 42, 118, 10},
      {8, 94, 191, 42, 5},
      {86, 229, 78, 48, 13},
      {165, 79, 215, 161, 0}
    },
    { {231, 13, 80, 70, 9},
      {172, 2, 33, 251, 9},
      {150, 32, 171, 14, 7},
      {157, 248, 68, 3, 5}
    },
    { {50, 149, 124, 126, 3},
      {108, 59, 182, 57, 5},
      {199, 227, 234, 148, 12},
      {169, 198, 221, 195, 6}
    },
    { {71, 130, 81, 20, 11},
      {188, 16, 153, 209, 0},
      {210, 136, 164, 30, 2},
      {8, 185, 144, 131, 13}
    },
    { {120, 242, 17, 22, 15},
      {95, 28, 249, 144, 1},
      {246, 136, 132, 241, 14},
      {128, 153, 243, 143, 10}
    },
    { {144, 251, 64, 119, 10},
      {105, 30, 155, 14, 9},
      {94, 224, 45, 240, 9},
      {151, 13, 151, 137, 6}
    },
    { {162, 111, 219, 64, 3},
      {53, 214, 44, 59, 8},
      {192, 45, 191, 100, 5},
      {29, 195, 70, 186, 12}
    },
    { {36, 36, 50, 53, 7},
      {141, 80, 226, 52, 4},
      {234, 196, 194, 66, 4},
      {34, 196, 112, 171, 1}
    },
    { {135, 138, 180, 213, 3},
      {140, 145, 152, 95, 14},
      {202, 178, 213, 30, 1},
      {127, 161, 152, 147, 1}
    },
    { {238, 169, 194, 236, 0},
      {171, 226, 50, 139, 10},
      {3, 116, 57, 87, 7},
      {93, 20, 196, 125, 5}
    },
    { {194, 91, 81, 134, 10},
      {120, 22, 9, 147, 11},
      {86, 24, 173, 164, 3},
      {220, 153, 6, 129, 14}
    },
    { {154, 26, 220, 22, 4},
      {106, 137, 204, 19, 9},
      {38, 131, 181, 133, 9},
      {156, 131, 57, 21, 6}
    },
    { {115, 131, 224, 101, 1},
      {44, 138, 58, 205, 4},
      {138, 96, 124, 28, 14},
      {43, 53, 197, 19, 4}
    },
    { {25, 137, 191, 169, 14},
      {18, 251, 87, 84, 14},
      {121, 95, 217, 25, 8},
      {114, 174, 173, 244, 8}
    },
    { {29, 136, 219, 124, 13},
      {190, 232, 215, 88, 8},
      {179, 237, 177, 27, 8},
      {17, 174, 177, 119, 13}
    },
    { {58, 57, 201, 98, 1},
      {119, 138, 38, 9, 9},
      {132, 105, 57, 197, 12},
      {153, 6, 69, 30, 14}
    },
    { {196, 95, 111, 141, 1},
      {252, 103, 12, 166, 14},
      {139, 31, 111, 162, 3},
      {118, 83, 14, 99, 15}
    },
    { {127, 212, 244, 88, 13},
      {230, 173, 241, 249, 4},
      {177, 162, 242, 191, 14},
      {41, 248, 251, 86, 7}
    },
    { {235, 94, 5, 198, 13},
      {94, 5, 105, 235, 11},
      {182, 58, 7, 173, 7},
      {221, 121, 106, 7, 10}
    },
    { {0, 222, 87, 31, 10},
      {120, 117, 153, 52, 9},
      {95, 142, 167, 176, 0},
      {146, 201, 154, 225, 14}
    },
    { {48, 182, 57, 119, 2},
      {89, 24, 190, 60, 5},
      {78, 233, 198, 208, 12},
      {163, 199, 209, 137, 10}
    },
    { {217, 17, 79, 9, 9},
      {118, 107, 5, 198, 0},
      {153, 15, 40, 137, 11},
      {6, 58, 13, 102, 14}
    },
    { {11, 138, 78, 254, 13},
      {46, 97, 223, 73, 11},
      {183, 247, 37, 29, 0},
      {217, 47, 184, 103, 4}
    },
    { {150, 200, 148, 195, 5},
      {132, 141, 80, 31, 11},
      {172, 50, 145, 54, 9},
      {223, 128, 171, 18, 1}
    },
    { {222, 86, 207, 27, 9},
      {246, 237, 141, 167, 1},
      {157, 143, 54, 167, 11},
      {142, 91, 27, 118, 15}
    },
    { {233, 20, 44, 234, 11},
      {70, 49, 39, 234, 7},
      {213, 115, 66, 137, 7},
      {229, 126, 72, 198, 2}
    },
    { {166, 193, 149, 63, 7},
      {156, 183, 242, 23, 1},
      {239, 202, 152, 54, 5},
      {142, 132, 254, 211, 9}
    },
    { {112, 249, 99, 53, 11},
      {125, 94, 179, 132, 12},
      {218, 204, 105, 240, 14},
      {50, 28, 215, 171, 14}
    },
    { {60, 210, 112, 229, 12},
      {234, 12, 123, 28, 6},
      {58, 112, 228, 179, 12},
      {99, 141, 227, 5, 7}
    },
    { {67, 44, 58, 151, 8},
      {9, 64, 133, 245, 15},
      {30, 149, 195, 76, 2},
      {250, 250, 16, 41, 0}
    },
    { {96, 54, 213, 83, 2},
      {113, 145, 168, 188, 1},
      {76, 170, 182, 192, 6},
      {131, 209, 88, 152, 14}
    },
    { {135, 118, 248, 196, 14},
      {233, 148, 77, 123, 6},
      {114, 49, 246, 238, 1},
      {109, 235, 34, 153, 7}
    },
    { {32, 85, 144, 158, 12},
      {72, 166, 225, 48, 3},
      {55, 144, 154, 160, 4},
      {192, 200, 118, 81, 2}
    },
    { {51, 149, 147, 18, 0},
      {80, 202, 176, 113, 1},
      {4, 140, 154, 156, 12},
      {136, 224, 213, 48, 10}
    },
    { {61, 129, 11, 57, 10},
      {146, 122, 183, 68, 0},
      {89, 205, 8, 27, 12},
      {2, 46, 213, 228, 9}
    },
    { {39, 67, 30, 187, 2},
      {128, 119, 174, 85, 3},
      {77, 215, 140, 46, 4},
      {202, 167, 94, 224, 1}
    },
    { {240, 133, 185, 233, 2},
      {16, 186, 54, 190, 6},
      {73, 121, 218, 16, 15},
      {103, 214, 197, 208, 8}
    },
    { {163, 11, 146, 226, 0},
      {0, 194, 42, 91, 11},
      {4, 116, 157, 12, 5},
      {221, 165, 68, 48, 0}
    },
    { {220, 20, 37, 232, 5},
      {214, 41, 66, 170, 6},
      {161, 122, 66, 131, 11},
      {101, 84, 41, 70, 11}
    },
    { {180, 171, 135, 231, 11},
      {157, 219, 59, 14, 11},
      {222, 126, 29, 82, 13},
      {215, 13, 205, 187, 9}
    },
    { {143, 183, 62, 39, 0},
      {203, 67, 30, 119, 5},
      {14, 71, 206, 223, 1},
      {174, 231, 140, 45, 3}
    },
    { {255, 211, 18, 95, 2},
      {202, 126, 184, 223, 1},
      {79, 164, 140, 191, 15},
      {143, 177, 215, 229, 3}
    },
    { {5, 76, 149, 162, 10},
      {144, 149, 3, 112, 11},
      {84, 90, 147, 42, 0},
      {208, 236, 10, 144, 9}
    },
    { {29, 121, 189, 67, 6},
      {211, 159, 68, 92, 13},
      {108, 43, 217, 235, 8},
      {179, 162, 47, 156, 11}
    },
    { {49, 220, 189, 155, 11},
      {84, 189, 181, 116, 15},
      {221, 155, 211, 184, 12},
      {242, 234, 219, 210, 10}
    },
    { {93, 131, 76, 204, 2},
      {170, 59, 28, 200, 2},
      {67, 51, 44, 27, 10},
      {65, 51, 141, 197, 5}
    },
    { {184, 110, 95, 128, 10},
      {51, 93, 45, 50, 10},
      {80, 31, 167, 97, 13},
      {84, 203, 75, 172, 12}
    },
    { {221, 181, 2, 55, 14},
      {203, 90, 211, 230, 1},
      {126, 196, 10, 219, 11},
      {134, 124, 181, 173, 3}
    },
    { {204, 212, 249, 32, 2},
      {242, 148, 22, 178, 4},
      {64, 73, 242, 179, 3},
      {36, 214, 130, 148, 15}
    },
    { {222, 175, 20, 123, 3},
      {135, 59, 154, 191, 9},
      {205, 226, 143, 87, 11},
      {159, 213, 157, 206, 1}
    },
    { {157, 32, 19, 22, 15},
      {159, 88, 193, 82, 1},
      {246, 140, 128, 75, 9},
      {132, 168, 49, 175, 9}
    },
    { {124, 189, 240, 175, 1},
      {239, 170, 50, 180, 15},
      {143, 80, 251, 211, 14},
      {242, 212, 197, 95, 7}
    },
    { {34, 195, 216, 114, 10},
      {32, 150, 191, 25, 1},
      {84, 225, 188, 52, 4},
      {137, 143, 214, 144, 4}
    },
    { {69, 199, 100, 214, 12},
      {168, 7, 217, 232, 7},
      {54, 178, 110, 58, 2},
      {225, 121, 190, 1, 5}
    },
    { {141, 147, 90, 116, 0},
      {234, 66, 158, 90, 0},
      {2, 229, 172, 155, 1},
      {5, 167, 148, 37, 7}
    },
    { {177, 161, 236, 16, 8},
      {33, 139, 181, 66, 4},
      {16, 131, 120, 88, 13},
      {36, 42, 221, 24, 4}
    },
    { {208, 121, 53, 51, 6},
      {81, 31, 194, 150, 13},
      {108, 202, 201, 224, 11},
      {182, 148, 63, 136, 10}
    },
    { {74, 81, 93, 164, 6},
      {122, 23, 70, 145, 2},
      {98, 91, 168, 165, 2},
      {72, 150, 46, 133, 14}
    },
    { {58, 45, 19, 35, 5},
      {23, 74, 98, 53, 9},
      {172, 76, 139, 69, 12},
      {154, 196, 101, 46, 8}
    },
    { {236, 65, 230, 22, 1},
      {174, 199, 160, 130, 5},
      {134, 134, 120, 35, 7},
      {164, 16, 94, 55, 5}
    },
    { {127, 172, 78, 24, 2},
      {163, 121, 180, 225, 8},
      {65, 135, 35, 95, 14},
      {24, 114, 217, 236, 5}
    },
    { {115, 198, 100, 107, 13},
      {36, 45, 123, 237, 5},
      {189, 98, 102, 60, 14},
      {171, 125, 235, 66, 4}
    },
    { {77, 251, 76, 21, 7},
      {239, 23, 220, 196, 8},
      {234, 131, 45, 251, 2},
      {18, 51, 190, 143, 7}
    },
    { {229, 15, 78, 90, 11},
      {164, 115, 173, 234, 9},
      {213, 167, 47, 10, 7},
      {149, 123, 92, 226, 5}
    },
    { {122, 161, 57, 166, 0},
      {27, 10, 54, 145, 7},
      {6, 89, 200, 85, 14},
      {232, 150, 197, 13, 8}
    },
    { {18, 122, 254, 87, 13},
      {109, 205, 205, 29, 13},
      {190, 167, 245, 228, 8},
      {187, 139, 59, 59, 6}
    },
    { {168, 230, 106, 159, 15},
      {47, 116, 253, 38, 7},
      {255, 149, 102, 113, 5},
      {230, 75, 242, 239, 4}
    },
    { {89, 155, 96, 189, 8},
      {106, 42, 155, 196, 14},
      {27, 208, 109, 153, 10},
      {114, 61, 149, 69, 6}
    },
    { {49, 234, 54, 102, 9},
      {13, 77, 59, 88, 13},
      {150, 102, 197, 120, 12},
      {177, 173, 203, 43, 0}
    },
    { {55, 203, 228, 166, 3},
      {172, 159, 58, 65, 15},
      {198, 82, 125, 62, 12},
      {248, 37, 207, 147, 5}
    },
    { {134, 60, 248, 42, 13},
      {229, 160, 71, 51, 13},
      {181, 65, 243, 198, 1},
      {188, 206, 32, 90, 7}
    },
    { {14, 181, 149, 30, 6},
      {219, 179, 208, 49, 1},
      {103, 138, 154, 215, 0},
      {136, 192, 188, 221, 11}
    },
    { {197, 251, 85, 118, 12},
      {249, 7, 219, 218, 9},
      {54, 234, 173, 250, 3},
      {149, 189, 190, 9, 15}
    },
    { {123, 128, 210, 239, 5},
      {46, 232, 114, 221, 3},
      {175, 116, 176, 29, 14},
      {203, 180, 225, 119, 4}
    },
    { {195, 179, 195, 50, 10},
      {113, 210, 155, 195, 1},
      {84, 204, 60, 220, 3},
      {140, 61, 148, 184, 14}
    },
    { {103, 62, 26, 116, 11},
      {205, 80, 175, 249, 8},
      {210, 229, 135, 206, 6},
      {25, 255, 80, 171, 3}
    },
    { {146, 201, 210, 150, 1},
      {44, 206, 144, 19, 11},
      {134, 148, 185, 52, 9},
      {220, 128, 151, 51, 4}
    },
    { {127, 25, 25, 12, 11},
      {222, 58, 37, 209, 8},
      {211, 9, 137, 143, 14},
      {24, 186, 69, 199, 11}
    },
    { {144, 112, 164, 195, 10},
      {65, 157, 1, 14, 7},
      {92, 50, 80, 224, 9},
      {231, 8, 11, 152, 2}
    },
    { {113, 67, 83, 178, 1},
      {52, 78, 170, 208, 3},
      {132, 220, 172, 40, 14},
      {192, 181, 87, 34, 12}
    },
    { {251, 75, 181, 78, 15},
      {30, 191, 105, 219, 13},
      {247, 42, 221, 45, 15},
      {189, 185, 111, 215, 8}
    },
    { {197, 129, 104, 9, 8},
      {160, 34, 21, 198, 4},
      {25, 1, 104, 26, 3},
      {38, 58, 132, 64, 5}
    },
    { {43, 241, 86, 121, 0},
      {99, 103, 178, 93, 0},
      {9, 230, 168, 253, 4},
      {11, 164, 222, 108, 6}
    },
    { {139, 50, 134, 244, 8},
      {75, 193, 139, 75, 2},
      {18, 246, 20, 205, 1},
      {77, 45, 24, 61, 2}
    },
    { {35, 88, 1, 235, 6},
      {80, 52, 98, 77, 11},
      {109, 120, 1, 172, 4},
      {219, 36, 98, 192, 10}
    },
    { {42, 83, 143, 50, 8},
      {82, 199, 175, 1, 1},
      {20, 207, 28, 165, 4},
      {136, 15, 94, 52, 10}
    },
    { {16, 109, 126, 84, 8},
      {41, 79, 133, 56, 12},
      {18, 167, 235, 96, 8},
      {49, 202, 31, 41, 4}
    },
    { {70, 253, 197, 231, 10},
      {249, 151, 19, 173, 11},
      {94, 122, 59, 246, 2},
      {219, 92, 142, 153, 15}
    },
    { {69, 106, 10, 93, 0},
      {137, 100, 140, 204, 8},
      {11, 165, 5, 106, 2},
      {19, 51, 18, 105, 1}
    },
    { {217, 50, 39, 125, 10},
      {91, 121, 139, 206, 4},
      {91, 238, 68, 201, 11},
      {39, 61, 25, 237, 10}
    },
    { {97, 59, 235, 152, 1},
      {117, 226, 172, 192, 14},
      {129, 157, 125, 200, 6},
      {112, 51, 84, 122, 14}
    },
    { {191, 19, 181, 70, 0},
      {218, 139, 40, 91, 5},
      {6, 42, 220, 143, 13},
      {173, 161, 77, 21, 11}
    },
    { {130, 186, 168, 203, 13},
      {69, 160, 93, 15, 15},
      {189, 49, 85, 212, 1},
      {255, 11, 160, 90, 2}
    },
    { {81, 205, 99, 27, 3},
      {52, 126, 144, 228, 13},
      {205, 140, 107, 56, 10},
      {178, 112, 151, 226, 12}
    },
    { {112, 171, 240, 76, 7},
      {45, 186, 120, 152, 12},
      {227, 32, 253, 80, 14},
      {49, 145, 229, 219, 4}
    },
    { {170, 102, 185, 24, 7},
      {23, 180, 236, 51, 4},
      {225, 137, 214, 101, 5},
      {44, 195, 114, 222, 8}
    },
    { {140, 20, 60, 97, 6},
      {194, 17, 70, 62, 4},
      {104, 99, 194, 131, 1},
      {39, 198, 40, 132, 3}
    },
    { {241, 160, 157, 75, 0},
      {17, 169, 52, 222, 1},
      {13, 43, 144, 88, 15},
      {135, 178, 201, 88, 8}
    },
    { {95, 27, 95, 220, 4},
      {250, 107, 204, 217, 10},
      {35, 191, 173, 143, 10},
      {89, 179, 61, 101, 15}
    },
    { {104, 72, 189, 153, 8},
      {18, 165, 165, 148, 14},
      {25, 155, 209, 33, 6},
      {114, 154, 90, 84, 8}
    },
    { {235, 245, 28, 134, 10},
      {75, 23, 53, 243, 3},
      {86, 19, 138, 253, 7},
      {204, 250, 206, 141, 2}
    },
    { {201, 92, 23, 53, 15},
      {94, 85, 195, 246, 8},
      {250, 206, 131, 169, 3},
      {22, 252, 58, 167, 10}
    },
    { {164, 229, 178, 170, 9},
      {133, 230, 51, 50, 7},
      {149, 84, 218, 114, 5},
      {228, 204, 198, 122, 1}
    },
    { {137, 187, 40, 109, 9},
      {79, 34, 31, 78, 12},
      {155, 97, 77, 217, 1},
      {55, 47, 132, 79, 2}
    },
    { {252, 83, 211, 81, 9},
      {246, 206, 169, 158, 0},
      {152, 172, 188, 163, 15},
      {7, 153, 87, 54, 15}
    },
    { {132, 250, 176, 250, 11},
      {197, 180, 155, 26, 15},
      {213, 240, 213, 242, 1},
      {245, 141, 146, 218, 3}
    },
    { {135, 156, 9, 173, 7},
      {220, 48, 86, 103, 10},
      {235, 89, 3, 158, 1},
      {94, 102, 160, 195, 11}
    },
    { {84, 19, 120, 184, 10},
      {224, 58, 143, 144, 6},
      {81, 209, 236, 130, 10},
      {96, 159, 21, 192, 7}
    },
    { {93, 207, 118, 108, 5},
      {174, 111, 90, 248, 12},
      {163, 102, 239, 59, 10},
      {49, 245, 175, 103, 5}
    },
    { {15, 128, 112, 181, 8},
      {170, 0, 147, 85, 6},
      {26, 208, 224, 31, 0},
      {106, 172, 144, 5, 5}
    },
    { {208, 181, 28, 15, 14},
      {73, 59, 85, 182, 1},
      {127, 3, 138, 208, 11},
      {134, 218, 173, 201, 2}
    },
    { {132, 202, 160, 119, 12},
      {136, 132, 219, 14, 13},
      {62, 224, 85, 50, 1},
      {183, 13, 178, 17, 1}
    },
    { {237, 211, 180, 253, 12},
      {202, 167, 251, 222, 6},
      {59, 242, 220, 187, 7},
      {103, 189, 254, 85, 3}
    },
    { {54, 161, 151, 8, 4},
      {145, 235, 112, 17, 0},
      {33, 14, 152, 86, 12},
      {8, 128, 237, 120, 9}
    },
    { {140, 65, 87, 242, 7},
      {182, 87, 194, 26, 3},
      {228, 254, 168, 35, 1},
      {197, 132, 62, 166, 13}
    },
    { {236, 146, 34, 145, 13},
      {198, 64, 249, 134, 6},
      {184, 148, 68, 147, 7},
      {102, 25, 240, 38, 3}
    },
    { {44, 51, 74, 17, 10},
      {227, 82, 173, 4, 0},
      {88, 133, 44, 195, 4},
      {2, 11, 84, 172, 7}
    },
    { {149, 119, 148, 119, 8},
      {201, 143, 139, 126, 1},
      {30, 226, 158, 234, 9},
      {135, 237, 31, 25, 3}
    },
    { {79, 158, 57, 137, 13},
      {214, 32, 93, 245, 14},
      {185, 25, 199, 159, 2},
      {122, 251, 160, 70, 11}
    },
    { {181, 203, 34, 59, 11},
      {132, 126, 187, 70, 13},
      {221, 196, 77, 58, 13},
      {182, 45, 215, 226, 1}
    },
    { {238, 226, 189, 59, 11},
      {151, 181, 191, 151, 5},
      {221, 203, 212, 119, 7},
      {174, 159, 218, 222, 9}
    },
    { {133, 212, 25, 195, 12},
      {208, 4, 85, 126, 3},
      {60, 57, 130, 186, 1},
      {199, 234, 162, 0, 11}
    },
    { {186, 35, 215, 19, 11},
      {55, 219, 169, 23, 1},
      {220, 142, 188, 69, 13},
      {142, 137, 93, 190, 12}
    },
    { {141, 72, 238, 157, 1},
      {174, 229, 132, 70, 14},
      {139, 151, 113, 43, 1},
      {118, 34, 26, 119, 5}
    },
    { {120, 69, 116, 14, 9},
      {46, 47, 33, 176, 5},
      {151, 2, 234, 33, 14},
      {160, 216, 79, 71, 4}
    },
    { {100, 231, 234, 47, 8},
      {169, 230, 63, 164, 5},
      {31, 69, 126, 114, 6},
      {162, 95, 198, 121, 5}
    },
    { {100, 118, 246, 62, 4},
      {233, 229, 234, 176, 5},
      {39, 198, 246, 226, 6},
      {160, 213, 122, 121, 7}
    },
    { {14, 238, 158, 149, 0},
      {139, 197, 156, 53, 10},
      {10, 151, 151, 119, 0},
      {90, 195, 154, 61, 1}
    },
    { {220, 175, 168, 110, 11},
      {143, 186, 31, 170, 13},
      {215, 97, 95, 83, 11},
      {181, 95, 133, 223, 1}
    },
    { {135, 109, 56, 242, 14},
      {129, 22, 199, 123, 15},
      {116, 241, 203, 110, 1},
      {253, 238, 54, 136, 1}
    },
    { {59, 235, 39, 76, 5},
      {31, 111, 120, 73, 12},
      {163, 46, 77, 125, 12},
      {57, 33, 239, 111, 8}
    },
    { {184, 191, 89, 23, 4},
      {123, 10, 252, 54, 9},
      {46, 137, 175, 209, 13},
      {150, 195, 245, 13, 14}
    },
    { {171, 32, 236, 13, 0},
      {43, 161, 36, 71, 4},
      {11, 3, 112, 77, 5},
      {46, 34, 72, 93, 4}
    },
    { {13, 85, 26, 83, 3},
      {198, 86, 132, 124, 1},
      {204, 165, 138, 171, 0},
      {131, 226, 22, 166, 3}
    },
    { {164, 73, 46, 167, 8},
      {136, 71, 39, 6, 15},
      {30, 87, 73, 34, 5},
      {246, 14, 78, 33, 1}
    },
    { {93, 108, 92, 103, 6},
      {171, 29, 70, 252, 9},
      {110, 99, 163, 107, 10},
      {147, 246, 43, 141, 5}
    },
    { {153, 4, 153, 70, 10},
      {26, 152, 5, 122, 1},
      {86, 41, 146, 9, 9},
      {133, 234, 1, 149, 8}
    },
    { {10, 157, 128, 226, 6},
      {66, 146, 82, 41, 11},
      {100, 112, 27, 149, 0},
      {217, 68, 164, 148, 2}
    },
    { {105, 134, 183, 41, 15},
      {22, 241, 123, 244, 4},
      {249, 78, 214, 25, 6},
      {34, 253, 232, 246, 8}
    },
    { {143, 173, 130, 44, 13},
      {143, 226, 83, 99, 8},
      {179, 68, 27, 95, 1},
      {28, 108, 164, 127, 1}
    },
    { {226, 67, 95, 38, 3},
      {60, 87, 46, 147, 1},
      {198, 79, 172, 36, 7},
      {140, 151, 78, 163, 12}
    },
    { {209, 155, 215, 78, 11},
      {124, 251, 25, 218, 9},
      {215, 46, 189, 152, 11},
      {149, 185, 141, 243, 14}
    },
    { {196, 218, 25, 133, 15},
      {220, 20, 93, 150, 10},
      {250, 25, 133, 178, 3},
      {86, 155, 162, 131, 11}
    },
    { {26, 193, 30, 251, 4},
      {2, 111, 214, 29, 3},
      {45, 247, 136, 53, 8},
      {203, 134, 191, 100, 0}
    },
    { {17, 45, 28, 17, 1},
      {5, 11, 132, 116, 8},
      {136, 131, 139, 72, 8},
      {18, 226, 29, 10, 0}
    },
    { {45, 76, 8, 144, 6},
      {130, 20, 228, 96, 10},
      {96, 145, 3, 43, 4},
      {80, 98, 114, 132, 1}
    },
    { {108, 235, 79, 250, 4},
      {179, 103, 254, 136, 11},
      {37, 255, 45, 115, 6},
      {209, 23, 254, 108, 13}
    },
    { {171, 203, 198, 115, 11},
      {38, 215, 187, 79, 9},
      {220, 230, 61, 61, 5},
      {159, 45, 222, 182, 4}
    },
    { {40, 32, 233, 173, 13},
      {63, 160, 103, 4, 6},
      {187, 89, 112, 65, 4},
      {98, 14, 96, 95, 12}
    },
    { {127, 250, 209, 178, 2},
      {243, 156, 186, 209, 11},
      {68, 216, 181, 255, 14},
      {216, 181, 211, 156, 15}
    },
    { {27, 102, 106, 88, 7},
      {39, 124, 204, 105, 4},
      {225, 165, 102, 109, 8},
      {41, 99, 51, 238, 4}
    },
    { {164, 137, 72, 44, 7},
      {172, 50, 118, 2, 8},
      {227, 65, 41, 18, 5},
      {20, 6, 228, 195, 5}
    },
    { {141, 42, 28, 71, 0},
      {139, 1, 12, 94, 9},
      {14, 35, 133, 75, 1},
      {151, 163, 8, 13, 1}
    },
    { {125, 182, 84, 216, 0},
      {227, 41, 184, 248, 2},
      {1, 178, 166, 219, 14},
      {65, 241, 217, 76, 7}
    },
    { {179, 222, 10, 136, 11},
      {68, 124, 61, 99, 10},
      {209, 21, 7, 188, 13},
      {92, 107, 195, 226, 2}
    },
    { {214, 120, 202, 34, 1},
      {229, 204, 6, 131, 9},
      {132, 69, 49, 230, 11},
      {156, 22, 3, 58, 7}
    },
    { {136, 97, 82, 97, 11},
      {39, 86, 3, 30, 0},
      {216, 100, 168, 97, 1},
      {7, 140, 6, 174, 4}
    },
    { {0, 238, 90, 84, 7},
      {45, 84, 220, 56, 8},
      {226, 165, 167, 112, 0},
      {17, 195, 178, 171, 4}
    },
    { {84, 105, 30, 139, 4},
      {129, 111, 68, 148, 11},
      {45, 23, 137, 98, 10},
      {210, 146, 47, 104, 1}
    },
    { {128, 254, 13, 106, 8},
      {81, 37, 31, 42, 9},
      {21, 107, 7, 240, 1},
      {149, 79, 138, 72, 10}
    },
    { {195, 127, 154, 209, 2},
      {65, 214, 140, 255, 10},
      {72, 181, 159, 236, 3},
      {95, 243, 22, 184, 2}
    },
    { {57, 235, 155, 203, 12},
      {19, 238, 125, 92, 11},
      {61, 61, 157, 121, 12},
      {211, 171, 231, 124, 8}
    },
    { {156, 205, 246, 33, 8},
      {162, 207, 19, 54, 12},
      {24, 70, 251, 51, 9},
      {54, 204, 143, 52, 5}
    },
    { {83, 192, 184, 251, 11},
      {4, 188, 151, 221, 7},
      {221, 241, 208, 60, 10},
      {235, 190, 147, 210, 0}
    },
    { {159, 93, 81, 153, 0},
      {242, 46, 128, 119, 10},
      {9, 152, 171, 175, 9},
      {94, 224, 23, 68, 15}
    },
    { {208, 152, 110, 185, 3},
      {100, 121, 150, 134, 14},
      {201, 215, 97, 144, 11},
      {118, 22, 153, 226, 6}
    },
    { {21, 133, 42, 206, 6},
      {136, 122, 84, 104, 7},
      {103, 53, 74, 26, 8},
      {225, 98, 165, 225, 1}
    },
    { {144, 212, 121, 19, 3},
      {116, 28, 148, 54, 5},
      {204, 137, 226, 176, 9},
      {166, 194, 147, 130, 14}
    },
    { {44, 22, 172, 202, 12},
      {194, 161, 109, 40, 7},
      {53, 51, 86, 131, 4},
      {225, 75, 104, 84, 3}
    },
    { {112, 226, 233, 181, 14},
      {57, 156, 255, 132, 6},
      {122, 217, 116, 112, 14},
      {98, 31, 243, 153, 12}
    },
    { {36, 160, 106, 178, 15},
      {165, 80, 247, 0, 7},
      {244, 213, 96, 82, 4},
      {224, 14, 240, 170, 5}
    },
    { {114, 1, 237, 44, 10},
      {56, 187, 39, 129, 4},
      {83, 75, 120, 4, 14},
      {40, 30, 77, 209, 12}
    },
    { {76, 99, 9, 66, 8},
      {147, 6, 13, 136, 1},
      {20, 41, 12, 99, 2},
      {129, 27, 6, 12, 9}
    },
    { {6, 185, 203, 230, 13},
      {253, 194, 87, 9, 11},
      {182, 125, 57, 214, 0},
      {217, 14, 164, 59, 15}
    },
    { {215, 246, 104, 218, 12},
      {225, 44, 221, 235, 7},
      {53, 177, 102, 254, 11},
      {237, 123, 179, 72, 7}
    },
    { {129, 109, 246, 113, 1},
      {37, 199, 130, 126, 12},
      {136, 230, 251, 104, 1},
      {55, 228, 30, 58, 4}
    },
    { {88, 117, 65, 240, 5},
      {119, 14, 194, 168, 2},
      {160, 248, 42, 225, 10},
      {65, 84, 55, 14, 14}
    },
    { {74, 115, 6, 33, 0},
      {67, 71, 10, 133, 0},
      {8, 70, 12, 229, 2},
      {10, 21, 14, 44, 2}
    },
    { {72, 9, 86, 13, 1},
      {46, 99, 0, 148, 8},
      {139, 6, 169, 1, 2},
      {18, 144, 12, 103, 4}
    },
    { {59, 153, 72, 245, 11},
      {110, 26, 183, 77, 10},
      {218, 241, 41, 157, 12},
      {91, 46, 213, 135, 6}
    },
    { {54, 150, 213, 148, 6},
      {248, 153, 248, 49, 2},
      {98, 154, 182, 150, 12},
      {72, 193, 249, 145, 15}
    },
    { {81, 34, 220, 136, 12},
      {33, 169, 77, 208, 2},
      {49, 19, 180, 72, 10},
      {64, 187, 41, 88, 4}
    },
    { {168, 69, 58, 219, 0},
      {2, 102, 164, 62, 7},
      {13, 181, 202, 33, 5},
      {231, 194, 86, 100, 0}
    },
    { {180, 120, 208, 7, 12},
      {233, 140, 97, 22, 9},
      {62, 0, 177, 226, 13},
      {150, 136, 99, 25, 7}
    },
    { {213, 175, 225, 90, 1},
      {181, 170, 152, 234, 13},
      {133, 168, 127, 90, 11},
      {181, 113, 149, 90, 13}
    },
    { {4, 255, 169, 156, 13},
      {221, 166, 221, 32, 14},
      {179, 153, 95, 242, 0},
      {112, 75, 182, 91, 11}
    },
    { {178, 170, 134, 106, 1},
      {5, 233, 58, 11, 9},
      {133, 102, 21, 84, 13},
      {157, 5, 201, 122, 0}
    },
    { {33, 202, 57, 14, 7},
      {28, 52, 124, 80, 13},
      {231, 9, 197, 56, 4},
      {176, 163, 226, 195, 8}
    },
    { {217, 149, 56, 136, 0},
      {66, 42, 20, 242, 6},
      {1, 17, 202, 153, 11},
      {100, 242, 133, 68, 2}
    },
    { {158, 66, 134, 121, 2},
      {130, 253, 138, 15, 0},
      {73, 230, 20, 39, 9},
      {15, 5, 27, 244, 1}
    },
    { {35, 126, 230, 40, 15},
      {101, 245, 107, 97, 12},
      {241, 70, 119, 236, 4},
      {56, 109, 106, 250, 6}
    },
    { {223, 217, 194, 111, 1},
      {238, 238, 18, 207, 9},
      {143, 100, 57, 191, 11},
      {159, 52, 135, 119, 7}
    },
    { {134, 37, 95, 146, 14},
      {177, 83, 197, 51, 3},
      {116, 159, 170, 70, 1},
      {204, 202, 60, 168, 13}
    },
    { {89, 24, 159, 35, 8},
      {82, 201, 7, 212, 9},
      {28, 79, 145, 137, 10},
      {146, 190, 9, 52, 10}
    },
    { {83, 168, 9, 113, 7},
      {21, 24, 214, 205, 8},
      {232, 233, 1, 92, 10},
      {27, 54, 177, 138, 8}
    },
    { {8, 171, 22, 118, 15},
      {15, 83, 219, 24, 9},
      {246, 230, 141, 81, 0},
      {145, 141, 188, 175, 0}
    },
    { {249, 140, 152, 28, 13},
      {14, 168, 245, 242, 8},
      {179, 129, 147, 25, 15},
      {20, 250, 241, 87, 0}
    },
    { {249, 136, 239, 149, 11},
      {62, 217, 181, 198, 14},
      {218, 159, 113, 25, 15},
      {118, 58, 217, 183, 12}
    },
    { {31, 2, 254, 30, 14},
      {170, 249, 205, 81, 5},
      {119, 135, 244, 15, 8},
      {168, 171, 57, 245, 5}
    },
    { {104, 234, 254, 163, 13},
      {39, 197, 127, 148, 15},
      {188, 87, 245, 113, 6},
      {242, 159, 234, 62, 4}
    },
    { {126, 145, 79, 240, 6},
      {242, 91, 246, 137, 2},
      {96, 255, 40, 151, 14},
      {73, 22, 253, 164, 15}
    },
    { {44, 149, 20, 92, 13},
      {206, 35, 241, 56, 0},
      {179, 162, 138, 147, 4},
      {1, 200, 252, 71, 3}
    },
    { {62, 147, 44, 208, 8},
      {194, 11, 189, 9, 6},
      {16, 179, 76, 151, 12},
      {105, 11, 221, 4, 3}
    },
    { {183, 92, 175, 40, 3},
      {212, 253, 38, 99, 12},
      {193, 79, 83, 174, 13},
      {60, 102, 75, 242, 11}
    },
    { {186, 245, 114, 60, 11},
      {111, 126, 179, 51, 4},
      {211, 196, 234, 245, 13},
      {44, 204, 215, 239, 6}
    },
    { {121, 203, 18, 140, 13},
      {14, 110, 121, 208, 10},
      {179, 20, 141, 57, 14},
      {80, 185, 231, 103, 0}
    },
    { {64, 87, 46, 30, 0},
      {72, 103, 140, 160, 5},
      {7, 135, 78, 160, 2},
      {160, 83, 30, 97, 2}
    },
    { {82, 212, 150, 97, 15},
      {68, 221, 83, 189, 0},
      {248, 102, 146, 180, 10},
      {11, 220, 171, 178, 2}
    },
    { {177, 112, 92, 13, 10},
      {105, 61, 37, 86, 0},
      {91, 3, 160, 232, 13},
      {6, 170, 75, 201, 6}
    },
    { {99, 49, 219, 103, 3},
      {125, 210, 38, 221, 1},
      {206, 109, 184, 204, 6},
      {139, 182, 68, 187, 14}
    },
    { {16, 154, 188, 196, 1},
      {76, 137, 28, 24, 14},
      {130, 51, 213, 144, 8},
      {113, 131, 137, 19, 2}
    },
    { {96, 60, 124, 104, 15},
      {101, 49, 103, 184, 12},
      {241, 99, 227, 192, 6},
      {49, 222, 104, 202, 6}
    },
    { {115, 50, 161, 255, 11},
      {93, 184, 171, 205, 7},
      {223, 248, 84, 204, 14},
      {235, 61, 81, 219, 10}
    },
    { {61, 143, 96, 230, 4},
      {170, 10, 122, 104, 15},
      {38, 112, 111, 27, 12},
      {241, 101, 229, 5, 5}
    },
    { {202, 97, 172, 50, 9},
      {7, 135, 135, 131, 5},
      {148, 195, 88, 101, 3},
      {172, 30, 30, 30, 0}
    },
    { {132, 25, 50, 28, 14},
      {200, 114, 193, 18, 12},
      {115, 132, 201, 130, 1},
      {52, 136, 52, 225, 3}
    },
    { {171, 167, 29, 138, 7},
      {23, 51, 124, 115, 3},
      {229, 27, 142, 93, 5},
      {204, 227, 236, 206, 8}
    },
    { {220, 105, 76, 91, 7},
      {167, 63, 196, 142, 9},
      {237, 163, 41, 99, 11},
      {151, 18, 63, 206, 5}
    },
    { {244, 156, 252, 224, 5},
      {228, 137, 118, 186, 14},
      {160, 115, 243, 146, 15},
      {117, 214, 233, 18, 7}
    },
    { {200, 69, 243, 60, 3},
      {62, 246, 130, 178, 4},
      {195, 204, 250, 33, 3},
      {36, 212, 22, 247, 12}
    },
    { {104, 74, 251, 82, 12},
      {50, 196, 237, 152, 13},
      {52, 173, 245, 33, 6},
      {177, 155, 114, 52, 12}
    },
    { {214, 20, 91, 29, 3},
      {252, 120, 132, 183, 0},
      {203, 141, 162, 134, 11},
      {14, 210, 17, 227, 15}
    },
    { {16, 253, 223, 173, 12},
      {121, 239, 87, 52, 10},
      {59, 95, 187, 240, 8},
      {82, 206, 175, 121, 14}
    },
    { {31, 139, 72, 20, 8},
      {170, 10, 157, 65, 8},
      {18, 129, 45, 31, 8},
      {24, 43, 149, 5, 5}
    },
    { {156, 50, 163, 233, 3},
      {215, 248, 10, 14, 6},
      {201, 124, 84, 195, 9},
      {103, 5, 1, 254, 11}
    },
    { {157, 108, 124, 56, 7},
      {167, 61, 198, 114, 12},
      {225, 195, 227, 107, 9},
      {52, 230, 59, 206, 5}
    },
    { {183, 3, 177, 221, 5},
      {156, 170, 232, 95, 6},
      {171, 184, 220, 14, 13},
      {111, 161, 117, 83, 9}
    },
    { {111, 221, 59, 152, 0},
      {210, 102, 180, 241, 14},
      {1, 157, 203, 191, 6},
      {120, 242, 214, 100, 11}
    },
    { {178, 62, 176, 218, 4},
      {65, 168, 232, 59, 15},
      {37, 176, 215, 196, 13},
      {253, 193, 113, 88, 2}
    },
    { {115, 173, 103, 162, 0},
      {49, 75, 50, 225, 15},
      {4, 94, 107, 92, 14},
      {248, 116, 205, 40, 12}
    },
    { {83, 97, 3, 190, 8},
      {25, 110, 131, 193, 3},
      {23, 220, 8, 108, 10},
      {200, 60, 23, 105, 8}
    },
    { {214, 122, 66, 252, 3},
      {237, 124, 138, 139, 10},
      {195, 244, 37, 230, 11},
      {93, 21, 19, 235, 7}
    },
    { {31, 40, 119, 191, 6},
      {187, 121, 194, 85, 15},
      {111, 222, 225, 79, 8},
      {250, 164, 57, 237, 13}
    },
    { {45, 103, 170, 137, 12},
      {131, 230, 109, 100, 6},
      {57, 21, 94, 107, 4},
      {98, 107, 102, 124, 1}
    },
    { {168, 55, 137, 133, 0},
      {91, 130, 44, 38, 2},
      {10, 25, 30, 193, 5},
      {70, 67, 68, 29, 10}
    },
    { {47, 235, 63, 25, 12},
      {147, 103, 253, 85, 12},
      {57, 143, 205, 127, 4},
      {58, 171, 254, 108, 9}
    },
    { {41, 160, 212, 118, 5},
      {47, 129, 242, 88, 1},
      {166, 226, 176, 89, 4},
      {129, 164, 248, 31, 4}
    },
    { {65, 177, 230, 167, 15},
      {109, 211, 83, 196, 7},
      {254, 86, 120, 216, 2},
      {226, 60, 172, 187, 6}
    },
    { {248, 93, 126, 140, 7},
      {110, 127, 100, 178, 14},
      {227, 23, 235, 161, 15},
      {116, 210, 111, 231, 6}
    },
    { {133, 48, 143, 35, 4},
      {209, 193, 70, 70, 1},
      {44, 79, 16, 202, 1},
      {134, 38, 40, 56, 11}
    },
    { {121, 131, 156, 71, 5},
      {14, 139, 124, 220, 1},
      {174, 35, 156, 25, 14},
      {131, 179, 237, 23, 0}
    },
    { {75, 124, 102, 113, 8},
      {99, 69, 131, 237, 12},
      {24, 230, 99, 237, 2},
      {59, 124, 26, 44, 6}
    },
    { {231, 237, 2, 5, 2},
      {137, 86, 48, 231, 8},
      {74, 4, 11, 126, 7},
      {30, 112, 198, 169, 1}
    },
    { {57, 33, 5, 61, 13},
      {31, 43, 227, 68, 0},
      {187, 202, 8, 73, 12},
      {2, 44, 125, 79, 8}
    },
    { {46, 92, 218, 7, 3},
      {238, 212, 36, 53, 9},
      {206, 5, 179, 167, 4},
      {154, 194, 66, 183, 7}
    },
    { {191, 65, 151, 115, 14},
      {146, 223, 227, 95, 1},
      {124, 238, 152, 47, 13},
      {143, 172, 127, 180, 9}
    },
    { {229, 242, 167, 113, 14},
      {209, 213, 251, 206, 4},
      {120, 238, 84, 250, 7},
      {39, 61, 250, 184, 11}
    },
    { {189, 13, 222, 45, 14},
      {170, 251, 103, 118, 8},
      {123, 71, 187, 11, 13},
      {22, 238, 109, 245, 5}
    },
    { {104, 73, 143, 23, 14},
      {26, 215, 229, 132, 9},
      {126, 143, 25, 33, 6},
      {146, 26, 126, 181, 8}
    },
    { {183, 56, 155, 230, 11},
      {221, 216, 39, 91, 11},
      {214, 125, 145, 206, 13},
      {221, 174, 65, 187, 11}
    },
    { {41, 127, 247, 175, 7},
      {127, 247, 106, 116, 15},
      {239, 94, 255, 233, 4},
      {242, 229, 110, 255, 14}
    },
    { {177, 201, 139, 131, 7},
      {20, 222, 116, 70, 11},
      {236, 29, 25, 56, 13},
      {214, 34, 231, 178, 8}
    },
    { {112, 239, 122, 3, 2},
      {33, 94, 60, 180, 13},
      {76, 5, 239, 112, 14},
      {178, 211, 199, 168, 4}
    },
    { {155, 66, 34, 76, 0},
      {10, 108, 8, 75, 4},
      {3, 36, 68, 45, 9},
      {45, 33, 3, 101, 0}
    },
    { {45, 220, 24, 13, 15},
      {206, 52, 117, 116, 8},
      {251, 1, 131, 187, 4},
      {18, 234, 226, 199, 3}
    },
    { {10, 189, 5, 237, 15},
      {95, 51, 83, 45, 10},
      {251, 122, 11, 213, 0},
      {91, 76, 172, 207, 10}
    },
    { {225, 190, 133, 89, 1},
      {85, 161, 184, 238, 8},
      {137, 170, 23, 216, 7},
      {23, 113, 216, 90, 10}
    },
    { {145, 183, 86, 114, 7},
      {101, 91, 218, 122, 1},
      {228, 230, 174, 216, 9},
      {133, 229, 189, 170, 6}
    },
    { {45, 22, 243, 125, 11},
      {254, 240, 171, 124, 4},
      {219, 236, 246, 139, 4},
      {35, 237, 80, 247, 15}
    },
    { {39, 176, 238, 193, 6},
      {225, 209, 116, 77, 6},
      {104, 55, 112, 222, 4},
      {107, 34, 232, 184, 7}
    },
    { {108, 133, 130, 249, 13},
      {134, 226, 243, 172, 2},
      {185, 244, 26, 19, 6},
      {67, 92, 244, 118, 1}
    },
    { {115, 83, 40, 110, 3},
      {76, 62, 46, 201, 5},
      {199, 97, 76, 172, 14},
      {169, 55, 71, 195, 2}
    },
    { {97, 193, 148, 57, 1},
      {4, 167, 178, 212, 0},
      {137, 194, 152, 56, 6},
      {2, 180, 222, 82, 0}
    },
    { {123, 144, 243, 154, 9},
      {118, 232, 177, 209, 7},
      {149, 156, 240, 157, 14},
      {232, 184, 209, 118, 14}
    },
    { {199, 152, 208, 205, 5},
      {236, 160, 80, 223, 10},
      {171, 48, 177, 158, 3},
      {95, 176, 160, 83, 7}
    },
    { {40, 109, 39, 77, 11},
      {31, 119, 33, 44, 12},
      {219, 46, 75, 97, 4},
      {51, 72, 78, 239, 8}
    },
    { {217, 203, 12, 182, 4},
      {10, 15, 222, 194, 11},
      {38, 211, 13, 57, 11},
      {212, 55, 191, 5, 0}
    },
    { {74, 39, 247, 51, 6},
      {51, 211, 202, 181, 5},
      {108, 206, 254, 69, 2},
      {170, 213, 60, 188, 12}
    },
    { {58, 192, 163, 225, 10},
      {18, 220, 51, 13, 6},
      {88, 124, 80, 53, 12},
      {107, 12, 195, 180, 8}
    },
    { {7, 221, 220, 229, 7},
      {236, 151, 86, 125, 10},
      {234, 115, 187, 190, 0},
      {91, 230, 174, 147, 7}
    },
    { {136, 88, 155, 155, 11},
      {86, 244, 133, 22, 11},
      {221, 157, 145, 161, 1},
      {214, 138, 18, 246, 10}
    },
    { {138, 23, 163, 27, 11},
      {86, 242, 137, 39, 5},
      {221, 140, 94, 133, 1},
      {174, 73, 20, 246, 10}
    },
    { {238, 5, 82, 209, 7},
      {166, 82, 224, 191, 2},
      {232, 180, 170, 7, 7},
      {79, 208, 116, 166, 5}
    },
    { {187, 95, 31, 51, 1},
      {86, 79, 174, 119, 9},
      {140, 207, 143, 173, 13},
      {158, 231, 95, 38, 10}
    },
    { {208, 60, 187, 190, 2},
      {89, 248, 134, 178, 15},
      {71, 221, 211, 192, 11},
      {244, 214, 17, 249, 10}
    },
    { {152, 244, 149, 172, 6},
      {91, 189, 82, 50, 2},
      {99, 90, 146, 241, 9},
      {68, 196, 171, 221, 10}
    },
    { {197, 139, 251, 112, 15},
      {180, 210, 223, 218, 12},
      {240, 237, 253, 26, 3},
      {53, 191, 180, 178, 13}
    },
    { {122, 253, 6, 239, 8},
      {75, 111, 51, 173, 11},
      {31, 118, 11, 245, 14},
      {219, 92, 207, 109, 2}
    },
    { {81, 109, 127, 193, 5},
      {53, 79, 68, 252, 14},
      {168, 63, 235, 104, 10},
      {115, 242, 47, 42, 12}
    },
    { {187, 149, 178, 239, 2},
      {74, 250, 50, 127, 7},
      {79, 116, 218, 157, 13},
      {239, 228, 197, 245, 2}
    },
    { {74, 73, 174, 244, 10},
      {10, 215, 135, 137, 14},
      {82, 247, 89, 37, 2},
      {121, 30, 30, 181, 0}
    },
    { {121, 22, 119, 209, 9},
      {118, 73, 169, 252, 6},
      {152, 190, 230, 137, 14},
      {99, 249, 89, 38, 14}
    },
    { {171, 195, 19, 250, 3},
      {22, 118, 186, 91, 3},
      {197, 252, 140, 61, 5},
      {205, 165, 214, 230, 8}
    },
    { {97, 72, 188, 236, 1},
      {12, 165, 38, 216, 14},
      {131, 115, 209, 40, 6},
      {113, 182, 74, 83, 0}
    },
    { {189, 91, 56, 250, 15},
      {198, 62, 239, 90, 15},
      {245, 241, 205, 171, 13},
      {245, 175, 119, 198, 3}
    },
    { {167, 203, 22, 91, 5},
      {132, 103, 248, 95, 9},
      {173, 166, 141, 62, 5},
      {159, 161, 254, 98, 1}
    },
    { {122, 215, 157, 221, 12},
      {90, 175, 253, 189, 2},
      {59, 187, 158, 181, 14},
      {75, 219, 255, 85, 10}
    },
    { {30, 33, 123, 48, 10},
      {179, 90, 135, 17, 4},
      {80, 205, 232, 71, 8},
      {40, 142, 21, 172, 13}
    },
    { {7, 106, 159, 128, 15},
      {149, 213, 77, 81, 10},
      {240, 31, 149, 110, 0},
      {88, 171, 42, 186, 9}
    },
    { {66, 20, 244, 73, 5},
      {100, 161, 64, 189, 4},
      {169, 34, 242, 132, 2},
      {43, 208, 40, 82, 6}
    },
    { {90, 77, 21, 204, 4},
      {26, 47, 64, 185, 10},
      {35, 58, 139, 37, 10},
      {89, 208, 47, 69, 8}
    },
    { {42, 237, 24, 136, 14},
      {3, 54, 117, 49, 10},
      {113, 17, 139, 117, 4},
      {88, 202, 230, 204, 0}
    },
    { {74, 162, 225, 254, 13},
      {63, 160, 219, 137, 7},
      {183, 248, 116, 85, 2},
      {233, 29, 176, 95, 12}
    },
    { {132, 142, 174, 178, 2},
      {128, 209, 158, 34, 15},
      {68, 215, 87, 18, 1},
      {244, 71, 152, 176, 1}
    },
    { {51, 40, 220, 225, 13},
      {37, 137, 103, 93, 10},
      {184, 115, 177, 76, 12},
      {91, 174, 105, 26, 4}
    },
    { {58, 219, 131, 61, 10},
      {90, 254, 187, 5, 8},
      {91, 204, 29, 181, 12},
      {26, 13, 215, 245, 10}
    },
    { {115, 169, 29, 159, 13},
      {29, 43, 245, 213, 11},
      {191, 155, 137, 92, 14},
      {218, 186, 253, 75, 8}
    },
    { {4, 109, 218, 103, 10},
      {169, 214, 7, 60, 9},
      {94, 101, 187, 98, 0},
      {147, 206, 6, 185, 5}
    },
    { {58, 203, 120, 189, 13},
      {46, 46, 255, 21, 14},
      {187, 209, 237, 53, 12},
      {122, 143, 247, 71, 4}
    },
    { {206, 153, 231, 26, 12},
      {242, 227, 209, 131, 13},
      {53, 142, 121, 151, 3},
      {188, 24, 188, 116, 15}
    },
    { {165, 37, 165, 84, 6},
      {153, 147, 224, 106, 4},
      {98, 170, 90, 74, 5},
      {37, 96, 124, 153, 9}
    },
    { {240, 63, 168, 232, 1},
      {69, 170, 46, 170, 14},
      {129, 113, 95, 192, 15},
      {117, 87, 69, 90, 2}
    },
    { {118, 229, 192, 251, 1},
      {165, 174, 178, 173, 3},
      {141, 240, 58, 118, 14},
      {203, 84, 215, 90, 5}
    },
    { {227, 99, 237, 110, 12},
      {57, 167, 111, 203, 5},
      {55, 107, 124, 108, 7},
      {173, 63, 110, 89, 12}
    },
    { {116, 249, 37, 224, 5},
      {213, 15, 114, 136, 14},
      {160, 122, 73, 242, 14},
      {113, 20, 239, 10, 11}
    },
    { {198, 217, 168, 122, 2},
      {192, 182, 150, 139, 13},
      {69, 225, 89, 182, 3},
      {189, 22, 150, 208, 3}
    },
    { {13, 96, 234, 117, 11},
      {175, 212, 135, 76, 4},
      {218, 229, 112, 107, 0},
      {35, 46, 18, 191, 5}
    },
    { {51, 227, 74, 122, 13},
      {37, 110, 255, 73, 1},
      {181, 229, 44, 124, 12},
      {137, 47, 247, 106, 4}
    },
    { {113, 122, 201, 96, 4},
      {113, 140, 110, 200, 8},
      {32, 105, 53, 232, 14},
      {17, 55, 99, 24, 14}
    },
    { {152, 138, 48, 249, 12},
      {2, 40, 219, 30, 14},
      {57, 240, 197, 17, 9},
      {119, 141, 177, 68, 0}
    },
    { {46, 163, 26, 179, 12},
      {131, 66, 255, 21, 3},
      {60, 213, 140, 87, 4},
      {202, 143, 244, 44, 1}
    },
    { {172, 182, 25, 142, 10},
      {219, 48, 61, 50, 3},
      {87, 25, 134, 211, 5},
      {196, 203, 192, 205, 11}
    },
    { {240, 101, 182, 107, 5},
      {5, 239, 98, 190, 5},
      {173, 102, 218, 96, 15},
      {167, 212, 111, 122, 0}
    },
    { {168, 198, 88, 212, 0},
      {42, 4, 188, 58, 2},
      {2, 177, 166, 49, 5},
      {69, 195, 210, 5, 4}
    },
    { {204, 173, 140, 128, 11},
      {135, 147, 21, 162, 10},
      {208, 19, 27, 83, 3},
      {84, 90, 140, 158, 1}
    },
    { {163, 169, 133, 130, 14},
      {17, 147, 113, 67, 11},
      {116, 26, 25, 92, 5},
      {220, 40, 236, 152, 8}
    },
    { {236, 44, 102, 29, 0},
      {171, 97, 160, 166, 12},
      {11, 134, 99, 67, 7},
      {54, 80, 88, 109, 5}
    },
    { {25, 228, 167, 179, 8},
      {19, 205, 147, 100, 7},
      {28, 222, 82, 121, 8},
      {226, 108, 155, 60, 8}
    },
    { {141, 26, 103, 29, 6},
      {250, 113, 200, 70, 12},
      {107, 142, 101, 139, 1},
      {54, 33, 56, 229, 15}
    },
    { {178, 43, 3, 194, 13},
      {21, 74, 105, 11, 11},
      {180, 60, 13, 68, 13},
      {221, 9, 101, 42, 8}
    },
    { {233, 107, 129, 42, 2},
      {19, 182, 42, 194, 9},
      {69, 72, 29, 105, 7},
      {148, 53, 70, 220, 8}
    },
    { {175, 188, 151, 185, 4},
      {211, 225, 242, 119, 10},
      {41, 222, 147, 223, 5},
      {94, 228, 248, 124, 11}
    },
    { {46, 29, 146, 160, 8},
      {194, 194, 35, 49, 10},
      {16, 84, 155, 135, 4},
      {88, 204, 68, 52, 3}
    },
    { {20, 26, 214, 160, 3},
      {228, 217, 10, 16, 10},
      {192, 86, 181, 130, 8},
      {80, 133, 9, 178, 7}
    },
    { {181, 12, 21, 252, 0},
      {152, 41, 162, 122, 10},
      {3, 250, 131, 10, 13},
      {85, 228, 89, 65, 9}
    },
    { {95, 40, 182, 146, 9},
      {135, 201, 129, 209, 15},
      {148, 150, 209, 79, 10},
      {248, 184, 25, 62, 1}
    },
    { {5, 73, 160, 42, 15},
      {132, 182, 67, 64, 13},
      {245, 64, 89, 42, 0},
      {176, 44, 38, 210, 1}
    },
    { {170, 29, 19, 140, 15},
      {94, 114, 97, 51, 10},
      {243, 28, 139, 133, 5},
      {92, 200, 100, 231, 10}
    },
    { {214, 197, 212, 242, 6},
      {160, 159, 210, 187, 3},
      {100, 242, 186, 54, 11},
      {205, 212, 191, 144, 5}
    },
    { {175, 39, 99, 231, 15},
      {191, 82, 107, 111, 7},
      {254, 124, 110, 79, 5},
      {239, 109, 100, 175, 13}
    },
    { {13, 30, 84, 39, 10},
      {234, 17, 11, 116, 9},
      {94, 66, 167, 139, 0},
      {146, 237, 8, 133, 7}
    },
    { {105, 13, 29, 67, 9},
      {22, 3, 37, 252, 9},
      {156, 43, 139, 9, 6},
      {147, 250, 76, 6, 8}
    },
    { {142, 118, 82, 211, 11},
      {231, 84, 137, 63, 3},
      {220, 180, 166, 231, 1},
      {207, 201, 18, 174, 7}
    },
    { {183, 231, 13, 198, 11},
      {157, 31, 61, 107, 3},
      {214, 59, 14, 126, 13},
      {205, 107, 207, 139, 9}
    },
    { {91, 61, 61, 135, 11},
      {95, 27, 5, 245, 15},
      {222, 27, 203, 205, 10},
      {250, 250, 13, 143, 10}
    },
    { {79, 134, 142, 113, 6},
      {130, 209, 222, 237, 0},
      {104, 231, 22, 31, 2},
      {11, 119, 184, 180, 1}
    },
    { {177, 31, 187, 167, 12},
      {88, 202, 111, 118, 15},
      {62, 93, 223, 136, 13},
      {246, 239, 101, 49, 10}
    },
    { {128, 174, 197, 152, 2},
      {49, 177, 152, 34, 10},
      {65, 154, 55, 80, 1},
      {84, 65, 152, 216, 12}
    },
    { {213, 231, 198, 181, 15},
      {173, 223, 219, 230, 2},
      {250, 214, 62, 122, 11},
      {70, 125, 191, 187, 5}
    },
    { {93, 45, 129, 118, 3},
      {159, 154, 130, 232, 9},
      {198, 232, 27, 75, 10},
      {145, 116, 21, 159, 9}
    },
    { {203, 168, 201, 76, 1},
      {63, 160, 20, 203, 8},
      {131, 41, 49, 93, 3},
      {29, 50, 128, 95, 12}
    },
    { {103, 230, 75, 17, 3},
      {181, 84, 188, 229, 0},
      {200, 141, 38, 126, 6},
      {10, 115, 210, 170, 13}
    },
    { {250, 67, 80, 92, 14},
      {42, 62, 233, 155, 0},
      {115, 160, 172, 37, 15},
      {13, 153, 119, 197, 4}
    },
    { {240, 150, 174, 222, 7},
      {76, 249, 252, 170, 7},
      {231, 183, 86, 144, 15},
      {229, 83, 249, 243, 2}
    },
    { {71, 218, 157, 211, 11},
      {212, 149, 157, 221, 11},
      {220, 187, 149, 190, 2},
      {219, 187, 154, 146, 11}
    },
    { {154, 5, 75, 133, 14},
      {58, 90, 69, 39, 2},
      {122, 29, 42, 5, 9},
      {78, 74, 37, 165, 12}
    },
    { {120, 108, 53, 241, 11},
      {23, 29, 163, 188, 14},
      {216, 250, 195, 97, 14},
      {115, 220, 91, 142, 8}
    },
    { {147, 175, 244, 140, 6},
      {41, 187, 88, 115, 14},
      {99, 18, 255, 92, 9},
      {124, 225, 173, 217, 4}
    },
    { {42, 176, 175, 52, 11},
      {95, 209, 183, 1, 4},
      {210, 207, 80, 213, 4},
      {40, 14, 216, 191, 10}
    },
    { {240, 11, 250, 99, 1},
      {36, 202, 46, 158, 13},
      {140, 101, 253, 0, 15},
      {183, 151, 69, 50, 4}
    },
    { {76, 166, 198, 133, 5},
      {175, 193, 88, 164, 2},
      {170, 22, 54, 83, 2},
      {66, 81, 168, 63, 5}
    },
    { {186, 20, 227, 14, 1},
      {126, 232, 32, 35, 5},
      {135, 12, 114, 133, 13},
      {172, 64, 65, 119, 14}
    },
    { {33, 208, 242, 145, 8},
      {96, 196, 177, 84, 6},
      {24, 148, 240, 184, 4},
      {98, 168, 210, 48, 6}
    },
    { {195, 163, 48, 100, 2},
      {9, 18, 26, 219, 4},
      {66, 96, 204, 92, 3},
      {45, 181, 132, 137, 0}
    },
    { {55, 157, 233, 153, 7},
      {244, 186, 244, 101, 14},
      {233, 153, 123, 158, 12},
      {122, 98, 245, 210, 15}
    },
    { {34, 132, 160, 94, 1},
      {12, 160, 176, 41, 5},
      {135, 160, 82, 20, 4},
      {169, 64, 208, 83, 0}
    },
    { {111, 213, 214, 14, 1},
      {238, 231, 48, 241, 1},
      {135, 6, 186, 191, 6},
      {136, 240, 206, 119, 7}
    },
    { {203, 84, 111, 171, 13},
      {118, 101, 71, 231, 7},
      {189, 95, 98, 173, 3},
      {238, 126, 42, 102, 14}
    },
    { {156, 42, 145, 229, 8},
      {155, 136, 11, 30, 10},
      {26, 120, 149, 67, 9},
      {87, 141, 1, 29, 9}
    },
    { {61, 210, 148, 126, 15},
      {206, 189, 251, 88, 1},
      {247, 226, 148, 187, 12},
      {129, 173, 251, 215, 3}
    },
    { {100, 83, 228, 51, 13},
      {228, 135, 235, 132, 5},
      {188, 194, 124, 162, 6},
      {162, 29, 126, 18, 7}
    },
    { {43, 198, 86, 24, 12},
      {34, 101, 249, 113, 0},
      {49, 134, 166, 61, 4},
      {8, 233, 250, 100, 4}
    },
    { {37, 39, 183, 207, 14},
      {153, 243, 105, 124, 7},
      {127, 62, 222, 74, 4},
      {227, 233, 108, 249, 9}
    },
    { {22, 36, 199, 38, 13},
      {189, 201, 67, 33, 1},
      {182, 78, 50, 70, 8},
      {136, 76, 41, 59, 13}
    },
    { {202, 176, 164, 248, 14},
      {67, 177, 211, 139, 6},
      {113, 242, 80, 213, 3},
      {109, 28, 184, 220, 2}
    },
    { {193, 236, 138, 99, 14},
      {1, 212, 87, 238, 9},
      {124, 101, 19, 120, 3},
      {151, 126, 162, 184, 0}
    },
    { {82, 142, 30, 232, 13},
      {4, 105, 95, 185, 10},
      {177, 119, 135, 20, 10},
      {89, 223, 169, 98, 0}
    },
    { {254, 163, 79, 210, 9},
      {183, 75, 189, 139, 3},
      {148, 191, 44, 87, 15},
      {205, 27, 221, 46, 13}
    },
    { {92, 145, 249, 82, 6},
      {242, 154, 212, 152, 5},
      {100, 169, 248, 147, 10},
      {161, 146, 181, 148, 15}
    },
    { {42, 103, 218, 12, 8},
      {43, 230, 45, 49, 0},
      {19, 5, 190, 101, 4},
      {8, 203, 70, 125, 4}
    },
    { {111, 147, 7, 201, 2},
      {210, 115, 56, 205, 2},
      {73, 62, 12, 159, 6},
      {75, 49, 204, 228, 11}
    },
    { {27, 127, 101, 218, 0},
      {115, 47, 136, 105, 15},
      {5, 186, 111, 237, 8},
      {249, 97, 31, 76, 14}
    },
    { {96, 94, 48, 82, 3},
      {68, 20, 168, 184, 13},
      {196, 160, 199, 160, 6},
      {177, 209, 82, 130, 2}
    },
    { {187, 3, 116, 51, 0},
      {34, 11, 170, 87, 5},
      {12, 194, 236, 13, 13},
      {174, 165, 93, 4, 4}
    },
    { {245, 150, 230, 89, 11},
      {228, 249, 185, 238, 4},
      {217, 166, 118, 154, 15},
      {39, 121, 217, 242, 7}
    },
    { {146, 81, 227, 36, 14},
      {120, 222, 67, 3, 4},
      {114, 76, 120, 164, 9},
      {44, 12, 39, 177, 14}
    },
    { {37, 129, 147, 161, 3},
      {148, 210, 50, 84, 2},
      {200, 92, 152, 26, 4},
      {66, 164, 196, 178, 9}
    },
    { {179, 35, 202, 164, 8},
      {41, 202, 47, 67, 2},
      {18, 85, 60, 76, 13},
      {76, 47, 69, 57, 4}
    },
    { {141, 73, 29, 60, 1},
      {158, 39, 134, 82, 8},
      {131, 203, 137, 43, 1},
      {20, 166, 30, 71, 9}
    },
    { {227, 51, 36, 195, 13},
      {69, 3, 105, 207, 7},
      {188, 50, 76, 204, 7},
      {239, 57, 108, 10, 2}
    },
    { {184, 54, 252, 47, 7},
      {111, 185, 110, 54, 5},
      {239, 67, 246, 193, 13},
      {166, 199, 105, 223, 6}
    },
    { {237, 209, 46, 107, 4},
      {194, 103, 118, 206, 5},
      {45, 103, 72, 187, 7},
      {167, 54, 238, 100, 3}
    },
    { {235, 71, 59, 247, 10},
      {26, 86, 175, 255, 7},
      {94, 253, 206, 45, 7},
      {239, 255, 86, 165, 8}
    },
    { {20, 242, 111, 82, 9},
      {245, 77, 157, 8, 5},
      {148, 175, 100, 242, 8},
      {161, 11, 155, 42, 15}
    },
    { {205, 127, 26, 185, 15},
      {199, 118, 207, 246, 10},
      {249, 213, 143, 235, 3},
      {86, 255, 54, 238, 3}
    },
    { {198, 53, 226, 175, 3},
      {237, 242, 2, 167, 7},
      {207, 84, 122, 198, 3},
      {238, 84, 4, 251, 7}
    },
    { {215, 249, 12, 215, 3},
      {205, 31, 148, 207, 11},
      {206, 179, 9, 254, 11},
      {223, 50, 159, 139, 3}
    },
    { {83, 21, 215, 208, 12},
      {112, 203, 193, 249, 2},
      {48, 190, 186, 140, 10},
      {73, 248, 61, 48, 14}
    },
    { {86, 18, 116, 213, 1},
      {236, 9, 136, 157, 6},
      {138, 178, 228, 134, 10},
      {107, 145, 25, 3, 7}
    },
    { {181, 26, 186, 149, 2},
      {200, 216, 172, 86, 14},
      {74, 149, 213, 138, 13},
      {118, 163, 81, 177, 3}
    },
    { {98, 62, 7, 58, 4},
      {81, 97, 234, 161, 9},
      {37, 206, 7, 196, 6},
      {152, 85, 120, 104, 10}
    },
    { {235, 252, 157, 72, 14},
      {83, 181, 117, 251, 8},
      {113, 43, 147, 253, 7},
      {29, 250, 234, 220, 10}
    },
    { {129, 51, 86, 47, 14},
      {105, 115, 75, 86, 1},
      {127, 70, 172, 200, 1},
      {134, 173, 44, 233, 6}
    },
    { {125, 43, 146, 2, 6},
      {131, 218, 104, 208, 9},
      {100, 4, 157, 75, 14},
      {144, 177, 101, 188, 1}
    },
    { {144, 85, 192, 202, 6},
      {96, 190, 64, 42, 3},
      {101, 48, 58, 160, 9},
      {197, 64, 39, 208, 6}
    },
    { {205, 132, 181, 92, 15},
      {158, 177, 209, 250, 4},
      {243, 170, 210, 27, 3},
      {37, 248, 184, 215, 9}
    },
    { {11, 54, 92, 9, 4},
      {99, 33, 76, 117, 0},
      {41, 3, 166, 205, 0},
      {10, 227, 40, 76, 6}
    },
    { {190, 98, 7, 5, 10},
      {155, 93, 41, 7, 0},
      {90, 14, 4, 103, 13},
      {14, 9, 75, 173, 9}
    },
    { {39, 38, 95, 35, 12},
      {177, 65, 111, 117, 1},
      {60, 79, 166, 78, 4},
      {138, 239, 104, 40, 13}
    },
    { {100, 138, 166, 99, 10},
      {128, 209, 59, 140, 13},
      {92, 102, 85, 18, 6},
      {179, 29, 200, 176, 1}
    },
    { {254, 101, 69, 2, 13},
      {183, 15, 97, 163, 1},
      {180, 10, 42, 103, 15},
      {140, 88, 111, 14, 13}
    },
    { {167, 99, 68, 48, 13},
      {165, 7, 235, 67, 0},
      {176, 194, 44, 110, 5},
      {12, 45, 126, 10, 5}
    },
    { {217, 192, 126, 10, 8},
      {34, 109, 21, 210, 5},
      {21, 7, 224, 57, 11},
      {164, 186, 139, 100, 4}
    },
    { {29, 104, 212, 20, 13},
      {175, 141, 193, 80, 8},
      {178, 130, 177, 107, 8},
      {16, 168, 59, 31, 5}
    },
    { {203, 197, 40, 112, 3},
      {6, 22, 150, 235, 4},
      {192, 225, 74, 61, 3},
      {45, 118, 150, 134, 0}
    },
    { {228, 40, 28, 188, 9},
      {141, 33, 167, 146, 10},
      {147, 211, 129, 66, 7},
      {84, 158, 88, 75, 1}
    },
    { {225, 221, 170, 221, 6},
      {72, 246, 244, 238, 14},
      {107, 181, 91, 184, 7},
      {119, 114, 246, 241, 2}
    },
    { {244, 239, 90, 230, 13},
      {173, 78, 127, 186, 11},
      {182, 117, 175, 114, 15},
      {213, 223, 231, 43, 5}
    },
    { {122, 159, 45, 115, 15},
      {86, 27, 255, 173, 13},
      {252, 235, 79, 149, 14},
      {187, 95, 253, 134, 10}
    },
    { {189, 215, 111, 86, 6},
      {250, 95, 252, 106, 5},
      {102, 175, 110, 187, 13},
      {165, 99, 255, 165, 15}
    },
    { {79, 139, 34, 80, 4},
      {130, 66, 216, 201, 12},
      {32, 164, 77, 31, 2},
      {57, 49, 180, 36, 1}
    },
    { {104, 156, 225, 61, 6},
      {122, 176, 242, 164, 12},
      {107, 200, 115, 145, 6},
      {50, 84, 240, 213, 14}
    },
    { {23, 125, 226, 145, 13},
      {229, 206, 193, 101, 14},
      {184, 148, 123, 238, 8},
      {122, 104, 55, 58, 7}
    },
    { {101, 27, 125, 109, 2},
      {248, 51, 46, 220, 12},
      {75, 107, 237, 138, 6},
      {51, 183, 76, 193, 15}
    },
    { {230, 30, 244, 128, 14},
      {224, 145, 105, 179, 14},
      {112, 18, 247, 134, 7},
      {124, 217, 104, 144, 7}
    },
    { {10, 131, 197, 183, 14},
      {58, 147, 219, 5, 3},
      {126, 218, 60, 21, 0},
      {202, 13, 188, 149, 12}
    },
    { {113, 216, 247, 205, 5},
      {124, 237, 112, 220, 14},
      {171, 62, 241, 184, 14},
      {115, 176, 235, 115, 14}
    },
    { {119, 102, 40, 235, 6},
      {129, 60, 110, 237, 7},
      {109, 113, 70, 110, 14},
      {235, 119, 99, 200, 1}
    },
    { {164, 71, 35, 16, 2},
      {144, 86, 168, 34, 4},
      {64, 140, 78, 34, 5},
      {36, 65, 86, 160, 9}
    },
    { {141, 28, 30, 136, 7},
      {198, 113, 68, 114, 10},
      {225, 23, 131, 139, 1},
      {84, 226, 40, 230, 3}
    },
    { {174, 5, 207, 117, 0},
      {186, 195, 166, 47, 0},
      {10, 239, 58, 7, 5},
      {15, 70, 92, 53, 13}
    },
    { {242, 185, 238, 5, 10},
      {105, 219, 53, 135, 12},
      {90, 7, 121, 212, 15},
      {62, 26, 205, 185, 6}
    },
    { {150, 184, 220, 80, 14},
      {225, 153, 213, 27, 8},
      {112, 163, 177, 214, 9},
      {29, 138, 185, 152, 7}
    },
    { {191, 140, 179, 54, 4},
      {154, 200, 242, 115, 13},
      {38, 204, 211, 31, 13},
      {188, 228, 241, 53, 9}
    },
    { {77, 139, 178, 107, 7},
      {134, 242, 90, 220, 13},
      {237, 100, 221, 27, 2},
      {179, 181, 164, 246, 1}
    },
    { {48, 111, 43, 129, 9},
      {21, 78, 45, 36, 14},
      {152, 29, 79, 96, 12},
      {114, 75, 71, 42, 8}
    },
    { {134, 223, 39, 101, 6},
      {216, 87, 90, 47, 12},
      {106, 110, 79, 182, 1},
      {63, 69, 174, 161, 11}
    },
    { {201, 63, 168, 244, 10},
      {75, 146, 143, 234, 14},
      {82, 241, 95, 201, 3},
      {117, 127, 20, 157, 2}
    },
    { {178, 101, 205, 76, 0},
      {57, 175, 36, 43, 0},
      {3, 43, 58, 100, 13},
      {13, 66, 79, 89, 12}
    },
    { {230, 107, 232, 211, 7},
      {165, 150, 236, 143, 15},
      {236, 177, 125, 102, 7},
      {255, 19, 118, 154, 5}
    },
    { {172, 157, 252, 25, 5},
      {230, 163, 244, 54, 12},
      {169, 131, 251, 147, 5},
      {54, 194, 252, 86, 7}
    },
    { {177, 19, 196, 190, 12},
      {104, 171, 235, 66, 3},
      {55, 210, 60, 136, 13},
      {196, 45, 125, 81, 6}
    },
    { {211, 223, 9, 161, 13},
      {84, 14, 95, 231, 10},
      {184, 89, 15, 188, 11},
      {94, 127, 167, 2, 10}
    },
    { {238, 69, 143, 67, 7},
      {150, 215, 100, 175, 1},
      {236, 47, 26, 39, 7},
      {143, 82, 110, 182, 9}
    },
    { {103, 202, 12, 176, 9},
      {132, 5, 191, 193, 10},
      {144, 211, 5, 62, 6},
      {88, 63, 218, 2, 1}
    },
    { {179, 49, 95, 108, 5},
      {125, 107, 102, 91, 0},
      {163, 111, 168, 204, 13},
      {13, 166, 109, 107, 14}
    },
    { {51, 113, 11, 210, 2},
      {81, 94, 164, 73, 3},
      {68, 189, 8, 236, 12},
      {201, 34, 87, 168, 10}
    },
    { {155, 233, 139, 184, 6},
      {19, 254, 214, 67, 10},
      {97, 221, 25, 125, 9},
      {92, 38, 183, 252, 8}
    },
    { {65, 191, 62, 12, 5},
      {77, 99, 92, 240, 12},
      {163, 7, 207, 216, 2},
      {48, 243, 172, 107, 2}
    },
    { {39, 180, 57, 11, 8},
      {209, 32, 53, 117, 5},
      {29, 9, 194, 222, 4},
      {170, 234, 192, 72, 11}
    },
    { {22, 194, 235, 111, 10},
      {184, 252, 31, 13, 5},
      {95, 109, 116, 54, 8},
      {171, 15, 131, 241, 13}
    },
    { {180, 216, 190, 76, 8},
      {200, 237, 53, 26, 12},
      {19, 39, 209, 178, 13},
      {53, 138, 203, 113, 3}
    },
    { {157, 108, 51, 41, 10},
      {147, 124, 3, 118, 12},
      {89, 76, 195, 107, 9},
      {54, 236, 3, 236, 9}
    },
    { {62, 93, 56, 87, 15},
      {206, 30, 229, 61, 13},
      {254, 161, 203, 167, 12},
      {187, 202, 119, 135, 3}
    },
    { {180, 100, 140, 48, 10},
      {129, 157, 167, 34, 0},
      {80, 195, 18, 98, 13},
      {4, 78, 91, 152, 1}
    },
    { {93, 101, 141, 153, 11},
      {151, 191, 133, 228, 2},
      {217, 155, 26, 107, 10},
      {66, 122, 31, 222, 9}
    },
    { {75, 57, 30, 253, 8},
      {75, 99, 135, 221, 10},
      {27, 247, 137, 205, 2},
      {91, 190, 28, 109, 2}
    },
    { {159, 102, 246, 244, 7},
      {175, 221, 202, 123, 6},
      {226, 246, 246, 111, 9},
      {109, 229, 59, 191, 5}
    },
    { {99, 251, 122, 188, 8},
      {105, 102, 191, 209, 14},
      {19, 213, 237, 252, 6},
      {120, 191, 214, 105, 6}
    },
    { {48, 213, 149, 195, 13},
      {84, 143, 113, 60, 3},
      {188, 58, 154, 176, 12},
      {195, 200, 239, 18, 10}
    },
    { {44, 228, 218, 91, 8},
      {163, 228, 181, 60, 1},
      {29, 165, 178, 115, 4},
      {131, 202, 210, 124, 5}
    },
    { {123, 79, 47, 122, 4},
      {18, 111, 238, 233, 13},
      {37, 239, 79, 45, 14},
      {185, 119, 127, 100, 8}
    },
    { {80, 165, 170, 211, 11},
      {5, 218, 149, 172, 7},
      {220, 181, 90, 80, 10},
      {227, 90, 149, 186, 0}
    },
    { {53, 99, 219, 99, 9},
      {181, 206, 47, 92, 1},
      {156, 109, 188, 106, 12},
      {131, 175, 71, 58, 13}
    },
    { {202, 17, 193, 89, 1},
      {118, 162, 128, 143, 0},
      {137, 168, 56, 133, 3},
      {15, 16, 20, 86, 14}
    },
    { {119, 153, 15, 22, 8},
      {216, 75, 181, 193, 9},
      {22, 143, 9, 158, 14},
      {152, 58, 221, 33, 11}
    },
    { {6, 232, 6, 138, 11},
      {133, 117, 17, 1, 11},
      {213, 22, 1, 118, 0},
      {216, 8, 138, 234, 1}
    },
    { {88, 242, 76, 249, 1},
      {103, 45, 158, 140, 2},
      {137, 243, 36, 241, 10},
      {67, 23, 155, 78, 6}
    },
    { {87, 131, 120, 211, 4},
      {160, 10, 220, 221, 7},
      {44, 177, 236, 30, 10},
      {235, 179, 181, 0, 5}
    },
    { {102, 199, 182, 12, 6},
      {136, 247, 120, 177, 4},
      {99, 6, 222, 54, 6},
      {40, 209, 238, 241, 1}
    },
    { {120, 143, 69, 72, 0},
      {50, 43, 56, 168, 8},
      {1, 42, 47, 17, 14},
      {17, 81, 205, 68, 12}
    },
    { {233, 239, 208, 58, 13},
      {39, 166, 251, 242, 9},
      {181, 192, 191, 121, 7},
      {148, 253, 246, 94, 4}
    },
    { {39, 212, 15, 163, 10},
      {208, 85, 55, 101, 3},
      {92, 95, 2, 190, 4},
      {202, 110, 202, 160, 11}
    },
    { {125, 42, 249, 128, 10},
      {179, 152, 45, 208, 14},
      {80, 25, 245, 75, 14},
      {112, 187, 65, 156, 13}
    },
    { {86, 144, 82, 117, 6},
      {232, 88, 210, 157, 0},
      {106, 228, 160, 150, 10},
      {11, 148, 177, 161, 7}
    },
    { {30, 225, 231, 61, 2},
      {187, 255, 146, 5, 4},
      {75, 206, 120, 119, 8},
      {42, 4, 159, 253, 13}
    },
    { {254, 157, 166, 210, 9},
      {198, 203, 177, 171, 15},
      {148, 182, 91, 151, 15},
      {253, 88, 221, 54, 3}
    },
    { {75, 222, 118, 252, 11},
      {110, 117, 155, 249, 14},
      {211, 246, 231, 189, 2},
      {121, 253, 154, 231, 6}
    },
    { {50, 21, 75, 223, 8},
      {120, 106, 165, 45, 3},
      {31, 189, 42, 132, 12},
      {203, 74, 85, 97, 14}
    },
    { {141, 94, 72, 11, 6},
      {226, 52, 76, 102, 9},
      {109, 1, 39, 171, 1},
      {150, 99, 34, 196, 7}
    },
    { {85, 153, 228, 23, 12},
      {232, 139, 209, 196, 13},
      {62, 130, 121, 154, 10},
      {178, 56, 189, 17, 7}
    },
    { {127, 178, 100, 131, 2},
      {227, 25, 56, 197, 7},
      {76, 18, 100, 223, 14},
      {234, 49, 201, 140, 7}
    },
    { {6, 201, 57, 111, 3},
      {156, 54, 22, 29, 13},
      {207, 105, 201, 54, 0},
      {187, 134, 134, 195, 9}
    },
    { {66, 202, 124, 45, 2},
      {40, 53, 30, 149, 12},
      {75, 67, 229, 52, 2},
      {58, 151, 138, 193, 4}
    },
    { {78, 192, 248, 186, 12},
      {162, 164, 215, 145, 7},
      {53, 209, 240, 55, 2},
      {232, 158, 178, 84, 5}
    },
    { {45, 113, 131, 30, 11},
      {223, 246, 161, 64, 1},
      {215, 140, 24, 235, 4},
      {128, 40, 86, 255, 11}
    },
    { {128, 130, 234, 70, 4},
      {40, 192, 92, 10, 5},
      {38, 37, 116, 16, 1},
      {165, 3, 160, 49, 4}
    },
    { {172, 206, 58, 52, 3},
      {142, 84, 190, 50, 12},
      {194, 197, 199, 51, 5},
      {52, 199, 210, 167, 1}
    },
    { {88, 160, 32, 181, 13},
      {15, 8, 211, 132, 6},
      {186, 208, 64, 81, 10},
      {98, 28, 177, 15, 0}
    },
    { {114, 159, 129, 35, 8},
      {80, 138, 59, 165, 9},
      {28, 72, 31, 148, 14},
      {154, 93, 197, 16, 10}
    },
    { {123, 5, 90, 20, 12},
      {42, 74, 229, 241, 0},
      {50, 133, 170, 13, 14},
      {8, 250, 117, 37, 4}
    },
    { {150, 248, 124, 233, 1},
      {229, 45, 22, 31, 14},
      {137, 115, 225, 246, 9},
      {127, 134, 139, 74, 7}
    },
    { {41, 99, 28, 75, 10},
      {3, 55, 45, 92, 1},
      {93, 35, 140, 105, 4},
      {131, 171, 78, 204, 0}
    },
    { {157, 226, 24, 237, 14},
      {139, 60, 95, 94, 2},
      {123, 113, 132, 123, 9},
      {71, 175, 163, 205, 1}
    },
    { {58, 230, 175, 66, 4},
      {19, 205, 124, 41, 5},
      {36, 47, 86, 117, 12},
      {169, 67, 235, 60, 8}
    },
    { {180, 56, 174, 163, 3},
      {197, 217, 38, 6, 15},
      {204, 87, 81, 194, 13},
      {246, 6, 73, 186, 3}
    },
    { {208, 70, 61, 144, 7},
      {20, 29, 204, 178, 6},
      {224, 155, 198, 32, 11},
      {100, 211, 59, 130, 8}
    },
    { {187, 32, 151, 39, 0},
      {27, 201, 34, 87, 1},
      {14, 78, 144, 77, 13},
      {142, 164, 73, 61, 8}
    },
    { {229, 143, 37, 62, 0},
      {152, 35, 186, 226, 13},
      {7, 202, 79, 26, 7},
      {180, 117, 220, 65, 9}
    },
    { {194, 117, 158, 53, 4},
      {73, 199, 198, 183, 0},
      {42, 199, 154, 228, 3},
      {14, 214, 62, 57, 2}
    },
    { {10, 66, 188, 132, 7},
      {14, 149, 76, 17, 6},
      {226, 19, 212, 37, 0},
      {104, 131, 42, 151, 0}
    },
    { {173, 67, 236, 130, 11},
      {166, 151, 45, 66, 7},
      {212, 19, 124, 43, 5},
      {228, 43, 78, 150, 5}
    },
    { {61, 136, 115, 251, 11},
      {182, 120, 179, 92, 15},
      {221, 252, 225, 27, 12},
      {243, 172, 209, 230, 13}
    },
    { {102, 161, 166, 184, 11},
      {133, 243, 179, 129, 6},
      {209, 214, 88, 86, 6},
      {104, 28, 220, 250, 1}
    },
    { {221, 141, 175, 200, 4},
      {146, 235, 84, 234, 14},
      {33, 63, 91, 27, 11},
      {117, 114, 173, 116, 9}
    },
    { {21, 7, 218, 203, 3},
      {164, 250, 12, 124, 3},
      {205, 53, 190, 10, 8},
      {195, 227, 5, 242, 5}
    },
    { {137, 254, 190, 99, 3},
      {71, 213, 30, 126, 13},
      {204, 103, 215, 249, 1},
      {183, 231, 138, 190, 2}
    },
    { {25, 7, 189, 144, 10},
      {18, 155, 141, 112, 6},
      {80, 155, 222, 9, 8},
      {96, 235, 29, 148, 8}
    },
    { {98, 241, 188, 178, 14},
      {65, 151, 247, 145, 7},
      {116, 211, 216, 244, 6},
      {232, 158, 254, 152, 2}
    },
    { {14, 147, 144, 200, 10},
      {194, 178, 25, 25, 2},
      {81, 48, 156, 151, 0},
      {73, 137, 132, 212, 3}
    },
    { {102, 251, 212, 124, 10},
      {233, 183, 187, 153, 8},
      {83, 226, 189, 246, 6},
      {25, 157, 222, 217, 7}
    },
    { {203, 127, 203, 186, 5},
      {119, 230, 206, 227, 11},
      {165, 221, 63, 237, 3},
      {220, 119, 54, 126, 14}
    },
    { {72, 77, 150, 114, 5},
      {6, 199, 194, 184, 9},
      {164, 230, 155, 33, 2},
      {145, 212, 62, 54, 0}
    },
    { {182, 201, 156, 184, 13},
      {132, 175, 247, 19, 10},
      {177, 211, 153, 54, 13},
      {92, 142, 255, 82, 1}
    },
    { {214, 73, 19, 109, 6},
      {152, 126, 66, 159, 8},
      {107, 108, 137, 38, 11},
      {31, 148, 39, 225, 9}
    },
    { {13, 229, 2, 166, 2},
      {139, 86, 18, 96, 3},
      {70, 84, 10, 123, 0},
      {192, 100, 134, 173, 1}
    },
    { {249, 154, 61, 119, 4},
      {90, 9, 254, 222, 13},
      {46, 235, 197, 153, 15},
      {183, 183, 249, 5, 10}
    },
    { {124, 20, 82, 167, 4},
      {234, 72, 98, 180, 3},
      {46, 84, 162, 131, 14},
      {194, 212, 97, 37, 7}
    },
    { {57, 81, 76, 89, 15},
      {102, 63, 229, 76, 0},
      {249, 163, 40, 169, 12},
      {3, 42, 127, 198, 6}
    },
    { {202, 63, 57, 87, 14},
      {91, 18, 205, 191, 13},
      {126, 169, 207, 197, 3},
      {191, 219, 52, 141, 10}
    },
    { {219, 212, 95, 243, 10},
      {114, 93, 151, 255, 3},
      {92, 255, 162, 189, 11},
      {207, 254, 155, 164, 14}
    },
    { {113, 246, 139, 38, 2},
      {89, 220, 62, 224, 1},
      {70, 77, 22, 248, 14},
      {128, 119, 195, 185, 10}
    },
    { {61, 34, 41, 13, 7},
      {159, 56, 108, 68, 4},
      {235, 9, 68, 75, 12},
      {34, 35, 97, 207, 9}
    },
    { {154, 77, 130, 207, 12},
      {10, 238, 65, 47, 11},
      {63, 52, 27, 37, 9},
      {223, 72, 39, 117, 0}
    },
    { {16, 141, 72, 139, 2},
      {32, 58, 20, 36, 11},
      {77, 17, 43, 16, 8},
      {210, 66, 133, 192, 4}
    },
    { {66, 66, 157, 48, 14},
      {16, 149, 207, 145, 0},
      {112, 203, 148, 36, 2},
      {8, 159, 58, 144, 8}
    },
    { {191, 242, 183, 46, 9},
      {223, 237, 59, 83, 5},
      {151, 78, 212, 255, 13},
      {172, 173, 203, 127, 11}
    },
    { {137, 69, 178, 16, 8},
      {2, 198, 129, 114, 4},
      {16, 132, 218, 41, 1},
      {36, 232, 22, 52, 0}
    },
    { {156, 83, 206, 106, 1},
      {230, 239, 14, 10, 1},
      {133, 103, 60, 163, 9},
      {133, 7, 15, 118, 7}
    },
    { {246, 110, 46, 178, 11},
      {133, 93, 175, 163, 15},
      {212, 215, 71, 102, 15},
      {252, 95, 91, 170, 1}
    },
    { {123, 135, 22, 82, 6},
      {2, 91, 248, 249, 1},
      {100, 166, 142, 29, 14},
      {137, 241, 253, 164, 0}
    },
    { {117, 48, 198, 172, 7},
      {237, 249, 98, 192, 2},
      {227, 86, 48, 202, 14},
      {64, 52, 105, 251, 7}
    },
    { {219, 148, 76, 224, 4},
      {98, 9, 86, 235, 2},
      {32, 115, 34, 157, 11},
      {77, 118, 169, 4, 6}
    },
    { {64, 156, 92, 188, 6},
      {104, 49, 214, 176, 10},
      {99, 211, 163, 144, 2},
      {80, 214, 184, 193, 6}
    },
    { {96, 119, 25, 168, 12},
      {81, 38, 111, 176, 2},
      {49, 89, 142, 224, 6},
      {64, 223, 102, 72, 10}
    },
    { {100, 136, 107, 16, 1},
      {180, 64, 180, 128, 12},
      {128, 141, 97, 18, 6},
      {48, 18, 208, 34, 13}
    },
    { {33, 158, 169, 148, 0},
      {88, 128, 188, 96, 14},
      {2, 153, 87, 152, 4},
      {112, 99, 208, 17, 10}
    },
    { {146, 100, 250, 242, 11},
      {37, 220, 135, 59, 7},
      {212, 245, 242, 100, 9},
      {237, 206, 19, 186, 4}
    },
    { {214, 14, 81, 55, 9},
      {188, 8, 139, 183, 9},
      {158, 200, 167, 6, 11},
      {158, 221, 17, 3, 13}
    },
    { {6, 232, 25, 146, 12},
      {145, 4, 213, 17, 11},
      {52, 153, 129, 118, 0},
      {216, 138, 178, 8, 9}
    },
    { {101, 196, 213, 201, 3},
      {180, 181, 48, 252, 2},
      {201, 58, 178, 58, 6},
      {67, 240, 202, 210, 13}
    },
    { {91, 1, 85, 227, 4},
      {50, 11, 66, 221, 3},
      {44, 122, 168, 13, 10},
      {203, 180, 45, 4, 12}
    },
    { {93, 64, 143, 102, 1},
      {158, 205, 6, 200, 1},
      {134, 111, 16, 43, 10},
      {129, 54, 11, 55, 9}
    },
    { {61, 239, 243, 168, 13},
      {183, 238, 123, 112, 14},
      {177, 92, 255, 123, 12},
      {112, 237, 231, 126, 13}
    },
    { {83, 37, 200, 86, 6},
      {41, 154, 196, 233, 1},
      {102, 161, 58, 76, 10},
      {137, 114, 53, 153, 4}
    },
    { {18, 10, 174, 82, 10},
      {0, 217, 141, 9, 13},
      {84, 167, 85, 4, 8},
      {185, 11, 25, 176, 0}
    },
    { {155, 29, 191, 200, 11},
      {86, 251, 5, 123, 14},
      {209, 63, 219, 141, 9},
      {125, 234, 13, 246, 10}
    },
    { {207, 183, 171, 86, 8},
      {219, 194, 157, 235, 5},
      {22, 173, 94, 223, 3},
      {173, 123, 148, 61, 11}
    },
    { {42, 141, 220, 32, 7},
      {38, 147, 118, 49, 8},
      {224, 67, 187, 21, 4},
      {24, 198, 236, 150, 4}
    },
    { {142, 106, 174, 19, 3},
      {135, 213, 140, 7, 13},
      {204, 135, 85, 103, 1},
      {190, 3, 26, 190, 1}
    },
    { {171, 253, 77, 159, 3},
      {127, 55, 180, 103, 11},
      {207, 155, 43, 253, 5},
      {222, 98, 222, 207, 14}
    },
    { {45, 9, 17, 6, 12},
      {154, 2, 97, 80, 9},
      {54, 8, 137, 11, 4},
      {144, 168, 100, 5, 9}
    },
    { {168, 207, 127, 12, 12},
      {58, 103, 125, 50, 12},
      {51, 15, 239, 49, 5},
      {52, 203, 238, 101, 12}
    },
    { {166, 83, 153, 76, 7},
      {220, 182, 108, 27, 0},
      {227, 41, 156, 166, 5},
      {13, 131, 102, 211, 11}
    },
    { {135, 166, 224, 109, 0},
      {169, 160, 26, 111, 4},
      {11, 96, 118, 94, 1},
      {47, 101, 128, 89, 5}
    },
    { {59, 232, 221, 151, 0},
      {59, 141, 180, 85, 11},
      {14, 155, 177, 125, 12},
      {218, 162, 219, 29, 12}
    },
    { {209, 178, 226, 250, 9},
      {101, 232, 155, 202, 7},
      {149, 244, 116, 216, 11},
      {229, 61, 145, 122, 6}
    },
    { {0, 188, 184, 78, 7},
      {77, 176, 84, 56, 13},
      {231, 33, 211, 208, 0},
      {177, 194, 160, 219, 2}
    },
    { {166, 167, 158, 98, 11},
      {133, 211, 63, 59, 1},
      {212, 103, 158, 86, 5},
      {141, 207, 204, 186, 1}
    },
    { {200, 134, 91, 1, 3},
      {54, 80, 28, 182, 0},
      {200, 13, 166, 17, 3},
      {6, 211, 128, 166, 12}
    },
    { {12, 255, 119, 177, 9},
      {247, 71, 155, 52, 14},
      {152, 222, 239, 243, 0},
      {114, 205, 158, 46, 15}
    },
    { {142, 29, 220, 148, 3},
      {238, 147, 132, 51, 10},
      {194, 147, 187, 135, 1},
      {92, 194, 28, 151, 7}
    },
    { {18, 244, 200, 158, 7},
      {109, 188, 212, 33, 3},
      {231, 145, 50, 244, 8},
      {200, 66, 179, 219, 6}
    },
    { {221, 245, 180, 168, 13},
      {199, 175, 83, 242, 6},
      {177, 82, 218, 251, 11},
      {100, 252, 175, 94, 3}
    },
    { {97, 194, 86, 250, 10},
      {32, 117, 187, 216, 3},
      {85, 246, 164, 56, 6},
      {193, 189, 218, 224, 4}
    },
    { {34, 2, 81, 216, 5},
      {52, 32, 232, 25, 2},
      {161, 184, 164, 4, 4},
      {73, 129, 112, 66, 12}
    },
    { {160, 222, 24, 47, 2},
      {72, 52, 62, 54, 9},
      {79, 65, 135, 176, 5},
      {150, 199, 194, 193, 2}
    },
    { {203, 116, 62, 141, 0},
      {75, 101, 4, 247, 6},
      {11, 23, 194, 237, 3},
      {110, 242, 10, 109, 2}
    },
    { {32, 93, 174, 186, 9},
      {68, 231, 167, 32, 15},
      {149, 215, 91, 160, 4},
      {240, 78, 94, 114, 2}
    },
    { {182, 151, 72, 253, 13},
      {236, 42, 255, 47, 2},
      {187, 241, 46, 150, 13},
      {79, 79, 245, 67, 7}
    },
    { {244, 165, 139, 213, 6},
      {153, 218, 244, 174, 2},
      {106, 189, 26, 82, 15},
      {71, 82, 245, 185, 9}
    },
    { {137, 147, 208, 12, 15},
      {110, 178, 89, 82, 0},
      {243, 0, 188, 153, 1},
      {4, 169, 164, 215, 6}
    },
    { {43, 161, 206, 80, 15},
      {39, 211, 245, 73, 0},
      {240, 167, 56, 93, 4},
      {9, 42, 252, 190, 4}
    },
    { {84, 31, 38, 148, 10},
      {200, 91, 137, 160, 14},
      {82, 150, 79, 130, 10},
      {112, 89, 29, 161, 3}
    },
    { {250, 73, 70, 153, 5},
      {38, 111, 224, 135, 10},
      {169, 150, 41, 37, 15},
      {94, 16, 127, 102, 4}
    },
    { {143, 119, 77, 200, 6},
      {243, 55, 76, 107, 2},
      {97, 59, 46, 239, 1},
      {77, 99, 46, 204, 15}
    },
    { {21, 73, 52, 191, 10},
      {136, 63, 131, 84, 15},
      {95, 210, 201, 42, 8},
      {242, 172, 31, 193, 1}
    },
    { {160, 179, 125, 121, 0},
      {113, 35, 190, 30, 4},
      {9, 235, 236, 208, 5},
      {39, 135, 220, 72, 14}
    },
    { {94, 29, 64, 6, 12},
      {234, 10, 65, 161, 9},
      {54, 0, 43, 135, 10},
      {152, 88, 37, 5, 7}
    },
    { {247, 121, 227, 64, 9},
      {245, 206, 33, 203, 12},
      {144, 44, 121, 238, 15},
      {61, 56, 71, 58, 15}
    },
    { {126, 60, 103, 41, 12},
      {243, 105, 99, 165, 12},
      {57, 78, 99, 199, 14},
      {58, 92, 105, 108, 15}
    },
    { {38, 10, 20, 204, 8},
      {136, 33, 41, 25, 10},
      {19, 50, 133, 6, 4},
      {89, 137, 72, 65, 1}
    },
    { {210, 180, 127, 62, 15},
      {125, 121, 215, 179, 5},
      {247, 207, 226, 212, 11},
      {172, 222, 185, 235, 14}
    },
    { {52, 74, 22, 15, 2},
      {136, 125, 40, 20, 9},
      {79, 6, 133, 34, 12},
      {146, 129, 75, 225, 1}
    },
    { {240, 88, 87, 246, 9},
      {124, 77, 163, 154, 11},
      {150, 254, 161, 160, 15},
      {213, 156, 91, 35, 14}
    },
    { {200, 222, 73, 90, 3},
      {118, 52, 156, 170, 9},
      {197, 169, 39, 177, 3},
      {149, 83, 146, 198, 14}
    },
    { {41, 152, 52, 89, 7},
      {70, 49, 240, 92, 12},
      {233, 162, 193, 153, 4},
      {51, 160, 248, 198, 2}
    },
    { {192, 28, 220, 163, 10},
      {96, 145, 7, 182, 11},
      {92, 83, 179, 128, 3},
      {214, 222, 8, 144, 6}
    },
    { {17, 134, 214, 198, 15},
      {44, 217, 89, 120, 3},
      {246, 54, 182, 24, 8},
      {193, 233, 169, 179, 4}
    },
    { {199, 106, 99, 224, 0},
      {177, 68, 10, 203, 14},
      {0, 124, 101, 110, 3},
      {125, 53, 2, 40, 13}
    },
    { {250, 63, 143, 86, 4},
      {91, 203, 236, 171, 9},
      {38, 175, 31, 197, 15},
      {157, 83, 125, 61, 10}
    },
    { {30, 208, 47, 67, 7},
      {214, 93, 84, 13, 5},
      {236, 47, 64, 183, 8},
      {171, 2, 171, 166, 11}
    },
    { {167, 132, 152, 85, 15},
      {140, 144, 245, 127, 0},
      {250, 161, 146, 30, 5},
      {15, 234, 240, 147, 1}
    },
    { {12, 209, 94, 200, 6},
      {226, 119, 84, 24, 2},
      {97, 55, 168, 179, 0},
      {65, 130, 174, 228, 7}
    },
    { {222, 242, 230, 20, 0},
      {235, 205, 152, 131, 4},
      {2, 134, 116, 247, 11},
      {44, 17, 155, 61, 7}
    },
    { {172, 114, 24, 123, 12},
      {195, 36, 239, 30, 1},
      {61, 225, 132, 227, 5},
      {135, 143, 114, 76, 3}
    },
    { {103, 201, 15, 119, 7},
      {156, 87, 246, 205, 9},
      {238, 239, 9, 62, 6},
      {155, 54, 254, 163, 9}
    },
    { {7, 91, 205, 34, 11},
      {244, 151, 15, 65, 9},
      {212, 75, 61, 174, 0},
      {152, 47, 14, 146, 15}
    },
    { {201, 190, 213, 237, 13},
      {127, 161, 91, 254, 10},
      {187, 122, 183, 217, 3},
      {87, 253, 168, 95, 14}
    },
    { {106, 147, 24, 117, 4},
      {74, 2, 254, 157, 0},
      {42, 225, 140, 149, 6},
      {11, 151, 244, 5, 2}
    },
    { {107, 212, 146, 123, 3},
      {70, 244, 178, 253, 1},
      {205, 228, 146, 189, 6},
      {139, 244, 210, 246, 2}
    },
    { {118, 132, 7, 115, 8},
      {144, 73, 179, 173, 1},
      {28, 238, 2, 22, 14},
      {139, 92, 217, 32, 9}
    },
    { {181, 12, 26, 120, 15},
      {132, 120, 231, 122, 8},
      {241, 229, 131, 10, 13},
      {21, 238, 113, 226, 1}
    },
    { {153, 76, 54, 246, 14},
      {10, 93, 195, 122, 15},
      {118, 246, 195, 41, 9},
      {245, 236, 59, 165, 0}
    },
    { {33, 245, 36, 87, 11},
      {77, 23, 177, 108, 5},
      {222, 162, 74, 248, 4},
      {163, 104, 222, 139, 2}
    },
    { {97, 170, 231, 238, 4},
      {57, 225, 122, 200, 15},
      {39, 126, 117, 88, 6},
      {241, 53, 232, 121, 12}
    },
    { {36, 1, 140, 232, 5},
      {132, 163, 102, 8, 2},
      {161, 115, 24, 2, 4},
      {65, 6, 108, 82, 1}
    },
    { {97, 41, 206, 35, 10},
      {33, 211, 39, 196, 9},
      {92, 71, 57, 72, 6},
      {146, 62, 76, 184, 4}
    },
    { {164, 226, 123, 199, 10},
      {185, 84, 61, 30, 7},
      {94, 61, 228, 114, 5},
      {231, 139, 194, 169, 13}
    },
    { {182, 249, 62, 111, 6},
      {201, 127, 118, 31, 13},
      {111, 103, 201, 246, 13},
      {191, 134, 239, 233, 3}
    },
    { {184, 158, 66, 250, 4},
      {98, 104, 250, 42, 11},
      {37, 244, 39, 145, 13},
      {213, 69, 241, 100, 6}
    },
    { {205, 168, 178, 163, 12},
      {131, 192, 83, 214, 15},
      {60, 84, 209, 91, 3},
      {246, 188, 160, 60, 1}
    },
    { {76, 57, 177, 159, 7},
      {223, 178, 192, 148, 15},
      {239, 152, 217, 195, 2},
      {242, 144, 52, 223, 11}
    },
    { {185, 129, 38, 244, 2},
      {10, 91, 178, 74, 6},
      {66, 246, 72, 25, 13},
      {101, 36, 221, 165, 0}
    },
    { {14, 11, 250, 182, 8},
      {170, 194, 143, 17, 15},
      {22, 213, 253, 7, 0},
      {248, 143, 20, 53, 5}
    },
    { {242, 177, 157, 60, 6},
      {89, 187, 246, 147, 0},
      {99, 203, 152, 212, 15},
      {12, 150, 253, 217, 10}
    },
    { {62, 75, 92, 90, 2},
      {162, 63, 172, 25, 9},
      {69, 163, 173, 39, 12},
      {153, 131, 95, 196, 5}
    },
    { {53, 13, 105, 45, 8},
      {184, 42, 39, 100, 12},
      {27, 73, 107, 10, 12},
      {50, 110, 69, 65, 13}
    },
    { {180, 35, 142, 37, 5},
      {141, 203, 110, 6, 0},
      {170, 71, 28, 66, 13},
      {6, 7, 109, 59, 1}
    },
    { {69, 98, 201, 154, 0},
      {177, 164, 140, 192, 3},
      {5, 153, 52, 106, 2},
      {192, 51, 18, 88, 13}
    },
    { {180, 56, 177, 59, 0},
      {209, 168, 162, 22, 13},
      {13, 200, 209, 194, 13},
      {182, 132, 81, 88, 11}
    },
    { {131, 20, 145, 146, 14},
      {80, 144, 193, 115, 3},
      {116, 152, 146, 140, 1},
      {204, 232, 48, 144, 10}
    },
    { {255, 9, 118, 5, 7},
      {174, 91, 96, 215, 12},
      {234, 6, 233, 15, 15},
      {62, 176, 109, 167, 5}
    },
    { {76, 85, 220, 22, 13},
      {238, 135, 197, 176, 1},
      {182, 131, 186, 163, 2},
      {128, 218, 62, 23, 7}
    },
    { {104, 114, 128, 201, 15},
      {71, 180, 105, 140, 2},
      {249, 48, 20, 225, 6},
      {67, 25, 98, 222, 2}
    },
    { {118, 171, 174, 160, 4},
      {129, 203, 126, 129, 14},
      {32, 87, 93, 86, 14},
      {120, 23, 237, 56, 1}
    },
    { {145, 170, 76, 41, 14},
      {33, 57, 95, 70, 8},
      {121, 67, 37, 88, 9},
      {22, 47, 169, 200, 4}
    },
    { {172, 153, 124, 202, 8},
      {226, 35, 53, 26, 15},
      {21, 51, 233, 147, 5},
      {245, 138, 204, 68, 7}
    },
    { {14, 211, 209, 214, 1},
      {254, 134, 152, 25, 3},
      {134, 184, 188, 183, 0},
      {201, 129, 150, 23, 15}
    },
    { {186, 245, 248, 135, 3},
      {111, 158, 52, 55, 7},
      {206, 17, 250, 245, 13},
      {238, 194, 199, 159, 6}
    },
    { {24, 203, 208, 54, 12},
      {42, 142, 219, 16, 9},
      {54, 192, 189, 49, 8},
      {144, 141, 183, 21, 4}
    },
    { {230, 136, 49, 122, 7},
      {148, 48, 242, 155, 13},
      {229, 232, 193, 22, 7},
      {189, 148, 240, 194, 9}
    },
    { {38, 93, 140, 24, 7},
      {196, 183, 228, 33, 8},
      {225, 131, 27, 166, 4},
      {24, 66, 126, 210, 3}
    },
    { {37, 253, 177, 16, 7},
      {213, 150, 240, 112, 12},
      {224, 136, 219, 250, 4},
      {48, 224, 246, 154, 11}
    },
    { {46, 105, 226, 33, 9},
      {167, 198, 35, 5, 12},
      {152, 68, 121, 103, 4},
      {58, 12, 70, 62, 5}
    },
    { {148, 212, 15, 127, 5},
      {220, 109, 214, 46, 1},
      {175, 239, 2, 178, 9},
      {135, 70, 187, 99, 11}
    },
    { {104, 227, 24, 161, 2},
      {3, 22, 62, 148, 2},
      {72, 81, 140, 113, 6},
      {66, 151, 198, 140, 0}
    },
    { {115, 38, 168, 12, 2},
      {9, 184, 44, 225, 4},
      {67, 1, 86, 76, 14},
      {40, 115, 65, 217, 0}
    },
    { {8, 225, 19, 56, 4},
      {19, 102, 210, 16, 0},
      {33, 204, 136, 113, 0},
      {0, 132, 182, 108, 8}
    },
    { {30, 74, 153, 58, 11},
      {150, 188, 143, 17, 9},
      {213, 201, 149, 39, 8},
      {152, 143, 19, 214, 9}
    },
    { {56, 123, 230, 193, 11},
      {103, 223, 41, 12, 14},
      {216, 54, 125, 225, 12},
      {115, 9, 79, 190, 6}
    },
    { {23, 164, 136, 165, 2},
      {137, 152, 22, 101, 2},
      {74, 81, 18, 94, 8},
      {74, 102, 129, 153, 1}
    },
    { {100, 67, 135, 200, 8},
      {144, 231, 41, 136, 2},
      {17, 62, 28, 34, 6},
      {65, 25, 78, 112, 9}
    },
    { {110, 7, 130, 143, 7},
      {142, 242, 104, 165, 3},
      {239, 20, 30, 7, 6},
      {202, 81, 100, 247, 1}
    },
    { {184, 128, 233, 2, 2},
      {50, 152, 52, 2, 5},
      {68, 9, 112, 17, 13},
      {164, 2, 193, 148, 12}
    },
    { {62, 14, 185, 83, 2},
      {146, 152, 172, 61, 13},
      {76, 169, 215, 7, 12},
      {187, 195, 81, 148, 9}
    },
    { {143, 140, 17, 176, 2},
      {146, 16, 146, 115, 10},
      {64, 216, 131, 31, 1},
      {92, 228, 144, 132, 9}
    },
    { {27, 141, 195, 155, 13},
      {54, 234, 209, 101, 11},
      {189, 156, 59, 29, 8},
      {218, 104, 181, 118, 12}
    },
    { {238, 130, 108, 30, 9},
      {174, 33, 189, 131, 5},
      {151, 131, 100, 23, 7},
      {172, 27, 216, 71, 5}
    },
    { {149, 96, 110, 240, 8},
      {161, 77, 135, 74, 6},
      {16, 247, 96, 106, 9},
      {101, 46, 27, 40, 5}
    },
    { {80, 228, 38, 86, 2},
      {9, 93, 144, 168, 5},
      {70, 166, 66, 112, 10},
      {161, 80, 155, 169, 0}
    },
    { {212, 77, 143, 180, 2},
      {152, 223, 134, 162, 10},
      {66, 223, 27, 34, 11},
      {84, 86, 31, 177, 9}
    },
    { {5, 79, 54, 9, 7},
      {132, 119, 72, 116, 12},
      {233, 6, 207, 42, 0},
      {50, 225, 46, 226, 1}
    },
    { {165, 246, 91, 233, 9},
      {245, 100, 63, 126, 2},
      {153, 125, 166, 250, 5},
      {71, 239, 194, 106, 15}
    },
    { {246, 67, 68, 234, 5},
      {164, 47, 106, 139, 3},
      {165, 114, 44, 38, 15},
      {205, 21, 111, 66, 5}
    },
    { {85, 81, 175, 40, 13},
      {212, 239, 71, 192, 4},
      {177, 79, 88, 170, 10},
      {32, 62, 47, 114, 11}
    },
    { {27, 94, 104, 63, 5},
      {110, 44, 206, 101, 13},
      {175, 193, 103, 173, 8},
      {186, 103, 51, 71, 6}
    },
    { {6, 118, 167, 222, 12},
      {217, 229, 201, 41, 7},
      {55, 190, 86, 230, 0},
      {233, 73, 58, 121, 11}
    },
    { {70, 176, 31, 231, 11},
      {221, 81, 23, 157, 3},
      {222, 127, 128, 214, 2},
      {203, 158, 136, 171, 11}
    },
    { {38, 80, 223, 195, 0},
      {240, 197, 36, 29, 3},
      {12, 63, 176, 166, 4},
      {203, 130, 74, 48, 15}
    },
    { {52, 1, 83, 23, 3},
      {188, 90, 160, 20, 1},
      {206, 140, 168, 2, 12},
      {130, 128, 85, 163, 13}
    },
    { {185, 183, 76, 10, 12},
      {99, 43, 125, 98, 1},
      {53, 3, 46, 217, 13},
      {132, 107, 237, 76, 6}
    },
    { {103, 124, 93, 21, 5},
      {253, 5, 228, 245, 8},
      {170, 139, 163, 238, 6},
      {26, 242, 122, 11, 15}
    },
    { {62, 14, 116, 210, 9},
      {166, 9, 169, 57, 15},
      {148, 178, 231, 7, 12},
      {249, 201, 89, 6, 5}
    },
    { {135, 91, 221, 123, 4},
      {240, 167, 206, 95, 9},
      {45, 235, 189, 174, 1},
      {159, 167, 62, 80, 15}
    },
    { {243, 250, 71, 202, 15},
      {117, 125, 121, 203, 11},
      {245, 62, 37, 252, 15},
      {221, 57, 235, 234, 14}
    },
    { {215, 194, 170, 201, 7},
      {132, 252, 92, 207, 6},
      {233, 53, 84, 62, 11},
      {111, 51, 163, 242, 1}
    },
    { {9, 112, 244, 98, 8},
      {99, 133, 3, 88, 5},
      {20, 98, 240, 233, 0},
      {161, 172, 10, 28, 6}
    },
    { {49, 139, 231, 50, 10},
      {48, 219, 187, 64, 13},
      {84, 206, 125, 24, 12},
      {176, 45, 221, 176, 12}
    },
    { {250, 59, 245, 187, 4},
      {115, 171, 234, 151, 15},
      {45, 218, 253, 197, 15},
      {254, 149, 125, 92, 14}
    },
    { {119, 186, 106, 0, 14},
      {225, 88, 125, 193, 12},
      {112, 5, 101, 222, 14},
      {56, 59, 225, 168, 7}
    },
    { {245, 7, 16, 43, 15},
      {132, 58, 107, 246, 1},
      {253, 64, 142, 10, 15},
      {134, 253, 101, 194, 1}
    },
    { {55, 185, 16, 190, 8},
      {201, 42, 179, 81, 11},
      {23, 208, 137, 222, 12},
      {216, 172, 213, 73, 3}
    },
    { {154, 36, 87, 204, 1},
      {63, 105, 0, 59, 2},
      {131, 62, 162, 69, 9},
      {77, 192, 9, 111, 12}
    },
    { {86, 238, 203, 19, 12},
      {177, 204, 221, 165, 9},
      {60, 141, 55, 118, 10},
      {154, 91, 179, 56, 13}
    },
    { {19, 93, 119, 167, 4},
      {120, 79, 66, 117, 15},
      {46, 94, 235, 172, 8},
      {250, 228, 47, 33, 14}
    },
    { {13, 146, 253, 48, 4},
      {242, 129, 222, 80, 4},
      {32, 203, 244, 155, 0},
      {32, 167, 184, 20, 15}
    },
    { {237, 122, 230, 34, 3},
      {231, 213, 42, 194, 13},
      {196, 70, 117, 235, 7},
      {180, 53, 74, 190, 7}
    },
    { {141, 125, 109, 37, 12},
      {251, 7, 71, 102, 12},
      {58, 75, 107, 235, 1},
      {54, 110, 46, 13, 15}
    },
    { {120, 62, 207, 103, 3},
      {127, 217, 46, 172, 9},
      {206, 111, 55, 193, 14},
      {147, 87, 73, 191, 14}
    },
    { {62, 116, 93, 252, 6},
      {251, 61, 230, 57, 2},
      {99, 251, 162, 231, 12},
      {73, 198, 123, 205, 15}
    },
    { {188, 170, 218, 5, 10},
      {171, 216, 61, 22, 8},
      {90, 5, 181, 83, 13},
      {22, 139, 193, 189, 5}
    },
    { {13, 162, 38, 20, 8},
      {139, 65, 153, 64, 4},
      {18, 134, 68, 91, 0},
      {32, 41, 152, 45, 1}
    },
    { {134, 217, 94, 7, 10},
      {232, 87, 21, 23, 9},
      {94, 7, 169, 182, 1},
      {158, 138, 142, 161, 7}
    },
    { {37, 126, 156, 27, 14},
      {193, 181, 237, 116, 9},
      {125, 131, 151, 234, 4},
      {146, 235, 122, 216, 3}
    },
    { {221, 222, 210, 16, 1},
      {230, 204, 152, 242, 8},
      {128, 132, 183, 187, 11},
      {20, 241, 147, 54, 7}
    },
    { {233, 1, 205, 160, 13},
      {54, 131, 103, 194, 2},
      {176, 91, 56, 9, 7},
      {68, 62, 108, 22, 12}
    },
    { {119, 152, 12, 57, 4},
      {192, 41, 246, 197, 8},
      {41, 195, 1, 158, 14},
      {26, 54, 249, 64, 3}
    },
    { {209, 14, 75, 139, 7},
      {52, 120, 76, 230, 11},
      {237, 29, 39, 8, 11},
      {214, 115, 33, 226, 12}
    },
    { {226, 178, 81, 251, 9},
      {117, 32, 187, 159, 3},
      {157, 248, 164, 212, 7},
      {207, 157, 208, 74, 14}
    },
    { {108, 39, 137, 18, 7},
      {151, 146, 236, 160, 1},
      {228, 137, 30, 67, 6},
      {128, 83, 116, 158, 9}
    },
    { {78, 213, 70, 229, 4},
      {234, 71, 82, 173, 2},
      {42, 118, 42, 183, 2},
      {75, 84, 174, 37, 7}
    },
    { {240, 6, 14, 244, 9},
      {12, 73, 175, 170, 2},
      {146, 247, 6, 0, 15},
      {69, 95, 89, 35, 0}
    },
    { {179, 69, 148, 196, 9},
      {12, 143, 33, 123, 2},
      {146, 50, 154, 44, 13},
      {77, 232, 79, 19, 0}
    },
    { {183, 99, 0, 71, 12},
      {137, 14, 105, 79, 1},
      {62, 32, 12, 110, 13},
      {143, 41, 103, 9, 1}
    },
    { {124, 196, 229, 3, 1},
      {182, 141, 48, 164, 5},
      {140, 10, 114, 51, 14},
      {162, 80, 203, 22, 13}
    },
    { {86, 1, 102, 16, 6},
      {160, 91, 192, 129, 4},
      {96, 134, 104, 6, 10},
      {40, 16, 61, 160, 5}
    },
    { {82, 75, 187, 242, 15},
      {20, 222, 207, 153, 15},
      {244, 253, 221, 36, 10},
      {249, 159, 55, 178, 8}
    },
    { {142, 50, 98, 30, 15},
      {239, 112, 201, 3, 5},
      {247, 132, 100, 199, 1},
      {172, 9, 48, 239, 7}
    },
    { {66, 68, 75, 195, 9},
      {52, 68, 5, 173, 3},
      {156, 61, 34, 36, 2},
      {203, 90, 2, 34, 12}
    },
    { {139, 101, 126, 225, 12},
      {35, 71, 71, 127, 6},
      {56, 119, 234, 109, 1},
      {111, 238, 46, 44, 4}
    },
    { {53, 42, 182, 84, 7},
      {141, 217, 232, 88, 12},
      {226, 166, 213, 74, 12},
      {49, 161, 121, 187, 1}
    },
    { {231, 165, 75, 233, 2},
      {177, 114, 54, 239, 2},
      {73, 125, 42, 94, 7},
      {79, 118, 196, 232, 13}
    },
    { {120, 30, 244, 21, 3},
      {110, 153, 168, 180, 12},
      {202, 130, 247, 129, 14},
      {50, 209, 89, 151, 6}
    },
    { {50, 242, 74, 129, 2},
      {97, 92, 60, 5, 2},
      {72, 21, 36, 244, 12},
      {74, 3, 195, 168, 6}
    },
    { {40, 17, 16, 119, 11},
      {78, 18, 163, 28, 1},
      {222, 224, 136, 129, 4},
      {131, 140, 84, 135, 2}
    },
    { {37, 203, 65, 84, 13},
      {188, 6, 249, 72, 8},
      {178, 168, 45, 58, 4},
      {17, 41, 246, 3, 13}
    },
    { {167, 103, 245, 103, 1},
      {189, 135, 42, 127, 5},
      {142, 106, 254, 110, 5},
      {175, 229, 78, 27, 13}
    },
    { {91, 165, 158, 70, 11},
      {15, 219, 21, 249, 1},
      {214, 39, 154, 93, 10},
      {137, 250, 141, 191, 0}
    },
    { {88, 61, 62, 106, 2},
      {67, 123, 6, 184, 13},
      {69, 103, 203, 193, 10},
      {177, 214, 13, 236, 2}
    },
    { {191, 77, 67, 125, 10},
      {186, 126, 163, 111, 8},
      {91, 236, 43, 47, 13},
      {31, 108, 87, 229, 13}
    },
    { {160, 164, 162, 247, 2},
      {9, 208, 178, 46, 7},
      {78, 244, 82, 80, 5},
      {231, 68, 208, 185, 0}
    },
    { {93, 52, 153, 184, 6},
      {211, 184, 198, 240, 2},
      {97, 217, 146, 203, 10},
      {64, 246, 49, 220, 11}
    },
    { {211, 7, 214, 157, 5},
      {44, 235, 200, 247, 2},
      {171, 150, 190, 12, 11},
      {78, 241, 61, 115, 4}
    },
    { {155, 172, 247, 16, 14},
      {51, 217, 209, 115, 12},
      {112, 142, 243, 93, 9},
      {60, 232, 185, 188, 12}
    },
    { {196, 108, 246, 252, 13},
      {173, 229, 195, 186, 14},
      {179, 246, 243, 98, 3},
      {117, 220, 58, 123, 5}
    },
    { {184, 1, 141, 118, 12},
      {26, 139, 231, 10, 1},
      {54, 235, 24, 1, 13},
      {133, 14, 125, 21, 8}
    },
    { {232, 151, 127, 181, 13},
      {126, 67, 255, 182, 6},
      {186, 223, 238, 145, 7},
      {102, 223, 252, 39, 14}
    },
    { {11, 60, 141, 188, 5},
      {95, 161, 198, 97, 10},
      {163, 219, 19, 205, 0},
      {88, 102, 56, 95, 10}
    },
    { {20, 20, 133, 139, 8},
      {208, 169, 1, 36, 3},
      {29, 26, 18, 130, 8},
      {194, 72, 9, 80, 11}
    },
    { {115, 104, 16, 10, 4},
      {1, 44, 96, 209, 9},
      {37, 0, 129, 108, 14},
      {152, 176, 99, 72, 0}
    },
    { {18, 132, 134, 239, 3},
      {12, 249, 18, 45, 3},
      {207, 118, 18, 20, 8},
      {203, 68, 137, 243, 0}
    },
    { {210, 252, 86, 125, 0},
      {105, 109, 146, 191, 8},
      {11, 230, 163, 244, 11},
      {31, 212, 155, 105, 6}
    },
    { {91, 207, 66, 130, 14},
      {34, 94, 89, 225, 11},
      {116, 20, 47, 61, 10},
      {216, 121, 167, 164, 4}
    },
    { {73, 183, 123, 167, 3},
      {127, 82, 30, 244, 7},
      {206, 93, 238, 217, 2},
      {226, 247, 132, 175, 14}
    },
    { {97, 229, 12, 169, 15},
      {5, 55, 119, 228, 2},
      {249, 83, 10, 120, 6},
      {66, 126, 238, 202, 0}
    },
    { {28, 221, 19, 226, 0},
      {210, 78, 18, 56, 11},
      {4, 124, 139, 179, 8},
      {209, 196, 135, 36, 11}
    },
    { {241, 23, 253, 83, 15},
      {116, 155, 237, 254, 5},
      {252, 171, 254, 136, 15},
      {167, 251, 125, 146, 14}
    },
    { {141, 51, 92, 19, 13},
      {231, 3, 205, 86, 1},
      {188, 131, 172, 203, 1},
      {134, 171, 60, 14, 7}
    },
    { {248, 223, 155, 48, 4},
      {82, 206, 254, 178, 8},
      {32, 205, 159, 177, 15},
      {20, 215, 247, 52, 10}
    },
    { {205, 115, 59, 196, 0},
      {219, 70, 12, 218, 6},
      {2, 61, 204, 235, 3},
      {101, 179, 6, 45, 11}
    },
    { {35, 29, 150, 151, 9},
      {76, 195, 161, 117, 11},
      {158, 150, 155, 140, 4},
      {218, 232, 92, 51, 2}
    },
    { {44, 151, 177, 71, 6},
      {218, 146, 120, 60, 5},
      {110, 40, 222, 147, 4},
      {163, 193, 228, 149, 11}
    },
    { {230, 105, 156, 80, 1},
      {133, 135, 164, 155, 8},
      {128, 163, 153, 102, 7},
      {29, 146, 94, 26, 1}
    },
    { {181, 161, 208, 86, 14},
      {169, 154, 241, 90, 1},
      {118, 160, 184, 90, 13},
      {133, 168, 245, 153, 5}
    },
    { {60, 125, 33, 88, 4},
      {211, 46, 224, 40, 12},
      {33, 168, 75, 227, 12},
      {49, 64, 119, 76, 11}
    },
    { {173, 20, 11, 20, 5},
      {222, 64, 228, 98, 0},
      {162, 141, 2, 139, 5},
      {4, 98, 112, 39, 11}
    },
    { {251, 9, 110, 114, 3},
      {38, 91, 166, 203, 13},
      {196, 231, 105, 13, 15},
      {189, 54, 93, 166, 4}
    },
    { {23, 132, 105, 191, 13},
      {188, 40, 215, 101, 7},
      {191, 217, 98, 30, 8},
      {234, 110, 177, 67, 13}
    },
    { {36, 48, 61, 47, 6},
      {217, 49, 102, 20, 5},
      {111, 75, 192, 194, 4},
      {162, 134, 104, 201, 11}
    },
    { {235, 214, 231, 21, 5},
      {126, 197, 248, 231, 4},
      {170, 142, 118, 189, 7},
      {46, 113, 250, 55, 14}
    },
    { {147, 91, 170, 215, 11},
      {76, 222, 141, 79, 15},
      {222, 181, 93, 172, 9},
      {255, 43, 23, 179, 2}
    },
    { {209, 95, 34, 205, 13},
      {76, 110, 73, 238, 14},
      {187, 52, 79, 168, 11},
      {119, 121, 39, 99, 2}
    },
    { {55, 178, 213, 83, 15},
      {245, 153, 249, 93, 1},
      {252, 170, 180, 222, 12},
      {139, 169, 249, 154, 15}
    },
    { {211, 101, 213, 98, 8},
      {49, 143, 3, 251, 1},
      {20, 106, 186, 108, 11},
      {141, 252, 15, 24, 12}
    },
    { {29, 118, 183, 146, 15},
      {215, 221, 201, 112, 7},
      {244, 158, 214, 235, 8},
      {224, 233, 59, 190, 11}
    },
    { {87, 185, 126, 171, 11},
      {229, 123, 23, 213, 15},
      {221, 87, 233, 222, 10},
      {250, 190, 141, 234, 7}
    },
    { {9, 11, 171, 164, 4},
      {26, 194, 78, 64, 14},
      {34, 93, 93, 9, 0},
      {112, 39, 36, 53, 8}
    },
    { {163, 131, 97, 118, 11},
      {60, 18, 187, 75, 5},
      {214, 232, 108, 28, 5},
      {173, 45, 212, 131, 12}
    },
    { {156, 250, 125, 156, 6},
      {251, 61, 220, 18, 14},
      {99, 155, 229, 243, 9},
      {116, 131, 187, 205, 15}
    },
    { {76, 134, 94, 122, 15},
      {166, 113, 223, 184, 1},
      {245, 231, 166, 19, 2},
      {129, 223, 184, 230, 5}
    },
    { {249, 13, 22, 148, 11},
      {14, 91, 161, 242, 10},
      {210, 150, 139, 9, 15},
      {84, 248, 93, 167, 0}
    },
    { {120, 93, 12, 212, 1},
      {78, 15, 164, 168, 10},
      {130, 179, 11, 161, 14},
      {81, 82, 95, 7, 2}
    },
    { {51, 98, 95, 192, 1},
      {53, 77, 44, 89, 2},
      {128, 63, 164, 108, 12},
      {73, 163, 75, 42, 12}
    },
    { {0, 213, 240, 148, 3},
      {108, 150, 144, 48, 6},
      {194, 144, 250, 176, 0},
      {96, 192, 150, 147, 6}
    },
    { {167, 185, 175, 176, 8},
      {209, 195, 183, 67, 14},
      {16, 223, 89, 222, 5},
      {124, 46, 220, 56, 11}
    },
    { {223, 59, 151, 88, 10},
      {211, 251, 137, 219, 8},
      {81, 174, 157, 207, 11},
      {29, 185, 29, 252, 11}
    },
    { {182, 138, 31, 89, 2},
      {144, 121, 188, 31, 8},
      {73, 175, 133, 22, 13},
      {31, 131, 217, 224, 9}
    },
    { {103, 50, 106, 212, 0},
      {233, 64, 172, 201, 6},
      {2, 181, 100, 206, 6},
      {105, 51, 80, 41, 7}
    },
    { {7, 184, 90, 134, 3},
      {237, 80, 20, 81, 11},
      {198, 21, 161, 222, 0},
      {216, 162, 128, 171, 7}
    },
    { {37, 38, 25, 247, 15},
      {157, 16, 239, 124, 3},
      {254, 249, 134, 74, 4},
      {195, 239, 112, 139, 9}
    },
    { {130, 206, 71, 104, 3},
      {52, 117, 26, 43, 8},
      {193, 110, 39, 52, 1},
      {29, 69, 138, 226, 12}
    },
    { {108, 230, 33, 121, 12},
      {147, 36, 251, 172, 4},
      {57, 232, 70, 115, 6},
      {35, 93, 242, 76, 9}
    },
    { {98, 230, 221, 55, 8},
      {57, 133, 191, 181, 1},
      {30, 203, 182, 116, 6},
      {138, 223, 218, 25, 12}
    },
    { {62, 92, 11, 17, 10},
      {210, 92, 165, 37, 8},
      {88, 141, 3, 167, 12},
      {26, 74, 83, 164, 11}
    },
    { {57, 245, 112, 80, 8},
      {99, 14, 177, 120, 4},
      {16, 160, 234, 249, 12},
      {33, 232, 215, 12, 6}
    },
    { {214, 87, 51, 142, 6},
      {216, 126, 72, 179, 7},
      {103, 28, 206, 166, 11},
      {236, 209, 39, 225, 11}
    },
    { {192, 98, 168, 92, 12},
      {9, 164, 205, 138, 4},
      {51, 161, 84, 96, 3},
      {37, 27, 50, 89, 0}
    },
    { {131, 220, 0, 222, 5},
      {76, 36, 208, 107, 11},
      {167, 176, 3, 188, 1},
      {221, 96, 178, 67, 2}
    },
    { {177, 33, 195, 250, 3},
      {53, 250, 162, 74, 3},
      {197, 252, 56, 72, 13},
      {197, 36, 85, 250, 12}
    },
    { {227, 149, 10, 113, 14},
      {64, 82, 247, 239, 0},
      {120, 229, 10, 156, 7},
      {15, 126, 244, 160, 2}
    },
    { {113, 29, 253, 190, 8},
      {120, 171, 167, 240, 15},
      {23, 219, 251, 136, 14},
      {240, 254, 93, 81, 14}
    },
    { {130, 171, 112, 191, 2},
      {41, 50, 154, 23, 15},
      {79, 208, 237, 84, 1},
      {254, 133, 148, 201, 4}
    },
    { {93, 123, 132, 66, 1},
      {199, 143, 8, 200, 9},
      {132, 34, 29, 235, 10},
      {145, 49, 15, 30, 3}
    },
    { {103, 239, 52, 200, 12},
      {129, 39, 121, 249, 14},
      {49, 50, 207, 126, 6},
      {121, 249, 238, 72, 1}
    },
    { {202, 53, 231, 230, 13},
      {127, 195, 67, 171, 7},
      {182, 126, 122, 197, 3},
      {237, 92, 44, 63, 14}
    },
    { {36, 1, 77, 111, 11},
      {188, 51, 39, 12, 1},
      {223, 107, 40, 2, 4},
      {131, 14, 76, 195, 13}
    },
    { {183, 239, 157, 161, 14},
      {145, 159, 127, 119, 10},
      {120, 91, 159, 126, 13},
      {94, 239, 239, 152, 9}
    },
    { {176, 76, 172, 188, 7},
      {12, 189, 230, 34, 14},
      {227, 211, 83, 32, 13},
      {116, 70, 123, 211, 0}
    },
    { {193, 166, 3, 154, 8},
      {17, 96, 153, 226, 3},
      {21, 156, 6, 88, 3},
      {196, 121, 144, 104, 8}
    },
    { {197, 1, 82, 163, 10},
      {160, 82, 3, 214, 3},
      {92, 84, 168, 10, 3},
      {198, 188, 4, 160, 5}
    },
    { {220, 202, 43, 43, 7},
      {150, 124, 94, 134, 13},
      {237, 77, 69, 51, 11},
      {182, 23, 163, 230, 9}
    },
    { {88, 6, 54, 143, 3},
      {14, 121, 8, 180, 7},
      {207, 22, 198, 1, 10},
      {226, 209, 9, 231, 0}
    },
    { {139, 223, 114, 151, 10},
      {106, 86, 153, 119, 15},
      {94, 148, 239, 189, 1},
      {254, 233, 150, 165, 6}
    },
    { {118, 72, 225, 171, 10},
      {176, 188, 35, 133, 15},
      {93, 88, 113, 38, 14},
      {250, 28, 67, 208, 13}
    },
    { {224, 20, 244, 247, 7},
      {108, 145, 226, 190, 7},
      {238, 242, 242, 128, 7},
      {231, 212, 120, 147, 6}
    },
    { {113, 170, 60, 249, 0},
      {1, 41, 190, 220, 14},
      {9, 243, 197, 88, 14},
      {115, 183, 217, 72, 0}
    },
    { {156, 171, 243, 108, 14},
      {187, 250, 91, 26, 12},
      {115, 108, 253, 83, 9},
      {53, 141, 165, 253, 13}
    },
    { {135, 84, 73, 90, 15},
      {244, 52, 197, 107, 1},
      {245, 169, 34, 174, 1},
      {141, 106, 50, 194, 15}
    },
    { {154, 153, 213, 71, 2},
      {122, 155, 16, 31, 9},
      {78, 42, 185, 149, 9},
      {159, 128, 141, 149, 14}
    },
    { {231, 44, 187, 69, 6},
      {153, 208, 100, 255, 12},
      {106, 45, 211, 78, 7},
      {63, 242, 96, 185, 9}
    },
    { {196, 176, 203, 147, 1},
      {245, 192, 148, 134, 3},
      {140, 157, 48, 210, 3},
      {198, 18, 144, 58, 15}
    },
    { {137, 237, 4, 93, 7},
      {15, 55, 208, 110, 8},
      {235, 162, 11, 121, 1},
      {23, 96, 190, 207, 0}
    },
    { {199, 169, 10, 58, 11},
      {133, 114, 151, 195, 9},
      {213, 197, 9, 94, 3},
      {156, 62, 148, 234, 1}
    },
    { {65, 90, 235, 196, 2},
      {120, 212, 12, 200, 14},
      {66, 61, 117, 168, 2},
      {113, 51, 2, 177, 14}
    },
    { {127, 85, 215, 48, 2},
      {242, 223, 162, 241, 0},
      {64, 206, 186, 175, 14},
      {8, 244, 95, 180, 15}
    },
    { {17, 75, 16, 122, 6},
      {0, 62, 202, 88, 9},
      {101, 224, 141, 40, 8},
      {145, 165, 55, 192, 0}
    },
    { {230, 235, 111, 39, 4},
      {185, 71, 126, 135, 13},
      {46, 79, 109, 118, 7},
      {190, 23, 238, 41, 13}
    },
    { {132, 244, 176, 135, 15},
      {205, 148, 81, 54, 7},
      {254, 16, 210, 242, 1},
      {230, 200, 162, 155, 3}
    },
    { {91, 222, 44, 8, 8},
      {66, 45, 29, 225, 12},
      {17, 3, 71, 189, 10},
      {56, 123, 139, 68, 2}
    },
    { {205, 253, 149, 194, 7},
      {215, 151, 80, 250, 11},
      {228, 58, 155, 251, 3},
      {213, 240, 174, 158, 11}
    },
    { {51, 193, 37, 123, 3},
      {20, 63, 178, 77, 5},
      {205, 234, 72, 60, 12},
      {171, 36, 223, 194, 8}
    },
    { {99, 27, 71, 141, 15},
      {124, 115, 105, 197, 10},
      {251, 30, 45, 140, 6},
      {90, 57, 108, 227, 14}
    },
    { {229, 239, 80, 81, 11},
      {165, 22, 185, 254, 8},
      {216, 160, 175, 122, 7},
      {23, 249, 214, 138, 5}
    },
    { {65, 141, 160, 190, 12},
      {8, 162, 211, 224, 15},
      {55, 208, 91, 24, 2},
      {240, 124, 180, 81, 0}
    },
    { {193, 184, 36, 139, 1},
      {69, 33, 16, 198, 15},
      {141, 18, 65, 216, 3},
      {246, 48, 136, 74, 2}
    },
    { {118, 222, 87, 133, 11},
      {252, 93, 57, 181, 10},
      {218, 30, 167, 182, 14},
      {90, 217, 203, 163, 15}
    },
    { {219, 201, 87, 218, 6},
      {50, 127, 208, 219, 11},
      {101, 190, 169, 61, 11},
      {221, 176, 191, 228, 12}
    },
    { {202, 225, 244, 5, 4},
      {43, 135, 80, 151, 4},
      {42, 2, 248, 117, 3},
      {46, 144, 174, 29, 4}
    },
    { {145, 76, 36, 42, 2},
      {0, 61, 2, 98, 13},
      {69, 66, 67, 40, 9},
      {180, 100, 11, 192, 0}
    },
    { {69, 163, 208, 168, 2},
      {161, 178, 26, 208, 2},
      {65, 80, 188, 90, 2},
      {64, 181, 132, 216, 5}
    },
    { {136, 154, 125, 93, 13},
      {126, 33, 221, 30, 12},
      {187, 171, 229, 145, 1},
      {55, 139, 184, 71, 14}
    },
    { {154, 61, 179, 211, 2},
      {83, 218, 128, 63, 15},
      {76, 188, 219, 197, 9},
      {255, 192, 21, 188, 10}
    },
    { {208, 44, 134, 112, 15},
      {5, 217, 195, 170, 8},
      {240, 230, 19, 64, 11},
      {21, 92, 57, 186, 0}
    },
    { {21, 123, 52, 80, 12},
      {193, 15, 201, 88, 12},
      {48, 162, 205, 234, 8},
      {49, 169, 63, 8, 3}
    },
    { {137, 13, 55, 91, 10},
      {18, 115, 129, 126, 13},
      {93, 174, 203, 9, 1},
      {183, 232, 28, 228, 8}
    },
    { {137, 27, 201, 59, 10},
      {114, 178, 143, 70, 9},
      {93, 201, 61, 137, 1},
      {150, 47, 20, 212, 14}
    },
    { {138, 2, 216, 10, 3},
      {38, 176, 12, 19, 1},
      {197, 1, 180, 5, 1},
      {140, 131, 0, 214, 4}
    },
    { {30, 93, 243, 65, 14},
      {242, 222, 65, 61, 12},
      {120, 44, 251, 167, 8},
      {59, 200, 39, 180, 15}
    },
    { {120, 109, 137, 131, 2},
      {19, 158, 36, 164, 11},
      {76, 25, 27, 97, 14},
      {210, 82, 71, 156, 8}
    },
    { {172, 247, 228, 126, 13},
      {239, 167, 251, 42, 5},
      {183, 226, 126, 243, 5},
      {165, 77, 254, 95, 7}
    },
    { {28, 22, 226, 90, 0},
      {226, 232, 136, 40, 5},
      {5, 164, 118, 131, 8},
      {161, 65, 17, 116, 7}
    },
    { {46, 169, 4, 13, 9},
      {143, 35, 49, 5, 8},
      {155, 2, 9, 87, 4},
      {26, 8, 204, 79, 1}
    },
    { {243, 38, 138, 238, 5},
      {13, 232, 110, 235, 3},
      {167, 117, 22, 76, 15},
      {205, 119, 97, 123, 0}
    },
    { {128, 232, 75, 204, 11},
      {61, 116, 21, 10, 10},
      {211, 61, 33, 112, 1},
      {85, 10, 130, 235, 12}
    },
    { {162, 190, 211, 245, 12},
      {121, 192, 251, 63, 10},
      {58, 252, 183, 212, 5},
      {95, 205, 240, 57, 14}
    },
    { {179, 172, 4, 251, 2},
      {1, 57, 178, 111, 11},
      {77, 242, 3, 92, 13},
      {223, 100, 217, 200, 0}
    },
    { {132, 89, 129, 25, 2},
      {208, 182, 128, 6, 8},
      {73, 136, 25, 162, 1},
      {22, 0, 22, 208, 11}
    },
    { {96, 167, 159, 201, 4},
      {17, 227, 124, 188, 2},
      {41, 63, 158, 80, 6},
      {67, 211, 236, 120, 8}
    },
    { {19, 143, 250, 197, 9},
      {44, 202, 29, 125, 14},
      {154, 53, 255, 28, 8},
      {123, 235, 133, 51, 4}
    },
    { {214, 125, 112, 144, 3},
      {229, 30, 128, 179, 14},
      {192, 144, 235, 230, 11},
      {124, 208, 23, 138, 7}
    },
    { {43, 138, 160, 242, 5},
      {6, 128, 250, 73, 15},
      {164, 240, 85, 29, 4},
      {249, 37, 240, 22, 0}
    },
    { {41, 131, 176, 59, 10},
      {2, 178, 187, 84, 5},
      {93, 192, 220, 25, 4},
      {162, 173, 212, 212, 0}
    },
    { {140, 215, 81, 28, 8},
      {250, 38, 153, 50, 0},
      {19, 136, 174, 179, 1},
      {4, 201, 150, 69, 15}
    },
    { {47, 222, 77, 2, 12},
      {242, 5, 125, 97, 9},
      {52, 11, 39, 191, 4},
      {152, 107, 234, 4, 15}
    },
    { {213, 4, 207, 5, 12},
      {184, 201, 69, 230, 0},
      {58, 15, 50, 10, 11},
      {6, 122, 41, 49, 13}
    },
    { {186, 231, 214, 153, 0},
      {35, 239, 184, 55, 2},
      {9, 150, 190, 117, 13},
      {78, 193, 223, 124, 4}
    },
    { {156, 243, 126, 164, 13},
      {239, 79, 95, 18, 6},
      {178, 87, 236, 243, 9},
      {100, 143, 175, 47, 7}
    },
    { {143, 79, 97, 177, 14},
      {178, 22, 203, 103, 14},
      {120, 216, 111, 47, 1},
      {126, 109, 54, 132, 13}
    },
    { {137, 255, 175, 206, 6},
      {91, 247, 92, 106, 15},
      {103, 63, 95, 249, 1},
      {245, 99, 174, 253, 10}
    },
    { {124, 126, 177, 202, 2},
      {211, 188, 40, 184, 15},
      {69, 56, 215, 227, 14},
      {241, 209, 67, 220, 11}
    },
    { {150, 194, 61, 167, 11},
      {156, 29, 31, 23, 7},
      {222, 91, 196, 54, 9},
      {238, 143, 139, 131, 9}
    }
};



static unsigned char DICT_APRILTAG_36h11_BYTES[][4][5] =
{
    { {33, 161, 70, 186, 11},
      {37, 115, 179, 64, 3},
      {213, 214, 40, 88, 4},
      {192, 44, 220, 234, 4}
    },
    { {146, 209, 143, 233, 11},
      {84, 255, 23, 15, 2},
      {217, 127, 24, 180, 9},
      {79, 14, 143, 242, 10}
    },
    { {112, 137, 1, 75, 11},
      {20, 58, 49, 140, 9},
      {221, 40, 9, 16, 14},
      {147, 24, 197, 194, 8}
    },
    { {25, 57, 121, 226, 7},
      {119, 26, 70, 88, 15},
      {228, 121, 233, 201, 8},
      {241, 166, 37, 142, 14}
    },
    { {68, 21, 61, 61, 7},
      {220, 51, 198, 180, 4},
      {235, 203, 202, 130, 2},
      {34, 214, 60, 195, 11}
    },
    { {53, 205, 91, 140, 15},
      {188, 126, 117, 112, 10},
      {243, 29, 171, 58, 12},
      {80, 234, 231, 227, 13}
    },
    { {161, 11, 165, 106, 0},
      {16, 163, 42, 74, 13},
      {5, 106, 93, 8, 5},
      {181, 37, 76, 80, 8}
    },
    { {43, 135, 74, 96, 8},
      {34, 66, 63, 105, 0},
      {16, 101, 46, 29, 4},
      {9, 111, 196, 36, 4}
    },
    { {181, 127, 184, 212, 4},
      {201, 142, 236, 122, 14},
      {34, 177, 223, 234, 13},
      {117, 227, 119, 25, 3}
    },
    { {78, 32, 181, 166, 4},
      {155, 129, 66, 145, 7},
      {38, 90, 208, 71, 2},
      {232, 148, 40, 29, 9}
    },
    { {97, 216, 151, 242, 12},
      {80, 197, 243, 216, 11},
      {52, 254, 145, 184, 6},
      {209, 188, 250, 48, 10}
    },
    { {171, 52, 105, 255, 12},
      {123, 32, 231, 111, 7},
      {63, 249, 98, 205, 5},
      {239, 110, 112, 77, 14}
    },
    { {89, 76, 164, 92, 2},
      {10, 189, 128, 232, 12},
      {67, 162, 83, 41, 10},
      {49, 112, 27, 213, 0}
    },
    { {250, 28, 45, 46, 2},
      {90, 57, 38, 163, 13},
      {71, 75, 67, 133, 15},
      {188, 86, 73, 197, 10}
    },
    { {151, 194, 75, 151, 2},
      {184, 92, 156, 71, 3},
      {78, 157, 36, 62, 9},
      {206, 35, 147, 161, 13}
    },
    { {117, 146, 134, 36, 10},
      {200, 217, 59, 192, 0},
      {82, 70, 20, 154, 14},
      {0, 61, 201, 177, 3}
    },
    { {28, 170, 254, 153, 10},
      {163, 249, 157, 20, 14},
      {89, 151, 245, 83, 8},
      {114, 139, 153, 252, 5}
    },
    { {50, 54, 221, 193, 6},
      {113, 153, 108, 61, 2},
      {104, 59, 182, 196, 12},
      {75, 195, 105, 152, 14}
    },
    { {136, 78, 82, 125, 6},
      {42, 116, 202, 62, 8},
      {107, 228, 167, 33, 1},
      {23, 197, 50, 229, 4}
    },
    { {87, 113, 168, 199, 14},
      {201, 158, 69, 205, 7},
      {126, 49, 88, 238, 10},
      {235, 58, 39, 153, 3}
    },
    { {245, 41, 37, 184, 1},
      {149, 43, 162, 194, 14},
      {129, 218, 73, 74, 15},
      {116, 52, 93, 74, 9}
    },
    { {46, 89, 204, 26, 1},
      {230, 167, 164, 1, 9},
      {133, 131, 57, 167, 4},
      {152, 2, 94, 86, 7}
    },
    { {127, 165, 138, 179, 1},
      {135, 202, 182, 229, 3},
      {140, 213, 26, 95, 14},
      {202, 118, 213, 62, 1}
    },
    { {235, 67, 56, 79, 9},
      {14, 38, 45, 223, 5},
      {159, 33, 204, 45, 7},
      {175, 187, 70, 71, 0}
    },
    { {57, 19, 213, 52, 5},
      {126, 139, 234, 80, 0},
      {162, 202, 188, 137, 12},
      {0, 165, 125, 23, 14}
    },
    { {215, 155, 26, 59, 5},
      {196, 106, 222, 215, 9},
      {173, 197, 141, 158, 11},
      {158, 183, 181, 98, 3}
    },
    { {13, 251, 231, 104, 13},
      {247, 231, 91, 72, 12},
      {177, 110, 125, 251, 0},
      {49, 45, 174, 126, 15}
    },
    { {92, 215, 160, 49, 13},
      {198, 142, 219, 164, 4},
      {184, 192, 94, 179, 10},
      {34, 93, 183, 22, 3}
    },
    { {185, 200, 247, 164, 11},
      {62, 221, 51, 82, 14},
      {210, 94, 241, 57, 13},
      {116, 172, 203, 183, 12}
    },
    { {70, 152, 126, 6, 11},
      {236, 81, 21, 145, 13},
      {214, 7, 225, 150, 2},
      {184, 154, 136, 163, 7}
    },
    { {110, 12, 93, 82, 7},
      {182, 17, 228, 185, 9},
      {228, 171, 163, 7, 6},
      {153, 210, 120, 134, 13}
    },
    { {91, 197, 103, 187, 4},
      {50, 111, 210, 229, 7},
      {45, 222, 106, 61, 10},
      {234, 116, 191, 100, 12}
    },
    { {30, 99, 2, 188, 2},
      {139, 126, 138, 1, 2},
      {67, 212, 12, 103, 8},
      {72, 5, 23, 237, 1}
    },
    { {138, 199, 208, 70, 10},
      {42, 150, 25, 59, 1},
      {86, 32, 190, 53, 1},
      {141, 201, 134, 149, 4}
    },
    { {166, 95, 226, 50, 6},
      {224, 214, 234, 35, 13},
      {100, 196, 127, 166, 5},
      {188, 69, 118, 176, 7}
    },
    { {84, 0, 235, 97, 6},
      {176, 216, 70, 140, 4},
      {104, 109, 112, 2, 10},
      {35, 22, 33, 176, 13}
    },
    { {247, 160, 102, 131, 6},
      {161, 89, 112, 199, 7},
      {108, 22, 96, 94, 15},
      {238, 48, 233, 168, 5}
    },
    { {45, 208, 175, 223, 6},
      {218, 245, 244, 76, 7},
      {111, 191, 80, 187, 4},
      {227, 34, 250, 245, 11}
    },
    { {59, 156, 182, 11, 1},
      {70, 233, 48, 117, 13},
      {141, 6, 211, 157, 12},
      {186, 224, 201, 118, 2}
    },
    { {173, 106, 28, 175, 9},
      {143, 37, 47, 86, 11},
      {159, 83, 133, 107, 5},
      {214, 175, 74, 79, 1}
    },
    { {147, 182, 46, 251, 5},
      {69, 105, 222, 111, 7},
      {173, 247, 70, 220, 9},
      {239, 103, 185, 106, 2}
    },
    { {187, 33, 13, 147, 13},
      {23, 11, 229, 71, 3},
      {188, 155, 8, 77, 13},
      {206, 42, 125, 14, 8}
    },
    { {222, 229, 80, 148, 11},
      {175, 30, 145, 179, 2},
      {210, 144, 170, 119, 11},
      {76, 216, 151, 143, 5}
    },
    { {160, 205, 20, 121, 11},
      {4, 55, 179, 62, 8},
      {217, 226, 139, 48, 5},
      {23, 204, 222, 194, 0}
    },
    { {42, 35, 171, 178, 7},
      {23, 210, 238, 1, 7},
      {228, 221, 92, 69, 4},
      {232, 7, 116, 190, 8}
    },
    { {79, 103, 244, 181, 15},
      {175, 151, 203, 245, 6},
      {250, 210, 254, 111, 2},
      {106, 253, 62, 159, 5}
    },
    { {88, 168, 21, 129, 8},
      {19, 9, 17, 148, 10},
      {24, 26, 129, 81, 10},
      {82, 152, 137, 12, 8}
    },
    { {116, 52, 39, 214, 4},
      {217, 73, 224, 168, 7},
      {38, 190, 66, 194, 14},
      {225, 80, 121, 41, 11}
    },
    { {29, 236, 64, 135, 4},
      {171, 12, 80, 100, 11},
      {46, 16, 35, 123, 8},
      {210, 96, 163, 13, 5}
    },
    { {137, 42, 179, 160, 2},
      {19, 208, 10, 82, 14},
      {64, 92, 213, 73, 1},
      {116, 165, 0, 188, 8}
    },
    { {56, 213, 116, 46, 14},
      {106, 63, 115, 48, 5},
      {119, 66, 234, 177, 12},
      {160, 204, 239, 197, 6}
    },
    { {12, 67, 86, 122, 1},
      {166, 103, 138, 24, 1},
      {133, 230, 172, 35, 0},
      {129, 133, 30, 102, 5}
    },
    { {93, 179, 17, 203, 1},
      {215, 42, 24, 220, 3},
      {141, 56, 140, 219, 10},
      {195, 177, 133, 78, 11}
    },
    { {201, 87, 202, 31, 9},
      {110, 230, 141, 230, 1},
      {159, 133, 62, 169, 3},
      {134, 123, 22, 119, 6}
    },
    { {162, 36, 87, 130, 3},
      {53, 81, 32, 51, 3},
      {196, 30, 162, 68, 5},
      {204, 192, 72, 170, 12}
    },
    { {41, 172, 233, 143, 3},
      {63, 176, 52, 100, 15},
      {207, 25, 115, 89, 4},
      {242, 98, 192, 223, 12}
    },
    { {204, 197, 44, 82, 8},
      {130, 7, 149, 170, 5},
      {20, 163, 74, 51, 3},
      {165, 90, 158, 4, 1}
    },
    { {218, 139, 53, 120, 12},
      {18, 43, 219, 155, 12},
      {49, 234, 205, 21, 11},
      {61, 157, 189, 68, 8}
    },
    { {213, 180, 185, 227, 6},
      {209, 152, 86, 254, 7},
      {108, 121, 210, 218, 11},
      {231, 246, 161, 152, 11}
    },
    { {172, 156, 221, 60, 14},
      {250, 177, 247, 50, 8},
      {115, 203, 179, 147, 5},
      {20, 206, 248, 213, 15}
    },
    { {203, 218, 168, 57, 1},
      {70, 164, 158, 199, 12},
      {137, 193, 85, 189, 3},
      {62, 55, 146, 86, 2}
    },
    { {25, 134, 37, 107, 1},
      {22, 41, 26, 108, 5},
      {141, 106, 70, 25, 8},
      {163, 101, 137, 70, 8}
    },
    { {186, 230, 204, 136, 9},
      {39, 173, 61, 35, 2},
      {145, 19, 54, 117, 13},
      {76, 75, 203, 94, 4}
    },
    { {223, 145, 177, 14, 5},
      {222, 170, 80, 211, 5},
      {167, 8, 216, 159, 11},
      {172, 176, 165, 87, 11}
    },
    { {161, 169, 213, 167, 5},
      {61, 131, 114, 86, 11},
      {174, 90, 185, 88, 5},
      {214, 164, 236, 27, 12}
    },
    { {43, 101, 26, 169, 13},
      {7, 102, 103, 117, 2},
      {185, 85, 138, 109, 4},
      {74, 238, 102, 110, 0}
    },
    { {77, 92, 46, 25, 13},
      {198, 101, 197, 228, 12},
      {185, 135, 67, 171, 2},
      {50, 122, 58, 102, 3}
    },
    { {91, 22, 55, 54, 11},
      {94, 89, 139, 241, 5},
      {214, 206, 198, 141, 10},
      {168, 253, 25, 167, 10}
    },
    { {247, 83, 214, 10, 8},
      {224, 239, 41, 211, 1},
      {21, 6, 188, 174, 15},
      {140, 185, 79, 112, 7}
    },
    { {94, 132, 179, 125, 2},
      {154, 248, 146, 189, 4},
      {75, 236, 210, 23, 10},
      {43, 212, 145, 245, 9}
    },
    { {252, 228, 58, 159, 2},
      {139, 124, 180, 182, 7},
      {79, 149, 194, 115, 15},
      {230, 210, 211, 237, 1}
    },
    { {187, 146, 109, 26, 6},
      {114, 57, 252, 67, 5},
      {101, 139, 100, 157, 13},
      {172, 35, 249, 196, 14}
    },
    { {237, 141, 35, 110, 5},
      {158, 98, 114, 234, 13},
      {167, 108, 75, 27, 7},
      {181, 116, 228, 103, 9}
    },
    { {106, 248, 70, 243, 7},
      {103, 85, 242, 141, 11},
      {236, 246, 33, 245, 6},
      {219, 20, 250, 174, 6}
    },
    { {234, 86, 165, 19, 8},
      {82, 133, 169, 167, 5},
      {28, 138, 86, 165, 7},
      {174, 89, 90, 20, 10}
    },
    { {93, 73, 113, 70, 12},
      {186, 14, 65, 216, 13},
      {54, 40, 233, 43, 10},
      {177, 184, 39, 5, 13}
    },
    { {71, 3, 96, 236, 10},
      {168, 50, 11, 201, 6},
      {83, 112, 108, 14, 2},
      {105, 61, 4, 193, 5}
    },
    { {140, 75, 205, 103, 10},
      {186, 151, 15, 14, 9},
      {94, 107, 61, 35, 1},
      {151, 15, 14, 149, 13}
    },
    { {41, 95, 57, 1, 14},
      {82, 22, 109, 116, 12},
      {120, 9, 207, 169, 4},
      {50, 235, 102, 132, 10}
    },
    { {246, 181, 119, 233, 7},
      {245, 123, 114, 191, 6},
      {233, 126, 234, 214, 15},
      {111, 212, 237, 234, 15}
    },
    { {82, 6, 2, 245, 10},
      {8, 88, 139, 173, 2},
      {90, 244, 6, 4, 10},
      {75, 93, 17, 161, 0}
    },
    { {181, 17, 148, 146, 14},
      {192, 155, 225, 82, 3},
      {116, 146, 152, 138, 13},
      {196, 168, 125, 144, 3}
    },
    { {72, 226, 153, 171, 7},
      {23, 180, 94, 148, 3},
      {237, 89, 148, 113, 2},
      {194, 151, 162, 222, 8}
    },
    { {183, 159, 87, 55, 10},
      {248, 91, 187, 119, 9},
      {94, 206, 175, 158, 13},
      {158, 237, 221, 161, 15}
    },
    { {168, 20, 198, 131, 14},
      {98, 209, 97, 38, 3},
      {124, 22, 50, 129, 5},
      {198, 72, 104, 180, 6}
    },
    { {45, 169, 0, 88, 3},
      {135, 50, 176, 72, 8},
      {193, 160, 9, 91, 4},
      {17, 32, 212, 206, 1}
    },
    { {212, 171, 164, 121, 7},
      {133, 187, 218, 142, 12},
      {233, 226, 93, 82, 11},
      {55, 21, 189, 218, 1}
    },
    { {152, 42, 14, 101, 12},
      {11, 73, 79, 14, 8},
      {58, 103, 5, 65, 9},
      {23, 15, 41, 45, 0}
    },
    { {244, 128, 204, 193, 5},
      {164, 137, 116, 142, 2},
      {168, 51, 48, 18, 15},
      {71, 18, 233, 18, 5}
    },
    { {11, 32, 234, 115, 2},
      {35, 208, 134, 77, 5},
      {76, 229, 112, 77, 0},
      {171, 38, 16, 188, 4}
    },
    { {169, 80, 103, 16, 10},
      {114, 85, 161, 66, 4},
      {80, 142, 96, 169, 5},
      {36, 40, 90, 164, 14}
    },
    { {4, 134, 226, 238, 1},
      {172, 224, 26, 40, 7},
      {135, 116, 118, 18, 0},
      {225, 69, 128, 115, 5}
    },
    { {176, 169, 114, 97, 5},
      {37, 74, 114, 30, 12},
      {168, 100, 233, 80, 13},
      {55, 132, 229, 42, 4}
    },
    { {231, 63, 58, 37, 7},
      {205, 82, 110, 247, 12},
      {234, 69, 207, 206, 7},
      {62, 247, 100, 171, 3}
    },
    { {23, 226, 5, 185, 8},
      {145, 45, 155, 69, 2},
      {25, 218, 4, 126, 8},
      {74, 45, 155, 72, 9}
    },
    { {236, 154, 178, 178, 4},
      {194, 192, 250, 146, 15},
      {36, 212, 213, 147, 7},
      {244, 149, 240, 52, 3}
    },
    { {217, 51, 158, 7, 6},
      {75, 219, 76, 214, 1},
      {110, 7, 156, 201, 11},
      {134, 179, 45, 189, 2}
    },
    { {235, 136, 116, 249, 9},
      {38, 33, 179, 223, 14},
      {153, 242, 225, 29, 7},
      {127, 188, 216, 70, 4}
    },
    { {148, 32, 165, 92, 12},
      {153, 169, 193, 10, 4},
      {51, 170, 80, 66, 9},
      {37, 8, 57, 89, 9}
    },
    { {227, 100, 255, 153, 2},
      {49, 245, 164, 247, 6},
      {73, 159, 242, 108, 7},
      {110, 242, 90, 248, 12}
    },
    { {138, 76, 184, 240, 10},
      {2, 148, 135, 59, 14},
      {80, 241, 211, 37, 1},
      {125, 206, 18, 148, 0}
    },
    { {247, 42, 238, 118, 6},
      {169, 217, 238, 203, 13},
      {102, 231, 117, 78, 15},
      {189, 55, 121, 185, 5}
    },
    { {115, 204, 107, 86, 9},
      {60, 76, 181, 233, 13},
      {150, 173, 99, 60, 14},
      {185, 122, 211, 35, 12}
    },
    { {201, 156, 166, 165, 9},
      {78, 193, 19, 230, 14},
      {154, 86, 83, 153, 3},
      {118, 124, 136, 55, 2}
    },
    { {54, 106, 12, 83, 5},
      {133, 13, 236, 13, 9},
      {172, 163, 5, 102, 12},
      {155, 3, 123, 10, 1}
    },
    { {80, 158, 81, 148, 3},
      {124, 24, 152, 176, 10},
      {194, 152, 167, 144, 10},
      {80, 209, 145, 131, 14}
    },
    { {186, 109, 34, 125, 1},
      {15, 110, 162, 47, 12},
      {139, 228, 75, 101, 13},
      {63, 68, 87, 111, 0}
    },
    { {210, 172, 241, 99, 11},
      {53, 152, 19, 191, 13},
      {220, 104, 243, 84, 11},
      {191, 220, 129, 154, 12}
    },
    { {217, 137, 254, 246, 8},
      {42, 203, 151, 218, 15},
      {22, 247, 249, 25, 11},
      {245, 190, 157, 53, 4}
    },
    { {100, 23, 7, 187, 2},
      {208, 115, 170, 164, 3},
      {77, 222, 14, 130, 6},
      {194, 85, 92, 224, 11}
    },
    { {249, 189, 49, 42, 11},
      {87, 58, 51, 242, 13},
      {213, 72, 203, 217, 15},
      {180, 252, 197, 206, 10}
    },
    { {119, 115, 156, 32, 7},
      {197, 159, 110, 209, 0},
      {224, 67, 156, 238, 14},
      {8, 183, 111, 154, 3}
    },
    { {104, 191, 141, 27, 15},
      {87, 179, 253, 164, 9},
      {253, 139, 31, 209, 6},
      {146, 91, 252, 222, 10}
    },
    { {222, 221, 132, 166, 14},
      {202, 159, 83, 163, 11},
      {118, 82, 27, 183, 11},
      {220, 92, 175, 149, 3}
    },
    { {66, 234, 236, 73, 11},
      {37, 181, 29, 141, 12},
      {217, 35, 117, 116, 2},
      {59, 27, 138, 218, 4}
    },
    { {48, 43, 104, 213, 3},
      {45, 26, 172, 12, 14},
      {202, 177, 109, 64, 12},
      {115, 3, 85, 139, 4}
    },
    { {78, 187, 147, 12, 14},
      {219, 242, 89, 145, 8},
      {115, 12, 157, 215, 2},
      {24, 153, 164, 253, 11}
    },
    { {67, 156, 93, 155, 5},
      {116, 33, 212, 245, 11},
      {173, 155, 163, 156, 2},
      {218, 242, 184, 66, 14}
    },
    { {225, 252, 146, 40, 13},
      {69, 228, 115, 242, 8},
      {177, 68, 147, 248, 7},
      {20, 252, 226, 122, 2}
    },
    { {248, 166, 168, 212, 10},
      {11, 152, 189, 170, 6},
      {82, 177, 86, 81, 15},
      {101, 91, 209, 157, 0}
    },
    { {181, 57, 2, 161, 8},
      {193, 74, 35, 70, 10},
      {24, 84, 9, 202, 13},
      {86, 44, 69, 40, 3}
    },
    { {177, 99, 71, 94, 14},
      {57, 127, 233, 74, 1},
      {119, 174, 44, 104, 13},
      {133, 41, 127, 233, 12}
    },
    { {173, 47, 90, 242, 9},
      {167, 66, 175, 122, 11},
      {148, 245, 175, 75, 5},
      {213, 239, 84, 46, 5}
    },
    { {170, 181, 220, 133, 0},
      {107, 131, 52, 55, 2},
      {10, 19, 186, 213, 5},
      {78, 194, 204, 29, 6}
    },
    { {179, 236, 37, 120, 6},
      {17, 61, 242, 107, 12},
      {97, 234, 67, 124, 13},
      {61, 100, 251, 200, 8}
    },
    { {71, 153, 196, 64, 5},
      {228, 131, 80, 201, 8},
      {160, 34, 57, 158, 2},
      {25, 48, 172, 18, 7}
    },
    { {139, 198, 141, 192, 4},
      {18, 133, 92, 107, 2},
      {32, 59, 22, 61, 1},
      {77, 99, 170, 20, 8}
    },
    { {112, 47, 21, 221, 14},
      {25, 59, 233, 188, 10},
      {123, 186, 143, 64, 14},
      {83, 217, 125, 201, 8}
    },
    { {180, 34, 147, 218, 9},
      {149, 232, 169, 26, 3},
      {149, 188, 148, 66, 13},
      {197, 137, 81, 122, 9}
    },
    { {26, 202, 208, 56, 2},
      {34, 188, 154, 17, 8},
      {65, 192, 181, 53, 8},
      {24, 133, 147, 212, 4}
    },
    { {129, 51, 99, 151, 1},
      {125, 66, 136, 70, 7},
      {142, 156, 108, 200, 1},
      {230, 33, 20, 43, 14}
    },
    { {14, 105, 42, 215, 0},
      {139, 70, 132, 13, 15},
      {14, 181, 73, 103, 0},
      {251, 2, 22, 45, 1}
    },
    { {138, 208, 92, 91, 9},
      {102, 37, 149, 31, 1},
      {157, 163, 160, 181, 1},
      {143, 138, 154, 70, 6}
    },
    { {133, 230, 84, 193, 11},
      {165, 21, 25, 126, 2},
      {216, 50, 166, 122, 1},
      {71, 233, 138, 138, 5}
    },
    { {61, 10, 194, 179, 11},
      {166, 216, 171, 68, 11},
      {220, 212, 53, 11, 12},
      {210, 45, 81, 182, 5}
    },
    { {75, 191, 160, 85, 2},
      {75, 146, 152, 237, 12},
      {74, 160, 95, 221, 2},
      {59, 113, 148, 157, 2}
    },
    { {114, 45, 222, 48, 6},
      {33, 219, 230, 177, 8},
      {96, 199, 187, 68, 14},
      {24, 214, 125, 184, 4}
    },
    { {54, 157, 26, 205, 2},
      {200, 122, 52, 61, 10},
      {75, 53, 139, 150, 12},
      {91, 194, 197, 225, 3}
    },
    { {162, 217, 148, 230, 0},
      {72, 135, 50, 27, 11},
      {6, 114, 153, 180, 5},
      {221, 132, 206, 17, 2}
    },
    { {95, 173, 145, 57, 13},
      {151, 170, 211, 245, 8},
      {185, 200, 155, 95, 10},
      {26, 252, 181, 94, 9}
    },
    { {87, 83, 90, 233, 6},
      {224, 126, 78, 221, 2},
      {105, 117, 172, 174, 10},
      {75, 183, 39, 224, 7}
    },
    { {95, 94, 241, 212, 6},
      {250, 156, 200, 249, 14},
      {98, 184, 247, 175, 10},
      {121, 241, 51, 149, 15}
    },
    { {235, 195, 230, 172, 12},
      {42, 231, 123, 195, 6},
      {51, 86, 124, 61, 7},
      {108, 61, 238, 117, 4}
    },
    { {10, 153, 200, 53, 13},
      {110, 130, 215, 5, 8},
      {186, 193, 57, 149, 0},
      {26, 14, 180, 23, 6}
    },
    { {84, 99, 228, 154, 5},
      {165, 175, 200, 128, 7},
      {165, 146, 124, 98, 10},
      {224, 17, 63, 90, 5}
    },
    { {37, 103, 104, 147, 10},
      {161, 22, 173, 100, 7},
      {92, 145, 110, 106, 4},
      {226, 107, 86, 136, 5}
    },
    { {171, 237, 55, 6, 4},
      {27, 71, 112, 115, 13},
      {38, 14, 203, 125, 5},
      {188, 224, 238, 45, 8}
    },
    { {89, 189, 222, 217, 4},
      {99, 235, 212, 252, 10},
      {41, 183, 187, 217, 10},
      {83, 242, 189, 124, 6}
    },
    { {100, 250, 4, 192, 9},
      {197, 5, 57, 136, 10},
      {144, 50, 5, 242, 6},
      {81, 25, 202, 10, 3}
    },
    { {37, 14, 68, 231, 5},
      {172, 1, 106, 108, 11},
      {174, 114, 39, 10, 4},
      {211, 101, 104, 3, 5}
    },
    { {229, 138, 55, 75, 12},
      {144, 97, 121, 222, 13},
      {61, 46, 197, 26, 7},
      {183, 185, 232, 96, 9}
    },
    { {36, 202, 165, 116, 1},
      {156, 133, 186, 8, 12},
      {130, 234, 85, 50, 4},
      {49, 5, 218, 19, 9}
    },
    { {228, 44, 246, 88, 8},
      {161, 225, 161, 186, 12},
      {17, 166, 243, 66, 7},
      {53, 216, 88, 120, 5}
    },
    { {110, 162, 56, 83, 8},
      {131, 0, 189, 157, 5},
      {28, 161, 196, 87, 6},
      {171, 155, 208, 12, 1}
    },
    { {247, 88, 19, 2, 5},
      {212, 76, 96, 211, 9},
      {164, 12, 129, 174, 15},
      {156, 176, 99, 34, 11}
    },
    { {12, 235, 51, 187, 13},
      {151, 102, 219, 20, 15},
      {189, 220, 205, 115, 0},
      {242, 141, 182, 110, 9}
    },
    { {120, 106, 232, 103, 0},
      {43, 140, 46, 140, 13},
      {14, 97, 117, 97, 14},
      {179, 23, 67, 29, 4}
    },
    { {240, 222, 140, 150, 9},
      {76, 141, 189, 162, 11},
      {150, 147, 23, 176, 15},
      {212, 91, 219, 19, 2}
    },
    { {38, 200, 30, 254, 6},
      {136, 117, 246, 25, 11},
      {103, 247, 129, 54, 4},
      {217, 134, 250, 225, 1}
    },
    { {105, 244, 137, 8, 6},
      {83, 180, 116, 224, 0},
      {97, 9, 18, 249, 6},
      {0, 114, 226, 220, 10}
    },
    { {63, 191, 146, 164, 13},
      {207, 202, 123, 113, 10},
      {178, 84, 159, 223, 12},
      {88, 237, 229, 63, 3}
    },
    { {214, 85, 90, 118, 0},
      {232, 78, 134, 187, 1},
      {6, 229, 170, 166, 11},
      {141, 214, 23, 33, 7}
    },
    { {156, 202, 226, 54, 13},
      {174, 204, 219, 2, 13},
      {182, 196, 117, 51, 9},
      {180, 13, 179, 55, 5}
    },
    { {178, 86, 193, 106, 3},
      {116, 188, 42, 43, 1},
      {197, 104, 54, 164, 13},
      {141, 69, 67, 210, 14}
    },
    { {215, 138, 199, 34, 1},
      {180, 201, 26, 195, 9},
      {132, 78, 53, 30, 11},
      {156, 53, 137, 50, 13}
    },
    { {45, 193, 254, 38, 2},
      {170, 215, 54, 80, 5},
      {70, 71, 248, 59, 4},
      {160, 166, 206, 181, 5}
    },
    { {175, 32, 81, 9, 0},
      {179, 32, 32, 87, 0},
      {9, 8, 160, 79, 5},
      {14, 160, 64, 76, 13}
    },
    { {147, 44, 8, 175, 14},
      {9, 56, 71, 103, 11},
      {127, 81, 3, 76, 9},
      {222, 110, 33, 201, 0}
    },
    { {83, 254, 202, 119, 5},
      {109, 204, 222, 237, 9},
      {174, 229, 55, 252, 10},
      {155, 119, 179, 59, 6}
    },
    { {93, 41, 36, 151, 10},
      {139, 27, 129, 196, 15},
      {94, 146, 73, 75, 10},
      {242, 56, 29, 141, 1}
    },
    { {47, 16, 26, 34, 11},
      {198, 80, 39, 81, 1},
      {212, 69, 128, 143, 4},
      {136, 174, 64, 166, 3}
    },
    { {198, 173, 22, 177, 4},
      {129, 67, 210, 183, 10},
      {40, 214, 139, 86, 3},
      {94, 212, 188, 40, 1}
    },
    { {210, 236, 164, 178, 0},
      {1, 141, 146, 163, 15},
      {4, 210, 83, 116, 11},
      {252, 84, 155, 24, 0}
    },
    { {166, 250, 96, 61, 4},
      {233, 36, 250, 7, 12},
      {43, 192, 101, 246, 5},
      {62, 5, 242, 73, 7}
    },
    { {48, 87, 195, 182, 9},
      {124, 206, 171, 32, 3},
      {150, 220, 62, 160, 12},
      {192, 77, 87, 51, 14}
    },
    { {232, 23, 181, 77, 2},
      {90, 179, 40, 190, 4},
      {75, 42, 222, 129, 7},
      {39, 209, 76, 213, 10}
    },
    { {64, 171, 252, 62, 0},
      {41, 163, 158, 144, 13},
      {7, 195, 253, 80, 2},
      {176, 151, 156, 89, 4}
    },
    { {245, 58, 209, 110, 7},
      {253, 184, 106, 218, 9},
      {231, 104, 181, 202, 15},
      {149, 181, 97, 219, 15}
    },
    { {54, 243, 160, 110, 8},
      {201, 174, 59, 9, 5},
      {23, 96, 92, 246, 12},
      {169, 13, 199, 89, 3}
    },
    { {39, 120, 46, 65, 8},
      {193, 69, 37, 77, 12},
      {24, 39, 65, 238, 4},
      {59, 42, 74, 40, 3}
    },
    { {62, 144, 98, 20, 6},
      {234, 88, 240, 1, 4},
      {98, 132, 96, 151, 12},
      {40, 0, 241, 165, 7}
    },
    { {1, 253, 122, 212, 14},
      {105, 86, 213, 120, 14},
      {114, 181, 235, 248, 0},
      {113, 234, 182, 169, 6}
    },
    { {200, 118, 91, 198, 4},
      {123, 68, 76, 186, 3},
      {38, 61, 166, 225, 3},
      {197, 211, 34, 45, 14}
    },
    { {173, 61, 130, 21, 15},
      {207, 210, 225, 102, 8},
      {250, 132, 27, 203, 5},
      {22, 104, 116, 191, 3}
    },
    { {150, 49, 122, 39, 14},
      {233, 90, 71, 23, 5},
      {126, 69, 232, 198, 9},
      {174, 142, 37, 169, 7}
    },
    { {5, 248, 19, 202, 6},
      {209, 116, 80, 88, 11},
      {101, 60, 129, 250, 0},
      {209, 160, 162, 232, 11}
    },
    { {14, 122, 220, 34, 14},
      {227, 149, 79, 17, 9},
      {116, 67, 181, 231, 0},
      {152, 143, 42, 156, 7}
    },
    { {191, 82, 196, 165, 0},
      {234, 141, 42, 71, 2},
      {10, 82, 52, 175, 13},
      {78, 37, 75, 21, 7}
    },
    { {114, 79, 106, 226, 14},
      {32, 94, 111, 169, 15},
      {116, 117, 111, 36, 14},
      {249, 95, 103, 160, 4}
    },
    { {84, 15, 215, 234, 8},
      {176, 235, 11, 184, 11},
      {21, 126, 191, 2, 10},
      {209, 221, 13, 112, 13}
    },
    { {156, 150, 47, 116, 1},
      {222, 73, 158, 42, 4},
      {130, 239, 70, 147, 9},
      {37, 71, 153, 39, 11}
    },
    { {38, 64, 66, 176, 1},
      {164, 68, 162, 1, 2},
      {128, 212, 32, 38, 4},
      {72, 4, 82, 34, 5}
    },
    { {25, 124, 183, 117, 4},
      {91, 205, 194, 124, 12},
      {42, 238, 211, 233, 8},
      {51, 228, 59, 61, 10}
    },
    { {31, 43, 184, 80, 13},
      {135, 138, 205, 89, 12},
      {176, 161, 221, 79, 8},
      {57, 171, 53, 30, 1}
    },
    { {200, 125, 108, 42, 13},
      {103, 39, 71, 162, 13},
      {181, 67, 107, 225, 3},
      {180, 94, 46, 78, 6}
    },
    { {220, 141, 45, 237, 14},
      {154, 59, 87, 174, 14},
      {123, 123, 75, 19, 11},
      {119, 94, 173, 197, 9}
    },
    { {203, 107, 1, 115, 8},
      {19, 6, 139, 207, 9},
      {28, 232, 13, 109, 3},
      {159, 61, 22, 12, 8}
    },
    { {190, 66, 16, 54, 8},
      {138, 12, 171, 19, 1},
      {22, 192, 132, 39, 13},
      {140, 141, 83, 5, 1}
    },
    { {187, 70, 34, 167, 7},
      {14, 92, 106, 103, 7},
      {238, 84, 70, 45, 13},
      {238, 101, 99, 167, 0}
    },
    { {115, 50, 38, 239, 8},
      {73, 105, 43, 205, 7},
      {31, 118, 68, 204, 14},
      {235, 61, 73, 105, 2}
    },
    { {73, 51, 173, 227, 9},
      {87, 131, 15, 204, 7},
      {156, 123, 92, 201, 2},
      {227, 63, 12, 30, 10}
    },
    { {31, 28, 137, 108, 7},
      {222, 184, 70, 105, 8},
      {227, 105, 19, 143, 8},
      {25, 102, 33, 215, 11}
    },
    { {137, 118, 101, 171, 10},
      {115, 53, 11, 102, 7},
      {93, 90, 102, 233, 1},
      {230, 109, 10, 204, 14}
    },
    { {146, 244, 107, 161, 2},
      {113, 92, 22, 39, 6},
      {72, 93, 98, 244, 9},
      {110, 70, 131, 168, 14}
    },
    { {41, 121, 129, 118, 7},
      {95, 150, 226, 72, 9},
      {230, 232, 25, 233, 4},
      {145, 36, 118, 159, 10}
    },
    { {173, 112, 44, 248, 13},
      {199, 37, 231, 74, 6},
      {177, 243, 64, 235, 5},
      {101, 46, 122, 78, 3}
    },
    { {71, 49, 152, 185, 0},
      {193, 162, 134, 213, 2},
      {9, 209, 152, 206, 2},
      {74, 182, 20, 88, 3}
    },
    { {236, 46, 142, 6, 11},
      {143, 209, 45, 162, 9},
      {214, 7, 23, 67, 7},
      {148, 91, 72, 191, 1}
    },
    { {54, 161, 214, 152, 0},
      {161, 235, 176, 17, 2},
      {1, 150, 184, 86, 12},
      {72, 128, 221, 120, 5}
    },
    { {165, 206, 134, 80, 15},
      {132, 213, 249, 106, 8},
      {240, 166, 23, 58, 5},
      {21, 105, 250, 178, 1}
    },
    { {99, 120, 112, 52, 8},
      {105, 4, 163, 209, 12},
      {18, 192, 225, 236, 6},
      {56, 188, 82, 9, 6}
    },
    { {243, 171, 65, 212, 6},
      {57, 26, 248, 203, 10},
      {98, 184, 45, 92, 15},
      {93, 49, 245, 137, 12}
    },
    { {230, 3, 42, 139, 5},
      {132, 98, 108, 135, 7},
      {173, 21, 76, 6, 7},
      {238, 19, 100, 98, 1}
    },
    { {76, 17, 238, 199, 3},
      {238, 211, 4, 140, 7},
      {206, 55, 120, 131, 2},
      {227, 18, 12, 183, 7}
    },
    { {38, 29, 229, 13, 6},
      {248, 179, 96, 37, 12},
      {107, 10, 123, 134, 4},
      {58, 64, 108, 209, 15}
    },
    { {41, 102, 22, 227, 2},
      {3, 85, 42, 124, 3},
      {76, 118, 134, 105, 4},
      {195, 229, 74, 172, 0}
    },
    { {55, 207, 241, 160, 10},
      {176, 158, 59, 113, 14},
      {80, 88, 255, 62, 12},
      {120, 237, 199, 144, 13}
    },
    { {196, 87, 132, 122, 11},
      {196, 183, 139, 170, 1},
      {213, 226, 30, 162, 3},
      {133, 93, 30, 210, 3}
    },
    { {119, 110, 82, 255, 9},
      {173, 108, 171, 253, 11},
      {159, 244, 167, 110, 14},
      {219, 253, 83, 107, 5}
    },
    { {152, 8, 25, 187, 9},
      {22, 40, 135, 22, 11},
      {157, 217, 129, 1, 9},
      {214, 142, 17, 70, 8}
    },
    { {80, 203, 43, 93, 8},
      {24, 110, 157, 140, 12},
      {27, 173, 77, 48, 10},
      {51, 27, 151, 97, 8}
    },
    { {139, 105, 24, 6, 3},
      {15, 22, 4, 83, 9},
      {198, 1, 137, 109, 1},
      {156, 162, 6, 143, 0}
    },
    { {18, 217, 194, 206, 9},
      {108, 238, 17, 9, 11},
      {151, 52, 57, 180, 8},
      {217, 8, 135, 115, 6}
    },
    { {21, 140, 46, 115, 8},
      {128, 73, 151, 108, 13},
      {28, 231, 67, 26, 8},
      {179, 110, 153, 32, 1}
    },
    { {107, 132, 1, 94, 10},
      {26, 48, 177, 233, 1},
      {87, 168, 2, 29, 6},
      {137, 120, 208, 197, 8}
    },
    { {103, 105, 93, 185, 15},
      {181, 55, 231, 213, 10},
      {249, 219, 169, 110, 6},
      {90, 190, 126, 202, 13}
    },
    { {62, 114, 116, 92, 1},
      {239, 45, 168, 25, 4},
      {131, 162, 228, 231, 12},
      {41, 129, 91, 79, 7}
    },
    { {1, 202, 26, 87, 1},
      {12, 68, 156, 92, 9},
      {142, 165, 133, 56, 0},
      {147, 163, 146, 35, 0}
    },
    { {34, 231, 3, 111, 5},
      {29, 102, 122, 45, 1},
      {175, 108, 14, 116, 4},
      {139, 69, 230, 107, 8}
    },
    { {213, 220, 237, 174, 1},
      {252, 173, 22, 226, 15},
      {135, 91, 115, 186, 11},
      {244, 118, 139, 83, 15}
    },
    { {128, 149, 50, 197, 11},
      {76, 82, 17, 62, 6},
      {218, 52, 202, 144, 1},
      {103, 200, 132, 163, 2}
    },
    { {247, 228, 233, 213, 4},
      {185, 140, 244, 239, 6},
      {42, 185, 114, 126, 15},
      {111, 114, 243, 25, 13}
    },
    { {89, 216, 79, 156, 11},
      {126, 125, 149, 192, 10},
      {211, 159, 33, 185, 10},
      {80, 58, 155, 231, 14}
    },
    { {244, 46, 208, 133, 1},
      {173, 136, 40, 182, 10},
      {138, 16, 183, 66, 15},
      {86, 209, 65, 27, 5}
    },
    { {93, 180, 169, 31, 10},
      {219, 184, 149, 228, 5},
      {95, 137, 82, 219, 10},
      {162, 122, 145, 221, 11}
    },
    { {13, 88, 229, 177, 15},
      {246, 149, 195, 68, 14},
      {248, 218, 113, 171, 0},
      {114, 44, 58, 150, 15}
    },
    { {105, 251, 159, 164, 3},
      {95, 215, 62, 208, 10},
      {194, 95, 157, 249, 6},
      {80, 183, 206, 191, 10}
    },
    { {41, 186, 81, 246, 0},
      {123, 0, 186, 88, 11},
      {6, 248, 165, 217, 4},
      {209, 165, 208, 13, 14}
    },
    { {88, 124, 103, 66, 7},
      {119, 93, 64, 168, 13},
      {228, 46, 99, 225, 10},
      {177, 80, 43, 174, 14}
    },
    { {216, 185, 2, 30, 13},
      {79, 106, 209, 130, 9},
      {183, 132, 9, 209, 11},
      {148, 24, 181, 111, 2}
    },
    { {187, 89, 68, 73, 14},
      {98, 63, 97, 79, 8},
      {121, 34, 41, 173, 13},
      {31, 40, 111, 196, 6}
    },
    { {170, 35, 31, 53, 10},
      {27, 83, 175, 23, 0},
      {90, 207, 140, 69, 5},
      {14, 143, 92, 173, 8}
    },
    { {13, 149, 148, 63, 4},
      {202, 163, 210, 116, 1},
      {47, 194, 154, 155, 0},
      {130, 228, 188, 85, 3}
    },
    { {222, 196, 97, 37, 7},
      {190, 28, 82, 167, 4},
      {234, 72, 98, 55, 11},
      {46, 84, 163, 135, 13}
    },
    { {5, 161, 105, 48, 13},
      {181, 2, 215, 64, 4},
      {176, 201, 104, 90, 0},
      {32, 46, 180, 10, 13}
    },
    { {226, 81, 46, 181, 8},
      {72, 71, 167, 135, 6},
      {26, 215, 72, 164, 7},
      {110, 30, 94, 33, 2}
    },
    { {171, 72, 226, 220, 0},
      {42, 228, 160, 75, 14},
      {3, 180, 113, 45, 5},
      {125, 32, 82, 117, 4}
    },
    { {12, 182, 132, 148, 15},
      {207, 145, 217, 32, 2},
      {242, 146, 22, 211, 0},
      {64, 73, 184, 159, 3}
    },
    { {146, 122, 107, 91, 1},
      {117, 108, 140, 15, 13},
      {141, 173, 101, 228, 9},
      {191, 3, 19, 106, 14}
    },
    { {187, 144, 243, 69, 0},
      {122, 200, 48, 95, 4},
      {10, 44, 240, 157, 13},
      {47, 160, 193, 53, 14}
    },
    { {99, 46, 143, 173, 0},
      {25, 225, 46, 229, 10},
      {11, 95, 23, 76, 6},
      {90, 119, 72, 121, 8}
    },
    { {194, 238, 175, 42, 7},
      {21, 245, 94, 163, 13},
      {229, 79, 87, 116, 3},
      {188, 87, 170, 250, 8}
    },
    { {30, 178, 175, 57, 13},
      {215, 233, 223, 5, 4},
      {185, 207, 84, 215, 8},
      {42, 15, 185, 126, 11}
    },
    { {81, 21, 72, 197, 2},
      {104, 26, 4, 236, 2},
      {74, 49, 42, 136, 10},
      {67, 114, 5, 129, 6}
    },
    { {70, 147, 213, 222, 4},
      {248, 163, 216, 153, 3},
      {39, 186, 188, 150, 2},
      {201, 145, 188, 81, 15}
    },
    { {136, 142, 145, 97, 13},
      {22, 128, 91, 62, 8},
      {184, 104, 151, 17, 1},
      {23, 205, 160, 22, 8}
    },
    { {104, 131, 111, 244, 15},
      {62, 83, 255, 136, 6},
      {242, 255, 108, 17, 6},
      {97, 31, 252, 167, 12}
    },
    { {81, 189, 46, 68, 1},
      {77, 75, 20, 232, 12},
      {130, 39, 75, 216, 10},
      {49, 114, 141, 43, 2}
    },
    { {122, 10, 196, 14, 9},
      {46, 169, 41, 129, 9},
      {151, 2, 53, 5, 14},
      {152, 25, 73, 87, 4}
    },
    { {139, 183, 98, 180, 10},
      {107, 82, 155, 99, 6},
      {82, 212, 110, 221, 1},
      {108, 109, 148, 173, 6}
    },
    { {28, 175, 131, 161, 10},
      {147, 218, 27, 36, 10},
      {88, 92, 31, 83, 8},
      {82, 77, 133, 188, 9}
    },
    { {194, 112, 11, 187, 13},
      {85, 100, 199, 135, 3},
      {189, 221, 0, 228, 3},
      {206, 30, 50, 106, 10}
    },
    { {251, 46, 208, 154, 10},
      {35, 184, 169, 243, 11},
      {85, 144, 183, 77, 15},
      {220, 249, 81, 220, 4}
    },
    { {248, 82, 2, 148, 14},
      {74, 92, 233, 130, 2},
      {114, 148, 4, 161, 15},
      {68, 25, 115, 165, 2}
    },
    { {122, 161, 4, 202, 6},
      {3, 59, 112, 137, 3},
      {101, 50, 8, 85, 14},
      {201, 16, 237, 204, 0}
    },
    { {106, 150, 210, 188, 11},
      {110, 240, 187, 177, 2},
      {211, 212, 182, 149, 6},
      {72, 221, 208, 247, 6}
    },
    { {40, 242, 236, 117, 14},
      {107, 149, 255, 12, 4},
      {122, 227, 116, 241, 4},
      {35, 15, 250, 157, 6}
    },
    { {96, 112, 101, 235, 4},
      {113, 37, 98, 140, 7},
      {45, 122, 96, 224, 6},
      {227, 20, 106, 72, 14}
    },
    { {122, 217, 179, 168, 12},
      {82, 238, 115, 145, 14},
      {49, 92, 217, 181, 14},
      {120, 156, 231, 116, 10}
    },
    { {53, 114, 214, 203, 15},
      {229, 253, 105, 92, 3},
      {253, 54, 180, 234, 12},
      {195, 169, 107, 250, 7}
    },
    { {212, 38, 239, 184, 15},
      {181, 249, 207, 162, 6},
      {241, 223, 118, 66, 11},
      {100, 95, 57, 250, 13}
    },
    { {100, 238, 77, 186, 10},
      {177, 53, 191, 160, 11},
      {85, 219, 39, 114, 6},
      {208, 95, 218, 200, 13}
    },
    { {152, 61, 122, 143, 1},
      {111, 106, 4, 54, 15},
      {143, 21, 235, 193, 9},
      {246, 194, 5, 111, 6}
    },
    { {140, 163, 31, 73, 9},
      {151, 99, 29, 30, 0},
      {153, 47, 140, 83, 1},
      {7, 139, 140, 110, 9}
    },
    { {2, 84, 87, 180, 6},
      {120, 85, 194, 49, 2},
      {98, 222, 162, 164, 0},
      {72, 196, 58, 161, 14}
    },
    { {213, 58, 228, 20, 9},
      {237, 137, 137, 194, 12},
      {146, 130, 117, 202, 11},
      {52, 57, 25, 27, 7}
    },
    { {157, 60, 100, 57, 6},
      {227, 57, 194, 102, 12},
      {105, 194, 99, 203, 9},
      {54, 100, 57, 204, 7}
    },
    { {121, 83, 50, 238, 5},
      {78, 110, 106, 216, 7},
      {167, 116, 204, 169, 14},
      {225, 181, 103, 103, 2}
    },
    { {112, 136, 77, 102, 4},
      {56, 9, 118, 136, 9},
      {38, 107, 33, 16, 14},
      {145, 22, 233, 1, 12}
    },
    { {29, 211, 82, 141, 8},
      {234, 110, 25, 84, 2},
      {27, 20, 172, 187, 8},
      {66, 169, 135, 101, 7}
    },
    { {216, 229, 29, 36, 6},
      {27, 31, 86, 178, 0},
      {98, 75, 138, 113, 11},
      {4, 214, 175, 141, 8}
    },
    { {214, 2, 156, 163, 8},
      {128, 137, 15, 151, 3},
      {28, 83, 148, 6, 11},
      {206, 159, 9, 16, 1}
    },
    { {217, 74, 182, 16, 12},
      {2, 205, 201, 210, 12},
      {48, 134, 213, 41, 11},
      {52, 185, 59, 52, 0}
    },
    { {118, 107, 184, 136, 0},
      {129, 174, 44, 145, 14},
      {1, 17, 221, 102, 14},
      {120, 147, 71, 88, 1}
    },
    { {253, 12, 107, 49, 9},
      {182, 72, 167, 230, 12},
      {152, 205, 99, 11, 15},
      {54, 126, 81, 38, 13}
    },
    { {59, 255, 139, 157, 1},
      {95, 238, 188, 101, 10},
      {139, 157, 31, 253, 12},
      {90, 99, 215, 127, 10}
    },
    { {231, 134, 244, 212, 14},
      {168, 145, 249, 251, 6},
      {114, 178, 246, 30, 7},
      {109, 249, 248, 145, 5}
    },
    { {212, 230, 185, 183, 9},
      {157, 140, 159, 182, 7},
      {158, 217, 214, 114, 11},
      {230, 223, 147, 27, 9}
    },
    { {207, 168, 187, 54, 9},
      {159, 192, 151, 211, 13},
      {150, 205, 209, 95, 3},
      {188, 190, 144, 63, 9}
    },
    { {32, 164, 54, 108, 6},
      {9, 113, 114, 56, 4},
      {99, 102, 194, 80, 4},
      {33, 196, 232, 233, 0}
    },
    { {92, 239, 185, 83, 6},
      {147, 158, 220, 188, 13},
      {108, 169, 223, 115, 10},
      {179, 211, 183, 156, 9}
    },
    { {15, 6, 182, 96, 5},
      {134, 193, 74, 121, 4},
      {160, 102, 214, 15, 0},
      {41, 229, 40, 54, 1}
    },
    { {187, 81, 210, 94, 7},
      {110, 254, 224, 91, 1},
      {231, 164, 184, 173, 13},
      {141, 160, 119, 247, 6}
    },
    { {247, 226, 98, 100, 1},
      {173, 76, 58, 203, 4},
      {130, 100, 100, 126, 15},
      {45, 53, 195, 43, 5}
    },
    { {129, 253, 169, 133, 5},
      {93, 134, 84, 102, 14},
      {170, 25, 91, 248, 1},
      {118, 98, 166, 27, 10}
    },
    { {34, 65, 219, 197, 9},
      {60, 198, 37, 29, 2},
      {154, 61, 184, 36, 4},
      {75, 138, 70, 51, 12}
    },
    { {86, 251, 75, 40, 0},
      {241, 110, 30, 129, 8},
      {1, 77, 45, 246, 10},
      {24, 23, 135, 104, 15}
    },
    { {237, 166, 76, 109, 0},
      {171, 33, 62, 238, 0},
      {11, 99, 38, 91, 7},
      {7, 119, 200, 77, 5}
    },
    { {206, 123, 8, 15, 4},
      {203, 38, 76, 135, 9},
      {47, 1, 13, 231, 3},
      {158, 19, 38, 77, 3}
    },
    { {97, 146, 238, 126, 9},
      {108, 225, 191, 200, 5},
      {151, 231, 116, 152, 6},
      {161, 63, 216, 115, 6}
    },
    { {148, 63, 245, 18, 4},
      {241, 139, 200, 50, 13},
      {36, 138, 255, 194, 9},
      {180, 193, 61, 24, 15}
    },
    { {136, 179, 190, 173, 15},
      {79, 243, 95, 22, 6},
      {251, 87, 220, 209, 1},
      {102, 143, 172, 255, 2}
    },
    { {155, 39, 229, 141, 0},
      {59, 171, 8, 103, 6},
      {11, 26, 126, 77, 9},
      {110, 97, 13, 93, 12}
    },
    { {190, 11, 69, 253, 2},
      {186, 59, 170, 15, 10},
      {75, 250, 45, 7, 13},
      {95, 5, 93, 197, 13}
    },
    { {105, 251, 58, 114, 10},
      {67, 86, 191, 216, 13},
      {84, 229, 205, 249, 6},
      {177, 191, 214, 172, 2}
    },
    { {97, 56, 58, 131, 2},
      {65, 80, 36, 212, 15},
      {76, 21, 193, 200, 6},
      {242, 178, 64, 168, 2}
    },
    { {102, 136, 99, 155, 3},
      {180, 112, 176, 133, 15},
      {205, 156, 97, 22, 6},
      {250, 16, 208, 226, 13}
    },
    { {84, 208, 28, 153, 2},
      {192, 61, 148, 148, 2},
      {73, 147, 128, 178, 10},
      {66, 146, 155, 192, 3}
    },
    { {144, 177, 12, 2, 10},
      {65, 27, 21, 2, 1},
      {84, 3, 8, 208, 9},
      {132, 10, 141, 136, 2}
    },
    { {246, 103, 252, 109, 1},
      {173, 175, 46, 191, 4},
      {139, 99, 254, 102, 15},
      {47, 215, 79, 91, 5}
    },
    { {122, 205, 117, 129, 3},
      {54, 31, 48, 181, 14},
      {200, 26, 235, 53, 14},
      {122, 208, 207, 134, 12}
    },
    { {58, 189, 7, 246, 1},
      {95, 75, 178, 41, 11},
      {134, 254, 11, 213, 12},
      {217, 68, 221, 47, 10}
    },
    { {84, 48, 18, 71, 3},
      {205, 88, 0, 156, 1},
      {206, 36, 128, 194, 10},
      {131, 144, 1, 171, 3}
    },
    { {6, 119, 148, 213, 8},
      {201, 135, 137, 61, 2},
      {26, 178, 158, 230, 0},
      {75, 201, 30, 25, 3}
    },
    { {168, 85, 177, 32, 1},
      {86, 134, 34, 50, 4},
      {128, 72, 218, 161, 5},
      {36, 196, 70, 22, 10}
    },
    { {156, 180, 215, 253, 11},
      {255, 249, 147, 62, 2},
      {219, 254, 178, 211, 9},
      {71, 204, 153, 255, 15}
    },
    { {49, 161, 147, 230, 10},
      {25, 218, 51, 88, 3},
      {86, 124, 152, 88, 12},
      {193, 172, 197, 185, 8}
    },
    { {61, 163, 183, 109, 4},
      {155, 235, 122, 92, 4},
      {43, 110, 220, 91, 12},
      {35, 165, 237, 125, 9}
    },
    { {65, 83, 241, 224, 4},
      {112, 134, 74, 216, 6},
      {32, 120, 252, 168, 2},
      {97, 181, 38, 16, 14}
    },
    { {85, 68, 51, 91, 5},
      {148, 108, 192, 252, 5},
      {173, 172, 194, 42, 10},
      {163, 240, 51, 98, 9}
    },
    { {146, 246, 18, 73, 13},
      {69, 108, 89, 63, 0},
      {185, 36, 134, 244, 9},
      {15, 201, 163, 106, 2}
    },
    { {38, 154, 53, 248, 10},
      {208, 49, 187, 25, 14},
      {81, 250, 197, 150, 4},
      {121, 141, 216, 192, 11}
    },
    { {4, 38, 119, 137, 6},
      {177, 113, 72, 52, 6},
      {105, 30, 230, 66, 0},
      {98, 193, 40, 232, 13}
    },
    { {148, 135, 205, 198, 15},
      {188, 155, 93, 42, 3},
      {246, 59, 62, 18, 9},
      {197, 75, 173, 147, 13}
    },
    { {78, 167, 231, 217, 12},
      {179, 227, 217, 173, 6},
      {57, 190, 126, 87, 2},
      {107, 89, 188, 124, 13}
    },
    { {46, 54, 115, 69, 11},
      {255, 80, 41, 61, 4},
      {218, 44, 230, 199, 4},
      {43, 201, 64, 175, 15}
    },
    { {58, 94, 143, 48, 4},
      {82, 205, 238, 33, 8},
      {32, 207, 23, 165, 12},
      {24, 71, 123, 52, 10}
    },
    { {118, 217, 250, 167, 7},
      {236, 222, 118, 149, 15},
      {238, 85, 249, 182, 14},
      {250, 150, 231, 179, 7}
    },
    { {100, 228, 234, 29, 15},
      {173, 244, 245, 164, 4},
      {251, 133, 114, 114, 6},
      {34, 90, 242, 251, 5}
    },
    { {18, 167, 180, 52, 13},
      {13, 139, 219, 49, 4},
      {178, 194, 222, 84, 8},
      {40, 205, 189, 27, 0}
    },
    { {73, 115, 217, 85, 3},
      {127, 150, 140, 220, 0},
      {202, 169, 188, 233, 2},
      {3, 179, 22, 159, 14}
    },
    { {205, 180, 24, 151, 5},
      {207, 0, 212, 246, 3},
      {174, 145, 130, 219, 3},
      {198, 242, 176, 15, 3}
    },
    { {139, 218, 118, 210, 14},
      {98, 85, 217, 91, 15},
      {116, 182, 229, 189, 1},
      {253, 169, 186, 164, 6}
    },
    { {245, 74, 77, 236, 8},
      {184, 45, 47, 202, 10},
      {19, 123, 37, 42, 15},
      {85, 63, 75, 65, 13}
    },
    { {129, 22, 22, 248, 10},
      {64, 113, 139, 122, 2},
      {81, 246, 134, 136, 1},
      {69, 237, 24, 224, 2}
    },
    { {64, 54, 244, 230, 11},
      {109, 145, 11, 184, 7},
      {214, 114, 246, 192, 2},
      {225, 221, 8, 155, 6}
    },
    { {39, 54, 40, 42, 6},
      {193, 48, 110, 97, 5},
      {101, 65, 70, 206, 4},
      {168, 103, 96, 200, 3}
    },
    { {164, 19, 210, 41, 15},
      {228, 242, 107, 22, 0},
      {249, 68, 188, 130, 5},
      {6, 141, 100, 242, 7}
    },
    { {144, 31, 228, 193, 7},
      {100, 155, 72, 46, 14},
      {232, 50, 127, 128, 9},
      {119, 65, 45, 146, 6}
    },
    { {154, 191, 206, 86, 14},
      {107, 219, 221, 43, 9},
      {118, 167, 63, 213, 9},
      {157, 75, 189, 189, 6}
    },
    { {45, 50, 91, 158, 15},
      {255, 112, 237, 80, 3},
      {247, 157, 164, 203, 4},
      {192, 171, 112, 239, 15}
    },
    { {217, 144, 31, 106, 9},
      {86, 105, 23, 218, 1},
      {149, 111, 128, 153, 11},
      {133, 190, 137, 102, 10}
    },
    { {58, 166, 79, 173, 14},
      {59, 121, 127, 37, 2},
      {123, 95, 38, 85, 12},
      {74, 79, 233, 237, 12}
    },
    { {101, 40, 141, 96, 11},
      {149, 145, 39, 200, 8},
      {208, 107, 17, 74, 6},
      {17, 62, 72, 154, 9}
    },
    { {10, 83, 233, 96, 11},
      {118, 150, 15, 9, 4},
      {208, 105, 124, 165, 0},
      {41, 15, 6, 150, 14}
    },
    { {233, 224, 156, 62, 4},
      {11, 165, 246, 210, 1},
      {39, 195, 144, 121, 7},
      {132, 182, 250, 93, 0}
    },
    { {77, 155, 120, 99, 13},
      {230, 2, 95, 220, 13},
      {188, 97, 237, 155, 2},
      {179, 191, 164, 6, 7}
    },
    { {217, 139, 180, 131, 3},
      {6, 155, 24, 214, 15},
      {204, 18, 221, 25, 11},
      {246, 177, 141, 150, 0}
    },
    { {52, 177, 129, 237, 7},
      {221, 186, 114, 12, 2},
      {235, 120, 24, 210, 12},
      {67, 4, 229, 219, 11}
    },
    { {91, 55, 63, 73, 0},
      {83, 107, 12, 253, 4},
      {9, 47, 206, 205, 10},
      {43, 243, 13, 108, 10}
    },
    { {133, 114, 183, 185, 1},
      {213, 229, 138, 86, 6},
      {137, 222, 212, 234, 1},
      {102, 165, 26, 122, 11}
    },
    { {44, 93, 5, 106, 4},
      {210, 39, 98, 40, 9},
      {37, 106, 11, 163, 4},
      {145, 68, 110, 68, 11}
    },
    { {228, 103, 18, 150, 13},
      {141, 70, 233, 178, 3},
      {182, 148, 142, 98, 7},
      {196, 217, 118, 43, 1}
    },
    { {42, 8, 214, 247, 8},
      {42, 193, 163, 29, 11},
      {30, 246, 177, 5, 4},
      {219, 140, 88, 53, 4}
    },
    { {69, 41, 73, 198, 1},
      {189, 2, 4, 200, 11},
      {134, 57, 41, 74, 2},
      {209, 50, 4, 11, 13}
    },
    { {37, 1, 236, 91, 7},
      {164, 179, 228, 76, 5},
      {237, 163, 120, 10, 4},
      {163, 34, 124, 210, 5}
    },
    { {184, 245, 71, 121, 12},
      {115, 111, 243, 46, 0},
      {57, 238, 42, 241, 13},
      {7, 76, 255, 108, 14}
    },
    { {163, 183, 234, 75, 3},
      {101, 242, 60, 111, 5},
      {205, 37, 126, 220, 5},
      {175, 99, 196, 250, 6}
    },
    { {68, 61, 107, 155, 4},
      {241, 98, 196, 164, 15},
      {45, 157, 107, 194, 2},
      {242, 82, 52, 104, 15}
    },
    { {190, 45, 203, 132, 8},
      {187, 202, 37, 35, 10},
      {18, 29, 59, 71, 13},
      {92, 74, 69, 61, 13}
    },
    { {181, 31, 196, 126, 12},
      {232, 171, 235, 106, 9},
      {55, 226, 63, 138, 13},
      {149, 109, 125, 81, 7}
    },
    { {111, 56, 207, 152, 8},
      {243, 225, 165, 193, 10},
      {17, 159, 49, 207, 6},
      {88, 58, 88, 124, 15}
    },
    { {144, 242, 254, 147, 4},
      {97, 205, 220, 22, 7},
      {44, 151, 244, 240, 9},
      {230, 131, 187, 56, 6}
    },
    { {7, 253, 84, 10, 11},
      {229, 55, 17, 113, 9},
      {213, 2, 171, 254, 0},
      {152, 232, 142, 202, 7}
    },
    { {238, 83, 195, 208, 8},
      {242, 198, 169, 139, 2},
      {16, 188, 60, 167, 7},
      {77, 25, 86, 52, 15}
    },
    { {247, 9, 142, 188, 10},
      {136, 251, 167, 195, 10},
      {83, 215, 25, 14, 15},
      {92, 62, 93, 241, 1}
    },
    { {101, 70, 124, 48, 9},
      {164, 5, 175, 240, 4},
      {144, 195, 230, 42, 6},
      {32, 255, 90, 2, 5}
    },
    { {174, 111, 215, 151, 14},
      {187, 215, 233, 55, 11},
      {126, 158, 191, 103, 5},
      {222, 201, 126, 189, 13}
    },
    { {139, 196, 115, 158, 9},
      {62, 100, 145, 115, 7},
      {151, 156, 226, 61, 1},
      {236, 232, 146, 103, 12}
    },
    { {151, 25, 48, 254, 4},
      {200, 42, 194, 91, 15},
      {39, 240, 201, 142, 9},
      {253, 164, 53, 65, 3}
    },
    { {154, 117, 45, 245, 7},
      {95, 31, 198, 47, 6},
      {234, 251, 74, 229, 9},
      {111, 70, 63, 143, 10}
    },
    { {189, 71, 254, 83, 14},
      {162, 223, 237, 126, 5},
      {124, 167, 254, 43, 13},
      {167, 235, 127, 180, 5}
    },
    { {111, 21, 48, 146, 8},
      {194, 2, 161, 241, 7},
      {20, 144, 202, 143, 6},
      {232, 248, 84, 4, 3}
    },
    { {93, 225, 204, 72, 9},
      {167, 175, 21, 200, 0},
      {145, 35, 56, 123, 10},
      {1, 58, 143, 94, 5}
    },
    { {17, 28, 227, 220, 13},
      {124, 232, 193, 104, 14},
      {179, 188, 115, 136, 8},
      {113, 104, 49, 115, 14}
    },
    { {164, 199, 225, 87, 12},
      {184, 134, 249, 46, 5},
      {62, 168, 126, 50, 5},
      {167, 73, 246, 17, 13}
    },
    { {85, 38, 145, 17, 2},
      {145, 152, 136, 244, 0},
      {72, 136, 150, 74, 10},
      {2, 241, 17, 152, 9}
    },
    { {157, 101, 17, 128, 5},
      {151, 14, 64, 114, 2},
      {160, 24, 138, 107, 9},
      {68, 224, 39, 14, 9}
    },
    { {102, 99, 193, 203, 3},
      {181, 182, 40, 141, 3},
      {205, 56, 60, 102, 6},
      {203, 17, 70, 218, 13}
    },
    { {191, 197, 148, 0, 2},
      {130, 159, 48, 115, 0},
      {64, 2, 154, 63, 13},
      {12, 224, 207, 148, 1}
    },
    { {58, 53, 0, 56, 7},
      {71, 58, 226, 33, 0},
      {225, 192, 10, 197, 12},
      {8, 68, 117, 206, 2}
    },
    { {118, 212, 246, 121, 12},
      {224, 237, 243, 189, 4},
      {57, 230, 242, 182, 14},
      {43, 220, 251, 112, 7}
    },
    { {68, 111, 38, 103, 11},
      {141, 87, 11, 172, 13},
      {222, 102, 79, 98, 2},
      {179, 93, 14, 171, 1}
    },
    { {23, 174, 89, 233, 7},
      {181, 56, 94, 125, 10},
      {233, 121, 167, 94, 8},
      {91, 231, 161, 202, 13}
    },
    { {54, 19, 45, 99, 14},
      {208, 27, 111, 13, 5},
      {124, 107, 76, 134, 12},
      {171, 15, 109, 128, 11}
    },
    { {142, 7, 73, 43, 15},
      {182, 50, 79, 39, 1},
      {253, 73, 46, 7, 1},
      {142, 79, 36, 198, 13}
    },
    { {24, 138, 205, 223, 1},
      {62, 169, 156, 12, 11},
      {143, 187, 53, 17, 8},
      {211, 3, 153, 87, 12}
    },
    { {237, 144, 82, 208, 2},
      {226, 80, 176, 218, 2},
      {64, 180, 160, 155, 7},
      {69, 176, 208, 164, 7}
    },
    { {91, 107, 208, 31, 5},
      {47, 174, 200, 213, 9},
      {175, 128, 189, 109, 10},
      {154, 177, 55, 95, 4}
    },
    { {132, 164, 208, 112, 14},
      {161, 144, 211, 58, 0},
      {112, 224, 178, 82, 1},
      {5, 204, 176, 152, 5}
    },
    { {201, 178, 163, 175, 4},
      {91, 224, 90, 198, 7},
      {47, 92, 84, 217, 3},
      {230, 53, 160, 125, 10}
    },
    { {171, 24, 133, 47, 12},
      {90, 161, 99, 71, 9},
      {63, 74, 17, 141, 5},
      {158, 44, 104, 85, 10}
    },
    { {157, 233, 24, 57, 0},
      {131, 46, 150, 86, 8},
      {9, 193, 137, 123, 9},
      {22, 166, 151, 76, 1}
    },
    { {53, 118, 238, 158, 0},
      {233, 237, 172, 96, 7},
      {7, 151, 118, 234, 12},
      {224, 99, 91, 121, 7}
    },
    { {232, 208, 13, 33, 0},
      {82, 5, 54, 134, 0},
      {8, 75, 0, 177, 7},
      {6, 22, 202, 4, 10}
    },
    { {112, 155, 90, 149, 12},
      {104, 74, 253, 148, 10},
      {58, 149, 173, 144, 14},
      {82, 155, 245, 33, 6}
    },
    { {169, 2, 42, 49, 4},
      {2, 64, 238, 70, 4},
      {40, 197, 68, 9, 5},
      {38, 39, 112, 36, 0}
    },
    { {160, 62, 162, 185, 12},
      {65, 224, 235, 38, 14},
      {57, 212, 87, 192, 5},
      {118, 77, 112, 120, 2}
    },
    { {30, 88, 158, 197, 14},
      {202, 221, 69, 29, 10},
      {122, 55, 145, 167, 8},
      {91, 138, 43, 181, 3}
    },
    { {116, 71, 14, 237, 13},
      {140, 111, 111, 172, 2},
      {187, 119, 14, 34, 14},
      {67, 95, 111, 99, 1}
    },
    { {54, 189, 68, 21, 9},
      {237, 11, 177, 37, 8},
      {154, 130, 43, 214, 12},
      {26, 72, 221, 11, 7}
    },
    { {183, 248, 210, 81, 6},
      {225, 220, 240, 95, 8},
      {104, 164, 177, 254, 13},
      {31, 160, 243, 184, 7}
    },
    { {220, 222, 75, 250, 5},
      {246, 108, 222, 170, 11},
      {165, 253, 39, 179, 11},
      {213, 87, 179, 102, 15}
    },
    { {158, 218, 153, 195, 0},
      {210, 140, 28, 31, 11},
      {12, 57, 149, 183, 9},
      {223, 131, 131, 20, 11}
    },
    { {143, 251, 152, 168, 11},
      {199, 182, 31, 83, 10},
      {209, 81, 157, 255, 1},
      {92, 175, 134, 222, 3}
    },
    { {154, 199, 119, 231, 1},
      {62, 79, 26, 63, 7},
      {142, 126, 238, 53, 9},
      {239, 197, 143, 39, 12}
    },
    { {141, 10, 83, 203, 3},
      {182, 112, 8, 94, 11},
      {205, 60, 165, 11, 1},
      {215, 161, 0, 230, 13}
    },
    { {201, 237, 29, 206, 13},
      {31, 39, 85, 250, 11},
      {183, 59, 139, 121, 3},
      {213, 250, 174, 79, 8}
    },
    { {194, 56, 83, 94, 10},
      {121, 112, 129, 155, 9},
      {87, 172, 161, 196, 3},
      {157, 152, 16, 233, 14}
    },
    { {238, 74, 166, 67, 7},
      {134, 213, 104, 143, 13},
      {236, 38, 85, 39, 7},
      {191, 17, 106, 182, 1}
    },
    { {205, 131, 69, 161, 14},
      {178, 19, 91, 198, 2},
      {120, 90, 44, 27, 3},
      {70, 61, 172, 132, 13}
    },
    { {90, 25, 224, 90, 14},
      {98, 186, 193, 137, 13},
      {117, 160, 121, 133, 10},
      {185, 24, 53, 212, 6}
    },
    { {150, 215, 128, 210, 5},
      {196, 142, 216, 43, 3},
      {164, 176, 30, 182, 9},
      {205, 65, 183, 18, 3}
    },
    { {226, 200, 133, 24, 5},
      {20, 165, 240, 131, 8},
      {161, 138, 17, 52, 7},
      {28, 16, 250, 82, 8}
    },
    { {197, 49, 79, 141, 11},
      {253, 115, 5, 198, 2},
      {219, 31, 40, 202, 3},
      {70, 58, 12, 235, 15}
    },
    { {164, 124, 55, 238, 1},
      {221, 101, 34, 58, 15},
      {135, 126, 195, 226, 5},
      {245, 196, 74, 107, 11}
    },
    { {29, 130, 198, 248, 12},
      {162, 233, 219, 72, 2},
      {49, 246, 52, 27, 8},
      {65, 45, 185, 116, 5}
    },
    { {50, 92, 92, 108, 8},
      {104, 45, 39, 57, 8},
      {19, 99, 163, 164, 12},
      {25, 206, 75, 65, 6}
    },
    { {93, 157, 27, 174, 0},
      {218, 106, 22, 240, 11},
      {7, 93, 139, 155, 10},
      {208, 246, 133, 101, 11}
    },
    { {149, 73, 224, 117, 14},
      {168, 158, 195, 78, 12},
      {122, 224, 121, 42, 9},
      {55, 44, 55, 145, 5}
    },
    { {138, 104, 69, 193, 5},
      {55, 5, 64, 15, 10},
      {168, 58, 33, 101, 1},
      {95, 0, 42, 14, 12}
    },
    { {173, 249, 241, 15, 14},
      {251, 182, 113, 86, 13},
      {127, 8, 249, 251, 5},
      {182, 168, 230, 221, 15}
    },
    { {149, 206, 243, 11, 0},
      {176, 236, 24, 118, 13},
      {13, 12, 247, 58, 9},
      {182, 225, 131, 112, 13}
    },
    { {113, 147, 59, 32, 13},
      {84, 74, 127, 208, 4},
      {176, 77, 204, 152, 14},
      {32, 191, 229, 34, 10}
    },
    { {31, 13, 227, 166, 15},
      {190, 218, 67, 97, 15},
      {246, 92, 123, 15, 8},
      {248, 108, 37, 183, 13}
    },
    { {89, 111, 91, 186, 6},
      {51, 126, 206, 240, 11},
      {101, 221, 175, 105, 10},
      {208, 247, 55, 236, 12}
    },
    { {205, 97, 217, 243, 4},
      {179, 134, 198, 222, 3},
      {44, 249, 184, 107, 3},
      {199, 182, 54, 28, 13}
    },
    { {164, 120, 173, 50, 8},
      {209, 133, 167, 2, 13},
      {20, 203, 81, 226, 5},
      {180, 14, 90, 24, 11}
    },
    { {218, 222, 64, 18, 15},
      {102, 28, 217, 163, 9},
      {244, 128, 39, 181, 11},
      {156, 89, 179, 134, 6}
    },
    { {209, 87, 100, 55, 0},
      {104, 15, 138, 230, 5},
      {14, 194, 110, 168, 11},
      {166, 117, 31, 1, 6}
    },
    { {58, 43, 245, 210, 2},
      {51, 155, 168, 25, 15},
      {68, 186, 253, 69, 12},
      {249, 129, 93, 156, 12}
    },
    { {35, 3, 227, 72, 13},
      {52, 226, 105, 73, 4},
      {177, 44, 124, 12, 4},
      {41, 41, 100, 114, 12}
    },
    { {65, 185, 191, 154, 9},
      {85, 227, 149, 208, 15},
      {149, 159, 217, 216, 2},
      {240, 186, 156, 122, 10}
    },
    { {90, 78, 147, 238, 3},
      {30, 252, 10, 185, 11},
      {199, 124, 151, 37, 10},
      {217, 213, 3, 247, 8}
    },
    { {70, 43, 68, 81, 2},
      {161, 19, 136, 141, 8},
      {72, 162, 45, 70, 2},
      {27, 17, 28, 136, 5}
    },
    { {78, 1, 113, 156, 1},
      {190, 34, 128, 145, 6},
      {131, 152, 232, 7, 2},
      {104, 144, 20, 71, 13}
    },
    { {115, 160, 217, 19, 7},
      {53, 152, 244, 213, 1},
      {236, 137, 176, 92, 14},
      {138, 178, 241, 154, 12}
    },
    { {147, 43, 65, 109, 13},
      {61, 42, 75, 79, 8},
      {187, 104, 45, 76, 9},
      {31, 45, 37, 75, 12}
    },
    { {255, 75, 97, 90, 7},
      {182, 62, 232, 203, 13},
      {229, 168, 109, 47, 15},
      {189, 49, 119, 198, 13}
    },
    { {44, 136, 15, 26, 12},
      {146, 97, 245, 0, 9},
      {53, 143, 1, 19, 4},
      {144, 10, 248, 100, 9}
    },
    { {111, 59, 182, 170, 1},
      {199, 227, 42, 209, 15},
      {133, 86, 221, 207, 6},
      {248, 181, 76, 126, 3}
    },
    { {71, 109, 33, 174, 6},
      {153, 54, 66, 225, 15},
      {103, 88, 75, 110, 2},
      {248, 116, 38, 201, 9}
    },
    { {249, 194, 144, 104, 8},
      {2, 172, 59, 218, 0},
      {17, 96, 148, 57, 15},
      {5, 189, 195, 84, 0}
    },
    { {118, 129, 17, 33, 0},
      {144, 10, 50, 149, 0},
      {8, 72, 136, 22, 14},
      {10, 148, 197, 0, 9}
    },
    { {98, 52, 88, 245, 5},
      {109, 0, 230, 189, 2},
      {170, 241, 162, 196, 6},
      {75, 214, 112, 11, 6}
    },
    { {243, 25, 25, 225, 15},
      {84, 26, 103, 223, 10},
      {248, 121, 137, 140, 15},
      {95, 190, 101, 130, 10}
    },
    { {239, 71, 167, 221, 1},
      {158, 231, 168, 239, 6},
      {139, 190, 94, 47, 7},
      {111, 113, 94, 119, 9}
    },
    { {198, 199, 119, 128, 8},
      {176, 71, 25, 179, 6},
      {16, 30, 238, 54, 3},
      {108, 217, 142, 32, 13}
    },
    { {238, 201, 111, 9, 6},
      {178, 119, 116, 135, 12},
      {105, 15, 105, 55, 7},
      {62, 18, 238, 228, 13}
    },
    { {155, 60, 155, 218, 15},
      {87, 248, 197, 123, 11},
      {245, 189, 147, 205, 9},
      {221, 234, 49, 254, 10}
    },
    { {111, 212, 50, 87, 3},
      {206, 84, 176, 253, 5},
      {206, 164, 194, 191, 6},
      {171, 240, 210, 167, 3}
    },
    { {96, 231, 208, 170, 9},
      {37, 166, 59, 176, 3},
      {149, 80, 190, 112, 6},
      {192, 221, 198, 90, 4}
    },
    { {6, 81, 205, 163, 4},
      {240, 135, 70, 5, 3},
      {44, 91, 56, 166, 0},
      {202, 6, 46, 16, 15}
    },
    { {226, 172, 192, 78, 7},
      {45, 176, 112, 171, 9},
      {231, 32, 51, 84, 7},
      {157, 80, 224, 219, 4}
    },
    { {148, 96, 209, 162, 2},
      {177, 156, 2, 18, 3},
      {68, 88, 176, 98, 9},
      {196, 132, 3, 152, 13}
    },
    { {130, 183, 155, 23, 7},
      {93, 210, 220, 55, 1},
      {238, 141, 158, 212, 1},
      {142, 195, 180, 187, 10}
    },
    { {78, 177, 137, 244, 11},
      {223, 146, 151, 137, 2},
      {210, 249, 24, 215, 2},
      {73, 30, 148, 159, 11}
    },
    { {203, 249, 195, 177, 14},
      {115, 214, 211, 199, 10},
      {120, 220, 57, 253, 3},
      {94, 60, 182, 188, 14}
    },
    { {111, 108, 100, 132, 1},
      {175, 5, 32, 225, 14},
      {130, 18, 99, 111, 6},
      {120, 112, 74, 15, 5}
    },
    { {229, 78, 80, 162, 14},
      {160, 20, 107, 242, 11},
      {116, 80, 167, 42, 7},
      {212, 253, 98, 128, 5}
    },
    { {99, 47, 48, 242, 3},
      {5, 18, 170, 249, 15},
      {196, 240, 207, 76, 6},
      {249, 245, 84, 138, 0}
    },
    { {179, 187, 61, 230, 12},
      {89, 11, 127, 91, 15},
      {54, 123, 205, 220, 13},
      {253, 175, 237, 9, 10}
    },
    { {99, 49, 245, 56, 3},
      {117, 179, 162, 209, 4},
      {193, 202, 248, 204, 6},
      {40, 180, 92, 218, 14}
    },
    { {188, 161, 218, 254, 5},
      {175, 234, 246, 26, 3},
      {167, 245, 184, 83, 13},
      {197, 134, 245, 127, 5}
    },
    { {209, 129, 108, 171, 9},
      {36, 43, 23, 198, 7},
      {157, 83, 104, 24, 11},
      {230, 62, 141, 66, 4}
    },
    { {13, 132, 7, 172, 10},
      {154, 113, 19, 96, 2},
      {83, 94, 2, 27, 0},
      {64, 108, 136, 229, 9}
    },
    { {121, 78, 29, 126, 5},
      {30, 45, 238, 248, 9},
      {167, 235, 135, 41, 14},
      {145, 247, 123, 71, 8}
    },
    { {146, 222, 101, 68, 2},
      {120, 29, 24, 43, 12},
      {66, 42, 103, 180, 9},
      {61, 65, 139, 129, 14}
    },
    { {101, 20, 151, 148, 9},
      {220, 193, 161, 240, 2},
      {146, 158, 146, 138, 6},
      {64, 248, 88, 51, 11}
    },
    { {51, 247, 81, 113, 2},
      {113, 30, 186, 125, 0},
      {72, 232, 174, 252, 12},
      {11, 229, 215, 136, 14}
    },
    { {60, 156, 233, 25, 5},
      {246, 168, 244, 36, 12},
      {169, 137, 115, 147, 12},
      {50, 66, 241, 86, 15}
    },
    { {126, 203, 129, 215, 10},
      {154, 158, 185, 141, 11},
      {94, 184, 29, 55, 14},
      {219, 25, 215, 149, 9}
    },
    { {237, 225, 64, 23, 8},
      {171, 6, 177, 198, 1},
      {30, 128, 40, 123, 7},
      {134, 56, 214, 13, 5}
    },
    { {192, 155, 14, 226, 12},
      {64, 67, 95, 138, 11},
      {52, 119, 13, 144, 3},
      {213, 31, 172, 32, 2}
    },
    { {20, 230, 31, 119, 4},
      {153, 77, 222, 60, 1},
      {46, 239, 134, 114, 8},
      {131, 199, 187, 41, 9}
    },
    { {109, 51, 102, 73, 10},
      {227, 115, 41, 204, 4},
      {89, 38, 108, 203, 6},
      {35, 57, 76, 236, 7}
    },
    { {134, 93, 208, 136, 0},
      {224, 166, 0, 51, 10},
      {1, 16, 187, 166, 1},
      {92, 192, 6, 80, 7}
    },
    { {247, 33, 75, 251, 8},
      {177, 106, 167, 207, 3},
      {29, 253, 40, 78, 15},
      {207, 62, 85, 104, 13}
    },
    { {216, 185, 194, 226, 2},
      {99, 218, 18, 138, 11},
      {68, 116, 57, 209, 11},
      {213, 20, 133, 188, 6}
    },
    { {153, 167, 7, 55, 15},
      {31, 91, 219, 102, 1},
      {254, 206, 14, 89, 9},
      {134, 109, 189, 175, 8}
    },
    { {192, 236, 249, 12, 9},
      {61, 164, 21, 178, 12},
      {147, 9, 243, 112, 3},
      {52, 218, 130, 91, 12}
    },
    { {187, 139, 38, 224, 1},
      {6, 75, 58, 75, 14},
      {128, 118, 77, 29, 13},
      {125, 37, 205, 38, 0}
    },
    { {5, 103, 143, 202, 12},
      {145, 231, 77, 104, 3},
      {53, 63, 30, 106, 0},
      {193, 107, 46, 120, 9}
    },
    { {51, 180, 131, 131, 4},
      {81, 200, 112, 101, 3},
      {44, 28, 18, 220, 12},
      {202, 96, 225, 56, 10}
    },
    { {235, 157, 132, 184, 3},
      {70, 179, 178, 227, 10},
      {193, 210, 27, 157, 7},
      {92, 116, 220, 214, 2}
    },
    { {124, 162, 119, 234, 2},
      {179, 121, 58, 152, 7},
      {69, 126, 228, 83, 14},
      {225, 149, 201, 236, 13}
    },
    { {2, 175, 95, 102, 11},
      {61, 83, 31, 57, 9},
      {214, 111, 175, 84, 0},
      {153, 207, 140, 171, 12}
    },
    { {74, 103, 140, 18, 2},
      {3, 151, 140, 161, 1},
      {68, 131, 30, 101, 2},
      {136, 83, 30, 156, 0}
    },
    { {18, 21, 146, 191, 3},
      {76, 250, 130, 53, 3},
      {207, 212, 154, 132, 8},
      {202, 196, 21, 243, 2}
    },
    { {204, 130, 180, 140, 9},
      {142, 161, 25, 146, 6},
      {147, 18, 212, 19, 3},
      {100, 153, 136, 87, 1}
    },
    { {119, 203, 233, 15, 12},
      {184, 174, 125, 197, 13},
      {63, 9, 125, 62, 14},
      {186, 59, 231, 81, 13}
    },
    { {94, 247, 215, 147, 1},
      {247, 207, 152, 181, 3},
      {140, 158, 190, 247, 10},
      {202, 209, 159, 62, 15}
    },
    { {17, 121, 71, 153, 5},
      {117, 111, 192, 68, 10},
      {169, 158, 41, 232, 8},
      {82, 32, 63, 106, 14}
    },
    { {52, 77, 95, 151, 0},
      {184, 79, 164, 52, 11},
      {14, 159, 171, 34, 12},
      {210, 194, 95, 33, 13}
    },
    { {223, 47, 18, 231, 2},
      {139, 90, 10, 255, 11},
      {78, 116, 143, 79, 11},
      {223, 245, 5, 173, 1}
    },
    { {17, 66, 119, 179, 7},
      {52, 93, 202, 84, 7},
      {236, 222, 228, 40, 8},
      {226, 165, 59, 162, 12}
    },
    { {11, 143, 167, 61, 5},
      {30, 227, 218, 101, 12},
      {171, 206, 95, 29, 0},
      {58, 101, 188, 119, 8}
    },
    { {54, 225, 111, 66, 3},
      {181, 95, 52, 9, 5},
      {196, 47, 104, 118, 12},
      {169, 2, 207, 170, 13}
    },
    { {5, 116, 2, 183, 14},
      {201, 84, 195, 100, 3},
      {126, 212, 2, 234, 0},
      {194, 108, 50, 169, 3}
    },
    { {195, 66, 66, 58, 7},
      {36, 116, 202, 195, 1},
      {229, 196, 36, 44, 3},
      {140, 53, 50, 226, 4}
    },
    { {117, 140, 52, 218, 11},
      {132, 57, 177, 248, 15},
      {213, 178, 195, 26, 14},
      {241, 248, 217, 194, 1}
    },
    { {103, 220, 97, 97, 4},
      {240, 4, 114, 237, 12},
      {40, 104, 99, 190, 6},
      {59, 116, 226, 0, 15}
    },
    { {41, 178, 34, 6, 8},
      {75, 64, 57, 64, 5},
      {22, 4, 68, 217, 4},
      {160, 41, 192, 45, 2}
    },
    { {113, 1, 175, 15, 10},
      {24, 251, 37, 196, 5},
      {95, 15, 88, 8, 14},
      {162, 58, 77, 241, 8}
    },
    { {84, 210, 43, 199, 5},
      {220, 76, 92, 140, 7},
      {174, 61, 68, 178, 10},
      {227, 19, 163, 35, 11}
    },
    { {144, 139, 163, 175, 9},
      {28, 234, 27, 6, 15},
      {159, 92, 93, 16, 9},
      {246, 13, 133, 115, 8}
    },
    { {235, 104, 201, 181, 1},
      {63, 132, 166, 199, 10},
      {138, 217, 49, 109, 7},
      {94, 54, 82, 31, 12}
    },
    { {38, 5, 212, 66, 12},
      {160, 131, 97, 57, 1},
      {52, 34, 186, 6, 4},
      {137, 200, 108, 16, 5}
    },
    { {106, 184, 139, 127, 4},
      {91, 224, 246, 141, 9},
      {47, 237, 17, 213, 6},
      {155, 22, 240, 125, 10}
    },
    { {32, 124, 118, 87, 13},
      {109, 69, 225, 60, 13},
      {190, 166, 227, 224, 4},
      {179, 200, 122, 43, 6}
    },
    { {55, 121, 64, 210, 0},
      {225, 14, 160, 73, 11},
      {4, 176, 41, 238, 12},
      {217, 32, 87, 8, 7}
    },
    { {172, 46, 183, 123, 6},
      {147, 241, 234, 62, 13},
      {109, 238, 215, 67, 5},
      {183, 197, 120, 252, 9}
    },
    { {62, 135, 49, 142, 12},
      {154, 42, 121, 49, 7},
      {55, 24, 206, 23, 12},
      {232, 201, 229, 69, 9}
    },
    { {71, 175, 234, 26, 10},
      {161, 242, 157, 225, 13},
      {85, 133, 127, 94, 2},
      {184, 123, 148, 248, 5}
    },
    { {70, 222, 53, 183, 4},
      {216, 5, 218, 181, 15},
      {46, 218, 199, 182, 2},
      {250, 213, 186, 1, 11}
    },
    { {72, 163, 107, 161, 0},
      {51, 66, 30, 132, 6},
      {8, 93, 108, 81, 2},
      {98, 23, 132, 44, 12}
    },
    { {240, 113, 21, 54, 11},
      {93, 31, 163, 146, 1},
      {214, 202, 136, 224, 15},
      {132, 156, 95, 139, 10}
    },
    { {150, 63, 135, 237, 4},
      {217, 235, 74, 47, 10},
      {43, 126, 31, 198, 9},
      {95, 69, 45, 121, 11}
    },
    { {216, 58, 185, 150, 5},
      {95, 136, 204, 146, 15},
      {166, 153, 213, 193, 11},
      {244, 147, 49, 31, 10}
    },
    { {123, 46, 181, 160, 15},
      {23, 153, 107, 241, 14},
      {240, 90, 215, 77, 14},
      {120, 253, 105, 158, 8}
    },
    { {182, 173, 250, 243, 2},
      {161, 218, 182, 63, 15},
      {76, 245, 251, 86, 13},
      {255, 198, 213, 184, 5}
    },
    { {135, 224, 122, 78, 10},
      {169, 116, 21, 91, 5},
      {87, 37, 224, 126, 1},
      {173, 170, 130, 233, 5}
    },
    { {123, 69, 24, 244, 14},
      {10, 30, 231, 249, 2},
      {114, 241, 138, 45, 14},
      {73, 254, 119, 133, 0}
    },
    { {144, 100, 163, 115, 8},
      {17, 204, 131, 46, 5},
      {28, 236, 82, 96, 9},
      {167, 76, 19, 56, 8}
    },
    { {60, 36, 44, 161, 2},
      {131, 25, 38, 36, 6},
      {72, 83, 66, 67, 12},
      {98, 70, 73, 140, 1}
    },
    { {150, 178, 92, 74, 4},
      {225, 41, 92, 27, 1},
      {37, 35, 164, 214, 9},
      {141, 131, 169, 72, 7}
    },
    { {139, 1, 216, 58, 12},
      {34, 162, 199, 83, 1},
      {53, 193, 184, 13, 1},
      {140, 174, 52, 84, 4}
    },
    { {229, 28, 206, 178, 5},
      {228, 193, 230, 226, 11},
      {164, 215, 51, 138, 7},
      {212, 118, 120, 50, 7}
    },
    { {72, 15, 88, 219, 0},
      {34, 34, 140, 188, 11},
      {13, 177, 175, 1, 2},
      {211, 211, 20, 68, 4}
    },
    { {186, 226, 149, 134, 7},
      {31, 157, 120, 19, 3},
      {230, 26, 148, 117, 13},
      {204, 129, 235, 159, 8}
    },
    { {137, 252, 15, 186, 0},
      {83, 101, 150, 98, 11},
      {5, 223, 3, 249, 1},
      {212, 102, 154, 108, 10}
    },
    { {221, 6, 71, 94, 9},
      {190, 105, 137, 234, 1},
      {151, 174, 38, 11, 11},
      {133, 121, 25, 103, 13}
    },
    { {247, 254, 70, 180, 12},
      {233, 77, 251, 227, 10},
      {50, 214, 39, 254, 15},
      {92, 125, 251, 41, 7}
    },
    { {67, 206, 240, 38, 13},
      {44, 132, 91, 241, 13},
      {182, 64, 247, 60, 2},
      {184, 253, 162, 19, 4}
    },
    { {99, 147, 88, 90, 2},
      {96, 50, 188, 217, 1},
      {69, 161, 172, 156, 6},
      {137, 179, 212, 192, 6}
    },
    { {135, 180, 4, 69, 6},
      {201, 17, 80, 111, 0},
      {106, 34, 2, 222, 1},
      {15, 96, 168, 137, 3}
    },
    { {79, 164, 124, 190, 10},
      {171, 49, 151, 241, 7},
      {87, 211, 226, 95, 2},
      {232, 254, 152, 205, 5}
    },
    { {118, 25, 249, 85, 10},
      {248, 154, 165, 157, 12},
      {90, 169, 249, 134, 14},
      {59, 154, 85, 145, 15}
    },
    { {81, 185, 161, 51, 0},
      {81, 138, 146, 196, 13},
      {12, 200, 89, 216, 10},
      {178, 52, 149, 24, 10}
    },
    { {21, 51, 99, 124, 7},
      {253, 122, 202, 72, 4},
      {227, 236, 108, 202, 8},
      {33, 37, 53, 235, 15}
    },
    { {117, 135, 169, 114, 10},
      {144, 154, 191, 232, 5},
      {84, 233, 94, 26, 14},
      {161, 127, 213, 144, 9}
    },
    { {96, 222, 17, 98, 2},
      {80, 20, 58, 184, 9},
      {68, 104, 135, 176, 6},
      {145, 213, 194, 128, 10}
    },
    { {125, 76, 230, 138, 14},
      {162, 253, 97, 224, 15},
      {117, 22, 115, 43, 14},
      {240, 120, 107, 244, 5}
    },
    { {175, 19, 94, 129, 12},
      {226, 67, 109, 87, 2},
      {56, 23, 172, 143, 5},
      {78, 171, 108, 36, 7}
    },
    { {49, 201, 182, 71, 12},
      {8, 207, 113, 92, 13},
      {62, 38, 217, 56, 12},
      {179, 168, 239, 49, 0}
    },
    { {172, 67, 216, 21, 5},
      {174, 134, 236, 22, 0},
      {170, 129, 188, 35, 5},
      {6, 131, 118, 23, 5}
    },
    { {216, 83, 225, 207, 11},
      {126, 190, 9, 142, 7},
      {223, 56, 124, 161, 11},
      {231, 25, 7, 215, 14}
    },
    { {79, 205, 197, 207, 0},
      {186, 167, 16, 237, 11},
      {15, 58, 59, 63, 2},
      {219, 112, 142, 85, 13}
    },
    { {238, 105, 111, 210, 13},
      {183, 71, 229, 139, 15},
      {180, 191, 105, 103, 7},
      {253, 26, 126, 46, 13}
    },
    { {183, 22, 24, 71, 5},
      {204, 8, 108, 127, 1},
      {174, 33, 134, 142, 13},
      {143, 227, 97, 3, 3}
    },
    { {185, 180, 184, 120, 1},
      {71, 168, 182, 122, 4},
      {129, 225, 210, 217, 13},
      {37, 230, 209, 94, 2}
    },
    { {50, 133, 121, 80, 5},
      {52, 10, 244, 57, 4},
      {160, 169, 234, 20, 12},
      {41, 194, 245, 2, 12}
    },
    { {149, 233, 71, 119, 1},
      {189, 79, 146, 78, 9},
      {142, 238, 41, 122, 9},
      {151, 36, 159, 43, 13}
    },
    { {154, 160, 48, 46, 2},
      {11, 56, 18, 19, 5},
      {71, 64, 192, 85, 9},
      {172, 132, 129, 205, 0}
    },
    { {236, 24, 40, 209, 7},
      {198, 16, 228, 142, 14},
      {232, 177, 65, 131, 7},
      {119, 18, 112, 134, 3}
    },
    { {188, 80, 121, 99, 5},
      {246, 12, 102, 30, 5},
      {172, 105, 224, 163, 13},
      {167, 134, 99, 6, 15}
    },
    { {135, 133, 193, 183, 9},
      {188, 130, 147, 103, 3},
      {158, 216, 58, 30, 1},
      {206, 108, 148, 19, 13}
    },
    { {211, 122, 173, 183, 10},
      {89, 157, 143, 199, 15},
      {94, 219, 85, 236, 11},
      {254, 63, 27, 153, 10}
    },
    { {129, 127, 2, 206, 0},
      {73, 102, 8, 106, 11},
      {7, 52, 15, 232, 1},
      {213, 97, 6, 105, 2}
    },
    { {253, 99, 223, 176, 10},
      {179, 223, 175, 210, 2},
      {80, 223, 188, 107, 15},
      {68, 191, 95, 188, 13}
    },
    { {81, 10, 193, 175, 2},
      {56, 184, 10, 196, 11},
      {79, 88, 53, 8, 10},
      {210, 53, 1, 209, 12}
    },
    { {61, 30, 223, 141, 3},
      {254, 249, 44, 116, 10},
      {203, 31, 183, 139, 12},
      {82, 227, 73, 247, 15}
    },
    { {244, 183, 58, 64, 2},
      {193, 90, 60, 186, 4},
      {64, 37, 206, 210, 15},
      {37, 211, 197, 168, 3}
    },
    { {55, 215, 137, 233, 12},
      {208, 174, 127, 109, 2},
      {57, 121, 30, 190, 12},
      {75, 111, 231, 80, 11}
    },
    { {143, 17, 199, 124, 10},
      {250, 243, 131, 75, 0},
      {83, 238, 56, 143, 1},
      {13, 44, 28, 245, 15}
    },
    { {196, 189, 3, 107, 14},
      {209, 114, 83, 174, 9},
      {125, 108, 11, 210, 3},
      {151, 92, 164, 232, 11}
    },
    { {125, 250, 153, 92, 9},
      {223, 172, 189, 216, 8},
      {147, 169, 149, 251, 14},
      {17, 187, 211, 95, 11}
    },
    { {123, 20, 232, 37, 1},
      {110, 136, 38, 229, 4},
      {138, 65, 114, 141, 14},
      {42, 118, 65, 23, 6}
    },
    { {202, 7, 14, 127, 4},
      {10, 99, 206, 175, 1},
      {47, 231, 14, 5, 3},
      {143, 87, 60, 101, 0}
    },
    { {92, 34, 219, 205, 5},
      {191, 232, 76, 156, 2},
      {171, 61, 180, 67, 10},
      {67, 147, 33, 127, 13}
    },
    { {160, 219, 114, 108, 8},
      {104, 102, 59, 26, 12},
      {19, 100, 237, 176, 5},
      {53, 141, 198, 97, 6}
    },
    { {107, 119, 118, 222, 2},
      {107, 119, 168, 249, 7},
      {71, 182, 238, 237, 6},
      {233, 241, 94, 237, 6}
    },
    { {128, 245, 142, 116, 8},
      {73, 199, 151, 42, 0},
      {18, 231, 26, 240, 1},
      {5, 78, 158, 57, 2}
    },
    { {225, 172, 31, 244, 10},
      {25, 81, 183, 250, 10},
      {82, 255, 131, 88, 7},
      {85, 254, 216, 169, 8}
    },
    { {27, 186, 234, 133, 12},
      {107, 200, 93, 69, 14},
      {58, 21, 117, 221, 8},
      {122, 43, 161, 61, 6}
    },
    { {38, 172, 85, 127, 12},
      {185, 33, 243, 61, 9},
      {63, 234, 163, 86, 4},
      {155, 204, 248, 73, 13}
    },
    { {229, 209, 239, 53, 7},
      {252, 215, 246, 198, 4},
      {234, 207, 120, 186, 7},
      {38, 54, 254, 179, 15}
    },
    { {31, 41, 212, 225, 0},
      {163, 139, 2, 93, 10},
      {8, 114, 185, 79, 8},
      {91, 164, 13, 28, 5}
    },
    { {69, 159, 187, 21, 11},
      {220, 210, 157, 244, 12},
      {218, 141, 223, 154, 2},
      {50, 251, 148, 179, 11}
    },
    { {212, 176, 73, 85, 9},
      {253, 8, 149, 142, 0},
      {154, 169, 32, 210, 11},
      {7, 26, 145, 11, 15}
    },
    { {108, 183, 10, 207, 8},
      {203, 98, 61, 172, 3},
      {31, 53, 14, 211, 6},
      {195, 91, 196, 109, 3}
    },
    { {248, 97, 122, 81, 8},
      {35, 78, 165, 158, 4},
      {24, 165, 232, 97, 15},
      {39, 154, 87, 44, 4}
    },
    { {195, 145, 218, 195, 4},
      {96, 194, 84, 223, 3},
      {44, 53, 184, 156, 3},
      {207, 178, 164, 48, 6}
    },
    { {81, 35, 94, 44, 12},
      {41, 107, 79, 208, 0},
      {51, 71, 172, 72, 10},
      {0, 191, 45, 105, 4}
    },
    { {116, 64, 124, 79, 14},
      {168, 61, 101, 156, 5},
      {127, 35, 224, 34, 14},
      {163, 154, 107, 193, 5}
    },
    { {223, 22, 9, 162, 4},
      {210, 8, 78, 227, 3},
      {36, 89, 6, 143, 11},
      {204, 119, 33, 4, 11}
    },
    { {206, 210, 125, 193, 7},
      {246, 21, 92, 159, 6},
      {232, 59, 228, 183, 3},
      {111, 147, 170, 134, 15}
    }
};

}//namespace
