/*
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%                                                                             %
%                                                                             %
%                 RRRR    AAA   N   N  DDDD    OOO   M   M                    %
%                 R   R  A   A  NN  N  D   D  O   O  MM MM                    %
%                 RRRR   AAAAA  N N N  D   D  O   O  M M M                    %
%                 R R    A   A  N  NN  D   D  O   O  M   M                    %
%                 R  R   A   A  N   N  DDDD    OOO   M   M                    %
%                                                                             %
%                                                                             %
%                   Methods to Generate Random Numbers                        %
%                                                                             %
%                                                                             %
%                             Software Design                                 %
%                               John Cristy                                   %
%                              December 2001                                  %
%                                                                             %
%                                                                             %
%  Copyright (C) 2003 ImageMagick Studio, a non-profit organization dedicated %
%  to making software imaging solutions freely available.                     %
%                                                                             %
%  Permission is hereby granted, free of charge, to any person obtaining a    %
%  copy of this software and associated documentation files ("ImageMagick"),  %
%  to deal in ImageMagick without restriction, including without limitation   %
%  the rights to use, copy, modify, merge, publish, distribute, sublicense,   %
%  and/or sell copies of ImageMagick, and to permit persons to whom the       %
%  ImageMagick is furnished to do so, subject to the following conditions:    %
%                                                                             %
%  The above copyright notice and this permission notice shall be included in %
%  all copies or substantial portions of ImageMagick.                         %
%                                                                             %
%  The software is provided "as is", without warranty of any kind, express or %
%  implied, including but not limited to the warranties of merchantability,   %
%  fitness for a particular purpose and noninfringement.  In no event shall   %
%  ImageMagick Studio be liable for any claim, damages or other liability,    %
%  whether in an action of contract, tort or otherwise, arising from, out of  %
%  or in connection with ImageMagick or the use or other dealings in          %
%  ImageMagick.                                                               %
%                                                                             %
%  Except as contained in this notice, the name of the ImageMagick Studio     %
%  shall not be used in advertising or otherwise to promote the sale, use or  %
%  other dealings in ImageMagick without prior written authorization from the %
%  ImageMagick Studio.                                                        %
%                                                                             %
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%
%
%
*/

/*
  Include declarations.
*/
#include "studio.h"
#include "signature.h"
#include "utility.h"

/*
  Global declarations.
*/
static SemaphoreInfo
  *random_semaphore = (SemaphoreInfo *) NULL;

static SignatureInfo
  *reservoir = (SignatureInfo *) NULL;

static unsigned long
  *roulette = (unsigned long *) NULL;

/*
  Forward declarations.
*/
static void
  InitializeRandomReservoir(void);

/*
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%                                                                             %
%                                                                             %
%                                                                             %
+   D e s t r o y R a n d o m R e s e r v i o r                               %
%                                                                             %
%                                                                             %
%                                                                             %
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%
%  DestroyRandomReservoir() deallocates memory associated with the random
%  reservoir.
%
%  The format of the DestroyRandomReservoir method is:
%
%      DestroyRandomReservoir(void)
%
%
*/
MagickExport void DestroyRandomReservoir(void)
{
  AcquireSemaphoreInfo(&random_semaphore);
  if (reservoir != (SignatureInfo *) NULL)
    {
      (void) memset(reservoir,0,sizeof(SignatureInfo));
      LiberateMemory((void **) &reservoir);
    }
  if (roulette != (unsigned long *) NULL)
    {
      (void) memset(roulette,0,sizeof(unsigned long));
      LiberateMemory((void **) &roulette);
    }
  DestroySemaphoreInfo(&random_semaphore);
}

/*
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%                                                                             %
%                                                                             %
%                                                                             %
%   D i s t i l l R a n d o m E v e n t                                       %
%                                                                             %
%                                                                             %
%                                                                             %
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%
%  DistillRandomEvent() distills randomness from an event and stores it int
%  the reservoir.  This method should be called before GetRandomKey() and it
%  should be called a number of times using different random events (e.g.
%  thread completion time, fine grained time-of-day clock in a tight loop,
%  keystroke timing, etc.) to build up sufficient randomness in the reservoir.
%
%  The format of the DistillRandomEvent method is:
%
%      DistillRandomEvent(const unsigned char *event,const size_t length)
%
%  A description of each parameter follows:
%
%    o event: A random event.
%
%    o length: The length of the event.
%
*/
MagickExport void DistillRandomEvent(const unsigned char *event,
  const size_t length)
{
  SignatureInfo
    digest_info;

  /*
    Distill a random event.
  */
  assert(event != (const unsigned char *) NULL);
  if ((reservoir == (SignatureInfo *) NULL) ||
      (roulette == (unsigned long *) NULL))
    {
      AcquireSemaphoreInfo(&random_semaphore);
      if (reservoir == (SignatureInfo *) NULL)
        reservoir=(SignatureInfo *) AcquireMemory(sizeof(SignatureInfo));
      if (roulette == (unsigned long *) NULL)
        roulette=(unsigned long *) AcquireMemory(sizeof(unsigned long));
      LiberateSemaphoreInfo(&random_semaphore);
      if ((reservoir == (SignatureInfo *) NULL) ||
          (roulette == (unsigned long *) NULL))
        MagickFatalError(ResourceLimitFatalError,"MemoryAllocationFailed",
          "UnableToDistillRandomEvent");
      (void) memset(reservoir,0,sizeof(SignatureInfo));
      (void) memset(roulette,0,sizeof(unsigned long));
    }
  AcquireSemaphoreInfo(&random_semaphore);
  GetSignatureInfo(&digest_info);
  UpdateSignature(&digest_info,(const unsigned char *) reservoir->digest,
    sizeof(reservoir->digest));
  UpdateSignature(&digest_info,event,length);
  FinalizeSignature(&digest_info);
  memcpy(reservoir->digest,digest_info.digest,sizeof(reservoir->digest));
  LiberateSemaphoreInfo(&random_semaphore);
}

/*
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%                                                                             %
%                                                                             %
%                                                                             %
%   G e t R a n d o m K e y                                                   %
%                                                                             %
%                                                                             %
%                                                                             %
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%
%  GetRandomKey() gets a random key from the reservoir.
%
%  The format of the GetRandomKey method is:
%
%      GetRandomKey(unsigned char *key,const size_t length)
%
%  A description of each parameter follows:
%
%    o key: The key.
%
%    o length: The key length.
%
%
*/
MagickExport void GetRandomKey(unsigned char *key,const size_t length)
{
  SignatureInfo
    digest_info;

  long
    n;

  assert(key != (unsigned char *) NULL);
  if ((roulette == (unsigned long *) NULL) ||
      (reservoir == (SignatureInfo *) NULL))
    InitializeRandomReservoir();
  AcquireSemaphoreInfo(&random_semaphore);
  for (n=length; n > 0; n-=sizeof(reservoir->digest))
  {
    GetSignatureInfo(&digest_info);
    UpdateSignature(&digest_info,(const unsigned char *) reservoir->digest,
      sizeof(reservoir->digest));
    UpdateSignature(&digest_info,(const unsigned char *) roulette,
      sizeof(roulette));
    FinalizeSignature(&digest_info);
    (*roulette)++;
    memcpy(key,digest_info.digest,
      n < (long) sizeof(reservoir->digest) ? n : sizeof(reservoir->digest));
    key+=sizeof(reservoir->digest);
  }
  LiberateSemaphoreInfo(&random_semaphore);
}

/*
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%                                                                             %
%                                                                             %
%                                                                             %
%   G e t R a n d o m V a l u e                                               %
%                                                                             %
%                                                                             %
%                                                                             %
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%
%  GetRandomValue() return a non-negative double-precision floating-point
%  value uniformly distributed over the interval [0.0, 1.0).
%
%  The format of the GetRandomValue method is:
%
%      double GetRandomValue(void)
%
%
*/
MagickExport double GetRandomValue(void)
{
  static long
    i = 8;

  static SignatureInfo
    digest_info;

  unsigned long
    key,
    range;

  if ((roulette == (unsigned long *) NULL) ||
      (reservoir == (SignatureInfo *) NULL))
    InitializeRandomReservoir();
  AcquireSemaphoreInfo(&random_semaphore);
  range=(unsigned long) (~0);
  do
  {
    if (i == 8)
      {
        GetSignatureInfo(&digest_info);
        UpdateSignature(&digest_info,(const unsigned char *) reservoir->digest,
          sizeof(reservoir->digest));
        UpdateSignature(&digest_info,(const unsigned char *) roulette,
          sizeof(roulette));
        FinalizeSignature(&digest_info);
        (*roulette)++;
        i=0;
      }
    key=digest_info.digest[i++];
  } while (key == range);
  LiberateSemaphoreInfo(&random_semaphore);
  return((double) key/range);
}

/*
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%                                                                             %
%                                                                             %
%                                                                             %
+   I n i t i a l i z e R a n d o m R e s e r v i o r                         %
%                                                                             %
%                                                                             %
%                                                                             %
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%
%  InitializeRandomReservoir() initializes the random reservoir with entropy.
%
%  The format of the InitializeRandomReservoir method is:
%
%      InitializeRandomReservoir(void)
%
%
*/
static void InitializeRandomReservoir(void)
{
  char
    filename[MaxTextExtent];

  int
    file;

  long
    pid;

  time_t
    nanoseconds,
    seconds;

  unsigned char
    random[MaxTextExtent];

  /*
    Initialize random reservoir.
  */
  seconds=time((time_t *) 0);
  nanoseconds=0;
#if defined(HAVE_GETTIMEOFDAY)
  {
    struct timeval
      timer;

    if (gettimeofday(&timer,0) == 0)
      {
        seconds=timer.tv_sec;
        nanoseconds=1000*timer.tv_usec;
      }
  }
#endif
#if defined(HAVE_CLOCK_GETTIME) && defined(CLOCK_HIGHRES)
  {
    struct timespec
      timer;

    if (clock_gettime(CLOCK_HIGHRES,&timer) == 0)
      {
        seconds=timer.tv_sec;
        nanseconds=timer.tv_nsec;
      }
  }
#endif
  DistillRandomEvent((const unsigned char *) &seconds,sizeof(time_t));
  DistillRandomEvent((const unsigned char *) &nanoseconds,sizeof(time_t));
  nanoseconds=0;
#if defined(HAVE_TIMES)
  {
    struct tms
      timer;

    (void) times(&timer);
    nanoseconds=timer.tms_utime+timer.tms_stime;
  }
#else
#if defined(WIN32)
  nanoseconds=NTElapsedTime()+NTUserTime();
#else
  nanoseconds=clock();
#endif
#endif
  DistillRandomEvent((const unsigned char *) &nanoseconds,sizeof(time_t));
  pid=getpid();
  DistillRandomEvent((const unsigned char *) &pid,sizeof(long));
  DistillRandomEvent((const unsigned char *) &roulette,sizeof(unsigned long *));
  (void) strcpy(filename,"magic");
  (void) tmpnam(filename);
  DistillRandomEvent((const unsigned char *) filename,MaxTextExtent);
  file=open("/dev/random",O_RDONLY | O_BINARY,0777);
  if (file == -1)
    return;
  (void) read(file,random,MaxTextExtent);
  (void) close(file);
  DistillRandomEvent(random,MaxTextExtent);
}
